/*
	wavio.cpp
	Routines for reading and writing wave files
	AML - 3-19-01
*/

#include <windows.h>
#include <stdio.h>
#include "wavio.h"

#define MM_OK     0
#define MM_ERROR -1

void DispWAVInfo(WAVEFORMATEX* fmt)
{
	printf("Channels: %i\n",fmt->nChannels);
	printf("SamplesPerSec: %i\n",fmt->nSamplesPerSec);
	printf("BitsPerSample: %i\n",fmt->wBitsPerSample);

	if (fmt->wFormatTag == WAVE_FORMAT_PCM)
		printf("PCM Format waveform.\n");
	else
		printf("Non-PCM Format waveform.\n");
}

bool SaveWAV(char* Filename,WAVEFORMATEX* fmt,void* data,int size, SamplerChunk* schunk, int schunkSize)
{
	HMMIO hmmio;	// handle to the multi-media file (our .wav)

	// Open the wave file
	hmmio = mmioOpen(Filename,NULL,MMIO_WRITE|MMIO_CREATE);

	if (!hmmio)
	{
		printf("ERROR: Failed to open '%s'.",Filename);
		return false;
	}

	// Create the main WAVE chunk
	MMCKINFO mmckinfo;
	ZeroMemory(&mmckinfo,sizeof(MMCKINFO));
	//mmckinfo.cksize=1;		// Size of the main chunk isn't getting set ???
	mmckinfo.fccType = mmioFOURCC('W','A','V','E');
	
	if (mmioCreateChunk(hmmio,&mmckinfo,MMIO_CREATERIFF)!=MM_OK)
	{
		printf("ERROR: Failed to create WAVE chunk.\n");
		return false;
	}

	// Create the format chunk
	MMCKINFO mmckinfoFmt;
	ZeroMemory(&mmckinfoFmt,sizeof(MMCKINFO));
	mmckinfoFmt.ckid    = mmioFOURCC('f','m','t',' ');
	//mmckinfoFmt.cksize  = sizeof(WAVEFORMATEX);		// This now varies

	if (mmioCreateChunk(hmmio,&mmckinfoFmt,0)!=MM_OK)
	{
		printf("ERROR: Failed to create 'fmt ' chunk.\n");
		mmioClose(hmmio,0);
		return false;
	}

	// Write out the format data
	
	// Handle PCM format WAVs
	if (fmt->wFormatTag == WAVE_FORMAT_PCM)
	{
		if (mmioWrite(hmmio,(char*)fmt,sizeof(PCMWAVEFORMAT))==MM_ERROR)
		{
			printf("ERROR: Failed to write PCM waveform format information.\n");
			mmioClose(hmmio,0);
			return false;
		}
	}
	else
	{
		// Write out standard non-PCM format block
		if (mmioWrite(hmmio,(char*)fmt,sizeof(WAVEFORMATEX))==MM_ERROR)
		{
			printf("ERROR: Failed to write out waveform format information.\n");
			mmioClose(hmmio,0);
			return false;
		}

		// Write out extended format information if any exists
		if (fmt->cbSize>0)
		{
			if (mmioWrite(hmmio,(((char*)fmt)+sizeof(WAVEFORMATEX)),fmt->cbSize)==MM_ERROR)
			{
				printf("ERROR: Failed to write out extended wave format data.\n");
				mmioClose(hmmio,0);
				return false;
			}
		}
	}

	if (mmioAscend(hmmio,&mmckinfoFmt,0)!=MM_OK)
	{
		printf("ERROR: Failed to ascend from 'fmt ' chunk.\n");
		mmioClose(hmmio,0);
		return false;
	}

	MMCKINFO mmckinfoData;
	ZeroMemory(&mmckinfoData,sizeof(MMCKINFO));
	mmckinfoData.ckid    = mmioFOURCC('d','a','t','a');
	mmckinfoData.cksize  = size;

	if (mmioCreateChunk(hmmio,&mmckinfoData,0)!=MM_OK)
	{
		printf("ERROR: Failed to create 'data' chunk.\n");
		return false;
	}

	// Write out the actual waveform data
	if (mmioWrite(hmmio,(char*)data,size)==MM_ERROR)
	{
		printf("ERROR: Failed to write waveform data.\n");
		return false;
	}

	if (mmioAscend(hmmio,&mmckinfoData,0)!=MM_OK)
	{
		printf("ERROR: Failed to ascend from data chunk.\n");
		return false;
	}

	// Create the sampler chunk so we can store loop info
	if (schunk && schunkSize)
	{
		ZeroMemory(&mmckinfoData,sizeof(MMCKINFO));
		mmckinfoData.ckid   = mmioFOURCC('s','m','p','l');
		
		// Sound forge uses the extended smpl data size field to have meaning, specifically
		// the state of the loop, it also seems to require that the chunk have some additonal
		// info afterwards, we'll pad it with 0's we'd need 2 bytes to adhear to spec anyways
		// since the extended data could be 0-2 depending on loop setting.  Perhaps SF has some
		// other settings, it appears to store some less relevant info afterwards as well

		//mmckinfoData.cksize = sizeof(SamplerChunk) + 4;
		mmckinfoData.cksize = schunkSize;

		if (mmioCreateChunk(hmmio,&mmckinfoData,0) != MM_OK)
		{
			printf("ERROR: Creating smpl chunk\n");
			return false;
		}

		if (mmioWrite(hmmio,(const char*)schunk,schunkSize) == MM_ERROR)
		{
			printf("ERROR: Failed to write smplr chunk\n");
			return false;
		}

		// Write out padding
		/*
		char pad[4];
		ZeroMemory(pad,4);

		if (mmioWrite(hmmio,(const char*)pad,4) == MM_ERROR)
		{
			printf("ERROR: Failed to write padding for smpl chunk\n");
			return false;
		}
		*/

		if (mmioAscend(hmmio,&mmckinfoData,0)!=MM_OK)
		{
			printf("ERROR: Failed to ascend from smplr chunk.\n");
			return false;
		}
	}

	if (mmioAscend(hmmio,&mmckinfo,0)!=MM_OK)
	{
		printf("ERROR: Failed to ascend from primary chunk.\n");
		return false;
	}

	mmioClose(hmmio,0);
	return true;
}

// Since the sampler chunk is variably sized, the user is responsible for freeing it
bool GetSamplerChunk(char* Filename, SamplerChunk** schunk, int* cbSize)
{
	HMMIO hmmio;	// handle to the multi-media file (our .wav)

	hmmio = mmioOpen(Filename,NULL,MMIO_READ);
	if (hmmio==NULL)
	{
		printf("ERROR: Couldn't open %s\n",Filename);
		return false;
	}

	MMCKINFO mmckinfoParent;
	ZeroMemory(&mmckinfoParent,sizeof(MMCKINFO));
	mmckinfoParent.fccType = mmioFOURCC('W','A','V','E');
	
	if (mmioDescend(hmmio,&mmckinfoParent,0,MMIO_FINDRIFF)!=MM_OK)
	{
		printf("ERROR: No waveform data exists in this file.\n");
		mmioClose(hmmio,0);
		return false;
	}

	MMCKINFO mmckinfoSubchunk;
	ZeroMemory(&mmckinfoSubchunk,sizeof(MMCKINFO));
	mmckinfoSubchunk.ckid = mmioFOURCC('s','m','p','l');

	if (mmioDescend(hmmio,&mmckinfoSubchunk,&mmckinfoParent,MMIO_FINDCHUNK)!=MM_OK)
	{
		mmioClose(hmmio,0);
		return true;
	}

	// Store the size of the chunk so we know how much extended data exists (if appropriate)
	if (cbSize)
		*cbSize = mmckinfoSubchunk.cksize;

	// Allocate memory for the sampler chunk
	*schunk = (SamplerChunk*)malloc(mmckinfoSubchunk.cksize);
	ZeroMemory(*schunk,mmckinfoSubchunk.cksize);

	// The sampler chunk was found read it
	if (mmioRead(hmmio,(char*)*schunk,*cbSize)==0)
	{
		printf("ERROR: Failed to load sampler chunk\n");
		mmioClose(hmmio,0);
		return false;
	}

	mmioClose(hmmio,0);

	return true;
}

// This function will look for the sampler WAV chunk to see if looping information
// is included and set bLooped if any loops exist
bool WAVLoops(char* Filename,bool* bLooped)
{
	HMMIO hmmio;	// handle to the multi-media file (our .wav)

	*bLooped=false;

	hmmio = mmioOpen(Filename,NULL,MMIO_READ);
	if (hmmio==NULL)
	{
		printf("ERROR: Couldn't open %s\n",Filename);
		return false;
	}

	MMCKINFO mmckinfoParent;
	ZeroMemory(&mmckinfoParent,sizeof(MMCKINFO));
	mmckinfoParent.fccType = mmioFOURCC('W','A','V','E');
	
	if (mmioDescend(hmmio,&mmckinfoParent,0,MMIO_FINDRIFF)!=MM_OK)
	{
		printf("ERROR: No waveform data exists in this file.\n");
		mmioClose(hmmio,0);
		return false;
	}

	MMCKINFO mmckinfoSubchunk;
	ZeroMemory(&mmckinfoSubchunk,sizeof(MMCKINFO));
	mmckinfoSubchunk.ckid = mmioFOURCC('s','m','p','l');

	if (mmioDescend(hmmio,&mmckinfoSubchunk,&mmckinfoParent,MMIO_FINDCHUNK)!=MM_OK)
	{
		mmioClose(hmmio,0);
		return true;
	}

	// The sampler chunk was found read it
	SamplerChunk schunk;

	if (mmioRead(hmmio,(char*)&schunk,sizeof(SamplerChunk))==0)
	{
		printf("ERROR: Failed to load sampler chunk\n");
		mmioClose(hmmio,0);
		return false;
	}

	if (schunk.cbSamplerData != 0)
	{
		/*
		char* extdata = (char*)malloc(schunk.cbSamplerData);

		if (mmioRead(hmmio,extdata,schunk.cbSamplerData)==0)
		{
			printf("ERROR: Couldn't load extended sampler data\n");
			mmioClose(hmmio,0);
			return false;
		}

		// For possible future use (if we do something special with extended sampler data)
		free(extdata);
		*/
	}

	// SoundForge seems kind of weird about this.  cbSamplerData is supposed to indicate
	// number of bytes of extended data in the sampler chunk but, SoundForge appears to be using
	// the field to specify loop type:
	//		0 extended bytes: means no looping
	//		1 extended bytes: means sustaining loop
	//		2 extended bytes: means sustaining with release

	if (schunk.cbSamplerData==0)
	{
		*bLooped=false;
		return true;
	}

	/*  For possible later use to read in loop markers
	if (schunk.cSampleLoops==0)
	{
		mmioClose(hmmio,0);
		return true;
	}

	if (schunk.cSampleLoops>1)
		printf("WARNING: There are multiple loops defined in this wave file.\n");

	// Read in the loops
	for(int i=0;i<schunk.cSampleLoops;i++)
	{
		SampleLoop sloop;

		if (mmioRead(hmmio,(char*)&sloop,sizeof(SampleLoop))==0)
		{
			printf("ERROR: Failed to load sample loop %i.\n",i);
			mmioClose(hmmio,0);
			return false;
		}

		// Warn if the start of the loop isn't zero since we only support a single full
		if (sloop.dwStart!=0)
			printf("WARNING: Loop %i does not start at begining of waveform, this will be ignored.\n");
	}
	*/

	mmioClose(hmmio,0);

	*bLooped=true;

	return true;
}

// Note: WAVEFORMATEX can contain extended data for non-PCM formats so, is allocated by LoadWAV
bool LoadWAV(char* Filename,WAVEFORMATEX** fmt,void** data,int* size)
{
	*data = 0;
	*size = 0;
	*fmt  = 0;

	HMMIO hmmio;	// handle to the multi-media file (our .wav)

	// Open the wave file
	hmmio = mmioOpen(Filename,NULL,MMIO_READ);

	if (!hmmio)
	{
		printf("Failed to open file.\n");
		return false;
	}

	MMCKINFO mmckinfoParent;
	ZeroMemory(&mmckinfoParent,sizeof(MMCKINFO));
	mmckinfoParent.fccType = mmioFOURCC('W','A','V','E');

	if (mmioDescend(hmmio,&mmckinfoParent,0,MMIO_FINDRIFF)!=MM_OK)
	{
		printf("ERROR: No waveform data exists in this file.\n");
		mmioClose(hmmio,0);
		return false;
	}

	MMCKINFO mmckinfoSubchunk;
	ZeroMemory(&mmckinfoSubchunk,sizeof(MMCKINFO));
	mmckinfoSubchunk.ckid = mmioFOURCC('f','m','t',' ');

	if (mmioDescend(hmmio,&mmckinfoSubchunk,&mmckinfoParent,MMIO_FINDCHUNK)!=MM_OK)
	{
		printf("ERROR: No waveform format data exists in this file.\n");
		mmioClose(hmmio,0);
		return false;
	}

	WAVEFORMAT format;
	void*      fmtdata;

	// Since the load process varies depending on if this is a PCM format wave or not
	// we must first load in a WAVEFORMAT structure check if it's PCM, if not, we have
	// to read extended waveformat info, and read a variable amount of additional data

	if (mmioRead(hmmio,(char*)&format,sizeof(WAVEFORMAT))==MM_ERROR)
	{
		printf("ERROR: Failed to read initial waveform format data.\n");
		mmioClose(hmmio,0);
		return false;
	}

	// Load for PCM waves
	if (format.wFormatTag == WAVE_FORMAT_PCM)
	{	
		if (mmckinfoSubchunk.cksize>sizeof(WAVEFORMAT))
		{
			// Allocate a PCM style format descriptor and load in the addition wBitsPerSample info
			fmtdata=malloc(sizeof(WAVEFORMATEX));
			memcpy(fmtdata,&format,sizeof(WAVEFORMAT));

			// Read wBitsPerSample data
			if (mmioRead(hmmio,(((char*)fmtdata)+sizeof(WAVEFORMAT)),sizeof(WORD))==MM_ERROR)
			{
				printf("ERROR: Failed to retrieve PCM wBitsPerSample info.\n");
				mmioClose(hmmio,0);
				return false;
			}

			WAVEFORMATEX* fmtdataex=(WAVEFORMATEX*)fmtdata;
			fmtdataex->cbSize = 0;
		}
		else
			return false;
	}
	else
	{
		// Load for non-PCM waves
		if (mmckinfoSubchunk.cksize>sizeof(WAVEFORMAT))
		{
			WORD cbSize;	// Additional parameter necessary to form WAVEFORMATEX 
							// (number of additional bytes to load)
			
			if (mmioRead(hmmio,(char*)&cbSize,sizeof(WORD))==MM_ERROR)
			{
				printf("ERROR: Failed to retrieve extended waveformat info.\n");
				mmioClose(hmmio,0);
				return false;
			}

			// Allocate extended waveformat data
			fmtdata = malloc(sizeof(WAVEFORMATEX)+cbSize);
			memcpy(fmtdata,&format,sizeof(WAVEFORMAT));
			WAVEFORMATEX* formatex = (WAVEFORMATEX*)fmtdata;
			formatex->cbSize = cbSize;

			if (cbSize>0)
			{
				// Read extended wave format data in
				if (mmioRead(hmmio,(((char*)fmtdata)+sizeof(WAVEFORMATEX)),cbSize)==MM_ERROR)
				{
					printf("ERROR: Failed to load extended waveformat data.\n");
					mmioClose(hmmio,0);
					return false;
				}
			}
		}
		else
		{
			// Allocate standard non-extended waveformat data
			fmtdata = malloc(sizeof(WAVEFORMATEX));
			memcpy(fmtdata,&format,sizeof(WAVEFORMATEX));
			WAVEFORMATEX* formatex = (WAVEFORMATEX*)fmtdata;
			formatex->cbSize = 0;
		}
	}

	mmioAscend(hmmio,&mmckinfoSubchunk,0);
	
	// Acquire the waveform data chunk
	MMCKINFO mmckinfoData;
	ZeroMemory(&mmckinfoData,sizeof(MMCKINFO));
	mmckinfoData.ckid = mmioFOURCC('d','a','t','a');

	if (mmioDescend(hmmio,&mmckinfoData,&mmckinfoParent,MMIO_FINDCHUNK)!=MM_OK)
	{
		printf("ERROR: Couldn't find waveform data in file.\n");
		mmioClose(hmmio,0);
		return false;
	}

	// Read in the actual waveform
	int   wavesize=mmckinfoData.cksize;
	char* wavedata=(char*)malloc(wavesize);
	
	if (mmioRead(hmmio,wavedata,wavesize)==MM_ERROR)
	{
		printf("ERROR: Couldn't read waveform data.\n");
		mmioClose(hmmio,0);
		return false;
	}
	
	*data = wavedata;
	*size = wavesize;
	*fmt  = (WAVEFORMATEX*)fmtdata;

	mmioClose(hmmio,0);
	return true;
}
