/*
	aspectresize.cpp
	App to resize and crop Targa images (also supports rotation)
	the non-major axis

	NOTE: Required to exist entirely in one file

	! Note shorts are assumed to be 2 bytes on target system !
	!      ints   are assumed to be 4 bytes                  !
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

void TCFixCMPWord( unsigned short* data );

enum ColorComponent
{
	RED,
	GREEN,
	BLUE,
};

void Assert(int val, char* buf)
{
	if (!val)
	{
		printf(buf);
		exit(-1);
	}
}

int	GetLog2( int value )
{
	int i;

	i = 0;
	if( value > 0 )
	{		
		while( value != 1 )
		{
			value >>= 1;
			i++;
		}

		return i;
	}
	
	return -1;
}

//////////// CVirtualOutputFile
//namespace IoUtils
//{
///////////////////// Header Start
enum VirtualFileMode
{
	vTEXT_MODE,
	vBINARY_MODE,
};

class CVirtualFile
{
protected:
	CVirtualFile();
	~CVirtualFile();

public:
	bool	Init( int size, char* pData = NULL );
	bool	Uninit();
	bool	SeekPos( int pos );
	int		TellPos() const;
	bool	Align( int alignSize );
	bool	Skip( int numBytes );

	inline char* GetBuffer()     { return mp_buffer; }
	inline int   GetBufferSize() { return (int)(mp_currentPos - mp_buffer); }

protected:
	char*	mp_buffer;
	int		m_size;
	char*	mp_currentPos;
	int		m_currentPos;
};

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

class CVirtualInputFile : public CVirtualFile
{
public:
	CVirtualInputFile();
	~CVirtualInputFile();

public:
	bool	Read( char* pData, int size );
	int		Load( const char* pFileName, int filemode = vBINARY_MODE );	// returns size of file read
};

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

class CVirtualOutputFile : public CVirtualFile
{
public:
	CVirtualOutputFile();
	~CVirtualOutputFile();

public:
	bool	Write( const char* pData, int size, bool reverseByteOrder = false );
	bool	WriteString( const char* pString, bool addNewLine );
	bool	Save( const char* pFileName, int filemode = vBINARY_MODE );	// returns 0 on failure
};

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

// some convenience functions for backwards-compatibility
bool write8(CVirtualOutputFile* pFileBuffer, unsigned char* pData, bool reverseByteOrder);
bool write16(CVirtualOutputFile* pFileBuffer, unsigned short* pData, bool reverseByteOrder);
bool write32(CVirtualOutputFile* pFileBuffer, unsigned int* pData, bool reverseByteOrder);

// whether a file exists
bool file_exists(const char* pFileName);

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

///////////////////// Header End

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CVirtualFile::CVirtualFile()
{
	mp_buffer = NULL;
	m_size = 0;
	mp_currentPos = 0;
	m_currentPos = 0;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CVirtualFile::~CVirtualFile()
{
	Uninit();
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CVirtualFile::Init( int size, char* pData )
{
	Assert( mp_buffer == NULL, "Buffer already exists!" );
	Assert( size >= 0, "Size must be positive: (%d bytes/%d)"/*, size, 333*/ );

	mp_buffer = new char[size];

	if ( pData )
	{
		memcpy(mp_buffer,pData,size);
	}
	else
	{
		//memset(mp_buffer,0,size);
	}

	m_size = size;
	mp_currentPos = &mp_buffer[0];
	m_currentPos = 0;

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CVirtualFile::Uninit()
{
	if ( mp_buffer )
	{
		delete[] mp_buffer;
	}

	m_size = 0;
	mp_currentPos = NULL;
	m_currentPos = 0;

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CVirtualFile::SeekPos( int pos )
{
	// GJ:  you're allowed to be out of bounds,
	// as long as you don't read or write to it...
	// this lets us do things like Skip() or
	// Align() past the last chunk of data to
	// the byte right after the end of the file...
#if 0
	if ( pos < 0 || pos >= m_size )
	{
		char error_msg[512];
		sprintf( error_msg, "Tried to seek to an out-of-bounds position (current position = %d; trying to seek position %d; size of file = %d)", m_currentPos, pos, m_size );
		Assert( 0, error_msg );
		return false;
	}
#endif

	Assert( mp_buffer != NULL, "Buffer doesn't exist!" );

	mp_currentPos = &mp_buffer[pos];
	m_currentPos = pos;

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CVirtualFile::Align( int alignSize )
{
	// set the cursor to the nearest X bytes
	int remainder = m_currentPos % alignSize;

	if( remainder )
	{
		return SeekPos( alignSize - remainder + m_currentPos );
	}	
	
	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CVirtualFile::Skip( int numBytes )
{
	// skip X number of bytes
	return SeekPos( m_currentPos + numBytes );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int CVirtualFile::TellPos() const
{
	return m_currentPos;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CVirtualInputFile::CVirtualInputFile() : CVirtualFile()
{
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CVirtualInputFile::~CVirtualInputFile()
{
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CVirtualInputFile::Read( char* pData, int size )
{
	if ( m_currentPos < 0 || m_currentPos > (m_size - size) )
	{
		char error_msg[512];
		sprintf( error_msg, "Tried to read data out of bounds (trying to read %d bytes at position %d; size of file = %d)", size, m_currentPos, m_size );
		Assert( 0, error_msg );
		return false;
	}

	if ( size == 0 )
	{
		// reading 0 bytes is valid
		// but it would be nice to figure out
		// who is actually doing this...
		return true;
	}

	Assert( pData != NULL, "No return data specified" );
	Assert( size > 0, "Size must be positive: (Tried to get %d bytes/%d)"/*, size, 555*/ );

#if 0
	memcpy(pData,mp_currentPos,size);
	pData+=size;
	mp_currentPos+=size;
	m_currentPos+=size;
#else

	for ( int i = 0; i < size; i++ )
	{
		*pData = *mp_currentPos;

		pData++;
		mp_currentPos++;
		m_currentPos++;
	}

#endif

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int CVirtualInputFile::Load( const char* pFileName, int fileMode )
{
	Assert( pFileName != NULL, "No filename" );
	
	FILE* pFile = NULL;
	
	switch ( fileMode )
	{
		case vTEXT_MODE:
			pFile = fopen( pFileName, "rt" );
			break;
		case vBINARY_MODE:
			pFile = fopen( pFileName, "rb" );
			break;
		default:
			printf( "Unrecognized file mode" );
			return 0;
	}

	int inSize = 0;
	
	if ( pFile )
	{
		int startIn = ftell( pFile );
		fseek( pFile, 0, SEEK_END ); 
		inSize = ftell( pFile ) - startIn;
		fseek( pFile, startIn, SEEK_SET );

		char* pBuffer = new char[inSize];

		fread( pBuffer, inSize, 1, pFile );

		this->Init( inSize, pBuffer );
	
		delete pBuffer;

		fclose( pFile );
	}

	return inSize;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CVirtualOutputFile::CVirtualOutputFile() : CVirtualFile()
{
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CVirtualOutputFile::~CVirtualOutputFile()
{
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CVirtualOutputFile::WriteString( const char* pString, bool addNewLine )
{
	int size = strlen(pString) + ( addNewLine ? 1 : 0 );

	if ( m_currentPos < 0 || m_currentPos > (m_size - size) )
	{
		char msg[1024];
		sprintf( msg, "Output buffer too small (raise from %dKb)\n", m_size/1024 );
		Assert( 0, msg );
		return false;
	}

	Assert( strlen(pString) < 1024, "Msg buffer too small!" );

	char msg[1024 + 1];	// add two for the potential new line

	sprintf( msg, "%s%s", pString, addNewLine ? "\n" : "" );
	return this->Write( msg, strlen(msg), false ); 
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CVirtualOutputFile::Write( const char* pData, int size, bool reverseByteOrder )
{
	if ( m_currentPos < 0 || m_currentPos > (m_size - size) )
	{
		char msg[1024];
		sprintf( msg, "Output buffer too small (raise from %dKb)\n", m_size/1024 );
		Assert( 0, msg );
		return false;
	}

	if( size == 0 )
	{
		return true;	// nothing to write
	}

	Assert( pData != NULL, "No return data specified" );
	Assert( size > 0, "Size must be positive" );

	if ( reverseByteOrder )
	{
		pData += ( size - 1 );

		for ( int i = 0; i < size; i++ )
		{
			*mp_currentPos = *pData;
	
			pData--;
			mp_currentPos++;
			m_currentPos++;
		}
	}
	else
	{
		memcpy( mp_currentPos, pData, size );
		mp_currentPos += size;
		m_currentPos += size;

#if 0
		for ( int i = 0; i < size; i++ )
		{
			*mp_currentPos = *pData;
	
			pData++;
			mp_currentPos++;
			m_currentPos++;
		}
#endif
	}

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CVirtualOutputFile::Save( const char* pFileName, int fileMode )
{
	Assert( pFileName != NULL, "No filename" );
	
	FILE* pFile = NULL;
	
	switch ( fileMode )
	{
		case vTEXT_MODE:
			pFile = fopen( pFileName, "wt" );
			break;
		case vBINARY_MODE:
			pFile = fopen( pFileName, "wb" );
			break;
		default:
			printf( "Unrecognized file mode" );
			return 0;
	}

	if ( !pFile )
	{
		char msg[256];
		sprintf( msg, "Could not open %s for output", pFileName );
		Assert( 0, msg );
		return 0;
	}

	fwrite( (const void*)mp_buffer, this->TellPos(), 1, pFile );
	fclose( pFile );

	// don't return number of bytes written, because
	// 0 is a valid number of bytes to be written out
	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool file_exists(const char* pFileName)
{
	FILE* pFile = fopen( pFileName, "rb" );
	
	if ( !pFile )
	{
		return false;
	}
	else
	{
		fclose( pFile );
		return true;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

// for debugging
#define	swap32(_p)	((uint32 *)_p)[0] = ( ((uint8 *)_p)[3] | (((uint8 *)_p)[2]<<8) | (((uint8 *)_p)[1]<<16) | (((uint8 *)_p)[0]<<24) );
#define	swap16(_p)	((uint16 *)_p)[0] = ( ((uint8 *)_p)[1] | (((uint8 *)_p)[0]<<8) );

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool write8(CVirtualOutputFile* pFile, unsigned char* pData, bool reverseByteOrder)
{
	Assert( pFile != NULL, "No buffer pointer!" );

	unsigned char theData = *pData;

	pFile->Write( (const char*)&theData, 1, reverseByteOrder );

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool write16(CVirtualOutputFile* pFile, unsigned short* pData, bool reverseByteOrder)
{
	Assert( pFile != NULL, "No buffer pointer!" );

	unsigned short theData = *pData;

#if 1
	pFile->Write( (const char*)&theData, 2, reverseByteOrder );
#else
	// tests the byte-swapping code
	swap16(&theData);
	pFile->Write( (const char*)&theData, 2, !reverseByteOrder );
#endif

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool write32(CVirtualOutputFile* pFile, unsigned int* pData, bool reverseByteOrder)
{
	Assert( pFile != NULL, "No buffer pointer!" );

	unsigned int theData = *pData;

#if 1
	pFile->Write( (const char*)&theData, 4, reverseByteOrder );
#else
	// tests the byte-swapping code
	swap32(&theData);
	pFile->Write( (const char*)&theData, 4, !reverseByteOrder );
#endif

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

//};
////////////////////////////////////////////////////////////////////////////////////////////////

//////////// GenCRC
static unsigned long CRCTable[256] = // CRC polynomial 0xedb88320
{
      0x00000000, 0x77073096, 0xee0e612c, 0x990951ba,
      0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
      0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
      0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
      0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de,
      0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
      0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec,
      0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
      0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
      0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
      0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940,
      0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
      0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116,
      0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
      0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
      0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
      0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a,
      0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
      0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818,
      0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
      0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
      0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
      0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c,
      0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
      0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2,
      0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
      0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
      0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
      0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086,
      0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
      0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4,
      0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
      0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
      0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
      0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8,
      0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
      0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe,
      0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
      0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
      0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
      0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252,
      0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
      0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60,
      0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
      0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
      0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
      0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04,
      0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
      0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a,
      0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
      0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
      0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
      0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e,
      0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
      0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c,
      0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
      0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
      0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
      0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0,
      0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
      0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6,
      0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
      0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
      0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};

// Generates a checksum from a name, case insensitive.
unsigned long GenerateCRC(const char *pName)
{
    // A checksum of zero is used to mean no name.
    if (!pName) 
	{
		return 0;
	}	
    
    // Initializing the CRC to all one bits avoids failure of detection
	// should entire data stream get cyclically bit-shifted by one position.
	// The calculation of the probability of this happening is left as
	// an exercise for the reader.
	unsigned long rc = 0xffffffff;
    int size=strlen(pName);

    for (int i=0; i<size; i++)
    {
        char ch=pName[i];
        // Convert to lower case.
        if (ch>='A' && ch<='Z') 
		{
			ch='a'+ch-'A';
		}	
        rc = CRCTable[(rc^ch) & 0xff] ^ ((rc>>8) & 0x00ffffff);
    }

	return rc;
}

unsigned long GenerateCRC(const char *pData, int size)
{	
	// A checksum of zero is used to mean no name.
    if( !pData ) 
	{
		return 0;
	}	
    
    // Initializing the CRC to all one bits avoids failure of detection
	// should entire data stream get cyclically bit-shifted by one position.
	// The calculation of the probability of this happening is left as
	// an exercise for the reader.
	unsigned long rc = 0xffffffff;
	int i;
    
    for( i = 0; i < size; i++ )
    {
        char ch=pData[i];
        rc = CRCTable[(rc^ch) & 0xff] ^ ((rc>>8) & 0x00ffffff);
    }

	return rc;
}
//////////////////////////////////////////////////////////// End GenCRC

///////////////////////// Start llist.h

template <class T>
struct Link
{
	T data;
	Link<T>* next;
	Link<T>* prev;
};

template <class T>
class LinkList
{
	Link<T>* head;
	Link<T>* tail;
	int      count;
	Link<T>* curpos;
	int      curposIndex;

public:
	LinkList()
	{
		curpos=head=tail=0;
		count=curposIndex=0;
	}

	LinkList(LinkList<T>& list)
	{
		curpos=head=tail=0;
		count=curposIndex=0;

		Link<T>* curNode = list.GetHead();

		while(curNode)
		{
			AddToTail(&curNode->data);
			curNode = curNode->next;
		}
	}

	LinkList<T>& operator = (LinkList<T>& list)
	{
		Clear();

		if (&list==this)
			return *this;

		Link<T>* curNode = list.GetHead();

		while(curNode)
		{
			AddToTail(&curNode->data);
			curNode = curNode->next;
		}

		return *this;
	}

	void Clear()
	{
		Link<T>* CurLink=head;
		Link<T>* tmpLink;

		while(CurLink)
		{
			tmpLink=CurLink->next;
			delete CurLink;
			CurLink=tmpLink;
		}

		head=tail=0;
		count=0;
		curposIndex=0;
	}

	~LinkList()
	{
		Clear();
	}

	inline Link<T>* GetHead() { return head;  }
	inline Link<T>* GetTail() { return tail;  }
	inline int      GetSize() { return count; }

	Link<T>* AddToTail(T* item)
	{
		Link<T>* newLink=new Link<T>;
		newLink->data=*item;
		newLink->prev=tail;
		newLink->next=0;

		if (head==0)
			head=newLink;

		if (tail)
			tail->next=newLink;

		tail=newLink;

		curposIndex = count;
		curpos = tail;
		count++;

		return newLink;
	}

	Link<T>* AddAfterLink(Link<T>* link, T* item)
	{
		if (link)
		{
			Link<T>* newLink=new Link<T>;
			newLink->data=*item;
			newLink->prev=link;
			newLink->next=link->next;

///
			if (link->next)
				link->next->prev = newLink;
///

			link->next=newLink;
	
			if (link==tail)
				tail=newLink;

			curposIndex = count;
			curpos = tail;
			count++;
			return newLink;
		}
		else
		{
			return AddToHead(item);
		}
	}

	Link<T>* Find(T* val)
	{
		Link<T>* curlink = head;

		while(curlink)
		{
			if (curlink->data==*val)
				return curlink;

			curlink=curlink->next;
		}

		return 0;
	}

	Link<T>* Find(bool (*Compare)(T* v1,T* v2),T* value)
	{
		Link<T>* curlink = head;

		while(curlink)
		{
			if (Compare(&curlink->data,value))
				return curlink;

			curlink=curlink->next;
		}

		return 0;
	}

	Link<T>* FindFrom(T* val,Link<T>* link=0)
	{
		Link<T>* curlink;

		if (!link)
			curlink = head;
		else
			curlink = link->next;

		while(curlink)
		{
			if (curlink->data==*val)
				return curlink;

			curlink=curlink->next;
		}

		return 0;
	}

	Link<T>* FindFrom(bool (*Compare)(T* v1,T* v2),T* value,Link<T>* link=0)
	{
		Link<T>* curlink;

		if (!link)
			curlink = head;
		else
			curlink = link->next;

		while(curlink)
		{
			if (Compare(&curlink->data,value))
				return curlink;

			curlink=curlink->next;
		}

		return 0;
	}

	Link<T>* FindRange(T* val,Link<T>* start=0,Link<T>* end=0)
	{
		Link<T>* curlink;

		if (!start)
			start = head;
		else
			curlink = start->next;

		if (!end)
			end = tail;

		while(curlink != end)
		{
			if (curlink->data==*val)
				return curlink;

			curlink = curlink->next;
		}

		return 0;
	}

	Link<T>* FindRange(bool (*Compare)(T* v1,T* v2),T* value,Link<T>* start=0,Link<T>* end=0)
	{
		Link<T>* curlink;

		if (!start)
			start = head;
		else
			curlink = start->next;

		if (!end)
			end = tail;

		while(curlink != end)
		{
			if (Compare(&curlink->data,value))
				return curlink;

			curlink = curlink->next;
		}

		return 0;		
	}

	Link<T>* FindRange(bool (*Compare)(T* v1,T* v2,void*), T* value, Link<T>* start=0, Link<T>* end=0, void* pData=0)
	{
		Link<T>* curlink;

		if (!start)
			start = head;
		else
			curlink = start->next;

		if (!end)
			end = tail;

		while(curlink != end)
		{
			if (Compare(&curlink->data,value,pData))
				return curlink;

			curlink = curlink->next;
		}

		return 0;		
	}

	Link<T>* AddToTail()
	{
		Link<T>* newLink=new Link<T>;
		newLink->prev=tail;
		newLink->next=0;

		if (head==0)
			head=newLink;

		if (tail)
			tail->next=newLink;

		tail=newLink;
		curposIndex = count;
		curpos = tail;
		count++;

		return newLink;
	}

	Link<T>* AddToTailUnique(T* item)
	{
		Link<T>* curlink=head;

		while(curlink)
		{
			if (curlink->data==*item)
				return 0;

			curlink=curlink->next;
		}

		return AddToTail(item);
	}

	Link<T>* AddToTailFindUnique(T* item)
	{
		Link<T>* curlink=head;

		while(curlink)
		{
			if (curlink->data==*item)
				return curlink;

			curlink=curlink->next;
		}

		return AddToTail(item);
	}

	inline Link<T>* AddUnique(T* item)
	{
		return AddToTailUnique(item);
	}

	inline Link<T>* Add(T* item)
	{
		return AddToTail(item);
	}

	inline Link<T>* Add()
	{
		return AddToTail();
	}

	Link<T>* AddToHead(T* item)
	{
		Link<T>* newLink=new Link<T>;
		newLink->data=*item;
		newLink->prev=0;
		newLink->next=head;

		if (tail==0)
			tail=newLink;

		if (head)
			head->prev=newLink;

		head=newLink;
		curposIndex = 0;
		curpos = head;
		count++;

		return newLink;
	}

	void Remove(Link<T>* link)	
	{
		count--;

		// Link is head
		if (link==head)
		{
			if (link==tail)
			{
				head=0;
				tail=0;
				delete link;
				return;
			}
			
			head=link->next;
			link->next->prev=0;
			delete link;
			curposIndex = 0;
			curpos = head;
			return;
		}

		// Link is middle
		if (link!=head && link!=tail)
		{
			link->next->prev=link->prev;
			link->prev->next=link->next;
			delete link;
		}

		// Link is tail
		if (link==tail)
		{
			tail=link->prev;
			link->prev->next=0;
			delete link;
		}

		curposIndex = count-1;
		curpos = tail;
	}

	Link<T>* GetLink(int i)
	{
		curpos=head;

		for(int j=0;j<i;j++)
			curpos=curpos->next;

		return curpos;
	}

	// Append list
	LinkList<T>& operator+= (LinkList<T>& list)
	{
		Link<T>* link = list.GetHead();

		int count = list.count;

		for(int i=0;i<count;i++)
		{
			AddToTail(&link->data);
			link=link->next;
		}

		return *this;	
	}

	LinkList<T>& operator+ (LinkList<T>& list)
	{
		Link<T>* link = list.GetHead();

		int count = list.count;

		for(int i=0;i<count;i++)
		{
			AddToTail(&link->data);
			link=link->next;
		}

		return *this;	
	}

	/*
	T& operator[] (int i)
	{
		curpos=head;

		for(int j=0;j<i;j++)
			curpos=curpos->next;

		return curpos->data;
	}
	*/

	T& operator[] (int i)
	{
		assert(i>-1);

		if (curpos==0)
		{
			curpos=head;
			curposIndex=0;
			return curpos->data;
		}

		assert(curpos!=0);

		if (curposIndex<i)
		{
			int dist=i-curposIndex;

			for(int j=0;j<dist;j++)
			{
				curpos=curpos->next;
				assert(curpos!=0);
			}

			curposIndex=i;
			return curpos->data;
		}

		if (curposIndex>i)
		{
			int dist=curposIndex-i;

			for(int j=0;j<dist;j++)
			{
				curpos=curpos->prev;
				assert(curpos!=0);
			}

			curposIndex=i;
			return curpos->data;
		}

		// curposIndex==i
		return curpos->data;
	}

	int GetIndex(Link<T>* link)
	{
		Link<T>* curlink = head;
		int index = 0;

		while(curlink)
		{
			if (curlink == link)
				return index;
			
			curlink = curlink->next;
			index++;
		}

		return -1;
	}
};

///////////////////////// End llist.h

///////////////////////// Color.h
#define MASK_R555  0x7C00
#define MASK_G555  0x3E0
#define MASK_B555  0x1F

#define MASK_R565  0xF800
#define MASK_G565  0x7E0
#define MASK_B565  0x1F

struct Point
{
	int x,y;

	inline Point(int sx, int sy)
	{
		x = sx;
		y = sy;
	}
};

template<class T>
struct Freq
{
	T            data;
	unsigned int freq;		// How frequent this data appeared within the sample area

	Freq() { freq = 0; }
};

struct color8		// Palletized texture
{
	unsigned char index;

	int operator== (color8& right)
	{
		return (index == right.index);
	}
};

struct color16_555
{
	short color;		// Contains the packed color in RGB555 format 5 bits (red, green, and blue)	

	inline unsigned char R()
	{
		return (color >> 10);
	}

	// Pack the red component into 555 format
	inline void R(unsigned char val)
	{
		color &= ~MASK_R555;		// Remove current red component
		color |= (val << 10);		// Pack in new value
	}

	inline unsigned char G()
	{
		return ((color & MASK_G555) >> 5);
	}

	inline void G(unsigned char val)
	{
		color &= ~MASK_G555;		// Remove current green component
		color |= (val << 5);		// Pack in new value
	}

	inline unsigned char B()
	{
		return (color & MASK_B555);
	}

	inline void B(unsigned char val)
	{
		color &= ~MASK_B555;		// Remove current blue component
		color |= val;				// Pack in new value
	}

	inline unsigned char operator[](int id)
	{
		switch(id)
		{
		case RED:
			return R();

		case GREEN:
			return G();			

		case BLUE:
			return B();
		}
	}

	int operator== (color16_555& right)
	{
		return (color == right.color);
	}
};

struct color16_565
{
	short color;		// Contains the packed color in RGB565 format 5 bits red, 6 bits green, 5 bits blue)

	inline unsigned char R()
	{
		return (color >> 10);
	}

	inline void R(unsigned char val)
	{
		color &= ~MASK_R565;		// Remove red component
		color |= (val << 10);		// Pack in new value
	}

	inline unsigned char G()
	{
		return ((color & MASK_G565) >> 5);
	}

	inline void G(unsigned char val)
	{
		color &= ~MASK_G565;		// Remove green component
		color |= (val << 5);		// Pack in new value
	}

	inline unsigned char B()
	{
		return (color & MASK_B565);
	}

	inline void B(unsigned char val)
	{
		color &= ~MASK_B565;		// Remove blue component
		color |= val;				// Pack in new value
	}

	inline unsigned char operator[](int id)
	{
		switch(id)
		{
		case RED:
			return R();

		case GREEN:
			return G();			

		case BLUE:
			return B();
		}
	}

	int operator== (color16_565& right)
	{
		return (color == right.color);
	}
};

struct color24
{
	unsigned char r,g,b;

	int operator== (color24& right)
	{
		return (r == right.r &&
			    g == right.g &&
				b == right.b);
	}
};

struct color32
{
	unsigned char a,r,g,b;

	int operator== (color32& right)
	{
		return (r == right.r &&
			    g == right.g &&
				b == right.b &&
				a == right.a);
	}
};
//////////////////////// End color.h


///////////////////////// NxTexture from PngConv to support our .img format ////////////////////////////////////////////
enum TextureFormat
{
	v32_BIT,
	v24_BIT,
	v16_BIT,
	v8_BIT,
	v4_BIT,		
	v8_BIT_GRAY,
	v4_BIT_GRAY,
	v2_BIT,
	v1_BIT,
	v2_BIT_GRAY,
	v1_BIT_GRAY,
};

enum TextureMipMax
{
	vMAX_MIP_LEVELS = 10
};

enum TextureFlags
{
	mFORCE_BYTE_PER_COMPONENT		=	0x0001,
	mCHANGE_FULLY_TRANSPARENT_COLOR	=	0x0002,
	mAUTO_GENERATE_MIPMAPS			=	0x0004,
	mTRANSPARENT					=	0x0008,
};	

class	NxTexture
{
public:
	NxTexture( void );
	~NxTexture( void );

	bool	operator==( NxTexture& texture );

	int GetBpp( void );
	int GetPaletteBpp( void );

	int	GetPixelFormat( void );
	int	GetPaletteFormat( void );

	char* GetTexelData( int mip_level );
	char* GetPaletteData( void );

	int	GetTexelDataSize( int mip_level );
	int GetPaletteDataSize( void );
	int	GetTotalDataSize( void );

	int	GetWidth( int mip_level );
	int GetHeight( int mip_level );

	void SetWidth( int mip_level, int width );
	void SetHeight( int mip_level, int height );

	int GetNumPaletteEntries( void );

	bool IsPaletted( void );

	char*	GetName( void );
	char*	GetMipPath( int mip_level );

	bool	Convert4BitPixelFormatTo8BitPixelFormat( void );
	bool	Convert16BitPaletteFormatTo32BitPaletteFormat( void );
	bool	Convert24BitPaletteFormatTo32BitPaletteFormat( void );
	bool	Convert32BitRGBAPaletteFormatTo32BitBGRAPaletteFormat( void );
	bool	Convert32BitRGBAPixelFormatTo32BitBGRAPixelFormat( void );
	bool	ConvertTo32BitPixelFormat( void );
	
	bool	SwizzleTextureForXbox( void );
	bool	SaveTextureForNGPS( char* path );

	bool	ShouldAutoGenerateMipMaps( void );
	bool	ShouldCompressComponents( void );
	bool	ShouldChangeTransparentColor( void );	
	
	void	SetFlags( int flags );
	int		GetFlags( void );

	void	SetAutoGenerateMipMaps( bool auto_generate );
	
	void	SetNumMipLevels( int num_mip_levels );
	int		GetNumMipLevels( void );

	void	GenerateChecksum( void );
	unsigned long	GetChecksum( void );

	void	SetValidity( bool valid );
	bool	IsValid( void );

//private:

	int	m_pixel_format;
	int	m_palette_format;

	int	m_orig_width[vMAX_MIP_LEVELS];
	int	m_orig_height[vMAX_MIP_LEVELS];

	int	m_width[vMAX_MIP_LEVELS];
	int	m_height[vMAX_MIP_LEVELS];

	int m_bpp;
	int m_palette_bpp;

	char*	m_texel_data[vMAX_MIP_LEVELS];
	char*	m_palette_data;

	int m_num_palette_entries;
	
	int		m_mip_levels;

	int		m_flags;	

	char	m_name[256];

	unsigned long m_checksum;

	bool	m_valid;
};

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////// ImageData.h

struct ImageData
{
	unsigned long width, height;

	int bit_depth;
	int nChannels;
	int imageSize;
	int pixel_depth;

	unsigned char* buf;
	
	color32*       palette;
	int            nPalEntries;

	ImageData()
	{
		width       = 0;
		height      = 0;

		bit_depth   = 0;
		nChannels   = 0;
		imageSize   = 0;
		pixel_depth = 0;

		buf         = NULL;
		palette     = NULL;
		nPalEntries = 0;
	}

	~ImageData()
	{
		if (buf)
			free(buf);

		if (palette)
			free(palette);
	}

	void CopyNoImage(ImageData* src)
	{
		bit_depth   = src->bit_depth;
		nChannels   = src->nChannels;
		imageSize   = src->imageSize;
		pixel_depth = src->pixel_depth;	

		width       = src->width;
		height      = src->height;

		// Copy palette
		nPalEntries = src->nPalEntries;
		
		for(int i = 0; i < nPalEntries; i++)
			palette[i] = src->palette[i];
	}
};

///////////////////// End ImageData.h


/////////////////// Quantizer.h

/*
	Quantizer.h
	Image quantization class (uses typical octtree method but, since we need alpha as well, this
	uses more memory in favor of speed with a hexideca tree (4-bit index keys) :)
*/

//#include "ImageData.h"
//#include "llist.h"
//#include "colors.h"
//#include <math.h>

struct HDNode
{
	HDNode*        pChild[16];
	HDNode*        pParent;
	bool           bLeaf;		// As the tree is pruned not every level will have a depth of 7
	int            sumRed;		// Sum of all red component values for this color
	int            sumGreen;	// Sum of all green component values for this color
	int            sumBlue;		// Sum of all blue component values for this color
	int            sumAlpha;	// Sum of all alpha values for this color
	int            nPixels;		// Number of pixels using this color

	int            index;		// The index of the average color for this node in the tree 
								// (filled in after final construction)  Speeds up conversion to nearest colors

	Link<HDNode*>* link;		// link into the leaflist for this node

	HDNode()
	{
		for(int i=0;i<16;i++)
			pChild[i] = 0;

		pParent  = NULL;
		bLeaf    = false;
		sumRed   = 0;
		sumGreen = 0;
		sumBlue  = 0;
		sumAlpha = 0;
		nPixels  = 0;
		index    = -1;
	}

	void DeleteChild(int n)
	{
		for(int i=0;i<16;i++)
			if (n != i && pChild[i] == pChild[n])
				pChild[i] = NULL;

		delete pChild[n];
		pChild[n] = NULL;
	}

	~HDNode()
	{
		// The standard closest color tree method can leave some of the palette unused
		// To better optimize the palette we'll build the full tree and only reduce two of the children 
		// at a time, which can result in some child pointers pointing to the same node
		// This will prevent from freeing multiple times.
		for(int i=0;i<16;i++)
		{
			if (pChild[i])
			{
				for(int j=i+1;j<16;j++)
					if (pChild[j] == pChild[i])
						pChild[j] = NULL;

				delete pChild[i];
				pChild[i] = NULL;
			}
		}
	}
};

struct TreeData
{
	HDNode*           root;
	LinkList<HDNode*> leafList;
	int               nPalEntries;
	color32*          pal;

	TreeData()
	{
		root = NULL;
		nPalEntries = 0;
		pal = NULL;
	}

	~TreeData()
	{
		if (root)
			delete root;

		if (pal)
			delete [] pal;
	}
};

// Color component distribution
// This struct will be built per pixel, allowing us to disperse 
struct ColorDist
{
	unsigned int rSum, gSum, bSum, aSum;
};

enum QuantizerFlag
{
	QUANT_PALETTE = 0x0001,		// Flag to convert the image to a palettized one (8-bit)
	QUANT_ALPHA   = 0x0002,		// Flag to store indexed alpha information in the image's tRNS chunk
};

class Quantizer
{
	ImageData*         srcImage;	// The source image that we're quantizing
	ImageData*         destImage;	// The image that we're generating
	HDNode*            root;		// Pointer to root of the color tree
	int                maxColors;	// Maximum number of colors allowed in palette
	LinkList<HDNode*>  leafList;	// Link list of leaf nodes in the tree for faster searches
	color32*           pal;			// The quantized palette
	int                palColors;	// Number of colors in final palette
	unsigned long      flags;

	void AddColor(HDNode** node,
		          HDNode*  parent,
				  int      level,
		          unsigned char r,unsigned char g,unsigned char b,unsigned char a=255,
				  LinkList<HDNode*>* list=NULL);

	void PruneTree(int nColors);
	void BuildPal();
	void CopyPal(ImageData* img);

	// Finds closest color match within the palette
	int GetClosestColor(unsigned char r,
		                unsigned char g,
						unsigned char b,
						unsigned char a=255,
						int           level=0,
						HDNode*       node=NULL);

	void Quant8Bit();
	void Quant16Bit();
	void Quant24Bit();
	void Quant32Bit();

	void QuantAdd();

	void FreeTree();

	// Finds squared distance between 2 colors in 4 dimensions
	inline int     Dist2_4D(unsigned char r1,
						    unsigned char g1,
							unsigned char b1,
							unsigned char a1,
							unsigned char r2,
							unsigned char g2,
							unsigned char b2,
							unsigned char a2)
{
	return (abs((int)r2-(int)r1)+
		    abs((int)g2-(int)g1)+
			abs((int)b2-(int)b1)+
			abs((int)a2-(int)a1));
}

public:
	Quantizer();
	~Quantizer();

	// Finds the closest linear match to a particular color from
	// a given point in the color tree
	void FindLinearMatch(unsigned char r,
						 unsigned char g,
						 unsigned char b,
						 unsigned char a,
		                 HDNode*  node, 
						 color32* bestMatch,
						 int* dist);

	inline color32 GetClosestColor32(unsigned char r,
		                             unsigned char g,
									 unsigned char b,
									 unsigned char a=255)
	{
		return pal[GetClosestColor(r,g,b,a)];
	}

	color32 FindColorMatch(TreeData* tdata,
		                   unsigned char r,
					       unsigned char g,
					       unsigned char b,
					       unsigned char a=255);

	color32 GetClosestColor32(HDNode* tree,
		                      unsigned char r,
							  unsigned char g,
							  unsigned char b,
							  unsigned char a=255,
							  int           level=0);
		                      
	TreeData* BuildColorTree(ImageData* image);

	void  AddImage(ImageData* image, int weight);
	void  ProcImage(ImageData* destImage, ImageData* srcImage, int nColors, unsigned long flags);
	int   NumColors();					// Returns number of unique colors in tree w/ leaf indexing
};

////////////////////////////////////////////////////////////////////////////////////////

/////////////////// Quantizer.cpp
Quantizer::Quantizer()
{
	palColors = 0;
	root = NULL;
	pal  = NULL;
	//fp = fopen("c:\\quantdbg.txt","w");
}

Quantizer::~Quantizer()
{
	//fclose(fp);

	if (root)
		delete root;

	if (pal)
		delete [] pal;
}

int Quantizer::NumColors()
{
	int total = 0;

	Link<HDNode*>* curlink = leafList.GetHead();

	while(curlink)
	{
		HDNode* root = curlink->data;

		// If there is a pixel count greater than 0 we know this is a leaf node
		/*
		if (root->nPixels>0)
		{
			fprintf(fp,"[%i,%i,%i,%i]\n",root->sumRed / root->nPixels,
									  root->sumGreen / root->nPixels,
									  root->sumBlue / root->nPixels,
									  root->sumAlpha / root->nPixels);
		}
		*/

		if (root->bLeaf)
			total += root->nPixels;

		curlink = curlink->next;
	}
	
	return total;
}

void Quantizer::Quant8Bit()
{
	color8* sBuf = (color8*)srcImage->buf;
	color8* dBuf = (color8*)destImage->buf;
	int size = srcImage->width * srcImage->height;

	// Build quant color tree
	color32 color;
	int i;

	for(i=0;i<size;i++)
	{
		color.r = srcImage->palette[sBuf->index].r;
		color.g = srcImage->palette[sBuf->index].g;
		color.b = srcImage->palette[sBuf->index].b;
		color.a = srcImage->palette[sBuf->index].a;

		AddColor(&root,NULL,0,color.r,color.g,color.b,color.a,&leafList);
		sBuf++;
	}

	// Prune color tree and build up quantized palette
	PruneTree(maxColors);
	BuildPal();

	CopyPal(destImage);

	sBuf = (color8*)srcImage->buf;
	dBuf = (color8*)destImage->buf;

	for(i=0;i<size;i++)
	{
		dBuf->index = GetClosestColor(color.r,color.g,color.b,color.a);
		sBuf++;
		dBuf++;
	}
}

void Quantizer::Quant16Bit()
{
	color16_555* sBuf = (color16_555*)srcImage->buf;
	color16_555* dBuf = (color16_555*)destImage->buf;
	int size = srcImage->width * srcImage->height;

	for(int i=0;i<size;i++)
	{
		AddColor(&root,NULL,0,sBuf->R(),sBuf->G(),sBuf->B());
		sBuf++;
	}

	// Prune color tree and build up quantized palette
	PruneTree(maxColors);
	BuildPal();

	// Pass 2: Reassign image colors to quantized values
	sBuf = (color16_555*)srcImage->buf;

	if (!flags & QUANT_PALETTE)
	{
		color16_555 color;
		int         index;

		for(int i=0;i<size;i++)
		{
			index = GetClosestColor(sBuf->R(),sBuf->G(),sBuf->B());
			color.R(pal[index].r);
			color.G(pal[index].g);
			color.B(pal[index].b);

			*dBuf = color;
			sBuf++;
			dBuf++;
		}
	}
	else
	{
		CopyPal(destImage);

		color8* dBuf = (color8*)destImage->buf;

		for(int i=0;i<size;i++)
		{
			dBuf->index = GetClosestColor(sBuf->R(),sBuf->G(),sBuf->B());
			sBuf++;
			dBuf++;
		}
	}
}

void Quantizer::Quant24Bit()
{
	color24* sBuf = (color24*)srcImage->buf;
	color24* dBuf = (color24*)destImage->buf;
	int size = srcImage->width * srcImage->height;

	int i;

	// Pass 1: Build color tree
	for(i=0;i<size;i++)
	{
		AddColor(&root,NULL,0,sBuf->r,sBuf->g,sBuf->b);
		sBuf++;
	}

	// Prune color tree and build up quantized palette
	PruneTree(maxColors);
	BuildPal();

	// Pass 2: Reassign image colors to quantized values
	sBuf = (color24*)srcImage->buf;

	if (!flags & QUANT_PALETTE)
	{
		color24 color;
		int     index;

		for(i=0;i<size;i++)
		{
			index = GetClosestColor(sBuf->r,sBuf->g,sBuf->b);
			color.r = pal[index].r;
			color.g = pal[index].g;
			color.b = pal[index].b;

			*dBuf = color;
			sBuf++;
			dBuf++;
		}
	}
	else
	{
		CopyPal(destImage);

		color8* dBuf = (color8*)destImage->buf;

		for(i=0;i<size;i++)
		{
			dBuf->index = GetClosestColor(sBuf->r,sBuf->g,sBuf->b);
			sBuf++;
			dBuf++;
		}
	}
}

void Quantizer::Quant32Bit()
{
	color32* sBuf = (color32*)srcImage->buf;
	color32* dBuf = (color32*)destImage->buf;
	int size = srcImage->width * srcImage->height;

	int i;

	// Pass 1: Build color tree
	for(i=0;i<size;i++)
	{
		AddColor(&root,NULL,0,sBuf->r,sBuf->g,sBuf->b,sBuf->a,&leafList);
		
		sBuf++;
	}

	// Prune color tree and build up quantized palette
	PruneTree(maxColors);
	BuildPal();

	// Pass 2: Reassign image colors to quantized values
	sBuf = (color32*)srcImage->buf;

	if (!flags & QUANT_PALETTE)
	{
		for(i=0;i<size;i++)
		{
			int idx = GetClosestColor(sBuf->r,sBuf->g,sBuf->b,sBuf->a);
			*dBuf = pal[idx];
			sBuf++;
			dBuf++;
		}
	}
	else
	{
		destImage->bit_depth = 8;
		CopyPal(destImage);

		color8* dBuf = (color8*)destImage->buf;

		for(i=0;i<size;i++)
		{
			dBuf->index = GetClosestColor(sBuf->r,sBuf->g,sBuf->b,sBuf->a);
			sBuf++;
			dBuf++;
		}
	}
}


void Quantizer::AddColor(HDNode** pnode,
						 HDNode*  parent,
						 int level,
						 unsigned char r,unsigned char g,unsigned char b,unsigned char a,
						 LinkList<HDNode*>* list)
{
	HDNode* node = *pnode;

	if (!node)
	{
		node = *pnode = new HDNode;
		node->bLeaf = (level == 8) ? true : false;

		node->pParent = parent;

		if (node->bLeaf)
		{
			if (!list)
				list = &leafList;

			node->link = list->AddToTail(&node);
		}
	}

	// We've reached the leaf level, assign color
	if (node->bLeaf)
	{
		node->sumRed   += r;
		node->sumGreen += g;
		node->sumBlue  += b;
		node->sumAlpha += a;

		node->nPixels++;
		return;
	}

	//int color = (r << 24) | (g << 16) | (b << 8) | a;

	/*
	int levelMask = 1 << level;
	
	int rBit = (r & levelMask) >> level;
	int gBit = (g & levelMask) >> level;
	int bBit = (b & levelMask) >> level;
	int aBit = (a & levelMask) >> level;
	*/

	int levelMask = 128 >> level;
	int invlevel = 7 - level;

	// Debug (remove later)
	int rBit = (r & levelMask) >> level;
	int gBit = (g & levelMask) >> level;
	int bBit = (b & levelMask) >> level;
	int aBit = (a & levelMask) >> level;


	// Build a 4-bit key from the rgba bit values for this level
	int child = (((r & levelMask) >> invlevel) << 3) | 
		        (((g & levelMask) >> invlevel) << 2) | 
				(((b & levelMask) >> invlevel) << 1) | 
				 ((a & levelMask) >> invlevel);

	AddColor(&node->pChild[child],node,++level,r,g,b,a,list);
}

void Quantizer::ProcImage(ImageData* destImage, ImageData* srcImage, int nColors, unsigned long flags)
{
	int depth;
	maxColors = nColors;

	this->srcImage  = srcImage;
	this->destImage = destImage;
	this->flags     = flags;

	if (root)
	{
		delete root;
		root = NULL;
	}

	leafList.Clear();

	depth = srcImage->bit_depth * srcImage->nChannels;

	destImage->CopyNoImage(srcImage);

	if (!flags & QUANT_PALETTE)
		destImage->buf = (unsigned char*)malloc(srcImage->imageSize);
	else
	{
		destImage->buf = (unsigned char*)malloc(srcImage->width*srcImage->height);
		destImage->imageSize = srcImage->width*srcImage->height;
		destImage->bit_depth = 8;
		destImage->nChannels = 1;
	}

	// Allocate memory for the leafNode array
	switch(depth)
	{
	case 8:
		Quant8Bit();
		break;

	case 16:
		Quant16Bit();
		break;

	case 24:
		Quant24Bit();
		break;

	case 32:
		Quant32Bit();
		break;
	}

	printf("Palette contains %i colors.\n",leafList.GetSize());

	PruneTree(maxColors);
}

// Reduce the size of the tree by closest color
void Quantizer::PruneTree(int nColors)
{
	int nNodes  = leafList.GetSize();
	int nReduce = nNodes - nColors;

	// If we're already at the proper number of colors we don't need
	// to prune the tree
	if (nNodes <= nColors)
		return;

	printf("Reducing tree by %i colors\n",nReduce);

	while(leafList.GetSize() > nColors)
	{
		// May want to change from head to get better distribution of reduced colors
		Link<HDNode*>* linkA = leafList.GetHead();

		if (!linkA)
			return;

		// Find a sibling
		HDNode* pParent   = linkA->data->pParent;
		int     iSibIndex = -1;

		for(int i=0;i<16;i++)
		{
			if (pParent->pChild[i] && pParent->pChild[i] != linkA->data)
			{
				iSibIndex = i;
				break;
			}
		}
		
		if (iSibIndex == -1)
		{
			// There are no siblings to combine colors with fold this child into its parent
			// and add it to the end of the leaf list (it will be reduced again if we don't find enough
			// colors by the time we get to the end of the list)
			pParent->bLeaf    = true;
			pParent->sumRed   = linkA->data->sumRed;
			pParent->sumGreen = linkA->data->sumGreen;
			pParent->sumBlue  = linkA->data->sumBlue;
			pParent->sumAlpha = linkA->data->sumAlpha;
			pParent->nPixels  = linkA->data->nPixels;

			// Clear out child list
			for(int i=0;i<16;i++)
				pParent->pChild[i] = NULL;

			delete linkA->data;
			leafList.Remove(linkA);
			pParent->link = leafList.AddToTail(&pParent);
			continue;
		}

		HDNode* pSibling = pParent->pChild[iSibIndex];

		// We've found a sibling, combine the two colors
		linkA->data->nPixels  += pSibling->nPixels;
		linkA->data->sumRed   += pSibling->sumRed;
		linkA->data->sumGreen += pSibling->sumGreen;
		linkA->data->sumBlue  += pSibling->sumBlue;
		linkA->data->sumAlpha += pSibling->sumAlpha;

		leafList.Remove(pSibling->link);
		pParent->DeleteChild(iSibIndex);
		pParent->pChild[iSibIndex] = linkA->data;
	}

	printf("Reduced colors to %i.\n",leafList.GetSize());
}

void Quantizer::BuildPal()
{
	if (pal)
		delete [] pal;

	palColors = leafList.GetSize();
	pal = new color32[palColors];

	HDNode* node;
	Link<HDNode*>* curlink = leafList.GetHead();

	int i = 0;

	while(curlink)
	{
		node = curlink->data;

		pal[i].r = node->sumRed / node->nPixels;
		pal[i].g = node->sumGreen / node->nPixels;
		pal[i].b = node->sumBlue / node->nPixels;
		pal[i].a = node->sumAlpha / node->nPixels;

		// Assign computed color index into tree for fast lookups when reassigning image
		node->index = i;

		i++;
		curlink = curlink->next;
	}
}

void Quantizer::CopyPal(ImageData* img)
{
	img->nPalEntries = palColors;

	if (img->palette)
		free(img->palette);

	img->palette = (color32*)malloc(sizeof(color32)*palColors);

	for(int i=0;i<palColors;i++)
	{
		img->palette[i] = pal[i];
	}
}

int Quantizer::GetClosestColor(unsigned char r,
		                       unsigned char g,
					           unsigned char b,
					           unsigned char a,
					           int           level,
					           HDNode*       node)
{
	if (!node)
		node = root;

	if (node->bLeaf)
		return node->index;
	
	int levelMask = 128 >> level;
	int invlevel  = 7 - level;
	
	// Build the 4-bit key from the rgba bit values for this level
	int child = (((r & levelMask) >> invlevel) << 3) | 
		        (((g & levelMask) >> invlevel) << 2) | 
				(((b & levelMask) >> invlevel) << 1) | 
				 ((a & levelMask) >> invlevel);
	
	if (node->pChild[child])
		return GetClosestColor(r,g,b,a,++level,node->pChild[child]);

	// Should never get here (Invalid tree construction)
	assert(0);
	return -1;
}

color32 Quantizer::GetClosestColor32(HDNode* node,
									 unsigned char r,
									 unsigned char g,
									 unsigned char b,
									 unsigned char a,
									 int           level)
{
	if (!node)
		node = root;

	if (node->bLeaf)
	{
		color32 color;
		color.r = (unsigned char)(node->sumRed / node->nPixels);
		color.g = (unsigned char)(node->sumGreen / node->nPixels);
		color.b = (unsigned char)(node->sumBlue / node->nPixels);
		color.a = (unsigned char)(node->sumAlpha / node->nPixels);

		return color;
	}
	
	int levelMask = 128 >> level;
	int invlevel  = 7 - level;
	
	// Build the 4-bit key from the rgba bit values for this level
	int child = (((r & levelMask) >> invlevel) << 3) | 
		        (((g & levelMask) >> invlevel) << 2) | 
				(((b & levelMask) >> invlevel) << 1) | 
				 ((a & levelMask) >> invlevel);
	
	if (node->pChild[child])
		return GetClosestColor32(node->pChild[child],r,g,b,a,++level);

	// The color doesn't exist in the color tree
	// we'll need to find the closest match from this level in the tree
	color32 closest;
	int     dist;

	closest.r = 0;
	closest.g = 0;
	closest.b = 0;
	closest.a = 0;
	
	dist = -1;

	if (node->pParent)
		FindLinearMatch(r,g,b,a,node->pParent,&closest,&dist);
	else
		FindLinearMatch(r,g,b,a,node,&closest,&dist);

	return closest;
}

void Quantizer::FindLinearMatch(unsigned char r,
								unsigned char g,
								unsigned char b,
								unsigned char a,
								HDNode*  node, 
								color32* bestMatch,
								int* dist)
{
	if (node->bLeaf)
	{
		color32 color;

		color.r = (unsigned char)(node->sumRed / node->nPixels);
		color.g = (unsigned char)(node->sumGreen / node->nPixels);
		color.b = (unsigned char)(node->sumBlue / node->nPixels);
		color.a = (unsigned char)(node->sumAlpha / node->nPixels);

		int d	 = Dist2_4D(r,g,b,a,
								color.r,color.g,color.b,color.a);

		if (*dist == -1 || d < *dist)
		{
			bestMatch->r = color.r;
			bestMatch->g = color.g;
			bestMatch->b = color.b;
			bestMatch->a = color.a;

			*dist = d;
		}

		return;
	}

	for(int i=0;i<16;i++)
		if (node->pChild[i])
			FindLinearMatch(r,g,b,a,node->pChild[i],bestMatch,dist);
}

color32 Quantizer::FindColorMatch(TreeData* tdata,
								  unsigned char r,
								  unsigned char g,
								  unsigned char b,
								  unsigned char a)
{
	int nEntries  = tdata->nPalEntries;
	color32*  pal = tdata->pal;

	int small = -1;
	int scolor;

	for(int i=0;i < nEntries; i++)
	{
		int d = Dist2_4D(r,g,b,a,
			                 pal[i].r,pal[i].g,pal[i].b,pal[i].a);

		if (small == -1 || d < small)
		{
			small  = d;
			scolor = i;
		}
	}

	return pal[scolor];
}


void Quantizer::AddImage(ImageData* image,int weight)
{
	
}

TreeData* Quantizer::BuildColorTree(ImageData* image)
{
	TreeData* tdata = new TreeData;
	int depth = image->bit_depth * image->nChannels;
	int size  = image->width * image->height;

	switch(depth)
	{
	case 8:
		{
			color8*           color   = (color8*)image->buf;
			color32*          palette = image->palette;

			for(int i=0;i<size;i++)
			{
				AddColor(&tdata->root,NULL,0,palette[color->index].r,
					                  palette[color->index].g,
									  palette[color->index].b,
									  palette[color->index].a,
									  &tdata->leafList);
				color++;
			}
		}
		break;

	case 16:
		{
			color16_555* color = (color16_555*)image->buf;
			
			for(int i=0;i<size;i++)
			{
				AddColor(&tdata->root,NULL,0,color->R(),color->G(),color->B(),255,&tdata->leafList);
				color++;
			}
		}
		break;

	case 24:
		{
			color24* color = (color24*)image->buf;

			for(int i=0;i<size;i++)
			{
				AddColor(&tdata->root,NULL,0,color->r,color->g,color->b,255,&tdata->leafList);
				color++;
			}
		}
		break;

	case 32:
		{
			color32* color = (color32*)image->buf;

			for(int i=0;i<size;i++)
			{
				AddColor(&root,NULL,0,color->r,color->g,color->b,color->a,&leafList);
				color++;
			}
		}
		break;
	}
	
	// Build up palette
	Link<HDNode*>* curlink = tdata->leafList.GetHead();

	tdata->nPalEntries = tdata->leafList.GetSize();
	tdata->pal = new color32[tdata->nPalEntries];

	int i=0;

	while(curlink)
	{
		HDNode* node = curlink->data;
		
		tdata->pal[i].r = (unsigned char)(node->sumRed / node->nPixels);
		tdata->pal[i].g = (unsigned char)(node->sumGreen / node->nPixels);
		tdata->pal[i].b = (unsigned char)(node->sumBlue / node->nPixels);
		tdata->pal[i].a = (unsigned char)(node->sumAlpha / node->nPixels);

		curlink = curlink->next;
		i++;
	}

	return tdata;
}

//////////////////////////// End Quantizer


////////////////////////////////////////////////////////

//#pragma pack(push,1)		// Force single-byte alignment
struct TGAHeader
{
	unsigned char  IDLength;
	unsigned char  ColorMapType;
	unsigned char  ImageType;
	unsigned short CMapStart;
	unsigned short CMapLength;
	unsigned char  CMapDepth;
	unsigned short XOffset;
	unsigned short YOffset;
	unsigned short Width;
	unsigned short Height;
	unsigned char  PixelDepth;
	unsigned char  ImageDescriptor;
};

struct TGAFooter
{
	unsigned long ExtensionOffset;
	unsigned long DeveloperOffset;
	char          Signature[18];
};

//#pragma pack(pop)

// These are the available ImageType's for the Targa format
#define TGATYPE_NODATA       0
#define TGATYPE_RAWPAL       1
#define TGATYPE_RAWTRUECOLOR 2
#define TGATYPE_RAWMONO      3
#define TGATYPE_PAL          9
#define TGATYPE_TRUECOLOR    10
#define TGATYPE_MONO         11

enum Axis2D
{
	AXIS_WIDTH,
	AXIS_HEIGHT,
	AXIS_NONE,
};

// Global options
bool bClosestNeighbor      = true;
bool bOutputImgPS2         = false;
bool bOutputTarga          = true;
bool bPalettizeImage       = false;
bool bNoCropping           = false;

int  iPreRotateIterations  = 0;
int  iPostRotateIterations = 0;

char crcString[256] = "";

void* ReduceImageSizeCN24(void* srcBuf,
					      unsigned int srcWidth, unsigned int srcHeight,
					      unsigned int destWidth, unsigned int destHeight)
{
	// Allocate memory for destination buffer (owned by caller)
	int memSize = destWidth * destHeight * 3;
	char* destBuf = (char*)malloc(memSize);

	// Calc reduction ratios
	float Xstep = (float)srcWidth / (float)destWidth;
	float Ystep = (float)srcHeight / (float)destHeight;

	unsigned int x,y;

	color24* Dbuf = (color24*)destBuf;
	color24* Sbuf = (color24*)srcBuf;

	for(y=0;y<destHeight;y++)
		for(x=0;x<destWidth;x++)
			Dbuf[y*destWidth+x] = Sbuf[(int)((int)(y*Ystep)*srcWidth+(float)x*Xstep)];

	return destBuf;
}

void* ReduceImageSizeCN32(void* srcBuf,
						  unsigned int srcWidth, unsigned int srcHeight,
						  unsigned int destWidth, unsigned int destHeight)
{
	// Allocate memory for destination buffer (owned by caller)
	int memSize = destWidth * destHeight * 4;
	char* destBuf = (char*)malloc(memSize);

	// Calc reduction ratios
	float Xstep = (float)srcWidth / (float)destWidth;
	float Ystep = (float)srcHeight / (float)destHeight;

	unsigned int x,y;

	color32* Dbuf = (color32*)destBuf;
	color32* Sbuf = (color32*)srcBuf;

	for(y=0;y<destHeight;y++)
		for(x=0;x<destWidth;x++)
			Dbuf[y*destWidth+x] = Sbuf[(int)((int)(y*Ystep)*srcWidth+(float)x*Xstep)];

	return destBuf;
}

void* Proc24bitBoxReduce(void* srcBuf,
						 unsigned int srcWidth, unsigned int srcHeight,
						 unsigned int destWidth, unsigned int destHeight)
{
	// Allocate memory for destination buffer (owned by caller)
	int memSize = destWidth * destHeight * 3;
	char* destBuf = (char*)malloc(memSize);

	// Calc reduction ratios
	float Xstep = (float)srcWidth / (float)destWidth;
	float Ystep = (float)srcHeight / (float)destHeight;

	unsigned int x,y,i;

	// Matrix for computing neighboring pixels
	const Point Matrix[8] = { Point(-1,-1),
							  Point( 0,-1),
							  Point( 1,-1),
							  Point(-1, 0),
							  Point( 1, 0),
							  Point(-1, 1),
							  Point( 0, 1),
							  Point( 1, 1) };
	
	color24* Dbuf = (color24*)destBuf;
	color24* Sbuf = (color24*)srcBuf;

	color24 bestColor;

	for(y=0;y<destHeight;y++)
		for(x=0;x<destWidth;x++)
		{
			unsigned int freqR = 0;		// Total Red component around sampled area
			unsigned int freqG = 0;		// Total Green component around sampled area
			unsigned int freqB = 0;		// Total Blue component around sampled area
			unsigned int nSamples = 1;	// Number of samples computed from matrix

			//Point ptSrc((int)((float)y*Ystep),(int)((float)x*Xstep));
			Point ptSrc((int)((float)x*Xstep),(int)((float)y*Ystep));

			// Assign the sample point color
			bestColor = Sbuf[ptSrc.y*srcWidth+ptSrc.x];

			freqR = bestColor.r;
			freqG = bestColor.g;
			freqB = bestColor.b;

			// Find neighboring pixel values
			for(i=0;i<8;i++)
			{
				Point ptSample(ptSrc.x+Matrix[i].x,
							   ptSrc.y+Matrix[i].y);

				if (ptSample.x > -1 && ptSample.x < (int)srcWidth &&
					ptSample.y > -1 && ptSample.y < (int)srcHeight)
				{
					// Find sample color for this location in the matrix
					color24 color  = Sbuf[(int)((float)ptSample.y*srcWidth+(float)ptSample.x)];

					freqR += color.r;
					freqG += color.g;
					freqB += color.b;
					nSamples++;
				}
			}

			// Find the average color for the surrounding area
			bestColor.r = (unsigned char)(freqR / nSamples);
			bestColor.g = (unsigned char)(freqG / nSamples);
			bestColor.b = (unsigned char)(freqB / nSamples);

			// Use the average color of the region
			Dbuf[y*destWidth+x] = bestColor;
		}

	return Dbuf;
}

void* Proc32bitBoxReduce(void* srcBuf,
						 unsigned int srcWidth, unsigned int srcHeight,
						 unsigned int destWidth, unsigned int destHeight)
{
	// Allocate memory for destination buffer (owned by caller)
	int memSize = destWidth * destHeight * 4;
	char* destBuf = (char*)malloc(memSize);

	// Calc reduction ratios
	float Xstep = (float)srcWidth / (float)destWidth;
	float Ystep = (float)srcHeight / (float)destHeight;

	unsigned int x,y,i;

	// Matrix for computing neighboring pixels
	const Point Matrix[8] = { Point(-1,-1),
							  Point( 0,-1),
							  Point( 1,-1),
							  Point(-1, 0),
							  Point( 1, 0),
							  Point(-1, 1),
							  Point( 0, 1),
							  Point( 1, 1) };
	
	color32* Dbuf = (color32*)destBuf;
	color32* Sbuf = (color32*)srcBuf;

	color32 bestColor;

	for(y=0;y<destHeight;y++)
		for(x=0;x<destWidth;x++)
		{
			unsigned int freqR = 0;		// Total Red component around sampled area
			unsigned int freqG = 0;		// Total Green component around sampled area
			unsigned int freqB = 0;		// Total Blue component around sampled area
			unsigned int freqA = 0;		// Total Alpha component around sampled area

			Point ptSrc((int)((float)x*Xstep),(int)((float)y*Ystep));

			// Assign the sample point color
			bestColor = Sbuf[ptSrc.y*srcWidth+ptSrc.x];

			freqR = bestColor.r;
			freqG = bestColor.g;
			freqB = bestColor.b;

			// Find neighboring pixel values
			for(i=0;i<8;i++)
			{
				Point ptSample(ptSrc.x+Matrix[i].x,
							   ptSrc.y+Matrix[i].y);

				if (ptSample.x > -1 && ptSample.x < (int)srcWidth &&
					ptSample.y > -1 && ptSample.y < (int)srcHeight)
				{
					// Find sample color for this location in the matrix
					color32 color  = Sbuf[(int)((float)ptSample.y*srcWidth+(float)ptSample.x)];

					freqR += color.r;
					freqG += color.g;
					freqB += color.b;
				}
			}

			// Find the average color for the surrounding area
			bestColor.r = (unsigned char)(freqR / 9);
			bestColor.g = (unsigned char)(freqG / 9);
			bestColor.b = (unsigned char)(freqB / 9);

			// Use the average color of the region
			Dbuf[y*destWidth+x] = bestColor;
		}

	return Dbuf;
}

void* Rotate90Image24(void* image,
					  unsigned short* pWidth, unsigned short* pHeight,
					  int mode)
{
	color24* srcImage = (color24*)image;
	unsigned short srcWidth  = *pWidth;
	unsigned short srcHeight = *pHeight;

	color24* destImage = (color24*)malloc(sizeof(color24) * srcWidth * srcHeight);

	unsigned short x, y;

	switch(mode)
	{
	case 1:	// 90 degrees CW		
		for(y = 0; y < srcHeight; y++)
			for(x = 0; x < srcWidth; x++)
				destImage[x * srcHeight + y] = srcImage[y * srcWidth + srcWidth - x];

		*pWidth  = srcHeight;
		*pHeight = srcWidth;
		break;

	case 2: // 180 degrees CW
		for(y = 0; y < srcHeight; y++)
			for(x = 0; x < srcWidth; x++)
				destImage[y * srcWidth + x] = srcImage[(srcHeight - y - 1) * srcWidth + srcWidth - x];

		break;

	case 3: // 270 degrees CW
		for(y = 0; y < srcHeight; y++)
			for(x = 0; x < srcWidth; x++)
				destImage[x * srcHeight + y] = srcImage[(srcHeight - y - 1) * srcWidth + x];

		*pWidth  = srcHeight;
		*pHeight = srcWidth;
		break;
	}
	
	return destImage;
}

void* Rotate90Image32(void* image,
					  unsigned short* pWidth, unsigned short* pHeight,
					  int mode)
{
	color32* srcImage = (color32*)image;
	unsigned short srcWidth  = *pWidth;
	unsigned short srcHeight = *pHeight;

	color32* destImage = (color32*)malloc(sizeof(color32) * srcWidth * srcHeight);

	unsigned short x, y;

	switch(mode)
	{
	case 1:	// 90 degrees CW		
		for(y = 0; y < srcHeight; y++)
			for(x = 0; x < srcWidth; x++)
				destImage[x * srcHeight + y] = srcImage[y * srcWidth + srcWidth - x];

		*pWidth  = srcHeight;
		*pHeight = srcWidth;
		break;

	case 2: // 180 degrees CW
		for(y = 0; y < srcHeight; y++)
			for(x = 0; x < srcWidth; x++)
				destImage[y * srcWidth + x] = srcImage[(srcHeight - y) * srcWidth + srcWidth - x];

		break;

	case 3: // 270 degrees CW
		for(y = 0; y < srcHeight; y++)
			for(x = 0; x < srcWidth; x++)
				destImage[x * srcHeight + y] = srcImage[(srcHeight - y) * srcWidth + x];

		*pWidth  = srcHeight;
		*pHeight = srcWidth;
		break;
	}
	
	return destImage;
}

// Builds image of destination size filled to the appropriate color with source image centered
void* Crop24(void* image, 
			 int   srcWidth,  int srcHeight,
			 int   destWidth, int destHeight)
{
	color24* srcImage = (color24*)image;

	if (destWidth > srcWidth)
		return NULL;

	if (destHeight > srcHeight)
		return NULL;
	
	color24* destImage = (color24*)malloc(sizeof(color24) * destWidth * destHeight);

	int x, y;

	// Compute the start offsets into the original image
	int preX = (int)((srcWidth - destWidth) / 2.0f);
	int preY = (int)((srcHeight - destHeight) / 2.0f);

	for(y = 0; y < destHeight; y++)
		for(x = 0; x < destWidth; x++)
		{
			// Width and height need to be cropped off from both directions
			destImage[y * destWidth + x] = srcImage[(y + preY) * srcWidth + x + preX];
		}
	
	return destImage;
}

void* Crop32(void* image,
			 int   srcWidth, int srcHeight,
			 int   destWidth, int destHeight)
{
	color32* srcImage = (color32*)image;

	if (destWidth > srcWidth)
		return NULL;

	if (destHeight > srcHeight)
		return NULL;

	color32* destImage = (color32*)malloc(sizeof(color32) * destWidth * destHeight);

	int x, y;

	// Compute the start offsets into the original image
	int preX = (int)((srcWidth - destWidth) / 2.0f);
	int preY = (int)((srcHeight - destHeight) / 2.0f);

	for(y = 0; y < destHeight; y++)
		for(x = 0; x < destWidth; x++)
		{
			// Width and height need to be cropped off from both directions
			destImage[y * destWidth + x] = srcImage[(y + preY) * srcWidth + x + preX];
		}
	
	return destImage;
}

void GetDimensions(int  origWidth,  int origHeight,
				   int  destWidth,  int destHeight,
				   int* scaleWidth, int* scaleHeight,
				   int* finalWidth, int* finalHeight)
{
	// Determine the major axis
	Axis2D axisMajor;

	if (origWidth == origHeight)
		axisMajor = AXIS_NONE;
	else 
		if (origWidth > origHeight)
			axisMajor = AXIS_WIDTH;
		else
			axisMajor = AXIS_HEIGHT;

	switch(axisMajor)
	{
	case AXIS_WIDTH:
		//if (bCropMajor)
		{
			*scaleHeight = destHeight;
			
			// Ensure aspect ratio is maintained
			float aspect = (float)origWidth / (float)origHeight;
			*scaleWidth  = (int)(aspect * (float)destHeight);
			
			*finalWidth  = destWidth;
			*finalHeight = destHeight;
			break;
		}
		break;

	case AXIS_HEIGHT:
		//if (bCropMajor)
		{
			*scaleWidth = destWidth;
			
			// Ensure aspect ratio is maintained
			float aspect = (float)origHeight / (float)origWidth;
			*scaleHeight = (int)(aspect * (float)destWidth);

			*finalWidth  = destWidth;
			*finalHeight = destHeight;
		}
		break;

	default:
		// If the aspect is 1:1 then a direct scaling is all that is needed
		*scaleWidth  = destWidth;
		*scaleHeight = destHeight;
		*finalWidth  = *scaleWidth;
		*finalHeight = *scaleHeight;
		break;
	}
}

//NxTexture* BufferToTexture(void* buf, int srcDepth

void SetImage(ImageData* img, int width, int height, int depth)
{
	img->width     = width;
	img->height    = height;

	switch(depth)
	{
	case 32:
		img->nChannels = 4;
		break;

	case 24:
		img->nChannels = 3;
		break;

	default:
		img->nChannels = 3;
		break;
	}
	
	img->imageSize   = width * height * img->nChannels;
	img->pixel_depth = depth;

	// Input is always true color for now
	img->palette     = NULL;
	img->nPalEntries = 0;
	img->bit_depth   = 8;
}

ImageData* CreateImage(void* buf, int width, int height, int depth)
{
	ImageData* newImage = new ImageData;

	newImage->width     = width;
	newImage->height    = height;

	switch(depth)
	{
	case 32:
		newImage->nChannels = 4;
		break;

	case 24:
		newImage->nChannels = 3;
		break;

	default:
		newImage->nChannels = 3;
		break;
	}
	
	newImage->imageSize   = width * height * newImage->nChannels;
	newImage->pixel_depth = depth;

	// Input is always true color for now
	newImage->palette     = NULL;
	newImage->nPalEntries = 0;
	newImage->bit_depth   = 8;

	newImage->buf = (unsigned char*)malloc(newImage->imageSize);
	memcpy(newImage->buf, buf, newImage->imageSize);

	return newImage;
}

NxTexture* ConvertImageDataToTexture(ImageData* img)
{
	NxTexture* newTexture = new NxTexture();

	// No additional mips only root level
	newTexture->m_mip_levels       = 0;
	newTexture->m_orig_width[0]    = img->width;
	newTexture->m_orig_height[0]   = img->height;
    newTexture->m_checksum         = GenerateCRC(crcString);
	
	newTexture->m_width[0]         = img->width;
	newTexture->m_height[0]        = img->height;

	newTexture->m_flags            = mFORCE_BYTE_PER_COMPONENT;

	if (img->nPalEntries > 0)
	{
		if (img->nPalEntries < 257)
			newTexture->m_pixel_format = v8_BIT;
		else
		{
			printf("Unexpected number of palette entries > 256\n");
			exit(-3);
		}

		switch(img->pixel_depth)
		{
		case 32:
			newTexture->m_palette_format = v32_BIT;
			newTexture->m_palette_bpp = 32;
			break;

		case 24:
			newTexture->m_palette_format = v24_BIT;
			newTexture->m_palette_bpp = 24;
			break;

		default:
			printf("Unknown pixel depth for palette (Expected 32/24)\n");
			exit(-3);
		}

		newTexture->m_bpp = 8;
	
		// Copy palette data
		//newTexture->m_num_palette_entries = img->nPalEntries;
		newTexture->m_num_palette_entries = 256;	// img.ps2 has a fixed palette size based on bitdepth
		newTexture->m_palette_data = new char[((newTexture->m_palette_bpp / 8) * 256)];
		
		unsigned char* pPalPos = (unsigned char*)newTexture->m_palette_data;

		for(int i = 0; i < img->nPalEntries; i++)
		{
			switch(newTexture->m_palette_bpp)
			{
			case 24:
				*pPalPos = img->palette[i].r;
				pPalPos++;
				*pPalPos = img->palette[i].g;
				pPalPos++;
				*pPalPos = img->palette[i].b;
				pPalPos++;
				break;

			case 32:
				*pPalPos = img->palette[i].r;
				pPalPos++;
				*pPalPos = img->palette[i].g;
				pPalPos++;
				*pPalPos = img->palette[i].b;
				pPalPos++;
				*pPalPos = img->palette[i].a;
				pPalPos++;
				break;

			default:
				printf("Unknown size when transfering palette.  Expected (32/24)\n");
				exit(-1);
			}	
		}

		// Generate fake palette entries (if our palette had less than 256 colors)
		for(int i = img->nPalEntries; i < 256; i++)
		{
			switch(newTexture->m_palette_bpp)
			{
			case 24:
				*pPalPos = 0;
				pPalPos++;
				*pPalPos = 0;
				pPalPos++;
				*pPalPos = 0;
				pPalPos++;
				break;

			case 32:
				*pPalPos = 0;
				pPalPos++;
				*pPalPos = 0;
				pPalPos++;
				*pPalPos = 0;
				pPalPos++;
				*pPalPos = 0;
				pPalPos++;
				break;

			default:
				printf("Unknown size when transfering palette.  Expected (32/24)\n");
				exit(-1);
			}	
		}

		// Copy texel data
		newTexture->m_texel_data[0] = new char[(img->imageSize)];
		memcpy(newTexture->m_texel_data[0], img->buf, img->imageSize);

		// Only 8-bit output for now
		//if (nPalEntries < 17)
		//	newTexture->m_pixel_format = v4_BIT;
	}
	else
	{
		// Non-paletted should be 24 or 32 bit
		switch(img->pixel_depth)
		{
		case 24:
			newTexture->m_pixel_format = v24_BIT;
			newTexture->m_bpp = 24;
			break;

		case 32:
			newTexture->m_pixel_format = v32_BIT;
			newTexture->m_bpp = 32;
			break;

		default:
			printf("Non expected pixel output format expecting 24/32 bit for non-paletted output\n");
			exit(-3);
		}

		// Copy texel data
		newTexture->m_texel_data[0] = (char*)malloc(img->imageSize);
		memcpy(newTexture->m_texel_data[0], img->buf, img->imageSize);
	}

	return newTexture;
}

void SwapRedBlue32(unsigned char* pData, int size)
{
	color32* color = (color32*)pData;
	unsigned char tmp;
	int unit_size = size / sizeof(color24);

	for(int i = 0; i < unit_size; i++)
	{
		tmp = (*color).r;
		(*color).r = (*color).b;
		(*color).b = tmp;
		color++;
	}
}

void SwapRedBlue24(unsigned char* pData, int size)
{
	color24* color = (color24*)pData;
	unsigned char tmp;
	int unit_size = size / sizeof(color24);

	for(int i = 0; i < unit_size; i++)
	{
		tmp = (*color).r;
		(*color).r = (*color).b;
		(*color).b = tmp;
		color++;
	}
}

int main(int argc, char* argv[])
{
	if (argc < 4)
	{
		printf("Targa Aspect Resize\n");
		printf("Neversoft Entertainment,  2003  aml\n\n");
		
		printf("aspectresize [srcfile] [outputfile] [destwidth] [destheight] {options}\n\n");

		printf("srcfile      - Source file to do conversion on (Must be 24/32 bit raw Targa)\n");
		printf("outputfile   - Destination file with scaled/cropped image\n");
		printf("               (This will be a 24/32 bit Targa or internal .img.ps2\n");
		printf("destwidth    - Desired width of output image in pixels\n");
		printf("destheight   - Desired height of output image in pixels\n");
		printf("-nc          - No cropping. Image will be stretched (aspect not preserved)\n");
		printf("-bf          - Use box filtering for scaling (defaults to closest neighbor)\n");
		printf("-br90        - Pre-rotate the image by 90 degrees clockwise (-br180, -br270)\n");
		printf("-er90        - Post-rotate the image by 90 degrees clockwise (-er180, -er270)\n");
		printf("-p[plat]     - Dumps output file as .img for given platform (defaults Targa)\n");
		printf("               p = PS2  g = Gamecube [No Support]  x = XBOX [No support]\n");
		
		printf("-pal         - Force output to be palettized (assumes 8-bit palette)\n");
		printf("-crc[string] - If exporting to .img format this is the checksum value that\n");
		printf("               should be embedded in the file (defaults to output filename)\n\n");

		printf("The minor axis will be assign destination size, the major axis will be assigned\n");
		printf("a scale to keep it 1:1.  The major axis will then be cropped.\n");
		return 0;
	}

	// Determine what options are set
	for(int opt = 5; opt < argc; opt++)
	{
		if (strstr(argv[opt], "-bf"))
			bClosestNeighbor = false;

		if (strstr(argv[opt],"-nc"))
			bNoCropping = true;

		if (strstr(argv[opt], "-br90"))
			iPreRotateIterations = 1;

		if (strstr(argv[opt], "-br180"))
			iPreRotateIterations = 2;

		if (strstr(argv[opt], "-br270"))
			iPreRotateIterations = 3;

		if (strstr(argv[opt], "-er90"))
			iPostRotateIterations = 1;

		if (strstr(argv[opt], "-er180"))
			iPostRotateIterations = 2;

		if (strstr(argv[opt], "-er270"))
			iPostRotateIterations = 3;

		if (strstr(argv[opt], "-pp"))
		{
			bOutputTarga  = false;
			bOutputImgPS2 = true;
		}

		if (strstr(argv[opt], "-pg"))
		{
			printf("Gamecube platform not supported! Aborted.\n");
			exit(1);
		}

		if (strstr(argv[opt], "-px"))
		{
			printf("XBOX platform not supported! Aborted.\n");
			exit(1);
		}

		if (strstr(argv[opt], "-pal"))
			bPalettizeImage = true;

		if (strstr(argv[opt], "-crc"))
			strcpy(crcString, argv[opt] + 4);
	}

	// Ensure that a crc string exists (if one doesn't, we use the output file)
	if (crcString[0] == 0)
		strcpy(crcString, argv[2]);

	// Acquire destination width and height
	int destWidth  = atoi(argv[3]);
	int destHeight = atoi(argv[4]);

	FILE* fp = fopen(argv[1], "rb");

	if (!fp)
	{
		printf("ERROR: Failed to open '%s' for input.\n", argv[1]);
		return -1;
	}

	TGAHeader header;
	//fread(&header, sizeof(TGAHeader), 1, fp);
	fread((char*) &header.IDLength, sizeof( unsigned char ), 1, fp );
	fread((char*) &header.ColorMapType, sizeof( unsigned char ), 1, fp );
	fread((char*) &header.ImageType, sizeof( unsigned char ), 1, fp );
	fread((char*) &header.CMapStart, sizeof( unsigned short ), 1, fp );
	fread((char*) &header.CMapLength, sizeof( unsigned short ), 1, fp );
	fread((char*) &header.CMapDepth, sizeof( unsigned char ), 1, fp );
	fread((char*) &header.XOffset, sizeof( unsigned short ), 1, fp );
	fread((char*) &header.YOffset, sizeof( unsigned short ), 1, fp );
	fread((char*) &header.Width, sizeof( unsigned short ), 1, fp );
	fread((char*) &header.Height, sizeof( unsigned short ), 1, fp );
	fread((char*) &header.PixelDepth, sizeof( unsigned char ), 1, fp );
	fread((char*) &header.ImageDescriptor, sizeof( unsigned char ), 1, fp );

	if (header.ImageType != TGATYPE_RAWTRUECOLOR)
	{
		printf("ERROR: Only raw non RLE encoded true color Targas are currently supported.\n");
		return -1;
	}

	if (header.PixelDepth != 24 && header.PixelDepth != 32)
	{
		printf("ERROR: Only 24 and 32 bit targas are currently supported.\n");
		return -1;
	}

	// Adjust dimensions appropriately
	// We only should be scaling the image on the major axis
	int scaleWidth, scaleHeight;
	int finalWidth, finalHeight;

	// Allocate space and load in the image data based on bit depth
	if (header.PixelDepth == 24)
	{
		color24* image = (color24*)malloc(sizeof(color24) * header.Width * header.Height);
		fread(image, sizeof(color24), header.Width * header.Height, fp);
		fclose(fp);

		// If we're exporting to non-targa we need to swap red and blue since Targa stores data BGR not RGB
		if (!bOutputTarga)
			SwapRedBlue24((unsigned char*)image, sizeof(color24)* header.Width * header.Height);

		// Perform pre-rotation
		color24* imageRot;

		if (iPreRotateIterations)
		{
			imageRot = (color24*)Rotate90Image24(image, &header.Width, &header.Height, iPreRotateIterations);
			memcpy(image, imageRot, sizeof(color24) * header.Width * header.Height);
			free(imageRot);
		}

		GetDimensions(header.Width, header.Height,
			          destWidth, destHeight,
			          &scaleWidth, &scaleHeight, 
					  &finalWidth, &finalHeight);

		color24* imageScale;
		color24* imageFinal;

		if (bNoCropping)
		{
			imageScale = NULL;
			
			if (!bClosestNeighbor)
			{
				imageFinal = (color24*)Proc24bitBoxReduce(image, 
														header.Width, header.Height,
														finalWidth, finalHeight);
			}
			else
			{
				imageFinal = (color24*)ReduceImageSizeCN24(image,
														header.Width, header.Height,
														finalWidth, finalHeight);
			}
		}
		else
		{
			if (!bClosestNeighbor)
			{
				imageScale = (color24*)Proc24bitBoxReduce(image, 
														header.Width, header.Height,
														scaleWidth, scaleHeight);
			}
			else
			{
				imageScale = (color24*)ReduceImageSizeCN24(image,
														header.Width, header.Height,
														scaleWidth, scaleHeight);
			}

			imageFinal = (color24*)Crop24(imageScale,
			                              scaleWidth, scaleHeight, 
								          finalWidth, finalHeight);
		}

		header.Width  = finalWidth;
		header.Height = finalHeight;

		// Perform post-rotation
		if (iPostRotateIterations)
		{
			imageRot = (color24*)Rotate90Image24(imageFinal, &header.Width, &header.Height, iPostRotateIterations);
			memcpy(imageFinal, imageRot, sizeof(color24) * header.Width * header.Height);
			free(imageRot);
		}

		// Palettize the image if appropriate
		ImageData* imageSrc  = CreateImage(imageFinal, header.Width, header.Height, 24);
		ImageData* imageDest = new ImageData();
		
		if (bPalettizeImage)
		{
			Quantizer quant;
			quant.ProcImage(imageDest, imageSrc, 256, QUANT_PALETTE);
			header.ImageType    = TGATYPE_RAWPAL;
			header.ColorMapType = 1;
			header.CMapStart    = 0;
			header.CMapLength   = imageDest->nPalEntries;
			header.CMapDepth    = header.PixelDepth;
			header.PixelDepth   = 8;
		}
		else
		{
			SetImage(imageDest, header.Width, header.Height, 24);

			imageDest->buf = (unsigned char*)malloc((header.PixelDepth / 8) * header.Width * header.Height);
			memcpy(imageDest->buf, imageFinal, (header.PixelDepth / 8) * header.Width * header.Height);
		}
		
		if (bOutputTarga)
		{
			fp = fopen(argv[2], "wb");

			if (!fp)
			{
				printf("Failed to open '%s' for writting.\n", argv[2]);
				exit(2);
			}

			/*fwrite(&header, sizeof(TGAHeader), 1, fp);*/
			fwrite((char*) &header.IDLength, sizeof( unsigned char ), 1, fp );
			fwrite((char*) &header.ColorMapType, sizeof( unsigned char ), 1, fp );
			fwrite((char*) &header.ImageType, sizeof( unsigned char ), 1, fp );
			fwrite((char*) &header.CMapStart, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.CMapLength, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.CMapDepth, sizeof( unsigned char ), 1, fp );
			fwrite((char*) &header.XOffset, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.YOffset, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.Width, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.Height, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.PixelDepth, sizeof( unsigned char ), 1, fp );
			fwrite((char*) &header.ImageDescriptor, sizeof( unsigned char ), 1, fp );

			// Write out the palette entries
			if (header.ColorMapType)
			{
				// Colormap depth will be either 24 or 32 bit
				switch(header.CMapDepth)
				{
				case 24:
					{
						int i;
						for(i = 0; i < imageDest->nPalEntries; i++)
						{
							fwrite(&imageDest->palette[i].r, 1, 1, fp);
							fwrite(&imageDest->palette[i].g, 1, 1, fp);
							fwrite(&imageDest->palette[i].b, 1, 1, fp);
						}
					}
					break;

				case 32:
					{
						for(int i = 0; i < imageDest->nPalEntries; i++)
						{
							fwrite(&imageDest->palette[i].r, 1, 1, fp);
							fwrite(&imageDest->palette[i].g, 1, 1, fp);
							fwrite(&imageDest->palette[i].b, 1, 1, fp);
							fwrite(&imageDest->palette[i].a, 1, 1, fp);
						}
					}
					break;

				default:
					printf("Unexpected quantized color depth!\n");
					return -1;
				}
				
			}

			fwrite(imageDest->buf, sizeof( unsigned char ) * (header.PixelDepth / 8), header.Width * header.Height, fp);
			fclose(fp);

			printf("Successfully wrote '%s' as Targa\n", argv[2]);
		}

		if (bOutputImgPS2)
		{
			NxTexture* pTexture = ConvertImageDataToTexture(imageDest);
			
			if (!pTexture->Convert24BitPaletteFormatTo32BitPaletteFormat())
			{
				printf("Failed to convert 24-bit palette to 32-bit palette.  Operation Aborted!\n");
				exit(3);
			}

			if (pTexture->SaveTextureForNGPS(argv[2]))
			{
				printf("Successfully wrote '%s' as img.ps2\n", argv[2]);
			}
			else
			{
				printf("Failed to open '%s' for writting.\n", argv[2]);
				exit(2);
			}

			delete pTexture;
		}

		if (image)
			free(image);

		if (imageScale)
			free(imageScale);

		if (imageFinal)
			free(imageFinal);

		delete imageSrc;
		delete imageDest;
	}

	/////////////////////////// 32-bit

	if (header.PixelDepth == 32)
	{
		color32* image = (color32*)malloc(sizeof(color32) * header.Width * header.Height);
		fread(image, sizeof(color32), header.Width * header.Height, fp);
		fclose(fp);

		// If we're exporting to non-targa we need to swap red and blue since Targa stores data BGR not RGB
		if (!bOutputTarga)
			SwapRedBlue32((unsigned char*)image, sizeof(color32)* header.Width * header.Height);

		// Perform pre-rotation
		color32* imageRot;

		if (iPreRotateIterations)
		{
			imageRot = (color32*)Rotate90Image32(image, &header.Width, &header.Height, iPreRotateIterations);
			memcpy(image, imageRot, sizeof(color32) * header.Width * header.Height);
			free(imageRot);
		}

		GetDimensions(header.Width, header.Height,
			          destWidth, destHeight,
			          &scaleWidth, &scaleHeight, 
					  &finalWidth, &finalHeight);

		color32* imageScale;
		color32* imageFinal;

		if (bNoCropping)
		{
			imageScale = NULL;

			if (!bClosestNeighbor)
			{
				imageFinal = (color32*)Proc32bitBoxReduce(image,
														header.Width, header.Height,
														finalWidth, finalHeight);
			}
			else
			{
				imageFinal = (color32*)ReduceImageSizeCN32(image,
														header.Width, header.Height,
														finalWidth, finalHeight);
			}
		}
		else
		{
			if (!bClosestNeighbor)
			{
				imageScale = (color32*)Proc32bitBoxReduce(image,
														header.Width, header.Height,
														scaleWidth, scaleHeight);
			}
			else
			{
				imageScale = (color32*)ReduceImageSizeCN32(image,
														header.Width, header.Height,
														scaleWidth, scaleHeight);
			}

			imageFinal = (color32*)Crop32(imageScale,
										  scaleWidth, scaleHeight,
										  finalWidth, finalHeight);
		}

		header.Width  = finalWidth;
		header.Height = finalHeight;

		// Perform post-rotation
		if (iPostRotateIterations)
		{
			imageRot = (color32*)Rotate90Image32(imageFinal, &header.Width, &header.Height, iPostRotateIterations);
			memcpy(imageFinal, imageRot, sizeof(color24) * header.Width * header.Height);
			free(imageRot);
		}

		// Palettize the image if appropriate
		ImageData* imageSrc  = CreateImage(imageFinal, header.Width, header.Height, 32);
		ImageData* imageDest = new ImageData();
		
		if (bPalettizeImage)
		{
			Quantizer quant;
			quant.ProcImage(imageDest, imageSrc, 256, QUANT_PALETTE);
			header.ImageType    = TGATYPE_RAWPAL;
			header.ColorMapType = 1;
			header.CMapStart    = 0;
			header.CMapLength   = imageDest->nPalEntries;
			header.CMapDepth    = header.PixelDepth;
			header.PixelDepth   = 8;
		}
		else
		{
			SetImage(imageDest, header.Width, header.Height, 32);

			imageDest->buf = (unsigned char*)malloc((header.PixelDepth / 8) * header.Width * header.Height);
			memcpy(imageDest->buf, imageFinal, (header.PixelDepth / 8) * header.Width * header.Height);
		}

		if (bOutputTarga)
		{
			fp = fopen(argv[2], "wb");

			if (!fp)
			{
				printf("Failed to open '%s' for writting.\n", argv[2]);
				exit(2);
			}

			// fwrite(&header, sizeof(TGAHeader), 1, fp);
			fwrite((char*) &header.IDLength, sizeof( unsigned char ), 1, fp );
			fwrite((char*) &header.ColorMapType, sizeof( unsigned char ), 1, fp );
			fwrite((char*) &header.ImageType, sizeof( unsigned char ), 1, fp );
			fwrite((char*) &header.CMapStart, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.CMapLength, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.CMapDepth, sizeof( unsigned char ), 1, fp );
			fwrite((char*) &header.XOffset, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.YOffset, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.Width, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.Height, sizeof( unsigned short ), 1, fp );
			fwrite((char*) &header.PixelDepth, sizeof( unsigned char ), 1, fp );
			fwrite((char*) &header.ImageDescriptor, sizeof( unsigned char ), 1, fp );

			// Write out the palette entries
			if (header.ColorMapType)
			{
				// Colormap depth will be either 24 or 32 bit
				switch(header.CMapDepth)
				{
				case 24:
					{
						int i;
						for(i = 0; i < imageDest->nPalEntries; i++)
						{
							fwrite(&imageDest->palette[i].r, 1, 1, fp);
							fwrite(&imageDest->palette[i].g, 1, 1, fp);
							fwrite(&imageDest->palette[i].b, 1, 1, fp);
						}
					}
					break;

				case 32:
					{
						for(int i = 0; i < imageDest->nPalEntries; i++)
						{
							fwrite(&imageDest->palette[i].r, 1, 1, fp);
							fwrite(&imageDest->palette[i].g, 1, 1, fp);
							fwrite(&imageDest->palette[i].b, 1, 1, fp);
							fwrite(&imageDest->palette[i].a, 1, 1, fp);
						}
					}
					break;

				default:
					printf("Unexpected quantized color depth!\n");
					return -1;
				}
				
			}

			fwrite(imageDest->buf, sizeof( unsigned char ) * (header.PixelDepth / 8), header.Width * header.Height, fp);
			fclose(fp);
			
			printf("Successfully wrote '%s'\n", argv[2]);
		}

		if (bOutputImgPS2)
		{
			NxTexture* pTexture = ConvertImageDataToTexture(imageDest);
			
			// No need to convert format should already be in 32-bit
			if (pTexture->m_palette_format != v32_BIT)
			{
				printf("Invalid palette format (not 32-bit)! Aborted.\n");
				exit(2);
			}

			if (pTexture->SaveTextureForNGPS(argv[2]))
			{
				printf("Successfully wrote '%s' as img.ps2\n", argv[2]);
			}
			else
			{
				printf("Failed to open '%s' for writting.\n", argv[2]);
				exit(2);
			}

			delete pTexture;
		}

		if (image)
			free(image);

		if (image)
			free(imageScale);

		if (imageFinal)
			free(imageFinal);

		delete imageSrc;
		delete imageDest;
	}
}

//////////////////////////////// NxTexture from PngConv for .img support Implementation ////////

#include <stdlib.h>
#include <stdio.h>

const unsigned int TEXTURE_VERSION = 2;

#define	GXPackedRGB565(r,g,b)   ((unsigned short)((((r)&0xf8)<<8)|(((g)&0xfc)<<3)|(((b)&0xf8)>>3)))

#define	GXPackedRGBA4(r,g,b,a)  ((unsigned short)((((r)&0xf0)<<8)|(((g)&0xf0)<<4)|(((b)&0xf0)   )|(((a)&0xf0)>>4)))

#define	GXPackedRGB5A3(r,g,b,a) ((unsigned short)((a)>=224 ? ((((r)&0xf8)<<7)|(((g)&0xf8)<<2)|(((b)&0xf8)>>3)|(1<<15)): ((((r)&0xf0)<<4)|(((g)&0xf0)   )|(((b)&0xf0)>>4)|(((a)&0xe0)<<7))))

/*>*******************************(*)*******************************<*/
// switch tuple and byte order within 16-bit words of an s3-packed tile
// to match hw.
// 1) switch 2-bit tuple order within bytes
//    from ( 0,1,2,3 ) to ( 3,2,1,0 ).
// 2) leave byte order within the word as is.
/*>*******************************(*)*******************************<*/
void TCFixCMPWord( unsigned short* data )
{
	unsigned short tmp;


	tmp = *data;

	// reverse tuple order within bytes
	*data = ( (tmp & 0x3 )   << 6 ) |
			( (tmp & 0xC )   << 2 ) |
			( (tmp & 0x30)   >> 2 ) |
			( (tmp & 0xC0)   >> 6 ) |

            ( (tmp & 0x300 ) << 6 ) |
			( (tmp & 0xC00 ) << 2 ) |
			( (tmp & 0x3000) >> 2 ) |
			( (tmp & 0xC000) >> 6 ) ;
}

/*>*******************************(*)*******************************<*/
// reverse the byte order within a block of bytes.
// do this only if TPL_BIG_END is defined
/*>*******************************(*)*******************************<*/ 
void TCFixEndian( unsigned char* src, unsigned int numBytes )
{
	unsigned char  tmp[8];  // large enough to hold a double
	unsigned int max, i;

//    TCAssertMsg( (numBytes <= 8), "TCFixEndian: numBytes > 8\n" );

	if( (numBytes == 0) || (numBytes == 1) )
	{
		return;
	}
					
	max = numBytes - 1;
	for(i=0; i< numBytes; i++)
	{
		tmp[(max - i)] = src[i];	
	}
	
	for(i=0; i< numBytes; i++)
	{
		src[i] = tmp[i];
	}
}

/*>*******************************(*)*******************************<*/ 
// TCPackTile_CMP
//
// pack a 2x2 tile block, each tile of 4x4 texels, into a single 
// 32B dst tile note: this assumes s3 algorithm pads out to a minimum 
// block size of 4x4 texels
/*>*******************************(*)*******************************<*/ 
void TCPackTile_CMP ( char * pData, int width, int height, unsigned int tileX, unsigned int tileY, unsigned short* dstPtr)
{
	unsigned int  x, y;
	unsigned short* srcPtr;
	unsigned short  tmp;
	unsigned int  srcTileOffset;
	unsigned int  subTileRows, subRowShorts;    // number of s3 4x4 tiles
	unsigned int  srcPadWidth, srcPadHeight;
	unsigned short* buffPtr;

	// set the padded size of the s3 source image out to a 4-texel boundary
	srcPadWidth  = ( (width  + 3) >> 2 );
	srcPadHeight = ( (height + 3) >> 2 );

	// number of bytes in a single row of 4x4 texel source tiles
	srcTileOffset = srcPadWidth * 8;

	// number of 4x4 (source) tile rows to copy ( will be 1 or 2 )
	subTileRows = 2;
	if( (srcPadHeight - tileY) < 2 )
		subTileRows = 1;

	// number of 4x4 tile cols to copy translated into number of short values
	// ( will be 4 or 8 )
	subRowShorts = 8;
	if( (srcPadWidth - tileX) < 2 )
		subRowShorts = 4;

	for( y=0; y < subTileRows; y++ )
	{
		srcPtr  = (unsigned short*)( (unsigned char*)(pData) + ((tileY + y) * srcTileOffset) + (tileX*8) ); 
		buffPtr = ( dstPtr + (y * 8) );        // 16 bytes per subRow = 8 shorts

		// process one or both 4x4 row tiles at once- 4 short each
		for( x=0; x < subRowShorts; x++ )
		{			
			switch( x )
			{

			// color table entries - switch bytes within a 16-bit world only
			case 0:	
			case 1:
			case 4:
			case 5:
				tmp = *srcPtr++;
				TCFixEndian( (unsigned char*)(&tmp), 2 );
				*buffPtr++ = tmp;
				break;
			
			// 2-bit color tuples;
			// reverse tuple order within bytes of a word
			case 2:
			case 3:
			case 6:
			case 7:
				tmp = *srcPtr++;
				TCFixCMPWord( &tmp );
				*buffPtr++ = tmp;
				break;

			} // end switch
		} // end for( subRowShorts )			
	} // end for( subTileRows )
}

/*>*******************************(*)*******************************<*/ 
// TCWriteTplImage_CMP
//
/*>*******************************(*)*******************************<*/ 
void FixCompressedTexture ( char * pData, int width, int height, char * pFixed )
{
	unsigned int tileRow, tileCol;
	unsigned int srcTileRows, srcTileCols;
	unsigned short* dstPtr;

	// each source tile is 4x4 texels, 8B
	srcTileRows   = ((height + 3) >> 2);
	srcTileCols   = ((width  + 3) >> 2);

	dstPtr = (unsigned short*)(pFixed);

	// each dst tile is 2x2 source tiles, so move by 2 each iteration
	for(tileRow = 0; tileRow < srcTileRows; tileRow += 2 )
	{
		for(tileCol = 0; tileCol < srcTileCols; tileCol += 2 )
		{
			TCPackTile_CMP( pData, width, height, tileCol, tileRow, dstPtr );
			dstPtr += 16; // 32B per dst tile, short ptr
		}
	}
}

NxTexture::NxTexture( void )
{
	int i;

	for( i = 0; i < vMAX_MIP_LEVELS; i++ )
	{
		m_width[i] = 0;
		m_height[i] = 0;
		m_texel_data[i] = NULL;
	}

	m_bpp = 0;
	m_palette_bpp = 0;
	m_pixel_format = v32_BIT;
	m_palette_format = v32_BIT;	
	m_palette_data = NULL;
	sprintf( m_name, "None" );
	//m_map = NULL;	
	m_mip_levels = 0;
	m_flags = 0;
	//m_transparent_color.r = 0.0f;
	//m_transparent_color.g = 0.0f;
	//m_transparent_color.b = 0.0f;
	m_valid = true;
}

NxTexture::~NxTexture( void )
{
	int i;

	for( i = 0; i < vMAX_MIP_LEVELS; i++ )
	{
		if( m_texel_data[i] )
		{			
			delete [] m_texel_data[i];
		}
	}

	if( m_palette_data )
	{
		delete [] m_palette_data;
	}
}

bool NxTexture::ShouldAutoGenerateMipMaps( void )
{
	return (( m_flags & mAUTO_GENERATE_MIPMAPS ) != 0 );
}

void NxTexture::SetNumMipLevels( int num_mip_levels )
{
	m_mip_levels = num_mip_levels;
}

int	NxTexture::GetNumMipLevels( void )
{
	return m_mip_levels;
}

int	NxTexture::GetPixelFormat( void )
{
	return m_pixel_format;
}

int	NxTexture::GetPaletteFormat( void )
{
	return m_palette_format;
}

char* NxTexture::GetTexelData( int mip_level )
{
	return m_texel_data[mip_level];
}

char* NxTexture::GetPaletteData( void )
{
	return m_palette_data;
}

char*	NxTexture::GetName( void )
{
	return m_name;
}

int	NxTexture::GetWidth( int mip_level )
{
	return m_width[mip_level];
}

int NxTexture::GetHeight( int mip_level )
{
	return m_height[mip_level];
}

void NxTexture::SetWidth( int mip_level, int width )
{
	m_width[mip_level] = width;
}
	
void NxTexture::SetHeight( int mip_level, int height )
{
	m_height[mip_level] = height;
}

bool NxTexture::IsPaletted( void )
{
	return (( m_pixel_format == v8_BIT ) || ( m_pixel_format == v4_BIT ));
}

int NxTexture::GetBpp( void )
{
	return m_bpp;
}

int NxTexture::GetPaletteBpp( void )
{
	return m_palette_bpp;
}

int NxTexture::GetNumPaletteEntries( void )
{
	return m_num_palette_entries;
}

int	NxTexture::GetTexelDataSize( int mip_level )
{
	int bits_of_texel_data;

	bits_of_texel_data = GetBpp() * GetWidth( mip_level) * GetHeight( mip_level );
	assert(( bits_of_texel_data % 8 ) == 0 );

	return bits_of_texel_data >> 3;
}

int NxTexture::GetPaletteDataSize( void )
{
	int bits_of_palette_data;

	bits_of_palette_data = GetPaletteBpp() * m_num_palette_entries;
	assert(( bits_of_palette_data % 8 ) == 0 );

	return bits_of_palette_data >> 3;
}

bool	NxTexture::ShouldCompressComponents( void )
{
	return (( m_flags & mFORCE_BYTE_PER_COMPONENT ) == 0 );
}

bool	NxTexture::ShouldChangeTransparentColor( void )
{
	return (( m_flags & mCHANGE_FULLY_TRANSPARENT_COLOR ) != 0 );
}

void	NxTexture::SetFlags( int flags )
{
	m_flags |= flags;
}

int		NxTexture::GetFlags( void )
{
	return m_flags;
}

unsigned long	NxTexture::GetChecksum( void )
{
	return m_checksum;
}

bool	NxTexture::operator==( NxTexture& texture )
{
	return( m_checksum == texture.GetChecksum());
}

int	NxTexture::GetTotalDataSize( void )
{
	int i;
	int total_size;

	total_size = GetPaletteDataSize();
	for( i = 0; i <= GetNumMipLevels(); i++ )
	{
		total_size += GetTexelDataSize( i );
	}

	return total_size;
}

void	NxTexture::SetValidity( bool valid )
{
	m_valid = valid;
}

bool	NxTexture::IsValid( void )
{
	return m_valid;
}

#define	vADC_ON		0
#define	vADC_OFF	0x00008000

#define vPSMCT32	0
#define vPSMCT24	1
#define vPSMCT16	2
#define vPSMT8		19
#define vPSMT4		20

bool NxTexture::Convert4BitPixelFormatTo8BitPixelFormat( void )
{
	if( !IsPaletted() || ( m_pixel_format != v4_BIT ))
	{
		return false;
	}

	for( int i = 0; i <= m_mip_levels; ++i )
	{
		unsigned int	texel_index			= 0;
		unsigned char*	p_8bit_texel_data	= new unsigned char[m_width[i] * m_height[i]];

		int texel_data_size = ( m_width[i] * m_height[i] ) / 2;
		for( int p = 0; p < texel_data_size; ++p )
		{
			// Read indices.
			unsigned int index0 = ((unsigned char)m_texel_data[i][p] >> 0 ) & 0xF;
			unsigned int index1 = ((unsigned char)m_texel_data[i][p] >> 4 ) & 0xF;

			p_8bit_texel_data[texel_index++] = index0;
			p_8bit_texel_data[texel_index++] = index1;
		}

		delete [] m_texel_data[i];

		m_texel_data[i]			= (char*)p_8bit_texel_data;
	}
			
	// Change the format.
	m_pixel_format	= v8_BIT;
	m_bpp			= 8;
	return true;
}



bool NxTexture::Convert16BitPaletteFormatTo32BitPaletteFormat( void )
{
	if( !IsPaletted() || ( m_palette_format != v16_BIT ))
	{
		return false;
	}

	unsigned int*	p_32bit_palette_data	= new unsigned int[m_num_palette_entries];

	for( int p = 0; p < m_num_palette_entries; ++p )
	{
		unsigned int entry	= ((unsigned short*)m_palette_data )[p];
		entry				= (( entry & 0x1F ) << 3 ) | ((( entry >> 5 ) & 0x1F ) << 11 ) | ((( entry >> 10 ) & 0x1F ) << 19 ) | (( entry & 0x8000 ) ? 0xFF000000 : 0x00 );

		p_32bit_palette_data[p] = entry;
	}

	delete [] m_palette_data;

	m_palette_data		= (char*)p_32bit_palette_data;
	m_palette_format	= v32_BIT;
	m_palette_bpp		= 32;

	return true;
}



bool NxTexture::Convert24BitPaletteFormatTo32BitPaletteFormat( void )
{
	if( !IsPaletted() || ( m_palette_format != v24_BIT ))
	{
		return false;
	}

	unsigned int	*p_32bit_palette_data	= new unsigned int[this->m_num_palette_entries];
	unsigned char	*p_24bit_palette_data	= (unsigned char*)m_palette_data;
	for( int p = 0; p < this->m_num_palette_entries; ++p )
	{
		unsigned int red	= *p_24bit_palette_data++;
		unsigned int grn	= *p_24bit_palette_data++;
		unsigned int blu	= *p_24bit_palette_data++;
		unsigned int entry	= red | ( grn << 8 ) | ( blu << 16 ) | 0xFF000000;

		p_32bit_palette_data[p] = entry;
	}

	delete [] m_palette_data;

	m_palette_data		= (char*)p_32bit_palette_data;
	m_palette_format	= v32_BIT;
	m_palette_bpp		= 32;

	return true;
}





bool NxTexture::Convert32BitRGBAPaletteFormatTo32BitBGRAPaletteFormat( void )
{
	if( !IsPaletted() || ( m_palette_format != v32_BIT ))
	{
		return false;
	}

	unsigned int *p_32bit_palette_data	= (unsigned int*)m_palette_data;
	for( int p = 0; p < this->m_num_palette_entries; ++p )
	{
		unsigned int old		= *p_32bit_palette_data;
		unsigned int red		= old & 0xFF;
		unsigned int grn		= ( old >> 8 ) & 0xFF;
		unsigned int blu		= ( old >> 16 ) & 0xFF;
		unsigned int alp		= ( old >> 24 ) & 0xFF;
		*p_32bit_palette_data++	= blu | ( grn << 8 ) | ( red << 16 ) | ( alp << 24 );
	}

	return true;
}



bool NxTexture::ConvertTo32BitPixelFormat( void )
{
	// Currently only supports 16 and 32 bit palettes (if a palette is present).
	if( IsPaletted() && ( m_palette_format != v32_BIT ) && ( m_palette_format != v24_BIT ) && ( m_palette_format != v16_BIT ))
	{
		return false;
	}

	switch( m_pixel_format )
	{
		case v4_BIT:
		{
			// Just convert to 8 bit pixel format, and fall through to next case.
			if( !Convert4BitPixelFormatTo8BitPixelFormat())
			{
				return false;
			}
		}
	
		case v8_BIT:
		{
			// Convert to 32 bit palette if not already.
			if( m_palette_format == v16_BIT )
			{
				if( !Convert16BitPaletteFormatTo32BitPaletteFormat())
				{
					return false;
				}
			}

			if( m_palette_format == v24_BIT )
			{
				if( !Convert24BitPaletteFormatTo32BitPaletteFormat())
				{
					return false;
				}
			}

			for( int i = 0; i <= this->m_mip_levels; ++i )
			{
				unsigned int	texel_index			= 0;
				unsigned int*	p_32bit_texel_data	= new unsigned int[m_width[i] * m_height[i]];

				int texel_data_size = m_width[i] * m_height[i];
				
				unsigned char*	p_texel_data	= (unsigned char*)m_texel_data[i];
				for( int p = 0; p < texel_data_size; ++p )
				{
					// Read indices.
					unsigned int entry;
					unsigned int index0					= *p_texel_data++;

					// Must be in 32 bit palette format at this stage.
					entry								= ((unsigned int*)m_palette_data )[index0];
					p_32bit_texel_data[texel_index++]	= entry;
				}

				delete [] m_texel_data[i];

				m_texel_data[i]			= (char*)p_32bit_texel_data;
			}
			
			// Now we can free up the palette data...
			delete [] m_palette_data;
			m_palette_data = NULL;

			// ...and change format..
			m_pixel_format	= v32_BIT;
			m_bpp			= 32;
			return true;
		}

		case v24_BIT:
		{
			for( int i = 0; i <= m_mip_levels; ++i )
			{
				unsigned int*	p_32bit_texel_data	= new unsigned int[m_width[i] * m_height[i]];
				unsigned char*	p_24bit_texel_data	= (unsigned char*)m_texel_data[i];

				for( int p = 0; p < ( m_width[i] * m_height[i] ); ++p )
				{
					// Read indices.
					unsigned int red		= *p_24bit_texel_data++;
					unsigned int grn		= *p_24bit_texel_data++;
					unsigned int blu		= *p_24bit_texel_data++;
					p_32bit_texel_data[p]	= red | ( grn << 8 ) | ( blu << 16 ) | 0xFF000000UL;
				}

				delete [] m_texel_data[i];

				m_texel_data[i]			= (char*)p_32bit_texel_data;
			}

			// Change the format.
			m_pixel_format	= v32_BIT;
			m_bpp			= 32;
			return true;
		}

		default:
		{
			break;
		}
	}

	// Cannot convert.
	return false;
}

bool NxTexture::Convert32BitRGBAPixelFormatTo32BitBGRAPixelFormat( void )
{
	if( m_pixel_format != v32_BIT )
	{
		return false;
	}

	for( int i = 0; i <= m_mip_levels; ++i )
	{
		unsigned int *p_32bit_texel_data	= (unsigned int*)m_texel_data[i];
		for( int p = 0; p < ( m_width[i] * m_height[i] ); ++p )
		{
			unsigned int old		= *p_32bit_texel_data;

			// Note - there appears to be an issue where zero-alpha pixels have their color set to 0x00FFFFFF.
			// Try setting to 0x0000000 in this case...
			if( old == 0x00FFFFFFUL )
				old = 0x00000000UL;
			
			unsigned int red		= old & 0xFF;
			unsigned int grn		= ( old >> 8 ) & 0xFF;
			unsigned int blu		= ( old >> 16 ) & 0xFF;
			unsigned int alp		= ( old >> 24 ) & 0xFF;
			*p_32bit_texel_data++	= blu | ( grn << 8 ) | ( red << 16 ) | ( alp << 24 );
		}
	}

	return true;
}

bool NxTexture::SaveTextureForNGPS( char *path )
{
	bool success = false;

	//printf("Opening file '%s'\n", path);
	static char	m_tex_buff[1024 * 1024 * 4];

	CVirtualOutputFile theOutputFile;

	// 10-meg buffer
	theOutputFile.Init(6*1024*1024);

	int j, k;
	
	int final_bpp, final_palette_bpp, max_level;
	unsigned long checksum;

	theOutputFile.Write((const char*) &TEXTURE_VERSION, sizeof( unsigned int ));
				
	checksum = m_checksum;
	theOutputFile.Write((const char*) &checksum, sizeof( unsigned long ));
	if( 1 /*m_checksum != vINVALID_CHECKSUM*/ )
	{
		int log_width, log_height;
		unsigned short orig_width, orig_height;
		int pixel_mode, clut_mode;								
		
		log_width = GetLog2( m_width[0] );
		log_height = GetLog2( m_height[0] );
		theOutputFile.Write((const char*) &log_width, sizeof( int ));
		theOutputFile.Write((const char*) &log_height, sizeof( int ));

		switch( m_pixel_format )
		{
			case v32_BIT:
				pixel_mode = vPSMCT32;
				final_bpp = 32;
				final_palette_bpp = 0;
				break;
			case v24_BIT:
				if( m_flags & mFORCE_BYTE_PER_COMPONENT )
				{
					pixel_mode = vPSMCT24;
					final_bpp = 24;
					final_palette_bpp = 0;						
				}
				else
				{
					pixel_mode = vPSMCT16;
					final_bpp = 16;
					final_palette_bpp = 0;
				}
				break;
			case v16_BIT:
				pixel_mode = vPSMCT16;
				final_bpp = 16;
				final_palette_bpp = 0;
				break;
			case v8_BIT:
				pixel_mode = vPSMT8;
				final_bpp = 8;						
				break;
			case v4_BIT:
				pixel_mode = vPSMT4;
				final_bpp = 4;
				break;
			default:
				goto save_error;
		}

		switch( m_palette_format )
		{
			case v32_BIT:
				clut_mode = vPSMCT32; 
				final_palette_bpp = 32;
				break;
			case v16_BIT:
				clut_mode = vPSMCT16;
				final_palette_bpp = 16;
				break;
			case v24_BIT:
				if( m_flags & mFORCE_BYTE_PER_COMPONENT )
				{
					clut_mode = vPSMCT32; 
					final_palette_bpp = 32;
				}
				else
				{
					clut_mode = vPSMCT16;
					final_palette_bpp = 16;						
				}
				break;
			default:
				goto save_error;
		}			

		// Get the max mip level
		max_level = m_mip_levels;

		// Get the original size
		orig_width = m_orig_width[0];
		orig_height = m_orig_height[0];
			
		// Write out pixel/clut storage modes
		theOutputFile.Write((const char*) &pixel_mode, sizeof( int ));
		theOutputFile.Write((const char*) &clut_mode, sizeof( int ));			
		theOutputFile.Write((const char*) &max_level, sizeof( int ));
		theOutputFile.Write((const char*) &orig_width, sizeof( short ));
		theOutputFile.Write((const char*) &orig_height, sizeof( short ));

		if( IsPaletted())
		{
			// 16-byte align palette data
			theOutputFile.Align( 16 );

			if( m_palette_format == v32_BIT )
			{
				// PS2's alpha range goes from 0 (fully-transparent) to 128 (fully-opaque)
				// so scale down the standard 0-255 range here
				for( int m = 0; m < GetPaletteDataSize(); m += 4 )
				{
					unsigned int cur_alpha;
					cur_alpha = (unsigned char) m_palette_data[m + 3];
					m_palette_data[m + 3] = ((cur_alpha * 128) / 255);
				}
				theOutputFile.Write((const char*) m_palette_data, GetPaletteDataSize() );
			}
			else
			{
				int size;

				if( m_flags & mFORCE_BYTE_PER_COMPONENT )
				{
					
					int num_entries;
					unsigned char* src;
					unsigned char* dst;

					// Right now these are the only two we expect
					assert( m_palette_format == v24_BIT );

					// Convert 24-bit palette to 32-bit palette						
					src = (unsigned char*) m_palette_data;
					dst = (unsigned char*) m_tex_buff;
					num_entries = m_num_palette_entries;
					for( j = 0; j < num_entries; j++ )
					{
						*dst++ = *src++;
						*dst++ = *src++;
						*dst++ = *src++;
						*dst++ = 128;	// fully opaque
					}

					size = (int) dst - (int) m_tex_buff;
				}
				else
				{
					int num_entries;
					unsigned char* src;
					unsigned short* dst;

					// Right now these are the only two we expect
					assert( m_palette_format == v24_BIT );

					// Convert 24-bit palette to 16-bit palette						
					src = (unsigned char*) m_palette_data;
					dst = (unsigned short*) m_tex_buff;
					num_entries = m_num_palette_entries;
					for( j = 0; j < num_entries; j++ )
					{
						unsigned char red, blue, green;
						unsigned short color;

						red = ( *src++ ) >> 3;
						green = ( *src++ ) >> 3;
						blue = ( *src++ ) >> 3;
							
						color = 0x8000 | red | ( green << 5 ) | ( blue << 10 );
						*dst++ = color;						
					}

					size = (int) dst - (int) m_tex_buff;
				}

				// Write out the texel data
				theOutputFile.Write((const char*) m_tex_buff, size );
			}
		}

		for( j = 0; j <= m_mip_levels; j++ )
		{
			// 16-byte align texture data
			theOutputFile.Align( 16 );
				
			// 24-bits will be converted to 16bit for PS2 unless overridden by user
			if(	( m_pixel_format == v24_BIT ) &&
				( !( m_flags & mFORCE_BYTE_PER_COMPONENT )))
			{
				unsigned char* src;
				unsigned short* dst;
				int num_pixels;

				src = (unsigned char*) m_texel_data[j];
				dst = (unsigned short*) m_tex_buff;
				num_pixels = m_height[j] * m_width[j];
				for( k = 0; k < num_pixels; k++ )
				{
					unsigned char red, blue, green;
					unsigned short color;

					red = ( *src++ ) >> 3;
					green = ( *src++ ) >> 3;
					blue = ( *src++ ) >> 3;
						
					color = red | ( green << 5 ) | ( blue << 10 );
					*dst++ = color;						
				}				

				// Write out the texel data
				theOutputFile.Write((const char*) m_tex_buff, (int) dst - (int) m_tex_buff );
			}
			else if( m_pixel_format == v32_BIT )
			{
				// PS2's alpha range goes from 0 (fully-transparent) to 128 (fully-opaque)
				// so scale down the standard 0-255 range here
				for( int m = 0; m < GetTexelDataSize(j); m += 4 )
				{
					unsigned int cur_alpha;
					cur_alpha = (unsigned char) m_texel_data[j][m + 3];
					m_texel_data[j][m + 3] = ((cur_alpha * 128) / 255);
				}

				// Write out the texel data
				theOutputFile.Write((const char*) m_texel_data[j], GetTexelDataSize(j) );
			}
			else
			{
				// Write out the texel data
				theOutputFile.Write((const char*) m_texel_data[j], GetTexelDataSize(j) );
			}					
		}
	}

	if ( !theOutputFile.Save( path ) )
	{
		goto save_error;
	}

	success = true;

save_error:

	return success;
}

////////////////////////////////////////////////////////////////////////////////////////////////
