/*
	aniqngen.cpp
	Animation range to .qn file generation tool
	2-19-01
*/

#include <stdio.h>
#include <stdlib.h>
#include <direct.h>
#include <io.h>
#include <sys/stat.h>
#include <string.h>
#include <conio.h>
#include <time.h>
#include "llist.h"

// GJ:  to prevent a script from getting too big
// (causing fragmentation of the script heap),
// we break up long auto-generated scripts into
// smaller ones.
// Example: 
// script animload_thps5_human
//		animload_thps5_human_01
//		animload_thps5_human_02
//		animload_thps5_human_03
//		animload_thps5_human_04
//		animload_thps5_human_05
// endscript
const int vMAX_ANIMS_PER_FUNCTION = 50;

enum	EANIMTYPE {
	ANIM_NORMAL = 0,
	ANIM_UNLOADABLE = 1,
};

struct FileRecord
{
	char AnimFile[256];
	char Suffix[256];
	char DirName[256];
	char SkelName[256];
	EANIMTYPE AnimType; 
	bool	is_ped;
	
	FileRecord()
	{
		AnimFile[0] = 0;
		Suffix[0]   = 0;
		DirName[0]  = 0;
		SkelName[0] = 0;
		AnimType = ANIM_NORMAL;
		is_ped = false;
	}
};

struct AnimRecord
{
	char AnimName[256];
	LinkList<FileRecord> list;

	bool HasUnloadableEntries();
};


char *stristr(const char *String, const char *Pattern)
{
      char *pptr, *sptr, *start;

      for (start = (char *)String; *start != 0; start++)
      {
            /* find start of pattern in string */
            for ( ; ((*start!=0) && (toupper(*start) != toupper(*Pattern))); start++)
                  ;

            pptr = (char *)Pattern;
            sptr = (char *)start;

            while (toupper(*sptr) == toupper(*pptr))
            {
                  sptr++;
                  pptr++;

                  /* if end of pattern then pattern was found */

                  if (0 == *pptr)
                        return (start);
            }
      }
      return NULL;
}



bool AnimRecord::HasUnloadableEntries()
{
	Link<FileRecord>* link = list.GetHead();

	while(link)
	{
		if (link->data.AnimType == ANIM_UNLOADABLE)
			return true;

		link = link->next;
	}

	return false;
}

LinkList<AnimRecord> DB;

#define  QNANIM_PATH "anims\\"			// Appended in name= on LoadAnim script autogen

void LCase(char* str)
{
	int len=strlen(str);

	for(int i=0;i<len;i++)
		str[i]=tolower(str[i]);
}

Link<AnimRecord>* AddAnim(char* Modelname,char* file,char* suffix,char* dir,char* skel,EANIMTYPE anim_type=ANIM_NORMAL, bool is_ped = true)
{
	FileRecord frec;
	char       tmpstr1[256];
	char       tmpstr2[256];
	strcpy(frec.AnimFile,file);
	strcpy(frec.Suffix,suffix);
	strcpy(frec.DirName,dir);
	strcpy(frec.SkelName,skel);
	frec.AnimType = anim_type;
	frec.is_ped = is_ped;

//	printf ("%d: %s\n",is_ped, Modelname);

	char modelNameLC[256];
	strcpy(modelNameLC, Modelname);

	_strlwr(modelNameLC);
	if (strstr(modelNameLC, "ped_m_walk4"))
	{
		int zzz;
		zzz = 0;
	}

	if (strstr(modelNameLC, "ped_m_run2"))
	{
		int zzz;
		zzz = 0;
	}

	// Search for the modelname
	Link<AnimRecord>* curNode=DB.GetHead();

	while(curNode)
	{
		strcpy(tmpstr1,curNode->data.AnimName);
		strcpy(tmpstr2,Modelname);

		LCase(tmpstr1);
		LCase(tmpstr2);

		if (strcmp(tmpstr1,tmpstr2)==0)
		{
			curNode->data.list.Add(&frec);
			return curNode;
		}

		curNode=curNode->next;
	}

	Link<AnimRecord>* arec=DB.Add();
	strcpy(arec->data.AnimName,Modelname);
	arec->data.list.Add(&frec);	

	return arec;
}

void GetPath(char* dst,int size,char* src)
{
	char* lstpos=strrchr(src,'\\');

	if (!lstpos)
	{
		if (size<3)
			return;

		dst[0]='.';
		dst[1]='\0';
		return;
	}

	int slen=strlen(src);

	for(int i=0;i<slen;i++)
	{
		if (i>size)
			return;

		if (src+i==lstpos)
		{
			dst[i]='\0';

			// If there aren't any \'s we probably have something like "c:" and one
			if (!strstr(dst,"\\"))
				strcat(dst,"\\");

			return;
		}

		dst[i]=src[i];
	}
}

void GetFile(char* dst,int size,char* src)
{
	char* lstpos=strrchr(src,'\\');

	int slen;

	if (!lstpos)
	{
		slen=strlen(src);

		for(int i=0;i<slen;i++)
		{
			if (i>size)
				return;

			dst[i]=src[i];
		}

		dst[i]='\0';
		return;
	}

	lstpos++;
	slen=strlen(lstpos);

	for(int i=0;i<slen;i++)
	{
		if (i>size)
			return;

		dst[i]=lstpos[i];
	}

	dst[i]='\0';
	return;
}

void Find(char* path,char* prefix,void (*fpCallback)(char* filename,void* data),void* pData)
{
	_finddata_t fdata;
	_chdir(path);

	char WorkingDir[256];
	_getcwd(WorkingDir,255);

	long hFindFirst=_findfirst("*.*",&fdata);

	if (hFindFirst==-1)
		return;

	int  hFile=0;

	while(hFile!=-1)
	{
		if (fdata.attrib & _A_SUBDIR)
		{
			if (strcmp(fdata.name,".")==0 ||
				strcmp(fdata.name,"..")==0)
			{
				hFile=_findnext(hFindFirst,&fdata);
				continue;
			}

			Find(fdata.name,prefix,fpCallback,pData);
			_chdir(WorkingDir);
		}

		char strLName[256];
		char strLPrefix[256];

		strcpy(strLName,fdata.name);
		strcpy(strLPrefix,prefix);

		LCase(strLName);
		LCase(strLPrefix);

		if (strstr(strLName,strLPrefix))
			fpCallback(fdata.name,pData);

		hFile=_findnext(hFindFirst,&fdata);
	}
}

void ProcAnr(char* filename,void* data)
{
	char WorkingDir[256];
	_getcwd(WorkingDir,255);

	FILE* fp=fopen(filename,"r");

	if (!fp)
	{
		printf("\nWARNING!: Failed to open '%s\\%s'.\n\tThe file is being ignored.",WorkingDir,filename);
		return;
	}
	else
		printf("\n\tProcessing %s\\%s...",WorkingDir,filename);

	char modelname[80];
	char suffixname[80];
	char dirname[80];
	char skelname[80];
	char boneset[80];

	char verID[6];
	int  version=0;
	int  active=1;
	EANIMTYPE anim_type=ANIM_NORMAL;

	verID[0]=fgetc(fp);
	verID[1]=fgetc(fp);
	verID[2]=fgetc(fp);
	verID[3]=fgetc(fp);
	verID[4]=fgetc(fp);	// Should be CR

	// Check if this is an animation range version 3 format file
	if (strstr(verID,"AnV3"))
	{
		version=3;
		fscanf(fp,"%s\n%s\n%s\n",modelname,suffixname,dirname);

		// ! Temporary change per Gary's suggestion REMOVE LATER !
		if (strstr(modelname,"Ped") || strstr(modelname,"human"))
		{
			version=4;
			strcpy(skelname,"human");
		}
	}
	else if (strstr(verID,"AnV4"))
	{
		version=4;
		fscanf(fp,"%s\n%s\n%s\n%s\n",modelname,suffixname,dirname,skelname);
	}
	else if (strstr(verID,"AnV5"))
	{
		version = 5;
		fscanf(fp,"%s\n%s\n%s\n%s\n",modelname,suffixname,dirname,skelname);
	}
	else if (strstr(verID,"AnV6"))
	{
		version = 6;
		fscanf(fp,"%s\n%s\n%s\n%s\n",modelname,suffixname,dirname,skelname);
	}
	else if (strstr(verID,"AnV7"))
	{
		version = 7;
		fscanf(fp,"%s\n%s\n%s\n%s\n",modelname,suffixname,dirname,skelname);
	}
	else if (strstr(verID,"AnV8"))
	{
		version = 8;
		//fscanf(fp,"%s\n%s\n%s\n%s\n%s\n",modelname,suffixname,dirname,skelname,boneset);
		fscanf(fp,"%s\n%s\n%s\n%s\n",modelname,suffixname,dirname,skelname);
		fgets(boneset, 256, fp);
	}
	else if (strstr(verID,"AnV9"))
	{
		version = 9;
		fscanf(fp,"%s\n%s\n%s\n%s\n",modelname,suffixname,dirname,skelname);
		fgets(boneset, 256, fp);
	}
	else
	{
		// If this isn't an animation range version 3 file we can't continue
		printf("\nWARNING!: File '%s\\%s'\n\t\t is using an outdated file format.\n\tPlease reexport this .anr file.  The file is being ignored.",WorkingDir,filename);
		fclose(fp);
		return;
	}

	if (strstr(modelname,"[none]"))
		modelname[0]='\0';

	if (strstr(suffixname,"[none]"))
		suffixname[0]='\0';

	if (strstr(skelname,"[none]"))
		skelname[0]='\0';

	// Scan in each animation
	while(!feof(fp))
	{
		char name[256];

		int ivalStart,ivalEnd;
		float fCompress,fTCompress;

		if (version>=6)
		{
			int nscanned;

			if (version == 6)
			{
				nscanned = fscanf(fp,"%s %i %i %f %f %i\n",name,&ivalStart,&ivalEnd,&fCompress,&fTCompress,&active);

				if (nscanned < 6)
					break;
			}
			else if (version == 7 || version == 8)
			{
				nscanned = fscanf(fp,"%s %i %i %f %f %i %i\n",name,&ivalStart,&ivalEnd,&fCompress,&fTCompress,&active,&anim_type);

				if (nscanned < 7)
					break;
			}
			else if (version == 9)
			{
				nscanned = fscanf(fp,"%s %i %i %f %f %i %i\n",name,&ivalStart,&ivalEnd,&fCompress,&fTCompress,&active,&anim_type);

				if (nscanned < 7)
					break;

				char partialAnim[256];
				fgets(partialAnim, 256, fp);
			}
		}
		else
		{
			int nscanned;
			nscanned = fscanf(fp,"%s %i %i %f %f\n",name,&ivalStart,&ivalEnd,&fCompress,&fTCompress);

			if (nscanned < 5)
				break;
		}

		if (version>=5)
		{
			// Read in per bone animation tolerance info (or rather skip over it)
			char lineBuf[1024];

			do
			{
				fgets(lineBuf,1023,fp);
			} while (strstr(lineBuf,"[End]")==0);
		}

		if (fCompress==0.0f)
			fCompress=0.999700f;

		if (fTCompress==0.0f)
			fTCompress=1.0f;

		// Detect special animations			
//		if (stristr(dirname,"thps5_skater_special"))
//		{
//			anim_type = ANIM_SPECIAL;
//		}

		// Detect pedestrian specific animations		
		bool is_ped = false;
		if (!strcmpi(dirname,"ped_m") || !strcmpi(dirname,"ped_sitting"))
		{
//			printf ("\n%s is a ped\n",dirname);
			is_ped = true;
		}
		else
		{
//			printf ("\n%s is NOT\n",dirname);		
		}

		if (version==3)
			AddAnim(modelname,name,suffixname,dirname,"",ANIM_NORMAL, is_ped);
		else if (version>=4)
		{
			if (active==1)
			{
				if (version<7)
					AddAnim(modelname,name,suffixname,dirname,skelname,ANIM_NORMAL, is_ped);
				else
					AddAnim(modelname,name,suffixname,dirname,skelname, anim_type, is_ped);
			}
		}
	}

	fclose(fp);
}

void WriteFunctions( Link<AnimRecord>* curNode, FILE* fp, int i, int startIndex, int numToWrite )
{
	bool bSetRef=false;
	char strSkelName[256];

	Link<FileRecord>* skelrec=curNode->data.list.GetHead();

	if (skelrec && strlen(skelrec->data.SkelName)>0)
	{
		bSetRef=true;

		if (skelrec)
		{
			if (skelrec->data.SkelName[0]==0)
			{
				// default to human
				fprintf(fp,"\tSetReferenceChecksum animload_human\n");
				strcpy(strSkelName,"human");
			}
			else
			{
				strcpy(strSkelName,skelrec->data.SkelName);
				fprintf(fp,"\tSetReferenceChecksum animload_%s\n",strSkelName);
			}
		}
	}
	else
	{
		printf("ERROR:  No skeleton?!?  Tell Gary!");
		exit(1);
	}

	Link<FileRecord>* subNode=curNode->data.list.GetHead();

	bool firstNode = true;
	int subNodeIndex = 0;
		
	while(subNode)
	{
		// Dump out the first set of subnodes that

		if ( 
				i == subNode->data.AnimType 
				|| ( i == 2 && !subNode->data.is_ped )
		)
		{
			if ( ( subNodeIndex >= startIndex ) && ( subNodeIndex < ( startIndex + numToWrite ) ) )
			{

				// Ken: Modified to run <LoadFunction> rather than LoadAnim so that
				// the same script can be used to unload anims, required for unloading 
				// the skater anims to free up memory when building a park.

				// GJ:  Also modified it to insert "<...>" parameter, so that the
				// "use_pip" flag can be passed in from a higher level.
				fprintf(fp,"\t%s<LoadFunction> <...> name=\"%s%s\\%s.ska\" descChecksum=%s\n",
						firstNode?"if ":"\t",
						QNANIM_PATH,subNode->data.DirName,subNode->data.AnimFile,
						subNode->data.AnimFile);
				firstNode = false;
			}

			if (strlen(subNode->data.SkelName)>0 &&
				strcmp(subNode->data.SkelName,strSkelName)!=0)
			{
				printf("WARNING: Inconsistent skeleton name for %s (%s).\n",subNode->data.AnimFile,subNode->data.SkelName);
			}

			subNodeIndex++;
		}

		subNode=subNode->next;
	}
			
	if (!firstNode)
	{
		fprintf(fp,"\tendif\n");
	}

	if (bSetRef)
	{
		fprintf(fp,"\tSetReferenceChecksum 0\n");
	}
}

void WriteHeader(FILE* fp)
{
	time_t ltime;
	time(&ltime);

	fprintf(fp,"// animqngen: auto-generated .qn load script\n");
	fprintf(fp,"// Created: %s\n",ctime(&ltime));
}

void DumpDB(FILE* fp)
{
	Link<AnimRecord>* curNode=DB.GetHead();

	// first pass:  only "thps5_human" anims are allowed
	// to be unloadable, so unset the unloadable flag
	// on any invalid anims
	while(curNode)
	{
		char lcase[256];
		strcpy(lcase, curNode->data.AnimName);
		_strlwr(lcase);

		if (!strstr(lcase, "thps5_human"))
		{
			if (curNode->data.HasUnloadableEntries())
			{
				printf("WARNING! '%s' has unloadable entries.  Currently, only 'thps5_human' should have unloadable entries! Ignoring!\n", (char*)curNode->data.AnimName);
			}

			Link<FileRecord>* link = curNode->data.list.GetHead();
			while(link)
			{
				link->data.AnimType = ANIM_NORMAL;
				link = link->next;
			}
		}

		curNode=curNode->next;
	}

	curNode=DB.GetHead();
	while(curNode)
	{
		for(int i = 0; i < 3; i++)
		{
			// whether to split the script into smaller
			// ones (to prevent the fragmented script heap 
			// from running out of memory when loading
			// large scripts)
			bool bSubdivideScript=false;

			int count = 0;

			Link<FileRecord>* link = curNode->data.list.GetHead();
			while(link)
			{
				if ( (i == link->data.AnimType) || ( i == 2 && !link->data.is_ped )  )
				{
					count++;
				}
				link = link->next;
			}

			if ( count > vMAX_ANIMS_PER_FUNCTION )
			{
				printf( "*** %s has %d anims after pass %d\n", curNode->data.AnimName, count, i );
				bSubdivideScript = true;
			}

			if (i == ANIM_NORMAL)
			{
				fprintf(fp,"script animload_%s LoadFunction=LoadAnim\n",curNode->data.AnimName);
			}
			else if (i == ANIM_UNLOADABLE)
			{
				char lcase[256];
				strcpy(lcase, curNode->data.AnimName);
				_strlwr(lcase);

				// Function is only generated if it has unloadable entries,
				// or if it's the thps5_human one
				if (!strstr(lcase, "thps5_human") && !curNode->data.HasUnloadableEntries())
					continue;

				// Warn the user if a non thps5_human has unloadable entries
				// TODO:  This should really be an assert...
				if (curNode->data.HasUnloadableEntries())
				{
					if (!strstr(lcase, "thps5_human"))
					{
						printf("WARNING! '%s' has unloadable entries.  Currently, only 'thps5_human' should have unloadable entries! Ignoring!\n", (char*)curNode->data.AnimName);

						fprintf(fp,"script animload_%s_unloadable LoadFunction=LoadAnim\n",curNode->data.AnimName);
						fprintf(fp,"endscript\n");
						continue;
					}
				}

				fprintf(fp,"script animload_%s_unloadable LoadFunction=LoadAnim\n",curNode->data.AnimName);
			}
			else 
			{

				char lcase[256];
				strcpy(lcase, curNode->data.AnimName);
				_strlwr(lcase);
				
				//printf ("*** %d: %s\n",curNode->data.is_ped, lcase);

				// skip if not the human anim, or if it is the ped										
				if (!strstr(lcase, "thps5_human"))// || curNode->is_ped)
					continue;
				
				fprintf(fp,"script animload_%s_net LoadFunction=LoadAnim\n",curNode->data.AnimName);
			}

			if ( bSubdivideScript )
			{
				for ( int j = 0; j < ((count-1)/vMAX_ANIMS_PER_FUNCTION)+1; j++ )
				{
					if (i == 0)
						fprintf(fp,"\tanimload_%s%s_%d <...>\n", curNode->data.AnimName, "", j );	
					else if ( i == 1)
						fprintf(fp,"\tanimload_%s%s_%d <...>\n", curNode->data.AnimName, "_unloadable", j );	
					else
						fprintf(fp,"\tanimload_%s%s_%d <...>\n", curNode->data.AnimName, "_net", j );	
				}
			}
			else
			{
				WriteFunctions( curNode, fp, i, 0, vMAX_ANIMS_PER_FUNCTION );
			}
			// end subdivide if

			fprintf(fp,"endscript\n\n");

			if ( bSubdivideScript )
			{
				for ( int j = 0; j < ((count-1)/vMAX_ANIMS_PER_FUNCTION)+1; j++ )
				{
					if (i == 0)
						fprintf(fp,"script animload_%s%s_%d LoadFunction=LoadAnim\n", curNode->data.AnimName, "", j );	
					else if ( i == 1)
						fprintf(fp,"script animload_%s%s_%d LoadFunction=LoadAnim\n", curNode->data.AnimName, "_unloadable", j );	
					else
						fprintf(fp,"script animload_%s%s_%d LoadFunction=LoadAnim\n", curNode->data.AnimName, "_net", j );	
					WriteFunctions( curNode, fp, i, j * vMAX_ANIMS_PER_FUNCTION, vMAX_ANIMS_PER_FUNCTION );
					fprintf(fp,"endscript\n\n");
				}
			}

		}	// end for
		

		curNode=curNode->next;
	}
}

int main(int argc, char* argv[])
{
	char path[256];

	if (argc<2)
	{
		printf("ANIQNGEN.EXE - Animation range to .qn file generation tool\n");
		printf("NeverSoft Entertainment, 2001\n\n");

		printf("Usage: aniqngen [filename] {path} || {anrfile}\n\n");

		printf("filename- name of the .qn file to generate\n");
		printf("path    - the path to scan recursively for .anr files\n");
		printf("anrfile - a specific anr file to convert to a .qn file\n");
		return 0;
	}
	
	if (argc==1)
	{
		printf("You must specify an output filename.\n");
		return 2;
	}

	if (argc>3)
	{
		printf("Unknown argument: %s\n",argv[3]);
		return 3;
	}

	FILE* fp=fopen(argv[1],"w");
	WriteHeader(fp);

	if (argc!=2)
	{
		if (strstr(argv[2],"."))
		{
			printf("Exporting %s for file %s...",argv[1],argv[2]);
			
			char path[256];
			char file[256];

			GetPath(path,255,argv[2]);
			_chdir(path);

			GetFile(file,255,argv[2]);

			ProcAnr(file,NULL);
			DumpDB(fp);
			fclose(fp);
			
			printf("\nDone.\n");
			return 1;
		}

		_chdir(argv[2]);
	}
	
	_getcwd(path,255);

	// Do the export
 	printf("Exporting %s...",argv[1]);	

	Find(path,".anr",ProcAnr,NULL);

	DumpDB(fp);
	fclose(fp);

	printf("\nDone.\n");

	return 1;
}
