/*
	TrackUI.h
	Track Visualization Graph Control

	This is used in the cutscene exporter to control when cameras
	are switched

	aml - 1-30-03
*/

#ifndef __TRACKUI__
#define __TRACKUI__

#include "max.h"
#include "notify.h"
#include "UIControl.h"
#include "../misc/llist.h"
#include "../misc/saveable.h"

#define CLASSNAME_TRACKUI "TrackUI"
#define UNSEL_BARCOLOR  RGB(166,233,245)
#define SEL_BARCOLOR    RGB(139,235,182)
#define SEL_SWAP        RGB(255,255,255)
#define TRACKUI_UPDATE  WM_USER + 0x0008

struct TrackUIRange
{
	int			  end;		// end time to display for the bar in the graph
	CStr		  name;		// Name of the object to display in the graph
	COLORREF	  color;	// The color used to render this range into the graph
	DWORD		  userData;	// User data attached to this particular item

	TrackUIRange()
	{
		end      = 0;
		color    = UNSEL_BARCOLOR;
		userData = 0;
	}

	TrackUIRange(TrackUIRange& right)
	{
		end      = right.end;
		name     = right.name;
		color    = right.color;
		userData = right.userData;
	}

	TrackUIRange& operator= (TrackUIRange& right)
	{
		end      = right.end;
		name     = right.name;
		color    = right.color;
		userData = right.userData;

		return *this;
	}

	int operator==(TrackUIRange& right)
	{
		return (end == right.end);
	}

	int operator<(TrackUIRange& right)
	{
		return (end < right.end);
	}

	int operator>(TrackUIRange& right)
	{
		return (end > right.end);
	}
};

struct TrackUIKey: public Saveable
{
	int      time;				// Time at which this key occurs in the sequence
	DWORD    userData;			// User data associated with this particular key
	CStr     name;
	CStr     userBuffer;		// String buffer not manipulated by TrackUI (for simplified management of user data)
	void*    pInternalUserData;	// User data maintained within TrackUIKey itself
	int      iUserDataSize;		// Size of internal user data
	COLORREF cDisplayColor;
	COLORREF cHighlightColor;
	
	TrackUIKey();
	TrackUIKey(const TrackUIKey& right);
	~TrackUIKey();

	void AllocUserData(int size);

	// For sorting purposes
	int operator== (const TrackUIKey& key) { return (time == key.time); }
	int operator>  (const TrackUIKey& key) { return (time > key.time);  }
	int operator<  (const TrackUIKey& key) { return (time > key.time);  }
	int operator>= (const TrackUIKey& key) { return (time >= key.time); }
	int operator<= (const TrackUIKey& key) { return (time <= key.time); }

	TrackUIKey& operator= (const TrackUIKey& right);

	int  GetSize();
	void Store(void* data);
	void Retrieve(void* data);

	CStr GetClass();
	CStr GetType();
	CStr GetProp(CStr name);
};

class TrackUI: public UIControl, public TimeChangeCallback
{
	HFONT                  hFont;							// Font used for names of Track ranges
	HMENU                  hMenu;							// Base menu when right clicking on a track key
	HMENU                  hPopupMenu;						// Popup menu displayed when right clicking oin a track key

	HWND                   hwndScroll;						// Scrollbar for scrolling left/right accross the timeline
	HWND                   hwndZoom;						// Scrollbar for zooming in/out on the timeline
	HWND                   hwndShowHide;					// Button to show/hide the UI buttons
	HWND                   hwndReset;						// Button to reset the zoom and scroll settings

	POINT                  mousePos;						
	int                    minFrame, maxFrame;				// Minimum and maximum times in frames
	RECT                   clientRect;						// Rectangle specifying the client area of the control
	int                    curFrame;						
	Link<TrackUIRange>*    selTrack;						// The currently selected time track
	Link<TrackUIRange>*    swapTrack;						// Track that is being swapped w/ selTrack in TAS_SWAPTRACK state
	int                    origStartTime;					// The original start time of the last poll of the timebar
	int                    origEndTime;						// The original end time of the last poll of the timebar
															// (The above two are used to determine when a full control render refresh are necessary)
															
	bool                   bMayAddKeys;						// True if the user is allowed to add new track keys
	bool                   bMayRemoveKeys;					// True if the user is allowed to remove existing track keys
	bool                   bUIVisible;						// True if the lower-right UI controls are visible
	
	TrackUIKey             defaultKey;						// This is the key that gets created when the user selects to add a
															// new key from the popup menu
															
	int                    iKeySelected;					// Indicates the current track key selected (-1 if none)
															// this gets updated by DrawGraph
															
	int                    graphStartTime;					// This is the start time in which bars are drawn on the graph

	float                  fZoomFactor;						// Multiple that the timeline scale will be zoomed by
	int                    scrollOffset;

	LinkList<TrackUIKey>   trackKeys;						// List of all the user added track keys
														
	Link<TrackUIKey>*      linkUpdateKey;					// Points to the key that is being updated in the TAS_KEYMOVE state
															// or during a key lock operation

	DWORD                  saveID;							// The chunkID to be used for storing the TrackUIKey data
	Animatable*            saveAnimatable;					// The node that key data will be saved into

	void(*fpTimeChangeCB)(TrackUI*, void*);					// Callback when current trackbar time has changed
	void* pTimeChangeData;									// Callback retained data

	void(*fpKeyAdded)(TrackUI*,Link<TrackUIKey>*,void*);	// Callback executed whenever a new key is added (should set userdata here)
	void* pKeyAddedData;									// Callback retained data

	void(*fpKeyRemoved)(TrackUI*,Link<TrackUIKey>*,void*);	// Callback executed immediately before a key is deleted (should clean up any userdata here)
	void* pKeyRemovedData;									// Callback retained data

	void(*fpPreKeyMoved)(TrackUI*,Link<TrackUIKey>*,void*);	// Callback executed immediately before a key is moved
	void* pPreKeyMovedData;									// Callback retained data

	void(*fpPostKeyMoved)(TrackUI*,Link<TrackUIKey>*,void*);// Callback executed immediately after a key is moved
	void* pPostKeyMovedData;								// Callback retained data

	void(*fpKeyChanged)(TrackUI*,void*);					// Callback executed whenever any keybased action takes place
	void* pKeyChangedData;									// Callback retained data

	enum TimeAdjustState
	{
		TAS_FREE,			// Cursor is free to move around the TrackUI control (no actions taken yet)
		TAS_TIMEMOVE,		// Cursor movement is adjusting track time data
		TAS_SWAPTRACK,		// Next click swaps selected track
		TAS_KEYMOVE,		// Cursor movement adjusts currently selected key time
		TAS_LOCK,			// No updates should occur in the lock state
	};

	TimeAdjustState        state;

	void (*fpDblClick)(TrackUI*,Link<TrackUIRange>*,void*);	// Callback that gets called when the user double clicks an item in the graph
	void *pDblClickData;									// Userdata attached to the click callback

	void(*fpAction)(Link<TrackUIKey>*,void*);				// Callback that gets called when the user double clicks a specific key
	void *pActionData;										// Callback retained data

	void Init();
	void DrawGraph(HDC hdc);
	void TimeChanged(TimeValue t);

	LRESULT WndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);
	void BuildControlUI();

	Link<TrackUIRange>* GetPointTrack();			// Retrieves the track that the mouse is currently overtop of

	static void TimeUnitsChanged(void *param, NotifyInfo* info);
	static void UnitsChanged(void *param, NotifyInfo* info);

	void CycleTracks();

public:
	LinkList<TrackUIRange> barlist;
	
	CLASSNAME(CLASSNAME_TRACKUI)

	TrackUI(HINSTANCE hInstance);
	TrackUI(HINSTANCE hInstance, HWND hwnd);
	TrackUI(TrackUI& cpopup);
	~TrackUI();

	TrackUI& operator = (TrackUI& right);

	void                RegisterDblClickCB(void(*fp)(TrackUI*,Link<TrackUIRange>*,void*), void* pData);
	Link<TrackUIRange>* AddItem(int end, CStr name, DWORD userData = 0, Link<TrackUIRange>* link = NULL);
	int                 TrackLength();
	void                SetStartTime(int time);
	inline int          GetStartTime() { return graphStartTime; }

	void                Clear();
	void                ClearBar();
	void                ClearKeys();
	void                PostUpdate();
	void				RegisterTimeChangeCB(void(*fp)(TrackUI*, void*), void* pData);
	Link<TrackUIRange>* FindTrack(TimeValue time);

	Link<TrackUIKey>*	AddKey(TrackUIKey* trackKey);
	void				RemoveKey(Link<TrackUIKey>* linkTrackKey);

	inline void         SetDefaultAddKey(TrackUIKey* trackKey) { defaultKey = *trackKey; }
	inline void         MayAddTrackKeys(bool bVal) { bMayAddKeys = bVal; }
	inline void         MayRemoveTrackKeys(bool bVal) { bMayRemoveKeys = bVal; }
	bool				KeySelected();					// True if a key is currently selected in the UI
	void                Refresh();

	// Locking a key highlights it and prevents any key manipulation actions (unless explicitly unlocked)
	void                LockAffectKey(int time);
	void				LockAffectKey(Link<TrackUIKey>* link);
	void                UnlockAffectKey();

	// Keyframe callback setting methods
	void				SetKeyAddedCB(void(*pFunc)(TrackUI*,Link<TrackUIKey>*,void*), void* pData);
	void				SetRemoveKeyCB(void(*pFunc)(TrackUI*,Link<TrackUIKey>*,void*), void* pData);
	void				SetPreKeyMovedCB(void(*pFunc)(TrackUI*,Link<TrackUIKey>*,void*), void* pData);
	void				SetPostKeyMovedCB(void(*pFunc)(TrackUI*,Link<TrackUIKey>*,void*), void* pData);
	
	// Occurs for any and all key based TrackUI changes (add/del/move, etc.)  client should poll data and update
	void                SetKeyChangedCB(void(*pFunc)(TrackUI*,void*), void* pData);

	inline LinkList<TrackUIKey>* GetKeyList() { return &trackKeys; }
	inline void CreateKeyList(LinkList<TrackUIKey>* list) { trackKeys = *list; Refresh(); }

	// Save/Load of key data
	inline void SetKeySaveChunk(DWORD id) { saveID = id; }

	int  CalcKeySize();				// Returns the total amount of memory in bytes used by TrackUI keys
	void SaveKeyData(Animatable* animbase);
	void LoadKeyData(Animatable* animbase);
	void LoadKeyData(Animatable* animbase, DWORD id);
	void SaveKeyData();
	void LoadKeyData();

	// Intended to be called by a dialog created when called from the KeyAction
	// This makes trackUI signal that the key has been changed
	void CallKeyUpdatedCB();

	// TrackUI UI
	void ShowHideUI();
	void KeyLeft();
	void KeyRight();
	void ZoomIn();
	void ZoomOut();
	void ResetZoomScroll();

	inline void SetKeyActionCB(void(*pFunc)(Link<TrackUIKey>*,void*), void* pData) { fpAction = pFunc; pActionData = pData; }
	inline void SetTrackSaveAnimatable(Animatable* animatable) { saveAnimatable = animatable; }
};

#endif
