/*
	ScriptIniParser.cpp
	Parses Script INI file
	1-28-01
*/

#include "ScriptIniParser.h"
#include "../path.h"
#include "ParseFuncs.h"
#include "nodecrc.h"
#include "../Trigger/Trigger.h"

ScriptIniParser::ScriptIniParser() : propDB(4)
{

}

ScriptIniParser::~ScriptIniParser()
{
	ConfigRec* curProp;

	int size=propDB.GetSize();
	propDB.IterateStart();

	for(int i=0;i<size;i++)
	{
		curProp=propDB.IterateNext();
		delete curProp;
	}
}

void ScriptIniParser::AddGlobalDefaults()
{
	Link<ConfigClass>* curclass = configDB.GetHead();

	CStr strGlobalDefaults = GetClassTypeRecord("Default","");

	while(curclass)
	{
		if (curclass->data.name!=CStr("Default"))
		{
			//curclass->data.strCmds=strGlobalDefaults+curclass->data.strCmds;
			AddGlobalDefaults(curclass->data.strCmds);
		}

		curclass=curclass->next;
	}
}

void ScriptIniParser::AddGlobalDefaults(CStr& propBuffer)
{
	LinkList<ConfigProp> cprops;
	DWORD                cflags;
	CStr				 unkBuf;
	CStr                 cluster;
	
	ParseConfigProps(&cprops,&cflags,propBuffer,&unkBuf,&cluster);
	AddGlobalDefaults(propBuffer,&cprops,&cflags,&unkBuf);
}

void ScriptIniParser::AddGlobalDefaults(CStr&                 propBuffer,
										LinkList<ConfigProp>* cprops,
										DWORD*                cflags,
										CStr*                 unkBuf,
										CStr&                 className,
										CStr&                 typeName)
{
	//ScriptIniParser::AddGlobalDefaults

	CStr tmpBuf;

	char eol[3];
	eol[0]=10;
	eol[1]=13;
	eol[2]=0;

	// Cluster and flag settings are currently not retained for default

	// Check each default property to ensure it exists, if it doesn't add it
	Link<ConfigProp>* curDEFprop=DEFcprops.GetHead();
	Link<ConfigProp>* linkLast=NULL;

	while(curDEFprop)
	{
		// Check if this property exists within the given buffer
		Link<ConfigProp>* curprop=cprops->GetHead();
		bool bExists=false;

		while(curprop)
		{
			if (curprop->data.name==curDEFprop->data.name)
			{
				bExists=true;
				break;
			}

			curprop=curprop->next;
		}

		if (!bExists)
		{
			// Don't add default if instructed to omit it
			CStr rec = GetClassTypeRecord(className,typeName);

			if (!IsInstr(rec,CStr("Omit ")+curDEFprop->data.name) &&
				!IsInstr(propBuffer,CStr("Omit ")+curDEFprop->data.name))
			{
				ConfigProp newProp = curDEFprop->data;

				// Add the property
				tmpBuf += curDEFprop->data.name + " = " + curDEFprop->data.value + CStr(eol);
				tmpBuf += curDEFprop->data.extdata + CStr(eol);

				if (!linkLast)
					linkLast = cprops->AddToHead(&newProp);
				else
					linkLast = cprops->AddAfterLink(linkLast, &newProp);
			}
		}
		else
		{
			if (curprop->data.extdata.Length()==0 &&
				curDEFprop->data.extdata.Length()>0)
			{
				tmpBuf += curDEFprop->data.extdata + CStr(eol);
			}
		}

		curDEFprop=curDEFprop->next;
	}

	// Combine Flags
	if (cflags)
		(*cflags) |= DEFcflags;

	// Global defaults should appear before user properties
	propBuffer = tmpBuf+propBuffer;
}

bool ScriptIniParser::ParseScriptIni(char* filename)
{
	// Whenever scripts.ini is reparsed, the current timestamp should
	// be updated for incremental script export (see NodeCRC)
	InitNodeCRCFileTimes();

	// Reset group paths
	strCurClassGroup = "";
	strCurTypeGroup = "";

	// Construct ini file path
	CStr strFile;
	
	if (!filename)
	{
		strFile=getenv(APP_ENV);

		// Verify that the environment variable is set
		if (strFile==CStr(""))
		{
			char ErrorBuf[256];
			sprintf(ErrorBuf,"Couldn't load '%s' the '%s' environment variable is not set.",SCRIPT_INI,APP_ENV);

			MessageBox(NULL,ErrorBuf,"ParseScriptIni",MB_ICONSTOP|MB_OK);
			return false;
		}

		strFile+=CStr(SCRIPT_PATH)+SCRIPT_INI;
	}
	else
		strFile = filename;

	FILE* fp=fopen(strFile,"r");

	if (!fp)
		return false;

	Reset();

	char lineBuf[512];
	int  pos=1000;
	CStr word=GetWordToken(fp,lineBuf,&pos);
	CStr word2;
	CStr className;

	while(word!=CStr(""))
	{
		// Parse Group
		if (word==CStr("Group"))
			ParseClassGroup(fp,lineBuf,pos);
		
		// Parse Class
		if (word==CStr("Class"))
			ParseClass(fp,lineBuf,pos);

		word=GetWordToken(fp,lineBuf,&pos);
	}

	fclose(fp);
	
	// Cache Default class information for faster access
	//DEFBuffer = GetClassTypeRecord("Default","");
	//ParseConfigProps(&DEFcprops,&DEFcflags,DEFBuffer,&DEFunkBuf,&DEFcluster);

	AddGlobalDefaults();

	return true;
}

void ScriptIniParser::ParseClassGroup(FILE* fp,char* lineBuf,int pos)
{
	CStr groupName = GetWordToken(fp,lineBuf,&pos);
	CStr word;

	CStr oldGroup = strCurClassGroup;

	if (strCurClassGroup.Length()==0)
		strCurClassGroup = groupName;
	else
		strCurClassGroup += CStr("/") + groupName;

	// Advance to start block
	if (!FileAdvance(fp,lineBuf,&pos,'{'))
		return;

	word = GetWordToken(fp,lineBuf,&pos);

	while(word!=CStr(""))
	{
		if (word==CStr("Group"))
		{
			ParseClassGroup(fp,lineBuf,pos);
			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		if (word==CStr("Class"))
		{
			ParseClass(fp,lineBuf,pos);
			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		if (word==CStr(" "))
		{
			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		if (word==CStr("}"))
		{
			strCurClassGroup = oldGroup;
			return;
		}

		word=GetWordToken(fp,lineBuf,&pos);
	}
}

void ScriptIniParser::ParseTypeGroup(ConfigClass* pcclass,FILE* fp,char* lineBuf,int pos)
{
	CStr groupName = GetWordToken(fp,lineBuf,&pos);
	CStr word;

	CStr oldGroup = strCurTypeGroup;

	if (strCurTypeGroup.Length()==0)
		strCurTypeGroup = groupName;
	else
		strCurTypeGroup += CStr("/") + groupName;

	// Advance to start block
	if (!FileAdvance(fp,lineBuf,&pos,'{'))
		return;

	word = GetWordToken(fp,lineBuf,&pos);

	while(word!=CStr(""))
	{
		if (word==CStr("Group"))
		{
			ParseTypeGroup(pcclass,fp,lineBuf,pos);
			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		if (word==CStr("Type"))
		{
			ParseType(pcclass,fp,lineBuf,pos);
			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		if (word==CStr(" "))
		{
			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		if (word==CStr("}"))
		{
			strCurTypeGroup = oldGroup;
			return;
		}

		word=GetWordToken(fp,lineBuf,&pos);
	}
}

void ScriptIniParser::ParseClass(FILE* fp,char* lineBuf,int pos)
{
	CStr className=GetWordToken(fp,lineBuf,&pos);
	CStr word,word2;

	Link<ConfigClass>* lcclass=configDB.Add();
	//lcclass->data.name=className;
	if (strCurClassGroup.Length()>0)
		lcclass->data.name = strCurClassGroup + CStr("/") + className;
	else
		lcclass->data.name = className;

	if (!FileAdvance(fp,lineBuf,&pos,'{'))
		return;										//	Advance to start block

	word=GetWordToken(fp,lineBuf,&pos);				//  Get the next line delim
	
	while(word!=CStr(""))
	{
		// Check if a grouping directive was encountered
		if (word==CStr("Group"))
		{
			ParseTypeGroup(&lcclass->data,fp,lineBuf,pos);
			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		// Check if type
		if (word==CStr("Type"))
		{
			ParseType(&lcclass->data,fp,lineBuf,pos);
			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		if (word==CStr(" "))
		{
			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		// Check if end of block
		if (word==CStr("}"))
		{
			// Add empty char if there is no record to distinguish record existance
			if (lcclass->data.strCmds.Length()==0 && lcclass->data.types.GetSize()==0)
				lcclass->data.strCmds=" ";

			// Add hashed DB record for class with pre parsed info (for speed)
			ConfigRec* newConfigRec = new ConfigRec;
			ParseConfigProps(&newConfigRec->props,
							 &newConfigRec->flags,
							 lcclass->data.strCmds,
							 &newConfigRec->strCluster,
							 &newConfigRec->strProgram);

			if (lcclass->data.name == CStr("Default"))
			{
				DEFBuffer = lcclass->data.strCmds;
				ParseConfigProps(&DEFcprops,&DEFcflags,DEFBuffer,&DEFunkBuf,&DEFcluster);
			}
			else
			{
				AddGlobalDefaults(lcclass->data.strCmds,
								  &newConfigRec->props,
								  &newConfigRec->flags,
								  NULL);
			}

			// Add Group string
			lcclass->data.strGroup = strCurClassGroup;

			char buf[256];
			sprintf(buf,"Added '%s' to hash table.\n",lcclass->data.name);
			OutputDebugString(buf);

			if (propDB.GetItem(lcclass->data.name))
				propDB.FlushItem(lcclass->data.name);

			propDB.PutItem(lcclass->data.name,newConfigRec);

			// Add another version using the original (non-groupped) name so
			// old defined objects will still have the appropriate default data
			ConfigRec* newConfigRec2 = new ConfigRec;
			*newConfigRec2 = *newConfigRec;
			
			if (propDB.GetItem(lcclass->data.name))
				propDB.FlushItem(lcclass->data.name);

			propDB.PutItem(className,newConfigRec2);
			return;
		}

		// Add data to buffer
		lcclass->data.strCmds+=StripLeft(lineBuf);
		pos=strlen(lineBuf);

		word=GetWordToken(fp,lineBuf,&pos);
	}

	// Add empty char if there is no record to distinguish record existance
	if (lcclass->data.strCmds.Length()==0 && lcclass->data.types.GetSize()==0)
		lcclass->data.strCmds=" ";

	// Add hashed DB record for class with pre parsed info (for speed)
	ConfigRec* newConfigRec = new ConfigRec;
	ParseConfigProps(&newConfigRec->props,
					 &newConfigRec->flags,
					 lcclass->data.strCmds,
					 &newConfigRec->strCluster,
					 &newConfigRec->strProgram);

	AddGlobalDefaults(lcclass->data.strCmds,
				      &newConfigRec->props,
					  &newConfigRec->flags,
					  NULL);

	char buf[256];
	sprintf(buf,"Added '%s' to hash table.\n",lcclass->data.name);
	OutputDebugString(buf);

	if (propDB.GetItem(lcclass->data.name))
		propDB.FlushItem(lcclass->data.name);

	propDB.PutItem(lcclass->data.name,newConfigRec);

	// Add another version using the original (non-groupped) name so
	// old defined objects will still have the appropriate default data
	ConfigRec* newConfigRec2 = new ConfigRec;
	*newConfigRec2 = *newConfigRec;

	if (propDB.GetItem(className))
		propDB.FlushItem(className);

	propDB.PutItem(className,newConfigRec2);
}

void ScriptIniParser::ParseType(ConfigClass* pcclass,FILE* fp,char* lineBuf,int pos)
{
	CStr typeName=GetWordToken(fp,lineBuf,&pos);
	CStr word,word2;

	Link<ConfigType>* lctype=pcclass->types.Add();
	//lctype->data.name=typeName;
	if (strCurTypeGroup.Length()>0)
		lctype->data.name = strCurTypeGroup + CStr("/") + typeName;
	else
		lctype->data.name = typeName;

	if (!FileAdvance(fp,lineBuf,&pos,'{'))
		return;								//	Advance to start block

	word=GetWordToken(fp,lineBuf,&pos);		//  Get the next line delim
	
	while(word!=CStr(""))
	{
		if (word==CStr(" "))
		{
			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		// Check for close block
		if (word==CStr("}"))
		{
			ConfigRec* newConfigRec = new ConfigRec;
			ParseConfigProps(&newConfigRec->props,
							 &newConfigRec->flags,
							 pcclass->strCmds+lctype->data.strCmds,
							 &newConfigRec->strCluster,
							 &newConfigRec->strProgram);

			AddGlobalDefaults(pcclass->strCmds+lctype->data.strCmds,
				              &newConfigRec->props,
							  &newConfigRec->flags,
							  NULL);

			lctype->data.strGroup = strCurTypeGroup;

			char buf[256];
			sprintf(buf,"Added '%s' to hash table.\n",pcclass->name+lctype->data.name);
			OutputDebugString(buf);

			if (propDB.GetItem(pcclass->name+lctype->data.name))
				propDB.FlushItem(pcclass->name+lctype->data.name);

			propDB.PutItem(pcclass->name+lctype->data.name,newConfigRec);

			// Add another version using the original (non-groupped) name so
			// old defined objects will still have the appropriate default data
			ConfigRec* newConfigRec2 = new ConfigRec;
			*newConfigRec2 = *newConfigRec;
			
			if (propDB.GetItem(pcclass->name+typeName))
				propDB.FlushItem(pcclass->name+typeName);

			propDB.PutItem(pcclass->name+typeName,newConfigRec2);
			return;
		}

		// Add data to buffer
		lctype->data.strCmds+=StripLeft(lineBuf);
		pos=strlen(lineBuf);

		word=GetWordToken(fp,lineBuf,&pos);
	}

	ConfigRec* newConfigRec = new ConfigRec;
	ParseConfigProps(&newConfigRec->props,
		             &newConfigRec->flags,
					 lctype->data.strCmds,
					 &newConfigRec->strCluster,
					 &newConfigRec->strProgram);

	AddGlobalDefaults(lctype->data.strCmds,
				      &newConfigRec->props,
					  &newConfigRec->flags,
					  NULL);

	char buf[256];
	sprintf(buf,"Added '%s' to hash table.\n",pcclass->name+lctype->data.name);
	OutputDebugString(buf);

	if (propDB.GetItem(pcclass->name+lctype->data.name))
		propDB.FlushItem(pcclass->name+lctype->data.name);

	propDB.PutItem(pcclass->name+lctype->data.name,newConfigRec);

	// Add another version using the original (non-groupped) name so
	// old defined objects will still have the appropriate default data
	ConfigRec* newConfigRec2 = new ConfigRec;
	*newConfigRec2 = *newConfigRec;
	
	if (propDB.GetItem(pcclass->name+typeName))
		propDB.FlushItem(pcclass->name+typeName);

	propDB.PutItem(pcclass->name+typeName,newConfigRec2);
}

CStr ScriptIniParser::GetDynUICmds(CStr className,CStr typeName,CStr propName)
{
	CStr PropBuffer=GetClassTypeRecord(className,typeName);

	// Build a new buffer consisting of all the Script.ini DB info partaining to
	// the given property name

	int pos=0;
	int buflen=PropBuffer.Length();

	CStr strAutoDuck;
	CStr strTemp;
	CStr eol="\n";

	while (pos<buflen)
	{
		// Get a line of text from the buffer
		strTemp=GetRemainLinePartial(PropBuffer,&pos);
	
		//if (strTemp==CStr(""))
		if (strTemp.Length() == 0)
			continue;

		// Skip if we reach a line that doesn't reference this property
		//if (!(Instr(strTemp,"// @")!=-1 && Instr(strTemp,propName)!=-1))
		if (!(strstr(strTemp,"// @") && strstr(strTemp,propName)))
			continue;

		strAutoDuck+=strTemp+eol;
	}

	return strAutoDuck;
}

// This will update the config prop extended data buffer (used for the Dynamic UI system) against the
// Dynamic UI commands stored in the script.ini file, if no special extended data exists it will be updated
// with the script.ini DynUI commands
void ScriptIniParser::GetDynUIProps(LinkList<ConfigProp>* cprops,CStr className,CStr typeName)
{
	Link<ConfigProp>* curNode=cprops->GetHead();

	while(curNode)
	{
		CStr DynCmdBuf=GetDynUICmds(className,typeName,curNode->data.name);
		
		if (curNode->data.extdata.Length()==0)
			curNode->data.extdata=DynCmdBuf;

		curNode=curNode->next;
	}
}

void ScriptIniParser::CombineDynUIProps(LinkList<ConfigProp>* cprops,CStr className,CStr typeName)
{
	Link<ConfigProp>* curNode=cprops->GetHead();

	while(curNode)
	{
		CStr DynCmdBuf=GetDynUICmds(className,typeName,curNode->data.name);

		if (curNode->data.extdata.Length()==0)
			curNode->data.extdata += DynCmdBuf;

		curNode=curNode->next;
	}
}

bool ScriptIniParser::GetDefaultProps(LinkList<ConfigProp>* cprops,CStr className,CStr typeName)
{
	// Abort if inconsistent
	if (className==CStr(vUNKNOWN) ||
		className==CStr("")       ||
		typeName==CStr(vUNKNOWN))
		return false;

	// Find the default record for this class/type
	CStr propBuffer=GetClassTypeRecord(className,typeName);
	
	// Abort if not found
	if (propBuffer==CStr(""))
		return false;

	// Build property list
	CStr scriptBuffer;
	DWORD cflags;

	scriptBuffer=ParseConfigProps(cprops,&cflags,propBuffer);
	return true;
}

// Build a list of omission properties for the given class and type
bool ScriptIniParser::GetOmissions(CStr origClass, CStr origType, LinkList<CStr>* omitList)
{
	CStr propBuffer = GetClassTypeRecord(origClass, origType);

	int  pos        = 0;
	int  len        = propBuffer.Length();
	bool bOmissions = false;
	CStr lineBuf;

	if (!omitList)
		return false;

	omitList->Clear();

	while(pos<len)
	{
		lineBuf=GetRemainLinePartial(propBuffer,&pos);

		char* strOmit = strstr(lineBuf, "Omit ");

		if (strOmit)
		{
			strOmit += 5;
			omitList->Add(&CStr(strOmit));
			bOmissions = true;
		}
	}

	return bOmissions;
}

bool ScriptIniParser::RemoveOmissions(CStr origClass, CStr origType, LinkList<ConfigProp>* props)
{
	CStr propBuffer = GetClassTypeRecord(origClass, origType);

	int  pos        = 0;
	int  len        = propBuffer.Length();
	bool bOmissions = false;
	CStr lineBuf;

	if (!props)
		return false;

	Link<ConfigProp>* curprop;
	ConfigProp cprop;

	while(pos<len)
	{
		lineBuf=GetRemainLinePartial(propBuffer,&pos);

		char* strOmit = strstr(lineBuf, "Omit ");

		if (strOmit)
		{
			strOmit += 5;
			cprop.name = strOmit;
		
			curprop = props->Find(&cprop);

			if (curprop)
			{
				props->Remove(curprop);
				bOmissions = true;
			}
		}
	}

	return bOmissions;	
}

DWORD ScriptIniParser::GetDefaultFlags(CStr className,CStr typeName)
{
	DWORD flags=0;

	/*
	if (GetDefault(className,typeName,"CreatedAtStart")==CStr("TRUE"))
		flags|=CCLASS_CREATEDATSTART;
	
	if (GetDefault(className,typeName,"AbsentInNetGames")==CStr("TRUE"))
		flags|=CCLASS_ABSENTINNETGAMES;

	if (GetDefault(className,typeName,"TrickObject")==CStr("TRUE"))
		flags|=CCLASS_TRICKOBJECT;

	if (GetDefault(className,typeName,"Occluder")==CStr("TRUE"))
		flags|=CCLASS_OCCLUDER;
	*/

	if (GetDefault(className,typeName,"TrickObject")==CStr("TRUE"))
		flags|=CCLASS_TRICKOBJECT;

	return flags;
}

CStr ScriptIniParser::GetDefault(CStr className,CStr typeName,CStr propName)
{
	//return CStr("");
	// Abort if inconsistent
	if (className==CStr(vUNKNOWN) ||
		className==CStr("")       ||
		typeName==CStr(vUNKNOWN))
		return CStr("");

	/* No longer need with hash table
	// Find the default record for this class/type
	CStr propBuffer=GetClassTypeRecord(className,typeName);
	
	// Abort if not found
	if (propBuffer==CStr(""))
		return CStr("");
	*/

	// Build property list
	//CStr scriptBuffer;
	CStr strClust;
	LinkList<ConfigProp> cprops;
	DWORD cflags;

	// Retrieve default from the hash table
	ConfigRec* crec = propDB.GetItem(className+typeName,FALSE);
	if (crec==NULL)
	{
		char buf[256];
		sprintf(buf,"Coludn't find '%s' in hash table.\n",(char*)(className+typeName));
		OutputDebugString(buf);

		return CStr("");
	}
	cflags = crec->flags;

	//scriptBuffer=ParseConfigProps(&cprops,&cflags,propBuffer,NULL,&strClust);

	// Process flag defaults
	/*
	if (propName==CStr("CreatedAtStart"))
		if (cflags & CCLASS_CREATEDATSTART)
			return CStr("TRUE");
		else
			return CStr("FALSE");

	if (propName==CStr("AbsentInNetGames"))
		if (cflags & CCLASS_ABSENTINNETGAMES)
			return CStr("TRUE");
		else
			return CStr("FALSE");

	if (propName==CStr("TrickObject"))
		if (cflags & CCLASS_TRICKOBJECT)
			return CStr("TRUE");
		else
			return CStr("FALSE");

	if (propName==CStr("Occluder"))
		if (cflags & CCLASS_OCCLUDER)
			return CStr("TRUE");
		else
			return CStr("FALSE");

	if (propName==CStr("Cluster"))
		return strClust;
	*/

	// Scan the property list for default properties
	Link<ConfigProp>* curNode=crec->props.GetHead();
	
	while(curNode)
	{
		if (curNode->data.name==propName)
		{
			CStr valLC = curNode->data.value;
			valLC.toLower();

			if (valLC == CStr("[none]"))
				return CStr("");

			return curNode->data.value;
		}

		curNode=curNode->next;
	}

	// Couldn't find property
	return CStr("");
}

void ScriptIniParser::Reset()
{
	DEFClass = "";
	DEFType  = "";
	lastDEFBuffer = "";

	configDB.Clear();
	propDB.FlushAllItems();
}

CStr ScriptIniParser::GetTypeRecord(ConfigClass* cclass, CStr typeName, bool bTypeOnly)
{
	// If there's no type, we've found it
	if (typeName==CStr(""))
		return cclass->strCmds;
	
	// We've found the class, now find the type
	Link<ConfigType>* curType=cclass->types.GetHead();

	while(curType)
	{
		if (curType->data.name == typeName)
		{
			// We've found the type, export it's property data with it's class data
			DEFClass      = cclass->name;
			DEFType       = typeName;

			if (bTypeOnly)
				lastDEFBuffer = curType->data.strCmds;
			else
				lastDEFBuffer = cclass->strCmds+curType->data.strCmds;

			return lastDEFBuffer;
		}

		curType=curType->next;
	}

	// We couldn't find the type.  Locate the first type that contains this name as a subitem
	if (!IsInstr(typeName,"/"))
	{
		curType=cclass->types.GetHead();
		
		while(curType)
		{
			// Strip any path information
			char* cpath = strrchr(curType->data.name,'/');

			if (cpath)
			{
				if (strcmp(cpath+1,typeName)==0)
				{
					// We've found the type, export it's property data with it's class data
					DEFClass      = cclass->name;
					DEFType       = typeName;

					if (bTypeOnly)
						lastDEFBuffer = curType->data.strCmds;
					else
						lastDEFBuffer = cclass->strCmds+curType->data.strCmds;

					return lastDEFBuffer;
				}
			}

			curType=curType->next;
		}
	}

	// Couldn't find type
	return CStr("");
}

CStr ScriptIniParser::GetClassRecord(CStr className)
{
	Link<ConfigClass>* curNode = configDB.GetHead();

	while(curNode)
	{
		if (curNode->data.name == className)
			return curNode->data.strCmds;

		curNode = curNode->next;
	}

	return CStr("");
}

CStr ScriptIniParser::GetClassTypeRecord(CStr className,CStr typeName)
{
	// Cached buffer temp removed
	//if (DEFClass == className && DEFType == typeName)
	//	return lastDEFBuffer;

	// Scan through configDB for the class
	Link<ConfigClass>* curNode=configDB.GetHead();

	while(curNode)
	{
		if (curNode->data.name==className)
			return GetTypeRecord(&curNode->data,typeName);

		curNode=curNode->next;
	}

	// Couldn't find class entry try subclass entries
	if (!IsInstr(className,"/"))
	{
		curNode=configDB.GetHead();

		while(curNode)
		{
			char* cpos = strrchr(curNode->data.name,'/');

			if (cpos)
			{
				if (strcmp(cpos+1,className)==0)
					return GetTypeRecord(&curNode->data,typeName);
			}

			curNode=curNode->next;
		}
	}

	// Couldn't find entry
	return CStr("");
}

ConfigClass* ScriptIniParser::GetConfigClass(CStr className)
{
	Link<ConfigClass>* curNode=configDB.GetHead();
	
	while(curNode)
	{
		if (curNode->data.name==className)
			return &curNode->data;

		curNode=curNode->next;
	}

	return NULL;
}

ConfigType* ScriptIniParser::GetConfigType(ConfigClass* pcclass, CStr typeName)
{
	Link<ConfigType>* curNode = pcclass->types.GetHead();

	while(curNode)
	{
		if (curNode->data.name == typeName)
			return &curNode->data;

		curNode = curNode->next;
	}

	return NULL;
}

void ScriptIniParser::GetPrograms(LinkList<ConfigProgram>* cprogs,CStr className,CStr typeName)
{
	// Find the default record for this class/type
	CStr strProgram;
	CStr propBuffer=GetClassTypeRecord(className,typeName);

	cprogs->Clear();
	ParseConfigProps(NULL,NULL,propBuffer,NULL,NULL,cprogs);
}

void ScriptIniParser::ConvertToDefaults(CStr className,CStr typeName,LinkList<ConfigProp>* config)
{
	Link<ConfigProp>* curNode=config->GetHead();

	while(curNode)
	{
		if (curNode->data.value==CStr(vRESTORE_DEFAULT_SEQ))
			curNode->data.value=GetDefault(className,typeName,curNode->data.name);

		curNode=curNode->next;
	}
}

void ScriptIniParser::ConvertToDefaultDelim(CStr className,CStr typeName,LinkList<ConfigProp>* config)
{
	Link<ConfigProp>* curNode=config->GetHead();

	while(curNode)
	{
		CStr strDefault=GetDefault(className,typeName,curNode->data.name);

		// If the property is set to it's default value mark it with the default delim
		if (strDefault==curNode->data.value)
			curNode->data.value=CStr(vRESTORE_DEFAULT_SEQ);

		curNode=curNode->next;
	}
}

bool ScriptIniParser::ClassExists(CStr className)
{
	Link<ConfigClass>* curNode=configDB.GetHead();

	while(curNode)
	{
		if (curNode->data.name==className)
			return true;

		// Try subclasses
		if (!IsInstr(className,"/"))
		{
			char* cpos = strrchr(curNode->data.name,'/');

			if (cpos)
			{
				if (strcmp(cpos+1,className)==0)
					return true;
			}
		}

		curNode=curNode->next;
	}

	return false;
}

bool ScriptIniParser::TypeExists(ConfigClass* cclass,CStr typeName)
{
	Link<ConfigType>* subNode=cclass->types.GetHead();

	// Search for the type
	while(subNode)
	{
		if (subNode->data.name==typeName)
			return true;

		// If the type contains only the root of the path it's treated as being existent
		if (!IsInstr(typeName,"/"))
		{
			char* cpos = strrchr(subNode->data.name,'/');
			
			if (cpos)
			{
				if (strcmp(cpos+1,typeName)==0)
					return true;
			}
		}

		subNode=subNode->next;
	}

	return false;
}

bool ScriptIniParser::TypeExists(CStr className,CStr typeName)
{
	Link<ConfigClass>* curNode=configDB.GetHead();

	while(curNode)
	{
		if (curNode->data.name==className)
			return TypeExists(&curNode->data,typeName);

		curNode=curNode->next;
	}

	// Class doesn't exist try subclasses
	if (!IsInstr(typeName,"/"))
	{
		curNode = configDB.GetHead();

		while(curNode)
		{
			char* cpos = strrchr(curNode->data.name,'/');

			if (cpos)
			{
				if (strcmp(cpos+1,typeName)==0)
					return TypeExists(&curNode->data,typeName);
			}

			curNode=curNode->next;
		}
	}

	return false;
}

void ScriptIniParser::RemoveDefaults(CStr origClass,
									 CStr origType,
									 LinkList<ConfigProp>* origProps)
{
	// Merge in the new properties to the original properties if any of the new properties either
	// do not exist or they contain a value that does not match the default property in the original list

	Link<ConfigProp>* linkOrig = origProps->GetHead();
	Link<ConfigProp>* linkNext;

	// Remove any items in the original list that match their defaults
	while(linkOrig)
	{
		linkNext = linkOrig->next;

		if (linkOrig->data.value == GetDefault(origClass,origType,linkOrig->data.name))
			origProps->Remove(linkOrig);

		linkOrig=linkNext;
	}
}

// Returns true if all the properties in the given property list are defaulted
bool ScriptIniParser::IsDefaulted(PropList* plist, CStr className, CStr typeName)
{
	int nprops = plist->NumProps();

	for(int i=0;i<nprops;i++)
	{
		CStr name = plist->GetName(i);
		CStr value;

		plist->GetValue(i,value);

		if (value != GetDefault(className,typeName,name))
			return false;
	}

	return true;
}

COLORREF ScriptIniParser::GetNodeColor(CStr className, CStr typeName)
{
	CStr propBuf = GetClassTypeRecord(className, typeName);

	// Scan each line for a nodecolor directive
	COLORREF color;
	char* cpos = strstr(propBuf, "// @nodecolor |");
	char  bufColor[256];

	if (!cpos)
		return 0;
	
	cpos += 15;
	cpos = strstr(cpos, "[");

	if (!cpos)
		return 0;

	cpos++;

	GetOption(cpos, bufColor, ',', 0);
	color = (COLORREF)((unsigned char)atoi(bufColor));
	GetOption(cpos, bufColor, ',', 1);
	color |= (COLORREF)((unsigned char)atoi(bufColor)) << 8;
	GetOption(cpos, bufColor, ',', 2);
	color |= (COLORREF)((unsigned char)atoi(bufColor)) << 16;

	return color;
}

bool ScriptIniParser::AdvanceToClassGroup(FILE* fp, char* lineBuf, int pos, CStr className)
{
	CStr groupName = GetWordToken(fp,lineBuf,&pos);
	CStr word;

	CStr oldGroup = strCurClassGroup;

	if (strCurClassGroup.Length()==0)
		strCurClassGroup = groupName;
	else
		strCurClassGroup += CStr("/") + groupName;

	// Advance to start block
	if (!FileAdvance(fp,lineBuf,&pos,'{'))
		return false;

	word = GetWordToken(fp,lineBuf,&pos);

	while(word!=CStr(""))
	{
		if (word==CStr("Group"))
		{
			if (AdvanceToClassGroup(fp, lineBuf, pos, className))
				return true;

			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		if (word==CStr("Class"))
		{
			if (AdvanceToClass(fp, lineBuf, pos, className))
				return true;

			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		if (word==CStr(" "))
		{
			word=GetWordToken(fp,lineBuf,&pos);
			continue;
		}

		if (word==CStr("}"))
		{
			strCurClassGroup = oldGroup;
			return false;
		}

		word=GetWordToken(fp,lineBuf,&pos);
	}

	return false;
}

bool ScriptIniParser::AdvanceToClass(FILE* fp, char* lineBuf, int pos, CStr className)
{
	CStr curclassName = GetWordToken(fp,lineBuf,&pos);

	int dbgPos = ftell(fp);

	if (!FileAdvance(fp,lineBuf,&pos,'{'))
		return false;

	dbgPos = ftell(fp);

	if (curclassName == className)
		return true;

	CStr word = GetWordToken(fp,lineBuf,&pos);				//  Get the next line delim	

	while(word!=CStr(""))
	{
		// Check if a grouping directive was encountered
		if (word==CStr("Group"))
		{
			if (AdvanceToClassGroup(fp, lineBuf, pos, className))
				return true;
			
			word=GetWordToken(fp, lineBuf, &pos);
			continue;
		}

		if (word==CStr("Class"))
		{
			if (AdvanceToClass(fp, lineBuf, pos, className))
				return true;

			word=GetWordToken(fp, lineBuf, &pos);
			continue;
		}

		if (word==CStr(" "))
		{
			word=GetWordToken(fp, lineBuf, &pos);
			continue;
		}

		if (word==CStr("}"))
			return false;

		word=GetWordToken(fp, lineBuf, &pos);
	}

	return false;
}

// This function will advance to a particular class in the scripts.ini file
// The file is assumed open.  The file pointer will be positioned immediately following
// the first '{' after the Class name in the class definition
bool ScriptIniParser::AdvanceToClass(FILE* fp, CStr className)
{
	if (!fp)
		return false;

	char lineBuf[512];
	int  pos=1000;
	CStr word=GetWordToken(fp,lineBuf,&pos);

	while(word!=CStr(""))
	{
		// Parse Group
		if (word==CStr("Group"))
			if (AdvanceToClassGroup(fp,lineBuf,pos,className))
				return true;
		
		// Parse Class
		if (word==CStr("Class"))
			if (AdvanceToClass(fp,lineBuf,pos,className))
				return true;

		word=GetWordToken(fp,lineBuf,&pos);
	}

	return false;
}

bool ScriptIniParser::IsViewable(INode* node, CStr className, CStr typeName)
{
	//return true;		// Temporarily disabled until we discuss how we really want this done  aml 6-3-03

	/*
	Object* obj = node->EvalWorldState(0).obj;
	CStr buf = GetClassTypeRecord(className, typeName);

	// If a trigger class is selected then [NonPlaceable] classes/types should not be listed
	if (obj->ClassID() == vTRIGGER_CLASS_ID)
	{
		if (strstr(buf, "[NonPlaceable]"))
			return false;
		else
			return true;
	}
	else
	{
		// If something else is selected besides a trigger then only [NonPlaceable] classes/types should be listed
		if (strstr(buf, "[NonPlaceable]"))
			return true;
		else
			return false;
	}
	*/

	// Anything marked as TriggerOnly should only be viewed in the PE if a trigger class is selected
	Object* obj = node->EvalWorldState(0).obj;
	CStr buf = GetClassTypeRecord(className, typeName);

	// If a non-trigger class is selected and the type record has a TriggerOnly directive it should not be listed
	if (obj->ClassID() != vTRIGGER_CLASS_ID)
	{
		if (strstr(buf, "[TriggerOnly]"))
			return false;
	}

	return true;
}

Link<ConfigClass>* ScriptIniParser::ConfigDBFind(char* className)
{
	Link<ConfigClass>* link = configDB.GetHead();

	while(link)
	{
		if (strcmp(link->data.name, className) == 0)
			return link;

		link = link->next;
	}

	return NULL;
}

CStr ScriptIniParser::GetEvalProp(INode* node, CStr propName)
{
	CStr propBuffer;
	char propVal[1024];
	node->GetUserPropBuffer(propBuffer);

	if (!GetProperty(propBuffer, propName, propVal) || strcmp(propVal, "!") == 0)
	{
		CStr className = GetClassName(propBuffer);
		CStr typeName  = GetTypeName(propBuffer);

		return GetDefault(className, typeName, propName);
	}

	return CStr(propVal);
}

