/*
	PieceBrowser.cpp
	Maintains a catalog of pieces to use and place to build up the level
*/

#include "PieceBrowser.h"
#include "Resource.h"
#include "path.h"
#include "PropEdit/ParseFuncs.h"
#include "PropEdit/ConfigData.h"
#include "../Export/SFP.h"
#include "../UI/MultiList.h"
#include "../misc/find.h"
#include "PartPreview.h"
#include "../Material/NExtMat.h"
#include "../Material/NExtMultiMat.h"
#include "../Texture/NExtTexture.h"
#include "imaterial.h"
#include "stdmat.h"  // For MultiMtl

#include <gl/gl.h>
#include <gl/glu.h>

#define  MODELNAME_CHUNK_ID 0x0001

HGLRC  hglrc;
extern PartPreviewDlg* pPartPreviewDlg;

// Version History
// v2 - Face Flags

#define PIECE_FILE_VERSION  0x0004				// Current version of the piece file .nsp (NeverSoft Part) to export
#define WM_REFRESHPARTLIST  WM_USER + 0x0050

#define UPDATE_WIDTH   0
#define UPDATE_HEIGHT  1
#define UPDATE_LENGTH  2

//#define DEBUGLINE(line) { FILE* fpDbg = fopen("c:\\pbdebug.txt","a"); fprintf(fpDbg, "%s\n", line); fclose(fpDbg); }
#define DEBUGLINE(line)

static PieceBrowserObjClassDesc pieceBrowserObjDesc;
ClassDesc2* GetPieceBrowserObjDesc() { return &pieceBrowserObjDesc; }

extern Interface* gInterface;

// pblocks
enum { piecebrowserobj_params, };

// params
enum 
{
		piecebrowserobj_width,
		piecebrowserobj_height,
		piecebrowserobj_len,
		piecebrowserobj_model,
		piecebrowserobj_lockaspect,
		piecebrowserobj_lockwidthheight,
		piecebrowserobj_resetdimensions,
		piecebrowserobj_lockdimensions,
		piecebrowserobj_lockwidth,
		piecebrowserobj_lockheight,
		piecebrowserobj_locklength,
		piecebrowserobj_buildrot,
		piecebrowserobj_rot,
};

// Define param block
static ParamBlockDesc2 param_blk_desc(piecebrowserobj_params,
									  _T("params"), 0, &pieceBrowserObjDesc,
									  P_AUTO_CONSTRUCT + P_AUTO_UI, PBLOCK_REF,

									  // rollout
									  IDD_PIECEBROWSER, IDS_PARAMS, 0, 0, NULL,

									  // params
									  piecebrowserobj_width, _T("width"), TYPE_FLOAT, 0, IDS_SFX_WIDTH,
									  p_default,		10.0f,
									  p_range,			0.0f, FLT_MAX,
									  p_ui,             TYPE_SPINNER, EDITTYPE_UNIVERSE,
										IDC_PB_EDITWIDTH, IDC_PB_SPINWIDTH, SPIN_AUTOSCALE,
									  end,

									  piecebrowserobj_height, _T("height"), TYPE_FLOAT, 0, IDS_SFX_HEIGHT,
									  p_default,        10.0f,
									  p_range,          0.0f, FLT_MAX,
									  p_ui,             TYPE_SPINNER, EDITTYPE_UNIVERSE,
										IDC_PB_EDITHEIGHT, IDC_PB_SPINHEIGHT, SPIN_AUTOSCALE,
									  end,

									  piecebrowserobj_len, _T("length"), TYPE_FLOAT, 0, IDS_SFX_LENGTH,
									  p_default,        10.0f,
									  p_range,          0.0f, FLT_MAX,
									  p_ui,             TYPE_SPINNER, EDITTYPE_UNIVERSE,
										IDC_PB_EDITLEN, IDC_ASPINNER, SPIN_AUTOSCALE,
									  end,

									  piecebrowserobj_model, _T("model"), TYPE_STRING, 0, IDS_SFX_MODEL,
									  p_default,        _T("test.nsp"),
									  end,

									  piecebrowserobj_lockaspect, _T("lockaspect"), TYPE_BOOL, 0, IDS_SFX_LOCKASPECT,
									  p_default,        TRUE,
									  p_ui,             TYPE_SINGLECHEKBOX, IDC_LOCKASPECT,
									  end,

									  piecebrowserobj_lockwidthheight, _T("CubeLock"), TYPE_BOOL, 0, IDS_SFX_CUBELOCK,
									  p_default,        TRUE,
									  p_ui,             TYPE_SINGLECHEKBOX, IDC_LOCKWIDTHHEIGHT,
									  end,

									  piecebrowserobj_resetdimensions, _T("ResetDimensions"), TYPE_BOOL, 0, IDS_SFX_RESETDIM,
									  p_default,		TRUE,
									  p_ui,				TYPE_SINGLECHEKBOX, IDC_RESETDIM,
									  end,

									  piecebrowserobj_lockdimensions, _T("LockDimensions"), TYPE_BOOL, 0, IDS_SFX_LOCKDIM,
									  p_default,        TRUE,
									  p_ui,             TYPE_SINGLECHEKBOX, IDC_LOCKDIMENSIONS,
									  end,

									  piecebrowserobj_lockwidth, _T("LockWidth"), TYPE_BOOL, 0, IDS_SFX_LOCKWIDTH,
									  p_default,        FALSE,
									  p_ui,             TYPE_SINGLECHEKBOX, IDC_LOCKWIDTH,
									  end,

									  piecebrowserobj_lockheight, _T("LockHeight"), TYPE_BOOL, 0, IDS_SFX_LOCKHEIGHT,
									  p_default,        FALSE,
									  p_ui,             TYPE_SINGLECHEKBOX, IDC_LOCKHEIGHT,
									  end,

									  piecebrowserobj_locklength, _T("LockLength"), TYPE_BOOL, 0, IDS_SFX_LOCKLENGTH,
									  p_default,        FALSE,
									  p_ui,             TYPE_SINGLECHEKBOX, IDC_LOCKLENGTH,
									  end,

									  piecebrowserobj_buildrot, _T("BuildRotation"), TYPE_BOOL, 0, IDS_SFX_BUILDROT,
									  p_default,        FALSE,
									  p_ui,             TYPE_SINGLECHEKBOX, IDC_BUILDROT,
									  end,

									  piecebrowserobj_rot, _T("Rotation"), TYPE_FLOAT, 0, IDS_SFX_ROT,
									  p_default,		0.0f,
									  p_range,			0.0f, 360.0f,
									  p_ui,				TYPE_SPINNER, EDITTYPE_FLOAT,
									    IDC_ROTVAL, IDC_ROTSPIN, SPIN_AUTOSCALE,
									  end,

									  end);

class PieceBrowserObjDlgProc: public ParamMap2UserDlgProc
{
	HWND hwnd;

	static void AddNSPFile(char* filename, void* data);
	void UpdateSpinner(PieceBrowserObj* obj, int Mode);

public:
	PieceBrowserObj *obj;

	PieceBrowserObjDlgProc(PieceBrowserObj *obj)
	{
		hwnd = NULL;
		this->obj = obj;
	}

	BOOL DlgProc(TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);

	void DeleteThis() { delete this; }

	void RefreshPartList();
	void SelTextUpdate();

	inline HWND GetHWND() { return hwnd; }
};

void PieceBrowserObjDlgProc::RefreshPartList()
{
	if (hwnd)
	{
		SendDlgItemMessage(hwnd,IDC_PARTLIST,LB_RESETCONTENT,0,0);

		obj->partList.Clear();
		obj->mlist->Reset();
		FindFull(obj->Path,".nsp",AddNSPFile,this);
	}
}

void PieceBrowserObjDlgProc::AddNSPFile(char* filename, void* data)
{
	PieceBrowserObjDlgProc* pthis = (PieceBrowserObjDlgProc*)data;

	// Remove the original path
	CStr strTempPath = pthis->obj->Path;
	strTempPath += "\\";

	CStr strAddName = ReplaceStr(filename,strTempPath,"");

	// Convert the file path into a hierarchy path to be added to the multilist
	strAddName = ReplaceStr(strAddName,"\\","/");

	pthis->obj->mlist->AddItem(strAddName);

	// Also add the piece to the part listbox
	CStr partName;
	
	if (strstr(strAddName,"/"))
		partName = strrchr(strAddName,'/') + 1;
	else
		partName = strAddName;

	if (partName.Length() > 0)
	{
		SendDlgItemMessage(pthis->hwnd,IDC_PARTLIST,LB_ADDSTRING,0,(LPARAM)(char*)partName);
		
		// ...and the part list
		pthis->obj->partList.Add(&CStr(filename));
	}
}

void PieceBrowserObjDlgProc::UpdateSpinner(PieceBrowserObj* obj, int Mode)
{
	float scale;
	float newvalue;
	BOOL  bLockAspect = TRUE;

	if (obj && obj->pblock)
	{
		// The dimensions should only be updated proportionately if the
		// aspect ratio isn't locked
		obj->pblock->GetValue(piecebrowserobj_lockaspect, 0, bLockAspect, FOREVER);

		if (bLockAspect)
		{
			// Determine the current scale of the object based on the original normalized aspect data
			// within the current piece
			switch(Mode)
			{
			case UPDATE_WIDTH:
				obj->pblock->GetValue(piecebrowserobj_width, 0, scale, FOREVER);
				scale /= obj->piece.width;

				newvalue = obj->piece.height * scale;
				obj->pblock->SetValue(piecebrowserobj_height, 0, newvalue);

				newvalue = obj->piece.length * scale;
				obj->pblock->SetValue(piecebrowserobj_len, 0, newvalue);
				break;

			case UPDATE_HEIGHT:
				obj->pblock->GetValue(piecebrowserobj_height, 0, scale, FOREVER);
				scale /= obj->piece.height;

				newvalue = obj->piece.width * scale;
				obj->pblock->SetValue(piecebrowserobj_width, 0, newvalue);

				newvalue = obj->piece.length * scale;
				obj->pblock->SetValue(piecebrowserobj_len, 0, newvalue);
				break;

			case UPDATE_LENGTH:
				obj->pblock->GetValue(piecebrowserobj_len, 0, scale, FOREVER);
				scale /= obj->piece.length;

				newvalue = obj->piece.width * scale;
				obj->pblock->SetValue(piecebrowserobj_width, 0, newvalue);

				newvalue = obj->piece.height * scale;
				obj->pblock->SetValue(piecebrowserobj_height, 0, newvalue);
				break;
			}
		}
		else
		{
			float scalar;

			switch(Mode)
			{
			case UPDATE_WIDTH:
				obj->pblock->GetValue(piecebrowserobj_width, 0, scalar, FOREVER);
				break;

			case UPDATE_HEIGHT:
				obj->pblock->GetValue(piecebrowserobj_height, 0, scalar, FOREVER);
				break;

			case UPDATE_LENGTH:
				obj->pblock->GetValue(piecebrowserobj_len, 0, scalar, FOREVER);
				break;
			}
		}
	}
}

BOOL PieceBrowserObjDlgProc::DlgProc(TimeValue t, IParamMap2 *map, HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	this->hwnd = hwnd;
	IParamBlock2 *pblock = map->GetParamBlock();

	switch(msg)
	{
	case CC_SPINNER_CHANGE:
		switch(LOWORD(wParam))
		{
		case IDC_PB_SPINWIDTH:
			//UpdateSpinner(pblock, UPDATE_WIDTH);
			UpdateSpinner(obj, UPDATE_WIDTH);
			break;

		case IDC_PB_SPINHEIGHT:
			//UpdateSpinner(pblock, UPDATE_HEIGHT);
			UpdateSpinner(obj, UPDATE_HEIGHT);
			break;

		case IDC_ASPINNER:
			//UpdateSpinner(pblock, UPDATE_LENGTH);
			UpdateSpinner(obj, UPDATE_LENGTH);
			break;
		}

		break;

	case WM_INITDIALOG:
		{
			TCHAR* strModel;
			obj->mlist->Attach(GetDlgItem(hwnd,IDC_COMPSEL));
			pblock->GetValue( piecebrowserobj_model, 0, strModel, FOREVER );
			RefreshPartList();
			obj->mlist->SetValue(strModel);
		}
		return TRUE;

	case WM_REFRESHPARTLIST:
		{
			RefreshPartList();
		}
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_STORE:
			{
				obj->StoreComponent();

			}
			return TRUE;

		case IDC_REFRESH:
			{
				RefreshPartList();
			}
			return TRUE;

		case IDC_PIECEPAL:
			{
				if (!obj->bPreviewInitialized)
				{
					pPartPreviewDlg->Init(obj);
					obj->bPreviewInitialized = true;
				}

				pPartPreviewDlg->Show();
				pPartPreviewDlg->Init();
			}
			return TRUE;

		case IDC_COMPSEL:
			{
				switch(HIWORD(wParam))
				{
				case CBN_SELCHANGE:
					{
						CStr strValue = obj->mlist->GetValue();
						pblock->SetValue(piecebrowserobj_model, 0, strValue);
						obj->RetrieveComponent();
					}
					return TRUE;
				}
			}
			break;

		case IDC_PARTEDIT:
			{
				switch(HIWORD(wParam))
				{
				case EN_CHANGE:
					{
						char buf[256];
						int  sel;

						SelTextUpdate();

						sel = SendDlgItemMessage(hwnd, IDC_PARTLIST, LB_GETCURSEL, 0, 0);
						SendDlgItemMessage(hwnd, IDC_PARTLIST, LB_GETTEXT, (WPARAM)sel, (LPARAM)buf);
						obj->mlist->SetValue(buf);
						pblock->SetValue(piecebrowserobj_model, 0, buf);
						obj->RetrieveComponent();
					}
				}
			}
			return TRUE;

		case IDC_PARTLIST:
			{
				switch(HIWORD(wParam))
				{
				case LBN_SELCHANGE:
					{
						char buf[256];
						int  sel;

						sel = SendDlgItemMessage(hwnd, IDC_PARTLIST, LB_GETCURSEL, 0, 0);
						SendDlgItemMessage(hwnd, IDC_PARTLIST, LB_GETTEXT, (WPARAM)sel, (LPARAM)buf);
						obj->mlist->SetValue(buf);
						pblock->SetValue(piecebrowserobj_model, 0, buf);
						obj->RetrieveComponent();
					}
					return TRUE;
				}
			}
			break;
		}

		break;
	}

	return FALSE;
}

void PieceBrowserObjDlgProc::SelTextUpdate()
{
	ICustEdit* IEdit = GetICustEdit(GetDlgItem(hwnd,IDC_PARTEDIT));

	char wildcard[256];
	char item[1024];

	IEdit->GetText(wildcard,255);
	strcat(wildcard,"*");

	// Select all the items in the part list that match the wildcard
	int nItems = SendDlgItemMessage(hwnd,IDC_PARTLIST,LB_GETCOUNT,0,0);

	for(int i = 0; i < nItems; i++)
	{
		SendDlgItemMessage(hwnd,IDC_PARTLIST,LB_GETTEXT,(WPARAM)i,(LPARAM)item);

		if (MatchPattern(CStr(item),CStr(wildcard)))
		{
			SendDlgItemMessage(hwnd,IDC_PARTLIST,LB_SETCURSEL,(WPARAM)i,0);
			ReleaseICustEdit(IEdit);
			return;
		}
	}

	ReleaseICustEdit(IEdit);
}

class PieceBrowserObjMouseCallback: public CreateMouseCallBack
{
protected:
	IPoint2          sp0, sp1;
	Point3           p0, p1, tp0;
	PieceBrowserObj *obj;
	BOOL             bLockAspect;
	BOOL             bLockWidthHeight;
	BOOL             bLockDimensions;
	BOOL             square;
	float            width, height, length;
	float            scalar;

public:
	int proc(ViewExp *vpt,
		     int     msg,
			 int     point,
			 int     flags,
			 IPoint2 m,
			 Matrix3 &mat);

	inline void SetObject(PieceBrowserObj *obj) { this->obj = obj; }
	inline void SetLockAspect(BOOL bLock)       { bLockAspect      = bLock; }
	inline void SetLockWidthHeight(BOOL bLock)  { bLockWidthHeight = bLock; }
	inline void SetLockDimensions(BOOL bLock)   { bLockDimensions  = bLock; }
	inline void SetScalar(float val)            { scalar = val; }

	void SetDimensions(float w, float h, float l);
};
PieceBrowserObjMouseCallback mouseCallback;

float NormalizeDimensions(float& w, float& h, float& l)
{
	// Normalize the dimensions
	float n;

	if (w > h)
		n = w;
	else
		n = h;

	if (l > n)
		n = l;

	w = w / n;
	h = h / n;
	l = l / n;

	return n;
}

void PieceBrowserObjMouseCallback::SetDimensions(float w, float h, float l)
{
	width  = w;
	height = h;
	length = l;
}

int PieceBrowserObjMouseCallback::proc(ViewExp *vpt,
									   int     msg,
									   int     point,
									   int     flags,
									   IPoint2 m,
									   Matrix3 &mat)
{
	Point3 d;

	if (msg == MOUSE_FREEMOVE)
	{
		vpt->SnapPreview(m,m,NULL, SNAP_IN_3D);
	}

	DEBUGLINE("MouseCallback Start");
	// TODO: Update for width and height
   Point3 p1;

	if (!obj)
	{
		DEBUGLINE("MouseCallback ABORT no object");
		return CREATE_ABORT;
	}

   if (!obj->GetLoadStatus())
   {
	   DEBUGLINE("MouseCallback ABORT GetLoadStatus");
	   obj->bInCreation = false;
		return CREATE_ABORT;
   }

	if (!obj->pblock)
	{
		DEBUGLINE("MouseCallback ABORT No pblock");
		obj->bInCreation = false;
		return CREATE_ABORT;
	}

	BOOL bLockWidth, bLockHeight, bLockLength, bBuildRot;
	obj->pblock->GetValue(piecebrowserobj_lockwidth, 0, bLockWidth, FOREVER );
	obj->pblock->GetValue(piecebrowserobj_lockheight, 0, bLockHeight, FOREVER );
	obj->pblock->GetValue(piecebrowserobj_locklength, 0, bLockLength, FOREVER );
	obj->pblock->GetValue(piecebrowserobj_buildrot, 0, bBuildRot, FOREVER );

	//obj->m_Rot.IdentityMatrix();

  if (bLockDimensions)
  {
		switch(msg)
		{
		case MOUSE_POINT:
		case MOUSE_MOVE:
			switch(point)
			{
			case 0:
				// If dimensions are locked we'll create the object on the first click
				if (bLockWidth && bLockHeight && bLockLength)
					return CREATE_STOP;

				if (!bLockWidth)
					obj->pblock->SetValue(
						piecebrowserobj_width,
						gInterface->GetTime(),
						width * scalar);

				if (!bLockHeight)
					obj->pblock->SetValue(
						piecebrowserobj_height,
						gInterface->GetTime(),
						height * scalar);

				if (!bLockLength)
					obj->pblock->SetValue(
						piecebrowserobj_len,
						gInterface->GetTime(),
						length * scalar);

				p0 = vpt->SnapPoint(m, m, NULL, SNAP_IN_PLANE);
				mat.SetTrans(p0);
				DEBUGLINE("MouseCallback CREATE_STOP");

				if (bBuildRot)
					return CREATE_CONTINUE;
				else
					return CREATE_STOP;

			case 2:
			{
					// Find the angle about the object's center
					Point3 angPt = vpt->SnapPoint(m,m,NULL,SNAP_IN_3D);
					Point3 vUp(0.0f, 0.0f, 1.0f);
					Point3 vRight(1.0f, 0.0f, 0.0f);
					Point3 ptOrigin = mat.GetTrans();
					
					angPt  = angPt - ptOrigin;
					float ang    = acos(DotProd(angPt, vRight) / (angPt.Length() * vRight.Length()));
					float degang = RadToDeg(ang);

					if (angPt.y < 0.0f)
						degang = 360.0f - degang;

					obj->pblock->SetValue(piecebrowserobj_rot, 0, degang);
					return CREATE_CONTINUE;
			}
		    case 3:
				return CREATE_STOP;
			}
		}
  }

  if (bLockAspect)
  {
	   switch(msg)
	   {
	   case MOUSE_POINT:
	   case MOUSE_MOVE:
		  switch(point)
		  {
		  case 0:
			 // If the aspect ratio is locked the piece should start at original dimensions
			  if (bLockWidth && bLockHeight && bLockLength)
				  return CREATE_STOP;

			  if (!bLockWidth)
				obj->pblock->SetValue(piecebrowserobj_width, gInterface->GetTime(), width * scalar);

			  if (!bLockHeight)
				obj->pblock->SetValue(piecebrowserobj_height, gInterface->GetTime(), height * scalar);

			  if (!bLockLength)
				obj->pblock->SetValue(piecebrowserobj_len, gInterface->GetTime(), length * scalar);

			 sp0 = m;
			 tp0 = p0 = vpt->SnapPoint(m, m, NULL, SNAP_IN_PLANE);
			 mat.SetTrans(p0);
			 break;
		  case 1:
			 {
				 mat.IdentityMatrix();
				 p1 = vpt->SnapPoint(m, m, NULL, SNAP_IN_PLANE);
				 mat.SetTrans(p0);
				 //float radius = Length(p1-p0); // calc the drag size
				 float dist = p1.x - p0.x;
				 
				 float size = dist + scalar;

				 // clamp it
				 if (size < 0)
					 size = 0.0f;

				if (bLockWidth && bLockHeight && bLockLength)
					return CREATE_STOP;

				 // set object size to drag size
				if (!bLockWidth)
					 obj->pblock->SetValue(
						piecebrowserobj_width,
						gInterface->GetTime(),
						width * size);

				if (!bLockHeight)
					obj->pblock->SetValue(
						piecebrowserobj_height,
						gInterface->GetTime(),
						height * size);

				if (!bLockLength)
					obj->pblock->SetValue(
						piecebrowserobj_len,
						gInterface->GetTime(),
						length * size);

				 param_blk_desc.InvalidateUI();

				 if(flags & MOUSE_CTRL)
				 {
					float ang = (float)atan2(p1.y-p0.y, p1.x-p0.x);
					mat.PreRotateZ(gInterface->SnapAngle(ang));
				 }

				 if(msg == MOUSE_POINT)
				 {
					 if (Length(m-sp0) < 1)
					 {
						 obj = NULL;
						 DEBUGLINE("MouseCallback CREATE_ABORT");
						 return CREATE_ABORT;
					 }

					 DEBUGLINE("MouseCallback CREATE_STOP");

					if (bBuildRot)
						return CREATE_CONTINUE;
					else
						return CREATE_STOP;
				 }
				 break;
			 }
		  case 2:
			  {
					// Find the angle about the object's center
					Point3 angPt = vpt->SnapPoint(m,m,NULL,SNAP_IN_3D);
					Point3 vUp(0.0f, 0.0f, 1.0f);
					Point3 vRight(1.0f, 0.0f, 0.0f);
					Point3 ptOrigin = mat.GetTrans();
					
					angPt  = angPt - ptOrigin;
					float ang    = acos(DotProd(angPt, vRight) / (angPt.Length() * vRight.Length()));
					float degang = RadToDeg(ang);

					if (angPt.y < 0.0f)
						degang = 360.0f - degang;

					obj->pblock->SetValue(piecebrowserobj_rot, 0, degang);
					return CREATE_CONTINUE;
			  }
		  case 3:
			  return CREATE_STOP;

			 //
		  }
		  break;
	   case MOUSE_ABORT:
		   obj = NULL;
		   DEBUGLINE("MouseCallback CREATE_ABORT");
		  return CREATE_ABORT;
	   }
	   DEBUGLINE("MouseCallback CREATE_CONTINUE");
	   return CREATE_CONTINUE;
  }

	DEBUGLINE("MouseCallback Phase II");

  if (bLockWidthHeight)
   {
	   switch(msg)
	   {
	   case MOUSE_POINT:
	   case MOUSE_MOVE:
		  switch(point)
		  {
		  case 0:
			 sp0 = m;
			 tp0 = p0 = vpt->SnapPoint(m, m, NULL, SNAP_IN_PLANE);
			 mat.SetTrans(p0);
			 break;
		  case 1:
			 {
				 mat.IdentityMatrix();
				 p1 = vpt->SnapPoint(m, m, NULL, SNAP_IN_PLANE);
				 mat.SetTrans(p0);
				 float radius = Length(p1-p0); // calc the drag size

				if (bLockWidth && bLockHeight && bLockLength)
					return CREATE_STOP;

				 // set object size to drag size
				if (!bLockWidth)
					 obj->pblock->SetValue(
						piecebrowserobj_width,
						gInterface->GetTime(),
						radius);

				if (!bLockHeight)
					obj->pblock->SetValue(
						piecebrowserobj_height,
						gInterface->GetTime(),
						radius);

				if (!bLockLength)
					obj->pblock->SetValue(
						piecebrowserobj_len,
						gInterface->GetTime(),
						radius);

				 param_blk_desc.InvalidateUI();

				 if(flags & MOUSE_CTRL)
				 {
					float ang = (float)atan2(p1.y-p0.y, p1.x-p0.x);
					mat.PreRotateZ(gInterface->SnapAngle(ang));
				 }

				 if(msg == MOUSE_POINT)
				 {
					 if (Length(m-sp0) < 1)
					 {
						 obj = NULL;
						 DEBUGLINE("MouseCallback CREATE_ABORT");
						 return CREATE_ABORT;
					 }

					 DEBUGLINE("MouseCallback CREATE_STOP");

					 if (bBuildRot)
						return CREATE_CONTINUE;
					 else
						return CREATE_STOP;
				 }
				 break;
			  }
		  case 2:
			  {
					// Find the angle about the object's center
					Point3 angPt = vpt->SnapPoint(m,m,NULL,SNAP_IN_3D);
					Point3 vUp(0.0f, 0.0f, 1.0f);
					Point3 vRight(1.0f, 0.0f, 0.0f);
					Point3 ptOrigin = mat.GetTrans();
					
					angPt  = angPt - ptOrigin;
					float ang    = acos(DotProd(angPt, vRight) / (angPt.Length() * vRight.Length()));
					float degang = RadToDeg(ang);

					if (angPt.y < 0.0f)
						degang = 360.0f - degang;

					obj->pblock->SetValue(piecebrowserobj_rot, 0, degang);
					return CREATE_CONTINUE;
			  }
		  case 3:
			  return CREATE_STOP;
		  }
		  break;

	   case MOUSE_ABORT:
		   obj = NULL;
		   DEBUGLINE("MouseCallback CREATE_ABORT");
		  return CREATE_ABORT;
	   }
	   DEBUGLINE("MouseCallback CREATE_CONTINUE");
	   return CREATE_CONTINUE;
   }
   else
   {
	   ////////////////////
	    if (msg == MOUSE_UNINIT)
			return TRUE;

		if (msg == MOUSE_ABORT)
		{
			obj->bInCreation = false;
			return CREATE_ABORT;
		}

		if (msg != MOUSE_FREEMOVE)
		{
			switch(point) 
			{
			case 0:
				sp0 = m;
				if (bLockWidth && bLockLength && bLockHeight)
					return CREATE_STOP;

				if (!bLockWidth)
					obj->pblock->SetValue(piecebrowserobj_width,0,0.0f);

				if (!bLockLength)
					obj->pblock->SetValue(piecebrowserobj_len,0,0.0f);

				if (!bLockHeight)
					obj->pblock->SetValue(piecebrowserobj_height,0,0.0f);

				// Flag the system that we are in a creation mode
				// so that we don't try to restore non-updated pblock values
				// if it is unappropriate
				obj->bInCreation = true;

				p0 = vpt->SnapPoint(m,m,NULL,SNAP_IN_3D);
				p1 = p0 + Point3(.01,.01,.01);
				mat.SetTrans(float(.5)*(p0+p1));				
				break;
			case 1:
				sp1 = m;
				p1 = vpt->SnapPoint(m,m,NULL,SNAP_IN_3D);
				p1.z = p0.z +(float).01; 
				if ((flags&MOUSE_CTRL)) 
				{
					mat.SetTrans(p0);
				} 
				else
				{
					mat.SetTrans(float(.5)*(p0+p1));
				} // ???

				d = p1-p0;
				
				square = FALSE;
				if (flags&MOUSE_CTRL) 
				{
					// Constrain to square base
					float len;
					if (fabs(d.x) > fabs(d.y)) len = d.x;
					else len = d.y;
					d.x = d.y = 2.0f * len;
					square = TRUE;
				}

				if (bLockWidth && bLockLength)
					break;

				if (!bLockWidth)
					obj->pblock->SetValue(piecebrowserobj_width,0,float(fabs(d.x)));

				if (!bLockLength)
					obj->pblock->SetValue(piecebrowserobj_len,0,float(fabs(d.y)));

				//obj->pblock->SetValue(piecebrowserobj_height,0,float(fabs(d.z)));
				//obj->pmapParam->Invalidate();
				param_blk_desc.InvalidateUI();

					if (msg==MOUSE_POINT && 
						(Length(sp1-sp0)<3 || Length(d)<0.1f)) 
					{
						obj->bInCreation = false;
						return CREATE_ABORT;
					}
				break;
			case 2:
				p1.z = p0.z + vpt->SnapLength(vpt->GetCPDisp(p1,Point3(0,0,1),sp1,m));

				d = p1-p0;
				
				// Constrain to square base
				float len;
				if (fabs(d.x) > fabs(d.y)) len = d.x;
				else len = d.y;
				d.x = d.y = 2.0f * len;					

				//obj->pblock->SetValue(piecebrowserobj_width,0,float(fabs(d.x)));
				//obj->pblock->SetValue(piecebrowserobj_len,0,float(fabs(d.y)));
				if (!bLockHeight)
					obj->pblock->SetValue(piecebrowserobj_height,0,float(d.z));
				else
					break;

				//obj->pmapParam->Invalidate();				
				//currentMap->Invalidate();
				param_blk_desc.InvalidateUI();
					
				if (msg==MOUSE_POINT) 
				{
					if (bBuildRot)
						return CREATE_CONTINUE;
					else
					{
						obj->bInCreation = false;
						return CREATE_STOP;
					}
				}

				break;
			case 3:
				{	
					// Find the angle about the object's center
					Point3 angPt = vpt->SnapPoint(m,m,NULL,SNAP_IN_3D);
					Point3 vUp(0.0f, 0.0f, 1.0f);
					Point3 vRight(1.0f, 0.0f, 0.0f);
					Point3 ptOrigin = mat.GetTrans();
					
					angPt  = angPt - ptOrigin;
					float ang    = acos(DotProd(angPt, vRight) / (angPt.Length() * vRight.Length()));
					float degang = RadToDeg(ang);

					if (angPt.y < 0.0f)
						degang = 360.0f - degang;

					obj->pblock->SetValue(piecebrowserobj_rot, 0, degang);
					return CREATE_CONTINUE;
				}
			case 4:
				obj->bInCreation = false;
				return CREATE_STOP;
			}
		}
   }

   return TRUE;
}

PieceBrowserObj::PieceBrowserObj()
{
	ip                  = NULL;
	pblock              = NULL;
	amtlLib             = NULL;
	curnode             = NULL;
	curmtl              = NULL;
	bPreviewInitialized = false;
	bPieceLoaded        = false;
	bInCreation         = false;

	// Read in any configuration file data
	ReadConfig();

	mlist = new MultiList(hInstance);
	mlist->EvalFullPath(true);

	pieceBrowserObjDesc.MakeAutoParamBlocks(this);
}

PieceBrowserObj::~PieceBrowserObj()
{
	DeleteAllRefsFromMe();

	if (amtlLib)
		delete amtlLib;

	// Shutdown piece preview window if opened
	/*
	if (previewDlg)
		delete previewDlg;

	if (dlgProc)
		delete dlgProc;
	*/
}

void PieceBrowserObj::RefAdded(RefMakerHandle rm)
{
	DEBUGLINE("PieceBrowserObj::RefAdded Start");

	if (rm->ClassID()==Class_ID(BASENODE_CLASS_ID,0))
	{
		INode* node=(INode*)rm;

		Object* obj=node->EvalWorldState(0).obj;
		
		if (obj && obj->ClassID() == vPIECEBROWSER_CLASS_ID)
		{
			nodeHandle = node->GetHandle();
			curnode = node;
			node->SetMtl((Mtl*)curmtl);

			// Assign face flags
			Object* obj = node->EvalWorldState(0).obj;

			//if (obj->CanConvertToType(triObjectClassID))
			{
				//TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

				//if (triObj)
				{
					//Mesh& mesh = triObj->GetMesh();
					BuildMesh(gInterface->GetTime());

					IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
					if (pFDMgr)
					{
						FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
						if( fdc == NULL ) 
						{
							fdc = new FaceFlagsData();
							pFDMgr->AddFaceDataChan( fdc );
							fdc->FacesCreated( 0, mesh.getNumFaces() );
						}

						// Now assign all the saved face data
						for(int i = 0; i < piece.nFlagTypes; i++)
						{
							fdc->m_data[i] = piece.flags[i];
						}
					}
					
					//if (triObj != obj)
					//	triObj->DeleteThis();
				}
			}

			//node->SetMtl((MultiMtl*)curmtl);
			
			/*
			if (curmtl)
			{
				IMaterial* imtl = (IMaterial*)node->GetInterface(IMATERIAL_INTERFACE_ID);
				imtl->SetINode(node);
			}
			*/
		}
	}

	DEBUGLINE("PieceBrowserObj::RefAdded End");
}

CreateMouseCallBack* PieceBrowserObj::GetCreateMouseCallBack()
{
	BOOL bLockAspect = TRUE, bLockWidthHeight = FALSE, bLockDimensions = FALSE;
	
	mouseCallback.SetObject(this);

	if (pblock)
	{
		pblock->GetValue( piecebrowserobj_lockaspect, 0, bLockAspect, FOREVER );
		pblock->GetValue( piecebrowserobj_lockwidthheight, 0, bLockWidthHeight, FOREVER );
		pblock->GetValue( piecebrowserobj_lockdimensions, 0, bLockDimensions, FOREVER );
	}

	mouseCallback.SetLockAspect(bLockAspect);
	mouseCallback.SetLockWidthHeight(bLockWidthHeight);
	mouseCallback.SetLockDimensions(bLockDimensions);
	mouseCallback.SetDimensions(piece.width, piece.height, piece.length);
	mouseCallback.SetScalar(piece.scalar);

	return &mouseCallback;
}

void PieceBrowserObj::BeginEditParams(IObjParam  *ip,
								      ULONG      flags,
								      Animatable *prev)
{
	DEBUGLINE("PieceBrowserObj::BeginEditParams Begin");

	this->ip = ip;

	char str[256];
	sprintf(str, "Interface: %x  This: %x  Flags: %i  Prev: %x", ip, this, flags, prev);
	DEBUGLINE(str);
	pieceBrowserObjDesc.BeginEditParams(ip, this, flags, prev);

	DEBUGLINE("Pre new PieceBrowserObjDlgProc");
	dlgProc = new PieceBrowserObjDlgProc(this);
	DEBUGLINE("Post new PieceBrowserObjDlgProc");

	param_blk_desc.SetUserDlgProc(dlgProc);
	DEBUGLINE("SetUserDlgProc");

	DEBUGLINE("Pre RetrieveComponent");
	if (!bPieceLoaded)
		RetrieveComponent();
	
	DEBUGLINE("PieceBrowserObj::BeginEditParams End");

	// Connect to the part preview window
	pPartPreviewDlg->Init(this);
}

void PieceBrowserObj::EndEditParams(IObjParam  *ip,
								    ULONG      flags,
								    Animatable *next)
{
	DEBUGLINE("PieceBrowserObj::EndEditParams Start");
	
	mouseCallback.SetObject(NULL);

	if (dlgProc)
		dlgProc->obj = NULL;

	// Disconnect piece preview window if opened
	pPartPreviewDlg->Init(NULL);
	
	pieceBrowserObjDesc.EndEditParams(ip, this, flags, next);
	this->ip = NULL;

	//  dlgProc appears to get deleted by EndEditParams
	dlgProc = NULL;

	DEBUGLINE("PieceBrowserObj::EndEditParams End");
}

RefTargetHandle PieceBrowserObj::Clone(RemapDir &remap)
{
	DEBUGLINE("PieceBrowserObj::Clone Start");
	PieceBrowserObj *obj = new PieceBrowserObj();
	obj->ReplaceReference(0, pblock->Clone(remap));
	return obj;
}

Point3 PieceBrowserObj::GetMinVerts(Mesh& cmesh)
{
	Point3 ptMin;

	if (cmesh.numVerts == 0)
		return Point3(0.0f,0.0f,0.0f);

	ptMin = cmesh.verts[0];

	for(int i = 0; i < cmesh.numVerts; i++)
	{
		if (cmesh.verts[i].x < ptMin.x)
			ptMin.x = cmesh.verts[i].x;

		if (cmesh.verts[i].y < ptMin.y)
			ptMin.y = cmesh.verts[i].y;

		if (cmesh.verts[i].z < ptMin.z)
			ptMin.z = cmesh.verts[i].z;
	}

	return ptMin;
}

Point3 PieceBrowserObj::GetMaxVerts(Mesh& cmesh)
{
	Point3 ptMax;

	if (cmesh.numVerts == 0)
		return Point3(0.0f,0.0f,0.0f);

	ptMax = cmesh.verts[0];

	for(int i = 0; i < cmesh.numVerts; i++)
	{
		if (cmesh.verts[i].x > ptMax.x)
			ptMax.x = cmesh.verts[i].x;

		if (cmesh.verts[i].y > ptMax.y)
			ptMax.y = cmesh.verts[i].y;

		if (cmesh.verts[i].z > ptMax.z)
			ptMax.z = cmesh.verts[i].z;
	}

	return ptMax;
}

void PieceBrowserObj::ScaleMesh(Mesh& modmesh, float width, float height, float length)
{
	DEBUGLINE("PieceBrowserObj::ScaleMesh Start");
	// The mesh should already be loaded into basemesh
	// we will scale the information as appropriate and copy it into the mesh object

	Point3 min = GetMinVerts(modmesh);
	Point3 max = GetMaxVerts(modmesh);

	float curWidth, curHeight, curLength;

	curWidth  = max.x - min.x;
	curHeight = max.z - min.z;
	curLength = max.y - min.y;
	
	// Scale all verts
	for(int i = 0; i < modmesh.numVerts; i++)
	{
		modmesh.verts[i].x = modmesh.verts[i].x * width / curWidth;
		modmesh.verts[i].z = modmesh.verts[i].z * height / curHeight;
		modmesh.verts[i].y = modmesh.verts[i].y * length / curLength;
	}

	modmesh.InvalidateGeomCache();
}

void PieceBrowserObj::BuildMesh(TimeValue t)
{
	DEBUGLINE("PieceBrowserObj::BuildMesh Start");

	if (!bPieceLoaded)
	{
		mesh.numVerts = 0;
		mesh.numFaces = 0;

		mesh.InvalidateTopologyCache();
		mesh.InvalidateGeomCache();

		return;
	}

	float width  = pblock->GetFloat(piecebrowserobj_width, t);
	float height = pblock->GetFloat(piecebrowserobj_height, t);
	float length = pblock->GetFloat(piecebrowserobj_len, t);
	float degrot = pblock->GetFloat(piecebrowserobj_rot, t);

	// The mesh should already be loaded into basemesh
	// we will scale the information as appropriate and copy it into the mesh object

	Point3 min = GetMinVerts(piece.mesh);
	Point3 max = GetMaxVerts(piece.mesh);

	float curWidth, curHeight, curLength;

	curWidth  = max.x - min.x;
	curHeight = max.z - min.z;
	curLength = max.y - min.y;
	

	mesh.DeepCopy(&piece.mesh,GEOM_CHANNEL | TOPO_CHANNEL | TEXMAP_CHANNEL |	MTL_CHANNEL | DISP_ATTRIB_CHANNEL | TM_CHANNEL);
	//mesh = piece.mesh;

	//mesh.setNumMaps(piece.nTextures);

	// Scale all verts
	for(int i = 0; i < mesh.numVerts; i++)
	{
		//int ID = mesh.getFaceMtlIndex(i);

		mesh.verts[i].x = mesh.verts[i].x * width / curWidth;
		mesh.verts[i].z = mesh.verts[i].z * height / curHeight;
		mesh.verts[i].y = mesh.verts[i].y * length / curLength;
		mesh.verts[i] = mesh.verts[i] * RotateZMatrix(DegToRad(degrot));
	}

	mesh.InvalidateTopologyCache();
	mesh.InvalidateGeomCache();
}

void PieceBrowserObj::FreeCaches()
{
	mesh.FreeAll();
}

void PieceBrowserObj::GetLocalBoundBox(TimeValue t,
									   INode     *node,
									   ViewExp   *vpt,
									   Box3      &box)
{
	BuildMesh(t);
	box = mesh.getBoundingBox();
}

void PieceBrowserObj::GetWorldBoundBox(TimeValue t,
								       INode     *node,
								       ViewExp   *vpt,
								       Box3      &box)
{
	BuildMesh(t);
	Matrix3 tm = node->GetObjectTM(t);
	box = mesh.getBoundingBox() * tm;
}


int PieceBrowserObj::Display(TimeValue t,INode *node, ViewExp *vpt, int flags)
{
	DEBUGLINE("PieceBrowserObj::Display Start");
	BuildMesh(t);

   // get the graphics window API and setup for drawing
   GraphicsWindow *gw = vpt->getGW();
   Material *mtl = gw->getMaterial();
   //DWORD limits = gw->getRndLimits();
   //gw->setRndLimits(GW_WIREFRAME | GW_BACKCULL);
   //gw->setRndLimits(GW_POLY_EDGES);

   // setup colors
   if(node->Selected()) {
      gw->setColor(TEXT_COLOR, 1.f, 1.f, 1.f);
      gw->setColor(LINE_COLOR, 1.f, 1.f, 1.f);
   }
   else if(!node->IsFrozen()) {
      gw->setColor(TEXT_COLOR, 0.5f, 0.5f, 1.f);
      gw->setColor(LINE_COLOR, 0.5f, 0.5f, 1.f);
   }

   // setup window transformation
   Matrix3 tm(1);
   tm = node->GetObjectTM(t);
   gw->setTransform(tm);

   // finally draw the mesh itself
   //mesh.render(gw, mtl, NULL, COMP_ALL);
   mesh.render(gw, node->Mtls(), (flags & USE_DAMAGE_RECT) ? &vpt->GetDammageRect() : NULL, COMP_ALL, node->NumMtls());

   //gw->setRndLimits(limits);
   DEBUGLINE("PieceBrowserObj::Display End");
   return 0;
}


int PieceBrowserObj::HitTest(TimeValue t,INode *node,int type,int crossing,int flags,IPoint2 *p,ViewExp *vpt)
{
	DEBUGLINE("PieceBrowserObj::HitTest Start");
   HitRegion hitRegion;
   Point2 pt((float)p[0].x, (float)p[0].y);
   GraphicsWindow *gw = vpt->getGW();

   BuildMesh(t);
   gw->setTransform(node->GetObjectTM(t));
   MakeHitRegion(hitRegion, type, crossing, 4, p);

   return mesh.select(
      gw,
      node->Mtls(),
      &hitRegion,
      flags & HIT_ABORTONHIT,
      node->NumMtls());

   DEBUGLINE("PieceBrowserObj::HitTest End");
}

ObjectState PieceBrowserObj::Eval(TimeValue t)
{
	return ObjectState(this);
}

// We will support conversion to triangle meshes
int PieceBrowserObj::CanConvertToType(Class_ID obtype)
{
	if (obtype == triObjectClassID ||
		obtype == vPIECEBROWSER_CLASS_ID)
		return 1;
	else
		return Object::CanConvertToType(obtype);
}

Object* PieceBrowserObj::ConvertToType(TimeValue t, Class_ID obtype)
{
	if (obtype == vPIECEBROWSER_CLASS_ID)
		return this;

	if (obtype == triObjectClassID)
	{
		TriObject *triob;
		BuildMesh(t);
		triob = CreateNewTriObject();
		Mesh& triobmesh = triob->GetMesh();
		triobmesh = mesh;
		//triob->GetMesh() = mesh;

		IFaceDataMgr* pFDMgrSrc = static_cast<IFaceDataMgr*>(triobmesh.GetInterface( FACEDATAMGR_INTERFACE ));

		if ( pFDMgrSrc )
		{
			FaceFlagsData* fdcSrc = dynamic_cast<FaceFlagsData*>( pFDMgrSrc->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));

			if ( fdcSrc == NULL )
			{
				// The mesh does not have our face-data channel so we will add it here
				fdcSrc = new FaceFlagsData();
				pFDMgrSrc->AddFaceDataChan( fdcSrc );			
			}

			fdcSrc->FacesCreated(0, triobmesh.numFaces);

			// Copy the face flags maintained within the piece database to the actual face flag channel
			for(int i = 0; i < piece.nFlagTypes; i++)
			{
				fdcSrc->m_data[i] = piece.flags[i];
			}
		}

////////////// RETURNHERE

		triob->SetChannelValidity(TOPO_CHAN_NUM,ObjectValidity(t));
		triob->SetChannelValidity(GEOM_CHAN_NUM,ObjectValidity(t));	
		triob->SetChannelValidity(MTL_CHAN_NUM,ObjectValidity(t));	// change
		return triob;
	}
	else
		return Object::ConvertToType(t,obtype);
}

void PieceBrowserObj::ReadConfig()
{
	CStr appDir=gInterface->GetDir(APP_PLUGCFG_DIR);
	appDir+="\\PieceBrowser.ini";

	char strRenderer[256];

	GetPrivateProfileString("Parts","PartDir",".",Path,PATH_SIZE,appDir);
	GetPrivateProfileString("Parts","Renderer","OpenGL",strRenderer, 255, appDir);

	if (strcmp(strRenderer,"OpenGL") == 0)
		rendererType = RENDERER_OPENGL;
	else
	if (strcmp(strRenderer,"Direct3D") == 0)
		rendererType = RENDERER_DIRECT3D;
	else
		rendererType = RENDERER_OPENGL;		// If type is unrecognized use OpenGL Renderer
}

void PieceBrowserObj::WriteConfig()
{
	CStr appDir=gInterface->GetDir(APP_PLUGCFG_DIR);
	appDir+="\\PieceBrowser.ini";

	WritePrivateProfileString("Parts","PartDir",Path,appDir);
}

void PieceBrowserObj::StoreComponent()
{
	// Pop up a save dialog to save the piece
	OPENFILENAME ofn;
	char filename[2048]="";

	ofn.lStructSize       = sizeof(ofn);
	ofn.hwndOwner         = gInterface->GetMAXHWnd();
	ofn.hInstance         = hInstance;
	ofn.lpstrFilter       = "NeverSoft Piece File (*.nsp)\0*.nsp\0\0";
	ofn.lpstrCustomFilter = NULL;
	ofn.nMaxCustFilter    = 0;
	ofn.nFilterIndex      = 0;
	ofn.lpstrFile         = filename;
	ofn.nMaxFile          = 256;
	ofn.lpstrFileTitle    = NULL;
	ofn.nMaxFileTitle     = 0;
	ofn.lpstrInitialDir   = NULL;
	ofn.lpstrTitle        = "Save piece";
	ofn.Flags             = OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset       = 0;
	ofn.nFileExtension    = 0;
	ofn.lpstrDefExt       = TEXT(".nsp");
	ofn.lCustData         = 0;
	ofn.lpfnHook          = NULL;
	ofn.lpTemplateName    = NULL;

	GetSaveFileName(&ofn);

	if (strlen(filename) > 0)
	{
		// In addition to saving the piece we also want to save a material library with the
		// material data for the selected node(s)
		SaveMtlLib(filename);
		
		if (SavePiece(filename))
			MessageBox(gInterface->GetMAXHWnd(),"The piece has been saved","Save Piece",MB_ICONINFORMATION|MB_OK);
		else
			MessageBox(gInterface->GetMAXHWnd(),"Failed to save piece!","Save Piece",MB_ICONSTOP|MB_OK);
	}
}

bool PieceBrowserObj::SaveMtlLib(char* filename)
{
	MtlBaseLib newMatLib;
	CStr strFilename = filename;
	strFilename.toLower();
	strFilename = ReplaceStr(filename,".nsp",".mat");

	// Add the currently selected nodes to the new material library
	int nNodes = gInterface->GetSelNodeCount();

	for(int i = 0; i < nNodes; i++)
	{
		INode* node = gInterface->GetSelNode(i);
		Mtl*    mtl = node->GetMtl();
		
		if (mtl)
			newMatLib.Add(mtl);
	}

	if (!gInterface->SaveMaterialLib(strFilename, &newMatLib))
		return false;

	return true;
}

void PieceBrowserObj::RetrieveComponent(CStr value)
{
	DEBUGLINE("PieceBrowserObj::RetrieveComponent Start");

	// Reteive the filename from the multilist control
	CStr filename;
	CStr basename;

	if (value.Length() == 0)
		basename = mlist->GetValue();
	else
		basename = value;
		
	filename = CStr(Path) + CStr("\\") + basename;
	filename = ReplaceStr(filename,"/","\\");

	if (LoadPiece(&piece, filename))
	{
		// Attempt to match up the material, if we can't we'll load in its material lib
		MtlBaseLib  mtlLib;
		int         mtlID;
		curmtl = NULL;

		// First try the loaded material library
		mtlLib = gInterface->GetMaterialLibrary();
		mtlID  = mtlLib.FindMtlByName(piece.mtlName);

		if (mtlID != -1)
		{
			curmtl = mtlLib[mtlID];
		}

		if (mtlID == -1)
		{
			/*
			MtlBaseLib* mtlLib;

			mtlLib = gInterface->GetSceneMtls();
			mtlID  = mtlLib->FindMtlByName(piece.mtlName);

			if (mtlID != -1)
			{
				curmtl = (*mtlLib)[mtlID];
			}
			else
			*/

			// GetSceneMtls doesn't appear to always give an accurate representation
			// of the materials in the scene, the below manually scans each node for the material
			curmtl = ForceFindSceneMtlByName(piece.mtlName);

			if (!curmtl)
			{
				CStr matFilename = filename;
				matFilename.toLower();

				matFilename = ReplaceStr(matFilename, ".nsp", ".mat");

				if (amtlLib)
					delete amtlLib;

				amtlLib = new MtlBaseLib;

				*amtlLib = gInterface->GetMaterialLibrary();		
				amtlLib->DeleteAll();

				if (piece.mtlName.Length() > 0)
				{
					if (!gInterface->LoadMaterialLib(matFilename, amtlLib))
					{
						char strErr[256];
						sprintf(strErr, "Couldn't find the material in memory, tried the material libraray '%s' but, it could not be loaded.", (char*)matFilename);

						MessageBox(gInterface->GetMAXHWnd(), strErr, "Material Library Not Found", MB_ICONWARNING | MB_OK);
					}

					mtlID  = amtlLib->FindMtlByName(piece.mtlName);

					if (mtlID != -1)
					{
						curmtl = (*amtlLib)[mtlID];
					}
					else
					{
						char strErr[256];
						sprintf(strErr, "A required material '%s' does not exist in the associated material libraray '%s'.", (char*)piece.mtlName, (char*)matFilename);

						MessageBox(gInterface->GetMAXHWnd(), strErr, "Material Not Found", MB_ICONWARNING | MB_OK);		
					}
				}
			}
		}

		if (curnode)
			curnode->SetMtl((Mtl*)curmtl);

		bPieceLoaded = true;
	}

	if (pblock)
	{
		BOOL bResetDimensions = FALSE;
		BOOL bLockWidth = FALSE;
		BOOL bLockHeight = FALSE;
		BOOL bLockLength = FALSE;

		pblock->GetValue(piecebrowserobj_resetdimensions, 0, bResetDimensions, FOREVER );
		pblock->GetValue(piecebrowserobj_lockwidth, 0, bLockWidth, FOREVER );
		pblock->GetValue(piecebrowserobj_lockheight, 0, bLockHeight, FOREVER );
		pblock->GetValue(piecebrowserobj_locklength, 0, bLockLength, FOREVER );

		if (bResetDimensions)
		{
			// We want the piece to maintain the average size of the current piece so
			// Grab the current dimensions and take the largest to get our scale
			float origWidth, origHeight, origLength;
			pblock->GetValue(piecebrowserobj_width, 0, origWidth, FOREVER );
			pblock->GetValue(piecebrowserobj_height, 0, origHeight, FOREVER );
			pblock->GetValue(piecebrowserobj_len, 0, origLength, FOREVER );

			float scale;

			if (origWidth > origHeight)
				scale = origWidth;
			else
				scale = origHeight;

			if (scale < origLength)
				scale = origLength;

			if (!bLockWidth)
				pblock->SetValue(piecebrowserobj_width,  0, piece.width  * scale);

			if (!bLockHeight)
				pblock->SetValue(piecebrowserobj_height, 0, piece.height * scale);

			if (!bLockLength)
				pblock->SetValue(piecebrowserobj_len, 0, piece.length * scale);
		}
	}

	gInterface->ForceCompleteRedraw();
	DEBUGLINE("PieceBrowserObj::RetrieveComponent End");
}

void WriteMtlName(int mtlID, Mtl* mtl, FILE* fp)
{
	if (mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
	{
		INExtMaterial* next_mtl = dynamic_cast<INExtMaterial*>(mtl);
		Texmap* tmap = next_mtl->GetTexmap(0);

		if (tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
		{
			INExtTexture* next_tex = dynamic_cast< INExtTexture* >( tmap );

			BitmapInfo* bmi = next_tex->GetBaseMap(0);

			const char* file;
			int len;

			if (bmi)
			{
				file = bmi->Name();
				len  = strlen(file);
			}
			else
			{
				file = "";
				len  = 0;
			}

			fwrite(&mtlID,sizeof(int),1,fp);
			fwrite(&len,sizeof(int),1,fp);
			fwrite(file,len,1,fp);
		}
		else
		{
			int len = 0;
			mtlID = 0;

			fwrite(&mtlID,sizeof(int), 1, fp);
			fwrite(&len, sizeof(int), 1, fp);
		}
	}
	else
	{
		int len = 0;
		mtlID = 0;

		fwrite(&mtlID,sizeof(int), 1, fp);
		fwrite(&len, sizeof(int), 1, fp);
	}
}

// Scan the given mesh to see if it uses the specified mtl ID
bool MtlIsUsed(int mtlID, Mesh& mesh)
{
	for(int i = 0; i < mesh.numFaces; i++)
	{
		if (mesh.getFaceMtlIndex(i) == mtlID)
			return true;
	}

	return false;
}

// This function counts up the number of unique material IDs within a given mesh
int NumUsedMtls(LinkList<int>& mtlList, Mesh& mesh, int maxMtl)
{
	for(int i = 0; i < mesh.numFaces; i++)
	{
		int mtlID = mesh.getFaceMtlIndex(i) % maxMtl;
		mtlList.AddToTailUnique(&mtlID);
	}

	return mtlList.GetSize();
}

void PieceBrowserObj::GetWidthHeightLength(Mesh& mesh, float& width, float& height, float& length)
{
	int nVerts = mesh.numVerts;
	Point3 ptMin, ptMax;

	if (nVerts == 0)
		return;

	ptMin = mesh.verts[0];
	ptMax = mesh.verts[0];

	for(int i = 1; i < nVerts; i++)
	{
		if (mesh.verts[i].x < ptMin.x)
			ptMin.x = mesh.verts[i].x;

		if (mesh.verts[i].y < ptMin.y)
			ptMin.y = mesh.verts[i].y;

		if (mesh.verts[i].z < ptMin.z)
			ptMin.z = mesh.verts[i].z;

		if (mesh.verts[i].x > ptMax.x)
			ptMax.x = mesh.verts[i].x;

		if (mesh.verts[i].y > ptMax.y)
			ptMax.y = mesh.verts[i].y;

		if (mesh.verts[i].z > ptMax.z)
			ptMax.z = mesh.verts[i].z;
	}

	width     = ptMax.x - ptMin.x;
	length    = ptMax.y - ptMin.y;
	height    = ptMax.z - ptMin.z;
}

bool PieceBrowserObj::SavePiece(char* Filename)
{
	FILE* fp = fopen(Filename,"wb");
	long version = PIECE_FILE_VERSION;

	if (!fp)
		return false;

	fwrite(&version,sizeof(long),1,fp);

	int nNodes = gInterface->GetSelNodeCount();

	fwrite(&nNodes,sizeof(int),1,fp);

	for(int i = 0; i < nNodes; i++)
	{
		INode* node = gInterface->GetSelNode(i);
		Object* obj = node->EvalWorldState(0).obj;

		// Should always be the case, but always best to check
		if ( obj->CanConvertToType(triObjectClassID) )
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

			Mesh& mesh = triObj->GetMesh();

			// Write out the vert list
			fwrite(&mesh.numVerts, sizeof(int), 1, fp);
			fwrite(mesh.verts, sizeof(Point3), mesh.numVerts, fp);

			// Write out the face list
			fwrite(&mesh.numFaces, sizeof(int), 1, fp);
			fwrite(mesh.faces, sizeof(Face), mesh.numFaces, fp);

			// For now we won't worry about multitexturing, just get the first set of texture coords
			// Write out texture coords
			fwrite(&mesh.numTVerts, sizeof(int), 1, fp);

			if (mesh.numTVerts > 0)
			{
				fwrite(mesh.tVerts,sizeof(UVVert), mesh.numTVerts, fp);
				fwrite(mesh.tvFace, sizeof(TVFace), mesh.numFaces, fp);
			}

			// !!
			// Write out the extended mapping channels for multi-texture v4
			int nMaps = mesh.getNumMaps();
			fwrite(&nMaps, sizeof(int), 1, fp);
			
			for(int channel = 2; channel < nMaps; channel++)
			{
				int nMapVerts, nMapFaces;
				
				nMapVerts = mesh.getNumMapVerts(channel);
				fwrite(&nMapVerts, sizeof(int), 1, fp);

				UVVert* uvv = mesh.mapVerts(channel);
				fwrite(uvv, sizeof(UVVert), nMapVerts, fp);

				TVFace* tvf = mesh.mapFaces(channel);

				if (tvf)
					nMapFaces = mesh.numFaces;
				else
					nMapFaces = 0;

				fwrite(&nMapFaces, sizeof(int), 1, fp);
				fwrite(tvf, sizeof(TVFace), nMapFaces, fp);
			}
			// !!

			// Need to write out the material ID for each face
			for(int curface = 0; curface < mesh.numFaces; curface++)
			{
				int mtlID = mesh.getFaceMtlIndex(curface);
				fwrite(&mtlID, sizeof(int), 1, fp);
			}

			// Write out the material name
			Mtl* mtl = node->GetMtl();
			CStr mtlName;
			
			if (mtl)
				mtlName = mtl->GetName();
			else
				mtlName = "";

			int mtlNameLen = mtlName.Length();

			fwrite(&mtlNameLen, sizeof(int), 1, fp);
			fwrite((char*)mtlName, sizeof(char), mtlNameLen, fp);

			// Write out the texture filename for the first pass texture

			// Need to handle multi-materials
			int numMtls;

			if (mtl)
			{
				if (mtl->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID ||
					mtl->ClassID() == Class_ID(MULTI_CLASS_ID,0))
				{
					// This object uses a multi material we must save out all used texture names
					//numMtls = mtl->NumSubMtls();
					LinkList<int> mtlList;

					numMtls = NumUsedMtls(mtlList, mesh, mtl->NumSubMtls());
					fwrite(&numMtls,sizeof(int),1,fp);

					// Need to put the material IDs in order
					mtlList.Sort();

					// Write out the mtl Names
					Link<int>* curlink = mtlList.GetHead();

					while(curlink)
					{
						Mtl* submtl = mtl->GetSubMtl(curlink->data);
						WriteMtlName(curlink->data,submtl,fp);

						curlink = curlink->next;
					}
				}
				else
				{
					// This object uses a single material
					numMtls = 1;
					fwrite(&numMtls,sizeof(int),1,fp);
					
					WriteMtlName(-1,mtl, fp);
				}
			}
			else
			{
				// This object doesn't have a material
				numMtls = 0;
				fwrite(&numMtls,sizeof(int),1,fp);
			}

			// Write out the face flags on the piece
			IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
			if ( pFDMgr != NULL )
			{
				FaceFlagsData* fdc  = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));

				if (fdc)
				{
					int size = fdc->m_data.Count();
					fwrite(&size, sizeof(int), 1, fp);

					FlagType ftype;
					for(int i = 0; i < size; i++)
					{
						ftype = fdc->m_data[i];
						fwrite(&ftype, sizeof(FlagType), 1, fp);
					}
				}
				else
				{
					int size = 0;
					fwrite(&size, sizeof(int), 1, fp);
				}
			}

			if (triObj != obj)
				triObj->DeleteThis();
		}
	}

	fclose(fp);
	return true;
}

bool PieceBrowserObj::LoadPiece(NSPiece* pPiece, char* Filename)
{
	DEBUGLINE("PieceBrowserObj::LoadPiece Start");
	FILE* fp = fopen(Filename,"rb");
	long version;

	if (!fp)
	{
		return false;
	}

	fread(&version,sizeof(long),1,fp);

	int nNodes;
	fread(&nNodes,sizeof(int),1,fp);

	for(int i = 0; i < nNodes; i++)
	{
		// Read in the verts
		int nVerts;
		fread(&nVerts,sizeof(int),1,fp);
		pPiece->mesh.setNumVerts(nVerts);

		fread(pPiece->mesh.verts, sizeof(Point3), nVerts, fp);

		// Read in the faces
		int nFaces;
		fread(&nFaces,sizeof(int),1,fp);
		pPiece->mesh.setNumFaces(nFaces);

		fread(pPiece->mesh.faces, sizeof(Face), nFaces, fp);

		int nTVerts;
		fread(&nTVerts, sizeof(int), 1, fp);
		pPiece->mesh.setNumTVerts(nTVerts);

		fread(pPiece->mesh.tVerts, sizeof(UVVert), nTVerts, fp);

		if (nTVerts > 0)
		{
			pPiece->mesh.setNumTVFaces(nFaces);
			fread(pPiece->mesh.tvFace, sizeof(TVFace), nFaces, fp);
		}
		else
		{
			pPiece->mesh.setNumTVFaces(0);
		}

		// !!
		// Read in the multitexture UV coords for v4
		if (version >= 0x0004)
		{
			int nChannels;
			fread(&nChannels, sizeof(int), 1, fp);

			pPiece->mesh.setNumMaps(nChannels, TRUE);

			for(int channel = 2; channel < nChannels; channel++)
			{
				// Write out map verts
				int nMapVerts, nMapFaces;
				fread(&nMapVerts, sizeof(int), 1, fp);

				if (nMapVerts > 0)
				{
					pPiece->mesh.setMapSupport(channel);
					pPiece->mesh.setNumMapVerts(channel, nMapVerts);
				}
				
				UVVert* uvv = pPiece->mesh.mapVerts(channel);
				fread(uvv, sizeof(UVVert), nMapVerts, fp);

				fread(&nMapFaces, sizeof(int), 1, fp);

				// Write out map faces
				TVFace* face = pPiece->mesh.mapFaces(channel);
				fread(face, sizeof(TVFace), nMapFaces, fp);
			}
		}

		// Read in the material ID for each face
		pPiece->Alloc();

		for(int curface = 0; curface < nFaces; curface++)
		{
			int mtlID;
			fread(&mtlID,sizeof(int),1,fp);
			pPiece->mesh.setFaceMtlIndex(curface, mtlID);
		}

		// Read in the material name so we can match it with the
		// material again if it exists within the current scene
		int   mtlNameLen;
		char* tmpName;

		fread(&mtlNameLen, sizeof(int), 1, fp);
		tmpName = new char[mtlNameLen + 1];
		fread(tmpName, sizeof(char), mtlNameLen, fp);
		tmpName[mtlNameLen] = '\0';
		pPiece->mtlName = tmpName;
		delete [] tmpName;

		// Read in the unique texture names
		int nUniqueTextures;
		fread(&nUniqueTextures,sizeof(int),1,fp);

		pPiece->AllocNames(nUniqueTextures);

		for(int i = 0; i < nUniqueTextures; i++)
		{
			int nLen;
			int texID;

			fread(&texID,sizeof(int),1,fp);
			fread(&nLen, sizeof(int), 1, fp);
			
			char* tempBuf = new char[nLen + 1];
			fread(tempBuf, sizeof(char), nLen, fp);
			tempBuf[nLen] = '\0';

			pPiece->texname[i]   = tempBuf;
			pPiece->texnameID[i] = texID;

			delete [] tempBuf;			
		}

		// Read in face flag data
		if (version >= 0x0002)
		{
			int nFlags;

			fread(&nFlags, sizeof(int), 1, fp);
			pPiece->AllocFlags(nFlags);

			for(int cface = 0; cface < nFlags; cface++)
			{
				fread(&pPiece->flags[cface], sizeof(FlagType), 1, fp);
			}
		}
	}

	fclose(fp);

	GetWidthHeightLength(pPiece->mesh, pPiece->width, pPiece->height, pPiece->length);
	pPiece->scalar = NormalizeDimensions(pPiece->width, pPiece->height, pPiece->length);

	// Assign the name of the piece
	char buf[_MAX_FNAME];
	_splitpath(Filename, NULL, NULL, buf, NULL);
	pPiece->pieceName = buf;

	DEBUGLINE("PieceBrowserObj::LoadPiece End");
	return true;
}

Mesh* PieceBrowserObj::GetRenderMesh(TimeValue t, INode *inode, View& view, BOOL& needDelete)
{
	needDelete = FALSE;
	return &mesh;
}

void PieceBrowserObj::Deform(Deformer *defProc, int useSel)
{
	int nv = NumPoints();
	int i;
	if ( useSel ) 
	{
		BitArray sel = mesh.VertexTempSel();
		float *vssel = mesh.getVSelectionWeights ();
		if (vssel) {
			for (i=0; i<nv; i++)
			{
				if(sel[i]) 
				{
					SetPoint(i,defProc->Map(i,GetPoint(i)));
					continue;
				}

				if (vssel[i]==0) continue;

				Point3 & A = GetPoint(i);
				Point3 dir = defProc->Map(i,A) - A;
				SetPoint(i,A+vssel[i]*dir);
			}
		} 
		else
		{
			for (i=0; i<nv; i++) 
			{
				if (sel[i]) 
					SetPoint(i,defProc->Map(i,GetPoint(i)));
			}
		}
	} 
	else 
	{
		for (i=0; i<nv; i++) 
			SetPoint(i,defProc->Map(i,GetPoint(i)));
	}

	PointsWereChanged();	
}

HWND PieceBrowserObj::GetHWND()
{
	return dlgProc->GetHWND();
}

void PieceBrowserObj::GetDeformBBox(TimeValue t, Box3& box, Matrix3 *tm,BOOL useSel )
{
	{	
	if ( tm || useSel ) {
		box.Init();
		BitArray sel;
		float *vssel = NULL;
		if ( useSel ) {
			sel = mesh.VertexTempSel();
			vssel = mesh.getVSelectionWeights ();
			}
		for ( int i = 0; i < mesh.getNumVerts(); i++ ) {
			if ( !useSel || sel[i] || (vssel&&vssel[i])) {
				if ( tm ) {
					box += *tm * mesh.getVert(i);
				} else {
					box += mesh.getVert(i);
					}
				}
			}
	} else {
		box = mesh.getBoundingBox();
		}
	}
}

LRESULT CALLBACK GLWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_PAINT:
		{
			HDC hdc = GetDC(hwnd);
			RECT rect;

			GetWindowRect(hwnd,&rect);

			wglMakeCurrent(hdc, hglrc);

				glMatrixMode(GL_PROJECTION);
				glLoadIdentity();
				gluPerspective(30.0, 1.0, 1.0, 10.0);

				glViewport(0,
						   0,
						   rect.right - rect.left,
						   rect.bottom - rect.top);

				glClearColor(0.0f,1.0f,0.0f,1.0f);
				glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

				glColor3f(0.0f,1.0f,0.0f);

				glMatrixMode(GL_MODELVIEW);
				glBegin(GL_QUADS);

					glVertex3f(0.0f,0.0f,0.0f);
					glVertex3f(1.0f,0.0f,0.0f);
					glVertex3f(1.0f,1.0f,0.0f);
					glVertex3f(0.0f,1.0f,0.0f);

				glEnd();

				SwapBuffers(hdc);

			wglMakeCurrent(NULL,NULL);
		}
		return TRUE;

	case WM_CREATE:

		return TRUE;
	}

	return DefWindowProc(hwnd,msg,wParam,lParam);
}

bool PieceBrowserObj::CreateGLWindow()
{
	DEBUGLINE("PieceBrowserObj::CreateGLWindow Start");
	// Creates a preview window for the pieces and sets it up for OpenGL
	WNDCLASS wndclass;
	ZeroMemory(&wndclass,sizeof(WNDCLASS));

	wndclass.style         = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
	wndclass.lpfnWndProc   = GLWndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = NULL;
	wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
	wndclass.lpszMenuName  = NULL;
	wndclass.lpszClassName = "OpenGLContainer";

	RegisterClass(&wndclass);

	HWND hwndGLMain = CreateWindow("OpenGLContainer",
		                           "Part Preview",
								   WS_DLGFRAME | WS_VISIBLE,
								   0,
								   0,
								   640,
								   480,
								   gInterface->GetMAXHWnd(),
								   NULL,
								   hInstance,
								   NULL);

	ZeroMemory(&wndclass,sizeof(WNDCLASS));
	wndclass.style         = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
	wndclass.lpfnWndProc   = GLWndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = NULL;
	wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
	wndclass.lpszMenuName  = NULL;
	wndclass.lpszClassName = "OpenGLPreview";

	RegisterClass(&wndclass);

	HWND hwndGL = CreateWindow("OpenGLPreview",
		                       "OpenGL Preview Test",
							   WS_CLIPCHILDREN | WS_CLIPSIBLINGS | WS_CHILD | WS_VISIBLE,
							   0,
							   0,
							   100,
							   100,
							   hwndGLMain,
							   NULL,
							   hInstance,
							   NULL);

	PIXELFORMATDESCRIPTOR pfd;
	ZeroMemory(&pfd, sizeof(PIXELFORMATDESCRIPTOR));

	pfd.nSize    = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags  = PFD_DOUBLEBUFFER   |
		           PFD_SUPPORT_OPENGL |
				   PFD_DRAW_TO_WINDOW;

	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 24;
	pfd.cDepthBits = 32;
	pfd.iLayerType = PFD_MAIN_PLANE;


	HDC  hdc = GetDC(hwndGL);

	int nPixelFormat = ChoosePixelFormat(hdc, &pfd);
	
	if (nPixelFormat == 0)
		return false;

	if (!SetPixelFormat(hdc, nPixelFormat, &pfd))
		return false;

	hglrc = wglCreateContext(hdc);

	InvalidateRect(hwndGL,NULL,TRUE);
	return true;
}

IOResult PieceBrowserObj::Load(ILoad* iload)
{
	char* strTemp;

	while (iload->OpenChunk() == IO_OK)
	{
		switch(iload->CurChunkID())
		{
		case MODELNAME_CHUNK_ID:
			{
				if (iload->ReadCStringChunk(&strTemp) != IO_OK)
					return IO_ERROR;

				RetrieveComponent(strTemp);
			}
			break;
		}

		if (iload->CloseChunk() != IO_OK)
			return IO_ERROR;	
	} 
	
	return IO_OK;
}

IOResult PieceBrowserObj::Save(ISave* isave)
{
	CStr strModel = mlist->GetValue();

	isave->BeginChunk(MODELNAME_CHUNK_ID);
	
	if (isave->WriteCString(strModel) != IO_OK)
		return IO_ERROR;

	isave->EndChunk();

	return IO_OK;
}

void PieceBrowserObj::ApplyUVWMap(int type, 
								  float utile, 
								  float vtile,
								  float wtile,
								  int uflip, 
								  int vflip,
								  int wflip, 
								  int cap, 
								  const Matrix3& tm,
								  int channel)
{
	mesh.ApplyUVWMap(type,
					 utile, 
					 vtile,
					 wtile,
					 uflip,
					 vflip,
					 wflip,
					 cap,
					 tm);

	mesh.InvalidateTopologyCache();
}

///////////
void PieceBrowserObj::NotifyPreCollapse(INode* node, IDerivedObject* derObj,int index)
{

}

void PieceBrowserObj::NotifyPostCollapse(INode* node, Object* uobj, IDerivedObject* derObj,int index)
{
	/*
	Object* pobj = node->EvalWorldState(0).obj;

	Class_ID cid = pobj->ClassID();

	if (!pobj->CanConvertToType(triObjectClassID))
		return;

	TriObject* obj = (TriObject*)pobj->ConvertToType(0,triObjectClassID);
	if (!obj)
		return;

	// Now that the stack is collapsed, we can reassign the face flag data that we stored
	Mesh& meshSrc = obj->GetMesh();

	IFaceDataMgr* pFDMgrSrc = static_cast<IFaceDataMgr*>(meshSrc.GetInterface( FACEDATAMGR_INTERFACE ));

	if ( pFDMgrSrc )
	{
		FaceFlagsData* fdcSrc = dynamic_cast<FaceFlagsData*>( pFDMgrSrc->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));

		if ( fdcSrc == NULL )
		{
			// The mesh does not have our face-data channel so we will add it here
			fdcSrc = new FaceFlagsData();
			pFDMgrSrc->AddFaceDataChan( fdcSrc );			
		}

		fdcSrc->FacesCreated(0, meshSrc.numFaces);

		// Copy the face flags maintained within the piece database to the actual face flag channel
		for(int i = 0; i < piece.nFlagTypes; i++)
		{
			fdcSrc->m_data[i] = piece.flags[i];
		}
	}
	*/
}

