/*
	PartPreview.cpp
	This is a piece preview dialog that allows up to 9 pieces to be displayed
	in 3D at once
*/

#include "PartPreview.h"
#include "PieceBrowser.h"
#include "Resource.h"
#include "../image/image.h"
#include "imagedata.h"
#include <gl/gl.h>
#include <gl/glu.h>

#define WM_UPDATE WM_USER + 0x001

extern HINSTANCE hInstance;

int PreviewWindowText[NUM_PREVIEW_WINDOWS] = { IDC_PREVIEWT1,
											   IDC_PREVIEWT2,
											   IDC_PREVIEWT3,
											   IDC_PREVIEWT4,
											   IDC_PREVIEWT5,
											   IDC_PREVIEWT6,
											   IDC_PREVIEWT7,
											   IDC_PREVIEWT8,
											   IDC_PREVIEWT9,
											 };

PartPreviewDlg::PartPreviewDlg(HINSTANCE hInstance, HWND hwndParent, PieceBrowserObj* pBrowser) :
	MSDlgWindow(hInstance,MAKEINTRESOURCE(IDD_PARTPREVIEW),hwndParent,false)
{
	pPieceBrowser     = pBrowser;
	rendererType      = pBrowser->GetRendererType();
	bActive           = false;
	numActiveWindows  = 0;

	GetConfig();
	CreatePreviewWindows();

	HWND  hwndScroll = GetDlgItem(hwnd,IDC_SCROLL);
	this->hwndParent = hwndParent;

	SetScrollRange(hwndScroll,SB_CTL,0,pBrowser->partList.GetSize() - NUM_PREVIEW_WINDOWS - 1,TRUE);
	
	Init();
}

PartPreviewDlg::PartPreviewDlg(HINSTANCE hInstance, HWND hwndParent) :
	MSDlgWindow(hInstance, MAKEINTRESOURCE(IDD_PARTPREVIEW), hwndParent, false)
{
	pPieceBrowser    = NULL;
	rendererType     = RENDERER_DIRECT3D;
	bActive          = false;
	numActiveWindows = 0;

	GetConfig();
	CreatePreviewWindows();

	this->hwndParent = hwndParent;
}

void PartPreviewDlg::GetConfig()
{
	CStr appDir=gInterface->GetDir(APP_PLUGCFG_DIR);
	appDir+="\\PieceBrowser.ini";

	char strRenderer[256];
	char strPath[256];

	GetPrivateProfileString("Parts","PartDir",".",strPath,PATH_SIZE,appDir);
	GetPrivateProfileString("Parts","Renderer","OpenGL",strRenderer, 255, appDir);

	if (strcmp(strRenderer,"OpenGL") == 0)
		rendererType = RENDERER_OPENGL;
	else
	if (strcmp(strRenderer,"Direct3D") == 0)
		rendererType = RENDERER_DIRECT3D;
	else
		rendererType = RENDERER_OPENGL;		// If type is unrecognized use OpenGL Renderer
}

void PartPreviewDlg::Init(PieceBrowserObj* obj)
{
	pPieceBrowser = obj;

	if (obj)
	{
		if (bActive)
			Show();
	}
	else
		Hide();
}

void PartPreviewDlg::Init()
{
	if (!pPieceBrowser)
		return;

	if (!bActive)
	{
		lastTexID         = 1;			// Last assigned texture ID for OpenGL
		selPreview        = -1;			// No piece is currently selected
		activeD3DRenderer = NULL;

		InitContexts();
		LoadMeshes();
		bActive = true;

		HWND  hwndScroll = GetDlgItem(hwnd, IDC_SCROLL);
		SetScrollRange(hwndScroll,SB_CTL,0,pPieceBrowser->partList.GetSize() - NUM_PREVIEW_WINDOWS - 1,TRUE);
	}
	else
	{
		DestroyContexts();
		bActive = false;
		Init();
	}
}

PartPreviewDlg::~PartPreviewDlg()
{
	if (bActive)
	{
		DestroyContexts();
	}
}

void PartPreviewDlg::DeletePreviewTextures(int context)
{
	// Delete any previously uploaded textures for the last mesh using
	// this GL context so we don't run out of V-RAM
	if (rendererType == RENDERER_OPENGL)
		DeletePreviewTexturesGL(context);
	else
	if (rendererType == RENDERER_DIRECT3D)
		DeletePreviewTexturesD3D(context);
}

void PartPreviewDlg::DeletePreviewTexturesGL(int context)
{
	if (meshPreviewWindow[context].nTextures > 0)
	{
		HDC hdc = GetDC(hwndPreviewWindow[context]);

		wglMakeCurrent(hdc,glrcPreviewWindow[context]);

		unsigned int* deltexnames = new unsigned int[meshPreviewWindow[context].nTextures];
		
		for(int i = 0; i < meshPreviewWindow[context].nTextures; i++)
			deltexnames[i] = meshPreviewWindow[context].texnameIDGL[i];
		
		glDeleteTextures(meshPreviewWindow[context].nTextures, (const unsigned int*)deltexnames);

		delete [] deltexnames;

		wglMakeCurrent(NULL,NULL);
		ReleaseDC(hwndPreviewWindow[context], hdc);
	}
}

void PartPreviewDlg::DeletePreviewTexturesD3D(int context)
{
	d3dPreviewWindow[context].FreeAll();
}

void PartPreviewDlg::LoadMeshes(int start)
{
	if (!pPieceBrowser)
		return;

	Link<CStr>* curpart = pPieceBrowser->partList.GetLink(start);
	int curmesh = 0;

	while(curpart && curmesh != NUM_PREVIEW_WINDOWS)
	{
		DeletePreviewTextures(curmesh);

		pPieceBrowser->LoadPiece(&meshPreviewWindow[curmesh], curpart->data);
		//pPieceBrowser->LoadPiece(&meshPreviewWindow[curmesh], "C:\\NSPPartTest\\halfpipe.nsp");

		//meshPreviewWindow[curmesh].mesh.setNumFaces(0);

		//pPieceBrowser->ScaleMesh(meshPreviewWindow[curmesh].mesh,200.0f,100.0f,100.0f);
		pPieceBrowser->ScaleMesh(meshPreviewWindow[curmesh].mesh,1.0f,1.0f,1.0f);

		if (rendererType == RENDERER_DIRECT3D)
			meshPreviewWindow[curmesh].BuildD3DMesh(d3dPreviewWindow[curmesh].GetNumDisplays());


		// Fill in the preview filename
		char* filename = strrchr(curpart->data,'\\');

		if (!filename)
			filename = (char*)curpart->data;
		else
			filename++;

		SetDlgItemText(hwnd,PreviewWindowText[curmesh],filename);

		// Upload textures used by the model
		if (rendererType == RENDERER_OPENGL)
		{
			for(int curtex = 0; curtex < meshPreviewWindow[curmesh].nTextures; curtex++)
			{
				meshPreviewWindow[curmesh].texnameIDGL[curtex] = UploadTexture(meshPreviewWindow[curmesh].texname[curtex], curmesh);
				//meshPreviewWindow[curmesh].texnameIDGL[curtex] = 0;
			}

			// Now that we've loaded the textures we need to fill out the texID array to correspond
			// to the OpenGL texture object name for each face of the mesh
			meshPreviewWindow[curmesh].AssignTexIDs();
		}

		if (rendererType == RENDERER_DIRECT3D)
		{
			d3dPreviewWindow[curmesh].UploadMesh(&meshPreviewWindow[curmesh].D3Dmesh);
		}

		curmesh++;
		curpart = curpart->next;
	}

	numActiveWindows = curmesh;
}

void PartPreviewDlg::ClearPieceSelection()
{
	selPreview = -1;
}

void PartPreviewDlg::UpdateScroll(HWND hwndScroll, int Code, int Pos)
{
	int curPos = GetScrollPos(hwndScroll, SB_CTL);
	int max = pPieceBrowser->partList.GetSize() - NUM_PREVIEW_WINDOWS;

	switch(Code)
	{
	case SB_PAGELEFT:
		curPos -= NUM_PREVIEW_WINDOWS;
		if (curPos < 0)
			curPos = 0;

		lastTexID = 1;
		LoadMeshes(curPos);
		ClearPieceSelection();
		break;

	case SB_LINELEFT:
	case SB_LEFT:
		curPos--;
		if (curPos < 0)
			curPos = 0;

		lastTexID = 1;
		LoadMeshes(curPos);
		ClearPieceSelection();
		break;

	case SB_PAGERIGHT:
		curPos += NUM_PREVIEW_WINDOWS;
		if (curPos > max)
			curPos = max;

		lastTexID = 1;
		LoadMeshes(curPos);
		ClearPieceSelection();
		break;

	case SB_LINERIGHT:
	case SB_RIGHT:
		curPos++;
		
		if (curPos > max)
			curPos = max;

		lastTexID = 1;
		LoadMeshes(curPos);
		ClearPieceSelection();
		break;

	case SB_THUMBPOSITION:
		curPos = Pos;

		lastTexID = 1;
		LoadMeshes(curPos);
		ClearPieceSelection();
		break;

	default:
		return;
	}

	SetScrollPos(hwndScroll,SB_CTL,curPos,TRUE);
}

void PartPreviewDlg::Shutdown()
{
	if (rendererType == RENDERER_DIRECT3D)
		DestroyContextsD3D();
	else
	if (rendererType == RENDERER_OPENGL)
		DestroyContextsGL();

	bActive = false;
}

BOOL PartPreviewDlg::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_SHOWWINDOW:
		if (wParam == TRUE)
			SetTimer(hwnd,1,25,NULL);
		else
			KillTimer(hwnd,1);

		return TRUE;

	case WM_ACTIVATE:
		if (wParam == WA_ACTIVE || wParam == WA_CLICKACTIVE)
			SetTimer(hwnd,1,25,NULL);
		else
			;//KillTimer(hwnd,1);

		return TRUE;

	case WM_CLOSE:
		KillTimer(hwnd,1);
		Shutdown();
		Hide();
		return TRUE;

	case WM_TIMER:
		if (wParam == 1)
			UpdatePreviews();

		return FALSE;

	case WM_HSCROLL:
		if (lParam)
		{
			UpdateScroll((HWND)lParam,LOWORD(wParam),HIWORD(wParam));
		}
		return FALSE;

	case WM_COMMAND:

		return FALSE;
	};

	return FALSE;
}

void PartPreviewDlg::CreatePreviewWindows()
{
	WNDCLASS wndclass;

	// Register a window class that is suitable for OpenGL
	ZeroMemory(&wndclass,sizeof(WNDCLASS));
	wndclass.style         = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
	wndclass.lpfnWndProc   = PreviewGLWndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = NULL;
	wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
	wndclass.lpszMenuName  = NULL;
	wndclass.lpszClassName = "OpenGLPreview";

	RegisterClass(&wndclass);

	// Construct all the preview windows (9)
	for(int i = 0; i < NUM_PREVIEW_WINDOWS; i++)
	{
		HWND hwndGL = CreateWindow("OpenGLPreview",
								   "",
								   WS_CLIPCHILDREN | WS_CLIPSIBLINGS | WS_CHILD | WS_VISIBLE | WS_BORDER,
								   0 + i * 105,
								   0,
								   100,
								   100,
								   hwnd,
								   (HMENU)i,
								   hInstance,
								   NULL);

		hwndPreviewWindow[i] = hwndGL;

		SetWindowLong(hwndGL,GWL_USERDATA,(LONG)this);
	}
}

bool PartPreviewDlg::InitContexts()
{
	if (rendererType == RENDERER_OPENGL)
	{
		PIXELFORMATDESCRIPTOR pfd;
		ZeroMemory(&pfd, sizeof(PIXELFORMATDESCRIPTOR));

		pfd.nSize    = sizeof(PIXELFORMATDESCRIPTOR);
		pfd.nVersion = 1;
		pfd.dwFlags  = PFD_DOUBLEBUFFER   |
					   PFD_SUPPORT_OPENGL |
					   PFD_DRAW_TO_WINDOW |
					   PFD_GENERIC_FORMAT;	// No hardware acceleration

		pfd.iPixelType = PFD_TYPE_RGBA;
		pfd.cColorBits = 24;
		pfd.cDepthBits = 32;
		pfd.iLayerType = PFD_MAIN_PLANE;

		// Initialize OpenGL to perform the piece preview renders
		// We need a context for each preview window
		for(int i = 0; i < NUM_PREVIEW_WINDOWS; i++)
		{
			HDC  hdc = GetDC(hwndPreviewWindow[i]);

			int nPixelFormat = ChoosePixelFormat(hdc, &pfd);
			
			if (nPixelFormat == 0)
				return false;

			if (!SetPixelFormat(hdc, nPixelFormat, &pfd))
				return false;

			glrcPreviewWindow[i] = wglCreateContext(hdc);

			wglMakeCurrent(hdc,glrcPreviewWindow[i]);
			const unsigned char* strRenderer = glGetString(GL_RENDERER);

			wglMakeCurrent(NULL,NULL);

			ReleaseDC(hwndPreviewWindow[i],hdc);
		}
	}
	else if (rendererType == RENDERER_DIRECT3D)
	{
		for(int i = 0; i < NUM_PREVIEW_WINDOWS; i++)
		{
			d3dPreviewWindow[i].InitD3D(hwndPreviewWindow[i]);
		}
	}

	return true;
}

void PartPreviewDlg::DestroyContextsGL()
{
	for(int i = 0; i < NUM_PREVIEW_WINDOWS; i++)
	{
		wglDeleteContext(glrcPreviewWindow[i]);
	}
}

void PartPreviewDlg::DestroyContextsD3D()
{
	for(int i = 0; i < NUM_PREVIEW_WINDOWS; i++)
	{
		d3dPreviewWindow[i].ShutdownD3D();
	}
}

void PartPreviewDlg::DestroyContexts()
{
	if (rendererType == RENDERER_OPENGL)
		DestroyContextsGL();
	else
	if (rendererType == RENDERER_DIRECT3D)
		DestroyContextsD3D();
}

void PartPreviewDlg::GLUpdatePreview(int i)
{
	//InvalidateRect(hwndPreviewWindow[i],NULL,FALSE);
	//SendMessage(hwndPreviewWindow[i],WM_UPDATE,0,0);
	HDC  hdc = GetDC(hwndPreviewWindow[i]);
	RECT rect;

	GetWindowRect(hwndPreviewWindow[i], &rect);

	wglMakeCurrent(hdc, glrcPreviewWindow[i]);

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_TEXTURE_2D);

	glEnable(GL_BLEND);
	glDepthFunc(GL_LESS);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(30.0, 1.0, 1.0, 5000.0);
	gluLookAt(0.0f,1.0f,-3.0f,
			  0.0f,0.2f,1.0f,
			  0.0f,1.0f,0.0f);

	glViewport(0,
			   0,
			   rect.right - rect.left,
			   rect.bottom - rect.top);

	glColor3f(1.0f,1.0f,1.0f);

	if (selPreview == i)
	{
		glClearColor(0.4f,0.4f,0.4f,1.0f);
	}
	else
	{
		glClearColor(0.0f,0.0f,0.0f,1.0f);
	}

	glShadeModel(GL_FLAT);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	//glColor4f(0.0f,0.5f,0.0f,0.5f);

	glMatrixMode(GL_MODELVIEW);
	//glLoadIdentity();

	glRotatef(0.5f,0.0f,1.0f,0.0f);

	DrawPiece(&meshPreviewWindow[i]);

	/*  TODO: Need to add rotation matrix code for blend so we can rotate the geometry
		seperate from the red quad for the highlight filter, or maybe I'll just leave it as clear color */
	/*
	if (Highlight)
	{
		//glDisable(GL_TEXTURE_2D);

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		gluOrtho2D(0.0f,1.0f,1.0f,0.0f);

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		glDisable(GL_TEXTURE_2D);
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);

		glBegin(GL_QUADS);

			glColor4f(1.0f,0.0f,0.0f,0.25f);
			glVertex2f(0.0f, 0.0f);
			glVertex2f(1.0f, 0.0f);
			glVertex2f(1.0f, 1.0f);
			glVertex2f(0.0f, 1.0f);

		glEnd();
	}
	else
	{

	}
	*/

	SwapBuffers(hdc);

	wglMakeCurrent(NULL,NULL);

	ReleaseDC(hwndPreviewWindow[i],hdc);
}

void PartPreviewDlg::D3DUpdatePreview(int i)
{
	activeD3DRenderer = &(d3dPreviewWindow[i]);

	if (selPreview == i)
		activeD3DRenderer->SetClearColor(0.4f, 0.4f, 0.4f);
	else
		activeD3DRenderer->SetClearColor(0.0f, 0.0f, 0.0f);

	if (i == 0)
	{
		int zzz;
		zzz = 0;
	}

	d3dPreviewWindow[i].Render(&meshPreviewWindow[i].D3Dmesh);
}

void PartPreviewDlg::UpdatePreviews()
{
	int i;

	if (rendererType == RENDERER_OPENGL)
	{
		//for(i = 0; i < NUM_PREVIEW_WINDOWS; i++)
		for(i = 0; i < numActiveWindows; i++)
			GLUpdatePreview(i);
	}
	else
	if (rendererType == RENDERER_DIRECT3D)
	{
		//for(i = 0; i < NUM_PREVIEW_WINDOWS; i++)
		for(i = 0; i < numActiveWindows; i++)
			D3DUpdatePreview(i);
	}
}

void PartPreviewDlg::DrawPieceGL(NSPiece* piece)
{
	// No optimizations for now
	Point3* verts = piece->mesh.verts;
	Face*   faces = piece->mesh.faces;
	TVFace* tvFaces = piece->mesh.tvFace;
	UVVert* tVerts = piece->mesh.tVerts;

	glMatrixMode(GL_MODELVIEW);

	//glDisable(GL_TEXTURE_2D);
	//glColor3f(1.0f,1.0f,1.0f);

//	if (piece->texID)
	//glBindTexture(GL_TEXTURE_2D,1);

	int lastID = -2;

	for(int i = 0; i < piece->mesh.numFaces; i++)
	{
		if (lastID != piece->texID[i])
		{
			glBindTexture(GL_TEXTURE_2D,piece->texID[i]);
			lastID = piece->texID[i];
		}

		glBegin(GL_TRIANGLES);

		glTexCoord2f(-tVerts[tvFaces[i].t[0]].x,
				     -tVerts[tvFaces[i].t[0]].y);

		glVertex3f(verts[faces[i].v[0]].x,
				   verts[faces[i].v[0]].z,
				   verts[faces[i].v[0]].y);

		glTexCoord2f(-tVerts[tvFaces[i].t[1]].x,
				     -tVerts[tvFaces[i].t[1]].y);

		glVertex3f(verts[faces[i].v[1]].x,
				   verts[faces[i].v[1]].z,
				   verts[faces[i].v[1]].y);

		glTexCoord2f(-tVerts[tvFaces[i].t[2]].x,
				     -tVerts[tvFaces[i].t[2]].y);

		glVertex3f(verts[faces[i].v[2]].x,
				   verts[faces[i].v[2]].z,
				   verts[faces[i].v[2]].y);

		glEnd();
	}
}

void PartPreviewDlg::DrawPieceD3D(NSPiece* piece)
{
	activeD3DRenderer->Render(&piece->D3Dmesh);
}

void PartPreviewDlg::DrawPiece(NSPiece* piece)
{
	if (rendererType == RENDERER_OPENGL)
		DrawPieceGL(piece);
	else
	if (rendererType == RENDERER_DIRECT3D)
		DrawPieceD3D(piece);
}

LRESULT PartPreviewDlg::PreviewGLWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_LBUTTONDOWN:
		{
			PartPreviewDlg* pthis = (PartPreviewDlg*)GetWindowLong(hwnd,GWL_USERDATA);

			if (pthis && pthis->pPieceBrowser)
			{
				int   ID = GetWindowLong(hwnd,GWL_ID);
				
				char pieceText[256];

				HWND hwndPBrowser = pthis->pPieceBrowser->GetHWND();

				// Update piece selection
				GetWindowText(GetDlgItem(pthis->hwnd,PreviewWindowText[ID]), pieceText, 255);
				int sel = SendDlgItemMessage(hwndPBrowser, IDC_PARTLIST, LB_FINDSTRING, (WPARAM)-1, (LPARAM)pieceText);
				SendDlgItemMessage(hwndPBrowser, IDC_PARTLIST, LB_SETCURSEL, (WPARAM)sel, 0);
				
				// Notify the parent window that the selection has been updated
				SendMessage(hwndPBrowser, WM_COMMAND, MAKEWPARAM(IDC_PARTLIST,LBN_SELCHANGE), 0);

				pthis->selPreview = ID;
			}
		}
		return TRUE;

	case WM_CREATE:

		return TRUE;
	}

	return DefWindowProc(hwnd,msg,wParam,lParam);
}

int PartPreviewDlg::UploadTexture(char* filename, int contextID)
{
	// We'll use my ImageData class from MipGen since it can abstract the format
	// into whatever bpp and palette into whatever is required (in this case 32-bit RGBA)

	ImageData img;
	ImageData img32;
	img.ReadPNG(filename);

	img.Convert32(&img32);

	int ID = -1;

	int width  = img32.width;
	int height = img32.height;

	HDC hdc = GetDC(hwndPreviewWindow[contextID]);

	wglMakeCurrent(hdc, glrcPreviewWindow[contextID]);

	glBindTexture(GL_TEXTURE_2D, lastTexID);

	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
	//glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);

	glPixelStorei(GL_UNPACK_ROW_LENGTH,0);
	glPixelStorei(GL_UNPACK_SKIP_ROWS,0);
	glPixelStorei(GL_UNPACK_SKIP_PIXELS,0);

	glPixelStorei(GL_UNPACK_ALIGNMENT,1);

	glTexImage2D(GL_TEXTURE_2D,
					0,					// Mip level 0
					4,					// Number of color components RBGA (4)
					width,
					height,
					0,					// No border
					GL_RGBA,
					//GL_BGRA_EXT,
					GL_UNSIGNED_BYTE,
					img32.buf);

	ID = lastTexID;
	lastTexID++;

	//glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

	wglMakeCurrent(NULL, NULL);

	ReleaseDC(hwndPreviewWindow[contextID],hdc);

	return ID;
}
