/**********************************************************************
 *<
	FILE: sample.cpp

	DESCRIPTION:  Sample implementation

	HISTORY: created November 11 1994

 *>	Copyright (c) 1994, All Rights Reserved.
 **********************************************************************/

#include <next.h>
#include "NExtParticle.h"
#include <props.h>
#include "dummy.h"
#include "Simpobj.h"
#include "notify.h"
#include "../PropEdit/PropEdit.h"

#define NONPLACEABLE_METACMD  "[NonPlaceable]"

#define FUNCSTART(x)
#define FUNCEND(x)

bool NodeExistsInList(INodeTab& list, INode* node);

#define FUNCSTART(x) \
	{ \
	char buf[256]; \
	sprintf(buf, "Entered %s (%i)\n", x, __LINE__); \
	OutputDebugString(buf); \
	}

#define FUNCEND(x) \
	{ \
	char buf[256]; \
	sprintf(buf, "Exited %s (%i)\n", x, __LINE__); \
	OutputDebugString(buf); \
	}


extern HINSTANCE hInstance;
extern Interface* gInterface;
extern  PropEditor* pPropEdit;
extern GUP*         gpGUP;

bool gbLockPosChange = false;
bool gbLockPMU = false;

static bool bParticleDelLock = false;

enum
{
	PM_SUB_PB,
	PM_SUB_MID,
	PM_SUB_END,
	PM_SUB_MID_ROT,
	PM_SUB_END_ROT,
	vNUM_PM_SUBS
};

enum
{
	PB_LIFETIME,
	PB_MIDTIME_PCT,
	PB_ACCEL_X,
	PB_ACCEL_Y,
	PB_ACCEL_Z,
	PB_VEL_X,
	PB_VEL_Y,
	PB_VEL_Z
};

char	ParticleProperties[vNUM_PROPERTIES][256] = 
{
	"Emission_Values/Lifetime",
	"Emission_Values/MidPointPCT",
	"Emission_Values/Acceleration_X",
	"Emission_Values/InitialVel_X",
	"Emission_Values/Acceleration_Y",
	"Emission_Values/InitialVel_Y",
	"Emission_Values/Acceleration_Z",
	"Emission_Values/InitialVel_Z",	
	"Emission_Values/UseMidPoint"	
};

//----------------------------------------------------------------------

class ParticleMaster: public ReferenceTarget, public IParticleMaster 
{
	public:
	// Object parameters

		IParamBlock *pblock;
		INode* masterNode;
		IndePosition* ctrl_mid, *ctrl_end;
		LocalEulerRotation* ctrl_mid_rot, *ctrl_end_rot;
		bool ignore_controller_updates;
		int  numNodes;
		ParticleMaster();
		~ParticleMaster();
		
		void RecalculatePositions( TimeValue t );
		void RecalculateParameters( TimeValue t, Interval& ivalid );

		INode* GetMasterNode() { return masterNode; }
		void SetMasterNode( INode* node ) { masterNode = node; }
		void SetLifetime(TimeValue t, float r, bool update_props, bool update_controls=true );
		void SetMidtimePCT(TimeValue t, float r, bool update_props, bool update_controls=true );		
		void SetAccelerationX(TimeValue t, float r, bool update_props, bool update_controls=true );
		void SetAccelerationY(TimeValue t, float r, bool update_props, bool update_controls=true );
		void SetAccelerationZ(TimeValue t, float r, bool update_props, bool update_controls=true );
		void SetVelocityX(TimeValue t, float r, bool update_props, bool update_controls=true );
		void SetVelocityY(TimeValue t, float r, bool update_props, bool update_controls=true );
		void SetVelocityZ(TimeValue t, float r, bool update_props, bool update_controls=true );
		void SetType( char* type );

		int GetNum(TimeValue t, Interval& valid = Interval(0,0) ); 			
		float GetLifetime(TimeValue t, Interval& valid = Interval(0,0) ); 	
		float GetMidtimePCT(TimeValue t, Interval& valid = Interval(0,0) ); 	
		float GetAccelerationX(TimeValue t, Interval& valid = Interval(0,0) ); 	
		float GetAccelerationY(TimeValue t, Interval& valid = Interval(0,0) ); 	
		float GetAccelerationZ(TimeValue t, Interval& valid = Interval(0,0) ); 	
		float GetVelocityX(TimeValue t, Interval& valid = Interval(0,0) ); 	
		float GetVelocityY(TimeValue t, Interval& valid = Interval(0,0) ); 	
		float GetVelocityZ(TimeValue t, Interval& valid = Interval(0,0) ); 	

		void UpdateNodeProperties( bool update_controls );

		void UpdateUI(TimeValue t);

		// Class vars
		static HWND hMasterParams;
		static IObjParam *iObjParams;
		static float dlgLifetime;
		static float dlgMidtimePCT;
		static Point3 dlgAccel;
		static Point3 dlgVel;
		static CStr	dlgType;
		static ISpinnerControl *lifeSpin;
		static ISpinnerControl *midtimePCTSpin;
		static ISpinnerControl *accelXSpin;
		static ISpinnerControl *accelYSpin;
		static ISpinnerControl *accelZSpin;
		static ISpinnerControl *velXSpin;
		static ISpinnerControl *velYSpin;
		static ISpinnerControl *velZSpin;		
		
		// From Animatable
		int NumSubs()  { return vNUM_PM_SUBS; }
		Animatable* SubAnim(int i);
		TSTR SubAnimName(int i);

		Class_ID ClassID() { return PARTICLEMASTER_CLASS_ID; }  
		SClass_ID SuperClassID() { return SYSTEM_CLASS_ID; }  
		void GetClassName(TSTR& s) { s = "NExt Particle";}
		void DeleteThis() { delete this; }		
		void BeginEditParams( IObjParam *ip, ULONG flags,Animatable *prev );
		void EndEditParams( IObjParam *ip, ULONG flags,Animatable *next );
		
		// From Reference Target
		RefTargetHandle Clone(RemapDir& remap = NoRemap());
		int NumRefs() { return vNUM_PM_REFS; };
		RefTargetHandle GetReference(int i);
		void SetReference(int i, RefTargetHandle rtarg);

		// IO
		IOResult Save(ISave *isave);
		IOResult Load(ILoad *iload);

		RefResult NotifyRefChanged(Interval, RefTargetHandle, PartID&, RefMessage);

		void AssignParticlePosProperties();
		void UpdateRefs();
};

//------------------------------------------------------

class ParticleMasterClassDesc:public ClassDesc {
	public:
	int 			IsPublic() { return 1; }
	void *			Create(BOOL loading = FALSE) { return new ParticleMaster(); }
	const TCHAR *	ClassName() { return "NExt Particle"; }
	int 			BeginCreate(Interface *i);
	int 			EndCreate(Interface *i);
	SClass_ID		SuperClassID() { return SYSTEM_CLASS_ID; }
	Class_ID		ClassID() { return PARTICLEMASTER_CLASS_ID; }
	const TCHAR* 	Category() { return _T("");  }
	};
static ParticleMasterClassDesc mcDesc;

ClassDesc* GetNExtParticleDesc() { return &mcDesc; }
//------------------------------------------------------

HWND ParticleMaster::hMasterParams = NULL;
IObjParam *ParticleMaster::iObjParams;

float ParticleMaster::dlgLifetime = 2.0f;
float ParticleMaster::dlgMidtimePCT = 50.0f;
Point3 ParticleMaster::dlgAccel	= Point3( 0.0f, -16.0f, 0.0f );
Point3 ParticleMaster::dlgVel	= Point3( 20.0f, 0.0f, 0.0f );
CStr ParticleMaster::dlgType = "Default";

ISpinnerControl *ParticleMaster::lifeSpin;
ISpinnerControl *ParticleMaster::midtimePCTSpin;
ISpinnerControl *ParticleMaster::accelXSpin;
ISpinnerControl *ParticleMaster::accelYSpin;
ISpinnerControl *ParticleMaster::accelZSpin;
ISpinnerControl *ParticleMaster::velXSpin;
ISpinnerControl *ParticleMaster::velYSpin;
ISpinnerControl *ParticleMaster::velZSpin;


ParticleMaster::ParticleMaster() {
	FUNCSTART("ParticleMaster::ParticleMaster");
	ParamBlockDesc desc[] = {
		{ TYPE_FLOAT, NULL, TRUE },	// Lifetime
		{ TYPE_FLOAT, NULL, TRUE },	// MidtimePCT
		{ TYPE_FLOAT, NULL, TRUE },	// AccelerationX
		{ TYPE_FLOAT, NULL, TRUE },	// AccelerationY
		{ TYPE_FLOAT, NULL, TRUE },	// AccelerationZ
		{ TYPE_FLOAT, NULL, TRUE },	// VelocityX
		{ TYPE_FLOAT, NULL, TRUE },	// VelocityY
		{ TYPE_FLOAT, NULL, TRUE }	// VelocityZ
		};

	ctrl_mid = NULL;
	ctrl_end = NULL;
	ctrl_mid_rot = NULL;
	ctrl_end_rot = NULL;
	ignore_controller_updates = false;

	MakeRefByID( FOREVER, 0, CreateParameterBlock( desc, 8 ) );	
	
	UpdateNodeProperties( false );
	numNodes = 2;
	masterNode = NULL;

	FUNCEND("ParticleMaster::ParticleMaster");
}

RefTargetHandle ParticleMaster::Clone(RemapDir& remap) {
	FUNCSTART("ParticleMaster::Clone");

	ParticleMaster* newm = new ParticleMaster();	
	newm->ReplaceReference(0,pblock->Clone(remap));
	newm->numNodes = numNodes;
	//remap.PatchPointer((RefTargetHandle*) &newm->ctrl_mid, (RefTargetHandle) ctrl_mid );
	newm->ctrl_mid = (IndePosition *) remap.CloneRef( ctrl_mid );
	newm->ctrl_end = (IndePosition *) remap.CloneRef( ctrl_end );
	if( ctrl_mid_rot )
	{
		newm->ctrl_mid_rot = (LocalEulerRotation *) remap.CloneRef( ctrl_mid_rot );
	}
	if( ctrl_end_rot )
	{
		newm->ctrl_end_rot = (LocalEulerRotation *) remap.CloneRef( ctrl_end_rot );
	}
	BaseClone(this, newm, remap);

	FUNCEND("ParticleMaster::Clone");
	return(newm);
}

ParticleMaster::~ParticleMaster() 
{
	FUNCSTART("ParticleMaster::~ParticleMaster");
	FUNCEND("ParticleMaster::~ParticleMaster");
}

void ParticleMaster::UpdateRefs()
{
	RecalculatePositions(0);
	NotifyDependents(FOREVER, PART_OBJ, REFMSG_CHANGE);	
}

void ParticleMaster::AssignParticlePosProperties()
{
	//INode* masterNode;
	//IndePosition* ctrl_mid, *ctrl_end;

	if (gbLockPosChange)
		return;

	Object* obj = masterNode->EvalWorldState(0).obj;

	if (obj->ClassID() == PARTICLE_BOX_CLASS_ID)
	{
		IParticleBox* pbox   = dynamic_cast<IParticleBox*>(obj);
		INode*        node   = pbox->GetObjectNode();
		IParticleBox* midBox = pbox->GetMidBox();
		IParticleBox* endBox = pbox->GetEndBox();

		if (!midBox || !endBox)
			return;

		INode* midBoxNode    = midBox->GetObjectNode();
		INode* endBoxNode    = endBox->GetObjectNode();

		if (!midBoxNode || !endBoxNode)
			return;

		Point3 posMidBox     = midBoxNode->GetNodeTM(0).GetTrans() - node->GetNodeTM(0).GetTrans();
		Point3 posEndBox     = endBoxNode->GetNodeTM(0).GetTrans() - node->GetNodeTM(0).GetTrans();

		Control* scale_ctrl;
		ScaleValue scale;
		Point3 midScale, endScale;

		scale_ctrl = midBoxNode->GetTMController()->GetScaleController();
		scale_ctrl->GetValue( 0, &scale, FOREVER );
		midScale = scale.s;

		scale_ctrl = endBoxNode->GetTMController()->GetScaleController();
		scale_ctrl->GetValue( 0, &scale, FOREVER );
		endScale = scale.s;

		SetParticleOrientation(node, posMidBox, posEndBox, midBox->GetWidth(),
														   midBox->GetHeight(),
														   midBox->GetLength(),
														   endBox->GetWidth(),
														   endBox->GetHeight(),
														   endBox->GetLength(),
														   midScale,
														   endScale);
	}
}

void ParticleMaster::UpdateUI(TimeValue t)
{
	FUNCSTART("ParticleMaster::UpdateUI");
	if ( hMasterParams ) 
	{
		lifeSpin->SetValue( GetLifetime(t), FALSE );
		midtimePCTSpin->SetValue( GetMidtimePCT(t), FALSE );
		accelXSpin->SetValue( GetAccelerationX(t), FALSE );
		accelYSpin->SetValue( GetAccelerationY(t), FALSE );
		accelZSpin->SetValue( GetAccelerationZ(t), FALSE );
		velXSpin->SetValue( GetVelocityX(t), FALSE );
		velYSpin->SetValue( GetVelocityY(t), FALSE );
		velZSpin->SetValue( GetVelocityZ(t), FALSE );
	
	}
	FUNCEND("ParticleMaster::UpdateUI");
}

TSTR ParticleMaster::SubAnimName( int i )
{
	FUNCSTART("ParticleMaster::SubAnimName");
	switch( i )
	{
		case PM_SUB_PB:
			{
				FUNCEND("ParticleMaster::SubAnimName");
				return GetString(IDS_DS_RINGARRAYPAR);
			}
		case PM_SUB_MID:
			{
				FUNCEND("ParticleMaster::SubAnimName");
				return "Mid Controller";
			}
		case PM_SUB_END:
			{
				FUNCEND("ParticleMaster::SubAnimName");
				return "End Controller";
			}
		case PM_SUB_MID_ROT:
			{
				FUNCEND("ParticleMaster::SubAnimName");
				return "Mid Rotation Controller";
			}
		case PM_SUB_END_ROT:
			{
				FUNCEND("ParticleMaster::SubAnimName");
				return "End Rotation Controller";
			}
		default:
			{
				FUNCEND("ParticleMaster::SubAnimName");
				return "";
			}
	}
	
		{
			FUNCEND("ParticleMaster::SubAnimName");
		return "";	
		}
	FUNCEND("ParticleMaster::SubAnimName");
}

Animatable* ParticleMaster::SubAnim( int i )
{
	FUNCSTART("ParticleMaster::SubAnim");
	switch( i )
	{
		case PM_SUB_PB:
			{
				FUNCEND("ParticleMaster::SubAnim");
				return pblock;
			}
		case PM_SUB_MID:
			{
				FUNCEND("ParticleMaster::SubAnim");
				return ctrl_mid;
			}
		case PM_SUB_END:
			{
				FUNCEND("ParticleMaster::SubAnim");
				return ctrl_end;
			}
		case PM_SUB_MID_ROT:
			{
				FUNCEND("ParticleMaster::SubAnim");
				return ctrl_mid_rot;
			}
		case PM_SUB_END_ROT:
			{
				FUNCEND("ParticleMaster::SubAnim");
				return ctrl_end_rot;
			}
		default:
			{
				FUNCEND("ParticleMaster::SubAnim");
				return NULL;
			}
	}
	FUNCEND("ParticleMaster::SubAnim");
}

RefTargetHandle ParticleMaster::GetReference(int i)  
{ 
	FUNCSTART("ParticleMaster::GetReference");
	switch( i )
	{
		case PM_REF_PB:
			{
				FUNCEND("ParticleMaster::GetReference");
				return pblock;
			}
		case PM_REF_MID_CTRL:
			{
				FUNCEND("ParticleMaster::GetReference");
				return ctrl_mid;
			}
		case PM_REF_END_CTRL:
			{
				FUNCEND("ParticleMaster::GetReference");
				return ctrl_end;
			}
		case PM_REF_MID_ROT_CTRL:
			{
				FUNCEND("ParticleMaster::GetReference");
				return ctrl_mid_rot;
			}
		case PM_REF_END_ROT_CTRL:
			{
				FUNCEND("ParticleMaster::GetReference");
				return ctrl_end_rot;
			}
		default:
			{
				FUNCEND("ParticleMaster::GetReference");
				return NULL;
			}
	}
	FUNCEND("ParticleMaster::GetReference");
}

void ParticleMaster::SetReference(int i, RefTargetHandle rtarg) 
{
	FUNCSTART("ParticleMaster::SetReference");
	switch( i )
	{
		case PM_REF_PB:
			pblock = (IParamBlock *)rtarg;
			break;
		case PM_REF_MID_CTRL:
			ctrl_mid = (IndePosition*)rtarg;
			break;
		case PM_REF_END_CTRL:
			ctrl_end = (IndePosition*)rtarg;
			break;
		case PM_REF_MID_ROT_CTRL:
			ctrl_mid_rot = (LocalEulerRotation*)rtarg;
			break;
		case PM_REF_END_ROT_CTRL:
			ctrl_end_rot = (LocalEulerRotation*)rtarg;
			break;
		default:
			break;
	}
	FUNCEND("ParticleMaster::SetReference");
}

RefResult ParticleMaster::NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, 
     PartID& partID, RefMessage message ) 
   {
	FUNCSTART("ParticleMaster::NotifyRefChanged");
	switch (message) 
	{
		case REFMSG_GET_PARAM_DIM: 
		{ 
			// the ParamBlock needs info to display in the tree view
			GetParamDim *gpd = (GetParamDim*)partID;
			switch (gpd->index) 
			{
				case PB_LIFETIME: gpd->dim = stdWorldDim; break;	  
					break;									
			}
			{
				FUNCEND("ParticleMaster::NotifyRefChanged");
				return REF_STOP; 
			}
		}

		case REFMSG_GET_PARAM_NAME: 
		{
			// the ParamBlock needs info to display in the tree view
			GetParamName *gpn = (GetParamName*)partID;
			switch (gpn->index) 
			{
				case PB_LIFETIME: gpn->name = "Lifetime";	break;
					
			}
			{
				FUNCEND("ParticleMaster::NotifyRefChanged");
				return REF_STOP; 
			}
		}
		
		case BOX_MOVED:
		{
			if (gbLockPMU)
				return REF_STOP;

			AssignParticlePosProperties();

			if( !ignore_controller_updates )
			{
				RecalculateParameters( 0, changeInt );
			}
			{
				FUNCEND("ParticleMaster::NotifyRefChanged");
				return REF_STOP;
			}
		}
	}
	{
		FUNCEND("ParticleMaster::NotifyRefChanged");
		return(REF_SUCCEED);
	}
	FUNCEND("ParticleMaster::NotifyRefChanged");
}

void ParticleMaster::RecalculateParameters( TimeValue t, Interval& ivalid )
{
	FUNCSTART("ParticleMaster::RecalculateParameters");

	//return;

	if (gbLockPosChange)
		return;

	Point3 x0, x1, x2;
	Point3 vel, acc;
	float t1, t2, t12, t22;

	if( ( ctrl_mid == NULL ) ||
		( ctrl_end == NULL ))
	{
		{
			FUNCEND("ParticleMaster::RecalculateParameters");
			return;
		}
	}

	x0.x = 0;
	x0.y = 0;
	x0.z = 0;
	
	ctrl_mid->GetValue( t, &x1, ivalid, CTRL_ABSOLUTE );
	ctrl_end->GetValue( t, &x2, ivalid, CTRL_ABSOLUTE );

	t1 = ( GetMidtimePCT( t, ivalid ) / 100.0f ) * GetLifetime( t, ivalid );
	t2 = GetLifetime( t, ivalid );
	t12 = ( t1 * t1 );
	t22 = ( t2 * t2 );
	
	vel =  ( t22 * (x1 - x0) - t12 * (x2 - x0)) / (( t1 * t2 ) * (t2 - t1));
	acc = 2 * ( t1 * (x2 - x0) - t2 * (x1 - x0)) / (( t1 * t2 ) * ((t2 - t1)));

	SetAccelerationX( t, acc.x, true, false );
	SetAccelerationY( t, acc.y, true, false );
	SetAccelerationZ( t, acc.z, true, false );
	
	SetVelocityX( t, vel.x, true, false );
	SetVelocityY( t, vel.y, true, false );
	SetVelocityZ( t, vel.z, true, false );
	FUNCEND("ParticleMaster::RecalculateParameters");
}

void ParticleMaster::RecalculatePositions( TimeValue t )
{
	FUNCSTART("ParticleMaster::RecalculatePositions");

	//return;

	if (gbLockPosChange)
		return;
	
	Point3 vel, acc;
	Point3 trans;
	float t2;
	
	vel.x = GetVelocityX( t, FOREVER );
	vel.y = -GetVelocityZ( t, FOREVER );
	vel.z = GetVelocityY( t, FOREVER );
	
	acc.x = GetAccelerationX( t, FOREVER );
	acc.y = -GetAccelerationZ( t, FOREVER );
	acc.z = GetAccelerationY( t, FOREVER );

	t2 = ( GetMidtimePCT( t, FOREVER ) / 100.0f ) * GetLifetime( t, FOREVER );
	trans = ( vel + ( acc * t2 / 2 )) * t2;
	if( ctrl_mid )
	{
		ctrl_mid->SetValueFromMaster( t, &trans, TRUE, CTRL_ABSOLUTE );
	}

	t2 = GetLifetime( t, FOREVER );
	trans = ( vel + ( acc * t2 / 2 )) * t2;
	if( ctrl_end )
	{
		ctrl_end->SetValueFromMaster( t, &trans, TRUE, CTRL_ABSOLUTE );
	}
	
	// Make sure spinners track when animating and in Motion Panel
	UpdateUI(t);
	FUNCEND("ParticleMaster::RecalculatePositions");
}

int ParticleMaster::GetNum(TimeValue t, Interval& valid ) { 	
	FUNCSTART("ParticleMaster::GetNum");
	return numNodes;
		FUNCEND("ParticleMaster::GetNum");
}



float ParticleMaster::GetLifetime(TimeValue t, Interval& valid ) { 	
	FUNCSTART("ParticleMaster::GetLifetime");
	float f;
	pblock->GetValue( PB_LIFETIME, t, f, valid );
	{
		FUNCEND("ParticleMaster::GetLifetime");
		return f;
	}
	FUNCEND("ParticleMaster::GetLifetime");
}

void ParticleMaster::SetLifetime(TimeValue t, float r,  bool update_props, bool update_controls ) 
{ 
	FUNCSTART("ParticleMaster::SetLifetime");
	if( r < 0.1f )
	{
		{
			FUNCEND("ParticleMaster::SetLifetime");
			return;
		}
	}
	pblock->SetValue( PB_LIFETIME, t, r );
	if( update_controls )
	{
		INode* start_node, *mid_node, *end_node;
		Point3 start_pos, mid_pos, end_pos;
		
		start_node = GetMasterNode();
		if( start_node )
		{
			Object* obj = start_node->EvalWorldState(0).obj;
			if( obj->ClassID() == PARTICLE_BOX_CLASS_ID )
			{
				IParticleBox* p_box, *p_mid, *p_end;				
				Point3 x0, x1, x2;
				Point3 vel, acc;
				float t1, t2, t12, t22, lifetime;				
				char value_str[64];
				
				p_box = dynamic_cast< IParticleBox* > ( obj );
				p_mid = p_box->GetMidBox();
				p_end = p_box->GetEndBox();
				mid_node = p_mid->GetObjectNode();
				end_node = p_end->GetObjectNode();

				start_pos = start_node->GetNodeTM(0).GetTrans();
				mid_pos = mid_node->GetNodeTM(0).GetTrans();
				end_pos = end_node->GetNodeTM(0).GetTrans();

				lifetime = r;
				x0 = start_pos;
				x1 = mid_pos;
				x2 = end_pos;
				
				t1 = ( GetMidtimePCT( t, FOREVER ) / 100.0f ) * lifetime;
				t2 = lifetime;
				t12 = ( t1 * t1 );
				t22 = ( t2 * t2 );
				
				vel =  ( t22 * (x1 - x0) - t12 * (x2 - x0)) / (( t1 * t2 ) * (t2 - t1));
				acc = 2 * ( t1 * (x2 - x0) - t2 * (x1 - x0)) / (( t1 * t2 ) * ((t2 - t1)));

				SetAccelerationX( t, acc.x, true, false );
				SetAccelerationY( t, acc.z, true, false );
				SetAccelerationZ( t, -acc.y, true, false );
				
				SetVelocityX( t, vel.x, true, false );
				SetVelocityY( t, vel.z, true, false );
				SetVelocityZ( t, -vel.y, true, false );

				sprintf( value_str, "%f", acc.x );
				pPropEdit->SetValue( CStr( ParticleProperties[vACCELERATION_X] ), value_str );
				sprintf( value_str, "%f", acc.z );
				pPropEdit->SetValue( CStr( ParticleProperties[vACCELERATION_Y] ), value_str );
				sprintf( value_str, "%f", -acc.y );
				pPropEdit->SetValue( CStr( ParticleProperties[vACCELERATION_Z] ), value_str );

				sprintf( value_str, "%f", vel.x );
				pPropEdit->SetValue( CStr( ParticleProperties[vVEL_X] ), value_str );
				sprintf( value_str, "%f", vel.z );
				pPropEdit->SetValue( CStr( ParticleProperties[vVEL_Y] ), value_str );
				sprintf( value_str, "%f", -vel.y );
				pPropEdit->SetValue( CStr( ParticleProperties[vVEL_Z] ), value_str );
			}
		}

		RecalculatePositions( t );
	}
	if( update_props )
	{
		if( GetMasterNode())
		{
			AddPropValue( GetMasterNode(), "Emission_Values",
							"LifetimeMax", r );
		}		
	}
		
	NotifyDependents(FOREVER, PART_OBJ, REFMSG_CHANGE);	
	FUNCEND("ParticleMaster::SetLifetime");
}

float ParticleMaster::GetMidtimePCT(TimeValue t, Interval& valid ) { 	
	FUNCSTART("ParticleMaster::GetMidtimePCT");
	float f;
	pblock->GetValue( PB_MIDTIME_PCT, t, f, valid );
	{
		FUNCEND("ParticleMaster::GetMidtimePCT");
		return f;
	}
	FUNCEND("ParticleMaster::GetMidtimePCT");
}

void ParticleMaster::SetMidtimePCT(TimeValue t, float r, bool update_props, bool update_controls )
{
	FUNCSTART("ParticleMaster::SetMidtimePCT");
	pblock->SetValue( PB_MIDTIME_PCT, t, r );
	if( update_controls )
	{
		RecalculatePositions( t );
	}
	if( update_props )
	{
		if( GetMasterNode())
		{
			AddPropValue( GetMasterNode(), "Emission_Values",
							"MidTimePct", r );
		}
	}
	
	NotifyDependents(FOREVER, PART_OBJ, REFMSG_CHANGE);	
	FUNCEND("ParticleMaster::SetMidtimePCT");
}

float ParticleMaster::GetAccelerationX(TimeValue t, Interval& valid ) { 	
	FUNCSTART("ParticleMaster::GetAccelerationX");
	float f;
	pblock->GetValue( PB_ACCEL_X, t, f, valid );
	{
		FUNCEND("ParticleMaster::GetAccelerationX");
		return f;
	}
	FUNCEND("ParticleMaster::GetAccelerationX");
}

void ParticleMaster::SetAccelerationX(TimeValue t, float r, bool update_props, bool update_controls )
{
	FUNCSTART("ParticleMaster::SetAccelerationX");
	pblock->SetValue( PB_ACCEL_X, t, r );
	if( update_controls )
	{
		RecalculatePositions( t );
	}

	if( update_props )
	{
		if( GetMasterNode())
		{
			AddPropValue( GetMasterNode(), "Emission_Values",
							"Acceleration_X", r );
		}
	}

	NotifyDependents(FOREVER, PART_OBJ, REFMSG_CHANGE);	
	FUNCEND("ParticleMaster::SetAccelerationX");
}

float ParticleMaster::GetAccelerationY(TimeValue t, Interval& valid ) { 	
	FUNCSTART("ParticleMaster::GetAccelerationY");
	float f;
	pblock->GetValue( PB_ACCEL_Y, t, f, valid );
	{
		FUNCEND("ParticleMaster::GetAccelerationY");
		return f;
	}
	FUNCEND("ParticleMaster::GetAccelerationY");
}

void ParticleMaster::SetAccelerationY(TimeValue t, float r, bool update_props, bool update_controls )
{
	FUNCSTART("ParticleMaster::SetAccelerationY");
	pblock->SetValue( PB_ACCEL_Y, t, r );
	if( update_controls )
	{
		RecalculatePositions( t );
	}
	if( update_props )
	{
		if( GetMasterNode())
		{
			AddPropValue( GetMasterNode(), "Emission_Values",
							"Acceleration_Y", r );
		}
	}
		
	NotifyDependents(FOREVER, PART_OBJ, REFMSG_CHANGE);	
	FUNCEND("ParticleMaster::SetAccelerationY");
}

float ParticleMaster::GetAccelerationZ(TimeValue t, Interval& valid ) { 	
	FUNCSTART("ParticleMaster::GetAccelerationZ");
	float f;
	pblock->GetValue( PB_ACCEL_Z, t, f, valid );
	{
		FUNCEND("ParticleMaster::GetAccelerationZ");
		return f;
	}
	FUNCEND("ParticleMaster::GetAccelerationZ");
}

void ParticleMaster::SetAccelerationZ(TimeValue t, float r, bool update_props, bool update_controls )
{
	FUNCSTART("ParticleMaster::SetAccelerationZ");
	pblock->SetValue( PB_ACCEL_Z, t, r );
	if( update_controls )
	{
		RecalculatePositions( t );
	}
	if( update_props )
	{
		if( GetMasterNode())
		{
			AddPropValue( GetMasterNode(), "Emission_Values",
							"Acceleration_Z", r );
		}
	}

	NotifyDependents(FOREVER, PART_OBJ, REFMSG_CHANGE);	
	FUNCEND("ParticleMaster::SetAccelerationZ");
}

float ParticleMaster::GetVelocityX(TimeValue t, Interval& valid ) { 	
	FUNCSTART("ParticleMaster::GetVelocityX");
	float f;
	pblock->GetValue( PB_VEL_X, t, f, valid );
	{
		FUNCEND("ParticleMaster::GetVelocityX");
		return f;
	}
	FUNCEND("ParticleMaster::GetVelocityX");
}

void ParticleMaster::SetVelocityX(TimeValue t, float r, bool update_props, bool update_controls )
{
	FUNCSTART("ParticleMaster::SetVelocityX");
	pblock->SetValue( PB_VEL_X, t, r );
	if( update_controls )
	{
		RecalculatePositions( t );
	}
	if( update_props )
	{
		if( GetMasterNode())
		{
			AddPropValue( GetMasterNode(), "Emission_Values",
							"InitialVel_X", r );					
		}
	}

	NotifyDependents(FOREVER, PART_OBJ, REFMSG_CHANGE);
	FUNCEND("ParticleMaster::SetVelocityX");
}

float ParticleMaster::GetVelocityY(TimeValue t, Interval& valid ) { 	
	FUNCSTART("ParticleMaster::GetVelocityY");
	float f;
	pblock->GetValue( PB_VEL_Y, t, f, valid );
	{
		FUNCEND("ParticleMaster::GetVelocityY");
		return f;
	}
	FUNCEND("ParticleMaster::GetVelocityY");
}

void ParticleMaster::SetVelocityY(TimeValue t, float r, bool update_props, bool update_controls )
{
	FUNCSTART("ParticleMaster::SetVelocityY");
	pblock->SetValue( PB_VEL_Y, t, r );
	if( update_controls )
	{
		RecalculatePositions( t );
	}
	if( update_props )
	{
		if( GetMasterNode())
		{
			AddPropValue( GetMasterNode(), "Emission_Values",
							"InitialVel_Y", r );
		}
	}
	
	NotifyDependents(FOREVER, PART_OBJ, REFMSG_CHANGE);	
	FUNCEND("ParticleMaster::SetVelocityY");
}

void ParticleMaster::UpdateNodeProperties( bool update_controls )
{
	FUNCSTART("ParticleMaster::UpdateNodeProperties");
	CStr value;
	value = pPropEdit->GetDefault( "ParticleObject", dlgType, ParticleProperties[vLIFETIME] );
	ParticleMaster::dlgLifetime = atof( value );
	value = pPropEdit->GetDefault( "ParticleObject", dlgType, ParticleProperties[vMID_TIME_PCT] );
	ParticleMaster::dlgMidtimePCT = atof( value );
	value = pPropEdit->GetDefault( "ParticleObject", dlgType, ParticleProperties[vACCELERATION_X] );
	ParticleMaster::dlgAccel.x = atof( value );
	value = pPropEdit->GetDefault( "ParticleObject", dlgType, ParticleProperties[vACCELERATION_Y] );
	ParticleMaster::dlgAccel.y = atof( value );
	value = pPropEdit->GetDefault( "ParticleObject", dlgType, ParticleProperties[vACCELERATION_Z] );					
	ParticleMaster::dlgAccel.z = atof( value );
	value = pPropEdit->GetDefault( "ParticleObject", dlgType, ParticleProperties[vVEL_X] );
	ParticleMaster::dlgVel.x = atof( value );
	value = pPropEdit->GetDefault( "ParticleObject", dlgType, ParticleProperties[vVEL_Y] );
	ParticleMaster::dlgVel.y = atof( value );
	value = pPropEdit->GetDefault( "ParticleObject", dlgType, ParticleProperties[vVEL_Z] );
	ParticleMaster::dlgVel.z = atof( value );
	value = pPropEdit->GetDefault( "ParticleObject", dlgType, ParticleProperties[vUSE_MIDPOINT] );	
	
	numNodes = 2;
	SetLifetime( TimeValue(0), dlgLifetime, update_controls, update_controls );
	SetMidtimePCT( TimeValue(0), dlgMidtimePCT, update_controls, update_controls );	
	SetAccelerationX( TimeValue(0), dlgAccel.x, update_controls, update_controls );
	SetAccelerationY( TimeValue(0), dlgAccel.y, update_controls, update_controls );
	SetAccelerationZ( TimeValue(0), dlgAccel.z, update_controls, update_controls );
	SetVelocityX( TimeValue(0), dlgVel.x, update_controls, update_controls );
	SetVelocityY( TimeValue(0), dlgVel.y, update_controls, update_controls );
	SetVelocityZ( TimeValue(0), dlgVel.z, update_controls, update_controls );
	FUNCEND("ParticleMaster::UpdateNodeProperties");
}

void ParticleMaster::SetType( char* type )
{
	FUNCSTART("ParticleMaster::SetType");
	dlgType = type;
	UpdateNodeProperties( true );
	FUNCEND("ParticleMaster::SetType");
}

float ParticleMaster::GetVelocityZ(TimeValue t, Interval& valid ) { 	
	FUNCSTART("ParticleMaster::GetVelocityZ");
	float f;
	pblock->GetValue( PB_VEL_Z, t, f, valid );
	{
		FUNCEND("ParticleMaster::GetVelocityZ");
		return f;
	}
	FUNCEND("ParticleMaster::GetVelocityZ");
}

void ParticleMaster::SetVelocityZ(TimeValue t, float r, bool update_props, bool update_controls )
{
	FUNCSTART("ParticleMaster::SetVelocityZ");
	pblock->SetValue( PB_VEL_Z, t, r );
	if( update_controls )
	{
		RecalculatePositions( t );
	}
	if( update_props )
	{
		if( GetMasterNode())
		{
			AddPropValue( GetMasterNode(), "Emission_Values",
							"InitialVel_Z", r );
		}
	}
	
	NotifyDependents(FOREVER, PART_OBJ, REFMSG_CHANGE);	
	FUNCEND("ParticleMaster::SetVelocityZ");
}

//--------------------------------------------------


INT_PTR CALLBACK MasterParamDialogProc( HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam )
	{
	FUNCSTART("MasterParamDialogProc");
	ParticleMaster *mc = (ParticleMaster *)GetWindowLongPtr( hDlg, GWLP_USERDATA );
	{
		FUNCEND("MasterParamDialogProc");
		if ( !mc && message != WM_INITDIALOG ) return FALSE;
	}
	
	assert(mc->iObjParams);
	switch ( message ) {
		case WM_INITDIALOG:
		{
			mc = (ParticleMaster *)lParam;
			SetWindowLongPtr( hDlg, GWLP_USERDATA, (LONG_PTR)mc );
			SetDlgFont( hDlg, mc->iObjParams->GetAppHFont() );
			
			mc->lifeSpin  = GetISpinner(GetDlgItem(hDlg,IDC_LIFETIME_SPINNER));
			mc->midtimePCTSpin  = GetISpinner(GetDlgItem(hDlg,IDC_MIDTIMEPCT_SPINNER));
			mc->accelXSpin  = GetISpinner(GetDlgItem(hDlg,IDC_ACCEL_X_SPINNER));
			mc->accelYSpin  = GetISpinner(GetDlgItem(hDlg,IDC_ACCEL_Y_SPINNER));
			mc->accelZSpin  = GetISpinner(GetDlgItem(hDlg,IDC_ACCEL_Z_SPINNER));
			mc->velXSpin  = GetISpinner(GetDlgItem(hDlg,IDC_VEL_X_SPINNER));
			mc->velYSpin  = GetISpinner(GetDlgItem(hDlg,IDC_VEL_Y_SPINNER));
			mc->velZSpin  = GetISpinner(GetDlgItem(hDlg,IDC_VEL_Z_SPINNER));

			mc->lifeSpin->SetLimits( 0.1f, 100000.0f, FALSE );
			mc->midtimePCTSpin->SetLimits( 0.0f, 100.0f, FALSE );
			
			mc->accelXSpin->SetLimits( -1000000.0f, 1000000.0f, FALSE );
			mc->accelYSpin->SetLimits( -1000000.0f, 1000000.0f, FALSE );
			mc->accelZSpin->SetLimits( -1000000.0f, 1000000.0f, FALSE );
			mc->velXSpin->SetLimits( -1000000.0f, 1000000.0f, FALSE );
			mc->velYSpin->SetLimits( -1000000.0f, 1000000.0f, FALSE );
			mc->velZSpin->SetLimits( -1000000.0f, 1000000.0f, FALSE );

			mc->lifeSpin->SetScale(float(0.1) );
			mc->midtimePCTSpin->SetScale(float(0.1) );
			mc->accelXSpin->SetScale(float(0.1) );
			mc->accelYSpin->SetScale(float(0.1) );
			mc->accelZSpin->SetScale(float(0.1) );
			mc->velXSpin->SetScale(float(0.1) );
			mc->velYSpin->SetScale(float(0.1) );
			mc->velZSpin->SetScale(float(0.1) );

			mc->lifeSpin->SetValue( mc->GetLifetime(mc->iObjParams->GetTime()), FALSE );
			mc->midtimePCTSpin->SetValue( mc->GetMidtimePCT(mc->iObjParams->GetTime()), FALSE );
			mc->accelXSpin->SetValue( mc->GetAccelerationX(mc->iObjParams->GetTime()), FALSE );
			mc->accelYSpin->SetValue( mc->GetAccelerationY(mc->iObjParams->GetTime()), FALSE );
			mc->accelZSpin->SetValue( mc->GetAccelerationZ(mc->iObjParams->GetTime()), FALSE );
			mc->velXSpin->SetValue( mc->GetVelocityX(mc->iObjParams->GetTime()), FALSE );
			mc->velYSpin->SetValue( mc->GetVelocityY(mc->iObjParams->GetTime()), FALSE );
			mc->velZSpin->SetValue( mc->GetVelocityZ(mc->iObjParams->GetTime()), FALSE );

			mc->lifeSpin->LinkToEdit( GetDlgItem(hDlg,IDC_LIFETIME), EDITTYPE_FLOAT );			
			mc->midtimePCTSpin->LinkToEdit( GetDlgItem(hDlg,IDC_MIDTIMEPCT), EDITTYPE_FLOAT );			
			
			mc->accelXSpin->LinkToEdit( GetDlgItem(hDlg,IDC_ACCEL_X), EDITTYPE_FLOAT );
			mc->accelYSpin->LinkToEdit( GetDlgItem(hDlg,IDC_ACCEL_Y), EDITTYPE_FLOAT );
			mc->accelZSpin->LinkToEdit( GetDlgItem(hDlg,IDC_ACCEL_Z), EDITTYPE_FLOAT );
			mc->velXSpin->LinkToEdit( GetDlgItem(hDlg,IDC_VEL_X), EDITTYPE_FLOAT );
			mc->velYSpin->LinkToEdit( GetDlgItem(hDlg,IDC_VEL_Y), EDITTYPE_FLOAT );
			mc->velZSpin->LinkToEdit( GetDlgItem(hDlg,IDC_VEL_Z), EDITTYPE_FLOAT );
			
			ConfigClass* cclass=pPropEdit->GetConfigClass( "ParticleObject" );
			if (cclass)
			{
				Link<ConfigType>* curNode=cclass->types.GetHead();

				SendDlgItemMessage( hDlg,IDC_TYPE_LIST,CB_RESETCONTENT,0,0);

				while(curNode)
				{
					// Don't add the types that have NonPlaceable in their directives
					if (!strstr(curNode->data.strCmds, NONPLACEABLE_METACMD))
					{
						SendDlgItemMessage(hDlg,IDC_TYPE_LIST,CB_ADDSTRING,(WPARAM)0,(LPARAM)(char*)curNode->data.name);
					}

					curNode=curNode->next;
				}
			}

			SendDlgItemMessage(hDlg, IDC_TYPE_LIST, CB_SELECTSTRING, (WPARAM)-1, (LPARAM)ParticleMaster::dlgType.data());
				{
					FUNCEND("MasterParamDialogProc");
				return FALSE;	// DB 2/27
				}
		}

		case WM_DESTROY:
			ReleaseISpinner( mc->lifeSpin );
			ReleaseISpinner( mc->accelXSpin );
			ReleaseISpinner( mc->accelYSpin );
			ReleaseISpinner( mc->accelZSpin );
			ReleaseISpinner( mc->velXSpin );
			ReleaseISpinner( mc->velYSpin );
			ReleaseISpinner( mc->velZSpin );
			mc->lifeSpin = NULL;
			mc->accelXSpin = NULL;
			mc->accelYSpin = NULL;
			mc->accelZSpin = NULL;
			mc->velXSpin = NULL;
			mc->velYSpin = NULL;
			mc->velZSpin = NULL;
			{
				FUNCEND("MasterParamDialogProc");
				return FALSE;
			}

		case CC_SPINNER_CHANGE:	{
			if (!theHold.Holding()) theHold.Begin();
			TimeValue t = mc->iObjParams->GetTime();
			switch ( LOWORD(wParam) ) {
				case IDC_LIFETIME_SPINNER:
					mc->SetLifetime(t, mc->lifeSpin->GetFVal(), true );
					break;
				case IDC_MIDTIMEPCT_SPINNER:
					mc->SetMidtimePCT(t, mc->midtimePCTSpin->GetFVal(), true );					
					break;
				case IDC_ACCEL_X_SPINNER:
					mc->SetAccelerationX(t, mc->accelXSpin->GetFVal(), true );					
					break;
				case IDC_ACCEL_Y_SPINNER:
					mc->SetAccelerationY(t, mc->accelYSpin->GetFVal(), true );					
					break;
				case IDC_ACCEL_Z_SPINNER:
					mc->SetAccelerationZ(t, mc->accelZSpin->GetFVal(), true );					
					break;
				case IDC_VEL_X_SPINNER:
					mc->SetVelocityX(t, mc->velXSpin->GetFVal(), true );					
					break;
				case IDC_VEL_Y_SPINNER:
					mc->SetVelocityY(t, mc->velYSpin->GetFVal(), true );					
					break;
				case IDC_VEL_Z_SPINNER:
					mc->SetVelocityZ(t, mc->velZSpin->GetFVal(), true );					
					break;
				}
			assert(mc->iObjParams);
			mc->iObjParams->RedrawViews(t, REDRAW_INTERACTIVE, mc);
			{
				FUNCEND("MasterParamDialogProc");
				return TRUE;
			}

			}
		case CC_SPINNER_BUTTONDOWN:
			theHold.Begin();
			{
				FUNCEND("MasterParamDialogProc");
				return TRUE;
			}

		case WM_CUSTEDIT_ENTER:
		case CC_SPINNER_BUTTONUP:
			if (HIWORD(wParam) || message==WM_CUSTEDIT_ENTER) 
				theHold.Accept(GetString(IDS_DS_PARAMCHG));
			else 
				theHold.Cancel();
			mc->iObjParams->RedrawViews(mc->iObjParams->GetTime(), REDRAW_END, mc);
			{
				FUNCEND("MasterParamDialogProc");
				return TRUE;
			}

		case WM_MOUSEACTIVATE:
			mc->iObjParams->RealizeParamPanel();
			{
				FUNCEND("MasterParamDialogProc");
				return FALSE;
			}

		case WM_LBUTTONDOWN:
		case WM_LBUTTONUP:
		case WM_MOUSEMOVE:
			mc->iObjParams->RollupMouseMessage(hDlg,message,wParam,lParam);
			{
				FUNCEND("MasterParamDialogProc");
				return FALSE;
			}

		case WM_COMMAND:			
		{
			switch(HIWORD(wParam))
			{
				case LBN_SELCHANGE:
				{
					switch(LOWORD(wParam))
					{
						case IDC_TYPE_LIST:
						{
							char bufType[256];
							int  index;

							index=SendDlgItemMessage(hDlg,IDC_TYPE_LIST,CB_GETCURSEL,0,0);
							SendDlgItemMessage(hDlg,IDC_TYPE_LIST,CB_GETLBTEXT,(WPARAM)index,(LPARAM)bufType);
							mc->SetType( bufType );
							gInterface->ForceCompleteRedraw();
							break;
						}
					}
					break;
				}
			}

			{
				FUNCEND("MasterParamDialogProc");
				return FALSE;
			}
		}

		default:
			{
				FUNCEND("MasterParamDialogProc");
				return FALSE;
			}
		}
		FUNCEND("MasterParamDialogProc");
}



void ParticleMaster::BeginEditParams( IObjParam *ip, ULONG flags,Animatable *prev )
{
	FUNCSTART("ParticleMaster::BeginEditParams");
	iObjParams = ip;
	
	if( !hMasterParams ) 
	{
		hMasterParams = ip->AddRollupPage( 
				hInstance, 
				MAKEINTRESOURCE(IDD_PARTICLE_PARAMS),				
				MasterParamDialogProc,
				GetString(IDS_RB_PARAMETERS), 
				(LPARAM)this );		
		ip->RegisterDlgWnd(hMasterParams);		
	} 
	else 
	{
		SetWindowLongPtr( hMasterParams, GWLP_USERDATA, (LONG_PTR)this );		

		// Init the dialog to our values.
		lifeSpin->SetValue(GetLifetime(ip->GetTime()),FALSE);
		midtimePCTSpin->SetValue(GetMidtimePCT(ip->GetTime()),FALSE);
		accelXSpin->SetValue(GetAccelerationX(ip->GetTime()),FALSE);
		accelYSpin->SetValue(GetAccelerationY(ip->GetTime()),FALSE);
		accelZSpin->SetValue(GetAccelerationZ(ip->GetTime()),FALSE);
		velXSpin->SetValue(GetVelocityX(ip->GetTime()),FALSE);
		velYSpin->SetValue(GetVelocityY(ip->GetTime()),FALSE);
		velZSpin->SetValue(GetVelocityZ(ip->GetTime()),FALSE);

	}
	FUNCEND("ParticleMaster::BeginEditParams");
}
		
void ParticleMaster::EndEditParams( IObjParam *ip, ULONG flags,Animatable *next )
{
	FUNCSTART("ParticleMaster::EndEditParams");
	if (hMasterParams==NULL) 
	{
		{
			FUNCEND("ParticleMaster::EndEditParams");
			return;
		}
	}

	dlgLifetime = lifeSpin->GetFVal();
	dlgMidtimePCT= midtimePCTSpin->GetFVal();
	dlgAccel.x = accelXSpin->GetFVal();
	dlgAccel.y = accelYSpin->GetFVal();
	dlgAccel.z = accelZSpin->GetFVal();
	dlgVel.x = velXSpin->GetFVal();
	dlgVel.y = velYSpin->GetFVal();
	dlgVel.z = velZSpin->GetFVal();
		
	if( flags&END_EDIT_REMOVEUI ) 
	{
		ip->UnRegisterDlgWnd(hMasterParams);
		ip->DeleteRollupPage(hMasterParams);
		hMasterParams = NULL;
	}
	else 
	{
		SetWindowLongPtr( hMasterParams, GWLP_USERDATA, 0 );
	}
	
	iObjParams = NULL;
	FUNCEND("ParticleMaster::EndEditParams");
}

#define NUMNODES_CHUNK 0x100

IOResult ParticleMaster::Save(ISave *isave) 
{
	FUNCSTART("ParticleMaster::Save");
	ULONG nb;
	isave->BeginChunk(NUMNODES_CHUNK);
	isave->Write(&numNodes,sizeof(numNodes), &nb);
	isave->EndChunk();
	{
		FUNCEND("ParticleMaster::Save");
		return IO_OK;
	}
	FUNCEND("ParticleMaster::Save");
}

IOResult ParticleMaster::Load(ILoad *iload) 
{
	FUNCSTART("ParticleMaster::Load");
	ULONG nb;
	IOResult res;
	while (IO_OK==(res=iload->OpenChunk())) 
	{
		switch(iload->CurChunkID())  
		{
			case NUMNODES_CHUNK: 
			{
				res = iload->Read(&numNodes,sizeof(numNodes), &nb);
			}
			break;
		}
		iload->CloseChunk();
		if (res!=IO_OK) 
			{
				FUNCEND("ParticleMaster::Load");
				return res;
			}
	}

	{
		FUNCEND("ParticleMaster::Load");
		return IO_OK;
	}
	FUNCEND("ParticleMaster::Load");
}

//----------------------------------------------------------------------
class ParticleMasterCreationManager : public MouseCallBack, ReferenceMaker 
{
	public:
		CreateMouseCallBack *createCB;	
		INode *node0;		
		ParticleMaster *theMaster;
		IObjCreate *createInterface;
		ClassDesc *cDesc;
		Matrix3 mat;  // the nodes TM relative to the CP
		IPoint2 pt0;
		Point3 center;
		BOOL attachedToNode;
		int lastPutCount;		

		void CreateNewMaster();
			
		int ignoreSelectionChange;

		int NumRefs() { return 1; }
		RefTargetHandle GetReference(int i);
		void SetReference(int i, RefTargetHandle rtarg);

		// StdNotifyRefChanged calls this, which can change the partID to new value 
		// If it doesnt depend on the particular message& partID, it should return
		// REF_DONTCARE
	    RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, 
	    	PartID& partID,  RefMessage message);

		void Begin( IObjCreate *ioc, ClassDesc *desc );
		void End();
		
		ParticleMasterCreationManager()
		{
			ignoreSelectionChange = FALSE;
		}
		int proc( HWND hwnd, int msg, int point, int flag, IPoint2 m );
	};

#define CID_BONECREATE	CID_USER + 1

class ParticleMasterCreateMode : public CommandMode 
{
		ParticleMasterCreationManager proc;
	public:
		void Begin( IObjCreate *ioc, ClassDesc *desc ) { proc.Begin( ioc, desc ); }
		void End() { proc.End(); }
		int Class() { return CREATE_COMMAND; }
		int ID() { return CID_BONECREATE; }
		MouseCallBack *MouseProc(int *numPoints) { *numPoints = 100000; return &proc; }
		ChangeForegroundCallback *ChangeFGProc() { return CHANGE_FG_SELECTED; }
		BOOL ChangeFG( CommandMode *oldMode ) { return (oldMode->ChangeFGProc() != CHANGE_FG_SELECTED); }
		void EnterMode() { SetCursor( LoadCursor( hInstance, MAKEINTRESOURCE(IDC_CROSS_HAIR) ) ); }
		void ExitMode() { SetCursor( LoadCursor(NULL, IDC_ARROW) ); }
		BOOL IsSticky() { return FALSE; }
};

static ParticleMasterCreateMode theParticleMasterCreateMode;

//ParticleMasterCreationManager::ParticleMasterCreationManager( IObjCreate *ioc, ClassDesc *desc )
void ParticleMasterCreationManager::Begin( IObjCreate *ioc, ClassDesc *desc )
{
	FUNCSTART("ParticleMasterCreationManager::Begin");
	createInterface = ioc;
	cDesc           = desc;
	createCB        = NULL;
	node0			= NULL;
	theMaster 		= NULL;
	attachedToNode = FALSE;

	CreateNewMaster();
	FUNCEND("ParticleMasterCreationManager::Begin");
}

void ParticleMasterCreationManager::SetReference(int i, RefTargetHandle rtarg) 
{ 
	FUNCSTART("ParticleMasterCreationManager::SetReference");
	switch(i) 
	{
		case 0: 
			node0 = (INode *)rtarg; 
			break;
		default: assert(0); 
	}
	FUNCEND("ParticleMasterCreationManager::SetReference");
}

RefTargetHandle ParticleMasterCreationManager::GetReference(int i) 
{ 
	FUNCSTART("ParticleMasterCreationManager::GetReference");
	switch(i) 
	{
		{
			FUNCEND("ParticleMasterCreationManager::GetReference");
			case 0: return (RefTargetHandle)node0;
		}
		default: assert(0); 
	}
	
	{
		FUNCEND("ParticleMasterCreationManager::GetReference");
		return NULL;
	}
	FUNCEND("ParticleMasterCreationManager::GetReference");
}

void ParticleMasterCreationManager::End()
{
	FUNCSTART("ParticleMasterCreationManager::End");
	if (theMaster) 
	{
		theMaster->EndEditParams( (IObjParam*)createInterface, 
	                    	          TRUE/*destroy*/, NULL );
		if ( !attachedToNode ) 
		{
			delete theMaster;
			theMaster = NULL;
			// DS 8/21/97: If something has been put on the undo stack since this object was created, we have to flush the undo stack.
			if (theHold.GetGlobalPutCount()!=lastPutCount) {
				GetSystemSetting(SYSSET_CLEAR_UNDO);
				}
		} else if ( node0 ) {
			 // Get rid of the references.
			DeleteAllRefsFromMe();
			}
		theMaster = NULL; //JH 9/15/97
		}	
		FUNCEND("ParticleMasterCreationManager::End");
}

RefResult ParticleMasterCreationManager::NotifyRefChanged(
	Interval changeInt,
	RefTargetHandle hTarget, 
	PartID& partID,  
	RefMessage message) 
	{
	FUNCSTART("ParticleMasterCreationManager::NotifyRefChanged");
	switch (message) {
		case REFMSG_TARGET_SELECTIONCHANGE:
		 	if ( ignoreSelectionChange ) {
				break;
				}
		 	if (theMaster) {
				// this will set node0 ==NULL;
				DeleteAllRefsFromMe();
				goto endEdit;
				}
			else
				{
					FUNCEND("ParticleMasterCreationManager::NotifyRefChanged");
					return REF_SUCCEED;  //JH 9.15.97 
				}
			// fall through

		case REFMSG_TARGET_DELETED:
			if (theMaster) {
				endEdit:
				theMaster->EndEditParams( (IObjParam*)createInterface, FALSE/*destroy*/,NULL );
				theMaster = NULL;
				node0 = NULL;
				CreateNewMaster();	
				attachedToNode = FALSE;
				}
			break;		
		}
	{
		FUNCEND("ParticleMasterCreationManager::NotifyRefChanged");
		return REF_SUCCEED;
	}
		FUNCEND("ParticleMasterCreationManager::NotifyRefChanged");
}

void ParticleMasterCreationManager::CreateNewMaster()
	{
	FUNCSTART("ParticleMasterCreationManager::CreateNewMaster");
	theMaster = new ParticleMaster();	
	
	// Start the edit params process
	theMaster->BeginEditParams( (IObjParam*)createInterface, BEGIN_EDIT_CREATE,NULL );
	lastPutCount = theHold.GetGlobalPutCount();
		FUNCEND("ParticleMasterCreationManager::CreateNewMaster");
}

#define BOXSZ 20.0f

static BOOL needToss;

int ParticleMasterCreationManager::proc( 
				HWND hwnd,
				int msg,
				int point,
				int flag,
				IPoint2 m )
	{	
	FUNCSTART("ParticleMasterCreationManager::proc");
	int res;
	INode *newNode[2],*startNode;	
	float r;
	ViewExp *vpx = createInterface->GetViewport(hwnd); 
	assert( vpx );


	switch ( msg ) {
		case MOUSE_POINT:
				{
				if (point==0) {
					pt0 = m;	

					assert(theMaster);

					// Clear the current set as not to confuse the balance check
					gInterface->ClearNodeSelection();

					mat.IdentityMatrix();
					if ( createInterface->SetActiveViewport(hwnd) ) {
						{
							FUNCEND("ParticleMasterCreationManager::proc");
							return FALSE;
						}
						}
					if (createInterface->IsCPEdgeOnInView()) { 
						{
							FUNCEND("ParticleMasterCreationManager::proc");
							return FALSE;
						}
						}
					if ( attachedToNode ) {
				   		// send this one on its way
				   		theMaster->EndEditParams( (IObjParam*)createInterface,0,NULL );
						
						// Get rid of the references.
						DeleteAllRefsFromMe();

						// new object
						CreateNewMaster();   // creates theMaster
						}

					needToss = theHold.GetGlobalPutCount()!=lastPutCount;

				   	theHold.Begin();	 // begin hold for undo
					mat.IdentityMatrix();
					center = vpx->SnapPoint(m,m,NULL,SNAP_IN_PLANE);
					mat.SetTrans(center);


					// Create a dummy object & node
					GenBoxObject *start_box = (GenBoxObject *)createInterface->
						CreateInstance(GEOMOBJECT_CLASS_ID,PARTICLE_BOX_CLASS_ID);//Class_ID(BOXOBJ_CLASS_ID,0));
					assert( start_box );
					startNode = createInterface->CreateObjectNode(start_box);
					theMaster->SetMasterNode( startNode );
					startNode->SetWireColor( RGB( 0, 255, 0 ));

					CStr props = CStr( "Class = ParticleObject\r\nType = Default\r\n" );
					//props += pPropEdit->GetClassTypeRecord( "ParticleObject", "Default" );
					props += pPropEdit->GetUniqueClassTypeRecord( "ParticleObject", "Default" );
					startNode->SetUserPropBuffer( props );					

					FILE* fpOut = fopen("c:\\blah.txt", "w");
					fprintf(fpOut, props);
					fclose(fpOut);

					start_box->SetParams(BOXSZ,BOXSZ,BOXSZ,1,1,1,FALSE); 
					start_box->MakeRefByID( FOREVER, PARTICLE_MASTER, theMaster );
					
					// Make two extra boxes to represent the mid and end spreads
					theMaster->ignore_controller_updates = true;
					for (int i=0; i<theMaster->numNodes; i++) 
					{
						// make a box object
						GenBoxObject *ob = (GenBoxObject *)createInterface->
						CreateInstance(GEOMOBJECT_CLASS_ID,PARTICLE_BOX_CLASS_ID);//Class_ID(BOXOBJ_CLASS_ID,0));
						ob->SetParams(BOXSZ,BOXSZ,BOXSZ,1,1,1,FALSE); 
												
						start_box->MakeRefByID( FOREVER, MID_BOX_REF + i, ob );						
						newNode[i] = createInterface->CreateObjectNode(ob);

						if( i == 0 )
						{
							newNode[i]->SetWireColor( RGB( 255, 255, 0 ));
						}
						else
						{
							newNode[i]->SetWireColor( RGB( 255, 0, 0 ));
						}

						IndePosition* pos_ctrl = new IndePosition( false, theMaster, i );
						theMaster->MakeRefByID( FOREVER, PM_REF_MID_CTRL + i, pos_ctrl );
						newNode[i]->GetTMController()->SetPositionController( pos_ctrl );
						LocalEulerRotation* rot_ctrl = new LocalEulerRotation;
						theMaster->MakeRefByID( FOREVER, PM_REF_MID_ROT_CTRL + i, rot_ctrl );
						newNode[i]->GetTMController()->SetRotationController( rot_ctrl );
												
						startNode->AttachChild(newNode[i]);						
					}					

					theMaster->ignore_controller_updates = false;
					theMaster->RecalculatePositions( 0 );
					// select the dummy node.
					attachedToNode = TRUE;

					// Reference the node so we'll get notifications.
					mat.SetTrans(vpx->SnapPoint(m,m,NULL,SNAP_IN_PLANE));
					createInterface->SetNodeTMRelConstPlane(startNode, mat);
					res = TRUE;			
					}
				else {
					// select a node so if go into modify branch, see params 
					ignoreSelectionChange = TRUE;
				   	createInterface->SelectNode( theMaster->GetMasterNode());
					ignoreSelectionChange = FALSE;
					theHold.Accept(IDS_DS_CREATE);
					res = FALSE;
					}
 				createInterface->RedrawViews(createInterface->GetTime(),REDRAW_NORMAL,theMaster);  
				}
				break;
		case MOUSE_MOVE:
			if (node0) {
				r = (float)fabs(vpx->SnapLength(vpx->GetCPDisp(center,Point3(0,1,0),pt0,m)));
				/*theMaster->SetRad(0,r);
				theMaster->radSpin->SetValue(r, FALSE );*/
				createInterface->RedrawViews(createInterface->GetTime(),REDRAW_NORMAL,theMaster);
				}
			res = TRUE;
			break;

// mjm - 3.2.99 - begin
		case MOUSE_FREEMOVE:
			SetCursor( LoadCursor( hInstance, MAKEINTRESOURCE(IDC_CROSS_HAIR) ) );
			break;
// mjm - end

		case MOUSE_PROPCLICK:
			// right click while between creations
			createInterface->RemoveMode(NULL);
			break;

		case MOUSE_ABORT:
			assert(theMaster);
			theMaster->EndEditParams( (IObjParam*)createInterface, 0,NULL );
			theHold.Cancel();  // undo the changes
			// DS 8/21/97: If something has been put on the undo stack since this object was created, we have to flush the undo stack.
			if (needToss) 
				GetSystemSetting(SYSSET_CLEAR_UNDO);
			DeleteAllRefsFromMe();
			CreateNewMaster();	
			createInterface->RedrawViews(createInterface->GetTime(),REDRAW_END,theMaster); 
			attachedToNode = FALSE;
			res = FALSE;						
			break;
		}
	
	createInterface->ReleaseViewport(vpx); 
	{
		FUNCEND("ParticleMasterCreationManager::proc");
		return res;
	}
		FUNCEND("ParticleMasterCreationManager::proc");
}

int ParticleMasterClassDesc::BeginCreate(Interface *i)
	{
	FUNCSTART("ParticleMasterClassDesc::BeginCreate");
	SuspendSetKeyMode();
	IObjCreate *iob = i->GetIObjCreate();
	
	theParticleMasterCreateMode.Begin( iob, this );
	iob->PushCommandMode( &theParticleMasterCreateMode );
	
	{
		FUNCEND("ParticleMasterClassDesc::BeginCreate");
		return TRUE;
	}
		FUNCEND("ParticleMasterClassDesc::BeginCreate");
}

int ParticleMasterClassDesc::EndCreate(Interface *i)
	{
	FUNCSTART("ParticleMasterClassDesc::EndCreate");
	ResumeSetKeyMode();
	theParticleMasterCreateMode.End();
	i->RemoveMode( &theParticleMasterCreateMode );
	{
		FUNCEND("ParticleMasterClassDesc::EndCreate");
		return TRUE;
	}
		FUNCEND("ParticleMasterClassDesc::EndCreate");
}

int		GetParticlePropertyIndex( CStr property )
{
	FUNCSTART("GetParticlePropertyIndex");
	int i;

	for( i = 0; i < vNUM_PROPERTIES; i++ )
	{
		if(	( stricmp( property, ParticleProperties[i] ) == 0 ))
		{
			{
				FUNCEND("GetParticlePropertyIndex");
				return i;
			}
		}
	}

	{
		FUNCEND("GetParticlePropertyIndex");
		return -1;
	}
	FUNCEND("GetParticlePropertyIndex");
}

void	HandleParticlePropertyChange( IParticleBox* box, int prop_index, CStr prop_val )
{
	FUNCSTART("HandleParticlePropertyChange");
	IParticleMaster* particle_master;

	particle_master = box->GetParticleMaster();

	switch( prop_index )
	{
		case vLIFETIME:
			particle_master->SetLifetime( 0, atof( prop_val ), false );
			break;
		case vMID_TIME_PCT:
			particle_master->SetMidtimePCT( 0, atof( prop_val ), false);
			break;
		case vACCELERATION_X:
			particle_master->SetAccelerationX( 0, atof( prop_val ), false);
			break;
		case vACCELERATION_Y:
			particle_master->SetAccelerationY( 0, atof( prop_val ), false);
			break;
		case vACCELERATION_Z:
			particle_master->SetAccelerationZ( 0, atof( prop_val ), false);
			break;
		case vVEL_X:
			particle_master->SetVelocityX( 0, atof( prop_val ), false);
			break;
		case vVEL_Y:
			particle_master->SetVelocityY( 0, atof( prop_val ), false);
			break;
		case vVEL_Z:
			particle_master->SetVelocityZ( 0, atof( prop_val ), false);
			break;
		case vUSE_MIDPOINT:
			INode* mid_node;

			mid_node = box->GetMidBox()->GetObjectNode();//particle_master->GetSlaveNode( 0 );
			if( mid_node )
			{
				if( stricmp( prop_val, "TRUE" ) == 0 )
				{
					mid_node->SetWireColor( RGB( 255, 255, 255 ));
				}
				else
				{
					mid_node->SetWireColor( RGB( 0, 0, 0 ));
				}
				gInterface->ForceCompleteRedraw();
			}
			break;
	}
	FUNCEND("HandleParticlePropertyChange");
}

void ResetValidity()
{
	int nNodeCount = gInterface->GetSelNodeCount();
	int i;

	for(i = 0; i < nNodeCount; i++)
	{
		INode* node = gInterface->GetSelNode(i);
		Object* obj = node->EvalWorldState(0).obj;

		if (obj && obj->ClassID() == PARTICLE_BOX_CLASS_ID)
		{
			IParticleBox* pbox    = dynamic_cast<IParticleBox*>(obj);

			if (!pbox)
				continue;

			IParticleBox* pboxMid = pbox->GetMidBox();
			IParticleBox* pboxEnd = pbox->GetEndBox();

			pbox->SetValid(TRUE);

			if (pboxMid)
				pboxMid->SetValid(TRUE);

			if (pboxEnd)
				pboxEnd->SetValid(TRUE);
		}
	}
}

static  void particle_cleanup(void *param, NotifyInfo *info)
{
	OutputDebugString("HANDLER: particle_cleanup\n");

	INode* node = (INode*)info->callParam;
	Object* obj = node->EvalWorldState(0).obj;
	
	if (obj && obj->ClassID() == PARTICLE_BOX_CLASS_ID)
	{
		IParticleBox* pbox = dynamic_cast<IParticleBox*>(obj);

		if (!pbox->IsValid())
		{
			MessageBox(gInterface->GetMAXHWnd(), "You have only copied a partial particle system.  This will NOT export.  You should delete this system and clone the entire system (all 3 boxes) if you want to make a copy.", "Unbalanced Particle System Clone", MB_ICONWARNING|MB_OK);
			ResetValidity();

			CStr name = node->GetName();
			pbox->DetachAssociations();
		}
	}
}

static	void particle_sel_change(void *param,NotifyInfo *info)
{
	FUNCSTART("particle_sel_change");
	int i, j;
	int num_sel;
	INode* root;
	
	num_sel = gInterface->GetSelNodeCount();
	root = gInterface->GetRootNode();

	UnregisterParticleNotifications();

	for( i = 0; i < num_sel; i++ )
	{
		INode* child, *node, *parent;
		int num_children;
		
		node = gInterface->GetSelNode( i );		
		Object* obj=node->EvalWorldState(0).obj;
		if( obj->ClassID()==PARTICLE_BOX_CLASS_ID)
		{
			if( node->GetParentNode() == root )
			{
				parent = node;
			}
			else
			{
				parent = node->GetParentNode();
			}

			if( parent->Selected())
			{
				num_children = parent->NumChildren();
				for( j = 0; j < num_children; j++ )
				{
					child = parent->GetChildNode( j );
					if( child->Selected() == false )
					{
						gInterface->SelectNode( child, FALSE );
					}
				}
			}
			else
			{
				num_children = parent->NumChildren();
				for( j = 0; j < num_children; j++ )
				{
					child = parent->GetChildNode( j );
					if( child->Selected())
					{
						gInterface->DeSelectNode( child );
					}
				}
			}			
		}
	}

	RegisterParticleNotifications();
	FUNCEND("particle_sel_change");
}

static	void particle_sel_pre_delete(void *param,NotifyInfo *info)
{	
	OutputDebugString("HANDLER: particle_sel_pre_delete\n");
	FUNCSTART("particle_sel_pre_delete");

	if (bParticleDelLock)
		return;

	bParticleDelLock = true;

	Tab< INode* > &sel_nodes = *( Tab<INode*> *) info->callParam;
	int i, j;
	int num_sel;
	INode* root;
	Tab< INode* > del_list;
	
	num_sel = sel_nodes.Count();
	root = gInterface->GetRootNode();

	for( i = 0; i < num_sel; i++ )
	{
		INode* child, *node, *parent;
		int num_children;
		
		node = sel_nodes[i];
		CStr name = node->GetName();

		Object* obj=node->EvalWorldState(0).obj;
		if( obj->ClassID()==PARTICLE_BOX_CLASS_ID)
		{
			IParticleBox* pbox = dynamic_cast<IParticleBox*>(obj);
			if (!pbox->IsValid())
				continue;

			// Abort if node properties are detached
			if (pbox->IsDetached())
			{
				bParticleDelLock = false;
				return;
			}

			if( node->GetParentNode() == root )
			{
				parent = node;
			}
			else
			{
				parent = node->GetParentNode();
			}

			num_children = parent->NumChildren();
			for( j = 0; j < num_children; j++ )
			{
				child = parent->GetChildNode( j );
				if( child->Selected() == false )
				{
					del_list.Append( 1, &child );					
				}
			}			

			if( parent->Selected() == false )
			{
				del_list.Append( 1, &parent );
			}			
		}
	}	

	// Prevent MAX from doing a double delete but, don't screw with
	// deletion for any collateral nodes not associated with the system
	// aml 7-10-03
	for( i = 0; i < sel_nodes.Count(); i++ )
	{
		for(j = 0; j < del_list.Count(); j++)
		{
			if (sel_nodes[i] == del_list[j])
			{
				sel_nodes.Delete(i, 1);
				i = -1;
				break;
			}
		}
	}

	// Ensure we don't delete any duplicates that exist within our same delete list
	INodeTab processedNodes;

	for( i = 0; i < del_list.Count(); i++ )
	{
		if (!NodeExistsInList(processedNodes, del_list[i]))
		{
			processedNodes.Append(1, &del_list[i]);
			gInterface->DeleteNode( del_list[i] );
		}
	}

	bParticleDelLock = false;

	FUNCEND("particle_sel_pre_delete");
}

void	RegisterParticleNotifications( void )
{
	FUNCSTART("RegisterParticleNotifications");
#ifndef DISABLE_NOTIFICATIONS
	RegisterNotification( particle_sel_pre_delete, NULL, NOTIFY_SEL_NODES_PRE_DELETE );
	RegisterNotification( particle_cleanup, NULL, NOTIFY_SCENE_ADDED_NODE );
#endif
	FUNCEND("RegisterParticleNotifications");
}

void	UnregisterParticleNotifications( void )
{
	FUNCSTART("UnregisterParticleNotifications");
	UnRegisterNotification( particle_sel_pre_delete, NULL, NOTIFY_SEL_NODES_PRE_DELETE );	
	UnRegisterNotification( particle_cleanup, NULL, NOTIFY_SCENE_ADDED_NODE );
	FUNCEND("UnregisterParticleNotifications");
}
