/*
	NodeCRC.cpp
	Functions for determining when node data has changed
	and needs updating to speed script export
*/

#include "NodeCRC.h"
#include "misc/gencrc.h"
#include "path.h"
#include "next.h"
#include "appdata.h"
#include "UserConfig.h"

#define NODECRC_VERSION 0x0001

FILETIME gScriptsTime;
FILETIME gPluginTime;

void InitNodeCRCFileTimes()
{
	// Stamp the time of scripts.ini into NodeCRC
	//CStr fname = getenv(APP_ENV);
	//fname += CStr(SCRIPT_PATH) + SCRIPT_INI;

	CStr fname = GetScriptsIniPath();
	
	HANDLE hFile = CreateFile(fname,
		                      GENERIC_READ,
							  0,
							  NULL,
							  OPEN_EXISTING,
							  0,
							  NULL);

	if (hFile)
	{
		GetFileTime(hFile, NULL, NULL, &gScriptsTime);
		CloseHandle(hFile);
	}
	else
	{
		gScriptsTime.dwLowDateTime  = 0;
		gScriptsTime.dwHighDateTime = 0;
	}

	// Stamp the time of the plugin build into the NodeCRC
	char filename[1024];
	
	HMODULE hModule = GetModuleHandle("next.gup");
	GetModuleFileName(hModule, filename, 1023);

	hFile = CreateFile(filename,
		               GENERIC_READ,
					   0,
					   NULL,
					   OPEN_EXISTING,
					   0,
					   NULL);

	if (hFile)
	{
		GetFileTime(hFile, NULL, NULL, &gPluginTime);
		CloseHandle(hFile);
	}
	else
	{
		gPluginTime.dwLowDateTime  = 0;
		gPluginTime.dwHighDateTime = 0;
	}
}

class NodeCRCDepEnumProc: public DependentEnumProc
{
	int proc(ReferenceMaker* rmaker);

public:
	Tab<ReferenceMaker*> refList;
};

int NodeCRCDepEnumProc::proc(ReferenceMaker* rmaker)
{
	refList.Append(1, &rmaker);

	return DEP_ENUM_CONTINUE;
}

NodeCRC::NodeCRC()
{
	nameCRC     = 0;
	propBufCRC  = 0;
	propBufCRC2 = 0;
	//refSum    = 0;

	TM.IdentityMatrix();

	scriptsTime.dwLowDateTime  = 0;
	scriptsTime.dwHighDateTime = 0;
	
	pluginTime.dwLowDateTime  = 0;
	pluginTime.dwHighDateTime = 0;

	nodeFlags = 0;
}

NodeCRC::NodeCRC(INode* node)
{
	CStr propBuf;
	NodeCRCDepEnumProc ncrcDepEnumerator;

	node->GetUserPropBuffer(propBuf);

	nameCRC    = GenerateCRC(node->GetName());
	propBufCRC = GenerateCRC(propBuf);
	propBufCRC2 = GenAdditiveCRC(propBuf);

	// Build up checksum for references
	/*
	refSum = 0;

	node->EnumDependents(&ncrcDepEnumerator);

	int nrefs = ncrcDepEnumerator.refList.Count();

	for(int i = 0; i < nrefs; i++)
	{
		if (ncrcDepEnumerator.refList[i]->SuperClassID() == BASENODE_CLASS_ID)
			refSum ^= ((INode*)ncrcDepEnumerator.refList[i])->GetHandle();
	}
	*/

	TM = node->GetNodeTM(0);

	// Update the time stamps
	scriptsTime = gScriptsTime;
	pluginTime  = gPluginTime;

	nodeFlags = 0;
}

NodeCRC::NodeCRC(NodeCRC& right)
{
	nameCRC     = right.nameCRC;
	propBufCRC  = right.propBufCRC;
	propBufCRC2 = right.propBufCRC2;
	//refSum      = right.refSum;

	TM          = right.TM;
	scriptsTime = right.scriptsTime;
	pluginTime  = right.pluginTime;

	nodeFlags   = right.nodeFlags;
}

NodeCRC& NodeCRC::operator =(NodeCRC& right)
{
	nameCRC     = right.nameCRC;
	propBufCRC  = right.propBufCRC;
	propBufCRC2 = right.propBufCRC2;
	//refSum      = right.refSum;

	TM          = right.TM;
	scriptsTime = right.scriptsTime;
	pluginTime  = right.pluginTime;

	nodeFlags   = right.nodeFlags;

	return *this;
}

int NodeCRC::operator ==(NodeCRC& right)
{
	if (nameCRC                    == right.nameCRC              &&
		propBufCRC                 == right.propBufCRC           &&
		propBufCRC2                == right.propBufCRC2          &&
		//refSum                     == right.refSum               &&
		TM                         == right.TM                    &&
		scriptsTime.dwLowDateTime  == gScriptsTime.dwLowDateTime  &&
		scriptsTime.dwHighDateTime == gScriptsTime.dwHighDateTime &&
		pluginTime.dwLowDateTime   == gPluginTime.dwLowDateTime   &&
		pluginTime.dwHighDateTime  == gPluginTime.dwHighDateTime)
		return true;

	return false;
}

struct NodeCRCAppData
{
	unsigned int version;
	NodeCRC      nodedata;
};

void NodeCRC::Store(INode* node)
{
	assert(node);

	// Dump the version of the CRC incase we decide to add more check fields later
	AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CRC);
	NodeCRCAppData*  nodeCRCMem;

	if (!appdata)
	{
		nodeCRCMem = (NodeCRCAppData*)malloc(sizeof(NodeCRCAppData));
		node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CRC, sizeof(NodeCRCAppData), nodeCRCMem);
	}
	else
	{
		node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CRC);
		nodeCRCMem = (NodeCRCAppData*)malloc(sizeof(NodeCRCAppData));
		node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CRC, sizeof(NodeCRCAppData), nodeCRCMem);
	}

	nodeCRCMem->version  = NODECRC_VERSION;
	nodeCRCMem->nodedata = *this;
}

bool NodeCRC::Retrieve(INode* node)
{
	assert(node);

	AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CRC);
	
	if (appdata && appdata->data)
	{
		// Ensure this is the proper version
		if (appdata->length != sizeof(NodeCRCAppData))
		{
			node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CRC);
			return false;
		}

		NodeCRCAppData* nodeCRCMem = (NodeCRCAppData*)appdata->data;

		if (nodeCRCMem->version == NODECRC_VERSION)
			*this = nodeCRCMem->nodedata;

		return true;
	}

	return false;
}

void AssignNodeOutput(INode* node, NodeCRC* ncrc, char* buf, char* bufScript)
{
	AppDataChunk* appdata;
	bool bAllocNCRC = false;
	char* newbuf;

	assert(node);
	assert(buf);

	if (!ncrc)
	{
		ncrc = new NodeCRC(node);
		bAllocNCRC = true;
	}

	ncrc->Store(node);

	appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTBUFFER);

	if (!appdata)
	{
		newbuf = (char*)malloc(strlen(buf) + 1);
		node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTBUFFER, strlen(buf) + 1, newbuf);
	}
	else
	{
		node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTBUFFER);
		newbuf = (char*)malloc(strlen(buf) + 1);
		node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTBUFFER, strlen(buf) + 1, newbuf);
	}

	strcpy(newbuf, buf);

	appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTSCRIPTBUFFER);

	if (!appdata)
	{
		newbuf = (char*)malloc(strlen(bufScript) + 1);
		node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTSCRIPTBUFFER, strlen(bufScript) + 1, newbuf);
	}
	else
	{
		node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTSCRIPTBUFFER);
		newbuf = (char*)malloc(strlen(bufScript) + 1);
		node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTSCRIPTBUFFER, strlen(bufScript) + 1, newbuf);
	}

	strcpy(newbuf, bufScript);

	if (bAllocNCRC)
		delete ncrc;
}

bool GetNodeOutput(INode* node, NodeCRC* ncrc, CStr& out)
{
	AppDataChunk* appdata;

	appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CRC);

	if (!appdata || !appdata->data)
		return false;

	NodeCRCAppData* nodeCRCMem = (NodeCRCAppData*)appdata->data;

	switch(nodeCRCMem->version)
	{
	case NODECRC_VERSION:
		{
			if (!ncrc || nodeCRCMem->nodedata == *ncrc)
			{
				appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTBUFFER);

				if (!appdata || !appdata->data)
					return false;

				out = (char*)appdata->data;
				return true;
			}
		}
		break;

	default:
		return false;
	}

	return false;
}

bool GetNodeOutputScript(INode* node, NodeCRC* ncrc, CStr& out)
{
	AppDataChunk* appdata;

	appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CRC);

	if (!appdata || !appdata->data)
		return false;

	NodeCRCAppData* nodeCRCMem = (NodeCRCAppData*)appdata->data;

	switch(nodeCRCMem->version)
	{
	case NODECRC_VERSION:
		{
			if (!ncrc || nodeCRCMem->nodedata == *ncrc)
			{
				appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTSCRIPTBUFFER);

				if (!appdata || !appdata->data)
					return false;

				out = (char*)appdata->data;
				return true;
			}
		}
		break;

	default:
		return false;
	}

	return false;
}

unsigned long GenAdditiveCRC(char* buf)
{
	unsigned int len = strlen(buf);
	unsigned int crc = 0;

	for(int i = 0; i < len; i++)
		crc += buf[i];

	return crc;
}
