/*
	MaxUtil.cpp
	Commonly used utility functions related to max
	aml - 10-17-02
*/

#include "maxutil.h"
#include "modstack.h"
#include <trigger/trigger.h>

extern Interface* gInterface;

bool NodeExistsInSel(INode* checkNode)
{
	int nSelCount = gInterface->GetSelNodeCount();

	for(int i = 0; i < nSelCount; i++)
	{
		INode* node = gInterface->GetSelNode(i);

		if (node == checkNode)
			return true;
	}

	return false;
}

void GetAllNodes(Tab<INode*>& nodes, INode* root)
{
	int kids = root->NumberOfChildren();
	
	for(int i=0;i<kids;i++)
	{
		INode* node = root->GetChildNode(i);
		GetAllNodes(nodes,node);
	}

	if (root != gInterface->GetRootNode())
		nodes.Append(1,&root);
}

void GetAllNodes(Tab<INode*>& nodes)
{ 
	GetAllNodes(nodes, gInterface->GetRootNode()); 
}

int GetNodeCountInt(INode* node)
{
	int kids = node->NumberOfChildren();
	int total = 0;

	for(int i = 0; i < kids; i++)
	{
		INode* subnode = node->GetChildNode(i);
		total += GetNodeCountInt(subnode);
	}

	return total + 1;
}

int GetNodeCount()
{
	INode* node = gInterface->GetRootNode();
	return GetNodeCountInt(node);
}

void GetSelSetNodes(INodeTab& nodes)
{
	int count = gInterface->GetSelNodeCount();

	for(int i=0;i<count;i++)
	{
		INode* node = gInterface->GetSelNode(i);
		nodes.Append(1, &node);
	}
}

ClassDesc* GetClassDesc(SClass_ID scid, Class_ID cid)
{
	DllDir& dlldir = gInterface->GetDllDir();
	ClassDirectory& classdir = dlldir.ClassDir();

	return classdir.FindClass(scid, cid);
}

Modifier* FindModifier(Object* obj, Class_ID cid)
{
	if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		IDerivedObject* dobj = (IDerivedObject*)obj;

		int nMods = dobj->NumModifiers();

		for(int j = 0; j < nMods; j++)
		{
			Modifier* mod = dobj->GetModifier(j);

			// Check if this modifier is the modifier we want
			if (mod->ClassID() == cid)
				return mod;
		}

		Object* objref = dobj->GetObjRef();
		
		if (objref)
			return FindModifier(objref, cid);
	}

	return NULL;
}

Modifier* FindModifier(INode* node, Class_ID cid)
{
	Object* obj = node->GetObjectRef();
	return FindModifier(obj, cid);
}

void RemapChannel(Mesh& mesh, int srcChannel, int destChannel)
{
	// Temporarily disabled until state of MultiRes code is determined
	return;
	
	UVVert* srcverts  = mesh.mapVerts(srcChannel);
	TVFace* srcfaces  = mesh.mapFaces(srcChannel);

	int nVerts = mesh.getNumMapVerts(srcChannel);

	// Ensure that the destination channel is initialized
	mesh.setMapSupport(destChannel);
	mesh.setNumMapVerts(destChannel, nVerts);

	UVVert* destverts = mesh.mapVerts(destChannel);
	TVFace* destfaces = mesh.mapFaces(destChannel);

	// Assign the channel data
	memcpy(destverts, srcverts, sizeof(UVVert) * nVerts);
	memcpy(destfaces, srcfaces, sizeof(TVFace) * mesh.numFaces);
}

void CopyChannelFrom(UVChannel& channel, Mesh& mesh, int chanID)
{
	// Temporarily disabled until state of MultiRes code is determined
	return;

	UVVert* srcverts = mesh.mapVerts(chanID);
	TVFace* srcfaces = mesh.mapFaces(chanID);

	if (srcverts && srcfaces)
	{
		int nVerts = mesh.getNumMapVerts(chanID);

		channel.setVerts(nVerts);
		channel.setFaces(mesh.numFaces);

		memcpy(channel.verts, srcverts, sizeof(UVVert) * nVerts);
		memcpy(channel.faces, srcfaces, sizeof(TVFace) * mesh.numFaces);
	}
}

void CopyChannelTo(UVChannel& channel, Mesh& mesh, int chanID)
{
	// Temporarily disabled until state of MultiRes code is determined
	return;

	if (channel.verts && channel.faces)
	{
		mesh.setMapSupport(chanID);
		mesh.setNumMapVerts(chanID, channel.nVerts);

		UVVert* destverts = mesh.mapVerts(chanID);
		TVFace* destfaces = mesh.mapFaces(chanID);

		memcpy(destverts, channel.verts, sizeof(UVVert) * channel.nVerts);
		memcpy(destfaces, channel.faces, sizeof(TVFace) * mesh.numFaces);
	}
}

// There seem to be some odd quirks in R5 with calling GetSceneMtls() where
// it doesn't always update.  Using the below to force locate a specific scene
// material by manually scanning all the scene nodes
Mtl* ForceFindSceneMtlByName(CStr name)
{
	INodeTab nodes;
	GetAllNodes(nodes);

	int count = nodes.Count();

	for(int i = 0; i < count; i++)
	{
		Mtl* nodemtl = nodes[i]->GetMtl();

		if (nodemtl)
		{
			if (nodemtl->GetName() == name)
				return nodemtl;
		}
	}

	return NULL;
}

bool VerifyParentExists(INode* nodeParent, INode* node)
{
	if (!node || !nodeParent)
		return false;

	if (node == nodeParent)
		return true;

	INode* root = gInterface->GetRootNode();

	while(node != root)
	{
		if (node == nodeParent)
			return true;

		node = node->GetParentNode();
	}

	return false;
}

Point3 GetNodePosition(INode* node)
{
	Matrix3 mat;
	Point3  pos;
	float   tmp;

	mat=node->GetNodeTM(0);		// Get node's transform matrix at time 0
	pos=mat.GetTrans();
	tmp=pos.z;
	pos.z=pos.y;
	pos.y=tmp;

	return pos;
}

bool	boxes_overlap( Box3& box1, Box3& box2 )
{
	if(	( box2.pmin.x > box1.pmax.x ) ||
		( box2.pmin.y > box1.pmax.y ) ||
		( box2.pmin.z > box1.pmax.z ) ||
		( box1.pmin.x > box2.pmax.x ) ||
		( box1.pmin.y > box2.pmax.y ) ||
		( box1.pmin.z > box2.pmax.z ))		
	{
		return false;
	}
	return true;
}

bool Overlapping( INode* node1, INode* node2, float obj_tolerance, float node_tolerance )
{
	bool tri_objects;
	Object* obj1, *obj2;
	
	tri_objects = false;
	obj1 = node1->EvalWorldState(0).obj;
	obj2 = node2->EvalWorldState(0).obj;
	if( obj1->CanConvertToType( triObjectClassID ))
	{
		if( !obj2->CanConvertToType( triObjectClassID ))
		{
			return false;
		}
		tri_objects = true;
	}
	else if( obj2->CanConvertToType( triObjectClassID ))
	{
		return false;
	}

	if( tri_objects )
	{
		Box3 box1, box2;
		Point3 width1, width2;
		Matrix3 tm;
		float num, den, ratio;
	
		tm = node1->GetObjTMAfterWSM(0);
		GetObjectExtents( node1, tm, box1.pmin, box1.pmax );

		tm = node2->GetObjTMAfterWSM(0);
		GetObjectExtents( node2, tm, box2.pmin, box2.pmax );

		// If the objects' bounding boxes don't intersect, don't even test
		// against the threshold
		if( boxes_overlap( box1, box2 ) == false )
		{
			return false;
		}		

		width1 = box1.Width();
		width2 = box2.Width();
		
		num = ( width1.Length() < width2.Length() ) ? width1.Length() : width2.Length();
		den = ( width1.Length() > width2.Length() ) ? width1.Length() : width2.Length();
		ratio = num / den;
		if(( 1.0f - ratio ) > obj_tolerance )
		{
			return false;
		}

		return true;
	}
	else
	{
		if( obj1->ClassID() == vTRIGGER_CLASS_ID )
		{
			Point3 pos1, pos2;
			float dist;

			if( obj2->ClassID() != vTRIGGER_CLASS_ID )
			{
				return false;
			}

			pos1 = GetNodePosition( node1 );
			pos2 = GetNodePosition( node2 );
			dist = (pos1 - pos2).Length();
			if( dist > node_tolerance )
			{
				return false;
			}
			
			return true;
		}
		else if( obj2->ClassID() == vTRIGGER_CLASS_ID )
		{
			return false;
		}		
	}

	return false;
}

void CheckForOverlappingNodes( float obj_tolerance, float node_tolerance )
{
	int i, j, num_sets;
	INodeTab new_nodes, old_nodes, overlapping_old, overlapping_new;
	bool found;
	Interface* ip;

	ip = GetCOREInterface();
	
	found = false;
	num_sets = ip->GetNumNamedSelSets();
	for( i = 0; i < num_sets; i++ )
	{
		TSTR set_name;

		set_name = ip->GetNamedSelSetName( i );
		if( strcmp( set_name, "MERGE - NEW" ) == 0)
		{
			found = true;
			ip->GetNamedSelSetList( new_nodes, i );
			break;
		}
	}
	if( !found )
	{
		return;
	}

	found = false;
	for( i = 0; i < num_sets; i++ )
	{
		TSTR set_name;

		set_name = ip->GetNamedSelSetName( i );
		if( strcmp( set_name, "MERGE - OLD" ) == 0)
		{
			found = true;
			ip->GetNamedSelSetList( old_nodes, i );
			break;
		}
	}
	if( !found )
	{
		return;
	}

	for( i = 0; i < new_nodes.Count(); i++ )
	{
		INode* new_node, *old_node;
		
		new_node = new_nodes[i];
		for( j = 0; j < old_nodes.Count(); j++ )
		{
			old_node = old_nodes[j];
			if( Overlapping( new_node, old_node, obj_tolerance, node_tolerance ))
			{
				overlapping_old.Append( 1, &old_node );
				overlapping_new.Append( 1, &new_node );
			}
		}		
	}

	ip->RemoveNamedSelSet(CStr("OVERLAPPING - OLD"));
	if( overlapping_old.Count() > 0 )
	{
		
		ip->AddNewNamedSelSet(overlapping_old,CStr("OVERLAPPING - OLD"));
	}

	ip->RemoveNamedSelSet(CStr("OVERLAPPING - NEW"));
	if( overlapping_new.Count() > 0 )
	{		
		ip->AddNewNamedSelSet(overlapping_new,CStr("OVERLAPPING - NEW"));
	}
}

bool GetObjectExtents(INode* node, Matrix3 tm, Point3& min, Point3& max)
{
	// Convert to a tri object and scan the mesh to determine it's XYZ extents  aml
	Object* obj=node->EvalWorldState(0).obj;

	if (obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID,0)))
	{
		TriObject* triobj=(TriObject*)obj->ConvertToType(0,Class_ID(TRIOBJ_CLASS_ID,0));

		Mesh& mesh=triobj->GetMesh();

		if (mesh.numVerts==0)
			return false;

		// Assign min/max to start
		min=max=mesh.verts[0];

		min=tm*min;
		max=tm*max;
		//Matrix3 tm=node->GetNodeTM(0);

		for(int i=0;i<mesh.numVerts;i++)
		{
			Point3 pt=tm*mesh.verts[i];

			if (pt.x<min.x)
				min.x=pt.x;

			if (pt.y<min.y)
				min.y=pt.y;

			if (pt.z<min.z)
				min.z=pt.z;

			if (pt.x>max.x)
				max.x=pt.x;

			if (pt.y>max.y)
				max.y=pt.y;

			if (pt.z>max.z)
				max.z=pt.z;
		}

		if (obj!=triobj)
			triobj->DeleteThis();

		return true;
	}

	return false;
}

INode* FindNode(CStr name, INode* node)
{
	int    nKids = node->NumberOfChildren();
	INode* rVal;

	for(int i = 0; i < nKids; i++)
	{
		INode* child = node->GetChildNode(i);
		rVal = FindNode(name, child);

		if (rVal)
			return rVal;
	}

	// Make sure check is case insensitive
	CStr nodeName = node->GetName();
	nodeName.toLower();
	name.toLower();

	if (nodeName == name)
		return node;

	return NULL;
}
