/*
	LODMod.cpp
	aml
*/

#include "LODMod.h"
#include "resource.h"
#include "../PropEdit/ParseFuncs.h"
#include "../misc/MaxUtil.h"
#include "modstack.h"

extern HINSTANCE hInstance;
extern Interface* gInterface;

#define vLODMOD_NODELIST_CHUNK   0x0001
#define vLODMOD_NUMLODS_CHUNK    0x0002

static LODModClassDesc gLODModDesc;
ClassDesc2* GetLODModClassDesc() { return &gLODModDesc; }

static LODModClientClassDesc gLODModClientDesc;
ClassDesc2* GetLODModClientClassDesc() { return &gLODModClientDesc; }


///////////// LODMod Implementation

LODMod::LODMod()
{
	hRollup    = NULL;
	pLODList   = NULL;
	numLODs    = 0;
	focusID    = -1;
}

IOResult LODMod::SaveLocalData(ISave *isave, LocalModData *ld) 
{ 
	unsigned long nb = 0;
	int patchID;

	// We'll save the name of the nodes we're attached to here
	// GetRefID allows us to save pointers that can be backpatched to the
	// new memory state when the file is reloaded  aml
	isave->BeginChunk(vLODMOD_NODELIST_CHUNK);
		
		int nCount = mainNodes.Count();
		if (isave->Write(&nCount, sizeof(int), &nb) != IO_OK)
			return IO_ERROR;

		for(int i = 0; i < nCount; i++)
		{
			patchID = isave->GetRefID(mainNodes[i]);
			
			if (isave->Write(&patchID, sizeof(int), &nb) != IO_OK)
				return IO_ERROR;
		}
	
	isave->EndChunk();

	isave->BeginChunk(vLODMOD_NUMLODS_CHUNK);

		if (isave->Write(&numLODs, sizeof(int), &nb) != IO_OK)
			return IO_ERROR;

	isave->EndChunk();
	return IO_OK; 
}  

IOResult LODMod::LoadLocalData(ILoad *iload, LocalModData **pld) 
{ 
	unsigned long nb = 0;

	while(iload->PeekNextChunkID() != 0)
	{
		if (iload->OpenChunk() != IO_OK)
			return IO_ERROR;

		switch(iload->CurChunkID())
		{
		case vLODMOD_NODELIST_CHUNK:
			{
				// Read in the nodelist
				int    nNodes;
				int    patchID;
				INode* node;

				if (iload->Read(&nNodes, sizeof(int), &nb) != IO_OK)
					return IO_ERROR;

				// Read in values to back patch
				for(int i = 0; i < nNodes; i++)
				{
					if (iload->Read(&patchID, sizeof(int), &nb) != IO_OK)
						return IO_ERROR;

					iload->RecordBackpatch(patchID, (void**)&node);
				}
			}
			break;

		case vLODMOD_NUMLODS_CHUNK:
			{
				if (iload->Read(&numLODs, sizeof(int), &nb) != IO_OK)
					return IO_ERROR;
			}
			break;
		}

		if (iload->CloseChunk() != IO_OK)
			return IO_ERROR;
	}

	return IO_OK; 
}  

// This function is meant to repatch the property data after a clone
// from the node's references
void LODMod::SetPropsFromReferences()
{
	ReadLODData();

	for(int i = 0; i < numLODs; i++)
	{
		INode* node = (INode*)GetReference(i);
		
		if (node)
		{
			CStr   name = node->GetName();
			pLODList->SetValue(i * 2, name);
		}
	}

	StoreLODData();
}

void LODMod::UpdateInternalPropsFromReferences(INodeTab& nodeList)
{
	// We need to apply the property update to all the nodes that are associated with us
	int nMainNodes = mainNodes.Count();
	int i;

	for(i = 0; i < nMainNodes; i++)
	{
		LinkList<ConfigProp> cprops;
		CStr                 propBuffer;

		mainNodes[i]->GetUserPropBuffer(propBuffer);
		ParseConfigProps(&cprops, NULL, propBuffer);

		Link<ConfigProp>* link = cprops.GetHead();

		while(link)
		{
			if (strstr(link->data.name,"LOD_Level"))
			{
				// Extract the LOD level #
				int level = 0;
				sscanf(link->data.name, "LOD_Level%i", &level);
				level--;

				if (level > -1)
					AddPropValue(mainNodes[i], "", link->data.name, nodeList[level]->GetName());
			}

			link = link->next;
		}
	}
}

void LODMod::StoreLODData()
{
	// We need to determine the node that we're applied to
	ModContextList mcList;
	INodeTab       nodeList;

	gInterface->GetModContexts(mcList, nodeList);
	int i, j;

	for(i = 0; i < nodeList.Count(); i++)
	{
		// Get the values from the LOD Editors property list
		// and add them to the node
		int nProps = pLODList->NumProps();

		for(j = 0; j < nProps; j++)
		{
			CStr value;
			CStr name  = pLODList->GetName(j);

			pLODList->GetValue(j, value);

			if (value == CStr(""))
				value = "[None]";

			nodeList[i]->SetUserPropString(name, value);
		}
	}
}

// Determines the highest LOD level in parsed config data
int LODMod::GetMAXLODLevels(LinkList<ConfigProp>& cprops)
{
	Link<ConfigProp>* link = cprops.GetHead();
	int maxLevel = 0;

	while(link)
	{
		if (strstr(link->data.name, "LOD_Level"))
		{
			int level = 0;
			sscanf(link->data.name, "LOD_Level%i", &level);
			
			if (level > maxLevel)
				maxLevel = level;
		}

		link = link->next;
	}

	return maxLevel;
}

void LODMod::ReadLODData()
{	
	ModContextList mcList;
	INodeTab       nodeList;

	gInterface->GetModContexts(mcList, nodeList);
	pLODList->Clear();
	refLODs.ZeroCount();

	// Copy the associated nodes into the associated node list
	// we use this to determine the remapped node we need to use when
	// a clone operation is being performed
	mainNodes = nodeList;

	DeleteAllRefsFromMe();
	numLODs = 0;
		
	// Only the first referenced node will be used in the event that LOD data
	// differs with the modifier applied to a group of 2 or more objects

	if (nodeList.Count() > 0)
	{
		LinkList<ConfigProp> cprops;
		CStr                 propBuffer;

		nodeList[0]->GetUserPropBuffer(propBuffer);
		ParseConfigProps(&cprops,NULL,propBuffer);

		// Scan through config props adding
		Link<ConfigProp>* link = cprops.GetHead();

		while(link)
		{
			if (strstr(link->data.name, "LOD_Level"))
			{
				int level = 0;
				sscanf(link->data.name, "LOD_Level%i", &level);

				numLODs++;
				pLODList->AddNode(link->data.name, "The LOD Representation node for the specified LOD level");

				// Rereference this object
				INode* node = gInterface->GetINodeByName(link->data.value);

				if (node)
				{
					int ref = FindRef(node);
					if (ref != 1)
						DeleteReference(ref);

					if (MakeRefByID(FOREVER, numLODs / 2, node) != REF_SUCCEED)
					{
						MessageBox(gInterface->GetMAXHWnd(), "MakeRefByID Failed!", "ChangeCB", MB_ICONWARNING|MB_OK);
					}

					refLODs.Append(1, &node);

					int idx = pLODList->NumProps() - 1;
					pLODList->SetValue(idx, node->GetName());
				}

				// Find the associated LOD_Dist within the config props
				// Since we want the LOD_Level to always be followed by LOD distance
				Link<ConfigProp>* link2 = cprops.GetHead();

				while(link2)
				{
					if (strstr(link2->data.name, "LOD_Dist"))
					{
						int distLevel = 0;
						sscanf(link2->data.name, "LOD_Dist%i", &distLevel);

						if (level == distLevel)
						{
							pLODList->AddSpinEdit(link2->data.name,0,5000,1,"Distances greater than this value will use the model representation in the corresponding slot");

							int idx = pLODList->NumProps() - 1;
							pLODList->SetValue(idx, link2->data.value);
						}
					}

					link2 = link2->next;
				}

			}
			
			link = link->next;
		}

		pLODList->BuildUI();
	}

	/*
	LinkList<ConfigProp> cprops;
	CStr                 propBuffer;
	
	numLODs = 0;

	if (!nodeMaster)
		return;

	nodeMaster->GetUserPropBuffer(propBuffer);
	ParseConfigProps(&cprops,NULL,propBuffer);

	Link<ConfigProp>* curprop = cprops.GetHead();

	// Set the name of the LOD nodeMaster
	plist->Clear();
	plist->AddNode("Master","The master node that contains the LOD system information.  This is LOD level 0");
	plist->SetValue(0,nodeMaster->GetName());

	while(curprop)
	{
		if (IsInstr(curprop->data.name,"LOD_Level"))
		{
			plist->AddNode(curprop->data.name,"The LOD Representation node for the specified LOD level");
			plist->SetValue(plist->GetCurIndex(),curprop->data.value);
			numLODs++;
		}

		if (IsInstr(curprop->data.name,"LOD_Dist"))
		{
			plist->AddSpinEdit(curprop->data.name,0,5000,1,"Distances greater than this value will use the model representation in the corresponding slot");
			plist->SetValue(plist->GetCurIndex(),curprop->data.value);
		}

		curprop = curprop->next;
	}

	plist->DestroyUI();
	plist->BuildUI();
	*/
}


BOOL CALLBACK LODMod::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	LODMod* pthis = (LODMod*)GetWindowLong(hwnd, GWL_USERDATA);

	switch(msg)
	{
	case WM_INITDIALOG:
		SetWindowLong(hwnd, GWL_USERDATA, lParam);
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_ADDLOD:
			pthis->AddLOD();
			return TRUE;

		case IDC_REMOVELOD:
			pthis->RemoveLOD();
			return TRUE;

		case IDC_SELCHILDREN:
			pthis->SelectChildren();
			return TRUE;

		case IDC_SELALL:
			pthis->SelectAll();
			return TRUE;
		}
	}

	return FALSE;
}

void LODMod::ChangeCB(PropList* pList, void* pData)
{
	LODMod* pthis = (LODMod*)pData;

	// We will make a reference for each modified node
	int id   = pList->GetLastMod();
	CStr val;
	pList->GetValue(id, val);
	
	INode* node = gInterface->GetINodeByName(val);

	// Every node that gets assigned will result in a reference being made
	if (node)
	{
		int ref = pthis->FindRef(node);

		if (ref != -1)
			pthis->DeleteReference(ref);

		if (pthis->MakeRefByID(FOREVER, id / 2, node) != REF_SUCCEED)
		{
			MessageBox(gInterface->GetMAXHWnd(), "MakeRefByID Failed!", "ChangeCB", MB_ICONWARNING|MB_OK);
		}

		pthis->refLODs[id/2] = node;

		// Apply the client modifier so the user can see how the references are setup
		if (!FindModifier(node, vLODMODCLIENT_CLASS_ID))
		{
			Modifier* mod = (Modifier*)CreateInstance(OSM_CLASS_ID, vLODMODCLIENT_CLASS_ID);
			ModContext* mc = new ModContext(new Matrix3(1), NULL, NULL);

			Object* obj = node->GetObjectRef();
			IDerivedObject* dobj = CreateDerivedObject(obj);
			dobj->AddModifier(mod);
			node->SetObjectRef(dobj);
		}
	}

	pthis->StoreLODData();
}

void LODMod::FocusCB(PropList* pList, int id, void* pData)
{
	LODMod* pthis = (LODMod*)pData;
	pthis->focusID = id;
}

void LODMod::AddLOD()
{
	// Add new LOD fields to the list
	char bufName[256];
	char bufDesc[256];
	sprintf(bufName,"LOD_Level%i",++numLODs);
	sprintf(bufDesc,"This is LOD Level Number %i",numLODs);

	pLODList->SaveValues();
	pLODList->DestroyUI();
	pLODList->AddNode(bufName,bufDesc);

	sprintf(bufName,"LOD_Dist%i",numLODs);
	sprintf(bufDesc,"Distances greater than this value will use the model representation in slot LOD_Level%i",numLODs);

	pLODList->AddSpinEdit(bufName,0,5000,1,bufDesc);
	pLODList->BuildUI();
	pLODList->RestoreValues();

	INode* node = NULL;
	refLODs.Append(1, &node);
}

void LODMod::RemoveLOD(int id)
{
	// LODs are added in twos.  We expect to get the first field in the sequence (0, 2, 4, 8)
	// so mod 2 must always be 0 if not move to the field before the given value
	if (id % 2 != 0)
		id--;

	if (id < 0 || id > pLODList->NumProps() || pLODList->NumProps() == 0)
	{
		MessageBox(gInterface->GetMAXHWnd(), "Failed to remove LOD level.  Level index is out of range", "Can't remove LOD level", MB_ICONWARNING|MB_OK);
		return;
	}

	pLODList->RemoveProp(id);
	pLODList->RemoveProp(id);
	numLODs--;

	// Renumber the LOD ranges
	int num = 1;

	for(int i=0;i<pLODList->NumProps();i++)
	{
		char name[256];

		if (i % 2 == 0)
		{
			sprintf(name,"LOD_Level%i",num);
			pLODList->SetName(i,name);
		}
		else
		{
			sprintf(name,"LOD_Dist%i",num);
			pLODList->SetName(i,name);
			num++;
		}
	}

	focusID-=2;
	if (focusID < 0)
		focusID = 0;

	pLODList->BuildUI();
	pLODList->ResetIndexToEnd();
}

void LODMod::RemoveLOD()
{
	if (focusID == -1)
	{
		MessageBox(gInterface->GetMAXHWnd(), "There are no LOD levels available to delete", "No LOD levels remain", MB_ICONWARNING|MB_OK);
		return;
	}

	RemoveLOD(focusID);
}

void LODMod::BeginEditParams( IObjParam  *ip, ULONG flags,Animatable *prev)
{
	if (!hRollup)
	{
		hRollup = gInterface->AddRollupPage(hInstance, MAKEINTRESOURCE(IDD_LODMOD), DlgProc, "LOD Modifier", (LONG)this);
		pLODList = new PropList(hInstance, 1);
		pLODList->Attach(GetDlgItem(hRollup, IDC_LODLIST));
		pLODList->SetChangeCB(ChangeCB, this);
		pLODList->SetFocusCB(FocusCB, this);
		pLODList->HasApply(FALSE);

		ReadLODData();
	}
}

void LODMod::EndEditParams( IObjParam *ip,ULONG flags,Animatable *next)
{
	if (hRollup)
	{
		gInterface->DeleteRollupPage(hRollup);
		hRollup = NULL;
	}

	if (pLODList)
	{
		delete pLODList;
		pLODList = NULL;
	}
}

RefTargetHandle LODMod::Clone(RemapDir& remap)
{
	LODMod* newLODMod = new LODMod;
	BaseClone(this, newLODMod, remap);

	// Clone our references so they match up with the new set of created objects
	// if the user copied the entire LOD set (TT182)
	INodeTab nodeRefList;
	int      nRefLODs = refLODs.Count();
	int      i;

	nodeRefList.SetCount(nRefLODs);

	for(i = 0; i < nRefLODs; i++)
	{
		INode* clone = (INode*) remap.FindMapping( refLODs[i] );
		
		if (clone == NULL)
			clone = (INode*) remap.CloneRef( refLODs[i] );

		if (clone)
		{
			if (GetReference(i))
				newLODMod->ReplaceReference(i, clone);
			else
				newLODMod->MakeRefByID(FOREVER, i, clone);

			nodeRefList[i] = clone;
		}
	}

	// We now need to go through and update the property buffers of the copied object
	// so that it syncs up with the new references
	int nNodes = mainNodes.Count();
	newLODMod->mainNodes.SetCount(nNodes);

	for(i = 0; i < nNodes; i++)
		remap.PatchPointer((RefTargetHandle*)&newLODMod->mainNodes[i], (RefTargetHandle)mainNodes[i]);

	CStr name = newLODMod->mainNodes[0]->GetName();

	newLODMod->UpdateInternalPropsFromReferences(nodeRefList);

	return newLODMod;
}

RefResult LODMod::NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, PartID& partID,  RefMessage message)
{
	// Need to handle case where one of the LOD levels gets deleted
	//if (message == REFMSG_REF_DELETED)
	//if (message == REFMSG_SEL_NODES_DELETED)
	if (message == REFMSG_TARGET_DELETED)
	{
		char buf[256] = "Name: [None]";

		if (hTarget->ClassID() == Class_ID(BASENODE_CLASS_ID, 0))
		{
			sprintf(buf, "Name: %s\n", (char*)((INode*)hTarget)->GetName());
		}		

		MessageBox(gInterface->GetMAXHWnd(), buf, "Target Deleted", MB_ICONWARNING|MB_OK);
	}

	return REF_SUCCEED;
}

CreateMouseCallBack* LODMod::GetCreateMouseCallBack()
{
	return NULL;
}

INode* LODMod::GetNode()
{
	CStr name;
	NotifyDependents(FOREVER, (unsigned long)&name, REFMSG_GET_NODE_NAME);

	INode* node = gInterface->GetINodeByName(name);
	return node;
}

void LODMod::SelectChildren()
{
	int nProps = pLODList->NumProps();
	INodeTab nodeList;

	for(int i = 0; i < nProps; i++)
	{
		CStr name = pLODList->GetName(i);

		if (strstr(name, "LOD_Level"))
		{
			CStr value;
			pLODList->GetValue(i, value);

			INode* node = gInterface->GetINodeByName(value);

			if (node)
				nodeList.Append(1, &node);
		}
	}

	gInterface->ClearNodeSelection();
	gInterface->SelectNodeTab(nodeList, TRUE, TRUE);
}

void LODMod::SelectAll()
{
	int nProps = pLODList->NumProps();
	int i;
	INodeTab nodeList;

	// Select children
	for(i = 0; i < nProps; i++)
	{
		CStr name = pLODList->GetName(i);

		if (strstr(name, "LOD_Level"))
		{
			CStr value;
			pLODList->GetValue(i, value);

			INode* node = gInterface->GetINodeByName(value);

			if (node)
				nodeList.Append(1, &node);
		}
	}

	// Select the main node(s)
	for(i = 0; i < mainNodes.Count(); i++)
		nodeList.Append(1, &mainNodes[i]);

	gInterface->SelectNodeTab(nodeList, TRUE, TRUE);
}

///////////////////////// LODMod Client Implementation //////////////////////////////////////////////////////

LODModClient::LODModClient()
{
	hRollup  = NULL;
	pLODList = NULL;
}

BOOL LODModClient::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	LODModClient* pthis = (LODModClient*)GetWindowLong(hwnd, GWL_USERDATA);

	switch(msg)
	{
	case WM_INITDIALOG:
		SetWindowLong(hwnd, GWL_USERDATA, lParam);
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_SELMASTER:
			pthis->SelMaster();
			return TRUE;

		case IDC_SELALL:
			pthis->SelAll();
			return TRUE;
		}

		return TRUE;
	}

	return FALSE;
}

void LODModClient::BeginEditParams( IObjParam  *ip, ULONG flags,Animatable *prev)
{
	if (!hRollup)
	{
		hRollup = gInterface->AddRollupPage(hInstance, MAKEINTRESOURCE(IDD_LODMODCLIENT), DlgProc, "LOD Client Modifier", (LONG)this);
		pLODList = new PropList(hInstance, 1);
		pLODList->Attach(GetDlgItem(hRollup, IDC_PROPLIST));
		pLODList->HasApply(FALSE);

		ReadLODData();
	}
}

void LODModClient::EndEditParams( IObjParam *ip,ULONG flags,Animatable *next)
{
	if (hRollup)
	{
		gInterface->DeleteRollupPage(hRollup);
		hRollup = NULL;
	}

	if (pLODList)
	{
		delete pLODList;
		pLODList = NULL;
	}
}

INode* LODModClient::GetMasterNode()
{
	// The master node references us so, we can look through our reference list to find it
	ModContextList mcList;
	INodeTab       nodeList;

	gInterface->GetModContexts(mcList, nodeList);

	// We'll only do this for the first node selected
	if (nodeList.Count() != 1)
		return NULL;

	RefList&     refList = nodeList[0]->GetRefList();
	RefListItem* refItem = refList.FirstItem();

	while(refItem)
	{
		if (refItem->maker->ClassID() == vLODMOD_CLASS_ID)
		{
			LODMod* lodMod = (LODMod*)refItem->maker;

			// Return the lodMod's original node
			return lodMod->GetNode();
		}

		refItem = refItem->next;
	}

	return NULL;
}

IOResult LODModClient::LoadLocalData(ILoad *iload, LocalModData **pld)
{
	return IO_OK;
}

IOResult LODModClient::SaveLocalData(ISave *isave, LocalModData *ld)
{
	return IO_OK;
}

RefTargetHandle LODModClient::Clone(RemapDir& remap)
{
	LODModClient* newLODMod = new LODModClient;
	BaseClone(this, newLODMod, remap);
	return newLODMod;
}

void LODModClient::SelMaster()
{
	INode* nodeMaster = GetMasterNode();
	gInterface->SelectNode(nodeMaster);
	gInterface->ForceCompleteRedraw();
}

void LODModClient::SelAll()
{
	LinkList<ConfigProp> cprops;
	CStr                 propBuffer;

	INode* nodeMaster = GetMasterNode();

	gInterface->ClearNodeSelection();

	nodeMaster->GetUserPropBuffer(propBuffer);
	ParseConfigProps(&cprops,NULL,propBuffer);

	gInterface->SelectNode(nodeMaster, FALSE);
	
	Link<ConfigProp>* link = cprops.GetHead();
	while(link)
	{
		if (strstr(link->data.name, "LOD_Level"))
		{
			INode* selNode = gInterface->GetINodeByName(link->data.value);
			gInterface->SelectNode(selNode, FALSE);
		}

		link = link->next;
	}

	gInterface->ForceCompleteRedraw();
}

void LODModClient::ReadLODData()
{	
	INode* nodeMaster = GetMasterNode();

	if (!nodeMaster)
	{
		OutputDebugString("No NodeMaster!\n");
		return;
	}

	SetDlgItemText(hRollup, IDC_MASTER, (char*)nodeMaster->GetName());

	pLODList->Clear();
		
	// Only the first referenced node will be used in the event that LOD data
	// differs with the modifier applied to a group of 2 or more objects

	LinkList<ConfigProp> cprops;
	CStr                 propBuffer;

	nodeMaster->GetUserPropBuffer(propBuffer);
	ParseConfigProps(&cprops,NULL,propBuffer);

	// Scan through config props adding
	Link<ConfigProp>* link = cprops.GetHead();

	while(link)
	{
		if (strstr(link->data.name, "LOD_Level"))
		{
			int level = 0;
			sscanf(link->data.name, "LOD_Level%i", &level);

			pLODList->AddNode(link->data.name, "The LOD Representation node for the specified LOD level");

			// Rereference this object
			INode* node = gInterface->GetINodeByName(link->data.value);

			if (node)
			{
				int idx = pLODList->NumProps() - 1;
				pLODList->SetValue(idx, node->GetName());
			}

			// Find the associated LOD_Dist within the config props
			// Since we want the LOD_Level to always be followed by LOD distance
			Link<ConfigProp>* link2 = cprops.GetHead();

			while(link2)
			{
				if (strstr(link2->data.name, "LOD_Dist"))
				{
					int distLevel = 0;
					sscanf(link2->data.name, "LOD_Dist%i", &distLevel);

					if (level == distLevel)
					{
						pLODList->AddSpinEdit(link2->data.name,0,5000,1,"Distances greater than this value will use the model representation in the corresponding slot");

						int idx = pLODList->NumProps() - 1;
						pLODList->SetValue(idx, link2->data.value);
					}
				}

				link2 = link2->next;
			}
		}
		
		link = link->next;
	}

	pLODList->BuildUI();

	// Disable all the properties in the list
	// They may only be modified from the master LOD modifier
	for(int i = 0; i < pLODList->NumProps(); i++)
		pLODList->DisableProp(i);
}
