/*
	ScriptOptimize.h
	This runs through the exported node array and converts it to a group
	of shared substructures as per Mick's Ruby script
  
	This will also generate new structures as appropriate (if so told)

	aml 5-29-03
*/

#ifndef __SCRIPTOPTIMIZE_H__
#define __SCRIPTOPTIMIZE_H__

#include <core/HashTable.h>
#include "../misc/llist.h"
#include "max.h"
#include "../UI/ProgressBar.h"

enum OptLevel
{
	OPTLEVEL_NONE,					// Reserved.  Optimization shouldn't occur at all

	OPTLEVEL_DEFINITIONS_ONLY,		// Only the definitions in node_compress.q will be used for
									// node array replacement (FASTEST)

	OPTLEVEL_LOW_COMPRESSION,		// The node array will be analyzed and structures will be generated
									// for any duplicate occurances of data in the node array.  Structures
									// will be generated on a first come basis for the full unique contents of 
									// complete nodes  (SLOWER)

	OPTLEVEL_HIGH_COMPRESSION,		// The node array will be replaced with definitions generated via the
									// last LOW_COMPRESSION mode export

	OPTLEVEL_PERIODIC_COMPRESSION,	// A low compression mode export will occur every X days every other day
									// a high compession mode export will occur.  This way the replacement database
									// is kept fairly well up-to-date without much of a speed hit

};

struct ReplaceRecord
{
	CStr name;
	CStr buffer;

	int operator == (ReplaceRecord& right) { return ( buffer == right.buffer ); }
};

struct NodeOptRecord
{
	CStr line;				// A line in the node array that we're tracking
	
	int  nOccurances;		// Number of times this specific line occurs in node array
	int  nPreOccurances;	// Number of times all previous lines plus this one exists in node array 
	int  nPostOccurances;	// Number of times this line plus all future lines exist in node array
};

class ScriptOptimizer
{
	OptLevel                             level;
	FILE*                                fp;			// .qn file
	CStr                                 fileBuf;		// Contains node array
	CStr                                 fileBuf2;		// Contains the contents of the .qn after the node array
	CStr                                 defFilename;	// Definitions filename (including path)
	ProgressBar*                         pProgress;		// Progress bar used for building replacement file
	Lst::HashTable<Link<NodeOptRecord>*> lookupTable;	// Hash table used for 
	LinkList< LinkList<NodeOptRecord> >  nodeOptList;	// List containing the occurance counts of each node array line
	LinkList<ReplaceRecord>              replaceList;	// List of all the names that should have their buffer replaced

	bool LoadDefs(char* filename, char* buf = NULL, int* size = NULL);
	void PerformDefReplace();

	// If buf is included the output file will be dumped to a buffer in memory for quicker access
	void PerformLowCompress(char* filename, char* buf = NULL, int* size = NULL);
	void PerformHighCompress(char* filename, char* buf = NULL, int* size = NULL);

	void StoreQNUpdateTime();
	bool QNNeedsFullUpdate(int nDays);

public:
	ScriptOptimizer();

	inline void SetOptimizeLevel(OptLevel level) { this->level = level; }
	void OptimizeScripts(char* filename);
};

#endif
