#include <Trigger/Trigger.h>
#include <export/export.h>
#include <export/ExportOptions.h>
#include <export/TextureExporter.h>
#include <export/SkinExport.h>
#include <Link/LinkUI.h>
#include <Export/Strip/strip.h>
#include <Misc/GenCrc.h>
#include <Misc/Util.h>
#include <Material/NextMat.h>
#include <Material/NextMultiMat.h>
#include <Texture/NExtTexture.h>
#include <Texture/NExtAnimatedTexture.h>
#include <Image/Image.h>
#include <Particle/NExtParticle.h>
#include <Wibble/Wibble.h>
#include <PropEdit/ParseFuncs.h>
#include <PropEdit/ConfigData.h>
#include "../PropEdit/LODBrowser.h"
#include "../UI/OktoAll.h"
#include <StdMat.h>
#include "ModelExportOptions.h"
#include "SkinExportOptions.h"
#include "modstack.h"
#include "../misc/maxutil.h"
#include "meshadj.h"
#include "../../genlib/virtualfile.h"
#include "../LODMod/LODMod.h"
#include "../WibbleWatcher.h"

// Note standard unit is an inch
#define HEDGE_MAXIMUM_SQ  (7071.067812f * 7071.067812f)		// Hypotenuse maximum units (sqrt(5000^2 + 5000^2))
#define EDGE_MAXIMUM_SQ   (5000.0f * 5000.0f)				// Non Hypotenuse edge maximum units

extern INode* FixPolyConvNode(INode* node);

//#define OUTPUT_DEBUG_VERTS
//#define WRITE_DEBUG_FILE
#define DEBUG_OUTPUT

#ifdef NDEBUG
#undef OUTPUT_DEBUG_VERTS
#undef WRITE_DEBUG_FILE
#undef DEBUG_OUTPUT
#endif

#include "PropEdit/PropEdit.h"
extern PropEditor* pPropEdit;

#include "MemDebug.h"

#ifdef TRUE
#undef TRUE
#define TRUE 1
#endif

#ifdef FALSE
#undef FALSE
#define FALSE 0
#endif

#ifdef DEBUG_OUTPUT
#define DEBUGOUT(a)           {FILE* fp = fopen("c:\\maxdebug,txt","a");   \
	                          fprintf(fp,"%s\n",(char*)(a));               \
                              fclose(fp);                                  \
							  OutputDebugString(a);}  
#else
#define DEBUGOUT(a)           
#endif

// MultiRes Defines
#define MULTIRES_CLASS_ID  Class_ID(0x6a9e4c6b, 0x494744dd)
#define PB_VERTEXCOUNT     1
#define PB_VERTEXPERCENT   2
#define PB_GENERATE        10

static bool s_already_warned_nested;

inline float LengthSqr(Point3& v)
{
	return (v.x * v.x + v.y * v.y + v.z * v.z);
}

bool	Exporter::IsExportable( INode* node )
{
	Object *obj;			

	obj = node->EvalWorldState(0).obj;

	CStr propBuffer;
	node->GetUserPropBuffer(propBuffer);

	// Don't export the node if it's been flagged to be omitted
	//if (strstr(propBuffer,"// CMD:OMIT") ||
	//	strstr(propBuffer,"// CMD:FORCEOMIT"))
	//	return false;

	if( ( obj->ClassID() == vLINK_OBJ_CLASS_ID ) ||
		( obj->ClassID() == vTRIGGER_CLASS_ID ))
	{
		return false;
	}

	if( obj->SuperClassID() == SHAPE_CLASS_ID )
	{
		return false;
	}

	if( obj->SuperClassID() == HELPER_CLASS_ID )
	{
		return false;
	}

	if( obj->ClassID() == PARTICLE_BOX_CLASS_ID )
	{
		return false;
	}

	return ( obj->CanConvertToType( triObjectClassID ) == TRUE);	
}

bool	Exporter::SaveMaterials( IoUtils::CVirtualOutputFile &file, GeoDatabase* db )
{
	int i, num_materials;

	num_materials = db->m_Materials.GetSize();
	file.Write((const char*) &num_materials, sizeof( int ));
	for( i = 0; i < num_materials; i++ )
	{
		NxMaterial* material;
		int flags, num_sequences, num_frames, mmag, mmin, j;

		material = &db->m_Materials[i];

		// Warn if the base texture of this material is a non-blended 32-bit texture
		// WORK:
		if (m_export_type != vQUICK_EXPORT_NOTEX &&
			m_export_type != vQUICK_UPDATE_NOTEX &&
			m_export_type != vQUICK_VIEW_NOTEX   &&
	        m_export_type != vRESET_AND_RELOAD_NOTEX)
		{
			unsigned long texCRC = material->m_Passes[0].m_TexChecksum[ vPLAT_PS2 ];
			ITextureExporter* tex_exp = GetTextureExporter();
			NxTexture* nxtex = tex_exp->GetTextureByChecksum(texCRC);

			if (nxtex && nxtex->GetBpp()==NxTexture::v32_BIT && material->m_Passes[0].m_BlendMode==vBLEND_MODE_DIFFUSE)
			{
				char strErr[256];
				sprintf(strErr,"WARNING!  The material '%s' contains a 32-bit non-blended PS2 basetexture '%s'.",material->m_Name,nxtex->GetName(0));
				MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Material Warning",MB_ICONWARNING|MB_OK);		
				ReportWarning( strErr );
			}
		}

		// Warn if the first-pass blend mode is not "diffuse" on a material that isn't flagged as transparent
		if ( material->m_Passes[0].m_BlendMode != vBLEND_MODE_DIFFUSE &&
			!material->m_Transparent )
		{
			char strErr[256];
			sprintf(strErr,"WARNING!  You have chosen a blend mode that may not sort correctly because you have not flagged the material '%s' as transparent.",material->m_Name);
			MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Material Warning",MB_ICONWARNING|MB_OK);
			ReportWarning( strErr );
		}

		flags = 0;
		file.Write((const char*) &material->m_Checksum, sizeof( unsigned long ));

		if( material->m_Transparent )
		{
			flags |= NxMaterial::mTRANSPARENT;
		}
		if( material->m_OneSided )
		{
			flags |= NxMaterial::mONE_SIDED;
		}
		if( material->m_TwoSided )
		{
			flags |= NxMaterial::mTWO_SIDED;
		}
		if( material->m_Invisible )
		{
			flags |= NxMaterial::mINVISIBLE;
		}
		if( material->m_NumWibbleSequences > 0 )
		{
			flags |= NxMaterial::mVC_WIBBLE;
		}
		if( material->m_UseSpecular )
		{
			flags |= NxMaterial::mSPECULAR;
		}

		// All newly written files contain the new format CRC as additional data
		flags |= NxMaterial::mNEWFORMATCRC;

		file.Write((const char*) &flags, sizeof( int ));
		file.Write((const char*) &material->m_NewChecksum, sizeof( unsigned long )); // new v15

		file.Write((const char*) &material->m_Terrain, sizeof( int ));
		file.Write((const char*) &material->m_AlphaCutoff, sizeof( int ));
		file.Write((const char*) &material->m_CutoffFunc, sizeof( int ));		// new v5
		file.Write((const char*) &material->m_DrawOrder, sizeof( float ));		
		file.Write((const char*) &material->m_Sorted, sizeof( bool ));
		file.Write((const char*) &material->m_grassify, sizeof( bool ));		// new v8
		file.Write((const char*) &material->m_grassHeight, sizeof( float ));	// new v9
		file.Write((const char*) &material->m_grassLayers, sizeof( int ));		// new v9
		file.Write((const char*) &material->m_water, sizeof( bool ));		// new vE

		file.Write((const char*) &material->m_BasePass, sizeof( int ));

		if (flags & NxMaterial::mSPECULAR)
		{
			file.Write((const char*) &material->m_SpecularPower, sizeof(float));
			file.Write((const char*) &material->m_SpecularColor, sizeof(Color));
		}

		// Write out VC wibble sequences
		if( flags & NxMaterial::mVC_WIBBLE )
		{
			int k;

			num_sequences = material->m_NumWibbleSequences;
			file.Write((const char*) &num_sequences, sizeof( int ));
			for( j = 0; j < num_sequences; j++ )
			{
				num_frames = material->m_WibbleSequences[j].m_NumFrames;
				file.Write((const char*) &num_frames, sizeof( int ));
				for( k = 0; k < num_frames; k++ )
				{
					WibbleKeyframe *frame;
					
					frame = &material->m_WibbleSequences[j].m_WibbleFrames[k];
					file.Write((const char*) &frame->m_Time, sizeof( int ));
					file.Write((const char*) &frame->m_Color, sizeof( AColor ));
				}
			}
		}

		file.Write((const char*) &material->m_NumPasses, sizeof( int ));
		for( j = 0; j < material->m_NumPasses; j++ )
		{
			NxMaterialPass* pass;

			pass = &material->m_Passes[j];
			// new flags field in 0x0A, but moved up here in version 0x0C
			file.Write((const char*) &pass->m_Flags, sizeof( unsigned long ));
			if( pass->m_Flags & NxMaterialPass::ANIMATED_TEXTURE )
			{
				int frame;
				NxAnimatedTexture* anim_tex;

				anim_tex = &pass->m_AnimatedTexture;
				file.Write((const char*) &anim_tex->m_NumKeyframes, sizeof( int ));
				file.Write((const char*) &anim_tex->m_Period, sizeof( int ));
				file.Write((const char*) &anim_tex->m_Iterations, sizeof( int ));				
				file.Write((const char*) &anim_tex->m_Phase, sizeof( int ));
				for( frame = 0; frame < anim_tex->m_NumKeyframes; frame++ )
				{
					NxAnimatedTextureKeyframe* key_frame;

					key_frame= &anim_tex->m_Keyframes[frame];
					file.Write((const char*) &key_frame->m_Time, sizeof( unsigned int ));
					file.Write((const char*) key_frame->m_TexChecksum, vNUM_PLATFORMS * sizeof( unsigned long ));					
				}				
			}
			else
			{
				file.Write((const char*) pass->m_TexChecksum, vNUM_PLATFORMS * sizeof( unsigned long ));
			}
			
			flags = 0;
			if( pass->m_TexChecksum[0] != 0 )
			{
				flags |= NxMaterial::mTEXTURED;
			}
			if( pass->m_MappingMode == vMAPPING_ENVIRONMENT )
			{
				flags |= NxMaterial::mENVIRONMENT;
			}
			if( pass->m_UVWibbleEnabled )
			{
				flags |= NxMaterial::mUV_WIBBLE;
			}
			
			// For now, everything is smooth
			flags |= NxMaterial::mSMOOTH;
			file.Write((const char*) &flags, sizeof( int ));
			// Write out alpha-blending params
			file.Write((const char*) &pass->m_BlendMode, sizeof( int ));
			file.Write((const char*) &pass->m_AddressModeU, sizeof( int ));
			file.Write((const char*) &pass->m_AddressModeV, sizeof( int ));
			file.Write((const char*) &pass->m_FixedAlpha, sizeof( int ));
			// Write out surface properties of the material
			file.Write((const char*) &pass->m_HasColor, sizeof( bool ));

			// Force color to identity if no color is defined
			if (!pass->m_HasColor)
			{
				const float ident = 0.5f;
				file.Write((const char*) &ident, sizeof( float ));	// r
				file.Write((const char*) &ident, sizeof( float ));	// g
				file.Write((const char*) &ident, sizeof( float ));	// b
				file.Write((const char*) &ident, sizeof( float ));	// a
			}
			else
			{
				file.Write((const char*) &pass->m_Color.r, sizeof( float ));
				file.Write((const char*) &pass->m_Color.g, sizeof( float ));
				file.Write((const char*) &pass->m_Color.b, sizeof( float ));
				file.Write((const char*) &pass->m_Color.a, sizeof( float ));
			}

			file.Write((const char*) &pass->m_Diffuse.r, sizeof( float ));
			file.Write((const char*) &pass->m_Diffuse.g, sizeof( float ));
			file.Write((const char*) &pass->m_Diffuse.b, sizeof( float ));
			file.Write((const char*) &pass->m_Specular.r, sizeof( float ));
			file.Write((const char*) &pass->m_Specular.g, sizeof( float ));
			file.Write((const char*) &pass->m_Specular.b, sizeof( float ));
			file.Write((const char*) &pass->m_Ambient.r, sizeof( float ));
			file.Write((const char*) &pass->m_Ambient.g, sizeof( float ));
			file.Write((const char*) &pass->m_Ambient.b, sizeof( float ));

			// Write out UV wibble params
			if( flags & NxMaterial::mUV_WIBBLE )
			{
				file.Write((const char*) &pass->m_UVel, sizeof( float ));
				file.Write((const char*) &pass->m_VVel, sizeof( float ));
				file.Write((const char*) &pass->m_UFrequency, sizeof( float ));
				file.Write((const char*) &pass->m_VFrequency, sizeof( float ));
				file.Write((const char*) &pass->m_UAmplitude, sizeof( float ));
				file.Write((const char*) &pass->m_VAmplitude, sizeof( float ));
				file.Write((const char*) &pass->m_UPhase, sizeof( float ));
				file.Write((const char*) &pass->m_VPhase, sizeof( float ));
			}

			if( flags & NxMaterial::mENVIRONMENT )
			{
				file.Write((const char*) &pass->m_EnvTileU, sizeof( float ));
				file.Write((const char*) &pass->m_EnvTileV, sizeof( float ));				
			}

			// Write out MMAG and MMIN		
			mmag = (int) pass->m_MagFilteringMode;
			mmin = (int) pass->m_MinFilteringMode;
			file.Write((const char*) &mmag, sizeof( int ));
			file.Write((const char*) &mmin, sizeof( int ));

			// Write out K and L		
			file.Write((const char*) &pass->m_MipMapK, sizeof( float ));						
		}		    
	}

	return true;
}

NxVertex::NxVertex( void )
{
	int i;

	m_Pos.x = 0;
	m_Pos.y = 0;
	m_Pos.z = 0;
	m_Normal.x = 0;
	m_Normal.y = 0;
	m_Normal.z = 0;
	
	for( i = 0; i < vMAX_MATERIAL_PASSES; i++ )
	{
		m_TexCoord[i].x = 0;
		m_TexCoord[i].y = 0;
		m_TexCoord[i].z = 0;
	}
	m_Color.r = 0;
	m_Color.g = 0;
	m_Color.b = 0;
	m_Color.a = 0;
	m_WibbleIndex = 0;
	m_WibbleOffset = 0;
	m_Pass = 0;
	m_FaceIndex = 0;
	m_Weighted = false;

	for( i = 0; i < vMAX_WEIGHTS_PER_VERTEX; i++ )
	{
		m_Weight[i] = 0;
		m_WeightedIndex[i] = 0;
	}
}

NxFace::NxFace( void )
{
	m_MatChecksum = 0;
	m_Vertex[0] = 0;
	m_Vertex[1] = 0;
	m_Vertex[2] = 0;
	m_Normal.x = 0;
	m_Normal.y = 0;
	m_Normal.z = 0;
	m_FaceFlags = 0;
	m_MinFaceFlags = 0;
	m_Index = 0;
	m_Pass = 0;
	m_LightmapChecksum = 0;
	m_LightmapUV[0].u = 0;
	m_LightmapUV[0].v = 0;
	m_LightmapUV[1] = m_LightmapUV[0];
	m_LightmapUV[2] = m_LightmapUV[0];
}

NxLODLevel::NxLODLevel( void )
{
	m_Distance = 0.0f;
	m_ObjectCRC = 0;
}

NxObject::NxObject( void )
{
	m_NumVerts = 0;
	m_Verts = NULL;
	m_NumFaces = 0;
	m_Faces = NULL;
	m_Name = "Default";
	m_BoundingBox.Init();
	m_Flags = 0;
	m_NumUVSets = 0;
	m_SkeletonChecksum = 0;

	m_NumMaxVerts = 0;
	m_MaxVerts    = NULL;

	m_LODVersion   = NxLODLevel::vVERSION_NUMBER;
	m_LODMaster.m_NumLODLevels = 0;
	m_LODMaster.m_LODLevels    = NULL;

	m_Version = NxObject::vVERSION_NUMBER;

	m_ParentCRC = 0;
	m_NumChildren = 0;
	m_ChildCRCs = NULL;

	m_LODLevels = 0;
	m_LODinfo   = NULL;

	m_NumUVChannels = 0;
	m_UVChannels = NULL;	
}

NxObject::~NxObject( void )
{
	if( m_Verts )
	{
		delete [] m_Verts;
	}

	if( m_Faces )
	{
		delete [] m_Faces;
	}	

	if ( (m_LODFlags & mMASTER) && m_LODMaster.m_LODLevels )
	{
		delete [] m_LODMaster.m_LODLevels;
	}

	if ( m_ChildCRCs )
	{
		delete [] m_ChildCRCs;
	}

	if ( m_LODinfo )
	{
		delete [] m_LODinfo;
	}

	if ( m_MaxVerts )
	{
		delete [] m_MaxVerts;
	}

	if ( m_UVChannels )
	{
		delete [] m_UVChannels;
	}
}

bool	NxVertex::operator==( NxVertex& vertex )
{
	int j,k,m;

	// Check position
	if (m_Pos[0] != vertex.m_Pos[0] ||
		m_Pos[1] != vertex.m_Pos[1] ||
		m_Pos[2] != vertex.m_Pos[2])
		return false;

	// Check UVs
	for(j=0;j<vMAX_MATERIAL_PASSES;j++)
	{
		if (m_TexCoord[j][0] != vertex.m_TexCoord[j][0] ||
			m_TexCoord[j][1] != vertex.m_TexCoord[j][1])
			return false;
	}
				
	// Check Color
	if (m_Color[0] != vertex.m_Color[0] ||
		m_Color[1] != vertex.m_Color[1] ||
		m_Color[2] != vertex.m_Color[2] ||
		m_Color[3] != vertex.m_Color[3])
		return false;

	if (m_WibbleIndex  != vertex.m_WibbleIndex  ||
		m_WibbleOffset != vertex.m_WibbleOffset ||
		m_Pass         != vertex.m_Pass         ||
		m_NumWeights   != vertex.m_NumWeights)
		return false;

	// Check WeightedIndicies
	for(k=0;k<m_NumWeights;k++)
	{
		if (m_WeightedIndex[k] != vertex.m_WeightedIndex[k])
			return false;
	}

	// Check Weights
	for(m=0;m<m_NumWeights;m++)
	{
		if (m_Weight[m]!=vertex.m_Weight[m])
			return false;
	}

	return true;

	/*
	int i;

	for( i = 0; i < vMAX_MATERIAL_PASSES; i++ )
	{
		if( m_TexCoord[i] != vertex.m_TexCoord[i] )
		{
			return false;
		}
	}

	if( m_Weighted != vertex.m_Weighted )
	{
		return false;
	}

	if( m_Weighted )
	{
		for( i = 0; i < vMAX_WEIGHTS_PER_VERTEX; i++ )
		{
			if( m_Weight[i] != vertex.m_Weight[i] )
			{
				return false;
			}
		}
	}

	if( ( m_Pos == vertex.m_Pos ) &&		
		( m_Normal == vertex.m_Normal ) &&		
		( m_Color == vertex.m_Color ) &&
		( m_WibbleIndex == vertex.m_WibbleIndex ) &&
		( m_WibbleOffset == vertex.m_WibbleOffset ) &&
		( m_Pass == vertex.m_Pass ))
	{
		return true;
	}
	return false;
	*/
}

void	NxObject::OptimizeVertList( void )
{	
	NxVertex* unique_verts;
	int i, j, num_unique_verts;
	int* remapped_verts;

	if( m_NumVerts == 0 )
	{
		// Nothing to optimize
		return;
	}
	
	num_unique_verts = 0;
	unique_verts = new NxVertex[ m_NumVerts ];
	remapped_verts = new int[ m_NumVerts ];

	// Initialize the "instructions" for remapping verts. By default, just remap to same
	// array position
	for( i = 0; i < m_NumVerts; i++ )
	{
		remapped_verts[i] = i;
	}

	for( i = 0; i < m_NumVerts; i++ )
	{
		bool unique;

		unique = true;
		// Check if this vertex is already represented in our list of unique verts
		for( j = 0; j < num_unique_verts; j++ )
		{			
			// If so, just point it to the already-existing vertex
			if( unique_verts[j] == m_Verts[i] )
			{
				remapped_verts[i] = j;
				unique = false;
				break;
			}			
		}
		if( unique )
		{
			// This is a new vertex. Add it to the list and remap the original to this spot
			unique_verts[num_unique_verts] = m_Verts[i];
			remapped_verts[i] = num_unique_verts;
			num_unique_verts++;
		}
	}

	// Now go through each face and change vertex indices to match remapped vert indices
	for( i = 0; i < m_NumFaces; i++ )
	{
		for( j = 0; j < 3; j++ )
		{
			m_Faces[i].m_Vertex[j] = remapped_verts[m_Faces[i].m_Vertex[j]];
		}
	}

	delete [] m_Verts;
	m_NumVerts = num_unique_verts;
	m_Verts = new NxVertex[ m_NumVerts ];

	for( i = 0; i < m_NumVerts; i++ )
	{
		m_Verts[i] = unique_verts[i];
	}

	delete [] unique_verts;
	delete [] remapped_verts;
}

void	NxObject::Dump( void )
{
	int i;

	DebugPrint( "Contents of object %s\n", m_Name );
	DebugPrint( "================================\n" );
	DebugPrint( "Vertices: %d Faces %d\n", m_NumVerts, m_NumFaces );
	DebugPrint( "Vertex Positions\n" );
	for( i = 0; i < m_NumVerts; i++ )
	{		
		DebugPrint( "Vert[%d]	X:%f Y:%f Z:%f\n", i, m_Verts[i].m_Pos.x, m_Verts[i].m_Pos.y, m_Verts[i].m_Pos.z );
	}
	DebugPrint( "\nVertex Colors\n" );
	for( i = 0; i < m_NumVerts; i++ )
	{		
		DebugPrint( "Vert[%d]	R:%f G:%f B:%f A:%f\n", i, m_Verts[i].m_Color.r, m_Verts[i].m_Color.g, m_Verts[i].m_Color.b, m_Verts[i].m_Color.a );
	}
	
	for( i = 0; i < vMAX_MATERIAL_PASSES; i++ )
	{		
		int j;

		DebugPrint( "\nVertex Map Coords: Pass %d\n", i );
		for( j = 0; j < m_NumVerts; j++ )
		{
			DebugPrint( "Vert[%d]	U:%f V:%f\n", j, m_Verts[i].m_TexCoord[i].x, m_Verts[i].m_TexCoord[i].y );
		}
	}

	DebugPrint( "\nVertex Normals\n" );
	for( i = 0; i < m_NumVerts; i++ )
	{
		DebugPrint( "Vert[%d]	X:%f Y:%f Z:%f\n", i, m_Verts[i].m_Normal.x, m_Verts[i].m_Normal.y, m_Verts[i].m_Normal.z );
	}
	
	DebugPrint( "\nFace Vertex Indices\n" );
	for( i = 0; i < m_NumFaces; i++ )
	{
		DebugPrint( "Face[%d]	[%d,%d,%d]\n", i, m_Faces[i].m_Vertex[0], m_Faces[i].m_Vertex[1], m_Faces[i].m_Vertex[2] );
	}

}

Mtl*	GetSubMaterial( Mtl* base_mtl, int mtl_id )
{
    if( base_mtl )
    {
        Class_ID clID = base_mtl->ClassID();

		if( ( base_mtl->ClassID() == Class_ID( MULTI_CLASS_ID, 0 )) ||
			( base_mtl->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID ))
        {            
            Mtl *sub_mtl= NULL;

            if( base_mtl->NumSubMtls())
            {
                if( mtl_id >= base_mtl->NumSubMtls()) 
				{
					mtl_id %= base_mtl->NumSubMtls();
				}
            }
            else
            {                
                return NULL;
            }
            
			sub_mtl = base_mtl->GetSubMtl( mtl_id );
			if( sub_mtl == NULL )
			{
				return NULL;
			}
            if(	( sub_mtl->ClassID() == Class_ID( MULTI_CLASS_ID, 0 )) ||				
				( sub_mtl->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID ))
            {
				char msg[1024];

				sprintf( msg, "%s is a nested Multi/Sub-Object Material within %s, Using a default", base_mtl->GetName(), sub_mtl->GetName());
				if( !s_already_warned_nested )
				{					
					MessageBoxAll( gInterface->GetMAXHWnd(), msg, "Nested Materials", MB_OK );
					s_already_warned_nested = true;					
				}
				ReportWarning( msg );
				return NULL;
            }
            
			return sub_mtl;            
        }
		else
		{
			return base_mtl;
		}
    }

    return NULL;
}

NxAnimatedTextureKeyframe::NxAnimatedTextureKeyframe( void )
{
	int i;

	m_Time = 0;
	for( i = 0; i < vNUM_PLATFORMS; i++ )
	{
		m_TexChecksum[i] = 0;
	}
}

NxAnimatedTexture::NxAnimatedTexture( void )
{
	m_NumKeyframes = 0;
	m_Period = 0;
	m_Iterations = 0;
	m_LoopType = vLOOP;
}

NxAnimatedTexture::~NxAnimatedTexture( void )
{
}

NxMaterialPass::NxMaterialPass( void )
{
	memset( this, 0, sizeof( NxMaterialPass ));

	m_BlendMode = vBLEND_MODE_DIFFUSE;
	m_MappingMode = vMAPPING_EXPLICIT;
	m_MagFilteringMode = vFILTERING_LINEAR;
	m_MinFilteringMode = vFILTERING_LINEAR;
	m_UVWibbleEnabled = false;
	m_UVel = 0.0f;
	m_VVel = 0.0f;
	m_UAmplitude = 0.0f;
	m_VAmplitude = 0.0f;
	m_UPhase = 0.0f;
	m_VPhase = 0.0f;
	m_UFrequency = 0.0f;
	m_VFrequency = 0.0f;
	m_EnvTileU = 3.0f;
	m_EnvTileV = 3.0f;
	m_Color = AColor( 0.5f, 0.5f, 0.5f, 0.5f );
	m_Ambient = AColor( 0.5f, 0.5f, 0.5f, 0.5f );
	m_Diffuse = AColor( 0.5f, 0.5f, 0.5f, 0.5f );
	m_Specular = AColor( 0.5f, 0.5f, 0.5f, 0.5f );
	m_MipMapK = -8.0f;	
	m_Flags = 0;
}

NxMaterial::NxMaterial( void )
{
	sprintf( m_Name, "default" );	
	m_MaxMtl = NULL;
	m_Checksum = 0;
	m_Transparent = false;
	m_BasePass = 1;
	m_OneSided = false;
	m_TwoSided = false;
	m_Invisible = false;
	m_Terrain = 0;
	m_NumPasses = 1;
	m_WarnedAlready = false;
	m_AlphaCutoff = 1;
	m_DrawOrder = 0;
	m_Sorted = false;
	m_NumWibbleSequences = 0;	
	m_water = false;
	m_grassify = false;
	m_grassHeight = 0;
	m_grassLayers = 0;
	m_SpecularPower = 0.0f;
	m_SpecularColor = Color(1.0f,1.0f,1.0f);
}

bool	NxMaterial::operator == ( NxMaterial& material )
{
	return ( m_Checksum == material.m_Checksum );
}

unsigned long	NxMaterial::GetNewCRC( void )
{
	return( GenerateCRC( m_Name) );
}

unsigned long	NxMaterial::GetCRC( void )
{
	/*int i;
	char data[128];
	char* ch;	

	ch = data;

	for( i = 0; i < m_NumPasses; i++ )
	{
		memcpy( ch, &m_Passes[i].m_TexChecksum, sizeof( unsigned long ));
		ch += sizeof( unsigned long );
		memcpy( ch, &m_Passes[i].m_BlendMode, sizeof( int ));
		ch += sizeof( int );
		memcpy( ch, &m_Passes[i].m_FixedAlpha, sizeof( int ));
		ch += sizeof( int );
		memcpy( ch, &m_Passes[i].m_MappingMode, sizeof( int ));
		ch += sizeof( int );
		memcpy( ch, &m_Passes[i].m_MinFilteringMode, sizeof( int ));
		ch += sizeof( int );
		memcpy( ch, &m_Passes[i].m_MagFilteringMode, sizeof( int ));
		ch += sizeof( int );
		if( m_Passes[i].m_UVWibbleEnabled )
		{
			memcpy( ch, &m_Passes[i].m_UVel, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &m_Passes[i].m_VVel, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &m_Passes[i].m_UAmplitude, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &m_Passes[i].m_VAmplitude, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &m_Passes[i].m_UPhase, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &m_Passes[i].m_VPhase, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &m_Passes[i].m_UFrequency, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &m_Passes[i].m_VFrequency, sizeof( float ));
			ch += sizeof( float );		
		}
		else
		{
			float pad;

			pad = 0;
			memcpy( ch, &pad, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &pad, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &pad, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &pad, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &pad, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &pad, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &pad, sizeof( float ));
			ch += sizeof( float );
			memcpy( ch, &pad, sizeof( float ));
			ch += sizeof( float );
		}

		memcpy( ch, &m_Passes[i].m_MipMapK, sizeof( float ));
		ch += sizeof( float );
		memcpy( ch, &m_Passes[i].m_MipMapL, sizeof( float ));
		ch += sizeof( float );
	}

	return( GenerateCRC( data, (int) ch - (int) data ));
	*/

	return( GenerateCRC( m_Name, strlen( m_Name )));
	//return GenerateCRC( m_Name );
}

NxMeshGroup::NxMeshGroup( void )
{
	m_TextureGroup = -1;
}

// Add a normal to the list if the smoothing group bits overlap, 
// otherwise create a new vertex normal in the list
void VNormal::AddNormal(Point3 &n,DWORD s) 
{
	if (!(s&smooth) && init) 
	{
		if( next )
		{
			next->AddNormal( n, s );
		}
		else 
		{
			next = new VNormal( n, s );
		}
	} 
	else 
	{
		norm   += n;
		smooth |= s;
		init    = TRUE;
	}
}

// Retrieves a normal if the smoothing groups overlap or there is 
// only one in the list
Point3 &VNormal::GetNormal(DWORD s) 
{
	if( smooth&s || !next )
	{
		return norm;
	}
	else
	{
		return next->GetNormal( s );
	}
}

// Normalize each normal in the list
void VNormal::Normalize() 
{
	VNormal *ptr = next, *prev = this;
	while( ptr )
	{
		if (ptr->smooth&smooth) 
		{
			norm += ptr->norm;
			prev->next = ptr->next;
			ptr->next = NULL;
			delete ptr;
			ptr = prev->next;
		} 
		else 
		{
			prev = ptr;
			ptr  = ptr->next;
		}
	}
	
	norm = ::Normalize(norm);
	if (next) next->Normalize();
}

GeoDatabase::GeoDatabase( void )
: m_Skinned( false ), m_DynamicallyLit( false )
{	
	bWarnedDiffuse = false;
	bWarnedSpecular = false;
	bWarnedGloss = false;

	m_Skinned        = false;
	m_DynamicallyLit = false;	
	m_SkinRootBone   = NULL;
}

GeoDatabase::~GeoDatabase( void )
{
	int i;

	for( i = 0; i < m_Objects.Count(); i++ )
	{
		delete m_Objects[i];
	}
}

unsigned long	GeoDatabase::AddMaterial( Mtl* material, bool& env_mapped, bool& single_sided, 
											char* preText )
{
	int i, j, k, pass;
	ITextureExporter* tex_exp;
	NxMaterial new_mat;
	Texmap* map;
	UVGen *uv_gen;
	int tex_flags;
	unsigned long tex_checksum;	
	ExportOptions* options;
	bool is_standard_material;
				
	options = GetExportOptions();
	
	assert( material );
	
	tex_exp = GetTextureExporter();

	single_sided = false;
	env_mapped = false;		
	is_standard_material = false;
	
	// Create a new material for comparison and, if it's unique, for addition 
	// to the material list	
	new_mat.m_MaxMtl = material;
	strcpy( new_mat.m_Name, preText);
	//strcat( new_mat.m_Name, "-");
	strcat( new_mat.m_Name, "_");
	strcat( new_mat.m_Name, material->GetName());

	if( material->ClassID() == NEXT_MATERIAL_CLASS_ID )
	{		
		INExtMaterial* next_mat = dynamic_cast<INExtMaterial*> ( material );

		new_mat.m_Transparent = next_mat->IsTransparent();
		new_mat.m_OneSided = next_mat->IsOneSided();
		new_mat.m_TwoSided = next_mat->IsTwoSided();
		new_mat.m_Terrain = next_mat->GetTerrainType();
		new_mat.m_NumPasses = next_mat->GetNumPasses();
		new_mat.m_AlphaCutoff = next_mat->GetAlphaCutoff();
		new_mat.m_CutoffFunc = next_mat->GetCutoffFunc();
		new_mat.m_Invisible = next_mat->IsInvisible();
		new_mat.m_BasePass = next_mat->GetBasePass();
		if( next_mat->IsSorted())
		{
			new_mat.m_DrawOrder = options->m_SortGroupId + (float)vBASE_ORDERED_TRANSPARENT_MATERIAL_DRAW_ID;
			new_mat.m_Sorted = true;
		}
		else
		{
			new_mat.m_DrawOrder = next_mat->GetDrawOrderId();
			new_mat.m_Sorted = false;
		}		

		if( next_mat->IsOneSided())
		{
			single_sided = true;
		}

		new_mat.m_water = next_mat->IsWater();
		new_mat.m_grassify = next_mat->IsGrassified();
		new_mat.m_grassHeight = next_mat->GetGrassHeight();
		new_mat.m_grassLayers = next_mat->GetGrassLayers();

		if(	( options->m_ExportType == ExportOptions::vEXPORT_SCENE ) &&
			( next_mat->GetUseSpecular() ))
		{
			char strErr[256];
			sprintf(strErr,"The material '%s' uses specular color properties.\nThis is an invalid option for levels.",new_mat.m_Name);
			// Warn if blend mode uses diffuse if not first pass
			if( !bWarnedSpecular)
			{				
				MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Invalid blending mode",MB_ICONWARNING|MB_OK);
				bWarnedSpecular = true;
			}
			ReportWarning( strErr );
			new_mat.m_UseSpecular = false;
		}
		else
		{
			new_mat.m_UseSpecular   = next_mat->GetUseSpecular() ? true : false;
		}
		new_mat.m_SpecularPower = next_mat->GetSpecularPower();
		new_mat.m_SpecularColor = next_mat->GetSpecularColor();

		// Passes aren't necessarily enabled in order, so this wasn't always working   aml
		//for( pass = 0; pass < new_mat.m_NumPasses; pass++ )

		int matPass = 0;
		//bool bFlagUnlit = false;

		for( pass = 0; pass < vMAX_MATERIAL_PASSES; pass++)
		{
			tex_checksum = 0;

			//if (next_mat->MapEnabled( pass ))
			if (next_mat->PassEnabled( pass ))
			{
				map = material->GetSubTexmap( pass );	

				if( map &&
					map->ClassID() == NEXT_TEXTURE_CLASS_ID )
				{
					int min_filter, mag_filter, mip_filter;
					bool is_mipped;

					INExtTexture* next_tex = dynamic_cast< INExtTexture* >( map );

					is_mipped = ( next_tex->GetMipType( vPLAT_PS2 ) != vMIP_TYPE_NONE );

					min_filter = next_tex->GetMinFilterMode();
					mag_filter = next_tex->GetMagFilterMode();
					mip_filter = next_tex->GetMipFilterMode();
					switch( min_filter )
					{
						case vFILTERING_NEAREST:
						{
							if( mip_filter == vFILTERING_NEAREST )
							{
								min_filter = vFILTERING_NEAREST_MIPMAP_NEAREST;
							}
							else
							{
								min_filter = vFILTERING_NEAREST_MIPMAP_LINEAR;						
							}
							break;
						}
						case vFILTERING_LINEAR:
						{
							if( mip_filter == vFILTERING_NEAREST )
							{
								min_filter = vFILTERING_LINEAR_MIPMAP_NEAREST;
							}
							else
							{
								min_filter = vFILTERING_LINEAR_MIPMAP_LINEAR;
							}						
							break;					
						}
					}
					new_mat.m_Passes[matPass].m_MinFilteringMode = (FilteringMode) min_filter;				
					new_mat.m_Passes[matPass].m_MagFilteringMode = (FilteringMode) next_tex->GetMagFilterMode();
					new_mat.m_Passes[matPass].m_MipMapK = next_tex->GetMipMapK();
					new_mat.m_Passes[matPass].m_AddressModeU = next_tex->GetAddressModeU();
					new_mat.m_Passes[matPass].m_AddressModeV = next_tex->GetAddressModeV();				
				}

				new_mat.m_Passes[matPass].m_BlendMode = next_mat->GetBlendMode( pass );	

				// Warn if blend mode uses diffuse if not first pass
				if (new_mat.m_Passes[matPass].m_BlendMode==vBLEND_MODE_DIFFUSE && pass!=0)					
				{
					char strErr[256];
					sprintf(strErr,"The material '%s' is unassigned (uses diffuse blending) in pass %i.\nThis will invalidate earlier mappings.",new_mat.m_Name,pass+1);
					if( !bWarnedDiffuse)
					{
						MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Invalid blending mode",MB_ICONWARNING|MB_OK);
						bWarnedDiffuse = true;
					}
					ReportWarning( strErr );
				}
				
				// Warn if blend mode is glossmap and we're exporting a level
				if( ( new_mat.m_Passes[matPass].m_BlendMode==vBLEND_MODE_GLOSS_MAP ) &&
					( options->m_ExportType == ExportOptions::vEXPORT_SCENE ))				
				{
					char strErr[256];
					sprintf(strErr,"The material '%s' uses a gloss map in pass %i.\nThis blend mode is not valid in level exports. Will default back to diffuse.",new_mat.m_Name,pass+1);
					
					if( !bWarnedGloss )
					{						
						MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Invalid blending mode",MB_ICONWARNING|MB_OK);
						bWarnedGloss = true;
					}

					ReportWarning( strErr );
					new_mat.m_Passes[matPass].m_BlendMode = vBLEND_MODE_DIFFUSE;
				}

				new_mat.m_Passes[matPass].m_FixedAlpha = next_mat->GetFixedValue( pass );			

				// Get wibble params
				new_mat.m_Passes[matPass].m_UVWibbleEnabled = next_mat->UVWibbleDataEnabled( pass );
				new_mat.m_Passes[matPass].m_UVel = next_mat->GetUVel( pass );
				new_mat.m_Passes[matPass].m_VVel = next_mat->GetVVel( pass );
				new_mat.m_Passes[matPass].m_UFrequency = next_mat->GetUFrequency( pass );
				new_mat.m_Passes[matPass].m_VFrequency = next_mat->GetVFrequency( pass );
				new_mat.m_Passes[matPass].m_UAmplitude = next_mat->GetUAmplitude( pass );
				new_mat.m_Passes[matPass].m_VAmplitude = next_mat->GetVAmplitude( pass );
				new_mat.m_Passes[matPass].m_UPhase = next_mat->GetUPhase( pass );
				new_mat.m_Passes[matPass].m_VPhase = next_mat->GetVPhase( pass );

				// Get Env Map tile values
				new_mat.m_Passes[matPass].m_EnvTileU = next_mat->GetEnvTileU( pass );
				new_mat.m_Passes[matPass].m_EnvTileV = next_mat->GetEnvTileV( pass );
				
				// Get color values
				new_mat.m_Passes[matPass].m_Color = next_mat->GetColor( pass );
				new_mat.m_Passes[matPass].m_HasColor = next_mat->ColorEnabled( pass );
				new_mat.m_Passes[matPass].m_Ambient = next_mat->GetAmbientColor( pass );
				new_mat.m_Passes[matPass].m_Diffuse = next_mat->GetDiffuseColor( pass );
				new_mat.m_Passes[matPass].m_Specular = next_mat->GetSpecularColor( pass );		
				if( next_mat->VertexAlphaEnabled( pass ) == FALSE )
				{
					new_mat.m_Passes[matPass].m_Flags |= NxMaterialPass::IGNORE_VERTEX_ALPHA;					
				}
				
				// Get mapping and filtering modes
				new_mat.m_Passes[matPass].m_MappingMode = (MappingMode) next_mat->GetMappingMode( pass );
				if( new_mat.m_Passes[matPass].m_MappingMode == vMAPPING_ENVIRONMENT )
				{
					env_mapped = true;
				}

				if (next_mat->GetColorLocked( pass ))
					new_mat.m_Passes[matPass].m_Flags |= NxMaterialPass::COLOR_LOCKED;
				else
					new_mat.m_Passes[matPass].m_Flags &= ~NxMaterialPass::COLOR_LOCKED;				

				ExportOptions* exportOptions = GetExportOptions();

				// if you are exporting a scene, all materials with env mapping should have
				// the unlit checkbox checked.  all other types of export should have the unlit
				// checkbox unchecked for materials with env mapping.
				/*
				if (env_mapped)
				{
					if (exportOptions->m_ExportType == ExportOptions::vEXPORT_SCENE)
						bFlagUnlit = true;						
				}
				*/

				if (exportOptions->m_ExportType == ExportOptions::vEXPORT_SCENE)
					new_mat.m_Passes[matPass].m_Flags |= NxMaterialPass::UNLIT;
				else
					new_mat.m_Passes[matPass].m_Flags &= ~NxMaterialPass::UNLIT;

				/*
				if (next_mat->GetUnlit( pass ))
					new_mat.m_Passes[matPass].m_Flags |= NxMaterialPass::UNLIT;
				else
					new_mat.m_Passes[matPass].m_Flags &= ~NxMaterialPass::UNLIT;
				*/

				//auto_gen_mips = !next_mat->ManualMip( pass );
				//if( next_mat->ManualMip( pass ))
				//{
				//	num_mip_levels = next_mat->NumMipLevels( pass );
				//}
				//
				//force_byte_per_component = ( next_mat->GetTextureForceFlags( pass ) & TEX_FORCE_32_BIT ) != 0;
				//change_trans_color = ( next_mat->GetTextureForceFlags( pass ) & TEX_CHANGE_TRANS_COLOR ) != 0;
				//if( change_trans_color )
				//{
				//	trans_color = next_mat->GetTransColor( pass );
				//}

				// Advance to next actual material
				matPass++;
			}
		}

		// If any pass meets the unlit condition it should affect all passes
		/*
		if (bFlagUnlit)
		{
			for( pass = 0; pass < vMAX_MATERIAL_PASSES; pass++)
				new_mat.m_Passes[pass].m_Flags |= NxMaterialPass::UNLIT;
		}
		else
		{
			for( pass = 0; pass < vMAX_MATERIAL_PASSES; pass++)
				new_mat.m_Passes[pass].m_Flags &= ~NxMaterialPass::UNLIT;
		}
		*/

		/*
		// Handle the unlit flag
		// if you are exporting a scene, all materials with env mapping should have
		// the unlit checkbox checked.  all other types of export should have the unlit
		// checkbox unchecked for materials with env mapping.
		// This should apply to all passes
		if (env_mapped)
		{
			if (exportOptions->m_ExportType == ExportOptions::vEXPORT_SCENE)
				new_mat.m_Passes[matPass].m_Flags |= NxMaterialPass::UNLIT;
			else
				new_mat.m_Passes[matPass].m_Flags &= ~NxMaterialPass::UNLIT;
		}
		else
			new_mat.m_Passes[matPass].m_Flags &= ~NxMaterialPass::UNLIT;
		*/
	}	
	else
	{
		is_standard_material = true;

		map = material->GetSubTexmap( 0 );
		new_mat.m_NumPasses = 1;
		new_mat.m_Transparent = false;		
		if( map )
		{
			// Get filtering mode
			if( map->ClassID() == Class_ID(BMTEX_CLASS_ID, 0))
			{
				switch(((BitmapTex*)map)->GetFilterType())
				{
				case FILTER_NADA:			
					new_mat.m_Passes[0].m_MinFilteringMode = vFILTERING_NEAREST;
					new_mat.m_Passes[0].m_MagFilteringMode = vFILTERING_NEAREST;				
					break;		
				case FILTER_PYR:			
					new_mat.m_Passes[0].m_MinFilteringMode = vFILTERING_LINEAR_MIPMAP_LINEAR;
					new_mat.m_Passes[0].m_MagFilteringMode = vFILTERING_LINEAR_MIPMAP_LINEAR;
					break;
				case FILTER_SAT:
				default:
					new_mat.m_Passes[0].m_MinFilteringMode = vFILTERING_LINEAR;
					new_mat.m_Passes[0].m_MagFilteringMode = vFILTERING_LINEAR;
					break;		
				}
			}		

			// Get Mapping mode
			if( uv_gen = map->GetTheUVGen())
			{
				int type;

				type = uv_gen->GetSlotType();
			
				switch( type )
				{
					case MAPSLOT_ENVIRON:
						new_mat.m_Passes[0].m_MappingMode = vMAPPING_ENVIRONMENT;
						break;
					case MAPSLOT_TEXTURE:
					default:
						new_mat.m_Passes[0].m_MappingMode = vMAPPING_EXPLICIT;
						env_mapped = true;
						break;
				}
			}	
		}
	}

	int matPass=0;

	// Passes aren't necessarily enabled in order, so this wasn't always working   
	// It now does the proper check  aml
	if( material->ClassID() == NEXT_MATERIAL_CLASS_ID )
	{
		INExtMaterial* next_mat = dynamic_cast<INExtMaterial*> ( material );
		bool env_mapped;

		for( pass = 0; pass < vMAX_MATERIAL_PASSES; pass++)
		{
			if (next_mat->MapEnabled( pass ))
			{
				env_mapped = ( new_mat.m_Passes[matPass].m_MappingMode == vMAPPING_ENVIRONMENT );
				map = material->GetSubTexmap( pass );
				if( map->ClassID() == NEXT_ANIMATEDTEXTURE_CLASS_ID )
				{
					int frame;
					INExtAnimatedTexture* anim_tex;
					NxAnimatedTexture* anim_pass;					

					new_mat.m_Passes[matPass].m_Flags |= NxMaterialPass::ANIMATED_TEXTURE;				
					anim_pass = &new_mat.m_Passes[matPass].m_AnimatedTexture;
					anim_tex = dynamic_cast<INExtAnimatedTexture*>( map );
					anim_pass->m_LoopType = anim_tex->GetLoopType();
					anim_pass->m_Phase = anim_tex->GetPhase();
					anim_pass->m_NumKeyframes = anim_tex->GetNumFrames();
					if( anim_pass->m_LoopType == vLOOPTYPE_LOOP )
					{
						anim_pass->m_NumKeyframes += 1;	// we need to add 1 more frame representing the last texture's hold duration
					}
					else if( anim_pass->m_LoopType == vLOOPTYPE_PINGPONG )
					{
						anim_pass->m_NumKeyframes *= 2;	// we will manually add the frames in reverse order so the engine just needs to support looping
					}
					Dbg_Assert( anim_pass->m_NumKeyframes <= vMAX_NUM_ANIMATED_FRAMES );

					anim_pass->m_Period = 0;
					anim_pass->m_Iterations = anim_tex->GetIterations();					
					for( frame = 0; frame < anim_tex->GetNumFrames(); frame++ )
					{
						tex_flags = 0;		
						tex_exp->AddTexture(	anim_tex->GetMap( frame ), tex_flags, 
												anim_pass->m_Keyframes[frame].m_TexChecksum, 
												!env_mapped );
						anim_pass->m_Keyframes[frame].m_Time = anim_tex->GetTime( frame );						
					}

					if(	( anim_pass->m_LoopType == vLOOPTYPE_LOOP ) ||
						( anim_pass->m_LoopType == vLOOPTYPE_PINGPONG ))
					{
						int duration, last_frame, hold_frame;

						last_frame = anim_tex->GetNumFrames() - 1;
						hold_frame = anim_tex->GetNumFrames();
						
						// The period of the animation will be the starting time
						// of the last frame plus its duration. We add one more
						// frame of animation to represent the last frame's hold 
						// duration						
						duration = anim_tex->GetDuration( last_frame );
						anim_pass->m_Keyframes[hold_frame].m_Time = 	
							anim_pass->m_Keyframes[last_frame].m_Time + 	duration;
						memcpy( anim_pass->m_Keyframes[hold_frame].m_TexChecksum,
							anim_pass->m_Keyframes[last_frame].m_TexChecksum, vNUM_PLATFORMS * sizeof( unsigned long ));
						anim_pass->m_Period = anim_pass->m_Keyframes[last_frame].m_Time + 
												duration;						
					}
					
					if( anim_pass->m_LoopType == vLOOPTYPE_PINGPONG )
					{
						int duration, src_frame, dst_frame;

						// We assume the extra frame has been added above to represent
						// the duration of the last frame. So, we start from the frame
						// after that
						dst_frame = anim_tex->GetNumFrames() + 1;

						// For ping pongs, we need to append all frames to the animation
						// in reverse order
						for( src_frame = anim_tex->GetNumFrames() - 2; src_frame >= 0; 
								src_frame--, dst_frame++ )
						{
							// Use the duration of the previous frame to determine
							// the start time of this frame
							duration = anim_tex->GetDuration( src_frame + 1 );
							anim_pass->m_Period += duration;

							anim_pass->m_Keyframes[dst_frame].m_Time = anim_pass->m_Period;
							memcpy( anim_pass->m_Keyframes[dst_frame].m_TexChecksum,
									anim_pass->m_Keyframes[src_frame].m_TexChecksum, 
									vNUM_PLATFORMS * sizeof( unsigned long ));							
						}						
					}
				}
				else
				{				
					tex_flags = 0;		
					tex_exp->AddTexture( map, tex_flags, new_mat.m_Passes[matPass].m_TexChecksum, !env_mapped );
				}
				matPass++;
			}
		}
	}
	else
	{
		// If this is a non NExt material it's a single pass (and we'll just use the Diffuse map)
		map = material->GetSubTexmap( ID_DI );

		tex_flags = 0;
		
		tex_exp->AddTexture( map, tex_flags, new_mat.m_Passes[0].m_TexChecksum );
	}

	new_mat.m_Checksum = new_mat.GetCRC();
	new_mat.m_NewChecksum = new_mat.GetNewCRC();

	//new_mat.m_Checksum = new_mat.GetNewCRC();
	//new_mat.m_NewChecksum = new_mat.GetNewCRC();

	// First check to see if we already have this material
	for( i = 0; i < m_Materials.GetSize(); i++ )
	{
		if( m_Materials[i] == new_mat )
		{
			// Now check to see if there are differing materials with identical names
			if( new_mat.m_MaxMtl != m_Materials[i].m_MaxMtl )
			{
				char msg[1024];
					
				sprintf( msg, "There are multiple materials with the name %s. This will cause problems.", new_mat.m_Name );
				//if( m_Materials[i].m_WarnedAlready == false )
				if( tex_exp->m_WarnedAboutDupeMaterials == false || 
					( tex_exp->m_WarnedAboutDupeMaterials == true && CStr(new_mat.m_Name) != tex_exp->m_LastWarnedMatName ))
				{
					
					MessageBoxAll( gInterface->GetMAXHWnd(), msg,
							"Multiple Named Materials", MB_OK|MB_TASKMODAL );

					m_Materials[i].m_WarnedAlready = false;
					tex_exp->m_WarnedAboutDupeMaterials = true;
					tex_exp->m_LastWarnedMatName = new_mat.m_Name;
				}
				ReportWarning( msg );
			}
			return m_Materials[i].m_Checksum;
		}
	}

	if( material->ClassID() == NEXT_MATERIAL_CLASS_ID )
	{		
		int num_frames;
		INExtMaterial* next_mat = dynamic_cast<INExtMaterial*> ( material );

		new_mat.m_NumWibbleSequences = next_mat->VCWibbleNumSequences();
		for( j = 0; j < new_mat.m_NumWibbleSequences; j++ )
		{		
			num_frames = next_mat->VCWibbleNumKeyframes( j );
			new_mat.m_WibbleSequences[j].m_NumFrames = num_frames;
			new_mat.m_WibbleSequences[j].m_WibbleFrames = new WibbleKeyframe[num_frames];
			for( k = 0; k < num_frames; k++ )
			{
				new_mat.m_WibbleSequences[j].m_WibbleFrames[k] = next_mat->VCWibbleKeyFrame( j, k );					
			}
		}		
	}
	
	if( is_standard_material )
	{
		char msg[1024];
					
		sprintf( msg, "Material %s is not a NExt Material.", new_mat.m_Name );
		MessageBox( gInterface->GetMAXHWnd(), msg,
				"Non-NExt Material", MB_OK|MB_TASKMODAL );

	}

	m_Materials.AddToTail( &new_mat );	
	
	return m_Materials[i].m_Checksum;
}

NxMaterial*	GeoDatabase::GetMaterialByChecksum( unsigned long checksum )
{
	int i;

	for( i = 0; i < m_Materials.GetSize(); i++ )
	{
		//if ( m_Materials[i].GetNewCRC() == checksum )
		if( m_Materials[i].GetCRC() == checksum )
		{
			return &m_Materials[i];
		}
	}

	return NULL;
}

// This function will convert the source UVs stored in the NxObject into
// a list of actual UVs that have been transformed by the UVGen for the
// material that corresponds to each face in the mesh    aml
// UPDATE 4-3-01: This will now update crop/placement positions for the UVs
void GeoDatabase::UpdateFaceUVs( INode* node, NxObject* object )
{
	int num_faces=object->m_NumFaces;

	Object*            obj = node->EvalWorldState(0).obj;
	assert( obj->CanConvertToType( triObjectClassID ));
	TriObject*   triObject = (TriObject *) obj->ConvertToType( 0, triObjectClassID );
	Mesh* mesh  = &triObject->mesh;

	for(int i=0;i<num_faces;i++)
	{
		int v0=object->m_Faces[i].m_Vertex[0];
		int v1=object->m_Faces[i].m_Vertex[1];
		int v2=object->m_Faces[i].m_Vertex[2];

		// Verify that the object verts are redundant 1:1
		assert(v1==v0+1 && v2==v1+1);

		// Determine the material for this face
		Mtl* actual_mtl = GetSubMaterial( node->GetMtl(), mesh->faces[i].getMatID());

		if (actual_mtl)
		{
			// Find the texture map
			if (actual_mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
			{
				INExtMaterial* next_mat = dynamic_cast<INExtMaterial*>(actual_mtl);
				int num_passes = next_mat->GetNumPasses();

				for(int pass=0;pass<num_passes;pass++)
				{
					// Don't process if there aren't texture coords for this pass
					if (pass >= object->m_NumUVSets)
						break;
					
					Texmap* tex = actual_mtl->GetSubTexmap(pass);

					if (tex)
					{
						INExtTexture* next_tex=NULL;

						if (tex->ClassID()==NEXT_TEXTURE_CLASS_ID)
							next_tex = dynamic_cast<INExtTexture*>(tex);
						
						// Now grab the UV Transformation Matrix
						Matrix3 uvtm;
						tex->GetUVTransform(uvtm);

						// Apply crop/placement UV values
						if (next_tex && next_tex->GetCropPlaceMode() != vNONE)
						{

							Point3 pt;
							float  u,v,w,h,fu1,fv1;
							u = next_tex->GetCropPlaceU();
							v = next_tex->GetCropPlaceV();
							w = next_tex->GetCropPlaceW();
							h = next_tex->GetCropPlaceH();

							switch(next_tex->GetCropPlaceMode())
							{
							case vCROP:
								// MAX has the UV box upside down just on vertical axis (ARGH!)
								v = v-(1.0f-h);
								
								if (v<0.0f)
									v=-v;

								// v0
								pt = object->m_Verts[v0].m_TexCoord[pass];
								pt.x = ((pt.x * w) + u);
								pt.y = ((pt.y * h) + v);
								
								object->m_Verts[v0].m_TexCoord[pass] = pt;

								// v1
								pt = object->m_Verts[v1].m_TexCoord[pass];
								pt.x = ((pt.x * w) + u);
								pt.y = ((pt.y * h) + v);

								object->m_Verts[v1].m_TexCoord[pass] = pt;

								// v2
								pt = object->m_Verts[v2].m_TexCoord[pass];
								pt.x = ((pt.x * w) + u);
								pt.y = ((pt.y * h) + v);
								
								object->m_Verts[v2].m_TexCoord[pass] = pt;
								break;

							case vPLACE:
								// MAX has the UV box upside down just on vertical axis (ARGH!)
								
								v = v-(1.0f-h);
								
								if (v<0.0f)
									v=-v;
								
								fu1 = u + w;
								fv1 = v + h;

								// v0
								pt = object->m_Verts[v0].m_TexCoord[pass];
								pt.x = 1.0f - ((fu1 -  pt.x)/w);
								pt.y = 1.0f - ((fv1 -  pt.y)/h);

								object->m_Verts[v0].m_TexCoord[pass] = pt;

								// v1
								pt = object->m_Verts[v1].m_TexCoord[pass];
								pt.x = 1.0f - ((fu1 -  pt.x)/w);
								pt.y = 1.0f - ((fv1 -  pt.y)/h);

								object->m_Verts[v1].m_TexCoord[pass] = pt;

								// v2
								pt = object->m_Verts[v2].m_TexCoord[pass];
								pt.x = 1.0f - ((fu1 -  pt.x)/w);
								pt.y = 1.0f - ((fv1 -  pt.y)/h);

								object->m_Verts[v2].m_TexCoord[pass] = pt;
								break;
							}
						}
						//else
						{
							// And finally, apply the UVGen transform to our source texture coordinates
							//object->m_Verts[v0].m_TexCoord[pass] = uvtm*object->m_Verts[v0].m_TexCoord[pass];
							//object->m_Verts[v1].m_TexCoord[pass] = uvtm*object->m_Verts[v1].m_TexCoord[pass];
							//object->m_Verts[v2].m_TexCoord[pass] = uvtm*object->m_Verts[v2].m_TexCoord[pass];

							object->m_Verts[v0].m_TexCoord[pass] = object->m_Verts[v0].m_TexCoord[pass] * uvtm;
							object->m_Verts[v1].m_TexCoord[pass] = object->m_Verts[v1].m_TexCoord[pass] * uvtm;
							object->m_Verts[v2].m_TexCoord[pass] = object->m_Verts[v2].m_TexCoord[pass] * uvtm;
						}
					}
				}
			}
			else
			{
				Texmap* tex = actual_mtl->GetSubTexmap(ID_DI);

				if (tex)
				{
					// Now grab the UV Transformation Matrix
					Matrix3 uvtm;
					tex->GetUVTransform(uvtm);

					// TODO: Apply UVs crop/place for non NExt materials

					// And apply to our source texture coordinates
					object->m_Verts[v0].m_TexCoord[0] = uvtm*object->m_Verts[v0].m_TexCoord[0];
					object->m_Verts[v1].m_TexCoord[0] = uvtm*object->m_Verts[v1].m_TexCoord[0];
					object->m_Verts[v2].m_TexCoord[0] = uvtm*object->m_Verts[v2].m_TexCoord[0];
				}
			}
		}
	}

	if (triObject!=obj)
		triObject->DeleteThis();
}

bool GeoDatabase::GetExtents(INode* node, Matrix3 tm, Point3& min, Point3& max)
{
	return GetObjectExtents( node, tm, min, max );	
}

int GeoDatabase::TranslateCASRemovalFlags(char* str)
{
	int flag=0;
	LinkList<CStr> elements;
	ParseArray(&elements,str);

	Link<CStr>* curNode=elements.GetHead();

	while(curNode)
	{
		int id=atoi(curNode->data);

		if (id!=0)
			flag |= (1 << (id-1));

		curNode=curNode->next;
	}

	return flag;
}

void GeoDatabase::RemapPass(int* passList, FlagType origFlags, FlagType& newFlags, FlagType testFlag, int ID)
{
	if (origFlags & testFlag)
	{
		switch(passList[ID])
		{
		case 0:
			newFlags &= ~mFD_PASS_1_DISABLED;
			break;
		case 1:
			newFlags |= mFD_PASS_2_ENABLED;
			break;
		case 2:
			newFlags |= mFD_PASS_3_ENABLED;
			break;
		case 3:
			newFlags |= mFD_PASS_4_ENABLED;
			break;
		}
	}
	else
	{
		switch(passList[ID])
		{
		case 0:
			newFlags |= mFD_PASS_1_DISABLED;
			break;
		case 1:
			newFlags &= ~mFD_PASS_2_ENABLED;
			break;
		case 2:
			newFlags &= ~mFD_PASS_3_ENABLED;
			break;
		case 3:
			newFlags &= ~mFD_PASS_4_ENABLED;
			break;
		}
	}
}

// This function steps through the face pass flags and readjusts them based on the enabled
// passes in the given material COMBINED WITH the "absent in net games" flag
FlagType GeoDatabase::AdjustMinPassFlag( Mtl* mtl, Mesh* mesh, int index)
{
	int      newPass[vMAX_MATERIAL_PASSES];		// Array that contains the converted pass value
	bool     netPass[vMAX_MATERIAL_PASSES];		// Array that contains the converted pass value
	int      lastPass=0;
	bool     bAbort = false;

	// Build up translation list of the original passes
	if( mtl && mtl->ClassID() == NEXT_MATERIAL_CLASS_ID )
	{		
		INExtMaterial* next_mat = dynamic_cast<INExtMaterial*> ( mtl );

		for(int pass = 0; pass < vMAX_MATERIAL_PASSES; pass++)
		{
			if (next_mat->PassEnabledInNetGames( pass ))
			{
				netPass[pass] = true;
			}
			else
			{
				netPass[pass] = false;
			}
			if (next_mat->PassEnabled( pass ))			
			{
				newPass[pass] = lastPass;
				lastPass++;
			}
			else
				newPass[pass] = -1;
		}
	}
	else
	{
		newPass[lastPass++] = 0;
		newPass[lastPass++] = -1;
		newPass[lastPass++] = -1;
		newPass[lastPass++] = -1;

		netPass[0] = true;
		netPass[1] = true;
		netPass[2] = true;
		netPass[3] = true;
	}

	// Now go through the faces and adjust their face pass flags to coincide with the new pass
	// values for the material

	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh->GetInterface( FACEDATAMGR_INTERFACE ));

	if ( !pFDMgr )
		return 0;
		
	FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));

	if( !fdc ) 
		return 0;

	int numFaces = mesh->getNumFaces();
	FlagType origFlags, newFlags;

	fdc->GetValue( index, origFlags );
	newFlags = origFlags;

	// Get rid of all pass flags
	newFlags &= ~(mFD_PASS_1_DISABLED|mFD_PASS_2_ENABLED|mFD_PASS_3_ENABLED|mFD_PASS_4_ENABLED);

	if (!mtl)
		return newFlags;

	// Remap passes 2-4
	if( netPass[1] )
	{
		RemapPass(newPass,origFlags,newFlags,mFD_PASS_2_ENABLED,1);
	}
	if( netPass[2] )
	{
		RemapPass(newPass,origFlags,newFlags,mFD_PASS_3_ENABLED,2);
	}
	if( netPass[3] )
	{
		RemapPass(newPass,origFlags,newFlags,mFD_PASS_4_ENABLED,3);
	}

	// Remap pass 1
	if (!(origFlags & mFD_PASS_1_DISABLED))
	{
		switch(newPass[0])
		{
		case 0:
			newFlags &= ~mFD_PASS_1_DISABLED;
			break;
		case 1:
			newFlags |= mFD_PASS_2_ENABLED;
			break;
		case 2:
			newFlags |= mFD_PASS_3_ENABLED;
			break;
		case 3:
			newFlags |= mFD_PASS_4_ENABLED;
			break;
		}
	}
	else
	{
		switch(newPass[0])
		{
		case 0:
			newFlags |= mFD_PASS_1_DISABLED;
			break;
		case 1:
			newFlags &= ~mFD_PASS_2_ENABLED;
			break;
		case 2:
			newFlags &= ~mFD_PASS_3_ENABLED;
			break;
		case 3:
			newFlags &= ~mFD_PASS_4_ENABLED;
			break;
		}
	}

	return newFlags;
}

// This function steps through the face pass flags and readjusts them based on the enabled
// passes in the given material
FlagType GeoDatabase::AdjustPassFlag( Mtl* mtl, Mesh* mesh, int index)
{
	int      newPass[vMAX_MATERIAL_PASSES];		// Array that contains the converted pass value
	int      lastPass=0;
	bool     bAbort = false;

	// Build up translation list of the original passes
	if( mtl && mtl->ClassID() == NEXT_MATERIAL_CLASS_ID )
	{		
		INExtMaterial* next_mat = dynamic_cast<INExtMaterial*> ( mtl );

		for(int pass = 0; pass < vMAX_MATERIAL_PASSES; pass++)
		{
			if (next_mat->PassEnabled( pass ))
			{
				newPass[pass] = lastPass;
				lastPass++;
			}
			else
				newPass[pass] = -1;
		}
	}
	else
	{
		newPass[lastPass++] = 0;
		newPass[lastPass++] = -1;
		newPass[lastPass++] = -1;
		newPass[lastPass++] = -1;
	}

	// Now go through the faces and adjust their face pass flags to coincide with the new pass
	// values for the material

	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh->GetInterface( FACEDATAMGR_INTERFACE ));

	if ( !pFDMgr )
		return 0;
		
	FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));

	if( !fdc ) 
		return 0;

	int numFaces = mesh->getNumFaces();
	FlagType origFlags, newFlags;

	fdc->GetValue( index, origFlags );
	newFlags = origFlags;

	// Get rid of all pass flags
	newFlags &= ~(mFD_PASS_1_DISABLED|mFD_PASS_2_ENABLED|mFD_PASS_3_ENABLED|mFD_PASS_4_ENABLED);

	if (!mtl)
		return newFlags;

	// Remap passes 2-4
	RemapPass(newPass,origFlags,newFlags,mFD_PASS_2_ENABLED,1);
	RemapPass(newPass,origFlags,newFlags,mFD_PASS_3_ENABLED,2);
	RemapPass(newPass,origFlags,newFlags,mFD_PASS_4_ENABLED,3);

	// Remap pass 1
	if (!(origFlags & mFD_PASS_1_DISABLED))
	{
		switch(newPass[0])
		{
		case 0:
			newFlags &= ~mFD_PASS_1_DISABLED;
			break;
		case 1:
			newFlags |= mFD_PASS_2_ENABLED;
			break;
		case 2:
			newFlags |= mFD_PASS_3_ENABLED;
			break;
		case 3:
			newFlags |= mFD_PASS_4_ENABLED;
			break;
		}
	}
	else
	{
		switch(newPass[0])
		{
		case 0:
			newFlags |= mFD_PASS_1_DISABLED;
			break;
		case 1:
			newFlags &= ~mFD_PASS_2_ENABLED;
			break;
		case 2:
			newFlags &= ~mFD_PASS_3_ENABLED;
			break;
		case 3:
			newFlags &= ~mFD_PASS_4_ENABLED;
			break;
		}
	}

	return newFlags;
}

bool	GeoDatabase::VerifyMeshTriSize( INode* node )
{
	Object* obj = node->EvalWorldState(0).obj;

	if (obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

		if (triObj)
		{
			Mesh& mesh = triObj->GetMesh();

			for(int i = 0; i < mesh.numFaces; i++)
			{
				// Ensure that no triangles exist that are larger than 5000 inches
				float edgeL[3];
				int   maxEdge, edgeA, edgeB;
				
				edgeL[0] = LengthSqr(mesh.verts[mesh.faces[i].v[0]] - mesh.verts[mesh.faces[i].v[1]]);
				edgeL[1] = LengthSqr(mesh.verts[mesh.faces[i].v[1]] - mesh.verts[mesh.faces[i].v[2]]);
				edgeL[2] = LengthSqr(mesh.verts[mesh.faces[i].v[2]] - mesh.verts[mesh.faces[i].v[0]]);

				// Determine hypotenuse
				if (edgeL[0] > edgeL[1])
				{
					if (edgeL[0] > edgeL[2])
					{
						edgeA   = 1;
						edgeB   = 2;
						maxEdge = 0;
					}
					else
					{
						edgeA   = 0;
						edgeB   = 1;
						maxEdge = 2;
					}
				}
				else
				{
					if (edgeL[1] > edgeL[2])
					{
						edgeA   = 0;
						edgeB   = 2;
						maxEdge = 1;
					}
					else
					{
						edgeA   = 0;
						edgeB   = 1;
						maxEdge = 2;
					}
				}

				// Verify that all the edges are within our constraints
				if (edgeL[edgeA] > EDGE_MAXIMUM_SQ ||
					edgeL[edgeB] > EDGE_MAXIMUM_SQ ||
					edgeL[maxEdge] > HEDGE_MAXIMUM_SQ)
				{
					char strErr[256];
					int  iResult;
					sprintf(strErr, "The node '%s' contains a triangle that exceeds the engine's limitation of %f inches per edge (%f inches per hypotenuse).\nDo you want to continue anyway?", (char*)node->GetName(), (float)sqrt(EDGE_MAXIMUM_SQ), (float)sqrt(HEDGE_MAXIMUM_SQ));
					iResult = MessageBoxAll(gInterface->GetMAXHWnd(), strErr, "Triangle exceeeds size limitations", MB_ICONWARNING|MB_YESNO);

					if (iResult == IDYES)
						return true;

					return false;
				}
			}

			if (triObj != obj)
				triObj->DeleteThis();
		}
	}

	return true;
}

NxObject*	GeoDatabase::CreateObjectFromNode( INode* node, float scale, ULONG flags )
{
	// Before attempting to convert the object test it for the poly conversion problem
	try
	{
		Object* obj = node->EvalWorldState(0).obj;
		PolyObject* polyobj = (PolyObject*)obj->ConvertToType(0, polyObjectClassID);

		if (polyobj != obj)
			polyobj->DeleteThis();
	}
	catch(...)
	{
		char bufErr[1024];
		sprintf(bufErr, "Failed to convert the object '%s' into a poly object!  Do you want to Fix the corruption?  Fixing may result in deleted map channels that have been corrupted.  Not fixing prevents the object from being exported (which otherwise would crash)", (char*)node->GetName());
		int valResult = MessageBox(gInterface->GetMAXHWnd(), bufErr, "!! Export Poly Conversion Error !!", MB_ICONWARNING|MB_YESNO);

		if (valResult == IDYES)
		{
			node = FixPolyConvNode(node);

			if (!node)
				return NULL;
		}
		else
			return NULL;
	}

	// Ensure that the triangles in the mesh are appropriately sized for the engine
	if (!VerifyMeshTriSize(node))
		return NULL;

	//VerifyMemory();

	NxObject* object;
	Object *obj;			
	TriObject* triObject;
	Mesh *mesh;
	int i, j, k, num_faces;
	Matrix3 matrix, matrix_no_trans;
	float min_z, max_z;
	bool two_pass;
	int last_checksum, last_id;
	Mtl* actual_mtl;
	bool env_mapped, single_sided;
	
	Modifier* pMultiRes = NULL;
	LODOptions lod_options;

	if (flags & EVALLOD_SKIN)
	{
		SkinExportOptions skin_options;
		GetSkinExportOptions( &skin_options );

		lod_options.m_AlwaysLOD      = skin_options.m_AlwaysLOD;
		lod_options.m_LODLowerLimit  = skin_options.m_LODLowerLimit;
		lod_options.m_LODNumLevels   = skin_options.m_LODNumLevels;
		lod_options.m_LODProgressive = skin_options.m_LODProgressive;

		// Apply multi-res modifier at bottom of stack (thus before Physique)
		pMultiRes = EvalLODData(node, &lod_options, true);
		
		if (pMultiRes)
			SetLODFull(pMultiRes);
		//	pMultiRes->DisableMod();
	}

	if (flags & EVALLOD_MODEL)
	{
		ModelExportOptions model_options;
		GetModelExportOptions( &model_options );

		lod_options.m_AlwaysLOD      = model_options.m_AlwaysLOD;
		lod_options.m_LODLowerLimit  = model_options.m_LODLowerLimit;
		lod_options.m_LODNumLevels   = model_options.m_LODNumLevels;
		lod_options.m_LODProgressive = model_options.m_LODProgressive;

		// Apply multi-res modifier at top of stack
		pMultiRes = EvalLODData(node, &lod_options);
		
		if (pMultiRes)
			SetLODFull(pMultiRes);
			//pMultiRes->DisableMod();
	}
	

	//DisableMultiRes(node);

	// Don't export SFP Nodes
	CStr propBuf;

	node->GetUserPropBuffer(propBuf);
	if (strstr(propBuf,"SFP_Effect"))
		return NULL;

	//VerifyMemory();

	DEBUGOUT(CStr("NAME: ") + node->GetName());

#ifdef OUTPUT_DEBUG_VERTS
	FILE* fp = fopen("c:\\vertdebug.txt","a");

	fprintf(fp,"Node: %s\n",node->GetName());
#endif

	////////////////////////////////////////////////////////
	matrix = node->GetObjTMAfterWSM(0);
	//matrix = node->GetNodeTM(0);

	//Matrix3 ntm = node->GetNodeTM(0);
	//Matrix3 ptm = node->GetParentTM(0);

	//matrix = ntm * Inverse(ptm);
	//matrix.IdentityMatrix();
	//matrix = RotateXMatrix(180);

	//matrix = node->GetNodeTM(0);

	////////////////////////////////////////////////////////


	Quat qOffset = node->GetObjOffsetRot();
	//qOffset.x = 0.23483;
	//qOffset.y = 0.58733;
	//qOffset.z = 0.2;
	//qOffset.w = 1.0f;

	//PreRotateMatrix(matrix, qOffset);

	matrix_no_trans = matrix;
	matrix_no_trans.NoTrans();
	matrix_no_trans.NoScale();	

	object = new NxObject;
	//GetLODUVs(pMultiRes, node, object);

	obj = node->EvalWorldState(0).obj;
	
	CStr name = node->GetName();		// REMOVE

	// Some objects genuinely can't be converted to tri objects.  Ignore these
	//assert( obj->CanConvertToType( triObjectClassID ));
	if (!obj->CanConvertToType( triObjectClassID ))
	{
		delete object;
		return NULL;
	}	

	triObject = (TriObject *) obj->ConvertToType( 0, triObjectClassID );
	mesh = &triObject->mesh;
	if(( mesh->numVerts == 0 ) || ( mesh->numFaces == 0 ))
	{
		if (obj->ClassID()==Class_ID(SPLINE3D_CLASS_ID,0))
		{
			// Make sure the triobject has been deleted if necessary  aml
			if (triObject!=obj)
				triObject->DeleteThis();
			
			delete object;
			return NULL;
		}
		
		char msg[128];

		sprintf( msg, "Object %s has zero verts or zero faces and will not be exported", node->GetName());
		MessageBoxAll( gInterface->GetMAXHWnd(), msg, "Warning No Geometry!", MB_OK|MB_TASKMODAL);
		ReportWarning( msg );

		// Make sure the triobject has been deleted if necessary  aml
		if (triObject!=obj)
			triObject->DeleteThis();

		delete object;
		return NULL;
	}

	num_faces = mesh->getNumFaces();
	two_pass = false;

	object->m_Name = node->GetName();
	object->m_NumFaces = num_faces;
	object->m_Faces = new NxFace[object->m_NumFaces];
	object->m_NumVerts = 3 * object->m_NumFaces;		// Initially, assume each vert is unique
	object->m_Verts = new NxVertex[object->m_NumVerts];
	
	//VerifyMemory();

	// Get the object's bounding box and convert it to our coordinate system	

	/*
	object->m_BoundingBox = mesh->getBoundingBox( &node->GetObjTMAfterWSM(0));
	min_z = -object->m_BoundingBox.pmax.y;
	max_z = -object->m_BoundingBox.pmin.y;
	object->m_BoundingBox.pmin.y = object->m_BoundingBox.pmin.z;
	object->m_BoundingBox.pmin.z = min_z;
	object->m_BoundingBox.pmax.y = object->m_BoundingBox.pmax.z;
	object->m_BoundingBox.pmax.z = max_z;
	*/
	
	DEBUGOUT("Pre BoundBox\n");

	// Get the actual extents (they differ if faces have been hidden)  aml
	Point3 min,max;
	Matrix3 tm;
	
	tm=node->GetObjTMAfterWSM(0);

	if (flags & LOCAL_COORDS)
		tm.NoTrans();

	GetExtents(node,tm,min,max);
	
	min_z = -max.y;
	max_z = -min.y;

	object->m_BoundingBox.pmin.y = min.z;
	object->m_BoundingBox.pmin.z = min_z;
	object->m_BoundingBox.pmin.x = min.x;

	object->m_BoundingBox.pmax.y = max.z;
	object->m_BoundingBox.pmax.z = max_z;
	object->m_BoundingBox.pmax.x = max.x;

	DEBUGOUT("Post BoundBox\n");

	for( i = 0; i < object->m_NumVerts; i++ )
	{
		object->m_Verts[i].m_FaceIndex = i / 3;
	}

	// Mark the expanded verts with the max verts that they represent
	for( i = 0; i < object->m_NumFaces; i++ )
	{
		object->m_Verts[(i * 3) + 0].m_MaxVert = mesh->faces[i].v[0];
		object->m_Verts[(i * 3) + 1].m_MaxVert = mesh->faces[i].v[1];
		object->m_Verts[(i * 3) + 2].m_MaxVert = mesh->faces[i].v[2];
	}

	//VerifyMemory();

	// Build lookup table for MAX Vert to expanded vert
	// (These must always reference the 1st matched vert so they're still valid
	//  after the exported verts are collapsed)
	// MUST BE ASSINGED PRIOR TO EvalLOD

	// Assign MAX vert lookup data
	object->m_NumMaxVerts = mesh->numVerts;

	if (object->m_MaxVerts)
		delete [] object->m_MaxVerts;
	
	object->m_MaxVerts    = new int[object->m_NumMaxVerts];
	
	for( i = 0; i < mesh->numVerts; i++ )
	{
		//for( j = 0; j < object->m_NumVerts; j++ )
		for( j = 0; j < object->m_NumFaces * 3; j++ )
		{
			if ( object->m_Verts[j].m_Pos == mesh->verts[i] )
			{
				object->m_MaxVerts[i] = j;
			}

			if ( object->m_Verts[j].m_MaxVert == i )
			{
				object->m_MaxVerts[i] = j;
				//break;
			}
		}
	}

	//VerifyMemory();

	/*
	for( i = 0; i < mesh->numVerts; i++ )
	{
		for( j = 0; j < object->m_NumVerts; j++ )
		{
			if ( object->m_Verts[i].m_Pos == mesh->verts[j] )
			{
				object->m_MaxVerts[i] = j;
//				break;
			}

			if ( object->m_Verts[j].m_MaxVert == i )
			{
				object->m_MaxVerts[i] = j;
				break;
			}

		}
	}
	*/

	DEBUGOUT("Post Vert\n");
	//return NULL;

	int dest_index = 0;

	for( i = 0; i < num_faces; i++ )
	{
		object->m_Faces[i].m_Index = i;
		object->m_Faces[i].m_Pass = 0;	// Initialize these to first-pass faces

		dest_index = i * 3;
		for( j = 0; j < 3; j++ )
		{
			object->m_Faces[i].m_Vertex[j] = dest_index;
			dest_index++;
		}
	}

	//VerifyMemory();

	DEBUGOUT("Post Faces 1\n");

//////// Pass one acquire positions /////////////////////////////////////////////////////////////////////
	for( i = 0; i < num_faces; i++ )
	{
		Face* face;
		int index;

		face = &mesh->faces[i];
		dest_index = i * 3;
		for( j = 0; j < 3; j++ )
		{	
			index = face->getVert( j );			

			object->m_Verts[ dest_index ].m_Pos = ( mesh->verts[index] );
			dest_index++;
		}
	}

	//VerifyMemory();

///////// Pass two rotate offsets
	// Apply pivot offset rotation to the stored points in the model
	if (flags & OFFSET_ROTATION)
	{
		Quat q = node->GetObjOffsetRot();
		Matrix3 mat;
		//float   rot[3];
		mat.IdentityMatrix();
		q.MakeMatrix(mat);
		//QuatToEuler(q, rot);
		//mat.RotateX(rot[2]);
		//mat.RotateY(-rot[1]);
		//mat.RotateZ(rot[0]);

		for(int i = 0; i < dest_index; i++)
			object->m_Verts[ i ].m_Pos = object->m_Verts[ i ].m_Pos * mat;
	}

	//VerifyMemory();

	// Pass 3 apply offset matricies
	for(i = 0; i < dest_index; i++ )
	{
		if (flags & LOCAL_COORDS)
		{
			if (flags & SKEL_EXPORT)
			{
				Matrix3 nodeTM = node->GetNodeTM(0);
				nodeTM.NoTrans();
				nodeTM.NoRot();
				//TEMP
				//object->m_Verts[ dest_index ].m_Pos = mesh->verts[index] * nodeTM;
				object->m_Verts[ i ].m_Pos = object->m_Verts[ i ].m_Pos * nodeTM;
				//object->m_Verts[ dest_index ].m_Pos = mesh->verts[index];
			}
			else
			{
				// If not exporting as a skeleton, but exporting local we should still include
				// the rotation of the node transform
				Matrix3 nodeTM = node->GetNodeTM(0);
				nodeTM.NoTrans();

				object->m_Verts[ i ].m_Pos = object->m_Verts[ i ].m_Pos * nodeTM;
			}
		}
		else
		{
			//TEMP
			object->m_Verts[ i ].m_Pos = object->m_Verts[ i ].m_Pos * matrix;
			//object->m_Verts[ dest_index ].m_Pos = ( mesh->verts[index] );
		}

		if (flags & OFFSET_PIVOT)
		{
			Point3 ptOffset = node->GetObjOffsetPos();
			object->m_Verts[ i ].m_Pos += ptOffset;
		}

		if (!(flags & NOCOORD_SWAP))
		{
			float tmp;
			tmp                          = object->m_Verts[ i ].m_Pos.z;
			object->m_Verts[ i ].m_Pos.z = -object->m_Verts[ i ].m_Pos.y;
			object->m_Verts[ i ].m_Pos.y = tmp;
		}


#ifdef OUTPUT_DEBUG_VERTS
		//if (flags & SKEL_EXPORT)
		{
			fprintf(fp,"Vert[%i]: %f,%f,%f\n",dest_index,
							                          object->m_Verts[ i ].m_Pos.x,
						                              object->m_Verts[ i ].m_Pos.y,
													  object->m_Verts[ i ].m_Pos.z);
		}
#endif
	}

	//VerifyMemory();

/////////////////////// End vert output
/*
	// Gather positions
	for( i = 0; i < num_faces; i++ )
	{
		Face* face;
		int index;
		float temp;

		face = &mesh->faces[i];
		dest_index = i * 3;
		for( j = 0; j < 3; j++ )
		{	
			index = face->getVert( j );			

			if (flags & LOCAL_COORDS)
			{
				if (flags & SKEL_EXPORT)
				{
					Matrix3 nodeTM = node->GetNodeTM(0);
					nodeTM.NoTrans();
					nodeTM.NoRot();
					//TEMP
					object->m_Verts[ dest_index ].m_Pos = mesh->verts[index] * nodeTM;
					//object->m_Verts[ dest_index ].m_Pos = mesh->verts[index];
					
				}
				else
					object->m_Verts[ dest_index ].m_Pos = ( mesh->verts[index] );
			}
			else
			{
				//TEMP
				object->m_Verts[ dest_index ].m_Pos = ( mesh->verts[index] * matrix );
				//object->m_Verts[ dest_index ].m_Pos = ( mesh->verts[index] );
			}

			if (flags & OFFSET_PIVOT)
			{
				Point3 ptOffset = node->GetObjOffsetPos();
				//TEMP
				object->m_Verts[ dest_index ].m_Pos += ptOffset;
				//object->m_Verts[ dest_index ].m_Pos.x +=  ptOffset.x;
				//object->m_Verts[ dest_index ].m_Pos.y +=  ptOffset.z;
				//object->m_Verts[ dest_index ].m_Pos.z += -ptOffset.y;
				//object->m_Verts[ dest_index ].m_Pos;
			}

#ifdef OUTPUT_DEBUG_VERTS
			//if (flags & SKEL_EXPORT)
			{
				fprintf(fp,"Vert[%i]: %f,%f,%f\n",dest_index,
							                              object->m_Verts[ dest_index ].m_Pos.x,
						                                  object->m_Verts[ dest_index ].m_Pos.y,
													      object->m_Verts[ dest_index ].m_Pos.z);
			}
#endif
			dest_index++;
		}
	}
*/
	/*
	// Apply pivot offset rotation to the stored points in the model
	if (flags & OFFSET_ROTATION)
	{
		Quat q = node->GetObjOffsetRot();
		Matrix3 mat;
		float   rot[3];
		mat.IdentityMatrix();
		//q.MakeMatrix(mat,true);
		QuatToEuler(q, rot);
		mat.RotateX(rot[2]);
		mat.RotateY(-rot[1]);
		mat.RotateZ(rot[0]);

		for(int i = 0; i < dest_index; i++)
			object->m_Verts[ i ].m_Pos = object->m_Verts[ i ].m_Pos * mat;
	}

	// Apply pivot offset
	if (flags & OFFSET_PIVOT)
	{
		Point3 ptOffset = node->GetObjOffsetPos();

		for(int i = 0; i < dest_index; i++)
		{
			//TEMP
			object->m_Verts[ dest_index ].m_Pos += ptOffset;
			//object->m_Verts[ dest_index ].m_Pos.x +=  ptOffset.x;
			//object->m_Verts[ dest_index ].m_Pos.y +=  ptOffset.z;
			//object->m_Verts[ dest_index ].m_Pos.z += -ptOffset.y;
			//object->m_Verts[ dest_index ].m_Pos;
		}
	}
	*/

#ifdef OUTPUT_DEBUG_VERTS
	fclose(fp);
#endif

	//VerifyMemory();

	DEBUGOUT("Post Positions\n");

	// Add occluder and shadow volume flags if applicable
	CStr propBuffer;
	node->GetUserPropBuffer(propBuffer);
	if (Instr(propBuffer,"Occluder")!=-1)
		object->m_Flags |= NxObject::mOCCLUDER;

	if (Instr(propBuffer, "ShadowVolume = TRUE")!=-1)
		object->m_Flags |= NxObject::mSHADOWVOLUME;

	// Either the colors existed or we fill them in with the defaults
	object->m_Flags |= NxObject::mCOLORED;

	if( m_DynamicallyLit && !(flags & NONORMALS))
	{
		object->m_Flags |= NxObject::mNORMALS;
	}

	// Flag the object as AbsentInNetGames
	if (GetPropValueDef(pPropEdit, node, "AbsentInNetGames") == CStr("TRUE"))
	{
		object->m_Flags |= NxObject::mABSENTINNETGAMES;
	}

	DEBUGOUT("Pre Vert Col\n");
	//return NULL;

	// Grab color data if it exists
	if( mesh->getNumVertCol() > 0 )
	{
		TVFace* alpha_faces;
		UVVert* alpha_verts;		

		alpha_faces = mesh->mapFaces( MAP_ALPHA );
		alpha_verts = mesh->mapVerts( MAP_ALPHA );
		for( i = 0; i < num_faces; i++ )
		{
			//TVFace* face;			
			int index, dest_index, alpha_index;		

			UVVert* colors = mesh->mapVerts(0);
			TVFace* faces  = mesh->mapFaces(0);

			//face = &mesh->vcFace[i];

			dest_index = i * 3;			
			for( j = 0; j < 3; j++ )
			{
				//index = face->getTVert( j );
				//index = faces[i].t[j];

				index = faces[i].t[j];

				// This doesn't seem to always be accurate, get from map channel 0 (color channel) instead
				//object->m_Verts[ dest_index ].m_Color.r = mesh->vertCol[index].x;
				//object->m_Verts[ dest_index ].m_Color.g = mesh->vertCol[index].y;
				//object->m_Verts[ dest_index ].m_Color.b = mesh->vertCol[index].z;

				object->m_Verts[ dest_index ].m_Color.r = colors[index].x;
				object->m_Verts[ dest_index ].m_Color.g = colors[index].y;
				object->m_Verts[ dest_index ].m_Color.b = colors[index].z;
			

				object->m_Verts[ dest_index ].m_Color.a = 1.0f;	// Default to opaque
				if( alpha_verts && alpha_faces )
				{
					alpha_index = alpha_faces[ i ].getTVert( j );
					// Alpha in the alpha map channel is stored as a percentage 
					// (i.e. a float between 0 and 1)
					object->m_Verts[ dest_index ].m_Color.a = alpha_verts[ alpha_index ].x;
				}	
				dest_index++;
			}
		}
	}
	else
	{
		// Old method wasn't taking into account alpha for faces without vert colors   aml
		TVFace* alpha_faces;
		UVVert* alpha_verts;		

		alpha_faces = mesh->mapFaces( MAP_ALPHA );
		alpha_verts = mesh->mapVerts( MAP_ALPHA );
		for( i = 0; i < num_faces; i++ )
		{
			int dest_index, alpha_index;		

			dest_index = i * 3;			
			for( j = 0; j < 3; j++ )
			{
				// No vert colors default to white
				object->m_Verts[ dest_index ].m_Color.r = 1.0f;
				object->m_Verts[ dest_index ].m_Color.g = 1.0f;
				object->m_Verts[ dest_index ].m_Color.b = 1.0f;

				if( alpha_verts && alpha_faces )
				{
					alpha_index = alpha_faces[ i ].getTVert( j );
					// Alpha in the alpha map channel is stored as a percentage 
					// (i.e. a float between 0 and 1)
					object->m_Verts[ dest_index ].m_Color.a = alpha_verts[ alpha_index ].x;
				}
				else
				{
					// Default to opaque
					object->m_Verts[ dest_index ].m_Color.a = 1.0f;			
				}

				dest_index++;
			}

		}

		/*				
		for( i = 0; i < object->m_NumVerts; i++ )
		{
			object->m_Verts[i].m_Color.r = 1.0f;
			object->m_Verts[i].m_Color.g = 1.0f;
			object->m_Verts[i].m_Color.b = 1.0f;
			object->m_Verts[i].m_Color.a = 1.0f;	// Default to opaque
		}
		*/
	}

	//VerifyMemory();

	DEBUGOUT("Post Vert Col\n");
	//return NULL;

	// Grab all wibble index info if it exists
	if( mesh->mapSupport( vWIBBLE_INDEX_CHANNEL ))
	{
		TVFace* wibble_faces;
		UVVert* wibble_verts;

		wibble_faces = mesh->mapFaces( vWIBBLE_INDEX_CHANNEL );
		wibble_verts = mesh->mapVerts( vWIBBLE_INDEX_CHANNEL );
		if( wibble_faces && wibble_verts )
		{
			TVFace* face;
			int index, dest_index;

			object->m_Flags |= NxObject::mVCWIBBLE;		

			for( i = 0; i < num_faces; i++ )
			{
				face = &wibble_faces[i];
				dest_index = i * 3;
				for( j = 0; j < 3; j++ )
				{
					index = face->getTVert( j );
					object->m_Verts[dest_index].m_WibbleIndex = *(int *) &wibble_verts[index];
					dest_index++;
				}
			}
		}
	}

	//VerifyMemory();

	DEBUGOUT("Post Wibble Index\n");

	// Grab all wibble offset info if it exists
	if( mesh->mapSupport( vWIBBLE_OFFSET_CHANNEL ))
	{
		TVFace* wibble_faces;
		UVVert* wibble_verts;

		wibble_faces = mesh->mapFaces( vWIBBLE_OFFSET_CHANNEL );
		wibble_verts = mesh->mapVerts( vWIBBLE_OFFSET_CHANNEL );
		if( wibble_faces && wibble_verts )
		{
			TVFace* face;
			int index, dest_index;

			for( i = 0; i < num_faces; i++ )
			{
				face = &wibble_faces[i];
				dest_index = i * 3;
				for( j = 0; j < 3; j++ )
				{
					index = face->getTVert( j );
					object->m_Verts[dest_index].m_WibbleOffset = *(int *) &wibble_verts[index];
					dest_index++;
				}
			}
		}
	}
	
	//VerifyMemory();
	DEBUGOUT("Post Wibble Offset\n");

	// Computer vertex normals according to smoothing groups
	{
		Face* face;	
		Point3* verts;
		Point3 v0, v1, v2;
		Tab<VNormal> vnorms;
		Tab<Point3> fnorms;

		face = mesh->faces;	
		verts = mesh->verts;

		int nVerts = mesh->getNumVerts();
		int nFaces = mesh->getNumFaces();

		vnorms.SetCount(nVerts);
		fnorms.SetCount(nFaces);

		// Compute face and vertex surface normals
		for( i = 0; i < nVerts; i++ ) 
		{
			vnorms[i] = VNormal();
		}

		for( i = 0; i < nFaces; i++, face++ ) 
		{
			// Calculate the surface normal
			v0 = verts[face->v[0]];
			v1 = verts[face->v[1]];
			v2 = verts[face->v[2]];
			fnorms[i] = (v1-v0)^(v2-v1);
			for( j = 0; j < 3; j++ ) 
			{
				vnorms[face->v[j]].AddNormal( fnorms[i], face->smGroup );
			}
			fnorms[i] = Normalize( fnorms[i] );
		}
		for( i = 0; i < nVerts; i++ ) 
		{
			vnorms[i].Normalize();
		}

		for( i = 0; i < nFaces; i++ )
		{
			Face* face;
			int index, dest_index;
			float temp;

			face = &mesh->faces[i];

			// If we've flipped the verts we will also need to flip face normals
			// NOTE! It's ! FLIP_VERTS because the system is usually flipped this way in the general case
			if (!(flags & FLIP_VERTS))
			{
				/* No normals will only apply to vertex normals
				if (flags & NONORMALS)
				{
					object->m_Faces[i].m_Normal.x = 0.0f;
					object->m_Faces[i].m_Normal.y = 0.0f;
					object->m_Faces[i].m_Normal.z = 0.0f;
				}
				else
				*/
				{
					object->m_Faces[i].m_Normal = fnorms[i] * matrix_no_trans;
					temp = object->m_Faces[i].m_Normal.y;
					object->m_Faces[i].m_Normal.y = object->m_Faces[i].m_Normal.z;
					object->m_Faces[i].m_Normal.z = -temp;
				}
			}

			dest_index = i * 3;

			for( j = 0; j < 3; j++ )
			{			
				index = face->getVert( j );			
				object->m_Verts[ dest_index ].m_Normal = vnorms[index].GetNormal( face->smGroup ) * matrix_no_trans;

				// If we've flipped the verts we will need to also flip the normals
				// NOTE!  It's ! FLIP_VERTS because the system is usually flipped this way in the general case  aml
				if (!(flags & FLIP_VERTS))
				{
					if (flags & NONORMALS)
					{
						object->m_Verts[ dest_index ].m_Normal.x = 0.0f;
						object->m_Verts[ dest_index ].m_Normal.y = 0.0f;
						object->m_Verts[ dest_index ].m_Normal.z = 0.0f;
					}
					else
					{
						temp = object->m_Verts[ dest_index ].m_Normal.y;
						object->m_Verts[ dest_index ].m_Normal.y = object->m_Verts[ dest_index ].m_Normal.z;
						object->m_Verts[ dest_index ].m_Normal.z = -temp;
					}
				}

				dest_index++;
			}
		}		
	}	
	
	//VerifyMemory();
	DEBUGOUT("Post Vert Norm\n");

	// Gather face flag data if it exists
	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh->GetInterface( FACEDATAMGR_INTERFACE ));
	if ( pFDMgr )
	{	
		/* Face flags are saved out later when passes are adjusted later in this func
		FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
		if( fdc ) 
		{
			for( i = 0; i < num_faces; i++ )
			{
				fdc->GetValue( i, object->m_Faces[i].m_FaceFlags );
			}
		}
		else
		{
			for( i = 0; i < num_faces; i++ )
			{
				object->m_Faces[i].m_FaceFlags = 0;
			}
		}
		//*/

		// Gather CAS face flag data if it exists
		// Get the face-data channel from the incoming mesh	
		CASFaceFlagsData* CASfdc = dynamic_cast<CASFaceFlagsData*>( pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));
		if( CASfdc ) 
		{
			for( i = 0; i < num_faces; i++ )
			{
				CASfdc->GetValue( i, object->m_Faces[i].m_CASFaceFlags );
			}
		}
		else
		{
			for( i = 0; i < num_faces; i++ )
			{
				object->m_Faces[i].m_CASFaceFlags = 0;
			}
		}
	}

	//VerifyMemory();
	DEBUGOUT("Post CAS Data\n");

	// Grab all texture coordinate data if it exists
	if( mesh->getNumTVerts() > 0 )
	{
		TVFace* face_lists[vMAX_MATERIAL_PASSES];
		UVVert* tvert_lists[vMAX_MATERIAL_PASSES];

		object->m_Flags |= NxObject::mTEXTURED;
				
		int pass_channel[vMAX_MATERIAL_PASSES];	// Stores the mapping channel used for each pass

		// BOOKMARK: UVVerts mapping channel 1, 2, 3, etc.  Needs changed to mapping channel  aml

		// This is going to require that each pass use maps 1, 2, 3, etc.
		// We want the user to be able to specify their own mapping channels  aml
		//face_lists[i] = mesh->mapFaces( i + 1 );
		//tvert_lists[i] = mesh->mapVerts( i + 1 );

		// Find the map channel the user selected for this pass
		// First we need to get at the material so we can ultimately get down to the textures
		// used for each pass of the NExtMaterial

		bool bUVWarn[vMAX_MATERIAL_PASSES] = { false };

#ifdef WRITE_DEBUG_FILE
		FILE* fpDebug=fopen("c:\\texdebug.txt","w");
#endif

		for( i = 0; i < object->m_NumFaces; i++)
		{
			Mtl* actual_mtl = GetSubMaterial( node->GetMtl(), mesh->faces[i].getMatID());
			
			if (!actual_mtl)
				actual_mtl = node->GetMtl();
			
			int channel = 1;
			int lastOutputPass = 0;
			int index, dest_index;
			TVFace* face;

			if (actual_mtl)
			{
				if (actual_mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
				{
					INExtMaterial* next_mat = dynamic_cast<INExtMaterial*>(actual_mtl);

					for(int pass = 0; pass < vMAX_MATERIAL_PASSES; pass++)
					{
						Texmap* tex = next_mat->GetTexmap(pass);

						if (tex && next_mat->MapEnabled(pass))
						{
							UVGen* uvgen = tex->GetTheUVGen();

							if (uvgen)
							{
								// Map channels are only used if the UVGen is set to explicit
								if (uvgen->GetUVWSource()==UVWSRC_EXPLICIT)
									channel = tex->GetMapChannel();
								else
									channel = pass + 1;
							}
							else
							{
								// No UVGen on material (assume incremental)
								channel = pass + 1;
							}

							// Grab data for current channel
							face_lists[lastOutputPass]  = mesh->mapFaces(channel);
							tvert_lists[lastOutputPass] = mesh->mapVerts(channel);

							/*
							if (face_lists[lastOutputPass] == NULL ||
								tvert_lists[lastOutputPass] == NULL)
							{
								if (!bUVWarn[pass])
								{
									char strBuf[256];
									sprintf(strBuf,"WARNING!  The mapping channel %i referenced by material '%s' pass %i does not appear to exist on object '%s'.  Pass %i will be disabled on this material.",channel,(char*)actual_mtl->GetName(),pass+1,(char*)node->GetName(),pass+1);
									MessageBoxAll(gInterface->GetMAXHWnd(),strBuf,"Missing UV Map Channel",MB_OK);
									bUVWarn[pass] = true;
								}
							}
							*/

							if( face_lists[lastOutputPass] && tvert_lists[lastOutputPass] )
							{
								face = &face_lists[lastOutputPass][i];
								dest_index = i * 3; 

								for( k = 0; k<3 ; k++)
								{
									index = face->getTVert( k );
									object->m_Verts[ dest_index ].m_TexCoord[lastOutputPass] = tvert_lists[lastOutputPass][index];

									// Verify UVs within range for GameCube
									if ((tvert_lists[lastOutputPass][index].x < -32.0f &&
										 tvert_lists[lastOutputPass][index].x > 31.0f) ||
										(tvert_lists[lastOutputPass][index].y < -32.0f &&
										 tvert_lists[lastOutputPass][index].y > 31.0f))
									{
										char sbuf[256], tbuf[256];
										sprintf(sbuf,"WARNING! %s has UV coordinates that fail to fall between -32 and 31.  This is unsupported on GameCube.",object->m_Name);
										sprintf(tbuf,"%s UVs out of range",object->m_Name);
										MessageBoxAll(gInterface->GetMAXHWnd(),sbuf,tbuf,MB_OK);
										ReportWarning( sbuf );
									}

#ifdef WRITE_DEBUG_FILE
									fprintf(fpDebug,"Face: %i Channel: %i Vert[%i].m_TexCoord[pass: %i] = [%f,%f]\n",i,channel,dest_index, lastOutputPass, tvert_lists[lastOutputPass][index].x, tvert_lists[lastOutputPass][index].y);
#endif
									dest_index++;
								}

								lastOutputPass++;
							}
							else	// Use default mapping coordinates
							{
								object->m_Verts[object->m_Faces[i].m_Vertex[0]].m_TexCoord[pass].x = 0.0f;
								object->m_Verts[object->m_Faces[i].m_Vertex[0]].m_TexCoord[pass].y = 0.0f;
								object->m_Verts[object->m_Faces[i].m_Vertex[0]].m_TexCoord[pass].z = 0.0f;

								object->m_Verts[object->m_Faces[i].m_Vertex[1]].m_TexCoord[pass].x = 0.0f;
								object->m_Verts[object->m_Faces[i].m_Vertex[1]].m_TexCoord[pass].y = 0.0f;
								object->m_Verts[object->m_Faces[i].m_Vertex[1]].m_TexCoord[pass].z = 0.0f;

								object->m_Verts[object->m_Faces[i].m_Vertex[2]].m_TexCoord[pass].x = 0.0f;
								object->m_Verts[object->m_Faces[i].m_Vertex[2]].m_TexCoord[pass].y = 0.0f;
								object->m_Verts[object->m_Faces[i].m_Vertex[2]].m_TexCoord[pass].z = 0.0f;
							
#ifdef WRITE_DEBUG_FILE
								fprintf(fpDebug,"Face: %i Channel: %i (Defaulted) Vert[0].m_TexCoord[pass: %i] = [%f,%f]\n",i,channel,lastOutputPass,0.0f,0.0f);
								fprintf(fpDebug,"Face: %i Channel: %i (Defaulted) Vert[1].m_TexCoord[pass: %i] = [%f,%f]\n",i,channel,lastOutputPass,0.0f,0.0f);
								fprintf(fpDebug,"Face: %i Channel: %i (Defaulted) Vert[2].m_TexCoord[pass: %i] = [%f,%f]\n",i,channel,lastOutputPass,0.0f,0.0f);
#endif
								lastOutputPass++;
							}
						}
					}
				}
				else
				{
					// Not a NExt material
					// We assume it's a standard material and only take the diffuse map
					Texmap* tex = actual_mtl->GetSubTexmap(ID_DI);

					if (tex)
					{
						UVGen* uvgen = tex->GetTheUVGen();

						if (uvgen)
						{
							if (uvgen->GetUVWSource()==UVWSRC_EXPLICIT)
							{
								int channel = tex->GetMapChannel();

								face_lists[0]   = mesh->mapFaces( 1 );
								tvert_lists[0]  = mesh->mapVerts( 1 );
								pass_channel[0] = 1;

								if ( face_lists[0] && tvert_lists[0] )
								{
									// Set the TVerts for this specific face
									face = &face_lists[0][i];
									dest_index = i * 3;

									for( k = 0; k<3 ; k++)
									{
										index = face->getTVert( k );
										object->m_Verts[ dest_index ].m_TexCoord[0] = tvert_lists[lastOutputPass][index];
										dest_index++;
									}

									lastOutputPass = 1;
								}
							}
						}
					}
					else
					{
						// No diffuse map, just pick the first channel
						face_lists[0]   = mesh->mapFaces( 1 );
						tvert_lists[0]  = mesh->mapVerts( 1 );
						pass_channel[0] = 1;

						if ( face_lists[0] && tvert_lists[0] )
						{
							// Set the TVerts for this specific face
							face = &face_lists[0][i];
							dest_index = i * 3;

							for( k = 0; k<3 ; k++)
							{
								index = face->getTVert( k );
								object->m_Verts[ dest_index ].m_TexCoord[0] = tvert_lists[lastOutputPass][index];
								dest_index++;
							}

							lastOutputPass = 1;
						}
					}

				}
			}
			else
			{
				// No material assigned, use first map channel
				face_lists[0]  = mesh->mapFaces( 1 );
				tvert_lists[0] = mesh->mapVerts( 1 );

				if ( face_lists[0] && tvert_lists[0] )
				{
					// Set the TVerts for this specific face
					face = &face_lists[0][i];
					for( k = 0; k<3 ; k++)
					{
						index = face->getTVert( k );
						dest_index = i * 3;
						object->m_Verts[ dest_index ].m_TexCoord[0] = tvert_lists[lastOutputPass][index];
						dest_index++;
					}

					lastOutputPass = 1;
				}
			}

			// Set the object's number of UV sets to the maximum number of UVs of any
			// given face within the mesh
			if ( object->m_NumUVSets < lastOutputPass)
				object->m_NumUVSets = lastOutputPass;
		}

#ifdef WRITE_DEBUG_FILE
		fclose(fpDebug);
#endif

	}

	//VerifyMemory();
	DEBUGOUT("Post Texture coord\n");

	s_already_warned_nested = false;

	// Get the name of the parent material
	Mtl* nodeMtl = node->GetMtl();
	CStr parentMtlName;

	if (nodeMtl)
		parentMtlName = nodeMtl->GetName();
	
	actual_mtl = NULL;
	last_checksum = -1;
	last_id = -1;
	env_mapped = false;
	single_sided = false;
	// Gather up all used materials
	for( i = 0; i < num_faces; i++ )
	{
		// As an optimization, don't go through the process of adding a material to the
		// database if we just dealt with it.
		if( last_id != mesh->faces[i].getMatID())
		{
			actual_mtl = GetSubMaterial( node->GetMtl(), mesh->faces[i].getMatID());
			last_id = mesh->faces[i].getMatID();

			env_mapped = false;
			single_sided = false;
			
			if (actual_mtl)
				object->m_Faces[i].m_MatChecksum = AddMaterial( actual_mtl, 
																env_mapped, 
																single_sided, 
																parentMtlName );
			else
				object->m_Faces[i].m_MatChecksum = 0;

			last_checksum = object->m_Faces[i].m_MatChecksum;
		}
		else
		{
			object->m_Faces[i].m_MatChecksum = last_checksum;
		}
		
		if( env_mapped && !(flags & NONORMALS))
		{
			object->m_Flags |= NxObject::mNORMALS;
		}
		if( single_sided )
		{
			// We will use normals in sceneconv to determine tristrip facing direction, even
			// if the mesh is not dynamically lit nor environment-mapped
			object->m_Flags |= NxObject::mSS_NORMALS;			
		}		

		// Assign face flags after reassigning the included pass flags to match material passes
		// now in sequential order   aml
		FlagType ftype = AdjustPassFlag(actual_mtl, mesh, i);

		// Handle skater shadow flag assignment
		if ((ftype & mFD_NON_COLLIDABLE) && !(ftype & mFD_SKATER_SHADOW) && !(ftype & mFD_NO_SKATER_SHADOW_WALL))
			ftype |= mFD_NO_SKATER_SHADOW;

		object->m_Faces[i].m_FaceFlags = ftype;

		// Assign face flags after reassigning the included pass flags to match material passes
		// now in sequential order   aml
		ftype = AdjustMinPassFlag(actual_mtl, mesh, i);		

		// Handle skater shadow flag assignment
		if ((ftype & mFD_NON_COLLIDABLE) && !(ftype & mFD_SKATER_SHADOW) && !(ftype & mFD_NO_SKATER_SHADOW_WALL))
			ftype |= mFD_NO_SKATER_SHADOW;

		object->m_Faces[i].m_MinFaceFlags = ftype;

		/* DEBUG
		if (!(object->m_Faces[i].m_FaceFlags & mFD_PASS_1_DISABLED))
			DEBUGOUT("Pass 1 Enabled\n");

		if ((object->m_Faces[i].m_FaceFlags & mFD_PASS_2_ENABLED))
			DEBUGOUT("Pass 2 Enabled\n");

		if ((object->m_Faces[i].m_FaceFlags & mFD_PASS_3_ENABLED))
			DEBUGOUT("Pass 3 Enabled\n");

		if ((object->m_Faces[i].m_FaceFlags & mFD_PASS_4_ENABLED))
			DEBUGOUT("Pass 4 Enabled\n");
		*/
	}

	//VerifyMemory();
	DEBUGOUT("Post Mtls\n");

	// Compute the mesh's transformed UVs from its material's UVGen	   aml
	UpdateFaceUVs(node, object);
	
	//VerifyMemory();
	DEBUGOUT("Post Face UVs\n");

	if( m_Skinned )
	{
		ISkinExporter* skin_exp;

		skin_exp = GetSkinExporter();

		if( skin_exp->GetSkinData( object, node, object->m_SkeletonChecksum, m_SkinRootBone ) == false )
		{
			delete object;
			return NULL;
		}

		object->m_Flags |= NxObject::mSKINNED;

		// From now on all newly created objects will use a variable number of weights
		object->m_Flags |= NxObject::mHAS4WEIGHTS;

		// Acquire CAS Removal flags
		CStr strRemoval = GetPropValue(node, "RemovalFlags");
		if (strRemoval.Length()>0)
		{
			object->m_Flags |= NxObject::mHASCASREMOVEFLAGS;
			object->m_CASRemoveFlags = TranslateCASRemovalFlags(strRemoval);
		}
		else
			object->m_CASRemoveFlags = 0;
	}	

	//VerifyMemory();
	DEBUGOUT("Post Skinned\n");

	// Acquire object level KBias
	if (node->GetUserPropFloat("MipMapKBias",object->m_KBias))
		object->m_Flags |= NxObject::mHASKBIAS;

	// Acquire LOD information
	object->m_LODVersion = NxLODLevel::vVERSION_NUMBER;
	object->m_LODMaster.m_NumLODLevels = LODBrowser::GetNumLODLevels(node);

	// LODs that are assigned to themselves as masters 
	int LODtype = LODBrowser::GetLODType(node);

	INode* nodeMaster = LODBrowser::GetLODMaster(node);

	if (nodeMaster != NULL &&
		nodeMaster == node)
		LODtype = LODBrowser::LODTYPE_MASTER;

	switch(LODtype)
	{
	case LODBrowser::LODTYPE_SLAVE:
		{
			object->m_Flags |= NxObject::mHASLODINFO;
			object->m_LODFlags = NxObject::mSLAVE;
		
			CStr name = node->GetName();

			INode* nodeMaster = LODBrowser::GetLODMaster(node);
			assert(nodeMaster != node);

			if (!nodeMaster)
				object->m_LODSlave.m_masterCRC = 0;
			else
				object->m_LODSlave.m_masterCRC = GenerateCRC(nodeMaster->GetName());
		}	
		break;

	case LODBrowser::LODTYPE_MASTER:
		{
			if (object->m_LODMaster.m_NumLODLevels > 0)
			{
				object->m_LODFlags = NxObject::mMASTER;
				object->m_Flags |= NxObject::mHASLODINFO;

				if (object->m_LODMaster.m_LODLevels)
					delete [] object->m_LODMaster.m_LODLevels;

				object->m_LODMaster.m_LODLevels = new NxLODLevel[object->m_LODMaster.m_NumLODLevels];

				LODBrowser::GetLODLevels(node,object->m_LODMaster.m_LODLevels);
			}
			else
				object->m_LODFlags = 0;
			
			break;
		}

	case LODBrowser::LODTYPE_NONE:
		{
			object->m_LODFlags = 0;
		}
		break;
	}

	// Acquire LOD data from new LOD modifier
	//Object* objLOD = node->EvalWorldState(0).obj;
	LODMod* lodMod = (LODMod*)FindModifier(node, vLODMOD_CLASS_ID);

	if (lodMod)
	{
		// This is a master LOD node
		LODtype = LODBrowser::LODTYPE_MASTER;
		object->m_LODMaster.m_NumLODLevels = lodMod->GetNumLODs();

		if (object->m_LODMaster.m_NumLODLevels > 0)
		{
			object->m_LODFlags = NxObject::mMASTER;
			object->m_Flags |= NxObject::mHASLODINFO;

			if (object->m_LODMaster.m_LODLevels)
				delete [] object->m_LODMaster.m_LODLevels;

			object->m_LODMaster.m_LODLevels = new NxLODLevel[object->m_LODMaster.m_NumLODLevels];

			// Fill in LOD level data  (LODBrowser and LOD Modifier both store in the same PE compatible type)
			LODBrowser::GetLODLevels(node,object->m_LODMaster.m_LODLevels);
		}
		else
			object->m_LODFlags = 0;
	}
	else
	{
		// If this node references a LODMod then its a slave LOD node
		RefList& refList = node->GetRefList();
		RefListItem* refItem = refList.FirstItem();
		LODMod* lodMOD = NULL;

		while(refItem)
		{
			if (refItem->maker->ClassID() == vLODMOD_CLASS_ID)
			{
				lodMOD = (LODMod*)refItem->maker;
				break;
			}

			refItem = refItem->next;
		}

		if (lodMOD)
		{
			// This is a slave LOD node
			LODtype = LODBrowser::LODTYPE_SLAVE;

			object->m_Flags |= NxObject::mHASLODINFO;
			object->m_LODFlags = NxObject::mSLAVE;
		
			CStr name = node->GetName();

			INode* nodeMaster = lodMOD->GetNode();
			assert(nodeMaster != node);

			if (nodeMaster)
			{
				CStr nodemasterName = nodeMaster->GetName();

				if (!nodeMaster)
					object->m_LODSlave.m_masterCRC = 0;
				else
					object->m_LODSlave.m_masterCRC = GenerateCRC(nodemasterName);
			}
		}
		else
		{
			// This is a non-LOD node
			LODtype = LODBrowser::LODTYPE_NONE;
		}
	}


	//VerifyMemory();
	DEBUGOUT("Post LOD\n");

	// Acquire relative information
	INode* nodeParent = node->GetParentNode();

	// Get parent
	if (nodeParent->IsRootNode())
		object->m_ParentCRC = 0;
	else
	{
		CStr name = nodeParent->GetName();
		object->m_ParentCRC = GenerateCRC(name);
	}

	// Get children
	int nKids = node->NumberOfChildren();

	object->m_NumChildren = nKids;

	if (object->m_ChildCRCs)
		delete [] object->m_ChildCRCs;

	object->m_ChildCRCs = new unsigned long[nKids];

	for(i = 0; i < nKids; i++)
	{
		CStr name = node->GetChildNode(i)->GetName();
		object->m_ChildCRCs[i] = GenerateCRC(name);
	}

	DEBUGOUT("Post Hierarchy\n");

	// Make sure the triobject has been deleted if necessary  aml
	if (triObject!=obj)
		triObject->DeleteThis();

	//VerifyMemory();
	DEBUGOUT("Post Deletion\n");

	//EnableMultiRes(node);

	// Extract the actual LOD info
	if (flags & EVALLOD_SKIN)
	{
		if (pMultiRes)
			pMultiRes->EnableMod();

		EvalLODData(pMultiRes, node, object, &lod_options);

		/*
		// Skins should not maintain MultiRes as it seems to screw up the consistency with
		// the Physique data for some reason
		if (pMultiRes)
		{
			int idx = GetMultiResIndex(node);
			
			if (idx > -1)
				RemoveModifier(node, idx);
		}
		*/
	}
	else if (flags & EVALLOD_MODEL)
	{
		if (pMultiRes)
			pMultiRes->EnableMod();

		EvalLODData(pMultiRes, node, object, &lod_options);
	}

	// Assign Billboard properties (if appropriate)
	if (!GetBillboardProperties(node, object))
		return NULL;

	//return NULL;
	
	//VerifyMemory();
	return object;
}

bool	GeoDatabase::GetBillboardProperties(INode* node, NxObject* object)
{
	float ftmp;
	CStr  propBuffer;
	node->GetUserPropBuffer(propBuffer);

	// Determinne the pivot axis
	Object* obj = node->EvalWorldState(0).obj;

	if (!obj->CanConvertToType(triObjectClassID))
		return false;

	TriObject* triobj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

	if (!triobj)
		return false;

	Mesh& mesh = triobj->GetMesh();
	AdjEdgeList elist(mesh);

	// Compute the normal axis billboard plane
	Matrix3 OTM = node->GetObjTMAfterWSM(0);
	Point3 e0   = (mesh.verts[mesh.faces[0].v[1]] * OTM) - (mesh.verts[mesh.faces[0].v[0]] * OTM);
	Point3 e1   = (mesh.verts[mesh.faces[0].v[2]] * OTM) - (mesh.verts[mesh.faces[0].v[0]] * OTM);
	Point3 norm = e1 ^ e0;
	norm = norm.Normalize();
	
	norm.x = (float)fabs(norm.x);
	norm.y = (float)fabs(norm.y);
	norm.z = (float)fabs(norm.z);

	CStr  className  = GetClassName(propBuffer);
	CStr  typeName   = GetTypeName(propBuffer);
	CStr  defValMode = pPropEdit->GetDefault(className, typeName, "Billboard/BillboardMode");
	CStr  defValObj  = pPropEdit->GetDefault(className, typeName, "Billboard/AxisObject");

	Matrix3 TM = node->GetNodeTM(0);
	object->m_PivotPos = TM.GetTrans() + node->GetObjOffsetPos();

	// Convert Pivot offset to the game's coordinate system
	ftmp = object->m_PivotPos.z;
	object->m_PivotPos.z = object->m_PivotPos.y;
	object->m_PivotPos.y = ftmp;

	object->m_BillboardOrigin = TM.GetTrans();

	// Convert Billboard Origin to the game's coordinate system
	ftmp = object->m_BillboardOrigin.z;
	object->m_BillboardOrigin.z = -object->m_BillboardOrigin.y;
	object->m_BillboardOrigin.y = ftmp;

	CStr strBillboardMode = GetPropValue(node, "Billboard/BillboardMode");

	if (strBillboardMode == CStr("!"))
		strBillboardMode = defValMode;

	if (strBillboardMode == CStr("") ||
		strBillboardMode == CStr("Off"))
	{
		object->m_BillboardType = NxObject::BBT_NOT_BILLBOARD;
		return true;
	}
	else
	{
		if (strstr(strBillboardMode, "Screen_Aligned"))
			object->m_BillboardType = NxObject::BBT_SCREEN_ALIGNED;
		else if (strstr(strBillboardMode, "X_Axis_WRLD"))
		{
			object->m_BillboardType = NxObject::BBT_AXIAL_ALIGNED;
			object->m_PivotAxis = Point3(1.0f, 0.0f, 0.0f);
		}
		else if (strstr(strBillboardMode, "Y_Axis_WRLD"))
		{
			object->m_BillboardType = NxObject::BBT_AXIAL_ALIGNED;
			object->m_PivotAxis = Point3(0.0f, 1.0f, 0.0f);
		}
		else if (strstr(strBillboardMode, "Z_Axis_WRLD"))
		{
			object->m_BillboardType = NxObject::BBT_AXIAL_ALIGNED;
			object->m_PivotAxis = Point3(0.0f, 0.0f, 1.0f);
		}
		else if (strstr(strBillboardMode, "X_Axis_LOC"))
		{
			object->m_BillboardType = NxObject::BBT_AXIAL_ALIGNED;

			// Choose the axis then rotate it with the node's transform
			// to orient the vector appropriately
			Matrix3 TMR = TM;
			TMR.NoTrans();

			object->m_PivotAxis = Point3(1.0f, 0.0f, 0.0f);
			object->m_PivotAxis = object->m_PivotAxis * TMR;

			if (object->m_PivotAxis == norm)
			{
				char strErr[256];
				sprintf(strErr, "Billboard node '%s' is being exported on Local X which is degenerate for the given geometry.  Cannot export.", (char*)node->GetName());
				MessageBoxAll(gInterface->GetMAXHWnd(), strErr, "Billboard Export Failed (X)", MB_ICONWARNING|MB_OK);
				ReportWarning( strErr );
				return false;				
			}
		}
		else if (strstr(strBillboardMode, "Y_Axis_LOC"))
		{
			object->m_BillboardType = NxObject::BBT_AXIAL_ALIGNED;

			Matrix3 TMR = TM;
			TMR.NoTrans();

			object->m_PivotAxis = Point3(0.0f, 1.0f, 0.0f);
			object->m_PivotAxis = object->m_PivotAxis * TMR;

			if (object->m_PivotAxis == norm)
			{
				char strErr[256];
				sprintf(strErr, "Billboard node '%s' is being exported on Local Y which is degenerate for the given geometry.  Cannot export.", (char*)node->GetName());
				MessageBoxAll(gInterface->GetMAXHWnd(), strErr, "Billboard Export Failed (Y)", MB_ICONWARNING|MB_OK);
				ReportWarning( strErr );
				return false;				
			}
		}
		else if (strstr(strBillboardMode, "Z_Axis_LOC"))
		{
			object->m_BillboardType = NxObject::BBT_AXIAL_ALIGNED;

			Matrix3 TMR = TM;
			TMR.NoTrans();

			object->m_PivotAxis = Point3(0.0f, 0.0f, 1.0f);
			object->m_PivotAxis = object->m_PivotAxis * TMR;

			if (object->m_PivotAxis == norm)
			{
				char strErr[256];
				sprintf(strErr, "Billboard node '%s' is being exported on Local Z which is degenerate for the given geometry.  Cannot export.", (char*)node->GetName());
				MessageBoxAll(gInterface->GetMAXHWnd(), strErr, "Billboard Export Failed (Z)", MB_ICONWARNING|MB_OK);
				ReportWarning( strErr );
				return false;				
			}
		}
		else if (strstr(strBillboardMode, "Short_Axis") ||
			     strstr(strBillboardMode, "Long_Axis"))
		{
			object->m_BillboardType = NxObject::BBT_AXIAL_ALIGNED;

			// Determinne the pivot axis
			int nEdges = elist.edges.Count();
			// We should assume we have a quad and as such we should have 5 edges
			// 4 + 1 shared hypotenuse between the two tris making it up

			if (nEdges == 5)
			{
				// Throw out the longest edge
				float edgeLen[5];
				int   edgeLong[2], edgeShort[2];
				int   excludeEdge = 0;

				edgeLong[0]  = edgeLong[1]  = 0;
				edgeShort[0] = edgeShort[1] = 0;

				int i;

				for(i = 0; i < 5; i++)
				{
					edgeLen[i] = Length(mesh.verts[elist.edges[i].v[1]] - mesh.verts[elist.edges[i].v[0]]);

					if (edgeLen[i] > edgeLen[excludeEdge])
						excludeEdge = i;
				}

				// Assign short and long edges to first non excluded edge
				if (excludeEdge != 0)
				{
					edgeLong[0]  = 0;
					edgeLong[1]  = 0;
					edgeShort[0] = 0;
					edgeShort[1] = 0;
				}
				else
				{
					edgeLong[0]  = 1;
					edgeLong[1]  = 1;
					edgeShort[0] = 1;
					edgeShort[1] = 1;
				}

				for(i = 0; i < 5; i++)
				{
					// Find longest edge
					if (i != excludeEdge)
					{
						if (edgeLen[i] < edgeLen[edgeShort[0]])
							edgeShort[0] = i;

						if (edgeLen[i] > edgeLen[edgeLong[0]])
							edgeLong[0] = i;

						if (edgeLen[i]   == edgeLen[edgeShort[0]] &&
							edgeShort[0] != i)
							edgeShort[1] = i;

						if (edgeLen[i]   == edgeLen[edgeLong[0]] &&
							edgeLong[0] != i)
							edgeLong[1] = i;
					}
				}

				// Set pivot axis
				if (strstr(strBillboardMode, "Short_Axis"))
				{
					Matrix3 OTM = node->GetObjTMAfterWSM(0);
					Point3 v0 = mesh.verts[elist.edges[edgeShort[0]].v[1]] * OTM;
					Point3 v1 = mesh.verts[elist.edges[edgeShort[0]].v[0]] * OTM;

					//object->m_PivotAxis = mesh.verts[elist.edges[edgeShort[0]].v[1]] - mesh.verts[elist.edges[edgeShort[0]].v[0]];
					object->m_PivotAxis = v0 - v1;
					object->m_PivotAxis = object->m_PivotAxis.Normalize();
				}
				else if (strstr(strBillboardMode, "Long_Axis"))
				{
					Matrix3 OTM = node->GetObjTMAfterWSM(0);
					Point3 v0 = mesh.verts[elist.edges[edgeLong[0]].v[1]] * OTM;
					Point3 v1 = mesh.verts[elist.edges[edgeLong[0]].v[0]] * OTM;

					//object->m_PivotAxis = mesh.verts[elist.edges[edgeLong[0]].v[1]] - mesh.verts[elist.edges[edgeLong[0]].v[0]];
					object->m_PivotAxis = v0 - v1;
					object->m_PivotAxis = object->m_PivotAxis.Normalize();
				}
			}
			else
			{
				char strErr[256];
				sprintf(strErr, "Billboard node '%s' does not contain the expected 5 edges (4 visible) for a quad.  Cannot export.", (char*)node->GetName());
				MessageBoxAll(gInterface->GetMAXHWnd(), strErr, "Billboard Export Failed", MB_ICONWARNING|MB_OK);
				ReportWarning( strErr );
				return false;
			}
		}
		else if (strstr(strBillboardMode, "User-Defined"))
		{
			CStr    strNode = GetPropValue(node, "Billboard/AxisObject");

			if (strNode == CStr("!"))
				strNode = defValObj;

			INode* axisNode = gInterface->GetINodeByName(strNode);

			if (!axisNode)
			{
				char strErr[256];
				sprintf(strErr, "Billboard node '%s' contains a User-Defined Pivot Axis but, user-defined node '%s' could not be found in the scene.", (char*)node->GetName(), (char*)strNode);
				MessageBoxAll(gInterface->GetMAXHWnd(), strErr, "Billboard Export Problem", MB_ICONWARNING|MB_OK);
				ReportWarning( strErr );
				return false;
			}

			Matrix3 TM2 = axisNode->GetNodeTM(0);
			Point3  nodeAxisPos = TM2.GetTrans();
			Point3  origAxisPos = TM.GetTrans();

			object->m_PivotAxis = nodeAxisPos - origAxisPos;
			object->m_PivotAxis = object->m_PivotAxis.Normalize();

			object->m_BillboardType = NxObject::BBT_AXIAL_ALIGNED;
		}
	}

	// Convert the pivot axis into the game's coordinate system
	ftmp = object->m_PivotAxis.z;
	object->m_PivotAxis.z = -object->m_PivotAxis.y;
	object->m_PivotAxis.y = ftmp;

	object->m_Flags |= NxObject::mBILLBOARD;
	return true;
}

// Deprecated:  Discreet has fixed MultiRes to work with multiple map channels   aml
void	GeoDatabase::GetLODUVs(Modifier* mod, INode* node, NxObject* nxobj)
{
	if (!mod || !node || !nxobj)
		return;

	// MultiRes generates a new set of UV coordinates for the default
	// UV channel 1 but, ignores all other channels.  We need the new
	// UV coords for all channels to get exported

	IParamBlock2* pblk = ((Animatable*)mod)->GetParamBlock(0);

	// Disabling the MultiRes modifier will give us access to the original UV data
	mod->DisableMod();

	Object* obj = node->EvalWorldState(0).obj;
	if (obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triobj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
		Mesh& mesh = triobj->GetMesh();

		int nMaps = mesh.getNumMaps();

		nxobj->m_NumUVChannels = nMaps;
		nxobj->m_UVChannels = new UVChannel[nMaps];

		if (nMaps > 2)
		{
			//CopyChannelFrom(nxobj->m_UVChannels[0], mesh, 0);
			//CopyChannelFrom(nxobj->m_UVChannels[1], mesh, 1);

			// Remap each channel in the mesh to channel 1 and reevaluate and store UVs
			for(int mapID = 2; mapID < nMaps; mapID++)
			{
				//RemapChannel(mesh, mapID, 1);
				mod->EnableMod();
				//pblk->SetValue(PB_GENERATE, 0, 1);	// Force MultiRes Generate Function

				Object* obj = node->EvalWorldState(0).obj;
				if (obj->CanConvertToType(triObjectClassID))
				{
					TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
					Mesh&        mesh = triObj->GetMesh();

					//CopyChannelFrom(nxobj->m_UVChannels[mapID], mesh, 1);
				}

				mod->DisableMod();
			}

			//CopyChannelTo(nxobj->m_UVChannels[1], mesh, 1);
		}
	}

	mod->EnableMod();
	pblk->SetValue(PB_GENERATE, 0, 1);
}

void	GeoDatabase::SetLODFull(Modifier* mod)
{
	float per = 100.0f;

	IParamBlock2* pblk = ((Animatable*)mod)->GetParamBlock(0);
	pblk->SetValue(PB_VERTEXPERCENT, 0, per);
}

void	GeoDatabase::EvalLODData(Modifier* mod, INode* node, NxObject* nxobj, LODOptions* lod_options)
{
	if (!mod || !node || !nxobj || !lod_options)
		return;

	// Abort if no LOD levels are to be exported
	if (lod_options->m_LODNumLevels == 0)
		return;

	IParamBlock2* pblk = ((Animatable*)mod)->GetParamBlock(0);

	// Go through and evaluate the object at different LOD levels
	
	float start = lod_options->m_LODLowerLimit;
	float step  = (100.0f - start) / (float)lod_options->m_LODNumLevels;

	int idx = 0;
	Interval interval = FOREVER;

	float origper, origcount;
	pblk->GetValue(PB_VERTEXPERCENT, 0, origper, interval);
	interval = FOREVER;
	pblk->GetValue(PB_VERTEXCOUNT, 0, origcount, interval);

	if (lod_options->m_LODProgressive)
	{
		Object* obj = node->EvalWorldState(0).obj;
		if (obj->CanConvertToType(triObjectClassID))
		{
			// Determine the number of verticies within the mesh
			TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
			Mesh& mesh = triObj->GetMesh();

			int nverts = mesh.numVerts;

			if (triObj != obj)
				triObj->DeleteThis();

			nxobj->m_LODLevels = nverts;
			nxobj->m_LODinfo   = new NxLODInfo[nverts];

			for(int cvert = nverts; cvert > 0; cvert--)
			{
				// If this mesh contains multipass UV coordinates we will need to swap the UVs
				// in and out of the primary UV list as multi-res only remaps the primary UVs
				// once MultiRes is on the stack our multitexture UVs are gone
				//
				// UPDATE: 4-21-03 Discreet has fixed this and has provided us with a new MultiRes plugin
				//                 swapping is no longer necessary to get consecutive pass UV map data  aml

				int nMaps = mesh.getNumMaps();
				//???????????

				pblk->SetValue(PB_VERTEXCOUNT, 0, cvert);

				Object* obj = node->EvalWorldState(0).obj;
				if (obj->CanConvertToType(triObjectClassID))
				{
					TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

					Mesh& mesh = triObj->GetMesh();

					nxobj->m_LODinfo[idx].numFaces = mesh.numFaces;
					nxobj->m_LODinfo[idx].faces = new NxLODFace[mesh.numFaces];

					for(int cface = 0; cface < mesh.numFaces; cface++)
					{
						// Converts from MAX Vertex indicies to the indicies of the expanded verts
						nxobj->m_LODinfo[idx].faces[cface].v[0] = nxobj->m_MaxVerts[mesh.faces[cface].v[0]];
						nxobj->m_LODinfo[idx].faces[cface].v[1] = nxobj->m_MaxVerts[mesh.faces[cface].v[1]];
						nxobj->m_LODinfo[idx].faces[cface].v[2] = nxobj->m_MaxVerts[mesh.faces[cface].v[2]];
					}

					if (triObj != obj)
						triObj->DeleteThis();

					idx++;
				}
			}
		}
	}
	else
	{
		nxobj->m_LODLevels = lod_options->m_LODNumLevels;
		nxobj->m_LODinfo   = new NxLODInfo[lod_options->m_LODNumLevels];

		for(float per = 100.0f; per >= start; per -= step)
		{
			pblk->SetValue(PB_VERTEXPERCENT, 0, per);

			Object* obj = node->EvalWorldState(0).obj;
			if (obj->CanConvertToType(triObjectClassID))
			{
				TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

				Mesh& mesh = triObj->GetMesh();

				nxobj->m_LODinfo[idx].numFaces = mesh.numFaces;
				nxobj->m_LODinfo[idx].faces = new NxLODFace[mesh.numFaces];

				//FILE* fpTest = fopen("c:\\lodtest.txt","w");

				for(int cface = 0; cface < mesh.numFaces; cface++)
				{
					/*
					fprintf(fpTest, "%i: (0) %i -> %i\n", cface, 
						                              mesh.faces[cface].v[0],
													  nxobj->m_MaxVerts[mesh.faces[cface].v[0]]);

					fprintf(fpTest, "%i: (1) %i -> %i\n", cface, 
						                              mesh.faces[cface].v[1],
													  nxobj->m_MaxVerts[mesh.faces[cface].v[1]]);

					fprintf(fpTest, "%i: (2) %i -> %i\n", cface, 
						                              mesh.faces[cface].v[2],
													  nxobj->m_MaxVerts[mesh.faces[cface].v[2]]);
					*/

					// Converts from MAX Vertex indicies to the indicies of the expanded verts
					nxobj->m_LODinfo[idx].faces[cface].v[0] = nxobj->m_MaxVerts[mesh.faces[cface].v[0]];
					nxobj->m_LODinfo[idx].faces[cface].v[1] = nxobj->m_MaxVerts[mesh.faces[cface].v[1]];
					nxobj->m_LODinfo[idx].faces[cface].v[2] = nxobj->m_MaxVerts[mesh.faces[cface].v[2]];

					/*  This is only applicable if you aren't exporting with transforms applied
					if (nxobj->m_Verts[nxobj->m_MaxVerts[mesh.faces[cface].v[0]]].m_Pos != 
						mesh.verts[mesh.faces[cface].v[0]] ||
						
						nxobj->m_Verts[nxobj->m_MaxVerts[mesh.faces[cface].v[1]]].m_Pos != 
						mesh.verts[mesh.faces[cface].v[1]] ||

						nxobj->m_Verts[nxobj->m_MaxVerts[mesh.faces[cface].v[2]]].m_Pos != 
						mesh.verts[mesh.faces[cface].v[2]])
					{
						char strErr[256];
						sprintf(strErr, "Node '%s' contains MultiRes info that could not be matched to original mesh.  Contact Adam", (char*)node->GetName());
						MessageBoxAll(gInterface->GetMAXHWnd(), strErr, "MultiRes Problem Encountered", MB_ICONWARNING|MB_OK);
					}
					*/

				}

				//fclose(fpTest);

				if (triObj != obj)
					triObj->DeleteThis();

				idx++;
			}
		}
	}

	nxobj->m_Flags |= NxObject::mHASINTLODINFO;

	pblk->SetValue(PB_VERTEXPERCENT, 0, origper);
	pblk->SetValue(PB_VERTEXCOUNT, 0, origcount);
}

void	GeoDatabase::RemoveModifier(INode* node, int index)
{
	// Check if the node has a MultiRes modifier applied to it
	Object* obj = node->GetObjectRef();

	if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		IDerivedObject* dobj = (IDerivedObject*)obj;
		dobj->DeleteModifier(index);
	}
}

Modifier*	GeoDatabase::EvalLODData(INode* node, LODOptions* lod_options, bool bBottom)
{
	// Check if the node has a MultiRes modifier applied to it
	Object*   obj = node->GetObjectRef();
	Modifier* mod = FindModifier(obj, MULTIRES_CLASS_ID);

	if (mod)
		return mod;

	// If we didn't find a MultiRes Modifier we'll add one
	if (lod_options->m_AlwaysLOD)
	{
		Modifier* mod = (Modifier*)CreateInstance(OSM_CLASS_ID, MULTIRES_CLASS_ID);
		ModContext* mc = new ModContext(new Matrix3(1), NULL, NULL);

		IDerivedObject* dobj;

		if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
		{
			dobj = (IDerivedObject*)obj;

			if (bBottom)
				dobj->AddModifier(mod, mc, dobj->NumModifiers());
			else
				dobj->AddModifier(mod, mc);
		}
		else
		{
			dobj = CreateDerivedObject(obj);
			
			if (bBottom)
				dobj->AddModifier(mod, mc, dobj->NumModifiers());
			else
				dobj->AddModifier(mod, mc);

			node->SetObjectRef(dobj);
		}

		Object* cobj = node->EvalWorldState(0).obj;

		IParamBlock2* pblk = ((Animatable*)mod)->GetParamBlock(0);
		pblk->SetValue(PB_GENERATE, 0, 1);	// Force MultiRes Generate Function

		// We now need to set the vert percent from 100 to something else and back again
		// due to an appearent bug in Multi-Res where UpdateTopology doesn't get called
		// when the generate button is pushed.  Which resulted in Physique not updating
		// itself with the newly generated vert list.  aml
		float    origper;
		Interval interval = FOREVER;
		pblk->GetValue(PB_VERTEXPERCENT, 0, origper, interval);
		
		pblk->SetValue(PB_VERTEXPERCENT, 0, 50.0f);
		Object* obj;
		obj = node->EvalWorldState(0).obj;
		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triobj = (TriObject*)node->EvalWorldState(0).obj;
			Mesh& mesh = triobj->GetMesh();

			mesh.InvalidateTopologyCache();
			mesh.InvalidateGeomCache();

			if (triobj != obj)
				triobj->DeleteThis();
		}

		pblk->SetValue(PB_VERTEXPERCENT, 0, origper);

		obj = node->EvalWorldState(0).obj;
		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triobj = (TriObject*)node->EvalWorldState(0).obj;
			Mesh& mesh = triobj->GetMesh();

			mesh.InvalidateTopologyCache();
			mesh.InvalidateGeomCache();

			if (triobj != obj)
				triobj->DeleteThis();
		}

		//EvalLODData(mod, node, nxobj);
		return mod;
	}

	return NULL;
}

int		GeoDatabase::DisableMultiRes(INode* node)
{
	Object* obj = node->GetObjectRef();

	if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		IDerivedObject* dobj = (IDerivedObject*)obj;

		int nMods = dobj->NumModifiers();

		for(int j = 0; j < nMods; j++)
		{
			Modifier* mod = dobj->GetModifier(j);

			// Check if this modifier is a MultiRes modifier
			if (mod->ClassID() == MULTIRES_CLASS_ID)
			{
				int state = mod->IsEnabled();
				mod->DisableMod();
				return state;
			}
		}
	}	

	return FALSE;
}

int		GeoDatabase::EnableMultiRes(INode* node)
{
	Object* obj = node->GetObjectRef();

	if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		IDerivedObject* dobj = (IDerivedObject*)obj;

		int nMods = dobj->NumModifiers();

		for(int j = 0; j < nMods; j++)
		{
			Modifier* mod = dobj->GetModifier(j);

			// Check if this modifier is a MultiRes modifier
			if (mod->ClassID() == MULTIRES_CLASS_ID)
			{
				int state = mod->IsEnabled();
				mod->EnableMod();
				return state;
			}
		}
	}
	
	return FALSE;
}

int		GeoDatabase::GetMultiResIndex(INode* node)
{
	Object* obj = node->GetObjectRef();

	if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		IDerivedObject* dobj = (IDerivedObject*)obj;

		int nMods = dobj->NumModifiers();

		for(int j = 0; j < nMods; j++)
		{
			Modifier* mod = dobj->GetModifier(j);

			// Check if this modifier is a MultiRes modifier
			if (mod->ClassID() == MULTIRES_CLASS_ID)
			{
				return j;
			}
		}
	}
	
	return -1;
}

void	ResetWarningFile( void )
{
	FILE* file;

	file = fopen( "c:\\export_warnings.txt", "w" );
	fclose( file );
}

void	ReportWarning( char* warning )
{
	FILE* file;

	file = fopen( "c:\\export_warnings.txt", "a" );
	fprintf( file, "%s\n", warning );
	fclose( file );
}