///////////////////////////////////////////////////////////////////////////////////////
//
// checksum.cpp		KSH 7 Oct 2002
//
// Checksum name lookup stuff
//
///////////////////////////////////////////////////////////////////////////////////////

#include <core/defines.h>
#include "checksum.h"
#include "stdafx.h"
#include "mycode.h"
#include <io.h>

#include <gel/scripting/parse.h>
#include <gel/scripting/checksum.h>

namespace Script
{

// This is an array of up to 256 checksums which are indexed into by the game code's
// WriteToBuffer function when writing out the contents of a CStruct.
// This was to save space because if a checksum is in this array it can be
// specified with one byte rather than 4.
// The array of checksums is defined by the script array WriteToBuffer_CompressionLookupTable_8
// in compress.q
// The script debugger does not know how to parse qb files though so it requests the
// array from the game code when it is started up.
static int s_num_8bit_indexed_checksums=0;
static uint32 sp_8bit_indexed_checksums[256];

void Set8BitIndexedChecksums(int num_checksums, uint32 *p_checksums)
{
	Dbg_MsgAssert(num_checksums<=256,("num_checksums too big, max is 256"));
	for (int i=0; i<num_checksums; ++i)
	{
		sp_8bit_indexed_checksums[i]=p_checksums[i];
	}
	s_num_8bit_indexed_checksums=num_checksums;
}

uint32 Get8BitIndexedChecksum(int index)
{
	Dbg_MsgAssert(index>=0 && index<s_num_8bit_indexed_checksums,("Bad index of %d sent to Get8BitIndexedChecksum",index));
	if (index<0 || index>=s_num_8bit_indexed_checksums)
	{
		return 0xe254d2cb; // Unresolved8BitCompressedName
	}
	return sp_8bit_indexed_checksums[index];
}

void LoadChecksumNamesFromQB(const char *p_qb_name)
{
	char p_message[1024];
	sprintf(p_message,"Parsing %s",p_qb_name);
	Printf(p_message);

	// Load the whole .q into memory for ease of parsing.
	FILE *ifp=fopen(p_qb_name,"rb");
	Dbg_MsgAssert(ifp!=NULL,("Could not open source file"));
	fseek(ifp,0,SEEK_END);
	long source_size=ftell(ifp);
	fseek(ifp,0,SEEK_SET);
	char *p_source=(char*)malloc(source_size);
	Dbg_MsgAssert(p_source!=NULL,("Could not allocate memory to hold input file"));
	int items_read=fread(p_source,source_size,1,ifp)
	Dbg_MsgAssert(items_read==1,("fread failed"));
	fclose(ifp);
	ifp=NULL;

	uint8 *p_token=(uint8*)p_source;
	while (*p_token != ESCRIPTTOKEN_ENDOFFILE)
	{
		if (*p_token==ESCRIPTTOKEN_CHECKSUM_NAME)
		{
			++p_token;
			uint32 checksum=*(uint32*)p_token;
			p_token+=4;
			AddChecksumName(checksum,(const char*)p_token);
			while (*p_token)
			{
				++p_token;
			}
			++p_token;
		}
		else
		{
			p_token=SkipToken(p_token);
		}
	}

	free(p_source);
}

// This checks the directory entry in p_file_info, and if it is a qb file, it parses it,
// if it is a subdirectory name, it calls LoadChecksumNamesFromPath on it.
void CheckDirectoryEntry(const char *p_path, const _finddata_t *p_file_info)
{
	char p_ext[_MAX_EXT];

	_splitpath( p_file_info->name, NULL, NULL, NULL, p_ext );
	if( p_file_info->attrib & _A_SUBDIR )
	{
		if( ( stricmp( p_file_info->name, "." ) != 0 ) &&
			( stricmp( p_file_info->name, ".." ) != 0 ))
		{
			char p_new_path[_MAX_PATH];

			sprintf( p_new_path, "%s\\%s", p_path, p_file_info->name );
			LoadChecksumNamesFromPath(p_new_path);
		} 
	}
	else if( stricmp( p_ext, ".qb" ) == 0 )
	{
		char p_qb_name[1024];
		sprintf(p_qb_name,"%s\\%s",p_path,p_file_info->name);
		LoadChecksumNamesFromQB(p_qb_name);
	}		
}

void LoadChecksumNamesFromPath(const char *p_path)
{
	char search_string[_MAX_PATH];
	sprintf( search_string, "%s\\*.*", p_path );

	struct _finddata_t file_info;
	long handle = _findfirst( search_string, &file_info );
	if( handle >= 0 )
	{
		CheckDirectoryEntry(p_path,&file_info);

		while( _findnext( handle, &file_info ) == 0 )
		{
			CheckDirectoryEntry(p_path,&file_info);
		}
	}
}

} // namespace Script

