/*
	Quantizer.cpp
	Image quantization class
*/

#include "Quantizer.h"
#include "ImageData.h"
#include "colors.h"
#include <stdlib.h>

FILE* fp;

Quantizer::Quantizer()
{
	palColors = 0;
	//fp = fopen("c:\\quantdbg.txt","w");
}

Quantizer::~Quantizer()
{
	//fclose(fp);

	if (root)
		delete root;

	if (pal)
		delete [] pal;
}

int Quantizer::NumColors()
{
	int total = 0;

	Link<HDNode*>* curlink = leafList.GetHead();

	while(curlink)
	{
		HDNode* root = curlink->data;

		// If there is a pixel count greater than 0 we know this is a leaf node
		/*
		if (root->nPixels>0)
		{
			fprintf(fp,"[%i,%i,%i,%i]\n",root->sumRed / root->nPixels,
									  root->sumGreen / root->nPixels,
									  root->sumBlue / root->nPixels,
									  root->sumAlpha / root->nPixels);
		}
		*/

		if (root->bLeaf)
			total += root->nPixels;

		curlink = curlink->next;
	}
	
	return total;
}

void Quantizer::Quant8Bit()
{
	color8* sBuf = (color8*)srcImage->buf;
	color8* dBuf = (color8*)destImage->buf;
	int size = srcImage->width * srcImage->height;

	// Build quant color tree
	color32 color;

	for(int i=0;i<size;i++)
	{
		color.r = srcImage->palette[sBuf->index].red;
		color.g = srcImage->palette[sBuf->index].green;
		color.b = srcImage->palette[sBuf->index].blue;

		if (flags & QUANT_ALPHA &&
			sBuf->index < srcImage->tRNSentries)
			color.a = srcImage->tRNSpal[sBuf->index];
		else
			color.a = 255;

		AddColor(&root,NULL,0,color.r,color.g,color.b,color.a,&leafList);
		sBuf++;
	}

	// Prune color tree and build up quantized palette
	PruneTree(maxColors);
	BuildPal();

	CopyPal(destImage);

	sBuf = (color8*)srcImage->buf;
	dBuf = (color8*)destImage->buf;

	for(i=0;i<size;i++)
	{
		// Account for indexed alpha in color matching
		if (flags & QUANT_ALPHA)
			if (sBuf->index < srcImage->tRNSentries)
				color.a = srcImage->tRNSpal[sBuf->index];
			else
				color.a = 255;

		dBuf->index = GetClosestColor(color.r,color.g,color.b,color.a);
		sBuf++;
		dBuf++;
	}
}

void Quantizer::Quant16Bit()
{
	color16_555* sBuf = (color16_555*)srcImage->buf;
	color16_555* dBuf = (color16_555*)destImage->buf;
	int size = srcImage->width * srcImage->height;

	for(int i=0;i<size;i++)
	{
		AddColor(&root,NULL,0,sBuf->R(),sBuf->G(),sBuf->B());
		sBuf++;
	}

	// Prune color tree and build up quantized palette
	PruneTree(maxColors);
	BuildPal();

	// Pass 2: Reassign image colors to quantized values
	sBuf = (color16_555*)srcImage->buf;

	if (!flags & QUANT_PALETTE)
	{
		color16_555 color;
		int         index;

		for(i=0;i<size;i++)
		{
			index = GetClosestColor(sBuf->R(),sBuf->G(),sBuf->B());
			color.R(pal[index].r);
			color.G(pal[index].g);
			color.B(pal[index].b);

			*dBuf = color;
			sBuf++;
			dBuf++;
		}
	}
	else
	{
		CopyPal(destImage);

		color8* dBuf = (color8*)destImage->buf;

		for(i=0;i<size;i++)
		{
			dBuf->index = GetClosestColor(sBuf->R(),sBuf->G(),sBuf->B());
			sBuf++;
			dBuf++;
		}
	}
}

void Quantizer::Quant24Bit()
{
	color24* sBuf = (color24*)srcImage->buf;
	color24* dBuf = (color24*)destImage->buf;
	int size = srcImage->width * srcImage->height;

	int i;

	// Pass 1: Build color tree
	for(i=0;i<size;i++)
	{
		AddColor(&root,NULL,0,sBuf->r,sBuf->g,sBuf->b);
		sBuf++;
	}

	// Prune color tree and build up quantized palette
	PruneTree(maxColors);
	BuildPal();

	// Pass 2: Reassign image colors to quantized values
	sBuf = (color24*)srcImage->buf;

	if (!flags & QUANT_PALETTE)
	{
		color24 color;
		int     index;

		for(i=0;i<size;i++)
		{
			index = GetClosestColor(sBuf->r,sBuf->g,sBuf->b);
			color.r = pal[index].r;
			color.g = pal[index].g;
			color.b = pal[index].b;

			*dBuf = color;
			sBuf++;
			dBuf++;
		}
	}
	else
	{
		CopyPal(destImage);

		color8* dBuf = (color8*)destImage->buf;

		for(i=0;i<size;i++)
		{
			dBuf->index = GetClosestColor(sBuf->r,sBuf->g,sBuf->b);
			sBuf++;
			dBuf++;
		}
	}
}

void Quantizer::Quant32Bit()
{
	color32* sBuf = (color32*)srcImage->buf;
	color32* dBuf = (color32*)destImage->buf;
	int size = srcImage->width * srcImage->height;

	int i;

	// Pass 1: Build color tree
	for(i=0;i<size;i++)
	{
		AddColor(&root,NULL,0,sBuf->r,sBuf->g,sBuf->b,sBuf->a,&leafList);
		
		sBuf++;
	}

	// Prune color tree and build up quantized palette
	PruneTree(maxColors);
	BuildPal();

	// Pass 2: Reassign image colors to quantized values
	sBuf = (color32*)srcImage->buf;

	if (!flags & QUANT_PALETTE)
	{
		for(i=0;i<size;i++)
		{
			int idx = GetClosestColor(sBuf->r,sBuf->g,sBuf->b,sBuf->a);
			*dBuf = pal[idx];
			sBuf++;
			dBuf++;
		}
	}
	else
	{
		destImage->color_type = PNG_COLOR_TYPE_PALETTE;
		destImage->bit_depth = 8;
		CopyPal(destImage);

		color8* dBuf = (color8*)destImage->buf;

		for(i=0;i<size;i++)
		{
			dBuf->index = GetClosestColor(sBuf->r,sBuf->g,sBuf->b,sBuf->a);
			sBuf++;
			dBuf++;
		}
	}
}


void Quantizer::AddColor(HDNode** pnode,
						 HDNode*  parent,
						 int level,
						 unsigned char r,unsigned char g,unsigned char b,unsigned char a,
						 LinkList<HDNode*>* list)
{
	HDNode* node = *pnode;

	if (!node)
	{
		node = *pnode = new HDNode;
		node->bLeaf = (level == 8) ? true : false;

		node->pParent = parent;

		if (node->bLeaf)
		{
			if (!list)
				list = &leafList;

			node->link = list->AddToTail(&node);
		}
	}

	// We've reached the leaf level, assign color
	if (node->bLeaf)
	{
		node->sumRed   += r;
		node->sumGreen += g;
		node->sumBlue  += b;
		node->sumAlpha += a;

		node->nPixels++;
		return;
	}

	//int color = (r << 24) | (g << 16) | (b << 8) | a;

	/*
	int levelMask = 1 << level;
	
	int rBit = (r & levelMask) >> level;
	int gBit = (g & levelMask) >> level;
	int bBit = (b & levelMask) >> level;
	int aBit = (a & levelMask) >> level;
	*/

	int levelMask = 128 >> level;
	int invlevel = 7 - level;

	// Debug (remove later)
	int rBit = (r & levelMask) >> level;
	int gBit = (g & levelMask) >> level;
	int bBit = (b & levelMask) >> level;
	int aBit = (a & levelMask) >> level;


	// Build a 4-bit key from the rgba bit values for this level
	int child = (((r & levelMask) >> invlevel) << 3) | 
		        (((g & levelMask) >> invlevel) << 2) | 
				(((b & levelMask) >> invlevel) << 1) | 
				 ((a & levelMask) >> invlevel);

	AddColor(&node->pChild[child],node,++level,r,g,b,a,list);
}

void Quantizer::ProcImage(ImageData* destImage, ImageData* srcImage, int nColors, unsigned long flags)
{
	int depth;
	maxColors = nColors;

	this->srcImage  = srcImage;
	this->destImage = destImage;
	this->flags     = flags;

	if (root)
	{
		delete root;
		root = NULL;
	}

	leafList.Clear();

	depth = srcImage->bit_depth * srcImage->nChannels;

	destImage->CopyNoImage(srcImage);

	if (!flags & QUANT_PALETTE)
		destImage->buf = (unsigned char*)malloc(srcImage->imageSize);
	else
	{
		destImage->buf = (unsigned char*)malloc(srcImage->width*srcImage->height);
		destImage->imageSize = srcImage->width*srcImage->height;
		destImage->color_type = PNG_COLOR_TYPE_PALETTE;
		destImage->bit_depth = 8;
		destImage->nChannels = 1;
	}

	// Allocate memory for the leafNode array
	switch(depth)
	{
	case 8:
		Quant8Bit();
		break;

	case 16:
		Quant16Bit();
		break;

	case 24:
		Quant24Bit();
		break;

	case 32:
		Quant32Bit();
		break;
	}

	printf("Leaf list: %i\n",leafList.GetSize());

	PruneTree(maxColors);
}

// Reduce the size of the tree by closest color
void Quantizer::PruneTree(int nColors)
{
	int nNodes  = leafList.GetSize();
	int nReduce = nNodes - nColors;

	// If we're already at the proper number of colors we don't need
	// to prune the tree
	if (nNodes <= nColors)
		return;

	printf("Reducing tree by %i colors\n",nReduce);

	while(leafList.GetSize() > nColors)
	{
		// May want to change from head to get better distribution of reduced colors
		Link<HDNode*>* linkA = leafList.GetHead();

		if (!linkA)
			return;

		// Find a sibling
		HDNode* pParent   = linkA->data->pParent;
		int     iSibIndex = -1;

		for(int i=0;i<16;i++)
		{
			if (pParent->pChild[i] && pParent->pChild[i] != linkA->data)
			{
				iSibIndex = i;
				break;
			}
		}
		
		if (iSibIndex == -1)
		{
			// There are no siblings to combine colors with fold this child into its parent
			// and add it to the end of the leaf list (it will be reduced again if we don't find enough
			// colors by the time we get to the end of the list)
			pParent->bLeaf    = true;
			pParent->sumRed   = linkA->data->sumRed;
			pParent->sumGreen = linkA->data->sumGreen;
			pParent->sumBlue  = linkA->data->sumBlue;
			pParent->sumAlpha = linkA->data->sumAlpha;
			pParent->nPixels  = linkA->data->nPixels;

			// Clear out child list
			for(int i=0;i<16;i++)
				pParent->pChild[i] = NULL;

			delete linkA->data;
			leafList.Remove(linkA);
			pParent->link = leafList.AddToTail(&pParent);
			continue;
		}

		HDNode* pSibling = pParent->pChild[iSibIndex];

		// We've found a sibling, combine the two colors
		linkA->data->nPixels  += pSibling->nPixels;
		linkA->data->sumRed   += pSibling->sumRed;
		linkA->data->sumGreen += pSibling->sumGreen;
		linkA->data->sumBlue  += pSibling->sumBlue;
		linkA->data->sumAlpha += pSibling->sumAlpha;

		leafList.Remove(pSibling->link);
		pParent->DeleteChild(iSibIndex);
		pParent->pChild[iSibIndex] = linkA->data;
	}

	printf("Reduced colors to %i.\n",leafList.GetSize());
}

void Quantizer::BuildPal()
{
	if (pal)
		delete [] pal;

	palColors = leafList.GetSize();
	pal = new color32[palColors];

	HDNode* node;
	Link<HDNode*>* curlink = leafList.GetHead();

	int i = 0;

	while(curlink)
	{
		node = curlink->data;

		pal[i].r = node->sumRed / node->nPixels;
		pal[i].g = node->sumGreen / node->nPixels;
		pal[i].b = node->sumBlue / node->nPixels;
		pal[i].a = node->sumAlpha / node->nPixels;

		// Assign computed color index into tree for fast lookups when reassigning image
		node->index = i;

		i++;
		curlink = curlink->next;
	}
}

void Quantizer::CopyPal(ImageData* img)
{
	img->nPalEntries = palColors;

	if (img->palette)
		free(img->palette);

	img->palette = (png_color*)malloc(sizeof(png_color)*palColors);

	for(int i=0;i<palColors;i++)
	{
		img->palette[i].red   = pal[i].r;
		img->palette[i].green = pal[i].g;
		img->palette[i].blue  = pal[i].b;
	}

	if (flags & QUANT_ALPHA)
	{
		img->tRNSentries = palColors;

		if (img->tRNSpal)
			free(img->tRNSpal);

		img->tRNSpal = (png_byte*)malloc(sizeof(png_byte)*palColors);

		for(int i=0;i<palColors;i++)
		{
			img->tRNSpal[i] = pal[i].a;
		}
	}
}

int Quantizer::GetClosestColor(unsigned char r,
		                       unsigned char g,
					           unsigned char b,
					           unsigned char a,
					           int           level,
					           HDNode*       node)
{
	if (!node)
		node = root;

	if (node->bLeaf)
		return node->index;
	
	int levelMask = 128 >> level;
	int invlevel  = 7 - level;
	
	// Build the 4-bit key from the rgba bit values for this level
	int child = (((r & levelMask) >> invlevel) << 3) | 
		        (((g & levelMask) >> invlevel) << 2) | 
				(((b & levelMask) >> invlevel) << 1) | 
				 ((a & levelMask) >> invlevel);
	
	if (node->pChild[child])
		return GetClosestColor(r,g,b,a,++level,node->pChild[child]);

	// Should never get here (Invalid tree construction)
	assert(0);
	return -1;
}

color32 Quantizer::GetClosestColor32(HDNode* node,
									 unsigned char r,
									 unsigned char g,
									 unsigned char b,
									 unsigned char a,
									 int           level)
{
	if (!node)
		node = root;

	if (node->bLeaf)
	{
		color32 color;
		color.r = (unsigned char)(node->sumRed / node->nPixels);
		color.g = (unsigned char)(node->sumGreen / node->nPixels);
		color.b = (unsigned char)(node->sumBlue / node->nPixels);
		color.a = (unsigned char)(node->sumAlpha / node->nPixels);

		return color;
	}
	
	int levelMask = 128 >> level;
	int invlevel  = 7 - level;
	
	// Build the 4-bit key from the rgba bit values for this level
	int child = (((r & levelMask) >> invlevel) << 3) | 
		        (((g & levelMask) >> invlevel) << 2) | 
				(((b & levelMask) >> invlevel) << 1) | 
				 ((a & levelMask) >> invlevel);
	
	if (node->pChild[child])
		return GetClosestColor32(node->pChild[child],r,g,b,a,++level);

	// The color doesn't exist in the color tree
	// we'll need to find the closest match from this level in the tree
	color32 closest;
	__int32 dist;

	closest.r = 0;
	closest.g = 0;
	closest.b = 0;
	closest.a = 0;
	
	dist = -1;

	if (node->pParent)
		FindLinearMatch(r,g,b,a,node->pParent,&closest,&dist);
	else
		FindLinearMatch(r,g,b,a,node,&closest,&dist);

	return closest;
}

void Quantizer::FindLinearMatch(unsigned char r,
								unsigned char g,
								unsigned char b,
								unsigned char a,
								HDNode*  node, 
								color32* bestMatch,
								__int32* dist)
{
	if (node->bLeaf)
	{
		color32 color;

		color.r = (unsigned char)(node->sumRed / node->nPixels);
		color.g = (unsigned char)(node->sumGreen / node->nPixels);
		color.b = (unsigned char)(node->sumBlue / node->nPixels);
		color.a = (unsigned char)(node->sumAlpha / node->nPixels);

		__int32 d	 = Dist2_4D(r,g,b,a,
								color.r,color.g,color.b,color.a);

		if (*dist == -1 || d < *dist)
		{
			bestMatch->r = color.r;
			bestMatch->g = color.g;
			bestMatch->b = color.b;
			bestMatch->a = color.a;

			*dist = d;
		}

		return;
	}

	for(int i=0;i<16;i++)
		if (node->pChild[i])
			FindLinearMatch(r,g,b,a,node->pChild[i],bestMatch,dist);
}

color32 Quantizer::FindColorMatch(TreeData* tdata,
								  unsigned char r,
								  unsigned char g,
								  unsigned char b,
								  unsigned char a)
{
	int nEntries  = tdata->nPalEntries;
	color32*  pal = tdata->pal;

	int small = -1;
	int scolor;

	for(int i=0;i < nEntries; i++)
	{
		__int32 d = Dist2_4D(r,g,b,a,
			                 pal[i].r,pal[i].g,pal[i].b,pal[i].a);

		if (small == -1 || d < small)
		{
			small  = d;
			scolor = i;
		}
	}

	return pal[scolor];
}


void Quantizer::AddImage(ImageData* image,int weight)
{
	
}

TreeData* Quantizer::BuildColorTree(ImageData* image)
{
	TreeData* tdata = new TreeData;
	int depth = image->bit_depth * image->nChannels;
	int size  = image->width * image->height;

	switch(depth)
	{
	case 8:
		{
			color8*           color   = (color8*)image->buf;
			png_color_struct* palette = image->palette;
			unsigned char     alpha;

			for(int i=0;i<size;i++)
			{
				if (image->btRNStransValid)
					alpha = image->tRNSpal[color->index];
				else
					alpha = 255;

				AddColor(&tdata->root,NULL,0,palette[color->index].red,
					                  palette[color->index].green,
									  palette[color->index].blue,
									  alpha,
									  &tdata->leafList);
				color++;
			}
		}
		break;

	case 16:
		{
			color16_555* color = (color16_555*)image->buf;
			
			for(int i=0;i<size;i++)
			{
				AddColor(&tdata->root,NULL,0,color->R(),color->G(),color->B(),255,&tdata->leafList);
				color++;
			}
		}
		break;

	case 24:
		{
			color24* color = (color24*)image->buf;

			for(int i=0;i<size;i++)
			{
				AddColor(&tdata->root,NULL,0,color->r,color->g,color->b,255,&tdata->leafList);
				color++;
			}
		}
		break;

	case 32:
		{
			color32* color = (color32*)image->buf;

			for(int i=0;i<size;i++)
			{
				AddColor(&root,NULL,0,color->r,color->g,color->b,color->a,&leafList);
				color++;
			}
		}
		break;
	}
	
	// Build up palette
	Link<HDNode*>* curlink = tdata->leafList.GetHead();

	tdata->nPalEntries = tdata->leafList.GetSize();
	tdata->pal = new color32[tdata->nPalEntries];

	int i=0;

	while(curlink)
	{
		HDNode* node = curlink->data;
		
		tdata->pal[i].r = (unsigned char)(node->sumRed / node->nPixels);
		tdata->pal[i].g = (unsigned char)(node->sumGreen / node->nPixels);
		tdata->pal[i].b = (unsigned char)(node->sumBlue / node->nPixels);
		tdata->pal[i].a = (unsigned char)(node->sumAlpha / node->nPixels);

		curlink = curlink->next;
		i++;
	}

	return tdata;
}
