/*
	MipMapGen.cpp
	Mip Map Generator class
*/

#include "MipMapGen.h"
#include "colors.h"
#include <windows.h>
#include <stdlib.h>
#include <io.h>
#include "ParseFuncs.h"

MipMapGen::MipMapGen()
{
	//srcList.Clear();
	outPath         = ".";
	destFile        = NULL;
	outPalFile      = NULL;
	inPalFile       = NULL;
	outDepth        = 24;
	curOutDepth     = 24;
	bGenMips	    = false;
	mipTargetWidth  = 4;
	mipTargetHeight = 4;
	scaleWidth      = 0;
	scaleHeight     = 0;
	bDumpTarga      = false;
	bUTransparent   = false;
	bVTransparent   = false;
	quantColors     = 0;
	bReduceCN       = false;
	bUseTransColor  = false;
	limitMips       = -1;
	bDelOldMips     = false;
	bReport         = false;

	transColor.r    = 255;
	transColor.g    = 255;
	transColor.b    = 255;
}

MipMapGen::~MipMapGen()
{

}


void MipMapGen::ResizeImageCN(ImageData* destImage, 
							  ImageData* srcImage,
							  unsigned int destWidth,
							  unsigned int destHeight)
{
	// Make a copy of the source image to where we'll modify it
	destImage->CopyNoImage(srcImage);

	int depth = srcImage->bit_depth * srcImage->nChannels;

	destImage->width     = destWidth;
	destImage->height    = destHeight;
	destImage->imageSize = destHeight * destWidth * (depth / 8);

	// Allocate memory for the new image
	destImage->buf = (unsigned char*)malloc(destImage->imageSize);

	// Calc reduction ratios
	float Xstep = (float)srcImage->width  / (float)destWidth;
	float Ystep = (float)srcImage->height / (float)destHeight;

	unsigned int x,y;

	switch(depth)
	{
	case 8:
		{
			color8* Dbuf = (color8*)destImage->buf;
			color8* Sbuf = (color8*)srcImage->buf;

			for(y=0;y<destHeight;y++)
				for(x=0;x<destWidth;x++)
					Dbuf[y*destWidth+x] = Sbuf[(int)((float)y*Ystep*srcImage->width+(float)x*Xstep)];
		}
		break;
	case 16:
		{
			// We will assume that all bit depths of 16 are packed 555 RGB
			color16_555* Dbuf = (color16_555*)destImage->buf;
			color16_555* Sbuf = (color16_555*)srcImage->buf;

			for(y=0;y<destHeight;y++)
				for(x=0;x<destWidth;x++)
					Dbuf[y*destWidth+x] = Sbuf[(int)((float)y*Ystep*srcImage->width+(float)x*Xstep)];
		}
		break;
	case 24:
		{
			color24* Dbuf = (color24*)destImage->buf;
			color24* Sbuf = (color24*)srcImage->buf;

			for(y=0;y<destHeight;y++)
				for(x=0;x<destWidth;x++)
					Dbuf[y*destWidth+x] = Sbuf[(int)((float)y*Ystep*srcImage->width+(float)x*Xstep)];
		}
		break;
	case 32:
		{
			color32* Dbuf = (color32*)destImage->buf;
			color32* Sbuf = (color32*)srcImage->buf;

			for(y=0;y<destHeight;y++)
				for(x=0;x<destWidth;x++)
					Dbuf[y*destWidth+x] = Sbuf[(int)((float)y*Ystep*srcImage->width+(float)x*Xstep)];
		}
		break;
	}

}

void MipMapGen::ResizeImageBox(ImageData* destImage, 
							   ImageData* srcImage,
							   unsigned int destWidth,
							   unsigned int destHeight)
{
	// Don't resample the image if the new size is the same as the old size
	if (destWidth == srcImage->width &&
		destHeight == srcImage->height)
	{
		*destImage = *srcImage;
		return;
	}

	// Make a copy of the source image to where we'll modify it
	destImage->CopyNoImage(srcImage);

	int depth = srcImage->bit_depth * srcImage->nChannels;

	destImage->width     = destWidth;
	destImage->height    = destHeight;
	destImage->imageSize = destHeight * destWidth * (depth / 8);
	//destImage->filter_type

	// Allocate memory for the new image
	destImage->buf = (unsigned char*)malloc(destImage->imageSize);

	switch(depth)
	{
	case 8:
		Proc8BitBoxReduce(destImage,srcImage);
		break;

	case 16:
		Proc16BitBoxReduce(destImage,srcImage);
		break;

	case 24:
		Proc24BitBoxReduce(destImage,srcImage);
		break;

	case 32:
		Proc32BitBoxReduce(destImage,srcImage);
		break;
	}
}

void MipMapGen::Proc8BitBoxReduce(ImageData* destImage, ImageData* srcImage)
{
	// Calc reduction ratios
	float Xstep = (float)srcImage->width / (float)destImage->width;
	float Ystep = (float)srcImage->height / (float)destImage->height;

	unsigned int x,y,i,j;

	// Matrix for computing neighboring pixels
	const Point Matrix[8] = { Point(-1,-1),
							  Point( 0,-1),
							  Point( 1,-1),
							  Point(-1, 0),
							  Point( 1, 0),
							  Point(-1, 1),
							  Point( 0, 1),
							  Point( 1, 1) };
	

	color8* Dbuf = (color8*)destImage->buf;
	color8* Sbuf = (color8*)srcImage->buf;

	for(y=0;y<destImage->height;y++)
	{
		for(x=0;x<destImage->width;x++)
		{
			Freq<color8> colors[9];
			unsigned int highFreq = 0;				// Highest color freq
			color8       bestColor;					// Color with highest freq

			Point ptSrc((int)((float)x*Xstep),(int)((float)y*Ystep));

			// Assign the sample point color
			bestColor = Sbuf[ptSrc.y*srcImage->width+ptSrc.x];
			colors[0].data   = bestColor;
			colors[0].freq   = 1;

			// Find neighboring pixel values
			for(i=0;i<8;i++)
			{
				Point ptSample(ptSrc.x+Matrix[i].x,
							   ptSrc.y+Matrix[i].y);

				if (ptSample.x > -1 && ptSample.x < (int)srcImage->width &&
					ptSample.y > -1 && ptSample.y < (int)srcImage->height)
				{
					// Find sample color for this location in the matrix
					color8 color  = Sbuf[(int)((float)ptSample.y*srcImage->width+(float)ptSample.x)];
					bool   bFound = false;

					// Check if color exists and update the frequency
					for(j=0;j<i+1;j++)
					{
						if (colors[j].data == color)
						{
							colors[j].freq++;

							// Track the most frequent color
							if (colors[j].freq > highFreq)
							{
								highFreq  = colors[j].freq;
								bestColor = colors[j].data;
							}

							bFound = true;
							break;
						}
					}

					// If we couldn't find it, add a new color
					if (!bFound)
						colors[i+1].data = color;
				}
			}

			// Use the color with the highest frequency
			Dbuf[y*destImage->width+x] = bestColor;
		}
	}
}

void MipMapGen::Proc16BitBoxReduce(ImageData* destImage, ImageData* srcImage)
{
	// Calc reduction ratios
	float Xstep = (float)srcImage->width / (float)destImage->width;
	float Ystep = (float)srcImage->height / (float)destImage->height;

	unsigned int x,y,i;

	// Matrix for computing neighboring pixels
	const Point Matrix[8] = { Point(-1,-1),
							  Point( 0,-1),
							  Point( 1,-1),
							  Point(-1, 0),
							  Point( 1, 0),
							  Point(-1, 1),
							  Point( 0, 1),
							  Point( 1, 1) };
	
	color16_555* Dbuf = (color16_555*)destImage->buf;
	color16_555* Sbuf = (color16_555*)srcImage->buf;

	color16_555 bestColor;

	for(y=0;y<destImage->height;y++)
	{
		for(x=0;x<destImage->width;x++)
		{
			unsigned int freqR = 0;		// Total Red component around sampled area
			unsigned int freqG = 0;		// Total Green component around sampled area
			unsigned int freqB = 0;		// Total Blue component around sampled area
			unsigned int nSamples = 1;

			Point ptSrc((int)((float)x*Xstep),(int)((float)y*Ystep));

			// Assign the sample point color
			bestColor = Sbuf[ptSrc.y*srcImage->width+ptSrc.x];

			freqR = bestColor.R();
			freqG = bestColor.G();
			freqB = bestColor.B();

			// Find neighboring pixel values
			for(i=0;i<8;i++)
			{
				Point ptSample(ptSrc.x+Matrix[i].x,
							   ptSrc.y+Matrix[i].y);

				if (ptSample.x > -1 && ptSample.x < (int)srcImage->width &&
					ptSample.y > -1 && ptSample.y < (int)srcImage->height)
				{
					// Find sample color for this location in the matrix
					color16_555 color  = Sbuf[(int)((float)ptSample.y*srcImage->width+(float)ptSample.x)];

					freqR += color.R();
					freqG += color.G();
					freqB += color.B();
					nSamples++;
				}
			}

			// Find the average color for the surrounding area
			bestColor.R((unsigned char)(freqR / nSamples));
			bestColor.G((unsigned char)(freqG / nSamples));
			bestColor.B((unsigned char)(freqB / nSamples));

			// Use the average color of the region
			Dbuf[y*destImage->width+x] = bestColor;
		}
	}
}

void MipMapGen::Proc24BitBoxReduce(ImageData* destImage, ImageData* srcImage)
{
	// Allocate memory for destination buffer (owned by caller)
	//int memSize = destImage->width * destImage->height * 3;
	//char* destBuf = (char*)malloc(memSize);

	// Calc reduction ratios
	float Xstep = (float)srcImage->width / (float)destImage->width;
	float Ystep = (float)srcImage->height / (float)destImage->height;

	unsigned int x,y,i;

	// Matrix for computing neighboring pixels
	const Point Matrix[8] = { Point(-1,-1),
							  Point( 0,-1),
							  Point( 1,-1),
							  Point(-1, 0),
							  Point( 1, 0),
							  Point(-1, 1),
							  Point( 0, 1),
							  Point( 1, 1) };
	
	color24* Dbuf = (color24*)destImage->buf;
	color24* Sbuf = (color24*)srcImage->buf;

	color24 bestColor;

	for(y=0;y<destImage->height;y++)
	{
		for(x=0;x<destImage->width;x++)
		{
			unsigned int freqR = 0;		// Total Red component around sampled area
			unsigned int freqG = 0;		// Total Green component around sampled area
			unsigned int freqB = 0;		// Total Blue component around sampled area
			unsigned int nSamples = 1;	// Number of samples computed from matrix

			Point ptSrc((int)((float)x*Xstep),(int)((float)y*Ystep));

			// Assign the sample point color
			bestColor = Sbuf[ptSrc.y*srcImage->width+ptSrc.x];

			freqR = bestColor.r;
			freqG = bestColor.g;
			freqB = bestColor.b;

			// Find neighboring pixel values
			for(i=0;i<8;i++)
			{
				Point ptSample(ptSrc.x+Matrix[i].x,
							   ptSrc.y+Matrix[i].y);

				if (ptSample.x > -1 && ptSample.x < (int)srcImage->width &&
					ptSample.y > -1 && ptSample.y < (int)srcImage->height)
				{
					// Find sample color for this location in the matrix
					color24 color  = Sbuf[(int)((float)ptSample.y*srcImage->width+(float)ptSample.x)];

					freqR += color.r;
					freqG += color.g;
					freqB += color.b;
					nSamples++;
				}
			}

			// Find the average color for the surrounding area
			bestColor.r = (unsigned char)(freqR / nSamples);
			bestColor.g = (unsigned char)(freqG / nSamples);
			bestColor.b = (unsigned char)(freqB / nSamples);

			// Use the average color of the region
			Dbuf[y*destImage->width+x] = bestColor;
		}
	}
}

void MipMapGen::Proc32BitBoxReduce(ImageData* destImage, ImageData* srcImage)
{
	// Calc reduction ratios
	float Xstep = (float)srcImage->width / (float)destImage->width;
	float Ystep = (float)srcImage->height / (float)destImage->height;

	unsigned int x,y,i;

	// Matrix for computing neighboring pixels
	const Point Matrix[8] = { Point(-1,-1),
							  Point( 0,-1),
							  Point( 1,-1),
							  Point(-1, 0),
							  Point( 1, 0),
							  Point(-1, 1),
							  Point( 0, 1),
							  Point( 1, 1) };
	
	color32* Dbuf = (color32*)destImage->buf;
	color32* Sbuf = (color32*)srcImage->buf;

	color32 bestColor;

	for(y=0;y<destImage->height;y++)
	{
		for(x=0;x<destImage->width;x++)
		{
			unsigned int freqR = 0;		// Total Red component around sampled area
			unsigned int freqG = 0;		// Total Green component around sampled area
			unsigned int freqB = 0;		// Total Blue component around sampled area
			unsigned int freqA = 0;		// Total Alpha component around sampled area
			unsigned int nSamples = 1;

			Point ptSrc((int)((float)x*Xstep),(int)((float)y*Ystep));

			// Assign the sample point color
			bestColor = Sbuf[ptSrc.y*srcImage->width+ptSrc.x];

			freqR = bestColor.r;
			freqG = bestColor.g;
			freqB = bestColor.b;
			freqA = bestColor.a;

			// Find neighboring pixel values
			for(i=0;i<8;i++)
			{
				Point ptSample(ptSrc.x+Matrix[i].x,
							   ptSrc.y+Matrix[i].y);

				if (ptSample.x > -1 && ptSample.x < (int)srcImage->width &&
					ptSample.y > -1 && ptSample.y < (int)srcImage->height)
				{
					// Find sample color for this location in the matrix
					color32 color  = Sbuf[(int)((float)ptSample.y*srcImage->width+(float)ptSample.x)];

					freqR += color.r;
					freqG += color.g;
					freqB += color.b;
					freqA += color.a;
					nSamples++;
				}
			}

			// Find the average color for the surrounding area
			bestColor.r = (unsigned char)(freqR / nSamples);
			bestColor.g = (unsigned char)(freqG / nSamples);
			bestColor.b = (unsigned char)(freqB / nSamples);
			bestColor.a = (unsigned char)(freqA / nSamples);

			// Use the average color of the region
			Dbuf[y*destImage->width+x] = bestColor;
		}
	}
}

// This does the standard filtered reduction on 32-bit images for channels R,G,B but,
// does a closest neighbor reduction for the alpha channel
void MipMapGen::Proc32BitBoxReduceACN(ImageData* destImage, 
									  ImageData* srcImage,
									  unsigned int destWidth,
									  unsigned int destHeight)
{
	// Don't resample the image if the new size is the same as the old size
	if (destWidth == srcImage->width &&
		destHeight == srcImage->height)
	{
		*destImage = *srcImage;
		return;
	}

	// Make a copy of the source image to where we'll modify it
	destImage->CopyNoImage(srcImage);

	int depth = srcImage->bit_depth * srcImage->nChannels;

	destImage->width     = destWidth;
	destImage->height    = destHeight;
	destImage->imageSize = destHeight * destWidth * (depth / 8);
	//destImage->filter_type

	// Allocate memory for the new image
	destImage->buf = (unsigned char*)malloc(destImage->imageSize);

	// Calc reduction ratios
	float Xstep = (float)srcImage->width / (float)destImage->width;
	float Ystep = (float)srcImage->height / (float)destImage->height;

	unsigned int x,y,i;

	// Matrix for computing neighboring pixels
	const Point Matrix[8] = { Point(-1,-1),
							  Point( 0,-1),
							  Point( 1,-1),
							  Point(-1, 0),
							  Point( 1, 0),
							  Point(-1, 1),
							  Point( 0, 1),
							  Point( 1, 1) };
	
	color32* Dbuf = (color32*)destImage->buf;
	color32* Sbuf = (color32*)srcImage->buf;

	color32 bestColor;

	for(y=0;y<destImage->height;y++)
	{
		for(x=0;x<destImage->width;x++)
		{
			unsigned int freqR = 0;		// Total Red component around sampled area
			unsigned int freqG = 0;		// Total Green component around sampled area
			unsigned int freqB = 0;		// Total Blue component around sampled area
			unsigned int freqA = 0;		// Total Alpha component around sampled area
			unsigned int nSamples = 1;

			Point ptSrc((int)((float)x*Xstep),(int)((float)y*Ystep));

			// Assign the sample point color
			bestColor = Sbuf[ptSrc.y*srcImage->width+ptSrc.x];

			freqR = bestColor.r;
			freqG = bestColor.g;
			freqB = bestColor.b;
			//freqA = bestColor.a;

			// Find neighboring pixel values
			for(i=0;i<8;i++)
			{
				Point ptSample(ptSrc.x+Matrix[i].x,
							   ptSrc.y+Matrix[i].y);

				if (ptSample.x > -1 && ptSample.x < (int)srcImage->width &&
					ptSample.y > -1 && ptSample.y < (int)srcImage->height)
				{
					// Find sample color for this location in the matrix
					color32 color  = Sbuf[(int)((float)ptSample.y*srcImage->width+(float)ptSample.x)];

					freqR += color.r;
					freqG += color.g;
					freqB += color.b;
					//freqA += color.a;
					nSamples++;
				}
			}

			// Find the average color for the surrounding area
			bestColor.r = (unsigned char)(freqR / nSamples);
			bestColor.g = (unsigned char)(freqG / nSamples);
			bestColor.b = (unsigned char)(freqB / nSamples);
			//bestColor.a = (unsigned char)(freqA / nSamples);

			// bestColor should have been initially assigned to the sample in the center of the
			// reduction box so a should be the natural closest neighbor alpha

			// verify
			//printf("a: %i\n",bestColor.a);

			// Use the average color of the region
			Dbuf[y*destImage->width+x] = bestColor;
		}
	}
}

void MipMapGen::Dither(ImageData* ditheredImage,
	                   ImageData* quantizedImage,
			           ImageData* origImage)
{
	// Dither the image using the Floyd-Steinberg error diffusion matrix
	int depth = origImage->bit_depth * origImage->nChannels;

	switch(depth)
	{
	case 8:
		Dither8(ditheredImage,quantizedImage,origImage);
		break;

	case 16:
		Dither16(ditheredImage,quantizedImage,origImage);
		break;

	case 24:
		Dither24(ditheredImage,quantizedImage,origImage);
		break;

	case 32:
		Dither32(ditheredImage,quantizedImage,origImage);
		break;
	}	
}

void MipMapGen::Dither8(ImageData* ditheredImage,
	                    ImageData* quantizedImage,
			            ImageData* origImage)
{
	assert(0);
}

void MipMapGen::Dither16(ImageData* ditheredImage,
	                     ImageData* quantizedImage,
			             ImageData* origImage)
{
	assert(0);
}

void MipMapGen::Dither24(ImageData* ditheredImage,
	                     ImageData* quantizedImage,
			             ImageData* origImage)
{
	// Matrix for determining position of neighboring pixels (Reduced to affected pixels)
	const Point PosMatrix[4] = { Point( 1, 0),
							     Point(-1, 1),
							     Point( 0, 1),
								 Point( 1, 1) };

	// Floyd-Steinberg error diffusion matrix showing percent of color error
	// that should be propagated over the above positions
	const float ErrMatrix[4] = { 7.0f / 7.0f,
		                         3.0f / 7.0f,
								 5.0f / 7.0f,
								 1.0f / 7.0f };

	*ditheredImage = *quantizedImage;

	color24* sBuf = (color24*)origImage->buf;
	color24* dBuf = (color24*)ditheredImage->buf;

	TreeData* ctree = quant.BuildColorTree(quantizedImage);

	//return;

	int x,y;
	int width  = origImage->width;
	int height = origImage->height;

	int rDiff, gDiff, bDiff;

	int ydir,xdir;
	ydir = 1;
	xdir = 1;

	for(y = 0; y < height; y+=ydir)
	{
		if (xdir > 0)
			x = 0;
		else
			x = width;

		int oldy = y;

		for(; (xdir > 0) ? (x < width) : (x > -1); x+=xdir)
		{
			y++;
			color24* ptColor = &sBuf[y*width+x];
			color24*  newClr = &dBuf[y*width+x];

//			rDiff = newClr->r - ptColor->r;
//			gDiff = newClr->g - ptColor->g;
//			bDiff = newClr->b - ptColor->b;

			rDiff = ptColor->r - newClr->r;
			gDiff = ptColor->g - newClr->g;
			bDiff = ptColor->b - newClr->b;


			// Update neighboring pixels
			for(int i=0;i<4;i++)
			{
				Point ptSample = Point(x+PosMatrix[i].x,y+PosMatrix[i].y);

				if (ptSample.x > -1 && ptSample.x < width &&
					ptSample.y > -1 && ptSample.y < height)
				{
					color24* ptNP = &dBuf[(y+PosMatrix[i].y)*width+x+PosMatrix[i].x];
					
					float tval;
					
					tval = ptNP->r + rDiff * ErrMatrix[i];

					// Perform clamping on the computed value
					if (tval < 0)
						tval = 0;

					if (tval > 255)
						tval = 255;

					ptNP->r = (unsigned char)tval;
					
					tval = ptNP->g + gDiff * ErrMatrix[i];

					// Perform clamping on the computed value
					if (tval < 0)
						tval = 0;

					if (tval > 255)
						tval = 255;

					ptNP->g = (unsigned char)tval;

					tval = ptNP->b + bDiff * ErrMatrix[i];

					// Perform clamping on the computed value
					if (tval < 0)
						tval = 0;

					if (tval > 255)
						tval = 255;

					ptNP->b = (unsigned char)tval;

					//color32 color;
					__int32 dist = -1;
					//quant.FindLinearMatch(ptNP->r,ptNP->g,ptNP->b,255,ctree,&color,&dist);
					//color = quant.FindColorMatch(ctree,ptNP->r,ptNP->g,ptNP->b);

					color32 color = quant.GetClosestColor32(ctree->root,ptNP->r,ptNP->g,ptNP->b);
					ptNP->r = color.r;
					ptNP->g = color.g;
					ptNP->b = color.b;

				}
			}
			y = oldy;
		}
	

		if (xdir == 1)
			xdir = -1;
		else
			xdir = 1;

	}

	delete ctree;
}

void MipMapGen::Dither32(ImageData* ditheredImage,
	                     ImageData* quantizedImage,
			             ImageData* origImage)
{
	assert(0);
}

void MipMapGen::MakeVTransparent(ImageData* destImage,
							     ImageData* srcImage)
{
	destImage->CopyNoImage(srcImage);
	
	int depth = srcImage->bit_depth * srcImage->nChannels;

	if (destImage->buf)
		free(destImage->buf);

	destImage->buf = (unsigned char*)malloc(srcImage->width*srcImage->height*depth);

	color32* sBuf = (color32*)srcImage->buf;
	color32* dBuf = (color32*)destImage->buf;

	unsigned int x,y;

	for(y=0;y<srcImage->height;y++)
	{
		for(x=0;x<srcImage->width;x++)
		{
			color32 color = sBuf[y*srcImage->width+x];

			if (x==0 || x==srcImage->width-1)
			{
				color.a = 0;
			}

			dBuf[y*srcImage->width+x] = color;
		}
	}
}

void MipMapGen::MakeVTransparentRight(ImageData* destImage,
									  ImageData* srcImage)
{
	destImage->CopyNoImage(srcImage);
	
	int depth = srcImage->bit_depth * srcImage->nChannels;

	if (destImage->buf)
		free(destImage->buf);

	destImage->buf = (unsigned char*)malloc(srcImage->width*srcImage->height*depth);

	color32* sBuf = (color32*)srcImage->buf;
	color32* dBuf = (color32*)destImage->buf;

	unsigned int x,y;

	for(y=0;y<srcImage->height;y++)
	{
		for(x=0;x<srcImage->width;x++)
		{
			color32 color = sBuf[y*srcImage->width+x];

			if (x==srcImage->width-1)
			{
				color.a = 0;
			}

			dBuf[y*srcImage->width+x] = color;
		}
	}
}

void MipMapGen::MakeVTransparentLeft(ImageData* destImage,
									 ImageData* srcImage)
{
	destImage->CopyNoImage(srcImage);
	
	int depth = srcImage->bit_depth * srcImage->nChannels;

	if (destImage->buf)
		free(destImage->buf);

	destImage->buf = (unsigned char*)malloc(srcImage->width*srcImage->height*depth);

	color32* sBuf = (color32*)srcImage->buf;
	color32* dBuf = (color32*)destImage->buf;

	unsigned int x,y;

	for(y=0;y<srcImage->height;y++)
	{
		for(x=0;x<srcImage->width;x++)
		{
			color32 color = sBuf[y*srcImage->width+x];

			if (x==0)
			{
				color.a = 0;
			}

			dBuf[y*srcImage->width+x] = color;
		}
	}
}

void MipMapGen::MakeUTransparentTop(ImageData* destImage,
									ImageData* srcImage)
{
	destImage->CopyNoImage(srcImage);
	
	int depth = srcImage->bit_depth * srcImage->nChannels;

	if (destImage->buf)
		free(destImage->buf);

	destImage->buf = (unsigned char*)malloc(srcImage->width*srcImage->height*depth);

	color32* sBuf = (color32*)srcImage->buf;
	color32* dBuf = (color32*)destImage->buf;

	unsigned int x,y;

	for(y=0;y<srcImage->height;y++)
	{
		if (y==0)
		{
			for(x=0;x<srcImage->width;x++)
			{
				color32 color = sBuf[y*srcImage->width+x];
				color.a = 0;
				dBuf[y*srcImage->width+x] = color;
			}
		}
		else
		{
			for(x=0;x<srcImage->width;x++)
			{
				color32 color = sBuf[y*srcImage->width+x];
				dBuf[y*srcImage->width+x] = color;
			}
		}
	}
}

void MipMapGen::MakeUTransparentBottom(ImageData* destImage,
									   ImageData* srcImage)
{
	destImage->CopyNoImage(srcImage);
	
	int depth = srcImage->bit_depth * srcImage->nChannels;

	if (destImage->buf)
		free(destImage->buf);

	destImage->buf = (unsigned char*)malloc(srcImage->width*srcImage->height*depth);

	color32* sBuf = (color32*)srcImage->buf;
	color32* dBuf = (color32*)destImage->buf;

	unsigned int x,y;

	for(y=0;y<srcImage->height;y++)
	{
		if (y==srcImage->height-1)
		{
			for(x=0;x<srcImage->width;x++)
			{
				color32 color = sBuf[y*srcImage->width+x];
				color.a = 0;
				dBuf[y*srcImage->width+x] = color;
			}
		}
		else
		{
			for(x=0;x<srcImage->width;x++)
			{
				color32 color = sBuf[y*srcImage->width+x];
				dBuf[y*srcImage->width+x] = color;
			}
		}
	}
}

void MipMapGen::MakeUTransparent(ImageData* destImage,
							     ImageData* srcImage)
{
	destImage->CopyNoImage(srcImage);
	
	int depth = srcImage->bit_depth * srcImage->nChannels;

	if (destImage->buf)
		free(destImage->buf);

	destImage->buf = (unsigned char*)malloc(srcImage->width*srcImage->height*depth);

	color32* sBuf = (color32*)srcImage->buf;
	color32* dBuf = (color32*)destImage->buf;

	unsigned int x,y;

	for(y=0;y<srcImage->height;y++)
	{
		if (y==0 || y==srcImage->height-1)
		{
			for(x=0;x<srcImage->width;x++)
			{
				color32 color = sBuf[y*srcImage->width+x];
				color.a = 0;
				dBuf[y*srcImage->width+x] = color;
			}
		}
		else
		{
			for(x=0;x<srcImage->width;x++)
			{
				color32 color = sBuf[y*srcImage->width+x];
				dBuf[y*srcImage->width+x] = color;
			}
		}
	}
}

void MipMapGen::DumpHelp()
{
#ifdef _DEBUG
	const char* sBuildMode = "Debug";
#else
	const char* sBuildMode = "Release";
#endif

	printf("MipGen - MipMap Generation Utility (%s) %s %s\n", sBuildMode, __DATE__, __TIME__);
	printf("NeverSoft Entertainment  2002\n\n");

	printf("MipGen [options] [SourceFile] {...}\n");
	printf("[SourceFile]\tSpecifies the source file(s) to use for the conversion\n");
	printf("\t\tYou may string together as many source files as necessary\n");
	printf("\t\tto perform the operation\n");
	printf("-d[DestFile]\tSpecifies the destination file to use for output\n");
	printf("-op[path]\tSpecifies output path to write mips to\n");
	printf("-p[file]\tUse the palette information from this image to quantize all\n");
	printf("-pf[file]\tUse the palette data from this palette file to quantize all\n");
	printf("-s[w]x[h]\tScale the image to the given size (i.e. 8x8,32x32,128x128,etc)\n");
	printf("-m\t\tGenerate mip maps for source\n"); //
	printf("-mt[w]x[h]\tTarget mip map size (smallest mip)\n");
	printf("-q[maxcolors]\tQuantize by similar colors\n");
	//printf("-qf[maxcolors]\tQuantize by color frequency\n");
	printf("-dp[palfile]\tDump palette data to file\n");
	printf("-D Dither the image (must be used with quantization)\n");
	printf("-b[depth]\tSpecify bit depth for output files\n");
	printf("-bo\tConverts bit depth to 8-bit per channel (32 w/ alpha 24 w/o)\n");
	printf("-t\t\tDump as raw Targa file (16 - 32-bit only)\n");
	printf("-rcn\t\tReduce image by closest neighbor\n");
	printf("-rbf\t\tReduce image using averaging box filter\n");
	printf("-ut\t\tMake U edges (horiz) transparent\n"); //
	printf("-vt\t\tMake V edges (vert) transparent\n"); //
	printf("-tc[r,g,b]\tSets the transparency to forced color\n");
	printf("-lm[num]\tLimits the number of mips to generate (-lm0 does base only)\n");
	printf("-dm\tDelete old mips\n");
	printf("\n");
}

void MipMapGen::ProcScale(ImageData* img)
{
	if (scaleWidth>0 || scaleHeight>0)
	{
		int sw,sh;
			
		if (scaleWidth>0)
			sw = scaleWidth;
		else
			sw = img->width;

		if (scaleHeight)
			sh = scaleHeight;
		else
			sh = img->height;		
	}
}

void MipMapGen::EdgeTransU(ImageData* image)
{

}

void MipMapGen::EdgeTransV(ImageData* image)
{
	
}

// Note this function should only be used on 32-bit image data
void MipMapGen::AssignTrans(ImageData* destimage, color24 transColor)
{
	int width = destimage->width;
	int height = destimage->height;
	int size = width * height;

	assert(destimage->pixel_depth == 32);

	for(int idx = 0; idx < size; idx++)
	{
		color32 color = ((color32*)destimage->buf)[idx];

		if (color.a == 0)
		{
			color.r = transColor.r;
			color.g = transColor.g;
			color.b = transColor.b;

			((color32*)destimage->buf)[idx] = color;
		}
	}
}

void MipMapGen::DeleteWildcard(char* path,char* wildcard)
{
	_finddata_t fdata;
	char delpath[512];
	char delfile[512];
	strcpy(delpath,path);
	strcat(delpath,"\\");
	strcat(delpath,wildcard);

	long hFF = _findfirst(delpath,&fdata);

	if (hFF != -1)
	{
		printf("\nDelete %s...",fdata.name);
		strcpy(delfile,path);
		strcat(delfile,"\\");
		strcat(delfile,fdata.name);	
		DeleteFile(delfile);

		while( _findnext(hFF,&fdata) == 0 )
		{
			printf("\nDelete %s...",fdata.name);
			strcpy(delfile,path);
			strcat(delfile,"\\");
			strcat(delfile,fdata.name);
			DeleteFile(delfile);
		}

		_findclose(hFF);
	}
}

// Returns number of mips of the last image processed
int MipMapGen::Proc()
{
	int i=0;
	int size = srcList.GetSize();
	bool bCutout;
	int curlevel = 0;

	Link<char*>* srcfile = srcList.GetHead();
	ImageData* srcimg = new ImageData[size];

	// Read all the source images into memory
	while(srcfile)
	{
		printf("\nReading %s...",srcfile->data);
		srcimg[i].ReadPNG(srcfile->data);
		
		if (bReport)
		{
			printf("Width: %i",srcimg[i].width);
			printf("Height: %i",srcimg[i].height);
			printf("Bitdepth: %i",srcimg[i].bit_depth);
			printf("Channels: %i",srcimg[i].nChannels);
			printf("ColorType: %i",srcimg[i].color_type);
			return 0;
		}

		// Process deletion of old mips
		if (bDelOldMips)
		{
			char bufDrv[10];
			char bufDir[512];
			char bufFName[128];
			char bufExt[128];
			_splitpath(srcfile->data,bufDrv,bufDir,bufFName,bufExt);

			char filename[256];

			strcpy(filename,bufFName);
			strcat(filename,"_auto32m*.png");

			DeleteWildcard(outPath,filename);
		}

		srcfile = srcfile->next;
		i++;
	}

	ImageData* refImage = new ImageData;

	// Process the loaded source images
	for(i = 0; i < size; i++)
	{
		// Convert to the appropriate bit depth first
		ImageData* srcimage  = new ImageData;
		ImageData* destimage = new ImageData;

		// Set output depth for current image
		if (outDepth == 0)
		{
			// Output as 24 bit if palettized with no alpha entries or simply
			// sampled as only 3 channels
			if (srcimg[i].nChannels < 4 && srcimg[i].tRNSentries == 0)
				curOutDepth = 24;
			else
				curOutDepth = 32;
		}
		else
			curOutDepth = outDepth;

		switch(curOutDepth)
		{
		case 16:
			srcimg[i].Convert16(srcimage);
			break;

		case 24:
			srcimg[i].Convert24(srcimage);
			break;

		case 32:
			srcimg[i].Convert32(srcimage);
			break;
		}

		bCutout = IsCutout(srcimage);

		// If a different color has been flagged to use as transparency
		// and we're (32-bit) assign it for alpha 0
		// (If this is a cutout texture we do this application last, if not, we do it now)
		if (bUseTransColor && curOutDepth == 32 &&
			!bCutout)
			AssignTrans(srcimage,transColor);

		*refImage = *srcimage;

		// Quantize palette
		if (quantColors > 0)
		{
			// We will quantize the image down to the given number of colors
			switch(curOutDepth)
			{
			case 8:
				quant.ProcImage(destimage,srcimage,quantColors,Quantizer::QUANT_PALETTE | Quantizer::QUANT_ALPHA);
				break;
			case 16:
			case 24:
			case 32:
				quant.ProcImage(destimage,srcimage,quantColors,0);
				break;
			}

			// Perform dithering
			if (bDither)
			{
				ImageData ditherimage;
				Dither(&ditherimage,destimage,srcimage);

				*destimage = ditherimage;
			}
		}

		char bufOutDepth[256];
		_itoa(curOutDepth,bufOutDepth,10);

		// Generate mip maps
		if (bGenMips)
		{
			unsigned int mipWidth = srcimg[i].width;
			unsigned int mipHeight = srcimg[i].height;
			
			curlevel = 0;
			char filename[512];

			do
			{
				if (curlevel > 0)
				{
					if (mipWidth != 1)
						mipWidth >>= 1;

					if (mipHeight != 1)
						mipHeight >>= 1;
				}

				if (limitMips == -1 ||
					curlevel <= limitMips)
				{
					printf("\nGenerating miplevel %i...",curlevel);

					// If we're outputting a 32-bit image with cutout alpha (only 0 || 255)
					// we will use the closest neighbor method for the alpha channel
					if (curOutDepth == 32 &&
						bCutout)
					{
						Proc32BitBoxReduceACN(destimage,srcimage,mipWidth,mipHeight);
					}
					else
					{
						if (bReduceCN)
							ResizeImageCN(destimage,srcimage,mipWidth,mipHeight);
						else
							ResizeImageBox(destimage,srcimage,mipWidth,mipHeight);
					}

					// Edge transparency
					if (curOutDepth == 32)
					{
						if (bVTransparent)
						{
							*srcimage = *destimage;
							MakeVTransparent(destimage,srcimage);
						}

						if (bUTransparent)
						{
							*srcimage = *destimage;
							MakeUTransparent(destimage,srcimage);
						}					

						// Maintain transparent U/V edge  (This is forced on for now to prevent plugin rebuild)
						if (IsTransUTop(refImage))
						{
							*srcimage = *destimage;
							MakeUTransparentTop(destimage,srcimage);
						}

						if (IsTransUBottom(refImage))
						{
							*srcimage = *destimage;
							MakeUTransparentBottom(destimage,srcimage);
						}

						if (IsTransVLeft(refImage))
						{
							*srcimage = *destimage;
							MakeVTransparentLeft(destimage,srcimage);
						}

						if (IsTransVRight(refImage))
						{
							*srcimage = *destimage;
							MakeVTransparentRight(destimage,srcimage);
						}
					}

					// If this is a cutout texture this is the point at which we should be
					// assigning the transparent alpha color (after mip generation)
					if (bUseTransColor && curOutDepth == 32 &&
						bCutout)
						AssignTrans(destimage,transColor);

					char bufMipLevel[256];
					_itoa(curlevel,bufMipLevel,10);
					strcpy(filename,srcList[i]);
					
					// Terminate at extension
					int slen = strlen(filename);

					char bufDrv[10];
					char bufDir[512];
					char bufFName[128];
					char bufExt[128];
					_splitpath(filename,bufDrv,bufDir,bufFName,bufExt);

					strcpy(filename,outPath);
					strcat(filename,"\\");
					strcat(filename,bufFName);
					
					if (curOutDepth == 24 || curOutDepth == 32)
					{
						// For now we're forcing the name to 32 regardless
						strcpy(bufOutDepth,"32");

						strcat(filename,"_auto");
						strcat(filename,bufOutDepth);
						strcat(filename,"m");
					}
					else
					{
						strcat(filename,"_autom");
					}

					strcat(filename,bufMipLevel);
					strcat(filename,".png");

					destimage->WritePNG(filename);
				}

				curlevel++;

			} while(mipWidth  > mipTargetWidth ||
				  mipHeight > mipTargetHeight);
		}
	}

	// Dump palette (This will always be the palette from the first specified source file
	// (Unless the palette is quantized, but we'll handle that later)
	if (outPalFile)
	{	
		Link<char*>* srcfile = srcList.GetHead();

		if (srcfile)
		{
			ImageData* img = new ImageData;

			if (!img->ReadPNG(srcfile->data))
				printf("WARNING! Failed to read PNG file.\n");

			if (!img->SavePal(outPalFile))
				printf("WARNING! Failed to save palette data to file.\n");
		}
	}

	if (refImage)
		delete refImage;

	return curlevel;
}

// Scan image to determine if a specific side has a constant 0 alpha  assumed 32-bit
bool MipMapGen::IsTransUTop(ImageData* image)
{
	color32* sBuf = (color32*)image->buf;
	unsigned int x,y;

	for(y=0;y<image->height;y++)
	{
		if (y==0)
		{
			for(x=0;x<image->width;x++)
			{
				color32 color = sBuf[y*image->width+x];
				
				if (color.a != 0)
					return false;
			}
		}
	}

	return true;
}

bool MipMapGen::IsTransUBottom(ImageData* image)
{
	color32* sBuf = (color32*)image->buf;
	unsigned int x,y;

	for(y=0;y<image->height;y++)
	{
		if (y==image->height-1)
		{
			for(x=0;x<image->width;x++)
			{
				color32 color = sBuf[y*image->width+x];
				
				if (color.a != 0)
					return false;
			}
		}
	}

	return true;
}

bool MipMapGen::IsTransU(ImageData* image)
{
	if (IsTransUTop(image) || IsTransUBottom(image))
		return true;

	return false;
}

bool MipMapGen::IsTransVLeft(ImageData* image)
{
	// Check left edge
	color32* sBuf = (color32*)image->buf;
	unsigned int x,y;

	for(y=0;y<image->height;y++)
	{
		for(x=0;x<image->width;x++)
		{
			color32 color = sBuf[y*image->width+x];

			if (x==0)
			{
				if (color.a != 0)
					return false;
			}
		}
	}

	return true;
}

bool MipMapGen::IsTransVRight(ImageData* image)
{
	// Check left edge
	color32* sBuf = (color32*)image->buf;
	unsigned int x,y;

	for(y=0;y<image->height;y++)
	{
		for(x=0;x<image->width;x++)
		{
			color32 color = sBuf[y*image->width+x];

			if (x==image->width-1)
			{
				if (color.a != 0)
					return false;
			}
		}
	}

	return true;
}

// The image is considered a cutout image if it uses only values of 0 and 255
// for the alpha channel, this function assumes 32-bit input
bool MipMapGen::IsCutout(ImageData* image)
{
	color32* sBuf = (color32*)image->buf;
	int size = image->width * image->height;

	for(int i=0; i < size; i++)
	{
		if (sBuf[i].a != 0 &&
			sBuf[i].a != 255)
			return false;
	}

	return true;
}

bool MipMapGen::IsTransV(ImageData* image)
{
	if (IsTransVLeft(image) || IsTransVRight(image))
		return true;

	return false;
}

void MipMapGen::Init(int argc,char** argv)
{
	// Build up list of source files to use
	for(int i=1;i<argc;i++)
	{
		if (argv[i][0] != '-')
			srcList.AddToTail(&argv[i]);
		else
		{
			// Check first 3 characters for option
			char opt[4];
			opt[0] = argv[i][0];
			opt[1] = argv[i][1];
			opt[2] = argv[i][2];
			opt[3] = 0;

			// Assign any options
			// 3 chr opts
			if (strcmps(argv[i],"-rcn") == 0)
			{
				bReduceCN = true;
			}

			if (strcmps(argv[i],"-rbf") == 0)
			{
				bReduceCN = false;
			}

			// 2 chr opts
			if (strcmps(argv[i],"-op") == 0)
			{
				outPath = argv[i] + 3;
				continue;
			}
			
			if (strcmps(argv[i],"-dp") == 0)
			{
				outPalFile = argv[i] + 3;
				continue;
			}

			if (strcmps(argv[i],"-pf") == 0)
			{
				inPalFile = argv[i] + 3;
				continue;
			}

			if (strcmps(argv[i],"-mt") == 0)
			{
				char buf[256];
				GetOption(argv[i] + 3,buf,'x',0);
				
				mipTargetWidth = atoi(buf);

				GetOption(argv[i] + 3,buf,'x',1);

				mipTargetHeight = atoi(buf);
				continue;
			}

			if (strcmps(argv[i],"-ut") == 0)
			{
				bUTransparent = true;
			}

			if (strcmps(argv[i],"-uv") == 0)
			{
				bVTransparent = true;
			}

			if (strcmps(argv[i],"-bo") == 0)
			{
				outDepth = 0;
				continue;
			}

			if (strcmps(argv[i],"-tc") == 0)
			{
				bUseTransColor = true;

				// Parse out the transparency color from the command line
				char bufR[256];
				char bufG[256];
				char bufB[256];
				GetOption(argv[i] + 3,bufR,',',0);
				GetOption(argv[i] + 3,bufG,',',1);
				GetOption(argv[i] + 3,bufB,',',2);

				transColor.r = atoi(bufR);
				transColor.g = atoi(bufG);
				transColor.b = atoi(bufB);
			}

			if (strcmps(argv[i],"-lm") == 0)
			{
				limitMips = atoi(argv[i] + 3);
				continue;
			}

			if (strcmps(argv[i],"-dm") == 0)
			{
				bDelOldMips = true;
				continue;
			}

			// 1 chr opts
			if (strcmps(argv[i],"-b") == 0)
			{
				outDepth = atoi(argv[i] + 2);
				continue;
			}

			if (strcmps(argv[i],"-m") == 0)
			{
				bGenMips = true;
				continue;
			}

			if (strcmps(argv[i],"-D") == 0)
			{
				bDither = true;
				continue;
			}

			if (strcmps(argv[i],"-d") == 0)
			{
				destFile = argv[i] + 2;
				continue;
			}

			if (strcmps(argv[i],"-s") == 0)
			{
				char buf[256];
				GetOption(argv[i] + 2,buf,'x',0);

				scaleWidth = atoi(buf);

				GetOption(argv[i] + 2,buf,'x',1);

				scaleHeight = atoi(buf);
				continue;
			}

			if (strcmps(argv[i],"-t") == 0)
			{
				bDumpTarga = true;
			}


			if (strcmps(argv[i],"-q") == 0)
			{
				quantColors = atoi(argv[i] + 2);
			}

			if (strcmps(argv[i],"-r") == 0)
			{
				bReport = true;
			}
		}
	}	
}

