#include "FuncEnter.h"

/*
	GapTool.cpp
	Gap layout tool
*/

#include "GapTool.h"
#include "Resource.h"
#include "../PropEdit/ParseFuncs.h"
#include "../UI/NodeSelect.h"
#include "path.h"
#include "../Export/SceneExportOptions.h"
#include "../PropEdit/ConfigData.h"
#include "../PropEdit/PropEdit.h"
#include "../misc/maxutil.h"

///////// Accelerator stuff
static GapToolClassDesc  theGapToolDesc;
static GapActionCB       gapActionCB;
static GapActionCB*      gapAccel=NULL;
static GapToolDlg*       ggapToolDlg;

// We'll get the mapfile data from the property editor instead (as it's already loaded)
// and we won't have to wait at startup then
extern PropEditor* pPropEdit;

ClassDesc2* GetGapToolDesc() { FUNC_ENTER("GetGapToolDesc");  return &theGapToolDesc; }

extern HINSTANCE hInstance;

bool RegGapActionAccelerators();
void UnRegGapActionAccelerators();

// action table
static ActionDescription s_gap_actions[] = {

	ID_GAP_OPEN,
	IDS_GAPTOOL_DESC,
    IDS_GAPTOOL_DESC,
    IDS_GAPTOOL_ACTIONS,
};

const ActionTableId gap_actions           = vGAPTOOL_SHORTCUT_ID;
const ActionContextId gap_actions_context = vGAPTOOL_SHORTCUT_ID;

void GapDlgCloseCB(MSDlgWindow* dlg, void* pData)
{ FUNC_ENTER("GapDlgCloseCB"); 	
	GapToolDlg* pthis = (GapToolDlg*)pData;

	if (ggapToolDlg)
	{
		delete ggapToolDlg;
		ggapToolDlg = NULL;
	}
}

BOOL GapActionCB::ExecuteAction(int id)
{ FUNC_ENTER("GapActionCB::ExecuteAction"); 
	// Register the NodeSelector class
	REGISTERCLASS(NodeSelect,hInstance);

	if (!ggapToolDlg)
	{
		ggapToolDlg = new GapToolDlg(hInstance,GetCOREInterface()->GetMAXHWnd(),GetCOREInterface(),pPropEdit);
		ggapToolDlg->Show();
		ggapToolDlg->SetCloseCB(GapDlgCloseCB, this);
	}
	else
	{
		ggapToolDlg->Show();
	}

	return TRUE;
}

ActionTable* GetGapActions( void )
{ FUNC_ENTER("GetGapActions"); 
    //TSTR name = GetString(IDS_LINK_ACTIONS);
	TSTR name = _T("Neversoft Gap Tool");
    HACCEL hAccel = LoadAccelerators(hInstance,
                                     MAKEINTRESOURCE(IDR_GAP_ACCELERATOR));
    int numOps = sizeof(s_gap_actions) / sizeof(s_gap_actions[0]);
    ActionTable* pTab;
    pTab = new ActionTable( gap_actions, gap_actions_context, name, hAccel, numOps,
                             s_gap_actions, hInstance);
    GetCOREInterface()->GetActionManager()->RegisterActionContext( gap_actions_context, name.data());

    return pTab;
}

bool RegGapActionAccelerators()
{ FUNC_ENTER("RegGapActionAccelerators"); 
	gapAccel=new GapActionCB;

	if (!GetCOREInterface()->GetActionManager()->ActivateActionTable(gapAccel, gap_actions))
	{
		MessageBox(NULL,"ActionTable Failed to instantiate","Gap Tool",MB_OK);
		return false;
	}

	return true;
}

void UnRegGapActionAccelerators()
{ FUNC_ENTER("UnRegGapActionAccelerators"); 
	delete gapAccel;
	gapAccel=NULL;
}
///////////////////// End Accelerator stuff

GapTool::GapTool()
{ FUNC_ENTER("GapTool::GapTool"); 
	
}

GapTool::~GapTool()
{ FUNC_ENTER("GapTool::~GapTool"); 

}

void GapTool::BeginEditParams(Interface *ip,IUtil *iu)
{ FUNC_ENTER("GapTool::BeginEditParams"); 
	this->ip = ip;

	// Register the NodeSelector class
	REGISTERCLASS(NodeSelect,hInstance);

	gapToolDlg = new GapToolDlg(hInstance,ip->GetMAXHWnd(),ip,pPropEdit);
	gapToolDlg->Show();
}

void GapTool::EndEditParams(Interface *ip,IUtil *iu)
{ FUNC_ENTER("GapTool::EndEditParams"); 
	delete gapToolDlg;
}

void GapTool::SelectionSetChanged(Interface *ip,IUtil *iu)
{ FUNC_ENTER("GapTool::SelectionSetChanged"); 
	if (gapToolDlg)
		gapToolDlg->SelSetChanged();
}

//////////////////// GapToolDlg

GapToolDlg::GapToolDlg(HINSTANCE hInstance,HWND hwndParent,Interface* ip, MapFileParser* pMapFileParser) :
	MSDlgWindow(hInstance, MAKEINTRESOURCE(IDD_GAPTOOL), hwndParent)
{ FUNC_ENTER("GapToolDlg::GapToolDlg"); 
	this->ip = ip;
	pbar = NULL;

	mapfile = pMapFileParser;

	assert( mapfile );

	bLockSelChange = false;
	bInternalSet = false;

	IEditGapID = GetICustEdit(GetDlgItem(hwnd,IDC_EDITGAPID));
	nsNodeA = new NodeSelect(hInstance);
	nsNodeB = new NodeSelect(hInstance);

	// These proplists are only used to maintain data
	plistStartGap = new PropList(hInstance,1);
	plistEndGap = new PropList(hInstance,1);

	plistStartGap->HasApply(false);
	plistEndGap->HasApply(false);

	plistStartGap->Attach(GetDlgItem(hwnd,IDC_STARTPROPS));
	plistEndGap->Attach(GetDlgItem(hwnd,IDC_ENDPROPS));

	nsNodeA->Attach(GetDlgItem(hwnd,IDC_NODEA));
	nsNodeB->Attach(GetDlgItem(hwnd,IDC_NODEB));

	// Set node A and B to the current selection
	int nNodes = ip->GetSelNodeCount();

	if (nNodes >= 1)
		nsNodeA->SetNode(ip->GetSelNode(0));

	if (nNodes >= 2)
		nsNodeB->SetNode(ip->GetSelNode(1));

	// For first release don't worry about gaps.ini
	/*
	// Load in the Gaps.ini file
	CStr strFile;

	strFile=getenv(APP_ENV);

	// Verify that the environment variable is set
	if (strFile==CStr(""))
	{
		char ErrorBuf[256];
		sprintf(ErrorBuf,"Couldn't load '%s' the '%s' environment variable is not set.",SCRIPT_INI,APP_ENV);

		MessageBox(ip->GetMAXHWnd(),ErrorBuf,"ParseGapIni",MB_ICONSTOP|MB_OK);
	}

	strFile+=CStr(SCRIPT_PATH)+GAP_INI;

	if (!ParseScriptIni(strFile))
		MessageBox(ip->GetMAXHWnd(),"Unable to parse the gaps.ini file.","Gap Layout Tool",MB_ICONWARNING|MB_OK);
	*/

	// Parse the map file so we can lookup the AutoDuck syntax for startgap/endgap commands
	//if (!mapfile->ParseTagDBFile())
	//	MessageBox(ip->GetMAXHWnd(),"Unable to parse the tagsdb file to evaluate StartGap/EndGap parameters","Gap Layout Tool",MB_ICONWARNING|MB_OK);

	// Obtain property data
	TypeDesc* tdesc = mapfile->GetScriptDB()->GetItem("StartGap");

	if (tdesc)
		mapfile->ParseScriptTags(plistStartGap,tdesc->ScriptFile,tdesc->Name);

	// Clear out the default GapID value
	plistStartGap->SetValue("GapID","");

	tdesc = mapfile->GetScriptDB()->GetItem("EndGap");

	if (tdesc)
		mapfile->ParseScriptTags(plistEndGap,tdesc->ScriptFile,tdesc->Name);

 	if (IsDlgButtonChecked(hwnd,IDC_INTGAPS)==BST_UNCHECKED)
		FindGapsStatus();

	// Reset gap text
	plistEndGap->SetValue("text","Unnamed Gap");

	ReadGapProps();
	BuildGapList();
	BuildGapTextList();

	plistStartGap->BuildUI();
	plistEndGap->BuildUI();
}

GapToolDlg::~GapToolDlg()
{ FUNC_ENTER("GapToolDlg::~GapToolDlg"); 
	DestroyWindow(hwnd);

	if (IEditGapID)
		ReleaseICustEdit(IEditGapID);

	if (nsNodeA)
		delete nsNodeA;

	if (nsNodeB)
		delete nsNodeB;

	if (plistStartGap)
		delete plistStartGap;

	if (plistEndGap)
		delete plistEndGap;
}


BOOL GapToolDlg::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("GapToolDlg::DlgProc"); 
	switch(msg)
	{
	case WM_CLOSE:
		Hide();

		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_APPLY:
			Apply(true);
			return TRUE;

		case IDC_ADDGAP:
			Apply();
			return TRUE;

		case IDOK:
			Apply();
			Hide();

			return TRUE;

		case IDCANCEL:
			Hide();

			return TRUE;

		case IDC_REMOVE:
			{
				char buf[256];
				IEditGapID->GetText(buf,255);
				//CStr str = buf;
				//str += CStr("_") + nsNodeA->GetValue();

				//RemoveGap(str);
				RemoveGap(buf);

				ReadGapProps();
				BuildGapList();
				BuildGapTextList();
			}
			return TRUE;

		case IDC_REFRESH:
		 	if (IsDlgButtonChecked(hwnd,IDC_INTGAPS)==BST_UNCHECKED)
				FindGapsStatus();

			ReadGapProps();
			BuildGapList();
			BuildGapTextList();
			return TRUE;

		case IDC_INTGAPS:
			if (IsDlgButtonChecked(hwnd, IDC_INTGAPS) == BST_CHECKED && bInternalSet)
				return TRUE;

			if (IsDlgButtonChecked(hwnd, IDC_INTGAPS) != BST_CHECKED && !bInternalSet)
				return TRUE;

			if (IsDlgButtonChecked(hwnd, IDC_INTGAPS) == BST_CHECKED)
				bInternalSet = true;
			else
				bInternalSet = false;

			GapListChanged();
			return TRUE;
		}

		switch(HIWORD(wParam))
		{
		case LBN_SELCHANGE:
			switch(LOWORD(wParam))
			{
			case IDC_GAPGROUPS:
				SelGapGroup();
				return TRUE;

			case IDC_STARTGAPS:
				SelGapStart();
				return TRUE;

			case IDC_GAPTEXT:
				SelGapText();
				return TRUE;

			case IDC_ENDGAPS:
				SelGapEnd();
				return TRUE;
			}
			break;
		
		case EN_CHANGE:
			switch(LOWORD(wParam))
			{
			case IDC_EDITGAPID:
				GapIDChanged();
				return TRUE;
			}
		}
	};

	return FALSE;
}

void GapToolDlg::FindRelatedGaps(CStr gapid,INode* root)
{ FUNC_ENTER("GapToolDlg::FindRelatedGaps"); 
	if (root==NULL)
		root = ip->GetRootNode();

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
	{
		INode* child = root->GetChildNode(i);
		FindRelatedGaps(gapid,child);
	}

	if (root == ip->GetRootNode())
		return;

	// Determine if this is a related gap node
	// and if so, add it to the gap list
	CStr propBuffer;

	root->GetUserPropBuffer(propBuffer);

	// Parse line by line to see if the line contains any startgap/endgap calls with our gapid
	int  pos = 0;
	int  len = propBuffer.Length();
	CStr line;

	while(pos<len)
	{
		line = GetRemainLineExact(propBuffer,&pos);
		line.toLower();

		if (IsInstr(line,"startgap") ||
			IsInstr(line,"endgap"))
		{
			// Determine if the gap 
		}
	}
}

void GapToolDlg::FindGapsStatus()
{ FUNC_ENTER("GapToolDlg::FindGapsStatus"); 
	int numNodes = GetNodeCount();
	
	if (pbar)
		delete pbar;

	pbar = new ProgressBar(hInstance, hwnd, "Finding Gaps...", 0, numNodes);
	FindGaps(NULL, 0, pbar);

	delete pbar;
	pbar = NULL;
}

void GapToolDlg::FindGaps(INode* root, int iProgress, ProgressBar* pbar)
{ FUNC_ENTER("GapToolDlg::FindGaps"); 
	if (root==NULL)
	{
		gapList.Clear();
		root = ip->GetRootNode();
	}

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
	{
		INode* child = root->GetChildNode(i);
		
		if (pbar)
			pbar->SetVal(++iProgress);

		FindGaps(child, iProgress);
	}

	if (root == ip->GetRootNode())
		return;

	// Determine if this is a related gap node
	// and if so, add it to the gap list
	CStr propBuffer;

	root->GetUserPropBuffer(propBuffer);

	// Limit gap detection to only alternate [level]Gaps.qn file
	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	CStr name = seo.m_SceneName + CStr("_Gaps.q");

	LinkList<ConfigScript> cscripts;
	ParseConfigProps(NULL,NULL,propBuffer,NULL,NULL,NULL,&cscripts);

	ParseGapData(root,&propBuffer);
}

void GapToolDlg::ParseGapData(INode* node,CStr* buf)
{ FUNC_ENTER("GapToolDlg::ParseGapData"); 
	CStr propBuffer;
	node->GetUserPropBuffer(propBuffer);

	// Parse line by line to see if the line contains any startgap/endgap calls with our gapid
	int  pos = 0;
	int  len = buf->Length();
	CStr line,lcline;

	while(pos<len)
	{
		line = GetRemainLineExact(*buf,&pos);
		lcline = line;
		lcline.toLower();

		if (IsInstr(lcline,"startgap") ||
			IsInstr(lcline,"endgap"))
		{
			GapDesc gdSearch;
			gdSearch.gapID = GetPropVal(line,"GapID");

			// See if this gap exists, if not add it to the database
			Link<GapDesc>* gapLink = gapList.Find(&gdSearch);

			if (!gapLink)
				gapLink = gapList.Add(&gdSearch);

			// Now add all the gap properties to the database
			GapInfo    gapInfo;
			ConfigProp cprop;
			PropList*  srcList;
			gapInfo.node = node;

			if (IsInstr(lcline,"startgap"))
			{
				srcList = plistStartGap;
				gapInfo.type = STARTGAP;
			}
			else
			{
				srcList = plistEndGap;
				gapInfo.type = ENDGAP;
			}

			// Process properties
			int nProps = srcList->NumProps();

			for(int i=0;i<nProps;i++)
			{
				cprop.name  = srcList->GetName(i);
				cprop.value = GetPropVal(line,cprop.name);

				if (srcList->GetType(i) == PROPTYPE_CHECK && cprop.value == CStr(""))
					cprop.value = "FALSE";

				gapInfo.props.AddUnique(&cprop);
			}

			gapLink->data.refs.Add(&gapInfo);
		}
	}	
}

void GapToolDlg::BuildGapList()
{ FUNC_ENTER("GapToolDlg::BuildGapList"); 
	// Add all the gap groups to the list
	Link<GapDesc>* curgap = gapList.GetHead();
	int index;

	SendDlgItemMessage(hwnd,IDC_GAPGROUPS,LB_RESETCONTENT,0,0);
	SendDlgItemMessage(hwnd,IDC_STARTGAPS,LB_RESETCONTENT,0,0);
	SendDlgItemMessage(hwnd,IDC_ENDGAPS,LB_RESETCONTENT,0,0);

	while(curgap)
	{
		index = SendDlgItemMessage(hwnd,IDC_GAPGROUPS,LB_ADDSTRING,0,(LPARAM)(char*)curgap->data.gapID);
		SendDlgItemMessage(hwnd,IDC_GAPGROUPS,LB_SETITEMDATA,(WPARAM)index,(LPARAM)curgap);
		curgap = curgap->next;
	}
}

void GapToolDlg::BuildGapTextList()
{ FUNC_ENTER("GapToolDlg::BuildGapTextList"); 
	// Add all the gap text to the list
	Link<GapDesc>* curgap = gapList.GetHead();
	int index;

	SendDlgItemMessage(hwnd,IDC_GAPTEXT,LB_RESETCONTENT,0,0);

	while(curgap)
	{
		Link<GapInfo>* curinfo = curgap->data.refs.GetHead();

		while(curinfo)
		{
			CStr strText;
			ConfigProp srch;
			srch.name = "text";

			Link<ConfigProp>* cprop = curinfo->data.props.Find(&srch);

			if (cprop)
			{
				index = SendDlgItemMessage(hwnd,IDC_GAPTEXT,LB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)cprop->data.value);

				if (index == -1)
				{
					// Add the gap text
					index = SendDlgItemMessage(hwnd,IDC_GAPTEXT,LB_ADDSTRING,0,(LPARAM)(char*)cprop->data.value);
				}

				SendDlgItemMessage(hwnd,IDC_GAPTEXT,LB_SETITEMDATA,(WPARAM)index,(LPARAM)curgap);
			}

			curinfo = curinfo->next;
		}

		curgap = curgap->next;
	}
}

void GapToolDlg::BuildStartEndGapLists()
{ FUNC_ENTER("GapToolDlg::BuildStartEndGapLists"); 
	char buf[256];
	int index = SendDlgItemMessage(hwnd,IDC_GAPGROUPS,LB_GETCURSEL,0,0);

	SendDlgItemMessage(hwnd,IDC_GAPGROUPS,LB_GETTEXT,(WPARAM)index,(LPARAM)buf);

	// Now we have the text of the gapthawt we're looking for
	// dump all the start and end gaps

	SendDlgItemMessage(hwnd,IDC_STARTGAPS,LB_RESETCONTENT,0,0);
	SendDlgItemMessage(hwnd,IDC_ENDGAPS,LB_RESETCONTENT,0,0);

	// Find the current gap in the gap list
	GapDesc gapSearch;
	gapSearch.gapID = buf;

	Link<GapDesc>* gapLink = gapList.Find(&gapSearch);

	if (!gapLink)
		return;

	Link<GapInfo>* curInfo = gapLink->data.refs.GetHead();

	while(curInfo)
	{
		CStr name = curInfo->data.node->GetName();
		int  index;

		switch(curInfo->data.type)
		{
		case STARTGAP:
			index = SendDlgItemMessage(hwnd,IDC_STARTGAPS,LB_ADDSTRING,0,(LPARAM)(char*)name);
			SendDlgItemMessage(hwnd,IDC_STARTGAPS,LB_SETITEMDATA,(WPARAM)index,(LPARAM)curInfo);
			break;

		case ENDGAP:
			index = SendDlgItemMessage(hwnd,IDC_ENDGAPS,LB_ADDSTRING,0,(LPARAM)(char*)name);
			SendDlgItemMessage(hwnd,IDC_ENDGAPS,LB_SETITEMDATA,(WPARAM)index,(LPARAM)curInfo);
			break;
		}

		curInfo = curInfo->next;
	}
}

void GapToolDlg::SelGapGroup()
{ FUNC_ENTER("GapToolDlg::SelGapGroup"); 
	BuildStartEndGapLists();

	int index = SendDlgItemMessage(hwnd,IDC_GAPGROUPS,LB_GETCURSEL,0,0);

	if (index == -1)
		return;

	// Select node corresponding to the first start gap and first end gap
	SendDlgItemMessage(hwnd,IDC_ENDGAPS,LB_SETCURSEL,0,0);
	SendDlgItemMessage(hwnd,IDC_STARTGAPS,LB_SETCURSEL,0,0);
	SelGapStart();
	SelGapEnd();

	Link<GapDesc>* gapLink = (Link<GapDesc>*)SendDlgItemMessage(hwnd,IDC_GAPGROUPS,LB_GETITEMDATA,(WPARAM)index,0);

	// Select all the nodes in the scene that are part of this gap group
	ip->ClearNodeSelection();

	Link<GapInfo>* curInfo = gapLink->data.refs.GetHead();

	// Only the first startgap and the first endgap will be used in the selection set
	// this way it won't screw things up when the user attempts to make changes to the
	// properties
	INode* nodeStartGap = NULL;
	INode* nodeEndGap = NULL;

	while(curInfo)
	{
		if (curInfo->data.type == STARTGAP)
			nodeStartGap = curInfo->data.node;

		if (curInfo->data.type == ENDGAP)
			nodeEndGap = curInfo->data.node;

		if (nodeStartGap && nodeEndGap)
			break;

		curInfo = curInfo->next;
	}

	// Select the first start and end gap from the gap group first so that
	// NodeA and nodeB selector resolve properly and updates will work
	ip->SelectNode(nodeStartGap,0);
	ip->SelectNode(nodeEndGap,0);

	// Select the rest of the set
	curInfo = gapLink->data.refs.GetHead();

	while(curInfo)
	{
		if (curInfo->data.node != nodeStartGap &&
			curInfo->data.node != nodeEndGap)
			ip->SelectNode(curInfo->data.node);

		curInfo = curInfo->next;
	}

	if (IsDlgButtonChecked(hwnd,IDC_INTGAPS)==BST_CHECKED)
		IEditGapID->SetText(gapLink->data.gapID);
	else
	{
		// Strip the last underscore and following data from the gapID
		/*
		CStr gapName = gapLink->data.gapID;
		char* term = strrchr(gapName,'_');
		
		if (term)
			*term = 0;
		*/
		
		//IEditGapID->SetText(gapName);
		IEditGapID->SetText(gapLink->data.gapID);
	}

	ip->ForceCompleteRedraw();
}

void GapToolDlg::SelGapText()
{ FUNC_ENTER("GapToolDlg::SelGapText"); 
	int selindex = SendDlgItemMessage(hwnd,IDC_GAPTEXT,LB_GETCURSEL,0,0);
	Link<GapDesc>* gapdesc = (Link<GapDesc>*)SendDlgItemMessage(hwnd,IDC_GAPTEXT,LB_GETITEMDATA,(WPARAM)selindex,0);

	int index = SendDlgItemMessage(hwnd,IDC_GAPGROUPS,LB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)gapdesc->data.gapID);
	SendDlgItemMessage(hwnd,IDC_GAPGROUPS,LB_SETCURSEL,(WPARAM)index,0);

	SelGapGroup();
}

void GapToolDlg::SelGapStart()
{ FUNC_ENTER("GapToolDlg::SelGapStart"); 
	//SendDlgItemMessage(hwnd,IDC_ENDGAPS,LB_SETCURSEL,(WPARAM)-1,0);

	int index = SendDlgItemMessage(hwnd,IDC_STARTGAPS,LB_GETCURSEL,0,0);
	Link<GapInfo>* gapInfo = (Link<GapInfo>*)SendDlgItemMessage(hwnd,IDC_STARTGAPS,LB_GETITEMDATA,(WPARAM)index,0);

	if (index == -1)
		return;

	AssignProps(plistStartGap,&gapInfo->data);

	nsNodeA->SetNode(gapInfo->data.node);

	ip->SelectNode(gapInfo->data.node,1);
	ip->ForceCompleteRedraw();
}

void GapToolDlg::SelGapEnd()
{ FUNC_ENTER("GapToolDlg::SelGapEnd"); 
	//SendDlgItemMessage(hwnd,IDC_STARTGAPS,LB_SETCURSEL,(WPARAM)-1,0);

	int index = SendDlgItemMessage(hwnd,IDC_ENDGAPS,LB_GETCURSEL,0,0);
	Link<GapInfo>* gapInfo = (Link<GapInfo>*)SendDlgItemMessage(hwnd,IDC_ENDGAPS,LB_GETITEMDATA,(WPARAM)index,0);

	if (index == -1)
		return;

	AssignProps(plistEndGap,&gapInfo->data);

	nsNodeB->SetNode(gapInfo->data.node);

	// Must lock the selection change so it doesn't clear out NodeA
	bLockSelChange = true;
	ip->SelectNode(gapInfo->data.node,1);
	bLockSelChange = false;

	ip->ForceCompleteRedraw();
}

void GapToolDlg::GapIDChanged()
{ FUNC_ENTER("GapToolDlg::GapIDChanged"); 
	char buf[256];
	IEditGapID->GetText(buf,255);

	// Chop off after _
	int len = strlen(buf);

	for(int i = 0; i < len; i++)
		if (buf[i] == '_')
		{
			buf[i] = 0;
			break;
		}

	if (plistStartGap)
		plistStartGap->SetValue("GapID",buf);

	if (plistEndGap)
		plistEndGap->SetValue("GapID",buf);
}

void GapToolDlg::SelSetChanged()
{ FUNC_ENTER("GapToolDlg::SelSetChanged"); 
	if (bLockSelChange)
		return;

	bLockSelChange = true;

	int nNodes = ip->GetSelNodeCount();

	if (nNodes>0)
		nsNodeA->SetValue(ip->GetSelNode(0)->GetName());

	if (nNodes>1)
		nsNodeB->SetValue(ip->GetSelNode(1)->GetName());

//	ReadGapProps();
//	BuildGapList();

	bLockSelChange = false;
}

CStr GapToolDlg::DumpListProps(CStr strGapObj, PropList* plist)
{ FUNC_ENTER("GapToolDlg::DumpListProps"); 
	CStr vals;
	int nProps = plist->NumProps();

	for(int i=0;i<nProps;i++)
	{
		CStr name, value;

		name = plist->GetName(i);
		plist->GetValue(i,value);

		int len = value.Length();

		// Make sure the text field has quotes around it
		if (name == CStr("text"))
		{
			if (len > 1)
			{
				if (value[0] != '\"')
					value = CStr("\"") + value;

				if (value[len - 1] != '\"')
					value += CStr("\"");
			}
		}

		if (name == CStr("GapID"))
			value += CStr("_") + strGapObj;
			//value += CStr("_") + nsNodeA->GetValue();

		value = Shave(value);

		if (value.Length() > 0 && value != CStr("<None>"))
		{
			if (value == CStr("TRUE"))
			{
				vals += CStr(" ") + name;
			}
			else if (value == CStr("FALSE"))
			{
				continue;
			}
			else
				vals += CStr(" ") + name + CStr("=") + value;
		}
	}

	return vals;
}

bool LineContainsSGap(CStr* v1,CStr* v2,void* pData)
{ FUNC_ENTER("LineContainsSGap"); 
	CStr lcase = *v1;
	CStr* strName = (CStr*)pData;
	lcase.toLower();

	CStr gprop1 = GetPropVal(*v1,"GapID");
	CStr gprop2 = *v2  + CStr("_") + *strName;

	if (gprop1 ==  gprop2 &&
		IsInstr(lcase,"startgap"))
		return true;

	return false;
}

bool LineContainsEGap(CStr* v1,CStr* v2,void* pData)
{ FUNC_ENTER("LineContainsEGap"); 
	CStr lcase = *v1;
	CStr* strName = (CStr*)pData;
	lcase.toLower();

	CStr gprop1 = GetPropVal(*v1,"GapID");
	CStr gprop2 = *v2  + CStr("_") + *strName;

	if (gprop1 ==  gprop2 &&
		IsInstr(lcase,"endgap"))
		return true;

	return false;
}

void GapToolDlg::Apply(bool bUpdateOnly)
{ FUNC_ENTER("GapToolDlg::Apply"); 
	INode* nodeA = nsNodeA->GetNode();
	INode* nodeB = nsNodeB->GetNode();

	if (!(nodeA && nodeB))
	{
		MessageBox(ip->GetMAXHWnd(),"You must have two nodes selected.","Gap Tool",MB_ICONWARNING|MB_OK);
		return;
	}

	UpdateGapFile(nodeA,nodeB,nodeA->GetName(),bUpdateOnly);

	if (IsDlgButtonChecked(hwnd,IDC_TWOWAY) == BST_CHECKED)
		UpdateGapFile(nodeB,nodeA,nodeB->GetName(),bUpdateOnly);

	ReadGapProps();
	BuildGapList();
	BuildGapTextList();
}

void GapToolDlg::UpdateGapFile(INode* nodeA,INode* nodeB,CStr strCtrlName,bool bUpdateOnly)
{ FUNC_ENTER("GapToolDlg::UpdateGapFile"); 
	// Assign the alternate script property for gaps to call the script we'll generate
	AssignAltBuf(nodeA,"Gaps",CStr("GapScript_")+nodeA->GetName());
	AssignAltBuf(nodeB,"Gaps",CStr("GapScript_")+nodeB->GetName());

	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	CStr strPath = CStr(getenv(APP_ENV)) + CStr(QN_PATH);
	//CStr name = strPath + seo.m_SceneName + CStr("\\") + seo.m_SceneName + CStr("_Scripts.q");
	CStr name = strPath + seo.m_SceneName + CStr("\\") + seo.m_SceneName + CStr("_Gaps.q");

	LinkList<CStr> listLines;
	BuildList(name,&listLines);

	CStr search = CStr("script GapScript_") + nodeA->GetName();
	Link<CStr>* link = listLines.Find(&search);

	// Build Startgap/Endgap lines
	CStr SGapLine = CStr("StartGap") + DumpListProps(strCtrlName,plistStartGap);
	CStr EGapLine = CStr("EndGap") + DumpListProps(strCtrlName,plistEndGap);
	CStr strEnd   = CStr("endscript\n");

	Link<CStr>* tlink = link;

	if (!link)
	{
		if (bUpdateOnly)
			return;

		tlink = listLines.AddAfterLink(tlink,&search);	
		tlink = listLines.AddAfterLink(tlink,&SGapLine);
		tlink = listLines.AddAfterLink(tlink,&strEnd);
	}
	else
	{
		search = CStr("endscript");
		Link<CStr>* linkEnd = listLines.FindFrom(&search,link);

		// See if the GapID we're adding already exists
		CStr strGap;
		plistStartGap->GetValue("GapID",strGap);

		Link<CStr>* gline;
//		CStr name = nodeA->GetName();

		do
		{
			gline = listLines.FindRange(LineContainsSGap,&strGap,link,linkEnd,&strCtrlName);
			
			if (gline)
			{
				gline->data = SGapLine;
				break;
			}
			else
				if (bUpdateOnly)
					return;
				else
					tlink = listLines.AddAfterLink(link,&SGapLine);
			
		} while(gline);
	}

	// Now process the end gap
	search = CStr("script GapScript_") + nodeB->GetName();
	link = listLines.Find(&search);

	if (!link)
	{
		tlink = NULL;
		tlink = listLines.AddAfterLink(tlink,&search);	
		tlink = listLines.AddAfterLink(tlink,&EGapLine);
		tlink = listLines.AddAfterLink(tlink,&strEnd);
	}
	else
	{
		search = CStr("endscript");
		Link<CStr>* linkEnd = listLines.FindFrom(&search,link);

		// See if the GapID we're adding already exists
		CStr strGap;
		plistEndGap->GetValue("GapID",strGap);

		Link<CStr>* gline;
		//CStr name = nodeB->GetName();
		//CStr name = nodeA->GetName();

		do
		{
			gline = listLines.FindRange(LineContainsEGap,&strGap,link,linkEnd,&strCtrlName);
			
			if (gline)
			{
				gline->data = EGapLine;
				break;
			}
			else
				tlink = listLines.AddAfterLink(link,&EGapLine);
			
		} while(gline);
	}

	// Data has been updated, dump out the file
	CStr buf = BuildBuf(&listLines);

	FILE* fp = fopen(name,"w");

	if (!fp)
	{
		char sbuf[256];
		sprintf(sbuf,"Failed to save '%s'.",(char*)name);
		MessageBox(ip->GetMAXHWnd(),sbuf,"GapTool Export",MB_ICONWARNING|MB_OK);
		return;
	}

	fputs(buf,fp);
	fclose(fp);
}

void GapToolDlg::AssignProps(PropList* plist, GapInfo* gapInfo)
{ FUNC_ENTER("GapToolDlg::AssignProps"); 
	Link<ConfigProp>* curprop = gapInfo->props.GetHead();

	plist->ClearValues();

	while(curprop)
	{
		CStr name = curprop->data.name;
		CStr value = curprop->data.value;
		name.toLower();

		if (IsDlgButtonChecked(hwnd,IDC_INTGAPS)!=BST_CHECKED)
		{
			if (name == CStr("gapid"))
			{
				char* term = strrchr(value,'_');
			
				if (term)
					*term = 0;
			}
		}

		plist->SetValue(curprop->data.name,value);
		curprop = curprop->next;
	}
}

void GapToolDlg::ReadGapProps()
{ FUNC_ENTER("GapToolDlg::ReadGapProps"); 
	INode* nodeA = nsNodeA->GetNode();

	if (!nodeA)
		return;

	// Scan the first line of the gap buffer for the script we'll scan to acquire properties from
	CStr buf = GetAltBuf(nodeA,"Gaps");
	
	TermLine(buf);

	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	CStr strPath = CStr(getenv(APP_ENV)) + CStr(QN_PATH);
	//CStr name = strPath + seo.m_SceneName + CStr("\\") + seo.m_SceneName + CStr("_Scripts.q");
	CStr name = strPath + seo.m_SceneName + CStr("\\") + seo.m_SceneName + CStr("_Gaps.q");

	LinkList<CStr> listLines;
	BuildList(name,&listLines);

	// Each script within the file should be named GapScript_<nodename>
	// We'll scan through each line looking for script GapScript_
	// build a sublist for the script and send it to ParseGapData to build up the database

	gapList.Clear();

	// Verify Gap script exists
	/*
	CStr search = CStr("script ") + buf;
	Link<CStr>* link = listLines.Find(&search);

	if (!link)
	{
		// The gap information no longer exists ??  It may have been removed from the file
		// Remove our reference to the missing data
		AssignAltBuf(nodeA,"Gaps","");
		return;
	}
	*/

	Link<CStr>* curlink = listLines.GetHead();
	INode* node;

	// Search for gap scripts
	while(curlink)
	{
		if (IsInstr(curlink->data,"script GapScript_"))
		{
			// Extract the name of the node the script affects
			CStr nodeName = curlink->data.Substr(17,curlink->data.Length());
			node = ip->GetINodeByName(nodeName);

			if (node)
			{
				// The node was found, build a buffer for this script
				CStr search = "endscript";
				Link<CStr>* endLink = listLines.FindFrom(&search,curlink);

				CStr buf = BuildPartialList(&listLines,curlink->next,endLink);

				// Build gap DB
				ParseGapData(node,&buf);

				curlink = endLink;
			}
		}

		curlink = curlink->next;
	}

	CheckDlgButton(hwnd,IDC_INTGAPS,BST_UNCHECKED);
}

void GapToolDlg::GapListChanged()
{ FUNC_ENTER("GapToolDlg::GapListChanged"); 
	if (IsDlgButtonChecked(hwnd,IDC_INTGAPS)==BST_CHECKED)
	{
		// Process internal gaps
		EnableWindow(GetDlgItem(hwnd,IDC_REMOVE),FALSE);
		EnableWindow(GetDlgItem(hwnd,IDC_APPLY),FALSE);
		FindGapsStatus();		
	}
	else
	{
		// Process external gaps
		EnableWindow(GetDlgItem(hwnd,IDC_REMOVE),TRUE);
		EnableWindow(GetDlgItem(hwnd,IDC_APPLY),TRUE);
		ReadGapProps();
	}

	BuildGapList();
	BuildGapTextList();
}

// This function will go through all the scripts in the gap file and identify the ones
// that are empty (contain no data) and then remove the reference to it within the
// related node's buffer
void GapToolDlg::CleanEmptyScripts()
{ FUNC_ENTER("GapToolDlg::CleanEmptyScripts"); 
	// Assign the alternate script property for gaps to call the script we'll generate
	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	CStr strPath = CStr(getenv(APP_ENV)) + CStr(QN_PATH);
	//CStr name = strPath + seo.m_SceneName + CStr("\\") + seo.m_SceneName + CStr("_Scripts.q");
	CStr name = strPath + seo.m_SceneName + CStr("\\") + seo.m_SceneName + CStr("_Gaps.q");

	LinkList<CStr> listLines;
	BuildList(name,&listLines);
	
	// Scan through the lines and remove any references to the given gapID
	Link<CStr>* curline = listLines.GetHead();
	Link<CStr>* nextline;

	while(curline)
	{
		nextline = curline->next;

		// Cleanup empty lines
		if (curline->data == CStr("") &&
			nextline &&
			nextline->data == CStr(""))
		{
			listLines.Remove(curline);
			curline = nextline;
			continue;
		}

		// Check if the script contains no data and should be removed
		if (Instr(curline->data,"script") == 0 &&
			nextline &&
			Instr(nextline->data,"endscript") == 0)
		{
			// Determine node name that the script is attached to
			int pos = Instr(curline->data,"_") + 1;
			CStr strNode = GetRemainLineExact(curline->data,&pos);
			INode* node = ip->GetINodeByName(strNode);

			if (node)
				RemoveAltBuf(node,"Gaps");

			Link<CStr>* nextnextline = nextline->next;
			listLines.Remove(curline);
			listLines.Remove(nextline);
			nextline = nextnextline;
		}

		curline = nextline;
	}

	CStr buf = BuildBuf(&listLines);
	FILE* fp = fopen(name,"w");

	if (!fp)
		return;

	fputs(buf,fp);
	fclose(fp);
}

void GapToolDlg::RemoveGap(CStr gapID)
{ FUNC_ENTER("GapToolDlg::RemoveGap"); 
	// Assign the alternate script property for gaps to call the script we'll generate
	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	CStr strPath = CStr(getenv(APP_ENV)) + CStr(QN_PATH);
	//CStr name = strPath + seo.m_SceneName + CStr("\\") + seo.m_SceneName + CStr("_Scripts.q");
	CStr name = strPath + seo.m_SceneName + CStr("\\") + seo.m_SceneName + CStr("_Gaps.q");

	LinkList<CStr> listLines;
	BuildList(name,&listLines);
	
	// Scan through the lines and remove any references to the given gapID
	Link<CStr>* curline = listLines.GetHead();

	while(curline)
	{
		Link<CStr>* nextline = curline->next;

		CStr GapID = GetPropVal(curline->data,"gapid");

		if (GapID == gapID)
			listLines.Remove(curline);

		curline = nextline;
	}

	// Data has been updated, dump out the file
	CStr buf = BuildBuf(&listLines);

	FILE* fp = fopen(name,"w");

	if (!fp)
	{
		char sbuf[256];
		sprintf(sbuf,"Failed to save '%s'.",(char*)name);
		MessageBox(ip->GetMAXHWnd(),sbuf,"GapTool Export",MB_ICONWARNING|MB_OK);
		return;
	}

	fputs(buf,fp);
	fclose(fp);

	CleanEmptyScripts();
}
