#include "FuncEnter.h"

/*
	ThumbnailList.cpp
	UI Control for displaying parameters in a list with a
	thumbnail image to the right of each entry
*/

#include "ThumbnailList.h"
#include "../ImageData.h"
#include "../misc/Util.h"
#include <winbase.h>

#define TNL_ITEM_HEIGHT 70

struct TBEntry
{
	ImageData img;
	char      imgFilename[256];	// Filename of the image being displayed for this entry
	HWND      hwndCombo;
	void*     extdata;			// The data added by a user of the control
	
	HANDLE    hLoadThread;		// Handle for the thread that's loading the image
	bool      bLoaded;			// True if the image has been loaded into memory (img is valid)

	HWND      hwndList;			// The list that contains the entry that the thread is currently processing
	int       index;			// Index of the entry being loaded in the list
	FILE*     fp;				// PNG filesystem pointer (if terminated should be closed by terminator)
	HANDLE    hEventDone;		// Signals when loading is complete

	TBEntry()
	{
		imgFilename[0] = 0;

		hwndCombo = NULL;
		extdata = NULL;

		hLoadThread = NULL;
		bLoaded = false;

		fp = NULL;

		hEventDone = CreateEvent(NULL, FALSE, FALSE, NULL);
	}

	~TBEntry()
	{ FUNC_ENTER("TBEntry::~TBEntry"); 
		if (hwndCombo)
			DestroyWindow(hwndCombo);

		if (hEventDone)
			CloseHandle(hEventDone);
	}
};

int FindTBIndex(HWND hwnd, TBEntry* scanEntry)
{
	int cnt = SendMessage(hwnd, LB_GETCOUNT, 0, 0);

	for(int i = 0; i < cnt; i++)
	{
		TBEntry* tbentry = (TBEntry*)SendMessage(hwnd, LB_GETITEMDATA, (WPARAM)i, 0);

		if (tbentry == scanEntry)
			return i;
	}

	return -1;
}

ThumbNailList::ThumbNailList(HINSTANCE hInstance) :
	UIControl(hInstance, CLASS)
{ FUNC_ENTER("ThumbNailList::ThumbNailList"); 

	InitializeCriticalSection(&cs);
}

ThumbNailList::ThumbNailList(HINSTANCE hInstance, HWND hwnd) :
	UIControl(hInstance, CLASS)
{ FUNC_ENTER("ThumbNailList::ThumbNailList"); 

	DeleteCriticalSection(&cs);
}

ThumbNailList::~ThumbNailList()
{ FUNC_ENTER("ThumbNailList::~ThumbNailList"); 

	// Release all item memory
	Clear();
}

// Thumbnail lists load and convert images in the background while other processing
// is occurring so as not to halt processing as loading all the images in a list can take awhile. 
// Each image loads in a seperate thread
DWORD WINAPI ThumbNailList::LoadImageThread(void* pData)
{
	TBEntry* pEntry = (TBEntry*)pData;
	ImageData img;
	
	if (!img.ReadPNG((char*)pEntry->imgFilename, NULL))
	{
		pEntry->bLoaded = true;
		delete pData;
		SetEvent(pEntry->hEventDone);
		return 0;
	}

	img.Convert24(&pEntry->img);
	SwapRGB(pEntry->img.buf, pEntry->img.imageSize);
	Flip(pEntry->img.buf, pEntry->img.width, pEntry->img.height);
	pEntry->bLoaded = true;

	// Redraw the control once the image finishes loading
	RECT rect;
	SendMessage(pEntry->hwndList, LB_GETITEMRECT, (WPARAM)pEntry->index, (LPARAM)&rect);
	InvalidateRect(pEntry->hwndList, &rect, FALSE);
	
	SetEvent(pEntry->hEventDone);
	return 1;
}

LRESULT ThumbNailList::WndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ FUNC_ENTER("ThumbNailList::WndProc"); 
	ThumbNailList* pthis = (ThumbNailList*)GetWindowLong(hwnd,GWL_USERDATA);

	switch(msg)
	{
	case WM_MEASUREITEM:
		{
			RECT wndRect;
			GetClientRect(hwnd,&wndRect);

			// Fill out the MEASUREITEMSTRUCT
			MEASUREITEMSTRUCT* pmis = (MEASUREITEMSTRUCT*)lParam;

			//pmis->CtlType    = ODT_LISTBOX;
			//pmis->CtlID      = (UINT)wParam;
			//pmis->itemID     = 0;				// Unused this is a fixed sized listbox
			pmis->itemWidth  = wndRect.right - wndRect.left;
			pmis->itemHeight = TNL_ITEM_HEIGHT;
		}
		return TRUE;

	case WM_CREATE:
		{
			SetWindowLong(hwnd, GWL_USERDATA, (LONG)((CREATESTRUCT*)lParam)->lpCreateParams);
			pthis = (ThumbNailList*)((CREATESTRUCT*)lParam)->lpCreateParams;

			if (pthis)
				pthis->Attach(hwnd);
		}
		return TRUE;

	case WM_DESTROY:
		{

		}
		return TRUE;

	// Handle Notifications
	case WM_COMMAND:
		switch(HIWORD(wParam))
		{
		case LBN_DBLCLK:
		case LBN_ERRSPACE:
		case LBN_KILLFOCUS:
		case LBN_SELCANCEL:
		case LBN_SELCHANGE:
		case LBN_SETFOCUS:
			{
				HWND hwndParent = GetParent(hwnd);
				int  ID = (int)GetWindowLong(hwnd,GWL_ID);

				SendMessage(hwndParent,msg,(WPARAM)MAKEWPARAM(ID,HIWORD(wParam)),(LPARAM)hwnd);
				return TRUE;
			}
		}
		break;

	case LB_SELITEMRANGE:
	case LB_ITEMFROMPOINT:
	case LB_SELECTSTRING:
	case LB_SETCURSEL:
	case LB_SETSEL:
	case LB_GETSEL:
	case LB_GETCOUNT:
	case LB_GETCURSEL:
	case LB_GETSELCOUNT:
	case LB_GETSELITEMS:
	case LB_GETTEXT:
	case LB_GETTEXTLEN:
		{
			// Forward message
			return SendMessage(pthis->hwndList,msg,wParam,lParam);
		}

	case LB_RESETCONTENT:
		{
			int count = SendMessage(pthis->hwndList,LB_GETCOUNT,0,0);

			for(int i=0;i<count;i++)
			{
				TBEntry* tbentry = (TBEntry*)SendMessage(pthis->hwndList,LB_GETITEMDATA,(WPARAM)i,0);
				
				if (tbentry)
					delete tbentry;
			}

			// Forward message
			return SendMessage(pthis->hwndList,msg,wParam,lParam);
		}

	case LB_DELETESTRING:
		{
			// Delete our extended color data first
			TBEntry* tbentry = (TBEntry*)SendMessage(pthis->hwndList,LB_GETITEMDATA,wParam,0);
			if (tbentry)
				delete tbentry;

			// Forward message
			return SendMessage(pthis->hwndList,msg,wParam,lParam);
		}

	case LB_GETITEMDATA:
		{
			TBEntry* tbentry = (TBEntry*)SendMessage(pthis->hwndList,LB_GETITEMDATA,wParam,0);
			if (tbentry && (LRESULT)tbentry != LB_ERR)
				return (LRESULT)tbentry->extdata;

			return LB_ERR;
		}

	case LB_SETITEMDATA:
		{
			TBEntry* tbentry = (TBEntry*)SendMessage(pthis->hwndList,LB_GETITEMDATA,wParam,0);
			if (tbentry)
			{
				tbentry->extdata = (void*)lParam;
				return 0;
			}

			return LB_ERR;
		}
		return TRUE;

	case LB_ADDSTRING:
		{
			if (pthis)
			{
				int index = SendMessage(pthis->hwndList,msg,wParam,lParam);

				if (index != LB_ERR)
				{
					// Add our color information
					// This gets cleaned up in RESETCONTENT, DELETESTRING, and DESTROY messages
					TBEntry* tbentry = new TBEntry;
					strcpy(tbentry->imgFilename, (char*)lParam);

					tbentry->hwndList = hwndList;
					tbentry->index    = index;

					// We'll start another thread at this time that will do the actual loading of
					// the image.  (Note may also need to cap memory usage with this)
					tbentry->hLoadThread = CreateThread(NULL, 0, LoadImageThread, tbentry, 0, NULL);

					SendMessage(pthis->hwndList,LB_SETITEMDATA,(WPARAM)index,(LPARAM)tbentry);
				}

				return index;
			}
		
			return LB_ERR;
		}

	case WM_DRAWITEM:
		{
			DRAWITEMSTRUCT* drawInfo = (DRAWITEMSTRUCT*)lParam;
			HBRUSH bkgBrush;

			// Don't draw if the space is empty
			if (drawInfo->itemID == -1)
				return TRUE;

			// Get the text we're supposed to render
			int txtLen = SendMessage(drawInfo->hwndItem,LB_GETTEXTLEN,(WPARAM)drawInfo->itemID,0);
			char* itemText = new char[txtLen+1];
			SendMessage(drawInfo->hwndItem,LB_GETTEXT,(WPARAM)drawInfo->itemID,(LPARAM)itemText);

			HDC hdc = drawInfo->hDC;

			//if (drawInfo->itemData)
			//{
				TBEntry* tbentry = (TBEntry*)SendMessage(drawInfo->hwndItem,LB_GETITEMDATA,(WPARAM)drawInfo->itemID,0);
				//SetTextColor(hdc, tbentry->color);
			//}

			if (drawInfo->itemState & ODS_SELECTED)
			{
				bkgBrush = CreateSolidBrush(RGB(195,220,239));
				SetBkColor(hdc, RGB(195,220,239));

				// Attempt to create an embedded dropdown combo box for properties if the entry is selected
				if (!tbentry->hwndCombo)
					tbentry->hwndCombo = BuildDropdown(hwnd, drawInfo->rcItem, 0);
			}
			else
			{
				bkgBrush = CreateSolidBrush(RGB(255,255,255));
				SetBkColor(hdc, RGB(255,255,255));
			}

			FillRect(hdc,&drawInfo->rcItem,bkgBrush);

			RECT textRect;
			SetRect(&textRect, 
				    drawInfo->rcItem.left + 66,		// left
					drawInfo->rcItem.top + 3,       // top
					drawInfo->rcItem.right,			// right
					drawInfo->rcItem.bottom - 3);	// bottom

			HFONT hFont = CreateFont(14,0,0,0,0,0,0,0,0,0,0,0, VARIABLE_PITCH | FF_SWISS, "");
			SelectObject(hdc, hFont);

			DrawText(hdc,itemText,strlen(itemText),&textRect,DT_CENTER|DT_VCENTER|DT_SINGLELINE);

			// Image shouldn't be renderered until loading is complete
			if (tbentry->bLoaded)
			{
				// Draw the PNG bitmap into the item rect
				HDC        hdcBmp = CreateCompatibleDC(hdc);
				BITMAPINFO bmi;
				bmi.bmiHeader.biSize          = sizeof(BITMAPINFOHEADER);
				bmi.bmiHeader.biWidth         = tbentry->img.width;
				bmi.bmiHeader.biHeight        = tbentry->img.height;
				bmi.bmiHeader.biPlanes        = 1;
				bmi.bmiHeader.biBitCount      = tbentry->img.pixel_depth;
				bmi.bmiHeader.biCompression   = BI_RGB;
				bmi.bmiHeader.biSizeImage     = tbentry->img.imageSize;
				bmi.bmiHeader.biXPelsPerMeter = 0;
				bmi.bmiHeader.biYPelsPerMeter = 0;
				bmi.bmiHeader.biClrUsed       = 0;
				bmi.bmiHeader.biClrImportant  = 0;

				unsigned char* pBmpData;
				HBITMAP   hBmp = CreateDIBSection(hdcBmp, &bmi, DIB_RGB_COLORS, (void**)&pBmpData, NULL, 0);

				SelectObject(hdcBmp, hBmp);
				memcpy(pBmpData, tbentry->img.buf, tbentry->img.imageSize);

				StretchBlt(hdc,
						   drawInfo->rcItem.left + 3,		// X
						   drawInfo->rcItem.top + 3,		// Y
						   64,		// width
						   64,		// height
						   hdcBmp,
						   0,
						   0,
						   tbentry->img.width,
						   tbentry->img.height,
						   SRCCOPY);

				if (drawInfo->itemState & ODS_FOCUS)
					DrawFocusRect(hdc,&drawInfo->rcItem);

				// Create a solid black pen to draw a border around the texture
				POINT pt;

				HPEN  hPen  = CreatePen(PS_SOLID, 1, 0);			
				SelectObject(hdc, hPen);

				MoveToEx(hdc, drawInfo->rcItem.left + 3, drawInfo->rcItem.top + 3, &pt);
				LineTo(hdc, drawInfo->rcItem.left + 66, drawInfo->rcItem.top + 3);
				LineTo(hdc, drawInfo->rcItem.left + 66, drawInfo->rcItem.top + 66);
				LineTo(hdc, drawInfo->rcItem.left + 3, drawInfo->rcItem.top + 66);
				LineTo(hdc, drawInfo->rcItem.left + 3, drawInfo->rcItem.top + 3);

				if (itemText)
					delete itemText;

				DeleteObject(hPen);
				DeleteObject(hBmp);
				DeleteDC(hdcBmp);
			}

			DeleteObject(bkgBrush);
			DeleteObject(hFont);
		}
		return TRUE;
	}

	return DefWindowProc(hwnd,msg,wParam,lParam);
}

void ThumbNailList::BuildControlUI()
{ FUNC_ENTER("ThumbNailList::BuildControlUI"); 
	RECT wndRect;
	GetClientRect(hwnd,&wndRect);

	int width  = wndRect.right - wndRect.left;
	int height = wndRect.bottom - wndRect.top;

	hwndList = CreateWindow("listbox",
		                    "",
							WS_CHILD|WS_VISIBLE|WS_VSCROLL|LBS_OWNERDRAWFIXED|LBS_HASSTRINGS|LBS_MULTIPLESEL,
							0,
							0,
							width,
							height,
							hwnd,
							(HMENU)0,
							hInstance,
							NULL);


}

HWND ThumbNailList::BuildDropdown(HWND hwndParent, RECT rect, int entryID)
{ FUNC_ENTER("ThumbNailList::BuildDropdown"); 
	HWND hwndCombo = CreateWindow("combobox",
		                          "",
								  WS_CHILD|WS_VISIBLE|WS_VSCROLL,
								  rect.left + 75,
								  rect.top + 4,
								  rect.right - rect.left - 75,
								  20,
								  hwndParent,
								  NULL,
								  hInstance,
								  NULL);

	ShowWindow(hwndCombo, SW_SHOW);

	return hwndCombo;
}

int  ThumbNailList::AddItem(char* name, char* filename, void* pData)
{ FUNC_ENTER("ThumbNailList::AddItem");

	int index = SendMessage(hwndList,LB_ADDSTRING,0,(LPARAM)name);

	if (index != LB_ERR)
	{
		// Add our color information
		// This gets cleaned up in RESETCONTENT, DELETESTRING, and DESTROY messages
		TBEntry* tbentry = new TBEntry;

		strcpy(tbentry->imgFilename, filename);
		tbentry->extdata = pData;

		tbentry->hwndList = hwndList;
		tbentry->index    = index;

		tbentry->hLoadThread = CreateThread(NULL, 0, LoadImageThread, tbentry, 0, NULL);
		SendMessage(hwndList,LB_SETITEMDATA,(WPARAM)index,(LPARAM)tbentry);
	}

	return index;
}

void ThumbNailList::RemoveItem(int index)
{ FUNC_ENTER("ThumbNailList::RemoveItem");

	// Delete our extended data first
	TBEntry* tbentry = (TBEntry*)SendMessage(hwndList, LB_GETITEMDATA, (WPARAM)index, 0);
	if (tbentry && tbentry != (TBEntry*)LB_ERR)
	{
		// Doing slow poll for now, until these other modes work properly
		// Load must finish before removal!
		
		// Unfortunately, looks like we'll have to wait for completion as just terminating
		// the thread seems to get file I/O / libPNG into some confused state
		// Could probably restructure things and put some hooks into libPNG for aborting but,
		// probably more pain than its worth for now
		
		while(!tbentry->bLoaded){ }

		//WaitForSingleObject(tbentry->hEventDone, INFINITE);	// Not getting signaled state ???

		// Since this is all multithreaded now, we'll need to terminate the associated
		// thread if it hasn't finished loading before we free the data its acting on

		/*
		if (!tbentry->bLoaded)
		{
			//SuspendThread(tbentry->hLoadThread);
			
			char buf[256];
			sprintf(buf, "Shutting down thread 0x%x (0x%x)\n", tbentry->hLoadThread, 0);
			OutputDebugString(buf);

			// Force any still open PNG file to close
			if (tbentry->fp)
				fclose(tbentry->fp);

			TerminateThread(tbentry->hLoadThread, 0);
			CloseHandle(tbentry->hLoadThread);
		}
		*/

		delete tbentry;
	}

	// Do the actual delete from the real list
	SendMessage(hwndList, LB_DELETESTRING, (WPARAM)index, 0);
}

void ThumbNailList::Clear()
{
	int cnt = SendMessage(hwndList, LB_GETCOUNT, 0, 0);

	for(int i = 0; i < cnt; i++)
		RemoveItem(i);
}
