#include "FuncEnter.h"

/*
	ScreenFX.cpp
	Front-Facing Poly Effect Object
	The object still maintains its data in the PE's property format and also thus uses the script exporter
	in its current state.  This way property dependancies for these objects can still be managed with all the
	PEs features.  This code is responsible for setting up a single screen FX instance, but the individual instance
	may be modifiable thru ScreenFX or the property editor for individual (or multi prop modifications at a time accross objects)
	7-16-01
*/

#include "ScreenFX.h"
#include "Resource.h"
#include "path.h"
#include "PropEdit/ParseFuncs.h"
#include "PropEdit/ConfigData.h"
#include "../Export/SFP.h"

static HWND hwndLastUI;
static ScreenFXObjClassDesc screenFXObjDesc;
ClassDesc2* GetScreenFXObjDesc() { FUNC_ENTER("GetScreenFXObjDesc");  return &screenFXObjDesc; }

extern Interface* gInterface;

// pblocks
enum { screenfxobj_params, };

// params
enum 
{
		screenfxobj_width,
		screenfxobj_height,
};

// Define param block
static ParamBlockDesc2 param_blk_desc(screenfxobj_params,
									  _T("params"), 0, &screenFXObjDesc,
									  P_AUTO_CONSTRUCT + P_AUTO_UI, PBLOCK_REF,

									  // rollout
									  IDD_SCREENFXDLG, IDS_PARAMS, 0, 0, NULL,

									  // params
									  screenfxobj_width, _T("width"), TYPE_FLOAT, 0, IDS_SFX_WIDTH,
									  p_default,		10.0f,
									  p_range,			0.0f, FLT_MAX,
									  p_ui,             TYPE_SPINNER, EDITTYPE_UNIVERSE,
										IDC_SFX_EDITWIDTH, IDC_SFX_SPINWIDTH, SPIN_AUTOSCALE,
									  end,

									  screenfxobj_height, _T("height"), TYPE_FLOAT, 0, IDS_SFX_HEIGHT,
									  p_default,        10.0f,
									  p_range,          0.0f, FLT_MAX,
									  p_ui,             TYPE_SPINNER, EDITTYPE_UNIVERSE,
										IDC_SFX_EDITHEIGHT, IDC_SFX_SPINHEIGHT, SPIN_AUTOSCALE,
									  end,

									  end);

class ScreenFXObjDlgProc: public ParamMap2UserDlgProc
{
public:
	ScreenFXObj *obj;

	ScreenFXObjDlgProc(ScreenFXObj *obj)
	{
		this->obj = obj;
	}

	void ReadEffectModes(HWND hwnd);

	BOOL DlgProc(TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);

	void DeleteThis() { FUNC_ENTER("ScreenFXObjDlgProc::DeleteThis");  delete this; }
};

void ScreenFXObjDlgProc::ReadEffectModes(HWND hwnd)
{ FUNC_ENTER("ScreenFXObjDlgProc::ReadEffectModes"); 
	FILE* fp;

	char* path = getenv(APP_ENV);

	if (!path)
		return;

	CStr strPath = CStr(path) + SCRIPT_PATH + CStr("effects.lst");

	fp = fopen(strPath,"r");

	if (!fp)
		return;

	SendDlgItemMessage(hwnd,IDC_EFFECTTYPE,CB_RESETCONTENT,0,0);

	char lineBuf[256];
	CStr strLine;

	fgets(lineBuf,255,fp);

	do 
	{
		strLine = Shave(lineBuf);
		
		if (strLine.Length()>0)
			SendDlgItemMessage(hwnd,IDC_EFFECTTYPE,CB_ADDSTRING,(WPARAM)0,(LPARAM)(char*)strLine);

		fgets(lineBuf,255,fp);

	} while(!feof(fp));

	fclose(fp);

	// Check the property buffer of the object, and assign the type selection if appropriate
	INode* updateNode = gInterface->GetINodeByHandle(obj->nodeHandle);
	
	if (updateNode)
	{
		CStr value;
		if (GetSFPEffect(updateNode,value))
		{
			int index = SendDlgItemMessage(hwnd,IDC_EFFECTTYPE,CB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)value);
			if (index != CB_ERR)
			{
				SendDlgItemMessage(hwnd,IDC_EFFECTTYPE,CB_SETCURSEL,(WPARAM)index,0);
			}
		}
	}
}

void ApplyEffectType(HWND hwnd,ULONG handle)
{ FUNC_ENTER("ApplyEffectType"); 
	char buf[256];
	int index = SendDlgItemMessage(hwnd,IDC_EFFECTTYPE,CB_GETCURSEL,0,0);
	SendDlgItemMessage(hwnd,IDC_EFFECTTYPE,CB_GETLBTEXT,(WPARAM)index,(LPARAM)buf);

	INode* updateNode = gInterface->GetINodeByHandle(handle);

	if (updateNode)
	{
		if (strcmp(buf,"None")==0)
			RemovePropBufProp(updateNode,"SFP_Effect");
		else
		{
			CStr propBuffer;
			updateNode->SetUserPropString("SFP_Effect",buf);
			updateNode->GetUserPropBuffer(propBuffer);
			
			if (!IsInstr(propBuffer,"// @nextparm | SFP_Effect"))
			{
				CStr eol="\r\n";

				propBuffer += CStr("// @nextparm | SFP_Effect | list | file | effects.lst | done") + eol;
				propBuffer += CStr("// CMD:FORCEOMIT") + eol;
				updateNode->SetUserPropBuffer(propBuffer);
			}
		}
	}
}

BOOL ScreenFXObjDlgProc::DlgProc(TimeValue t, IParamMap2 *map, HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ FUNC_ENTER("ScreenFXObjDlgProc::DlgProc"); 
	switch(msg)
	{
	case WM_INITDIALOG:
		hwndLastUI = hwnd;
		ReadEffectModes(hwnd);
		return TRUE;

	case WM_COMMAND:
		switch(HIWORD(wParam))
		{
		case CBN_SELCHANGE:
			switch(LOWORD(wParam))
			{
			case IDC_EFFECTTYPE:
				{
					ApplyEffectType(hwnd,obj->nodeHandle);
					return TRUE;
				}
			}
			break;
		}
	}

	return FALSE;
}

class ScreenFXObjMouseCallback: public CreateMouseCallBack
{
protected:
	IPoint2 sp0;
	Point3  p0;
	ScreenFXObj *obj;

public:
	int proc(ViewExp *vpt,
		     int     msg,
			 int     point,
			 int     flags,
			 IPoint2 m,
			 Matrix3 &mat);

	void SetObject(ScreenFXObj *obj) { FUNC_ENTER("ScreenFXObjMouseCallback::SetObject");  this->obj = obj; }
};
ScreenFXObjMouseCallback mouseCallback;

int ScreenFXObjMouseCallback::proc(ViewExp *vpt,
								   int     msg,
								   int     point,
								   int     flags,
								   IPoint2 m,
								   Matrix3 &mat)
{ FUNC_ENTER("ScreenFXObjMouseCallback::proc"); 
	// TODO: Update for width and height
   Point3 p1;

   switch(msg)
   {
   case MOUSE_POINT:
   case MOUSE_MOVE:
      switch(point)
      {
      case 0:
         sp0 = m;
         p0 = vpt->SnapPoint(m, m, NULL, SNAP_IN_PLANE);
         mat.SetTrans(p0);
         break;
      case 1:
         mat.IdentityMatrix();
         p1 = vpt->SnapPoint(m, m, NULL, SNAP_IN_PLANE);
         mat.SetTrans(p0);
         float radius = Length(p1-p0); // calc the drag size

         // set object size to drag size
         obj->pblock->SetValue(
            screenfxobj_width,
            obj->ip->GetTime(),
            radius);

        obj->pblock->SetValue(
            screenfxobj_height,
            obj->ip->GetTime(),
            radius);

         param_blk_desc.InvalidateUI();

         if(flags & MOUSE_CTRL)
         {
            float ang = (float)atan2(p1.y-p0.y, p1.x-p0.x);
            mat.PreRotateZ(obj->ip->SnapAngle(ang));
         }

         if(msg == MOUSE_POINT)
		 {
			 if (Length(m-sp0) < 1)
				 return CREATE_ABORT;

			ApplyEffectType(hwndLastUI,obj->nodeHandle);

            return CREATE_STOP;
		 }
         break;
      }
      break;
   case MOUSE_ABORT:
      return CREATE_ABORT;
   }
   return CREATE_CONTINUE;
}

ScreenFXObj::ScreenFXObj()
{ FUNC_ENTER("ScreenFXObj::ScreenFXObj"); 
	ip     = NULL;
	pblock = NULL;

	screenFXObjDesc.MakeAutoParamBlocks(this);
}

ScreenFXObj::~ScreenFXObj()
{ FUNC_ENTER("ScreenFXObj::~ScreenFXObj"); 
	DeleteAllRefsFromMe();
}

void ScreenFXObj::RefAdded(RefMakerHandle rm)
{ FUNC_ENTER("ScreenFXObj::RefAdded"); 
	if (rm->ClassID()==Class_ID(BASENODE_CLASS_ID,0))
	{
		INode* node=(INode*)rm;

		Object* obj=node->EvalWorldState(0).obj;
		
		if (obj && obj->ClassID() == vSCREENFXOBJ_CLASS_ID)
		{
			nodeHandle = node->GetHandle();
		}
	}
}

CreateMouseCallBack* ScreenFXObj::GetCreateMouseCallBack()
{ FUNC_ENTER("ScreenFXObj::GetCreateMouseCallBack"); 
	mouseCallback.SetObject(this);
	return &mouseCallback;
}

void ScreenFXObj::BeginEditParams(IObjParam  *ip,
								  ULONG      flags,
								  Animatable *prev)
{ FUNC_ENTER("ScreenFXObj::BeginEditParams"); 
	this->ip = ip;
	screenFXObjDesc.BeginEditParams(ip, this, flags, prev);

	param_blk_desc.SetUserDlgProc(new ScreenFXObjDlgProc(this));
}

void ScreenFXObj::EndEditParams(IObjParam  *ip,
								ULONG      flags,
								Animatable *next)
{ FUNC_ENTER("ScreenFXObj::EndEditParams"); 
	screenFXObjDesc.EndEditParams(ip, this, flags, next);
	this->ip = NULL;
}

RefTargetHandle ScreenFXObj::Clone(RemapDir &remap)
{ FUNC_ENTER("ScreenFXObj::Clone"); 
	ScreenFXObj *obj = new ScreenFXObj();
	obj->ReplaceReference(0, pblock->Clone(remap));
	return obj;
}

void ScreenFXObj::BuildMesh(TimeValue t)
{ FUNC_ENTER("ScreenFXObj::BuildMesh"); 
	float width  = pblock->GetFloat(screenfxobj_width, t);
	float height = pblock->GetFloat(screenfxobj_height, t);

	mesh.setNumVerts(4);
	mesh.setNumFaces(4);
	mesh.setVert(0, Point3(-0.5 * width, -0.5 * height, 0.0));
	mesh.setVert(1, Point3(0.5 * width, -0.5 * height, 0.0));
	mesh.setVert(2, Point3(-0.5 * width, 0.5 * height, 0.0));
	mesh.setVert(3, Point3(0.5 * width, 0.5 * height, 0.0));

	mesh.faces[0].setVerts(0,1,2);
	mesh.faces[0].setEdgeVisFlags(1,0,1);
	mesh.faces[0].setSmGroup(0);
	mesh.faces[1].setVerts(3,2,1);
	mesh.faces[1].setEdgeVisFlags(1,0,1);
	mesh.faces[1].setSmGroup(0);

	// Do it in reverse too so its two sided
	mesh.faces[2].setVerts(2,1,0);
	mesh.faces[2].setEdgeVisFlags(1,0,1);
	mesh.faces[2].setSmGroup(0);
	mesh.faces[3].setVerts(1,2,3);
	mesh.faces[3].setEdgeVisFlags(1,0,1);
	mesh.faces[3].setSmGroup(0);

	mesh.InvalidateGeomCache();
}

void ScreenFXObj::FreeCaches()
{ FUNC_ENTER("ScreenFXObj::FreeCaches"); 
	mesh.FreeAll();
}

void ScreenFXObj::GetLocalBoundBox(TimeValue t,
								   INode     *node,
								   ViewExp   *vpt,
								   Box3      &box)
{ FUNC_ENTER("ScreenFXObj::GetLocalBoundBox"); 
	BuildMesh(t);
	box = mesh.getBoundingBox();
}

void ScreenFXObj::GetWorldBoundBox(TimeValue t,
								   INode     *node,
								   ViewExp   *vpt,
								   Box3      &box)
{ FUNC_ENTER("ScreenFXObj::GetWorldBoundBox"); 
	BuildMesh(t);
	Matrix3 tm = node->GetObjectTM(t);
	box = mesh.getBoundingBox() * tm;
}

int ScreenFXObj::Display(TimeValue t,INode *node, ViewExp *vpt, int flags)
{ FUNC_ENTER("ScreenFXObj::Display"); 
	BuildMesh(t);

   // get the graphics window API and setup for drawing
   GraphicsWindow *gw = vpt->getGW();
   Material *mtl = gw->getMaterial();
   DWORD limits = gw->getRndLimits();
   gw->setRndLimits(GW_WIREFRAME | GW_BACKCULL);

   // setup colors
   if(node->Selected()) {
      gw->setColor(TEXT_COLOR, 1.f, 1.f, 1.f);
      gw->setColor(LINE_COLOR, 1.f, 1.f, 1.f);
   }
   else if(!node->IsFrozen()) {
      gw->setColor(TEXT_COLOR, 0.5f, 0.5f, 1.f);
      gw->setColor(LINE_COLOR, 0.5f, 0.5f, 1.f);
   }

   // setup window transformation
   Matrix3 tm(1);
   tm = node->GetObjectTM(t);
   gw->setTransform(tm);

   // finally draw the mesh itself
   mesh.render(gw, mtl, NULL, COMP_ALL);

   gw->setRndLimits(limits);
   return 0;
}

int ScreenFXObj::HitTest(TimeValue t,INode *node,int type,int crossing,int flags,IPoint2 *p,ViewExp *vpt)
{ FUNC_ENTER("ScreenFXObj::HitTest"); 
   HitRegion hitRegion;
   Point2 pt((float)p[0].x, (float)p[0].y);
   GraphicsWindow *gw = vpt->getGW();

   BuildMesh(t);
   gw->setTransform(node->GetObjectTM(t));
   MakeHitRegion(hitRegion, type, crossing, 4, p);

   return mesh.select(
      gw,
      node->Mtls(),
      &hitRegion,
      flags & HIT_ABORTONHIT,
      node->NumMtls());
}

ObjectState ScreenFXObj::Eval(TimeValue t)
{ FUNC_ENTER("ScreenFXObj::Eval"); 
	return ObjectState(this);
}

// We will support conversion to triangle meshes
int ScreenFXObj::CanConvertToType(Class_ID obtype)
{ FUNC_ENTER("ScreenFXObj::CanConvertToType"); 
	if (obtype == triObjectClassID ||
		obtype == vSCREENFXOBJ_CLASS_ID)
		return 1;
	else
		return Object::CanConvertToType(obtype);
}

Object* ScreenFXObj::ConvertToType(TimeValue t, Class_ID obtype)
{ FUNC_ENTER("ScreenFXObj::ConvertToType"); 
	if (obtype == vSCREENFXOBJ_CLASS_ID)
		return this;

	if (obtype == triObjectClassID)
	{
		TriObject *triob;
		BuildMesh(t);
		triob = CreateNewTriObject();
		triob->GetMesh() = mesh;
		triob->SetChannelValidity(TOPO_CHAN_NUM,ObjectValidity(t));
		triob->SetChannelValidity(GEOM_CHAN_NUM,ObjectValidity(t));		
		return triob;
	}
	else
		return Object::ConvertToType(t,obtype);
}
