#include "FuncEnter.h"

/*
	RTFEditor.cpp
	This class provides an RTF text editor with syntax highlighting
	and parameter completion features
	5-15-01
*/

#include "RTFEditor.h"
#include "ParseFuncs.h"
#include "RichEdit.h"
#include "../UI/RichText.h"
#include "../UI/PropList.h"
#include "../Resource.h"
#include "../path.h"

RTFEditor::RTFEditor(HINSTANCE hInstance, HWND hwnd) : keywordDB(8)
{ FUNC_ENTER("RTFEditor::RTFEditor"); 
	bAllocRichText = TRUE;
	bAllocPropList = TRUE;
	pRichText = new RichText(hInstance, hwnd);

	pPopupParams=new PropList(hInstance,pRichText->GetHWND(),0,0);
	Init();
}

RTFEditor::RTFEditor(RichText* pRichText) : keywordDB(8)
{ FUNC_ENTER("RTFEditor::RTFEditor"); 
	bAllocRichText = FALSE;
	bAllocPropList = TRUE;
	this->pRichText = pRichText;

	pPopupParams=new PropList(hInstance,pRichText->GetHWND(),0,0);
	Init();
}

RTFEditor::RTFEditor() : keywordDB(8)
{ FUNC_ENTER("RTFEditor::RTFEditor"); 
	bAllocRichText = FALSE;
	bAllocPropList = FALSE;

	Init();
}

RTFEditor::~RTFEditor()
{ FUNC_ENTER("RTFEditor::~RTFEditor"); 
	// Delete popup menu
	DestroyMenu(hMenu);

	if (bAllocPropList)
		delete pPopupParams;

	if (bAllocRichText)
		delete pRichText;
}

void RTFEditor::Init()
{ FUNC_ENTER("RTFEditor::Init"); 
	// Add some quick default keywords
	// Yes, they're hardcoded for now, but there are so few
	// <...> begin break else endif endscript if repeat script
	AddKeyword("<...>",0);
	AddKeyword("begin",0);
	AddKeyword("break",0);
	AddKeyword("else",0);
	AddKeyword("endif",0);
	AddKeyword("endscript",0);
	AddKeyword("if",0);
	AddKeyword("repeat",0);
	AddKeyword("script",0);
	AddKeyword("switch",0);
	AddKeyword("case",0);
	AddKeyword("default",0);
	AddKeyword("endswitch",0);

	bSyntaxHighlighting = TRUE;

	//pPopupParams=new PropList(hInstance,hwnd,0,0);

	// Load the popup menu resource
	hMenu=LoadMenu(hInstance,MAKEINTRESOURCE(IDR_SCRIPTPOPUP));
	hPopupMenu=GetSubMenu(hMenu,0);

	fpRTFChange = NULL;
	pChangeData = NULL;

	bAllowAccelerators = FALSE;
}

bool RTFEditor::AddKeyword(CStr Name,COLORREF color)
{ FUNC_ENTER("RTFEditor::AddKeyword"); 
	KeywordDesc* pKeyword=new KeywordDesc;

	pKeyword->Name=Name;
	pKeyword->color=color;

	if (keywordDB.GetItem(Name))
		keywordDB.FlushItem(Name);

	return keywordDB.PutItem(Name,pKeyword);
}

void RTFEditor::DeleteKeyword(CStr Name)
{ FUNC_ENTER("RTFEditor::DeleteKeyword"); 
	if (Name==CStr(""))
		return;

	KeywordDesc* pKeyword=keywordDB.GetItem(Name);
	
	if (pKeyword)
	{
		delete pKeyword;
		keywordDB.FlushItem(Name);
	}
}

void RTFEditor::SyntaxCheck(int lineNum)
{ FUNC_ENTER("RTFEditor::SyntaxCheck"); 
	CStr text=pRichText->GetLine(lineNum);
	CStr strNewLine;

	// DEBUG: Scan the string for out of range characters
	/*
	for(int i = 0; i < text.Length(); i++)
	{
		if (text[i] < 30 || text[i] > 126)
		{
			int zzz;
			zzz = 0;
		}
	}
	*/

	//SetDlgItemText(hwnd,IDC_LINETXT,(char*)text);

	strNewLine =SRTF_COLOR_HEADER+SRTF_DEFAULT;
	strNewLine+=SyntaxRTF(text);
	strNewLine+=SRTF_END;

	//SetDlgItemText(hwnd,IDC_LINERTF,(char*)strNewLine);

	if (bSyntaxHighlighting)
		pRichText->SetLine(strNewLine,lineNum);
}

CStr RTFEditor::ParseEscapes(CStr line)
{ FUNC_ENTER("RTFEditor::ParseEscapes"); 
	line=InsertEscapes(line,'\\','\\');
	line=InsertEscapes(line,'{','\\');
	line=InsertEscapes(line,'}','\\');

	return line;
}

CStr RTFEditor::SyntaxRTF(CStr text)
{ FUNC_ENTER("RTFEditor::SyntaxRTF"); 
	// Parse the line of text for script database elements
	CStr strNewLine;
	CStr strWord;
	char schr[2]="\0";	// Two nulls (one explicit, one implicit)
	//char leftBuf[512]="";
	char leftBuf[1024]="";
	int  leftPos=0;
	int  pos=0;
	int  wrdStart=0;
	int  textLen;

	// Add escape codes to line if it uses codes for RTF blocks
	text=ParseEscapes(text);
	textLen=text.Length();

	strWord=GetWord(text,&pos,&wrdStart);
	
	while(strWord!=CStr(""))
	{

		TypeDesc* tdesc=scriptDB.GetItem(strWord);

		// If the entered word exists in the type info database process it
		if (tdesc)
		{
			// Highlight the text
			//MessageBox(NULL,strWord,"Word IDed",MB_OK);

			// Copy the content up to the start of this word
			for(int i=leftPos;i<wrdStart;i++)
			{
				schr[0]=text[i];
				strNewLine+=CStr(schr);
			}

			// Begin RTF colored text based on type
			switch(tdesc->type)
			{
			case DATA_SCRIPT:
				strNewLine+=SRTF_BLUE;
				break;

			case DATA_INTEGER:
				strNewLine+=SRTF_BLUE;
				break;

			case DATA_FLOAT:
			case DATA_VECTOR:
			case DATA_PAIR:
			case DATA_STRING:
			case DATA_ARRAY:
				strNewLine+=SRTF_RED;
				break;

			case DATA_LOCALSTRING:
			case DATA_STRUCT:
			case DATA_NAME:
				strNewLine+=SRTF_BOLD;
				strNewLine+=SRTF_GREEN;
				break;

			default:
				strNewLine+=SRTF_GREEN;
			}

			// Dump the word
			strNewLine+=tdesc->Name;

			// Return to default state text
			strNewLine+=SRTF_DEFAULT;

			// Advance left position to end of word and continue
			leftPos=wrdStart+tdesc->Name.Length();

			strWord=GetWord(text,&pos,&wrdStart);
			continue;
		}

		// Check for keyword
		KeywordDesc* kdesc=keywordDB.GetItem(strWord);

		if (kdesc)
		{
			// Highlight the text
			//MessageBox(NULL,strWord,"Word IDed",MB_OK);

			// Copy the content up to the start of this word
			for(int i=leftPos;i<wrdStart;i++)
			{
				schr[0]=text[i];
				strNewLine+=CStr(schr);
			}

			// Begin RTF colored text based on type
			strNewLine+=SRTF_RED+SRTF_BOLD;

			// Dump the word
			strNewLine+=kdesc->Name;

			// Return to default state text
			strNewLine+=SRTF_DEFAULT;

			// Advance left position to end of word and continue
			leftPos=wrdStart+kdesc->Name.Length();

			strWord=GetWord(text,&pos,&wrdStart);
			continue;
		}

		// Check for comment
		if (strWord==CStr("//") ||
			  (strWord.Length()>1 && strWord[0]=='/' && strWord[1]=='/') ||
			  (strWord.Length()>0 && strWord[0]==';') )
		{
			// Dump the rest of the line as a comment
			strNewLine+=SRTF_GREEN;
			break;
		}

		// If it's not a type, just copy it
		// Copy the content up to the word, then the word itself
		for(int i=leftPos;i<wrdStart;i++)
		{
			schr[0]=text[i];
			strNewLine+=CStr(schr);
		}
		strNewLine+=strWord;
		leftPos=wrdStart+strWord.Length();

		strWord=GetWord(text,&pos,&wrdStart);
	}

	// Copy the rest of the line (terminating spaces?)
	for(int i=leftPos;i<textLen;i++)
	{
		schr[0]=text[i];
		strNewLine+=CStr(schr);
	}

	strNewLine+=SRTF_DEFAULT;

	return strNewLine;
}

void RTFEditor::SetText(CStr text)
{ FUNC_ENTER("RTFEditor::SetText"); 
	pRichText->SetText(BuildTextRTF(text));
}

CStr RTFEditor::GetText()
{ FUNC_ENTER("RTFEditor::GetText"); 
	return pRichText->GetText();
}

CStr RTFEditor::BuildTextRTF(CStr text)
{ FUNC_ENTER("RTFEditor::BuildTextRTF"); 
	CStr strLine;
	CStr strNewText;
	int  pos=0;
	int  textLen=text.Length();

	strNewText =SRTF_COLOR_HEADER+SRTF_DEFAULT;

	while(pos<textLen)
	{
		//strLine=GetRemainLinePartial(text,&pos);
		strLine=GetRemainLineExact(text,&pos);
		strNewText+=SyntaxRTF(strLine);
		strNewText+=SRTF_ENDLINE;
	}

	strNewText+=SRTF_END;

	return strNewText;
}

CStr RTFEditor::GetWord(char* lineBuf,int* xPos,int* wrdStart)
{ FUNC_ENTER("RTFEditor::GetWord"); 
	char wordBuf[512];
	int  wordStart,wordEnd;
	int  lineLen;

	lineLen=strlen(lineBuf);

	// Check to see if there are no words remaining
	if (*xPos>=lineLen)
		return CStr("");

	// Now scan the line to the left for spaces to determine the word start pos


	// Scan to the right in search of a word
	for(wordStart=*xPos;wordStart<lineLen;wordStart++)
		if (lineBuf[wordStart]!=' '  &&
			lineBuf[wordStart]!='\t' &&
			lineBuf[wordStart]!='\r' &&
			lineBuf[wordStart]!='\n')
		{
			//wordStart++;
			break;
		}

	// Now scan the line to the right for spaces to determine the word end pos
	for(wordEnd=wordStart;wordEnd<lineLen;wordEnd++)
		if (lineBuf[wordEnd]==' '  ||
			lineBuf[wordEnd]=='\t' ||
			lineBuf[wordEnd]=='\r' ||
			lineBuf[wordEnd]=='\n')
		{
			//wordEnd--;
			break;
		}

	// Now build the word
	int i=0;

	for(int linePos=wordStart;linePos<wordEnd;linePos++)
	{
		// Words beyond buffer size will be automatically truncated
		if (i == 510)
			break;

		wordBuf[i++]=lineBuf[linePos];
	}

	wordBuf[i]='\0';

	*xPos=wordEnd+1;

	if (wrdStart)
		*wrdStart=wordStart;

	return CStr(wordBuf);
}

void RTFEditor::ProcParamList()
{ FUNC_ENTER("RTFEditor::ProcParamList"); 
	CStr strWord=pRichText->GetWord();
	
	if (strWord==CStr(""))
		return;

	TypeDesc* tdesc=scriptDB.GetItem(strWord);

	if (tdesc)
	{
		int indexPos=pRichText->GetCurPos();
		indexPos-=strWord.Length();

		POINT pt=pRichText->GetScreenPos(indexPos);

		// If a popup parameter list already exists ditch the old one
		if (pPopupParams)
		{
			//delete pPopupParams;
			pPopupParams->Hide();
		}

		FuncName=strWord;

		//pPopupParams=new PropList(hInstance,hwnd,pt.x,pt.y+15);
		pPopupParams->Clear();
		pPopupParams->SetPos(pt.x,pt.y+15);
		
		// Set our callback function so we can intercept the user pressing apply
		pPopupParams->SetApplyCB(PopupParamsApplyCB,this);
		pPopupParams->SetChangeCB(PopupParamsChangeCB,this);
		pPopupParams->SetExtListCB(PopupParamsExtListCB,this);

		ParseScriptTags(pPopupParams,tdesc->ScriptFile,tdesc->Name);	
		
		if (pPopupParams->NumProps()!=0)
			pPopupParams->BuildUI();

		//SetFocus(GetDlgItem(hwnd,IDC_RICHEDIT1));
		SetFocus(pRichText->GetHWND());
	}
}

LRESULT RTFEditor::ProcMessage(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("RTFEditor::ProcMessage"); 
	MSGFILTER* msgfilter;
	int        lineNum;
	POINT      ptCursor;

	if (!bAllowAccelerators)
		DisableAccelerators();

	switch(msg)
	{
	case WM_COMMAND:
		if ((HWND)lParam == pRichText->GetHWND())
		{
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				bChangeMade=TRUE;
				bScriptChanged=TRUE;

				// Hide any parameter popup window if not locked
				if (pPopupParams && !bLockParamPopup)
					pPopupParams->Hide();
				else
					bLockParamPopup=FALSE;

				/*
				if (pPopupParams)
				{
					delete pPopupParams;
					pPopupParams=NULL;
				}
				*/

				// Perform syntax highlighting on this line
				// (and the line above due to enter breaks)
				lineNum=pRichText->GetLineNum();

				if (lineNum>0)
					SyntaxCheck(pRichText->GetLineNum()-1);

				SyntaxCheck(pRichText->GetLineNum());
				
				// Popup a function parameter list if necessary
				if (!pPopupParams->IsVisible())
					ProcParamList();

				// Execute the change callback if available
				if (fpRTFChange)
					fpRTFChange(this,pChangeData);

				return TRUE;
			}
		}
		break;

	case WM_NOTIFY:
		if (((NMHDR*)lParam)->hwndFrom == pRichText->GetHWND())
		{
			// Intercept the RichText control's virtual key methods
			// so we can do syntax highlighting on the fly
			msgfilter=(MSGFILTER*)lParam;
			switch(msgfilter->msg)
			{
			case WM_RBUTTONUP:
				GetCursorPos(&ptCursor);
				TrackPopupMenu(hPopupMenu,TPM_RIGHTBUTTON,
					           ptCursor.x,ptCursor.y,
							   0,hwnd,NULL);
				break;

			case WM_KEYDOWN:
				switch(msgfilter->wParam)
				{
				case VK_UP:
				case VK_DOWN:
				case VK_LEFT:
				case VK_RIGHT:
				case VK_SPACE:
				case VK_RETURN:
				case VK_BACK:
					if (pPopupParams)
					{
						//delete pPopupParams;
						//pPopupParams=NULL;
						pPopupParams->Hide();
					}
					return TRUE;

				case VK_ESCAPE:
					bLockParamPopup=FALSE;

					if (pPopupParams->IsVisible())
					{
						//delete pPopupParams;
						//pPopupParams=NULL;
						pPopupParams->Hide();
					}
					else
					{
						bAllowAccelerators = TRUE;
						EnableAccelerators();
						SendMessage(hwnd,WM_CLOSE,0,0);
					}

					return TRUE;

				case VK_F1:
				case VK_F2:
					SpawnWinHelp();
					return TRUE;
				}

				return TRUE;
			}	
			
		}
	}

	return FALSE;
}

void RTFEditor::PopupParamsApplyCB(PropList* pPropList,void* pData)
{ FUNC_ENTER("RTFEditor::PopupParamsApplyCB"); 
	RTFEditor* pthis=(RTFEditor*)pData;
	CStr        eol="\r\n";	

	if (!pthis || !pPropList)
		return;

	// Close the parameter popup
	pthis->bLockParamPopup=FALSE;
	pPropList->Hide();

	pthis->pRichText->GoToEnd();
	pthis->pRichText->InsertText(eol);

	/*
	CStr strParams=pthis->GetParamCompleteText(pPropList);
	strParams+=CStr("\r\n");

	pthis->pRichText->InsertText(strParams);
	*/

	//SetFocus(GetDlgItem(pthis->hwnd,IDC_RICHEDIT1));
	SetFocus(pthis->pRichText->GetHWND());
}

void RTFEditor::PopupParamsChangeCB(PropList* pPropList,void* pData)
{ FUNC_ENTER("RTFEditor::PopupParamsChangeCB"); 
	RTFEditor*  pthis   =(RTFEditor*)pData;
	char        tmpbuf[512];

	if (!pthis || !pPropList)
		return;

	if (pthis->bLockParamComplete)
	{
		pthis->bLockParamComplete=FALSE;
		return;
	}

	CStr strParams=pthis->GetParamCompleteText(pPropList);
	CStr strLine  =pthis->pRichText->GetLine();
	CStr strNewLine;

	int pos=Instr(strLine,pthis->FuncName);

	if (pos!=-1)
	{
		strcpy(tmpbuf,(char*)strLine);
		tmpbuf[pos+pthis->FuncName.Length()]='\0';
		
		strLine=CStr(tmpbuf)+strParams;

		pthis->bLockParamPopup=TRUE;
		
		strNewLine =SRTF_COLOR_HEADER+SRTF_DEFAULT;
		strNewLine+=pthis->SyntaxRTF(strLine);
		strNewLine+=SRTF_END;

		if (pthis->bSyntaxHighlighting)
			pthis->pRichText->SetLine((char*)strNewLine);
	}
}

void RTFEditor::PopupParamsExtListCB(CStr name,CStr value,void* pData)
{ FUNC_ENTER("RTFEditor::PopupParamsExtListCB"); 
	CStr strLine,strLineLower;
	CStr buf;
	CStr strSearch1, strSearch2;
	CStr Keyword;
	char lineBuf[512];
	bool bFileTop=true;

	enum SearchMode
	{
		SEARCH_SCRIPT,
		SEARCH_PARAM,
		SEARCH_DONE,
	};

	SearchMode mode=SEARCH_SCRIPT;


	RTFEditor* pthis=(RTFEditor*)pData;
	
	// Determine the script file that contains this value
	TypeDesc* tdesc=pthis->scriptDB.GetItem(pthis->FuncName);

	strSearch1=CStr("// @script ");
	strSearch2=CStr("| ")+pthis->FuncName+CStr(" |");
	strSearch2.toLower();

	if (tdesc)
	{
		// Scan for the AutoDuck description
		FILE* fp=fopen(tdesc->ScriptFile,"r");

		if (!fp)
		{
			char ErrMsg[256];
			sprintf(ErrMsg,"Couldn't update extensible list values.\nFile: %s",(char*)tdesc->ScriptFile);
			MessageBox(pthis->pRichText->GetHWND(),ErrMsg,"Extensible List Update",MB_ICONSTOP|MB_OK);
			return;
		}

		while(fgets(lineBuf,512,fp))
		{
			strLine=lineBuf;
			strLineLower=lineBuf;
			strLineLower.toLower();

			if (mode!=SEARCH_DONE)
			{
				if (IsInstr(strLineLower,strSearch1) &&
					IsInstr(strLineLower,strSearch2))
				{
					if (mode==SEARCH_SCRIPT)
					{
						// Now look for our @nextparm line
						strSearch1=CStr("// @nextparm | ")+name;
						strSearch1.toLower();
						strSearch2=" ";
						mode=SEARCH_PARAM;
					}
					else if (mode==SEARCH_PARAM)
					{
						// We've now found our @nextparm line
						// Replace "| done" with the new parameter
						
						if (!IsInstr(value,"| done"))
						{
							strLine=ReplaceStr(strLine,"| done",CStr("| ")+value);
							strLine=ReplaceStr(strLine,"\n","");
							strLine=ReplaceStr(strLine,"\r","");
							strLine+=" | done\n";
						}

						mode=SEARCH_DONE;
					}
				}
			}

			// Store the file contents in memory so we can rewrite it
			// with the new value
			buf+=strLine;
		}

		fclose(fp);

		// Now rewrite the new file
		fp=fopen(tdesc->ScriptFile,"w");
		fwrite((char*)buf,buf.Length(),1,fp);
		fclose(fp);
	}
}

void RTFEditor::SpawnWinHelp()
{ FUNC_ENTER("RTFEditor::SpawnWinHelp"); 
	CStr strHelpFile=getenv(HELP_ENV);

	if (strHelpFile==CStr(""))
	{
		char ErrorMsg[256];
		sprintf(ErrorMsg,"Environment variable '%s' is not defined.",HELP_ENV);
		MessageBox(pRichText->GetHWND(),ErrorMsg,"Couldn't locate help file",MB_ICONSTOP|MB_OK);
		return;
	}

	WinHelp(pRichText->GetHWND(),
			strHelpFile,
			HELP_KEY,
			(DWORD)(char*)pRichText->GetWord());
}

CStr RTFEditor::GetParamCompleteText(PropList* pPropList)
{ FUNC_ENTER("RTFEditor::GetParamCompleteText"); 
	CStr strName;
	CStr strParams;
	CStr strValue;
	
	// Dump the contents of all the property list values out to a parameter string
	int numProps=pPropList->NumProps();

	for(int i=0;i<numProps;i++)
	{
		if (pPropList->GetType(i)==PROPTYPE_CHECK)
		{
			if (pPropList->GetValue(i,strValue))
			{
				if (strValue==CStr("TRUE"))
					strParams+=CStr(" ")+pPropList->GetName(i);
			}
			else
				return CStr("");
		}
		else
		{
			if (!pPropList->GetValue(i,strValue))
				return CStr("");

			if (strValue==CStr("") || strValue==CStr(" ") || strValue==CStr("<None>"))
				continue;

			strName = pPropList->GetName(i);
			
			if (strName.Length()>0)
				strParams+=CStr(" ")+pPropList->GetName(i)+CStr("=");
			else
				strParams+=CStr(" ");

			// Determine if we should put quotes around it
			switch(pPropList->GetType(i))
			{
			//case PROPTYPE_EDIT:
			case PROPTYPE_LIST:
			case PROPTYPE_EXTLIST:
			case PROPTYPE_FILE:
				strValue=CStr("\"")+strValue+CStr("\"");
				break;

			default:
				break;
			}

			strParams+=strValue;
		}
	}

	return strParams;
}

void RTFEditor::InitScriptDB()
{ FUNC_ENTER("RTFEditor::InitScriptDB"); 
	CStr strMapFile=getenv(APP_ENV);

	strMapFile+=CStr(SCRIPT_PATH)+MAP_FILE;
	ParseMapFile(strMapFile);
	ParseTagDBFile();
}

CStr RTFEditor::LoadTextStr(char* Filename)
{ FUNC_ENTER("RTFEditor::LoadTextStr"); 
	CStr  fileBuf;
	char  lineBuf[512];
	FILE* fp;
	int   size;

	fp=fopen(Filename,"r");

	if (!fp)
		return CStr("");

	// Check the file size
	fseek(fp,0,SEEK_END);
	size=ftell(fp);

	if (size>RICHTEXT_SIZELIMIT)
	{
		char strErr[256];

		fclose(fp);
		sprintf(strErr,"This file is over %iK bytes and is too large to be displayed using this editor.",RICHTEXT_SIZELIMIT/1024);

		MessageBox(pRichText->GetHWND(),strErr,"File too large",MB_ICONWARNING);
		return CStr("");
	}

	fseek(fp,0,SEEK_SET);

	while(!feof(fp))
	{
		fgets(lineBuf,512,fp);
		fileBuf+=lineBuf;
	}
	fclose(fp);

	return fileBuf;
}

bool RTFEditor::LoadText(char* Filename)
{ FUNC_ENTER("RTFEditor::LoadText"); 
	// Load the data into a CStr and have SetText add it
	CStr fileBuf=LoadTextStr(Filename);

	if (fileBuf==CStr(""))
		return false;

	SetText(fileBuf);
	return true;
}

void RTFEditor::GoToScriptDefinition(CStr keyword)
{ FUNC_ENTER("RTFEditor::GoToScriptDefinition"); 
	// Ensure that you don't lose anything
	TypeDesc* tdesc=scriptDB.GetItem(keyword);
		
	if (tdesc)
	{
		pRichText->Clear();

		if (!LoadText(tdesc->ScriptFile))
		{
			char errMsg[256];
			sprintf(errMsg,"The qscript file '%s' defining '%s' could not be found.",(char*)tdesc->ScriptFile,(char*)keyword,MB_ICONSTOP|MB_OK);
			MessageBox(pRichText->GetHWND(),errMsg,"Script file not found",MB_ICONSTOP|MB_OK);
		}
		else
		{
			// Find the definition of the script in the file
			CStr strSearch=CStr("Script ")+keyword;
			int  index=pRichText->Search(strSearch,RTS_WHOLEWORD);
			
			if (index!=-1)
				pRichText->HighlightLine(index);
		}
		return;
	}
	else
	{
		char errMsg[256];
		sprintf(errMsg,"The keyword '%s' does not exist in the map file database.",(char*)keyword);
		MessageBox(pRichText->GetHWND(),errMsg,"Keyword not found",MB_ICONWARNING|MB_OK);
	}
}

void RTFEditor::AllowChanges(BOOL mode)
{ FUNC_ENTER("RTFEditor::AllowChanges"); 
	if (mode)
		pRichText->EnableText();
	else
		pRichText->DisableText();
}

void RTFEditor::SetChangeCB(void(*fpCallback)(RTFEditor*,void*),void* pData)
{ FUNC_ENTER("RTFEditor::SetChangeCB"); 
	fpRTFChange = fpCallback;
	pChangeData = pData;
}
