#include "FuncEnter.h"

/*
	PropUpdater.cpp
	Property Updater

	The property updater is responsible for updating the properties of all objects within
	the level when property fields are added/deleted
*/

#include "PropUpdater.h"
#include "Next.h"
#include "ConfigData.h"
#include "../UI/ProgressBar.h"
#include "../UI/ColorListBox.h"
#include "appdata.h"
#include <windows.h>
#include "path.h"
#include "../PropEdit/ParseFuncs.h"
#include "../Export/ExportOptions.h"
#include "appdata.h"
#include "../Misc/MaxUtil.h"
#include "../Link/LinkUI.h"				// For vLINK_OBJ_CLASS_ID
#include "../Particle/NExtParticle.h"
#include "PropEdit.h"
#include "UserConfig.h"

#define COLOR_CLASS   RGB(0,0,128)		// Color that classes (without types) are displayed in
#define COLOR_TYPE    RGB(0,128,0)		// Color that classes with types are displayed in
#define COLOR_REMOVED RGB(0,0,0)		// Color of classes or types that have been removed from scripts.ini

// Purge functions from DebugTool.cpp
extern void PurgeScriptCache(INode* node = NULL);
extern void PurgeNodeCache(INode* node = NULL);

extern PropEditor* pPropEdit;

PropUpdater::PropUpdater(HINSTANCE hInstance, HWND hwnd, Interface* ip) :
	ModalDlgWindow(hInstance, MAKEINTRESOURCE(IDD_PROPUPDATER), hwnd), crcTable(8)
{ FUNC_ENTER("PropUpdater::PropUpdater"); 
	this->ip = ip;
	this->hInstance = hInstance;
	this->hwndParent = hwndParent;
	
	// Assign default options
	bAddNewProps               = true;
	bDeleteAllOld              = false;
	bDeleteRemovedDefaultProps = true;
	bUpdateOmissions           = true;
	bDiscardRemovedClasses     = false;
	bUpdateTriggerColors       = false;

	ColorListBox::Register(hInstance);

	pCTList = NULL;
	pNList  = NULL;

#ifndef DISABLE_NOTIFICATIONS
	RegisterNotification(LoadCB,this,NOTIFY_FILE_POST_OPEN);
	RegisterNotification(PreMergeCB,this,NOTIFY_FILE_PRE_MERGE);
	RegisterNotification(MergeCB,this,NOTIFY_FILE_POST_MERGE);
#endif

	bCancel = false;
}

PropUpdater::~PropUpdater()
{ FUNC_ENTER("PropUpdater::~PropUpdater"); 
	if (pCTList)
		delete pCTList;

	if (pNList)
		delete pNList;

	UnRegisterNotification(LoadCB,this,NOTIFY_FILE_POST_OPEN);
	UnRegisterNotification(PreMergeCB,this,NOTIFY_FILE_PRE_MERGE);
	UnRegisterNotification(MergeCB,this,NOTIFY_FILE_POST_MERGE);
}

void PropUpdater::SetOptions()
{ FUNC_ENTER("PropUpdater::SetOptions"); 
	if (bAddNewProps)
		CheckDlgButton(hwnd, IDC_ADDNEWPROPS, BST_CHECKED);
	else
		CheckDlgButton(hwnd, IDC_ADDNEWPROPS, BST_UNCHECKED);

	if (bDeleteAllOld)
		CheckDlgButton(hwnd, IDC_DELOLDPROPS, BST_CHECKED);
	else
		CheckDlgButton(hwnd, IDC_DELOLDPROPS, BST_UNCHECKED);

	if (bDeleteRemovedDefaultProps)
		CheckDlgButton(hwnd, IDC_DELOLDIFDEF, BST_CHECKED);
	else
		CheckDlgButton(hwnd, IDC_DELOLDIFDEF, BST_UNCHECKED);

	if (bUpdateOmissions)
		CheckDlgButton(hwnd, IDC_DELOMITPROPS, BST_CHECKED);
	else
		CheckDlgButton(hwnd, IDC_DELOMITPROPS, BST_UNCHECKED);

	if (bDiscardRemovedClasses)
		CheckDlgButton(hwnd, IDC_DISCARDDELNODES, BST_CHECKED);
	else
		CheckDlgButton(hwnd, IDC_DISCARDDELNODES, BST_UNCHECKED);

	if (bUpdateTriggerColors)
		CheckDlgButton(hwnd, IDC_UPDATECOLORS, BST_CHECKED);
	else
		CheckDlgButton(hwnd, IDC_UPDATECOLORS, BST_UNCHECKED);
}

void PropUpdater::GetOptions()
{ FUNC_ENTER("PropUpdater::GetOptions"); 
	if (IsDlgButtonChecked(hwnd, IDC_ADDNEWPROPS) == BST_CHECKED)
		bAddNewProps = true;
	else
		bAddNewProps = false;

	if (IsDlgButtonChecked(hwnd, IDC_DELOLDPROPS) == BST_CHECKED)
		bDeleteAllOld = true;
	else
		bDeleteAllOld = false;

	if (IsDlgButtonChecked(hwnd, IDC_DELOLDIFDEF) == BST_CHECKED)
		bDeleteRemovedDefaultProps = true;
	else
		bDeleteRemovedDefaultProps = false;

	if (IsDlgButtonChecked(hwnd, IDC_DELOMITPROPS) == BST_CHECKED)
		bUpdateOmissions = true;
	else
		bUpdateOmissions = false;

	if (IsDlgButtonChecked(hwnd, IDC_DISCARDDELNODES) == BST_CHECKED)
		bDiscardRemovedClasses = true;
	else
		bDiscardRemovedClasses = false;

	if (IsDlgButtonChecked(hwnd, IDC_UPDATECOLORS) == BST_CHECKED)
		bUpdateTriggerColors = true;
	else
		bUpdateTriggerColors = false;
}

BOOL PropUpdater::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ FUNC_ENTER("PropUpdater::DlgProc"); 
	switch(msg)
	{
	case WM_INITDIALOG:
		pCTList = new ColorListBox(hInstance);
		pCTList->Attach(GetDlgItem(hwnd, IDC_CTLIST), LBS_SORT|LBS_EXTENDEDSEL);
		
		pNList = new ColorListBox(hInstance);
		pNList->Attach(GetDlgItem(hwnd, IDC_NLIST), LBS_SORT|LBS_EXTENDEDSEL);

		SetOptions();
		CheckDlgButton(hwnd, IDC_PROCINSTANCES, BST_CHECKED);

		LoadClassTypeCRCs();
		PopulateClassTypeList();
		SelAll(IDC_CTLIST);
		PopulateNodeList();
		SelAll(IDC_NLIST);

		// Force the user to do a full update if force scripts.ini updates is set
		if (bForceScriptUpdates)
		{
			EnableWindow(GetDlgItem(hwnd, IDC_ADDNEWPROPS), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_DELOLDPROPS), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_DELOLDIFDEF), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_DELOMITPROPS), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_DISCARDDELNODES), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_UPDATECOLORS), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_OK), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_CANCEL), FALSE);

			// Full update will automatically be performed now
			bCancel = false;
			DoFullUpdate();
			EndDialog(hwnd, 0);
		}

		return TRUE;

	case WM_COMMAND:
		{
			switch(LOWORD(wParam))
			{
			case IDC_OK:
				bCancel = false;
				BuildUpdateList();
				EndDialog(hwnd, 0);
				return TRUE;

			case IDC_CANCEL:
				bCancel = true;
				EndDialog(hwnd, 0);
				return TRUE;

			case IDC_CTSELALL:
				SelAll(IDC_CTLIST);
				PopulateNodeList();
				return TRUE;

			case IDC_CTSELNONE:
				SelNone(IDC_CTLIST);
				PopulateNodeList();
				return TRUE;

			case IDC_CTSELINVERT:
				SelInvert(IDC_CTLIST);
				PopulateNodeList();
				return TRUE;

			case IDC_NSELALL:
				SelAll(IDC_NLIST);
				return TRUE;

			case IDC_NSELNONE:
				SelNone(IDC_NLIST);
				return TRUE;

			case IDC_NSELINVERT:
				SelInvert(IDC_NLIST);
				return TRUE;

			case IDC_ADDNEWPROPS:
				bAddNewProps = !bAddNewProps;
				SetOptions();
				return TRUE;

			case IDC_DELOLDPROPS:
				bDeleteAllOld = !bDeleteAllOld;

				if (bDeleteAllOld && bDeleteRemovedDefaultProps)
					bDeleteRemovedDefaultProps = false;
				
				SetOptions();
				return TRUE;

			case IDC_DELOLDIFDEF:;
				bDeleteRemovedDefaultProps = !bDeleteRemovedDefaultProps;

				if (bDeleteRemovedDefaultProps && bDeleteAllOld)
					bDeleteAllOld = false;
				
				SetOptions();
				return TRUE;

			case IDC_DELOMITPROPS:
				bUpdateOmissions = !bUpdateOmissions;
				SetOptions();
				return TRUE;

			case IDC_DISCARDDELNODES:
				bDiscardRemovedClasses = !bDiscardRemovedClasses;
				SetOptions();
				return TRUE;

			case IDC_UPDATECOLORS:
				bUpdateTriggerColors = !bUpdateTriggerColors;
				SetOptions();
				return TRUE;

			case IDC_FULLUPDATE:
				bCancel = false;
				DoFullUpdate();
				EndDialog(hwnd, 0);
				return TRUE;

			case IDC_CTLIST:
				{
					switch(HIWORD(wParam))
					{
					case LBN_SELCHANGE:
						PopulateNodeList();
						return TRUE;
					}
				}
				break;
			}
		}
		break;

	case WM_CLOSE:
		EndDialog(hwnd, 0);
		return TRUE;
	}

	return FALSE;
}

void PropUpdater::LoadCB(void* param,NotifyInfo* ninfo)
{ FUNC_ENTER("PropUpdater::LoadCB"); 
	OutputDebugString("HANDLER: LoadCB (PropUpdater)\n");
#ifdef DEBUG_CUTSCENEDB
	MessageBox(gInterface->GetMAXHWnd(), "PropUpdater::LoadCB Called\nMake NOTE!!", "Debug tracking for Cutscene DB Clears", MB_OK);
#endif

	PropUpdater* pthis = (PropUpdater*)param;
	pthis->lastNodeID = 0;

	INode* scene = gInterface->GetRootNode();
	AppDataChunk*    appdata;

	appdata = scene->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_DATA);

	pthis->ProcChanges();
}

void PropUpdater::PreMergeCB(void* param,NotifyInfo* ninfo)
{ FUNC_ENTER("PropUpdater::PreMergeCB"); 
	OutputDebugString("HANDLER: PreMergeCB (PropUpdater)\n");
#ifdef DEBUG_CUTSCENEDB
	MessageBox(gInterface->GetMAXHWnd(), "PropUpdater::PreMerge Called\nMake NOTE!!", "Debug tracking for Cutscene DB Clears", MB_OK);
#endif

	// Don't perform this action if the merge is due to an XRef
	if (ninfo->callParam != NULL)
		return;

	PropUpdater* pthis = (PropUpdater*)param;
	pthis->FindLastNodeID();
}

void PropUpdater::MergeCB(void* param,NotifyInfo* ninfo)
{ FUNC_ENTER("PropUpdater::MergeCB"); 
	OutputDebugString("HANDLER: MergeCB (PropUpdater)\n");
#ifdef DEBUG_CUTSCENEDB
	MessageBox(gInterface->GetMAXHWnd(), "PropUpdater::PostMerge Called\nMake NOTE!!", "Debug tracking for Cutscene DB Clears", MB_OK);
#endif
	
	if (ninfo->callParam != NULL)
		return;

	PropUpdater* pthis = (PropUpdater*)param;
	pthis->ProcChanges(true);
}

// Is file 2 newer than file one
bool PropUpdater::FileIsNewer( char* file1, char* file2 )
{ FUNC_ENTER("PropUpdater::FileIsNewer"); 
	HANDLE file_handle_1, file_handle_2;
	WIN32_FIND_DATA find_data_1, find_data_2;
	
	file_handle_1 = FindFirstFile( file1, &find_data_1 );  
	if( file_handle_1 == INVALID_HANDLE_VALUE )
	{
		return true;
	}

	file_handle_2 = FindFirstFile( file2, &find_data_2 );  
	if( file_handle_2 == INVALID_HANDLE_VALUE )
	{
		FindClose( file_handle_1 );
		return false;
	}
	
	FindClose( file_handle_1 );
	FindClose( file_handle_2 );
	return( CompareFileTime( &find_data_2.ftLastWriteTime, &find_data_1.ftLastWriteTime ) > 0 );
}

void PropUpdater::FindLastNodeID(INode* node)
{ FUNC_ENTER("PropUpdater::FindLastNodeID"); 
	if (!node)
	{
		node = ip->GetRootNode();
		lastNodeID = 0;
	}

	int nKids = node->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child = node->GetChildNode(i);
		FindLastNodeID(child);
	}

	if (lastNodeID < node->GetHandle())
		lastNodeID = node->GetHandle();
}

CStr PropUpdater::GetScriptIniName()
{ FUNC_ENTER("PropUpdater::GetScriptIniName"); 
	CStr strFile=getenv(APP_ENV);

	// Verify that the environment variable is set
	if (strFile==CStr(""))
	{
		char ErrorBuf[256];
		sprintf(ErrorBuf,"Couldn't load '%s' the '%s' environment variable is not set.",SCRIPT_INI,APP_ENV);

		MessageBox(NULL,ErrorBuf,"ParseScriptIni",MB_ICONSTOP|MB_OK);
		return CStr("");
	}

	//strFile+=CStr(SCRIPT_PATH)+SCRIPT_INI;
	strFile = GetScriptsIniPath();

	return strFile;
}

int PropUpdater::CountNodes(INode* node,int num)
{ FUNC_ENTER("PropUpdater::CountNodes"); 
	if (node == NULL)
		node = ip->GetRootNode();

	int numKids = node->NumberOfChildren();

	for(int i=0;i<numKids;i++)
	{
		INode* childNode = node->GetChildNode(i);
		num = CountNodes(childNode,num);
		num++;
	}

	return num;
}

COLORREF PropUpdater::ParseColor(char* line)
{ FUNC_ENTER("PropUpdater::ParseColor"); 
	COLORREF color;
	char* cpos = strstr(line, "// @nodecolor |");
	char  bufColor[256];

	if (!cpos)
		return 0;
	
	cpos += 15;
	cpos = strstr(cpos, "[");

	if (!cpos)
		return 0;

	cpos++;

	GetOption(cpos, bufColor, ',', 0);
	color = (COLORREF)((unsigned char)atoi(bufColor));
	GetOption(cpos, bufColor, ',', 1);
	color |= (COLORREF)((unsigned char)atoi(bufColor)) << 8;
	GetOption(cpos, bufColor, ',', 2);
	color |= (COLORREF)((unsigned char)atoi(bufColor)) << 16;

	return color;
}

void PropUpdater::ProcPostParticleChanges()
{ FUNC_ENTER("PropUpdater::ProcPostParticleChanges"); 
	Link<INode*>* link = particleList.GetHead();

	while(link)
	{
		CStr propBuffer;
		link->data->GetUserPropBuffer(propBuffer);

		CStr className = GetClassName(propBuffer);
		CStr typeName  = GetTypeName(propBuffer);

		LinkList<ConfigProp> cprops;

		CStr strScript = ParseConfigProps(&cprops,NULL,propBuffer);

		Link<ConfigProp>* linkProp = cprops.GetHead();
		int prop_index;

		while(linkProp)
		{
			// Convert value to default if necessary
			if (linkProp->data.value == CStr("!"))
				linkProp->data.value = pPropEdit->GetDefault(className, typeName, linkProp->data.name);

			if(( prop_index = GetParticlePropertyIndex( linkProp->data.name )) >= 0 )
			{
				Object* obj = link->data->EvalWorldState(0).obj;

				if (obj)
				{
					IParticleBox* box;
						
					box = dynamic_cast< IParticleBox* >( obj );				
					HandleParticlePropertyChange( box, prop_index, linkProp->data.value );
				}
			}

			linkProp = linkProp->next;
		}

		link = link->next;
	}

	if (particleList.GetSize() > 0)
		gInterface->ForceCompleteRedraw();
}

void PropUpdater::ForceParticleBoxesToDefault(ScriptIniParser* pScriptIniParser)
{ FUNC_ENTER("PropUpdater::ForceParticleBoxesToDefault"); 
	INodeTab nodeList;
	GetAllNodes(nodeList);

	for(int i = 0; i < nodeList.Count(); i++)
	{
		Object* obj = nodeList[i]->EvalWorldState(0).obj;

		if (obj->ClassID() == PARTICLE_BOX_CLASS_ID)
		{
			CStr propBuffer;
			nodeList[i]->GetUserPropBuffer(propBuffer);

			CStr className = GetClassName(propBuffer);
			CStr typeName  = GetTypeName(propBuffer);

			LinkList<ConfigProp> cprops, DEFprops;
			LinkList<ConfigScript> cscripts;
			DWORD cflags;
			CStr strUnk, strClusterBuf;
			LinkList<ConfigProgram> progs;

			CStr strScript = ParseConfigProps(&cprops,&cflags,propBuffer,&strUnk,&strClusterBuf,&progs,&cscripts);
			pScriptIniParser->ConvertToDefaultDelim(className, typeName, &cprops);

			DumpNode(nodeList[i],
					 className,
					 typeName,
					 cflags,
					 &cprops,
					 strUnk,
					 strScript,
					 strClusterBuf,
					 &progs,
					 &cscripts);
		}
	}
}

void PropUpdater::ProcNodes()
{ FUNC_ENTER("PropUpdater::ProcNodes"); 
	particleList.Clear();

	Link<INode*>* link = updateList.GetHead();
	bool changed_particle = false;

	while(link)
	{
		ULONG nodeID = link->data->GetHandle();

		if (nodeID <= lastNodeID)
			return;

		CStr propBuffer;
		link->data->GetUserPropBuffer(propBuffer);

		bool IsPUInstance;

		if (IsDlgButtonChecked(hwnd, IDC_PROCINSTANCES) == BST_CHECKED)
			IsPUInstance = (strstr(propBuffer, "PUInstance = TRUE")) ? true : false;
		else
			IsPUInstance = false;

		if (propBuffer.Length()>0)
		{
			CStr className = GetClassName(propBuffer);
			CStr typeName  = GetTypeName(propBuffer);

			LinkList<ConfigProp> cprops, DEFprops;
			LinkList<ConfigScript> cscripts;
			DWORD cflags;
			CStr strUnk, strClusterBuf;
			LinkList<ConfigProgram> progs;

			CStr strScript = ParseConfigProps(&cprops,&cflags,propBuffer,&strUnk,&strClusterBuf,&progs,&cscripts);
			GetDefaultProps(&DEFprops,className,typeName);

			Link<ConfigProp>* defprop = DEFprops.GetHead();
			Link<ConfigProp>* curprop;

			CStr ctrBuf = GetClassTypeRecord(className, typeName);

			if (bUpdateTriggerColors)
			{
				//CStr propBuffer = GetClassTypeRecord(className,typeName);
				COLORREF color = ParseColor(ctrBuf);

				if (color != 0)
					link->data->SetWireColor(color);
			}

			// Update program type changes if a program was changed from static to volatile
			// or vice-versa
			LinkList<ConfigProgram> progsDEF;
			ParseConfigProps(NULL,NULL,ctrBuf,NULL,NULL,&progsDEF, &cscripts);

			Link<ConfigProgram>* prglink = progs.GetHead();

			while(prglink)
			{
				Link<ConfigProgram>* defprog = progsDEF.Find(&prglink->data);

				if (defprog)
					prglink->data.type = defprog->data.type;

				prglink = prglink->next;
			}

			// Update the names of default properties to correspond to category name changes (TT275)
			CStr defName, propName;
			char* loc;

			// Update default property categories
			while(defprop)
			{
				// Strip name of categorization
				if (loc = strstr(defprop->data.name,"/"))
					defName = loc + 1;
				else
					defName = defprop->data.name;

				curprop = cprops.GetHead();
				
				while(curprop)
				{
					// Strip name of categorization
					if (loc = strstr(curprop->data.name,"/"))
						propName = loc + 1;
					else
						propName = curprop->data.name;

					if (defName == propName)
					{
						// Update the property name to its corresponding default category
						curprop->data.name = defprop->data.name;
					}

					curprop = curprop->next;
				}

				defprop = defprop->next;
			}

			defprop = DEFprops.GetHead();

			// Ensure that all the default properties exist in the node
			// and that all the properties exist in the same default order
			// as defined in scripts.ini
			LinkList<ConfigProp> cprops2;
			
			while(defprop)
			{
				curprop = cprops.Find(&defprop->data);

				// If the property doesn't exist, add it
				if (!curprop)
				{
					ConfigProp newProp = defprop->data;
					newProp.value = CStr("!");
					//cprops.Add(&newProp);
					//cprops2.Add(&newProp);		
					cprops2.AddUnique(&newProp);
				}
				else
				{
					if (bAddNewProps)
						cprops2.AddUnique(&curprop->data);
						//cprops2.Add(&curprop->data);
				}

				defprop = defprop->next;
			}

			// Now add all the properties that weren't defined
			Link<ConfigProp>* linkRetainedProp = cprops.GetHead();

			while(linkRetainedProp)
			{
				if (!cprops2.Find(&linkRetainedProp->data))
				{
					ConfigProp newProp = linkRetainedProp->data;
					cprops2.AddUnique(&newProp);
					//cprops2.Add(&newProp);
				}

				linkRetainedProp = linkRetainedProp->next;
			}

			// If a property has been omitted in the default data, remove it
			if (bUpdateOmissions)
			{
				LinkList<CStr> omitList;

				// TODO: Store this information parsed with the rest of the default data
				//       So it doesn't have to be reconstructed for each node in the scene
				//		 during the update process

				GetOmissions(className, typeName, &omitList);
				
				Link<CStr>* curlink = omitList.GetHead();

				while(curlink)
				{
					ConfigProp        cprop;
					Link<ConfigProp>* curprop;
					
					cprop.name = curlink->data;
					curprop = cprops2.Find(&cprop);

					if (curprop)
						cprops2.Remove(curprop);

					curlink = curlink->next;
				}
			}

			if (bDeleteRemovedDefaultProps)
			{
				// Ensure that all the properties for this object exist in
				// the default property list if the property is defaulted
				// if it doesn't, remove it

				Link<ConfigProp>* curprop = cprops2.GetHead();
				Link<ConfigProp>* nextprop;

				while(curprop)
				{
					nextprop = curprop->next;

					if (curprop->data.value == CStr("!"))
					{
						if (!DEFprops.Find(&curprop->data))
							cprops2.Remove(curprop);
					}

					curprop = nextprop;
				}
			}

			// Any properties not contained in scripts.ini will be discarded 
			// (POTENTIALLY VERY DANGEROUS, user defined props will be destroyed)
			if (bDeleteAllOld)
			{
				Link<ConfigProp>* curprop = cprops2.GetHead();
				Link<ConfigProp>* nextprop;

				while(curprop)
				{
					nextprop = curprop->next;

					if (!DEFprops.Find(&curprop->data))
						cprops2.Remove(curprop);

					curprop = nextprop;
				}
			}

			// Handle updating particle system node defaults
			Object* obj = link->data->EvalWorldState(0).obj;

			if (obj->ClassID() == PARTICLE_BOX_CLASS_ID)
			{
				particleList.Add(&link->data);
				ConvertToDefaultDelim(className, typeName, &cprops2);
			}

			DumpNode(link->data,
					 className,
					 typeName,
					 cflags,
					 &cprops2,
					 strUnk,
					 strScript,
					 strClusterBuf,
					 &progs,
					 &cscripts);

			// If this is a property updater instance, then we'll need to copy the new nextparticle metacommands over
			if (IsPUInstance)
			{
				Point3 ptMid, ptEnd, ptMidScale, ptEndScale;
				float  fMidWidth, fMidHeight, fMidLength, fEndWidth, fEndHeight, fEndLength;

				GetParticleOrientation(ctrBuf, &ptMid, &ptEnd, &fMidWidth,
					                                           &fMidHeight,
															   &fMidLength,
															   &fEndWidth,
															   &fEndHeight,
															   &fEndLength,
															   &ptMidScale,
															   &ptEndScale);

				SetParticleOrientation(link->data, ptMid, ptEnd, fMidWidth,
					                                             fMidHeight,
																 fMidLength,
																 fEndWidth,
																 fEndHeight,
																 fEndLength,
																 ptMidScale,
																 ptEndScale);
			}
		}

		pProgress->SetVal(pProgress->GetVal()+1);		

		link = link->next;
	}

	// Delete any nodes flagged for deletion (if appropriate)
	if (bDiscardRemovedClasses)
	{
		Link<INode*>* link = delList.GetHead();

		while(link)
		{
			gInterface->DeleteNode(link->data, FALSE);
			link = link->next;
		}

		delList.Clear();
	}

	if (changed_particle)
		gInterface->ForceCompleteRedraw();
}

// For backward compat.
void PropUpdater::ProcNode(INode* parentNode)
{ FUNC_ENTER("PropUpdater::ProcNode"); 
	INodeTab nodeList;
	GetAllNodes(nodeList);

	updateList.Clear();
	particleList.Clear();

	int cnt = nodeList.Count();

	for(int i = 0; i < cnt; i++)
		updateList.Add(&nodeList[i]);

	ProcNodes();
}

bool PropUpdater::DoUpdate(WIN32_FIND_DATA timestampDest)
{ FUNC_ENTER("PropUpdater::DoUpdate"); 
	ReferenceTarget* scene = ip->GetScenePointer();

	int bResult;

	if (!bForceScriptUpdates)
		bResult = MessageBox(ip->GetMAXHWnd(),"The scripts.ini file has been updated.  Would you like to update the property data in existing nodes?","Scripts.ini Updated",MB_ICONQUESTION|MB_YESNO);
	else
		bResult = IDYES;

	if (bResult==IDNO)
	{
		//int bResult2 = MessageBox(ip->GetMAXHWnd(),"Do you want to continue to be prompted about this?","Scripts.ini Updated",MB_ICONQUESTION|MB_YESNO);

		//if (bResult2==IDYES)
			return false;
	}

	WIN32_FIND_DATA* newFindData = (WIN32_FIND_DATA*)malloc(sizeof(WIN32_FIND_DATA));
	*newFindData = timestampDest;

	scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_SCRIPTS_INI_TIMESTAMP_ID);
	scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_SCRIPTS_INI_TIMESTAMP_ID,
						   sizeof(WIN32_FIND_DATA),newFindData);	

	if (bResult==IDYES)
	{
		/*
		int bResult;

		if (!bForceScriptUpdates)
			bResult = MessageBox(ip->GetMAXHWnd(),"Do you want to also update trigger node colors to match scripts.ini defaults?","Scripts.ini Updated",MB_ICONQUESTION|MB_YESNO);
		else
			bResult = IDYES;

		if (bResult == IDYES)
			bUpdateTriggerColors = true;
		else
			bUpdateTriggerColors = false;

		if (!bForceScriptUpdates)
			bResult = MessageBox(ip->GetMAXHWnd(), "Do you want to sync newly added omissions?  Note: This will irrevokably remove data now flagged as omitted in scripts.ini!", "Sync Omissions", MB_ICONQUESTION|MB_YESNO);
		else
			bResult = IDYES;

		if (bResult == IDYES)
			bUpdateOmissions = true;
		else
			bUpdateOmissions = false;

		if (!bForceScriptUpdates)
			bResult = MessageBox(ip->GetMAXHWnd(), "Do you want to remove old defaulted properties that no longer exist in scripts.ini?", "Remove Old Defaulted Properties", MB_ICONQUESTION|MB_YESNO);
		else
			bResult = IDYES;

		if (bResult == IDYES)
			bDeleteRemovedDefaultProps = true;
		else
			bDeleteRemovedDefaultProps = false;
		*/

		return true;
	}

	return false;	
}

bool PropUpdater::ShouldUpdate(bool bForce)
{ FUNC_ENTER("PropUpdater::ShouldUpdate"); 
	ReferenceTarget* scene = ip->GetScenePointer();
	AppDataChunk* appdata;

	HANDLE handle;
	WIN32_FIND_DATA timestampSrc, timestampDest;

	appdata = scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_SCRIPTS_INI_TIMESTAMP_ID);

	if (appdata && appdata->data)
	{
		timestampSrc = *((WIN32_FIND_DATA*)appdata->data);

		handle = FindFirstFile( GetScriptIniName(), &timestampDest );  

		if (bForce)
		{
			FindClose(handle);

			// Second Chance Cache Check
			// If scripts.ini is getting updated force the script and node caches to reset
			PurgeScriptCache();
			PurgeNodeCache();

			return DoUpdate(timestampDest);
		}

		if ( CompareFileTime( &timestampDest.ftLastWriteTime, &timestampSrc.ftLastWriteTime ) > 0 )
		{
			FindClose(handle);

			// Second Chance Cache Check
			// If scripts.ini is getting updated force the script and node caches to reset
			PurgeScriptCache();
			PurgeNodeCache();
			
			return DoUpdate(timestampDest);
		}

		FindClose(handle);
		return false;
	}

	handle = FindFirstFile( GetScriptIniName(), &timestampSrc );

	WIN32_FIND_DATA* newFindData = (WIN32_FIND_DATA*)malloc(sizeof(WIN32_FIND_DATA));
	*newFindData = timestampSrc;

	scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_SCRIPTS_INI_TIMESTAMP_ID);
	scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_SCRIPTS_INI_TIMESTAMP_ID,
		                   sizeof(WIN32_FIND_DATA),newFindData);	

	if (bForce)
	{
		// Second Chance Cache Check
		// If scripts.ini is getting updated force the script and node caches to reset
		PurgeScriptCache();
		PurgeNodeCache();

		return DoUpdate(timestampSrc);
	}

	return false;
}

void PropUpdater::ProcChanges(bool bForce)
{ FUNC_ENTER("PropUpdater::ProcChanges"); 
	if (bCancel)
	{
		bCancel = false;
		return;
	}
	
	// If you are flagged to export animations or skin properties
	// should not be updateable
	ExportOptions* expOptions = GetExportOptions();

	if (expOptions->m_ExportType == ExportOptions::vEXPORT_ANIM ||
		expOptions->m_ExportType == ExportOptions::vEXPORT_SKIN)
		return;

	/*
	// If the file doesn't exist, copy it
	CStr strFilename = ip->GetDir(APP_PLUGCFG_DIR);
	strFilename += "\\scripts.bak";
	FILE* fp;

	fp = fopen(strFilename,"r");

	if (!fp)
	{
		// Our backup file doesn't exist, create it and return (no property updates necessary)
		CopyFile(GetScriptIniName(),strFilename,FALSE);
		return;
	}

	fclose(fp);

	if (FileIsNewer(strFilename,GetScriptIniName()))
	*/

	CStr appDir=ip->GetDir(APP_PLUGCFG_DIR);
	appDir+="\\PropEdit.ini";

	if (GetPrivateProfileInt("Settings","ForceScriptUpdates",0,(char*)appDir)==1)
		bForceScriptUpdates = TRUE;
	else
		bForceScriptUpdates = FALSE;

	bool bRetry;

	if (ShouldUpdate(bForce))
	{
		ParseScriptIni();

		do
		{
			bRetry = false;
			Show();

			if (bDeleteAllOld)
			{
				int val = MessageBox(gInterface->GetMAXHWnd(), "WARNING! Delete All Old Properties is set!  This option will destroy any manually added user property data that does not exist in scripts.ini.  Data will be unrecoverable.  Are you absolutely sure you want to do this?",  "Delete All Old Properties", MB_ICONWARNING|MB_YESNO);

				if (val == IDNO)
				{
					bRetry = true;
					continue;
				}
			}

			if (bDiscardRemovedClasses)
			{
				int val = MessageBox(gInterface->GetMAXHWnd(), "WARNING! Discard Nodes with deleted classes is set!  This option will delete nodes assigned to any class/type names not in scripts.ini from the scene.  Data will be unrecoverable.  Are you absolutely sure you want to do this?", "Discard nodes with deleted classes", MB_ICONWARNING|MB_YESNO);

				if (val == IDNO)
				{
					bRetry = true;
					continue;
				}
			}

		} while (bRetry);

		//int numNodes = CountNodes(NULL,0);
		int numNodes = updateList.GetSize();
		pProgress = new ProgressBar(hInstance,hwndParent,"Updating Node Properties",0,numNodes);
		ProcNodes();

		delete pProgress;
	}

	DumpClassTypeCRCs();
}

// In the event that the Default class was modified then all properties must be updated
// This resets the class type list and fills it with all properties
void PropUpdater::PopulateClassTypeListAll()
{ FUNC_ENTER("PropUpdater::PopulateClassTypeListAll"); 
	// Classes appear blue, types appear green
	classUpdateList.Clear();
	typeUpdateList.Clear();

	pCTList->Clear();

	Link<ConfigClass>* link = configDB.GetHead();

	while(link)
	{
		Link<ConfigType>* linkType = link->data.types.GetHead();

		if (!linkType)
		{
			// This is a class only entry
			int idx = pCTList->AddItem(link->data.name, COLOR_CLASS);
			ConfigClass* pcclass = &link->data;
			Link<ConfigClass*>* updateLink = classUpdateList.Add(&pcclass);
			pCTList->SetItemData(idx, (DWORD)updateLink);
		}

		while(linkType)
		{
			int idx = pCTList->AddItem(link->data.name + CStr("-") + linkType->data.name, COLOR_TYPE);
			ConfigType* pctype = &linkType->data;
			Link<ConfigType*>* updateLink = typeUpdateList.Add(&pctype);
			pCTList->SetItemData(idx, (DWORD)updateLink);

			linkType = linkType->next;
		}

		link = link->next;
	}
}

void PropUpdater::PopulateClassTypeList()
{ FUNC_ENTER("PropUpdater::PopulateClassTypeList"); 
	// Classes appear blue, types appear green
	classUpdateList.Clear();
	typeUpdateList.Clear();

	pCTList->Clear();

	Link<ConfigClass>* link = configDB.GetHead();

	while(link)
	{
		Link<ConfigType>* linkType = link->data.types.GetHead();

		if (!linkType)
		{
			// This is a class only entry
			unsigned long* pOldCRC = crcTable.GetItem(GenerateCRC(link->data.name));
			unsigned long   newCRC = GenerateCRC(GetClassTypeRecord(link->data.name));

			if (!(pOldCRC && *pOldCRC == newCRC))
			{
				/*
				if (strstr(link->data.name, "Default"))
				{
					PopulateClassTypeListAll();
					return;
				}
				*/

				int idx = pCTList->AddItem(link->data.name, COLOR_CLASS);
				ConfigClass* pcclass = &link->data;
				Link<ConfigClass*>* updateLink = classUpdateList.Add(&pcclass);
				pCTList->SetItemData(idx, (DWORD)updateLink);
			}
		}

		while(linkType)
		{
			unsigned long* pOldCRC = crcTable.GetItem(GenerateCRC(link->data.name + CStr("-") + linkType->data.name));
			unsigned long   newCRC = GenerateCRC(GetClassTypeRecord(link->data.name, linkType->data.name));

			// Only add items to the list that have been updated
			if (!(pOldCRC && *pOldCRC == newCRC))
			{
				int idx = pCTList->AddItem(link->data.name + CStr("-") + linkType->data.name, COLOR_TYPE);
				ConfigType* pctype = &linkType->data;
				Link<ConfigType*>* updateLink = typeUpdateList.Add(&pctype);
				pCTList->SetItemData(idx, (DWORD)updateLink);
			}

			linkType = linkType->next;
		}

		link = link->next;
	}
}

void PropUpdater::BuildSelLists()
{ FUNC_ENTER("PropUpdater::BuildSelLists"); 
	selClassList.Clear();
	selTypeList.Clear();

	// Node list is populated by finding all the nodes that have
	// been flagged for update from the class/type list
	int cnt = pCTList->GetCount();

	for(int j = 0; j < cnt; j++)
	{
		if (pCTList->IsSelected(j))
		{
			// If the color is blue we know its a Class rather than type
			if (pCTList->GetItemColor(j) == COLOR_CLASS)
			{
				ConfigClass* pcclass = ((Link<ConfigClass*>*)pCTList->GetItemData(j))->data;
				selClassList.Add(&pcclass);
			}

			if (pCTList->GetItemColor(j) == COLOR_TYPE)
			{
				ConfigType* pctype = ((Link<ConfigType*>*)pCTList->GetItemData(j))->data;
				selTypeList.Add(&pctype);
			}
		}
	}
}

void PropUpdater::PopulateNodeList()
{ FUNC_ENTER("PropUpdater::PopulateNodeList"); 
	BuildSelLists();

	INodeTab     nodeList;
	CStr         propBuffer, className, typeName;
	ConfigClass* pcclass;
	ConfigType*  pctype;

	GetAllNodes(nodeList);

	pNList->Clear();

	for(int i = 0; i < nodeList.Count(); i++)
	{
		// Trigger links are excluded from property updates
		Object* obj = nodeList[i]->EvalWorldState(0).obj;

		if (obj->ClassID() == vLINK_OBJ_CLASS_ID)
			continue;

		// If this node contains one of the listed class/types for update it should be added
		// to the node update list
		
		nodeList[i]->GetUserPropBuffer(propBuffer);
		className = GetClassName(propBuffer);
		typeName = GetTypeName(propBuffer);

		pcclass = GetConfigClass(className);
		
		if (!pcclass)
		{
			// The class no longer exists in scripts.ini
			// (Item gets added to the list but, appears in black)
			pNList->AddItem(nodeList[i]->GetName(),COLOR_REMOVED,(DWORD)nodeList[i]);
			continue;
		}

		if (typeName.Length() > 0)
		{
			pctype = GetConfigType(pcclass, typeName);

			if (!pctype)
			{
				// The type no longer exists in scripts.ini
				// (Item gets added to the list but, appears in black)
				pNList->AddItem(nodeList[i]->GetName(),COLOR_REMOVED,(DWORD)nodeList[i]);
				continue;
			}

			// Check if the type exists within the current class/type selection set
			if (!selTypeList.Find(&pctype))
				continue;
		}
		else
		{
			// This is a class only prop check for it in he selection class list
			if (!selClassList.Find(&pcclass))
				continue;
		}

		// At this point we know the class/type does exist within the current class/type selection set
		// and it can be added to the node list
		if (pctype)
			pNList->AddItem(nodeList[i]->GetName(),COLOR_TYPE,(DWORD)nodeList[i]);
		else
			pNList->AddItem(nodeList[i]->GetName(),COLOR_CLASS,(DWORD)nodeList[i]);
	}
}

int PropUpdater::GetTotalEntries()
{ FUNC_ENTER("PropUpdater::GetTotalEntries"); 
	Link<ConfigClass>* link = configDB.GetHead();
	int size = 0;
	int nTypes;

	while(link)
	{
		nTypes = link->data.types.GetSize();
		size += nTypes;

		// If there are no types for this class we still need
		// to add an entry so there's space for the class info
		if (nTypes == 0)
			size++;

		link = link->next;
	}

	return size;
}

void PropUpdater::DumpClassTypeCRCs()
{ FUNC_ENTER("PropUpdater::DumpClassTypeCRCs"); 
	INode* node = ip->GetRootNode();
	int nEntries = GetTotalEntries();

	unsigned char* pData = (unsigned char*)malloc(nEntries * sizeof(unsigned long) * 2);
	unsigned char* pos = pData;
	node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CLASSTYPE_CRC_STATE);

	// Build the CRC list
	Link<ConfigClass>* link = configDB.GetHead();

	while(link)
	{
		Link<ConfigType>* linkType = link->data.types.GetHead();

		if (!linkType)
		{
			// This is a class only entry
			*((unsigned long*)pos) = GenerateCRC(link->data.name);
			pos += sizeof(unsigned long);
			*((unsigned long*)pos) = GenerateCRC(GetClassTypeRecord(link->data.name));
			pos += sizeof(unsigned long);
		}

		while(linkType)
		{
			*((unsigned long*)pos) = GenerateCRC(link->data.name + CStr("-") + linkType->data.name);
			pos += sizeof(unsigned long);
			*((unsigned long*)pos) = GenerateCRC(GetClassTypeRecord(link->data.name, linkType->data.name));
			pos += sizeof(unsigned long);

			linkType = linkType->next;
		}

		link = link->next;
	}

	node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CLASSTYPE_CRC_STATE, nEntries * sizeof(unsigned long) * 2, pData);
}

void PropUpdater::LoadClassTypeCRCs()
{ FUNC_ENTER("PropUpdater::LoadClassTypeCRCs"); 
	INode* node = ip->GetRootNode();

	crcTable.FlushAllItems();
	crcList.Clear();

	AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CLASSTYPE_CRC_STATE);
	if (appdata && appdata->data)
	{
		int nEntries = appdata->length / (sizeof(unsigned long) * 2);
		unsigned char* pos = (unsigned char*)appdata->data;

		unsigned long crcName, crcBuffer;
		Link<unsigned long>* link;

		for(int i = 0; i < nEntries; i++)
		{
			crcName = *((unsigned long*)pos);
			pos += sizeof(unsigned long);
			crcBuffer = *((unsigned long*)pos);
			pos += sizeof(unsigned long);

			// Add the entry to the CRC list and hashtable
			link = crcList.Add(&crcBuffer);
			crcTable.PutItem(crcName, &link->data);
		}
	}
}

void PropUpdater::SelAll(int id)
{ FUNC_ENTER("PropUpdater::SelAll"); 
	int cnt = SendDlgItemMessage(hwnd, id, LB_GETCOUNT, 0, 0);
	SendDlgItemMessage(hwnd, id, LB_SELITEMRANGE, (WPARAM)(BOOL)TRUE, MAKELPARAM(0, cnt));
}

void PropUpdater::SelNone(int id)
{ FUNC_ENTER("PropUpdater::SelNone"); 
	int cnt = SendDlgItemMessage(hwnd, id, LB_GETCOUNT, 0, 0);
	SendDlgItemMessage(hwnd, id, LB_SELITEMRANGE, (WPARAM)(BOOL)FALSE, MAKELPARAM(0, cnt));	
}

void PropUpdater::SelInvert(int id)
{ FUNC_ENTER("PropUpdater::SelInvert"); 
	int cnt = SendDlgItemMessage(hwnd, id, LB_GETCOUNT, 0, 0);

	for(int i = 0; i < cnt; i++)
	{
		if (SendDlgItemMessage(hwnd, id, LB_GETSEL, (WPARAM)i, 0))
			SendDlgItemMessage(hwnd, id, LB_SETSEL, (WPARAM)(BOOL)FALSE, (LPARAM)(UINT)i);
		else
			SendDlgItemMessage(hwnd, id, LB_SETSEL, (WPARAM)(BOOL)TRUE, (LPARAM)(UINT)i);
	}
}

void PropUpdater::BuildUpdateList()
{ FUNC_ENTER("PropUpdater::BuildUpdateList"); 
	updateList.Clear();
	delList.Clear();

	int cnt = pNList->GetCount();

	for(int i = 0; i < cnt; i++)
	{
		if (pNList->IsSelected(i))
		{
			INode* node = (INode*)pNList->GetItemData(i);

			if (pNList->GetItemColor(i) == COLOR_REMOVED)
				delList.Add(&node);
			else
				updateList.Add(&node);
		}
	}
}

void PropUpdater::DoFullUpdate()
{ FUNC_ENTER("PropUpdater::DoFullUpdate"); 
	bAddNewProps               = true;
	bDeleteAllOld              = false;
	bDeleteRemovedDefaultProps = true;
	bUpdateOmissions           = true;
	bDiscardRemovedClasses     = false;
	bUpdateTriggerColors       = false;

	SelAll(IDC_CTLIST);
	PopulateNodeList();
	SelAll(IDC_NLIST);
	BuildUpdateList();
}
