/*
	PropEdit.cpp
	Adam Lippmann
	12-12-01 

	See PropEditMSearch.cpp MultiSearch methods
*/

#include "PropEdit.h"
#include "ParseFuncs.h"
#include "Resource.h"
#include "Link/LinkUI.h"
#include "FindInFiles.h"
#include "Trigger/Trigger.h"
#include "PropUpdater.h"
#include "LODBrowser.h"
#include "../UI/MultiList.h"
#include "../UI/MultiPropList.h"
#include "ReplaceInNodes.h"
#include "PropMerge.h"
#include "ClusterFind.h"
#include <stdio.h>
#include <core/StringHashTable.h>
#include "../misc/BrandCheck.h"
#include "../PropEdit/CmdTypes.h"
#include "../Particle/NExtParticle.h"
#include "../UI/InputDlg.h"
#include "../UI/OKtoAll.h"
#include "../misc/util.h"
#include <io.h>
#include "UserConfig.h"

#define NO_PREINCLUDES
#define NO_BASEPREINCLUDES
#include "FuncEnter.h"

#ifndef WRITE_ACCESS
#define  WRITE_ACCESS     0x02
#endif

#define PROPFLAG_NEWPROP        0x01
#define FLOATPROP_WIDTH         300
#define FLOATPROP_HEIGHT        263
#define PROPSTART                 0
//#define PROPSTART               5
//#define PROP_CREATEDATSTART     0
//#define PROP_ABSENTINNETGAMES   1
//#define PROP_TRICKOBJECT        2
//#define PROP_CLUSTER            3
//#define PROP_OCCLUDER	        4

#define VK_L 0x4C
#define VK_P 0x50

#define DEBUGDUMP(x) { FILE* fp = fopen("c:\\debugdump.txt","w"); fprintf(fp, "%s\n", (char*)x); fclose(fp); }

static HWND    hwndPropEdit;
static HWND    hwndLOD;

bool bParticleUpdateLock = false;

extern bool gbLockPMU;
extern bool gbLockPosChange;
extern BOOL gBackgroundProcesses;

class ParticleEnum: public DependentEnumProc
{
public:
	Tab<ReferenceMaker*> reflist;
	
	int proc(ReferenceMaker *rmaker)
	{
		reflist.Append(1,&rmaker);
		return DEP_ENUM_CONTINUE;
	}
};


bool IsChildOfPE(HWND hwnd)
{ FUNC_ENTER("IsChildOfPE"); 
	if (hwnd == hwndPropEdit)
		return true;

	if (hwnd == NULL)
		return false;

	return IsChildOfPE(GetParent(hwnd));
}

bool IsChildOfLOD(HWND hwnd)
{ FUNC_ENTER("IsChildOfLOD"); 
	if (hwnd == hwndLOD)
		return true;

	if (hwnd == NULL)
		return false;

	return IsChildOfLOD(GetParent(hwnd));
}

BOOL CALLBACK EnumChildProc(HWND hwnd,LPARAM lParam)
{ FUNC_ENTER("EnumChildProc"); 
	PropEditor* pthis=(PropEditor*)lParam;
	pthis->RegisterWnd(hwnd);

	return TRUE;
}

void CALLBACK PropEditor::TimerProc(HWND hwnd,UINT msg,UINT id,DWORD time)
{ FUNC_ENTER("PropEditor::TimerProc"); 
	// Reclaim a pointer to the property editor from the window
	PropEditor* pPropEdit = (PropEditor*)GetWindowLong(hwnd,GWL_USERDATA);

	if (pPropEdit)
	{
		if (GetAsyncKeyState(VK_CONTROL) &&
			GetAsyncKeyState(VK_L))
		{
			if (hwndLOD)
				SetFocus(hwndLOD);
		}

		if (GetAsyncKeyState(VK_CONTROL) &&
			GetAsyncKeyState(VK_P))
		{
			if (hwndPropEdit)
				SetFocus(hwndPropEdit);
		}
	}
}

PropEditor::PropEditor(HINSTANCE hInstance,Interface* ip) :
	ttip(hInstance), addCB(ip), SubClassWndMan(SubWndReg)
{ FUNC_ENTER("PropEditor::PropEditor"); 
	this->hInstance=hInstance;
	this->ip=ip;

	bLockSelection=FALSE;		// Accept selection updates
	bLockPropBuffer=FALSE;		// Accept property buffer copy requests
	bLockParamComplete=FALSE;
	bLockClassUpdate=FALSE;
	bLockSelectAdd=FALSE;
	
	bAutoSelect=FALSE;			// Auto selection defaults to off
	bChangeMade=FALSE;			// No changes have been made yet
	bApplyOnClose=FALSE;		// True if auto applying changes when the property editor closes
	bApplyChangedOnly=TRUE;		// Only apply changes if the field was changed
	bAddConflict=TRUE;			// Add new fields when changing accross conflicts
	bForceScriptUpdates=FALSE;	// True if scripts.ini updates should be forced when they occur

	bSubSelApply=FALSE;
	bSubSelRestore=TRUE;
	bPropertyAutoFill=TRUE;		// True if properties for default class are auto filled in on blank nodes
								// This needs disabled by SRLT because the property editor will assign its
								// default class before SRLT can assign the appropriate properties to the object

	bImmediateApply=FALSE;		// True if properties are applied immediately whenever the data in a property
								// field has been changed

	bRefreshViewport=FALSE;

	bLockImmediateApply=FALSE;

	pPopupParams=NULL;
	pDefaultProps=NULL;
	pMSearch=NULL;
	pScriptBrowse=NULL;
	pFindNode=NULL;

	fpCloseNotify=NULL;
	pCloseNotifyData=NULL;

	bClassConflict=FALSE;
	bTypeConflict=FALSE;
	bChangeMade=FALSE;

	bDisplayTree=FALSE;

	bMapUnloaded=FALSE;

	bNoTreeview=FALSE;

	bSyntaxHighlighting=TRUE;
	bLockImmediateApply=TRUE;

	ResetModFlags();

	flags = 0;
	bOmit = false;

	LoadLibrary("RichEd32.dll");

	// PropView must be created before the window so the PropList class is registered with Windows
	//propView=new PropList(hInstance,2);
	propView = new MultiPropList(hInstance,2);

	// Create the class/type browser multilist controls
	classBrowser=new MultiList(hInstance);
	typeBrowser=new MultiList(hInstance);

	classBrowser->DispFullPath(true);
	typeBrowser->DispFullPath(true);
	classBrowser->EvalFullPath(true);
	typeBrowser->EvalFullPath(true);
	//classBrowser->SetOutDelim("/");
	//typeBrowser->SetOutDelim("/");

	hwnd=CreateDialog(hInstance,MAKEINTRESOURCE(IDD_PROPEDIT),ip->GetMAXHWnd(),RedirectDlgProc);
	//hwnd=CreateDialog(hInstance,MAKEINTRESOURCE(IDD_PROPEDIT2),ip->GetMAXHWnd(),RedirectDlgProc);
	SetWindowLong(hwnd,GWL_USERDATA,(LONG)this);

	classBrowser->Attach(GetDlgItem(hwnd,IDC_OBJCLASSM));
	typeBrowser->Attach(GetDlgItem(hwnd,IDC_OBJTYPEM));

	pPropUpdater = new PropUpdater(hInstance,ip->GetMAXHWnd(),ip);

	pPopupParams=new PropList(hInstance,hwnd,0,0);

	pFindNode=new FindNodeDlg(ip,hInstance,hwnd);
	pCompareDlg=new PropCompareDlg(hInstance,hwnd);
	pAddPropDlg=new AddPropDlg(hInstance,hwnd);
	
	pExtScriptDlg=new ExtScriptDlg(hInstance,hwnd);
	pRenameDlg=new NodeRenameDlg(hInstance,hwnd);

	pClusterFindDlg=new ClusterFindDlg(hInstance, hwnd);

	// Attach the property view window
	propView->UIControl::Attach(GetDlgItem(hwnd,IDC_PROPVIEW));
	propView->HasApply(false);
	propView->SetChangeCB(PropViewChangeCB,this);
	propView->SetEscapeCB(EscapeCB,this);
	propView->SetDblClickCB(PropViewDblClickCB,this);
	propView->SetStaticCB(SubPropProc,this);

	// Make this an "Always on Top" window
	//SetWindowPos(hwnd,HWND_TOPMOST,0,0,0,0,SWP_NOMOVE|SWP_NOSIZE);

	// Initialize the TreeView control
	pTreeView=new TreeView(hInstance,GetDlgItem(hwnd,IDC_TREE1));
	pRichText=new RichText(hInstance,GetDlgItem(hwnd,IDC_RICHEDIT1));

	pFindFilesDlg = new FindInFilesDlg(hInstance,hwnd,this,pRichText);
	pReplaceInNodesDlg = new ReplaceInNodesDlg(hInstance,hwnd,ip);

	pTVIScripts     =pTreeView->AddField("Scripts");
	pTVIIntergers   =pTreeView->AddField("Integers");
	pTVIFloats      =pTreeView->AddField("Floats");
	pTVIVectors     =pTreeView->AddField("Vectors");
	pTVIPairs       =pTreeView->AddField("Pairs");
	pTVIStrings     =pTreeView->AddField("Strings");
	pTVILocalStrings=pTreeView->AddField("LocalStrings");
	pTVIArrays      =pTreeView->AddField("Arrays");
	pTVIStructs     =pTreeView->AddField("Structs");
	pTVINames       =pTreeView->AddField("Names");

	hMenuProps=LoadMenu(hInstance,MAKEINTRESOURCE(IDR_PROPPOPUP));
	hPropsPopupMenu=GetSubMenu(hMenuProps,0);

	
	RefreshScriptIni(false);

	// Set the default to environment object
	SetDlgItemText(hwnd,IDC_OBJCLASSM,DEFAULT_CLASS);

	LoadConfig();

	// Parse qcomp.map
	CStr strMapFile=getenv(APP_ENV);

	if (strMapFile==CStr(""))
	{
		char ErrorBuf[256];
		sprintf(ErrorBuf,"Couldn't load '%s' the '%s' environment variable is not set.",MAP_FILE,APP_ENV);

		MessageBox(NULL,ErrorBuf,"Property Editor",MB_ICONSTOP|MB_OK);
	}
	else
	{
		strMapFile+=CStr(SCRIPT_PATH)+MAP_FILE;
		ParseMapFile(strMapFile);
		ParseTagDBFile();
		//ParseMapFile("c:\\ParseInfo\\qcomp.map");
	}

	// Turn off Treeview mode if necessary
	if (bNoTreeview)
	{
		HMENU hMenuBar=GetMenu(hwnd);
		EnableMenuItem(hMenuBar, IDM_DISPTREEVIEW, MF_GRAYED);
	}

	pPropMergeDlg=new PropMergeDlg(hInstance,hwnd,&configDB,PropMergeCB,this);

	ParseUserProps();
	AcquireSelSet();

	// Subclass the Globals list
	OldGlobalProc=(WNDPROC)GetWindowLong(GetDlgItem(hwnd,IDC_GLOBALLIST),GWL_WNDPROC);
	SetWindowLong(GetDlgItem(hwnd,IDC_GLOBALLIST),GWL_WNDPROC,(LONG)SubGlobalProc);

	// Registering the dialog with MAX screws up tab and enter (bad MAX, bad)
	//ip->RegisterDlgWnd(hwnd);

	//if (bSubSelRestore)
	//	NodeListRestore();

	// Register windows
	RegisterWnd(GetDlgItem(hwnd,IDC_NODELIST));
	RegisterWnd(GetDlgItem(hwnd,IDC_TREE1));
	RegisterWnd(GetDlgItem(hwnd,IDC_GLOBALLIST));
	RegisterWnd(GetDlgItem(hwnd,IDC_FILTERLIST));
	RegisterWnd(GetDlgItem(hwnd,IDC_FILTERNAME));
	RegisterWnd(GetDlgItem(hwnd,IDC_FILTERDATA));
	//RegisterWnd(GetDlgItem(hwnd,IDC_OBJCLASSM));
	//RegisterWnd(GetDlgItem(hwnd,IDC_OBJTYPE));

	//EnumChildWindows(GetDlgItem(hwnd,IDC_OBJCLASS),EnumChildProc,(LPARAM)this);
	//EnumChildWindows(GetDlgItem(hwnd,IDC_OBJTYPE),EnumChildProc,(LPARAM)this);

	// We'll subclass the nodelist window again to popup tooltip windows
	HWND hwndNodeList = GetDlgItem(hwnd,IDC_NODELIST);
	OldNodeListProc = (WNDPROC)GetWindowLong(hwndNodeList,GWL_WNDPROC);
	SetWindowLong(hwndNodeList,GWL_WNDPROC,(LONG)ttipWndProc);

	// Intercept any text editor changes so we can store the updates
	// based on the context mode that the PE is set to
	RTFEditor::SetChangeCB(ProcEditChange,this);

	LODview = new LODBrowser(hInstance,ip->GetMAXHWnd(),ip,this);

	//SubclassMAXClasses();

	// I hate to setup keyboard shortcuts by polling the keyboard state every so often
	// but, unfortunately we can't use accelerators because they need to be registered
	// through MAX which would force us to register the window, and then MAX will overtake
	// the behavior of our controls.  Preventing rather critical features such as the
	// use of the TAB key to actually tab over in the edit window.
#ifndef DISABLE_PROPKEY_TIMER
	if (gBackgroundProcesses)
		SetTimer(hwnd,0,300,TimerProc);
#endif
}

PropEditor::~PropEditor()
{ FUNC_ENTER("PropEditor::~PropEditor"); 
	int size,i;
	TypeDesc  *curScript;
	KeywordDesc *curKeyword;

	//UnSubclassMAXClasses();

	SaveConfig();

	// De-subclass the node list
	HWND hwndNodeList = (HWND)GetWindowLong(hwnd,IDC_NODELIST);
	SetWindowLong(hwndNodeList,GWL_WNDPROC,(LONG)OldNodeListProc);

	// Uninstall notification callback
	UnRegisterSelChange();

	// Shutdown TreeView control
	delete pTreeView;
	
	// Shutdown RichText control
	delete pRichText;

	CloseChildren();

	if (pFindFilesDlg)
		delete pFindFilesDlg;

	if (pFindNode)
		delete pFindNode;

	if (pCompareDlg)
		delete pCompareDlg;

	if (pAddPropDlg)
		delete pAddPropDlg;

	if (pExtScriptDlg)
		delete pExtScriptDlg;

	if (pRenameDlg)
		delete pRenameDlg;

	if (propView)
		delete propView;

	if (pPopupParams)
		delete pPopupParams;

	if (pPropUpdater)
		delete pPropUpdater;

	if (LODview)
		delete LODview;

	if (classBrowser)
		delete classBrowser;

	if (typeBrowser)
		delete typeBrowser;

	if (pReplaceInNodesDlg)
		delete pReplaceInNodesDlg;

	if (pPropMergeDlg)
		delete pPropMergeDlg;

	if (pClusterFindDlg)
		delete pClusterFindDlg;

	// Clear out the contents of the hash tables
	size=scriptDB.GetSize();
	scriptDB.IterateStart();

	for(i=0;i<size;i++)
	{
		curScript=scriptDB.IterateNext();
		delete curScript;
	}

	size=keywordDB.GetSize();
	keywordDB.IterateStart();

	for(i=0;i<size;i++)
	{
		curKeyword=keywordDB.IterateNext();
		delete curKeyword;
	}

	// Kill the keyboard shorcut poller
	KillTimer(hwnd,0);
}

void PropEditor::RefreshScriptIni(bool bWarn)
{ FUNC_ENTER("PropEditor::RefreshScriptIni"); 
	if (bWarn && ip->GetSelNodeCount()>0)
	{
		int rVal=MessageBox(hwnd,"WARNING!  This operation will cause the current selection to be lost.\nAre you sure?","Refresh Script.ini",MB_ICONWARNING|MB_YESNO);

		if (rVal==IDNO)
			return;
	}

	pPropUpdater->ForceParticleBoxesToDefault(this);
	TriggerUI::DestroyRetainedData();

	ip->ClearNodeSelection();

	SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_RESETCONTENT,0,0);
	SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_RESETCONTENT,0,0);

	// Detect any changes to scripts.ini and then update any affected nodes
	pPropUpdater->ProcChanges();
	ParseScriptIni();
	pPropUpdater->ProcPostParticleChanges();

	UpdateClassList();
}

void PropEditor::UpdateClassList()
{ FUNC_ENTER("PropEditor::UpdateClassList"); 
	// Add the script data to the selection combo box
	Link<ConfigClass>* cfgClass=configDB.GetHead();

	SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_RESETCONTENT,0,0);

	while(cfgClass)
	{
		if (cfgClass->data.name!=CStr("Default"))
		{
			CStr itemName;
			
			//if (cfgClass->data.strGroup.Length()>0)
			//	itemName = cfgClass->data.strGroup + CStr("/") + cfgClass->data.name;
			//else
				itemName = cfgClass->data.name;

			//LONG index=SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_ADDSTRING,0,(LPARAM)(char*)cfgClass->data.name);
			bool bViewable = true;

			for(int i = 0; i < selSet.GetSize(); i++)
			{
				if (!IsViewable(selSet[i], itemName, ""))
				{
					bViewable = false;
					break;
				}
			}

			if (bViewable)
			{
				LONG index=SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_ADDSTRING,0,(LPARAM)(char*)itemName);
				SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_SETITEMDATA,index,(LPARAM)cfgClass);
			}
		}

		cfgClass=cfgClass->next;
	}
}

void PropEditor::RegisterSelChange()
{ FUNC_ENTER("PropEditor::RegisterSelChange"); 
	// Install notification callback
#ifndef DISABLE_NOTIFICATIONS
	int rVal=RegisterNotification(MAXSelChange,this,NOTIFY_SELECTIONSET_CHANGED);

	if (!rVal)
		MessageBox(NULL,"Couldn't Register Selection Callback","PropEditor",MB_ICONSTOP|MB_OK);
#endif
}

void PropEditor::UnRegisterSelChange()
{ FUNC_ENTER("PropEditor::UnRegisterSelChange"); 
	UnRegisterNotification(MAXSelChange,this,NOTIFY_SELECTIONSET_CHANGED);
}

void PropEditor::ResetModFlags()
{ FUNC_ENTER("PropEditor::ResetModFlags"); 
	bClassChanged=FALSE;
	bTypeChanged=FALSE;
	bScriptChanged=FALSE;
}

void AddCallback::proc(INodeTab &nodeTab)
{ FUNC_ENTER("AddCallback::proc"); 
	// Select all the additional nodes that were selected in the dialog
	for(int i=0;i<nodeTab.Count();i++)
		ip->SelectNode(nodeTab[i],0);
}

BOOL CALLBACK PropEditor::RedirectDlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("PropEditor::RedirectDlgProc"); 
	PropEditor* thisclass=(PropEditor*)GetWindowLong(hwnd,GWL_USERDATA);
	return thisclass->DlgProc(hwnd,msg,wParam,lParam);
}

LRESULT PropEditor::SubPropProc(PropList* plist,HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam,void* pData)
{ FUNC_ENTER("PropEditor::SubPropProc"); 
	POINT pt;
	PropEditor* pthis=(PropEditor*)pData;

	switch(msg)
	{
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDM_CONFLICTBROWSER:
			pthis->PropViewDblClickCB(plist,plist->GetPropFromHWnd(hwnd),pthis);
			return TRUE;

		case IDM_MODPROP:
			pthis->ModifyProp(plist,plist->GetPropFromHWnd(hwnd));
			return TRUE;

		case IDM_DELPROP:
			pthis->DeleteProp(plist,plist->GetPropFromHWnd(hwnd));
			return TRUE;

		case IDM_RESETDEFAULT:
			{
				int  index = plist->GetPropFromHWnd(hwnd);
				CStr strDefault = pthis->GetDefault(pthis->GetClass(),pthis->GetType(),plist->GetName(index));
				plist->SetValue(index,strDefault);
				plist->SetColor(index,PROPCOLOR_GREEN);
				plist->SetMod(index,true);
			}
			return TRUE;
		}

	case WM_RBUTTONDOWN:
		// Bring up the popup menu
		GetCursorPos(&pt);
		TrackPopupMenu(pthis->hPropsPopupMenu,TPM_RIGHTBUTTON,
			           pt.x,pt.y,
					   0,hwnd,NULL);
		return TRUE;
	}

	return FALSE;
}

LRESULT PropEditor::SubGlobalProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("PropEditor::SubGlobalProc"); 
	HWND        hwndPropEditor=GetParent(hwnd);
	PropEditor* pthis=(PropEditor*)GetWindowLong(hwndPropEditor,GWL_USERDATA);
	POINT       pt;
	int         index;
	int         count;
	int         i;
	char        buf[256];

	switch(msg)
	{
	case WM_RBUTTONDOWN:
		GetCursorPos(&pt);
		ScreenToClient(hwnd,&pt);
		index=SendMessage(hwnd,LB_ITEMFROMPOINT,0,(LPARAM)MAKELPARAM(pt.x,pt.y));
		count=SendMessage(hwnd,LB_GETCOUNT,0,0);

		// Deselect everything
		for(i=0;i<count;i++)
			SendMessage(hwnd,LB_SETSEL,(WPARAM)FALSE,(LPARAM)i);

		// Select the item closest to the cursor
		SendMessage(hwnd,LB_SETSEL,(WPARAM)TRUE,(LPARAM)index);

		// Bring up the popup menu
		GetCursorPos(&pt);
		TrackPopupMenu(pthis->hPopupMenu,TPM_RIGHTBUTTON,
			           pt.x,pt.y,
					   0,hwnd,NULL);
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDM_GETDESC:
			index=0;
			SendMessage(hwnd,LB_GETSELITEMS,(WPARAM)1,(LPARAM)&index);
			SendMessage(hwnd,LB_GETTEXT,(WPARAM)index,(LPARAM)buf);
			pthis->GetScriptDescription(buf);
			return TRUE;

		case IDM_GETDEF:
			index=0;
			SendMessage(hwnd,LB_GETSELITEMS,(WPARAM)1,(LPARAM)&index);
			SendMessage(hwnd,LB_GETTEXT,(WPARAM)index,(LPARAM)buf);
			pthis->GoToScriptDefinition(buf);
			return TRUE;

		case IDM_GOTODEFNEW:
			index=0;
			SendMessage(hwnd,LB_GETSELITEMS,(WPARAM)1,(LPARAM)&index);
			SendMessage(hwnd,LB_GETTEXT,(WPARAM)index,(LPARAM)buf);			
			pthis->GoToScriptDefinitionNew(buf);
			return TRUE;
		}
	}

	return CallWindowProc(pthis->OldGlobalProc,hwnd,msg,wParam,lParam);
}

BOOL PropEditor::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("PropEditor::DlgProc"); 
	CStr       strHelpFile;
	CStr       strBuf;

	if (LOWORD(wParam)==IDC_RICHEDIT1)
		if (RTFEditor::ProcMessage(hwnd,msg,wParam,lParam))
			return TRUE;

	switch(msg)
	{
	case WM_KEYDOWN:
		switch(wParam)
		{
		case VK_ESCAPE:
			SendMessage(hwnd,WM_CLOSE,0,0);
			return TRUE;
		}

		return FALSE;

	case WM_RBUTTONDOWN:
		MessageBox(hwnd,"RBUTTONDOWN!","BLAH",MB_OK);
		return TRUE;

	case WM_CLOSE:
		propView->GetCategory(strLastPanel);
		ttip.Close();
		propView->CloseTips();
		
		SetFocus(ip->GetMAXHWnd());
		EnableAccelerators();

		// Close any open popups
		if (PerformApplyOnClose())
		{
			CloseChildren();
			EndDialog(hwnd,0);
			return TRUE;
		}
		else
		{
			SaveConfig();
			CloseChildren();
			EndDialog(hwnd,0);
			//UnRegisterSelChange();
			
			// MAX subclasses the window and intercepts VK_TAB and VK_RETURN
			// so we're not given MAX access to subclass our window
			//ip->UnRegisterDlgWnd(hwnd);
			EnableAccelerators();

			if (fpCloseNotify)
				fpCloseNotify(this,pCloseNotifyData);
		}

		Hide();
		return FALSE;

	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			EnableAccelerators();
		else
			DisableAccelerators();
		
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		// File Menu
		case IDM_LOAD:
			LoadDlg();
			return TRUE;

		case IDM_SAVE:
			SaveDlg();
			return TRUE;

		case IDM_EXIT:
			SendMessage(hwnd,WM_CLOSE,0,0);
			return TRUE;

		// Edit Menu
		case IDM_CUT:
			pRichText->ClipboardCut();
			return TRUE;

		case IDM_COPY:
			pRichText->ClipboardCopy();
			return TRUE;

		case IDM_PASTE:
			pRichText->ClipboardPaste();
			return TRUE;

		case IDM_SEARCH_CUR:
			pRichText->SearchDlg();
			return TRUE;

		case IDM_REPLACE_CUR:
			pRichText->ReplaceDlg();
			return TRUE;

		case IDM_SEARCH_SEL:
			if (pMSearch)
				delete pMSearch;

			pMSearch=new MSearchDlg(ip,this,hInstance,hwnd,MSEARCH_SEARCH);
			return TRUE;

		case IDM_REPLACE_SEL:
			if (pMSearch)
				delete pMSearch;

			pMSearch=new MSearchDlg(ip,this,hInstance,hwnd,MSEARCH_REPLACE);
			return TRUE;

		case IDM_FINDNODE:
			FindNode();
			return TRUE;

		case IDM_FINDINFILES:
			FindInFiles();
			return TRUE;

		case IDM_REPLACEINNODES:
			ReplaceInNodes();
			return TRUE;

		// View Menu
		case IDM_DISPTREEVIEW:
			DispTreeView();
			return TRUE;

		case IDM_DISPLIST:
			DispGlobalList();
			return TRUE;

		// Settings Menu
		case IDM_DEFAULTPROPS:
			DefaultPropDlg();
			return TRUE;

		case IDM_LODPROPS:
			OpenLODBrowser();
			return TRUE;

		case IDM_AUTOSELECT:
			AutoSelect();
			return TRUE;

		case IDM_SUBSELAPPLY:
			SubSelApply();
			return TRUE;

		case IDM_SUBSELRESTORE:
			SubSelRestore();
			return TRUE;

		case IDM_APPLYONCLOSE:
			ApplyOnClose();
			return TRUE;

		case IDM_APPLYCHANGED:
			ApplyChangedOnly();
			return TRUE;

		case IDM_ADDCONFLICT:
			AddConflict();
			return TRUE;

		case IDM_NAMEEXPAND:
			ToggleNameExpansion();
			return TRUE;

		case IDM_FORCESCRIPTUPDATE:
			ForceScriptUpdates();

			// ForceScriptUpdates is applied immediately because
			// the property updater gets it from the config file
			// when launched
			SaveConfig();
			return TRUE;

		case IDM_IMMEDIATEAPPLY:
			ImmediateApply();
			return TRUE;

		case IDM_REFRESHVIEWPORT:
			RefreshViewport();
			return TRUE;

		// Special Menu
		case IDM_CLEARPROPS:
			ClearProps();
			return TRUE;

		case IDM_ADDPROP:
			AddProp();
			return TRUE;

		case IDM_ADDLINK:
			AddLink();
			return TRUE;

		case IDM_ADDLOD:
			AddLODLevel();
			return TRUE;

		case IDM_MERGEPROPS:
			OpenPropMergeDlg();
			return TRUE;

		case IDM_ADDSFP:
			AddSFP();
			return TRUE;

		case IDM_STOREPROPS:
			StoreProps();
			return TRUE;

		case IDM_CLUSTERFIND:
			OpenClusterFinder();
			return TRUE;

		case IDM_REFRESHSCRIPTS:
			pPropMergeDlg->ResetProps();
			RefreshScriptIni(true);
			return TRUE;

		case IDM_ABOUT:
			{
				char buf[1024];
				char date[256], time[256], msg[256];

				if (!VerifyAuthenticity(date, time, msg))
					sprintf(buf,"NExt.Gup Plugin\nBuilt: %s %s\n\nNo Branding Info (Beta Release)",__DATE__,__TIME__);
				else
					sprintf(buf,"NExt.Gup Plugin\nBuilt: %s %s\n\nBrand Time: %s %s\n%s",__DATE__,__TIME__,date,time,msg);

				MessageBox(hwnd,buf,"About...",MB_ICONINFORMATION|MB_OK);
				return TRUE;
			}

		// Popup Right Click Menu
		case IDM_GETDESC:
			GetScriptDescription(pRichText->GetWord());
			return TRUE;

		case IDM_GETDEF:
			GoToScriptDefinition(pRichText->GetWord());
			return TRUE;

		case IDM_GOTODEFNEW:
			GoToScriptDefinitionNew(pRichText->GetWord());
			return TRUE;

		case IDM_PARAMPOPUP:
			//ProcParamList();
			return TRUE;

		// Buttons
		case IDC_RESTOREDEFAULTS:
			RestoreDefaults();
			return TRUE;

		case IDC_APPLY:
			// Don't compile script if script is in undefined state
			if (pRichText->GetText()!=CStr(vUNKNOWN) && !ErrorCheck())
				Apply(TRUE);

			return TRUE;

		case IDC_CANCEL:
			CloseChildren();

			SaveConfig();
			EndDialog(hwnd,0);
			UnRegisterSelChange();
			//ip->UnRegisterDlgWnd(hwnd);
			EnableAccelerators();

			if (fpCloseNotify)
				fpCloseNotify(this,pCloseNotifyData);

			return TRUE;

		case IDC_ADDSCRIPT:
			AddScript();
			return TRUE;

		case IDC_FINDSCRIPT:
			FindScript();
			return TRUE;

		case IDC_CONTEXTLIST:
			switch(HIWORD(wParam))
			{
			case CBN_SELCHANGE:
				{
					int index = SendDlgItemMessage(hwnd,IDC_CONTEXTLIST,CB_GETCURSEL,0,0);
					Link<ConfigScript>* link = (Link<ConfigScript>*)SendDlgItemMessage(hwnd,IDC_CONTEXTLIST,CB_GETITEMDATA,(WPARAM)index,0);

					if (link)
						RTFEditor::SetText(link->data.buffer);
					else
						RTFEditor::SetText(editScript);

					return TRUE;
				}
			}
			break;

		case IDC_OBJCLASSM:
			switch(HIWORD(wParam))
			{
			case CBN_SELCHANGE:
				{
					CStr className, typeName;
					/*
					CStr oldstrCluster;
					propView->GetValue(PROP_CLUSTER,oldstrCluster);
					*/
					DWORD oldFlags = GetPropViewFlags();

					// Save the data before we acquire the new selection information
					char buf[256];

					// Get class (this takes advantage of the fact that the text update isn't immediate)
					GetDlgItemTextLast(hwnd,IDC_OBJCLASSM,buf,256);
					className=CStr(buf);

					// Get type
					GetDlgItemText(hwnd,IDC_OBJTYPEM,buf,256);  // DBLCHK
					//GetDlgItemTextLast(hwnd,IDC_OBJTYPEM,buf,256);
					typeName=CStr(buf);
			
					// Update
					if (UpdateClassSelection())
					{
						LinkList<ConfigProp> oldprops;
						bClassChanged=TRUE;

						int bResult;
						
						//if (propView->WasUpdated())
						if (!bLockPropBuffer && !IsDefaulted(propView,className,typeName))
							bResult=MessageBox(hwnd,"Would you like to retain changed Default Properties?","Retain Default Prop. Data",MB_ICONQUESTION|MB_YESNO);
						else
							bResult=IDNO;

						if (bResult==IDYES)
							oldprops = props;

						//if (ip->GetSelNodeCount()>0)
						{
							// Fill in property data with new defaults from configDB
							if (pRichText->HasText())
								CopyProps(FALSE);
							else
								CopyProps(TRUE);

							LinkList<int> updateFields;

							if (bResult==IDYES)
							{
								MergeProps(className,typeName,&oldprops,&updateFields);
								flags = oldFlags;
								//strCluster = oldstrCluster;
							}

							UpdatePropView();
							AssignUpdated(&updateFields);
						}
					}
				
				ResetLastText(hwnd,IDC_OBJCLASSM);
				ResetLastText(hwnd,IDC_OBJTYPEM);

				// If ImmediateApply is checked then apply the properties now
				if (bImmediateApply && !bLockImmediateApply)
					Apply(FALSE);

				return TRUE;
				}
			}
			break;

		case IDC_OBJTYPEM:
			switch(HIWORD(wParam))
			{
			case CBN_SELCHANGE:
				{
					bChangeMade=TRUE;
					bTypeChanged=TRUE;

					int bResult;

					// Previously removed for TT1851 updated to exist but only for single selection changes
					//if (!bLockPropBuffer && pRichText->HasText() && selSet.GetSize() == 1)
					if (!bLockPropBuffer && pRichText->HasText())
					{
						bResult=MessageBox(hwnd,"The editor has a script currently displayed.\nWould you like to retain the script?","Retain Script Data",MB_ICONQUESTION|MB_YESNO);

						if (bResult==IDNO)
						{
							editScript = "";
							editScripts.Clear();
							ResetContext();

							pRichText->Clear();
						}
					}
					//*/

					// Save the data before we acquire the new selection information
					CStr className,typeName;
					char buf[256];

					// Get class (this takes advantage of the fact that the text update isn't immediate)
					GetDlgItemTextLast(hwnd,IDC_OBJCLASSM,buf,256);
					className=CStr(buf);

					// Get type
					GetDlgItemTextLast(hwnd,IDC_OBJTYPEM,buf,256);
					typeName=CStr(buf);

					//if (propView->WasUpdated())
					if (!bLockPropBuffer && !IsDefaulted(propView,className,typeName))
						bResult=MessageBox(hwnd,"Would you like to retain changed Default Properties?","Retain Default Prop. Data",MB_ICONQUESTION|MB_YESNO);
					else
						bResult=IDNO;

					LinkList<ConfigProp> oldProps;
					//CStr oldstrCluster;
					//VERIFY
					//propView->GetValue(PROP_CLUSTER,strCluster);
					//propView->GetValue(PROP_CLUSTER,oldstrCluster);
					DWORD oldFlags = GetPropViewFlags();

					if (bResult==IDYES)
					{
						// Store values before update so we can merge them afterward
						oldProps = props;

						char buf[256];

						// Get class (this takes advantage of the fact that the update isn't immediate)
						GetDlgItemTextLast(hwnd,IDC_OBJCLASSM,buf,256);
						className=CStr(buf);

						// Get type
						GetDlgItemTextLast(hwnd,IDC_OBJTYPEM,buf,256);
						typeName=CStr(buf);
					}

					// Fill in property data with new defaults from configDB
					if (pRichText->HasText())
						CopyProps(FALSE);
					else
						CopyProps(TRUE);

					LinkList<int> updateFields;

					if (bResult==IDYES)
					{
						MergeProps(className,typeName,&oldProps,&updateFields);
						flags = oldFlags;
						//strCluster = oldstrCluster;
					}

					UpdatePropView();
					AssignUpdated(&updateFields);
					ResetLastText(hwnd,IDC_OBJTYPEM);

					// If ImmediateApply is checked then apply the properties now
					if (bImmediateApply && !bLockImmediateApply)
						Apply(FALSE);

					// Particle objects require special processing when changing to their types since
					// they need to reposition/resize the individual boxes within the system
					if (className == CStr("ParticleObject"))
					{
						GetDlgItemText(hwnd,IDC_OBJTYPEM,buf,256);
						UpdateParticleObjVisualization(buf);
					}

					return TRUE;
				}
			}
			break;

		case IDC_ADDNODE:
			if (bLockSelectAdd)
				return TRUE;

			bLockSelection = TRUE;
			LockSelectAdd();
			ip->DoHitByNameDialog(&addCB);
			UnLockSelectAdd();
			bLockSelection = FALSE;
			AcquireSelSet(FALSE);
			break;

		case IDC_REMOVENODE:
			RemoveSelSet();
			break;

		case IDC_RESTORESCRIPT:
			ParseUserProps();
			break;

		case IDC_NODELIST:
			switch(HIWORD(wParam))
			{
			case LBN_SELCHANGE:
				ProcNodeListSelChange();
				return TRUE;

			case LBN_DBLCLK:
				RenameNode();
				return TRUE;
			}
			break;

		case IDC_GLOBALLIST:
			switch(HIWORD(wParam))
			{
			case LBN_DBLCLK:
				AddScript();
				return TRUE;
			}
			break;

		case IDC_ADDFILTER:
			AddFilter();
			return TRUE;

		case IDC_REMOVEFILTER:
			RemoveFilter();
			return TRUE;

		case IDC_FILTERLIST:
			switch(HIWORD(wParam))
			{
			case LBN_SELCHANGE:
				FilterChange();
				return TRUE;
			}
/*
		case IDC_RICHEDIT1:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				bChangeMade=TRUE;
				bScriptChanged=TRUE;

				// Hide any parameter popup window if not locked
				if (pPopupParams && !bLockParamPopup)
					pPopupParams->Hide();
				else
					bLockParamPopup=FALSE;

				/*
				if (pPopupParams)
				{
					delete pPopupParams;
					pPopupParams=NULL;
				}
				*/
/*
				// Perform syntax highlighting on this line
				// (and the line above due to enter breaks)
				lineNum=pRichText->GetLineNum();

				if (lineNum>0)
					SyntaxCheck(pRichText->GetLineNum()-1);

				SyntaxCheck(pRichText->GetLineNum());
				
				// Popup a function parameter list if necessary
				if (!pPopupParams->IsVisible())
					ProcParamList();

				return TRUE;
			}
			break;
		}

	case WM_NOTIFY:
		switch(wParam)
		{
		case IDC_RICHEDIT1:
			// Intercept the RichText control's virtual key methods
			// so we can do syntax highlighting on the fly
			msgfilter=(MSGFILTER*)lParam;
			switch(msgfilter->msg)
			{
			case WM_RBUTTONUP:
				GetCursorPos(&ptCursor);
				TrackPopupMenu(hPopupMenu,TPM_RIGHTBUTTON,
					           ptCursor.x,ptCursor.y,
							   0,hwnd,NULL);
				break;

			case WM_KEYDOWN:
				switch(msgfilter->wParam)
				{
				case VK_UP:
				case VK_DOWN:
				case VK_LEFT:
				case VK_RIGHT:
				case VK_SPACE:
				case VK_RETURN:
				case VK_BACK:
					if (pPopupParams)
					{
						//delete pPopupParams;
						//pPopupParams=NULL;
						pPopupParams->Hide();
					}
					return TRUE;

				case VK_ESCAPE:
					bLockParamPopup=FALSE;

					if (pPopupParams->IsVisible())
					{
						//delete pPopupParams;
						//pPopupParams=NULL;
						pPopupParams->Hide();
					}
					else
						SendMessage(hwnd,WM_CLOSE,0,0);

					return TRUE;

				case VK_F1:
				case VK_F2:
					SpawnWinHelp();
					return TRUE;
				}

				return TRUE;
			}	
			break;
*/
		}
	}

	return FALSE;
}

void PropEditor::MAXSelChange(void *param,NotifyInfo *info)
{ FUNC_ENTER("PropEditor::MAXSelChange"); 
	OutputDebugString("HANDLER: MAXSelChange (PropEditor)\n");
	// Don't update PE if link manager is locked, to save time when selecting
	// long chains of nodes
	if (!GetLinkMan()->IsLocked())
		((PropEditor*)param)->SelectionSetChanged();
}

void PropEditor::SelectionSetChanged(BOOL bNoApply)
{ FUNC_ENTER("PropEditor::SelectionSetChanged"); 
	//UpdateClassList();

	if (bLockSelection)
		return;

	AssignEmptyPropBuffers();

	if (propView)
	{
		if (ip->GetSelNodeCount() == 1)
			propView->SetRuleObj(ip->GetSelNode(0));
		else
			propView->SetRuleObj(NULL);
	}

	if (bLockSelection)
		return;

	// If AutoSelection is enabled, we must apply the current object
	if (!bNoApply && bAutoSelect)
		Apply(FALSE);

	// Disable acknowledgement when the selection changes until we're finished compositing the list
	classBrowser->DisableAck();
	typeBrowser->DisableAck();

	AcquireSelSet();

	// Clear property list if nothing is selected
	if (selSet.GetSize()==0)
	{
		ClearProps(FALSE);
		editScripts.Clear();
		pRichText->Clear();
	}

	bClassConflict=FALSE;
	bTypeConflict=FALSE;

	if (bSubSelRestore)
		NodeListRestore();
	else
	// If AutoSelection is enable, we must restore
	if (bAutoSelect)
		ParseUserProps();

	delprops.Clear();

	// Reenable class and type change acknowledgments
	classBrowser->EnableAck();
	typeBrowser->EnableAck();
}

void PropEditor::AssignEmptyPropBuffers()
{ FUNC_ENTER("PropEditor::AssignEmptyPropBuffers"); 
	if (!bPropertyAutoFill)
		return;

	int  nNodes = ip->GetSelNodeCount();
	CStr defBuf = GetClassTypeRecord(DEFAULT_CLASS, "");

	LinkList<ConfigProp>    cprops;
	LinkList<ConfigProgram> cprogs;
	LinkList<ConfigScript>  cscripts;
	CStr                    cmds;
	DWORD                   flags;
	CStr                    cluster;
	CStr                    scriptBuf;

	scriptBuf = ParseConfigProps(&cprops, &flags, defBuf, &cmds, &cluster, &cprogs, &cscripts);

	for(int i = 0; i < nNodes; i++)
	{
		INode* node = ip->GetSelNode(i);

		CStr propBuffer;
		node->GetUserPropBuffer(propBuffer);

		// If any selected property buffer is empty fill it in with the default props
		if (IsBlank(propBuffer))
		{
			DumpNode(node,
			 DEFAULT_CLASS,
			 "",
			 flags,
			 &cprops,
			 cmds,
			 scriptBuf,
			 cluster,
			 &cprogs,
			 &cscripts);
		}
	}
}

void PropEditor::RemoveSelSet()
{ FUNC_ENTER("PropEditor::RemoveSelSet"); 
	HWND listbox=GetDlgItem(hwnd,IDC_NODELIST);

	// Scan through the list and deselect any of the items currently selected in it
	int LstCount=SendMessage(listbox,LB_GETCOUNT,0,0);

	// Deny any selection updates while we're changing the selection
	bLockSelection=TRUE;

	for(int i=0;i<LstCount;i++)
	{
		if (SendMessage(listbox,LB_GETSEL,i,0))
		{
			char str[256];
			assert(SendMessage(listbox,LB_GETTEXT,i,(LONG)str)!=LB_ERR);

			INode* node=(INode*)SendMessage(listbox,LB_GETITEMDATA,i,0);
			ip->DeSelectNode(node);
		}
	}

	// Turn selection updates back on
	bLockSelection=FALSE;
	AcquireSelSet();
}

void PropEditor::AcquireSelSet(BOOL bDelOld)
{ FUNC_ENTER("PropEditor::AcquireSelSet"); 
	// Check if we're allowed to accept a selection change
	if (bLockSelection)
		return;
	
	selSet.Clear();

	HWND listbox=GetDlgItem(hwnd,IDC_NODELIST);

	// Dump the selected nodes in the Max scene to our selection list box
	int nSelNodes=ip->GetSelNodeCount();

	if (bDelOld)
		SendMessage(listbox,LB_RESETCONTENT,0,0);	// Clear ListBox

	for(int i=0;i<nSelNodes;i++)
	{
		int index;

		INode* node=ip->GetSelNode(i);
		
		// Don't allow selection of linkobjects
		Object* obj=node->EvalWorldState(0).obj;
		if (obj->ClassID()==vLINK_OBJ_CLASS_ID)
			continue;
		if( obj->ClassID()==PARTICLE_BOX_CLASS_ID)
		{
			if( node->GetParentNode() != gInterface->GetRootNode())
			{
				continue;
			}			
		}

		selSet.Add(&node);
		
		index=SendMessage(listbox,LB_ADDSTRING,0,(LONG)(char*)node->GetName());
		
		// Store pointer to the node in the listbox so we can reference it later
		SendMessage(listbox,LB_SETITEMDATA,index,(LPARAM)node);
	}

	SendMessage(listbox,LB_SETSEL,TRUE,(LPARAM)-1);

	ResetContext();

	if (nSelNodes != 1)
		EnableWindow(GetDlgItem(hwnd,IDC_CONTEXTLIST),FALSE);
	else
		EnableWindow(GetDlgItem(hwnd,IDC_CONTEXTLIST),TRUE);

	UpdateClassList();
}

void PropEditor::GetSelNodes()
{ FUNC_ENTER("PropEditor::GetSelNodes"); 
	HWND listbox=GetDlgItem(hwnd,IDC_NODELIST);

	// Dump the selected subnodes in the list out to the selection set table
	selSet.Clear();

	int numSel=SendMessage(listbox,LB_GETSELCOUNT,0,0);
	INT*  sels=new INT[numSel];

	SendMessage(listbox,LB_GETSELITEMS,(WPARAM)numSel,(LPARAM)sels);

	// Add each selected node in the nodelist to the selSet table
	for(int i=0;i<numSel;i++)
	{
		INode* node=(INode*)SendMessage(listbox,LB_GETITEMDATA,(WPARAM)sels[i],0);
		selSet.Add(&node);
	}
}

void PropEditor::GetScriptDescription(CStr keyword)
{ FUNC_ENTER("PropEditor::GetScriptDescription"); 
	if (keyword==CStr(""))
		return;

	TypeDesc* tdesc=scriptDB.GetItem(keyword);
	CStr      strDesc;

	if (tdesc)
	{
		strDesc=ParseScriptDesc(tdesc->ScriptFile,tdesc->Name);
		if (strDesc.Length()>0)
		{
			POINT pt=pRichText->GetScreenPos();
			ttip.CreateTip(strDesc,pt.x,pt.y);
		}
	}
}

void PropEditor::GoToScriptDefinition(CStr keyword)
{ FUNC_ENTER("PropEditor::GoToScriptDefinition"); 
	// Ensure that you don't lose anything
	if (pRichText->HasText())
	{
		int bResult;

		bResult=MessageBox(hwnd,"This command will load a new script into the edit window.\nWould you like to apply your script before loading the new one?","Apply Script Data",MB_ICONQUESTION|MB_YESNOCANCEL);

		if (bResult==IDCANCEL)
			return;

		if (bResult==IDYES)
			Apply(FALSE);

		TypeDesc* tdesc=scriptDB.GetItem(keyword);
		
		if (tdesc)
		{
			pRichText->Clear();

			if (!LoadText(tdesc->ScriptFile))
			{
				char errMsg[256];
				sprintf(errMsg,"The qscript file '%s' defining '%s' could not be found.",(char*)tdesc->ScriptFile,(char*)keyword,MB_ICONSTOP|MB_OK);
				MessageBox(hwnd,errMsg,"Script file not found",MB_ICONSTOP|MB_OK);
			}
			else
			{
				SetTitle(TITLEMODE_FILE, tdesc->ScriptFile);

				// Find the definition of the script in the file
				CStr strSearch=CStr("Script ")+keyword;
				int  index=pRichText->Search(strSearch,RTS_WHOLEWORD);
				
				if (index!=-1)
					pRichText->HighlightLine(index);
			}
			return;
		}
		else
		{
			char errMsg[256];
			sprintf(errMsg,"The keyword '%s' does not exist in the map file database.",(char*)keyword);
			MessageBox(hwnd,errMsg,"Keyword not found",MB_ICONWARNING|MB_OK);
		}
	}
}

void PropEditor::GoToScriptDefinitionNew(CStr keyword)
{ FUNC_ENTER("PropEditor::GoToScriptDefinitionNew"); 
	pExtScriptDlg->Show();

	TypeDesc* tdesc=scriptDB.GetItem(keyword);
		
	if (tdesc)
	{
		pExtScriptDlg->Clear();

		CStr fileBuf=LoadTextStr(tdesc->ScriptFile);

		if (fileBuf==CStr(""))
		{
			char errMsg[256];
			sprintf(errMsg,"The qscript file '%s' defining '%s' could not be found (or contains no data).",(char*)tdesc->ScriptFile,(char*)keyword,MB_ICONSTOP|MB_OK);
			MessageBox(hwnd,errMsg,"Script file not found",MB_ICONSTOP|MB_OK);
		}
		else
		{
			pExtScriptDlg->SetTitle(tdesc->ScriptFile);
			fileBuf=BuildTextRTF(fileBuf);

			pExtScriptDlg->SetText(fileBuf);
			pExtScriptDlg->Highlight(keyword);
		}
		return;
	}
	else
	{
		char errMsg[256];
		sprintf(errMsg,"The keyword '%s' does not exist in the map file database.",(char*)keyword);
		MessageBox(hwnd,errMsg,"Keyword not found",MB_ICONWARNING|MB_OK);
	}
}

bool PropEditor::UpdateClassSelection(BOOL bWarn)
{ FUNC_ENTER("PropEditor::UpdateClassSelection"); 
	if (bLockClassUpdate)
		return false;

	char className[256];

	// Check to ensure that change is valid
	if (!VerifyClassValidity())
	{
		//char buf[256];
		int  index;

		// Restore the class selection back to what the text is currently equal to (the last selection)
		GetDlgItemTextLast(hwnd,IDC_OBJCLASS,className,256);
		index=SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_FINDSTRING,(WPARAM)-1,(LPARAM)className);
		SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_SETCURSEL,(WPARAM)index,0);
		return false;
	}

	bChangeMade=TRUE;

	// Previously removed for TT1851 updated to exist but only for single selection changes
	/*
	if (bWarn && !bLockPropBuffer && pRichText->HasText() && selSet.GetSize() == 1)
	{
		int bResult;

		bResult=MessageBox(hwnd,"The editor has a script currently displayed.\nWould you like to retain the script?","Retain Script Data",MB_ICONQUESTION|MB_YESNO);

		if (bResult==IDNO)
		{
			editScripts.Clear();
			ResetContext();

			pRichText->Clear();
		}
	}
	//*/

	int index=SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_GETCURSEL,0,0);
	if (index>-1)
	{
		Link<ConfigClass>* cclass=(Link<ConfigClass>*)SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_GETITEMDATA,index,0);
		Link<ConfigType>*  ctypes=cclass->data.types.GetHead();

		SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_RESETCONTENT,0,0);

		// Enable the type window if there's anything going in it
		if (ctypes==NULL)
			EnableWindow(GetDlgItem(hwnd,IDC_OBJTYPEM),FALSE);
		else
			EnableWindow(GetDlgItem(hwnd,IDC_OBJTYPEM),TRUE);


		while(ctypes)
		{
			CStr itemName;

			//if (ctypes->data.strGroup.Length()>0)
			//	itemName = ctypes->data.strGroup + CStr("/") + ctypes->data.name;
			//else
				itemName = ctypes->data.name;

			// Items only get added if they're available options for all selected objects
			Link<INode*>* linkNode = selSet.GetHead();
			bool bViewable = true;

			while(linkNode)
			{
				if (!IsViewable(linkNode->data, GetClass(), itemName))
				{
					bViewable = false;
					break;
				}

				linkNode = linkNode->next;
			}

			if (bViewable)
			{
				int index=SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_ADDSTRING,0,(LPARAM)(char*)itemName);
				SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_SETITEMDATA,index,(LPARAM)ctypes);
			}

			ctypes=ctypes->next;
		}

		// Select the first item in the type list
		if (typeBrowser->GetAck())
		{
			typeBrowser->DisableAck();
			SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_SETCURSEL,0,0);
			typeBrowser->EnableAck();
		}
		else
		{
			SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_SETCURSEL,0,0);
		}
	}

	return true;
}

void PropEditor::AddScript()
{ FUNC_ENTER("PropEditor::AddScript"); 
	if (bDisplayTree)
	{
		TypeDesc* tdesc=(TypeDesc*)pTreeView->GetSelection();

		if (tdesc)
			pRichText->InsertText(tdesc->Name);
	}
	else
	{
		int selCount=SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_GETSELCOUNT,0,0);
		INT* sel=new INT[selCount];
		SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_GETSELITEMS,(WPARAM)selCount,(LPARAM)sel);

		for(int i=0;i<selCount;i++)
		{
			char buf[256];
			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_GETTEXT,(WPARAM)sel[i],(LPARAM)buf);

			if (i==selCount-1)
				pRichText->InsertText(buf);
			else
			{
				strcat(buf,"\n");
				pRichText->InsertText(buf);
			}
		}

		delete sel;
	}
}

Link<ConfigProp>* PropEditor::FindProp(LinkList<ConfigProp>* proplist,CStr name)
{ FUNC_ENTER("PropEditor::FindProp"); 
	Link<ConfigProp>* curNode=proplist->GetHead();

	while(curNode)
	{
		if (curNode->data.name==name)
			return curNode;

		curNode=curNode->next;
	}

	return NULL;
}

void PropEditor::ApplyNodeChanges(INode* node)
{ FUNC_ENTER("PropEditor::ApplyNodeChanges"); 
	LinkList<ConfigProp>    origprops;
	DWORD                   origflags;
	CStr                    origCmds;
	CStr                    origScript;
	LinkList<ConfigScript>  origScripts;
	CStr                    propBuffer;
	CStr                    origClass;
	CStr                    origType;
	CStr                    origCluster;
	LinkList<ConfigProgram> origPrograms;
	char					buf[256];
	bool					changed_particle;
	bool                    bUpdateParticleOrientation = false;

	// Store particle system position if appropriate
	Point3 ptMid, ptEnd, ptMidScale, ptEndScale;
	float  fMidWidth, fMidHeight, fMidLength, fEndWidth, fEndHeight, fEndLength;

	node->GetUserPropBuffer(propBuffer);
	changed_particle = false;

	if (node->EvalWorldState(0).obj->ClassID() == PARTICLE_BOX_CLASS_ID)
	{
		if (GetParticleOrientation(propBuffer, &ptMid, &ptEnd, &fMidWidth,
			                                                   &fMidHeight,
											                   &fMidLength,
											                   &fEndWidth,
											                   &fEndHeight,
											                   &fEndLength,
															   &ptMidScale,
															   &ptEndScale))
																bUpdateParticleOrientation = true;
	}

	// Check to see if this is an unassigned node, if so copy all data over
	if (propBuffer.Length()==0 || bClassChanged)
	{
		bClassChanged=true;
		bTypeChanged=true;
	}

	// Get original class and type
	if (!node->GetUserPropString("Class",origClass))
		node->GetUserPropString("class",origClass);

	if (!node->GetUserPropString("Type",origType))
		node->GetUserPropString("type",origType);

	//AddGlobalDefaults(propBuffer);
	//XXorigScript=ParseConfigProps(&origprops,&origflags,propBuffer,&origCmds,&origCluster,&origPrograms,&origScripts);
	origScript=ParseNodeConfigProps(&origprops,&origflags,node,&origCmds,&origCluster,&origPrograms,&origScripts);
	AddGlobalDefaults(propBuffer,&origprops,&origflags,&origCmds,origClass,origType);

	if (strCmds != origCmds)
		origCmds = strCmds;

	// Update LOD links
	LODBrowser::UpdateLODs(&origprops,propView,node);

	// Load the data from the UI fields into props
	GetDefaultProps();

	// Update class
	if (bClassChanged || !bApplyChangedOnly)
	{
		GetDlgItemTextLast(hwnd,IDC_OBJCLASSM,buf,256);
		origClass=CStr(buf);
	}

	// Update type
	if (bTypeChanged || !bApplyChangedOnly)
	{
		GetDlgItemText(hwnd,IDC_OBJTYPEM,buf,256);
		origType=CStr(buf);
	}

	// Programs must exist in scripts.ini they cannot be propagated within the editor
	// doing so will cause problems with multiple selection updates
	GetPrograms(&origPrograms,origClass,origType);
	//origPrograms = programs;

	int exception=(!bApplyChangedOnly || bClassChanged || bTypeChanged);

	// Update cluster
//	if (propView->WasUpdated(PROP_CLUSTER) || exception)
//		propView->GetValue(PROP_CLUSTER,origCluster);

	// Update flags
//	if (propView->WasUpdated(PROP_CREATEDATSTART) || exception)		// CreatedAtStart
//	{
//		if (flags & CCLASS_CREATEDATSTART)
//			origflags|=CCLASS_CREATEDATSTART;
//		else
//			origflags&=~CCLASS_CREATEDATSTART;
//	}

//	if (propView->WasUpdated(PROP_ABSENTINNETGAMES) || exception)	// AbsentInNetGames
//	{
//		if (flags & CCLASS_ABSENTINNETGAMES)
//			origflags|=CCLASS_ABSENTINNETGAMES;
//		else
//			origflags&=~CCLASS_ABSENTINNETGAMES;
//	}

//	if (propView->WasUpdated(PROP_TRICKOBJECT) || exception)		// TrickObject
//	{
//		if (flags & CCLASS_TRICKOBJECT)
//			origflags|=CCLASS_TRICKOBJECT;
//		else
//			origflags&=~CCLASS_TRICKOBJECT;
//	}

//	if (propView->WasUpdated(PROP_OCCLUDER) || exception)		// NonCollidable
//	{
//		if (flags & CCLASS_OCCLUDER)
//			origflags|=CCLASS_OCCLUDER;
//		else
//			origflags&=~CCLASS_OCCLUDER;
//	}

	// If the class or type changed we reset all prior properties
	// MAYBE: make optional
	if (bClassChanged || bTypeChanged)
		origprops.Clear();

	// Update any properties that needed to be deleted
	Link<CStr>* curDelProp=delprops.GetHead();

	while(curDelProp)
	{
		Link<ConfigProp>* link=FindProp(&origprops,curDelProp->data);
		
		if (link)
			origprops.Remove(link);
		
		curDelProp=curDelProp->next;
	}

	// Update properties
	int numProps=propView->NumProps();

	for(int i=PROPSTART;i<numProps;i++)
	{
		if (propView->WasUpdated(i) || (propView->GetFlags(i) & PROPFLAG_NEWPROP) || exception)
		{
			CStr name,value,extdata;
			name    = propView->GetName(i);
			extdata = propView->GetUserData(i);
			propView->GetValue(i,value);
	
			// Scan to see if the property is contained within the object
			Link<ConfigProp>* link=FindProp(&origprops,propView->GetName(i));

			// Update the property
			if (link)
				link->data.value=value;
			else
			{
				// The property doesn't exist so we'll have to add it (or ignore)
				if (bAddConflict || bClassChanged || bTypeChanged || propView->GetFlags(i) & PROPFLAG_NEWPROP)
				{
					ConfigProp cprop;
					cprop.name=name;
					cprop.value=value;
					cprop.extdata=extdata;
				
					origprops.Add(&cprop);
				}
			}
		}
	}

	// Update Script
	if (bScriptChanged || exception)
	{
		// Don't modify the script if the script is in an undefined state
		if (pRichText->GetText()!=CStr(vUNKNOWN))
			origScript = editScript;
			//origScript=pRichText->GetText();
	}

	// Update the alternate scripts if script context is available
	if (IsWindowEnabled(GetDlgItem(hwnd,IDC_CONTEXTLIST)))
		origScripts = editScripts;

	// Update omission mode
	if (IsWindowEnabled(GetDlgItem(hwnd,IDC_OMIT)))
	{
		if (IsDlgButtonChecked(hwnd,IDC_OMIT)==BST_CHECKED)
		{
			if (!IsInstr(origCmds,"// CMD:OMIT") &&
				!IsInstr(origCmds,"// CMD:FORCEOMIT"))
			{
				CStr eol = "\r\n";
				origCmds += CStr("// CMD:OMIT") + eol;
			}
		}
		else
		{
			origCmds = ReplaceStr(origCmds,"// CMD:OMIT","");
			origCmds = ReplaceStr(origCmds,"// CMD:FORCEOMIT","");
		}
	}

	DumpNode(node,
			 origClass,
			 origType,
			 origflags,
			 &origprops,
			 origCmds,
			 origScript,
			 origCluster,
			 &origPrograms,
			 &origScripts);

	// Propagate particle system position
	if (node->EvalWorldState(0).obj->ClassID() == PARTICLE_BOX_CLASS_ID)
	{
		// Scan through properties for particle update
		for(int i=PROPSTART;i<numProps;i++)
		{
			if (propView->WasUpdated(i) || (propView->GetFlags(i) & PROPFLAG_NEWPROP) || exception)
			{
				int prop_index;			
				CStr name,value,extdata;
				name    = propView->GetName(i);
				extdata = propView->GetUserData(i);
				propView->GetValue(i,value);
		
				if (!bParticleUpdateLock)
				{
					if(( prop_index = GetParticlePropertyIndex( name )) >= 0 )
					{
						Object* obj=node->EvalWorldState(0).obj;

						if (obj->ClassID()==PARTICLE_BOX_CLASS_ID)
						{				
							IParticleBox* box;
							
							box = dynamic_cast< IParticleBox* >( obj );
							
							changed_particle = true;

							if (!bClassChanged && !bTypeChanged)
								HandleParticlePropertyChange( box, prop_index, value );
						}
					}
				}

			}
		}

		if (bUpdateParticleOrientation)
			SetParticleOrientation(node, ptMid, ptEnd, fMidWidth,
				                                       fMidHeight,
												       fMidLength,
												       fEndWidth,
												       fEndHeight,
												       fEndLength,
													   ptMidScale,
													   ptEndScale);
	}	// End if ParticleBox

	if( changed_particle )
	{
		gInterface->RedrawViews( gInterface->GetTime());
	}

	// Update the color of the node based on selection
	COLORREF node_color = GetNodeColor(origClass, origType);
	COLORREF prevColor  = node->GetWireColor();

	if (node_color != prevColor &&
		node_color != 0)
	{
		node->SetWireColor(node_color);
		ip->ForceCompleteRedraw();
	}
}

void PropEditor::Apply(BOOL bAlert,INode* node)
{ FUNC_ENTER("PropEditor::Apply"); 
	if (node==NULL)
	{
		int numNodes=selSet.GetSize();

		for(int i=0;i<numNodes;i++)
		{
			INode* node=selSet[i];
			ApplyNodeChanges(node);
		}
		
		//if (bAlert)
		//	MessageBox(hwnd,"Your changes have been applied.","Changes Applied",MB_ICONINFORMATION);
	}
	else
		ApplyNodeChanges(node);

	ResetModFlags();
	delprops.Clear();

	// Refresh the viewports if appropriate
	if (bRefreshViewport)
		ip->ForceCompleteRedraw();
}

/*
void PropEditor::Apply(BOOL bAlert,INode* node)
{
	CStr applyBuf;
	CStr eol="\r\n";
	char buf[256];

	if (bApplyChangedOnly)
		BuildChangedProps();

	// Copy default property list UI data into the props list incase the
	// user made any changes
	GetDefaultProps();

	// Dump out class info
	GetDlgItemText(hwnd,IDC_OBJCLASS,buf,256);
	
	// Default to environment object if none selected
	if (strlen(buf)==0)
		strcpy(buf,"EnvironmentObject");

	applyBuf+=CStr("Class = ")+CStr(buf)+eol;

	// Dump out type info
	GetDlgItemText(hwnd,IDC_OBJTYPE,buf,256);
	
	if (strlen(buf)>0)
		applyBuf+=CStr("Type = ")+CStr(buf)+eol;

	// If there are no properties copy them over from the configDB
	//if (props.GetSize()==0)
	//	CopyProps();


	// Dump out flags
	if (flags & CCLASS_CREATEDATSTART)
		applyBuf+=CStr("CreatedAtStart")+eol;

	if (flags & CCLASS_ABSENTINNETGAMES)
		applyBuf+=CStr("AbsentInNetGames")+eol;

	if (flags & CCLASS_TRICKOBJECT)
		applyBuf+=CStr("TrickObject")+eol;

	
	// Dump out parsed properties
	Link<ConfigProp>* lprop=props.GetHead();

	while(lprop)
	{
		applyBuf+=lprop->data.name+CStr(" = ")+lprop->data.value+eol;
		lprop=lprop->next;
	}

	// Dump out default commands (not properties)
	// These were added via MAX's user property edit box or through scripts.ini
	applyBuf+=strCmds;

	// Dump out script
	applyBuf+=CStr("script")+eol;
	applyBuf+=pRichText->GetText()+eol;
	applyBuf+=CStr("endscript")+eol;


	// Set properties on all selected nodes
	if (node==NULL)
	{
		int numNodes=selSet.GetSize();

		for(int i=0;i<numNodes;i++)
		{
			INode* node=selSet[i];
			node->SetUserPropBuffer(applyBuf);
		}
		
		if (bAlert)
			MessageBox(hwnd,"Your changes have been applied.","Changes Applied",MB_ICONINFORMATION);
	}
	else
	{
		node->SetUserPropBuffer(applyBuf);
	}
}
*/

void PropEditor::SetTitle(int mode, char* name)
{ FUNC_ENTER("PropEditor::SetTitle"); 
	char windowTitle[128];
	char nodeName[90];

	// Truncate if over 80 characters long (sheesh)
	for(int i=0;i<80;i++)
		nodeName[i]=name[i];

	nodeName[i]='\0';

	if (strlen(name)>80)
		strcat(nodeName,"...");

	switch(mode)
	{
	case TITLEMODE_NODE:
		sprintf(windowTitle,"Property Editor - [Node: %s]",nodeName);
		break;

	case TITLEMODE_FILE:
		sprintf(windowTitle,"Property Editor - [File: %s]",nodeName);
		break;
	}
	
	SetWindowText(hwnd,windowTitle);
}

bool PropEditor::ParseUserProps(INode* node)
{ FUNC_ENTER("PropEditor::ParseUserProps"); 
	CStr propBuffer;
	CStr scriptBuffer;
	int numNodes=ip->GetSelNodeCount();
	bool bCopyProps=false;
	bChangeMade=FALSE;

	if (node==NULL)
	{
		// The primary display can only display one object (for now)
		if (numNodes==0)
		{
			SetWindowText(hwnd,"Property Editor - [No Nodes Selected]");
			return false;
		}
		
		if (numNodes>1)
		{
			SetWindowText(hwnd,"Property Editor - [Multiple Nodes Selected]");
			return false;
		}

		node=ip->GetSelNode(0);
	}

	SetTitle(TITLEMODE_NODE, node->GetName());

	// Scan the MAX User Property buffer for all the properties used
	// and our script data if it exists
	node->GetUserPropBuffer(propBuffer);
	//AddGlobalDefaults(propBuffer);
	//XXscriptBuffer=ParseConfigProps(&props,&flags,propBuffer,&strCmds,&strCluster,&programs,&scripts);
	scriptBuffer=ParseNodeConfigProps(&props,&flags,node,&strCmds,&strCluster,&programs,&scripts);
	editScripts = scripts;
	bOmit = ProcOmit(propBuffer);

	// Update the Context browser
	ResetContext();

	if (numNodes==1)
	{
		EnableWindow(GetDlgItem(hwnd,IDC_CONTEXTLIST),true);
		//EnableWindow(GetDlgItem(hwnd,IDC_OMIT),true);

		Link<ConfigScript>* curscript = editScripts.GetHead();
		int index;

		while(curscript)
		{
			index = SendDlgItemMessage(hwnd,IDC_CONTEXTLIST,CB_ADDSTRING,0,(LPARAM)(char*)curscript->data.name);
			SendDlgItemMessage(hwnd,IDC_CONTEXTLIST,CB_SETITEMDATA,(WPARAM)index,(LPARAM)curscript);

			curscript = curscript->next;
		}
	}
	else
	{
		EnableWindow(GetDlgItem(hwnd,IDC_CONTEXTLIST),false);
		EnableWindow(GetDlgItem(hwnd,IDC_OMIT),false);
	}

	if (propBuffer.Length()==0)
		bCopyProps=true;

	// Add the parsed data to the UI
	CStr ClassProp,TypeProp;

	// Set to correct defaults if no data in node
	ClassProp = GetClassProp(node);
	TypeProp  = GetTypeProp(node);

	AddGlobalDefaults(propBuffer,&props,&flags,&strCmds,ClassProp,TypeProp);

	// Check dynamic UI data against script.ini
	GetDynUIProps(&props,ClassProp,TypeProp);

	// Convert default delimited '!' properties to their actual defaults
	ConvertToDefaults(ClassProp,TypeProp,&props);

	// Get the program line from the default database (if appropriate)
	if (programs.GetSize()==0)
		GetPrograms(&programs,ClassProp,TypeProp);

	// Prevent copy request that would destroy the data we just parsed with
	// the default data in configDB
	bLockPropBuffer=TRUE;

	SetDlgItemText(hwnd,IDC_OBJCLASSM,ClassProp);
	int index=SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)ClassProp);
	
	if (index!=CB_ERR)
	{
		SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_SETCURSEL,index,0);
		UpdateClassSelection(FALSE);
	}
	
	SetDlgItemText(hwnd,IDC_OBJTYPEM,TypeProp);
	index=SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)TypeProp);

	if (index!=CB_ERR)
		SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_SETCURSEL,index,0);

	// Allow property copy requests again
	bLockPropBuffer=FALSE;

	if (bCopyProps)
		CopyProps();

	SetText(scriptBuffer);
	DefaultPropDlg(TRUE);
	UpdatePropView();

	// Reset changes made flag
	bChangeMade=FALSE;
	ResetModFlags();

	return true;
}

bool PropEditor::UpdateDefaultCondition(PropList* pPropList,int ID)
{ FUNC_ENTER("PropEditor::UpdateDefaultCondition"); 
	// Possibly update color if object returned to default property state
	CStr defvalue;

	PropColor color = propView->GetColor(ID);
	CStr       name = propView->GetName(ID);	
	CStr      value;

	pPropList->GetValue(ID,value);
	
	defvalue=GetDefault(GetClass(),GetType(),name);

	//if (defvalue==CStr("") && name!=CStr("Cluster"))
	//	return false;

	// Check if restore to default code was used
	if (value==CStr(vRESTORE_DEFAULT_SEQ))
	{
		propView->SetValue(ID,defvalue);
		
		if (color==PROPCOLOR_GRAY)
			propView->SetColor(ID,PROPCOLOR_GREEN);
	}

	if (pPropList->GetType(ID) == PROPTYPE_SPINEDIT)
	{
		if (atof(value) == atof(defvalue))
		{
			if (color == PROPCOLOR_GRAY)
				pPropList->SetColor(ID, PROPCOLOR_GREEN);
		}
		else
		{
			if (color == PROPCOLOR_GREEN)
				propView->SetColor(ID, PROPCOLOR_GRAY);
		}
	}
	else
	{
		if (value==defvalue)
		{
			if (color==PROPCOLOR_GRAY)
				propView->SetColor(ID,PROPCOLOR_GREEN);
		}
		else
		{
			if (color==PROPCOLOR_GREEN)
				propView->SetColor(ID,PROPCOLOR_GRAY);
		}	
	}

	return true;
}

void PropEditor::PropViewChangeCB(PropList* pPropList,void* pData)
{ FUNC_ENTER("PropEditor::PropViewChangeCB"); 
	PropEditor* pthis=(PropEditor*)pData;
	pthis->bChangeMade=TRUE;	
	
	// Check the cluster to see if we need to update TrickObject
/*
	if (pthis->propView->GetLastMod()==PROP_TRICKOBJECT)
	{
		CStr strVal;
		pPropList->GetValue(PROP_TRICKOBJECT,strVal);

		if (strVal==CStr("FALSE"))
			pPropList->SetValue(PROP_CLUSTER,"");

		pthis->UpdateDefaultCondition(pPropList,PROP_CLUSTER);
	}
	
	if (pthis->propView->GetLastMod()==PROP_CLUSTER)
	{
		CStr strVal;
		pPropList->GetValue(PROP_CLUSTER,strVal);

		if (strVal==CStr(""))
			pPropList->SetValue(PROP_TRICKOBJECT,"FALSE");
		else
			pPropList->SetValue(PROP_TRICKOBJECT,"TRUE");

		pPropList->SetMod(PROP_TRICKOBJECT,true);

		pthis->UpdateDefaultCondition(pPropList,PROP_TRICKOBJECT);
	}
*/

	// Check the TrickObjects default state
	pthis->UpdateDefaultCondition(pPropList,pthis->propView->GetLastMod());

	// Call any attached MAXScripts
	// (These scripts should only be called if a single object is set to be updated, if its
	//  called with multiple objects being selected the possibility exists for the script
	//  to be ran on objects of differing types)

	if (SendDlgItemMessage(pthis->hwnd, IDC_NODELIST, LB_GETSELCOUNT, 0, 0) == 1)
	{
		Link<ConfigProgram>* link = pthis->programs.GetHead();
		CStr temp;

		// Post the node currently selected in the node list so MAXScript can access it
		int item;
		SendDlgItemMessage(pthis->hwnd, IDC_NODELIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);
		INode* selNode = (INode*)SendDlgItemMessage(pthis->hwnd, IDC_NODELIST, LB_GETITEMDATA, (WPARAM)item, 0);
		
		pthis->PostMAXScriptNode(selNode);

		//if (pthis->ip->GetSelNodeCount()>0)
		//	pthis->PostMAXScriptNode(pthis->ip->GetSelNode(0));

		while(link)
		{
			pthis->ClearAltScripts();
			//pthis->AllocMemMaps();
		
			// Don't update scripts unless there is an active node to act upon
			if (pthis->ip->GetSelNodeCount()>0)
			{
				pthis->UpdateDynUIScripts(pPropList,link->data.filename);

				// Update the script window with a template (if appropriate)
				temp += pthis->GetColorTemplate();
				if (temp!=CStr(""))
				{
					pthis->bSyntaxHighlighting=FALSE;
					pthis->pRichText->Disable();

					char eol[2];
					eol[0] = 13;
					eol[1] = 0;

					temp = ReplaceStr(temp,CStr(eol),SRTF_ENDLINE);

					temp = SRTF_COLOR_HEADER + SRTF_DEFAULT + SRTF_ITALIC + SRTF_BOLD +
						   temp + SRTF_BOLDOFF + SRTF_ITALICOFF + SRTF_END;

					pthis->pRichText->SetText(temp);
				}
				else
				{
					pthis->bSyntaxHighlighting=TRUE;
					pthis->pRichText->Enable();
				}
			}
			else
			{
				pthis->bSyntaxHighlighting=FALSE;
				pthis->pRichText->Disable();

				CStr temp;
				temp = SRTF_COLOR_HEADER + SRTF_DEFAULT + SRTF_ITALIC + SRTF_BOLD +
					   CStr("A node must be selected for DynUI's attached to templated scripts.") + 
					   SRTF_BOLDOFF + SRTF_ITALICOFF + SRTF_END;

				pthis->pRichText->SetText(temp);
			}

			link = link->next;
		}
	}

	pthis->lastProps.Clear();
	pPropList->GetValues(&pthis->lastProps);

	// Handle command lists
	int lastMod = pPropList->GetLastMod();
	LinkList<PropCmd>* cmdList = pPropList->GetCmdList(lastMod);
	Link<PropCmd>* curCmd = cmdList->GetHead();
	CStr value;
	
	if (curCmd)
	{
		if (pPropList->GetValue(lastMod, value))
		{
			while(curCmd)
			{
				switch(curCmd->data.id)
				{
				case CMDTYPE_OMITON:
					if (curCmd->data.params[0] == value)
						CheckDlgButton(pthis->hwnd, IDC_OMIT, BST_CHECKED);

					break;

				case CMDTYPE_OMITOFF:
					if (curCmd->data.params[0] == value)
						CheckDlgButton(pthis->hwnd, IDC_OMIT, BST_UNCHECKED);

					break;
				}

				curCmd = curCmd->next;
			}
		}
	}

	// If ImmediateApply is checked then apply the properties now
	if (pthis->bImmediateApply && !pthis->bLockImmediateApply)
		pthis->Apply(FALSE);
}

void PropEditor::PropViewDblClickCB(PropList* pPropList,int index,void* pData)
{ FUNC_ENTER("PropEditor::PropViewDblClickCB"); 
	PropEditor* pthis=(PropEditor*)pData;

	if (pthis->pCompareDlg)
	{
		POINT pos;
		GetCursorPos(&pos);

		pthis->pCompareDlg->Move(pos.x,pos.y);
		pthis->AddConflicts(pPropList,index);
		pthis->pCompareDlg->Show();
	}
}

void PropEditor::AddConflicts(PropList* plist,int index)
{ FUNC_ENTER("PropEditor::AddConflicts"); 
	pCompareDlg->Clear();
	pCompareDlg->Attach(plist,index);

	HWND nodelist=GetDlgItem(hwnd,IDC_NODELIST);

	int                   numSels=SendMessage(nodelist,LB_GETSELCOUNT,0,0);
	INT*                  sels=new INT[numSels];
	INode*                compNode;
	CStr                  propBuffer;
	CStr                  compClass,compType;
	LinkList<ConfigProp>  cprops;
	CStr                  scriptBuffer;
	DWORD                 cflags;
	CStr                  strPropName;
	CStr                  strPropValue;
	CStr                  strClust;
	CStr                  strCmds;

	strPropName=plist->GetName(index);
	plist->GetValue(index,strPropValue);
	pCompareDlg->Clear();

	SendMessage(nodelist,LB_GETSELITEMS,(WPARAM)numSels,(LPARAM)sels);

	for(int i=0;i<numSels;i++)
	{
		cprops.Clear();

		compNode=(INode*)SendMessage(nodelist,LB_GETITEMDATA,(WPARAM)sels[i],0);
		compNode->GetUserPropBuffer(propBuffer);
		//AddGlobalDefaults(propBuffer);
		//XXscriptBuffer=ParseConfigProps(&cprops,&cflags,propBuffer,&strCmds,&strClust);
		scriptBuffer=ParseNodeConfigProps(&cprops,&cflags,compNode,&strCmds,&strClust);

		// Get class and type so we can look up the defaults
		compClass = GetClassProp(compNode);
		compType  = GetTypeProp(compNode);

		AddGlobalDefaults(propBuffer,&cprops,&cflags,&strCmds,compClass,compType);

		if (propBuffer.Length()==0)
		{
			CStr defaultBuf=GetClassTypeRecord(compClass,compType);
			scriptBuffer=ParseConfigProps(&cprops,&cflags,defaultBuf,NULL,&strClust);
		}

		// Check dynamic UI data against script.ini
		GetDynUIProps(&cprops,compClass,compType);

		ConvertToDefaults(compClass,compType,&cprops);

		// Handle flags
/*
		if (index==PROP_CREATEDATSTART)	// CCLASS_CREATEDATSTART
		{
			if (cflags & CCLASS_CREATEDATSTART)
				pCompareDlg->AddConflict(compNode->GetName(),"TRUE");
			else
				pCompareDlg->AddConflict(compNode->GetName(),"FALSE");
		}
	
		if (index==PROP_ABSENTINNETGAMES)  // CCLASS_ABSENTINNETGAMES
		{
			if (cflags & CCLASS_ABSENTINNETGAMES)
				pCompareDlg->AddConflict(compNode->GetName(),"TRUE");
			else
				pCompareDlg->AddConflict(compNode->GetName(),"FALSE");
		}

		if (index==PROP_TRICKOBJECT)  // CCLASS_TRICKOBJECT
		{
			if (cflags & CCLASS_TRICKOBJECT)
				pCompareDlg->AddConflict(compNode->GetName(),"TRUE");
			else
				pCompareDlg->AddConflict(compNode->GetName(),"FALSE");
		}

		if (index==PROP_CLUSTER)
		{
			pCompareDlg->AddConflict(compNode->GetName(),strClust);
		}

		if (index==PROP_OCCLUDER)
		{
			if (cflags & CCLASS_OCCLUDER)
				pCompareDlg->AddConflict(compNode->GetName(),"TRUE");
			else
				pCompareDlg->AddConflict(compNode->GetName(),"FALSE");
		}
*/

		Link<ConfigProp>* curNode=cprops.GetHead();

		while(curNode)
		{
			if (curNode->data.name==strPropName)
				pCompareDlg->AddConflict(compNode->GetName(),curNode->data.value);

			curNode=curNode->next;
		}
	}
}

void PropEditor::DefaultPropDlg(BOOL bKeepActive)
{ FUNC_ENTER("PropEditor::DefaultPropDlg"); 
	RECT  rect;
	int   x,y;
	HMENU hMenuBar=GetMenu(hwnd);

	GetWindowRect(hwnd,&rect);
	x=rect.left;
	y=rect.top+20;

	if (!bKeepActive)
	{
		if (pDefaultProps)
		{	
			// Toggle the check mark on the menu off
			CheckMenuItem(hMenuBar,IDM_DEFAULTPROPS,MF_UNCHECKED);

			// Close the property list
			delete pDefaultProps;
			pDefaultProps=NULL;

			// Reestablish the internal property view window
			HWND hwndPVUI;
			RECT rectPVUI;
			POINT ULpos,BRpos;
			
			hwndPVUI=GetDlgItem(hwnd,IDC_ORIGSIZE);
			GetWindowRect(hwndPVUI,&rectPVUI);
			
			ULpos.x=rectPVUI.left;
			ULpos.y=rectPVUI.top;
			BRpos.x=rectPVUI.right;
			BRpos.y=rectPVUI.bottom;

			ScreenToClient(hwnd,&ULpos);
			ScreenToClient(hwnd,&BRpos);

			hwndPVUI=GetDlgItem(hwnd,IDC_RICHEDIT1);
			MoveWindow(hwndPVUI,
					   ULpos.x,
					   ULpos.y,
					   BRpos.x-ULpos.x,
					   BRpos.y-ULpos.y,
					   TRUE);

			hwndPVUI=GetDlgItem(hwnd,IDC_PVFRAME);
			ShowWindow(hwndPVUI,SW_SHOW);

			// Reconnect propview window
			propView=oldPropView;
			//CopyProps();
			//UpdatePropView();
			SelectionSetChanged(TRUE);

			hwndPVUI=GetDlgItem(hwnd,IDC_PROPVIEW);
			ShowWindow(hwndPVUI,SW_SHOW);
			return;
		}
		else
		{
			// Toggle the check mark on the menu on
			CheckMenuItem(hMenuBar,IDM_DEFAULTPROPS,MF_CHECKED);
		}
	}
	else
	{
		if (!pDefaultProps)
			return;

		x=pDefaultProps->GetX();
		y=pDefaultProps->GetY();

		pDefaultProps->SetX(x);
		pDefaultProps->SetY(y);
		pDefaultProps->SetWidth(FLOATPROP_WIDTH);
		pDefaultProps->SetHeight(FLOATPROP_HEIGHT);
		return;
	}

	// Create the new property list window
	pDefaultProps=new MultiPropList(hInstance,hwnd,x,y,FLOATPROP_WIDTH,FLOATPROP_HEIGHT,"Default Properties");
	pDefaultProps->HasApply(false);
	pDefaultProps->SetChangeCB(PropViewChangeCB,this);
	pDefaultProps->SetEscapeCB(EscapeCB,this);
	pDefaultProps->SetDblClickCB(PropViewDblClickCB,this);
	pDefaultProps->SetStaticCB(SubPropProc,this);


	if (propView)
		oldPropView=propView;

	HWND hwndPVUI;
	RECT rectPVUI;
	POINT ULpos,BRpos;

	// Remove the Property list from the GUI since it's now floating
	hwndPVUI=GetDlgItem(hwnd,IDC_PROPVIEW);
	ShowWindow(hwndPVUI,SW_HIDE);
	hwndPVUI=GetDlgItem(hwnd,IDC_PVFRAME);
	ShowWindow(hwndPVUI,SW_HIDE);
	
	hwndPVUI=GetDlgItem(hwnd,IDC_FULLSIZE);
	GetWindowRect(hwndPVUI,&rectPVUI);
	
	ULpos.x=rectPVUI.left;
	ULpos.y=rectPVUI.top;
	BRpos.x=rectPVUI.right;
	BRpos.y=rectPVUI.bottom;

	ScreenToClient(hwnd,&ULpos);
	ScreenToClient(hwnd,&BRpos);

	hwndPVUI=GetDlgItem(hwnd,IDC_RICHEDIT1);
	MoveWindow(hwndPVUI,
		       ULpos.x,
			   ULpos.y,
			   BRpos.x-ULpos.x,
			   BRpos.y-ULpos.y,
			   TRUE);

	propView=pDefaultProps;
	//CopyProps();
	//UpdatePropView();
	SelectionSetChanged(TRUE);
}

void PropEditor::UpdatePropView()
{ FUNC_ENTER("PropEditor::UpdatePropView"); 
	bSyntaxHighlighting=TRUE;
	pRichText->Enable();
	propView->Clear();

//	propView->AddCheck("CreatedAtStart","The object is created when the game starts.");
//	propView->AddCheck("AbsentInNetGames","The object isn't created when playing network games.");
//	propView->AddCheck("TrickObject","Unknown");
//	propView->AddNode("Cluster","Cluster assigned if TrickObject");
//	propView->AddCheck("Occluder","...");

	// Add all the parsed properties to this list
	Link<ConfigProp>* curProp=props.GetHead();

	while(curProp)
	{
		AddPropViewProp(&curProp->data);
		curProp=curProp->next;
	}

	lastProps.Clear();
	propView->GetValues(&lastProps);
	
	bLockImmediateApply=TRUE;
	propView->BuildUI();
	
	lastProps.Clear();
	propView->GetValues(&lastProps);

	// Assign property values
	curProp=props.GetHead();

	// Assign checks if flags are set
//	if (strCluster.Length()>0)
//	{
//		flags|=CCLASS_TRICKOBJECT;
//		propView->SetValue(PROP_TRICKOBJECT,"TRUE");
//	}

//	propView->SetValue(PROP_CLUSTER,strCluster);

//	if (flags & CCLASS_CREATEDATSTART)
//		propView->SetValue(PROP_CREATEDATSTART,"TRUE");

//	if (flags & CCLASS_ABSENTINNETGAMES)
//		propView->SetValue(PROP_ABSENTINNETGAMES,"TRUE");

//	if (flags & CCLASS_TRICKOBJECT)
//		propView->SetValue(PROP_TRICKOBJECT,"TRUE");

//	if (flags & CCLASS_OCCLUDER)
//		propView->SetValue(PROP_OCCLUDER,"TRUE");

	int index=0;
	while(curProp)
	{
		propView->SetValue(index,curProp->data.value);
		propView->SetUserData(index,curProp->data.extdata);
		index++;
		curProp=curProp->next;
	}

	propView->ResetUpdateStatus();

	// Update default highlighting
	CompareDefaults(GetClass(),GetType());
	bLockImmediateApply=FALSE;
}

CStr PropEditor::GetClass()
{ FUNC_ENTER("PropEditor::GetClass"); 
	// Have to grab object class and object type from the current selection in the combo box
	// because the text update isn't immediate

	CStr strClass;
	char buf[256];
	int  index;

	index=SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_GETCURSEL,0,0);
	
	if (index==CB_ERR)
	{
		GetDlgItemTextLast(hwnd,IDC_OBJCLASSM,buf,256);
		return CStr(buf);
	}

	SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_GETLBTEXT,(WPARAM)index,(LPARAM)buf);
	strClass=CStr(buf);

	return strClass;
}

CStr PropEditor::GetType()
{ FUNC_ENTER("PropEditor::GetType"); 
	// Have to grab object class and object type from the current selection in the combo box
	// because the text update isn't immediate

	CStr strType;
	char buf[256];
	int  index;

	index=SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_GETCURSEL,0,0);

	if (index==CB_ERR)
	{
		GetDlgItemText(hwnd,IDC_OBJTYPEM,buf,256);
		return CStr(buf);
	}
	
	SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_GETLBTEXT,(WPARAM)index,(LPARAM)buf);
	strType=CStr(buf);

	return strType;
}

bool PropEditor::AutoSelect()
{ FUNC_ENTER("PropEditor::AutoSelect"); 
	HMENU hMenuBar=GetMenu(hwnd);

	bAutoSelect=!bAutoSelect;

	if (bAutoSelect)
		CheckMenuItem(hMenuBar,IDM_AUTOSELECT,MF_CHECKED);		
	else
		CheckMenuItem(hMenuBar,IDM_AUTOSELECT,MF_UNCHECKED);

	return bAutoSelect;
}

bool PropEditor::ToggleNameExpansion()
{ FUNC_ENTER("PropEditor::ToggleNameExpansion"); 
	HMENU hMenuBar=GetMenu(hwnd);

	bNameExpansion=!bNameExpansion;

	if (bNameExpansion)
		CheckMenuItem(hMenuBar,IDM_NAMEEXPAND,MF_CHECKED);
	else
		CheckMenuItem(hMenuBar,IDM_NAMEEXPAND,MF_UNCHECKED);

	return bNameExpansion;
}

bool PropEditor::SubSelApply()
{ FUNC_ENTER("PropEditor::SubSelApply"); 
	HMENU hMenuBar=GetMenu(hwnd);

	bSubSelApply=!bSubSelApply;

	if (bSubSelApply)
		CheckMenuItem(hMenuBar,IDM_SUBSELAPPLY,MF_CHECKED);		
	else
		CheckMenuItem(hMenuBar,IDM_SUBSELAPPLY,MF_UNCHECKED);

	return bSubSelApply;
}

bool PropEditor::SubSelRestore()
{ FUNC_ENTER("PropEditor::SubSelRestore"); 
	HMENU hMenuBar=GetMenu(hwnd);

	bSubSelRestore=!bSubSelRestore;

	if (bSubSelRestore)
		CheckMenuItem(hMenuBar,IDM_SUBSELRESTORE,MF_CHECKED);		
	else
		CheckMenuItem(hMenuBar,IDM_SUBSELRESTORE,MF_UNCHECKED);

	return bSubSelRestore;
}

bool PropEditor::ApplyChangedOnly()
{ FUNC_ENTER("PropEditor::ApplyChangedOnly"); 
	HMENU hMenuBar=GetMenu(hwnd);

	bApplyChangedOnly=!bApplyChangedOnly;

	if (bApplyChangedOnly)
		CheckMenuItem(hMenuBar,IDM_APPLYCHANGED,MF_CHECKED);
	else
		CheckMenuItem(hMenuBar,IDM_APPLYCHANGED,MF_UNCHECKED);

	return bApplyChangedOnly;
}

bool PropEditor::ForceScriptUpdates()
{ FUNC_ENTER("PropEditor::ForceScriptUpdates"); 
	HMENU hMenuBar=GetMenu(hwnd);

	bForceScriptUpdates = !bForceScriptUpdates;

	if (bForceScriptUpdates)
		CheckMenuItem(hMenuBar,IDM_FORCESCRIPTUPDATE,MF_CHECKED);
	else
		CheckMenuItem(hMenuBar,IDM_FORCESCRIPTUPDATE,MF_UNCHECKED);

	return bForceScriptUpdates;
}

bool PropEditor::ImmediateApply()
{ FUNC_ENTER("PropEditor::ImmediateApply"); 
	HMENU hMenuBar = GetMenu(hwnd);

	bImmediateApply = !bImmediateApply;

	if (bImmediateApply)
		CheckMenuItem(hMenuBar,IDM_IMMEDIATEAPPLY,MF_CHECKED);
	else
		CheckMenuItem(hMenuBar,IDM_IMMEDIATEAPPLY,MF_UNCHECKED);

	return bImmediateApply;
}

bool PropEditor::RefreshViewport()
{ FUNC_ENTER("PropEditor::RefreshViewport"); 
	HMENU hMenuBar = GetMenu(hwnd);

	bRefreshViewport = !bRefreshViewport;

	if (bRefreshViewport)
		CheckMenuItem(hMenuBar,IDM_REFRESHVIEWPORT,MF_CHECKED);
	else
		CheckMenuItem(hMenuBar,IDM_REFRESHVIEWPORT,MF_UNCHECKED);

	return bRefreshViewport;
}

bool PropEditor::AddConflict()
{ FUNC_ENTER("PropEditor::AddConflict"); 
	HMENU hMenuBar=GetMenu(hwnd);

	bAddConflict=!bAddConflict;

	if (bAddConflict)
		CheckMenuItem(hMenuBar,IDM_ADDCONFLICT,MF_CHECKED);
	else
		CheckMenuItem(hMenuBar,IDM_ADDCONFLICT,MF_UNCHECKED);

	return bAddConflict;
}

void PropEditor::CopyProps(BOOL bSetScript)
{ FUNC_ENTER("PropEditor::CopyProps"); 
	Link<ConfigClass>* lclass=NULL;
	Link<ConfigType>*  ltype=NULL;
	CStr               propBuffer;

	// Don't copy properties if the buffer is locked
	if (bLockPropBuffer)
		return;

	props.Clear();
	strCmds="";
	strCluster="";
	flags=0;

	// Dump out configDB flags
	int clsindex=SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_GETCURSEL,0,0);

	if (clsindex!=CB_ERR)
	{
		lclass=(Link<ConfigClass>*)SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_GETITEMDATA,clsindex,0);

		DEBUGDUMP(lclass->data.strCmds);

		if (lclass)
			propBuffer=lclass->data.strCmds;
	}

	// Dump out properties from configDB
	int index=SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_GETCURSEL,0,0);

	if (index!=CB_ERR)
	{
		ltype=(Link<ConfigType>*)SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_GETITEMDATA,index,0);

		//if (ltype)
		//	propBuffer+=ltype->data.strCmds;

		propBuffer = CombinePropBuffers(propBuffer, ltype->data.strCmds, lclass->data.name, ltype->data.name);
	}

	AddGlobalDefaults(propBuffer);
	CStr strScript;
	
	strScript=ParseConfigProps(&props,&flags,propBuffer,&strCmds,&strCluster,&programs,&scripts);

	AddGlobalDefaults(propBuffer,&props,&flags,&strCmds,CStr(DEFAULT_CLASS),CStr(""));
	bOmit = ProcOmit(propBuffer);

	if (bSetScript)
	{
		pRichText->SetText(strScript);
		editScript = strScript;
		editScripts = scripts;
	}

	if (lclass)
	{
		if (ltype)
			ConvertToDefaults(lclass->data.name,ltype->data.name,&props);
		else
			ConvertToDefaults(lclass->data.name,"",&props);
	}
}

void PropEditor::GetPropViewProps()
{ FUNC_ENTER("PropEditor::GetPropViewProps"); 
	// Copy flags out of the property list
	flags=0;

	CStr Result;
	/*
	propView->GetValue(PROP_CREATEDATSTART,Result);

	if (Result==CStr("TRUE"))
		flags|=CCLASS_CREATEDATSTART;

	propView->GetValue(PROP_ABSENTINNETGAMES,Result);

	if (Result==CStr("TRUE"))
		flags|=CCLASS_ABSENTINNETGAMES;

	propView->GetValue(PROP_TRICKOBJECT,Result);

	if (Result==CStr("TRUE"))
		flags|=CCLASS_TRICKOBJECT;

	propView->GetValue(PROP_OCCLUDER,Result);

	if (Result==CStr("TRUE"))
		flags|=CCLASS_OCCLUDER;
*/

	// Copy the data out of the property list back into the props list
	int numProps=propView->NumProps();
	assert(numProps>=PROPSTART-1);

	
	props.Clear();
	
	// Get the property value for cluster
/*
	ConfigProp cprop;
	cprop.name=propView->GetName(PROP_CLUSTER);
	cprop.extdata=propView->GetUserData(PROP_CLUSTER);
	propView->GetValue(PROP_CLUSTER,cprop.value);
*/

	for(int i=PROPSTART;i<numProps;i++)
	{
		ConfigProp cprop;
		cprop.name=propView->GetName(i);
		cprop.extdata=propView->GetUserData(i);
		propView->GetValue(i,cprop.value);
		props.Add(&cprop);
	}
}

void PropEditor::GetDefaultProps()
{ FUNC_ENTER("PropEditor::GetDefaultProps"); 
	if (!pDefaultProps)
		GetPropViewProps();
		return;

	// Copy flags out of the property list
	flags=0;
	
	CStr Result;
	/*
	pDefaultProps->GetValue(PROP_CREATEDATSTART,Result);

	if (Result==CStr("TRUE"))
		flags|=CCLASS_CREATEDATSTART;

	pDefaultProps->GetValue(PROP_ABSENTINNETGAMES,Result);

	if (Result==CStr("TRUE"))
		flags|=CCLASS_ABSENTINNETGAMES;

	pDefaultProps->GetValue(PROP_TRICKOBJECT,Result);

	if (Result==CStr("TRUE"))
		flags|=CCLASS_TRICKOBJECT;

	pDefaultProps->GetValue(PROP_OCCLUDER,Result);

	if (Result==CStr("TRUE"))
		flags|=CCLASS_OCCLUDER;
	*/

	// Copy the data out of the property list back into the props list
	int numProps=pDefaultProps->NumProps();
	assert(numProps>=PROPSTART-1);

	Link<ConfigProp>* lprop=props.GetHead();

	for(int i=PROPSTART;i<numProps;i++)
	{
		pDefaultProps->GetValue(i,lprop->data.value);
		lprop=lprop->next;
	}
}

void PropEditor::LoadDlg()
{ FUNC_ENTER("PropEditor::LoadDlg"); 
	CStr FileName;

	if (pRichText->HasText())
	{
		int rVal=MessageBox(hwnd,"This operation will clear the current script loaded in the editor.\nAre you sure you want to do this?","Load External File",MB_YESNO|MB_ICONQUESTION);

		if (rVal==IDNO)
			return;
	}
	
	pRichText->LoadDlg("Load Script File","Script Files (*.q)\0*.q\0All Files (*.*)\0*.*\0\0",".q",FileName);

	if (FileName.Length()>0)
		if (LoadText(FileName))
		{
			SetTitle(TITLEMODE_FILE,FileName);
		}
}

void PropEditor::SaveDlg()
{ FUNC_ENTER("PropEditor::SaveDlg"); 
	CStr FileName;

	if (!pRichText->HasText())
		return;

	pRichText->SaveDlg("Save Script File","*.q");
}

void PropEditor::FindScript()
{ FUNC_ENTER("PropEditor::FindScript"); 
	if (pScriptBrowse)
		delete pScriptBrowse;

	pScriptBrowse=new ScriptBrowse(hInstance,hwnd,this);
}

void PropEditor::FindNode()
{ FUNC_ENTER("PropEditor::FindNode"); 
	// This is a modal dialog so this should block
	// Must lock selection so we don't compare until we select the last node
	bLockSelection = true;
	pFindNode->Show();
	bLockSelection = false;
	SelectionSetChanged();
}

void PropEditor::CloseChildren()
{ FUNC_ENTER("PropEditor::CloseChildren"); 
	// Shutdown MultiSearch
	if (pMSearch)
	{
		delete pMSearch;
		pMSearch=NULL;
	}
	
	// Shutdown popup parameter list
	if (pPopupParams)
	{
		//delete pPopupParams;
		pPopupParams->Hide();
	}

	// Shutdown default properties param sheet
	if (pDefaultProps)
	{
		DefaultPropDlg(FALSE);

		if (pDefaultProps)
		{
			delete pDefaultProps;
			pDefaultProps=NULL;
			propView=oldPropView;
		}
	}

	// Shutdown the script browser
	if (pScriptBrowse)
	{
		delete pScriptBrowse;
		pScriptBrowse=NULL;
	}

	// Shutdown Find in files
	if (pFindFilesDlg)
		pFindFilesDlg->Hide();

	// Shutdown FindANode(tm)
	if (pFindNode)
		pFindNode->Hide();

	// Shutdown External script dialog
	if (pExtScriptDlg)
		pExtScriptDlg->Hide();

	// Shutdown Compare Popup Dialog
	if (pCompareDlg)
		pCompareDlg->Hide();

	// Shutdown AddProp Dialog
	if (pAddPropDlg)
		pAddPropDlg->Hide();

	// Shutdown Rename Dialog
	if (pRenameDlg)
		pRenameDlg->Hide();
}

bool PropEditor::ErrorCheck()
{ FUNC_ENTER("PropEditor::ErrorCheck"); 
	// Don't do error check if there's no text
	if (!pRichText->HasText())
		return false;

	// Build qcomp directory path string
	CStr strQCompDir=getenv(APP_ENV);

	// Verify that the environment variable is set
	if (strQCompDir==CStr(""))
	{
		char ErrorBuf[256];
		sprintf(ErrorBuf,"The '%s' environment variable is not set.",APP_ENV);

		MessageBox(NULL,ErrorBuf,"Can't find 'qcomp.exe'",MB_ICONSTOP|MB_OK);
		return false;
	}

	strQCompDir+=QCOMP_PATH;
	
	// Save the script file to disk as a temporary file so we can compile it
	CStr strSave=CStr(getenv(APP_ENV))+"\\temp.qn";
	
	FILE* fp=fopen(strSave,"w");

	fprintf(fp,"script TestScript\n");

	if (!fp || !pRichText->SaveTXT(fp))
	{
		MessageBox(hwnd,"Unable to save temporary script file 'temp.qn'","Script Compile/ErrorCheck",MB_ICONWARNING|MB_OK);
		return false;
	}

	fprintf(fp,"\nendscript\n");
	fclose(fp);

	// Make sure that any left over error file has been deleted
	DeleteFile("qcomp.err");

	// Spawn the qcomp compiler
	// Have to use ShellExecuteEx or CreateProcess so we get a handle 
	// to the process and can thus know when it's safe to read the error file in
	// Alt: For error pipe redirection
	CStr strParam=strSave+" -nomessageboxes";
	CStr strQCompEXE=strQCompDir+"\\qcomp.exe";

	SHELLEXECUTEINFO sei;
	ZeroMemory(&sei,sizeof(SHELLEXECUTEINFO));
	sei.cbSize=sizeof(SHELLEXECUTEINFO);
	sei.fMask=SEE_MASK_NOCLOSEPROCESS;
	sei.hwnd=hwnd;
	sei.lpVerb="open";
	//sei.lpFile="qcomp";
	sei.lpFile=(char*)strQCompEXE;
	sei.lpParameters=(char*)strParam;
	sei.lpDirectory=NULL;
	sei.nShow=SW_HIDE;
	
	if (!ShellExecuteEx(&sei))
	{
		DWORD err=GetLastError();
		MessageBox(hwnd,"Error spawning qcomp script compiler.","Script Compile/ErrorCheck",MB_ICONWARNING|MB_OK);
		return false;
	}

	// Wait for completion (or 10 secs. 10000 ms)
	if (WaitForInputIdle(sei.hProcess,10000)==WAIT_TIMEOUT)
	{
		MessageBox(hwnd,"Script compilation timed out after 10 seconds.","Script Compile/ErrorCheck",MB_ICONWARNING|MB_OK);
		return false;
	}

	FILE* errFile;

	errFile=fopen("qcomp.err","r");

	// If the file doesn't exist exit (no errors)
	if (!errFile)
		return false;

	// Parse the error
	char buf[128];
	char errline[128];
	char errname[128];
	int  line;
	
	fgets(buf,128,errFile);		// ignore 1st line
	fgets(errline,128,errFile);
	fgets(errname,128,errFile);

	line=atoi(errline)-1;
	_itoa(line,errline,10);

	fclose(errFile);

	// Delete temporary .qn file
	DeleteFile((char*)strSave);

	// Select the line containing the error
	pRichText->HighlightLineNum(line-1);
	sprintf(buf,"%s%s",errname,errline);
	MessageBox(hwnd,buf,"QComp compilation error",MB_ICONWARNING|MB_OK);

	return true;
}

void PropEditor::NodeListRestore()
{ FUNC_ENTER("PropEditor::NodeListRestore"); 
	HWND nodelist=GetDlgItem(hwnd,IDC_NODELIST);

	int  numSels=SendMessage(nodelist,LB_GETSELCOUNT,0,0);
	INT* sels=new INT[numSels];
		
	SendMessage(nodelist,LB_GETSELITEMS,(WPARAM)numSels,(LPARAM)sels);

	// Find fist selection with valid compare settings
	CStr compClass;
	CStr compType;
	int  i;

	if (numSels==0)
		return;

	if (numSels==1)
	{
		// Get the node information
		INode* selNode=(INode*)SendMessage(nodelist,LB_GETITEMDATA,(WPARAM)sels[0],0);
		ParseUserProps(selNode);

		// Acquire class and type for highlighting
		if (!selNode->GetUserPropString("Class",compClass))
			selNode->GetUserPropString("class",compClass);

		if (!selNode->GetUserPropString("Type",compType))
			selNode->GetUserPropString("type",compType);

		// Highlight defaults
		bClassConflict=FALSE;
		bTypeConflict=FALSE;

		CompareDefaults(compClass,compType);
		return;
	}

	bLockClassUpdate=TRUE;

	INode* compNode;

	for(i=0;i<numSels;i++)
	{
		compNode=(INode*)SendMessage(nodelist,LB_GETITEMDATA,(WPARAM)sels[i],0);
		
		// Must have Class
		/*
		if (!compNode->GetUserPropString("Class",compClass))
			if (!compNode->GetUserPropString("class",compClass))
				continue;
		*/

		compClass = GetClassProp(compNode);
		compType  = GetTypeProp(compNode);
		
		ParseUserProps(compNode);
		break;
	}

	AddCompareProps();

	// Reset colors
	int numProps=propView->NumProps();

	for(i=0;i<numProps;i++)
		propView->SetColor(i,PROPCOLOR_GRAY);

	// Now see if the selected nodes share commonality
	bool bCommon=true;
	bool bCommonClass=true;
	bool bCommonType=true;
	bool bCommonScript;

	for(i=0;i<numSels;i++)
	{
		INode* node=(INode*)SendMessage(nodelist,LB_GETITEMDATA,(WPARAM)sels[i],0);
		
		// Skip the comparison node
		//if (node==compNode)
		//	continue;

		CStr nodeClass,nodeType;

		nodeClass = GetClassProp(node);
		nodeType  = GetTypeProp(node); 

		if (nodeClass != compClass)
		{
			bCommon=false;
			bCommonClass=false;
		}

		if (nodeType != compType)
		{
			bCommon=false;
			bCommonType=false;
		}

		if (!CompareProps(node))
			bCommon=false;
	}

	bCommonScript=CompareScripts();

	if (!bCommonScript)
		bCommon=false;

	if (bCommonClass)
	{
		SetDlgItemText(hwnd,IDC_OBJCLASSM,compClass);

		int index=SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)compClass);
		
		if (index!=CB_ERR)
		{
			SendDlgItemMessage(hwnd,IDC_OBJCLASSM,CB_SETCURSEL,index,0);
			UpdateClassSelection(FALSE);
		}
	}
	else
		SetDlgItemText(hwnd,IDC_OBJCLASSM,vUNKNOWN);

	if (bCommonType)
	{
		SetDlgItemText(hwnd,IDC_OBJTYPEM,compType);

		int index=SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)compType);

		if (index!=CB_ERR)
			SendDlgItemMessage(hwnd,IDC_OBJTYPEM,CB_SETCURSEL,index,0);
	}
	else
		SetDlgItemText(hwnd,IDC_OBJTYPEM,vUNKNOWN);

	if (!bCommonScript)
		pRichText->SetText(vUNKNOWN);

	if (bCommon)
	{
		if (numSels>1)
			SetWindowText(hwnd,"Property Editor - [NodeList: Multiple Consistent Objects Selected]");
	}
	else
	{

		SetWindowText(hwnd,"Property Editor - [NodeList: Multiple Inconsistent Objects Selected]");
	}

	bClassConflict=!bCommonClass;
	bTypeConflict=!bCommonType;

	CompareDefaults(compClass,compType);

	bLockClassUpdate=FALSE;
	UpdateClassSelection();		// was commented

	delete sels;
}

bool PropEditor::CompareScripts()
{ FUNC_ENTER("PropEditor::CompareScripts"); 
	LinkList<ConfigProp> cprops;
	DWORD cflags;
	CStr  compBuffer;
	CStr  scriptBuffer;
	CStr  propBuffer;
	
	HWND nodelist=GetDlgItem(hwnd,IDC_NODELIST);
	int  numSels=SendMessage(nodelist,LB_GETSELCOUNT,0,0);

	if (numSels<1)
		return true;

	INT* sels=new INT[numSels];
		
	SendMessage(nodelist,LB_GETSELITEMS,(WPARAM)numSels,(LPARAM)sels);
	
	// Get comparison script
	INode* compNode=(INode*)SendMessage(nodelist,LB_GETITEMDATA,(WPARAM)sels[0],0);
	//XXcompNode->GetUserPropBuffer(propBuffer);
	//XXcompBuffer=ParseConfigProps(&cprops,&cflags,propBuffer);
	compBuffer=ParseNodeConfigProps(&cprops,&cflags,compNode);

	for(int i=0;i<numSels;i++)
	{
		compNode=(INode*)SendMessage(nodelist,LB_GETITEMDATA,(WPARAM)sels[i],0);
		//XXcompNode->GetUserPropBuffer(propBuffer);
		//XXscriptBuffer=ParseConfigProps(&cprops,&cflags,propBuffer);
		scriptBuffer=ParseNodeConfigProps(&cprops,&cflags,compNode);

		// If both buffers are considered blank then they will not result in a false result
		if (!IsBlank(scriptBuffer) || !IsBlank(compBuffer))
			if (scriptBuffer!=compBuffer)
				return false;
	}

	return true;
}

bool PropEditor::CompareDefaults(CStr className,CStr typeName)
{ FUNC_ENTER("PropEditor::CompareDefaults"); 
	// Cannot compare defaults unless there is a class and type
	if (className==CStr(vUNKNOWN) ||
		typeName==CStr(vUNKNOWN)  ||
		bClassConflict ||
		bTypeConflict)
		return false;

	int numProps=propView->NumProps();

	for(int i=0;i<numProps;i++)
	{
		// All other colors take precedence, skip this (it's already colored)
		if (propView->GetColor(i)!=PROPCOLOR_GRAY)
			continue;

		CStr name;
		CStr value;
		CStr defvalue;
		
		name=propView->GetName(i);
		propView->GetValue(i,value);

		defvalue=GetDefault(className,typeName,name);

		// Spinner controls are a special case as we want their defaults to be accepted
		// based on their values rather than the default string itself (i.e. 20 should be the same as 20.0)
		if (propView->GetType(i) == PROPTYPE_SPINEDIT)
		{
			if (atof(value) == atof(defvalue))
				propView->SetColor(i, PROPCOLOR_GREEN);

			continue;
		}

		// Color code the value green if it matches the default value
		// Only color green if it's not red, blue, or purple (those take precedence)
		if (value==defvalue)
			propView->SetColor(i,PROPCOLOR_GREEN);
	}

	return true;
}

void PropEditor::AddCompareProps()
{ FUNC_ENTER("PropEditor::AddCompareProps"); 
	HWND nodelist=GetDlgItem(hwnd,IDC_NODELIST);
	CStr propBuffer;
	CStr scriptBuffer;
	CStr clusterBuffer;
	CStr strCmds;
	LinkList<ConfigProp> cprops;
	LinkList<ConfigProp> allprops;
	Lst::StringHashTable< Link<ConfigProp>* > allpropsNames(8);
	DWORD cflags;
	int   index=PROPSTART;

	int  numSels=SendMessage(nodelist,LB_GETSELCOUNT,0,0);
	INT* sels=new INT[numSels];
		
	SendMessage(nodelist,LB_GETSELITEMS,(WPARAM)numSels,(LPARAM)sels);

	props.Clear();
	propView->Clear();

	// Add the explicit UI properties
	/*
	propView->AddCheck("CreatedAtStart","The object is created when the game starts.");
	propView->AddCheck("AbsentInNetGames","The object isn't created when playing network games.");
	propView->AddCheck("TrickObject","Unknown");
	propView->AddNode("Cluster","Cluster assigned if TrickObject");
	propView->AddCheck("Occluder","...");

	ConfigProp newProp;
	newProp.name="CreatedAtStart";
	Link<ConfigProp>* propCreatedAtStart=allprops.AddUnique(&newProp);
	allpropsNames.PutItem(newProp.name, &propCreatedAtStart);
	
	newProp.name="AbsentInNetGames";
	Link<ConfigProp>* propAbsentInNetGames=allprops.AddUnique(&newProp);
	allpropsNames.PutItem(newProp.name, &propAbsentInNetGames);

	newProp.name="TrickObject";
	Link<ConfigProp>* propTrickObject=allprops.AddUnique(&newProp);
	allpropsNames.PutItem(newProp.name, &propTrickObject);

	newProp.name="Cluster";
	Link<ConfigProp>* propCluster=allprops.AddUnique(&newProp);
	allpropsNames.PutItem(newProp.name, &propCluster);

	newProp.name="Occluder";
	Link<ConfigProp>* propOccluder=allprops.AddUnique(&newProp);
	allpropsNames.PutItem(newProp.name, &propOccluder);
	*/

	CStr commonClass, commonType;
	bool bCommon = true;

	for(int i=0;i<numSels;i++)
	{
		CStr compClass,compType;
		INode* compNode=(INode*)SendMessage(nodelist,LB_GETITEMDATA,(WPARAM)sels[i],0);
		compNode->GetUserPropBuffer(propBuffer);
		AddGlobalDefaults(propBuffer);
		scriptBuffer=ParseConfigProps(&cprops,&cflags,propBuffer,&strCmds,&clusterBuffer);

		// Get class type name for default lookup
		compClass = GetClassProp(compNode);
		compType  = GetTypeProp(compNode);

		if (i == 0)
		{
			commonClass = compClass;
			commonType  = compType;
		}
		else
		{
			if (commonClass != compClass ||
				commonType  != compType)
				bCommon = false;
		}

		AddGlobalDefaults(propBuffer,&cprops,&cflags,&strCmds,compClass,compType);

		if (propBuffer.Length()==0)
		{
			CStr defaultBuf=GetClassTypeRecord(compClass,compType);
			scriptBuffer=ParseConfigProps(&cprops,&cflags,defaultBuf,NULL,&clusterBuffer);
		}

		// nw
		GetDynUIProps(&cprops,compClass,compType);
		ConvertToDefaults(compClass,compType,&cprops);

		Link<ConfigProp>* curProp=cprops.GetHead();
		while(curProp)
		{
			if (!allpropsNames.GetItem(curProp->data.name, false))
			{
				//Link<ConfigProp>* link = allprops.AddUnique(&curProp->data);
				Link<ConfigProp>* link = allprops.AddToTail(&curProp->data);
				allpropsNames.PutItem(curProp->data.name, &link);
			}

			curProp=curProp->next;
		}
	}

	// If the selected nodes share commonality between class and type
	// they share the same omission list and we may omit unused properties
	if (bCommon)
		RemoveOmissions(commonClass, commonType, &allprops);

//	allprops.Remove(propCreatedAtStart);
//	allprops.Remove(propAbsentInNetGames);
//	allprops.Remove(propTrickObject);
//	allprops.Remove(propCluster);
//	allprops.Remove(propOccluder);

	// Build the combined property UI
	Link<ConfigProp>* curProp=allprops.GetHead();

	while(curProp)
	{
		AddPropViewProp(&curProp->data);
		curProp=curProp->next;
	}

	lastProps.Clear();
	propView->GetValues(&lastProps);
	
	bLockImmediateApply=TRUE;
	propView->BuildUI();

	// Fill out the flag property values
	/*
	if (cflags & CCLASS_CREATEDATSTART)
		propView->SetValue(PROP_CREATEDATSTART,"TRUE");
	else
		propView->SetValue(PROP_CREATEDATSTART,"FALSE");

	if (cflags & CCLASS_ABSENTINNETGAMES)
		propView->SetValue(PROP_ABSENTINNETGAMES,"TRUE");
	else
		propView->SetValue(PROP_ABSENTINNETGAMES,"FALSE");

	if (cflags & CCLASS_TRICKOBJECT)
		propView->SetValue(PROP_TRICKOBJECT,"TRUE");
	else
		propView->SetValue(PROP_TRICKOBJECT,"FALSE");
	
	if (cflags & CCLASS_OCCLUDER)
		propView->SetValue(PROP_OCCLUDER,"TRUE");
	else
		propView->SetValue(PROP_OCCLUDER,"FALSE");

	// Fill out the cluster field
	propView->SetValue(PROP_CLUSTER,clusterBuffer);
	*/

	// Fill out the rest of the property values
	curProp=allprops.GetHead();

	while(curProp)
	{
		propView->SetValue(index++,curProp->data.value);
		curProp=curProp->next;
	}

	propView->ResetUpdateStatus(); // 10-11-02
	bLockImmediateApply=FALSE;
}

CompareResult PropEditor::CompareProp(CStr propName,CStr propVal,LinkList<ConfigProp>* propList)
{ FUNC_ENTER("PropEditor::CompareProp"); 
	Link<ConfigProp>* curNode=propList->GetHead();

	while(curNode)
	{
		if (propName==curNode->data.name)
		{
			if (propVal==curNode->data.value)
				return COMPR_SAMEVALUE;

			return COMPR_FIELDEXISTS;
		}

		curNode=curNode->next;
	}

	return COMPR_NOFIELDEXISTS;
}

bool PropEditor::CompareProps(INode* node)
{ FUNC_ENTER("PropEditor::CompareProps"); 
	DWORD                cflags;
	LinkList<ConfigProp> cprops;
	CStr                 scriptBuffer;
	CStr                 propBuffer;
	CStr                 compClass,compType;
	CStr                 strClust;
	CStr                 strCmds;
	bool                 bCommon=true;

	node->GetUserPropBuffer(propBuffer);
	//AddGlobalDefaults(propBuffer);
	//XXscriptBuffer=ParseConfigProps(&cprops,&cflags,propBuffer,&strCmds,&strClust);
	scriptBuffer=ParseNodeConfigProps(&cprops,&cflags,node,&strCmds,&strClust);

	// Get class type name for default lookup
	compClass = GetClassProp(node);
	compType  = GetTypeProp(node);

	AddGlobalDefaults(propBuffer,&cprops,&cflags,&strCmds,compClass,compType);

	if (propBuffer.Length()==0)
	{
		CStr defaultBuf=GetClassTypeRecord(compClass,compType);
		scriptBuffer=ParseConfigProps(&cprops,&cflags,defaultBuf,NULL,&strClust);
	}

	ConvertToDefaults(compClass,compType,&cprops);

	// Compare flags
	/*
	if ((cflags & CCLASS_CREATEDATSTART)!=(flags & CCLASS_CREATEDATSTART))
	{
		propView->SetColor(PROP_CREATEDATSTART,PROPCOLOR_RED);
		bCommon=false;
	}

	if ((cflags & CCLASS_ABSENTINNETGAMES)!=(flags & CCLASS_ABSENTINNETGAMES))
	{
		propView->SetColor(PROP_ABSENTINNETGAMES,PROPCOLOR_RED);
		bCommon=false;
	}

	if ((cflags & CCLASS_TRICKOBJECT)!=(flags & CCLASS_TRICKOBJECT))
	{
		propView->SetColor(PROP_TRICKOBJECT,PROPCOLOR_RED);
		bCommon=false;
	}

	if ((cflags & CCLASS_OCCLUDER)!=(flags & CCLASS_OCCLUDER))
	{
		propView->SetColor(PROP_OCCLUDER,PROPCOLOR_RED);
		bCommon=false;
	}

	if (strClust!=strCluster)
	{
		propView->SetColor(PROP_CLUSTER,PROPCOLOR_RED);
		bCommon=false;
	}
	*/

	int numProps=propView->NumProps();

	// Compare properties
	for(int index=PROPSTART;index<numProps;index++)
	{
		CStr propName;
		CStr propValue;

		propName=propView->GetName(index);
		propView->GetValue(index,propValue);

		switch(CompareProp(propName,propValue,&cprops))
		{
		case COMPR_SAMEVALUE:
			// No action necessary (similar)
			break;
		case COMPR_FIELDEXISTS:
			// Field exists but different value (change red)
			if (propView->GetColor(index)==PROPCOLOR_PURPLE)
				break;
			
			if (propView->GetColor(index)==PROPCOLOR_BLUE)
				propView->SetColor(index,PROPCOLOR_PURPLE);
			else
				propView->SetColor(index,PROPCOLOR_RED);
			break;
		case COMPR_NOFIELDEXISTS:
			// Field doesn't exist
			if (propView->GetColor(index)==PROPCOLOR_PURPLE)
				break;

			if (propView->GetColor(index)==PROPCOLOR_RED)
				propView->SetColor(index,PROPCOLOR_PURPLE);
			else
				propView->SetColor(index,PROPCOLOR_BLUE);
			break;
		}
	}

	return bCommon;
}

void PropEditor::ProcNodeListSelChange()
{ FUNC_ENTER("PropEditor::ProcNodeListSelChange"); 
	// Disable acknowledgement when the selection changes until we're finished compositing the list
	classBrowser->DisableAck();
	typeBrowser->DisableAck();

	if (bSubSelApply && bChangeMade)
		Apply(FALSE);

	GetSelNodes();
	UpdateClassList();

	CStr lastValue = classBrowser->GetLastValue();

	bClassConflict=FALSE;
	bTypeConflict=FALSE;

	if (bSubSelRestore)
		NodeListRestore();

	// Reenable acknowledgement when the selection changes
	classBrowser->EnableAck();
	typeBrowser->EnableAck();
}

void PropEditor::DispTreeView()
{ FUNC_ENTER("PropEditor::DispTreeView"); 
	HMENU hMenuBar=GetMenu(hwnd);

	bDisplayTree=TRUE;

	CheckMenuItem(hMenuBar,IDM_DISPTREEVIEW,MF_CHECKED);
	CheckMenuItem(hMenuBar,IDM_DISPLIST,MF_UNCHECKED);

	ShowWindow(GetDlgItem(hwnd,IDC_TREE1),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_GLOBALLIST),SW_HIDE);
	ShowWindow(GetDlgItem(hwnd,IDC_STATICFILTERS),SW_HIDE);
	ShowWindow(GetDlgItem(hwnd,IDC_STATICFILTER),SW_HIDE);
	ShowWindow(GetDlgItem(hwnd,IDC_STATICNAME),SW_HIDE);
	ShowWindow(GetDlgItem(hwnd,IDC_ADDFILTER),SW_HIDE);
	ShowWindow(GetDlgItem(hwnd,IDC_REMOVEFILTER),SW_HIDE);
	ShowWindow(GetDlgItem(hwnd,IDC_FILTERLIST),SW_HIDE);
	ShowWindow(GetDlgItem(hwnd,IDC_FILTERNAME),SW_HIDE);
	ShowWindow(GetDlgItem(hwnd,IDC_FILTERDATA),SW_HIDE);
}

void PropEditor::DispGlobalList()
{ FUNC_ENTER("PropEditor::DispGlobalList"); 
	HMENU hMenuBar=GetMenu(hwnd);

	bDisplayTree=FALSE;

	CheckMenuItem(hMenuBar,IDM_DISPLIST,MF_CHECKED);
	CheckMenuItem(hMenuBar,IDM_DISPTREEVIEW,MF_UNCHECKED);

	ShowWindow(GetDlgItem(hwnd,IDC_TREE1),SW_HIDE);
	ShowWindow(GetDlgItem(hwnd,IDC_GLOBALLIST),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_STATICFILTERS),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_STATICFILTER),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_STATICNAME),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_ADDFILTER),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_REMOVEFILTER),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_FILTERLIST),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_FILTERNAME),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_FILTERDATA),SW_SHOW);
}

void PropEditor::ApplyOnClose()
{ FUNC_ENTER("PropEditor::ApplyOnClose"); 
	HMENU hMenuBar=GetMenu(hwnd);

	bApplyOnClose=!bApplyOnClose;

	if (bApplyOnClose)
		CheckMenuItem(hMenuBar,IDM_APPLYONCLOSE,MF_CHECKED);
	else
		CheckMenuItem(hMenuBar,IDM_APPLYONCLOSE,MF_UNCHECKED);
}

bool PropEditor::PerformApplyOnClose()
{ FUNC_ENTER("PropEditor::PerformApplyOnClose"); 
	if (bApplyOnClose)
		Apply(FALSE);
	else
	{
		if (bChangeMade)
		{
			if ((bClassChanged && GetClass()!=CStr(vUNKNOWN)) ||
				(bTypeChanged && GetType()!=CStr(vUNKNOWN))   ||
				(bScriptChanged && pRichText->GetText()!=CStr(vUNKNOWN)) ||
				propView->WasUpdated())
			{
				int result=MessageBox(hwnd,"Would you like to apply your changes?","Close Property Editor",MB_ICONWARNING|MB_YESNOCANCEL);

				if (result==IDCANCEL)
					return true;

				if (result==IDYES)
					Apply(FALSE);
			}
		}
	}

	return false;
}

void PropEditor::AddFilter()
{ FUNC_ENTER("PropEditor::AddFilter"); 
	CStr strName,strFilter;

	char csName[256];
	char csFilter[256];

	// Get the information for the filter to be added
	GetDlgItemText(hwnd,IDC_FILTERNAME,csName,256);
	GetDlgItemText(hwnd,IDC_FILTERDATA,csFilter,256);

	AddFilter(csName,csFilter);
}

int PropEditor::AddFilter(char* Name,char* Value)
{ FUNC_ENTER("PropEditor::AddFilter"); 
	CStr strName=Name;
	CStr strFilter=Value;

	// If no name is given give it the name of the filter
	if (strName==CStr(""))
		strName=strFilter;

	// Add the new filter
	Link<Filter>* filter= filterList.Add();
	filter->data.name   = strName;
	filter->data.filter = strFilter;

	// Add the filter to the filter list
	int index=SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_ADDSTRING,0,(LPARAM)(char*)strName);
	SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_SETITEMDATA,(WPARAM)index,(LPARAM)filter);

	// Select the newly added filter
	SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_SETSEL,(WPARAM)TRUE,(LPARAM)(UINT)index);

	SetDlgItemText(hwnd,IDC_FILTERNAME,"");
	SetDlgItemText(hwnd,IDC_FILTERDATA,"");

	UpdateGlobalList();

	return index;
}

void PropEditor::RemoveFilter()
{ FUNC_ENTER("PropEditor::RemoveFilter"); 
	int selCount=SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETSELCOUNT,0,0);
	
	if (selCount<1)
		return;

	INT sel;
	
	while (selCount>0)	
	{
		SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETSELITEMS,(WPARAM)1,(LPARAM)&sel);

		Link<Filter>* filter=(Link<Filter>*)SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETITEMDATA,(WPARAM)sel,0);
		filterList.Remove(filter);

		SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_DELETESTRING,(WPARAM)sel,0);

		selCount=SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETSELCOUNT,0,0);
	}

	UpdateGlobalList();
}

void PropEditor::UpdateGlobalList()
{ FUNC_ENTER("PropEditor::UpdateGlobalList"); 
	int numFilters=SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETSELCOUNT,0,0);

	INT* sels=new INT[numFilters];
	SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETSELITEMS,(WPARAM)numFilters,(LPARAM)sels);

	// Clear out the current list
	if (!bMapUnloaded)
		SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_RESETCONTENT,0,0);

	// Add everything back from the scriptDB
	TypeDesc  *curScript;
	
	int size=scriptDB.GetSize();
	scriptDB.IterateStart();

	for(int i=0;i<size;i++)
	{
		BOOL bFilterMatch=FALSE;

		curScript=scriptDB.IterateNext();
		
		if (numFilters>0)
		{
			// See if this script matches any of the selected filters
			for(int j=0;j<numFilters;j++)
			{
				Link<Filter>* filter=(Link<Filter>*)SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETITEMDATA,(WPARAM)sels[j],0);
				if (MatchPattern(curScript->Name,filter->data.filter))
				{
					bFilterMatch=TRUE;
					break;
				}
			}
		}
		else
		{
			// No filters means everything gets added
			bFilterMatch=TRUE;
		}

		if (bFilterMatch)
		{
			// Add the item to the global list
			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,0,(LPARAM)(char*)curScript->Name);
		}
	}

	delete sels;
}

void PropEditor::FilterChange()
{ FUNC_ENTER("PropEditor::FilterChange"); 
	UpdateGlobalList();

	int selCount=SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETSELCOUNT,0,0);

	if (selCount==1)
	{
		INT sel;
		SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETSELITEMS,(WPARAM)1,(LPARAM)&sel);
		Link<Filter>* filter=(Link<Filter>*)SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETITEMDATA,(WPARAM)sel,0);
		SetDlgItemText(hwnd,IDC_FILTERNAME,(char*)filter->data.name);
		SetDlgItemText(hwnd,IDC_FILTERDATA,(char*)filter->data.filter);
	}
}

void PropEditor::SetCloseNotify(void (*Func)(PropEditor*,void*),void* pdata)
{ FUNC_ENTER("PropEditor::SetCloseNotify"); 
	fpCloseNotify=Func;
	pCloseNotifyData=pdata;
}

void PropEditor::AddPropViewProp(ConfigProp* curProp)
{ FUNC_ENTER("PropEditor::AddPropViewProp"); 
	CStr strDesc,strDefault;

	if (curProp->extdata.Length()>0)
	{
		RTFEditor::ParsePropTags(curProp->extdata,curProp->name,strDesc,strDefault);
		if (!RTFEditor::ParseExtendedTags(propView,
							              curProp->extdata,
					                      "blah",
							              curProp->name,
							              strDesc,
							              strDefault))
		{
			if (strDefault.Length()>0)
				propView->AddEdit(curProp->name,strDefault);
			else
				propView->AddEdit(curProp->name,"No extended tags exist in the node data for this property.");

			RTFEditor::ParseExtendedTagsReq(propView,curProp->extdata,"blah",curProp->name,strDesc,strDefault);
		}
	}
	else
	{
		if (strDefault.Length()>0)
			propView->AddEdit(curProp->name,strDefault);
		else
			propView->AddEdit(curProp->name,"No extended tags exist in the node data for this property.");
	}

	propView->SetUserData(propView->GetCurIndex(), curProp->extdata);
}

void PropEditor::RestoreDefaults()
{ FUNC_ENTER("PropEditor::RestoreDefaults"); 
	CStr className,typeName;
	char buf[256];

	// Get class
	GetDlgItemTextLast(hwnd,IDC_OBJCLASSM,buf,256);
	className=CStr(buf);

	// Get type
	GetDlgItemTextLast(hwnd,IDC_OBJTYPEM,buf,256);
	typeName=CStr(buf);

	// Cannot compare defaults unless there is a class and type
	if (className==CStr(vUNKNOWN) ||
		typeName==CStr(vUNKNOWN))
		return;

	int numProps=propView->NumProps();

	if (numProps==0)
	{
		UpdateClassSelection(FALSE);
		CopyProps();
		UpdatePropView();
	}

	for(int i=4;i<numProps;i++)
	{
		CStr name;
		CStr value;
		CStr defvalue;
		
		name=propView->GetName(i);
		defvalue=GetDefault(className,typeName,name);

		if (defvalue!=CStr(""))
		{
			propView->SetValue(i,defvalue);
			propView->SetColor(i,PROPCOLOR_GREEN);
		}
	}

	// Add any properties that don't exist
	LinkList<ConfigProp> cprops;
	ScriptIniParser::GetDefaultProps(&cprops,className,typeName);

	Link<ConfigProp>* curProp=cprops.GetHead();

	bLockImmediateApply=TRUE;
	while(curProp)
	{
		// Check if this property exists within propview
		int numProps=propView->NumProps();
		bool bExists=false;

		for(int i=0;i<numProps;i++)
		{
			if (propView->GetName(i)==curProp->data.name)
			{
				bExists=true;
				break;
			}
		}
	
		if (!bExists)
		{
			int index;

			propView->SaveValues();
			propView->DestroyUI();

			AddPropViewProp(&curProp->data);
			propView->SetMod(propView->GetCurIndex(),true);
			index=propView->GetCurIndex();
			lastProps.Clear();
			propView->GetValues(&lastProps);
			
			propView->BuildUI();
			propView->RestoreValues();
			propView->SetValue(index,curProp->data.value);
			propView->SetColor(index,PROPCOLOR_GREEN);
			propView->SetFlags(index,PROPFLAG_NEWPROP);
		}

		curProp=curProp->next;
	}
	bLockImmediateApply=FALSE;
}

void PropEditor::ClearProps(BOOL bWarn)
{ FUNC_ENTER("PropEditor::ClearProps"); 
	int result;

	bChangeMade=TRUE;

	if (bWarn)
		result=MessageBox(hwnd,"Are you sure you wish to clear all property data?","Clear Properties",MB_ICONWARNING|MB_YESNO);
	else
		result=IDYES;

	if (result==IDYES)
	{
		flags=0;
		props.Clear();
		propView->Clear();

		// Add the required properties
		//propView->AddCheck("CreatedAtStart","The object is created when the game starts.");
		//propView->AddCheck("AbsentInNetGames","The object isn't created when playing network games.");
		//propView->AddCheck("TrickObject","Unknown");
		//propView->AddNode("Cluster","Cluster assigned if TrickObject");
		//propView->AddCheck("Occluder","...");

		lastProps.Clear();
		propView->GetValues(&lastProps);

		bLockImmediateApply=TRUE;
		propView->BuildUI();

		CompareDefaults(GetClass(),GetType());
		bLockImmediateApply=FALSE;
	}
}

bool PropEditor::SaveConfig()
{ FUNC_ENTER("PropEditor::SaveConfig"); 
	CStr appDir=ip->GetDir(APP_PLUGCFG_DIR);
	appDir+="\\PropEdit.ini";

	// View Section
	if (bDisplayTree)
		WritePrivateProfileString("View","TreeView","1",(char*)appDir);
	else
		WritePrivateProfileString("View","TreeView","0",(char*)appDir);

	// Settings Section
	if (bAutoSelect)
		WritePrivateProfileString("Settings","AutoApplyRestoreMAX","1",(char*)appDir);
	else
		WritePrivateProfileString("Settings","AutoApplyRestoreMAX","0",(char*)appDir);

	if (bSubSelApply)
		WritePrivateProfileString("Settings","ApplySubSelChange","1",(char*)appDir);
	else
		WritePrivateProfileString("Settings","ApplySubSelChange","0",(char*)appDir);

	if (bSubSelRestore)
		WritePrivateProfileString("Settings","UpdateViaSubSel","1",(char*)appDir);
	else
		WritePrivateProfileString("Settings","UpdateViaSubSel","0",(char*)appDir);

	if (bApplyChangedOnly)
		WritePrivateProfileString("Settings","ApplyOnlyChanged","1",(char*)appDir);
	else
		WritePrivateProfileString("Settings","ApplyOnlyChanged","0",(char*)appDir);

	if (bApplyOnClose)
		WritePrivateProfileString("Settings","AutoApplyOnClose","1",(char*)appDir);
	else
		WritePrivateProfileString("Settings","AutoApplyOnClose","0",(char*)appDir);

	if (bAddConflict)
		WritePrivateProfileString("Settings","AddPropertiesDuringConflict","1",(char*)appDir);
	else
		WritePrivateProfileString("Settings","AddPropertiesDuringConflict","0",(char*)appDir);

	if (bNameExpansion)
		WritePrivateProfileString("Settings","NameExpansion","1",(char*)appDir);
	else
		WritePrivateProfileString("Settings","NameExpansion","0",(char*)appDir);

	if (bForceScriptUpdates)
		WritePrivateProfileString("Settings","ForceScriptUpdates","1",(char*)appDir);
	else
		WritePrivateProfileString("Settings","ForceScriptUpdates","0",(char*)appDir);

	if (bImmediateApply)
		WritePrivateProfileString("Settings","ImmediateApply","1",(char*)appDir);
	else
		WritePrivateProfileString("Settings","ImmediateApply","0",(char*)appDir);

	if (bRefreshViewport)
		WritePrivateProfileString("Settings","RefreshViewport","1",(char*)appDir);
	else
		WritePrivateProfileString("Settings","RefreshViewport","0",(char*)appDir);

	// Filters Section
	int numFilters=SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETCOUNT,0,0);
	for(int i=0;i<numFilters;i++)
	{
		Link<Filter>* filter=(Link<Filter>*)SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETITEMDATA,(WPARAM)i,0);

		if (filter)
			WritePrivateProfileString("Filters",(char*)filter->data.name,(char*)filter->data.filter,(char*)appDir);

		if (SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_GETSEL,(WPARAM)i,0))
			WritePrivateProfileString("FilterSelection",(char*)filter->data.name,"1",(char*)appDir);
		else
			WritePrivateProfileString("FilterSelection",(char*)filter->data.name,"0",(char*)appDir);
	}

	return true;
}

bool PropEditor::LoadConfig()
{ FUNC_ENTER("PropEditor::LoadConfig"); 
	CStr appDir=ip->GetDir(APP_PLUGCFG_DIR);
	appDir+="\\PropEdit.ini";

	// View Section
	if (GetPrivateProfileInt("View","TreeView",0,(char*)appDir)==1)
		DispTreeView();
	else
		DispGlobalList();

	// Settings Section
	if (GetPrivateProfileInt("Settings","AutoApplyRestoreMAX",1,(char*)appDir)==1)
		bAutoSelect=FALSE;
	else
		bAutoSelect=TRUE;

	AutoSelect();

	if (GetPrivateProfileInt("Settings","ApplySubSelChange",0,(char*)appDir)==1)
		bSubSelApply=FALSE;
	else
		bSubSelApply=TRUE;

	SubSelApply();

	if (GetPrivateProfileInt("Settings","UpdateViaSubSel",1,(char*)appDir)==1)
		bSubSelRestore=FALSE;
	else
		bSubSelRestore=TRUE;
	
	SubSelRestore();
	
	if (GetPrivateProfileInt("Settings","ApplyOnlyChanged",1,(char*)appDir)==1)
		bApplyChangedOnly=FALSE;
	else
		bApplyChangedOnly=TRUE;

	ApplyChangedOnly();

	if (GetPrivateProfileInt("Settings","AutoApplyOnClose",0,(char*)appDir)==1)
		bApplyOnClose=FALSE;
	else
		bApplyOnClose=TRUE;

	ApplyOnClose();
	
	if (GetPrivateProfileInt("Settings","AddPropertiesDuringConflict",1,(char*)appDir)==1)
		bAddConflict=FALSE;
	else
		bAddConflict=TRUE;

	if (GetPrivateProfileInt("Settings","NameExpansion",1,(char*)appDir)==1)
		bNameExpansion=FALSE;
	else
		bNameExpansion=TRUE;

	if (GetPrivateProfileInt("Settings","ForceScriptUpdates",0,(char*)appDir)==1)
		bForceScriptUpdates = FALSE;
	else
		bForceScriptUpdates = TRUE;

	ForceScriptUpdates();

	if (GetPrivateProfileInt("Settings","ImmediateApply",0,(char*)appDir)==1)
		bImmediateApply = FALSE;
	else
		bImmediateApply = TRUE;

	ImmediateApply();

	if (GetPrivateProfileInt("Settings","RefreshViewport",0,(char*)appDir)==1)
		bRefreshViewport = FALSE;
	else
		bRefreshViewport = TRUE;

	RefreshViewport();

	ToggleNameExpansion();

	AddConflict();

	// Filters section
	char secNames[5120];	// 5K buffer for filter names
	int  bufLen=GetPrivateProfileSection("Filters",secNames,5120,(char*)appDir);
	
	CStr strName;

	for(int i=0;i<bufLen;i++)
	{
		char valbuf[256];
		char buf[2];
		buf[0]=0;
		buf[1]='\0';

		buf[0]=secNames[i];

		if (secNames[i]=='\0')
		{
			if (strName.Length()>1 && Instr(strName,"=")>0)
				strName=Left(strName,Instr(strName,"="));

			GetPrivateProfileString("Filters",(char*)strName,"*",valbuf,256,(char*)appDir);
			int index=AddFilter((char*)strName,valbuf);

			// Select if appropriate
			if (GetPrivateProfileInt("FilterSelection",(char*)strName,1,(char*)appDir)==1)
				SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_SETSEL,(WPARAM)TRUE,(LPARAM)index);
			else
				SendDlgItemMessage(hwnd,IDC_FILTERLIST,LB_SETSEL,(WPARAM)FALSE,(LPARAM)index);

			strName="";
			continue;
		}

		strName+=buf;
	}

	if (GetPrivateProfileInt("View","NoTreeview",0,(char*)appDir)==1)
		bNoTreeview=TRUE;
	else
		bNoTreeview=FALSE;

	DispGlobalList();
	UpdateGlobalList();
	return true;
}

bool PropEditor::AddNewProp(AddPropDlg* pAddProp)
{ FUNC_ENTER("PropEditor::AddNewProp"); 
	PropType   ptype  = pAddProp->GetType();
	ConfigProp cprop;
	
	CStr     strName  = pAddProp->GetProp();
	CStr     strDesc  = pAddProp->GetDesc();
	CStr     strValue = pAddProp->GetValue();
	CStr     strCategory;
	CStr     eol="\r\n";

	propView->GetCategory(strCategory);

	if (pAddProp->UseCategory())
	{
		if (strCategory.Length() > 0 && strCategory != CStr("Default") && strCategory != CStr("All"))
			strName = strCategory + CStr("/") + strName;
	}

	if (!IsAlphaNum(strName))
	{
		MessageBox(hwnd,"The property name contains invalid characters.  You may only use alpha-numeric characters and no spaces!","Invalid Name",MB_ICONWARNING|MB_OK);
		return false;
	}
	
	if (Instr(strDesc,"|")!=-1)
	{
		MessageBox(hwnd,"The property description contains invalid characters.  You may not use pipes. '|'","Invalid Description",MB_ICONWARNING|MB_OK);
		return false;
	}

	if (GetDynUICmds(GetClass(),GetType(),strName).Length()>0)
	{
		MessageBox(hwnd,"This property's UI has been defined in script.ini.\nSince this is a commonly used property, you must changed it in script.ini to maintain project consistency.","Commonly used UI element update",MB_ICONWARNING|MB_OK);
		return false;
	}

	switch(ptype)
	{
	case PROPTYPE_UNDEFINED:
		return false;

	case PROPTYPE_EDIT:
		propView->AddEdit(strName,strDesc);
		break;

	case PROPTYPE_LIST:
		propView->AddList(strName,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | list | done")+eol;
		break;

	case PROPTYPE_EXTLIST:
		propView->AddExtList(strName,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | extlist | done")+eol;
		break;

	case PROPTYPE_SLIDER:
		propView->AddSlider(strName,1,100,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | slider | 1 | 100")+eol;
		break;

	case PROPTYPE_SCRIPT:
		propView->AddScript(strName,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | script")+eol;
		break;

	case PROPTYPE_FILE:
		propView->AddFile(strName,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | file | . | All Files (*.*) | *.* | done")+eol;
		break;

	case PROPTYPE_COLOR:
		propView->AddColor(strName,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | color")+eol;
		break;

	case PROPTYPE_CHECK:
		propView->AddCheck(strName,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | check")+eol;
		break;

	case PROPTYPE_SPINEDIT:
		propView->AddSpinEdit(strName,0,100,1,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | spinedit | 0 | 100 | 1")+eol;
		break;

	case PROPTYPE_STATIC:
		propView->AddStatic(strName,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | static")+eol;
		break;

	case PROPTYPE_SLIDERNUM:
		propView->AddSliderNum(strName,0,100,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | slidernum | 0 | 100")+eol;
		break;

	case PROPTYPE_NODE:
		propView->AddNode(strName,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | node")+eol;
		break;

	case PROPTYPE_MULTI:
		propView->AddMulti(strName,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | multi | None | done")+eol;
		break;

	case PROPTYPE_FLAGS:
		propView->AddFlags(strName,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | flags | None | done")+eol;
		break;

	case PROPTYPE_MTL:
		propView->AddMtl(strName,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | material")+eol;
		break;

	case PROPTYPE_COLORPOPUP:
		propView->AddColorPopup(strName,1,4,strDesc);
		cprop.extdata=CStr("// @nextparm | ")+strName+CStr(" | colorpopup | 1 | 4")+eol;
		break;
	}

	if (strDesc.Length()>0)
		cprop.extdata+=CStr("// @nextdesc | ")+strName+CStr(" | ")+strDesc+eol;
	
	if (strValue.Length()>0)
		cprop.extdata+=CStr("// @nextdef | ")+strName+CStr(" | ")+strValue+eol;

	cprop.name  = strName;
	cprop.value = strValue;

	int index=propView->GetCurIndex();
	propView->SetUserData(index,cprop.extdata);

	props.Add(&cprop);
	return true;
}

void PropEditor::ModPropCB(AddPropDlg* pAddProp,void* pData)
{ FUNC_ENTER("PropEditor::ModPropCB"); 
	PropEditor* pthis=(PropEditor*)pData;

	pthis->bChangeMade=TRUE;

	// Add the property
	if (!pthis->AddNewProp(pAddProp))
		return;

	// Delete the property
	int index=pAddProp->GetIntUserData();
	pthis->delprops.Add(&pAddProp->GetStrUserData());

	pthis->propView->SaveValues();
	pthis->propView->DestroyUI();
	pthis->propView->DeleteProp(index);

	CStr strPropName = pAddProp->GetProp();
	CStr strValue    = pAddProp->GetValue();
	CStr strDesc     = pAddProp->GetDesc();
	CStr strCategory;

	PropType type    = pAddProp->GetType();

	pthis->propView->GetCategory(strCategory);

	if (pAddProp->UseCategory())
	{
		if (strCategory.Length() > 0 && strCategory != CStr("Default") && strCategory != CStr("All"))
			strPropName = strCategory + CStr("/") + strPropName;
	}

	index=pthis->propView->GetCurIndex();
	pthis->propView->SetMod(index,true);
	
	pthis->propView->SetMod(index,true);
	pthis->lastProps.Clear();
	pthis->propView->GetValues(&pthis->lastProps);

	pthis->bLockImmediateApply=TRUE;
	pthis->propView->BuildUI();
	
	pthis->propView->RestoreValues();
	pthis->propView->SetValue(index,strValue);
	pthis->propView->SetColor(index,PROPCOLOR_WHITE);
	pthis->propView->SetFlags(index,PROPFLAG_NEWPROP);
	pthis->bLockImmediateApply=FALSE;
}

void PropEditor::AddPropCB(AddPropDlg* pAddProp,void* pData)
{ FUNC_ENTER("PropEditor::AddPropCB"); 
	PropEditor* pthis=(PropEditor*)pData;

	pthis->bChangeMade=TRUE;

	CStr strPropName = pAddProp->GetProp();
	CStr strValue    = pAddProp->GetValue();
	CStr strDesc     = pAddProp->GetDesc();

	PropType type    = pAddProp->GetType();
	int      index;

	if (!pthis->AddNewProp(pAddProp))
		return;

	pthis->propView->SaveValues();
	pthis->propView->DestroyUI();

	index=pthis->propView->GetCurIndex();
	pthis->propView->SetMod(index,true);
	pthis->lastProps.Clear();
	pthis->propView->GetValues(&pthis->lastProps);

	pthis->bLockImmediateApply=TRUE;
	pthis->propView->BuildUI();
	
	pthis->propView->RestoreValues();
	pthis->propView->SetValue(index,strValue);
	pthis->propView->SetColor(index,PROPCOLOR_WHITE);
	pthis->propView->SetFlags(index,PROPFLAG_NEWPROP);
	pthis->bLockImmediateApply=FALSE;
}

void PropEditor::AddSFP()
{ FUNC_ENTER("PropEditor::AddSFP"); 
	ListData* ldata = propView->AddList("SFP_Effect","Screen Facing Poly Effect Mode");

	CheckDlgButton(hwnd,IDC_OMIT,BST_CHECKED);

	// Load the effects.lst file
	CStr strPath = CStr(getenv(APP_ENV)) + CStr(SCRIPT_PATH);

	FILE* fp = fopen(strPath + "effects.lst","r");

	if (!fp)
	{
		MessageBox(hwnd,"Failed to load the effects.lst file","Add Screen Facing Poly Props",MB_OK);
		return;
	}

	char buf[256];
	while(!feof(fp))
	{
		fgets(buf,255,fp);
		StripToken(buf);
		
		if (strlen(buf)>0)
			ldata->list.Add(&CStr(buf));
	}

	fclose(fp);

	CStr        eol="\r\n";
	ConfigProp  cprop;
	cprop.name     = "SFP_Effect";
	cprop.value    = "";
	cprop.extdata  = CStr("// @nextparm | SFP_Effect | list | file | effects.lst | done")+eol;
	cprop.extdata += CStr("// @nextdesc | SFP_Effect | Screen Facing Poly Effect Mode")+eol;

	int index=propView->GetCurIndex();
	propView->SetUserData(index,cprop.extdata);

	props.Add(&cprop);

	propView->SaveValues();
	propView->DestroyUI();

	index = propView->GetCurIndex();
	propView->SetMod(index,true);
	lastProps.Clear();
	propView->GetValues(&lastProps);

	bLockImmediateApply=TRUE;
	propView->BuildUI();
	propView->RestoreValues();
	
	propView->SetColor(index,PROPCOLOR_WHITE);
	propView->SetFlags(index,PROPFLAG_NEWPROP);
	bLockImmediateApply=FALSE;
}

void PropEditor::AddLink()
{ FUNC_ENTER("PropEditor::AddLink"); 
	// Scan through the properties and determine the highest numbered manual link
	int nProps = propView->NumProps();
	int maxLink = -1;

	for(int i=0;i<nProps;i++)
	{
		CStr name = propView->GetName(i);
		name.toLower();

		if (name.Substr(0,5)==CStr("link_"))
		{
			int linkID;
			sscanf(name,"link_%i",&linkID);

			if (linkID > maxLink)
				maxLink = linkID;
		}
	}

	// Add a new manual link
	maxLink++;
	char newName[256];
	sprintf(newName,"Link_%i",maxLink);

	propView->AddNode(newName,"This is a manually defined node link");

	CStr        eol="\r\n";
	ConfigProp  cprop;
	cprop.name     = newName;
	cprop.value    = "";
	cprop.extdata  = CStr("// @nextparm | ")+newName+CStr(" | node")+eol;
	cprop.extdata += CStr("// @nextdesc | ")+newName+CStr(" | This is a manually defined node link")+eol;
	
	int index=propView->GetCurIndex();
	propView->SetUserData(index,cprop.extdata);

	props.Add(&cprop);

	propView->SaveValues();
	propView->DestroyUI();

	index = propView->GetCurIndex();
	propView->SetMod(index,true);
	lastProps.Clear();
	propView->GetValues(&lastProps);

	bLockImmediateApply=TRUE;
	propView->BuildUI();
	propView->RestoreValues();
	
	propView->SetColor(index,PROPCOLOR_WHITE);
	propView->SetFlags(index,PROPFLAG_NEWPROP);
	bLockImmediateApply=FALSE;
}

void PropEditor::AddLODLevel()
{
	// Scan through the properties and determine the highest numbered manual link
	int nProps = propView->NumProps();
	int maxLink = -1;

	for(int i=0;i<nProps;i++)
	{
		CStr name = propView->GetName(i);
		name.toLower();

		if (name.Substr(0,4)==CStr("lod_"))
		{
			int linkID;
			sscanf(name,"lod_%i",&linkID);

			if (linkID > maxLink)
				maxLink = linkID;
		}
	}

	// Add a new manual link
	maxLink++;
	char newName[256];
	sprintf(newName,"LOD_%i",maxLink);

	propView->AddSpinEdit(newName, 0.0f, 99999.0f, 0.01f, "This is a manually defined LOD level");

	CStr        eol="\r\n";
	ConfigProp  cprop;
	cprop.name     = newName;
	cprop.value    = "";
	cprop.extdata  = CStr("// @nextparm | ")+newName+CStr(" | spinedit | 0.0 | 99999.0 | 0.01")+eol;
	cprop.extdata += CStr("// @nextdesc | ")+newName+CStr(" | This is a manually defined LOD level")+eol;
	
	int index=propView->GetCurIndex();
	propView->SetUserData(index,cprop.extdata);

	props.Add(&cprop);

	propView->SaveValues();
	propView->DestroyUI();

	index = propView->GetCurIndex();
	propView->SetMod(index,true);
	lastProps.Clear();
	propView->GetValues(&lastProps);

	bLockImmediateApply=TRUE;
	propView->BuildUI();
	propView->RestoreValues();
	
	propView->SetColor(index,PROPCOLOR_WHITE);
	propView->SetFlags(index,PROPFLAG_NEWPROP);
	bLockImmediateApply=FALSE;
}

void PropEditor::AddProp()
{ FUNC_ENTER("PropEditor::AddProp"); 
	pAddPropDlg->SetTitle("Add Property");
	pAddPropDlg->SetProp("");
	pAddPropDlg->SetValue("");
	pAddPropDlg->SetDesc("");
	pAddPropDlg->SetType(PROPTYPE_EDIT);
	
	pAddPropDlg->SetOKCB(AddPropCB,this);
	pAddPropDlg->Show();
}

void PropEditor::RenameCB(NodeRenameDlg* renameDlg,void* pData)
{ FUNC_ENTER("PropEditor::RenameCB"); 
	int index,newindex;
	PropEditor* pthis=(PropEditor*)pData;
	INode* node;

	index=SendDlgItemMessage(pthis->hwnd,IDC_NODELIST,LB_GETCURSEL,0,0);
	node=(INode*)SendDlgItemMessage(pthis->hwnd,IDC_NODELIST,LB_GETITEMDATA,(WPARAM)index,0);

	node->SetName(renameDlg->GetNodeName());

	newindex=SendDlgItemMessage(pthis->hwnd,IDC_NODELIST,LB_INSERTSTRING,(WPARAM)index,(LPARAM)(char*)renameDlg->GetNodeName());
	SendDlgItemMessage(pthis->hwnd,IDC_NODELIST,LB_SETITEMDATA,(WPARAM)newindex,(LPARAM)node);
	SendDlgItemMessage(pthis->hwnd,IDC_NODELIST,LB_SETSEL,(WPARAM)TRUE,(LPARAM)newindex);

	SendDlgItemMessage(pthis->hwnd,IDC_NODELIST,LB_DELETESTRING,(WPARAM)index+1,0);
}

void PropEditor::RenameNode()
{ FUNC_ENTER("PropEditor::RenameNode"); 
	int    index;
	INode* node;

	index=SendDlgItemMessage(hwnd,IDC_NODELIST,LB_GETCURSEL,0,0);
	node=(INode*)SendDlgItemMessage(hwnd,IDC_NODELIST,LB_GETITEMDATA,(WPARAM)index,0);

	if (!node)
		return;

	pRenameDlg->RegOKCB(RenameCB,this);
	pRenameDlg->SetNodeName(node->GetName());
	pRenameDlg->Show();
}

void PropEditor::DeleteProp(PropList* plist,int index)
{ FUNC_ENTER("PropEditor::DeleteProp"); 
	if (index<PROPSTART)
	{
		MessageBox(hwnd,"This is a permanent property that can't be deleted.","Permanent Property",MB_ICONWARNING|MB_OK);
		return;
	}

	delprops.Add(&plist->GetName(index));

	propView->SaveValues();
	propView->DestroyUI();
	propView->DeleteProp(index);
	lastProps.Clear();
	propView->GetValues(&lastProps);
	
	bLockImmediateApply=TRUE;
	propView->BuildUI();
	propView->RestoreValues();
	bLockImmediateApply=FALSE;
}

void PropEditor::ModifyProp(PropList* plist,int index)
{ FUNC_ENTER("PropEditor::ModifyProp"); 
	if (index<PROPSTART)
	{
		MessageBox(hwnd,"This is a permanent property that can't be modified.","Permanent Property",MB_ICONWARNING|MB_OK);
		return;
	}

	if (GetDynUICmds(GetClass(),GetType(),propView->GetName(index)).Length()>0)
	{
		MessageBox(hwnd,"This property's UI has been defined in script.ini.\nSince this is a commonly used property, you must changed it in script.ini to maintain project consistency.","Commonly used UI element update",MB_ICONWARNING|MB_OK);
		return;
	}

	CStr strVal;
	propView->GetValue(index,strVal);

	pAddPropDlg->SetTitle("Modify Property");
	pAddPropDlg->SetProp(propView->GetName(index));
	pAddPropDlg->SetDesc(propView->GetDesc(index));
	pAddPropDlg->SetType(propView->GetType(index));
	pAddPropDlg->SetValue(strVal);
	pAddPropDlg->SetIntUserData(index);
	pAddPropDlg->SetStrUserData(propView->GetName(index));
	pAddPropDlg->SetOKCB(ModPropCB,this);
	pAddPropDlg->Show();
}

void PropEditor::FindInFiles()
{ FUNC_ENTER("PropEditor::FindInFiles"); 
	pFindFilesDlg->Show();
}

HWND FindPropEditor(HWND hwnd)
{ FUNC_ENTER("FindPropEditor"); 
	HWND    hwndParent=GetParent(hwnd);

	char strWndTxt[256];
	GetWindowText(hwndParent,strWndTxt,254);

	while(!strstr(strWndTxt,"Property Editor"))
	{
		hwndParent=GetParent(hwndParent);
		GetWindowText(hwndParent,strWndTxt,254);
	}

	return hwndParent;
}

LRESULT CALLBACK PropEditor::SubWndReg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("PropEditor::SubWndReg"); 
	WNDPROC oldWndProc=(WNDPROC)GetWindowLong(hwnd,GWL_USERDATA);
	HWND    hwndParent;

	switch(msg)
	{
	case WM_KEYDOWN:
		switch(wParam)
		{
		case VK_ESCAPE:
			hwndParent=FindPropEditor(hwnd);

			if (hwndParent)
				SendMessage(hwndParent,WM_CLOSE,0,0);

			return TRUE;
		}
	}

	return CallWindowProc(oldWndProc,hwnd,msg,wParam,lParam);
}

bool PropEditor::VerifyClassValidity()
{ FUNC_ENTER("PropEditor::VerifyClassValidity"); 
	// Check the current class selection to ensure it's valid for all selected nodes
	CStr className=GetClass();

	int  numSels=selSet.GetSize();

	for(int i=0;i<numSels;i++)
	{
		Object* obj=selSet[i]->EvalWorldState(0).obj;

		if (obj->ClassID()==vTRIGGER_CLASS_ID)
		{
			if (className==CStr(LEVELGEOM_CLASS))
			{
				//char strErr[256];
				//sprintf(strErr,"Your currently selected nodes for modification contain a Trigger Node '%s' which cannot be converted to a '%s'.\nThe operation was aborted.",(char*)selSet[i]->GetName(),LEVELGEOM_CLASS);
				//MessageBox(hwnd,strErr,"Invalid Trigger Object Conversion",MB_ICONWARNING|MB_OK);
				return false;
			}
		}

		if (obj->ClassID()==Class_ID(SIMPLE_CAM_CLASS_ID,0) ||
			obj->ClassID()==Class_ID(LOOKAT_CAM_CLASS_ID,0))
		{
			if (className!=CStr(CAMERA_CLASS))
			{
				char strErr[256];
				sprintf(strErr,"Your current selection contains a camera '%s' which cannot be converted from a '%s'.\nThe operation was aborted.",(char*)selSet[i]->GetName(),CAMERA_CLASS);
				MessageBox(hwnd,strErr,"Invalid Camera Object Conversion",MB_ICONWARNING|MB_OK);
				return false;
			}
		}
	}

	return true;
}

/*
void PropEditor::RemoveDefaults(CStr origClass,
								CStr origType,
								LinkList<ConfigProp>* origProps)
{
	// Merge in the new properties to the original properties if any of the new properties either
	// do not exist or they contain a value that does not match the default property in the original list

	Link<ConfigProp>* linkOrig = origProps->GetHead();
	Link<ConfigProp>* linkNext;
	int               index = 0;

	// Remove any items in the original list that match their defaults
	while(linkOrig)
	{
		linkNext = linkOrig->next;

		CStr value;
		if (!propView->GetValue(index+PROPSTART,value))
			value = linkOrig->data.value;

		if (value == GetDefault(origClass,origType,linkOrig->data.name))
			origProps->Remove(linkOrig);

		linkOrig=linkNext;
		index++;
	}
}
*/

void PropEditor::MergeProps(CStr className, CStr typeName, LinkList<ConfigProp>* cprops, LinkList<int>* updateFields)
{ FUNC_ENTER("PropEditor::MergeProps"); 
	// Go through the properties and update them
	Link<ConfigProp>* linkOrig = cprops->GetHead();
	Link<ConfigProp>* linkNew;
	bool              bAddProp;
	int               index=0;

	// Go through all orig props...
	while(linkOrig)
	{
		linkNew = props.GetHead();
		bAddProp = true;

		// If the prop exists within the property list set it's value
		while(linkNew)
		{
			if (linkNew->data.name == linkOrig->data.name)
			{
				CStr value;
				if (propView->GetValue(index+PROPSTART,value))	
				{
					// If the value is set at it's default, don't assign it
					if (value != GetDefault(className,typeName,linkNew->data.name))
					{
						if (updateFields)
						{
							int val = index+PROPSTART;
							updateFields->AddToTail(&val);
						}

						linkNew->data.value = value;
					}
				}

				bAddProp = false;
				break;
			}

			linkNew = linkNew->next;
		}

		// If not... add it to the property list
		if (bAddProp)
		{
			CStr value;
			propView->GetValue(index+PROPSTART,value);

			// Only add if not assigned to a default value
			if (value != GetDefault(className,typeName,linkOrig->data.name))
			{
				ConfigProp cprop = linkOrig->data;
				cprop.value = value;

				props.AddToTail(&cprop);

				if (updateFields)
				{
					int val = index+PROPSTART;
					updateFields->AddToTail(&val);
				}
			}
		}

		index++;
		linkOrig = linkOrig->next;
	}
}

void PropEditor::MergePropsDEF(CStr className, CStr typeName, LinkList<ConfigProp>* cprops, LinkList<int>* updateFields)
{ FUNC_ENTER("PropEditor::MergePropsDEF"); 
	// Go through the properties and update them
	Link<ConfigProp>* linkOrig = cprops->GetHead();
	Link<ConfigProp>* linkNew;
	bool              bAddProp;
	int               index=0;

	// Go through all orig props...
	while(linkOrig)
	{
		linkNew = props.GetHead();
		bAddProp = true;

		// If the prop exists within the property list set it's value
		while(linkNew)
		{
			if (linkNew->data.name == linkOrig->data.name)
			{
				CStr value;
				if (propView->GetValue(index+PROPSTART,value))	
				{
					// If the value is set at it's default, don't assign it
					if (value != GetDefault(className,typeName,linkNew->data.name))
					{
						if (updateFields)
						{
							int val = index+PROPSTART;
							updateFields->AddToTail(&val);
						}

						linkNew->data.value = value;
					}
				}

				bAddProp = false;
				break;
			}

			linkNew = linkNew->next;
		}

		// If not... add it to the property list
		if (bAddProp)
		{
			CStr value;
			propView->GetValue(index+PROPSTART,value);

			ConfigProp cprop = linkOrig->data;
			cprop.value = value;

			props.AddToTail(&cprop);

			if (updateFields)
			{
				int val = index+PROPSTART;
				updateFields->AddToTail(&val);
			}
			
		}

		index++;
		linkOrig = linkOrig->next;
	}
}

void PropEditor::AssignUpdated(LinkList<int>* fields)
{ FUNC_ENTER("PropEditor::AssignUpdated"); 
	Link<int>* link = fields->GetHead();

	while(link)
	{
		propView->SetMod(link->data,true);
		link = link->next;
	}
}

DWORD PropEditor::GetPropViewFlags()
{ FUNC_ENTER("PropEditor::GetPropViewFlags"); 
	DWORD pvFlags = 0;
	CStr value;
	/*
	propView->GetValue(PROP_CREATEDATSTART,value);

	if (value==CStr("TRUE"))
		pvFlags |= CCLASS_CREATEDATSTART;

	propView->GetValue(PROP_ABSENTINNETGAMES,value);

	if (value==CStr("TRUE"))
		pvFlags |= CCLASS_ABSENTINNETGAMES;

	propView->GetValue(PROP_TRICKOBJECT,value);

	if (value==CStr("TRUE"))
		pvFlags |= CCLASS_TRICKOBJECT;

	propView->GetValue(PROP_OCCLUDER,value);

	if (value==CStr("TRUE"))
		pvFlags |= CCLASS_OCCLUDER;
	*/

	return pvFlags;
}

bool PropEditor::ParseMapFile(char* Filename)
{ FUNC_ENTER("PropEditor::ParseMapFile"); 
	FILE* fp;
	char  buf[256];
	char  bufName[256];
	char  bufFileName[256];

	// This will cause a ton of data to be added to the TreeView control
	// Disable redrawing on the control to speed things up
	//pTreeView->DisableRedraw();

	DataType mode;

	enum LineMode
	{
		LMODE_NAME,
		LMODE_FILENAME
	};

	fp=fopen(Filename,"r");

	if (!fp)
	{
		//MessageBox(NULL,"Unable to open map file\nScript information will be unavailable","ParseMapFile",MB_OK|MB_ICONWARNING);
		SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,(WPARAM)0,(LPARAM)"Couldn't load qcomp.map");
		EnableWindow(GetDlgItem(hwnd,IDC_GLOBALLIST),FALSE);
		bMapUnloaded=TRUE;
		return false;
	}

	// Determine the size of the file
	fseek(fp,0,SEEK_END);
	int lastPos=ftell(fp);
	fseek(fp,0,SEEK_SET);

	// Create Progress Bar
	pProgBar=new ProgressBar(hInstance,NULL,"Loading Map File...",0,lastPos);

	// First two lines contain header information
	fgets(buf,256,fp);
	fgets(buf,256,fp);

	// Parse out the information line by line
	while(fgets(buf,256,fp))
	{
		// Clear out name buffers
		bufName[0]='\0';
		bufFileName[0]='\0';

		// Update Progress Bar
		pProgBar->SetVal(ftell(fp));

		// Determine if there's a change of type
		if (strstr(buf,"<Scripts>"))
		{
			pProgBar->SetCaption("Loading Scripts...");
			mode=DATA_SCRIPT;
			continue;
		}

		if (strstr(buf,"<Integers>"))
		{
			pProgBar->SetCaption("Loading Integers...");
			mode=DATA_INTEGER;
			continue;
		}

		if (strstr(buf,"<Floats>"))
		{
			pProgBar->SetCaption("Loading Floats...");
			mode=DATA_FLOAT;
			continue;
		}

		if (strstr(buf,"<Vectors>"))
		{
			pProgBar->SetCaption("Loading Vectors...");
			mode=DATA_VECTOR;
			continue;
		}

		if (strstr(buf,"<Pairs>"))
		{
			pProgBar->SetCaption("Loading Pairs...");
			mode=DATA_PAIR;
			continue;
		}

		if (strstr(buf,"<Strings>"))
		{
			pProgBar->SetCaption("Loading Strings...");
			mode=DATA_STRING;
			continue;
		}

		if (strstr(buf,"<LocalStrings>"))
		{
			pProgBar->SetCaption("Loading LocalStrings...");
			mode=DATA_LOCALSTRING;
			continue;
		}

		if (strstr(buf,"<Arrays>"))
		{
			pProgBar->SetCaption("Loading Arrays...");
			mode=DATA_ARRAY;
			continue;
		}

		if (strstr(buf,"<Structures>"))
		{
			pProgBar->SetCaption("Loading Structures...");
			mode=DATA_STRUCT;
			continue;
		}

		if (strstr(buf,"<Names>"))
		{
			pProgBar->SetCaption("Loading Names...");
			mode=DATA_NAME;
			continue;
		}

		int slen=strlen(buf);
		int namePos=0;
		int filePos=0;

		LineMode curMode=LMODE_NAME;

		for(int i=0;i<slen;i++)
		{
			switch(curMode)
			{
			case LMODE_NAME:
				if (buf[i]!=' ')
				{
					if (buf[i]!=29 && buf[i]!=10)
						bufName[namePos++]=buf[i];
				}
				else
				{
					bufName[namePos]='\0';
					curMode=LMODE_FILENAME;
				}
				break;
			
			case LMODE_FILENAME:
				if (buf[i]!=' ')
				{
					if (buf[i]!=29 && buf[i]!=10)
						bufFileName[filePos++]=buf[i];
				}
				break;
			}

			bufFileName[filePos]='\0';
		}

		// Don't add the entity if it's empty
		if (bufName[0]=='\0' || bufFileName[0]=='\0')
			continue;

		// Add type to script db
		TypeDesc* tdesc=AddType(bufName,bufFileName,mode);
		tdesc->type=mode;

		if (!tdesc)
		{
			MessageBox(NULL,"Failed to load Map file Hash table add failed.","ParseMapFile",MB_OK);
			return false;
		}

		switch(mode)
		{
		case DATA_SCRIPT:
			if (!bNoTreeview)
				pTVIScripts->AddItem(bufName,tdesc,TV_SORT);

			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,0,(LPARAM)(char*)bufName);
			break;

		case DATA_INTEGER:
			if (!bNoTreeview)
				pTVIIntergers->AddItem(bufName,tdesc,TV_SORT);

			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,0,(LPARAM)(char*)bufName);
			break;

		case DATA_FLOAT:
			if (!bNoTreeview)
				pTVIFloats->AddItem(bufName,tdesc,TV_SORT);

			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,0,(LPARAM)(char*)bufName);
			break;

		case DATA_VECTOR:
			if (!bNoTreeview)
				pTVIVectors->AddItem(bufName,tdesc,TV_SORT);

			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,0,(LPARAM)(char*)bufName);
			break;

		case DATA_PAIR:
			if (!bNoTreeview)
				pTVIPairs->AddItem(bufName,tdesc,TV_SORT);

			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,0,(LPARAM)(char*)bufName);
			break;

		case DATA_STRING:
			if (!bNoTreeview)
				pTVIStrings->AddItem(bufName,tdesc,TV_SORT);

			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,0,(LPARAM)(char*)bufName);
			break;

		case DATA_LOCALSTRING:
			if (!bNoTreeview)
				pTVILocalStrings->AddItem(bufName,tdesc,TV_SORT);

			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,0,(LPARAM)(char*)bufName);
			break;

		case DATA_ARRAY:
			if (!bNoTreeview)
				pTVIArrays->AddItem(bufName,tdesc,TV_SORT);

			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,0,(LPARAM)(char*)bufName);
			break;

		case DATA_STRUCT:
			if (!bNoTreeview)
				pTVIStructs->AddItem(bufName,tdesc,TV_SORT);

			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,0,(LPARAM)(char*)bufName);
			break;

		case DATA_NAME:
			if (!bNoTreeview)
				pTVINames->AddItem(bufName,tdesc,TV_SORT);

			SendDlgItemMessage(hwnd,IDC_GLOBALLIST,LB_ADDSTRING,0,(LPARAM)(char*)bufName);
			break;
		}
	}

	fclose(fp);

	delete pProgBar;
	pProgBar=NULL;

	// Reenable drawing on the TreeView control now that our items are added
	//pTreeView->EnableRedraw();
	//ShowWindow(hwnd,SW_SHOW);

	return true;
}

// This subclassed wndproc can be used to construct a floating tooltip to expand out
// entries in any listbox
LRESULT PropEditor::ttipWndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("PropEditor::ttipWndProc"); 
	HINSTANCE hInstance = (HINSTANCE)GetWindowLong(hwnd,GWL_HINSTANCE);
	HWND   hwndPE = FindPropEditor(hwnd);
	PropEditor* pthis = (PropEditor*)GetWindowLong(hwndPE,GWL_USERDATA);

	static ToolTip tip(hInstance);
	static POINT ptLast;
	RECT  rect;
	POINT pt;
	GetCursorPos(&pt);
	GetWindowRect(hwnd,&rect);

	switch(msg)
	{
	case WM_MOUSEMOVE:
		if (!pthis->bNameExpansion)
			return TRUE;
		
		if (GetCapture()!=hwnd)
			SetCapture(hwnd);

		if (!PtInRect(&rect,pt))
		{
			tip.Close();
			ReleaseCapture();
			return TRUE;
		}

		if (!(pt.x == ptLast.x && pt.y == ptLast.y))
		{
			char buf[2048];
			POINT ptClient = pt;
			
			ScreenToClient(hwnd,&ptClient);
			int index = CallWindowProc(pthis->OldNodeListProc,hwnd,LB_ITEMFROMPOINT,0,MAKELPARAM(ptClient.x,ptClient.y));

			RECT itemRect;

			CallWindowProc(pthis->OldNodeListProc,hwnd,LB_GETITEMRECT,(WPARAM)index,(LPARAM)&itemRect);

			if (index < 0 ||
				!PtInRect(&itemRect,ptClient) ||
				CallWindowProc(pthis->OldNodeListProc,hwnd,LB_GETSEL,(WPARAM)index,0)==FALSE)
			{
				//ReleaseCapture();			
				tip.Close();

				return CallWindowProc(pthis->OldNodeListProc,hwnd,LB_GETTEXT,(WPARAM)index,(LPARAM)buf);;
			}
			
			CallWindowProc(pthis->OldNodeListProc,hwnd,LB_GETTEXT,(WPARAM)index,(LPARAM)buf);

			if (strlen(buf)>1)
			{
				// Only create the tooltip if the item we're on top of is selected in the list
				if (CallWindowProc(pthis->OldNodeListProc,hwnd,LB_GETSEL,(WPARAM)index,0))
				{
					tip.CreateTip(buf,pt.x+15,pt.y);
					SetFocus(hwnd);
				}
			}
			else
			{
				tip.Close();
				ReleaseCapture();
			}

			ptLast = pt;
		}

		return TRUE;
	}

	return CallWindowProc(pthis->OldNodeListProc,hwnd,msg,wParam,lParam);
}

void PropEditor::OpenLODBrowser()
{ FUNC_ENTER("PropEditor::OpenLODBrowser"); 
	if (LODview && !LODview->IsActive())
	{
		LODview->Show();
		return;
	}

	if (LODview)
		LODview->Hide();

	LODview->Show();
}

void PropEditor::ReplaceInNodes()
{ FUNC_ENTER("PropEditor::ReplaceInNodes"); 
	pReplaceInNodesDlg->Show();
}

void PropEditor::ProcEditChange(RTFEditor* rtf,void* pData)
{ FUNC_ENTER("PropEditor::ProcEditChange"); 
	PropEditor* pthis = (PropEditor*)pData;

	int index = SendDlgItemMessage(pthis->hwnd,IDC_CONTEXTLIST,CB_GETCURSEL,0,0);

	if (index == CB_ERR || index == 0)
	{
		pthis->editScript = rtf->GetText();
		return;
	}

	if (IsWindowEnabled(GetDlgItem(pthis->hwnd,IDC_CONTEXTLIST)))
		pthis->editScripts[index-1].buffer = rtf->GetText();
	else
		pthis->editScript = rtf->GetText();
}

void PropEditor::ResetContext()
{ FUNC_ENTER("PropEditor::ResetContext"); 
	SendDlgItemMessage(hwnd,IDC_CONTEXTLIST,CB_RESETCONTENT,0,0);
	SendDlgItemMessage(hwnd,IDC_CONTEXTLIST,CB_ADDSTRING,0,(LPARAM)"Internal");
	SendDlgItemMessage(hwnd,IDC_CONTEXTLIST,CB_SETITEMDATA,0,0);
	SendDlgItemMessage(hwnd,IDC_CONTEXTLIST,CB_SETCURSEL,0,0);
}

bool PropEditor::ProcOmit(CStr& propBuffer)
{ FUNC_ENTER("PropEditor::ProcOmit"); 
	if (ip->GetSelNodeCount() > 1)
	{
		EnableWindow(GetDlgItem(hwnd,IDC_OMIT),FALSE);
		return false;
	}

	EnableWindow(GetDlgItem(hwnd,IDC_OMIT),TRUE);

	if (IsInstr(propBuffer,"// CMD:OMIT"))
	{
		CheckDlgButton(hwnd,IDC_OMIT,BST_CHECKED);
		return true;
	}

	if (IsInstr(propBuffer,"// CMD:FORCEOMIT"))
	{
		CheckDlgButton(hwnd,IDC_OMIT,BST_CHECKED);
		EnableWindow(GetDlgItem(hwnd,IDC_OMIT),FALSE);
		return true;
	}

	CheckDlgButton(hwnd,IDC_OMIT,BST_UNCHECKED);
	return false;
}

bool PropEditor::PopupParamsOpen()
{ FUNC_ENTER("PropEditor::PopupParamsOpen"); 
	if (pPopupParams)
		return pPopupParams->IsVisible();

	return false;
}

void PropEditor::ClosePopupParams()
{ FUNC_ENTER("PropEditor::ClosePopupParams"); 
	if (pPopupParams)
		pPopupParams->Hide();
}

void PropEditor::EscapeCB(PropList* pPropList, void* pData)
{ FUNC_ENTER("PropEditor::EscapeCB"); 
	PropEditor* pthis = (PropEditor*)pData;

	if (pthis->PopupParamsOpen())
		pthis->ClosePopupParams();
	else
		SendMessage(pthis->hwnd, WM_CLOSE, 0, 0);
}

///////// Functions related to subclassing MAX Edit controls so we can intercept the user pressing ESC
static WNDPROC MAXEditWndProc;

LRESULT CALLBACK ESCSubclassProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("ESCSubclassProc"); 
	// EDITDUMMY unfortunately doesn't get WM_KEYDOWN so we'll have to query
	// the driver with each update
	if (GetAsyncKeyState(VK_ESCAPE))
	{
		OutputDebugString("Subclassed Edit: PropEditor force close !!!\n");

		if (IsWindowVisible(hwnd))
		{
			if (IsChildOfPE(hwnd))
			{
				PropEditor* pe = (PropEditor*)GetWindowLong(hwndPropEdit,GWL_USERDATA);
				
				if (pe->PopupParamsOpen())
					pe->ClosePopupParams();
				else
					SendMessage(hwndPropEdit,WM_CLOSE,0,0);
			}
		}
	}

	return CallWindowProc(MAXEditWndProc,hwnd,msg,wParam,lParam);
}

static WNDPROC oldProc;

void PropEditor::SubclassMAXClasses()
{ FUNC_ENTER("PropEditor::SubclassMAXClasses"); 
	// Subclass all MAX custom edit controls so we can intercept the escape key
	// First we need to get our hands on MAX's hInstance
	HWND hwnd = ip->GetMAXHWnd();
	HINSTANCE hInstance = (HINSTANCE)GetWindowLong(hwnd,GWL_HINSTANCE);

	// Intercept their edit control message sink and patch it with ours
	// MAX constructs appears to construct a dummy window class that handles management
	// of keystrokes/focus within the CustEdit control.  Special thanks to Spy++ :)
	WNDCLASSEX wndclass;
	GetClassInfoEx(hInstance,"EDITDUMMY",&wndclass);

	// We'll create a fake window temporarily to get at the wndow class
	HWND hwndTmp = CreateWindow("EDITDUMMY","Temp",0,0,0,0,0,NULL,NULL,hInstance,0);

	DWORD rVal = SetClassLong(hwndTmp,GCL_WNDPROC,(LONG)ESCSubclassProc);

	MAXEditWndProc = wndclass.lpfnWndProc;
	wndclass.lpfnWndProc = ESCSubclassProc;

	DestroyWindow(hwndTmp);

	hwndPropEdit = this->hwnd;
	hwndLOD = this->LODview->GetHWND();
}

void PropEditor::UnSubclassMAXClasses()
{ FUNC_ENTER("PropEditor::UnSubclassMAXClasses"); 
	HWND hwnd = ip->GetMAXHWnd();
	HINSTANCE hInstance = (HINSTANCE)GetWindowLong(hwnd,GWL_HINSTANCE);

	// Repatch their message sink back to what it was
	WNDCLASSEX wndclass;
	GetClassInfoEx(hInstance,"EDITDUMMY",&wndclass);
	wndclass.lpfnWndProc = MAXEditWndProc;

	UnregisterClass("EDITDUMMY",hInstance);
	RegisterClassEx(&wndclass);
}
////////////////////////////////////// End MAX Subclassing

void PropEditor::OpenPropMergeDlg()
{ FUNC_ENTER("PropEditor::OpenPropMergeDlg"); 
	pPropMergeDlg->Show();
}

bool ConfigPropValCompare(ConfigProp* ca, ConfigProp* cb)
{ FUNC_ENTER("ConfigPropValCompare"); 
	char* nameA = strrchr(ca->name,'/');
	char* nameB = strrchr(cb->name,'/');

	if (nameA == NULL)
		nameA = ca->name;
	else
		nameA++;

	if (nameB == NULL)
		nameB = cb->name;
	else
		nameB++;

	return (strcmp(nameA,nameB) == 0) ? true : false;
}

void PropEditor::PropMergeCB(PropMergeDlg* pMergeDlg,void* pData)
{ FUNC_ENTER("PropEditor::PropMergeCB"); 
	LinkList<int> updateFields;
	LinkList<ConfigProp> cprops;
	LinkList<ConfigProgram> progs;

	PropEditor* pthis = (PropEditor*)pData;

	CStr className = pMergeDlg->GetClass();
	CStr typeName = pMergeDlg->GetType();
	CStr catName = pMergeDlg->GetCat();

	CStr buffer = pthis->GetClassTypeRecord(className,typeName);

	ParseConfigProps(&cprops,NULL,buffer,NULL,NULL,&progs,NULL);

	// Combine the new properties into the current proplist
	Link<ConfigProp>* curprop = cprops.GetHead();

	int maxSize = pthis->props.GetSize() + cprops.GetSize();
	bool ModArray = new bool[maxSize];

	while(curprop)
	{
		// Find property in main list
		Link<ConfigProp>* origprop = pthis->props.Find(ConfigPropValCompare,&curprop->data);

		if (origprop)
		{
			if (origprop->data.value != curprop->data.value)
			{
				origprop->data.value = curprop->data.value;
				origprop->data.userData = 1;
				pthis->strCmds += origprop->data.extdata;
			}
		}
		else
		{
			CStr newName = catName + CStr("/") + curprop->data.name;
			curprop->data.extdata = ReplaceStr(curprop->data.extdata,curprop->data.name,newName);
			curprop->data.name = newName;
			curprop->data.userData = 1;
			pthis->strCmds += curprop->data.extdata;
			pthis->props.AddToTail(&curprop->data);
		}

		curprop = curprop->next;
	}

	//pthis->props += cprops;

	pthis->ConvertToDefaults(className,typeName,&cprops);

	pthis->propView->SaveValues();
	pthis->propView->DestroyUI();
	//pthis->propView->Clear();
	
	// Keeped flagged properties rather than clearing
	while(pthis->propView->NumProps() > PROPSTART)
		pthis->propView->RemoveProp(PROPSTART);


	pthis->MapFileParser::BuildPropList(pthis->propView,&pthis->props);

	// Flag appropriate properties as modified	
	
	//pthis->propView->RestoreValues();
	pthis->bLockImmediateApply=TRUE;
	pthis->propView->BuildUI();
	pthis->propView->RestoreValues();

	int nProps = pthis->propView->NumProps();

	// Force set all values
	curprop = pthis->props.GetHead();
	while(curprop)
	{
		pthis->propView->SetValue(curprop->data.name,curprop->data.value);
		curprop = curprop->next;
	}

	for(int i=0;i<nProps;i++)
	{
		pthis->UpdateDefaultCondition(pthis->propView,i);
	}

	pthis->programs += progs;

	// Set modification flags
	curprop = pthis->props.GetHead();

	i = 0;
	while(curprop)
	{
		if (curprop->data.userData == 1)
			pthis->propView->SetMod(i,true);

		curprop = curprop->next;
		i++;
	}

	//pthis->MergePropsDEF(className,typeName,&pthis->props,&updateFields);
	//pthis->AssignUpdated(&updateFields);
	pthis->bLockImmediateApply=FALSE;
}

void PropEditor::OpenClusterFinder()
{ FUNC_ENTER("PropEditor::OpenClusterFinder"); 
	pClusterFindDlg->Show();
	pClusterFindDlg->ScanClusters();
}

void PropEditor::Show()
{ FUNC_ENTER("PropEditor::Show"); 
	EnableAutoPropAssign();
	ShowWindow(hwnd,SW_SHOW);
	SetFocus(hwnd);
	RegisterSelChange();
	SelectionSetChanged(TRUE);

	// Retain the last property category selected
	if (propView)
		propView->SetCategory(strLastPanel);
}

void PropEditor::SetValue( CStr& name, CStr value )
{ FUNC_ENTER("PropEditor::SetValue"); 
	if (propView)
	{
		propView->SetValue( name, value );
	}
}

// This function adds the current settings of an object into scripts.ini as a new type
void PropEditor::StoreProps()
{ FUNC_ENTER("PropEditor::StoreProps"); 
	char buf[256];
	GetDlgItemTextLast(hwnd,IDC_OBJCLASSM,buf,256);
	CStr className = CStr(buf);

	//int bResult;

	if (className.Length() == 0)
	{
		MessageBox(hwnd, "A class must currently be selected to use this option", "Store Properties to Scripts.ini Type", MB_ICONWARNING|MB_OK);
		return;
	}

	if (propView->NumProps() == 0)
	{
		MessageBox(hwnd, "Properties must exist within the property list to use this option", "Store Properties to Scripts.ini Type", MB_ICONWARNING|MB_OK);
		return;
	}

	InputDlg* inpDlg = new InputDlg(hInstance, hwnd, "Enter Type Name", "", "Store Properties to Scripts.ini Type");
	inpDlg->Show();		// Blocks

	if (inpDlg->WasCancelled())
		return;

	CStr typeName = inpDlg->GetInput();

	delete inpDlg;

	// Determine if the class/type combination is already inuse
	if (GetClassTypeRecord(className, typeName).Length() > 0)
	{
		char strErr[256];
		/*
		sprintf(strErr, "Class: '%s' Type: '%s' is already in use.  Do you want to overwrite it in scripts.ini?", (char*)className, (char*)typeName);
		bResult = MessageBox(hwnd, strErr, "Class/Type already exists in scripts.ini", MB_YESNO|MB_ICONWARNING);

		if (bResult != IDYES)
			return;
		*/

		sprintf(strErr, "Class: '%s' Type: '%s' is already in use. Operation Aborted.", (char*)className, (char*)typeName);
		MessageBox(hwnd, strErr, "Class/Type already exists in scripts.ini", MB_YESNO|MB_ICONWARNING);
		return;
	}

	// Open scripts.ini
	CStr strFile = getenv(APP_ENV);

	if (strFile.Length() == 0)
	{
		char strErr[256];
		sprintf(strErr, "The '%s' environment variable must be set to continue.  Operation Aborted.", APP_ENV);
		MessageBox(hwnd, strErr, "Environment variable not set", MB_ICONWARNING|MB_OK);
		return;
	}

	//strFile += CStr(SCRIPT_PATH) + SCRIPT_INI;
	strFile = GetScriptsIniPath();
	
	FILE* fp = fopen(strFile, "r+");

	if (!fp)
	{
		// Check the file attributes to see if the .qn is read-only
		if (_access(strFile,WRITE_ACCESS)!=0)
		{
			int  result;

			result = MessageBoxAll( hwnd, "The scripts.ini file is set to read-only.  How would you like to proceed?", "Scripts.ini Read-only Warning", MB_CHECKOUT_UNLOCK );
			if( result == IDWRITABLE )
			{
				// break the lock, if necessary
				SetFileAttributes( strFile, FILE_ATTRIBUTE_NORMAL );
			}
			else if( result == IDCHECKOUT )
			{
				char *args[4];
				int   process_result;
				DWORD attribs;

				args[0] = "p4";
				args[1] = "edit";
				args[2] = strFile;
				args[3] = NULL;
				//process_result = spawnvp( _P_WAIT, args[0], args );
				process_result = ExecuteCommand( args );
				attribs = GetFileAttributes( strFile );
				if(( attribs & FILE_ATTRIBUTE_READONLY ) || ( attribs == -1 ))
				{
					MessageBox(NULL,"An error occurred while trying to check the file out. Perhaps someone else has it checked out already?","Update Scripts.ini (Store Properties)",MB_ICONSTOP|MB_OK);
					return;
				}
			}
			else
			{
				return;
			}
		}
		else
		{
			MessageBox(hwnd, "Failed to open scripts.ini.  The operation has been aborted", "Scripts.ini open failed", MB_ICONSTOP|MB_OK);
			return;
		}

		fp = fopen(strFile, "r+");
		
		if (!fp)
		{
			MessageBox(hwnd, "Scripts.ini file open still failed.  Operation aborted.", "File open failed", MB_ICONSTOP|MB_OK);
			return;
		}
	}

	fclose(fp);
	fp = fopen(strFile, "rb");

	// Scan through the file until we find our class
	if (!AdvanceToClass(fp, className))
	{
		char strErr[256];
		sprintf(strErr, "Failed to locate class '%s' in the scripts.ini file.", (char*)className);
		MessageBox(hwnd, strErr, "Couldn't find class", MB_ICONWARNING|MB_OK);
		fclose(fp);
		return;
	}

	// Couldn't append in middle of file
	// Read the file into memory instead
	int   insertPos = ftell(fp);
	fseek(fp, 0, SEEK_END);
	int   size      = ftell(fp);

	char* bufPre  = new char[insertPos];
	char* bufPost = new char[size - insertPos];

	fseek(fp, 0, SEEK_SET);
	fread(bufPre, insertPos, 1, fp);
	fread(bufPost, size - insertPos, 1, fp);
	fclose(fp);

	fp = fopen(strFile, "wb");

	if (!fp)
	{
		MessageBox(hwnd, "Failed to reopen scripts.ini for writing.  Operation Aborted!", "Failed to open Scripts.ini", MB_ICONSTOP|MB_OK);
		return;
	}

	fwrite(bufPre, insertPos, 1, fp);

	// At this point we should be able to insert a new class
/*
	if (bResult == IDYES)
	{
		// Find the old class and replace it

	}
	else
*/
	{
		// We're adding a new type here
		char  sdate[20];
		char  stime[20];
		char  user[256];
		DWORD usersize = 255;
		GetUserName(user, &usersize);

		_strdate(sdate);
		_strtime(stime);

		char CRLF[2];
		CRLF[0] = 0x0D;
		CRLF[1] = 0x0A;

		fwrite(CRLF, 2, 1, fp);
		fprintf(fp, "\t\t; / This type was autogenerated %s %s by %s \\", sdate, stime, user);
		fwrite(CRLF, 2, 1, fp);
		fprintf(fp, "\t\t; ----------------------- < Start %s > -----------------------", (char*)typeName);
		fwrite(CRLF, 2, 1, fp);
		fprintf(fp, "\t\tType %s", (char*)typeName);
		fwrite(CRLF, 2, 1, fp);
		fprintf(fp, "\t\t{");
		fwrite(CRLF, 2, 1, fp);

		// Write out all the properties
		int nProps = propView->NumProps();

		for(int i = 0; i < nProps; i++)
		{
			CStr propName = propView->GetName(i);
			CStr propVal;

			propView->GetValue(i, propVal);
			if (propVal.Length() == 0)
				propVal = "[none]";

			fprintf(fp, "\t\t\t%s = %s", (char*)propName, (char*)propVal);
			fwrite(CRLF, 2, 1, fp);
		}

		// Need to insert directives for particle systems
		if (className == CStr("ParticleObject"))
		{
			Point3 ptMid, ptEnd, ptMidScale, ptEndScale;
			float  fMidWidth, fMidHeight, fMidLength, fEndWidth, fEndHeight, fEndLength;

			// Find the first selected node that's a particle system
			INode* pNode = NULL;

			int cnt = ip->GetSelNodeCount();

			for(int i = 0; i < cnt; i++)
			{
				INode* node = ip->GetSelNode(i);

				if (node->EvalWorldState(0).obj->ClassID() == PARTICLE_BOX_CLASS_ID)
				{
					pNode = node;
					break;
				}
			}

			if (pNode)
			{
				CStr buf;

				ComputeParticleOrientation(pNode, &ptMid, &ptEnd, &fMidWidth,
																  &fMidHeight,
																  &fMidLength,
																  &fEndWidth,
																  &fEndHeight,
																  &fEndLength,
																  &ptMidScale,
																  &ptEndScale);

				buf = BuildParticleOrientation(ptMid, ptEnd, fMidWidth,
					                                         fMidHeight,
														     fMidLength,
														     fEndWidth,
														     fEndHeight,
														     fEndLength,
															 ptMidScale,
															 ptEndScale);

				fprintf(fp, "%s", (char*)buf);
			}
		}

		fprintf(fp, "\t\t}");
		fwrite(CRLF, 2, 1, fp);
		fprintf(fp, "\t\t; ----------------------- < End %s > -------------------------", (char*)typeName);
		fwrite(CRLF, 2, 1, fp);
	}

	// Write the rest of the file
	fwrite(bufPost, size - insertPos, 1, fp);

	// Cleanup memory
	delete [] bufPre;
	delete [] bufPost;

	fclose(fp);

	Apply(false);

	// Store the selected nodes and restore them after scripts.ini refreshes
	int nSelNodes = ip->GetSelNodeCount();
	INode** nodes = new INode*[nSelNodes];
	int i;

	for(i = 0; i < nSelNodes; i++)
	{
		nodes[i] = ip->GetSelNode(i);

		// Reassign the type to correspond with our newly created type
		nodes[i]->SetUserPropString("Type", typeName);
	}

	RefreshScriptIni(false);

	ip->ClearNodeSelection();

	for(i = 0; i < nSelNodes; i++)
		ip->SelectNode(nodes[i], 0);
}

void PropEditor::UpdateParticleObjVisualization(CStr typeName)
{ FUNC_ENTER("PropEditor::UpdateParticleObjVisualization"); 
	int count = ip->GetSelNodeCount();

	// Scan through the selected objects and find the objects that are particle systems
	for(int i = 0; i < count; i++)
	{
		INode* node = ip->GetSelNode(i);
		
		Object* obj = node->EvalWorldState(0).obj;
		if (obj->ClassID() == PARTICLE_BOX_CLASS_ID)
		{
			IParticleMaster* pMaster;
			IParticleBox*    pbox    = dynamic_cast<IParticleBox*>(obj);

			pMaster = pbox->GetParticleMaster();

			IParticleBox* pboxMid = pbox->GetMidBox();
			IParticleBox* pboxEnd = pbox->GetEndBox();

			if (!pboxMid || !pboxEnd)
				return;

			INode* mainBoxNode    = pbox->GetObjectNode();
			INode* midBoxNode     = pboxMid->GetObjectNode();
			INode* endBoxNode     = pboxEnd->GetObjectNode();

			if (!mainBoxNode || !midBoxNode || !endBoxNode)
				return;

			CStr buf = GetClassTypeRecord("ParticleObject", typeName);


			Point3 ptMid, ptEnd, ptMidScale, ptEndScale;
			float fMidWidth, fMidHeight, fMidLength;
			float fEndWidth, fEndHeight, fEndLength;

			//pMaster->LockPosChange(true);

			

			if (!GetParticleOrientation(buf, &ptMid, &ptEnd, &fMidWidth,
															 &fMidHeight,
															 &fMidLength,
															 &fEndWidth,
															 &fEndHeight,
															 &fEndLength,
															 &ptMidScale,
															 &ptEndScale))
															 return;

			// Assign dimensions
			pboxMid->SetWidth(fMidWidth);
			pboxMid->SetHeight(fMidHeight);
			pboxMid->SetLength(fMidLength);

			pboxEnd->SetWidth(fEndWidth);
			pboxEnd->SetHeight(fEndHeight);
			pboxEnd->SetLength(fEndLength);

			gbLockPMU = TRUE;
			gbLockPosChange = TRUE;

			// Assign positions
			Matrix3 tm;
			tm = midBoxNode->GetNodeTM(0);
			tm.SetScale(ptMidScale);
			tm.SetTrans(ptMid + mainBoxNode->GetNodeTM(0).GetTrans());
			
			midBoxNode->SetNodeTM(0, tm);

			//pMaster->UpdateRefs();

			tm = endBoxNode->GetNodeTM(0);
			tm.SetScale(ptEndScale);
			tm.SetTrans(ptEnd + mainBoxNode->GetNodeTM(0).GetTrans());
			
			endBoxNode->SetNodeTM(0, tm);

			//pMaster->LockPosChange(false);

			//pMaster->UpdateRefs();
			ip->ForceCompleteRedraw();

			//GetParticleOrientation(propBuffer, &ptMid, &ptEnd, &fMid);
			//pboxMid->SetWidth(

			gbLockPMU = FALSE;
			gbLockPosChange = FALSE;
		}
	}
}

