#include "FuncEnter.h"

/*
	ConfigData.cpp

	Functions to parse configuration data (property, etc.) into lists
*/

#include "ConfigData.h"
#include "PropBufGen.h"
#include "ParseFuncs.h"
#include "NExt.h"
#include "appdata.h"
#include "../misc/gencrc.h"
#include "../Material/TerrainTypes.h"
#include "PropEdit.h"
#include "../Misc/Util.h"
#include "../Particle/NExtParticle.h"

extern PropEditor* pPropEdit;
extern bool gbLockPosChange;

bool GetProgram(CStr& lineBuf,LinkList<ConfigProgram>* list)
{ FUNC_ENTER("GetProgram"); 
	if (!list)
		return false;

	CStr lcLineBuf=lineBuf;
	lcLineBuf.toLower();

	ConfigProgram           cprog;
	char*                   bufScan;
	ConfigProgram::ProgType type;
	int                     pos;

	for(int i = 0; i < 2; i++)
	{
		switch(i)
		{
		case 0:
			bufScan = "// @program |";
			type = ConfigProgram::CFGPROG_STATIC;
			break;

		case 1:
			bufScan = "// @programvolatile |";
			type = ConfigProgram::CFGPROG_VOLATILE;
			break;
		}
			
		pos = Instr(lineBuf, bufScan);

		if (pos==-1)
			continue;

		CStr   buf;
		pos += strlen(bufScan) + 1;
		buf = GetRemainLinePartial(lineBuf,&pos);

		char val[512];
		GetOption(buf,val,'|',0);
		StripToken(val);
		cprog.filename = val;

		GetOption(buf,val,'|',1);

		if (val)
		{
			StripToken(val);
			cprog.scriptContext = val;
		}

		cprog.type = type;
		list->AddToTail(&cprog);
		return true;
	}

	return false;
}

CStr GetCluster(CStr& lineBuf)
{ FUNC_ENTER("GetCluster"); 
	CStr lcLineBuf=lineBuf;
	lcLineBuf.toLower();
	
	int TrickObjPos=Instr(lcLineBuf,"trickobject");
	int ClusterPos=Instr(lcLineBuf,"cluster");
	int EqPos=Instr(lineBuf,"=");

	if (TrickObjPos>-1         &&
		ClusterPos>-1          &&
		EqPos>-1               &&
		ClusterPos>TrickObjPos &&
		EqPos>TrickObjPos)
	{
		return StripLeft(lineBuf.Substr(EqPos+1,lineBuf.Length()));
	}

	return CStr("");
}

bool HasAutoDuck(char* str)
{ FUNC_ENTER("HasAutoDuck"); 
	if (Instr(str,"// @")!=-1)
		return true;

	return false;
}

void AttachExtendedData(LinkList<ConfigProp>* cplist,CStr& buf)
{ FUNC_ENTER("AttachExtendedData"); 
	// Oodles faster than old version
	char* curpos = strstr(buf, "// @next");
	char  eolval, endbarval;
	char  *startbar, *endbar, *namecmp, *eol;
	Link<ConfigProp>* curNode;
	char namecmpbuf[256];

	while(curpos)
	{		
		// Find start bar or eol
		startbar = curpos;
		eol = curpos;

		while(*startbar != 0    &&
			  *startbar != '\n' &&
			  *startbar != '\r')
		{
			if (*startbar == '|')
			{
				// Find end bar
				endbar = startbar + 1;
				eol    = endbar;

				while(*endbar != 0    &&
					  *endbar != '\n' &&
					  *endbar != '\r' &&
					  *endbar != '|')    { endbar++; }
				
				endbarval = *endbar;

				// Find end of line
				eol = endbar + 1;

				while(*eol != 0    &&
					  *eol != '\r' &&
					  *eol != '\n')   { eol++; }

				eolval = *eol;
				*eol   = 0;

				*endbar = 0;
				curNode = cplist->GetHead();

				namecmp = startbar + 1;
				strcpy(namecmpbuf, namecmp);
				StripToken(namecmpbuf);

				while(curNode)
				{
					//if (strstr(namecmp, curNode->data.name))
					if (strcmp(namecmpbuf, curNode->data.name) == 0)
					{
						// Add the current line from curpos to the first 0 \r\n
						*endbar = endbarval;
						curNode->data.extdata += CStr(curpos) + CStr("\r\n");
						*endbar = 0;
					}

					curNode = curNode->next;
				}

				*eol = eolval;
				*endbar = endbarval;

				endbar++;	
				break;
			}

			startbar++;
		}

		curpos = strstr(++eol, "// @next");
	}


		/*
		// Build up lines in the buffer that relate to this particular property
		int pos=0;

		while(pos<buflen)
		{
			CStr strTemp=GetRemainLinePartial(strAutoDuck,&pos);
			CStr strSearch;
			
			for(int i=0;i<7;i++)
			{
				switch(i)
				{
				case 0:
					strSearch=CStr("// @nextparm | ")+curNode->data.name;
					break;
				case 1:
					strSearch=CStr("// @nextdesc | ")+curNode->data.name;
					break;
				case 2:
					strSearch=CStr("// @nextdef | ")+curNode->data.name;
					break;
				case 3:
					strSearch=CStr("// @nextint | ")+curNode->data.name;
					break;
				case 4:
					strSearch=CStr("// @nextintdef | ")+curNode->data.name;
					break;
				case 5:
					strSearch=CStr("// @nextreq | ")+curNode->data.name;
					break;
				case 6:
					strSearch=CStr("// @nextcmd | ")+curNode->data.name;
					break;
				}

				if (Instr(strTemp,strSearch)!=-1)
					curNode->data.extdata+=strTemp+CStr("\r\n");
			}
		}

		curNode=curNode->next;
	}
	*/
}

CStr ParseNodeConfigProps(LinkList<ConfigProp>* cplist,DWORD* flags,INode* node,CStr* unkBuf,CStr* clusterBuf,LinkList<ConfigProgram>* programs,
					      LinkList<ConfigScript>* scriptList, int* terrain)
{ FUNC_ENTER("ParseNodeConfigProps"); 
	// The idea here is that we'll do duplicate the original non-parsed buffer and compare it against the current
	// buffer, if they're different the node needs reparsed (because a user or tool, modified the data in the
	// node's property buffer)  if they're the same, we can simply retrieve our already parsed data and hand it
	// off

	CStr propBuffer;
	CStr scriptBuffer;
	node->GetUserPropBuffer(propBuffer);

	//return ParseConfigProps(cplist, flags, propBuffer, unkBuf, clusterBuf, programs, scriptList);
	unsigned long CRC = GenerateCRC(propBuffer);

	//AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CONFIGPROPS_ORIGBUFFER);
	AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CONFIGPROPS_ORIGCRC);
	AppDataChunk* appdataTerrain = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_TERRAIN_TYPE);

	unsigned long* intCRC = NULL;

	if (appdata && appdata->data)
		intCRC = (unsigned long*)appdata->data;

	if (intCRC && *intCRC == CRC)
	{
		// The current buffer matches the cached buffer, we can use the cached data instead of reparsing
		appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CONFIGPROPS_CACHEBUFFER);

		if (appdata && appdata->data)
		{
			unsigned char* pos = (unsigned char*)appdata->data;

			GetList<ConfigProp>(&pos, cplist);

			if (flags)
				memcpy(flags, pos, sizeof(DWORD));

			pos += sizeof(DWORD);

			if (unkBuf)
				GetString(&pos, *unkBuf);
			else
				pos += GetStringSize(&pos);

			if (clusterBuf)
				GetString(&pos, *clusterBuf);
			else
				pos += GetStringSize(&pos);

			if (programs)
				GetList<ConfigProgram>(&pos, programs);
			else
				pos += ListSizeConfigProgram(&pos);
				//pos += ListSize<ConfigProgram>(&pos);

			if (scriptList)
				GetList<ConfigScript>(&pos, scriptList);
			else
				pos += ListSizeConfigScript(&pos);
				//pos += ListSize<ConfigScript>(&pos);

			GetString(&pos, scriptBuffer);

			// Get the terrain type
			if (appdataTerrain && appdataTerrain->data)
			{
				if (terrain)
					*terrain = *((int*)appdataTerrain->data);
			}
			else
			{
				int* memTerrain = (int*)malloc(sizeof(int));
				*memTerrain = GetTerrainType(propBuffer, cplist);
				node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_TERRAIN_TYPE, sizeof(int), memTerrain);

				if (terrain)
					*terrain = *memTerrain;
			}

			return scriptBuffer;
		}
	}

	// Cached data doesn't match reparse the data and update the cached data
	DWORD                   pncpFlags;
	CStr                    pncpUnkBuf;
	CStr                    pncpClusterBuf;
	LinkList<ConfigProgram> pncpPrograms;
	LinkList<ConfigScript>  pncpScriptList;
	int                     terrainType;

	//scriptBuffer = ParseConfigProps(cplist, &flags, propBuffer, unkBuf, clusterBuf, programs, scriptList);
	scriptBuffer = ParseConfigProps(cplist, 
		                            &pncpFlags, 
									propBuffer,
									&pncpUnkBuf,
									&pncpClusterBuf, 
									&pncpPrograms,
									&pncpScriptList,
									&terrainType);

	int bufSize = ListSize(cplist) +
		          sizeof(DWORD) +
				  GetStringSize(pncpUnkBuf) +
				  GetStringSize(pncpClusterBuf) +
				  ListSize(&pncpPrograms) +
				  ListSize(&pncpScriptList) + 
				  //ListSize<ConfigProgram>(&pncpPrograms) +
				  //ListSize<ConfigScript>(&pncpScriptList) +
				  GetStringSize(scriptBuffer);

	unsigned char* data = (unsigned char*)malloc(bufSize);
	unsigned char* pos = data;

	WriteList<ConfigProp>(&pos, cplist);

	memcpy(pos, &pncpFlags, sizeof(DWORD));
	pos += sizeof(DWORD);

	WriteString(&pos, pncpUnkBuf);
	WriteString(&pos, pncpClusterBuf);

	WriteList<ConfigProgram>(&pos, &pncpPrograms);
	WriteList<ConfigScript>(&pos, &pncpScriptList);

	WriteString(&pos, scriptBuffer);

	node->RemoveAppDataChunk(vNEXT_CLASS_ID,
		                       GUP_CLASS_ID,
							 vNAPP_NODE_CONFIGPROPS_CACHEBUFFER);

	node->AddAppDataChunk(vNEXT_CLASS_ID,
		                    GUP_CLASS_ID, 
						  vNAPP_NODE_CONFIGPROPS_CACHEBUFFER,
						  bufSize,
						  data);

	node->RemoveAppDataChunk(vNEXT_CLASS_ID,
		                       GUP_CLASS_ID,
							 vNAPP_NODE_CONFIGPROPS_ORIGCRC);

	/*
	char* propBufferChunk = (char*)malloc(propBuffer.Length() + 1);
	memcpy(propBufferChunk, (char*)propBuffer, propBuffer.Length() + 1);

	node->AddAppDataChunk(vNEXT_CLASS_ID, 
		                    GUP_CLASS_ID, 
						  vNAPP_NODE_CONFIGPROPS_ORIGBUFFER,
						  propBuffer.Length() + 1,
						  (char*)propBufferChunk);
	*/

	unsigned long* appDataMem = (unsigned long*)malloc(sizeof(unsigned long));

	node->AddAppDataChunk(vNEXT_CLASS_ID,
		                    GUP_CLASS_ID,
						  vNAPP_NODE_CONFIGPROPS_ORIGCRC,
						  sizeof(unsigned long),
						  appDataMem);

	// Add appdata for terrain type
	node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_TERRAIN_TYPE);

	int* memTerrain = (int*)malloc(sizeof(int));
	*memTerrain = terrainType;

	node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_TERRAIN_TYPE, sizeof(int), memTerrain);

	*appDataMem = CRC;

	if (flags)
		*flags = pncpFlags;

	if (unkBuf)
		*unkBuf = pncpUnkBuf;

	if (clusterBuf)
		*clusterBuf = pncpClusterBuf;

	if (programs)
		*programs = pncpPrograms;

	if (scriptList)
		*scriptList = pncpScriptList;

	// Assign the terrain type for the node
	if (terrain)
		*terrain = terrainType;

	return scriptBuffer;
}

void AddLegacyProps(DWORD flags, CStr strCluster, LinkList<ConfigProp>* cplist)
{ FUNC_ENTER("AddLegacyProps"); 
	if (cplist)
	{
		if (flags & CCLASS_OCCLUDER)
		{
			// Add default parameter
			ConfigProp cprop;

			cprop.name  = "Occluder";
			cprop.value = "TRUE";
			cplist->AddToHeadUnique(&cprop);
		}

		if (flags & CCLASS_TRICKOBJECT)
		{
			// Add default parameter
			ConfigProp cprop;

			cprop.name  = "Cluster";
			cprop.value = strCluster;
			cplist->AddToHeadUnique(&cprop);
		}

		if (flags & CCLASS_ABSENTINNETGAMES)
		{
			// Add default parameter
			ConfigProp cprop;

			cprop.name  = "AbsentInNetGames";
			cprop.value = "TRUE";
			cplist->AddToHeadUnique(&cprop);
		}

		if (flags & CCLASS_CREATEDATSTART)
		{
			// Add default parameter
			ConfigProp cprop;

			cprop.name    = "CreatedAtStart";
			cprop.value   = "TRUE";
			cplist->AddToHeadUnique(&cprop);
		}
	}
}

int GetTerrainType(CStr propBuffer, LinkList<ConfigProp>* cplist)
{ FUNC_ENTER("GetTerrainType"); 
	Link<ConfigProp>* link = cplist->GetHead();
	CStr LCname, LCvalue, LCvalue2;

	while(link)
	{
		if (strstr(link->data.name, "/"))
			LCname = strrchr(link->data.name, '/') + 1;
		else
			LCname = link->data.name;

		LCname.toLower();

		if (link->data.name == CStr("terraintype"))
		{
			// Look up the value in the list of available terrain types and find
			// the matching index
			LCvalue = link->data.value;

			// Need to account for possibility that the property may be defaulted
			if (LCvalue == CStr("!"))
				LCvalue = pPropEdit->GetDefault(GetClassName(propBuffer), GetTypeName(propBuffer), link->data.name);

			LCvalue.toLower();

			for(int i = 0; i < vNUM_TERRAIN_TYPES; i++)
			{
				LCvalue2 = CStr("terrain_") + terrain_types[i];
				LCvalue2.toLower();

				if (LCvalue == LCvalue2)
					return i;
			}
		}

		link = link->next;
	}

	return -1;
}

int GetTerrainType(CStr propBuffer, CStr name, CStr value)
{ FUNC_ENTER("GetTerrainType"); 
	CStr LCvalue, LCvalue2;
	LCvalue = value;

	if (LCvalue == CStr("!"))
	{
		CStr strClass = GetClassName(propBuffer);
		CStr strType  = GetTypeName(propBuffer);

		LCvalue = pPropEdit->GetDefault(GetClassName(propBuffer), GetTypeName(propBuffer), name);
	}

	LCvalue.toLower();

	for(int i = 0; i < vNUM_TERRAIN_TYPES; i++)
	{
		LCvalue2 = CStr("terrain_") + terrain_types[i];
		LCvalue2.toLower();

		if (LCvalue == LCvalue2)
			return i;
	}

	return -1;
}

// Inlined because only used in ParseConfigProps (seperated for code readability)
inline bool ValidPropLine(char* lineBuf)
{ FUNC_ENTER("ValidPropLine"); 
	// TODO: Optimize ME!

	if (CountChar(lineBuf,'=')>=1)
	{
		int pos2=Instr(lineBuf,"=");

		CStr strName=Left(lineBuf,pos2);
		char token[512];
		strcpy(token,strName);
		StripToken(token);
		strName=token;

		// If the property name contains invalid characters then this isn't a valid prop line
		if (strstr(strName, "//") ||
			strstr(strName, ";")  ||
			strstr(strName, "@"))
			return false;

		return true;
	}

	return false;
}

// WARNING! When used on a lot of nodes this function can start to become rather intensive
//          Use new ParseNodeConfigProps to use data caching system
CStr ParseConfigProps(LinkList<ConfigProp>* cplist,DWORD* flags,CStr& propBuffer,CStr* unkBuf,CStr* clusterBuf,LinkList<ConfigProgram>* programs,
					  LinkList<ConfigScript>* scriptList, int* terrain)
{ FUNC_ENTER("ParseConfigProps"); 
	int   len=propBuffer.Length();
	int   lastpos=0,origpos=0,pos=0,pos2;
	CStr  scriptBuffer;
	CStr  strCmds;
	CStr  strCmdsAD;
	CStr  strName;
	CStr  strValue;
	CStr  lineBuf;
	CStr  stripLineBuf;
	CStr  word;
	CStr  strCluster;
	char  token[512];
	char  eol[]="\r\n";
	DWORD intflags = 0;

	if (unkBuf)
		*unkBuf="";

	if (clusterBuf)
		*clusterBuf="";

	if (flags)
		*flags=0;

	if (programs)
		programs->Clear();

	if (cplist)
		cplist->Clear();

	if (scriptList)
		scriptList->Clear();

	if (terrain)
		*terrain = -1;

	while(pos<len)
	{
		origpos=pos;
		lineBuf=GetRemainLinePartial(propBuffer,&pos);

		// Check flags
		// Flags can now only be accepted if they are the only line contents
		stripLineBuf=StripLeft(lineBuf);
		
		// Set flags for legacy properties
		if (stripLineBuf==CStr("CreatedAtStart"))
		{				
			intflags |= CCLASS_CREATEDATSTART;
			continue;
		}
		if (stripLineBuf==CStr("AbsentInNetGames"))
		{
			intflags |= CCLASS_ABSENTINNETGAMES;
			continue;
		}
		if (stripLineBuf==CStr("TrickObject"))
		{
			intflags |= CCLASS_TRICKOBJECT;
			continue;
		}
		if (stripLineBuf==CStr("Occluder"))
		{
			intflags |= CCLASS_OCCLUDER;
			continue;
		}

		// Flags should no longer exist
		if (flags)
			*flags = 0;

		// Check default properties
		// If there is one (and only one) = and only 2 or 3 words
		// It's a default parameter, otherwise, it's a command string
		CStr leftEq;
		int  EqPos=Instr(lineBuf,"=");

		if (EqPos>0)
			leftEq=Left(lineBuf,EqPos-1);
		
		// This will force lines with x=y to x = y while not changiing x = y to x  =  y
		lineBuf=ReplaceStr(lineBuf," = ","=");
		lineBuf=ReplaceStr(lineBuf,"="," = ");
		//int words=NumWords(lineBuf);
		//if (CountChar(lineBuf,'=')==1 && (words==2 || words==3))

		//if (CountChar(lineBuf,'=')==1 && NumWords(leftEq)==1)
		if (ValidPropLine(lineBuf))
		{
			pos2=Instr(lineBuf,"=");

			if (pos2!=-1)
			{
				// Get property name (left of ='s)
				strName=Left(lineBuf,pos2);
				strcpy(token,strName);
				StripToken(token);
				strName=token;

				// Get property value (right of ='s)
				strValue=lineBuf.Substr(pos2+1,lineBuf.Length());
						
				strcpy(token,strValue);
				StripToken(token);
				strValue=token;

				CStr valLC = strValue;
				valLC.toLower();

				if (valLC == CStr("[none]"))
					strValue = "";

				// If the value is blank it may have been dropped through multiple lines
				// or it may just not equal anything (error condition).

				// unfortunately this means we also have to handle nesting of brackets
				// and potential syntax errors
				if (IsBlank(strValue))
				{
					int oldpos=pos;
					int nstart=GetNestStart(propBuffer,&pos);

					if (nstart==-1)
						pos=oldpos;
					else
					{
						strValue=GetGroup(propBuffer,&pos);
						strValue=OneLineConvert(strValue);
						pos++;

						if (strValue==CStr(""))
							pos=oldpos;
					}
				}

				// Don't add Class and Type they're added elsewhere
				if (strName!=CStr("Class") &&
					strName!=CStr("class") &&
					strName!=CStr("Type")  &&
					strName!=CStr("type"))
				{
					if (cplist)
					{
						// Add default parameter
						ConfigProp cprop;

						cprop.name =strName;
						cprop.value=strValue;

						// Property will not be readded if it already exists, but altered
						// to contain the new value
						Link<ConfigProp>* linkProp = cplist->Find(&cprop);

						if (linkProp)
							linkProp->data.value = strValue;
						else
							cplist->Add(&cprop);
					}

					// Determine terrain value
					if (terrain)
					{
						CStr strNameLC;

						if (strstr(strName,"/"))
							strNameLC = strrchr(strName, '/') + 1;
						else
							strNameLC = strName;

						strNameLC.toLower();

						if (strNameLC == CStr("terraintype"))
							*terrain = GetTerrainType(propBuffer, strName, strValue);
					}

					// Handle new format cluster buffer
					if (clusterBuf)
					{
						CStr strNameLC = strName;
						strNameLC.toLower();
						
						if (strcmp(strNameLC,"trickobject")==0)
						{
							*clusterBuf = strValue;

							if (flags)
								(*flags)|=CCLASS_TRICKOBJECT;
						}
					}
				}
			}
		}
		else
		{
			lastpos=pos;
			pos=origpos;
			if (GetToken(propBuffer,&pos,token," \n\t"))
			{
				StripToken(token);
				word=CStr(token);

				// Check script buffer
				if (word==CStr("script"))
				{
					// Extract the script
					word=GetRemainLineExact(propBuffer,&pos);

					while(word!=CStr("endscript") && pos<propBuffer.Length())
					{
						scriptBuffer+=word;
						scriptBuffer+=CStr(eol);

						word=GetRemainLineExact(propBuffer,&pos);
					}
				}
				else if (word==CStr("altscript"))
				{
					ConfigScript cscript;

					//GetToken(propBuffer,&pos,token," ");
					//StripToken(token);
					//cscript.name     = token;
					//cscript.filename = GetRemainLineExact(propBuffer,&pos);
					cscript.name = GetRemainLineExact(propBuffer,&pos);
					
					word = GetRemainLineExact(propBuffer,&pos);

					while(word!=CStr("endscript") && pos<propBuffer.Length())
					{
						cscript.buffer += word;
						cscript.buffer += CStr(eol);

						word = GetRemainLineExact(propBuffer,&pos);
					}

					if (scriptList)
						scriptList->AddUnique(&cscript);
				}
				else
				{
					// Check Unknown commands
					// Line must represent command
					// Readvance position and add it to the command buffer
					pos=lastpos;

					// Check for @program lines
					if (!GetProgram(lineBuf,programs))
					{
						// Check for TrickObject Cluster= lines
						if (clusterBuf)
						{
							CStr cluster;

							if (lineBuf.Length()>0)
							{
								cluster=GetCluster(lineBuf);

								if (cluster.Length()>0)
								{
									if (flags)
										(*flags)|=CCLASS_TRICKOBJECT;

									*clusterBuf=cluster;
								}
								else
								{
									// Omit lines are not considered unknown cmds
									if (!strstr(lineBuf, "Omit ") &&
										!strstr(lineBuf, "[NonPlaceable]") &&
										!strstr(lineBuf, "[TriggerOnly]"))
									{
										if (HasAutoDuck(lineBuf))
											strCmdsAD+=lineBuf+CStr(eol);
										else
											strCmds+=lineBuf+CStr(eol);
									}
								}
							}

						}
						else
							if (lineBuf.Length()>0)
							{
								if (!strstr(lineBuf, "Omit "))
								{
									if (HasAutoDuck(lineBuf))
										strCmdsAD+=lineBuf+CStr(eol);
									else
										strCmds+=lineBuf+CStr(eol);
								}
							}
					}
				}
			}
			else
			{
				if (cplist)
				{
					AttachExtendedData(cplist,strCmdsAD);
					//cplist->Sort();
				}

				if (unkBuf)
					*unkBuf=strCmds;

				AddLegacyProps(intflags, strCluster, cplist);
				return scriptBuffer;
			}
		}
	}

	if (cplist)
	{
		AttachExtendedData(cplist,strCmdsAD);
		//cplist->Sort();
	}

	if (unkBuf)
		*unkBuf=strCmds;

	AddLegacyProps(intflags, strCluster, cplist);
	return scriptBuffer;
}

void ParseReqScripts(LinkList<CStr>* list,CStr& propBuffer)
{ FUNC_ENTER("ParseReqScripts"); 
	int pos = 0;
	int len = propBuffer.Length();

	char token[512];
	CStr word;

	while(pos<len)
	{
		if (GetToken(propBuffer,&pos,token," \n\t"))
		{
			StripToken(token);
			word = token;

			if (word == CStr("altscript"))
			{
				GetToken(propBuffer,&pos,token," ");		// Skip the script name
			
				CStr filename = GetRemainLineExact(propBuffer,&pos);
				list->Add(&filename);
			}
		}
	}
}

// Use this as opposed to MAX's GetUserPropString to get a value that may contain spaces
CStr GetPropValue(INode* node, CStr propName)
{ FUNC_ENTER("GetPropValue"); 
	CStr propBuffer;
	LinkList<ConfigProp> cprops;

	node->GetUserPropBuffer(propBuffer);
	ParseConfigProps(&cprops,NULL,propBuffer);
	
	Link<ConfigProp>* curnode=cprops.GetHead();

	while(curnode)
	{
		if (curnode->data.name == propName)
		{
			if (curnode->data.value == CStr("[none]"))
				return CStr("");

			return curnode->data.value;
		}

		curnode=curnode->next;
	}

	return CStr("");
}

CStr GetPropValueDef(ScriptIniParser* pparser, INode* node, CStr propName)
{ FUNC_ENTER("GetPropValueDef"); 
	CStr rVal = GetPropValue(node, propName);

	CStr propBuffer, className, typeName;
	LinkList<ConfigProp> cprops;

	node->GetUserPropBuffer(propBuffer);
	ParseConfigProps(&cprops,NULL,propBuffer);
	className = GetClassName(propBuffer);
	typeName  = GetTypeName(propBuffer);
	
	pparser->ConvertToDefaults(className, typeName, &cprops);

	Link<ConfigProp>* curnode=cprops.GetHead();

	while(curnode)
	{
		if (curnode->data.name == propName)
		{
			if (curnode->data.value == CStr("[none]"))
				return CStr("");

			return curnode->data.value;
		}

		curnode=curnode->next;
	}

	return CStr("");
}

// AddPropValue functions add a property to the node list's prop buffer so other tools can store their
// data in a format that's compatible with the property editor
void AddPropValue(INode* node, CStr category, CStr propName, CStr propValue, CStr extdata)
{ FUNC_ENTER("AddPropValue"); 
	CStr propBuffer;
	CStr eol = "\r\n";

	if (propValue == CStr(""))
		propValue = "[none]";

	if (category.Length() > 0)
		RemovePropBufProp(node, category + CStr("/") + propName);
	else
		RemovePropBufProp(node, propName);

	node->GetUserPropBuffer(propBuffer);

	if (category.Length() > 0)
		propBuffer += category + CStr("/") + propName + CStr(" = ") + propValue + eol;
	else
		propBuffer += propName + CStr(" = ") + propValue + eol;

	if (extdata.Length() > 0)
	{
		if (category.Length() > 0)
		{
			propBuffer += CStr("// @nextparm | ") + category + CStr("/") + propName + CStr(" | ") + extdata + eol;
			propBuffer += CStr("// @nextint | ") + category + CStr("/") + propName + eol;
		}
		else
		{
			propBuffer += CStr("// @nextparm | ") + propName + CStr(" | ") + extdata + eol;
			propBuffer += CStr("// @nextint | ") + propName + eol;
		}
	}

	node->SetUserPropBuffer(propBuffer);
}

void AddPropValue(INode* node, CStr category, CStr propName, float value, CStr extdata)
{ FUNC_ENTER("AddPropValue"); 
	char val[256];
	sprintf(val, "%f", value);

	CStr propBuffer;
	CStr eol = "\r\n";

	if (category.Length() > 0)
		RemovePropBufProp(node, category + CStr("/") + propName);
	else
		RemovePropBufProp(node, propName);

	node->GetUserPropBuffer(propBuffer);

	if (category.Length() > 0)
		propBuffer += category + CStr("/") + propName + CStr(" = ") + CStr(val) + eol;
	else
		propBuffer += propName + CStr(" = ") + CStr(val) + eol;

	if (extdata.Length() > 0)
	{
		if (category.Length() > 0)
		{
			propBuffer += CStr("// @nextparm | ") + category + CStr("/") + propName + CStr(" | ") + extdata + eol;
			propBuffer += CStr("// @nextint | ") + category + CStr("/") + propName + eol;
		}
		else
		{
			propBuffer += CStr("// @nextparm | ") + propName + CStr(" | ") + extdata + eol;
			propBuffer += CStr("// @nextint | ") + propName + eol;
		}
	}

	node->SetUserPropBuffer(propBuffer);
}

void AddPropValue(INode* node, CStr category, CStr propName, int value, CStr extdata)
{ FUNC_ENTER("AddPropValue"); 
	char val[256];
	sprintf(val, "%i", value);

	CStr propBuffer;
	CStr eol = "\r\n";

	if (category.Length() > 0)
		RemovePropBufProp(node, category + CStr("/") + propName);
	else
		RemovePropBufProp(node, propName);

	node->GetUserPropBuffer(propBuffer);

	if (category.Length() > 0)
		propBuffer += category + CStr("/") + propName + CStr(" = ") + CStr(val) + eol;
	else
		propBuffer += propName + CStr(" = ") + CStr(val) + eol;

	if (extdata.Length() > 0)
	{
		propBuffer += CStr("// @nextparm | ") + category + CStr("/") + propName + CStr(" | ") + extdata + eol;
		propBuffer += CStr("// @nextint | ") + category + CStr("/") + propName + eol;
	}
	else
	{
		propBuffer += CStr("// @nextparm | ") + propName + CStr(" | ") + extdata + eol;
		propBuffer += CStr("// @nextint | ") + propName + eol;
	}

	node->SetUserPropBuffer(propBuffer);
}

// Breaks an array (fmt [ 1 2 3 4 ]) or (fmt [ 1, 2, dsasdf, 5 ]), etc. into it's components
void ParseArray(LinkList<CStr>* list,CStr value)
{ FUNC_ENTER("ParseArray"); 
	list->Clear();

	CStr nameRun;
	bool bStart=false;
	bool bInRun=false;

	for(int i=0;i<value.Length();i++)
	{
		if (value[i]=='[')
		{
			bStart=true;
			continue;
		}

		if (bStart)
		{
			if (value[i]==']')
			{
				// Add final element if in run
				if (bInRun)
				{
					list->AddToTail(&nameRun);
					nameRun="";
					bInRun=false;
				}
				break;
			}

			if (value[i]==' ' || value[i]==',')
			{
				if (bInRun)
				{
					list->AddToTail(&nameRun);
					nameRun="";
					bInRun=false;
				}
			}
			else
			{
				char tmpStr[2];
				tmpStr[0]=value[i];
				tmpStr[1]=0;

				nameRun+=tmpStr;
				bInRun=true;
			}
		}
	}
}

bool GetProperty(char* buf, char* prop, char* out)
{ FUNC_ENTER("GetProperty"); 
	char* eq;
	int   proplen  = strlen(prop);
	char* pos      = strstr(buf, prop);

	*out = 0;

	if (!pos)
	{
		return false;
	}

	eq = pos + proplen;

	while(*eq != '=') 
	{ 
		if (*eq == 0    ||
			*eq == '\r' ||
			*eq == '\n')
		{
			return false;
		}

		eq++;
	}
	
	eq++;

	// Copy the value to the output buffer
	while(*eq != 0    &&
		  *eq != '\r' &&
		  *eq != '\n')
	{
		if (*eq != ' ')
		{
			*out = *eq;
			out++;
		}

		eq++;
	}

	*out = 0;
	return true;
}

static bigTime = 0;

CStr GetClassName(char* propBuffer)
{ FUNC_ENTER("GetClassNameA"); 
	char* eq;
	char* pos = strstr(propBuffer, "Class");
	char  buf[256];
	char* out = buf;

	*out = 0;

	if (!pos)
	{
		pos = strstr(propBuffer, "class");

		if (!pos)
		{
			return CStr("");
		}
	}

	eq = pos + 5;

	while(*eq != '=') 
	{ 
		if (*eq == 0    ||
			*eq == '\r' ||
			*eq == '\n')
		{
			return CStr("");
		}

		eq++;
	}

	eq++;

	// Copy the value to the output buffer
	while(*eq != 0    &&
		  *eq != '\r' &&
		  *eq != '\n')
	{
		if (*eq != ' ')
		{
			*out = *eq;
			out++;
		}

		eq++;
	}

	*out = 0;

	return CStr(buf);

	/*
	CStr propCopy=propBuffer;
	propCopy.toLower();

	int pos=Instr(propCopy,"class");

	if (pos!=-1)
	{
		pos=Instr(propCopy,"=",pos);
		pos++;
		
		return StripLeft(GetRemainLinePartial(propBuffer,&pos));
	}

	return CStr("");
	*/
}

CStr GetTypeName(char* propBuffer)
{ FUNC_ENTER("GetTypeName"); 
	char* eq;
	char* pos = strstr(propBuffer, "Type");
	char  buf[256];
	char* out = buf;

	// Ensure that Type is at the beginning of the property name
	if (pos && pos != (char*)propBuffer &&
		*(pos-1) != 0 &&
		*(pos-1) != 10 &&
		*(pos-1) != 13 &&
		*(pos-1) != ' ')
		pos = NULL;

	*out = 0;

	if (!pos)
	{
		pos = strstr(propBuffer, "type");

		// Ensure that type is at the beginning of the property name
		if (pos && pos != (char*)propBuffer &&
			*(pos-1) != 0  &&
			*(pos-1) != 10 &&
			*(pos-1) != 13 &&
			*(pos-1) != ' ')
			pos = NULL;

		if (!pos)
			return CStr("");
	}

	eq = pos + 4;

	while(*eq != '=') 
	{ 
		if (*eq == 0    ||
			*eq == '\r' ||
			*eq == '\n')
		{
			return CStr("");
		}

		eq++;
	}

	eq++;

	// Copy the value to the output buffer
	while(*eq != 0    &&
		  *eq != '\r' &&
		  *eq != '\n')
	{
		if (*eq != ' ')
		{
			*out = *eq;
			out++;
		}

		eq++;
	}

	*out = 0;
	return CStr(buf);

	/*
	CStr propCopy=propBuffer;
	propCopy.toLower();

	int pos=Instr(propCopy,"type");

	if (pos!=-1)
	{
		pos=Instr(propCopy,"=",pos);
		if (pos == -1)
			return CStr("");
		
		pos++;
		
		return StripLeft(GetRemainLinePartial(propBuffer,&pos));
	}

	return CStr("");	
	*/
}

bool RemovePropBufProp(INode* node, CStr propName)
{ FUNC_ENTER("RemovePropBufProp"); 
	CStr propBuffer;
	bool bFoundProp = false;
	node->GetUserPropBuffer(propBuffer);

	LinkList<CStr> listLines;

	PropBufGen::BuildList(&listLines,propBuffer);

	Link<CStr>* curline = listLines.GetHead();
	Link<CStr>* nextline;

	while(curline)
	{
		nextline = curline->next;

		int pos = Instr(curline->data,propName);

		if (pos != -1)
		{
			// Need to remove any metacommands that refer to the property too
			char* metacmd = strstr(curline->data, "// @next");

			if (metacmd)
			{
				// Advance to end of known match
				metacmd += 8;

				// Find start of first field (which should contain prop name)
				while(*metacmd != '|' && *metacmd != 0)
					metacmd++;

				if (*metacmd != 0)
				{
					metacmd++;
					char* fieldend = metacmd;

					// Find end of the first field
					while(*fieldend != '|' && *fieldend != 0)
						fieldend++;

					char tmp = *fieldend;
					*fieldend = '\0';

					if (strstr(metacmd, propName))
					{
						listLines.Remove(curline);
						curline = nextline;
						continue;
					}

					*fieldend = tmp;
				}
			}

			// See if the next non space character is an equal sign
			for(int i=pos+propName.Length();i<curline->data.Length();i++)
			{
				if (curline->data[i] != ' ')
				{
					if (curline->data[i] == '=')
					{
						// We've found a line containing the property (remove it)
						bFoundProp = true;
						listLines.Remove(curline);
					}

					// This is not a valid property definition (next line)
					break;
				}
			}
		}

		curline = nextline;
	}

	// We've scanned through the buffer removing all lines containing the given property
	// reassemble the buffer and exit
	propBuffer = PropBufGen::BuildBufCRLF(&listLines);
	node->SetUserPropBuffer(propBuffer);

	return bFoundProp;
}

//
void SetParticleOrientation(INode* node, Point3 ptMid, Point3 ptEnd, float  fMidWidth,
							                                         float  fMidHeight,
																	 float  fMidLength,
																	 float  fEndWidth,
																	 float  fEndHeight,
																	 float  fEndLength,
																	 Point3 ptMidScale,
																	 Point3 ptEndScale)
{ FUNC_ENTER("SetParticleOrientation"); 
	gbLockPosChange = true;

	CStr propBuffer;
	node->GetUserPropBuffer(propBuffer);

	char buf[256];
	sprintf(buf, "Node: %s PtMid: (%f,%f,%f)  PtEnd: (%f,%f,%f)\n", (char*)node->GetName(), ptMid.x, ptMid.y, ptMid.z, ptEnd.x, ptEnd.y, ptEnd.z);
	OutputDebugString(buf);

	sprintf(buf, "MidWidth: %f  MidHeight: %f  MidLength: %f\n", fMidWidth, fMidHeight, fMidLength);
	OutputDebugString(buf);

	sprintf(buf, "EndWidth: %f  EndHeight: %f  EndLength: %f\n", fEndWidth, fEndHeight, fEndLength);
	OutputDebugString(buf);
	OutputDebugString("--------------------------------------------\n");

	// Scan through the property buffer and if any of our expected @nextparticle
	// lines exist, we will need to remove them before adding new lines
	LinkList<CStr> listBuf;
	Link<CStr>*    startLine = NULL;			// Lines should be added before any 'script' line

	BuildList(&listBuf, propBuffer);

	Link<CStr>* link = listBuf.GetHead();

	while(link)
	{
		if (strstr(link->data, "// @nextparticle"))
		{
			Link<CStr>* nextLink = link->next;
			listBuf.Remove(link);
			link = nextLink;
			continue;
		}

		if (!startLine && link->data == CStr("script"))
			startLine = link;

		link = link->next;
	}

	// Now add our new particle orientation lines
	char arg1[256], arg2[256], arg3[256];
	//char buf[256];

	// Add mid position
	sprintf(arg1, "%f", ptMid.x);
	sprintf(arg2, "%f", ptMid.y);
	sprintf(arg3, "%f", ptMid.z);

	strcpy(buf, "// @nextparticle | midpos | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	listBuf.AddBeforeLink(startLine, &CStr(buf));

	// Add mid dimensions
	sprintf(arg1, "%f", fMidWidth);
	sprintf(arg2, "%f", fMidHeight);
	sprintf(arg3, "%f", fMidLength);

	strcpy(buf, "// @nextparticle | middim | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	listBuf.AddBeforeLink(startLine, &CStr(buf));

	// Add mid scale
	sprintf(arg1, "%f", ptMidScale.x);
	sprintf(arg2, "%f", ptMidScale.y);
	sprintf(arg3, "%f", ptMidScale.z);

	strcpy(buf, "// @nextparticle | midscale | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	listBuf.AddBeforeLink(startLine,&CStr(buf));

	// Add end position
	sprintf(arg1, "%f", ptEnd.x);
	sprintf(arg2, "%f", ptEnd.y);
	sprintf(arg3, "%f", ptEnd.z);

	strcpy(buf, "// @nextparticle | endpos | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	listBuf.AddBeforeLink(startLine, &CStr(buf));

	// Add end dimensions
	sprintf(arg1, "%f", fEndWidth);
	sprintf(arg2, "%f", fEndHeight);
	sprintf(arg3, "%f", fEndLength);

	strcpy(buf, "// @nextparticle | enddim | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	listBuf.AddBeforeLink(startLine, &CStr(buf));

	// Add end scale
	sprintf(arg1, "%f", ptEndScale.x);
	sprintf(arg2, "%f", ptEndScale.y);
	sprintf(arg3, "%f", ptEndScale.z);

	strcpy(buf, "// @nextparticle | endscale | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	listBuf.AddBeforeLink(startLine, &CStr(buf));

	propBuffer = BuildBufCRLF(&listBuf);
	node->SetUserPropBuffer(propBuffer);

	gbLockPosChange = false;
}

CStr BuildParticleOrientation(Point3 ptMid, Point3 ptEnd, float  fMidWidth,
							                              float  fMidHeight,
														  float  fMidLength,
														  float  fEndWidth,
														  float  fEndHeight,
														  float  fEndLength,
														  Point3 ptMidScale,
														  Point3 ptEndScale)
{ FUNC_ENTER("BuildParticleOrientation"); 
	gbLockPosChange = true;

	CStr strBuffer;
	char CRLF[3];
	CRLF[0] = 13;
	CRLF[1] = 10;
	CRLF[2] = 0;

	// Now add our new particle orientation lines
	char arg1[256], arg2[256], arg3[256];
	char buf[256];

	// Add mid position
	sprintf(arg1, "%f", ptMid.x);
	sprintf(arg2, "%f", ptMid.y);
	sprintf(arg3, "%f", ptMid.z);

	strcpy(buf, "// @nextparticle | midpos | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	strBuffer += CStr(buf) + CStr(CRLF);

	// Add mid dimensions
	sprintf(arg1, "%f", fMidWidth);
	sprintf(arg2, "%f", fMidHeight);
	sprintf(arg3, "%f", fMidLength);

	strcpy(buf, "// @nextparticle | middim | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	// Add mid scale
	sprintf(arg1, "%f", ptMidScale.x);
	sprintf(arg2, "%f", ptMidScale.y);
	sprintf(arg3, "%f", ptMidScale.z);

	strBuffer += CStr(buf) + CStr(CRLF);

	strcpy(buf, "// @nextparticle | midscale | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	strBuffer += CStr(buf) + CStr(CRLF);

	// Add end position
	sprintf(arg1, "%f", ptEnd.x);
	sprintf(arg2, "%f", ptEnd.y);
	sprintf(arg3, "%f", ptEnd.z);

	strcpy(buf, "// @nextparticle | endpos | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	strBuffer += CStr(buf) + CStr(CRLF);

	// Add end dimensions
	sprintf(arg1, "%f", fEndWidth);
	sprintf(arg2, "%f", fEndHeight);
	sprintf(arg3, "%f", fEndLength);

	strcpy(buf, "// @nextparticle | enddim | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	strBuffer += CStr(buf) + CStr(CRLF);

	// Add end scale
	sprintf(arg1, "%f", ptEndScale.x);
	sprintf(arg2, "%f", ptEndScale.y);
	sprintf(arg3, "%f", ptEndScale.z);

	strcpy(buf, "// @nextparticle | endscale | ");
	strcat(buf, arg1);
	strcat(buf, " | ");
	strcat(buf, arg2);
	strcat(buf, " | ");
	strcat(buf, arg3);

	strBuffer += CStr(buf) + CStr(CRLF);
		
	gbLockPosChange = false;

	return strBuffer;
}

bool ComputeParticleOrientation(INode* node, Point3* pptMid, Point3* pptEnd, float*  pfMidWidth,
								                                             float*  pfMidHeight,
																			 float*  pfMidLength,
																			 float*  pfEndWidth,
																			 float*  pfEndHeight,
																			 float*  pfEndLength,
																			 Point3* pptMidScale,
																			 Point3* pptEndScale)
{ FUNC_ENTER("ComputeParticleOrientation"); 
	Object* obj = node->EvalWorldState(0).obj;

	if (obj->ClassID() == PARTICLE_BOX_CLASS_ID)
	{
		IParticleBox* pbox   = dynamic_cast<IParticleBox*>(obj);
		INode*        node   = pbox->GetObjectNode();
		IParticleBox* midBox = pbox->GetMidBox();
		IParticleBox* endBox = pbox->GetEndBox();

		if (!midBox || !endBox)
			return false;

		INode* midBoxNode    = midBox->GetObjectNode();
		INode* endBoxNode    = endBox->GetObjectNode();

		if (!midBoxNode || !endBoxNode)
			return false;

		// Must be relative coordinates to the root of the system
		Point3 posMidBox     = midBoxNode->GetNodeTM(0).GetTrans() - node->GetNodeTM(0).GetTrans();
		Point3 posEndBox     = endBoxNode->GetNodeTM(0).GetTrans() - node->GetNodeTM(0).GetTrans();

		if (pptMid)
			*pptMid = posMidBox;

		if (pptEnd)
			*pptEnd = posEndBox;

		if (pfMidWidth)
			*pfMidWidth = midBox->GetWidth();

		if (pfMidHeight)
			*pfMidHeight = midBox->GetHeight();

		if (pfMidLength)
			*pfMidLength = midBox->GetLength();

		if (pfEndWidth)
			*pfEndWidth = endBox->GetWidth();

		if (pfEndHeight)
			*pfEndHeight = endBox->GetHeight();

		if (pfEndLength)
			*pfEndLength = endBox->GetLength();

		if (pptMidScale)
		{
			Control* scale_ctrl;
			ScaleValue scale;

			scale_ctrl = midBoxNode->GetTMController()->GetScaleController();
			scale_ctrl->GetValue( 0, &scale, FOREVER );

			*pptMidScale = scale.s;
		}

		if (pptEndScale)
		{
			Control* scale_ctrl;
			ScaleValue scale;

			scale_ctrl = endBoxNode->GetTMController()->GetScaleController();
			scale_ctrl->GetValue( 0, &scale, FOREVER );

			*pptEndScale = scale.s;
		}

		return true;
	}

	return false;
}

bool GetParticleOrientation(char* extdata, Point3* pptMid, Point3* pptEnd, float*  pfMidWidth,
							                                               float*  pfMidHeight,
																		   float*  pfMidLength,
																		   float*  pfEndWidth,
																		   float*  pfEndHeight,
																		   float*  pfEndLength,
																		   Point3* pptMidScale,
																		   Point3* pptEndScale)
{ FUNC_ENTER("GetParticleOrientation"); 
	enum
	{
		GPO_MIDPOS    = 0x0001,
		GPO_ENDPOS    = 0x0002,
		GPO_MIDWIDTH  = 0x0004,
		GPO_MIDHEIGHT = 0x0008,
		GPO_MIDLENGTH = 0x0010,
		GPO_ENDWIDTH  = 0x0020,
		GPO_ENDHEIGHT = 0x0040,
		GPO_ENDLENGTH = 0x0080,
		GPO_MIDSCALE  = 0x0100,
		GPO_ENDSCALE  = 0x0200,
	};

	char  *pos = strstr(extdata, "// @nextparticle");
	char  mode[256];
	char  arg1[256];
	char  arg2[256];
	char  arg3[256];
	int   availFlags = 0;
	int   usedFlags = 0;

	if (pptMid)
		availFlags |= GPO_MIDPOS;

	if (pptEnd)
		availFlags |= GPO_ENDPOS;

	if (pfMidWidth)
		availFlags |= GPO_MIDWIDTH;

	if (pfMidHeight)
		availFlags |= GPO_MIDHEIGHT;

	if (pfMidLength)
		availFlags |= GPO_MIDLENGTH;

	if (pfEndWidth)
		availFlags |= GPO_ENDWIDTH;

	if (pfEndHeight)
		availFlags |= GPO_ENDHEIGHT;

	if (pfEndLength)
		availFlags |= GPO_ENDLENGTH;

	if (pptMidScale)
		availFlags |= GPO_MIDSCALE;

	if (pptEndScale)
		availFlags |= GPO_ENDSCALE;

	while(pos)
	{
		GetOption(pos, mode, '|', 1);
		GetOption(pos, arg1, '|', 2);
		GetOption(pos, arg2, '|', 3);
		GetOption(pos, arg3, '|', 4);

		StripToken(mode);
		StripToken(arg1);
		StripToken(arg2);
		StripToken(arg3);

		if (strcmp(mode, "midpos") == 0)
		{
			if (pptMid)
			{
				pptMid->x = atof(arg1);
				pptMid->y = atof(arg2);
				pptMid->z = atof(arg3);
			}

			usedFlags |= GPO_MIDPOS;
		}

		if (strcmp(mode, "middim") == 0)
		{
			if (pfMidWidth)
			{
				*pfMidWidth = atof(arg1);
				usedFlags |= GPO_MIDWIDTH;
			}

			if (pfMidHeight)
			{
				*pfMidHeight = atof(arg2);
				usedFlags |= GPO_MIDHEIGHT;
			}

			if (pfMidLength)
			{
				*pfMidLength = atof(arg3);
				usedFlags |= GPO_MIDLENGTH;
			}
		}

		if (strcmp(mode, "midscale") == 0)
		{
			if (pptMidScale)
			{
				pptMidScale->x = atof(arg1);
				pptMidScale->y = atof(arg2);
				pptMidScale->z = atof(arg3);
				usedFlags |= GPO_MIDSCALE;
			}
		}

		if (strcmp(mode, "endpos") == 0)
		{
			if (pptEnd)
			{
				pptEnd->x = atof(arg1);
				pptEnd->y = atof(arg2);
				pptEnd->z = atof(arg3);

				usedFlags |= GPO_ENDPOS;
			}
		}

		if (strcmp(mode, "enddim") == 0)
		{
			if (pfEndWidth)
			{
				*pfEndWidth = atof(arg1);
				usedFlags |= GPO_ENDWIDTH;
			}

			if (pfEndHeight)
			{
				*pfEndHeight = atof(arg2);
				usedFlags |= GPO_ENDHEIGHT;
			}

			if (pfEndLength)
			{
				*pfEndLength = atof(arg3);
				usedFlags |= GPO_ENDLENGTH;
			}
		}

		if (strcmp(mode, "endscale") == 0)
		{
			if (pptEndScale)
			{
				pptEndScale->x = atof(arg1);
				pptEndScale->y = atof(arg2);
				pptEndScale->z = atof(arg3);
				usedFlags |= GPO_ENDSCALE;
			}
		}

		pos = strstr(pos + 1, "// @nextparticle");
	}

	// If we get all requested points return true, if not false
	if (availFlags == usedFlags)
		return true;

	return false;
}

/*
CStr ParseConfigProps(LinkList<ConfigProp>* cplist,DWORD* flags,CStr propBuffer)
{
	int   len=propBuffer.Length();
	int   pos=0,pos2;
	CStr  word,word2;
	CStr  scriptBuffer;
	char  token[512];
	char  eol[]="\r\n";

	*flags=0;
	cplist->Clear();

	while(GetToken(propBuffer,&pos,token," \n\t"))
	{
		StripToken(token);
		word=CStr(token);

		// Scan ahead for '=' token
		pos2=pos;
		GetToken(propBuffer,&pos2,token," \n\t");
		StripToken(token);

		if (token[0]=='=')
		{
			ConfigProp cprop;

			// Extract property
			pos=pos2;
			word2=GetRemainLinePartial(propBuffer,&pos);

			// Don't add Class and Type they're added elsewhere
			if (word!=CStr("Class") &&
				word!=CStr("class") &&
				word!=CStr("Type")  &&
				word!=CStr("type"))
			{
				cprop.name =word;
				cprop.value=word2;
				cplist->Add(&cprop);
			}

			continue;
		}

		if (word==CStr("script"))
		{
			// Extract the script
			word=GetRemainLineExact(propBuffer,&pos);

			while(word!=CStr("endscript"))
			{
				scriptBuffer+=word;
				scriptBuffer+=CStr(eol);

				word=GetRemainLineExact(propBuffer,&pos);
			}

			break;
		}

		if (word==CStr("CreatedAtStart"))
			(*flags)|=CCLASS_CREATEDATSTART;

		if (word==CStr("AbsentInNetGames"))
			(*flags)|=CCLASS_ABSENTINNETGAMES;

		if (word==CStr("TrickObject"))
			(*flags)|=CCLASS_TRICKOBJECT;
	}

	return scriptBuffer;
}
*/

