/*
	InterLink.cpp
	Provides functionality for IPC between MAX and Level Editor
	Ideally I was hoping to write this into a seperate GUP but,
	since this would mean duplicating all the scene export code
	in either InterMaxLink.gup or the LevelEditor I'm going to
	build this into next.gup.  It will at least ensure consistency
	between scene export formats .scn and via shared memory.

	9-29-03 - aml
*/

#include "InterLink.h"
#include "..\..\InterMaxLink\InterMaxLink.h"
#include "InterLinkCodes.h"
#include "max.h"
#include "..\export\SceneExport.h"
#include "..\Misc\Saveable.h"

#define UPDATE_INTERVAL  250	// Time between InterLink message update polls in milliseconds
#define CLIENT_NAME      "MAX"	// Name of this app as it should be identified with other applications

#define WINDOW_IDENTITY  "InterLinkClassMAXServerWindow"  // Name of the invisible window for managing
														  // communication in and out of MAX for the Level Editor
														  // The Level Editor will look for this window when starting
														  // And messages will be sent between application for simple
														  // 2-way conversations

static class InterLinkSys* gpInterLinkSys = NULL;

extern HINSTANCE hInstance;

class InterLinkSys
{
	HWND    hwnd;				// Window to propagate data to timer proc
	HWND	hwndLevelEditor;	// Window handle of the Level Editor
	UINT    timerID;
	HANDLE  hThread;
	DWORD   dwThreadID;
	NxScene tempScene;
	bool    bUpdateComplete;	// True if thread has completed updating (a new update may be initiated)

	void imlTimerUpdate();
	void PostObject(NxObject* obj);
	void PostMtl(NxMaterial* mtl);

	static int imlDataNotify(void* pData, DWORD id, DWORD size, int arg1, int arg2);
	static LRESULT WndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);

	static DWORD WINAPI UpdateObjThread(void* pData);

	void FindLevelEditorWindow();

public:
	InterLinkSys();
	~InterLinkSys();
};

bool StartInterLinkServices()
{
	gpInterLinkSys = new InterLinkSys;
	return true;
}

void StopInterLinkServices()
{
	delete gpInterLinkSys;
	gpInterLinkSys = NULL;
}

InterLinkSys::InterLinkSys()
{
	bUpdateComplete = true;

	imlRegisterClient("MAX", imlDataNotify);

	// Create fake window to handle timer messages
	WNDCLASSEX wcex;

	wcex.cbSize         = sizeof(WNDCLASSEX); 
	wcex.style			= 0;
	wcex.lpfnWndProc	= (WNDPROC)WndProc;
	wcex.cbClsExtra		= 0;
	wcex.cbWndExtra		= 0;
	wcex.hInstance		= hInstance;
	wcex.hIcon			= NULL;
	wcex.hCursor		= NULL;
	wcex.hbrBackground	= NULL;
	wcex.lpszMenuName	= NULL;
	wcex.lpszClassName	= "InterLinkClass";
	wcex.hIconSm		= NULL;

	RegisterClassEx(&wcex);

	hwnd = CreateWindow("InterLinkClass",
		                "InterLinkClassMAXServerWindow",
						0,
						0,
						0,
						0,
						0,
						NULL,
						NULL,
						hInstance,
						NULL);

	SetWindowLong(hwnd, GWL_USERDATA, (LONG)this);

	timerID = SetTimer(hwnd, 0, UPDATE_INTERVAL, NULL);
}

InterLinkSys::~InterLinkSys()
{
	KillTimer(hwnd, timerID);
	DestroyWindow(hwnd);

	imlUnlockSection(0,0);
	imlUnregisterClient();
}

void InterLinkSys::FindLevelEditorWindow()
{
	// Scan through all available top level windows on the system and find
	// the one associated with the level editor
}

LRESULT InterLinkSys::WndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	InterLinkSys* pthis = (InterLinkSys*)GetWindowLong(hwnd, GWL_USERDATA);

	switch(msg)
	{
	case WM_IML_ATTACH:
		{
			pthis->hwndLevelEditor = (HWND)lParam;
			SendMessage(pthis->hwndLevelEditor, WM_IML_ATTACH_ACK, 0, 0);
		}
		break;

	case WM_IML_DETACH:
		{
			HWND tempHwnd = pthis->hwndLevelEditor;

			pthis->hwndLevelEditor = NULL;
			SendMessage(tempHwnd, WM_IML_DETACH_ACK, 0, 0);
		}
		break;

	case WM_TIMER:
		{
			InterLinkSys* pthis = (InterLinkSys*)GetWindowLong(hwnd, GWL_USERDATA);
			pthis->imlTimerUpdate();
		}
		break;
	}

	return DefWindowProc(hwnd, msg, wParam, lParam);
}

int InterLinkSys::imlDataNotify(void* pData, DWORD id, DWORD size, int arg1, int arg2)
{
	switch(id)
	{
	case NOTIFYDATA_NEWGEOMBUILD:
		{
			OutputDebugString("Got request sent to MAX to build Geom, but unimplemented");
			imlBlockAck();
		}
		break;
	};

	return 0;
}

DWORD InterLinkSys::UpdateObjThread(void* pData)
{
	InterLinkSys* pthis = (InterLinkSys*)pData;

	Interface* gInterface = GetCOREInterface();

	// Take the first selected object and post it's geom data to connected clients
	if (gInterface->GetSelNodeCount() > 0)
	{
		INode* node = gInterface->GetSelNode(0);

		if (node)
		{
			NxObject* nxobj = pthis->tempScene.CreateObjectFromNode(node);
			
			if (nxobj)
			{
				pthis->PostObject(nxobj);
				delete nxobj;
			}
		}
	}

	pthis->bUpdateComplete = true;
	return 1;
}

void InterLinkSys::imlTimerUpdate()
{
	imlUpdate();	

	if (bUpdateComplete)
	{
		bUpdateComplete = false;
		//hThread = CreateThread(NULL, 0, UpdateObjThread, this, 0, &dwThreadID);
		//SetThreadPriority(hThread, THREAD_PRIORITY_BELOW_NORMAL);
		//SetThreadPriority(hThread, THREAD_PRIORITY_LOWEST);
		UpdateObjThread(this);
	}
}

#define WriteLimit(pPos, sizein, data, cursize, maxsize) \
	if (cursize + sizein > maxsize)   \
	{                                 \
		imlUnlockSection(0, cursize); \
		return;						  \
	}                                 \
	else							  \
		cursize += Write(pPos, sizein, data);

#define WriteStringLimit(pPos, str, cursize, maxsize) \
	if (strlen(str) + 5 + cursize > maxsize)   \
	{                                 \
		imlUnlockSection(0, cursize); \
		return;						  \
	}                                 \
	else							  \
		cursize += WriteString(pPos, str);

 

void InterLinkSys::PostObject(NxObject* obj)
{
	unsigned char* pMem = (unsigned char*)imlLockSection();
	unsigned int   maxSize = imlGetBlockSize();

	// Abort post if mem locked
	if (!pMem)
		return;

	unsigned char* pPos = pMem;
	int size = 0;

	WriteLimit(&pPos, sizeof(int), &obj->m_NumFaces, size, maxSize);
	WriteLimit(&pPos, sizeof(NxFace) * obj->m_NumFaces, obj->m_Faces, size, maxSize);
	WriteLimit(&pPos, sizeof(int), &obj->m_NumVerts, size, maxSize);
	WriteLimit(&pPos, sizeof(NxVertex) * obj->m_NumVerts, obj->m_Verts, size, maxSize);

	WriteStringLimit(&pPos, obj->m_Name, size, maxSize);

	WriteLimit(&pPos, sizeof(float), &obj->m_BoundingBox.pmin.x, size, maxSize);
	WriteLimit(&pPos, sizeof(float), &obj->m_BoundingBox.pmin.y, size, maxSize);
	WriteLimit(&pPos, sizeof(float), &obj->m_BoundingBox.pmin.z, size, maxSize);
	WriteLimit(&pPos, sizeof(float), &obj->m_BoundingBox.pmax.x, size, maxSize);
	WriteLimit(&pPos, sizeof(float), &obj->m_BoundingBox.pmax.y, size, maxSize);
	WriteLimit(&pPos, sizeof(float), &obj->m_BoundingBox.pmax.z, size, maxSize);

	WriteLimit(&pPos, sizeof(int), &obj->m_Flags, size, maxSize);
	WriteLimit(&pPos, sizeof(int), &obj->m_NumUVSets, size, maxSize);
	WriteLimit(&pPos, sizeof(unsigned long), &obj->m_SkeletonChecksum, size, maxSize);
	WriteLimit(&pPos, sizeof(int), &obj->m_CASRemoveFlags, size, maxSize);
	WriteLimit(&pPos, sizeof(float), &obj->m_KBias, size, maxSize);
	WriteLimit(&pPos, sizeof(int), &obj->m_ParentMatrixIndex, size, maxSize);

	WriteLimit(&pPos, sizeof(unsigned int), &obj->m_LODVersion, size, maxSize);
	WriteLimit(&pPos, sizeof(int), &obj->m_LODFlags, size, maxSize);
	
	if (obj->m_LODFlags & NxObject::mMASTER)
	{
		WriteLimit(&pPos, sizeof(int), &obj->m_LODMaster.m_NumLODLevels, size, maxSize);
		WriteLimit(&pPos, sizeof(NxLODLevel) * obj->m_LODMaster.m_NumLODLevels, obj->m_LODMaster.m_LODLevels, size, maxSize);
	}

	if (obj->m_LODFlags & NxObject::mSLAVE)
	{
		WriteLimit(&pPos, sizeof(unsigned long), &obj->m_LODSlave.m_masterCRC, size, maxSize);
	}

	WriteLimit(&pPos, sizeof(unsigned int), &obj->m_Version, size, maxSize);
	WriteLimit(&pPos, sizeof(int), &obj->m_ParentCRC, size, maxSize);
	
	WriteLimit(&pPos, sizeof(int), &obj->m_NumChildren, size, maxSize);
	WriteLimit(&pPos, sizeof(unsigned long) * obj->m_NumChildren, obj->m_ChildCRCs, size, maxSize);

	WriteLimit(&pPos, sizeof(int), &obj->m_LODLevels, size, maxSize);
	WriteLimit(&pPos, sizeof(NxLODInfo) * obj->m_LODLevels, obj->m_LODinfo, size, maxSize);

	WriteLimit(&pPos, sizeof(NxObject::BillboardType), &obj->m_BillboardType, size, maxSize);
	WriteLimit(&pPos, sizeof(Point3), &obj->m_BillboardOrigin, size, maxSize);
	WriteLimit(&pPos, sizeof(Point3), &obj->m_PivotPos, size, maxSize);
	WriteLimit(&pPos, sizeof(Point3), &obj->m_PivotAxis, size, maxSize);

	imlUnlockSection(NOTIFYDATA_NEWOBJ, size);
	//imlWaitForReads();
}

void InterLinkSys::PostMtl(NxMaterial* pMtl)
{
	unsigned char* pMem    = (unsigned char*)imlLockSection();
	unsigned char* pPos    = pMem;
	int size = 0;
	unsigned int   maxsize = imlGetBlockSize();

	CStr str = pMtl->m_Name;
	size += WriteString(&pPos, str);
	// m_MaxMtl must be relinked by m_Name on completion

	WriteLimit(&pPos, sizeof(unsigned long), &pMtl->m_Checksum, size, maxsize);
	WriteLimit(&pPos, sizeof(unsigned long), &pMtl->m_NewChecksum, size, maxsize);	
	WriteLimit(&pPos, sizeof(bool), &pMtl->m_Transparent, size, maxsize);
	WriteLimit(&pPos, sizeof(int), &pMtl->m_NumPasses, size, maxsize);
	WriteLimit(&pPos, sizeof(int), &pMtl->m_AlphaCutoff, size, maxsize);
	WriteLimit(&pPos, sizeof(int), &pMtl->m_Terrain, size, maxsize);
	WriteLimit(&pPos, sizeof(bool), &pMtl->m_WarnedAlready, size, maxsize);
	WriteLimit(&pPos, sizeof(bool), &pMtl->m_OneSided, size, maxsize);
	WriteLimit(&pPos, sizeof(bool), &pMtl->m_TwoSided, size, maxsize);
	WriteLimit(&pPos, sizeof(bool), &pMtl->m_Invisible, size, maxsize);
	WriteLimit(&pPos, sizeof(float), &pMtl->m_DrawOrder, size, maxsize);
	WriteLimit(&pPos, sizeof(int), &pMtl->m_BasePass, size, maxsize);
	WriteLimit(&pPos, sizeof(bool), &pMtl->m_Sorted, size, maxsize);
	WriteLimit(&pPos, sizeof(int), &pMtl->m_CutoffFunc, size, maxsize);
	WriteLimit(&pPos, sizeof(bool), &pMtl->m_water, size, maxsize);
	WriteLimit(&pPos, sizeof(bool), &pMtl->m_grassify, size, maxsize);
	WriteLimit(&pPos, sizeof(float), &pMtl->m_grassHeight, size, maxsize);
	WriteLimit(&pPos, sizeof(int), &pMtl->m_grassLayers, size, maxsize);
	WriteLimit(&pPos, sizeof(int), &pMtl->m_NumWibbleSequences, size, maxsize);
	
	for(int i = 0; i < pMtl->m_NumWibbleSequences; i++)
	{
		WriteLimit(&pPos, sizeof(int), &pMtl->m_WibbleSequences[i].m_NumFrames, size, maxsize);
		WriteLimit(&pPos, sizeof(WibbleKeyframe) * pMtl->m_WibbleSequences[i].m_NumFrames, pMtl->m_WibbleSequences[i].m_WibbleFrames, size, maxsize);
	}

	// Fortunately NxMaterial pass contains no variably sized fields so we can do one big memcpy
	WriteLimit(&pPos, sizeof(NxMaterialPass) * vMAX_MATERIAL_PASSES, pMtl->m_Passes, size, maxsize);

	WriteLimit(&pPos, sizeof(bool), &pMtl->m_UseSpecular, size, maxsize);
	WriteLimit(&pPos, sizeof(float), &pMtl->m_SpecularPower, size, maxsize);
	WriteLimit(&pPos, sizeof(float) * 3, pMtl->m_SpecularColor, size, maxsize);

	imlUnlockSection(NOTIFYDATA_NEWMTL, size);
}
