#include "FuncEnter.h"

#include <next.h>
#include <misc/util.h>
#include <Misc/GenCrc.h>
#include <Image/image.h>
#include <Image/quant.h>
#include <Export/Export.h>
#include <stdmat.h>
#include <process.h>
#include <stdio.h>
#include <fstream.h>
#include <io.h>
#include "../ui/OktoAll.h"

#define TOP_TRANSPARENT		0x01
#define BOTTOM_TRANSPARENT	0x02
#define LEFT_TRANSPARENT	0x04
#define RIGHT_TRANSPARENT	0x08

void	NxTexture::initialize( void )
{ FUNC_ENTER("NxTexture::initialize"); 
	int i;

	for( i = 0; i < vMAX_MIP_LEVELS; i++ )
	{
		m_width[i] = 0;
		m_height[i] = 0;
		m_texel_data[i] = NULL;
		m_bpp[i] = 0;
		m_pixel_format[i] = v32_BIT;
		m_name[i][0] = '\0';
		memset( &m_filetime[i], 0, sizeof( FILETIME ));
	}

	m_grayscale = false;
	m_palette_bpp = 0;	
	m_palette_format = v32_BIT;	
	m_palette_data = NULL;
	sprintf( m_name[0], "None" );
	m_map = NULL;	
	m_mip_levels = 0;
	m_flags = 0;
	m_transparent_color.r = 0.0f;
	m_transparent_color.g = 0.0f;
	m_transparent_color.b = 0.0f;
	m_valid = true;
	m_platform_flags = 0;
	
	for(i=0;i<vMAX_MIP_LEVELS; i++)
		m_useBasePal[i]=FALSE;
}

NxTexture::NxTexture( void )
{ FUNC_ENTER("NxTexture::NxTexture"); 
	initialize();
}

NxTexture::~NxTexture( void )
{ FUNC_ENTER("NxTexture::~NxTexture"); 
	int i;

	for( i = 0; i < vMAX_MIP_LEVELS; i++ )
	{
		if( m_texel_data[i] )
		{			
			delete [] m_texel_data[i];
		}
	}

	if( m_palette_data )
	{
		delete [] m_palette_data;
	}
}

bool NxTexture::ShouldAutoGenerate32BitMips( void )
{ FUNC_ENTER("NxTexture::ShouldAutoGenerate32BitMips"); 
	if ( m_flags & mAUTO_GENERATE_NGC_MIPS ||
		 m_flags & mAUTO_GENERATE_XBOX_MIPS)
		 return true;

	return false;
}

bool NxTexture::ShouldAutoGenerateMipMaps( void )
{ FUNC_ENTER("NxTexture::ShouldAutoGenerateMipMaps"); 
	return (( m_flags & mAUTO_GENERATE_MIPMAPS ) != 0 );

//	if ( m_flags & mAUTO_GENERATE_MIPMAPS ||
//		 m_flags & mAUTO_GENERATE_NGC_MIPS ||
//		 m_flags & mAUTO_GENERATE_XBOX_MIPS)
//		 return true;
//
//	return false;
}

void NxTexture::SetNumMipLevels( int num_mip_levels )
{ FUNC_ENTER("NxTexture::SetNumMipLevels"); 
	m_mip_levels = num_mip_levels;
}

int	NxTexture::GetNumMipLevels( void )
{ FUNC_ENTER("NxTexture::GetNumMipLevels"); 
	return m_mip_levels;
}

int	NxTexture::GetPixelFormat( void )
{ FUNC_ENTER("NxTexture::GetPixelFormat"); 
	return m_pixel_format[0];
}

void NxTexture::SetPixelFormat( int format )
{ FUNC_ENTER("NxTexture::SetPixelFormat"); 
	m_pixel_format[0] = format;
}

void NxTexture::SetPaletteFormat( int format )
{ FUNC_ENTER("NxTexture::SetPaletteFormat"); 
	m_palette_format = format;
}

int	NxTexture::GetPaletteFormat( void )
{ FUNC_ENTER("NxTexture::GetPaletteFormat"); 
	return m_palette_format;
}

char* NxTexture::GetTexelData( int mip_level )
{ FUNC_ENTER("NxTexture::GetTexelData"); 
	return m_texel_data[mip_level];
}

char* NxTexture::GetPaletteData( void )
{ FUNC_ENTER("NxTexture::GetPaletteData"); 
	return m_palette_data;
}

void NxTexture::SetTexelData( int mip_level, char* data )
{ FUNC_ENTER("NxTexture::SetTexelData"); 
	m_texel_data[mip_level] = new char[ GetTexelDataSize( mip_level ) ];
	memcpy( m_texel_data[mip_level], data, GetTexelDataSize( mip_level ));
}

void NxTexture::SetPaletteData( char* data )
{ FUNC_ENTER("NxTexture::SetPaletteData"); 
	// If we have not allocated space for palette data, assume the caller wants
	// us to actually set our palette data pointer to this new data pointer
	if( m_palette_data == NULL )
	{
		m_palette_data = data;
	}
	else
	{
		memcpy( m_palette_data, data, GetPaletteDataSize());
	}
}

void	NxTexture::SetName( char* name, int level )
{ FUNC_ENTER("NxTexture::SetName"); 
	assert( name );
	strcpy( m_name[level], name );
}

char*	NxTexture::GetName( int level )
{ FUNC_ENTER("NxTexture::GetName"); 
	return m_name[level];
}

int	NxTexture::GetWidth( int mip_level )
{ FUNC_ENTER("NxTexture::GetWidth"); 
	return m_width[mip_level];
}

int NxTexture::GetHeight( int mip_level )
{ FUNC_ENTER("NxTexture::GetHeight"); 
	return m_height[mip_level];
}

void NxTexture::SetWidth( int mip_level, int width )
{ FUNC_ENTER("NxTexture::SetWidth"); 
	m_width[mip_level] = width;
}
	
void NxTexture::SetHeight( int mip_level, int height )
{ FUNC_ENTER("NxTexture::SetHeight"); 
	m_height[mip_level] = height;
}

bool NxTexture::IsPaletted( void )
{ FUNC_ENTER("NxTexture::IsPaletted"); 
	return (( m_pixel_format[0] == v8_BIT ) || ( m_pixel_format[0] == v4_BIT ));
}

void NxTexture::SetBpp( int bpp )
{ FUNC_ENTER("NxTexture::SetBpp"); 
	m_bpp[0] = bpp;
}

int NxTexture::GetBpp( void )
{ FUNC_ENTER("NxTexture::GetBpp"); 
	return m_bpp[0];
}

void NxTexture::SetPaletteBpp( int bpp )
{ FUNC_ENTER("NxTexture::SetPaletteBpp"); 
	m_palette_bpp = bpp;
}

int NxTexture::GetPaletteBpp( void )
{ FUNC_ENTER("NxTexture::GetPaletteBpp"); 
	return m_palette_bpp;
}

FILETIME	NxTexture::GetFileTime( int mip_level )
{ FUNC_ENTER("NxTexture::GetFileTime"); 
	return m_filetime[mip_level];
}

void		NxTexture::SetFileTime( int mip_level, FILETIME time )
{ FUNC_ENTER("NxTexture::SetFileTime"); 
	m_filetime[mip_level] = time;
}

void NxTexture::SetNumPaletteEntries( int entries )
{ FUNC_ENTER("NxTexture::SetNumPaletteEntries"); 
	m_num_palette_entries = entries;
}

int NxTexture::GetNumPaletteEntries( void )
{ FUNC_ENTER("NxTexture::GetNumPaletteEntries"); 
	return m_num_palette_entries;
}

int	NxTexture::GetBytesPerRow( int mip_level )
{ FUNC_ENTER("NxTexture::GetBytesPerRow"); 
	int bytes_per_row;

	bytes_per_row = ( GetWidth( mip_level ) * GetBpp() + 7 ) >> 3;
	return bytes_per_row;
}

int	NxTexture::GetTexelDataSize( int mip_level )
{ FUNC_ENTER("NxTexture::GetTexelDataSize"); 
	return ( GetHeight( mip_level ) * GetBytesPerRow( mip_level ));	
}

int NxTexture::GetPaletteDataSize( void )
{ FUNC_ENTER("NxTexture::GetPaletteDataSize"); 
	int bits_of_palette_data;

	bits_of_palette_data = GetPaletteBpp() * m_num_palette_entries;
	assert(( bits_of_palette_data % 8 ) == 0 );

	return bits_of_palette_data >> 3;
}

void	NxTexture::create_transparent_mip_edges( int trans_flags, unsigned char trans_index )
{ FUNC_ENTER("NxTexture::create_transparent_mip_edges"); 
	int i, j;
	char* dst;
	unsigned char byte;

	assert( GetPaletteFormat() == v32_BIT );
	assert(( GetPixelFormat() == v4_BIT ) || ( GetPixelFormat() == v8_BIT ));

	if( GetPixelFormat() == v4_BIT )
	{			
		for( i = 1; i <= m_mip_levels; i++ )
		{
			if( trans_flags & TOP_TRANSPARENT )
			{		
				byte = trans_index;
				byte |= ( trans_index << 4 );
				dst = m_texel_data[i];	
				for( j = 0; j < (( m_width[i] + 1 ) >> 1 ); j++ )
				{									
					*dst++ = byte;
				}
			}
			if( trans_flags & BOTTOM_TRANSPARENT )
			{
				byte = trans_index;
				byte |= ( trans_index << 4 );
				dst = m_texel_data[i] + ((( m_width[i] * ( m_height[i] - 1 )) + 1 ) >> 1 );	
				for( j = 0; j < (( m_width[i] + 1 ) >> 1 ); j++ )
				{									
					*dst++ = byte;
				}
			}
			if( trans_flags & LEFT_TRANSPARENT )
			{
				dst = m_texel_data[i];	
				for( j = 0; j < m_height[i]; j++ )
				{									
					byte = *dst;
					byte &= 0x0F;
					byte |= ( trans_index << 4 );
					*dst = byte;
					dst += (( m_width[i] + 1 ) >> 1 );
				}			
			}
			if( trans_flags & RIGHT_TRANSPARENT )
			{
				dst = m_texel_data[i] + (( m_width[i] - 1 ) >> 1 );	
				for( j = 0; j < m_height[i]; j++ )
				{									
					byte = *dst;
					byte &= 0xF0;
					byte |= trans_index;
					*dst = byte;
					dst += (( m_width[i] + 1 ) >> 1 );
				}			
			}
		}
	}
	else	// 8-bit
	{
		byte = trans_index;
		for( i = 1; i <= m_mip_levels; i++ )
		{
			if( trans_flags & TOP_TRANSPARENT )
			{
				dst = m_texel_data[i];	
				for( j = 0; j < m_width[i]; j++ )
				{									
					*dst++ = byte;
				}
			}
			if( trans_flags & BOTTOM_TRANSPARENT )
			{
				dst = m_texel_data[i] + ( m_width[i] * ( m_height[i] - 1 ));	
				for( j = 0; j < m_width[i]; j++ )
				{									
					*dst++ = byte;
				}
			}
			if( trans_flags & LEFT_TRANSPARENT )
			{
				dst = m_texel_data[i];	
				for( j = 0; j < m_height[i]; j++ )
				{									
					*dst = byte;
					dst += m_width[i];
				}
			}
			if( trans_flags & RIGHT_TRANSPARENT )
			{
				dst = m_texel_data[i] + ( m_width[i] - 1 );	
				for( j = 0; j < m_height[i]; j++ )
				{									
					*dst = byte;
					dst += m_width[i];
				}
			}
		}
	}
}

int NxTexture::get_transparent_edge_flags( void )
{ FUNC_ENTER("NxTexture::get_transparent_edge_flags"); 
	int trans_flags;
	bool all_trans;
	char* dst;
	int i;

	assert( GetPaletteFormat() == v32_BIT );
	assert(( GetPixelFormat() == v4_BIT ) || ( GetPixelFormat() == v8_BIT ));

	trans_flags = 0;
	if( GetPixelFormat() == v4_BIT )
	{
		dst = m_texel_data[0];	
		all_trans = true;
		// First, test the top row
		for( i = 0; i < (( m_width[0] + 1 ) >> 1 ); i++ )
		{
			unsigned char byte, lo_nibble, hi_nibble;

			byte = *dst;
			lo_nibble = byte & 0x0F;
			hi_nibble = byte >> 4;

			if( m_palette_data[ ( lo_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			if( m_palette_data[ ( hi_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst++;		
		}

		if( all_trans )
		{
			trans_flags |= TOP_TRANSPARENT;
		}

		// Next, test the bottom row
		dst = m_texel_data[0] + ((( m_width[0] * ( m_height[0] - 1 )) + 1 ) >> 1 );	
		all_trans = true;
		for( i = 0; i < (( m_width[0] + 1 ) >> 1 ); i++ )
		{
			unsigned char byte, lo_nibble, hi_nibble;

			byte = *dst;
			lo_nibble = byte & 0x0F;
			hi_nibble = byte >> 4;

			if( m_palette_data[ ( lo_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			if( m_palette_data[ ( hi_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst++;		
		}

		if( all_trans )
		{
			trans_flags |= BOTTOM_TRANSPARENT;
		}

		// Next, test the left column
		dst = m_texel_data[0];
		all_trans = true;
		for( i = 0; i < m_height[0]; i++ )
		{
			unsigned char byte, hi_nibble;

			byte = *dst;
			hi_nibble = byte >> 4;

			if( m_palette_data[ ( hi_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst += (( m_width[0] + 1 ) >> 1 );
		}

		if( all_trans )
		{
			trans_flags |= LEFT_TRANSPARENT;
		}

		// Lastly, test the right column
		dst = m_texel_data[0] + (( m_width[0] - 1 ) >> 1 );
		all_trans = true;
		for( i = 0; i < m_height[0]; i++ )
		{
			unsigned char byte, lo_nibble;

			byte = *dst;
			lo_nibble = byte & 0x0F;

			if( m_palette_data[ ( lo_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst += (( m_width[0] + 1 ) >> 1 );
		}

		if( all_trans )
		{
			trans_flags |= RIGHT_TRANSPARENT;
		}
	}
	else	// 8-bit
	{
		dst = m_texel_data[0];	
		all_trans = true;
		// First, test the top row
		for( i = 0; i < m_width[0]; i++ )
		{
			unsigned char byte;

			byte = *dst;
			
			if( m_palette_data[ ( byte * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst++;		
		}

		if( all_trans )
		{
			trans_flags |= TOP_TRANSPARENT;
		}

		// Next, test the bottom row
		dst = m_texel_data[0] + ( m_width[0] * ( m_height[0] - 1 ));	
		all_trans = true;
		for( i = 0; i < m_width[0]; i++ )
		{
			unsigned char byte;

			byte = *dst;

			if( m_palette_data[ ( byte* 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}			

			dst++;		
		}

		if( all_trans )
		{
			trans_flags |= BOTTOM_TRANSPARENT;
		}

		// Next, test the left column
		dst = m_texel_data[0];
		all_trans = true;
		for( i = 0; i < m_height[0]; i++ )
		{
			unsigned char byte;

			byte = *dst;			

			if( m_palette_data[ ( byte * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst += m_width[0];
		}

		if( all_trans )
		{
			trans_flags |= LEFT_TRANSPARENT;
		}

		// Lastly, test the right column
		dst = m_texel_data[0] + ( m_width[0] - 1 );
		all_trans = true;
		for( i = 0; i < m_height[0]; i++ )
		{
			unsigned char byte;

			byte = *dst;	

			if( m_palette_data[ ( byte * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst += m_width[0];
		}

		if( all_trans )
		{
			trans_flags |= RIGHT_TRANSPARENT;
		}
	}

	return trans_flags;
}

bool NxTexture::auto_generate_8_bit_palette_mipmaps( bool has_alpha )
{ FUNC_ENTER("NxTexture::auto_generate_8_bit_palette_mipmaps"); 
	char* path;
	int i, j, k, width, height;
	int trans_flags;
	unsigned char trans_index;

	path = GetName( 0 );
	if( load_png( path, 0 ) == false )
	{
		char error_msg[128];

		sprintf( error_msg, "Could not load %s", path );
		MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "8-bit PNG Load Error!", MB_OK );
		ReportWarning( error_msg );
		return false;
	}			
	
	assert( GetPixelFormat() == v8_BIT );

	if( has_alpha )
	{
		trans_flags = get_transparent_edge_flags();
		// If one of the edges is transparent, we'll need to fill the corresponding edges on mips
		// so we'll need a transparent palette entry
		if( trans_flags != 0 )
		{
			for( i = 0; i < m_num_palette_entries; i++ )
			{
				if( m_palette_data[ (i * 4) + 3 ] == 0 )
				{
					trans_index = i;
					break;
				}
			}
		}
	}

	width = m_width[0];
	height = m_height[0];
	for( i = 1; i <= m_mip_levels; i++ )
	{	
		int tex_data_size, dst_idx, src_idx, src_tex_x, src_tex_y;
		char* src_texel;
		float tex_x, tex_y, ratio_x, ratio_y;

		width >>= 1;
		height >>= 1;

		if(	( width < vMIN_MIP_WIDTH ) &&
			( height < vMIN_MIP_HEIGHT ))
		{
			m_mip_levels = i - 1;
			break;
		}
		if( width < vMIN_MIP_WIDTH )
		{
			width = vMIN_MIP_WIDTH;
		}
		if( height < vMIN_MIP_HEIGHT )
		{
			height = vMIN_MIP_HEIGHT;
		}

		m_width[i] = width;
		m_height[i] = height;

		tex_data_size = GetTexelDataSize( i );
		m_texel_data[i] = new char[tex_data_size];

		src_texel = GetTexelData( 0 );
		tex_y = 0.5f;
		dst_idx = 0;
		for( j = 0; j < height; j++ )
		{
			tex_x = 0.5f;
			for( k = 0; k < width; k++ )
			{				
				ratio_x = (float) tex_x / width;
				ratio_y = (float) tex_y / height;
				
				src_tex_x = (int) ( m_width[0] * ratio_x );
				src_tex_y = (int) ( m_height[0] * ratio_y );

				src_idx = ( src_tex_y * m_width[0] ) + src_tex_x;
					 
				m_texel_data[i][dst_idx] = src_texel[src_idx];

				tex_x += 1.0f;
				dst_idx++;
			}
			tex_y += 1.0f;
		}
	}

	if( has_alpha && ( trans_flags != 0 ))
	{
		create_transparent_mip_edges( trans_flags, trans_index );
	}

	//Dump();
	return true;
}

bool NxTexture::auto_generate_4_bit_palette_mipmaps( bool has_alpha )
{ FUNC_ENTER("NxTexture::auto_generate_4_bit_palette_mipmaps"); 
	int size;
	char* dst;
	char* path;
	int i, j, k, width, height;
	int trans_flags;	
	unsigned char trans_index;

	path = GetName( 0 );
	if( load_png( path, 0 ) == false )
	{
		char error_msg[128];

		sprintf( error_msg, "Could not load %s", path );
		MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "4-bit PNG Load Error!", MB_OK );
		ReportWarning( error_msg );
		return false;
	}			
	
	assert( GetPixelFormat() == v4_BIT );

	// Earlier, we swapped the nibbles to be in the order that the target platforms wanted them.
	// But here, before we down-sample, temporarily revert the nibble ordering
	size = GetTexelDataSize( 0 );
	dst = m_texel_data[0];
	for( i = 0; i < size; i++ )
	{
		unsigned char byte, lo_nibble;
		
		byte = *dst;
		lo_nibble = byte & 0x0F;
		byte >>= 4;
		byte |= ( lo_nibble << 4 );
		*dst++ = byte;
	}

	if( has_alpha )
	{
		// Figure out if any of the four edges of this texture are completely transparent. If so, 
		// the mips need to be transparent on those edges as well just in case the artist wants to
		// use clamping on this texture
		trans_flags = get_transparent_edge_flags();
		// If one of the edges is transparent, we'll need to fill the corresponding edges on mips
		// so we'll need a transparent palette entry
		if( trans_flags != 0 )
		{
			for( i = 0; i < m_num_palette_entries; i++ )
			{
				if( m_palette_data[ (i * 4) + 3 ] == 0 )
				{
					trans_index = i;
					break;
				}
			}
		}
	}

	width = m_width[0];
	height = m_height[0];
	for( i = 1; i <= m_mip_levels; i++ )
	{	
		int tex_data_size, dst_idx, src_idx, src_tex_x, src_tex_y, src_byte, dst_byte;
		char* src_texel;
		unsigned char src_color;
		float tex_x, tex_y, ratio_x, ratio_y;

		width >>= 1;
		height >>= 1;

		if(	( width < vMIN_MIP_WIDTH ) &&
			( height < vMIN_MIP_HEIGHT ))
		{
			m_mip_levels = i - 1;
			break;
		}			
		if( width < vMIN_MIP_WIDTH )
		{
			width = vMIN_MIP_WIDTH;
		}
		if( height < vMIN_MIP_HEIGHT )
		{
			height = vMIN_MIP_HEIGHT;
		}
		
		m_width[i] = width;
		m_height[i] = height;

		tex_data_size = GetTexelDataSize( i );
		m_texel_data[i] = new char[tex_data_size];

		// Clear out the texel data
		for( j = 0; j < tex_data_size; j++ )
		{
			m_texel_data[i][j] = 0;
		}

		src_texel = GetTexelData( 0 );
		tex_y = 0.5f;
		dst_idx = 0;
		for( j = 0; j < height; j++ )
		{
			tex_x = 0.5f;
			for( k = 0; k < width; k++ )
			{	
				ratio_x = (float) tex_x / width;
				ratio_y = (float) tex_y / height;
				
				src_tex_x = (int) ( m_width[0] * ratio_x );
				src_tex_y = (int) ( m_height[0] * ratio_y );

				src_idx = ( src_tex_y * m_width[0] ) + src_tex_x;

				src_byte = src_idx >> 1;	// 4 bits per texel, so byte index is halved
				dst_byte = dst_idx >> 1;	// 4 bits per texel, so byte index is halved
					 
				// Decide which nibble to access
				if( src_idx % 2 )
				{
					src_color = src_texel[src_byte];
					src_color &= 0xF;
				}
				else
				{
					src_color = src_texel[src_byte];
					src_color >>= 4;					
				}

				// Decide which nibble to write to
				if( dst_idx % 2 )
				{
					m_texel_data[i][dst_byte] |= src_color;
				}
				else
				{
					m_texel_data[i][dst_byte] |= ( src_color << 4 );					
				}				

				tex_x += 1.0f;
				dst_idx++;
			}
			tex_y += 1.0f;
		}
	}
	
	if( has_alpha && ( trans_flags != 0 ))
	{
		create_transparent_mip_edges( trans_flags, trans_index );
	}

	// Now swap nibble order so that they are in the order that the targets expect them
	for( i = 0; i <= m_mip_levels; i++ )
	{
		size = GetTexelDataSize( i );
		dst = m_texel_data[i];
	
		for( j = 0; j < size; j++ )
		{
			unsigned char byte, lo_nibble;
			
			byte = *dst;
			lo_nibble = byte & 0x0F;
			byte >>= 4;
			byte |= ( lo_nibble << 4 );
			*dst++ = byte;
		}
	}

	//Dump();
	return true;
}

int FindNumMipFiles( char* file )
{ FUNC_ENTER("FindNumMipFiles"); 
	char buf[256];
	int slen = strlen(file);
	strcpy(buf,file);

	buf[slen-5] = '*';

	_finddata_t fdata;

	long fhandle = _findfirst(buf,&fdata);

	if (!fhandle)
		return 0;

	int nfiles = 1;

	while ( _findnext(fhandle,&fdata) != -1 )
		nfiles++;
	
	_findclose(fhandle);

	return nfiles;
}

int CalcNumMips(int width, int height, int minWidth = 1, int minHeight = 1)
{ FUNC_ENTER("CalcNumMips"); 
	int curWidth = width;
	int curHeight = height;
	int nLevels = 0;

	while( curWidth > minWidth ||
		   curHeight > minHeight)
	{
		if (curWidth > minWidth)
			curWidth >>= 1;

		if (curHeight > minHeight)
			curHeight >>= 1;

		nLevels++;
	}

	return nLevels;
}

bool AllMipsExist(char* filename, char* outPath, int nMips)
{ FUNC_ENTER("AllMipsExist"); 
   char drive[_MAX_DRIVE];
   char dir[_MAX_DIR];
   char fname[_MAX_FNAME];
   char ext[_MAX_EXT];
   char checkfile[512];
   char mipID[10];

	_splitpath(filename,drive,dir,fname,ext);

	for(int i = 0; i < nMips; i++)
	{
		strcpy(checkfile,outPath);
		strcat(checkfile,"\\");
		strcat(checkfile,fname);
		strcat(checkfile,"_auto32m");

		_itoa(i,mipID,10);
		strcat(checkfile,mipID);
		strcat(checkfile,".png");

		if (!FileExists(checkfile))
			return false;
	}

	return true;
}

bool NxTexture::SaveImage( char* filename )
{ FUNC_ENTER("NxTexture::SaveImage"); 
	/*
	FILE* fp = fopen(Filename, "wb");

	if (!fp)
		return false;

	// Initialize the LibPng library
	// (Use default error handling callbacks)
	png_structp png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,
		                                          NULL,NULL,NULL);

	png_infop info_ptr = png_create_info_struct(png_ptr);

	if (!info_ptr)
	{
		png_destroy_read_struct(&png_ptr,
			                    (png_infopp)NULL,
								(png_infopp)NULL);

		fclose(fp);
		return false;
	}

	png_infop end_info = png_create_info_struct(png_ptr);

	if (!end_info)
	{
		png_destroy_read_struct(&png_ptr,
			                    &info_ptr,
								(png_infopp)NULL);

		return false;
	}

	png_init_io(png_ptr,fp);

	// Color type modes
	//
	// PNG_COLOR_TYPE_GRAY
	// PNG_COLOR_TYPE_PALETTE
	// PNG_COLOR_TYPE_RGB
	// PNG_COLOR_TYPE_RGB_ALPHA
	// PNG_COLOR_TYPE_GRAY_ALPHA

	png_set_IHDR(png_ptr,info_ptr,
	             width,height,
				 bit_depth,
				 color_type,
				 interlace_type,
				 compression_type,
				 filter_type);

	//png_write_png(png_ptr,info_ptr,PNG_TRANSFORM_IDENTITY,NULL);

	if (nPalEntries > 0)
		png_set_PLTE(png_ptr,info_ptr,palette,nPalEntries);

	if (nPals > 0)
		png_set_sPLT(png_ptr,info_ptr,pals,nPals);

	// Build up row pointers
	png_bytep* pRows = (png_bytep*)malloc(sizeof(png_bytep)*height);
	unsigned char* pPos = (unsigned char*)buf;

	for(unsigned int i=0;i<height;i++)
	{
		pRows[i] = pPos;
		pPos += (int)((float)width*(((float)nChannels*(float)bit_depth)/8.0f));
	}

	// Set tRNS data
	if (btRNStransValid)
		png_set_tRNS(png_ptr,info_ptr,tRNSpal,tRNSentries,&tRNStrans_values);
	else
		if (tRNSentries>0)
			png_set_tRNS(png_ptr,info_ptr,tRNSpal,tRNSentries,NULL);

	/*
	// Copy the scanline data in reverse order to flip the image
	unsigned int halfsize = height/2;
	
	for(i=0;i<halfsize;i++)
	{
		png_bytep ptemp = pRows[height-i-1];
		pRows[height-i-1] = pRows[i];
		pRows[i] = ptemp;
	}
	/

	png_write_info(png_ptr,info_ptr);

	//png_write_png(png_ptr,info_ptr,PNG_TRANSFORM_IDENTITY,NULL);

	png_write_image(png_ptr,pRows);

	png_write_end(png_ptr,info_ptr);
	png_destroy_write_struct(&png_ptr,&info_ptr);

	free(pRows);
	*/
	return true;
}

// Merges together 2 alchemy palette file so we can prevent alchemy from remapping the
// files to a different bit depth if it felt the optimum palette shouldn't use all the entries
// Temporary till MipGen integrated for PS2 mips
void MergePalFiles(char* destFile, char* fileA, char* fileB)
{ FUNC_ENTER("MergePalFiles"); 
	struct PalEntry
	{
		unsigned char r,g,b;
	};
	
	// Load first palette file
	FILE* fp = fopen(fileA,"r");

	if (!fp)
		return;

	int  nColors1, nColors2;
	int  i;

	fscanf(fp,"PAL\n%i\n",&nColors1);
	PalEntry* fileAPal = new PalEntry[nColors1];

	for(i = 0; i < nColors1; i++)
	{
		fscanf(fp,"%i %i %i\n", &fileAPal[i].r, &fileAPal[i].g, &fileAPal[i].b);
	}

	fclose(fp);

	// Load second palette file
	fp = fopen(fileB,"r");

	if (!fp)
		return;

	fscanf(fp,"PAL\n%i\n",&nColors2);
	PalEntry* fileBPal = new PalEntry[nColors2];

	for(i = 0; i < nColors2; i++)
	{
		fscanf(fp,"%i %i %i\n", &fileBPal[i].r, &fileBPal[i].g, &fileBPal[i].b);
	}

	fclose(fp);

	// Now generate the combined palette file
	fp = fopen(destFile,"w");

	if (!fp)
		return;

	int maxColors = (nColors1 > nColors2) ? nColors1 : nColors2;

	fprintf(fp,"PAL\n%i\n",maxColors);
	int curEntry = 0;

	for(i = 0; i < maxColors; i++)
	{
		// All entries from the optimal palette and any left over from the original
		if (i < nColors1)
			fprintf(fp,"%i %i %i\n", fileAPal[i].r, fileAPal[i].g, fileAPal[i].b);
		else
		{
			fprintf(fp,"%i %i %i\n", fileBPal[curEntry].r, fileBPal[curEntry].g, fileBPal[curEntry].b);
			curEntry++;
		}
	}

	fclose(fp);

	delete [] fileAPal;
	delete [] fileBPal;
}

bool NxTexture::LoadImage( bool bBuildPostGenMips )
{ FUNC_ENTER("NxTexture::LoadImageA"); 
	char ext[_MAX_EXT];
	int i;
	char* path;


	char palette_path[_MAX_PATH];
	char ult_mip_path[_MAX_PATH];
	char temp_mip_path[_MAX_PATH];
	char mip_path[_MAX_PATH];
	char drive[_MAX_DRIVE];
	char cur_path[_MAX_PATH];
	char cur_file[_MAX_FNAME];			
	int width, height;
	NxTexture tmp_tex;
	bool should_regen;


	path = GetName( 0 );

///////////////////	
	if (bBuildPostGenMips)
	{
		// Determine the base filename
		char ult_mip_path[_MAX_PATH];
		char mip_path[_MAX_PATH];
		char drive[_MAX_DRIVE];
		char cur_path[_MAX_PATH];
		char cur_file[_MAX_FNAME];			
		char ext[_MAX_EXT];

		_splitpath( path, drive, cur_path, cur_file, ext );
		sprintf( ult_mip_path, "%s%smips", drive, cur_path );
		CreateDirectory( ult_mip_path, NULL );
		
		if( ShouldAutoGenerateMipMaps() )
		{
			//sprintf( mip_path, "c:/temp/%s_autom0%s", cur_file, ext );
			sprintf( mip_path, "%s\\%s_autom0%s", ult_mip_path, cur_file, ext );

			// Copy the highest level of detail to the mips directory
			if( FileIsNewer( mip_path, path ))
			{
				CopyFile( path, mip_path, false );
			}			
		}

		//if( ShouldAutoGenerateMipMaps())
		if( ShouldAutoGenerate32BitMips() )
		{
			//sprintf( mip_path, "c:/temp/%s_autom0%s", cur_file, ext );
			sprintf( mip_path, "%s\\%s_autom0%s", ult_mip_path, cur_file, ext );

			// Copy the highest level of detail to the mips directory
			/*
			if( FileIsNewer( mip_path, path ))
			{
				CopyFile( path, mip_path, false );
			}
			*/

			int numLevels = 0;

			// Perform export if mip 0 doesn't exist or source file time is
			// greater than mip 0 file time

			char* project_root = getenv( APP_ENV );
			if( project_root == NULL )
			{
				char strWarnBuf[256];
				sprintf(strWarnBuf,"You must set up your %s environment variable", APP_ENV);
				MessageBox( gInterface->GetMAXHWnd(), strWarnBuf,
								"Error!", MB_OK|MB_TASKMODAL);
				return false;
			}

			char bufPath[512];
			char bufOutPath[512] = "-op";
			char bufOutFile[512];
			strcat(bufOutPath,ult_mip_path);
			strcpy(bufOutFile,ult_mip_path);
			strcat(bufOutFile,"\\");
			strcat(bufOutFile,cur_file);
			strcat(bufOutFile,"_auto32m0.png");

			// Ensure no mips are set for read-only access
			ClearDirReadOnlyAttribs(ult_mip_path);

			sprintf( bufPath, "%s\\bin\\win32\\mipgen.exe", project_root );

			// Acquire the last write times for mipgen and the PNG file so we can determine
			// if the mip should be regenerated, if mipgen is newer it should always regenerate
			width = GetWidth( 0 );
			height = GetHeight( 0 );
			
			int nCalcLevels = CalcNumMips(width,height);
			
			if ( !AllMipsExist( path, bufOutPath + 3, nCalcLevels ) ||
				//!FileExists(bufOutFile) ||
				  FileIsNewer( bufOutFile, bufPath ) ||
				  FileIsNewer( bufOutFile, path )
				  )
			{
				char* args[10];
				char  bufTransColor[256];
				int   result;

				args[0] = bufPath;
				args[1] = path;			// Original png file to process
				args[2] = "-m";			// generate mip map levels
				//args[3] = "-b32";		// resample to 32-bit rgba
				args[3] = "-bo";		// resample to 24 or 32-bit based on alpha
				args[4] = "-mt1x1";		// build mip levels down to 1 pixel width and height
				args[5] = bufOutPath;	// Dump mips to output directory
				args[6] = "-dm";		// Delete old mips

				// Support fully transparent color if so checked
				if (ShouldChangeTransparentColor())
				{
					Color color = GetTransparentColor();
					sprintf(bufTransColor,"-tc%i,%i,%i",(int)(color.r * 255.0f),(int)(color.g * 255.0f),(int)(color.b * 255.0f));
					args[7] = bufTransColor;
					args[8] = NULL;
				}
				else
				{
					args[7] = NULL;
				}

				//result = ( spawnv( _P_WAIT, args[0], args ) == 0 );
				result = ExecuteCommand( args );

				OutputDebugString(path);
				OutputDebugString("\n");
			}

		}
		else
		{
			char* project_root = getenv( APP_ENV );
			if( project_root == NULL )
			{
				char strWarnBuf[256];
				sprintf(strWarnBuf,"You must set up your %s environment variable", APP_ENV);
				MessageBox( gInterface->GetMAXHWnd(), strWarnBuf,
								"Error!", MB_OK|MB_TASKMODAL);
				return false;
			}

			char bufPath[512];
			char bufOutPath[512] = "-op";
			char bufOutFile[512];
			strcat(bufOutPath,ult_mip_path);
			strcpy(bufOutFile,ult_mip_path);
			strcat(bufOutFile,"\\");
			strcat(bufOutFile,cur_file);
			strcat(bufOutFile,"_auto32m0.png");

			// Ensure no mips are set for read-only access
			ClearDirReadOnlyAttribs(ult_mip_path);

			sprintf( bufPath, "%s\\bin\\win32\\mipgen.exe", project_root );

			if ( !FileExists(bufOutFile) ||
				  FileIsNewer( bufOutFile, bufPath ) ||
				  FileIsNewer( bufOutFile, path )
				  )
			{		
				// This map shouldn't have mips auto-generated, however, we'll need to have the base with auto32_m0
				// so it can be found on pass 2.  Call mipgen with mip limit 0
				char* args[11];
				char  bufTransColor[256];
				int   result;

				args[0] = bufPath;
				args[1] = path;			// Original png file to process
				args[2] = "-m";			// generate mip map levels
				//args[3] = "-b32";		// resample to 32-bit rgba
				args[3] = "-bo";		// resample to 24 or 32-bit based on alpha
				args[4] = "-mt1x1";		// build mip levels down to 1 pixel width and height
				args[5] = bufOutPath;	// Dump mips to output directory
				args[6] = "-dm";		// Delete old mips
				args[7] = "-lm0";		// Limit to only base texture

				// Support fully transparent color if so checked
				if (ShouldChangeTransparentColor())
				{
					Color color = GetTransparentColor();
					sprintf(bufTransColor,"-tc%i,%i,%i",(int)(color.r * 255.0f),(int)(color.g * 255.0f),(int)(color.b * 255.0f));
					args[8] = bufTransColor;
					args[9] = NULL;
				}
				else
				{
					args[8] = NULL;
				}

				//result = ( spawnv( _P_WAIT, args[0], args ) == 0 );
				result = ExecuteCommand( args );

				OutputDebugString(path);
				OutputDebugString("\n");
			}
		}
	}
////////

	if( FileExists( path ) == false )
	{
		char error_msg[128];
			
		sprintf( error_msg, "Could not find '%s'", path );
		MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "LoadImage Error!", MB_OK );
		ReportWarning( error_msg );
		return false;
	}

	_splitpath( path, NULL, NULL, NULL, ext );
	if( stricmp( ext, ".png" ) == 0 )
	{
		// If the texture is post generated it can have additional mips that we should load
		// we'll do this by following the convention of 
		if( m_flags & mPOST_GENERATED_TEXTURE)
		{
			// Load the first image
			if ( load_png( path, 0 ) == false )
			{
				char error_msg[128];
			
				sprintf( error_msg, "Could not load %s", path );
				MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "LoadImage Error!", MB_OK );
				ReportWarning( error_msg );
				return false;
			}

			// Now compute the number of mip levels to use
			//m_mip_levels = FindNumMipFiles( path ) - 1;	// Compute instead of file usage
			width = GetWidth( 0 );
			height = GetHeight( 0 );
			
			m_mip_levels = CalcNumMips(width, height, 1, 1);
			m_num_palette_entries = 0;

			if (m_mip_levels > 10)
				m_mip_levels = 10;

			m_draw_order = 0;

			for(int i = 1; i < m_mip_levels + 1; i++)
			{
				// Determine the base filename
				char bufDrive[_MAX_DRIVE];
				char bufPath[_MAX_DIR];
				char bufName[_MAX_FNAME];
				char bufExt[_MAX_EXT];

				_splitpath(path,bufDrive,bufPath,bufName,bufExt);
				// Remove '0' from name
				int snamelen = strlen(bufName);
				bufName[snamelen - 1] = 0;

				// Load in the mip
				char bufLevel[20];
				sprintf(bufLevel,"%i",i);

				CStr filename = CStr(bufDrive) + CStr(bufPath) + bufName + CStr(bufLevel) + bufExt;

				if ( load_png( filename, i ) == false )
				{
					char error_msg[128];
				
					sprintf( error_msg, "Could not load %s", path );
					MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "LoadImage Error!", MB_OK );
					ReportWarning( error_msg );
					return false;
				}
			}
		}
		else
		if( m_mip_levels == 0 )
		{
			if( load_png( path, 0 ) == false )
			{
				char error_msg[128];
			
				sprintf( error_msg, "Could not load %s", path );
				MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "LoadImage Error!", MB_OK );
				ReportWarning( error_msg );
				return false;
			}				
		}
		else
		{
		
			if( !tmp_tex.load_png( path, 0 ))
			{
				return false;
			}

			// First, we need to load the image to see if it's paletted and has alpha. We handle
			// these differently
			if( ( ShouldAutoGenerateMipMaps()) &&
				( tmp_tex.IsPaletted()))								
			{				
				if(( tmp_tex.GetPaletteFormat() == v32_BIT ) || tmp_tex.IsGrayscale())
				{
					if( tmp_tex.GetPixelFormat() == v8_BIT )
					{			
						return auto_generate_8_bit_palette_mipmaps( tmp_tex.GetPaletteFormat() == v32_BIT );						
					}
					else if( tmp_tex.GetPixelFormat() == v4_BIT )
					{
						return auto_generate_4_bit_palette_mipmaps( tmp_tex.GetPaletteFormat() == v32_BIT );						
					}
				}
			}			
////////

			width = tmp_tex.m_width[0];		// aml
			height = tmp_tex.m_height[0];	// aml

			switch(tmp_tex.GetPixelFormat())
			{
			case v4_BIT:
				m_bpp[0] = 4;
				break;

			case v8_BIT:
				m_bpp[0] = 8;
				break;

			case v16_BIT:
				m_bpp[0] = 16;
				break;

			case v24_BIT:
				m_bpp[0] = 24;
				break;

			case v32_BIT:
				m_bpp[0] = 32;
				break;
			}

			m_width[0]  = width;
			m_height[0] = height;

			should_regen = false;
			for( i = 1; i <= m_mip_levels; i++ )
			{	
				width >>= 1;
				height >>= 1;

				if(	( width < vMIN_MIP_WIDTH ) &&
					( height < vMIN_MIP_HEIGHT ))
				{
					m_mip_levels = i - 1;
					break;
				}			

				if( width < vMIN_MIP_WIDTH )
				{
					width = vMIN_MIP_WIDTH;
				}
				if( height < vMIN_MIP_HEIGHT )
				{
					height = vMIN_MIP_HEIGHT;
				}

				switch(tmp_tex.GetPixelFormat())
				{
				case v4_BIT:
					m_bpp[i] = 4;
					break;

				case v8_BIT:
					m_bpp[i] = 8;
					break;

				case v16_BIT:
					m_bpp[i] = 16;
					break;

				case v24_BIT:
					m_bpp[i] = 24;
					break;

				case v32_BIT:
					m_bpp[i] = 32;
					break;
				}

				// First generate the mips, if they need to be generated256
				if( ShouldAutoGenerateMipMaps())
				{			
					_splitpath( path, drive, cur_path, cur_file, ext );

					sprintf( ult_mip_path, "%s%smips\\%s_autom%d%s", drive, cur_path, cur_file, i, ext );
					sprintf( temp_mip_path, "c:/temp/%s_autom%d%s", cur_file, i, ext );

					// Only generate it if it's actually newer
					if( FileIsNewer( ult_mip_path, path ))
					{
						int result;
						char* args[12];
						char width_str[32], height_str[32];
						char alch_path[_MAX_PATH];
						char* project_root;

						should_regen = true;
						project_root = getenv( APP_ENV );
						if( project_root == NULL )
						{
							char strWarnBuf[256];
							sprintf(strWarnBuf,"You must set up your %s environment variable", APP_ENV);
							MessageBox( gInterface->GetMAXHWnd(), strWarnBuf,
											"Error!", MB_OK|MB_TASKMODAL);
							return false;
						}

						sprintf( alch_path, "%s\\bin\\win32\\alchlong.exe", project_root );
						sprintf( width_str, "%d", width );
						sprintf( height_str, "%d", height );
						args[0] = alch_path;
						args[1] = "---n";
						args[2] = "-o";
						args[3] = "-d4";	// Stevenson & Arce dithering
						args[4] = "-Xd";	// Lanczos filtering when scaling X
						args[5] = width_str;
						args[6] = "-Yd";	// Lanczos filtering when scaling Y
						args[7] = height_str;
						args[8] = path;
						args[9] = ult_mip_path;
						//args[9] = temp_mip_path;
						args[10] = "---f";
						args[11] = NULL;

						//result = ( spawnv( _P_WAIT, args[0], args ) == 0 );				
						result = ExecuteCommand( args );

						// >> If we're building XBox or GCN textures we should do it now with MipGen

					}
				}				
			}

			if( ShouldAutoGenerateMipMaps())
			{
				sprintf( palette_path, "%s%smips\\auto%s.pal", drive, cur_path, cur_file );			
				// Only generate palette and re-palettize if palette is out of date
				//if( FileIsNewer( palette_path, path ))
				if( should_regen )
				{
					int result;
					char* pal_args[10];
					char pal_file[_MAX_PATH];

					char* first_args[8];
					char* second_args[12];
					char mip_wildcard[_MAX_PATH];
					char dest_dir[_MAX_PATH];
					char alch_path[_MAX_PATH];
					char* project_root;

					project_root = getenv( APP_ENV );
					if( project_root == NULL )
					{
						char strWarnBuf[256];
						sprintf(strWarnBuf,"You must set up your %s environment variable", APP_ENV);
						MessageBox( gInterface->GetMAXHWnd(), strWarnBuf,
										"Error!", MB_OK|MB_TASKMODAL);
						return false;
					}

					sprintf( alch_path, "%s\\bin\\win32\\alchlong.exe", project_root );

					// Create palette file for the basetexture
					sprintf( pal_file, "%s%smips\\autoBase%s.pal", drive, cur_path, cur_file );
						
					pal_args[0] = alch_path;
					pal_args[1] = "-o";
					pal_args[2] = path;
					pal_args[3] = "-l";
					pal_args[4] = pal_file;
					pal_args[5] = NULL;

					//result = ( spawnvp( _P_WAIT, alch_path, pal_args ) == 0 );
					result = ExecuteCommand( pal_args );

					// Create a generic palette for all the mips
					sprintf( mip_wildcard, "c:\\temp\\%s_autom*%s", cur_file, ext );
					first_args[0] = alch_path;
					first_args[1] = "-o";
					first_args[2] = "--";
					first_args[3] = mip_wildcard;				
					first_args[4] = "-L";
					first_args[5] = palette_path;				
					first_args[6] = NULL;

					//result = ( spawnvp( _P_WAIT, first_args[0], first_args ) == 0 );				
					result = ExecuteCommand( first_args );

					// Merge the palette files together so we don't change bit depth from original
					char remapPalName[256];
					
					sprintf(remapPalName, "%s%smips\\auto%s_remap.pal", drive, cur_path, cur_file );	
					MergePalFiles(remapPalName,palette_path,pal_file);

					// Now apply the created palette
					sprintf( dest_dir, "%s%smips\\", drive, cur_path );
					second_args[0] = alch_path;
					second_args[1] = "-o";
					second_args[2] = "---n";
					//second_args[3] = "-8";
					second_args[3] = "--";
					second_args[4] = mip_wildcard;
					second_args[5] = "-f";

					// Apply the appropriate palette file
					//second_args[7] = palette_path;
					second_args[6] = remapPalName;

					for(int i=0;i<m_mip_levels;i++)
					{
						if (UsesBasePal(i))
						{
							second_args[6] = pal_file;
							break;
						}
					}

					// Don't remap palette unless remapping to base palette
					if (strcmp(second_args[6],pal_file) == 0)
					{
						second_args[7] = dest_dir;
						second_args[8] = NULL;

						//result = ( spawnvp( _P_WAIT, second_args[0], second_args ) == 0 );
						result = ExecuteCommand( second_args );
					}
				}

				// Now load each mip level
				for( i = 0; i <= m_mip_levels; i++ )
				{
					sprintf( mip_path, "%s%smips\\%s_autom%d%s", drive, cur_path, cur_file, i, ext );
					if( load_png( mip_path, i ) == false )
					{
						return false;
					}			
				
					// If any of the images we load are grayscale, abort the alchemy path. Auto-generate
					// the mips ourselves. Alchemy does a terrible job on grayscale images
					if( IsGrayscale())
					{
						if( tmp_tex.GetPixelFormat() == v8_BIT )
						{			
							return auto_generate_8_bit_palette_mipmaps( tmp_tex.GetPaletteFormat() == v32_BIT );						
						}
						else if( tmp_tex.GetPixelFormat() == v4_BIT )
						{
							return auto_generate_4_bit_palette_mipmaps( tmp_tex.GetPaletteFormat() == v32_BIT );						
						}					
					}

					if( i > 0 )
					{
						int expwidth  = m_width[i-1] >> 1;
						int expheight = m_height[i-1] >> 1;

						if (expwidth<vMIN_MIP_WIDTH)
							expwidth=vMIN_MIP_WIDTH;

						if (expheight<vMIN_MIP_HEIGHT)
							expheight=vMIN_MIP_HEIGHT;

						if( ( m_width[i] != expwidth) ||
							( m_height[i] != expheight))
						{
							char error_msg[128];

							sprintf( error_msg, "Mipmap %s is not half-size of its parent in each dimension!\nExpected Width: %i\nExpected Height: %i\nActual Width: %i\nActual Height: %i\nMip Level: %i\nMip Path: %s", 
								path, 
								m_width[i - 1] >> 1,
								m_height[i - 1] >> 1,
								m_width[i],
								m_height[i],
								i,
								mip_path
							);

							MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "MipMap Error", MB_OK|MB_ICONWARNING );							
							ReportWarning( error_msg );
						}
					}
				}
			}			
			else
			{
				//////////// PROBLEM POINT
				NxTexture	mips[vMAX_MIP_LEVELS];
				NxTexture*	all_levels[vMAX_MIP_LEVELS];
				int prev_width, prev_height;

				if( load_png( GetName(0), 0 ) == false )
				{
					char error_msg[128];
		
					sprintf( error_msg, "Could not load %s", GetName(i));
					MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "Mip Load Error!", MB_OK );
					ReportWarning( error_msg );
					return false;
				}

				prev_width = m_width[0];
				prev_height = m_height[0];
				all_levels[0] = this;
				// Now load each mip level
				for( i = 1; i <= m_mip_levels; i++ )
				{
					if( mips[i-1].load_png( GetName(i), 0 ) == false )
					{
						char error_msg[128];
			
						sprintf( error_msg, "Could not load %s", GetName(i));
						MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "Mip Load Error!", MB_OK );
						ReportWarning( error_msg );
						return false;
					}

					if( mips[i-1].GetBpp() != GetBpp())
					{
						char error_msg[128];
			
						sprintf( error_msg, "Mipmap %s is not in the same format as its basemap : %s", GetName(i), GetName( 0 ));
						MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "Mip Format Error!", MB_OK );
						ReportWarning( error_msg );
						return false;
					}

					all_levels[i] = &mips[i-1];
					if( ( mips[i-1].m_width[0] != ( prev_width >> 1 )) ||
						( mips[i-1].m_height[0] != ( prev_height >> 1 )))
					{
						char error_msg[128];

						sprintf( error_msg, "Mipmap %s is not half-size of its parent in each dimension!\nExpected Width: %i\nExpected Height: %i\nActual Width: %i\nActual Height: %i\nMip Level: %i\nMip Path: %s\nMip Name: %s", 
							path, 
							prev_width >> 1,
							prev_height >> 1,
							mips[i-1].m_width[0],
							mips[i-1].m_height[0],
							i,
							mip_path,
							(char*)GetName(i)
							);

						MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "Half-size MipMap Error", MB_OK|MB_ICONWARNING );							
						ReportWarning( error_msg );
						return false;
					}										

					prev_width = mips[i-1].m_width[0];
					prev_height = mips[i-1].m_height[0];
				}

				if( NxPalettizeMips( all_levels, m_mip_levels + 1 ) == false )
				{
					char error_msg[128];

					sprintf( error_msg, "Internal palette quantizer error on %s. Notify Steve\n", path );
					MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "Quantizer MipMap Error", MB_OK|MB_ICONWARNING );							
					ReportWarning( error_msg );
					return false;
				}
			}
		}
		return true;
	}
	else
	{
		char error_msg[256];
		sprintf(error_msg,"The file format is unsupported for '%s'.  You must use a valid .PNG format file.",path);

		MessageBoxAll(gInterface->GetMAXHWnd(), error_msg, "Bitmap format unrecognized",MB_ICONWARNING|MB_OK);
		ReportWarning( error_msg );
	}

	return false;
}

BitmapInfo*	NxTexture::GetMap( void )
{ FUNC_ENTER("NxTexture::GetMap"); 
	return m_map;
}

void	NxTexture::SetMap( BitmapInfo* map )
{ FUNC_ENTER("NxTexture::SetMap"); 
	m_map = map;
}

bool	NxTexture::ShouldCompressComponents( void )
{ FUNC_ENTER("NxTexture::ShouldCompressComponents"); 
	return (( m_flags & mFORCE_BYTE_PER_COMPONENT ) == 0 );
}

bool	NxTexture::ShouldChangeTransparentColor( void )
{ FUNC_ENTER("NxTexture::ShouldChangeTransparentColor"); 
	return (( m_flags & mCHANGE_FULLY_TRANSPARENT_COLOR ) != 0 );
}

Color	NxTexture::GetTransparentColor( void )
{ FUNC_ENTER("NxTexture::GetTransparentColor"); 
	return m_transparent_color;
}

void	NxTexture::SetTransparentColor( Color trans_color )
{ FUNC_ENTER("NxTexture::SetTransparentColor"); 
	m_transparent_color = trans_color;
}

void	NxTexture::SetFlags( int flags )
{ FUNC_ENTER("NxTexture::SetFlags"); 
	m_flags |= flags;
}

int		NxTexture::GetFlags( void )
{ FUNC_ENTER("NxTexture::GetFlags"); 
	return m_flags;
}

#define FILENAME_ONLY	// Garrett: removes the directory path before the checksum calculation

void	NxTexture::GenerateChecksum( void )
{ FUNC_ENTER("NxTexture::GenerateChecksum"); 
	m_checksum = 0;
	if( m_map == NULL )
	{		
		return;
	}

#ifdef FILENAME_ONLY
	// Find base name
	const char *pFileName = GetName(0);
	int idx	= strlen(pFileName);
	while ((idx > 0) && pFileName[idx - 1] != '\\' && pFileName[idx - 1] != '/')
		--idx;

	const char *p_base_name = &(pFileName[idx]);
	m_checksum = GenerateCRC(p_base_name);
#else
	m_checksum = GenerateCRC( GetName( 0 ));		
#endif

	// Here, alter the checksum according to the platform for which this texture is meant. One texture may have
	// different mipmaps per platform
	if( m_platform_flags & mPLAT_PS2 )
	{
		m_checksum *= 2;
	}
	if( m_platform_flags & mPLAT_NGC )
	{
		m_checksum *= 3;
	}
	if( m_platform_flags & mPLAT_XBOX )
	{
		m_checksum *= 4;
	}
}

unsigned long	NxTexture::GetChecksum( void )
{ FUNC_ENTER("NxTexture::GetChecksum"); 
	return m_checksum;
}

void	NxTexture::SetChecksum( unsigned long checksum )
{ FUNC_ENTER("NxTexture::SetChecksum"); 
	m_checksum = checksum;
}

bool	NxTexture::operator==( NxTexture& texture )
{ FUNC_ENTER("NxTexture::operator=="); 
	return( m_checksum == texture.GetChecksum());
}

int	NxTexture::GetTotalDataSize( void )
{ FUNC_ENTER("NxTexture::GetTotalDataSize"); 
	int i;
	int total_size;

	total_size = GetPaletteDataSize();
	for( i = 0; i <= GetNumMipLevels(); i++ )
	{
		total_size += GetTexelDataSize( i );
	}

	return total_size;
}

void	NxTexture::SetValidity( bool valid )
{ FUNC_ENTER("NxTexture::SetValidity"); 
	m_valid = valid;
}

bool	NxTexture::IsValid( void )
{ FUNC_ENTER("NxTexture::IsValid"); 
	return m_valid;
}

bool	NxTexture::IsGrayscale( void )
{ FUNC_ENTER("NxTexture::IsGrayscale"); 
	return m_grayscale;
}

void	NxTexture::SetPlatformFlags( int flags )
{ FUNC_ENTER("NxTexture::SetPlatformFlags"); 
	m_platform_flags = flags;
}

int		NxTexture::GetPlatformFlags( void )
{ FUNC_ENTER("NxTexture::GetPlatformFlags"); 
	return m_platform_flags;
}

BOOL	NxTexture::UsesBasePal(int mip)
{ FUNC_ENTER("NxTexture::UsesBasePal"); 
	return m_useBasePal[mip];
}

void	NxTexture::SetUseBasePal(int mip,BOOL bUseBasePal)
{ FUNC_ENTER("NxTexture::SetUseBasePal"); 
	m_useBasePal[mip] = bUseBasePal;
}

void	NxTexture::Dump( void )
{ FUNC_ENTER("NxTexture::Dump"); 
	fstream file;
	int i, size, offset;
	char filename[_MAX_PATH];

	// For now, only write out 24-bit textures
	if( GetPaletteFormat() != v24_BIT )
	{
		return;
	}

	for( i = 0; i <= m_mip_levels; i++ )
	{
		unsigned short field;
		int j;
		char* pal;

		sprintf( filename, "c:/temp/auto%d.bmp", i );
		file.open( filename, ios::out | ios::binary );

		field = 0x4d42;
		file.write((const char*) &field, sizeof( unsigned short ));

		size = 26 /*combined header size*/ + GetTexelDataSize( i ) + GetPaletteDataSize();
		file.write((const char*) &size, sizeof( int ));

		field = 0;
		file.write((const char*) &field, sizeof( unsigned short ));
		file.write((const char*) &field, sizeof( unsigned short ));

		offset = 26 /*combined header size*/ + GetPaletteDataSize();
		file.write((const char*) &offset, sizeof( int ));

		size = 12;	// header size
		file.write((const char*) &size, sizeof( unsigned int ));

		field = GetWidth( i );
		file.write((const char*) &field, sizeof( unsigned short ));

		field = GetHeight( i );
		file.write((const char*) &field, sizeof( unsigned short ));

		field = 1;	// num planes
		file.write((const char*) &field, sizeof( unsigned short ));
	
		field = GetBpp();
		file.write((const char*) &field, sizeof( unsigned short ));

		pal = GetPaletteData();
		for( j = 0; j < GetNumPaletteEntries(); j++ )
		{
			file.write((const char*) &pal[( j * 3 ) + 2], sizeof( char ));
			file.write((const char*) &pal[( j * 3 ) + 1], sizeof( char ));
			file.write((const char*) &pal[(j * 3 )], sizeof( char ));
		}

		file.write((const char*) GetTexelData( i ), GetTexelDataSize( i ));
		file.close();
	}	
}
