#include "FuncEnter.h"

/*
	ScriptKeyEditor.cpp
	Popup script key editor window for Cutscene Exporter (and possibly others later)
	aml - 5-19-03
*/

#include "ScriptKeyEditor.h"
#include "Resource.h"
#include "path.h"
#include "MemDebug.h"

ScriptKeyEditor::ScriptKeyEditor(HINSTANCE hInstance, HWND hwndParent) :
//	ModalDlgWindow(hInstance, MAKEINTRESOURCE(IDD_SCRIPTKEYEDITOR), hwndParent)
	MSDlgWindow(hInstance, MAKEINTRESOURCE(IDD_SCRIPTKEYEDITOR), hwndParent)
{ FUNC_ENTER("ScriptKeyEditor::ScriptKeyEditor"); 
	editor = new RTFEditor(hInstance, GetDlgItem(hwnd, IDC_RICHEDIT1));
	timeAdjust = NULL;

	// Initialize parameter auto completion/dynUI Autoduck interface builder
	CStr strMapFile=getenv(APP_ENV);

	if (strMapFile==CStr(""))
	{
		char ErrorBuf[256];
		sprintf(ErrorBuf,"Couldn't load '%s' the '%s' environment variable is not set.",MAP_FILE,APP_ENV);

		MessageBox(NULL,ErrorBuf,"Property Editor",MB_ICONSTOP|MB_OK);
	}
	else
	{
		strMapFile+=CStr(SCRIPT_PATH)+MAP_FILE;
		editor->ParseMapFile(strMapFile);
		editor->ParseTagDBFile();
	}

	TimeChangeCB    = NULL;
	pTimeChangeData = NULL;
	UpdateCB        = NULL;
	pUpdateData     = NULL;
	CancelCB        = NULL;
	pCancelData     = NULL;
	KeyChangeCB     = NULL;
	pKeyChangeData  = NULL;
	linkKey         = NULL;
	trackUI         = NULL;

	bLockTextUpdate = false;

	ITimeEdit = GetICustEdit(GetDlgItem(hwnd, IDC_TIMEEDIT));
	ITimeSpin = GetISpinner(GetDlgItem(hwnd, IDC_TIMESPIN));
	ITimeSpin->LinkToEdit(GetDlgItem(hwnd, IDC_TIMEEDIT), EDITTYPE_INT);
	ITimeSpin->SetLimits(0.0f, 999999.0f);

	INameEdit = GetICustEdit(GetDlgItem(hwnd, IDC_KEYNAME));

	keyClass = new MultiList(hInstance);
	keyClass->Attach(GetDlgItem(hwnd, IDC_KEYCLASS));
	keyType = new MultiList(hInstance);
	keyType->Attach(GetDlgItem(hwnd, IDC_KEYTYPE));

	keyClass->EnableAck();
	keyType->EnableAck();

	pPropList = new PropList(hInstance, 2);
	pPropList->Attach(GetDlgItem(hwnd, IDC_PROPLIST));
	pPropList->SetChangeCB(PropChangeCB, this);
	pPropList->SetManualRedraw(true);

	keyClass->SetCurSel(1);
	ShowWindow(GetDlgItem(hwnd, IDC_RICHEDIT1), SW_SHOW);
	ShowWindow(GetDlgItem(hwnd, IDC_PROPLIST), SW_HIDE);

	ParseScriptIni("c:\\skate5\\data\\scripts\\scriptkeys.ini");
	AddKeyClasses();
}

ScriptKeyEditor::~ScriptKeyEditor()
{ FUNC_ENTER("ScriptKeyEditor::~ScriptKeyEditor"); 
	ReleaseICustEdit(INameEdit);
	ReleaseICustEdit(ITimeEdit);
	ReleaseISpinner(ITimeSpin);

	delete editor;
	delete keyClass;
	delete keyType;
}

void ScriptKeyEditor::AddKeyClasses()
{ FUNC_ENTER("ScriptKeyEditor::AddKeyClasses"); 
	Link<ConfigClass>* link = configDB.GetHead();

	keyClass->Reset();
	keyType->Reset();

	keyClass->AddItem(SCRIPTED_KEY_NAME);
	keyClass->SetValue(SCRIPTED_KEY_NAME);

	while(link)
	{
		keyClass->AddItem(link->data.name);		
		link = link->next;
	}
}

void ScriptKeyEditor::AddKeyTypes()
{ FUNC_ENTER("ScriptKeyEditor::AddKeyTypes"); 
	Link<ConfigClass>* linkClass = ConfigDBFind(keyClass->GetValue());

	keyType->Reset();

	if (linkClass)
	{
		Link<ConfigType>* linkType = linkClass->data.types.GetHead();

		while(linkType)
		{
			keyType->AddItem(linkType->data.name);
			linkType = linkType->next;
		}
	}
}

BOOL ScriptKeyEditor::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ FUNC_ENTER("ScriptKeyEditor::DlgProc"); 
	if (LOWORD(wParam)==IDC_RICHEDIT1)
		if (editor->ProcMessage(hwnd,msg,wParam,lParam))
			return TRUE;

	switch(msg)
	{
	case CC_SPINNER_CHANGE:
		if (timeAdjust)
			*timeAdjust = ITimeSpin->GetIVal() * GetTicksPerFrame();

		// Update the current key text in the key list combo box
		UpdateCurKeyText();

		if (TimeChangeCB)
			TimeChangeCB(ITimeSpin->GetIVal(), pTimeChangeData);

		return TRUE;

	case WM_CLOSE:
		if (CancelCB)
			CancelCB(this, pCancelData);

		Hide();
		return TRUE;

	case WM_COMMAND:
		{
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				{
					switch(LOWORD(wParam))
					{
					case IDC_KEYNAME:
						{
							if (linkKey)
							{
								char buf[256];
								INameEdit->GetText(buf, 255);
								linkKey->data.name = buf;

								if (trackUI)
									trackUI->Refresh();

								UpdateCurKeyText();
							}

							return TRUE;
						}
					}

				}

			case CBN_SELCHANGE:
				{
					switch(LOWORD(wParam))
					{
					case IDC_KEYCLASS:
						AddKeyTypes();

						// Select the first type in the list after a class change
						keyType->SetCurSel(0);

						UpdateClassType();
						return TRUE;

					case IDC_KEYTYPE:
						UpdateClassType();
						return TRUE;

					case IDC_KEYLIST:
						{
							// If selection has changed in the key list auto apply the changes to the current key
							// before the actual switch is made
							ApplyTrackKeyUserData();
							
							if (UpdateCB)
								UpdateCB(this, pUpdateData);
							//////////////////////////////

							bLockTextUpdate = true;

							int idx = SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_GETCURSEL, 0, 0);
							Link<TrackUIKey>* link = (Link<TrackUIKey>*)SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_GETITEMDATA, (WPARAM)idx, 0);

							if (KeyChangeCB)
								KeyChangeCB(this, link, pKeyChangeData);

							SetKeyAdjust(link);

							bLockTextUpdate = false;
							return TRUE;
						}
					}
				}
			}

			switch(LOWORD(wParam))
			{
			case IDC_APPLY:
				ApplyTrackKeyUserData();
				
				if (UpdateCB)
					UpdateCB(this, pUpdateData);

				Hide();
				return TRUE;

			case IDC_CANCEL:
				if (CancelCB)
					CancelCB(this, pCancelData);
					
				Hide();
				return TRUE;
			}
		}
		break;
	}

	return FALSE;
}

void ScriptKeyEditor::SetTimeAdjust(int* time)
{ FUNC_ENTER("ScriptKeyEditor::SetTimeAdjust"); 
	timeAdjust = time;
	ITimeSpin->SetValue(*time / GetTicksPerFrame(), TRUE);

	// Scan through the key list and select the key that matches our time
	int count = SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_GETCOUNT, 0, 0);
	bLockTextUpdate = true;

	for(int i = 0; i < count; i++)
	{
		Link<TrackUIKey>* link = (Link<TrackUIKey>*)SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_GETITEMDATA, (WPARAM)i, 0);

		if (link && link->data.time == *time)
		{
			// This key matches our time.  Select it
			SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_SETCURSEL, (WPARAM)i, 0);
			bLockTextUpdate = false;
			return;
		}
	}

	bLockTextUpdate = false;
}

void ScriptKeyEditor::SetTimeChangeCB(void(*pFunc)(int,void*), void* pData)
{ FUNC_ENTER("ScriptKeyEditor::SetTimeChangeCB"); 
	TimeChangeCB    = pFunc;
	pTimeChangeData = pData;
}

void ScriptKeyEditor::SetUpdateCB(void(*pFunc)(ScriptKeyEditor*,void*),void* pData)
{ FUNC_ENTER("ScriptKeyEditor::SetUpdateCB"); 
	UpdateCB = pFunc;
	pUpdateData = pData;
}

void ScriptKeyEditor::SetCancelCB(void(*pFunc)(ScriptKeyEditor*,void*),void* pData)
{ FUNC_ENTER("ScriptKeyEditor::SetCancelCB"); 
	CancelCB = pFunc;
	pCancelData = pData;
}

void ScriptKeyEditor::SetKeyChangeCB(void(*pFunc)(ScriptKeyEditor*,Link<TrackUIKey>* link,void*),void* pData)
{ FUNC_ENTER("ScriptKeyEditor::SetKeyChangeCB"); 
	KeyChangeCB = pFunc;
	pKeyChangeData = pData;
}

void ScriptKeyEditor::BuildKeyList(TrackUI* trackUI)
{ FUNC_ENTER("ScriptKeyEditor::BuildKeyList"); 
	this->trackUI = trackUI;
	
	SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_RESETCONTENT, 0, 0);
	
	bLockTextUpdate = true;

	LinkList<TrackUIKey>* list = trackUI->GetKeyList();
	list->Sort();

	Link<TrackUIKey>* link = list->GetHead();

	while(link)
	{
		char buf[256];
		sprintf(buf, "%s (Time: %i)", (char*)link->data.name, link->data.time / GetTicksPerFrame());

		int idx = SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_ADDSTRING, 0, (LPARAM)buf);
		SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_SETITEMDATA, (WPARAM)idx, (LPARAM)(DWORD)link);

		link = link->next;
	}

	bLockTextUpdate = false;
}

void ScriptKeyEditor::UpdateClassType()
{ FUNC_ENTER("ScriptKeyEditor::UpdateClassType"); 
	CStr value = keyClass->GetValue();

	if (value == CStr(SCRIPTED_KEY_NAME))
	{
		ShowWindow(GetDlgItem(hwnd, IDC_RICHEDIT1), SW_SHOW);
		ShowWindow(GetDlgItem(hwnd, IDC_PROPLIST), SW_HIDE);	

		if (linkKey)
		{
			linkKey->data.userData = SCRIPTKEY_SCRIPT;
			editor->SetText(linkKey->data.userBuffer);
		}
	}
	else
	{
		ShowWindow(GetDlgItem(hwnd, IDC_RICHEDIT1), SW_HIDE);
		ShowWindow(GetDlgItem(hwnd, IDC_PROPLIST), SW_SHOW);

		if (linkKey)
			linkKey->data.userData = SCRIPTKEY_GENERATED;

		// Now build the property list UI from the data loaded from scriptkeys.ini
		CStr strClassRec = GetClassTypeRecord(keyClass->GetValue(), keyType->GetValue());
			
		// Acquire the properties and add them to the property list
		LinkList<ConfigProp> props;
		ParseConfigProps(&props,NULL,strClassRec);
		//ParseConfigProps(&props,&flags,strClassRec,&strCmds,&strCluster,&programs);

		pPropList->Clear();
		pPropList->HasApply(false);
		BuildPropList(pPropList, &props);
		PropChangeCB(pPropList, this);
		pPropList->BuildUI();
	}
}

void ScriptKeyEditor::SetKeyAdjust(Link<TrackUIKey>* key)
{ FUNC_ENTER("ScriptKeyEditor::SetKeyAdjust"); 
	timeAdjust = &key->data.time;

	// Select this key in the key list
	int count = SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_GETCOUNT, 0, 0);

	bLockTextUpdate = true;

	for(int i = 0; i < count; i++)
	{
		Link<TrackUIKey>* link = (Link<TrackUIKey>*)SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_GETITEMDATA, (WPARAM)i, 0);

		if (link == key)
		{
			SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_SETCURSEL, (WPARAM)i, 0);
			linkKey = link;
			LoadKeyData();

			ITimeSpin->SetValue(key->data.time / GetTicksPerFrame(), TRUE);
			INameEdit->SetText(key->data.name);

			bLockTextUpdate = false;
			return;
		}
	}

	// The key should always be found in the list!
	bLockTextUpdate = false;
	assert(0);
}

void ScriptKeyEditor::UpdateCurKeyText()
{ FUNC_ENTER("ScriptKeyEditor::UpdateCurKeyText"); 
	if (bLockTextUpdate)
		return;

	bLockTextUpdate = true;

	if (linkKey)
	{
		char buf[256];
		sprintf(buf, "%s (Time: %i)", (char*)linkKey->data.name, linkKey->data.time / GetTicksPerFrame());
		
		int idxOld = SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_GETCURSEL, 0, 0);
		if (idxOld == -1)
		{
			bLockTextUpdate = false;
			return;
		}

		SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_DELETESTRING, (WPARAM)idxOld, 0);
		int idxNew = SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_INSERTSTRING, (WPARAM)idxOld-1, (LPARAM)buf);

		SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_SETITEMDATA, (WPARAM)idxNew, (LPARAM)linkKey);
		SendDlgItemMessage(hwnd, IDC_KEYLIST, CB_SETCURSEL, (WPARAM)idxNew, 0);
	}

	bLockTextUpdate = false;
}

void ScriptKeyEditor::ApplyTrackKeyUserData()
{ FUNC_ENTER("ScriptKeyEditor::ApplyTrackKeyUserData"); 
	if (!linkKey)
		return;

	CStr value = keyClass->GetValue();

	if (value == CStr(SCRIPTED_KEY_NAME))	
	{
		linkKey->data.userBuffer = editor->GetText();
		linkKey->data.userData   = SCRIPTKEY_SCRIPT;
		WriteKeyData();
	}
	else
	{
		linkKey->data.userBuffer = "";
		linkKey->data.userData   = SCRIPTKEY_GENERATED;
		WriteKeyData();
	}
}

CStr ScriptKeyEditor::GetScriptBuffer()
{ FUNC_ENTER("ScriptKeyEditor::GetScriptBuffer"); 
	if (linkKey->data.userData == SCRIPTKEY_SCRIPT)
		return linkKey->data.userBuffer;
	
	LinkList<ConfigProp> cprops;
	CStr strClass    = keyClass->GetValue();
	CStr strType     = keyType->GetValue();
	CStr strClassRec = GetClassTypeRecord(strClass, strType);

	BuildConfigList(&cprops, pPropList);
	CStr strTemplate = BuildTemplateBuffer(&cprops, strClassRec);
	return ProcessTemplateBuffer(&cprops, strTemplate);
}

/*
CStr ScriptKeyEditor::GetScriptBuffer(Link<TrackUIKey>* link)
{
	if (link->data.userData == SCRIPTKEY_SCRIPT)
		return link->data.userBuffer;

	LinkList<ConfigProp> cprops;
	CStr strClass    = keyClass->GetValue();
	CStr strType     = keyType->GetValue();
	CStr strClassRec = GetClassTypeRecord(strClass, strType);

	BuildConfigList(&cprops, pPropList);
	CStr strTemplate = BuildTemplateBuffer(&cprops, strClassRec);
	
	return ProcessTemplateBuffer(&cprops, strTemplate);
}
*/

CStr ScriptKeyEditor::GetScriptBuffer(Link<TrackUIKey>* link)
{ FUNC_ENTER("ScriptKeyEditor::GetScriptBuffer"); 
	if (link->data.userData == SCRIPTKEY_SCRIPT)
		return link->data.userBuffer;

	LinkList<ConfigProp> cprops;
	CStr strClass;
	CStr strType;
	CStr strClassRec;

//////////////////////////////////////////////////////////////////////////////

	if (link->data.pInternalUserData)
	{
		unsigned char* pos = (unsigned char*)link->data.pInternalUserData;
		DWORD version = *((DWORD*)pos);
		pos += sizeof(DWORD);
		GetString(&pos, strClass);
		GetString(&pos, strType);

		GetList<ConfigProp>(&pos, &cprops);
	}
	else
	{
		strClass = keyClass->GetValue();
		strType = keyType->GetValue();
	}

/////////////////////////////////////////////////////////////////////////////

	strClassRec = GetClassTypeRecord(strClass, strType);

	CStr strTemplate = BuildTemplateBuffer(&cprops, strClassRec);	
	return ProcessTemplateBuffer(&cprops, strTemplate);
}

void ScriptKeyEditor::PropChangeCB(PropList* pPropList, void* pData)
{ FUNC_ENTER("ScriptKeyEditor::PropChangeCB"); 
	ScriptKeyEditor* pthis = (ScriptKeyEditor*)pData;

	if (pthis->linkKey)
		pthis->linkKey->data.userBuffer = pthis->GetScriptBuffer(pthis->linkKey);
}

void ScriptKeyEditor::WriteKeyData()
{ FUNC_ENTER("ScriptKeyEditor::WriteKeyData"); 
	VerifyMemory();

	LinkList<ConfigProp> cprops;
	CStr strClass    = keyClass->GetValue();
	CStr strType     = keyType->GetValue();

	DWORD version = SCRIPTKEY_VERSION;

	BuildConfigList(&cprops, pPropList);
	int propSize  = sizeof(DWORD) + GetStringSize(strClass) + GetStringSize(strType) + ListSize<ConfigProp>(&cprops);

	linkKey->data.AllocUserData(propSize);
	unsigned char* pos = (unsigned char*)linkKey->data.pInternalUserData;
	*((DWORD*)pos) = version;
	pos += sizeof(DWORD);
	WriteString(&pos, strClass);
	WriteString(&pos, strType);

	WriteList<ConfigProp>(&pos, &cprops);

	VerifyMemory();

//	if (trackUI)
//		trackUI->SaveKeyData();
}

void ScriptKeyEditor::LoadKeyData()
{ FUNC_ENTER("ScriptKeyEditor::LoadKeyData"); 
	VerifyMemory();

	LinkList<ConfigProp> cprops;
	CStr strClass;
	CStr strType;

	if (!linkKey->data.pInternalUserData)
	{
		ShowWindow(GetDlgItem(hwnd, IDC_RICHEDIT1), SW_SHOW);
		ShowWindow(GetDlgItem(hwnd, IDC_PROPLIST), SW_HIDE);

		keyClass->SetValue(SCRIPTED_KEY_NAME);

		editor->SetText(GetScriptBuffer(linkKey));
		return;
	}
	else
	{
		ShowWindow(GetDlgItem(hwnd, IDC_RICHEDIT1), SW_HIDE);
		ShowWindow(GetDlgItem(hwnd, IDC_PROPLIST), SW_SHOW);
	}

	unsigned char* pos = (unsigned char*)linkKey->data.pInternalUserData;
	DWORD version = *((DWORD*)pos);
	pos += sizeof(DWORD);
	GetString(&pos, strClass);
	GetString(&pos, strType);

	GetList<ConfigProp>(&pos, &cprops);

	keyClass->SetValue(strClass);
	AddKeyTypes();
	keyType->SetValue(strType);

	UpdateClassType();
	AssignPropValues(pPropList, &cprops);

	VerifyMemory();

	//editor->SetText(GetScriptBuffer(linkKey));
}

void ScriptKeyEditor::Show()
{ FUNC_ENTER("ScriptKeyEditor::Show"); 
	POINT pt;
	GetCursorPos(&pt);
	SetWindowPos(hwnd, HWND_TOP, pt.x, pt.y, 0, 0, SWP_NOSIZE);

	MSDlgWindow::Show();
}
