/*
	CamExporter.cpp
	Camera Exporter
*/

#include <Engine/Engine.h>
#include "CamExporter.h"
#include <Sk/Gamenet/ExportMsg.h>
#include "../Misc/GenCrc.h"
#include "../Misc/Util.h"
#include <io.h>
#include <sys/stat.h>
#include <direct.h>
#include "CamExporter.h"
#include "GenExport.h"
#include "../../../../include/AnimExportFmt.h"
#include "Resource.h"
#include "../UI/ListView.h"
#include "../PropEdit/ParseFuncs.h"
#include "path.h"
#include "Appdata.h"
#include "SceneExportOptions.h"
#include "../UI/OKtoAll.h"
#include "SceneExport.h"
#include <process.h>
#include "../misc/mth.h"
#include "MemDebug.h"

#define NO_PREINCLUDES
#define NO_BASEPREINCLUDES
#include "FuncEnter.h"

#define COLID_CAMNAME    0
//#define COLID_MAXNAME    1
//#define COLID_START      2
//#define COLID_END        3
// MAX Name field has been removed
#define COLID_START      1
#define COLID_END        2


#define COLID_KEYFRAME   0
#define COLID_SCRIPT     1
#define COLID_KEYTIME    2
#define COLID_SCRIPTFILE 3

#define WRITE_ACCESS           0x02
#define DEFAULT_COMPRESS       0.999700f
#define DEFAULT_TCOMPRESS      1.000000f
#define DEFAULT_FCOMPRESS      1.000000f
#define CAMERA_CRC             0xc4e311fa

#define USERSCRIPT_NAME		   "[User]"

static CameraExporter theCameraExporter;
CameraExporter* GetCamExporter() { FUNC_ENTER("GetCamExporter");  return &theCameraExporter; }


struct CamExportSettings
{
	bool bDebug;		// Output debug information
	bool bCompress;		// Compress the keyframe data
	bool bNoDbls;		// Compress 2 key sequences to 1 key
	bool bExportQN;		// Export a QN file containing all exported cameras
	char strPrefix[256];// Export prefix name
	bool bExportAll;	// True if all cameras should be exported
	bool bCompressTime;	// Time should be compressed (no floats)   // Now Always On
	bool bCleanup;		// Deletes unsed camera files from the output directory
	bool bArchive;		// Archives the camera animation *.ska files
	bool bNetUpdate;	// True if a network update should be sent when selected
};

/////////// CamAddKeyDlg  ////////////////////////////////////////////

CamAddKeyDlg::CamAddKeyDlg(HINSTANCE hInstance,HWND hwndParent,Interface* ip) :
	MSDlgWindow(hInstance,MAKEINTRESOURCE(IDD_CAMOPTADDKEY),hwndParent)
{ FUNC_ENTER("CamAddKeyDlg::CamAddKeyDlg"); 
	this->ip=ip;

	scriptEditor = new RTFEditor(hInstance,GetDlgItem(hwnd,IDC_RICHEDIT1));
	scriptEditor->InitScriptDB();

	fpOK=NULL;
	pOKData=NULL;
	bAbort=false;
	bScriptsLoaded=false;

	IEdit=GetICustEdit(GetDlgItem(hwnd,IDC_EDITTIME));
	IEditFile=GetICustEdit(GetDlgItem(hwnd,IDC_EDITFILE));
	ISpin=GetISpinner(GetDlgItem(hwnd,IDC_SPINTIME));

	ISpin->LinkToEdit(GetDlgItem(hwnd,IDC_EDITTIME),EDITTYPE_FLOAT);
}

CamAddKeyDlg::~CamAddKeyDlg()
{ FUNC_ENTER("CamAddKeyDlg::~CamAddKeyDlg"); 
	ReleaseISpinner(ISpin);
	ReleaseICustEdit(IEdit);
	ReleaseICustEdit(IEditFile);

	delete scriptEditor;
}

CStr CamAddKeyDlg::GetScriptFile()
{ FUNC_ENTER("CamAddKeyDlg::GetScriptFile");  
	if (scriptFile.Length()==0)
		return CStr("[None]");

	return scriptFile; 
}

void CamAddKeyDlg::Show()
{ FUNC_ENTER("CamAddKeyDlg::Show"); 
	MSDlgWindow::Show();

	if (!bScriptsLoaded)
	{
		Refresh();
		bScriptsLoaded=true;
	}
}

BOOL CamAddKeyDlg::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("CamAddKeyDlg::DlgProc"); 
	if (LOWORD(wParam)==IDC_RICHEDIT1)
		if (scriptEditor->ProcMessage(hwnd,msg,wParam,lParam))
			return TRUE;

	switch(msg)
	{
	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			;//EnableAccelerators();
		else
			DisableAccelerators();
		
		return TRUE;

	case CC_SPINNER_CHANGE:
		SpinChange();
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDOK:
			Hide();
			if (fpOK)
				fpOK(this,pOKData);
			
			scriptFile="";

			return TRUE;

		case IDCANCEL:
			Hide();
			return TRUE;

		case IDC_RESETSCRIPT:
			SetScript("");
			SetScriptCode("");
			scriptEditor->AllowChanges(TRUE);
			return TRUE;

		case IDC_REFRESH:
			Refresh();
			return TRUE;

		case IDC_EDITFILE:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				{
					char buf[256];
					IEditFile->GetText(buf,256);
					if (strlen(buf)==0)
					{
						scriptEditor->AllowChanges(TRUE);
						scriptEditor->SetText("");
					}

					DisableAccelerators();

					break;
				}
			}

		case IDC_SCRIPTLIST:
			switch(HIWORD(wParam))
			{
			case LBN_SELCHANGE:
				SelChange();
				return TRUE;
			}
		}
	}

	return FALSE;
}

bool CamAddKeyDlg::MatchContent(FILE* fp,char* wildcard,char* buf,int* line)
{ FUNC_ENTER("CamAddKeyDlg::MatchContent"); 
	char  tempBuf[512];

	if (!fp)
		return false;

	while(!feof(fp))
	{
		fgets(tempBuf,510,fp);
		(*line)++;

		CStr strBuf=tempBuf;
		strBuf.toLower();

		CStr strWildcard=wildcard;
		strWildcard.toLower();

		if (MatchPattern(strBuf,strWildcard))
		{
			tempBuf[511]='\0';
			strcpy(buf,tempBuf);
			return true;
		}

		// Check for cancel
		MSG msg;

		if (PeekMessage(&msg,GetDlgItem(hwnd,IDC_CANCEL),0,0,PM_REMOVE))
		{
			if (msg.message==WM_LBUTTONDOWN)
			{
				bAbort=true;
				return false;
			}
		}
	}
	
	return false;
}

void CamAddKeyDlg::Refresh()
{ FUNC_ENTER("CamAddKeyDlg::Refresh"); 
	CStr strPath=getenv(APP_ENV);
	strPath+=SCRIPT_PATH;
	strPath=strPath.Substr(0,strPath.Length()-1);

	int count=CountFiles(strPath,"*.q");
	SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_SETRANGE,0,MAKELPARAM(0,count));
	SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_SETPOS,0,0);
	SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_RESETCONTENT,0,0);
	scriptFileDB.Clear();

	bAbort=false;
	Find(strPath,"*.q","script *");
}


int CamAddKeyDlg::CountFiles(char* sdir,char* prefix)
{ FUNC_ENTER("CamAddKeyDlg::CountFiles"); 
	_finddata_t fdata;
	int         count=0;

	_chdir(sdir);
	long handle=_findfirst("*.*",&fdata);
	int  bContinue;

	if (handle)
		bContinue=0;

	while(bContinue==0)
	{
		// Recurse subdirectories
		if (fdata.attrib & _A_SUBDIR)
		{
			if (strcmp(fdata.name,".")!=0 &&
				strcmp(fdata.name,"..")!=0)
			{
				char strPath[1024];
				strcpy(strPath,sdir);
				strcat(strPath,"\\");
				strcat(strPath,fdata.name);

				count+=CountFiles(strPath,prefix);
				_chdir(sdir);
			}
		}
		else
		{
			char strPath[1024];
			strcpy(strPath,sdir);
			strcat(strPath,"\\");
			strcat(strPath,fdata.name);

			if (MatchPattern(CStr(fdata.name),CStr(prefix)))
				count++;
		}

		bContinue=_findnext(handle,&fdata);
	}

	_findclose(handle);

	return count;
}

void CamAddKeyDlg::Find(char* sdir,char* prefix,char* search)
{ FUNC_ENTER("CamAddKeyDlg::Find"); 
	_finddata_t fdata;

	_chdir(sdir);
	long handle=_findfirst("*.*",&fdata);
	int  bContinue=-1;

	if (handle)
		bContinue=0;

	while(bContinue==0)
	{
		// Recurse subdirectories
		if (fdata.attrib & _A_SUBDIR)
		{
			if (strcmp(fdata.name,".")!=0 &&
				strcmp(fdata.name,"..")!=0)
			{
				char strPath[1024];
				strcpy(strPath,sdir);
				strcat(strPath,"\\");
				strcat(strPath,fdata.name);

				Find(strPath,prefix,search);
				_chdir(sdir);
			}
		}
		else
		{
			char strPath[2048];
			strcpy(strPath,sdir);
			strcat(strPath,"\\");
			strcat(strPath,fdata.name);

			if (MatchPattern(CStr(fdata.name),CStr(prefix)))
			{
				FILE* fp;
				char buf[1024]="";
				int  line=0;
				fp=fopen(fdata.name,"r");

				while(MatchContent(fp,search,buf,&line))
				{
					char strNew[1024];
					char strNum[256];
					_itoa(line,strNum,10);

					/*
					strcpy(strNew,strPath);
					strcat(strNew," (");
					strcat(strNew,strNum);
					strcat(strNew,")        ");
					strcat(strNew,buf);
					*/

					CStr strBuf=buf;
					strBuf.toLower();

					int pos=Instr(strBuf,"script ");
					if (pos!=-1 && pos<strBuf.Length())
					{
						strcpy(strNew,buf+pos+7);

						// Terminate after first word
						for(int i=0;i<strlen(strNew);i++)
						{
							if ((strNew[i]==' ' &&  !LineContainsAll(strNew,' ')) ||
								strNew[i]=='\t' ||
								strNew[i]=='\n' ||
								strNew[i]=='\r')
							{
								strNew[i]='\0';
								break;
							}
						}

						CStr str=StripLeft(strNew);

						if (str.Length()>0)
						{
							int index=SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_ADDSTRING,0,(LPARAM)(char*)str);
						
							// Add filename to the newly added entry
							char path[512];
							_getcwd(path,511);

							CStr strFilename=CStr(path)+CStr("\\")+CStr(fdata.name);
							Link<CStr>* link=scriptFileDB.Add(&strFilename);
							SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_SETITEMDATA,(WPARAM)index,(LPARAM)link);
						}
					}

					// Check for cancel
					MSG msg;

					if (PeekMessage(&msg,GetDlgItem(hwnd,IDC_CANCEL),0,0,PM_REMOVE))
					{
						if (msg.message==WM_LBUTTONDOWN)
							return;
					}
				}

				fclose(fp);

				if (bAbort)
					return;

				// Check for cancel
				MSG msg;

				if (PeekMessage(&msg,GetDlgItem(hwnd,IDC_CANCEL),0,0,PM_REMOVE))
				{
					if (msg.message==WM_LBUTTONDOWN)
					{
						_findclose(handle);
						return;
					}
				}

				// Update progress bar
				int val=SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_GETPOS,0,0);
				SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_SETPOS,(WPARAM)val+1,0);
			}
		}

		bContinue=_findnext(handle,&fdata);
	}

	_findclose(handle);
}

void CamAddKeyDlg::SetOKCB(void(*Func)(CamAddKeyDlg*,void*),void* pData)
{ FUNC_ENTER("CamAddKeyDlg::SetOKCB"); 
	fpOK=Func;
	pOKData=pData;
}

void CamAddKeyDlg::SetLimits(int min,int max)
{ FUNC_ENTER("CamAddKeyDlg::SetLimits"); 
	ISpin->SetLimits(min,max);
}
	
int CamAddKeyDlg::GetTime()
{ FUNC_ENTER("CamAddKeyDlg::GetTime"); 
	return IEdit->GetFloat();
}

void CamAddKeyDlg::SetTime(int time)
{ FUNC_ENTER("CamAddKeyDlg::SetTime"); 
	ISpin->SetValue(time,TRUE);
}

CStr CamAddKeyDlg::GetScript()
{ FUNC_ENTER("CamAddKeyDlg::GetScript"); 
	char buf[256];
	IEditFile->GetText(buf,256);
	return CStr(buf);
}

CStr CamAddKeyDlg::GetScriptCode()
{ FUNC_ENTER("CamAddKeyDlg::GetScriptCode"); 
	return scriptEditor->GetText();	
}

void CamAddKeyDlg::SetScript(CStr strScript)
{ FUNC_ENTER("CamAddKeyDlg::SetScript"); 
	IEditFile->SetText(strScript);
}

void CamAddKeyDlg::SetScriptCode(CStr strScriptCode)
{ FUNC_ENTER("CamAddKeyDlg::SetScriptCode"); 
	scriptEditor->SetText(strScriptCode);
	if (strScriptCode==CStr(""))
		scriptEditor->AllowChanges(TRUE);
}

void CamAddKeyDlg::SelChange()
{ FUNC_ENTER("CamAddKeyDlg::SelChange"); 
	char buf[256];

	int index=SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_GETCURSEL,0,0);
	SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_GETTEXT,(WPARAM)index,(LPARAM)buf);

	// Set the script name
	IEditFile->SetText(buf);

	// Set the script filename
	Link<CStr>* link=(Link<CStr>*)SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_GETITEMDATA,(WPARAM)index,0);
	scriptFile=link->data;

	// Load the script into the script editor window
	if (strlen(buf)>0)
	{
		scriptEditor->AllowChanges(TRUE);
		scriptEditor->GoToScriptDefinition(buf);
		scriptEditor->AllowChanges(FALSE);
	}
	else
	{
		scriptEditor->AllowChanges(TRUE);
		scriptEditor->SetText("");
	}
}

void CamAddKeyDlg::SpinChange()
{ FUNC_ENTER("CamAddKeyDlg::SpinChange"); 
	char     buf[256];
	int      val       = ISpin->GetIVal();
	Interval animrange = ip->GetAnimRange();
	
	sprintf(buf,"%f",((float)val-(animrange.Start()/GetTicksPerFrame()))/(float)GetFrameRate());

	SetDlgItemText(hwnd,IDC_TIME,buf);

	ip->SetTime(val*GetTicksPerFrame());
}

float CamAddKeyDlg::GetTimeSec()
{ FUNC_ENTER("CamAddKeyDlg::GetTimeSec"); 
	char buf[256];
	GetDlgItemText(hwnd,IDC_TIME,buf,255);

	return atof(buf);
}

/////////// CameraExportDlg  //////////////////////////////////////////

FILE* fpOutput;

CameraExportDlg::CameraExportDlg(HINSTANCE hInstance,HWND hwndParent,Interface* ip,ExportOptionsDlg* expopt) :
	ResizeDlgWindow(hInstance,MAKEINTRESOURCE(IDD_CAMOPTIONS),hwndParent)
{ FUNC_ENTER("CameraExportDlg::CameraExportDlg"); 
	this->ip=ip;
	this->expopt = expopt;

	scriptEditor = new RTFEditor(hInstance,GetDlgItem(hwnd,IDC_RICHEDIT1));
	scriptEditor->InitScriptDB();

	// Assign cam specific defaults
	Interval interval=ip->GetAnimRange();
	camsettings_defaults.start          = interval.Start()/GetTicksPerFrame();
	camsettings_defaults.end            = interval.End()/GetTicksPerFrame();
	camsettings_defaults.fRotTolerance  = DEFAULT_COMPRESS;
	camsettings_defaults.fTranTolerance = DEFAULT_TCOMPRESS;
	camsettings_defaults.fFovTolerance  = DEFAULT_FCOMPRESS;

	lastItem=0;

	bLockCamList=false;
	bLockKeyList=false;
	bLockNodeSelect=false;
	bLockLoadSave=false;
	bLockSelChange=false;
	bSwapCoordSystem=true;		// Swap camera export to game coordinate system

	lastCamNode=NULL;

	// Create the AddKey Dialog
	pAddKeyDlg=new CamAddKeyDlg(hInstance,hwnd,ip);
	pAddKeyDlg->SetOKCB(AddKeyCB,this);

	// Register for notification on MAX Selection change
	// Install notification callback
#ifndef DISABLE_NOTIFICATIONS
	int rVal=RegisterNotification(MAXSelChange,this,NOTIFY_SELECTIONSET_CHANGED);

	if (!rVal)
		MessageBox(NULL,"Couldn't Register Selection Callback","Camera Exporter",MB_ICONSTOP|MB_OK);

	rVal = RegisterNotification(CloseExportDlg,this,NOTIFY_FILE_PRE_OPEN);

	if (!rVal)
		MessageBox(NULL,"Couldn't Register open Callback","Camera Exporter",MB_ICONSTOP|MB_OK);

	rVal = RegisterNotification(CloseExportDlg,this,NOTIFY_SYSTEM_PRE_RESET);

	if (!rVal)
		MessageBox(NULL,"Couldn't Register reset Callback","Camera Exporter",MB_ICONSTOP|MB_OK);
#endif

	// Initialize ListView control
	lview=new ListView(GetDlgItem(hwnd,IDC_KEYLIST));
	camList=new ListView(GetDlgItem(hwnd,IDC_CAMLIST));

	camList->AddColumn("Camera Name",250);
	//camList->AddColumn("MAX Name",100);
	camList->AddColumn("Start");
	camList->AddColumn("End");

	lview->AddColumn("KeyFrame");
	lview->AddColumn("Script",150);
	lview->AddColumn("KeyTime");
	lview->AddColumn("ScriptFile",270);

	// Connect custom edit/spinners
	editQErr    = GetICustEdit(GetDlgItem(hwnd,IDC_QERR));
	editTErr    = GetICustEdit(GetDlgItem(hwnd,IDC_TERR));
	editFErr    = GetICustEdit(GetDlgItem(hwnd,IDC_FERR));
	editStart   = GetICustEdit(GetDlgItem(hwnd,IDC_SFRAMEEDIT));
	editEnd     = GetICustEdit(GetDlgItem(hwnd,IDC_EFRAMEEDIT));
	editKeyTime = GetICustEdit(GetDlgItem(hwnd,IDC_KEYTIMEEDIT));

	spinQErr    = GetISpinner(GetDlgItem(hwnd,IDC_QERRS));
	spinTErr    = GetISpinner(GetDlgItem(hwnd,IDC_TERRS));
	spinFErr    = GetISpinner(GetDlgItem(hwnd,IDC_FERRS));
	spinStart   = GetISpinner(GetDlgItem(hwnd,IDC_SFRAMESPIN));
	spinEnd     = GetISpinner(GetDlgItem(hwnd,IDC_EFRAMESPIN));
	spinKeyTime = GetISpinner(GetDlgItem(hwnd,IDC_KEYTIMESPIN));

	// Attach spinners to the edit boxes
	spinQErr->SetLimits(-5.000000f,5.000000f);
	spinTErr->SetLimits(-5.000000f,5.000000f);
	spinFErr->SetLimits(-5.000000f,5.000000f);
	spinQErr->SetScale(0.000001f);
	spinTErr->SetScale(0.000001f);
	spinFErr->SetScale(0.000001f);
	
	spinKeyTime->SetLimits(0.0f,999999.0f);
	spinKeyTime->SetScale(1.0f);

	GetRange();

	spinQErr->LinkToEdit(GetDlgItem(hwnd,IDC_QERR),EDITTYPE_FLOAT);
	spinTErr->LinkToEdit(GetDlgItem(hwnd,IDC_TERR),EDITTYPE_FLOAT);
	spinFErr->LinkToEdit(GetDlgItem(hwnd,IDC_FERR),EDITTYPE_FLOAT);
	
	spinStart->LinkToEdit(GetDlgItem(hwnd,IDC_SFRAMEEDIT),EDITTYPE_INT);
	spinEnd->LinkToEdit(GetDlgItem(hwnd,IDC_EFRAMEEDIT),EDITTYPE_INT);

	spinKeyTime->LinkToEdit(GetDlgItem(hwnd,IDC_KEYTIMEEDIT),EDITTYPE_INT);

	spinQErr->SetValue(DEFAULT_COMPRESS,TRUE);
	spinTErr->SetValue(DEFAULT_TCOMPRESS,TRUE);
	spinFErr->SetValue(DEFAULT_FCOMPRESS,TRUE);

	spinStart->SetValue(interval.Start()/GetTicksPerFrame(),TRUE);
	spinEnd->SetValue(interval.End()/GetTicksPerFrame(),TRUE);

	// Set Defaults
	CheckDlgButton(hwnd,IDC_OUTPUTDBG,BST_CHECKED);
	CheckDlgButton(hwnd,IDC_USECOMPRESSION,BST_CHECKED);
	CheckDlgButton(hwnd,IDC_COMPDBLS,BST_CHECKED);

	SetResizeModes(7,
				   IDC_QERR,		RSW_SCALE_X,	0,0,0,0,
				   IDC_FERR,		RSW_SCALE_X,	0,0,0,0,
				   IDC_TERR,		RSW_SCALE_X,	0,0,0,0,
				   IDC_SFRAMEEDIT,	RSW_SCALE_X,	0,0,0,0,
				   IDC_EFRAMEEDIT,	RSW_SCALE_X,	0,0,0,0,
				   IDC_PREFIX,		RSW_SCALE_X,	0,0,0,0,
				   IDC_EDITNAME,	RSW_SCALE_X,	0,0,0,0);

	genExporter=new GenExporter(ip);
	GetCamData();
	Show();
}

CameraExportDlg::~CameraExportDlg()
{ FUNC_ENTER("CameraExportDlg::~CameraExportDlg"); 
	// Unregister MAX selection change notification
	UnRegisterNotification(MAXSelChange,this,NOTIFY_SELECTIONSET_CHANGED);
	UnRegisterNotification(CloseExportDlg,this,NOTIFY_FILE_PRE_OPEN);
	UnRegisterNotification(CloseExportDlg,this,NOTIFY_SYSTEM_PRE_RESET);

	delete genExporter;

	if (lview)
		delete lview;

	if (pAddKeyDlg)
		delete pAddKeyDlg;

	// Unregister MAX UI controls
	ReleaseISpinner(spinQErr);
	ReleaseISpinner(spinTErr);
	ReleaseISpinner(spinFErr);
	ReleaseISpinner(spinStart);
	ReleaseISpinner(spinEnd);
	ReleaseISpinner(spinKeyTime);
	ReleaseICustEdit(editQErr);
	ReleaseICustEdit(editTErr);
	ReleaseICustEdit(editFErr);
	ReleaseICustEdit(editStart);
	ReleaseICustEdit(editEnd);
	ReleaseICustEdit(editKeyTime);

	delete scriptEditor;

	//fprintf(fpOutput,"Exit ::~CameraExporter()\n");
	//fclose(fpOutput);
}

void CameraExportDlg::CloseExportDlg(void *param, NotifyInfo *info)
{ FUNC_ENTER("CameraExportDlg::CloseExportDlg"); 
	OutputDebugString("HANDLER: CloseExportDlg\n");
	CameraExportDlg* pthis = (CameraExportDlg*)param;
	pthis->Hide();

	if (pthis->expopt)
	{
		// Emulate pressing of the OK button on the export options dialog
		SendMessage(pthis->expopt->GetHWND(),WM_COMMAND,MAKEWPARAM(IDOK,0),(LPARAM)GetDlgItem(pthis->expopt->GetHWND(),IDOK));
	}
}

void CameraExportDlg::Show()
{ FUNC_ENTER("CameraExportDlg::Show"); 
	LoadSettings();

	AcquireSelSet();

	if (selSet.Count()==1)
	{
		GetCamData();
		SelChange(TRUE);
		//LoadFromNode(selSet[0]);
	}
	else
		lview->Clear();
	
	MSDlgWindow::Show();
}

void CameraExportDlg::Hide()
{ FUNC_ENTER("CameraExportDlg::Hide"); 
	if (selSet.Count()==1)
		SaveToNode(selSet[0]);

	SaveSettings();
	MSDlgWindow::Hide();
}

void CameraExportDlg::UpdateScript()
{ FUNC_ENTER("CameraExportDlg::UpdateScript"); 
	int index = lview->GetSel();

	if (index!=-1)
	{
		Link<CStr>* link = (Link<CStr>*)lview->GetItemData(index);
		if (link)
			link->data = scriptEditor->GetText();
	}
}

BOOL CameraExportDlg::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("CameraExportDlg::DlgProc"); 
	NMHDR* nmhdr;

	ResizeDlgWindow::DlgProc(hwnd,msg,wParam,lParam);

	switch(msg)
	{
	case WM_COMMAND:
		switch(HIWORD(wParam))
		{
		case EN_CHANGE:
			switch(LOWORD(wParam))
			{
			case IDC_RICHEDIT1:
				UpdateScript();
				break;
			}
			break;
		}
		break;
	}

	if (LOWORD(wParam)==IDC_RICHEDIT1)
		if (scriptEditor->ProcMessage(hwnd,msg,wParam,lParam))
			return TRUE;

	switch(msg)
	{
	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			;//EnableAccelerators();
		else
			DisableAccelerators();
		
		return TRUE;

	case WM_CLOSE:
		Hide();
		return TRUE;

	case WM_NOTIFY:
		nmhdr=(NMHDR*)lParam;

		switch(nmhdr->code)
		{
		// Always allow inlist editing of the labels
		case LVN_BEGINLABELEDIT:
			return FALSE;

		case LVN_ENDLABELEDIT:
			UpdateLabel((NMLVDISPINFO*)lParam);
			return TRUE;
		}

		switch(nmhdr->idFrom)
		{
		case IDC_CAMLIST:
			if (!bLockCamList)
			{
				switch(nmhdr->code)
				{
				case LVN_ITEMCHANGED:
					CamSelChange((LPNMLISTVIEW)lParam);
					return TRUE;
				}
			}

			return FALSE;

		case IDC_KEYLIST:
			switch(nmhdr->code)
			{
			case LVN_ITEMCHANGED:
				ListViewUpdate((LPNMLISTVIEW)lParam);
				return TRUE;
			}

			return FALSE;
		}

		return FALSE;

	case CC_SPINNER_CHANGE:
		switch(LOWORD(wParam))
		{
		case IDC_SFRAMESPIN:
		case IDC_EFRAMESPIN:
			if (!bLockSelChange)
				UpdateRange();
			return TRUE;

		case IDC_KEYTIMESPIN:
			UpdateKeyTime();
			return TRUE;
		}

		return FALSE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_DONE:
			SendMessage(hwnd,WM_CLOSE,0,0);
			//Hide();
			return TRUE;

		case IDC_EXPORT:
			if (!DoExport())
			{
				MessageBox(NULL,"Export Failed!","Camera Exporter",MB_ICONSTOP|MB_OK);
			}
			else
			{
				//SendNetMessage();
			}

			return TRUE;

		case IDC_ADDSCRIPT:
			AddScript();
			return TRUE;

		case IDC_DELSCRIPT:
			DeleteScript();
			return TRUE;

		case IDC_GETRANGE:
			GetRange();
			return TRUE;

		case IDC_RESORT:
			ResortList();
			return TRUE;

		case IDC_SFRAMEEDIT:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				UpdateRange();
				return TRUE;
			}
			return FALSE;

		case IDC_EFRAMEEDIT:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				UpdateRange();
				return TRUE;
			}
			return FALSE;

		case IDC_EDITNAME:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				UpdateName();
				return TRUE;
			}
		}
	}

	return FALSE;
}

void CameraExportDlg::MAXSelChange(void *param,NotifyInfo *info)
{ FUNC_ENTER("CameraExportDlg::MAXSelChange"); 
	OutputDebugString("HANDLER: MAXSelChange\n");
	CameraExportDlg* pthis=(CameraExportDlg*)param;
	
	if (!pthis->bLockNodeSelect)
		pthis->SelChange();
}

CStr CameraExportDlg::GetSaveFile()
{ FUNC_ENTER("CameraExportDlg::GetSaveFile"); 
	OPENFILENAME ofn;
	char filename[256]="";

	// Fill out the arduously long openfilename struct
	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=hwnd;
	ofn.hInstance=hInstance;
	ofn.lpstrFilter="Animation Files (*.ska)\0*.ska\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=NULL;
	ofn.nMaxFileTitle=0;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle="Enter Filename to save camera data as";
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt="*.ska";
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetSaveFileName(&ofn);
	return CStr(filename);
}

void CameraExportDlg::GetCamData()
{ FUNC_ENTER("CameraExportDlg::GetCamData"); 
	genExporter->ClearSearchKeys();
	genExporter->ClearExportData();

	SearchKey skey;
	skey.cid=Class_ID(SIMPLE_CAM_CLASS_ID,0);
	skey.fps=60;
	skey.conttype=CONTTYPE_FLOAT;
	skey.animStr="FOV";
	skey.parmStr="";

	genExporter->AddSearchKey(skey);
	skey.cid=Class_ID(LOOKAT_CAM_CLASS_ID,0);
	genExporter->AddSearchKey(skey);
	//skey.cid=Class_ID(TARGET_CLASS_ID,0);
	//skey.animStr="";
	//genExporter->AddSearchKey(skey);

	Tab<ExportInfo>* expData=genExporter->Export();

	bLockSelChange=true;
	camList->Clear();

	if (expData)
	{
		int i;
		int count=expData->Count();

		for(i=0;i<count;i++)
		{
			int index=-1;
			ExportInfo& data=(*expData)[i];

			CamInfo newCamInfo;
			newCamInfo.expdata=&(*expData)[i];
			Link<CamInfo>* camlink=camDB.Add(&newCamInfo);

			// Acquire the camera name and range
			CamSpecificExportSettings* settings;
			AppDataChunk* appdata;
			appdata=data.node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID);

			if (appdata)
			{
				char strStart[256],strEnd[256];

				settings=(CamSpecificExportSettings*)appdata->data;
				_itoa(settings->start,strStart,10);
				_itoa(settings->end,strEnd,10);

				Interval interval=ip->GetAnimRange();

				if (strlen(strStart)==0)
					_itoa(interval.Start()*GetTicksPerFrame(),strStart,10);

				if (strlen(strEnd)==0)
					_itoa(interval.End()*GetTicksPerFrame(),strEnd,10);

				CStr name=data.node->GetName();

				// Only MAX name is allowed for camera now
				strcpy(settings->strUserName, data.node->GetName());
				index=camList->AddItem(settings->strUserName, COLID_CAMNAME);
				//camList->AddSubItem(data.node->GetName(), COLID_MAXNAME);
				camList->AddSubItem(strStart, COLID_START);
				camList->AddSubItem(strEnd, COLID_END);
			}
			else
			{
				char strStart[256],strEnd[256];

				Interval interval=ip->GetAnimRange();

				_itoa(interval.Start()/GetTicksPerFrame(),strStart,10);
				_itoa(interval.End()/GetTicksPerFrame(),strEnd,10);

				CStr name=data.node->GetName();

				index=camList->AddItem(data.node->GetName(), COLID_CAMNAME);
				//camList->AddSubItem(data.node->GetName(), COLID_MAXNAME);
				camList->AddSubItem(strStart,2, COLID_START);
				camList->AddSubItem(strEnd,3, COLID_END);

				SetDlgItemText(hwnd,IDC_EDITNAME,(char*)data.node->GetName());
			}


			camList->SetItemData(index,(DWORD)camlink);
		}
	}

	camList->Sort(Sort);
	LoadCamSelection();
	bLockSelChange=false;
}

void CameraExportDlg::SaveCamSelection()
{ FUNC_ENTER("CameraExportDlg::SaveCamSelection"); 
	// Go through the list and mark the selection state of all cameras
	int size=camList->GetCount();

	for(int i=0;i<size;i++)
	{
		Link<CamInfo>* camLink=(Link<CamInfo>*)camList->GetItemData(i);

		if (!camLink || !camLink->data.expdata)
			return;

		INode* node=camLink->data.expdata->node;

		node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_SELECTED_ID);
		bool* pSel=(bool*)malloc(sizeof(bool));

		*pSel=camList->IsSelected(i);
		node->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_SELECTED_ID,sizeof(bool),pSel);
	}
}

void CameraExportDlg::LoadCamSelection()
{ FUNC_ENTER("CameraExportDlg::LoadCamSelection"); 
	// Go through the list and mark the selection state of all cameras
	int size=camList->GetCount();

	for(int i=0;i<size;i++)
	{
		Link<CamInfo>* camLink=(Link<CamInfo>*)camList->GetItemData(i);

		if (!camLink || !camLink->data.expdata)
			return;

		INode* node=camLink->data.expdata->node;

		AppDataChunk* appdata;
		appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_SELECTED_ID);

		if (appdata && (*((bool*)appdata->data))==true)
			camList->Select(i);
		else
			camList->UnSelect(i);
	}
}

void CameraExportDlg::CancelCB(void* pData)
{ FUNC_ENTER("CameraExportDlg::CancelCB"); 
	CameraExportDlg* pthis=(CameraExportDlg*)pData;
	pthis->bAbort=true;
}

bool CameraExportDlg::DoExport()
{ FUNC_ENTER("CameraExportDlg::DoExport"); 
	MessageBoxReset("Export Frames Invalid");

	int count  = camList->GetCount();
	FILE* fpQN = NULL;

	Apply();

	// Get UI Settings //////////////////////////////////////////////////////
	bool bCompress, bOneFrame, bDebug, bCompressedTime;

	if (IsDlgButtonChecked(hwnd,IDC_USECOMPRESSION)==BST_CHECKED)
		bCompress=true;
	else
		bCompress=false;

	if (IsDlgButtonChecked(hwnd,IDC_COMPDBLS)==BST_CHECKED)
		bOneFrame=true;
	else
		bOneFrame=false;

	if (IsDlgButtonChecked(hwnd,IDC_OUTPUTDBG)==BST_CHECKED)
		bDebug=true;
	else
		bDebug=false;

//	if (IsDlgButtonChecked(hwnd,IDC_COMPRESSEDTIME)==BST_CHECKED)
		// Always on from now on
		bCompressedTime=true;
//	else
//		bCompressedTime=false;

	float errorQ=editQErr->GetFloat();
	float errorT=editTErr->GetFloat();
	float errorF=editFErr->GetFloat();

	/////////////////////////////////////////////////////////////////////////

	bAbort=false;

	progBar=new ProgressBar(hInstance,hwnd,"Exporting Cameras...",0,count);
	progBar->SetCancelCB(CancelCB,this);
	progBar->AllowCancel(TRUE);

	// Build path
	CStr strSceneName=GetSceneName();
	char bufPrefix[256];
	char bufName[256];
	CStr strFile;
	CStr strPath=getenv(APP_ENV);
	strPath+=CStr(CAM_PATH)+strSceneName;

	// Attempt to create the directory (incase it doesn't exist)
	_mkdir(strPath);

	/*
	if (IsDlgButtonChecked(hwnd,IDC_OUTPUTQN)==BST_CHECKED)
	{
		CStr strQNFile=strPath+CStr("\\Cam")+strSceneName+CStr(".qn");

		fpQN=fopen((char*)strQNFile,"w");

		fprintf(fpQN,"script LoadCameras\n");

		if (!fpQN)
		{
			char strErr[256];
			sprintf(strErr,"Failed to open '%s' for output.\nOperation Aborted,",(char*)strQNFile);

			MessageBox(hwnd,strErr,"Failed to output .qn file",MB_ICONWARNING|MB_OK);
			return false;
		}
	}
	*/

	strPath+="\\";
	GetDlgItemText(hwnd,IDC_PREFIX,bufPrefix,255);
	strPath+=bufPrefix;

	if (count==0 && IsDlgButtonChecked(hwnd,IDC_EXPSEL)==BST_CHECKED)
	{
		delete progBar;

		if (fpQN)
			fclose(fpQN);

		MessageBox(hwnd,"No camera was selected to export.","Export camera",MB_ICONWARNING|MB_OK);
		return false;
	}

	for(int i=0;i<count;i++)
	{
		camList->GetItem(i,bufName,255,0);

		strFile=strPath+CStr(bufName)+CStr(".ska");

		char strProg[256];
		sprintf(strProg,"Exporting %s...",(char*)strFile);

		progBar->SetCaption(strProg);
		progBar->SetVal(i);

		// Get start and end range for animation
		char strStart[256],strEnd[256];

		camList->GetItem(i,strStart,255,COLID_START);
		camList->GetItem(i,strEnd,255,COLID_END);

		if (atoi(strEnd)<atoi(strStart))
		{
			delete progBar;

			if (fpQN)
				fclose(fpQN);

			MessageBoxAll(hwnd,"Your end export frame is less than your start export frame.\nOperation Aborted.","Export Frames Invalid",MB_ICONWARNING|MB_OK);
			return false;
		}

		if (IsDlgButtonChecked(hwnd,IDC_EXPSEL)==BST_CHECKED)
		{
			// Export only selected
			if (camList->IsSelected(i))
			{
				CamInfo* pCamInfo=&(((Link<CamInfo>*)camList->GetItemData(i))->data);

				if (!Export(strFile,pCamInfo,atoi(strStart),atoi(strEnd),
					        errorQ,errorT,errorF,
							bCompress,bOneFrame,bDebug,bCompressedTime))
				{
					if (fpQN)
						fclose(fpQN);

					delete progBar;
					return false;
				}

				// Send Network update message to game
				if (IsDlgButtonChecked(hwnd,IDC_SENDNET)==BST_CHECKED)
					SendNetMessage(i);

				BuildTargetFile(i);

				if (fpQN)
					fprintf(fpQN,"\tLoadAsset \"%s\" %s\n",(char*)strFile,bufName);
			}
		}
		else
		{
			// Export all
			CamInfo* pCamInfo=&(((Link<CamInfo>*)camList->GetItemData(i))->data);

			if (!Export(strFile,pCamInfo,atoi(strStart),atoi(strEnd),
				        errorQ,errorT,errorF,
						bCompress,bOneFrame,bDebug,bCompressedTime))
			{
				if (fpQN)
				{
					fprintf(fpQN,"endscript\n");
					fclose(fpQN);
				}

				delete progBar;
				return false;
			}

			// Send Network update message to game
			if (IsDlgButtonChecked(hwnd,IDC_SENDNET)==BST_CHECKED)
				SendNetMessage(i);

			BuildTargetFile(i);

			if (fpQN)
				fprintf(fpQN,"\tLoadAsset \"%s\" %s\n",(char*)strFile,bufName);
		}

		// Check for abort
		if (bAbort)
		{
			if (fpQN)
			{
				fprintf(fpQN,"endscript\n");
				fclose(fpQN);
			}

			delete progBar;
			return false;
		}
	}

	if (fpQN)
	{
		fprintf(fpQN,"endscript\n");
		fclose(fpQN);
	}

	// Perform cleanup if appropriate
	if (IsDlgButtonChecked(hwnd,IDC_CLEANUP)==BST_CHECKED)
		Cleanup(strPath);

	// Perform archiving if appropriate
	if (IsDlgButtonChecked(hwnd,IDC_ARCHIVE)==BST_CHECKED)
		Archive(strPath);

	delete progBar;

	// Dump out a .q file for the cameras with keyframes
	// so we can do a .qbr update
	theCameraExporter.DoQBRUpdate();

	return true;
}

bool CameraExportDlg::CameraExists(char* name)
{ FUNC_ENTER("CameraExportDlg::CameraExists"); 
	CStr strName = name;
	strName.toLower();

	strName = ReplaceStr(strName,".ska","");

	int count = camList->GetCount();

	for(int i=-0;i<count;i++)
	{
		char bufName[256];
		camList->GetItem(i,bufName,255,0);

		CStr strBufName = bufName;
		strBufName.toLower();

		if (strBufName == strName)
			return true;
	}

	return false;
}

void CameraExportDlg::Cleanup(char* path)
{ FUNC_ENTER("CameraExportDlg::Cleanup"); 
	char curdir[1024];
	_getcwd(curdir, 1023);
	_chdir(path);

	_finddata_t fileinfo;

	long fhandle = _findfirst("*.ska",&fileinfo);

	if (!CameraExists(fileinfo.name))
		DeleteFile(fileinfo.name);

	while(_findnext(fhandle,&fileinfo)==0)
	{
		if (!CameraExists(fileinfo.name))
			DeleteFile(fileinfo.name);
	}

	_findclose(fhandle);
	_chdir(curdir);
}

void CameraExportDlg::Archive(char* path)
{ FUNC_ENTER("CameraExportDlg::Archive"); 
	char curdir[1024];
	_getcwd(curdir, 1023);
	_chdir(path);

	char zpath[256];
	CStr strZipName;
	
	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	strZipName  = seo.m_SceneName;
	strZipName += "Cams.zip";

	char* args[5];

	DeleteFile(strZipName);

	sprintf( zpath, "%s..\\..\\..\\bin\\win32\\pkzip25.exe", path );
	args[0] = zpath;
	args[1] = "-add";
	args[2] = (char*)strZipName;
	//args[3] = "*.ska";
	args[4] = NULL;

	// Launch pkzip25 to archive the .ska files in the
	_finddata_t fileinfo;

	long fhandle = _findfirst("*.ska",&fileinfo);

	if (CameraExists(fileinfo.name))
	{
		args[3] = fileinfo.name;
		//spawnvp( _P_WAIT, args[0], args );
		ExecuteCommand( args );
	}

	while(_findnext(fhandle,&fileinfo)==0)
	{
		if (CameraExists(fileinfo.name))
		{
			args[3] = fileinfo.name;
			//spawnvp( _P_WAIT, args[0], args );
			ExecuteCommand( args );
		}
	}

	_chdir(curdir);
}

void CameraExportDlg::SelChange(bool bNoApply)
{ FUNC_ENTER("CameraExportDlg::SelChange"); 
	SetFocus(hwnd);
	UpdateScript();

	if (bLockSelChange)
		return;

	if (!bNoApply)
		Apply();
	
	bLockCamList=true;

	// Unselect everything in the camera list
	camList->UnSelectAll();

	int numSelNodes=ip->GetSelNodeCount();

	for(int i=0;i<numSelNodes;i++)
	{
		INode* node=ip->GetSelNode(i);

		Object* obj=node->EvalWorldState(0).obj;

		if (obj->ClassID()==Class_ID(SIMPLE_CAM_CLASS_ID,0) ||
			obj->ClassID()==Class_ID(LOOKAT_CAM_CLASS_ID,0))
		{
			// Highlight the selected camera
			// Scan through the listview window to find our node

			int count=camList->GetCount();
			for(int i=0;i<count;i++)
			{
				Link<CamInfo>* link=(Link<CamInfo>*)camList->GetItemData(i);

				if (link->data.expdata->node==node)
				{
					camList->Select(i);
					lastCamNode=node;
					break;
				}
			}
		}
	}

	AcquireSelSet();
	if (selSet.Count()==1)
		LoadFromNode(selSet[0]);
	else
	{
		scriptDB.Clear();
		lview->Clear();
		scriptEditor->SetText("");
	}

	bLockCamList=false;
}

void CameraExportDlg::CamSelChange(NMLISTVIEW* pNotify)
{ FUNC_ENTER("CameraExportDlg::CamSelChange"); 
	UpdateScript();

	if (bLockSelChange)
		return;

	if (pNotify->uChanged & LVIF_STATE)
	{
		Apply();
		SetFocus(hwnd);
		bLockNodeSelect=true;

		// Select/Unselect camera that changed highlighted in the list
		Link<CamInfo>* link;
		link=(Link<CamInfo>*)camList->GetItemData(pNotify->iItem);

		if (camList->IsSelected(pNotify->iItem))
			ip->SelectNode(link->data.expdata->node,0);
		else
			ip->DeSelectNode(link->data.expdata->node);

		ip->ForceCompleteRedraw();

		AcquireSelSet();
		if (selSet.Count()==1)
			LoadFromNode(selSet[0]);
		else
		{
			scriptDB.Clear();
			lview->Clear();
			scriptEditor->SetText("");
		}

		// Send a network update message if necessary
		/*
		if (selSet.Count()==1 && IsDlgButtonChecked(hwnd,IDC_SENDNET)==BST_CHECKED)
		{
			Net::MsgViewObjSetCamAnimFile msg;
			char bufName[256];
			char bufPrefix[256];
			GetDlgItemText(hwnd,IDC_PREFIX,bufPrefix,255);

			camList->GetItem(pNotify->iItem,bufName,255);

			CStr sceneName = GetSceneName();
			CStr fileName  = CStr(CAMREL_PATH)+sceneName+CStr("\\")+CStr(bufPrefix)+CStr(bufName)+".ska";
			CStr absFileName = CStr(getenv(APP_ENV)) + CStr("\\data\\")+fileName;

			// First verify that the animation actually exists
			FILE* fp;
			fp = fopen(absFileName,"rb");

			if (!fp)
			{
				int bResult = MessageBox(gInterface->GetMAXHWnd(),"The selected camera doesn't exist.  Would you like to export it?","Couldn't send network update",MB_ICONQUESTION|MB_YESNO);

				if (bResult == IDYES)
				{
					ExportSingleCam(camList->GetSelItem());
				}
				else
				{
					bLockNodeSelect=false;
					return;
				}
			}
			else
				fclose(fp);

			strcpy(msg.m_Filename,(char*)fileName);
			msg.m_checksum=GenerateCRC(bufName);

			gClient->EnqueueMessageToServer( Net::vMSG_ID_VIEWOBJ_SET_CAMANIM_FILE, sizeof( Net::MsgViewObjSetCamAnimFile ),
											 &msg );
		}
		*/
	}

	bLockNodeSelect=false;
}

void CameraExportDlg::AddScript()
{ FUNC_ENTER("CameraExportDlg::AddScript"); 
	if (selSet.Count()>1)
	{
		MessageBox(hwnd,"You may only have one camera selected when adding a script key.","Multiple Cameras Selected",MB_ICONWARNING|MB_OK);
		return;
	}

	Interval interval=ip->GetAnimRange();

	pAddKeyDlg->Show();
	pAddKeyDlg->SetLimits((int)interval.Start()/GetTicksPerFrame(),(int)interval.End()/GetTicksPerFrame());
	pAddKeyDlg->SetScript("");
	pAddKeyDlg->SetScriptCode("");
}

void CameraExportDlg::DeleteScript()
{ FUNC_ENTER("CameraExportDlg::DeleteScript"); 
	int index=lview->GetSel();
	lview->DeleteItem(index);
}

void CameraExportDlg::AddKeyCB(CamAddKeyDlg* pDlg,void* pData)
{ FUNC_ENTER("CameraExportDlg::AddKeyCB"); 
	char buf[256];

	CameraExportDlg* pthis=(CameraExportDlg*)pData;

	_itoa(pDlg->GetTime(),buf,10);

	int index = pthis->lview->AddItem(buf);

	CStr scriptName = pDlg->GetScript();

	if (scriptName == CStr("") ||
		scriptName == CStr(USERSCRIPT_NAME))
	{
		pthis->lview->AddSubItem(USERSCRIPT_NAME, COLID_SCRIPT);

		float timesec=pDlg->GetTimeSec();
		sprintf(buf,"%f",timesec);

		pthis->lview->AddSubItem(buf, COLID_KEYTIME);
		pthis->lview->AddSubItem("N/A", COLID_SCRIPTFILE);

		Link<CStr>* link = pthis->scriptDB.AddToTail(&pDlg->GetScriptCode());
		pthis->lview->SetItemData(index,(DWORD)link);
	}
	else
	{
		pthis->lview->AddSubItem((char*)pDlg->GetScript(), COLID_SCRIPT);

		float timesec=pDlg->GetTimeSec();
		sprintf(buf,"%f",timesec);

		pthis->lview->AddSubItem(buf, COLID_KEYTIME);
		pthis->lview->AddSubItem((char*)pDlg->GetScriptFile(), COLID_SCRIPTFILE);
	}
}

void CameraExportDlg::ListViewUpdate(NMLISTVIEW* pNotify)
{ FUNC_ENTER("CameraExportDlg::ListViewUpdate"); 
	if (pNotify->uNewState & LVIS_SELECTED)
	{
		char buf[256];

		lview->GetItem(pNotify->iItem,buf,255);

		// Set current animation frame
		ip->SetTime(atof(buf)*GetTicksPerFrame());
	}

	// Update the displayed keytime
	char keytime[256];
	lview->GetItem(pNotify->iItem,keytime,255,0);
	bLockKeyList = true;
	spinKeyTime->SetValue(atoi(keytime),TRUE);
	bLockKeyList = false;

	// Update the displayed script
	char filename[1024];
	lview->GetItem(pNotify->iItem,filename,1023,3);

	lview->SetSel(pNotify->iItem);

	if (CStr(filename) == CStr("N/A"))
	{
		Link<CStr>* link = (Link<CStr>*)lview->GetItemData(pNotify->iItem);
		scriptEditor->AllowChanges(TRUE);
		scriptEditor->SetText(link->data);
	}
	else
	{
		char scriptname[1024];
		lview->GetItem(pNotify->iItem,scriptname,1023,1);
		scriptEditor->GoToScriptDefinition(scriptname);
		scriptEditor->AllowChanges(FALSE);
	}
}

void CameraExportDlg::SaveToNode(INode* node)
{ FUNC_ENTER("CameraExportDlg::SaveToNode"); 
	if (bLockLoadSave)
		return;

	bLockLoadSave=true;

	if (!node)
	{
		bLockLoadSave=false;
		return;
	}

	CStr name=node->GetName();

	node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID);
	node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_ID);

	CamSpecificExportSettings* settings=(CamSpecificExportSettings*)malloc(sizeof(CamSpecificExportSettings));

	settings->fRotTolerance=editQErr->GetFloat();
	settings->fTranTolerance=editTErr->GetFloat();
	settings->fFovTolerance=editFErr->GetFloat();
	settings->start=editStart->GetInt();
	settings->end=editEnd->GetInt();
	
	char bufName[256];
	GetDlgItemText(hwnd,IDC_EDITNAME,bufName,255);
	strCamName=bufName;
	strcpy(settings->strUserName,(char*)strCamName);

	node->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID,sizeof(CamSpecificExportSettings),settings);

	CStr strStream;
	char buf[2048];
	char frameNum[256];
	char scriptName[512];
	char scriptFile[1024];
	
	int count=lview->GetCount();

	// Build data stream
	for(int i=0;i<count;i++)
	{
		lview->GetItem(i,frameNum,255);
		lview->GetItem(i,scriptName,511,1);
		lview->GetItem(i,scriptFile,1023,3);

		sprintf(buf,"%s\n%s\n%i\n",scriptName,scriptFile,atoi(frameNum));
		strStream+=buf;

		if (CStr(scriptName)==CStr(USERSCRIPT_NAME))
		{
			Link<CStr>* bufLink = (Link<CStr>*)lview->GetItemData(i);

			char buf2[256];
			_itoa(bufLink->data.Length(),buf2,10);
			strcat(buf2,"\n");

			// Add length to the stream
			strStream+=buf2;
			strStream+=bufLink->data;
		}
	}

	// Allocate the memory for MAX and copy our string into it
	char* newBuf=(char*)malloc(strStream.Length()+1);
	strcpy(newBuf,(char*)strStream);

	node->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_ID,strStream.Length()+1,newBuf);

	bLockLoadSave=false;
}

void CameraExportDlg::LoadFromNode(INode* node)
{ FUNC_ENTER("CameraExportDlg::LoadFromNode"); 
	if (bLockLoadSave)
		return;

	bLockLoadSave=true;

	lview->Clear();
	scriptEditor->SetText("");
	scriptDB.Clear();

	if (!node)
	{
		bLockLoadSave=false;
		return;
	}

	AppDataChunk* appdata;

	CamSpecificExportSettings* settings;
	appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID);

	if (appdata)
		settings=(CamSpecificExportSettings*)appdata->data;
	else
	{
		settings=&camsettings_defaults;
		strcpy(settings->strUserName,(char*)node->GetName());
	}

	spinQErr->SetValue(settings->fRotTolerance,TRUE);
	spinTErr->SetValue(settings->fTranTolerance,TRUE);
	spinFErr->SetValue(settings->fFovTolerance,TRUE);

	spinStart->SetValue(settings->start,TRUE);
	spinEnd->SetValue(settings->end,TRUE);

	strCamName=settings->strUserName;
	SetDlgItemText(hwnd,IDC_EDITNAME,(char*)strCamName);
	SendDlgItemMessage(hwnd,IDC_EDITNAME,EM_SETSEL,(WPARAM)strCamName.Length(),(LPARAM)strCamName.Length());
	//camList->AddSubItem(settings->strUserName,0,lastItem);

	appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_ID);

	if (!appdata)
	{
		bLockLoadSave=false;
		return;
	}

	int   len=appdata->length;
	char* buf=(char*)appdata->data;

	int   frameNum;
	char  sframeNum[256];
	char  scriptName[512];
	char  scriptFile[1024];
	char  frameTime[256];
	int   extlen;				// Length of extended data for attached user scripts
	char  token[1024];
	int   pos = 0;

	while(1)
	{
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		strcpy(scriptName,token);
		
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		strcpy(scriptFile,token);
		
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		frameNum = atoi(token);
		
		//sscanf(buf,"%s\n%s\n%i\n%i\n",scriptName,scriptFile,&frameNum,&extlen);

		// Compute frame time
		Interval animrange = ip->GetAnimRange();
		sprintf(frameTime,"%f",((float)frameNum-(animrange.Start()/GetTicksPerFrame()))/(float)GetFrameRate());
		sprintf(sframeNum,"%i",frameNum);

		// Add this entry to listview
		int index = lview->AddItem(sframeNum);
		lview->AddSubItem(scriptName, COLID_SCRIPT);
		lview->AddSubItem(frameTime, COLID_KEYTIME);
		lview->AddSubItem(scriptFile, COLID_SCRIPTFILE);

		//buf+=Instr(buf,"\n")+1;
		//buf+=Instr(buf,"\n")+1;
		//buf+=Instr(buf,"\n")+1;

		if (CStr(scriptName)==CStr(USERSCRIPT_NAME))
		{
			char* sbuf;
			GetToken(buf,&pos,token,"\n");
			extlen = atoi(token);
			
			sbuf = new char[extlen+1];

			for(int i=0;i<extlen;i++)
				sbuf[i]=buf[pos++];

			sbuf[i]='\0';
			
			Link<CStr>* link = scriptDB.Add(&CStr(sbuf));
			lview->SetItemData(index,(DWORD)link);

			delete sbuf;
		}
	}

	bLockLoadSave=false;
}

void CameraExportDlg::SaveSettings()
{ FUNC_ENTER("CameraExportDlg::SaveSettings"); 
	CamExportSettings* settings=(CamExportSettings*)malloc(sizeof(CamExportSettings));

	ReferenceTarget* scene=ip->GetScenePointer();
	scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_SETTINGS_ID);

	if (IsDlgButtonChecked(hwnd,IDC_OUTPUTDBG)==BST_CHECKED)
		settings->bDebug=true;
	else
		settings->bDebug=false;

	if (IsDlgButtonChecked(hwnd,IDC_USECOMPRESSION)==BST_CHECKED)
		settings->bCompress=true;
	else
		settings->bCompress=false;

	if (IsDlgButtonChecked(hwnd,IDC_COMPDBLS)==BST_CHECKED)
		settings->bNoDbls=true;
	else
		settings->bNoDbls=false;

	if (IsDlgButtonChecked(hwnd,IDC_OUTPUTQN)==BST_CHECKED)
		settings->bExportQN=true;
	else
		settings->bExportQN=false;

	if (IsDlgButtonChecked(hwnd,IDC_EXPALL)==BST_CHECKED)
		settings->bExportAll=true;
	else
		settings->bExportAll=false;

//	if (IsDlgButtonChecked(hwnd,IDC_COMPRESSEDTIME)==BST_CHECKED)
		// Always on from now on
		settings->bCompressTime=true;
//	else
//		settings->bCompressTime=false;

	if (IsDlgButtonChecked(hwnd,IDC_SENDNET) == BST_CHECKED)
		settings->bNetUpdate = true;
	else
		settings->bNetUpdate = false;

	if (IsDlgButtonChecked(hwnd,IDC_CLEANUP)==BST_CHECKED)
		settings->bCleanup=true;
	else
		settings->bCleanup=false;

	if (IsDlgButtonChecked(hwnd,IDC_ARCHIVE)==BST_CHECKED)
		settings->bArchive=true;
	else
		settings->bArchive=false;

	GetDlgItemText(hwnd,IDC_PREFIX,settings->strPrefix,255);

	scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_SETTINGS_ID,sizeof(CamExportSettings),settings);

	SaveCamSelection();
}

void CameraExportDlg::LoadSettings()
{ FUNC_ENTER("CameraExportDlg::LoadSettings"); 
	CamExportSettings* settings;
	AppDataChunk* appdata;

	ReferenceTarget* scene=ip->GetScenePointer();

	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_SETTINGS_ID);
	
	if (appdata)
	{
		settings=(CamExportSettings*)appdata->data;

		if (settings->bDebug)
			CheckDlgButton(hwnd,IDC_OUTPUTDBG,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_OUTPUTDBG,BST_UNCHECKED);

		if (settings->bCompress)
			CheckDlgButton(hwnd,IDC_USECOMPRESSION,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_USECOMPRESSION,BST_UNCHECKED);

		if (settings->bNoDbls)
			CheckDlgButton(hwnd,IDC_COMPDBLS,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_COMPDBLS,BST_UNCHECKED);

		if (settings->bExportQN)
			CheckDlgButton(hwnd,IDC_OUTPUTQN,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_OUTPUTQN,BST_UNCHECKED);

//		if (settings->bCompressTime)
//			CheckDlgButton(hwnd,IDC_COMPRESSEDTIME,BST_CHECKED);
//		else
//			CheckDlgButton(hwnd,IDC_COMPRESSEDTIME,BST_UNCHECKED);

		SetDlgItemText(hwnd,IDC_PREFIX,settings->strPrefix);

		if (settings->bNetUpdate)
			CheckDlgButton(hwnd,IDC_SENDNET,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_SENDNET,BST_UNCHECKED);

		if (settings->bExportAll)
			CheckRadioButton(hwnd,IDC_EXPSEL,IDC_EXPALL,IDC_EXPALL);
		else
			CheckRadioButton(hwnd,IDC_EXPSEL,IDC_EXPALL,IDC_EXPSEL);

		if (settings->bCleanup)
			CheckDlgButton(hwnd,IDC_CLEANUP,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_CLEANUP,BST_UNCHECKED);

		if (settings->bArchive)
			CheckDlgButton(hwnd,IDC_ARCHIVE,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_ARCHIVE,BST_UNCHECKED);
	}
	else
	{
		CheckDlgButton(hwnd,IDC_OUTPUTDBG,BST_UNCHECKED);
		CheckDlgButton(hwnd,IDC_USECOMPRESSION,BST_UNCHECKED);
		CheckDlgButton(hwnd,IDC_COMPDBLS,BST_UNCHECKED);
		CheckDlgButton(hwnd,IDC_OUTPUTQN,BST_CHECKED);		// Output QN set as default
		CheckRadioButton(hwnd,IDC_EXPSEL,IDC_EXPALL,IDC_EXPALL);
		
		CheckDlgButton(hwnd,IDC_CLEANUP,BST_UNCHECKED);
		CheckDlgButton(hwnd,IDC_ARCHIVE,BST_UNCHECKED);
	}
}

void CameraExportDlg::AcquireSelSet()
{ FUNC_ENTER("CameraExportDlg::AcquireSelSet"); 
	int numNodes=ip->GetSelNodeCount();

	selSet.ZeroCount();

	for(int i=0;i<numNodes;i++)
	{
		INode* node=ip->GetSelNode(i);

		Object* obj=node->EvalWorldState(0).obj;

		if (obj->ClassID()==Class_ID(SIMPLE_CAM_CLASS_ID,0) ||
			obj->ClassID()==Class_ID(LOOKAT_CAM_CLASS_ID,0))
		{
			selSet.Append(1,&node);
		}
	}
}

void CameraExportDlg::Apply()
{ FUNC_ENTER("CameraExportDlg::Apply"); 
	int count=selSet.Count();

	if (count>1)
	{
		scriptDB.Clear();
		lview->Clear();
		return;
	}

	for(int i=0;i<count;i++)
	{
		// Apply the listview contents (scripted keys) to each node
		SaveToNode(selSet[i]);
	}
}

void CameraExportDlg::GetRange()
{ FUNC_ENTER("CameraExportDlg::GetRange"); 
	Interval interval=ip->GetAnimRange();
	spinStart->SetLimits(interval.Start()/GetTicksPerFrame(),interval.End()/GetTicksPerFrame());
	spinEnd->SetLimits(interval.Start()/GetTicksPerFrame(),interval.End()/GetTicksPerFrame());
}

void CameraExportDlg::UpdateRange()
{ FUNC_ENTER("CameraExportDlg::UpdateRange"); 
	// Determine current selection
	int count=camList->GetCount();
	int index;
	int selCount=0;

	for(int i=0;i<count;i++)
	{
		if (camList->IsSelected(i))
		{
			index=i;
			selCount++;
		}
	}

	if (selCount>1)
	{
		MessageBox(hwnd,"You can only update the frame range of one camera at a time.","Multiple Cameras Selected",MB_ICONWARNING|MB_OK);
		return;
	}

	char strStart[256];
	char strEnd[256];

	int start=editStart->GetInt();
	int end=editEnd->GetInt();

	_itoa(start,strStart,10);
	_itoa(end,strEnd,10);

	camList->AddSubItem(strStart, COLID_START, index);
	camList->AddSubItem(strEnd, COLID_END, index);
}

void CameraExportDlg::UpdateLabel(NMLVDISPINFO* dispinfo)
{ FUNC_ENTER("CameraExportDlg::UpdateLabel"); 
	return;		// Disabled now that game camera renaming is removed
	
	int index=dispinfo->item.iItem;

	if (dispinfo->item.pszText)
	{
		camList->AddSubItem(dispinfo->item.pszText, COLID_CAMNAME, index);
		strCamName=dispinfo->item.pszText;
		lastItem=index;
		SetDlgItemText(hwnd,IDC_EDITNAME,dispinfo->item.pszText);
	}
}

void CameraExportDlg::UpdateName()
{ FUNC_ENTER("CameraExportDlg::UpdateName"); 
	return;		// Disabled now that game camera renaming is removed

	int index=camList->GetSelItem();
	char buf[256];
	GetDlgItemText(hwnd,IDC_EDITNAME,buf,255);

	camList->AddSubItem(buf, COLID_CAMNAME,index);
	int buflen=strlen(buf);
}

int CameraExportDlg::Sort(DWORD dw1,DWORD dw2,ListView* pList,void* pExtData)
{ FUNC_ENTER("CameraExportDlg::Sort"); 
	Link<CamInfo>* camlink1=(Link<CamInfo>*)dw1;
	Link<CamInfo>* camlink2=(Link<CamInfo>*)dw2;

	if (!camlink1 || !camlink2)
		return 0;

	INode* node1=camlink1->data.expdata->node;
	INode* node2=camlink2->data.expdata->node;

	AppDataChunk *appdata1, *appdata2;

	appdata1=node1->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID);
	appdata2=node2->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID);

	if (!appdata1 || !appdata2)
		return 0;
	
	CamSpecificExportSettings* settings1=(CamSpecificExportSettings*)appdata1->data;
	CamSpecificExportSettings* settings2=(CamSpecificExportSettings*)appdata2->data;
	
	CStr strName1=settings1->strUserName;
	CStr strName2=settings2->strUserName;

	// Switch both to lower case so sorting is case-insensitive
	strName1.toLower();
	strName2.toLower();

	if (strName1<strName2)
		return -1;

	if (strName1==strName2)
		return 0;

	return 1;
}

void CameraExportDlg::ResortList()
{ FUNC_ENTER("CameraExportDlg::ResortList"); 
	Apply();
	camList->Sort(Sort);
}

bool CameraExportDlg::ExportSingleCam(int i)
{ FUNC_ENTER("CameraExportDlg::ExportSingleCam"); 
	FILE* fpQN = NULL;

	if (i==-1)
		return false;

	Apply();

	// Get UI Settings //////////////////////////////////////////////////////
	bool bCompress, bOneFrame, bDebug, bCompressedTime;

	if (IsDlgButtonChecked(hwnd,IDC_USECOMPRESSION)==BST_CHECKED)
		bCompress=true;
	else
		bCompress=false;

	if (IsDlgButtonChecked(hwnd,IDC_COMPDBLS)==BST_CHECKED)
		bOneFrame=true;
	else
		bOneFrame=false;

	if (IsDlgButtonChecked(hwnd,IDC_OUTPUTDBG)==BST_CHECKED)
		bDebug=true;
	else
		bDebug=false;

//	if (IsDlgButtonChecked(hwnd,IDC_COMPRESSEDTIME)==BST_CHECKED)
		// Always on from now on
		bCompressedTime=true;
//	else
//		bCompressedTime=false;

	float errorQ=editQErr->GetFloat();
	float errorT=editTErr->GetFloat();
	float errorF=editFErr->GetFloat();

	/////////////////////////////////////////////////////////////////////////

	bAbort=false;

	// Build path
	CStr strSceneName=GetSceneName();
	char bufPrefix[256];
	char bufName[256];
	CStr strFile;
	CStr strPath=getenv(APP_ENV);
	strPath+=CStr(CAM_PATH)+strSceneName;

	// Attempt to create the directory (incase it doesn't exist)
	_mkdir(strPath);

	strPath+="\\";
	GetDlgItemText(hwnd,IDC_PREFIX,bufPrefix,255);
	strPath+=bufPrefix;

	camList->GetItem(i,bufName,255,0);

	strFile=strPath+CStr(bufName)+CStr(".ska");

	// Get start and end range for animation
	char strStart[256],strEnd[256];

	camList->GetItem(i,strStart,255,COLID_START);
	camList->GetItem(i,strEnd,255,COLID_END);

	if (atoi(strEnd)<atoi(strStart))
	{
		delete progBar;

		if (fpQN)
			fclose(fpQN);

		MessageBox(hwnd,"Your end export frame is less than your start export frame.\nOperation Aborted.","Export Frames Invalid",MB_ICONWARNING|MB_OK);
		return false;
	}

	CamInfo* pCamInfo=&(((Link<CamInfo>*)camList->GetItemData(i))->data);

	if (!Export(strFile,pCamInfo,atoi(strStart),atoi(strEnd),
			    errorQ,errorT,errorF,
				bCompress,bOneFrame,bDebug,bCompressedTime))
	{
		if (fpQN)
			fclose(fpQN);

		delete progBar;
		return false;
	}

	if (fpQN)
		fprintf(fpQN,"\tLoadAsset \"%s\" %s\n",(char*)strFile,bufName);

	return true;
}

void CameraExportDlg::SendNetMessage(int i)
{ FUNC_ENTER("CameraExportDlg::SendNetMessage"); 
	bLockNodeSelect = true;

	if (i == -1)
		i = camList->GetSelItem();

	Net::MsgViewObjSetCamAnimFile msg;
	char bufName[256];
	char bufPrefix[256];
	GetDlgItemText(hwnd,IDC_PREFIX,bufPrefix,255);

	camList->GetItem(i,bufName,255);

	CStr sceneName = GetSceneName();
	CStr fileName  = CStr(CAMREL_PATH)+sceneName+CStr("\\")+CStr(bufPrefix)+CStr(bufName)+".ska";
	CStr absFileName = CStr(getenv(APP_ENV)) + CStr("\\data\\")+fileName;

	// First verify that the animation actually exists
	FILE* fp;
	fp = fopen(absFileName,"rb");

	if (!fp)
	{
		int bResult = MessageBox(gInterface->GetMAXHWnd(),"The selected camera doesn't exist.  Would you like to export it?","Couldn't send network update",MB_ICONQUESTION|MB_YESNO);

		if (bResult == IDYES)
		{
			ExportSingleCam(camList->GetSelItem());
		}
		else
		{
			bLockNodeSelect=false;
			return;
		}
	}
	else
		fclose(fp);

	strcpy(msg.m_Filename,(char*)fileName);
	msg.m_checksum=GenerateCRC(bufName);

	//gClient->EnqueueMessageToServer( Net::vMSG_ID_VIEWOBJ_SET_CAMANIM_FILE, sizeof( Net::MsgViewObjSetCamAnimFile ), &msg,
	//								 Net::NORMAL_PRIORITY, 	Net::QUEUE_SEQUENCED );

	Net::MsgDesc msgdesc;
	msgdesc.m_Id       = Net::vMSG_ID_VIEWOBJ_SET_CAMANIM_FILE;
	msgdesc.m_Length   = sizeof( Net::MsgViewObjSetCamAnimFile );
	msgdesc.m_Data     = &msg;
	msgdesc.m_Priority = Net::NORMAL_PRIORITY;
	msgdesc.m_Queue    = Net::QUEUE_DEFAULT;
	msgdesc.m_GroupId  = Net::GROUP_ID_DEFAULT;
	msgdesc.m_Singular = false;
	msgdesc.m_Delay    = i * 250;

	gClient->EnqueueMessageToServer( &msgdesc );
	
	bLockNodeSelect = false;
}

void CameraExportDlg::BuildTargetFile(int i)
{ FUNC_ENTER("CameraExportDlg::BuildTargetFile"); 
	bLockNodeSelect = true;

	if (i == -1)
		i = camList->GetSelItem();

	char bufName[256];
	char bufPrefix[256];
	GetDlgItemText(hwnd,IDC_PREFIX,bufPrefix,255);

	camList->GetItem(i,bufName,255);

	CStr sceneName = GetSceneName();
	CStr fileName  = CStr(CAMREL_PATH)+sceneName+CStr("\\")+CStr(bufPrefix)+CStr(bufName)+".ska";
	CStr absFileName = CStr(getenv(APP_ENV)) + CStr("\\data\\")+fileName;

	// First verify that the animation actually exists
	FILE* fp;
	fp = fopen(absFileName,"rb");

	if (!fp)
	{
		int bResult = MessageBox(gInterface->GetMAXHWnd(),"The selected camera doesn't exist.  Would you like to export it?","Couldn't send network update",MB_ICONQUESTION|MB_YESNO);

		if (bResult == IDYES)
		{
			ExportSingleCam(camList->GetSelItem());
		}
		else
		{
			bLockNodeSelect=false;
			return;
		}
	}
	else
		fclose(fp);

	// Execute animation converter
	char  ancPath[256];
	char* args[4];
	CStr  strFileCmd = CStr("-f") + absFileName;
	sprintf( ancPath, "%s\\bin\\win32\\animconv.exe", getenv(APP_ENV) );

	args[0] = ancPath;
	args[1] = strFileCmd;
	switch( GetQuickviewPlatform())
	{
		case vQUICKVIEW_XBOX:
			args[2] = "-px";
			break;
		case vQUICKVIEW_PS2:
			args[2] = "-pp";
			break;
		case vQUICKVIEW_NGC:
			args[2] = "-pg";
			break;
		case vQUICKVIEW_PCVIEW:
			args[2] = "-pc";
			break;
	}
	args[3] = NULL;

	//int result = ( spawnvp( _P_WAIT, ancPath, args ) == 0 );
	int result = ExecuteCommand( args );

	if( GetQuickviewPlatform() == vQUICKVIEW_XBOX )
	{
		char path[_MAX_PATH], copy_path[_MAX_PATH];

		sprintf( path, "%s.xbx", absFileName );
		sprintf( copy_path, "%s\\bin\\win32\\copyx.bat", getenv(APP_ENV) );
		args[0] = copy_path;
		args[1] = path;
		args[2] = NULL;

		//result = spawnv( _P_WAIT, args[0], args );
		result = ExecuteCommand( args );
	}
	else if( GetQuickviewPlatform() == vQUICKVIEW_NGC )
	{
		char path[_MAX_PATH], copy_path[_MAX_PATH];
		char* proj_root, *path_ptr;

		proj_root = getenv( APP_ENV );

		sprintf( path, "%s.ngc", absFileName );
		path_ptr = strstr( strlwr( path ), strlwr( proj_root ));
		if( path_ptr )
		{
			path_ptr += strlen( proj_root );
			path_ptr += strlen( "\\data" );
		}
		sprintf( copy_path, "%s\\ndata%s", path_ptr );		
		CopyFile( path, copy_path, false );
	}
	
	bLockNodeSelect = false;
}

void CameraExportDlg::UpdateKeyTime()
{ FUNC_ENTER("CameraExportDlg::UpdateKeyTime"); 
	if (bLockKeyList)
		return;

	int sel = lview->GetSel();

	if (sel==-1)
		return;

	int val = spinKeyTime->GetIVal();
	char buf[256];
	sprintf(buf,"%i",val);
	lview->AddSubItem(buf,COLID_KEYFRAME,sel);
}

///////////////////// Camera Exporter
CameraExporter::CameraExporter()
{ FUNC_ENTER("CameraExporter::CameraExporter"); 
	genExporter=NULL;
	bSwapCoordSystem=true;		// Swap camera export to game coordinate system
	bNoParentTransform=true;	// Cameras should not be exported with respect to their parents
}

CameraExporter::~CameraExporter()
{ FUNC_ENTER("CameraExporter::~CameraExporter"); 
}

bool CameraExporter::ExportNode(FILE* fpExport,FILE* fpDebug,INode* node,int numExtFrames,
								float errorQ,float errorT, bool bUseCompression, bool bCompressTime, bool bOneFrame, 
								unsigned char* pMemMapFileData,
								unsigned int*  pMemMapFilePos)
{ FUNC_ENTER("CameraExporter::ExportNode"); 
	//VerifyMemory();

	int fps3DS=GetFrameRate();								// Frames per second MAX is set to
	Interval animRange    = gInterface->GetAnimRange();

	int num3DSFrames = end-start;
	int reqFrames    = num3DSFrames*OUTPUT_FPS/fps3DS;		// Number of frames at the game's frame rate
	
	unsigned int wsize = 0;									// Size of last write operation (when storing to mem mapped file)

	if (skeleton)
		delete skeleton;
	skeleton=new CSkeletonData(node,false,true,true);
	rootbone=node;

	//VerifyMemory();

	this->bCompressTime = bCompressTime;

	INode* root=node;
	int numNodes=skeleton->GetCount();

	GetAnim(root,start,end,errorQ,errorT,NULL,bUseCompression);
	
	if (bOneFrame)
		RemoveDblKeys(numNodes,num3DSFrames);

	// Compute total QFrames and TFrames
	int QKeys=0;
	int TKeys=0;
	int i;

	for(i=0;i<numNodes;i++)
	{
		QKeys+=numCompressedQFrames[i];
		TKeys+=numCompressedTFrames[i];		
	}
	
	//VerifyMemory();

	// Output the stored data in THPS4 format
	//CSkeletonData* skeleton=new CSkeletonData(node);

	// Output file header
	SHeader header;
	header.version=ANIMFILE_VERSION;
	header.flags=INTERMEDIATE_FORMAT;
	
	if (!bUseCompression)
		header.flags|=UNCOMPRESSED_FORMAT;

	if (bCompressTime)
		header.flags|=COMPRESSEDTIME;

	// Cameras always had pre rotated root, assign the flag for it
	header.flags|=PREROTATEDROOT;

	header.flags|=CONTAINS_CAMDATA;
	header.flags|=CUSTOMKEYSAT60FPS;

	header.skeletonName=CAMERA_CRC;			// hard-coded default for now
	header.duration=(float)num3DSFrames/(float)fps3DS;
	header.numBones=numNodes;
	header.numQKeys=QKeys;
	header.numTKeys=TKeys;

	header.numUserDefinedKeys=numExtFrames;

	//VerifyMemory();

	if (fpExport)
	{
		if (!header.Write(fpExport))
		{
			delete skeleton;
			skeleton=NULL;
			return false;
		}
	}
	else
	{
		if (!header.Write(pMemMapFileData, &wsize))
		{
			delete skeleton;
			skeleton = NULL;
			return false;
		}

		*pMemMapFilePos += wsize;
	}

	//VerifyMemory();

	// Output Skeleton data

	// Output using CSkeleton instead
	if (fpExport)
	{
		if (!skeleton->Write(fpExport))
		{
			delete skeleton;
			skeleton=NULL;
			return false;
		}
	}
	else
	{
		wsize = skeleton->Write(pMemMapFileData);

		if (!wsize)
		{
			delete skeleton;
			skeleton=NULL;
			return false;
		}

		*pMemMapFilePos += wsize;
	}

	//VerifyMemory();
	///////////////////////////////////////////////////

	// Output compressed rotation data
	SQData sqdata(numNodes,QKeys);
	int QKeyPos=0;

	for(i=0;i<numNodes;i++)
	{
		sqdata.numKeysPerBone[i]=numCompressedQFrames[i];
		
		for(int j=0;j<numCompressedQFrames[i];j++)
		{
			//int curFrame=j+i*reqFrames;
			int curFrame=j+i*num3DSFrames;

			// If the compress time flag is specified we need to multiply time time by the FPS
			// rate to convert it into the frame number as opposed to the time into the animation
			if (bCompressTime)
			{
				// This is still going to get dumped out as a float, so we need to make sure
				// the compiler makes the representation of the float match an int
				// We can do this since they're both 4 bytes
				int* val = (int*)&sqdata.theQKeys[QKeyPos].time;
				float ftmp = (compQframes[curFrame].time * (float)OUTPUT_FPS);
				char buf[256];
				sprintf(buf,"Pre float is: %f\n",ftmp);
				OutputDebugString(buf);
				*val = Round(ftmp);
				
				//if ( (*val % 2) == 1 )
				//	assert(0);
			}
			else
				sqdata.theQKeys[QKeyPos].time = compQframes[curFrame].time;

			sqdata.theQKeys[QKeyPos].qx   = compQframes[curFrame].q[X];
			sqdata.theQKeys[QKeyPos].qy   = compQframes[curFrame].q[Y];
			sqdata.theQKeys[QKeyPos].qz   = compQframes[curFrame].q[Z];
			sqdata.theQKeys[QKeyPos].real = compQframes[curFrame].q[W];

			if (fpDebug)
			{
				if (bCompressTime)
				{
					char sbuf[256];
					sprintf(sbuf,"---> Q[node=%i frame=%i].time=%i\n",i,j,*((int*)&sqdata.theQKeys[QKeyPos].time));
					fprintf(fpDebug,"Q[node=%i frame=%i].time=%i\n",i,j,*((int*)&sqdata.theQKeys[QKeyPos].time));
					OutputDebugString(sbuf);
					
					sprintf(sbuf,"---> actual time: %f\n",compQframes[curFrame].time);
					sprintf(sbuf,"---> new time: %f\n",compQframes[curFrame].time * 60.0f);
					OutputDebugString(sbuf);
				}
				else
					fprintf(fpDebug,"Q[node=%i frame=%i].time=%f\n",i,j,compQframes[curFrame].time);

				fprintf(fpDebug,"Q[node=%i frame=%i].qx=%f\n",i,j,compQframes[curFrame].q[X]);
				fprintf(fpDebug,"Q[node=%i frame=%i].qy=%f\n",i,j,compQframes[curFrame].q[Y]);
				fprintf(fpDebug,"Q[node=%i frame=%i].qz=%f\n",i,j,compQframes[curFrame].q[Z]);
				fprintf(fpDebug,"Q[node=%i frame=%i].real=%f\n\n",i,j,compQframes[curFrame].q[W]);
			}

			QKeyPos++;
		}
	}

	//VerifyMemory();

	if (fpExport)
	{
		if (!sqdata.Write(fpExport))
			return false;
	}
	else
	{
		if (!sqdata.Write(pMemMapFileData, &wsize))
			return false;

		*pMemMapFilePos += wsize;
	}

	// Output compressed translation data
	STData stdata(numNodes,TKeys);
	int TKeyPos=0;

	//VerifyMemory();

	for(i=0;i<numNodes;i++)
	{
		stdata.numKeysPerBone[i]=numCompressedTFrames[i];
		
		for(int j=0;j<numCompressedTFrames[i];j++)
		{
			//int curFrame=j+i*reqFrames;
			int curFrame=j+i*num3DSFrames;

			if (bCompressTime)
			{
				// This is still going to get dumped out as a float, so we need to make sure
				// the compiler makes the representation of the float match an int
				// We can do this since they're both 4 bytes
				int* val = (int*)&stdata.theTKeys[TKeyPos].time;
				float ftmp = compTframes[curFrame].time * (float)OUTPUT_FPS;
				*val = Round(ftmp);

				//if ( *val % 2 == 1 )
				//	assert(0);
			}
			else
				stdata.theTKeys[TKeyPos].time = compTframes[curFrame].time;

			stdata.theTKeys[TKeyPos].tx   = compTframes[curFrame].t[0];
			stdata.theTKeys[TKeyPos].ty   = compTframes[curFrame].t[1];
			stdata.theTKeys[TKeyPos].tz   = compTframes[curFrame].t[2];

			if (fpDebug)
			{
				if (bCompressTime)
					fprintf(fpDebug,"T[node=%i frame=%i].time=%i\n",i,j,*((int*)&stdata.theTKeys[TKeyPos].time));
				else
					fprintf(fpDebug,"T[node=%i frame=%i].time=%f\n",i,j,compTframes[curFrame].time);

				fprintf(fpDebug,"T[node=%i frame=%i].tx=%f\n",i,j,compTframes[curFrame].t[0]);
				fprintf(fpDebug,"T[node=%i frame=%i].ty=%f\n",i,j,compTframes[curFrame].t[1]);
				fprintf(fpDebug,"T[node=%i frame=%i].tz=%f\n\n",i,j,compTframes[curFrame].t[2]);

				char test[256];
				sprintf(test,"T[node=%i frame=%i].time=%f\n",i,j,compTframes[curFrame].time);
				OutputDebugString(test);
				sprintf(test,"T[node=%i frame=%i].tx=%f\n",i,j,compTframes[curFrame].t[0]);
				OutputDebugString(test);
				sprintf(test,"T[node=%i frame=%i].ty=%f\n",i,j,compTframes[curFrame].t[1]);
				OutputDebugString(test);
				sprintf(test,"T[node=%i frame=%i].tz=%f\n\n",i,j,compTframes[curFrame].t[2]);
				OutputDebugString(test);
			}

			TKeyPos++;
		}
	}

	//VerifyMemory();

	if (fpExport)
	{
		if (!stdata.Write(fpExport))
			return false;
	}
	else
	{
		if (!stdata.Write(pMemMapFileData, &wsize))
			return false;

		*pMemMapFilePos += wsize;
	}

	delete skeleton;
	skeleton=NULL;

	//VerifyMemory();
	return true;
}

GenKey<float>* CameraExporter::GetFOVKeys(INode* node, int start, int end, int* numKeys,
										  float error, bool bOneFrame)
{ FUNC_ENTER("CameraExporter::GetFOVKeys"); 
	GenKey<float>* origkeys=NULL;
	GenKey<float>* fovkeys=NULL;
	int            numFOVkeys;

	int            numExtraFOVkeys = 1;	// 26Mar03 JCB - Allow for extra key on final frame.
	float          finalTime;			// 26Mar03 JCB - Preserve time of final frame.

	//CameraObject* camobj = (CameraObject*)node->EvalWorldState(0).obj;

	int      fpsMAX       = GetFrameRate();
	int      numFrames    = (end - start) + 1; // 26Mar03 JCB - Assume that 'end' frame should be played.
	
	// 26Mar03 JCB - This was giving results rounded to the nearest second, now fixed.
	numFOVkeys = (((float) numFrames)/((float) fpsMAX))*60.0f;	// NumReqFrames

	origkeys = new GenKey<float>[numFOVkeys];

	//float startTime    = (float)start / (float)fpsMAX;
	float startTime = 0;	// Relative to start of sequence now
	float ratePerFrame = 60.0f / fpsMAX;

	// Ensure that the object we're exporting is actually a camera
	Object* obj = (Object*)node->EvalWorldState(0).obj;

	if (obj->SuperClassID() == CAMERA_CLASS_ID)
	{
		for(int i = 0; i < numFOVkeys; i++)
		{
			int maxFrame = start + (i * numFrames) / numFOVkeys;

			// Camera seems to need reevaluated at each time to produce accurate results from GetFOV ???
			CameraObject* camobj = (CameraObject*)node->EvalWorldState(GetTicksPerFrame()*maxFrame).obj;

			origkeys[i].key  = camobj->GetFOV(GetTicksPerFrame()*maxFrame);
			origkeys[i].time = (float)i / 60.0f + startTime;		// seconds
			//origkeys[i].time = start * ratePerFrame + i;			// frames
		}
	}
	else
	{
		char buf[256];
		sprintf(buf, "The object '%s' is flagged as a camera but isn't an actual MAX camera.  No FOV data will be exported for this special camera", (char*)node->GetName());
		MessageBox(gInterface->GetMAXHWnd(), buf, "Non Max camera being exported", MB_OK);
		
		// Assign all frames to FOV 30
		for(int i = 0; i < numFOVkeys; i++)
		{
			origkeys[i].key  = 30.0f;
			origkeys[i].time = (float)i / 60.0f + startTime;		// seconds
		}
	}

	finalTime = origkeys[(numFOVkeys - 1)].time; // 26Mar03 JCB - Preserve time of final frame.

	// Compress FOV data
	// Not allowing FOV data to be compressed anymore
	/*
	if (error!=0.0f)
	{
		if (!fovkeys)
		{
			fovkeys=new GenKey<float>[(numFOVkeys+numExtraFOVkeys)]; // 26Mar03 JCB - Allocate extra key.
			numFOVkeys=CompressFloatKeys( fovkeys, origkeys, numFOVkeys, error );
		}
	}
	else
	*/
	{
		fovkeys=new GenKey<float>[(numFOVkeys+numExtraFOVkeys)]; // 26Mar03 JCB - Allocate extra key.

		for(int i=0;i<numFOVkeys;i++)
		{
			fovkeys[i].time = origkeys[i].time;
			fovkeys[i].key  = origkeys[i].key;
		}
	}
	
	// Compress FOV doubles
	/*
	if (bOneFrame)
	{
		bool bMatch = true;

		for(int i = 0; i < numFOVkeys; i++)
		{
			if (fovkeys[0].key != fovkeys[i].key)
			{
				bMatch = false;
				break;
			}
		}

		if (bMatch)
			numFOVkeys = 1;
	}
	*/

	// Get rid of duplicate FOV keys
	if (bOneFrame)
	{
		bool bMatch = true;

		int i, j;
		int cur  = 0;
		int next = 0;
		
		// i represents source key
		// j represents compare key
		for(i = 0; i < numFOVkeys; i++)
		{
			// Add source key
			fovkeys[cur] = fovkeys[i];
			cur++;

			for(j = i + 1; j < numFOVkeys; j++)
			{
				if (fovkeys[i].key != fovkeys[j].key)
				{
					// Key differs add the new key to sequence
					// This is our new source key, it will be added to the sequence
					i = j - 1;
					break;
				}

				i = j;
			}
		}

		numFOVkeys = cur;
	}

	// 01Jul03 JCB - This MUST be done AFTER removing duplicate keys as it DELIBERATELY creates a
	// 01Jul03 JCB - duplicate key on the final frame.
	//
	// 26Mar03 JCB - Add an extra key to the final frame so that playing the camera backwards works.
	//
	// Only if there is at least one key.
	if (numFOVkeys > 0)
	{
		// Only if there isn't already a key on the final frame.
		if (fovkeys[(numFOVkeys-1)].time != finalTime)
		{
			fovkeys[numFOVkeys].time = finalTime;
			fovkeys[numFOVkeys].key  = fovkeys[(numFOVkeys-1)].key;
			numFOVkeys++;
		}
	}

	// Original keys are no longer needed only compressed array is required
	delete [] origkeys;

	if (numKeys)
		*numKeys=numFOVkeys;

	return fovkeys;
}

GenKey<float>* CameraExporter::GetFOVKeys(CamInfo* camInfo,int* numKeys,
										  float error, bool bOneFrame)
{ FUNC_ENTER("CameraExporter::GetFOVKeys"); 
	GenKey<float>* origkeys=NULL;
	GenKey<float>* fovkeys=NULL;
	int            numFOVkeys;

	int            numExtraFOVkeys = 1;	// 26Mar03 JCB - Allow for extra key on final frame.
	float          finalTime;			// 26Mar03 JCB - Preserve time of final frame.

	if (!camInfo || !camInfo->expdata)
	{
		if (numKeys)
			*numKeys = 0;

		return NULL;
	}

	// Check the export data to see if we got any keyframes
	// if we didn't there's no controller and we'll have to grab the one frame manually
	if (camInfo->expdata->exportList[0].numKeys==0)
	{
		INode* node=camInfo->expdata->node;
		CameraObject* camobj=(CameraObject*)node->EvalWorldState(0).obj;

		Interval interval=gInterface->GetAnimRange();

		fovkeys=new GenKey<float>[(2+numExtraFOVkeys)]; // 26Mar03 JCB - Allocate extra key.

		fovkeys[0].time = 0.0f;
		fovkeys[0].key  = camobj->GetFOV(0);

		fovkeys[1].time = (((float) (interval.End() - interval.Start())) / (float) GetTicksPerFrame()) / (float) GetFrameRate();
		fovkeys[1].key  = camobj->GetFOV(0);

		numFOVkeys=2;

		finalTime = fovkeys[(numFOVkeys - 1)].time; // 26Mar03 JCB - Preserve time of final frame.
	}
	else
	{
		/*
		numFOVkeys=camInfo->expdata->exportList[0].numKeys;

		// Convert to proper key sequence
		origkeys=new GenKey<float>[numFOVkeys];
		int curKey=0;
		
		for(int i=0;i<numFOVkeys;i++)
		{
			float keyTime=camInfo->expdata->exportList[0].keys[i].time;
			int   keyFrame=keyTime*GetFrameRate();

			// Only add keys if they're within
			if (keyFrame>=start && keyFrame<=end)
			{
				origkeys[curKey].time=keyTime;
				origkeys[curKey].key =*((float*)camInfo->expdata->exportList[0].keys[i].key);
				curKey++;
			}
		}

		numFOVkeys=curKey;
		*/

		INode* node=camInfo->expdata->node;
		CamSpecificExportSettings* settings;
		AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID);

		if (appdata && appdata->data)
			settings=(CamSpecificExportSettings*)appdata->data;
		else
		{
			if (numKeys)
				*numKeys = 0;

			return NULL;
		}

		CameraObject* camobj = (CameraObject*)node->EvalWorldState(0).obj;

		int      fpsMAX       = GetFrameRate();
		int      numFrames    = (settings->end - settings->start) + 1; // 26Mar03 JCB - Assume that 'end' frame should be played.
		
		// 26Mar03 JCB - This was giving results rounded to the nearest second, now fixed.
		numFOVkeys = (((float) numFrames)/((float) fpsMAX))*60.0f;	// NumReqFrames

		origkeys=new GenKey<float>[(numFOVkeys+numExtraFOVkeys)];

		for(int i = 0; i < numFOVkeys; i++)
		{
			int maxFrame = i * numFrames / numFOVkeys;

			// Camera seems to need reevaluated at each time to produce accurate results from GetFOV ???
			CameraObject* camobj = (CameraObject*)node->EvalWorldState(GetTicksPerFrame()*maxFrame).obj;

			origkeys[i].key  = camobj->GetFOV(GetTicksPerFrame()*maxFrame);
			origkeys[i].time = (float)i/60.0f;
		}

		finalTime = origkeys[(numFOVkeys - 1)].time; // 26Mar03 JCB - Preserve time of final frame.

		// Compress FOV data
		if (error!=0.0f)
		{
			if (!fovkeys)
			{
				fovkeys=new GenKey<float>[(numFOVkeys+numExtraFOVkeys)]; // 26Mar03 JCB - Allocate extra key.
				numFOVkeys=CompressFloatKeys( fovkeys, origkeys, numFOVkeys, error );
			}
		}
		else
		{
			fovkeys=new GenKey<float>[(numFOVkeys+numExtraFOVkeys)]; // 26Mar03 JCB - Allocate extra key.

			for(i=0;i<numFOVkeys;i++)
			{
				fovkeys[i].time=camInfo->expdata->exportList[0].keys[i].time;
				fovkeys[i].key =*((float*)camInfo->expdata->exportList[0].keys[i].key);
			}
		}
	}
	
	// Compress FOV doubles
	if (bOneFrame)
	{
		bool bMatch = true;

		for(int i = 0; i < numFOVkeys; i++)
		{
			if (fovkeys[0].key != fovkeys[i].key)
			{
				bMatch = false;
				break;
			}
		}

		if (bMatch)
			numFOVkeys = 1;
	}

	// 26Mar03 JCB - Add an extra key on the final frame so that playing the camera backwards works.
	//
	// Only if there is at least one key.
	if (numFOVkeys > 0)
	{
		// Only if there isn't already a key on the final frame.
		if (fovkeys[(numFOVkeys-1)].time != finalTime)
		{
			fovkeys[numFOVkeys].time = finalTime;
			fovkeys[numFOVkeys].key  = fovkeys[(numFOVkeys-1)].key;
			numFOVkeys++;
		}
	}

	// Original keys are no longer needed only compressed array is required
	delete [] origkeys;

	if (numKeys)
		*numKeys=numFOVkeys;

	return fovkeys;
}


int CameraExporter::GetKeyCount(char* buf)
{ FUNC_ENTER("CameraExporter::GetKeyCount"); 
	int   len=strlen(buf);

	int   frameNum;
	int   extlen;				// Length of extended data for attached user scripts
	char  token[1024];
	int   pos = 0;
	char  scriptName[1024];
	int   count = 0;

	while(1)
	{
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		strcpy(scriptName,token);
		
		if (!GetToken(buf,&pos,token,"\n"))
			break;
	
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		frameNum = atoi(token);
		
		if (CStr(scriptName)==CStr(USERSCRIPT_NAME))
		{
			GetToken(buf,&pos,token,"\n");
			extlen = atoi(token);		
			pos+=extlen;
		}

		count++;
	}

	return count;
}

GenKey<ScriptKey>* CameraExporter::GetScriptKeys(CamInfo* camInfo,int* numKeys)
{ FUNC_ENTER("CameraExporter::GetScriptKeys"); 
	GenKey<ScriptKey>* scriptKeys;

	// Get the node for the current camera
	if (!camInfo || !camInfo->expdata || !camInfo->expdata->node)
	{
		if (numKeys)
			*numKeys=0;

		return NULL;
	}
	
	INode* node=camInfo->expdata->node;

	AppDataChunk* appdata;
	appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_ID);

	if (!appdata)
	{
		if (numKeys)
			*numKeys=0;

		return NULL;
	}

	int   len=appdata->length;
	char* buf=(char*)appdata->data;

	int   frameNum;
	char  scriptName[512];
	char  scriptFile[1024];
	char  token[1024];

	int   numItems=GetKeyCount(buf);
	int   extlen;
	int   pos = 0;

	if (numKeys)
		*numKeys=numItems;

	if (numItems==0)
		return NULL;

	scriptKeys=new GenKey<ScriptKey>[numItems];

	for(int i=0;i<numItems;i++)
	{
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		strcpy(scriptName,token);
		
		if (!GetToken(buf,&pos,token,"\n"))
			break;
	
		strcpy(scriptFile,token);

		if (!GetToken(buf,&pos,token,"\n"))
			break;

		frameNum = atoi(token);

		// Compute frame time
		Interval animrange = gInterface->GetAnimRange();

		scriptKeys[i].time = (frameNum-(animrange.Start()/GetTicksPerFrame())/(float)GetFrameRate());
		
		scriptKeys[i].key.name   =  scriptName;
		scriptKeys[i].key.file   =  scriptFile;
		scriptKeys[i].key.frame  =  frameNum;

		if (CStr(scriptName)==CStr(USERSCRIPT_NAME))
		{
			GetToken(buf,&pos,token,"\n");
			extlen = atoi(token);		
			pos+=extlen;
		}
	}

	return scriptKeys;
}

bool CameraExporter::Export(char* filename,CamInfo* camInfo,int start,int end,
							float errorQ,float errorT,float errorF,
							bool bCompress,bool bOneFrame, bool bDebug, bool bCompressTime)
{ FUNC_ENTER("CameraExporter::Export"); 
	//FILE *fpExport=NULL,*fpDebug=NULL;
	FILE* fpDebug = NULL;
	
	// Export the camera
	if (camInfo)
	{
		fpExport=fopen(filename,"wb");

		if (!fpExport)
		{
			if (bRemoveReadOnly)
			{
				if (_access(filename,WRITE_ACCESS)!=0)
				{
					if (_chmod(filename, _S_IREAD | _S_IWRITE)==-1)
					{
						MessageBox(NULL,"Couldn't change file attribute setting from read only.","Export Animations",MB_ICONSTOP|MB_OK);
						return false;
					}

					fpExport=fopen(filename,"wb");
				}

				if (!fpExport)
					return false;
			}
			else if( bCheckOut )
			{
				char *args[4];
				int process_result;

				args[0] = "p4";
				args[1] = "edit";
				args[2] = filename;
				args[3] = NULL;
				//process_result = spawnvp( _P_WAIT, args[0], args );
				process_result = ExecuteCommand( args );
				if (_access(filename,WRITE_ACCESS)!=0)
				{
					MessageBox(NULL,"An error occurred while trying to check the file out. Perhaps someone else has it checked out already?","Export Cameras",MB_ICONSTOP|MB_OK);
					return false;
				}
			}
			else
			{
				// Check the file attributes to see if the file is read-only
				if (_access(filename,WRITE_ACCESS)!=0)
				{
					char ErrMsg[256];
					int  rVal;
					sprintf(ErrMsg,"The file '%s' does not have write access enabled.\nWould you like to manually remove the Read-Only flag on all items?\nOr would you like to check this file out?",(char*)filename);
					rVal=MessageBoxAll(NULL,ErrMsg,"Export Animations",MB_CHECKOUT_UNLOCK);

					if (rVal==IDWRITABLE)
					{
						bRemoveReadOnly=true;
						rVal=IDNO;

						if (_chmod(filename, _S_IREAD | _S_IWRITE)==-1)
						{
							MessageBox(NULL,"An error occurred while changing the file's permission settings!","Export Animations",MB_ICONSTOP|MB_OK);
							return false;
						}
					}
					else if( rVal == IDCHECKOUT )
					{
						char *args[4];
						int process_result;

						bCheckOut = true;
						rVal=IDNO;					

						args[0] = "p4";
						args[1] = "edit";
						args[2] = filename;
						args[3] = NULL;
						//process_result = spawnvp( _P_WAIT, args[0], args );
						process_result = ExecuteCommand( args );
						if (_access(filename,WRITE_ACCESS)!=0)
						{
							MessageBox(NULL,"An error occurred while trying to check the file out. Perhaps someone else has it checked out already?","Export Cameras",MB_ICONSTOP|MB_OK);
							return false;
						}
					}
				}
			}

			fpExport=fopen(filename,"wb");

			if (!fpExport)
			{
				char strErr[256];
				sprintf(strErr, "Failed to open '%s' for output.");
				MessageBox(gInterface->GetMAXHWnd(), strErr, "Couldn't open export file", MB_ICONSTOP|MB_OK);
				return false;
			}
		}

		// Acquire the script and FOV keys
		GenKey<float>*     fovkeys;
		GenKey<ScriptKey>* scriptkeys;
		int numFOVkeys;
		int numScriptKeys;
		int numExtKeys;

		this->start = start;
		this->end   = end+1;

		scriptkeys = GetScriptKeys(camInfo,&numScriptKeys);
		fovkeys    = GetFOVKeys(camInfo,&numFOVkeys, false);
		numExtKeys = numFOVkeys+numScriptKeys;

		// Open debug file if requested
		if (bDebug)
			fpDebug=fopen("c:\\debug.txt","w");

		// Export the camera position data
		if (!ExportNode(fpExport,fpDebug,camInfo->expdata->node,numExtKeys,
			            errorQ, errorT, bCompress, bCompressTime, bOneFrame))
		{
			fclose(fpExport);

			if (fovkeys)
				delete [] fovkeys;

			if (scriptkeys)
				delete [] scriptkeys;

			return false;
		}

		if (fpDebug)
			fprintf(fpDebug,"\nFOV Keys:\n");

		// Write out the fov keys
		for(int i=0;i<numFOVkeys;i++)
		{
			// If the compress time flag is specified we need to multiply time time by the FPS
			// rate to convert it into the frame number as opposed to the time into the animation
			if (bCompressTime)
			{
				// This is still going to get dumped out as a float, so we need to make sure
				// the compiler makes the representation of the float match an int
				// We can do this since they're both 4 bytes
				int* val = (int*)&fovkeys[i].time;
				float ftmp = ( fovkeys[i].time * (float)OUTPUT_FPS );
				*val = Round(ftmp);
			}

			if (fpDebug)
			{
				fprintf(fpDebug,"%i\n",i);

				if (bCompressTime)
					fprintf(fpDebug,"Time: %i\n",*((int*)&fovkeys[i].time));
				else
					fprintf(fpDebug,"Time: %f\n",fovkeys[i].time);
				
				fprintf(fpDebug,"Key: %f (Deg: %f)\n",fovkeys[i].key, RadToDeg(fovkeys[i].key));
			}

			if (!ChangeFovKey(fovkeys[i].time,fovkeys[i].key).Write(fpExport))
			{
				if (fovkeys)
					delete [] fovkeys;

				if (scriptkeys)
					delete [] scriptkeys;

				return false;
			}
		}

		// Clear compressed keys now that they've been written
		if (fovkeys)
			delete [] fovkeys;

		if (fpDebug)
			fprintf(fpDebug,"\nScript Keys:\n");

		// Now write out the change script keys
		for(i=0;i<numScriptKeys;i++)
		{
			if (bCompressTime)
			{
				// Script time was always the keyframe
				// represent as int instead of float
				int* val = (int*)&scriptkeys[i].time;
				*val = scriptkeys[i].time;
			}

			if (fpDebug)
			{
				fprintf(fpDebug,"%i\n",i);

				if (bCompressTime)
					fprintf(fpDebug,"Time: %i\n",*((int*)&scriptkeys[i].time));
				else
					fprintf(fpDebug,"Time: %f\n",scriptkeys[i].time);

				fprintf(fpDebug,"Name: %s (0x%x)\n",(char*)scriptkeys[i].key.name,GenerateCRC((char*)scriptkeys[i].key.name));

				char buf[256];
				sprintf(buf,"Time: %f\n",scriptkeys[i].time);
				OutputDebugString(buf);

				sprintf(buf,"Name: %s (0x%x)\n",(char*)scriptkeys[i].key.name,GenerateCRC((char*)scriptkeys[i].key.name));
				OutputDebugString(buf);
			}

			CStr name;

			if (scriptkeys[i].key.name == CStr(USERSCRIPT_NAME))
			{
				char buf[256];
				_splitpath(filename,NULL,NULL,buf,NULL);
				name += buf;
				name += "Frame";
				sprintf(buf,"%i",scriptkeys[i].key.frame);
				name += buf;
			}
			else
			{
				name = scriptkeys[i].key.name;
			}

			unsigned long crc = GenerateCRC((char*)name);

			if (!RunScriptKey(scriptkeys[i].time,GenerateCRC((char*)name)).Write(fpExport))
			{
				if (scriptkeys)
					delete [] scriptkeys;

				return false;
			}
		}

		// Close debug file
		if (fpDebug)
			fclose(fpDebug);

		fclose(fpExport);

		// Free allocated keys
		if (scriptkeys)
			delete [] scriptkeys;

		return true;
	}

	return false;
}

bool CameraExporter::ExportableNodesExist( void )
{ FUNC_ENTER("CameraExporter::ExportableNodesExist"); 
	bool nodes_exist;

	genExporter=new GenExporter(gInterface);
	genExporter->ClearSearchKeys();
	genExporter->ClearExportData();

	SearchKey skey;
	skey.cid=Class_ID(SIMPLE_CAM_CLASS_ID,0);
	skey.fps=60;
	skey.conttype=CONTTYPE_FLOAT;
	skey.animStr="FOV";
	skey.parmStr="";

	genExporter->AddSearchKey(skey);
	skey.cid=Class_ID(LOOKAT_CAM_CLASS_ID,0);
	genExporter->AddSearchKey(skey);

	nodes_exist = genExporter->ExportableNodesExist();
	delete genExporter;
	return nodes_exist;
}

bool CameraExporter::DoExport( ExportType type )
{ FUNC_ENTER("CameraExporter::DoExport"); 
	// Do general export for cameras (so we can go through each one)
	genExporter=new GenExporter(gInterface);
	genExporter->ClearSearchKeys();
	genExporter->ClearExportData();

	SearchKey skey;
	skey.cid=Class_ID(SIMPLE_CAM_CLASS_ID,0);
	skey.fps=60;
	skey.conttype=CONTTYPE_FLOAT;
	skey.animStr="FOV";
	skey.parmStr="";

	genExporter->AddSearchKey(skey);
	skey.cid=Class_ID(LOOKAT_CAM_CLASS_ID,0);
	genExporter->AddSearchKey(skey);

	Tab<ExportInfo>* expData=genExporter->Export();

	if (!expData)
	{
		delete genExporter;
		return false;
	}

	// Assign default settings
	Interval interval=gInterface->GetAnimRange();
	CamSpecificExportSettings  default_settings;

	default_settings.start          = interval.Start()/GetTicksPerFrame();
	default_settings.end            = interval.End()/GetTicksPerFrame();
	default_settings.fRotTolerance  = DEFAULT_COMPRESS;
	default_settings.fTranTolerance = DEFAULT_TCOMPRESS;
	default_settings.fFovTolerance  = DEFAULT_FCOMPRESS;

	CamExportSettings default_exportSettings;

	default_exportSettings.bDebug        = false;
	default_exportSettings.bCompress     = true;
	default_exportSettings.bNoDbls       = true;
	default_exportSettings.bExportQN     = true;
	default_exportSettings.bArchive      = true;
	default_exportSettings.bCleanup      = false;
	default_exportSettings.bCompressTime = true;
	//default_exportSettings.bExportAll    = true;
	default_exportSettings.bExportAll    = false;
	default_exportSettings.bExportQN     = false;
	default_exportSettings.bNoDbls       = true;
	default_exportSettings.strPrefix[0]  = 0;

	// Get the camera export settings
	CamExportSettings* exportSettings;
	ReferenceTarget* scene=gInterface->GetScenePointer();

	AppDataChunk* appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_SETTINGS_ID);
	
	// If there are no camera export settings in the scene, use the defaults
	if (appdata)
		exportSettings=(CamExportSettings*)appdata->data;
	else
		exportSettings=&default_exportSettings;

	int numCams=expData->Count();
	
	// Determine the file path
	CStr strSceneName=GetSceneName();
	CStr strFile;
	CStr strPath=getenv(APP_ENV);

	if (strPath==CStr(""))
	{
		char strErr[256];
		sprintf(strErr,"Your '%s' environment variable isn't set.  Please set it to continue.\nOperation aborted.",APP_ENV);

		MessageBox(gInterface->GetMAXHWnd(),strErr,"Camera Exporter",MB_ICONWARNING|MB_OK);
		delete genExporter;
		return false;
	}

	strPath=CStr(getenv(APP_ENV))+CStr(CAM_PATH)+strSceneName;

	// Attempt to create the directory (incase it doesn't exist)
	_mkdir(strPath);

	strPath=CStr(getenv(APP_ENV))+CStr(CAM_PATH)+strSceneName;
	strPath+="\\";
	strPath+=exportSettings->strPrefix;

	// Go through all the cameras and export them
	for(int i=0;i<numCams;i++)
	{
		INode* node=(*expData)[i].node;

		CamSpecificExportSettings* settings;
		AppDataChunk* appdata;
		appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID);

		// If the camera doesn't have any data attached we will assume it to use the defaults
		if (appdata)
			settings=(CamSpecificExportSettings*)appdata->data;
		else
		{
			settings=&default_settings;
			strcpy(settings->strUserName,(char*)node->GetName());
		}

		CamInfo newCamInfo;
		newCamInfo.expdata=&(*expData)[i];

		strFile=strPath+CStr(settings->strUserName)+CStr(".ska");

		// Verify that the start/end range is valid
		if (settings->end < settings->start)
		{
			char strErr[256];
			sprintf(strErr,"The end frame is less than the start frame for camera '%s' (Node: %s).\nExport failed.",settings->strUserName,(char*)node->GetName());
			MessageBox(gInterface->GetMAXHWnd(),strErr,"Camera Exporter",MB_ICONWARNING|MB_OK);
			ReportWarning( strErr );
			return false;
		}

		if (!Export(strFile,
			        &newCamInfo,
					settings->start,
					settings->end,
			        settings->fRotTolerance,
					settings->fTranTolerance,
					settings->fFovTolerance,
					exportSettings->bCompress,
					exportSettings->bNoDbls,
					exportSettings->bDebug,
					exportSettings->bCompressTime))
		{
			delete genExporter;		
			return false;
		}

		// Create the corresponding .ps2 file for the .ska animation we just created
		BuildTargetFile(strFile);

		// Send a network update message for the camera we're exporting
		SendNetMessage(settings->strUserName,i*250);
	}

	delete genExporter;

	// Send a QBR update
	if (type == vCAM_EXPORT_NOSCRIPTS)
		DoQBRUpdate(false);
	else
		DoQBRUpdate(true);
	
	return true;
}

bool CameraExporter::Save()
{ FUNC_ENTER("CameraExporter::Save"); 
	return true;
}

CStr CameraExporter::GetSceneName()
{ FUNC_ENTER("CameraExporter::GetSceneName"); 
	// This is temporary, to get the scene name
	const int vMAX_LENGTH_SCENE_NAME = 64;
	const int vVERSION  = 0x0002;
	const int vVERSION2 = 0x0003;
	const int vVERSION3 = 0x0004;
	const int vVERSION4 = 0x0005;
	//const int vVERSION5 = 0x0006;

	ReferenceTarget* scene = gInterface->GetScenePointer();
	char scene_name[vMAX_LENGTH_SCENE_NAME];
	char* options_data;
	AppDataChunk* pad;
	int version;

	pad = scene->GetAppDataChunk( vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_SCENE_EXPORT_OPTIONS_ID );

	if (pad)
	{
		options_data = (char*) pad->data;

		version=*((int*)options_data);
		if (version == vVERSION || version == vVERSION2 || version == vVERSION3 || version >= vVERSION4 )
		{
			options_data += sizeof( int );
			options_data += sizeof( bool );
			options_data += sizeof( bool );
			options_data += sizeof( bool );
			options_data += sizeof( bool );
			options_data += sizeof( bool );
			options_data += sizeof( bool );
			memcpy( scene_name, options_data, vMAX_LENGTH_SCENE_NAME );
		}
		else
		{
			MessageBox(gInterface->GetMAXHWnd(),"WARNING!  An unrecognized SceneExportOptions data chunk version was detected.\nUnable to retrieve scene name.","Version out of date",MB_ICONWARNING|MB_OK);
			return CStr("default");
		}

		return CStr(scene_name);
	}

	return CStr("default");
}

bool CameraExporter::ExportQN(FILE* fp)
{ FUNC_ENTER("CameraExporter::ExportQN"); 
	if (!fp)
		return false;

	// Do general export for cameras (so we can go through each one)
	genExporter=new GenExporter(gInterface);
	genExporter->ClearSearchKeys();
	genExporter->ClearExportData();

	SearchKey skey;
	skey.cid=Class_ID(SIMPLE_CAM_CLASS_ID,0);
	skey.fps=60;
	skey.conttype=CONTTYPE_FLOAT;
	skey.animStr="FOV";
	skey.parmStr="";

	genExporter->AddSearchKey(skey);
	skey.cid=Class_ID(LOOKAT_CAM_CLASS_ID,0);
	genExporter->AddSearchKey(skey);

	Tab<ExportInfo>* expData=genExporter->Export();

	if (!expData)
	{
		delete genExporter;
		return false;
	}

	// Get the camera export settings
	CamExportSettings* exportSettings;
	ReferenceTarget* scene=gInterface->GetScenePointer();

	AppDataChunk* appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_SETTINGS_ID);
	
	// If there are no camera export settings in the scene, use the defaults
	CStr strPrefix;
	
	if (appdata)
	{
		exportSettings=(CamExportSettings*)appdata->data;
		strPrefix=exportSettings->strPrefix;
	}
	else
		strPrefix="";

	int numCams=expData->Count();

	// Determine the file path
	CStr strSceneName=GetSceneName();
	CStr strFile;
	CStr strPath=getenv(APP_ENV);

	if (strPath==CStr(""))
	{
		char strErr[256];
		sprintf(strErr,"Your '%s' environment variable isn't set.  Please set it to continue.\nOperation aborted.",APP_ENV);

		MessageBox(gInterface->GetMAXHWnd(),strErr,"Camera Exporter",MB_ICONWARNING|MB_OK);
		delete genExporter;
		return false;
	}

	//strPath=CStr(getenv(APP_ENV))+CStr(CAM_PATH)+strSceneName;
	strPath=CStr(CAMREL_PATH)+strSceneName;
	strPath+="\\";
	strPath+=strPrefix;

	// Write .qn header
	fprintf(fp,"script LoadCameras\n");

	int i;

	for(i=0;i<numCams;i++)
	{
		CStr   camName;
		INode* node=(*expData)[i].node;

		CamSpecificExportSettings* settings;
		AppDataChunk* appdata;
		appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID);

		// If the camera doesn't have any data attached we will assume it to use the defaults
		if (appdata)
		{
			settings=(CamSpecificExportSettings*)appdata->data;
			camName=settings->strUserName;
		}
		else
		{
			camName=node->GetName();
		}

		// Determine the camera animation filename
		strFile=strPath+camName+CStr(".ska");

		// We now have the camera name, write a line for it to the .qn file
		fprintf(fp,"\tLoadAsset \"%s\" %s\n",(char*)strFile,(char*)camName);
	}

	// Write .qn footer
	fprintf(fp,"endscript\n\n");

	// Now generate the user defined camera scripts
	for(i=0;i<numCams;i++)
	{
		INode* node=(*expData)[i].node;
		
		// Get the user name for the camera (export name)
		CStr name;
		appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID);

		if (appdata)
		{
			name = ((CamSpecificExportSettings*)appdata->data)->strUserName;
		}
		else
		{
			name = node->GetName();
		}

		appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_CAMERA_EXPORT_ID);

		if (!appdata)
			continue;

		int   len=appdata->length;
		char* buf=(char*)appdata->data;

		char  sframeNum[256];
		char  scriptName[512];
		int   extlen;				// Length of extended data for attached user scripts
		char  token[1024];
		int   pos = 0;

		while(1)
		{
			if (!GetToken(buf,&pos,token,"\n"))
				break;

			strcpy(scriptName,token);
			
			if (!GetToken(buf,&pos,token,"\n"))	// scriptFile
				break;
			
			if (!GetToken(buf,&pos,token,"\n"))
				break;

			strcpy(sframeNum,token);

			if (CStr(scriptName)==CStr(USERSCRIPT_NAME))
			{
				char* sbuf;
				GetToken(buf,&pos,token,"\n");
				extlen = atoi(token);
				
				sbuf = new char[extlen+1];

				for(int i=0;i<extlen;i++)
					sbuf[i]=buf[pos++];

				sbuf[i]='\0';
				
				CStr sname, strBuf;
				sname = name;
				sname += "Frame";
				sname += sframeNum;

				strBuf = ReplaceStr(sbuf,"\r\n","\n");

				fprintf(fp,"script %s\n",(char*)sname);
				fprintf(fp,"%s\n",(char*)strBuf);
				fprintf(fp,"endscript\n\n");

				delete sbuf;
			}
		}
	}

	return true;
}

void CameraExporter::SendNetMessage(char* camname,int delay)
{ FUNC_ENTER("CameraExporter::SendNetMessage"); 
	Net::MsgViewObjSetCamAnimFile msg;

	CStr sceneName = GetSceneName();
	CStr fileName  = CStr(CAMREL_PATH)+sceneName+CStr("\\")+CStr(camname)+".ska";
	CStr absFileName = CStr(getenv(APP_ENV)) + CStr("\\data\\")+fileName;

	strcpy(msg.m_Filename,(char*)fileName);
	msg.m_checksum=GenerateCRC(camname);

	//gClient->EnqueueMessageToServer( Net::vMSG_ID_VIEWOBJ_SET_CAMANIM_FILE, sizeof( Net::MsgViewObjSetCamAnimFile ), &msg,
	//								 Net::NORMAL_PRIORITY, 	Net::QUEUE_SEQUENCED );

	Net::MsgDesc msgdesc;
	msgdesc.m_Id       = Net::vMSG_ID_VIEWOBJ_SET_CAMANIM_FILE;
	msgdesc.m_Length   = sizeof( Net::MsgViewObjSetCamAnimFile );
	msgdesc.m_Data     = &msg;
	msgdesc.m_Priority = Net::NORMAL_PRIORITY;
	msgdesc.m_Queue    = Net::QUEUE_DEFAULT;
	msgdesc.m_GroupId  = Net::GROUP_ID_DEFAULT;
	msgdesc.m_Singular = false;
	msgdesc.m_Delay    = delay;

	gClient->EnqueueMessageToServer( &msgdesc );
}

void CameraExporter::DoQBRUpdate(bool bGenerateScripts)
{ FUNC_ENTER("CameraExporter::DoQBRUpdate"); 
	return;

	if (bGenerateScripts)
	{
		Exporter* exporter = GetSceneExporter();
		exporter->DoExport( vSCRIPT_EXPORT );
	}

	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	CStr dirPath = getenv(APP_ENV);
	dirPath += CStr(QN_PATH);
	dirPath += seo.m_SceneName + CStr("\\") + seo.m_SceneName + ".qn";

	// Execute qcomp to build .qb
	char  path[256];
	char* args[3];
	sprintf( path, "%s\\bin\\win32\\qcomp.exe", getenv(APP_ENV) );

	args[0] = path;
	args[1] = dirPath;
	args[2] = NULL;

	//spawnvp( _P_WAIT, path, args );
	ExecuteCommand( args );

	dirPath[dirPath.Length()-1] = 'b';

	// Execute rq to send to game
	sprintf( path, "%s\\bin\\win32\\rq.exe", getenv(APP_ENV) );

	args[0] = path;
	args[1] = dirPath;
	args[2] = NULL;

	//spawnvp( _P_WAIT, path, args );
	ExecuteCommand( args );
}

void CameraExporter::BuildTargetFile(char* camname)
{ FUNC_ENTER("CameraExporter::BuildTargetFile"); 
	// Execute animation converter
	char  ancPath[256];
	char* args[4];
	CStr  strFileCmd = CStr("-f") + camname;
	sprintf( ancPath, "%s\\bin\\win32\\animconv.exe", getenv(APP_ENV) );

	args[0] = ancPath;
	args[1] = strFileCmd;
	if( GetQuickviewPlatform() == vQUICKVIEW_XBOX )
	{
		args[2] = "-px";
	}
	else 
	{
		args[2] = "-pp";
	}
	args[3] = NULL;

	//int result = ( spawnvp( _P_WAIT, ancPath, args ) == 0 );
	ExecuteCommand( args );
}
