/*
	AnimExporter.h
	Animation Exporter

	Encapsulates animation export functionality

	1-11-01
*/

#ifndef __ANIMEXPORTER__
#define __ANIMEXPORTER__

#include <core/math.h>
#include "skeleton.h"
#include "GenExport.h"
#include "max.h"
#include "../UI/ProgressBar.h"

#define  OUTPUT_FPS     60		// Number of frames per second to output animation as

class TolData;

namespace Gfx
{
	struct SAnimQFrame
	{
		Mth::Quat		q;
		float			time;
		bool            bFlip[3];
		unsigned long   flags;

		SAnimQFrame()
		{
			bFlip[0]=false;
			bFlip[1]=false;
			bFlip[2]=false;

			flags = 0;
		}
	};

	struct SAnimTFrame
	{
		float			t[3];	
		float			time;
		unsigned long   flags;

		SAnimTFrame()
		{
			flags = 0;
		}
	};
}

#define LINEAR_TRUE   -1
#define LINEAR_FALSE   0

class AnimExporter
{
protected:
	Gfx::SAnimQFrame* Qframes;		// Uncompressed Rotation list  (Quaternions) for current node
	Gfx::SAnimTFrame* Tframes;		// Uncompressed Translation list for current node

	Gfx::SAnimQFrame* compQframes;	// Compressed Rotation list (Quaternions) for current node
	Gfx::SAnimTFrame* compTframes;	// Compressed Translation list for current node

	int* numCompressedQFrames;
	int* numCompressedTFrames;

	int        nReverseTKeys;		// Number of Translation keys added due to reversal of keyframe direction between individual keys
	int        nReverseFKeys;		// Number of Float keys added due to reversal of keyframe direction between individual keys
	int        nReverseQKeys;		// Number of Rotation keys added due to reversal of keyframe direction between individual keys

	FILE*      fpExport;			// File stream being exported
	FILE*      fpDebug;				// Debug File stream
	FILE*      fpKeyRevDebug;		// Key reversal information debug file

	Interface* ip;
	int        start,end;			// Start and end frames of MAX animation
	int        curFrame;			// Current position within the frames arrays
	int        numFrames;			// Number of animation frames in memory
	int        numNodes;			// Number of nodes in the skeleton animation SKA data being exported

	bool       bRemoveReadOnly;		// True if read only flag should be removed on all
	bool	   bCheckOut;			// True if we should attempt to check out all read-only files
	bool       bSwapCoordSystem;	// True if coordinates from MAX should be adjusted to game

	bool       bFixReverseKeyInterpolation;	// True if compression should account for adding additional keys if a sequence
											// that is determined linear varies between +/- between keys

	CSkeletonData* skeleton;		// Skeleton data for current animation
	INode*         rootbone;		// The root object of the skeleton
	INode*         pelvisbone;		// The pelvis bone in the skeleton (if applicable)
	
	bool       bCompressTime;		// True if times should be given as integers
	bool       bRotateRoot;			// True if root node should be rotated 90 degrees
	bool       bTestRotate;			// Just a test to figure out whats going on with current weird rotation in game (Remove later)
	bool       bNoParentTransform;	// True if the keys should NOT consider the parent's transformation

	void BuildNodeQFrames(Gfx::SAnimQFrame* qframes,INode* node,bool bExportFromPelvis,ProgressBar* pbar=NULL);
	void BuildNodeTFrames(Gfx::SAnimTFrame* qframes,INode* node,bool bExportFromPelvis,ProgressBar* pbar=NULL);

	void BuildNodeQTFrames(Gfx::SAnimQFrame* qframes,
		                   Gfx::SAnimTFrame* tframes,
						   INode*            node,
						   bool              bExportFromPelvis,
						   ProgressBar*      pbar);


	int BuildAllQFrames(Gfx::SAnimQFrame* qframes,INode* node,int numFrames,int ID);
	int BuildAllTFrames(Gfx::SAnimTFrame* tframes,INode* node,int numFrames,int ID);

	int BuildAllQTFrames(Gfx::SAnimQFrame* qframes,
		                 Gfx::SAnimTFrame* tframes,
						 INode*            node,
						 int               numFrames,
						 int               ID);

	void GetQFrames(Gfx::SAnimQFrame* qframes, INode* root, bool bExportFromPelvis, bool bOneNode=false, ProgressBar* pbar=NULL);	// Builds a list of qframes for the given node and it's children
	void GetTFrames(Gfx::SAnimTFrame* tframes, INode* root, bool bExportFromPelvis, bool bOneNode=false, ProgressBar* pbar=NULL);	// Builds a list of tframes for the given node and it's children

	void GetQTFrames(Gfx::SAnimQFrame* qframes,
		             Gfx::SAnimTFrame* tframes,
					 INode*            root,
					 bool              bExportFromPelvis,
					 bool              bOneNode,
					 ProgressBar*      pbar);

	static void FreeFrames(Gfx::SAnimQFrame** frame);	// Frees list returned by GetQFrame
	static void FreeFrames(Gfx::SAnimTFrame** frame);	// Frees list returned by GetTFrame

	int GetNodeName(CStr* strName,INode* node,int ID);
	int GetParentName(CStr* strName,INode* node,int ID);

	void RemoveDblKeys(int numNodes,int numFrames);
	void RemoveDblKeys(int numNodes, int numFrames,
						 Gfx::SAnimQFrame* compQframes,
						 Gfx::SAnimTFrame* compTframes,
						 int* numCompressedQFrames,
						 int* numCompressedTFrames);

	bool VerifyRoot();
	INode* GetRoot(INode* node);
	bool ExportNodes(INode* root);
	int  CountNodes(INode* node);
	void GetNodeNames(CStr* strNames,INode* root);
	void GetParentNames(CStr* strNames,INode* root);

	int  BuildFrames(INode* node);									// Builds frames for the given node

	bool GetAnim(INode*       root,
		         int          start,
				 int          end,
				 float        errorQ,
				 float        errorT,
				 TolData*     tdata,
				 bool         bCompress=true,
				 ProgressBar* pbar=NULL,
				 void (*PreCompCB)(void*) = NULL,
				 void* pPreCompData = NULL,
				 bool         bExportFromPelvis = false);

	bool GetAnim(INode*        root,
		          int          start,
				  int          end,
				  float        errorQ,
				  float        errorT,
				  TolData*     tdata,
				  bool         bCompress,
				  int**        numCompressedQFrames, Gfx::SAnimQFrame** compQFrames,
				  int**        numCompressedTFrames, Gfx::SAnimTFrame** compTFrames,
				  int*         numNodes,
				  ProgressBar* pbar = NULL,
				  void (*PreCompCB)(void*) = NULL,
				  void* pPreCompData = NULL,
				  bool  bExportFromPelvis = false);

	void HideDummy(INode* node);
	void UnhideDummy(INode* node);

	void UnhideDummies();
	void HideDummies();

	int CompressRawQFrameData( Gfx::SAnimQFrame* p_newdata, Gfx::SAnimQFrame* p_data, int num_keys, double error, ProgressBar* pbar=NULL );
	int CompressRawTFrameData( Gfx::SAnimTFrame* p_newdata, Gfx::SAnimTFrame* p_data, int num_keys, double error, ProgressBar* pbar=NULL );
	int CompressFloatKeys( GenKey<float>* newvals, GenKey<float>* vals, int num_keys, double error );

	void FixEulerFlips(Gfx::SAnimQFrame* pdata,int numKeys);

	bool ApproxEqual(float v1,float v2);

	int IsTLinear( const Gfx::SAnimTFrame* p_data, unsigned int start, unsigned int end, double error_x, double error_y, double error_z );
	int IsQLinear( const Gfx::SAnimQFrame* p_data, unsigned int start, unsigned int end, double error );
	int IsFloatLinear( const GenKey<float>* vals, unsigned int start, unsigned int end, double error );

public:
	AnimExporter();
	~AnimExporter();

	bool Export(INode* root,char* Filename,char* ModelName,int start,int end);
	bool ExportAnim(HWND            hwnd,
		            INode*          root,
					char*           Filename,
					char*           ModelName,
					int             start,
					int             end,
					float           errorQ,
					float           errorT,
					TolData*        tdata,
					bool            bOneFrame=false,
					bool            bDebug=false,
					bool            bUseCompression=false,
					bool            bCompressTime=false,
					bool            bRotateRoot=false,
					PartialAnimSet* pPartialAnim = NULL);

	// Retrieves the animation checksum from a .ska file
	unsigned int GetChecksum(char* Filename);
};

#endif
