/*
	PropList.cpp
	Property List Window Class
	
	Adam Lippmann
	adam@neversoft.com
*/

#include "PropList.h"
#include "PropListInt.h"
#include "Resource.h"
#include "SpinEdit.h"
#include "ScriptSelect.h"
#include "NodeSelect.h"
#include "../UI/SlideNum.h"
#include "../UI/MultiList.h"
#include "../UI/FlagSelect.h"
#include "../UI/MtlSelect.h"
#include "../UI/Static.h"
#include "../UI/ColorPopup.h"
#include <stdio.h>

#define PROP_SPACING          4     // The vertical spacing in pixels between object property fields
#define PROP_DROPDOWNHEIGHT 300		// The drop down height 
#define STATIC_ID           100		// Base offset for static IDs
#define PROP_HEIGHT          20		// Height of the property and text fields
#define PROP_HSPACING         5		// Horizontal spacing in pixels between field desc and field
#define PROP_LEFTSPACING      2		// Number of pixels from the left of the window edge to display fields
#define PROP_RIGHTSPACING     4		// Number of pixels from the right of the window edge to display fields
#define PROP_TOP              2		// Number of pixels from the top of the window to the first field
#define SCROLL_WIDTH         10		// Width of scrollbar
#define SCROLL_SPACE          5

#define TXT_HIGHLIGHT        RGB(215,215,215)
#define TXT_HIGHLIGHT_RED    RGB(250,200,200)
#define TXT_HIGHLIGHT_GREEN  RGB(200,250,200)
#define TXT_HIGHLIGHT_BLUE   RGB(200,200,250)
#define TXT_HIGHLIGHT_PURPLE RGB(250,200,250)
#define TXT_HIGHLIGHT_WHITE  RGB(255,255,255)

PropList::PropList(HINSTANCE hInstance,int cols) :
	ttip(hInstance)
{
	this->hInstance = hInstance;

	hwndParent=NULL;
	hwndScroll=NULL;
	x=y=0;
	numCols=cols;

	// Clear out callbacks
	fpFocus=NULL;
	pFocusData=NULL;
	
	fpApply=NULL;
	pApplyData=NULL;

	fpDblClick=NULL;
	pDblClickData=NULL;

	fpChange=NULL;
	pChangeData=NULL;

	fpExtList=NULL;
	pExtListData=NULL;

	fpSProc=NULL;
	pSProcData=NULL;
	OldStaticProc=NULL;

	bHasSliders=FALSE;
	bHasApply=true;

	bSetValLocked=true;

	lastModID=-1;
	curIndex=-1;
	linkAddPos = props.GetTail();

	// Initialize the property list window class
	wndclass.style         = CS_HREDRAW|CS_VREDRAW;
	wndclass.lpfnWndProc   = RedirectWndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = LoadIcon(NULL,IDI_APPLICATION);
	wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH)GetStockObject(LTGRAY_BRUSH);
	wndclass.lpszMenuName  = NULL;
	wndclass.lpszClassName = "PropList";
	
	RegisterClass(&wndclass);
	
	// Register our own custom control interfaces to be used in the list
	SpinEdit::Register(hInstance);

	// Initialize the brush used to paint the background in the property name child windows
	hBGBrush=CreateSolidBrush(TXT_HIGHLIGHT);
	hBGBrushR=CreateSolidBrush(TXT_HIGHLIGHT_RED);
	hBGBrushG=CreateSolidBrush(TXT_HIGHLIGHT_GREEN);
	hBGBrushB=CreateSolidBrush(TXT_HIGHLIGHT_BLUE);
	hBGBrushP=CreateSolidBrush(TXT_HIGHLIGHT_PURPLE);
	hBGBrushW=CreateSolidBrush(TXT_HIGHLIGHT_WHITE);

	bFloatingWindow=false;
	bAttachedWindow=false;
}

PropList::PropList(HINSTANCE hInstance,HWND hwdParent,int x,int y,int width,int height,CStr DlgText) :
	ttip(hInstance)
{
	this->hInstance = hInstance;

	numCols=1;
	hwndParent=hwdParent;
	hwndScroll=NULL;
	this->x=x;
	this->y=y;
	this->width=width;
	origWidth=width;
	this->height=height;

	// Clear out callbacks
	fpFocus=NULL;
	pFocusData=NULL;
	
	fpApply=NULL;
	pApplyData=NULL;

	fpChange=NULL;
	pChangeData=NULL;

	fpExtList=NULL;
	pExtListData=NULL;

	fpDblClick=NULL;
	pDblClickData=NULL;

	fpSProc=NULL;
	pSProcData=NULL;
	OldStaticProc=NULL;

	bHasSliders=FALSE;
	bHasApply=true;

	bSetValLocked=true;

	lastModID=-1;
	curIndex=-1;
	linkAddPos = props.GetTail();

	// Initialize the property list window class
	wndclass.style         = CS_HREDRAW|CS_VREDRAW;
	wndclass.lpfnWndProc   = RedirectWndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = LoadIcon(NULL,IDI_APPLICATION);
	wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH)GetStockObject(LTGRAY_BRUSH);
	wndclass.lpszMenuName  = NULL;
	wndclass.lpszClassName = "PropList";
	
	RegisterClass(&wndclass);

	if (DlgText==CStr(""))
	{
		// Create the property list floating window
		hwnd=CreateWindow("PropList",
						  NULL,
						  WS_POPUP|WS_DLGFRAME,
						  x,y,
						  width,height,
						  hwdParent,
						  NULL,
						  hInstance,
						  NULL);

		bFloatingWindow=false;
	}
	else
	{
		// Create a floating property list window
		hwnd=CreateWindow("PropList",
						  DlgText,
						  WS_DLGFRAME,
						  x,y,
						  width,height,
						  hwdParent,
						  NULL,
						  hInstance,
						  NULL);

		bFloatingWindow=true;
	}

	SetWindowLong(hwnd,GWL_USERDATA,(LONG)this);

	// Set window as topmost popup
	SetWindowPos(hwnd,HWND_TOPMOST,x,y,width,height,0);

	if (DlgText!=CStr(""))
		ShowWindow(hwnd,SW_SHOW);

	RECT winRect;
	GetClientRect(hwnd,&winRect);

	winRect.left+=PROP_LEFTSPACING;
	winRect.top +=PROP_TOP;

	rectRefTXT.left=winRect.left;
	rectRefTXT.top=winRect.top;
	rectRefTXT.right=winRect.left+(width/2)-PROP_HSPACING;
	rectRefTXT.bottom=rectRefTXT.top+PROP_HEIGHT;

	rectRefFLD.left=winRect.left+width/2;
	rectRefFLD.top =winRect.top;
	rectRefFLD.right=rectRefFLD.left+(width/2)-PROP_HSPACING;
	rectRefFLD.bottom=rectRefFLD.top+PROP_HEIGHT;

	rectRefTXT.right-=(SCROLL_WIDTH+SCROLL_SPACE)/2;
	rectRefFLD.right-=(SCROLL_WIDTH+SCROLL_SPACE)/2;

	//width+SCROLL_WIDTH+SCROLL_SPACE

	TXTWidth=rectRefTXT.right-rectRefTXT.left;
	FLDWidth=rectRefFLD.right-rectRefFLD.left;

	TXTHeight=rectRefTXT.bottom-rectRefTXT.top;
	FLDHeight=rectRefFLD.bottom-rectRefFLD.top;
	
	// Initialize the brush used to paint the background in the property name child windows
	hBGBrush=CreateSolidBrush(TXT_HIGHLIGHT);
	hBGBrushR=CreateSolidBrush(TXT_HIGHLIGHT_RED);
	hBGBrushG=CreateSolidBrush(TXT_HIGHLIGHT_GREEN);
	hBGBrushB=CreateSolidBrush(TXT_HIGHLIGHT_BLUE);
	hBGBrushP=CreateSolidBrush(TXT_HIGHLIGHT_PURPLE);
	hBGBrushW=CreateSolidBrush(TXT_HIGHLIGHT_WHITE);

	bAttachedWindow=false;
}

PropList::~PropList()
{
	ttip.Close();
	DeleteObject(hBGBrush);
	DeleteObject(hBGBrushR);
	DeleteObject(hBGBrushG);
	DeleteObject(hBGBrushB);
	DeleteObject(hBGBrushP);
	DeleteObject(hBGBrushW);
	DestroyWindow(hwnd);
}

void PropList::SaveValues()
{
	savedValues.Clear();

	int numProps=props.GetSize();

	for(int i=0;i<numProps;i++)
	{
		CStr value;
		GetValue(i,value);
		savedValues.Add(&value);
	}
}

void PropList::GetValues(LinkList<CStr>* list)
{
	int numProps = props.GetSize();

	for(int i=0;i<numProps;i++)
	{
		CStr value;
		GetValue(i,value);
		list->Add(&value);
	}
}

void PropList::SetValues(LinkList<CStr>* list)
{
	int numProps = list->GetSize();

	for(int i=0;i<numProps;i++)
		SetValue(i,(*list)[i]);
}

void PropList::RestoreValues()
{
	Link<CStr>* curNode=savedValues.GetHead();
	int  index=0;
	bool bUpdated;

	while(curNode)
	{
		// Set values and retain modification status
		bUpdated=WasUpdated(index);
		SetValue(index,curNode->data);
		SetMod(index++,bUpdated);
		curNode=curNode->next;
	}
}

void PropList::DestroyUI()
{
	bSetValLocked=true;

	// Destroy all windows
	Link<Property>* plink=props.GetHead();

	while(plink)
	{
		// Destroy UI child windows
		
		// Remove subclass on the property text window
		SetWindowLong(plink->data.hwndTxt,GWL_WNDPROC,(LONG)OldStaticProc);

		DestroyWindow(plink->data.hwndTxt);
		DestroyWindow(plink->data.hwndFld);
		
		plink=plink->next;
	}

	if (hwndApply)
		DestroyWindow(hwndApply);

	if (hwndScroll)
		DestroyWindow(hwndScroll);
}

void PropList::Clear()
{
	bSetValLocked=true;

	// Destroy all windows
	Link<Property>* plink=props.GetHead();
	ColorData*      cdata;

	DestroyUI();

	while(plink)
	{
		// Handle any special cleanup
		switch(plink->data.ptype)
		{
		case PROPTYPE_COLOR:
			cdata=(ColorData*)plink->data.pdata;
			ReleaseIColorSwatch(cdata->IColor);
			cdata->IColor=NULL;
			break;
		}

		plink=plink->next;
	}

	// Reset property management list
	props.Clear();
	
	// Reset temporary property value buffer
	propVals.Clear();

	width=origWidth;

	// Reset last modified flag
	lastModID=-1;
	curIndex=-1;
	linkAddPos = props.GetTail();
}

void PropList::AddStatic(CStr propName,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_STATIC;
	newProp.pdata = 0;

	StaticCtl* sc = new StaticCtl(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = sc;
	
	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
}

void PropList::AddEdit(CStr propName,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_EDIT;
	newProp.pdata      =NULL;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = props.AddAfterLink(linkAddPos,&newProp);
}

SpinEdit* PropList::AddSpinEdit(CStr propName,float min,float max,float incr,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_SPINEDIT;
	newProp.pdata = 0;

	SpinEdit* spe = new SpinEdit;

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = spe;
	
	spe->SetLimits(min,max);
	spe->SetIncr(incr);

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return spe;
}

SliderNum* PropList::AddSliderNum(CStr propName,int min,int max,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_SLIDERNUM;
	newProp.pdata = 0;

	SliderNum* sn = new SliderNum(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = sn;
	
	sn->SetLimits(min,max);

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return sn;
}

ColorPopup* PropList::AddColorPopup(CStr propName,int min,int max,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_COLORPOPUP;
	newProp.pdata = 0;

	ColorPopup* cp = new ColorPopup(hInstance);

	cp->SetLimits(min,max);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = cp;
	
	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return cp;
}

ListData* PropList::AddList(CStr propName,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;
	
	newProp.ptype      =PROPTYPE_LIST;
	newProp.pdata      =0;

	// List data must be added after the property is created
	// so it doesn't get deleted by the init property destructor
	Link<Property>* link=props.AddAfterLink(linkAddPos,&newProp);;
	link->data.pdata    =new ListData;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;

	return (ListData*)link->data.pdata;
}

ListData* PropList::AddExtList(CStr propName,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_EXTLIST;
	newProp.pdata      =0;

	// List data must be added after the property is created
	// so it doesn't get deleted by the init property destructor
	Link<Property>* link=props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata    =new ListData;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return (ListData*)link->data.pdata;
}

void PropList::AddSlider(CStr propName,int min,int max,CStr ToolTip)
{
	Property    newProp;
	SliderData* sdata;

	bHasSliders=TRUE;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_SLIDER;
	newProp.pdata      =0;

	// Slider data must be added after the property is created
	// so it doesn't get deleted by the init property destructor
	Link<Property>* link  =props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata=sdata=new SliderData;

	sdata->min=min;
	sdata->max=max;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
}

FileData* PropList::AddFile(CStr propName,CStr ToolTip)
{
	Property   newProp;
	FileData*  fdata;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_FILE;
	newProp.pdata      =0;

	// File data must be added after the property is created
	// so it doesn't get deleted by the init property destructor
	Link<Property>* link  =props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata=fdata=new FileData;

	fdata->Fltr[0]='\0';
	fdata->Fltr[1]='\0';

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return fdata;
}

ColorData* PropList::AddColor(CStr propName,CStr ToolTip,COLORREF Color,CStr DlgText)
{
	Property   newProp;
	ColorData* cdata;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_COLOR;
	newProp.pdata      =0;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	// Color data must be added after the property is created
	// so it doesn't get deleted by the init property destructor
	Link<Property>* link  =props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata=cdata=new ColorData;

	cdata->IColor  =NULL;
	cdata->DefColor=Color;
	cdata->DlgText =DlgText;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return cdata;
}

void PropList::AddCheck(CStr propName,CStr ToolTip)
{
	Property   newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_CHECK;
	newProp.pdata      =0;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = props.AddAfterLink(linkAddPos,&newProp);
}

ScriptSelect* PropList::AddScript(CStr propName,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_SCRIPT;
	newProp.pdata = 0;

	ScriptSelect* ss = new ScriptSelect(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = ss;
	
	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return ss;
}

NodeSelect* PropList::AddNode(CStr propName,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_NODE;
	newProp.pdata = 0;

	NodeSelect* ns = new NodeSelect(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = ns;
	
	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return ns;
}

MultiList* PropList::AddMulti(CStr propName,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_MULTI;
	newProp.pdata = 0;

	MultiList* ml = new MultiList(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = ml;
	
	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return ml;
}

FlagSelect* PropList::AddFlags(CStr propName,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_FLAGS;
	newProp.pdata = 0;

	FlagSelect* fs = new FlagSelect(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = fs;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return fs;
}

void PropList::AddMtl(CStr propName,CStr ToolTip)
{
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_MTL;
	newProp.pdata = 0;

	MtlSelect* ms = new MtlSelect(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = ms;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
}

LRESULT CALLBACK PropList::RedirectWndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	PropList* thisclass=(PropList*)GetWindowLong(hwnd,GWL_USERDATA);

	if (msg == WM_CREATE)
		return TRUE;

	if (thisclass)
		return thisclass->WndProc(hwnd,msg,wParam,lParam);

	return DefWindowProc(hwnd,msg,wParam,lParam);
}

LRESULT PropList::WndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	int ID=LOWORD(wParam);
	CStr FileTitle;
	FileData* fdata;
	RECT winRectPos;
	RECT winRectSize;
	char buf[128];
	int  index;

	//DisableAccelerators();

	switch(msg)
	{
	case WM_CREATE:
		return TRUE;

	case WM_SHOWWINDOW:
		if ((BOOL)wParam)
			DisableAccelerators();

		return TRUE;

	case WM_CHILDUPDATED:
		props[(int)wParam].bWasUpdated=true;
		lastModID=(int)wParam;

		if (fpChange)
			fpChange(this, pChangeData);

		return TRUE;

	case WM_CHILDGOTFOCUS:
		if (fpFocus)
			fpFocus(this,(int)wParam,pFocusData);

		return TRUE;

	case WM_CTLCOLORSTATIC:
		return ProcColor((HWND)lParam);

	case WM_VSCROLL:
		if (lParam)
		{
			UpdateUI(UpdateScroll((HWND)lParam,LOWORD(wParam),HIWORD(wParam)));
		}
		return TRUE;

	case CC_COLOR_CHANGE:
	case CC_SLIDER_CHANGE:
		props[ID].bWasUpdated=true;
		lastModID=ID;

		if (fpChange)
			fpChange(this, pChangeData);

		return TRUE;

	case WM_COMMAND:
		// Check if the apply button was pressed and if so,
		// call the attached callback function
		if (fpApply && (ID==IDC_ACCEPT || (HWND)lParam==hwndApply))
		{
			ProcExtLists();

			fpApply(this,pApplyData);
			return TRUE;
		}

		switch(HIWORD(wParam))
		{
		case CBN_SELENDOK:
		case CBN_SELCHANGE:		// This is the same as STN_DBLCLK  (have to process both)

			// Handle STN_DBLCLK
			if (ID>=STATIC_ID)
			{
				if (fpDblClick)
				{
					ttip.Close();
					fpDblClick(this,ID-STATIC_ID,pDblClickData);
				}

				return TRUE;
			}

			props[ID].bWasUpdated=true;
			lastModID=ID;

			// Force the combobox to change it's text now
			index=SendMessage((HWND)lParam,CB_GETCURSEL,0,0);
			if (index != CB_ERR)
			{
				SendMessage((HWND)lParam,CB_GETLBTEXT,(WPARAM)index,(LPARAM)buf);
				SetWindowText((HWND)lParam,buf);
			}

			if (fpChange)
				fpChange(this, pChangeData);

			return TRUE;

		case CBN_EDITCHANGE:
		case EN_CHANGE:
			props[ID].bWasUpdated=true;
			lastModID=ID;

			if (fpChange)
				fpChange(this, pChangeData);

			return TRUE;

		case CBN_SETFOCUS:
		case EN_SETFOCUS:
			if (fpFocus)
				fpFocus(this, ID, pFocusData);

			return TRUE;

		case STN_CLICKED:	// Static label clicked (Tooltip request)
			// Check if actually file property
			if (ID<STATIC_ID)
			{
				if (props[ID].ptype==PROPTYPE_FILE)
				{
					fdata=(FileData*)props[ID].pdata;
					OpenFileDlg(hwnd,&fdata->FileName,&FileTitle,CStr("Select file for ")+props[ID].strPropName,fdata->Fltr);

					if (FileTitle==CStr(""))
						SetDlgItemText(hwnd,ID,"None");
					else
						SetDlgItemText(hwnd,ID,(char*)FileTitle);

					props[ID].bWasUpdated=true;
					lastModID=ID;
				}
			}
			
			if (ID>=STATIC_ID)
			{
				GetWindowRect(props[ID-STATIC_ID].hwndTxt,&winRectPos);
				GetClientRect(hwnd,&winRectSize);

				ttip.CreateTip(props[ID-STATIC_ID].strToolTip,
					           winRectPos.left+15,winRectPos.top+15,
							   winRectSize.right,100);

				return TRUE;
			}
			else
			{
				props[ID].bWasUpdated=true;
				lastModID=ID;

				if (fpChange)
					fpChange(this, pChangeData);
			}

			return TRUE;
		};
	}

	return DefWindowProc(hwnd,msg,wParam,lParam);
}

void PropList::OpenFileDlg(HWND hwnd,CStr* File,CStr* FileTitle,char* Title,char* Filter)
{
	OPENFILENAME ofn;
	char filename[256]="";
	char filetitle[128]="";

	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=hwnd;
	ofn.hInstance=hInstance;
	ofn.lpstrFilter=Filter;
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=filetitle;
	ofn.nMaxFileTitle=128;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle=Title;
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=TEXT(".q");
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetOpenFileName(&ofn);

	*File=CStr(filename);
	*FileTitle=CStr(filetitle);
}

int PropList::GetVisiblePropCount()
{
	Link<Property>* curprop = props.GetHead();
	int count = 0;

	while(curprop)
	{
		if (curprop->data.bVisible)
			count++;

		curprop = curprop->next;
	}

	return count;
}

void PropList::UpdateUI(int scrollpos)
{
	// We now have to reorder the UI windows based on where the scroll bar is
	RECT rect;
	GetWindowRect(hwnd,&rect);
	
	if (scrollpos==-1)
		return;

	//int pos=SendMessage(hwndScroll,SBM_GETPOS,0,0);
	int pos=scrollpos;

	int width=rect.right-rect.left+PROP_RIGHTSPACING;
	int height=rect.bottom-rect.top;
	
	int numItems=((height)/((FLDHeight+PROP_SPACING)/numCols));
	
	int disppos=0;	// Display position

	//int listCount=props.GetSize();
	int listCount=GetVisiblePropCount();

	// Force full rows unless last item
	if (numItems%numCols>0 &&
		numItems<listCount)
		numItems-=numItems%numCols;

	int nProps = props.GetSize();

	for(int c=0;c<nProps;c++)
	{
		ShowWindow(props[c].hwndTxt,SW_HIDE);
		ShowWindow(props[c].hwndFld,SW_HIDE);
	}

	for(int i=0,curprop=0;curprop<nProps;curprop++)
	{
		if (props[curprop].bVisible)
		{
			// Should be added to display
			if (i>=scrollpos && i<scrollpos+numItems)
			{
				MoveWindow(props[curprop].hwndTxt,
						   GetXTXTPos(disppos),
						   GetYTXTPos(disppos),
						   TXTWidth,TXTHeight,TRUE);

				// Ensure proper height if dropdown listbox
				if (props[curprop].ptype==PROPTYPE_LIST ||
					props[curprop].ptype==PROPTYPE_EXTLIST)
				{
					// Expand the field width if there is no scroll bar
						MoveWindow(props[curprop].hwndFld,
								   GetXFLDPos(disppos),
								   GetYFLDPos(disppos),
								   FLDWidth,FLDHeight+PROP_DROPDOWNHEIGHT,TRUE);
				}
				else
				{
					MoveWindow(props[curprop].hwndFld,
							   GetXFLDPos(disppos),
							   GetYFLDPos(disppos),
							   FLDWidth,FLDHeight,TRUE);
				}

				// Advance to next display position
				disppos++;

				ShowWindow(props[curprop].hwndTxt,SW_SHOW);
				ShowWindow(props[curprop].hwndFld,SW_SHOW);
			}

			i++;
		}
	}

	if (nProps>curprop)
		for(i=curprop;i<nProps;i++)
		{
			ShowWindow(props[i].hwndTxt,SW_HIDE);
			ShowWindow(props[i].hwndFld,SW_HIDE);
		}

	// Reposition the apply button
	if (bHasApply)
	{
		MoveWindow(hwndApply,
			       rectRefTXT.left,(FLDHeight+PROP_SPACING)*disppos,
				   TXTWidth*2+10,TXTHeight,TRUE);
	}
}

void PropList::SetScrollRange()
{
	if (hwndScroll)
	{
		int listSize=GetVisiblePropCount();

		SendMessage(hwndScroll,SBM_SETRANGE,(WPARAM)0,(LPARAM)listSize-((listSize%numCols==0) ? numCols : 1));
	}
}

void PropList::BuildUI()
{
	// Calc maximum properties that can fit on the property list window
	//RECT rect;
	//GetWindowRect(hwnd,&rect);

	//int width=rect.right-rect.left+PROP_RIGHTSPACING;
	//int height=rect.bottom-rect.top;

	if (!bAttachedWindow)
		width+=PROP_RIGHTSPACING;

	int listSize=props.GetSize();

	if (listSize==0)
		return;

	// Compute the required height
	int reqHeight = ((bHasApply) ? listSize+1 : listSize) * (FLDHeight+PROP_SPACING)+9;

	FLDWidthS = FLDWidth;

	// Construct a scroll bar if the desired height is greater than the UI window
	if (reqHeight>height*numCols)
	{
		int swidth,sheight;

		if (bAttachedWindow)
		{
			swidth=width-SCROLL_WIDTH+SCROLL_SPACE;
			sheight=height;
		}
		else
		{
			swidth=width+SCROLL_WIDTH;
			
			if (bFloatingWindow)
				sheight=height-25;
			else
				sheight=height;
		}

		hwndScroll=CreateWindow("scrollbar",
			                    "scroll",
								WS_CHILD|WS_VISIBLE|SBS_VERT|SBS_RIGHTALIGN,
								//rect.right-rect.left-20,rect.bottom-rect.top,
								0,0,
								swidth,sheight,
								hwnd,
								(HMENU)0,
								hInstance,
								NULL);

		SetScrollRange();

		EnableWindow(hwndScroll,TRUE);
	}
//	else
//		if (numCols==1 && bAttachedWindow)
//			FLDWidthS -= SCROLL_WIDTH+SCROLL_SPACE+12;

	int numItems = ((height)/((FLDHeight+PROP_SPACING)/numCols));

	// Force full rows unless last item
	if (numItems%numCols>0 &&
		numItems<listSize)
		numItems-=numItems%numCols;

	for(int i=0;i<listSize;i++)
	{
		int theID=STATIC_ID+i;

		// Create the property text window
		props[i].hwndTxt=CreateWindow("static",
			(props[i].strDispName.Length() == 0) ? props[i].strPropName : props[i].strDispName,
									  WS_CHILD|WS_BORDER|SS_NOTIFY,
									  GetXTXTPos(i),
									  GetYTXTPos(i),
									  TXTWidth,
									  TXTHeight,
									  hwnd,
									  (HMENU)theID,
									  hInstance,
									  NULL);

		// Subclass the property text window
		OldStaticProc=(WNDPROC)GetWindowLong(props[i].hwndTxt,GWL_WNDPROC);
		SetWindowLong(props[i].hwndTxt,GWL_WNDPROC,(LONG)StaticProc);

		switch(props[i].ptype)
		{
		case PROPTYPE_EDIT:
			props[i].hwndFld=BuildEditUI(i);
			break;

		case PROPTYPE_LIST:
			props[i].hwndFld=BuildListUI(i,(ListData*)props[i].pdata);
			break;

		case PROPTYPE_EXTLIST:
			props[i].hwndFld=BuildExtListUI(i,(ListData*)props[i].pdata);
			break;

		case PROPTYPE_SLIDER:
			props[i].hwndFld=BuildSliderUI(i,(SliderData*)props[i].pdata);
			break;

		case PROPTYPE_FILE:
			props[i].hwndFld=BuildFileUI(i);
			break;

		case PROPTYPE_SCRIPT:
			props[i].hwndFld=BuildScriptUI(i);
			break;

		case PROPTYPE_COLOR:
			props[i].hwndFld=BuildColorUI(i,(ColorData*)props[i].pdata);
			break;

		case PROPTYPE_CHECK:
			props[i].hwndFld=BuildCheckUI(i);
			break;

		case PROPTYPE_SPINEDIT:
			props[i].hwndFld=BuildSpinEditUI(i);
			break;

		case PROPTYPE_STATIC:
			props[i].hwndFld=BuildStaticUI(i);
			break;

		case PROPTYPE_SLIDERNUM:
			props[i].hwndFld=BuildSliderNumUI(i);
			break;

		case PROPTYPE_NODE:
			props[i].hwndFld=BuildNodeUI(i);
			break;

		case PROPTYPE_MULTI:
			props[i].hwndFld=BuildMultiUI(i);
			break;

		case PROPTYPE_FLAGS:
			props[i].hwndFld=BuildFlagsUI(i);
			break;

		case PROPTYPE_MTL:
			props[i].hwndFld=BuildMtlUI(i);
			break;

		case PROPTYPE_COLORPOPUP:
			props[i].hwndFld=BuildColorPopup(i);
			break;
		}

		ShowWindow(props[i].hwndTxt,SW_HIDE);
		ShowWindow(props[i].hwndFld,SW_HIDE);

		/*
		if (i<numItems)
		{
			ShowWindow(props[i].hwndTxt,SW_SHOW);
			ShowWindow(props[i].hwndFld,SW_SHOW);
		}

		if (!props[i].bVisible)
		{
			ShowWindow(props[i].hwndTxt,SW_HIDE);
			ShowWindow(props[i].hwndFld,SW_HIDE);
		}
		*/
	}

	// Calculate the proper window height
	int propHeight=(listSize+1)*(FLDHeight+PROP_SPACING)+9;
	//SetWindowPos(hwnd,HWND_TOPMOST,0,0,width,propHeight,SWP_NOMOVE|SWP_SHOWWINDOW);
	
	//SetWindowPos(hwnd,HWND_TOPMOST,x,y,width+SCROLL_WIDTH+SCROLL_SPACE,propHeight,SWP_SHOWWINDOW);
	
	if (!bAttachedWindow)
	{
		if (reqHeight>height*numCols)
			//SetWindowPos(hwnd,HWND_TOPMOST,x,y,width,height,SWP_SHOWWINDOW);
			SetWindowPos(hwnd,HWND_TOPMOST,x,y,width+SCROLL_WIDTH+SCROLL_SPACE,height,SWP_SHOWWINDOW);
		else
			SetWindowPos(hwnd,HWND_TOPMOST,x,y,width,propHeight,SWP_SHOWWINDOW);
	}
	
	// Create the accept button at the bottom of the property list
	if (bHasApply)
	{
		hwndApply=CreateWindow("button",
			                    "&Apply",
								WS_CHILD|WS_VISIBLE,
								rectRefTXT.left,(FLDHeight+PROP_SPACING)*i+4,
								TXTWidth*2+10,TXTHeight,
								hwnd,
								(HMENU)IDC_ACCEPT,
								hInstance,
								NULL);
	}

	if (!fpApply)
		EnableWindow(hwndApply,FALSE);

	bSetValLocked=false;
	RestoreInternalValues();
	UpdateUI(0);
}

int PropList::GetXTXTPos(int index)
{
	int col=index%numCols;
	return rectRefTXT.left+((TXTWidth+PROP_HSPACING)+(FLDWidth+PROP_HSPACING))*col;
}

int PropList::GetXFLDPos(int index)
{
	int col=index%numCols;
	return rectRefFLD.left+((TXTWidth+PROP_HSPACING)+(FLDWidth+PROP_HSPACING))*col;
}

int PropList::GetYTXTPos(int index)
{
	//int col=index%numCols;
	return (FLDHeight+PROP_SPACING)*(index/numCols);
	//(FLDHeight+PROP_SPACING)*i
}

int PropList::GetYFLDPos(int index)
{
	//int col=index%numCols;
	return (FLDHeight+PROP_SPACING)*(index/numCols);
}

HWND PropList::BuildEditUI(int index)
{
	HWND hwndEdit;

	hwndEdit=CreateWindow("CustEdit",
				          props[index].strPropName,
						  WS_CHILD,
						  GetXFLDPos(index),
						  GetYFLDPos(index),
						  FLDWidthS,
						  FLDHeight,
						  hwnd,
						  (HMENU)index,
						  hInstance,
						  NULL);

	return hwndEdit;
}

HWND PropList::BuildStaticUI(int index)
{
	HWND hwndStatic;

	StaticCtl* sc = (StaticCtl*)props[index].pdata;

	hwndStatic=CreateWindow("StaticCtl",
                            "",
							WS_CHILD|WS_BORDER,
							GetXFLDPos(index),
							GetYFLDPos(index),
							FLDWidthS,
							FLDHeight,
							hwnd,
							(HMENU)index,
							hInstance,
							sc);

	sc->Attach(hwndStatic);

	return hwndStatic;
}

HWND PropList::BuildSpinEditUI(int index)
{
	HWND hwndSpinEdit;

	SpinEdit* spe = (SpinEdit*)props[index].pdata;

	hwndSpinEdit=CreateWindow(SPIN_EDIT_NAME,
							  props[index].strPropName,
							  WS_CHILD,
							  GetXFLDPos(index),
							  GetYFLDPos(index),
							  FLDWidthS,
							  FLDHeight,
							  hwnd,
							  (HMENU)index,
							  hInstance,
							  spe);

	return hwndSpinEdit;
}

HWND PropList::BuildSliderNumUI(int index)
{
	HWND hwndSliderNum;

	SliderNum* sn = (SliderNum*)props[index].pdata;

	hwndSliderNum = CreateWindow(SLIDENUM_CLASSNAME,
								 props[index].strPropName,
								 WS_CHILD,
								 GetXFLDPos(index),
								 GetYFLDPos(index),
								 FLDWidthS,
								 FLDHeight,
								 hwnd,
								 (HMENU)index,
								 hInstance,
								 sn);

	return hwndSliderNum;
}

HWND PropList::BuildListUI(int index,ListData* data)
{
	HWND hwndEdit;

	hwndEdit=CreateWindow("combobox",
				          props[index].strPropName,
						  WS_CHILD|CBS_DROPDOWNLIST|CBS_SORT|WS_VSCROLL,
						  GetXFLDPos(index),
						  GetYFLDPos(index),
						  FLDWidthS,
						  FLDHeight+PROP_DROPDOWNHEIGHT,
						  hwnd,
						  (HMENU)index,
						  hInstance,
						  NULL);

	int numEntries=data->list.GetSize();

	// Add all the entries to the list
	for(int j=0;j<numEntries;j++)
		SendMessage(hwndEdit,CB_ADDSTRING,0,(LPARAM)(char*)data->list[j]);

	// Set the starting selection for the combo box
	SendMessage(hwndEdit,CB_SETCURSEL,data->curSel,0);

	return hwndEdit;
}

HWND PropList::BuildExtListUI(int index,ListData* data)
{
	HWND hwndEdit;

	hwndEdit=CreateWindow("combobox",
				          props[index].strPropName,
						  WS_CHILD|CBS_DROPDOWN|CBS_AUTOHSCROLL|CBS_SORT|WS_VSCROLL,
						  GetXFLDPos(index),
						  GetYFLDPos(index),
						  FLDWidthS,
						  FLDHeight+PROP_DROPDOWNHEIGHT,
						  hwnd,
						  (HMENU)index,
						  hInstance,
						  NULL);

	int numEntries=data->list.GetSize();

	// Add all the entries to the list
	for(int j=0;j<numEntries;j++)
		SendMessage(hwndEdit,CB_ADDSTRING,0,(LPARAM)(char*)data->list[j]);

	// Set the starting selection for the combo box
	SendMessage(hwndEdit,CB_SETCURSEL,data->curSel,0);

	return hwndEdit;
}

HWND PropList::BuildSliderUI(int index,SliderData* data)
{
	HWND            hwndSlider;
	ISliderControl* ISlider;

	hwndSlider=CreateWindow("SliderControl",
				            props[index].strPropName,
						    WS_CHILD,
						    GetXFLDPos(index),
							GetYFLDPos(index),
						    FLDWidthS,
							FLDHeight,
						    hwnd,
						    (HMENU)index,
						    hInstance,
						    NULL);

	ISlider=GetISlider(hwndSlider);
	ISlider->SetLimits(data->min,data->max);
	ReleaseISlider(ISlider);

	return hwndSlider;
}

HWND PropList::BuildFileUI(int index)
{
	HWND            hwndFile;

	hwndFile=CreateWindow("button",
		                  "None",
						  WS_CHILD,
						  GetXFLDPos(index),
						  GetYFLDPos(index),
						  FLDWidthS,
						  FLDHeight,
						  hwnd,
						  (HMENU)index,
						  hInstance,
						  NULL);

	// Store the location of the name so the file dialog will know later on
	// when the button is pushed
	//LONG val=SetWindowLong(hwndFile,GWL_ID,100);
	//DWORD err=GetLastError();

	return hwndFile;						  
}

HWND PropList::BuildColorUI(int index,ColorData* cdata)
{
	HWND            hwndColor;

	hwndColor=CreateWindow("ColorSwatch",
		                   props[index].strPropName,
						   WS_CHILD,
						   GetXFLDPos(index),
						   GetYFLDPos(index),
						   FLDWidthS,
						   FLDHeight,
						   hwnd,
						   (HMENU)index,
						   hInstance,
						   NULL);

	cdata->IColor=GetIColorSwatch(hwndColor,cdata->DefColor,cdata->DlgText);

	return hwndColor;
}

HWND PropList::BuildCheckUI(int index)
{
	HWND            hwndCheck;

	hwndCheck=CreateWindow("button",
		                   props[index].strPropName,
						   WS_CHILD|BS_AUTOCHECKBOX,
						   GetXFLDPos(index),
						   GetYFLDPos(index),
						   FLDWidthS,FLDHeight,
						   hwnd,
						   (HMENU)index,
						   hInstance,
						   NULL);

	return hwndCheck;
}

HWND PropList::BuildScriptUI(int index)
{
	HWND           hwndScript;

	ScriptSelect* ss = (ScriptSelect*)props[index].pdata;

	hwndScript = CreateWindow("ScriptSelect",
		                      props[index].strPropName,
							  WS_CHILD,
							  GetXFLDPos(index),
							  GetYFLDPos(index),
							  FLDWidthS,FLDHeight,
							  hwnd,
							  (HMENU)index,
							  hInstance,
							  ss);

	return hwndScript;
}

HWND PropList::BuildNodeUI(int index)
{
	HWND           hwndNode;

	NodeSelect* ns = (NodeSelect*)props[index].pdata;

	hwndNode = CreateWindow("NodeSelector",
		                    props[index].strPropName,
							WS_CHILD,
							GetXFLDPos(index),
							GetYFLDPos(index),
							FLDWidthS,FLDHeight,
							hwnd,
							(HMENU)index,
							hInstance,
							ns);

	return hwndNode;
}

HWND PropList::BuildMultiUI(int index)
{
	HWND           hwndMulti;

	MultiList* ml = (MultiList*)props[index].pdata;

	hwndMulti = CreateWindow("MultiList",
		                      props[index].strPropName,
							  WS_CHILD,
							  GetXFLDPos(index),
							  GetYFLDPos(index),
							  FLDWidthS,FLDHeight,
							  hwnd,
							  (HMENU)index,
							  hInstance,
							  ml);

	return hwndMulti;
}

HWND PropList::BuildFlagsUI(int index)
{
	HWND           hwndFlags;

	FlagSelect* fs = (FlagSelect*)props[index].pdata;

	hwndFlags = CreateWindow("FlagSelect",
		                      props[index].strPropName,
							  WS_CHILD,
							  GetXFLDPos(index),
							  GetYFLDPos(index),
							  FLDWidthS,FLDHeight,
							  hwnd,
							  (HMENU)index,
							  hInstance,
							  fs);

	return hwndFlags;
}

HWND PropList::BuildMtlUI(int index)
{
	HWND           hwndFlags;

	FlagSelect* fs = (FlagSelect*)props[index].pdata;

	hwndFlags = CreateWindow("MtlSelect",
		                      props[index].strPropName,
							  WS_CHILD,
							  GetXFLDPos(index),
							  GetYFLDPos(index),
							  FLDWidthS,FLDHeight,
							  hwnd,
							  (HMENU)index,
							  hInstance,
							  fs);

	return hwndFlags;
}

HWND PropList::BuildColorPopup(int index)
{
	HWND           hwndColorPopup;

	ColorPopup* cp = (ColorPopup*)props[index].pdata;

	hwndColorPopup = CreateWindow("ColorPopup",
		                          props[index].strPropName,
								  WS_CHILD,
								  GetXFLDPos(index),
								  GetYFLDPos(index),
								  FLDWidthS,FLDHeight,
								  hwnd,
								  (HMENU)index,
								  hInstance,
								  cp);

	return hwndColorPopup;
}

CStr PropList::GetName(int index)
{
	return props[index].strPropName;
}

PropType PropList::GetType(int index)
{
	return props[index].ptype;
}

bool PropList::GetValue(CStr name,CStr& propVal)
{
	name.toLower();

	CStr val;
	bool bFound = false;

	Link<Property>* prop = props.GetHead();

	while(prop)
	{
		val = prop->data.strPropName;
		val.toLower();

		if (name == val)
		{
			bFound = true;
			break;
		}

		prop = prop->next;
	}

	if (bFound)
	{
		int index = props.GetIndex(prop);
		return GetValue(index,propVal);
	}

	return false;
}

bool PropList::GetValue(int index,CStr& strVal)
{
	// If there are unadded properties return their value
	if (propVals.GetSize()>0)
	{
		Link<PropVal>* curNode=propVals.GetHead();

		while(curNode)
		{
			if (curNode->data.index==index)
			{
				strVal=curNode->data.value;
				return true;
			}

			curNode=curNode->next;
		}
	}
		

	Property* propval=&props[index];

	if (!propval)
		return false;

	ICustEdit*      IEdit;
	ISliderControl* ISlider;
	IColorSwatch*   IColor;
	SpinEdit*       sedit;

	char buf[512];
	int  val;

	COLORREF cval;

	if (!propval->hwndFld)
		return false;

	switch(propval->ptype)
	{
	case PROPTYPE_EDIT:
		IEdit=GetICustEdit(propval->hwndFld);

		if (IEdit)
		{
			IEdit->GetText(buf,512);
			strVal=buf;
			ReleaseICustEdit(IEdit);
		}
		else
			return false;

		break;

	case PROPTYPE_STATIC:
		{
			StaticCtl* sc = (StaticCtl*)propval->pdata;
			if (sc)
				strVal = sc->GetValue();
		}
		break;

	case PROPTYPE_SPINEDIT:
		sedit = (SpinEdit*)propval->pdata;
		if (sedit)
		{
			sedit->GetValue(strVal);
		}
		break;

	case PROPTYPE_MULTI:
		{
			MultiList* ml = (MultiList*)propval->pdata;
			if (ml)
				strVal = ml->GetValue();
		}
		break;

	case PROPTYPE_FLAGS:
		{
			FlagSelect* fs = (FlagSelect*)propval->pdata;
			if (fs)
				strVal = fs->GetValue();
		}
		break;

	case PROPTYPE_SLIDERNUM:
		{
			SliderNum* snum = (SliderNum*)propval->pdata;
			if (snum)
			{
				sprintf(buf,"%i",snum->GetValue());
				strVal=buf;
			}
		}
		break;

	case PROPTYPE_LIST:
		GetWindowText(propval->hwndFld,buf,512);
		strVal=buf;
		break;

	case PROPTYPE_EXTLIST:
		GetWindowText(propval->hwndFld,buf,512);
		strVal=buf;
		break;

	case PROPTYPE_SLIDER:
		ISlider=GetISlider(propval->hwndFld);

		if (ISlider)
		{
			val=ISlider->GetIVal();
			ReleaseISlider(ISlider);
			_itoa(val,buf,10);
			strVal=buf;
		}
		else
			return false;

		break;

	case PROPTYPE_SCRIPT:
		GetWindowText(propval->hwndFld,buf,512);
		strVal=buf;		
		break;

	case PROPTYPE_NODE:
		{
			NodeSelect* ns = (NodeSelect*)propval->pdata;
			strVal = ns->GetValue();
		}
		break;

	case PROPTYPE_FILE:
		GetWindowText(propval->hwndFld,buf,512);
		strVal=buf;
		break;

	case PROPTYPE_COLOR:
		IColor=((ColorData*)propval->pdata)->IColor;

		if (IColor)
		{
			cval=IColor->GetColor();
			sprintf(buf,"[%i,%i,%i]",GetRValue(cval),GetGValue(cval),GetBValue(cval));
			strVal=buf;
		}
		else
			return false;

		break;

	case PROPTYPE_CHECK:
		if (SendMessage(propval->hwndFld,BM_GETCHECK,0,0)==BST_CHECKED)
			strVal="TRUE";
		else
			strVal="FALSE";

		break;

	case PROPTYPE_MTL:
		{
			MtlSelect* ms = (MtlSelect*)propval->pdata;
			strVal = ms->GetValue();
		}
		break;

	case PROPTYPE_COLORPOPUP:
		{
			ColorPopup* cp = (ColorPopup*)propval->pdata;
			strVal = cp->GetValue();
		}
	}

	return true;
}

void PropList::RestoreInternalValues()
{
	Link<PropVal>* curNode=propVals.GetHead();

	while(curNode)
	{
		SetValue(curNode->data.index,curNode->data.value);
		curNode=curNode->next;
	}

	propVals.Clear();
}

bool PropList::SetValue(CStr strName,CStr strValue)
{
	Property srchProp;
	srchProp.strPropName = strName;

	Link<Property>* prop = props.Find(&srchProp);

	if (prop)
		return SetValue(props.GetIndex(prop),strValue);

	return false;
}

bool PropList::SetValue(int index,CStr strValue)
{
	if (index>props.GetSize()-1)
		return false;

	if (bSetValLocked)
	{
		PropVal newPropVal;

		newPropVal.index=index;
		newPropVal.value=strValue;

		propVals.Add(&newPropVal);
		return false;
	}

	// Must be a reference so the property doesn't get copied
	// (if it's copied there will be 2 pointers to the property data, since the property
	//  destroys it's own data, the data would get destroyed twice and cause a crash down the line)
	Property& propval=props[index];

	ICustEdit*      IEdit;
	ISliderControl* ISlider;
	IColorSwatch*   IColor;
	SpinEdit*       sedit;

	int  val;

	int red,green,blue;

	switch(propval.ptype)
	{
	case PROPTYPE_EDIT:
		IEdit=GetICustEdit(propval.hwndFld);

		if (!IEdit)
			return false;

		IEdit->SetText(strValue);
		ReleaseICustEdit(IEdit);
		break;

	case PROPTYPE_STATIC:
		{
			StaticCtl* sc = (StaticCtl*)propval.pdata;

			if (sc)
				sc->SetValue(strValue);
		}
		break;

	case PROPTYPE_SPINEDIT:
		sedit=(SpinEdit*)propval.pdata;

		if (sedit)
		{
			sedit->SetValue(atof(strValue));
		}
		break;

	case PROPTYPE_MULTI:
		{
			MultiList* ml = (MultiList*)propval.pdata;
			
			if (ml)
				ml->SetValue(strValue);
		}
		break;

	case PROPTYPE_FLAGS:
		{
			FlagSelect* fs = (FlagSelect*)propval.pdata;

			if (fs)
				fs->SetValue(strValue);
		}
		break;

	case PROPTYPE_SLIDERNUM:
		{
			SliderNum* snum = (SliderNum*)propval.pdata;

			if (snum)
			{
				snum->SetValue(atoi(strValue));
			}
		}
		break;

	case PROPTYPE_LIST:
		val=SendMessage(propval.hwndFld,CB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)strValue);
		if (val==CB_ERR)
			return false;

		SendMessage(propval.hwndFld,CB_SETCURSEL,(WPARAM)val,0);
		break;

	case PROPTYPE_EXTLIST:
		val=SendMessage(propval.hwndFld,CB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)strValue);
		
		if (val!=CB_ERR)
			SendMessage(propval.hwndFld,CB_SETCURSEL,(WPARAM)val,0);
		else
			SetWindowText(propval.hwndFld,(char*)strValue);
		break;

	case PROPTYPE_SLIDER:
		ISlider=GetISlider(propval.hwndFld);
		ISlider->SetValue(atoi(strValue),FALSE);
		ReleaseISlider(ISlider);
		break;

	case PROPTYPE_SCRIPT:
		SetWindowText(propval.hwndFld,(char*)strValue);
		break;

	case PROPTYPE_NODE:
		{
			NodeSelect* ns = (NodeSelect*)propval.pdata;
			ns->SetValue(strValue);
		}
		break;

	case PROPTYPE_FILE:
		SetWindowText(propval.hwndFld,(char*)strValue);
		break;

	case PROPTYPE_COLOR:
		IColor=((ColorData*)propval.pdata)->IColor;
		
		if (!IColor)
			return false;

		sscanf((char*)strValue,"[%i,%i,%i]",&red,&green,&blue);
		IColor->SetColor(RGB(red,green,blue));
		break;

	case PROPTYPE_CHECK:
		if (strValue==CStr("TRUE"))
		{
			SendMessage(propval.hwndFld,BM_SETCHECK,(WPARAM)BST_CHECKED,0);
			return true;
		}
		
		if (strValue==CStr("FALSE"))
		{
			SendMessage(propval.hwndFld,BM_SETCHECK,(WPARAM)BST_UNCHECKED,0);
			return true;
		}
		
		return false;

	case PROPTYPE_MTL:
		{
			MtlSelect* ms = (MtlSelect*)propval.pdata;
			ms->SetValue(strValue);
			return true;
		}

	case PROPTYPE_COLORPOPUP:
		{
			ColorPopup* cp = (ColorPopup*)propval.pdata;
			cp->SetValue(strValue);
		}
	}

	return true;
}

void PropList::SetApplyCB(void(*Func)(PropList*,void*),void* pData)
{
	fpApply=Func;
	pApplyData=pData;

	if (Func)
		EnableWindow(hwndApply,TRUE);
	else
		EnableWindow(hwndApply,FALSE);
}

void PropList::SetChangeCB(void(*Func)(PropList*,void*),void* pData)
{
	fpChange=Func;
	pChangeData=pData;
}

void PropList::SetExtListCB(void(*Func)(CStr,CStr,void*),void* pData)
{
	fpExtList=Func;
	pExtListData=pData;
}

void PropList::SetDblClickCB(void(*Func)(PropList*,int,void*),void* pData)
{
	fpDblClick=Func;
	pDblClickData=pData;
}

void PropList::SetFocusCB(void(*Func)(PropList*,int,void*),void* pData)
{
	fpFocus=Func;
	pFocusData=pData;
}

void PropList::SetStaticCB(LRESULT(*Func)(PropList*,HWND,UINT,WPARAM,LPARAM,void*),void* pData)
{
	fpSProc=Func;
	pSProcData=pData;
}

// Get position functions 
int PropList::GetX()
{
	RECT rect;
	GetWindowRect(hwnd,&rect);
	return rect.left;
}

int PropList::GetY()
{
	RECT rect;
	GetWindowRect(hwnd,&rect);
	return rect.top;
}
///

void PropList::ProcExtLists()
{
	char curval[128];
	char buf[128];

	if (!fpExtList)
		return;

	// Scan through the property list for extensible lists
	int numProps=props.GetSize();

	for(int i=0;i<numProps;i++)
	{
		if (props[i].ptype==PROPTYPE_EXTLIST)
		{
			HWND propHwnd=props[i].hwndFld;			
			GetWindowText(propHwnd,curval,128);

			// See if this value already exists
			int numItems=SendMessage(propHwnd,CB_GETCOUNT,0,0);

			for(int j=0;j<numItems;j++)
			{
				SendMessage(propHwnd,CB_GETLBTEXT,(WPARAM)j,(LPARAM)buf);
				
				if (strcmp(curval,buf)==0)
					return;
			}

			// The value doesn't exist, call the callback to add it
			fpExtList(props[i].strPropName,CStr(curval),pExtListData);
		}
	}
}

int PropList::UpdateScroll(HWND scrollBar,int code,int pos)
{
	RECT rect;
	GetWindowRect(hwnd,&rect);

	int width=rect.right-rect.left+PROP_RIGHTSPACING;
	int height=rect.bottom-rect.top;
	
	int scrollpos=SendMessage(scrollBar,SBM_GETPOS,0,0);
	int numItems=((height)/((FLDHeight+PROP_SPACING)/numCols));

	int listCount=props.GetSize();

	switch(code)
	{
	case SB_LINEUP:
		scrollpos-=numCols;
		break;

	case SB_LINEDOWN:
		scrollpos+=numCols;
		break;

	case SB_PAGEUP:
		scrollpos-=numItems;

		if (scrollpos%numCols>0 &&
			scrollpos<=listCount)
			scrollpos-=scrollpos%numCols;
		break;

	case SB_PAGEDOWN:
		scrollpos+=numItems;

		if (scrollpos%numCols>0 &&
			scrollpos<=listCount)
			scrollpos-=scrollpos%numCols;
		break;

	case SB_THUMBPOSITION:
		scrollpos=pos;

		if (scrollpos%numCols>0)
			scrollpos-=(scrollpos%numCols);		
		break;

	default:
		return scrollpos;
	};

	if (scrollpos<0)
		scrollpos=0;
	
	if (scrollpos>listCount-numCols)
	{
		if (listCount%numCols==0)
			scrollpos=listCount-numCols;
		else
			scrollpos=listCount-listCount%numCols;
	}

	SendMessage(scrollBar,SBM_SETPOS,(WPARAM)scrollpos,(LPARAM)TRUE);
	return scrollpos;
}

void PropList::Attach(HWND hwndThis)
{
	hwnd=hwndThis;
	SetWindowLong(hwnd,GWL_USERDATA,(LONG)this);

	RECT winRect;
	GetClientRect(hwnd,&winRect);

	width =winRect.right-winRect.left;
	origWidth = width;
	height=winRect.bottom-winRect.top;

	winRect.left+=PROP_LEFTSPACING;
	winRect.top +=PROP_TOP;

	rectRefTXT.left=winRect.left;
	rectRefTXT.top=winRect.top;
	rectRefTXT.right=winRect.left+(width/2)-PROP_HSPACING;
	rectRefTXT.bottom=rectRefTXT.top+PROP_HEIGHT;

	//rectRefFLD.left=winRect.left+width/numCols;
	rectRefFLD.left=rectRefTXT.right+PROP_HSPACING;
	rectRefFLD.top =winRect.top;
	rectRefFLD.right=rectRefFLD.left+(width/2)-PROP_HSPACING;
	rectRefFLD.bottom=rectRefFLD.top+PROP_HEIGHT;

	rectRefTXT.right-=(SCROLL_WIDTH+SCROLL_SPACE);
	rectRefFLD.right-=(SCROLL_WIDTH+SCROLL_SPACE);

	TXTWidth=rectRefTXT.right-rectRefTXT.left;
	FLDWidth=rectRefFLD.right-rectRefFLD.left;

	TXTHeight=rectRefTXT.bottom-rectRefTXT.top;
	FLDHeight=rectRefFLD.bottom-rectRefFLD.top;

	TXTWidth/=numCols;
	FLDWidth/=numCols;

	rectRefFLD.left=rectRefTXT.left+TXTWidth+PROP_HSPACING;
	rectRefFLD.right=rectRefFLD.left+FLDWidth;

	bAttachedWindow=true;
}

BOOL PropList::ProcColor(HWND hwndCtl)
{
	int ID=GetWindowLong(hwndCtl,GWL_ID);

	if (ID>=STATIC_ID)
	{
		switch(props[ID-STATIC_ID].color)
		{
		case PROPCOLOR_GRAY:
			return (BOOL)hBGBrush;

		case PROPCOLOR_RED:
			return (BOOL)hBGBrushR;

		case PROPCOLOR_GREEN:
			return (BOOL)hBGBrushG;

		case PROPCOLOR_BLUE:
			return (BOOL)hBGBrushB;

		case PROPCOLOR_PURPLE:
			return (BOOL)hBGBrushP;

		case PROPCOLOR_WHITE:
			return (BOOL)hBGBrushW;
		}
	}

	return (BOOL)hBGBrush;
}

PropColor PropList::GetColor(int index)
{
	return props[index].color;
}

bool PropList::SetColor(int index,PropColor color)
{
	if (index>props.GetSize()-1)
		return false;

	props[index].color=color;

	// Refresh if changed after UI built
	if (props[index].hwndTxt)
		InvalidateRect(props[index].hwndTxt,NULL,TRUE);

	return true;
}

void PropList::HasApply(bool bValue)
{
	bHasApply=bValue;
}

bool PropList::WasUpdated(int index)
{
	return props[index].bWasUpdated;
}

bool PropList::WasUpdated()
{
	Link<Property>* curProp=props.GetHead();

	while(curProp)
	{
		if (curProp->data.bWasUpdated)
			return true;

		curProp=curProp->next;
	}

	return false;
}

void PropList::ResetUpdateStatus()
{
	Link<Property>* curProp=props.GetHead();

	while(curProp)
	{
		curProp->data.bWasUpdated=false;
		curProp=curProp->next;
	}

	lastModID = -1;
}

void PropList::SetFlags(int index,int flags)
{
	props[index].flags=flags;
}

DWORD PropList::GetFlags(int index)
{
	if (index >= props.GetSize())
		__asm int 3;

	return props[index].flags;
}

void PropList::SetMod(int index,bool bVal)
{
	if (index > props.GetSize() - 1)
		return;

	props[index].bWasUpdated=bVal;
	lastModID = index;					// TEST:
}

int PropList::GetPropFromHWnd(HWND hwndScan)
{
	int index=0;
	Link<Property>* curProp=props.GetHead();

	while(curProp)
	{
		if (curProp->data.hwndTxt==hwndScan ||
			curProp->data.hwndFld==hwndScan)
			return index;

		index++;
		curProp=curProp->next;
	}

	return -1;
}

LRESULT CALLBACK PropList::StaticProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	HWND hwndParent=GetParent(hwnd);
	PropList* pthis=(PropList*)GetWindowLong(hwndParent,GWL_USERDATA);

	if (pthis->fpSProc)
	{
		LRESULT rVal=pthis->fpSProc(pthis,hwnd,msg,wParam,lParam,pthis->pSProcData);

		if (rVal!=FALSE)
			return rVal;
	}

	return CallWindowProc(pthis->OldStaticProc,hwnd,msg,wParam,lParam);
}

void PropList::DeleteProp(int index, bool bDelSaved)
{
	int curindex=0;
	Link<Property>* curProp=props.GetHead();

	// Delete property
	while(curProp)
	{
		if (index==curindex)
		{
			props.Remove(curProp);
			break;
		}

		curindex++;
		curProp=curProp->next;
	}

	// Delete saved value
	if (bDelSaved)
	{
		Link<CStr>* curVal=savedValues.GetHead();

		curindex=0;
		while(curVal)
		{
			if (index==curindex)
			{
				savedValues.Remove(curVal);
				break;
			}

			curindex++;
			curVal=curVal->next;
		}
	}

	curIndex = props.GetSize()-1;
	linkAddPos = props.GetTail();
}

CStr PropList::GetDesc(int index)
{
	return props[index].strToolTip;
}

void PropList::SetDesc(int index,CStr str)
{
	props[index].strToolTip=str;
}

void PropList::SetUserData(int index,CStr str)
{
	props[index].strExtData=str;
}

void PropList::SetUserVal(int index,DWORD val)
{
	props[index].userData = val;
}

CStr PropList::GetUserData(int index)
{
	return props[index].strExtData;
}

DWORD PropList::GetUserVal(int index)
{
	return props[index].userData;
}

int PropList::GetReqHeight(int numProps)
{
	return (numProps+1)*(FLDHeight+PROP_SPACING)+9;
}

void PropList::SetAddPos(int i)
{
	Link<Property>* curProp = props.GetHead();

	for(int c=0;c<i;c++)
		curProp = curProp->next;

	if (curProp!=NULL)
		linkAddPos = curProp;
	else
		linkAddPos = props.GetTail();
}

void PropList::RemoveProp(int ID)
{
	Link<Property>* tmpAddlink;
	int             tmpPosIndex;
	int             scrollVal;

	Link<Property>* curProp = props.GetHead();

	for(int c=0;c<ID;c++)
		curProp = curProp->next;

	if (curProp)
	{
		tmpAddlink = curProp->prev;
		
		if (tmpAddlink)
			tmpPosIndex = FindProp(tmpAddlink);
		else
			tmpPosIndex = 0;

		//SaveValues();
		scrollVal = GetScrollVal();	
		DestroyUI();
		//DeleteProp(ID,false);
		DeleteProp(ID);
		//BuildUI();
		//RestoreValues();
		
		if (scrollVal!=-1)
			SetScrollVal(scrollVal);

		curIndex   = tmpPosIndex;
		linkAddPos = tmpAddlink;
	}
}

int PropList::FindProp(Link<Property>* prop)
{
	Link<Property>* curProp = props.GetHead();
	int c=0;

	while(curProp)
	{
		if (curProp==prop)
			return c;

		curProp=curProp->next;
		c++;
	}

	return -1;
}

int PropList::GetScrollVal()
{
	if (hwndScroll)
		return SendMessage(hwndScroll,SBM_GETPOS,0,0);

	return -1;
}

void PropList::SetScrollVal(int val)
{
	if (hwndScroll && val != -1)
		SendMessage(hwndScroll,SBM_SETPOS,(WPARAM)val,(LPARAM)TRUE);	

	UpdateUI(val);
}

int PropList::GetCurIndex()
{
	return curIndex;
}

int PropList::ResetIndexToEnd()
{
	curIndex = props.GetSize()-1;
	linkAddPos = props.GetTail();
	return curIndex;
}

bool PropList::PropExists(CStr name)
{
	Link<Property>* curprop = props.GetHead();

	while(curprop)
	{
		if (curprop->data.strPropName==name)
			return true;

		curprop=curprop->next;
	}

	return false;
}

void PropList::SetPos(int x,int y)
{
	this->x=x;
	this->y=y;
	SetWindowPos(hwnd,HWND_TOPMOST,x,y,width,height,0);
}

void PropList::SetName(int i,CStr name)
{
	props[i].strPropName = name;

	if (props[i].hwndTxt)
		SetWindowText(props[i].hwndTxt,name);
}

PropList& PropList::operator = (PropList& right)
{
	props       = right.props;
	savedValues = right.savedValues;
	propVals    = right.propVals;

	return *this;
}

void PropList::ClearValues()
{
	int numProps = props.GetSize();

	for(int i=0;i<numProps;i++)
		SetValue(i,"");
}

void PropList::AddProperty(Property* prop)
{
	Property newProp;

	newProp.hwndTxt     = NULL;
	newProp.hwndFld     = NULL;
	newProp.strPropName = prop->strPropName;
	newProp.strToolTip  = prop->strToolTip;
	newProp.strExtData  = prop->strExtData;
	newProp.ptype       = prop->ptype;
	newProp.color       = prop->color;
	newProp.bWasUpdated = prop->bWasUpdated;
	newProp.flags       = prop->flags;
	newProp.userData    = prop->userData;

	switch(prop->ptype)
	{
	case PROPTYPE_UNDEFINED:
		newProp.pdata = NULL;
		break;

	case PROPTYPE_EDIT:
		newProp.pdata = NULL;
		break;

	case PROPTYPE_LIST:
		newProp.pdata = new ListData(*((ListData*)prop->pdata));
		break;

	case PROPTYPE_EXTLIST:
		newProp.pdata = new ListData(*((ListData*)prop->pdata));
		break;

	case PROPTYPE_SLIDER:
		newProp.pdata = NULL;
		break;

	case PROPTYPE_SCRIPT:
		newProp.pdata = new ScriptSelect(*((ScriptSelect*)prop->pdata));
		break;

	case PROPTYPE_FILE:
		newProp.pdata = new FileData(*((FileData*)prop->pdata));
		break;

	case PROPTYPE_COLOR:
		newProp.pdata = new ColorData(*((ColorData*)prop->pdata));
		break;

	case PROPTYPE_CHECK:
		newProp.pdata = NULL;		
		break;

	case PROPTYPE_SPINEDIT:
		newProp.pdata = new SpinEdit(*((SpinEdit*)prop->pdata));
		break;

	case PROPTYPE_STATIC:
		newProp.pdata = NULL;
		break;

	case PROPTYPE_SLIDERNUM:
		newProp.pdata = new SliderNum(*((SliderNum*)prop->pdata));
		break;

	case PROPTYPE_NODE:
		newProp.pdata = new NodeSelect(*((NodeSelect*)prop->pdata));
		break;

	case PROPTYPE_MULTI:
		newProp.pdata = new MultiList(*((MultiList*)prop->pdata));
		break;

	case PROPTYPE_FLAGS:
		newProp.pdata = new FlagSelect(*((FlagSelect*)prop->pdata));
		break;
		
	case PROPTYPE_MTL:
		newProp.pdata = new MtlSelect(*((MtlSelect*)prop->pdata));
		break;
	}
}

