/*
	TriggerLayout.cpp
	Trigger Layout Tool/UI
*/

#include "Trigger.h"
#include "TriggerLayout.h"
#include "Resource.h"
#include "Next.h"
#include "AppData.h"

#ifdef TRUE
#undef TRUE
#define TRUE 1
#endif
#ifdef FALSE
#undef FALSE
#define FALSE 0
#endif

float Dist2(Point3 pt1, Point3 pt2)
{
	return ((pt2.x-pt1.x)*(pt2.x-pt1.x))+((pt2.y-pt1.y)*(pt2.y-pt1.y))+((pt2.z-pt1.z)*(pt2.z-pt1.z));
}

class PickSplineNode : public PickModeCallback,
		public PickNodeCallback 
{
public:			
	TriggerLayout *trigLayout;
	PickSplineNode() { trigLayout = NULL; }
	BOOL HitTest(IObjParam *ip,HWND hWnd,ViewExp *vpt,IPoint2 m,int flags);		
	BOOL Pick(IObjParam *ip,ViewExp *vpt);		
	void EnterMode(IObjParam *ip);
	void ExitMode(IObjParam *ip);
	BOOL Filter(INode *node);
	PickNodeCallback *GetFilter() {return this;}
	BOOL RightClick(IObjParam *ip,ViewExp *vpt) {return TRUE;}
};
static PickSplineNode theSplinePickMode;


BOOL PickSplineNode::Filter(INode *node)
{
	Object* obj=node->EvalWorldState(0).obj;

	if (obj->ClassID()==Class_ID(SPLINE3D_CLASS_ID,0))
		return TRUE;

	return FALSE;
}

BOOL PickSplineNode::HitTest(IObjParam *ip,HWND hWnd,ViewExp *vpt,IPoint2 m,int flags)
{	
	if (ip->PickNode(hWnd,m,this))
		return TRUE;
	else
		return FALSE;
}

BOOL PickSplineNode::Pick(IObjParam *ip,ViewExp *vpt)
{
	INode *node = vpt->GetClosestHit();
	
	if (node)
	{
		trigLayout->SetSpline(node);
		ExitMode(ip);
	}

	return FALSE;
}

void PickSplineNode::EnterMode(IObjParam *ip)
{
	if (trigLayout && trigLayout->ip && trigLayout->IBSplineSelect)
	{
		trigLayout->ResetTrigList();
		trigLayout->IBSplineSelect->SetCheck(TRUE);
	}

	GetCOREInterface()->PushPrompt("Select Spline Target");
}

void PickSplineNode::ExitMode(IObjParam *ip)
{
	if (trigLayout && trigLayout->ip && trigLayout->IBSplineSelect)
		trigLayout->IBSplineSelect->SetCheck(FALSE);

	GetCOREInterface()->PopPrompt();
}

/////////////////////////////////////////////////////////////////////////////////////////////////

class PickGeomNode : public PickModeCallback,
		public PickNodeCallback 
{
public:			
	TriggerLayout *trigLayout;
	PickGeomNode() { trigLayout = NULL; }
	BOOL HitTest(IObjParam *ip,HWND hWnd,ViewExp *vpt,IPoint2 m,int flags);		
	BOOL Pick(IObjParam *ip,ViewExp *vpt);		
	void EnterMode(IObjParam *ip);
	void ExitMode(IObjParam *ip);		
	BOOL Filter(INode *node);
	PickNodeCallback *GetFilter() {return this;}
	BOOL RightClick(IObjParam *ip,ViewExp *vpt) {return TRUE;}
};
static PickGeomNode theGeomPickMode;


BOOL PickGeomNode::Filter(INode *node)
{
	Object* obj=node->EvalWorldState(0).obj;

	if (obj->SuperClassID()==GEOMOBJECT_CLASS_ID)
		return TRUE;

	return FALSE;
}

BOOL PickGeomNode::HitTest(IObjParam *ip,HWND hWnd,ViewExp *vpt,IPoint2 m,int flags)
{	
	if (ip->PickNode(hWnd,m,this))
		return TRUE;
	else
		return FALSE;
}

BOOL PickGeomNode::Pick(IObjParam *ip,ViewExp *vpt)
{
	INode *node = vpt->GetClosestHit();
	
	if (node)
	{
		trigLayout->SetGeom(node);
		ExitMode(ip);
	}

	return FALSE;
}

void PickGeomNode::EnterMode(IObjParam *ip)
{
	if (trigLayout && trigLayout->ip && trigLayout->IBGeomSelect)
	{
		trigLayout->ResetTrigList();
		trigLayout->IBGeomSelect->SetCheck(TRUE);
	}

	GetCOREInterface()->PushPrompt("Select Spline Target");
}

void PickGeomNode::ExitMode(IObjParam *ip)
{
	if (trigLayout && trigLayout->ip && trigLayout->IBGeomSelect)
		trigLayout->IBGeomSelect->SetCheck(FALSE);

	GetCOREInterface()->PopPrompt();
}

//////////////////////////////////////////////////////////////////////////////////////////////////


BOOL TriggerLayout::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	TriggerLayout* pthis=(TriggerLayout*)GetWindowLong(hwnd,GWL_USERDATA);

	switch(msg)
	{
	case CC_SPINNER_CHANGE:

		if (IsDlgButtonChecked(hwnd,IDC_INTERACTIVE)==BST_CHECKED)
		{
			pthis->UpdateTriggers();
			return TRUE;
		}

		return FALSE;
	
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_UPDATE:
			pthis->UpdateTriggers();
			return TRUE;

		case IDC_PICKSPLINE:
			pthis->ip->SetPickMode(&theSplinePickMode);
			return TRUE;

		case IDC_PICKGEOM:
			pthis->ip->SetPickMode(&theGeomPickMode);
			return TRUE;

		case IDC_CLEAR:
			pthis->ClearTriggers();
			return TRUE;

		case IDC_LIMITMATCH:
		case IDC_ASSIGNPROPS:
		case IDC_INTERACTIVE:
		case IDC_REVERSE:
		case IDC_LINKNODES:
			pthis->UpdateTriggers();
			return TRUE;
		}
	}

	return FALSE;
}

TriggerLayout::TriggerLayout()
{
	ip=NULL;
	theSplinePickMode.trigLayout=this;
	theGeomPickMode.trigLayout=this;

	hwndLayout=NULL;

	nodeSpline=NULL;
	nodeGeom=NULL;

	IEditX=NULL;
	IEditY=NULL;
	IEditZ=NULL;
	IEditDistrib=NULL;
	
	ISpinX=NULL;
	ISpinY=NULL;
	ISpinZ=NULL;
	ISpinDistrib=NULL;

	IBSplineSelect=NULL;
	IBGeomSelect=NULL;
}

TriggerLayout::~TriggerLayout()
{

}

void TriggerLayout::BeginEditParams(IObjParam *ip, ULONG flags, Animatable* prev)
{
	this->ip=ip;
	
	hwndLayout=ip->AddRollupPage(hInstance,MAKEINTRESOURCE(IDD_TRIGGERSPLINE),DlgProc,"Trigger Layout",0,APPENDROLL_CLOSED);
	SetWindowLong(hwndLayout,GWL_USERDATA,(LONG)this);

	// Setup UI
	IEditX=GetICustEdit(GetDlgItem(hwndLayout,IDC_EDITX));
	IEditY=GetICustEdit(GetDlgItem(hwndLayout,IDC_EDITY));
	IEditZ=GetICustEdit(GetDlgItem(hwndLayout,IDC_EDITZ));

	IEditDistrib=GetICustEdit(GetDlgItem(hwndLayout,IDC_EDITDISTRIB));

	ISpinX=GetISpinner(GetDlgItem(hwndLayout,IDC_SPINX));
	ISpinY=GetISpinner(GetDlgItem(hwndLayout,IDC_SPINY));
	ISpinZ=GetISpinner(GetDlgItem(hwndLayout,IDC_SPINZ));

	ISpinDistrib=GetISpinner(GetDlgItem(hwndLayout,IDC_SPINDISTRIB));

	IBSplineSelect=GetICustButton(GetDlgItem(hwndLayout,IDC_PICKSPLINE));
	IBGeomSelect=GetICustButton(GetDlgItem(hwndLayout,IDC_PICKGEOM));

	ISpinX->SetLimits(0.0f,100.0f);
	ISpinY->SetLimits(0.0f,100.0f);
	ISpinZ->SetLimits(0.0f,100.0f);
	
	ISpinDistrib->SetScale(0.001f);
	ISpinDistrib->SetLimits(0.001f,1.000f);
	ISpinDistrib->SetValue(0.1f,TRUE);

	ISpinX->LinkToEdit(GetDlgItem(hwndLayout,IDC_EDITX),EDITTYPE_FLOAT);
	ISpinY->LinkToEdit(GetDlgItem(hwndLayout,IDC_EDITY),EDITTYPE_FLOAT);
	ISpinZ->LinkToEdit(GetDlgItem(hwndLayout,IDC_EDITZ),EDITTYPE_FLOAT);

	ISpinDistrib->LinkToEdit(GetDlgItem(hwndLayout,IDC_EDITDISTRIB),EDITTYPE_FLOAT);

	IBSplineSelect->SetType(CBT_CHECK);
	IBGeomSelect->SetType(CBT_CHECK);
}

void TriggerLayout::EndEditParams(IObjParam *ip, ULONG flags, Animatable* next)
{
	if (hwndLayout)
	{
		ip->DeleteRollupPage(hwndLayout);
		hwndLayout=NULL;
	}

	ReleaseICustEdit(IEditX);
	IEditX=NULL;

	ReleaseICustEdit(IEditY);
	IEditY=NULL;

	ReleaseICustEdit(IEditZ);
	IEditZ=NULL;

	ReleaseICustEdit(IEditDistrib);
	IEditDistrib=NULL;

	ReleaseISpinner(ISpinX);
	ISpinX=NULL;

	ReleaseISpinner(ISpinY);
	ISpinY=NULL;

	ReleaseISpinner(ISpinZ);
	ISpinZ=NULL;

	ReleaseISpinner(ISpinDistrib);
	ISpinDistrib=NULL;
	
	ReleaseICustButton(IBSplineSelect);
	IBSplineSelect=NULL;

	ReleaseICustButton(IBGeomSelect);
	IBGeomSelect=NULL;
}

void TriggerLayout::SetSpline(INode* newSpline)
{
	nodeSpline=newSpline;
	SetDlgItemText(hwndLayout,IDC_SPLINENAME,(char*)newSpline->GetName());
}

void TriggerLayout::SetGeom(INode* newGeom)
{
	nodeGeom=newGeom;
	SetDlgItemText(hwndLayout,IDC_GEOMNAME,(char*)newGeom->GetName());
}

void TriggerLayout::UpdateTriggers()
{
	int time=ip->GetTime();

	if (!nodeSpline)
	{
		MessageBox(hwndLayout,"No spline has been selected.","Couldn't update triggers",MB_ICONWARNING|MB_OK);
		return;
	}

	// Delete all the triggers created during the last update
	ClearTriggers();

	// Now, lets get the spline and geometry objects
	ShapeObject* shapeObj=(ShapeObject*)nodeSpline->EvalWorldState(time).obj;

	Matrix3 mat;
	mat=nodeSpline->GetNodeTM(time);

	int numCurves=shapeObj->NumberOfCurves();		// TODO: Process multi-curve objects
	int numVerts =shapeObj->NumberOfVertices(0);

	if (shapeObj)
	{
		// Get values from UI
		float tolX=IEditX->GetFloat();
		float tolY=IEditY->GetFloat();
		float tolZ=IEditZ->GetFloat();

		float distrib=IEditDistrib->GetFloat();

		float per=0.0f;

		TriObject* triObj=NULL;
		bool*      dupVert=NULL;
		Object*    obj=NULL;

		if (nodeGeom)
		{
			obj=(Object*)nodeGeom->EvalWorldState(time).obj;			

			// Everything must convert to Tri objects (but, just for sure)
			if (!obj->CanConvertToType(triObjectClassID))	
				return;

			// Connect the trigger objects to the closest matching
			// verticies of the selected object
			triObj=(TriObject*)obj->ConvertToType(time,triObjectClassID);

			if (!triObj)
				return;

			Mesh& mesh=triObj->GetMesh();

			// Objects must contain verticies
			if (mesh.numVerts==0)
				return;

			// TODO: Switch to bit arrays.  This is VERY temporary
			dupVert=new bool[mesh.numVerts];

			for(int i=0;i<mesh.numVerts;i++)
				dupVert[i]=false;
		}

		// Interpolate along the spline and add triggers as we go
		while(per<=1.0f)
		{
			Point3  pos;
			pos=shapeObj->InterpCurve3D(time,0,per,PARAM_NORMALIZED);
			Matrix3 tm=mat;
			Point3 trans;

			// Determine the actual interpolated node position on the spline
			trans=pos*nodeSpline->GetObjectTM(time);
			
			if (!nodeGeom)
			{
				// Connect the trigger objects along the spline
				tm.SetTrans(trans);

				// Rotate the new trigger to be inline with the selected viewport
				Matrix3 aTM, coordSysTM;
				ViewExp *ve=ip->GetActiveViewport();
				ve->GetAffineTM(aTM);
				coordSysTM=Inverse(aTM);
				coordSysTM.NoTrans();
				tm=coordSysTM;
				tm.SetTrans(trans);

				INode* node=AddTrigger(tm);

				// Copy offset transform
				node->SetObjOffsetPos(nodeSpline->GetObjOffsetPos());
				node->SetObjOffsetRot(nodeSpline->GetObjOffsetRot());
				node->SetObjOffsetScale(nodeSpline->GetObjOffsetScale());
			}
			else
			{
				Mesh& mesh=triObj->GetMesh();

				// Step through the mesh looking for a vertex that's closest to the spline pt
				Matrix3 objTM=nodeGeom->GetObjectTM(time);

				Point3 shortVert=mesh.verts[0]*objTM;
				float  shortDist=Dist2(trans,shortVert);
				int    shortIndex=0;

				for(int i=0;i<mesh.numVerts;i++)
				{
					Point3 fVert=mesh.verts[i]*objTM;
					float  fDist=Dist2(trans,fVert);

					if (fDist<shortDist)
					{
						// Ignore this vert if it falls outside of our tolerance ranges
						if (IsDlgButtonChecked(hwndLayout,IDC_LIMITMATCH)==BST_CHECKED)
						{
							if ( (fVert.x>trans.x-tolX && fVert.x<trans.x+tolX) &&
								 (fVert.y>trans.y-tolY && fVert.y<trans.y+tolY) &&
								 (fVert.z>trans.z-tolZ && fVert.z<trans.z+tolZ) )
							{
								shortVert=fVert;
								shortDist=fDist;
								shortIndex=i;		
							}
						}
						else
						{
							shortVert=fVert;
							shortDist=fDist;
							shortIndex=i;
						}
					}
				}

				// Create trigger if not duplicate
				if (!dupVert[shortIndex])
				{
					// We now have the vertex that's the shortest distance from the spline pt
					// Add a trigger here
					tm.SetTrans(shortVert);

					INode* node=AddTrigger(tm);
					dupVert[shortIndex]=true;

					// Copy offset transform
					node->SetObjOffsetPos(nodeSpline->GetObjOffsetPos());
					node->SetObjOffsetRot(nodeSpline->GetObjOffsetRot());
					node->SetObjOffsetScale(nodeSpline->GetObjOffsetScale());
				}
			}

			if (per==1.0f)
				break;

			per+=distrib;

			if (per>1.0f)
				per=1.0f;
		}

		// Destroy tri object (if necessary)
		if (triObj!=obj)
			triObj->DeleteThis();

		// Delete duplicate vert tracking array
		if (nodeGeom)
			delete [] dupVert;
	}

	if (IsDlgButtonChecked(hwndLayout,IDC_LINKNODES)==BST_CHECKED)
		LinkTriggers();

	ip->ForceCompleteRedraw();
}

INode* TriggerLayout::AddTrigger(Matrix3 tm)
{
	if (!ip)
		return NULL;

	// Build a new trigger object
	Trigger* triggerObj = (Trigger*)ip->CreateInstance(GEOMOBJECT_CLASS_ID,vTRIGGER_CLASS_ID);

	if (!triggerObj)
		return NULL;

	INode*   newNode   	= ip->CreateObjectNode(triggerObj);	

	if (!newNode)
		return NULL;

	// Assign the extended properties (name, color)
	ReferenceTarget* scene=ip->GetScenePointer();

	AppDataChunk* appdata;
	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_EXTTRIGGERSETTINGS_ID);

	CStr name;

	if (appdata)
	{
		ExtTriggerData* pExtData=(ExtTriggerData*)appdata->data;
		
		// Set name
		if (strlen(pExtData->name)==0)
			strcpy(pExtData->name,"TRG_");

		name=pExtData->name;

		newNode->SetWireColor(pExtData->color);
	}

	ip->MakeNameUnique(name);
	newNode->SetName(name);
	newNode->SetNodeTM(0,tm);

	// Assign the default property chunk to this trigger if so selected
	if (IsDlgButtonChecked(hwndLayout,IDC_ASSIGNPROPS)==BST_CHECKED)
	{
		// Grab the trigger default assign property buffer from the scene
		appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_TRIGGERPROP_ID);

		if (appdata)
			newNode->SetUserPropBuffer((char*)appdata->data);
	}

	trigList.Append(1,&newNode);

	return newNode;
}

void TriggerLayout::LinkTriggers()
{
	// Go through and select all the triggers and then link them
	// using the link manager interface

	int count=trigList.Count();

	ip->ClearNodeSelection();

	if (IsDlgButtonChecked(hwndLayout,IDC_REVERSE)==BST_CHECKED)
	{
		for(int i=count-1;i>=0;i--)
			ip->SelectNode(trigList[i],0);
	}
	else
	{
		for(int i=0;i<count;i++)
			ip->SelectNode(trigList[i],0);
	}
	
	ILinkMan* linkman=GetLinkMan();
	linkman->ChainLink(false);
}

void TriggerLayout::ClearTriggers()
{
	int count=trigList.Count();

	for(int i=count-1;i>=0;i--)			// Delete in reverse order for faster update (less dependancies)
		ip->DeleteNode(trigList[i]);

	trigList.ZeroCount();
	ip->ForceCompleteRedraw();
}
