/*
	FindNode.cpp
	FindAnode (tm) Dialog

	Adam Lippmann
	12-21-01
*/

#include "FindNode.h"
#include "Resource.h"
#include "ParseFuncs.h"
#include "QueryBuilder.h"

FindNodeDlg::FindNodeDlg(Interface* ip,HINSTANCE hInstance,HWND hwndParent) :
	MSDlgWindow(hInstance,MAKEINTRESOURCE(IDD_FINDNODE),hwndParent)
{
	this->ip=ip;
	numFoundNodes=0;
	
	ICustEdit* IEdit=GetICustEdit(GetDlgItem(hwnd,IDC_WILDCARD));
	IEdit->WantReturn(TRUE);
	ReleaseICustEdit(IEdit);

	qbDlg = NULL;
}

FindNodeDlg::~FindNodeDlg()
{
	if (qbDlg)
		delete qbDlg;
}

BOOL FindNodeDlg::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	ICustEdit* IEdit;

	switch(msg)
	{
	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			EnableAccelerators();
		else
			DisableAccelerators();
		
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_QOK:
			DoSearch();
			return TRUE;

		case IDC_QCANCEL:
			Hide();
			return TRUE;

		case IDC_ADDQUERY:
			AddQuery();
			return TRUE;

		case IDC_REMOVEQUERY:
			RemoveQuery();
			return TRUE;

		case IDC_VISSEARCH:
			OpenVisSearch();
			return TRUE;
		}

		switch(HIWORD(wParam))
		{
		case EN_CHANGE:
			IEdit=GetICustEdit(GetDlgItem(hwnd,IDC_WILDCARD));

			if (IEdit->GotReturn())
			{
				AddQuery();
				return TRUE;
			}

			ReleaseICustEdit(IEdit);
		}

		return FALSE;

	case WM_CLOSE:
		Hide();
		return TRUE;
	}

	return FALSE;
}

void FindNodeDlg::AddQuery()
{
	char buf[256];

	ICustEdit* IEdit=GetICustEdit(GetDlgItem(hwnd,IDC_WILDCARD));

	IEdit->GetText(buf,256);

	// Ensure there are no '/'s being added
	int buflen=strlen(buf);

	for(int i=0;i<buflen;i++)
		if (buf[i]=='/')
		{
			MessageBox(hwnd,"/'s are an invalid character.","Invalid Character",MB_ICONSTOP|MB_OK);
			return;
		}

	SendDlgItemMessage(hwnd,IDC_QUERYLIST,LB_ADDSTRING,0,(LPARAM)buf);

	IEdit->SetText("");

	ReleaseICustEdit(IEdit);
}

void FindNodeDlg::RemoveQuery()
{
	int index=SendDlgItemMessage(hwnd,IDC_QUERYLIST,LB_GETCURSEL,0,0);

	if (index!=LB_ERR)
	{
		SendDlgItemMessage(hwnd,IDC_QUERYLIST,LB_DELETESTRING,(WPARAM)index,0);
	}
}

void FindNodeDlg::DoSearch()
{
	char str[128];

	// Ensure that a query has been added before starting search
	int count=SendDlgItemMessage(hwnd,IDC_QUERYLIST,LB_GETCOUNT,0,0);
	
	if (count==0)
	{
		MessageBox(hwnd,"No queries have been added to the list.","FindANode",MB_ICONWARNING|MB_OK);
		return;
	}

	numFoundNodes=0;

	ip->ClearNodeSelection();

	INode* root    =ip->GetRootNode();
	int numChildren=root->NumberOfChildren();

	for(int i=0;i<numChildren;i++)
	{
		INode* node=root->GetChildNode(i);
		CheckNode(node);
	}

	ip->ForceCompleteRedraw();

	sprintf(str,"Found %i nodes.",numFoundNodes);
	MessageBox(hwnd,str,"FindANode(tm)",MB_ICONINFORMATION|MB_OK);
}

void FindNodeDlg::CheckNode(INode* node)
{
	int numChildren=node->NumberOfChildren();

	for(int i=0;i<numChildren;i++)
	{
		INode* child=node->GetChildNode(i);
		CheckNode(child);
	}

	char QueryBuf[256];
	CStr propBuffer;
	CStr strQuery;
	CStr strLine;
	int  pos;
	int  propBufSize;
	BOOL bAbort;

	int ListCount=SendDlgItemMessage(hwnd,IDC_QUERYLIST,LB_GETCOUNT,0,0);

	node->GetUserPropBuffer(propBuffer);
	propBufSize=propBuffer.Length();

	// See if this node meets the criteria
	// It must match on at least one line for EVERY Query in the list
	for(int j=0;j<ListCount;j++)
	{
		SendDlgItemMessage(hwnd,IDC_QUERYLIST,LB_GETTEXT,(WPARAM)j,(LPARAM)QueryBuf);
		strQuery=QueryBuf;

		// Parse out each line of the property buffer
		// Any line can match, but at least one MUST match
		pos=0;
		strLine=GetRemainLineExact(propBuffer,&pos);
		bAbort=TRUE;

		ConvertSlashes(strLine);

		while(pos<propBufSize)
		{
			if (MatchPattern(strLine,strQuery))
			{
				bAbort=FALSE;
				break;
			}

			strLine=GetRemainLineExact(propBuffer,&pos);
			ConvertSlashes(strLine);
		}

		if (bAbort)
			return;
	}

	// At this point the Node's properties have matched all the query patterns
	// on at least one line in their property buffer

	// Select the node
	ip->SelectNode(node,0);
	numFoundNodes++;
}

void FindNodeDlg::ConvertSlashes(CStr& str)
{
	// For whatever reason, the MAX MatchPattern will not work correctly
	// if the comparison line contains /'s.   We will convert the /'s to
	// _'s so everything works.

	int len=str.Length();

	for(int i=0;i<len;i++)
		if (str[i]=='/')
			str[i]='_';
}

void FindNodeDlg::OpenVisSearch()
{
	qbDlg = new QueryBuilderDlg(hInstance,hwnd,ip);
	qbDlg->RegOKCB(VisOKCB,this);
	qbDlg->Show();
}

void FindNodeDlg::VisOKCB(QueryBuilderDlg* qb,void* pdata)
{
	// The Visual Query Builder calls this function when OK is pressed
	FindNodeDlg* pthis = (FindNodeDlg*)pdata;


}
