/*
	FlagReassigner.cpp
	A utility plugin to store and reassign face channel data
*/

#include "FlagReassigner.h"
#include "NExt.h"
#include "Resource.h"
#include "Appdata.h"

static FlagReassignerClassDesc theFlagReassignerDesc;
ClassDesc2* GetFlagReassignerDesc() { return &theFlagReassignerDesc; }

FlagReassigner* gpthis;				// only 1 instance of a tool may be instantiated at once so this is okay

FlagReassigner::FlagReassigner()
{
	mtlIDs   = NULL;
	numFaces = NULL;
	numSel   = 0;
}

FlagReassigner::~FlagReassigner()
{
	ClearData();
}

void FlagReassigner::ClearData()
{
	if (mtlIDs)
	{
		delete [] mtlIDs;
		mtlIDs = NULL;
	}
}

BOOL FlagReassigner::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	switch(msg)
	{
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_STORE:
			//EnableWindow(GetDlgItem(hwnd,IDC_APPLY),TRUE);
			//EnableWindow(GetDlgItem(hwnd,IDC_STORE),FALSE);
			gpthis->StoreFlagData();
			return TRUE;

		case IDC_APPLY:
			//EnableWindow(GetDlgItem(hwnd,IDC_STORE),TRUE);
			//EnableWindow(GetDlgItem(hwnd,IDC_APPLY),FALSE);
			gpthis->ApplyFlagData();
			return TRUE;
		}
	}

	return FALSE;
}

void FlagReassigner::BeginEditParams(Interface* ip, IUtil* iu)
{
	this->ip=ip;
	this->iu=iu;

	gpthis = this;
	hwndRollup = ip->AddRollupPage(hInstance,MAKEINTRESOURCE(IDD_FLAGREASSIGNER),DlgProc,"Face Flag Reassigner");
}

void FlagReassigner::EndEditParams(Interface* ip, IUtil* iu)
{
	ip->DeleteRollupPage(hwndRollup);
}

void FlagReassigner::SelectionSetChanged(Interface* ip, IUtil* iu)
{

}

void FlagReassigner::StoreFlagData()
{
	OutputDebugString("Pre -------------------------------------------------\n");
	DebugFaces();

	numSel = ip->GetSelNodeCount();

	if (numSel==0)
	{
		MessageBox(ip->GetMAXHWnd(),"You must have an object selected to use this tool.","No objects selected",MB_ICONWARNING|MB_OK);
		return;
	}

	if (numSel>1)
	{
		MessageBox(ip->GetMAXHWnd(),"This tool may only be used with one object at a time.","Multiple objects selected",MB_ICONWARNING|MB_OK);
		return;
	}

	// Allocate data
	ClearData();

	for(int i=0;i<numSel;i++)
	{
		INode* node = ip->GetSelNode(i);
		Object* obj = node->EvalWorldState(0).obj;

		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh& mesh = triObj->GetMesh();

			// Now scan through all the faces of the object and pick out the flag data
			numFaces = mesh.numFaces;

			// Acquire access to face flags data
			IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));

			if (!pFDMgr)
			{
				char strErr[256];
				sprintf(strErr,"The object '%s' is not compatible with face flag data.",(char*)node->GetName());
				MessageBox(ip->GetMAXHWnd(),strErr,"Incompatible object in selection",MB_ICONWARNING|MB_OK);
				return;
			}

			FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>(pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
			CASFaceFlagsData* CASfdc = dynamic_cast<CASFaceFlagsData*>(pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));

			// Allocate memory for mtlIDs
			mtlIDs = new MtlID[numFaces];
			
			if (fdc)
				fdata.FacesCreated(0,numFaces);
			
			if (CASfdc)
				fdataCAS.FacesCreated(0,numFaces);

			int mtlindex  = 0;

			for(int face=0;face<numFaces;face++)
			{
				if (fdc)
				{
					FlagType flagtype;

					fdc->GetValue( face, flagtype );
					fdata.SetValue( face, flagtype );
				}

				if (CASfdc)
				{
					CASFlagType casflagtype;

					CASfdc->GetValue( face, casflagtype );
					fdataCAS.SetValue( face, casflagtype );
				}

				// Store mtlIDs
				mtlIDs[face] = mesh.getFaceMtlIndex(face);

				// Assign the material to a sequential value
				mesh.setFaceMtlIndex(face,++mtlindex);
			}

			if (triObj != obj)
				triObj->DeleteThis();
		}

		SaveFlagData(node);
	}

	OutputDebugString("Post -------------------------------------------------\n");
	DebugFaces();

	ip->ForceCompleteRedraw();
}

void FlagReassigner::ApplyFlagData()
{
	OutputDebugString("Pre -------------------------------------------------\n");
	DebugFaces();

	int numSel = ip->GetSelNodeCount();

	if (numSel==0)
	{
		MessageBox(ip->GetMAXHWnd(),"You must have an object selected to use this tool.","No objects selected",MB_ICONWARNING|MB_OK);
		return;
	}

	if (numSel>1)
	{
		MessageBox(ip->GetMAXHWnd(),"This tool may only be used with one object at a time.","Multiple objects selected",MB_ICONWARNING|MB_OK);
		return;
	}

	for(int i=0;i<numSel;i++)
	{
		INode* node = ip->GetSelNode(i);
		Object* obj = node->EvalWorldState(0).obj;

		LoadFlagData(node);

		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);

			// Now scan through all the faces of the object and attach the flag data
			Mesh& mesh = triObj->GetMesh();
			numFaces = mesh.numFaces;

			// Acquire access to face flags data
			IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));

			if (!pFDMgr)
			{
				char strErr[256];
				sprintf(strErr,"The object '%s' is not compatible with face flag data.",(char*)node->GetName());
				MessageBox(ip->GetMAXHWnd(),strErr,"Incompatible object in selection",MB_ICONWARNING|MB_OK);
				return;
			}

			FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>(pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
			CASFaceFlagsData* CASfdc = dynamic_cast<CASFaceFlagsData*>(pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));

			if (fdc == NULL)
			{
				// The Face data channel doesn't exist, add it
				fdc = new FaceFlagsData();
				pFDMgr->AddFaceDataChan( fdc );
				fdc->FacesCreated( 0, numFaces );
			}

			if (CASfdc == NULL)
			{
				// The CAS Face data channel doesn't exist, add it
				CASfdc = new CASFaceFlagsData();
				pFDMgr->AddFaceDataChan( fdc );
				CASfdc->FacesCreated( 0, numFaces );
			}

			int mtlindex  = 0;

			for(int face=0;face<numFaces;face++)
			{
				// We can tell the appropriate flags that should be applied
				// by the material ID - 1 since we reassigned them sequentially before

				MtlID matID = mesh.getFaceMtlIndex(face);
				int   faceID = matID - 1;

				FlagType flagtype;

				fdata.GetValue( faceID, flagtype );
				fdc->SetValue( face, flagtype );

				CASFlagType casflagtype;

				fdataCAS.GetValue( faceID, casflagtype );
				CASfdc->SetValue( face, casflagtype );

			// Assign the face material back to its original value
				mesh.setFaceMtlIndex(face,mtlIDs[faceID]);
			}

			if (triObj != obj)
				triObj->DeleteThis();			
		}

		DeleteFlagData(node);
	}

	OutputDebugString("Post -------------------------------------------------\n");
	DebugFaces();

	ip->ForceCompleteRedraw();
}

void FlagReassigner::LoadFlagData(INode* node)
{
	AppDataChunk* appdata;
	appdata = node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_FLAG_ASSIGNMENT_ID);

	ClearData();

	if (appdata && appdata->data)
	{
		char* pos          = (char*)appdata->data;
		int   numFlags;
		int   numCASFlags;

		numFaces = *((int*)pos);
		pos += sizeof(int);
		numFlags = *((int*)pos);
		pos += sizeof(int);
		numCASFlags = *((int*)pos);
		pos += sizeof(int);

		fdata.AllFacesDeleted();
		fdata.FacesCreated(0,numFlags);
		
		int i;

		for(i=0;i<numFlags;i++)
		{
			FlagType ftype = *((FlagType*)pos);
			pos += sizeof(FlagType);

			fdata.SetValue(i,ftype);
		}

		fdataCAS.AllFacesDeleted();
		fdataCAS.FacesCreated(0,numCASFlags);

		for(i=0;i<numCASFlags;i++)
		{
			CASFlagType ftype = *((CASFlagType*)pos);
			pos += sizeof(CASFlagType);

			fdataCAS.SetValue(i,ftype);
		}

		mtlIDs = new MtlID[numFaces];

		for(i=0;i<numFaces;i++)
		{
			mtlIDs[i] = *((MtlID*)pos);
			pos += sizeof(MtlID);
		}
	}
}

void FlagReassigner::SaveFlagData(INode* node)
{
	node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_FLAG_ASSIGNMENT_ID);

	int size = sizeof(int) +  // numFaces
			   sizeof(int) +  // numFlags
			   sizeof(int) +  // numCASFlags
		       sizeof(FlagType) * fdata.Count() +
			   sizeof(CASFlagType) * fdataCAS.Count() +
			   sizeof(MtlID) * numFaces;

	int i;
	int numFlags    = fdata.Count();
	int numCASFlags = fdataCAS.Count();

	char* data = (char*)malloc(size);
	char* pos  = data;
	memcpy(pos,&numFaces,sizeof(int));
	pos += sizeof(int);
	memcpy(pos,&numFlags,sizeof(int));
	pos += sizeof(int);
	memcpy(pos,&numCASFlags,sizeof(int));
	pos += sizeof(int);

	for(i=0;i<fdata.Count();i++)
	{
		memcpy(pos,&fdata.m_data[i],sizeof(FlagType));
		pos += sizeof(FlagType);
	}

	for(i=0;i<fdataCAS.Count();i++)
	{
		memcpy(pos,&fdataCAS.m_data[i],sizeof(CASFlagType));
		pos += sizeof(CASFlagType);
	}

	for(i=0;i<numFaces;i++)
	{
		memcpy(pos,&mtlIDs[i],sizeof(MtlID));
		pos += sizeof(MtlID);
	}

	node->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_FLAG_ASSIGNMENT_ID,size,data);
}

void FlagReassigner::DeleteFlagData(INode* node)
{
	node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_FLAG_ASSIGNMENT_ID);
}

void FlagReassigner::DebugFaces()
{
	numSel = ip->GetSelNodeCount();

	// Allocate data
	ClearData();

	for(int i=0;i<numSel;i++)
	{
		INode* node = ip->GetSelNode(i);
		Object* obj = node->EvalWorldState(0).obj;

		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh& mesh = triObj->GetMesh();

			for(int j=0;j<mesh.numFaces;j++)
			{
				char buf[256];
				sprintf(buf,"Face %i:  Material: %i\n",j,mesh.getFaceMtlIndex(j));
				OutputDebugString(buf);
			}
		}
	}
}
