#include <Engine/Engine.h>
#include <Sk/Gamenet/ExportMsg.h>
#include <Export/Export.h>
#include <Export/ModelExport.h>
#include <Export/SkinExport.h>
#include <Export/TextureExporter.h>
#include <Export/Strip/Strip.h>
#include <Misc/GenCrc.h>
#include <Misc/HelperFuncs.h>
#include <Export/Skeleton.h>
#include "phyexp.h"
#include <process.h>
#include "path.h"
#include "../UI/OKtoAll.h"

extern bool gbAbort;	// Flag to force entire export to abort
extern CStr GetModelName();

class SkinExporter : public ISkinExporter
{
public:
	bool	SpawnPostExportProcessor( void );
	void	EnumerateExportableNodes( void );
	bool	SaveTextures( void );
	bool	SaveSkin( void );
	bool	SaveGeometry( fstream& file );
	bool	GetSkinData( NxObject* pObject, INode* pNode, unsigned long& checksum );

	// From Exporter
	bool	DoExport( ExportType type );
	bool	Save( void );

	void		enumerate_selected_nodes( void );
	void		enumerate_visible_nodes( INode* root );
	void		enumerate_selection_set_nodes( void );

private:
	Tab <INode *>		m_exportable_nodes;	
	SkinExportOptions	m_export_options;
	NxModel*			m_model;
	bool                m_LitVertWarned;		// Set true if user has been warned of lit verticies
};

static	SkinExporter	s_skin_exporter;

// Utility functions for skin export...
static void ShowSkinExportErrorMessage(char* pErrorMessage)
{
    MessageBoxAll( gInterface->GetMAXHWnd(),
                   pErrorMessage,
                   "Skin Export",
                   MB_OK|MB_ICONERROR);
}

void	SkinExporter::enumerate_selected_nodes( void )
{
	int i;
	INode* sel_node;
	
	for( i = 0; i < gInterface->GetSelNodeCount(); i++ )
	{
		sel_node = gInterface->GetSelNode( i );
		if( IsExportable( sel_node ))
		{
			m_exportable_nodes.Append( 1, &sel_node );
		}
	}
}

void	SkinExporter::enumerate_visible_nodes( INode* root )
{
	INode* child;
	int i;
	
	for( i = 0; i < root->NumberOfChildren(); i++ )
	{
		child = root->GetChildNode( i );
		// Add only visible, non-frozen nodes
		if(	( child->IsHidden() == false ) &&
			( child->IsFrozen() == false ))
		{
			if( IsExportable( child ))
			{
				m_exportable_nodes.Append( 1, &child );			
			}
		}

		// Recursively add all visible children
		if( child->NumberOfChildren() > 0 )
		{
			enumerate_visible_nodes( child );
		}
	}
}

void	SkinExporter::enumerate_selection_set_nodes( void )
{
	int i, num_sel_sets;

	// If they haven't entered a set, just return
	if( m_export_options.m_ExportSet == TSTR( "" ))
	{
		return;
	}

	// Find the set that matches the name of the proposed export set
	num_sel_sets = gInterface->GetNumNamedSelSets();
	for( i = 0; i < num_sel_sets; i++ )
	{
		if( m_export_options.m_ExportSet == TSTR( gInterface->GetNamedSelSetName( i )))
		{
			int j, num_nodes;

			num_nodes = gInterface->GetNamedSelSetItemCount( i );
			for( j = 0; j < num_nodes; j++ )
			{
				INode* sel_node;

				sel_node = gInterface->GetNamedSelSetItem( i, j );

				m_exportable_nodes.Append( 1, &sel_node );
			}

			break;
		}
	}
}

void	SkinExporter::EnumerateExportableNodes( void )
{
	m_exportable_nodes.ZeroCount();
	
	if( m_export_options.m_ExportSelected )
	{
		enumerate_selected_nodes();
	}
	else if( m_export_options.m_ExportVisibleOnly )
	{
		enumerate_visible_nodes( gInterface->GetRootNode());
	}
	else if( m_export_options.m_ExportSelectionSet )
	{
		enumerate_selection_set_nodes();
	}	
}

ISkinExporter*	GetSkinExporter( void )
{
	return &s_skin_exporter;
}

DWORD WINAPI SkinExportProgressFunc(LPVOID arg) 
{
    return(0);
}

bool	SkinExporter::SaveGeometry( fstream& file )
{
	int i, j, k, num_obj, num_faces, num_verts;

	num_obj = m_model->m_Objects.Count();
	file.write((const char*) &num_obj, sizeof( int ));
	gInterface->ProgressStart(_T("Exporting Geometry"), TRUE, 
							SkinExportProgressFunc, NULL );
	for( i = 0; i < num_obj; i++ )
	{
		NxVertex* vert_list;
		NxFace* face_list;
		NxObject* object;
		Point3 center;
		float radius;
		unsigned long checksum;

		object = m_model->m_Objects[i];
		vert_list = object->m_Verts;
		face_list = object->m_Faces;
		num_faces = object->m_NumFaces;
		num_verts = object->m_NumVerts;

		
		// Write out the object's checksum		
		checksum = GenerateCRC( object->m_Name );
		file.write((const char*) &checksum, sizeof( unsigned long ));		

		// Write out the flags which describe the object's properties
		object->m_Flags |= NxObject::mHASVERSIONINFO;
		file.write((const char*) &object->m_Flags, sizeof( int ));

		// Write out the object's skeleton checksum
		file.write((const char*) &object->m_SkeletonChecksum, sizeof( unsigned long ));		

		// Write out the number of sets of UVs
		file.write((const char*) &object->m_NumUVSets, sizeof( int ));

		// Write out bounding box/sphere data
		file.write((const char*) &object->m_BoundingBox.pmin, sizeof( Point3 ));
		file.write((const char*) &object->m_BoundingBox.pmax, sizeof( Point3 ));

		center = object->m_BoundingBox.Center();
		radius = (( object->m_BoundingBox.pmax - object->m_BoundingBox.pmin ).Length()) / 2.0f;
		file.write((const char*) &center, sizeof( Point3 ));
		file.write((const char*) &radius, sizeof( float ));

		file.write((const char*) &num_verts, sizeof( int ));
		for( j = 0; j < num_verts; j++ )
		{
			NxVertex* vert;			
			
			vert = &vert_list[j];

			// Write out first texture coordinate set
			if( object->m_Flags & NxObject::mTEXTURED )
			{
				for( k = 0; k < object->m_NumUVSets; k++ )
				{
					file.write((const char*) &vert->m_TexCoord[k].x, sizeof( float ));
					file.write((const char*) &vert->m_TexCoord[k].y, sizeof( float ));
				}
			}

			// Write out colors
			if( object->m_Flags & NxObject::mCOLORED )
			{
				if (!m_LitVertWarned &&
					 m_export_options.m_WarnOnLitVerts &&
					 !(vert->m_Color.r == 1.0f &&
					   vert->m_Color.g == 1.0f &&
					   vert->m_Color.b == 1.0f))
				{
					MessageBoxAll(gInterface->GetMAXHWnd(),"WARNING!  You are exporting a skin with lit verticies.","Skin exported with lit verticies",MB_ICONWARNING|MB_OK);
					m_LitVertWarned = true;
				}

				file.write((const char*) &vert->m_Color.r, sizeof( float ));
				file.write((const char*) &vert->m_Color.g, sizeof( float ));
				file.write((const char*) &vert->m_Color.b, sizeof( float ));
				file.write((const char*) &vert->m_Color.a, sizeof( float ));
			}				

			// Write out normals
			if( object->m_Flags & NxObject::mNORMALS )
			{					
				file.write((const char*) &vert->m_Normal.x, sizeof( float ));
				file.write((const char*) &vert->m_Normal.y, sizeof( float ));
				file.write((const char*) &vert->m_Normal.z, sizeof( float ));
			}

			// Write out weighting info
			if( object->m_Flags & NxObject::mSKINNED )
			{
				if( vert->m_NumWeights == 0 )
				{
					int l;

					l = 1;
				}
				file.write((const char*) &vert->m_Weight[0], sizeof( float ));
				file.write((const char*) &vert->m_Weight[1], sizeof( float ));
				file.write((const char*) &vert->m_Weight[2], sizeof( float ));

				file.write((const char*) &vert->m_WeightedIndex[0], sizeof( int ));
				file.write((const char*) &vert->m_WeightedIndex[1], sizeof( int ));
				file.write((const char*) &vert->m_WeightedIndex[2], sizeof( int ));			
			}

			// Write out vc wibble data
			if( object->m_Flags & NxObject::mVCWIBBLE )
			{
				file.write((const char*) &vert->m_WibbleIndex, sizeof( char ));
				file.write((const char*) &vert->m_WibbleOffset, sizeof( char ));	
			}

			// Write out positions				
			file.write((const char*) &vert->m_Pos.x, sizeof( float ));
			file.write((const char*) &vert->m_Pos.y, sizeof( float ));
			file.write((const char*) &vert->m_Pos.z, sizeof( float ));		
		}

		file.write((const char*) &num_faces, sizeof( int ));
		for( j = 0; j < num_faces; j++ )
		{
			// Write out the material checksum for this face
			file.write((const char*) &face_list[j].m_MatChecksum, sizeof( unsigned long ));

			// Write out the face normal
			file.write((const char*) &face_list[j].m_Normal, sizeof( Point3 ));

			// Versioning data isn't saved with NxObject.  So we are using this flag
			// to indicate the addition of CAS Face data
			face_list[j].m_FaceFlags |= mFD_CASFACEFLAGSEXIST;

			// Write out the face flags
			file.write((const char*) &face_list[j].m_FaceFlags, sizeof( FlagType ));

			// Write out the CAS face flags
			file.write((const char*) &face_list[j].m_CASFaceFlags, sizeof( CASFlagType ));

			// Write out the vertex indices
			file.write((const char*) face_list[j].m_Vertex, sizeof( int ) * 3 );
		}

		if (object->m_Flags & NxObject::mHASCASREMOVEFLAGS)
		{
			// Write out CAS face removal flags
			file.write((const char*)&object->m_CASRemoveFlags, sizeof( int ));
		}

		if (object->m_Flags & NxObject::mHASKBIAS)
		{
			// Write out the KBias
			file.write((const char*)&object->m_KBias,sizeof( float ));
		}

		// Write out LOD data if appropriate
		if (object->m_Flags & NxObject::mHASLODINFO)
		{
			// Write out the LOD info
			file.write((const char*)&object->m_LODVersion,sizeof(unsigned int));
			file.write((const char*)&object->m_LODFlags,sizeof(int));

			if (object->m_LODFlags & NxObject::mMASTER)
			{
				file.write((const char*)&object->m_LODMaster.m_NumLODLevels,sizeof(int));
				file.write((const char*)object->m_LODMaster.m_LODLevels,sizeof(NxLODLevel) * object->m_LODMaster.m_NumLODLevels);
			}

			if (object->m_LODFlags & NxObject::mSLAVE)
			{
				file.write((const char*)&object->m_LODSlave.m_masterCRC,sizeof(int));
			}
		}

		// Write out version info
		file.write((const char*)&object->m_Version,sizeof(unsigned int));

		// Write out relative info v1
		file.write((const char*)&object->m_ParentCRC,sizeof(unsigned long));
		file.write((const char*)&object->m_NumChildren,sizeof(int));
		file.write((const char*)object->m_ChildCRCs,sizeof(unsigned long) * object->m_NumChildren);

		if( gInterface->GetCancel())
		{
			gInterface->ProgressEnd();
			file.close();
			return false;
		}
		gInterface->ProgressUpdate( i * 100 / num_obj );
	}

	gInterface->ProgressEnd();
	
	return true;
}

bool	SkinExporter::SpawnPostExportProcessor( void )
{
	int result;
	char *args[4];
	char path[_MAX_PATH], conv_path[_MAX_PATH];	
	TSTR scene_name, scene_dir, base_name;
	char* project_root;
	
	project_root = getenv( vSKATE4_ENVIRON_VAR );
	if( project_root == NULL )
	{
		MessageBox( gInterface->GetMAXHWnd(), "You must set up your SKATE4_PATH environment variable",
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_VIEW ||
		m_export_type == vQUICK_VIEW_NOTEX)
	{
		base_name = "quick";
	}
	else
	{
		base_name = m_export_options.m_SkinName;
	}	

	/*
	scene_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name;	
	scene_name = scene_dir + TSTR( "/" ) + base_name + TSTR( ".skin" );
	*/

	if (m_export_options.m_DirPath.Length()==0)
		scene_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name;
	else
		scene_dir = m_export_options.m_DirPath;

	scene_name = scene_dir + TSTR( "/" ) + base_name + TSTR( ".skin" );

	sprintf( path, "-f%s", scene_name );
	sprintf( conv_path, "%s\\bin\\win32\\sceneconv.exe", project_root );

	args[0] = conv_path;
	args[1] = "-pp";
	args[2] = path;	
	args[3] = NULL; 

	result = ( spawnv( _P_WAIT, args[0], args ) == 0 );					
	return true;
}

bool	SkinExporter::SaveTextures( void )
{
	TSTR texfile_name, usgfile_name, base_name;
	char* project_root;	
	ITextureExporter* tex_exp;
	
	tex_exp = GetTextureExporter();
	project_root = getenv( vSKATE4_ENVIRON_VAR );
	if( project_root == NULL )
	{
		MessageBox( gInterface->GetMAXHWnd(), "You must set up your SKATE4_PATH environment variable",
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_VIEW ||
		m_export_type == vQUICK_VIEW_NOTEX)
	{
		base_name = "quick";
	}
	else
	{
		base_name = m_export_options.m_SkinName;
	}

	if (strlen(m_export_options.m_DirPath)>0)
	{
		texfile_name = TSTR(m_export_options.m_DirPath) + TSTR( "/" ) + base_name + TSTR( ".tex" );
		usgfile_name = TSTR(m_export_options.m_DirPath) + TSTR( "/" )+ base_name + TSTR( ".usg" );

		if (m_export_options.m_DirWarn)
			if (!tex_exp->VerifyTexturePaths( base_name ))
				return false;
	}
	else
	{
		texfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name + TSTR( "/" ) + base_name + TSTR( ".tex" );
		usgfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name + TSTR( "/" ) + base_name + TSTR( ".usg" );

		if (m_export_options.m_DirWarn)
			if (!tex_exp->VerifyTexturePaths( TSTR( "models/" ) + base_name ))
				return false;
	}
	
	return tex_exp->SaveTextureDictionary( texfile_name, usgfile_name );
}

bool	SkinExporter::SaveSkin( void )
{
	fstream file;
	TSTR skn_name, skin_dir, base_name;
	char* project_root;
	int version;
	DWORD attribs;
	bool sky_export;
	
	project_root = getenv( vSKATE4_ENVIRON_VAR );
	if( project_root == NULL )
	{
		MessageBox( gInterface->GetMAXHWnd(), "You must set up your SKATE4_PATH environment variable",
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_VIEW ||
		m_export_type == vQUICK_VIEW_NOTEX)
	{
		base_name = "quick";

		skin_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name;
		CreateDirectory( skin_dir, NULL );	
		CreateDirectory( "c:/temp", NULL );	// for image processing

		skn_name = skin_dir + TSTR( "/" ) + TSTR( base_name ) + TSTR( ".skin" );
	}
	else
	{
		base_name = m_export_options.m_SkinName;
	}

	if (m_export_options.m_DirPath.Length()==0)
	{
		skin_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name;
		CreateDirectory( skin_dir, NULL );	
		CreateDirectory( "c:/temp", NULL );	// for image processing

		skn_name = skin_dir + TSTR( "/" ) + base_name + TSTR( ".skin" );
	}
	else
	{
		skin_dir = m_export_options.m_DirPath;
		CreateDirectory( skin_dir, NULL );	
		CreateDirectory( "c:/temp", NULL );	// for image processing
		
		skn_name = skin_dir + TSTR( "/" ) + base_name + TSTR( ".skin" );
	}

	attribs = GetFileAttributes( skn_name );
	if( attribs != -1 )
	{
		if( attribs & FILE_ATTRIBUTE_READONLY )
		{
			char message[256];
			
			sprintf( message, "You are about to save over a Read-Only file, %s, perhaps you should check-out the file first", skn_name );
			if( MessageBoxAll( gInterface->GetMAXHWnd(), message, "Read-only Warning", MB_OKCANCEL ) == IDCANCEL )
			{
				return false;
			}
		}
	}

	// break the lock, if necessary
	SetFileAttributes( skn_name, FILE_ATTRIBUTE_NORMAL );
	file.open( skn_name, ios::out | ios::binary );

	version = NxMaterial::vVERSION_NUMBER;
	file.write((const char*) &version, sizeof( int ));
	version = NxMesh::vVERSION_NUMBER;
	file.write((const char*) &version, sizeof( int ));
	version = NxVertex::vVERSION_NUMBER;
	file.write((const char*) &version, sizeof( int ));
	sky_export = false;
	file.write((const char*) &sky_export, sizeof( bool ));
	if( SaveMaterials( file, m_model ) == false )
	{
		file.close();
		return false;
	}

	if( SaveGeometry( file ) == false )
	{
		file.close();
		return false;	
	}	

	file.close();
	return true;	
}

bool	SkinExporter::Save( void )
{
	return( SaveSkin() && SaveTextures());		
}

bool	SkinExporter::DoExport( ExportType type )
{
	CStr export_name;
	IModelExporter* model_exp;
	ITextureExporter* tex_exp;

	// Reset all warning mesages
	MessageBoxResetAll();

	model_exp = GetModelExporter();
	tex_exp = GetTextureExporter();
	tex_exp->Reset();	
		 
	GetSkinExportOptions( &m_export_options );
	
	EnumerateExportableNodes();
	m_export_type = type;

	m_LitVertWarned = false;

	if( m_exportable_nodes.Count() == 0 )
	{
		// Nothing to export
		return false;
	}

	if( !m_exportable_nodes[0]->GetUserPropString( "ExportName", export_name ))
	{
		// If the object has no name, don't export it
		ShowSkinExportErrorMessage( "You must assign the ExportName property to this object" );
		return false;
	}

	strcpy( m_export_options.m_SkinName, export_name );

	// At this point, we have gathered a list of nodes which represent the exportable geometry.
	// Now parse these nodes and store relavent data in our own generic database
	m_model = model_exp->ExtractModelData( m_exportable_nodes, true, m_export_options.m_Scale );
	// If something went wrong or it was cancelled, don't continue
	if( m_model == NULL )
	{
		return false;
	}

	// Load the textures needed for the current export set
	tex_exp->LoadTextureData();

	// Here output the model and its materials, either via the network or to disk
	if( Save())
	{		
		if( SpawnPostExportProcessor())
		{
			// Here do quickview stuff
			if( m_export_type == vRESET_AND_RELOAD ||
				m_export_type == vRESET_AND_RELOAD_NOTEX)
			{
				char exec_str[128];
				char *viewer_root;

				if( viewer_root = getenv( vSKATE4_ENVIRON_VAR ))
				{			
					char viewer_app[_MAX_DIR + _MAX_FNAME];

					sprintf( viewer_app, "%s\\build\\NGPSgnu\\%s", viewer_root, vVIEWER_APP_NAME );
					sprintf( exec_str, "ps2run -r %s", viewer_app );
					WinExec( exec_str, SW_HIDE );
				}
			}
			else if( m_export_type == vQUICK_VIEW )
			{
				Net::MsgQuickview msg;

				// The scn, tex and usg files have been exported to temp "quick" files.
				// Tell the engine, via the network, to load these files into the current scene
				sprintf( msg.m_Filename, m_export_options.m_SkinName );
				gClient->EnqueueMessageToServer( Net::vMSG_ID_QUICKVIEW, sizeof( Net::MsgQuickview ),
													&msg );
			}
		}
	}

	return true;
}

int	get_remapped_vertex_index( INode* pNode, int nxVertexIndex )
{
	Object* pObject = pNode->EvalWorldState(0).obj;
	MaxAssert( pObject->CanConvertToType( triObjectClassID ));
	TriObject* pTriObject = (TriObject*) pObject->ConvertToType( 0, triObjectClassID );
	Mesh* pMesh = &pTriObject->mesh;

	// given the nxvertex index, we can get the face number
	int faceNum = nxVertexIndex / 3;
	int vertNum = nxVertexIndex % 3;

	// given the face number, we can get the MAX vertex index
	MaxAssert( faceNum >= 0 && faceNum < pMesh->getNumFaces() );
	Face* pFace = &pMesh->faces[faceNum];
	return pFace->v[vertNum];
}

bool	SkinExporter::GetSkinData( NxObject* pObject, INode* pNode, unsigned long& checksum )
{
    // Validate pointers
    MaxAssert( pObject );
    MaxAssert( pNode );

	Modifier* pPhysiqueModifier = FindPhysiqueModifier( pNode );
    
	checksum = 0;
    if ( !pPhysiqueModifier )
    {
        ShowSkinExportErrorMessage( "Selected object does not contain physique modifier" );
        return false;
    }
    
    // should be able to grab the interface to the exporters safely...
    IPhysiqueExport* pPhysiqueExport = (IPhysiqueExport*)pPhysiqueModifier->GetInterface(I_PHYINTERFACE);    
    MaxAssert( pPhysiqueExport );

    // create a ModContext Export Interface for the specific node of the Physique Modifier
    IPhyContextExport* pModContextExport = (IPhyContextExport*)pPhysiqueExport->GetContextInterface(pNode);
    MaxAssert( pModContextExport );

    // as of CS2, only rigid vertices were supported by Character Studio
    // the new version of CS may have better support, but that's a feature
    // that can be implemented at a later date...
    pModContextExport->ConvertToRigid( true );
    pModContextExport->AllowBlending( true );

    int numVerts = pObject->m_NumVerts;//pModContextExport->GetNumberVertices();    

    // the physique modifier should have reported the
    // same number of verts as the NxObject
//    MaxAssert( pObject->m_NumVerts == numVerts );

    bool badVertexTypeFound = false;
	bool tooManyWeightsFound = false;
	int  errWeights;
	int  i;

	CSkeletonData* pSkeletonData = NULL;

    for ( i = 0; i < numVerts; i++ )
    {
		int remapped_index = get_remapped_vertex_index( pNode, i );

        // GJ:  Theoretically, we should be getting the
        // verts in the same order as grabbing them from the mesh.
        // If not, we'll have to re-map them somehow (perhaps
        // by comparing the actual coordinates).

        NxVertex* pNxVertex = &(pObject->m_Verts[i]);

        IPhyVertexExport* pVertexExport;
        pVertexExport = pModContextExport->GetVertexInterface( remapped_index );

		if (!pVertexExport)
		{
			OutputDebugString("\\Export\\SkinExport.cpp: Physique pVertexExport interface not found!");
			return false;
		}

        MaxAssert( pVertexExport );

        // In THPS3, the exporter had a max limit of 4 weights per vertex.
        // Now, the exporter will not care about the limit;
        // it's up to the platform-specific processor to compress
        // them down to N verts.

        switch ( pVertexExport->GetVertexType() )
        {
            case RIGID_BLENDED_TYPE:
            {
                pNxVertex->m_Weighted = true;
                IPhyBlendedRigidVertex* pBlendedVertex = (IPhyBlendedRigidVertex*)pVertexExport;
                int numWeights = pBlendedVertex->GetNumberNodes();
				if ( numWeights > NxVertex::vMAX_WEIGHTS_PER_VERTEX )
				{
					tooManyWeightsFound = true;
					errWeights=numWeights;
					break;
				}

				pNxVertex->m_NumWeights = numWeights;
				float runningWeightTotal = 0.0f;
				for ( int j = 0; j < numWeights; j++ )
                {
                    INode* pBone = pBlendedVertex->GetNode(j);
					if ( !pSkeletonData )
					{
						pSkeletonData = new CSkeletonData( pBone );
					}
                    pNxVertex->m_WeightedIndex[j] = pSkeletonData->GetBoneIndex( pBone );
                    pNxVertex->m_Weight[j] = pBlendedVertex->GetWeight( j );
					runningWeightTotal += pBlendedVertex->GetWeight( j );
                }
				// the total weight for all the verts should equal 1.0
				MaxAssert( (short)( runningWeightTotal * 4096.0f + 0.5f ) == 4096 );
            }
            break;
            case RIGID_TYPE:
            {
                pNxVertex->m_Weighted = true;
                IPhyRigidVertex* pRigidVertex = (IPhyRigidVertex*)pVertexExport;
                INode* pBone = pRigidVertex->GetNode();
				if ( !pSkeletonData )
				{
					pSkeletonData = new CSkeletonData( pBone );
				}
				pNxVertex->m_NumWeights = 1;
                pNxVertex->m_WeightedIndex[0] = pSkeletonData->GetBoneIndex( pBone );
                pNxVertex->m_Weight[0] = 1.0f;
            }
            break;
            default:  
                // unrecognized type
                badVertexTypeFound = true;
                break;
        }

        pModContextExport->ReleaseVertexInterface( pVertexExport );
        pVertexExport = NULL;

        // NOT SURE IF THE FOLLOWING LINE IS NECESSARY:
        pModContextExport->ReleaseVertexInterface( pVertexExport );
        pVertexExport = NULL;
    }

	checksum = pSkeletonData->GetChecksum();
	delete pSkeletonData;
	
    pPhysiqueExport->ReleaseContextInterface( pModContextExport );            

    pPhysiqueModifier->ReleaseInterface( I_PHYINTERFACE, pPhysiqueExport );                    

    // TODO:
    // NORMALIZE THE WEIGHTS?
    // CHECK FOR > MAX WEIGHTS
    
    if ( badVertexTypeFound )
    {
        ShowSkinExportErrorMessage( "Unrecognized vertex type found (must be rigid or rigid-blended." );
        return false;
    }

	if ( tooManyWeightsFound )
    {
		char strErr[256];
		sprintf(strErr,"Too many weights found on vertex.\nNode Name: %s\nVert Index: %i\nNum Weights: %i (Limit: %i)",
			    (char*)pNode->GetName(),i,errWeights,NxVertex::vMAX_WEIGHTS_PER_VERTEX);

        ShowSkinExportErrorMessage( strErr );

		// Force the entire export to abort
		gbAbort=true;

        return false;
    }

	return true;
}
