/*
	CamExporter.h
	Camera Exporter
*/

#ifndef __CAMEXPORTER__
#define __CAMEXPORTER__

#include "AnimExporter.h"
#undef assert
#include "../PropEdit/RTFEditor.h"
#include "max.h"
#include "notify.h"
#include "../UI/ResizeWindow.h"
#include "../UI/ProgressBar.h"
#include "../misc/llist.h"
#include "skeleton.h"
#include "Export.h"
#include "ExportOptions.h"

class GenExporter;
class ListView;
class RTFEditor;
struct ExportInfo;

struct CamSpecificExportSettings  // vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID
{
	float fRotTolerance;	// Rotation compression tolerance
	float fTranTolerance;	// Translation compression tolerance
	float fFovTolerance;	// Field of view compression tolerance
	int   start;			// Start frame
	int   end;              // End frame
	char  strUserName[256];	// Name of the camera assigned by user (export identifier)
};

class CamAddKeyDlg: public MSDlgWindow
{
	Interface*        ip;					// MAX Interface pointer
	CStr              scriptFile;			// The script filename of the currently selected script

	void (*fpOK)(CamAddKeyDlg*,void*);		// Callback executed when user presses OK
	void *pOKData;							// Callback retained data

	ICustEdit*       IEdit;
	ICustEdit*       IEditFile;
	RTFEditor*       scriptEditor;

	ISpinnerControl* ISpin;
	LinkList<CStr>   scriptFileDB;			// Stores the names of all the script files referenced in the script list

	bool             bAbort;				// True if script search was aborted
	bool             bScriptsLoaded;		// True if recursive search done for scripts

	BOOL DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

	bool MatchContent(FILE* fp,char* wildcard,char* buf,int* line);
	void Find(char* sdir,char* prefix,char* search);
	int  CountFiles(char* sdir,char* prefix);

	void SelChange();

public:
	void  Show();

	void SetOKCB(void(*Func)(CamAddKeyDlg*,void*),void* pData);
	
	void  SetLimits(int min,int max);
	int   GetTime();
	float GetTimeSec();
	void  SetTime(int time);

	CStr  GetScript();
	void  SetScript(CStr strScript);
	void  SpinChange();

	CStr  GetScriptCode();
	void  SetScriptCode(CStr strScriptCode);

	void  Refresh();

	CStr GetScriptFile();
	inline void SetScriptFile(CStr file) { scriptFile=file; }

	CamAddKeyDlg(HINSTANCE hInstance,HWND hwndParent,Interface* ip);
	~CamAddKeyDlg();
};

struct ScriptKey
{
	CStr  name;		// For user-defined scripts name = [User] and file contains the script
	CStr  file;
	int   frame;
};

struct CamInfo
{
	ExportInfo*          expdata;
	LinkList<ScriptKey>  scriptKeys;
};

class CameraExporter: public Exporter, public AnimExporter
{
	GenExporter* genExporter;		// Pointer the generalized keyframe data exporter (used for FOV keys)

	int GetKeyCount(char* buf);

protected:
	GenKey<ScriptKey>* GetScriptKeys(CamInfo* camInfo,int* numKeys);

	GenKey<float>* GetFOVKeys(CamInfo* camInfo,int* numKeys=NULL,
		                      float error=0.0f, bool bOneFrame=true);

	bool ExportNode(FILE* fpExport,FILE* fpDebug,INode* node,int numExtFrames,
					float errorQ,float errorT,
					bool bUseCompression, bool bCompressTime, bool bOneFrame);
	
	bool Export(char* filename,CamInfo* camInfo,int start,int end,
		        float errorQ,float errorT,float errorF,
				bool bCompress,bool bOneFrame,bool bDebug,bool bCompressTime);

	CStr GetSceneName();
	void SendNetMessage(char* camname,int delay);

	void BuildPS2File(char* camname);

public:
	CameraExporter();
	~CameraExporter();

	bool ExportQN(FILE* fp);
	bool DoExport( ExportType type );
	bool Save();

	void DoQBRUpdate(bool bGenerateScripts=true);
};

CameraExporter* GetCamExporter();

class CameraExportDlg: public ResizeDlgWindow, public CameraExporter
{
	Interface*   ip;						// MAX Interface pointer
	ExportOptionsDlg* expopt;				// Stores the export options dialog, so we can force it to close
	GenExporter* genExporter;				// Pointer the generalized keyframe data exporter (used for FOV keys)

	RTFEditor*   scriptEditor;

	LinkList<CStr> scriptDB;				// Script buffers attached to each export node

	CamSpecificExportSettings camsettings_defaults;

	ListView*    lview;						// UI Interfaces
	ListView*    camList;

	ICustEdit* editQErr;
	ICustEdit* editTErr;
	ICustEdit* editFErr;
	ICustEdit* editStart;
	ICustEdit* editEnd;
	ICustEdit* editKeyTime;

	ISpinnerControl* spinQErr;
	ISpinnerControl* spinTErr;
	ISpinnerControl* spinFErr;
	ISpinnerControl* spinStart;
	ISpinnerControl* spinEnd;
	ISpinnerControl* spinKeyTime;

	ProgressBar*     progBar;
	
	CStr             strCamName;			// Name of the current camera
	int              lastItem;				// Item described by strCamName

	CamAddKeyDlg*    pAddKeyDlg;			// Add key dialog

	LinkList<CamInfo> camDB;				// Database of info attached to each camera

	Tab<INode*>      selSet;				// List of nodes to apply changes to

	INode*           lastCamNode;			// The last camera node selected
	bool             bLockCamList;			// Locks update calls for the camera list (camList)
	bool             bLockKeyList;			// Locks update calls for the keyframe list (lview)
	bool             bLockNodeSelect;		// Locks update calls from camera list selection
	bool             bLockLoadSave;
	bool             bLockSelChange;		// Lock update on any selection changes
	bool             bAbort;				// True if the export was aborted

	BOOL DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);
	bool DoExport();
	CStr GetSaveFile();

	static void MAXSelChange(void *param,NotifyInfo *info);
	static void AddKeyCB(CamAddKeyDlg* pDlg,void* pData);

	void SelChange(bool bNoApply=false);	// MAX selection change
	void CamSelChange(NMLISTVIEW* pNotify);	// Camera list box selection change

	void AddScript();
	void DeleteScript();
	void ListViewUpdate(NMLISTVIEW* pNotify);

	void SaveToNode(INode* node=NULL);		// Saves the script keyframe info to the current camera node
	void LoadFromNode(INode* node=NULL);	// Loads the script keyframe info from the current camera node

	void SaveSettings();					// Save the general export settings to the scene
	void LoadSettings();					// Loads the general export settings from the scene
	
	void Apply();
	void AcquireSelSet();
	void GetRange();
	void UpdateRange();
	void UpdateLabel(NMLVDISPINFO* pInfo);	// Process changing the name of a field in the listview window (camList)
	void UpdateName();

	void SaveCamSelection();
	void LoadCamSelection();

	void UpdateScript();

	bool ExportSingleCam(int i);

	void SendNetMessage(int i=-1);
	void BuildPS2File(int i=-1);

	static void CancelCB(void* pData);
	static int Sort(DWORD dw1,DWORD dw2,ListView* pData,void* pExtData);

	//void Resize(int width,int height);

	void UpdateKeyTime();

	static void CloseExportDlg(void *param, NotifyInfo *info);

	bool CameraExists(char* name);

public:
	CameraExportDlg(HINSTANCE hInstance,HWND hwndParent,Interface* ip,ExportOptionsDlg* expopt=NULL);
	~CameraExportDlg();

	void GetCamData();
	void ResortList();

	void Cleanup(char* path);
	void Archive(char* path);

	void Show();
	void Hide();
};

#endif
