 /**********************************************************************  
 *<
	FILE: Restore.cpp

	DESCRIPTION: Editable Polygon Mesh Object - RestoreObjects.

	CREATED BY: Steve Anderson

	HISTORY: created April 2000

 *>	Copyright (c) 2000, All Rights Reserved.
 **********************************************************************/

#include "EPoly.h"
#include "PolyEdit.h"

namespace EditPoly
{

//#define EPOLY_RESTORE_DEBUG_PRINT

// --- Restore objects ---------------------------------------------

// Not a real restore object - just used in drags to restore geometric
// and map vertex changes.
TempMoveRestore::TempMoveRestore (EditPolyObject *em) {
	init.SetCount (em->mm.numv);
	active.SetSize (em->mm.numv);
	active.ClearAll ();
	for (int i=0; i<em->mm.numv; i++) {
		if (em->mm.v[i].GetFlag (MN_DEAD)) continue;
		init[i] = em->mm.v[i].p;
	}

	mapInit = new Tab<UVVert>[em->mm.numm + NUM_HIDDENMAPS];
	for (int mp=-NUM_HIDDENMAPS; mp<em->mm.numm; mp++) {
		if (em->mm.M(mp)->GetFlag (MN_DEAD)) continue;
		int nmp = NUM_HIDDENMAPS + mp;
		mapInit[nmp].SetCount (em->mm.M(mp)->numv);
		for (i=0; i<em->mm.M(mp)->numv; i++) {
			mapInit[nmp][i] = em->mm.M(mp)->v[i];
		}
	}
}

TempMoveRestore::~TempMoveRestore () {
	delete [] mapInit;
}

void TempMoveRestore::Restore (EditPolyObject *em) {
	if (!init.Count()) return;
	for (int i=0; i<em->mm.numv; i++) {
		if (em->mm.v[i].GetFlag (MN_DEAD)) continue;
		em->mm.v[i].p = init[i];
	}

	for (int mp=-NUM_HIDDENMAPS; mp<em->mm.numm; mp++) {
		if (em->mm.M(mp)->GetFlag (MN_DEAD)) continue;
		int nmp = NUM_HIDDENMAPS + mp;
		for (i=0; i<em->mm.M(mp)->numv; i++) {
			em->mm.M(mp)->v[i] = mapInit[nmp][i];
		}
	}
}

// -----------------------------------

ComponentFlagRestore::ComponentFlagRestore (EditPolyObject *editObj, int selLevel) {
	ep = editObj;
	sl = selLevel;
	int i;

	switch (sl) {
	case MNM_SL_OBJECT:
		return;

	case MNM_SL_VERTEX:
		undo.SetCount (ep->mm.numv);
		for (i=0; i<undo.Count(); i++) {
			undo[i] = ep->mm.v[i].ExportFlags ();
		}
		break;

	case MNM_SL_EDGE:
		undo.SetCount (ep->mm.nume);
		for (i=0; i<undo.Count(); i++) {
			undo[i] = ep->mm.e[i].ExportFlags ();
		}
		break;

	case MNM_SL_FACE:
		undo.SetCount (ep->mm.numf);
		for (i=0; i<undo.Count(); i++) {
			undo[i] = ep->mm.f[i].ExportFlags ();
		}
		break;
	}
}

void ComponentFlagRestore::Restore (int isUndo) {
	int i, max = undo.Count ();

	switch (sl) {
	case MNM_SL_OBJECT:
		return;

	case MNM_SL_VERTEX:
		redo.SetCount (ep->mm.numv);
		for (i=0; i<redo.Count(); i++) {
			redo[i] = ep->mm.v[i].ExportFlags ();
			if (i<max) ep->mm.v[i].ImportFlags (undo[i]);
		}
		break;

	case MNM_SL_EDGE:
		redo.SetCount (ep->mm.nume);
		for (i=0; i<redo.Count(); i++) {
			redo[i] = ep->mm.e[i].ExportFlags ();
			if (i<max) ep->mm.e[i].ImportFlags (undo[i]);
		}
		break;

	case MNM_SL_FACE:
		redo.SetCount (ep->mm.numf);
		for (i=0; i<redo.Count(); i++) {
			redo[i] = ep->mm.f[i].ExportFlags ();
			if (i<max) ep->mm.f[i].ImportFlags (undo[i]);
		}
		break;
	}

	// We may have changed topology or selection ...
	ep->LocalDataChanged (PART_TOPO|PART_SELECT|PART_GEOM);
	// sca - or geometry - such as bounding boxes.
}

void ComponentFlagRestore::Redo () {
	int i, max = undo.Count ();

	switch (sl) {
	case MNM_SL_OBJECT:
		return;

	case MNM_SL_VERTEX:
		if (max>ep->mm.numv) max = ep->mm.numv;
		for (i=0; i<max; i++) ep->mm.v[i].ImportFlags (redo[i]);
		break;

	case MNM_SL_EDGE:
		if (max>ep->mm.nume) max = ep->mm.nume;
		for (i=0; i<max; i++) ep->mm.e[i].ImportFlags (redo[i]);
		break;

	case MNM_SL_FACE:
		if (max>ep->mm.numf) max = ep->mm.numf;
		for (i=0; i<max; i++) ep->mm.f[i].ImportFlags (redo[i]);
		break;
	}

	// We may have changed topology or selection ...
	ep->LocalDataChanged (PART_TOPO|PART_SELECT|PART_GEOM);
	// sca - or geometry - such as bounding boxes.
}

//-----------------------------------------------------

TopoChangeRestore::TopoChangeRestore (EditPolyObject *e) {
	ep = e;
	vfacBefore = NULL;
	vfacAfter = NULL;
	vedgBefore = NULL;
	vedgAfter = NULL;
	mapVertID = NULL;
	mapFaceID = NULL;
	mapVertsBefore = NULL;
	mapVertsAfter = NULL;
	mapFacesBefore = NULL;
	mapFacesAfter = NULL;
	vertexDataCount = edgeDataCount = 0;
	vertexDataBefore = NULL;
	vertexDataAfter = NULL;
	edgeDataBefore = NULL;
	edgeDataAfter = NULL;
}

TopoChangeRestore::~TopoChangeRestore () {
	if (vfacBefore) delete [] vfacBefore;
	if (vfacAfter) delete [] vfacAfter;
	if (vedgBefore) delete [] vedgBefore;
	if (vedgAfter) delete [] vedgAfter;
	if (mapVertID) delete [] mapVertID;
	if (mapFaceID) delete [] mapFaceID;
	if (mapVertsBefore) delete [] mapVertsBefore;
	if (mapVertsAfter) delete [] mapVertsAfter;
	if (vertexDataBefore) delete [] vertexDataBefore;
	if (vertexDataAfter) delete [] vertexDataAfter;
	if (edgeDataBefore) delete [] edgeDataBefore;
	if (edgeDataAfter) delete [] edgeDataAfter;
	if (mapFacesBefore) {
		for (int mp=0; mp<numMvBefore.Count(); mp++) {
			Tab<MNMapFace> & mf = mapFacesBefore[mp];
			for (int i=0; i<mf.Count(); i++) mf[i].Init ();
		}
		delete [] mapFacesBefore;
	}
	if (mapFacesAfter) {
		for (int mp=0; mp<numMvBefore.Count(); mp++) {
			Tab<MNMapFace> & mf = mapFacesAfter[mp];
			for (int i=0; i<mf.Count(); i++) mf[i].Init ();
		}
		delete [] mapFacesAfter;
	}
}

void TopoChangeRestore::After () {
	numvBefore = start.numv;
	numeBefore = start.nume;
	numfBefore = start.numf;
	numvAfter = ep->mm.numv;
	numeAfter = ep->mm.nume;
	numfAfter = ep->mm.numf;
	int mp, nhm=NUM_HIDDENMAPS;
	int mapCount = nhm + ep->mm.numm;	// Note: assuming that start has <= numm.
	numMvBefore.SetCount (mapCount);
	numMvAfter.SetCount (mapCount);
	mapVertID = new Tab<int>[mapCount];
	mapFaceID = new Tab<int>[mapCount];
	mapVertsBefore = new Tab<UVVert>[mapCount];
	mapVertsAfter = new Tab<UVVert>[mapCount];
	mapFacesBefore = new Tab<MNMapFace>[mapCount];
	mapFacesAfter = new Tab<MNMapFace>[mapCount];
	for (mp=-nhm; mp<ep->mm.numm; mp++) {
		int nmp = nhm + mp;
		if (ep->mm.M(mp)->GetFlag (MN_DEAD)) {
			numMvBefore[nmp] = numMvAfter[nmp] = 0;
		} else {
			numMvBefore[nmp] = start.M(mp)->numv;
			numMvAfter[nmp] = ep->mm.M(mp)->numv;
		}
	}

	// Record vertex changes:
	int i, j, ct, minv = (numvBefore < numvAfter) ? numvBefore : numvAfter;
	for (i=0; i<minv; i++) {
		bool differs=FALSE;
		if (!(start.v[i] == ep->mm.v[i])) differs=TRUE;
		if (!differs && (start.vfac[i].Count() != ep->mm.vfac[i].Count())) differs=TRUE;
		if (!differs) {
			for (j=0; j<start.vfac[i].Count(); j++) if (start.vfac[i][j] != ep->mm.vfac[i][j]) break;
			if (j<start.vfac[i].Count()) differs=TRUE;
		}
		if (!differs && (start.vedg[i].Count() != ep->mm.vedg[i].Count())) differs = TRUE;
		if (!differs) {
			for (j=0; j<start.vedg[i].Count(); j++) if (start.vedg[i][j] != ep->mm.vedg[i][j]) break;
			if (j<start.vedg[i].Count()) differs=TRUE;
		}
		if (!differs) continue;
		// Something's changed:
		vertID.Append (1, &i, minv/10);
	}
	vertsBefore.SetCount (vertID.Count()+numvBefore-minv);
	vertsAfter.SetCount (vertID.Count()+numvAfter-minv);
	if (ct=vertsBefore.Count()) {
		vfacBefore = new Tab<int>[ct];
		vedgBefore = new Tab<int>[ct];
	}
	if (ct=vertsAfter.Count()) {
		vfacAfter = new Tab<int>[ct];
		vedgAfter = new Tab<int>[ct];
	}
	ct=vertID.Count();
	for (i=0; i<ct; i++) {
		int j = vertID[i];
		vertsBefore[i] = start.v[j];
		vfacBefore[i] = start.vfac[j];
		vedgBefore[i] = start.vedg[j];
		vertsAfter[i] = ep->mm.v[j];
		vfacAfter[i] = ep->mm.vfac[j];
		vedgAfter[i] = ep->mm.vedg[j];
	}
	for (i=minv; i<numvBefore; i++) {
		int j = i-minv+ct;
		vertsBefore[j] = start.v[i];
		vfacBefore[j] = start.vfac[i];
		vedgBefore[j] = start.vedg[i];
	}
	for (i=minv; i<numvAfter; i++) {
		int j = i-minv+ct;
		vertsAfter[j] = ep->mm.v[i];
		vfacAfter[j] = ep->mm.vfac[i];
		vedgAfter[j] = ep->mm.vedg[i];
	}

	// Don't get fancy with the vertex and edge data, just back it up.
	vertexDataCount = start.vdSupport.GetSize ();
	if (ep->mm.vdSupport.GetSize() > vertexDataCount) vertexDataCount = ep->mm.vdSupport.GetSize();
	if (vertexDataCount) {
		vertexDataBefore = new Tab<float>[vertexDataCount];
		vertexDataAfter = new Tab<float>[vertexDataCount];
		for (mp=0; mp<vertexDataCount; mp++) {
			if (mp == VDATA_SELECT) continue;	// Please don't bother storing soft selection data.
			float *ovd = start.vertexFloat (mp);
			float *nvd = ep->mm.vertexFloat (mp);
			if (ovd && start.numv) {
				vertexDataBefore[mp].SetCount (start.numv);
				memcpy (vertexDataBefore[mp].Addr(0), ovd, start.numv*sizeof(float));
			}
			if (nvd && ep->mm.numv) {
				vertexDataAfter[mp].SetCount (ep->mm.numv);
				memcpy (vertexDataAfter[mp].Addr(0), nvd, ep->mm.numv*sizeof(float));
			}
		}
	}

	edgeDataCount = start.edSupport.GetSize ();
	if (ep->mm.edSupport.GetSize() > edgeDataCount) edgeDataCount = ep->mm.edSupport.GetSize();
	if (edgeDataCount) {
		edgeDataBefore = new Tab<float>[edgeDataCount];
		edgeDataAfter = new Tab<float>[edgeDataCount];
		for (mp=0; mp<edgeDataCount; mp++) {
			float *oed = start.edgeFloat (mp);
			float *ned = ep->mm.edgeFloat (mp);
			if (oed && start.nume) {
				edgeDataBefore[mp].SetCount (start.nume);
				memcpy (edgeDataBefore[mp].Addr(0), oed, start.nume*sizeof(float));
			}
			if (ned && ep->mm.nume) {
				edgeDataAfter[mp].SetCount (ep->mm.nume);
				memcpy (edgeDataAfter[mp].Addr(0), ned, ep->mm.nume*sizeof(float));
			}
		}
	}

	// Record Edge changes:
	int mine = (numeBefore < numeAfter) ? numeBefore : numeAfter;
	for (i=0; i<mine; i++) {
		if (start.e[i] == ep->mm.e[i]) continue;
		// Something's changed:
		edgeID.Append (1, &i, mine/10);
	}
	edgesBefore.SetCount (edgeID.Count()+numeBefore-mine);
	edgesAfter.SetCount (edgeID.Count()+numeAfter-mine);
	for (i=0; i<edgeID.Count(); i++) {
		int j = edgeID[i];
		edgesBefore[i] = start.e[j];
		edgesAfter[i] = ep->mm.e[j];
	}
	for (i=mine; i<numeBefore; i++) {
		int j = i-mine+edgeID.Count();
		edgesBefore[j] = start.e[i];
	}
	for (i=mine; i<numeAfter; i++) {
		int j = i-mine+edgeID.Count();
		edgesAfter[j] = ep->mm.e[i];
	}

	// Record face changes:
	int minf = (numfBefore < numfAfter) ? numfBefore : numfAfter;
	for (i=0; i<minf; i++) {
		if (start.f[i] == ep->mm.f[i]) continue;
		// Something's changed:
		faceID.Append (1, &i, minf/10);
	}
	MNFace ftemp;
	ftemp.SetAlloc (0);
	facesBefore.Resize (faceID.Count()+numfBefore-minf);
	facesAfter.Resize (faceID.Count()+numfAfter-minf);
	for (i=0; i<faceID.Count(); i++) {
		int j = faceID[i];
		facesBefore.Append (1, &ftemp);
		facesBefore[i] = start.f[j];
		facesAfter.Append (1, &ftemp);
		facesAfter[i] = ep->mm.f[j];
	}
	for (i=minf; i<numfBefore; i++) {
		int j = i-minf+faceID.Count();
		facesBefore.Append (1, &ftemp);
		facesBefore[j] = start.f[i];
	}
	for (i=minf; i<numfAfter; i++) {
		int j = i-minf+faceID.Count();
		facesAfter.Append (1, &ftemp);
		facesAfter[j] = ep->mm.f[i];
	}

	// Record Map Vertex and Face changes:
	for (int nmp=0; nmp<numMvBefore.Count(); nmp++) {
		mp = nmp - NUM_HIDDENMAPS;
		if (ep->mm.M(mp)->GetFlag (MN_DEAD)) continue;
		int minmv = numMvBefore[nmp];
		if (minmv > numMvAfter[nmp]) minmv = numMvAfter[nmp];
		for (i=0; i<minmv; i++) {
			if (start.M(mp)->v[i] == ep->mm.M(mp)->v[i]) continue;
			// Something's changed:
			mapVertID[nmp].Append (1, &i, minmv/10);
		}

		mapVertsBefore[nmp].SetCount (mapVertID[nmp].Count()+numMvBefore[nmp]-minmv);
		mapVertsAfter[nmp].SetCount (mapVertID[nmp].Count()+numMvAfter[nmp]-minmv);

		ct=mapVertID[nmp].Count();
		for (i=0; i<ct; i++) {
			int j = mapVertID[nmp][i];
			mapVertsBefore[nmp][i] = start.M(mp)->v[j];
			mapVertsAfter[nmp][i] = ep->mm.M(mp)->v[j];
		}
		for (i=minmv; i<numMvBefore[nmp]; i++) {
			mapVertsBefore[nmp][i-minmv+ct] = start.M(mp)->v[i];
		}
		for (i=minmv; i<numMvAfter[nmp]; i++) {
			mapVertsAfter[nmp][i-minmv+ct] = ep->mm.M(mp)->v[i];
		}

		// Faces.
		// (Use same minf, numfbefore, numfafter, etc, as for main mesh.)
		mapFaceID[nmp].SetCount(0);
		for (i=0; i<minf; i++) {
			if (start.M(mp)->f[i] == ep->mm.M(mp)->f[i]) continue;
			// Something's changed:
			mapFaceID[nmp].Append (1, &i, minf/10);
		}

		MNMapFace mftemp;
		mftemp.SetAlloc(0);

		mapFacesBefore[nmp].Resize (mapFaceID[nmp].Count()+numfBefore-minf);
		mapFacesAfter[nmp].Resize (mapFaceID[nmp].Count()+numfAfter-minf);
		for (i=0; i<mapFaceID[nmp].Count(); i++) {
			int j = mapFaceID[nmp][i];
			mapFacesBefore[nmp].Append (1, &mftemp);
			mapFacesBefore[nmp][i] = start.M(mp)->f[j];
			mapFacesAfter[nmp].Append (1, &mftemp);
			mapFacesAfter[nmp][i] = ep->mm.M(mp)->f[j];
		}
		for (i=minf; i<numfBefore; i++) {
			int j = i-minf+mapFaceID[nmp].Count();
			mapFacesBefore[nmp].Append (1, &mftemp);
			mapFacesBefore[nmp][j] = start.M(mp)->f[i];
		}
		for (i=minf; i<numfAfter; i++) {
			int j = i-minf+mapFaceID[nmp].Count();
			mapFacesAfter[nmp].Append (1, &mftemp);
			mapFacesAfter[nmp][j] = ep->mm.M(mp)->f[i];
		}
	}

	// Get rid of bulky extra mesh
	start.ClearAndFree ();
}

void TopoChangeRestore::Restore (int isUndo) {
	// Undo vertex changes:
#ifdef EPOLY_RESTORE_DEBUG_PRINT
	DebugPrint ("Undoing a topological change.\n");
	ep->mm.MNDebugPrint ();
#endif
	if (numvBefore != numvAfter) ep->mm.setNumVerts (numvBefore);
	for (int i=0; i<vertID.Count (); i++) {
		int j=vertID[i];
		ep->mm.v[j] = vertsBefore[i];
		ep->mm.vfac[j] = vfacBefore[i];
		ep->mm.vedg[j] = vedgBefore[i];
	}
	int offset = vertID.Count() - numvAfter;
	for (i=numvAfter; i<numvBefore; i++) {
		int j=i+offset;
		ep->mm.v[i] = vertsBefore[j];
		ep->mm.vfac[i] = vfacBefore[j];
		ep->mm.vedg[i] = vedgBefore[j];
	}

	// Restore the vertex data:
	for (int mp=0; mp<vertexDataCount; mp++) {
		if (mp == VDATA_SELECT) continue;
		if (vertexDataBefore[mp].Count() == 0) {
			ep->mm.setVDataSupport (mp, false);
			continue;
		}
		ep->mm.setVDataSupport (mp, true);
		float *vd = ep->mm.vertexFloat (mp);
		if (vd && numvBefore) memcpy (vd, vertexDataBefore[mp].Addr(0), numvBefore * sizeof(float));
	}

	// Undo edge changes:
	if (numeBefore != numeAfter) ep->mm.setNumEdges (numeBefore);
	for (i=0; i<edgeID.Count (); i++) ep->mm.e[edgeID[i]] = edgesBefore[i];
	offset = edgeID.Count() - numeAfter;
	for (i=numeAfter; i<numeBefore; i++) ep->mm.e[i] = edgesBefore[i+offset];

	// Restore the edge data:
	for (mp=0; mp<edgeDataCount; mp++) {
		if (edgeDataBefore[mp].Count() == 0) {
			ep->mm.setEDataSupport (mp, false);
			continue;
		}
		ep->mm.setEDataSupport (mp, true);
		float *ed = ep->mm.edgeFloat (mp);
		if (ed && numeBefore) memcpy (ed, edgeDataBefore[mp].Addr(0), numeBefore * sizeof(float));
	}

	// Undo face changes:
	if (numfBefore != numfAfter) ep->mm.setNumFaces (numfBefore);
	for (i=0; i<faceID.Count (); i++) ep->mm.f[faceID[i]] = facesBefore[i];
	offset = faceID.Count() - numfAfter;
	for (i=numfAfter; i<numfBefore; i++) {
		ep->mm.f[i].Init ();
		ep->mm.f[i] = facesBefore[i+offset];
	}

	// Undo map changes:
	int nhm = NUM_HIDDENMAPS;
	for (mp=-nhm; mp<ep->mm.numm; mp++) {
		if (ep->mm.M(mp)->GetFlag (MN_DEAD)) continue;
		int nmp = mp + nhm;

		// Map Vertices:
		if (numMvBefore[nmp] != numMvAfter[nmp]) {
			ep->mm.M(mp)->setNumVerts (numMvBefore[nmp]);
		}
		for (i=0; i<mapVertID[nmp].Count (); i++) {
			int j=mapVertID[nmp][i];
			ep->mm.M(mp)->v[j] = mapVertsBefore[nmp][i];
		}
		int offset = mapVertID[nmp].Count() - numMvAfter[nmp];
		for (i=numMvAfter[nmp]; i<numMvBefore[nmp]; i++) {
			ep->mm.M(mp)->v[i] = mapVertsBefore[nmp][i+offset];
		}

		// Map Faces:
		if (numfBefore != numfAfter) ep->mm.M(mp)->setNumFaces (numfBefore);
		for (i=0; i<mapFaceID[nmp].Count(); i++) {
			ep->mm.M(mp)->f[mapFaceID[nmp][i]] = mapFacesBefore[nmp][i];
		}
		offset = mapFaceID[nmp].Count() - numfAfter;
		for (i=numfAfter; i<numfBefore; i++) {
			ep->mm.M(mp)->f[i].Init ();
			ep->mm.M(mp)->f[i] = mapFacesBefore[nmp][i+offset];
		}
	}

#ifdef EPOLY_RESTORE_DEBUG_PRINT
	DebugPrint ("Topological change undone.\n");
	ep->mm.MNDebugPrint ();
#endif
	DbgAssert (ep->mm.CheckAllData ());
	ep->LocalDataChanged (PART_ALL-PART_SUBSEL_TYPE);
}

void TopoChangeRestore::Redo () {
#ifdef EPOLY_RESTORE_DEBUG_PRINT
	DebugPrint ("Undoing a topological change.\n");
	ep->mm.MNDebugPrint ();
#endif
	// Redo vertex changes:
	if (numvBefore != numvAfter) ep->mm.setNumVerts (numvAfter);
	for (int i=0; i<vertID.Count (); i++) {
		int j = vertID[i];
		ep->mm.v[j] = vertsAfter[i];
		ep->mm.vfac[j] = vfacAfter[i];
		ep->mm.vedg[j] = vedgAfter[i];
	}
	int offset = vertID.Count() - numvBefore;
	for (i=numvBefore; i<numvAfter; i++) {
		int j = i+offset;
		ep->mm.v[i] = vertsAfter[j];
		ep->mm.vfac[i] = vfacAfter[j];
		ep->mm.vedg[i] = vedgAfter[j];
	}

	// Redo the vertex data:
	for (int mp=0; mp<vertexDataCount; mp++) {
		if (mp == VDATA_SELECT) continue;
		if (vertexDataAfter[mp].Count() == 0) {
			ep->mm.setVDataSupport (mp, false);
			continue;
		}
		ep->mm.setVDataSupport (mp, true);
		float *vd = ep->mm.vertexFloat (mp);
		if (vd && numvAfter) memcpy (vd, vertexDataAfter[mp].Addr(0), numvAfter * sizeof(float));
	}

	// Redo edge changes:
	if (numeBefore != numeAfter) ep->mm.setNumEdges (numeAfter);
	for (i=0; i<edgeID.Count (); i++) ep->mm.e[edgeID[i]] = edgesAfter[i];
	offset = edgeID.Count() - numeBefore;
	for (i=numeBefore; i<numeAfter; i++) ep->mm.e[i] = edgesAfter[i+offset];

	// Redo the edge data:
	for (mp=0; mp<edgeDataCount; mp++) {
		if (edgeDataAfter[mp].Count() == 0) {
			ep->mm.setEDataSupport (mp, false);
			continue;
		}
		ep->mm.setEDataSupport (mp, true);
		float *ed = ep->mm.edgeFloat (mp);
		if (ed && numeAfter) memcpy (ed, edgeDataAfter[mp].Addr(0), numeAfter * sizeof(float));
	}

	// Redo face changes:
	if (numfBefore != numfAfter) ep->mm.setNumFaces (numfAfter);
	for (i=0; i<faceID.Count (); i++) {
		int j=faceID[i];
		ep->mm.f[j] = facesAfter[i];
	}
	offset = faceID.Count() - numfBefore;
	for (i=numfBefore; i<numfAfter; i++) {
		ep->mm.f[i].Init ();
		ep->mm.f[i] = facesAfter[i+offset];
	}

	// Redo map changes:
	int nhm = NUM_HIDDENMAPS;
	for (mp=-nhm; mp<ep->mm.numm; mp++) {
		int nmp = nhm + mp;
		if (ep->mm.M(mp)->GetFlag (MN_DEAD)) continue;

		// Map Vertices:
		if (numMvBefore[nmp] != numMvAfter[nmp]) {
			ep->mm.M(mp)->setNumVerts (numMvAfter[nmp]);
		}
		for (i=0; i<mapVertID[nmp].Count (); i++) {
			int j=mapVertID[nmp][i];
			ep->mm.M(mp)->v[j] = mapVertsAfter[nmp][i];
		}
		int offset = mapVertID[nmp].Count() - numMvBefore[nmp];
		for (i=numMvBefore[nmp]; i<numMvAfter[nmp]; i++) {
			ep->mm.M(mp)->v[i] = mapVertsAfter[nmp][i+offset];
		}

		// Map Faces:
		if (numfBefore != numfAfter) ep->mm.M(mp)->setNumFaces (numfAfter);
		for (i=0; i<mapFaceID[nmp].Count(); i++) {
			ep->mm.M(mp)->f[mapFaceID[nmp][i]] = mapFacesAfter[nmp][i];
		}
		offset = mapFaceID[nmp].Count() - numfBefore;
		for (i=numfBefore; i<numfAfter; i++) {
			ep->mm.M(mp)->f[i].Init ();
			ep->mm.M(mp)->f[i] = mapFacesAfter[nmp][i+offset];
		}
	}

	DbgAssert (ep->mm.CheckAllData ());
	ep->LocalDataChanged (PART_ALL-PART_SUBSEL_TYPE);
}

MapChangeRestore::MapChangeRestore (EditPolyObject *e, int mapchannel) {
	ep = e;
	mp = mapchannel;
	if ((ep->mm.numm <= mp) || (ep->mm.M(mp)->GetFlag (MN_DEAD))) {
		preVerts.SetCount (0);
		preFaces.SetCount (0);
		mapFlagsBefore = MN_DEAD;
	} else {
		preVerts.SetCount (ep->mm.M(mp)->numv);
		preFaces.SetCount (ep->mm.numf);
		if (ep->mm.M(mp)->numv) memcpy (preVerts.Addr(0), ep->mm.M(mp)->v, ep->mm.M(mp)->numv * sizeof(UVVert));
		for (int i=0; i<ep->mm.numf; i++) {
			preFaces[i].Init ();
			if (ep->mm.f[i].GetFlag (MN_DEAD)) continue;
			preFaces[i] = ep->mm.M(mp)->f[i];
		}
		mapFlagsBefore = ep->mm.M(mp)->ExportFlags ();
	}
}

MapChangeRestore::~MapChangeRestore () {
	for (int i=0; i<preFaces.Count(); i++) preFaces[i].Clear ();
	for (i=0; i<fbefore.Count(); i++) fbefore[i].Clear ();
	for (i=0; i<fafter.Count(); i++) fafter[i].Clear ();
}

bool MapChangeRestore::ReduceMem () {
	numvBefore = preVerts.Count();
	numvAfter = ep->mm.M(mp)->numv;
	numfBefore = preFaces.Count();
	numfAfter = ep->mm.M(mp)->numf;
	mapFlagsAfter = ep->mm.M(mp)->ExportFlags ();

	// Record vertex changes:
	int i, minv = (numvBefore < numvAfter) ? numvBefore : numvAfter;
	for (i=0; i<minv; i++) {
		if (preVerts[i] == ep->mm.M(mp)->v[i]) continue;
		// Something's changed:
		vertID.Append (1, &i, minv/10);
	}
	vbefore.SetCount (vertID.Count()+numvBefore-minv);
	vafter.SetCount (vertID.Count()+numvAfter-minv);
	int ct=vertID.Count();

	for (i=0; i<ct; i++) {
		int j = vertID[i];
		vbefore[i] = preVerts[j];
		vafter[i] = ep->mm.M(mp)->v[j];
	}
	for (i=minv; i<numvBefore; i++) {
		int j = i-minv+ct;
		vbefore[j] = preVerts[i];
	}
	for (i=minv; i<numvAfter; i++) {
		int j = i-minv+ct;
		vafter[j] = ep->mm.M(mp)->v[i];
	}

	// Record Face changes:
	int minf = ep->mm.numf;
	if (numfBefore < minf) minf = numfBefore;
	for (i=0; i<minf; i++) {
		if (preFaces[i] == ep->mm.M(mp)->f[i]) continue;
		// Something's changed:
		faceID.Append (1, &i, minf/10);
	}

	ct=faceID.Count ();
	fbefore.SetCount (ct + numfBefore-minf);
	fafter.SetCount (ct + numfAfter-minf);

	for (i=0; i<ct; i++) {
		int j = faceID[i];
		fbefore[i].Init ();
		fafter[i].Init ();
		fbefore[i] = preFaces[j];
		fafter[i] = ep->mm.M(mp)->f[j];
	}
	for (i=minf; i<numfBefore; i++) {
		int j = i-minf+ct;
		fbefore[j].Init ();
		fbefore[j] = preFaces[i];
	}
	for (i=minf; i<numfAfter; i++) {
		int j = i-minf+ct;
		fbefore[j].Init ();
		fafter[j] = ep->mm.M(mp)->f[i];
	}

	for (i=0; i<preFaces.Count(); i++) preFaces[i].Clear();
	preFaces.ZeroCount ();
	preFaces.Shrink ();
	preVerts.ZeroCount ();
	preVerts.Shrink ();

	if (mapFlagsAfter != mapFlagsAfter) return true;
	if (numfAfter != numfBefore) return true;
	if (numvAfter != numvBefore) return true;
	if (vbefore.Count() || vafter.Count() || fbefore.Count() || fafter.Count()) return true;
	// no change whatsoever.
	return false;
}

void MapChangeRestore::Restore (int isUndo) {
	if ((preFaces.Count() > 0) || (numvAfter == 0)) ReduceMem ();

	// Undo vertex changes:
	if (numvBefore != numvAfter) ep->mm.M(mp)->setNumVerts (numvBefore);
	for (int i=0; i<vertID.Count (); i++) {
		ep->mm.M(mp)->v[vertID[i]] = vbefore[i];
	}
	int offset = vertID.Count() - numvAfter;
	for (i=numvAfter; i<numvBefore; i++) {
		ep->mm.M(mp)->v[i] = vbefore[i+offset];
	}

	// Undo face changes:
	if (numfBefore != numfAfter) ep->mm.M(mp)->setNumFaces (numfBefore);
	for (i=0; i<faceID.Count(); i++) {
		ep->mm.M(mp)->f[faceID[i]] = fbefore[i];
	}
	offset = faceID.Count() - numfAfter;
	for (i=numfAfter; i<numfBefore; i++) {
		ep->mm.M(mp)->f[i] = fbefore[i+offset];
	}

	// Set flags to earlier state:
	ep->mm.M(mp)->ImportFlags (mapFlagsBefore);

	ep->LocalDataChanged (PART_VERTCOLOR);
}

void MapChangeRestore::Redo () {
	// Redo vertex changes:
	if (numvBefore != numvAfter) ep->mm.M(mp)->setNumVerts (numvAfter);
	for (int i=0; i<vertID.Count (); i++) {
		ep->mm.M(mp)->v[vertID[i]] = vafter[i];
	}
	int offset = vertID.Count() - numvBefore;
	for (i=numvBefore; i<numvAfter; i++) {
		ep->mm.M(mp)->v[i] = vafter[i+offset];
	}

	// Undo face changes:
	if (numfBefore != numfAfter) ep->mm.M(mp)->setNumFaces (numfAfter);
	for (i=0; i<faceID.Count(); i++) {
		ep->mm.M(mp)->f[faceID[i]] = fafter[i];
	}
	offset = faceID.Count() - numfBefore;
	for (i=numfBefore; i<numfAfter; i++) {
		ep->mm.M(mp)->f[i] = fafter[i+offset];
	}

	// Set flags to earlier state:
	ep->mm.M(mp)->ImportFlags (mapFlagsAfter);

	ep->LocalDataChanged (PART_VERTCOLOR);
}

//---------------------------------------------------

MapVertRestore::MapVertRestore (EditPolyObject *e, int mapchannel) {
	ep = e;
	mp = mapchannel;
	if ((ep->mm.numm <= mp) || (ep->mm.M(mp)->GetFlag (MN_DEAD))) {
		preVerts.SetCount (0);
	} else {
		preVerts.SetCount (ep->mm.M(mp)->numv);
		if (ep->mm.M(mp)->numv)
			memcpy (preVerts.Addr(0), ep->mm.M(mp)->v, ep->mm.M(mp)->numv * sizeof(UVVert));
	}
}

bool MapVertRestore::ReduceMem () {
	int numv = ep->mm.M(mp)->numv;

	// Record vertex changes:
	int i;
	vertID.ZeroCount();
	for (i=0; i<numv; i++) {
		if (preVerts[i] == ep->mm.M(mp)->v[i]) continue;
		// Something's changed:
		vertID.Append (1, &i, numv/10);
	}
	int ct=vertID.Count();
	vbefore.SetCount (ct);
	vafter.SetCount (ct);

	for (i=0; i<ct; i++) {
		int j = vertID[i];
		vbefore[i] = preVerts[j];
		vafter[i] = ep->mm.M(mp)->v[j];
	}

	// Clear out preVerts array
	preVerts.ZeroCount ();
	preVerts.Shrink ();

	if (ct) return true;
	// no change whatsoever.
	return false;
}

void MapVertRestore::Restore (int isUndo) {
	if (preVerts.Count() > 0) ReduceMem ();
	// Undo vertex changes:
	for (int i=0; i<vertID.Count (); i++) {
		ep->mm.M(mp)->v[vertID[i]] = vbefore[i];
	}
	ep->LocalDataChanged ((mp>0) ? PART_TEXMAP : PART_VERTCOLOR);
}

void MapVertRestore::Redo () {
	for (int i=0; i<vertID.Count (); i++) {
		ep->mm.M(mp)->v[vertID[i]] = vafter[i];
	}
	ep->LocalDataChanged ((mp>0) ? PART_TEXMAP : PART_VERTCOLOR);
}

//---------------------------------------------------

CreateOnlyRestore::CreateOnlyRestore (EditPolyObject *ep) {
	epol = ep;
	ovnum = ep->mm.numv;
	oenum = ep->mm.nume;
	ofnum = ep->mm.numf;
	int nhm = NUM_HIDDENMAPS;
	omvnum.SetCount (ep->mm.numm + nhm);
	omfnum.SetCount (ep->mm.numm + nhm);
	omapsUsed.SetSize (ep->mm.numm + nhm);
	for (int mp=-nhm; mp<ep->mm.numm; mp++) {
		int nmp = mp + nhm;
		MNMap *map = ep->mm.M(mp);
		if (map->GetFlag (MN_DEAD)) {
			omvnum[nmp] = 0;
			omfnum[nmp] = 0;
			omapsUsed.Clear (nmp);
		} else {
			omvnum[nmp] = map->numv;
			omfnum[nmp] = map->numf;
			omapsUsed.Set (nmp);
		}
	}
	nvfac = NULL;
	nvedg = NULL;
	nmverts = NULL;
	nmfaces = NULL;
	afterCalled = false;
}

CreateOnlyRestore::~CreateOnlyRestore () {
	if (nvfac) delete [] nvfac;
	if (nvedg) delete [] nvedg;
	if (nmverts) delete [] nmverts;
	if (nmfaces) delete [] nmfaces;
}

void CreateOnlyRestore::After () {
	int i, nvnum = epol->mm.numv - ovnum;
	if (nvnum>0) {
		nverts.ZeroCount ();
		nverts.Append (nvnum, epol->mm.V(ovnum));
		if (!nvfac) nvfac = new Tab<int>[nvnum];
		if (!nvedg) nvedg = new Tab<int>[nvnum];
		for (i=0; i<nvnum; i++) {
			nvfac[i] = epol->mm.vfac[i+ovnum];
			nvedg[i] = epol->mm.vedg[i+ovnum];
		}
	}

	int nenum = epol->mm.nume - oenum;
	if (nenum>0) {
		nedges.ZeroCount ();
		nedges.Append (nenum, epol->mm.E(oenum));
	}

	int nfnum = epol->mm.numf - ofnum;
	if (nfnum>0) {
		nfaces.ZeroCount ();
		nfaces.Resize (nfnum);
		MNFace temp;
		temp.Init();
		for (i=0; i<nfnum; i++) {
			nfaces.Append (1, &temp);
			nfaces[i] = epol->mm.f[i+ofnum];
		}
	}

	int mp, nhm = NUM_HIDDENMAPS;
	if (epol->mm.numm + nhm > omvnum.Count()) {
		int oldnumm = omvnum.Count();
		omvnum.SetCount (epol->mm.numm + nhm);
		omfnum.SetCount (epol->mm.numm + nhm);
		omapsUsed.SetSize (epol->mm.numm + nhm, true);
		for (int nmp=oldnumm; nmp<epol->mm.numm + nhm; nmp++) {
			omvnum[nmp] = 0;
			omfnum[nmp] = 0;
			omapsUsed.Clear (nmp);
		}
	}
	nmapsUsed.SetSize (epol->mm.numm + nhm);
	if (!nmverts) nmverts = new Tab<UVVert>[epol->mm.numm + nhm];
	if (!nmfaces) nmfaces = new Tab<MNMapFace>[epol->mm.numm + nhm];
	for (mp=-nhm; mp<epol->mm.numm; mp++) {
		int nmp = nhm + mp;
		MNMap *map = epol->mm.M(mp);
		nmapsUsed.Set (nmp, !map->GetFlag (MN_DEAD));
		if (map->GetFlag (MN_DEAD)) continue;
		int nmvnum = map->numv - omvnum[nmp];
		if (nmvnum>0) {
			nmverts[nmp].SetCount (nmvnum);
			memcpy (nmverts[nmp].Addr(0), (void*)(map->v+omvnum[nmp]), nmvnum*sizeof(UVVert));
		}

		int nmfnum = map->numf - omfnum[nmp];
		if (nmfnum>0) {
			nmfaces[nmp].SetCount (nmfnum);
			for (i=0; i<nmfnum; i++) {
				nmfaces[nmp][i].Init ();
				nmfaces[nmp][i] = map->f[i+omfnum[nmp]];
			}
		}
	}

	afterCalled = true;
}

void CreateOnlyRestore::Restore (BOOL isUndo) {
	if (isUndo && !afterCalled) After ();

	// Simplest possible deletion of new components:
	epol->mm.setNumVerts (ovnum);
	epol->mm.setNumEdges (oenum);
	epol->mm.setNumFaces (ofnum);
	for (int mp=-NUM_HIDDENMAPS; mp<epol->mm.numm; mp++) {
		if (epol->mm.M(mp)->GetFlag (MN_DEAD)) continue;
		int nmp = mp + NUM_HIDDENMAPS;
		epol->mm.M(mp)->setNumFaces (omfnum[nmp]);
		epol->mm.M(mp)->setNumVerts (omvnum[nmp]);
		if (!omapsUsed[nmp]) epol->mm.M(mp)->SetFlag (MN_DEAD);
	}
	epol->LocalDataChanged (PART_TOPO|PART_GEOM|PART_SELECT);
	DbgAssert (epol->mm.CheckAllData ());
}

void CreateOnlyRestore::Redo () {
	// Put back the new components:
	int i;
	if (nverts.Count()) {
		epol->mm.setNumVerts (ovnum + nverts.Count());
		memcpy (epol->mm.V(ovnum), nverts.Addr(0), nverts.Count()*sizeof(MNVert));
		for (i=0; i<nverts.Count(); i++) {
			epol->mm.vedg[i+ovnum] = nvedg[i];
			epol->mm.vfac[i+ovnum] = nvfac[i];
		}
	}

	if (nedges.Count()) {
		epol->mm.setNumEdges (oenum + nedges.Count());
		memcpy (epol->mm.E(oenum), nedges.Addr(0), nedges.Count()*sizeof(MNEdge));
	}

	if (nfaces.Count()) {
		epol->mm.setNumFaces (ofnum + nfaces.Count());
		for (i=0; i<nfaces.Count(); i++) {
			epol->mm.f[i+ofnum].Init ();
			epol->mm.f[i+ofnum] = nfaces[i];
		}
	}

	int mp, nhm=NUM_HIDDENMAPS;
	for (mp=-nhm; mp<epol->mm.numm; mp++) {
		int nmp = nhm + mp;
		if (!nmapsUsed[nmp]) continue;
		MNMap *map = epol->mm.M(mp);
		map->ClearFlag (MN_DEAD);
		map->setNumFaces (omfnum[nmp] + nmfaces[nmp].Count());
		map->setNumVerts (omvnum[nmp] + nmverts[nmp].Count());
		for (i=0; i<nmfaces[nmp].Count(); i++) {
			map->f[i+omfnum[nmp]].Init ();
			map->f[i+omfnum[nmp]] = nmfaces[nmp][i];
		}
		if (nmverts[nmp].Count()) memcpy ((void *)(map->v + omvnum[nmp]), nmverts[nmp].Addr(0), nmverts[nmp].Count()*sizeof(UVVert));
	}

	epol->LocalDataChanged (PART_TOPO|PART_GEOM|PART_SELECT);
	DbgAssert (epol->mm.CheckAllData ());
}

MeshVertRestore::MeshVertRestore(EditPolyObject *ep) {
	this->ep = ep;
	undo.SetCount(ep->mm.numv);
	for (int i=0; i<ep->mm.numv; i++) undo[i] = ep->mm.v[i].p;
}

void MeshVertRestore::Restore(int isUndo) {
	int i;
	if (isUndo) {
		redo.SetCount(ep->mm.numv);
		for (i=0; i<ep->mm.numv; i++) redo[i] = ep->mm.v[i].p;
	}
	for (i=0; i<ep->mm.numv; i++) ep->mm.v[i].p = undo[i];
	ep->LocalDataChanged (PART_GEOM);
}

void MeshVertRestore::Redo() {
	int i;
	for (i=0; i<ep->mm.numv; i++) ep->mm.v[i].p = redo[i];
	ep->LocalDataChanged (PART_GEOM);
}

MtlIDRestore::MtlIDRestore (EditPolyObject *ep) {
	this->ep = ep;
	undo.SetCount(ep->mm.numf);
	for (int i=0; i<ep->mm.numf; i++) undo[i] = ep->mm.f[i].material;
}

void MtlIDRestore::After () {
	redo.SetCount(ep->mm.numf);
	for (int i=0; i<ep->mm.numf; i++) redo[i] = ep->mm.f[i].material;
}

void MtlIDRestore::Restore(int isUndo) {
	if (redo.Count() == 0) After ();
	for (int i=0; i<ep->mm.numf; i++) ep->mm.f[i].material = undo[i];
	ep->InvalidateSurfaceUI();
	ep->NotifyDependents(FOREVER, PART_TOPO, REFMSG_CHANGE);
}

void MtlIDRestore::Redo() {
	for (int i=0; i<ep->mm.numf; i++) ep->mm.f[i].material = redo[i];
	ep->InvalidateSurfaceUI();
	ep->NotifyDependents(FOREVER, PART_TOPO, REFMSG_CHANGE);
}

SmGroupRestore::SmGroupRestore (EditPolyObject *e) {
	ep = e;
	osmg.SetCount (ep->mm.numf);
	for (int i=0; i<ep->mm.numf; i++) osmg[i] = ep->mm.f[i].smGroup;
}

void SmGroupRestore::After () {
	nsmg.SetCount (ep->mm.numf);
	for (int i=0; i<ep->mm.numf; i++) nsmg[i] = ep->mm.f[i].smGroup;
}

void SmGroupRestore::Restore (int isUndo) {
	if (!nsmg.Count ()) After ();
	int i, min = ep->mm.numf;
	if (osmg.Count() < min) min = osmg.Count ();
	for (i=0; i<min; i++) ep->mm.f[i].smGroup = osmg[i];
	ep->LocalDataChanged (PART_TOPO);
}

void SmGroupRestore::Redo () {
	int i, min = ep->mm.numf;
	if (nsmg.Count() < min) min = nsmg.Count ();
	for (i=0; i<min; i++) ep->mm.f[i].smGroup = nsmg[i];
	ep->LocalDataChanged (PART_TOPO);
}

CollapseDeadVertsRestore::CollapseDeadVertsRestore (EditPolyObject *e) {
	undovedg = NULL;
	undovfac = NULL;
	ep = e;
	startNum = ep->mm.numv;
	int allocAmt = ep->mm.numv/10;
	int ct=0;
	for (int i=0; i<ep->mm.numv; i++) {
		if (!ep->mm.v[i].GetFlag (MN_DEAD)) continue;
		positions.Append (1, &i, allocAmt);
		ct++;
	}
	if (!ct) return;
	undo.SetCount (ct);
	undovedg = new Tab<int>[ct];
	undovfac = new Tab<int>[ct];
	for (i=0; i<ct; i++) {
		undo[i] = ep->mm.v[positions[i]];
		undovedg[i] = ep->mm.vedg[positions[i]];
		undovfac[i] = ep->mm.vfac[positions[i]];
	}
}

void CollapseDeadVertsRestore::Restore (int isUndo) {
#ifdef EPOLY_RESTORE_DEBUG_PRINT
	MyDebugPrint ();
	DebugPrint ("Restoring CollapseDeadVerts - starting mesh:\n");
	ep->mm.MNDebugPrint ();
#endif
	DbgAssert (ep->mm.CheckAllData ());
	int i, j, k, ct = positions.Count();
	if (!ct) return;
	int newNumV = ep->mm.numv + ct;

	// Figure out the renumbering scheme for non-deleted faces:
	Tab<int> renum;
	renum.SetCount (ep->mm.numv);
	for (i=0, j=0, k=0; i<ep->mm.numv; i++) {
		while ((j<ct) && (positions[j] == k)) {
			// leave room for a deleted face:
			k++;
			j++;
		}
		renum[i] = k;
		k++;
	}
	DbgAssert (k+(ct-j) == newNumV);

	// Correct vertex references in edges, faces:
	DbgAssert (ep->mm.GetFlag (MN_MESH_FILLED_IN));

	for (i=0; i<ep->mm.nume; i++) {
		if (ep->mm.e[i].GetFlag (MN_DEAD)) continue;
		ep->mm.e[i].v1 = renum[ep->mm.e[i].v1];
		ep->mm.e[i].v2 = renum[ep->mm.e[i].v2];
	}

	for (i=0; i<ep->mm.numf; i++) {
		if (ep->mm.f[i].GetFlag (MN_DEAD)) continue;
		for (j=0; j<ep->mm.f[i].deg; j++) ep->mm.f[i].vtx[j] = renum[ep->mm.f[i].vtx[j]];
	}

	// Insert room for deleted vertices:
	int oldNumV = ep->mm.numv;
	ep->mm.setNumVerts (newNumV);
	for (i=oldNumV-1; i>=0; i--) {
		if (renum[i]==i) continue;
		ep->mm.v[renum[i]] = ep->mm.v[i];
		ep->mm.vfac[renum[i]] = ep->mm.vfac[i];
		ep->mm.vedg[renum[i]] = ep->mm.vedg[i];
	}

	// Replace deleted vertices:
	for (i=0; i<ct; i++) {
		ep->mm.v[positions[i]] = undo[i];
		ep->mm.vfac[positions[i]] = undovfac[i];
		ep->mm.vedg[positions[i]] = undovedg[i];
	}

#ifdef EPOLY_RESTORE_DEBUG_PRINT
	DebugPrint ("Restoring CollapseDeadVerts - ending mesh:\n");
	ep->mm.MNDebugPrint ();
#endif

	DbgAssert (ep->mm.CheckAllData ());
	ep->LocalDataChanged (PART_TOPO|PART_SELECT);
}

void CollapseDeadVertsRestore::Redo () {
#ifdef EPOLY_RESTORE_DEBUG_PRINT
	MyDebugPrint ();
	DebugPrint ("Redoing CollapseDeadVerts - starting mesh:\n");
	ep->mm.MNDebugPrint ();
#endif
	int i, j, k, ct = positions.Count();
	if (!ct) return;
	DbgAssert (ep->mm.CheckAllData ());
	int newNumV = ep->mm.numv - ct;
	int oldNumV = ep->mm.numv;

	// Figure out the renumbering scheme for non-deleted faces:
	Tab<int> renum;
	renum.SetCount (ep->mm.numv);
	for (i=0, j=0, k=0; i<ep->mm.numv; i++) {
		if ((j<ct) && (positions[j] == i)) {
			// this face will be deleted.
			renum[i] = -1;
			j++;
			continue;
		}
		renum[i] = k;
		k++;
	}
	DbgAssert (k == newNumV);

	// Correct vertex references in edges, faces:
	DbgAssert (ep->mm.GetFlag (MN_MESH_FILLED_IN));

	for (i=0; i<ep->mm.nume; i++) {
		if (ep->mm.e[i].GetFlag (MN_DEAD)) continue;
		ep->mm.e[i].v1 = renum[ep->mm.e[i].v1];
		ep->mm.e[i].v2 = renum[ep->mm.e[i].v2];
	}

	for (i=0; i<ep->mm.numf; i++) {
		if (ep->mm.f[i].GetFlag (MN_DEAD)) continue;
		for (j=0; j<ep->mm.f[i].deg; j++) ep->mm.f[i].vtx[j] = renum[ep->mm.f[i].vtx[j]];
	}

	// Remove deleted vertices:
	for (i=0; i<oldNumV; i++) {
		if (renum[i] == i) continue;
		if (renum[i] == -1) continue;
		ep->mm.v[renum[i]] = ep->mm.v[i];
		ep->mm.vfac[renum[i]] = ep->mm.vfac[i];
		ep->mm.vedg[renum[i]] = ep->mm.vedg[i];
	}

	ep->mm.setNumVerts (newNumV);

#ifdef EPOLY_RESTORE_DEBUG_PRINT
	DebugPrint ("Redoing CollapseDeadVerts - ending mesh:\n");
	ep->mm.MNDebugPrint ();
#endif

	DbgAssert (ep->mm.CheckAllData ());
	ep->LocalDataChanged (PART_TOPO|PART_SELECT);
}

void CollapseDeadVertsRestore::MyDebugPrint () {
	int i, ct = positions.Count();
	DebugPrint ("CollapseDeadVertsRestore Debug Printout:\n");
	DebugPrint ("  Before: %d vertices.  After: %d vertices.  %d vertices culled.\n", startNum, startNum-ct, ct);

	for (i=0; i<ct; i++) {
		DebugPrint ("%d  ", positions[i]);
		if (i%5==4) DebugPrint ("\n");
	}
	if (i%5!=0) DebugPrint ("\n");
}

CollapseDeadEdgesRestore::CollapseDeadEdgesRestore (EditPolyObject *e) {
	ep = e;
	int allocAmt = ep->mm.nume/10;
	int ct=0;
	for (int i=0; i<ep->mm.nume; i++) {
		if (!ep->mm.e[i].GetFlag (MN_DEAD)) continue;
		positions.Append (1, &i, allocAmt);
		ct++;
	}
	if (!ct) return;
	undo.SetCount (ct);
	for (i=0; i<ct; i++) undo[i] = ep->mm.e[positions[i]];
}

void CollapseDeadEdgesRestore::Restore (int isUndo) {
#ifdef EPOLY_RESTORE_DEBUG_PRINT
	DebugPrint ("Restoring CollapseDeadEdges\n");
#endif
	DbgAssert (ep->mm.CheckAllData ());
	int i, j, k, ct = positions.Count();
	if (!ct) return;
	int newNumE = ep->mm.nume + ct;

	// Figure out the renumbering scheme for non-deleted edges:
	Tab<int> renum;
	renum.SetCount (ep->mm.nume);
	for (i=0, j=0, k=0; i<ep->mm.nume; i++) {
		while ((j<ct) && (positions[j] == k)) {
			// leave room for a deleted edge:
			k++;
			j++;
		}
		renum[i] = k;
		k++;
	}
	DbgAssert (k+(ct-j) == newNumE);

	// Correct edge references in faces, verts:
	DbgAssert (ep->mm.GetFlag (MN_MESH_FILLED_IN));

	for (i=0; i<ep->mm.numf; i++) {
		if (ep->mm.f[i].GetFlag (MN_DEAD)) continue;
		int *ee = ep->mm.f[i].edg;
		for (j=0; j<ep->mm.f[i].deg; j++) ee[j] = renum[ee[j]];
	}

	for (i=0; i<ep->mm.numv; i++) {
		if (ep->mm.v[i].GetFlag (MN_DEAD)) continue;
		for (j=0; j<ep->mm.vedg[i].Count(); j++) ep->mm.vedg[i][j] = renum[ep->mm.vedg[i][j]];
	}

	// Insert room for deleted edges:
	int oldNumE = ep->mm.nume;
	ep->mm.setNumEdges (newNumE);
	for (i=oldNumE-1; i>=0; i--) {
		if (renum[i]==i) continue;
		ep->mm.e[renum[i]] = ep->mm.e[i];
	}

	// Replace deleted edges:
	for (i=0; i<ct; i++) ep->mm.e[positions[i]] = undo[i];

	DbgAssert (ep->mm.CheckAllData ());
	ep->LocalDataChanged (PART_TOPO|PART_SELECT);
}

void CollapseDeadEdgesRestore::Redo () {
	int i, j, k, ct = positions.Count();
	if (!ct) return;
	DbgAssert (ep->mm.CheckAllData ());
	int newNumE = ep->mm.nume - ct;
	int oldNumE = ep->mm.nume;

	// Figure out the renumbering scheme for non-deleted faces:
	Tab<int> renum;
	renum.SetCount (ep->mm.nume);
	for (i=0, j=0, k=0; i<ep->mm.nume; i++) {
		if ((j<ct) && (positions[j] == i)) {
			// this face will be deleted.
			renum[i] = -1;
			j++;
			continue;
		}
		renum[i] = k;
		k++;
	}
	DbgAssert (k == newNumE);

	// Correct edge references in faces, verts:
	DbgAssert (ep->mm.GetFlag (MN_MESH_FILLED_IN));

	for (i=0; i<ep->mm.numf; i++) {
		if (ep->mm.f[i].GetFlag (MN_DEAD)) continue;
		int *ee = ep->mm.f[i].edg;
		for (j=0; j<ep->mm.f[i].deg; j++) ee[j] = renum[ee[j]];
	}

	for (i=0; i<ep->mm.numv; i++) {
		if (ep->mm.v[i].GetFlag (MN_DEAD)) continue;
		for (j=0; j<ep->mm.vedg[i].Count(); j++) ep->mm.vedg[i][j] = renum[ep->mm.vedg[i][j]];
	}

	// Remove deleted faces and map faces:
	for (i=0; i<oldNumE; i++) {
		if (renum[i] == i) continue;
		if (renum[i] == -1) continue;
		ep->mm.e[renum[i]] = ep->mm.e[i];
	}

	ep->mm.setNumEdges (newNumE);
	DbgAssert (ep->mm.CheckAllData ());
	ep->LocalDataChanged (PART_TOPO|PART_SELECT);
}

CollapseDeadFacesRestore::CollapseDeadFacesRestore (EditPolyObject *e) {
	ep = e;
	int allocAmt = ep->mm.numf/10;
	int ct=0;
	for (int i=0; i<ep->mm.numf; i++) {
		if (!ep->mm.f[i].GetFlag (MN_DEAD)) continue;
		positions.Append (1, &i, allocAmt);
		MNFace mntemp;
		undo.Append (1, &mntemp, allocAmt);
		undo[ct] = ep->mm.f[i];
		ct++;
	}
	if (!ct) return;
	undoMap.SetCount (ep->mm.numm+NUM_HIDDENMAPS);
	for (int mp=-NUM_HIDDENMAPS; mp<ep->mm.numm; mp++) {
		int nmp = mp + NUM_HIDDENMAPS;
		undoMap[nmp] = NULL;
		if (ep->mm.M(mp)->GetFlag (MN_DEAD)) continue;
		undoMap[nmp] = new MNMapFace[ct];
		for (i=0; i<ct; i++) {
			undoMap[nmp][i].Init();
			undoMap[nmp][i] = ep->mm.M(mp)->f[positions[i]];
		}
	}
}

CollapseDeadFacesRestore::~CollapseDeadFacesRestore () {
	for (int mp=0; mp<undoMap.Count(); mp++) {
		if (!undoMap[mp]) continue;
		for (int i=0; i<undo.Count(); i++) {
			undoMap[mp][i].Clear();
		}
		delete [] undoMap[mp];
	}
}

void CollapseDeadFacesRestore::Restore (int isUndo) {
#ifdef EPOLY_RESTORE_DEBUG_PRINT
	DebugPrint ("Restoring CollapseDeadFaces\n");
#endif
	int i, j, k, ct = positions.Count();
	if (!ct) return;
	int newNumF = ep->mm.numf + ct;

	// Figure out the renumbering scheme for non-deleted faces:
	Tab<int> renum;
	renum.SetCount (ep->mm.numf);
	for (i=0, j=0, k=0; i<ep->mm.numf; i++) {
		while ((j<ct) && (positions[j] == k)) {
			// leave room for a deleted face:
			k++;
			j++;
		}
		renum[i] = k;
		k++;
	}
	DbgAssert (k+(ct-j) == newNumF);

	// Correct face references in edges, verts:
	if (ep->mm.GetFlag (MN_MESH_FILLED_IN)) {
		for (i=0; i<ep->mm.nume; i++) {
			if (ep->mm.e[i].GetFlag (MN_DEAD)) continue;
			ep->mm.e[i].f1 = renum[ep->mm.e[i].f1];
			if (ep->mm.e[i].f2>-1) ep->mm.e[i].f2 = renum[ep->mm.e[i].f2];
		}

		for (i=0; i<ep->mm.numv; i++) {
			if (ep->mm.v[i].GetFlag (MN_DEAD)) continue;
			for (j=0; j<ep->mm.vfac[i].Count(); j++) ep->mm.vfac[i][j] = renum[ep->mm.vfac[i][j]];
		}
	}

	// Insert room for deleted faces and map faces:
	int mp, oldNumF = ep->mm.numf;
	ep->mm.setNumFaces (newNumF);
	for (mp=-NUM_HIDDENMAPS; mp<ep->mm.numm; mp++) {
		if (ep->mm.M(mp)->GetFlag (MN_DEAD)) continue;
		ep->mm.M(mp)->setNumFaces (newNumF);
	}
	for (i=oldNumF-1; i>=0; i--) {
		if (renum[i]==i) continue;
		ep->mm.f[renum[i]] = ep->mm.f[i];
		ep->mm.f[i].Clear ();
		for (mp=-NUM_HIDDENMAPS; mp<ep->mm.numm; mp++) {
			if (ep->mm.M(mp)->GetFlag (MN_DEAD)) continue;
			ep->mm.M(mp)->f[renum[i]].Clear ();
			memcpy (ep->mm.M(mp)->F(renum[i]), ep->mm.M(mp)->F(i), sizeof(MNMapFace));
			ep->mm.M(mp)->f[i].Init();
		}
	}

	// Replace deleted faces, map faces:
	for (i=0; i<ct; i++) {
		ep->mm.f[positions[i]] = undo[i];
	}
	for (mp=-NUM_HIDDENMAPS; mp<ep->mm.numm; mp++) {
		if (ep->mm.M(mp)->GetFlag (MN_DEAD)) continue;
		int nmp = mp + NUM_HIDDENMAPS;
		if (undoMap[nmp] == NULL) continue;
		for (i=0; i<ct; i++) {
			ep->mm.M(mp)->f[positions[i]] = undoMap[nmp][i];
		}
	}

	ep->LocalDataChanged (PART_TOPO|PART_SELECT);
}

void CollapseDeadFacesRestore::Redo () {
	int i, j, k, ct = positions.Count();
	if (!ct) return;
	int newNumF = ep->mm.numf - ct;
	int mp, oldNumF = ep->mm.numf;

	// Figure out the renumbering scheme for non-deleted faces:
	Tab<int> renum;
	renum.SetCount (ep->mm.numf);
	for (i=0, j=0, k=0; i<ep->mm.numf; i++) {
		if ((j<ct) && (positions[j] == i)) {
			// this face will be deleted.
			renum[i] = -1;
			j++;
			continue;
		}
		renum[i] = k;
		k++;
	}
	DbgAssert (k == newNumF);

	// Correct face references in edges, verts:
	if (ep->mm.GetFlag (MN_MESH_FILLED_IN)) {
		for (i=0; i<ep->mm.nume; i++) {
			if (ep->mm.e[i].GetFlag (MN_DEAD)) continue;
			ep->mm.e[i].f1 = renum[ep->mm.e[i].f1];
			if (ep->mm.e[i].f2>-1) ep->mm.e[i].f2 = renum[ep->mm.e[i].f2];
		}

		for (i=0; i<ep->mm.numv; i++) {
			if (ep->mm.v[i].GetFlag (MN_DEAD)) continue;
			for (j=0; j<ep->mm.vfac[i].Count(); j++) ep->mm.vfac[i][j] = renum[ep->mm.vfac[i][j]];
		}
	}

	// Remove deleted faces and map faces:
	for (i=0; i<oldNumF; i++) {
		if (renum[i] == i) continue;
		if (renum[i] == -1) continue;
		ep->mm.f[renum[i]] = ep->mm.f[i];
		ep->mm.f[i].Clear ();
		for (mp=-NUM_HIDDENMAPS; mp<ep->mm.numm; mp++) {
			if (ep->mm.M(mp)->GetFlag (MN_DEAD)) continue;
			ep->mm.M(mp)->f[renum[i]].Clear ();
			memcpy (ep->mm.M(mp)->F(renum[i]), ep->mm.M(mp)->F(i), sizeof(MNMapFace));
			ep->mm.M(mp)->f[i].Init();
		}
	}

	ep->mm.setNumFaces (newNumF);
	for (mp=-NUM_HIDDENMAPS; mp<ep->mm.numm; mp++) {
		if (ep->mm.M(mp)->GetFlag (MN_DEAD)) continue;
		ep->mm.M(mp)->setNumFaces (newNumF);
	}
	ep->LocalDataChanged (PART_TOPO|PART_SELECT);
}

PerDataRestore::PerDataRestore (EditPolyObject *eo, int mslevel, int chan) {
	ep = eo;
	msl = mslevel;
	channel = chan;
	float *vd=NULL;
	switch (msl) {
	case MNM_SL_VERTEX:
		vd = ep->mm.vertexFloat (chan);
		num = ep->mm.numv;
		break;
	case MNM_SL_EDGE:
		vd = ep->mm.edgeFloat (chan);
		num = ep->mm.nume;
		break;
	}
	if (!vd) return;
	oldData.SetCount (num);
	memcpy (oldData.Addr(0), vd, num * sizeof(float));
}

void PerDataRestore::After () {
	if (!ep) return;
	float *vd=NULL;
	switch (msl) {
	case MNM_SL_VERTEX: vd = ep->mm.vertexFloat (channel); break;
	case MNM_SL_EDGE: vd = ep->mm.edgeFloat (channel); break;
	}
	// Create redo data.
	newData.SetCount (num);
	memcpy (newData.Addr(0), vd, num*sizeof(float));
}

void PerDataRestore::Restore (int isUndo) {
	if (!ep) return;
	float *vd=NULL;
	switch (msl) {
	case MNM_SL_VERTEX: vd = ep->mm.vertexFloat (channel); break;
	case MNM_SL_EDGE: vd = ep->mm.edgeFloat (channel); break;
	}
	if (vd && !newData.Count()) After ();

	if (oldData.Count()) {
		if (!vd) {
			if (msl==MNM_SL_VERTEX) {
				ep->mm.setVDataSupport (channel, true);
				vd = ep->mm.vertexFloat (channel);
			} else {
				ep->mm.setEDataSupport (channel, true);
				vd = ep->mm.edgeFloat (channel);
			}
			if (!vd) return;
		}
		memcpy (vd, oldData.Addr(0), num * sizeof(float));
	} else {
		if (msl == MNM_SL_VERTEX) {
			ep->mm.setVDataSupport (channel, false);
		} else {
			ep->mm.setEDataSupport (channel, false);
		}
	}

	if (ep->ip) ep->UpdatePerDataDisplay (ep->ip->GetTime(), msl, channel);
	ep->LocalDataChanged (PART_GEOM);
}

void PerDataRestore::Redo () {
	if (!ep) return;
	if (!newData.Count()) {
		switch (msl) {
		case MNM_SL_VERTEX: ep->mm.setVDataSupport (channel, false);
		case MNM_SL_EDGE: ep->mm.setEDataSupport (channel, false);
		}
		return;
	}

	float *vd=NULL;
	switch (msl) {
	case MNM_SL_VERTEX: vd = ep->mm.vertexFloat (channel); break;
	case MNM_SL_EDGE: vd = ep->mm.edgeFloat (channel); break;
	}
	if (!vd) {
		switch (msl) {
		case MNM_SL_VERTEX:
			ep->mm.setVDataSupport (channel, true);
			vd = ep->mm.vertexFloat (channel);
			break;
		case MNM_SL_EDGE:
			ep->mm.setEDataSupport (channel, true);
			vd = ep->mm.edgeFloat (channel);
		}
	}
	if (!vd) return;

	memcpy (vd, newData.Addr(0), num * sizeof(float));

	if (ep->ip) ep->UpdatePerDataDisplay (ep->ip->GetTime(), msl, channel);
	ep->LocalDataChanged (PART_GEOM);
}

}	// end namespace EditPoly
