#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <iostream>
#include <fstream.h>

#include "Utility.h"
#include "VirtualFile.h"

#define	vVERSION_STR	"1.0"
#define vEXE_NAME		"LibMaker"

struct SFileInfo
{
	int		fileSize;
	uint32	fileName;
	uint32	extName;

	char*	pDataBuf;
	int		patchAddress;
	int		patchOffset;
	char	actualFileName[_MAX_PATH];
};

const int MAX_FILES = 1024;
SFileInfo g_file_list[MAX_FILES];
int g_num_files = 0;

void	print_usage()
{
#ifdef _DEBUG
	printf( "%s v%s (%s)\n", vEXE_NAME, vVERSION_STR, "Debug" );
#else
	printf( "%s v%s (%s)\n", vEXE_NAME, vVERSION_STR, "Release" );
#endif
	printf( "Neversoft Entertainment, 2002\n" );
	printf( "\nUsage: LibMaker -b<batchfile> -f<filename> -p[p | g | x] -o\n" );
	printf( "Where p = PS2, g = GameCube, x = Xbox\n" );	
	printf( "Ex: LibMaker -fdata\\anims\\test\\test.ska -pp\n" );
	printf( "    LibMaker -bc:\\skate5\\tools\\src\\libmaker\\sample.txt -pp\n" );
	printf( "Note:  Files in batch list should omit the initial 'c:\\skate5\\data')\n" );
}

void split_file_ext( const char* pSrcBuf, char* pFileBuf, char* pExtBuf )
{
	char temp_buf[_MAX_PATH];
	strcpy( pFileBuf, pSrcBuf );
	strcpy( pExtBuf, "" );
	
	for ( int i = strlen( pFileBuf )-1; i >= 0; i-- )
	{
		if ( pFileBuf[i] == '.' )
		{
			strcpy( pExtBuf, &pFileBuf[i+1] );
			pFileBuf[i] = NULL;
			break;
		}
	}

	printf( "splitting '%s' into '%s' and '%s'\n", pSrcBuf, pFileBuf, pExtBuf );
}

bool output_files( const char* pDstName, int platform )
{
	fstream outstream( pDstName, ios::out | ios::binary );
	
	int versionNum = 1;
	outstream.write( (const char*)&versionNum, sizeof(int) );

	outstream.write( (const char*)&g_num_files, sizeof(int) );

	int i;

	for ( i = 0; i < g_num_files; i++ )
	{
		// FILE OFFSET
		// FILE SIZE
		// CHECKSUM OF FILENAME, MINUS EXTENSION
		// CHECKSUM OF EXTENSION

		SFileInfo* pFileInfo = &g_file_list[i];
		pFileInfo->patchAddress = outstream.tellp();
		int dummyOffset = 0;
		outstream.write( (const char*)&dummyOffset, sizeof( int ) );
		outstream.write( (const char*)&pFileInfo->fileSize, sizeof( int ) );
		outstream.write( (const char*)&pFileInfo->fileName, sizeof( int ) );
		outstream.write( (const char*)&pFileInfo->extName, sizeof( int ) );
	}

	for ( i = 0; i < g_num_files; i++ )
	{
		SFileInfo* pFileInfo = &g_file_list[i];
		pFileInfo->patchOffset = outstream.tellp();

		// write out file data
		printf( "Writing out %d bytes for file %d: %s\n", pFileInfo->fileSize, i, pFileInfo->actualFileName );
		outstream.write( (const char*)pFileInfo->pDataBuf, pFileInfo->fileSize ); 
	
		// pad it, if necessary
		if ( outstream.tellp() & 0x3 )
		{
			char dummy;
			int pad_size = outstream.tellp() & 0x3;
			printf( "Writing out %d bytes for padding\n", pad_size );
			outstream.write( (const char*)&dummy, 4 - pad_size );
		}
	}
	
	streampos end_pos = outstream.tellp();

	// patch the headers
	for ( i = 0; i < g_num_files; i++ )
	{
		SFileInfo* pFileInfo = &g_file_list[i];
		outstream.seekp( pFileInfo->patchAddress, ios::beg );
		outstream.write( (const char*)&pFileInfo->patchOffset, sizeof( int ) );
	}

	outstream.seekp( end_pos, ios::beg );

	outstream.close();

	return true;
}

void add_file( const char* fileName )
{
	char pFileName[_MAX_PATH];
	strcpy( pFileName, fileName );
	for ( int i = strlen( pFileName )-1; i >= 0; i-- )
	{
		if ( pFileName[i] == ' ' )
		{
			pFileName[i] = 0;
		}
		else
		{
			break;
		}
	}

	char* proj_path = getenv( "PROJ_ROOT" );
	if( proj_path == NULL )
	{		
		printf( "You must first define your PROJ_ROOT environment variable\n" );
	}

	char pFullPathName[_MAX_PATH];
	if ( proj_path[strlen(proj_path)-1] == '\\'
		|| proj_path[strlen(proj_path)-1] == '/' )
	{
		sprintf( pFullPathName, "%sdata\\", proj_path );	
	}
	else
	{
		sprintf( pFullPathName, "%s\\data\\", proj_path );		
	}

	strcat( pFullPathName, pFileName );

	fstream instream( pFullPathName, ios::in | ios::nocreate | ios::binary );
	if ( !instream.is_open() )
	{
		printf( "Couldn't open %s\n", pFullPathName );
		exit(1);
	}

	if ( g_num_files > MAX_FILES )
	{
		printf( "Couldn't add %s.  Too many files in list!\n", pFullPathName );
		instream.close();
		exit(1);
	}

	printf( "Adding %s to LIB\n", pFullPathName );
		
	SFileInfo* pFileInfo = &g_file_list[g_num_files];	

	char file_name[_MAX_PATH];
	char ext_name[_MAX_PATH];
	split_file_ext( pFileName, file_name, ext_name );
	pFileInfo->fileName = Crc::GenerateCRCFromString( file_name );
	pFileInfo->extName = Crc::GenerateCRCFromString( ext_name );
	strcpy( pFileInfo->actualFileName, pFileName );

	instream.seekp( 0, ios::end );
	pFileInfo->fileSize = instream.tellp();
	instream.seekp( 0, ios::beg );
	pFileInfo->pDataBuf = new char[pFileInfo->fileSize];
	instream.read( (char*)pFileInfo->pDataBuf, pFileInfo->fileSize );
	instream.close();
	pFileInfo->patchAddress = 0xDEADDEAD;
	pFileInfo->patchOffset = 0xFACEFACE;
	g_num_files++;
}

void delete_files()
{
	for ( int i = 0; i < g_num_files; i++ )
	{
		SFileInfo* pFileInfo = &g_file_list[i];
		delete[] pFileInfo->pDataBuf;
		pFileInfo->pDataBuf = NULL;
	}

	g_num_files = 0;
}

char* get_dest_name( char* pSrcName )
{
	static char dest_name[_MAX_PATH];
	char dummy_ext[_MAX_PATH];
	strcpy( dest_name, pSrcName );

	split_file_ext( pSrcName, dest_name, dummy_ext ); 
	
	printf( "dest_name = %s\n", dest_name );

	int idx = strlen(dest_name);
	dest_name[idx++] = '.';
	dest_name[idx++] = 'L';
	dest_name[idx++] = 'I';
	dest_name[idx++] = 'B';
	dest_name[idx++] = NULL;

	printf( "dest_name = %s\n", dest_name );

	return dest_name;
}

void	main( int argc, char* argv[] )
{
	char* pSrcName = NULL;
	char pDstName[_MAX_PATH];
	char* pBatchFileName = NULL;
	int platform = Utils::vPLATFORM_NONE;

	char* pArg = NULL;

	if ( pArg = Utils::TestArg( argc, argv, '?' ) )
	{
		print_usage();
		goto failure;
	}

	if ( pArg = Utils::TestArg( argc, argv, 'P' ) )
	{
		switch ( *pArg )
		{
			case 'P':
			case 'p':
				platform = Utils::vPLATFORM_PS2;
				break;
			case 'G':
			case 'g':
				platform = Utils::vPLATFORM_NGC;
				break;
			case 'X':
			case 'x':
				platform = Utils::vPLATFORM_XBOX;
				break;
		}
	}
	
	if ( platform == Utils::vPLATFORM_NONE )
	{
		print_usage();
		goto failure;
	}

	if ( pArg = Utils::TestArg( argc, argv, 'B' ) )
	{
		pBatchFileName = pArg;

		fstream batchStream( pBatchFileName, ios::in | ios::nocreate );
		if ( !batchStream.is_open() )
		{
			printf( "Error!  Could not open %s\n", pBatchFileName );
			batchStream.close();
			goto failure;
		}

		while ( !batchStream.eof() )
		{
			char lineBuf[256];
			batchStream.getline( &lineBuf[0], 256 );
			if ( batchStream.eof() )
			{
				break;
			}

//			run_utility( lineBuf, platform );
			add_file( lineBuf );
		}

		batchStream.close();
		strcpy( pDstName, get_dest_name( pBatchFileName ) );
	}
	else if ( pArg = Utils::TestArg( argc, argv, 'F' ) )
	{
		pSrcName = pArg;
//		run_utility( pSrcName, platform );
		add_file( pSrcName );
		strcpy( pDstName, get_dest_name( pSrcName ) );		
	}

	if( ( pSrcName == NULL && pBatchFileName == NULL ))
	{	
		print_usage();
		goto failure;
	}

	if ( !output_files( pDstName, platform ) )
	{
		printf( "Output failed!\n" );
		goto failure;
	}

success:
	delete_files();
	printf( "Success!\n" );
	return;

failure:
	delete_files();
	printf( "Failure!\n" );
	exit(1);
}