//****************************************************************************
//* MODULE:         Tools/LibConv
//* FILENAME:       libconv.cpp
//* OWNER:          Gary Jesdanun
//* CREATION DATE:  1/15/2003
//****************************************************************************

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

// Includes
#include <core\defines.h>

#include <io.h>
#include "stdlib.h"
#include "stdarg.h"
#include "..\genlib\utility.h"

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

// Defines

// If the lib format changes, this will need to be updated to match that
// used by the exporter, otherwise libconv.exe will assert.
#define LIB_VERSION 1

struct SLibHeader
{
	uint32 mVersionNumber;
	uint32 mNumFiles;
};

struct SLibFileInfo
{
	uint32 mOffset;
	uint32 mFileSize; // This is the original size, not padded to a multiple of 4
	uint32 mFileNameChecksum;
	uint32 mExtensionChecksum;
};

#define MAX_OUTPUT_EXTENSION_LENGTH 10
#define MAX_OUTPUT_EXTENSIONS 10
#define MAX_COMMAND_LENGTH 500
#define MAX_SOURCE_EXTENSION_LENGTH 20
struct SRule
{
	// The checksum of the source file extension, ie "ska"
	uint32 mSourceExtension;
	char mpSourceExtension[MAX_SOURCE_EXTENSION_LENGTH];

	int mNumOutputExtensions;
	char mpOutputExtensions[MAX_OUTPUT_EXTENSION_LENGTH*MAX_OUTPUT_EXTENSIONS];
	char mpCommand[MAX_COMMAND_LENGTH];
};

#define MAX_RULES 100
#define MAX_FILES_IN_LIB 5000

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

// Globals
int gDebugMode=0;
int gPlatform = Utils::vPLATFORM_NONE;
SRule gpRules[MAX_RULES];
int gNumRules=0;

SLibFileInfo gpOutputLibFileInfo[MAX_FILES_IN_LIB];
uint8 *gppOutputLibFiles[MAX_FILES_IN_LIB];
int gNumFilesInOutputLib=0;

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

// Prototypes
void CleanUp();
#define APPEND_ZERO true
#define DONT_APPEND_ZERO false
#define DONT_ASSERT false
uint8 *LoadFile(const char *p_fileName, uint32 *p_size, bool append_zero=false, bool assert=true);
void LoadAndParseIniFile(const char *p_ini_file_name);
void ProcessLibFile(const char *p_file_name);
void ProcessBunchOfLibFiles(const char *p_list_file_name);
SRule *FindRuleForGivenExtension(uint32 extensionChecksum);
void CreateTempFile(uint8 *p_libFile, SLibFileInfo *p_fileInfo, SRule *p_rule);
void WriteOutputLib(const char *p_inputLibName);
const char *GenerateTempFileName(SLibFileInfo *p_fileInfo, SRule *p_rule);
const char *ResolvePercent(const char *p_percent, const char *p_tempFileName=NULL);
const char *ResolvePercents(const char *p_source, const char *p_tempFileName=NULL);
void ExecuteCommand(SLibFileInfo *p_fileInfo, SRule *p_rule);
void LoadFileForInclusionInOutputLib(SLibFileInfo *p_fileInfo, SRule *p_rule, uint32 extension_index);
char *SkipWhiteSpace(char *p_ch);
char *CopyStringUntilEndOfLine(char *p_source, char *p_dest, int destSpaceLeft);

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void CleanUp()
{
	for (int i=0; i<gNumFilesInOutputLib; ++i)
	{
		free(gppOutputLibFiles[i]);
		gppOutputLibFiles[i]=NULL;
	}
	gNumFilesInOutputLib=0;

	if ( gDebugMode )
	{
		printf("@@@ Debug mode:  Not deleting temporary files\n");
	}
	else
	{
		printf("@@@ Deleting temporary files\n");
		system("del lctemp0x*");
	}
	printf("@@@ Done\n");
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

class LibConverter
{
public:
	virtual ~LibConverter();
};

LibConverter::~LibConverter()
{
	CleanUp();
}

// guarantees that the files will be cleaned up
// if the program asserts...
static LibConverter s_lib_converter;

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void SwapEndianness(uint32 *p_data, int numWords)
{
	for (int i=0; i<numWords; ++i)
	{
		uint32 a=p_data[i];
		a=(((a>>0)&0xff)<<24) |
		  (((a>>8)&0xff)<<16) |
		  (((a>>16)&0xff)<<8) |
		  (((a>>24)&0xff)<<0);
		p_data[i]=a;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

uint8 *LoadFile(const char *p_fileName, uint32 *p_size, bool append_zero, bool assert)
{
	FILE *ifp=fopen(p_fileName,"rb");
	if (ifp==NULL)
	{
		if (assert)
		{
			printf("File = '%s'\n",p_fileName);
			Utils::Assert(0,"Could not open source file");
		}
		else
		{
			*p_size=0;
			return NULL;
		}
	}

	fseek(ifp,0,SEEK_END);
	long file_size=ftell(ifp);
	fseek(ifp,0,SEEK_SET);
	uint8 *p_buffer=(uint8*)malloc(append_zero ? file_size+1:file_size);
	Utils::Assert(p_buffer!=NULL,"Could not allocate memory to hold input file");
	Utils::Assert(fread(p_buffer,file_size,1,ifp)==1,"fread failed");
	fclose(ifp);
	ifp=NULL;

	if (append_zero)
	{
		p_buffer[file_size]=0;
	}

	*p_size=file_size;
	return p_buffer;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

char *SkipWhiteSpace(char *p_ch)
{
	while (	*p_ch && (*p_ch==' ' || *p_ch=='\t' || *p_ch==0x0d || *p_ch==0x0a))
	{
		++p_ch;
	}
	return p_ch;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

char *SkipSpacesAndTabs(char *p_ch)
{
	while (	*p_ch && (*p_ch==' ' || *p_ch=='\t'))
	{
		++p_ch;
	}
	return p_ch;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

char *CopyStringUntilWhiteSpaceOrCloseBracket(char *p_source, char *p_dest, int destSpaceLeft)
{
	while (	*p_source!=0 &&
			*p_source!=' ' && 
			*p_source!='\t' && 
			*p_source!=0x0d &&
			*p_source!=0x0a &&
			*p_source!=']')
	{
		Utils::Assert(destSpaceLeft,"String in [] too long for buffer");
		*p_dest++=*p_source++;
		--destSpaceLeft;
	}
	Utils::Assert(destSpaceLeft,"String in [] too long for buffer");
	*p_dest=0;

	return p_source;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

char *CopyStringUntilEndOfLine(char *p_source, char *p_dest, int destSpaceLeft)
{
	while (	*p_source!=0 &&
			*p_source!=0x0d &&
			*p_source!=0x0a )
	{
		Utils::Assert(destSpaceLeft,"String in [] too long for buffer");
		*p_dest++=*p_source++;
		--destSpaceLeft;
	}
	Utils::Assert(destSpaceLeft,"String in [] too long for buffer");
	*p_dest=0;

	return p_source;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

// Loads the ini file and parses it, translating the info contained in it into an
// array of SRule's.
void LoadAndParseIniFile(const char *p_ini_file_name)
{
	printf("@@@ Parsing ini file '%s'\n",p_ini_file_name);

	uint8 *p_ini_file=NULL;
	uint32 ini_file_size=0;

	// Load the file into memory, appending a zero so that the
	// end can be detected easily whilst parsing.
	p_ini_file=LoadFile(p_ini_file_name,&ini_file_size,APPEND_ZERO);
	// ini_file_size is not actually used beyond this point.

	char *p_ch=(char*)p_ini_file;
	p_ch=SkipWhiteSpace(p_ch);
	while (*p_ch)
	{
		Utils::Assert(gNumRules<MAX_RULES,"Too many rules defined in libconv.ini");
		SRule *p_new_rule=&gpRules[gNumRules++];
		p_new_rule->mpSourceExtension[0]=0;
		p_new_rule->mSourceExtension=0;
		p_new_rule->mNumOutputExtensions=0;
		p_new_rule->mpCommand[0]=0;

		// WS=whitespace, ie, spaces, tabs or end-of-lines
		// First, expect [, WS, source extension, WS, ]

		Utils::Assert(*p_ch=='[',"Expected '['");
		++p_ch;
		p_ch=SkipWhiteSpace(p_ch);
		p_ch=CopyStringUntilWhiteSpaceOrCloseBracket(p_ch,p_new_rule->mpSourceExtension,MAX_SOURCE_EXTENSION_LENGTH);
		p_ch=SkipWhiteSpace(p_ch);
		Utils::Assert(*p_ch==']',"Expected ']'");
		++p_ch;

		// Store the checksum of the source extension.
		p_new_rule->mSourceExtension=Crc::GenerateCRCFromString(p_new_rule->mpSourceExtension);


		// Now, on the same line as the last ], expect a list of output extensions.
		// ST=spaces or tabs, but not end-of-lines
		// So we're expecting:
		// ST, [, WS, output extension, WS, ], ST, [, WS, output extension, WS, ], etc...
		// until an end-of-line is reached, which terminates the list.
		p_ch=SkipSpacesAndTabs(p_ch);
		while (*p_ch && *p_ch!=0x0d && *p_ch!=0x0a) // while not end-of-line
		{
			Utils::Assert(p_new_rule->mNumOutputExtensions<MAX_OUTPUT_EXTENSIONS,"Too many output extensions");
			Utils::Assert(*p_ch=='[',"Expected '['");
			++p_ch;
			p_ch=SkipWhiteSpace(p_ch);
			p_ch=CopyStringUntilWhiteSpaceOrCloseBracket(p_ch,
				p_new_rule->mpOutputExtensions+MAX_OUTPUT_EXTENSION_LENGTH*p_new_rule->mNumOutputExtensions,
				MAX_OUTPUT_EXTENSION_LENGTH);

			p_ch=SkipWhiteSpace(p_ch);
			Utils::Assert(*p_ch==']',"Expected ']'");
			++p_ch;
			++p_new_rule->mNumOutputExtensions;

			p_ch=SkipSpacesAndTabs(p_ch);
		}
		// Skip over the end-of-line, and any white space at the start of the next line.
		p_ch=SkipWhiteSpace(p_ch);

		// Now a rule may be defined, so if so, load the string into mpCommand.
		if (*p_ch)
		{
			if (*p_ch=='[')
			{
				// We've hit the start of the next source extension declaration,
				// so no rule was defined, so leave mpCommand containing the empty string.
			}
			else
			{
				// Read in the command, then skip to the next source 
				// extension declaration.
				p_ch=CopyStringUntilEndOfLine(p_ch,p_new_rule->mpCommand,MAX_COMMAND_LENGTH);
				p_ch=SkipWhiteSpace(p_ch);
				// If *p_ch is not zero then it should be [ at this point,
				// and the top of the loop will check this.
			}
		}
	}

	// Finished, so the ini file is no longer needed in memory.
	free(p_ini_file);
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

SRule *FindRuleForGivenExtension(uint32 extensionChecksum)
{
	SRule *p_rule=gpRules;

	for (int i=0; i<gNumRules; ++i)
	{
		if (p_rule->mSourceExtension==extensionChecksum)
		{
			return p_rule;
		}
		++p_rule;
	}

	return NULL;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void DebugDisplayRule(SRule *p_rule)
{
	printf("Source extension = 0x%08x\n",p_rule->mSourceExtension);
	for (int i=0; i<p_rule->mNumOutputExtensions; ++i)
	{
		printf("Output extension = '%s'\n",p_rule->mpOutputExtensions+MAX_OUTPUT_EXTENSION_LENGTH*i);
	}
	printf("Command='%s'\n\n",p_rule->mpCommand);
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void DebugDisplayRules()
{
	for (int i=0; i<gNumRules; ++i)
	{
		DebugDisplayRule(&gpRules[i]);
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

const char *GenerateTempFileName(SLibFileInfo *p_fileInfo, SRule *p_rule)
{
	static char sp_name[1024];
	sprintf(sp_name,"lctemp0x%08x.%s",p_fileInfo->mFileNameChecksum,p_rule->mpSourceExtension);
	return sp_name;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void CreateTempFile(uint8 *p_libFile, SLibFileInfo *p_fileInfo, SRule *p_rule)
{
	const char *p_file_name=GenerateTempFileName(p_fileInfo,p_rule);

	if ( gDebugMode )
	{
		printf("@@@ Creating temp file '%s', size %d\n",p_file_name,p_fileInfo->mFileSize);
	}

	Utils::Assert((p_fileInfo->mOffset&3)==0,"Bad file offset, not a multiple of 4");

	FILE *ofp=fopen(p_file_name,"wb");
	Utils::Assert(ofp!=NULL,"\nCould not open temporary file for writing.");
	Utils::Assert(fwrite(p_libFile+p_fileInfo->mOffset,p_fileInfo->mFileSize,1,ofp)==1,"fwrite failed");
	fclose(ofp);
	ofp=NULL;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

const char *ResolvePercent(const char *p_percent, const char *p_tempFileName)
{
	Utils::Assert(*p_percent=='%',"Expected p_percent to point to a '%'");
	++p_percent;

	static char sp_buf[1024];
	sp_buf[0]=0;

	switch (*p_percent)
	{
	case 'f':
		if (p_tempFileName)
		{
			strcpy(sp_buf,p_tempFileName);
		}
		break;
	case 'p':
		switch (gPlatform)
		{
		case Utils::vPLATFORM_XBOX:
			strcpy(sp_buf,"-px");
			break;
		case Utils::vPLATFORM_NGC:
			strcpy(sp_buf,"-pg");
			break;
		case Utils::vPLATFORM_PS2:
			strcpy(sp_buf,"-pp");
			break;
		default:
			break;
		}
		break;
	case 'e':
		strcat(sp_buf,".");
		strcat(sp_buf,Utils::GetPlatformExt(gPlatform));
		break;
	default:
		break;		
	}

	return sp_buf;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

const char *ResolvePercents(const char *p_source, const char *p_tempFileName)
{
	static char sp_resolved_string[1024];
	char *p_dest=sp_resolved_string;

	while (*p_source)
	{
		if (*p_source=='%')
		{
			const char *p_replacement_text=ResolvePercent(p_source,p_tempFileName);
			p_source+=2;
			
			while (*p_replacement_text)
			{
				*p_dest++=*p_replacement_text++;
			}
		}
		else
		{
			*p_dest++=*p_source++;
		}
	}
	*p_dest=0;

	return sp_resolved_string;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void ExecuteCommand(SLibFileInfo *p_fileInfo, SRule *p_rule)
{
	const char *p_temp_file_name=GenerateTempFileName(p_fileInfo,p_rule);
	const char *p_command=ResolvePercents(p_rule->mpCommand,p_temp_file_name);

	printf("@@@ Executing '%s'\n",p_command);
	printf("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
	
	if ( system(p_command) != 0 )
	{
		printf( "Libconv failed while running:  %s\n", p_command );
		CleanUp();
		exit(1);
	}
	printf("\n");
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int get_file_info_index( uint32 fileNameChecksum, uint32 extensionChecksum, bool assertOnFail = true )
{
	for ( int i = 0; i < gNumFilesInOutputLib; i++ )
	{
		if ( gpOutputLibFileInfo[i].mFileNameChecksum == fileNameChecksum 
			&& gpOutputLibFileInfo[i].mExtensionChecksum == extensionChecksum )
		{
			return i;
		}
	}

	char msg[256];
	sprintf( msg, "Couldn't find file info 0x%08x 0x%08x", fileNameChecksum, extensionChecksum );
	Utils::Assert( !assertOnFail, msg );
	return -1;
}

void SortOutputLib()
{
	// this rearranges the output lib into the following order:
	// CIF, CAM, OBA, SKA, MDL/TEX, SKIN/TEX/CAS, OTHER
	// which is necessary for the shrinking to work properly
	// (ideally, this would be specified in the INI file
	// somehow, but I don't have time to deal with it right now)
	SLibFileInfo* pTempFileInfo = new SLibFileInfo[gNumFilesInOutputLib];
	uint8** ppTempFile = (uint8**)malloc(gNumFilesInOutputLib*sizeof(uint8*));
	int tempFileCount = 0;

	bool* pDataCopied = new bool[gNumFilesInOutputLib];
	for ( int i = 0; i < gNumFilesInOutputLib; i++ )
	{
		pDataCopied[i] = false;
	}

	/*
	for ( int i = 0; i < gNumFilesInOutputLib; i++ )
	{
		printf( "Found %08x %08x\n", 
			gpOutputLibFileInfo[i].mFileNameChecksum, 
			gpOutputLibFileInfo[i].mExtensionChecksum );
	}
	*/

	for ( int i = 0; i < gNumFilesInOutputLib; i++ )
	{
		if ( pDataCopied[i] )
		{
			continue;
		}

		if ( gpOutputLibFileInfo[i].mFileNameChecksum == 0x00000000 
			&&  gpOutputLibFileInfo[i].mExtensionChecksum != CRCD(0x5ac14717,"CIF") )
		{
			pTempFileInfo[tempFileCount] = gpOutputLibFileInfo[i]; 
			ppTempFile[tempFileCount] = gppOutputLibFiles[i];
			pDataCopied[i] = true;
			tempFileCount++;
		}
	}

	for ( int i = 0; i < gNumFilesInOutputLib; i++ )
	{
		if ( pDataCopied[i] )
		{
			continue;
		}

		if ( gpOutputLibFileInfo[i].mExtensionChecksum == CRCD(0xeab51346,"SKA") )
		{
			pTempFileInfo[tempFileCount] = gpOutputLibFileInfo[i]; 
			ppTempFile[tempFileCount] = gppOutputLibFiles[i];
			pDataCopied[i] = true;
			tempFileCount++;
		}
	}

	for ( int i = 0; i < gNumFilesInOutputLib; i++ )
	{
		if ( pDataCopied[i] )
		{
			continue;
		}

		if ( gpOutputLibFileInfo[i].mExtensionChecksum == CRCD(0x524fd4e,"MDL") 
			||  gpOutputLibFileInfo[i].mExtensionChecksum == CRCD(0xe7308c1f,"GEOM") )
		{
			// find the associated TEX file
			int ref_index = get_file_info_index( gpOutputLibFileInfo[i].mFileNameChecksum, CRCD(0x1512808d,"TEX") );
			pTempFileInfo[tempFileCount] = gpOutputLibFileInfo[ref_index];
			ppTempFile[tempFileCount] = gppOutputLibFiles[ref_index];
			pDataCopied[ref_index] = true;
			tempFileCount++;

			pTempFileInfo[tempFileCount] = gpOutputLibFileInfo[i]; 
			ppTempFile[tempFileCount] = gppOutputLibFiles[i];
			pDataCopied[i] = true;
			tempFileCount++;
		}
	}

	for ( int i = 0; i < gNumFilesInOutputLib; i++ )
	{
		if ( pDataCopied[i] )
		{
			continue;
		}

		if ( gpOutputLibFileInfo[i].mExtensionChecksum == CRCD(0xfd8697e1,"SKIN") )
		{
			// find the associated CAS file, if any
			int ref_index = get_file_info_index( gpOutputLibFileInfo[i].mFileNameChecksum, CRCD(0xffc529f4,"CAS"), false );
			if ( ref_index != -1 )
			{
				pTempFileInfo[tempFileCount] = gpOutputLibFileInfo[ref_index];
				ppTempFile[tempFileCount] = gppOutputLibFiles[ref_index];
				pDataCopied[ref_index] = true;
				tempFileCount++;
			}

			// find the associated WGT file, if any
			ref_index = get_file_info_index( gpOutputLibFileInfo[i].mFileNameChecksum, CRCD(0x2cd4107d,"WGT"), false );
			if ( ref_index != -1 )
			{
				pTempFileInfo[tempFileCount] = gpOutputLibFileInfo[ref_index];
				ppTempFile[tempFileCount] = gppOutputLibFiles[ref_index];
				pDataCopied[ref_index] = true;
				tempFileCount++;
			}

			// find the associated TEX file
			ref_index = get_file_info_index( gpOutputLibFileInfo[i].mFileNameChecksum, CRCD(0x1512808d,"TEX") );
			pTempFileInfo[tempFileCount] = gpOutputLibFileInfo[ref_index];
			ppTempFile[tempFileCount] = gppOutputLibFiles[ref_index];
			pDataCopied[ref_index] = true;
			tempFileCount++;

			pTempFileInfo[tempFileCount] = gpOutputLibFileInfo[i]; 
			ppTempFile[tempFileCount] = gppOutputLibFiles[i];
			pDataCopied[i] = true;
			tempFileCount++;
		}
	}

	// now do the rest of the files
	for ( int i = 0; i < gNumFilesInOutputLib; i++ )
	{
		if ( !pDataCopied[i] )
		{
			pTempFileInfo[tempFileCount] = gpOutputLibFileInfo[i];
			ppTempFile[tempFileCount] = gppOutputLibFiles[i];
			pDataCopied[i] = true;
			tempFileCount++;
		}
	}

	Utils::Assert( tempFileCount == gNumFilesInOutputLib, "Couldn't reorder files, extra files found?" );

	// copy it back
	memcpy( &gpOutputLibFileInfo[0], pTempFileInfo, gNumFilesInOutputLib * sizeof(SLibFileInfo) );
	memcpy( &gppOutputLibFiles[0], ppTempFile, gNumFilesInOutputLib * sizeof(uint8*) );

	/*
	for ( int i = 0; i < gNumFilesInOutputLib; i++ )
	{
		printf( "Found %08x %08x %d %d\n", 
			gpOutputLibFileInfo[i].mFileNameChecksum, 
			gpOutputLibFileInfo[i].mExtensionChecksum,
			gpOutputLibFileInfo[i].mFileSize,
			gpOutputLibFileInfo[i].mOffset );
	}
	*/

	delete[] pTempFileInfo;
	delete[] pDataCopied;
	free(ppTempFile);
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void ProcessLibFile(const char *p_file_name)
{
	printf("@@@ Processing lib file '%s'\n",p_file_name);

	uint32 lib_file_size=0;
	uint32 *p_lib_file=(uint32*)LoadFile(p_file_name,&lib_file_size);

	SLibHeader *p_header=(SLibHeader*)p_lib_file;
	Utils::Assert(p_header->mVersionNumber==LIB_VERSION,"Bad version number in lib file.\nLIB_VERSION in tools\\src\\libconv\\libconv.h may need to be updated.");
	Utils::Assert(p_header->mNumFiles<=MAX_FILES_IN_LIB,"Too many files in lib");

	printf("@@@ %d source files\n\n",p_header->mNumFiles);

	// First, create all the temporary files.
	SLibFileInfo *p_file_info=(SLibFileInfo*)(p_header+1);
	for (uint32 i=0; i<p_header->mNumFiles; ++i)
	{
		// Check for duplicate temporary file names
		SLibFileInfo *p_dup_file_info=(SLibFileInfo*)(p_header+1);
		uint32 thisFileChecksum = p_file_info->mFileNameChecksum+p_file_info->mExtensionChecksum;
		for (uint32 j=0; j<i; ++j)
		{
			SLibFileInfo *p_curr_file_info=(SLibFileInfo*)(p_header+1)+j;
			uint32 otherFileChecksum = p_curr_file_info->mFileNameChecksum+p_curr_file_info->mExtensionChecksum;
			if ( thisFileChecksum == otherFileChecksum )
			{
				Utils::Assert( 0, "Trying to create 2 temporary files with same name (object=%08x)\n", p_file_info->mFileNameChecksum );
			}
		}

		// Find the rule for this file extension
		SRule *p_rule=FindRuleForGivenExtension(p_file_info->mExtensionChecksum);
		if (p_rule)// && p_rule->mpCommand[0])
		{
			// A command is defined, so generate the temporary file
			CreateTempFile((uint8*)p_lib_file,p_file_info,p_rule);
		}

		++p_file_info;
	}
	printf("\n");

	// Now run all the commands
	p_file_info=(SLibFileInfo*)(p_header+1);
	for (uint32 i=0; i<p_header->mNumFiles; ++i)
	{
		// Find the rule for this file extension
		SRule *p_rule=FindRuleForGivenExtension(p_file_info->mExtensionChecksum);
		if (p_rule && p_rule->mpCommand[0])
		{
			// A command is defined, so run it.
			printf("@@@ Running command '%s'\n",p_rule->mpCommand);
			ExecuteCommand(p_file_info,p_rule);
		}

		++p_file_info;
	}

	// Look up which of the newly generated files need to be
	// included in the output lib, and load them into memory.
	p_file_info=(SLibFileInfo*)(p_header+1);

	for (uint32 i=0; i<p_header->mNumFiles; ++i)
	{
		// Find the rule for this file extension
		SRule *p_rule=FindRuleForGivenExtension(p_file_info->mExtensionChecksum);

		if (p_rule)
		{
			for (int e=0; e<p_rule->mNumOutputExtensions; ++e)
			{
				LoadFileForInclusionInOutputLib(p_file_info,p_rule,e);
			}
		}

		++p_file_info;
	}

	// this builds the dependency file
	char BaseName[1024];
	char ExtName[1024];
	Utils::SplitFileExt(p_file_name,&BaseName[0],&ExtName[0]);

	char DepName[1024];
	sprintf(DepName,"%s.dep.%s",BaseName,Utils::GetPlatformExt(gPlatform));

	char systemCmd[256];
	sprintf(systemCmd,"del %s",DepName);
	system(systemCmd);
	sprintf(systemCmd,"copy lctemp0x00000000.dep.%s %s",Utils::GetPlatformExt(gPlatform),DepName);
	system(systemCmd);

	SortOutputLib();

	WriteOutputLib(p_file_name);

	free(p_lib_file);

	// so that globals and temp files are reset
	CleanUp();
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void ProcessBunchOfLibFiles(const char *p_list_file_name)
{
	printf("@@@ Processing bunch of lib files listed in '%s'\n",p_list_file_name);

	uint32 file_size=0;
	uint8 *p_list_file=LoadFile(p_list_file_name,&file_size,APPEND_ZERO);

	char p_lib_file_name[1000];
	char *p_ch=(char*)p_list_file;
	p_ch=SkipWhiteSpace(p_ch);

	while (*p_ch)
	{
		p_ch=CopyStringUntilEndOfLine(p_ch,p_lib_file_name,1000);
		p_ch=SkipWhiteSpace(p_ch);
		ProcessLibFile(p_lib_file_name);
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void LoadFileForInclusionInOutputLib(SLibFileInfo *p_fileInfo, SRule *p_rule, uint32 extension_index)
{
	// Calculate the full file name of the file to be loaded
	char p_file_name[1024];
	Utils::Assert(extension_index<MAX_OUTPUT_EXTENSIONS,"Bad extension index");
	const char *p_unresolved_extension=p_rule->mpOutputExtensions+MAX_OUTPUT_EXTENSION_LENGTH*extension_index;
	const char *p_full_extension=ResolvePercents(p_unresolved_extension);
	// If the unresolved extension had any %e's in it, these will have got resolved to
	// the appropriate extension for the platform.
	// Eg, if the unresolved extension was 'ska%e' the full extension will now be 'ska.ps2'
	// If there were no %e's, the full extension will be the same as the unresolved.
	sprintf(p_file_name,"lctemp0x%08x.%s",p_fileInfo->mFileNameChecksum,p_full_extension);

	if (stricmp(p_full_extension,"lib")==0)
	{
		// This is a special case. If the output file has extension lib, then
		// instead of including it as is, assume that it has the same file format
		// as the lib format that libconv uses, and include each of the files
		// included within it. So in other words it gets expanded.

		printf("@@@ Expanding '%s'\n",p_file_name);
		uint32 file_size=0;
		uint8 *p_lib=LoadFile(p_file_name,&file_size,DONT_APPEND_ZERO);

		SLibHeader *p_header=(SLibHeader*)p_lib;
		if (gPlatform==Utils::vPLATFORM_NGC)
		{
			SwapEndianness((uint32*)p_header,sizeof(SLibHeader)/4);
		}
		Utils::Assert(p_header->mVersionNumber==LIB_VERSION,"Lib version number mismatch, expected %d, got %d",LIB_VERSION,p_header->mVersionNumber);
		int num_files=p_header->mNumFiles;

		SLibFileInfo *p_contained_file_info=(SLibFileInfo*)(p_header+1);
		if (gPlatform==Utils::vPLATFORM_NGC)
		{
			SwapEndianness((uint32*)p_contained_file_info,num_files*(sizeof(SLibFileInfo)/4));
		}

		uint8 *p_contained_file=(uint8*)(p_contained_file_info+num_files);
		for (int i=0; i<num_files; ++i)
		{
			int size=p_contained_file_info->mFileSize;

			// Make a copy of the contained file, and store the pointer
			// in gppOutputLibFiles.
			// Need to make a copy because the memory blocks pointed to by
			// the pointers in gppOutputLibFiles will get freed later.
	
			uint8 *p_new=(uint8*)malloc(size);
			memcpy(p_new,p_contained_file,size);

			Utils::Assert(gNumFilesInOutputLib<MAX_FILES_IN_LIB,"Too many files in output lib");
			// mOffset gets calculated by WriteOutputLib, just initialising it here.
			gpOutputLibFileInfo[gNumFilesInOutputLib].mOffset=0;
			gpOutputLibFileInfo[gNumFilesInOutputLib].mFileNameChecksum=p_contained_file_info->mFileNameChecksum;
			gppOutputLibFiles[gNumFilesInOutputLib]=p_new;
			gpOutputLibFileInfo[gNumFilesInOutputLib].mFileSize=size;
			gpOutputLibFileInfo[gNumFilesInOutputLib].mExtensionChecksum=p_contained_file_info->mExtensionChecksum;
			++gNumFilesInOutputLib;

			p_contained_file+=(size+3)&~3;
			++p_contained_file_info;
		}

		free(p_lib);
	}
	else
	{
		// Load the file into memory
		if ( gDebugMode )
		{
			printf("@@@ Adding '%s' to lib file\n",p_file_name);
		}
		Utils::Assert(gNumFilesInOutputLib<MAX_FILES_IN_LIB,"Too many files in output lib");

		// mOffset gets calculated by WriteOutputLib, just initialising it here.
		gpOutputLibFileInfo[gNumFilesInOutputLib].mOffset=0;
		gpOutputLibFileInfo[gNumFilesInOutputLib].mFileNameChecksum=p_fileInfo->mFileNameChecksum;

		uint32 file_size=0;
		gppOutputLibFiles[gNumFilesInOutputLib]=LoadFile(p_file_name,&file_size,DONT_APPEND_ZERO,DONT_ASSERT);
		gpOutputLibFileInfo[gNumFilesInOutputLib].mFileSize=file_size;

		// Calculate p_non_plat_extension which is the full extension with any platform
		// specific bit removed from the end. Eg, 'ska.ps2' becomes 'ska'
		// Need to do this because the extension checksum in the lib is required to
		// not include the platform specific bit. 
		char p_non_plat_extension[100];
		const char *p_source=p_full_extension;	
		char *p_dest=p_non_plat_extension;
		while (*p_source && *p_source!='.')
		{
			*p_dest++=*p_source++;
		}
		*p_dest=0;

		gpOutputLibFileInfo[gNumFilesInOutputLib].mExtensionChecksum=Crc::GenerateCRCFromString(p_non_plat_extension);
		++gNumFilesInOutputLib;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void WriteOutputLib(const char *p_inputLibName)
{
	char p_output_lib_name[1024];
	sprintf(p_output_lib_name,"%s.%s",p_inputLibName,Utils::GetPlatformExt(gPlatform));

	FILE *ofp=fopen(p_output_lib_name,"wb");
	Utils::Assert(ofp!=NULL,"\nCould not open output lib file for writing.");

	SLibHeader header;
	header.mVersionNumber=LIB_VERSION;
	header.mNumFiles=gNumFilesInOutputLib;
	if (gPlatform==Utils::vPLATFORM_NGC)
	{
		SwapEndianness((uint32*)&header,sizeof(SLibHeader)/4);
	}
	Utils::Assert(fwrite(&header,sizeof(SLibHeader),1,ofp)==1,"fwrite failed");
	if (gPlatform==Utils::vPLATFORM_NGC)
	{
		SwapEndianness((uint32*)&header,sizeof(SLibHeader)/4);
	}

	// Calculate the mOffset members
	uint32 offset=sizeof(SLibHeader)+gNumFilesInOutputLib*sizeof(SLibFileInfo);
	for (int i=0; i<gNumFilesInOutputLib; ++i)
	{
		gpOutputLibFileInfo[i].mOffset=offset;
		offset+=gpOutputLibFileInfo[i].mFileSize;
		offset=(offset+3)&~3;
	}

	if (gPlatform==Utils::vPLATFORM_NGC)
	{
		SwapEndianness((uint32*)gpOutputLibFileInfo,gNumFilesInOutputLib*(sizeof(SLibFileInfo)/4));
	}

	if (gNumFilesInOutputLib)
	{
		Utils::Assert(fwrite(gpOutputLibFileInfo,gNumFilesInOutputLib*sizeof(SLibFileInfo),1,ofp)==1,"fwrite failed");
	}

	if (gPlatform==Utils::vPLATFORM_NGC)
	{
		SwapEndianness((uint32*)gpOutputLibFileInfo,gNumFilesInOutputLib*(sizeof(SLibFileInfo)/4));
	}

	for (int i=0; i<gNumFilesInOutputLib; ++i)
	{
		uint32 size=gpOutputLibFileInfo[i].mFileSize;
		// Write out the file data
		if (size)
		{
			Utils::Assert(ftell(ofp)==gpOutputLibFileInfo[i].mOffset,"File offset does not match ??");
			Utils::Assert(fwrite(gppOutputLibFiles[i],size,1,ofp)==1,"fwrite failed");
		}

		// then write out the padding to a long word boundary
		int pad_size=(4-(size&3))&3;
		if (pad_size)
		{
			uint8 p_pad[3]={0,0,0};
			Utils::Assert(fwrite(p_pad,pad_size,1,ofp)==1,"fwrite failed");
		}
	}

	fclose(ofp);
	ofp=NULL;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void print_usage()
{
	printf("LibConv.exe, %s  %s\n\n",__DATE__,__TIME__);
	printf("Required: -f<filename of source.lib file>\n");
	printf("          or -b<txt file containing list of .lib files>\n");
	printf("Required: -pp,-px or -pg, indicating PS2,XBox or GameCube\n");
	printf("Optional: -i<name of ini file>\n");
	printf("          if omitted it will use %%PROJ_ROOT%%\\bin\\win32\\libconv.ini\n\n");

	printf("LibConv will process each of the files contained in the lib file using the\n");
	printf("rules defined in the ini file.\n");
	printf("The processed files wil be put into a new lib file with the same name as\n");
	printf("the original, but with the extension .ps2, .xbx or .ngc appended, depending\n");
	printf("on whether -pp, -px or -pg was specified on the command line.\n\n");
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int main(int argc, char* argv[])
{
	//-------------------------------------------------------------
	if (argc==1)
	{
		// When no args, print help info.
		print_usage();
		return 0;
	}
	//-------------------------------------------------------------

	//-------------------------------------------------------------
	// Read the platform defined by the -p arg.
	char *p_platform = Utils::TestArg( argc, argv, 'P' );
	if ( p_platform )
	{
		switch ( *p_platform )
		{
			case 'X':
			case 'x':
				gPlatform = Utils::vPLATFORM_XBOX;
				break;
			case 'G':
			case 'g':
				gPlatform = Utils::vPLATFORM_NGC;
				break;
			case 'P':
			case 'p':
				gPlatform = Utils::vPLATFORM_PS2;
				break;
		}
	}
	Utils::Assert( gPlatform != Utils::vPLATFORM_NONE, "A platform must be set using -pp,-px or -pg" );
	//-------------------------------------------------------------

	//-------------------------------------------------------------
	if ( Utils::TestArg( argc, argv, 'D' ) )
	{
		gDebugMode = true;
	}
	//-------------------------------------------------------------

	//-------------------------------------------------------------
	// get the proj root name
	char* p_proj_root = getenv( "PROJ_ROOT" );
	Utils::Assert( p_proj_root != NULL, "LibConv.exe requires that the environment variable PROJ_ROOT is set." );
	//-------------------------------------------------------------

	//-------------------------------------------------------------
	// load up INI file (the PS2 version has a different INI file 
	// because it writes out GEOM files instead of MDL files...
	// ideally, we would have platform-specific sections inside
	// one shared INI file)
	char ini_file_name[_MAX_PATH];
	switch( gPlatform )
	{
		case Utils::vPLATFORM_PS2:
			sprintf( ini_file_name, "%s\\bin\\win32\\libconv_ps2.ini", p_proj_root );
			break;
		case Utils::vPLATFORM_XBOX:
		case Utils::vPLATFORM_NGC:
		default:
			sprintf( ini_file_name, "%s\\bin\\win32\\libconv.ini", p_proj_root );
			break;
	}
	
	// now check for -I override
	char* p_ini_file_name = Utils::TestArg( argc, argv, 'I' );
	if ( p_ini_file_name )
	{
		strcpy( ini_file_name, p_ini_file_name );
	}

	LoadAndParseIniFile(ini_file_name);
	//-------------------------------------------------------------

	//-------------------------------------------------------------
	char *p_file_name=Utils::TestArg( argc, argv, 'F' );
	if (p_file_name)
	{
		ProcessLibFile(p_file_name);
	}

	char *p_list_file_name=Utils::TestArg( argc, argv, 'B' );
	if (p_list_file_name)
	{
		ProcessBunchOfLibFiles(p_list_file_name);
	}
	//-------------------------------------------------------------

	//-------------------------------------------------------------
	CleanUp();
	//-------------------------------------------------------------

	// success!
	return 0;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/