/*
	WorldDB.h
	This class is responsible for maintaining the world database
	It allows objects to be added to the world on the fly
	and is responsible for spatial partitioning

	aml - 9-26-03
*/

#ifndef __WORLDDB__
#define __WORLDDB__

#include "WorldStructs.h"
#include "NxMaterial.h"
#include "llist.h"
#include <stdlib.h>
#include <stdio.h>

// The world data within the LevelEditor will get broken up into an
// OctTree so only reasonable sections of data get sent to the renderer at once

// One octnode describes a cube region that contains 8 smaller regions
// The 8 smaller regions are equally sized (1/8th the volume) and 

// The OctNode pointers refer to the representation as follows
#define OCTNODE_TBL   0		// Top-Back-Left
#define OCTNODE_TBR   1		// Top-Back-Right
#define OCTNODE_TFL   2     // Top-Front-Left
#define OCTNODE_TFR   3		// Top-Front-Right
#define OCTNODE_BBL   4		// Bottom-Back-Left
#define OCTNODE_BBR   5		// Bottom-Back-Right
#define OCTNODE_BFL   6		// Bottom-Front-Left
#define OCTNODE_BFR   7		// Bottom-Front-Right

// Region Flags
// These flags are for managing duplicate scenarios where a poly exists
// in multiple octants.  We can 
// If geom data intersects more than one of the 8 regions of its parent
enum
{
	OCTSHARE_TBL = 0x01,
	OCTSHARE_TBR = 0x02,
	OCTSHARE_TFL = 0x04,
	OCTSHARE_TFR = 0x08,
	OCTSHARE_BBL = 0x10,
	OCTSHARE_BBR = 0x20,
	OCTSHARE_BFL = 0x40,
	OCTSHARE_BFR = 0x80,
};

// Picture this as a set of eight blocks.  The eight block set
// stacked into a cube represents one octnode

//                    ---------------------------
//                   /             /            /|
//                  /    TBL      /   TBR      / |
//                 /_____________/___________ /  |
//                /             /|           /|  |
//               /             / |          / |  |
//              |-------------|--|---------|  |  |
//              |      TFL    |  |__TFR___ |  | /|
//              |             |  /         |  |/ |
//              |             | /          | /|  |
//  BBL         |_____________|/___________|/ | -|------ BBR
//  is occluded |             |            |  |  |
//  behind BFL  |             |            |  | /
//  and under   |      BFL    |     BFR    |  |/ 
//  TBL         |             |            |  /  
//              |             |            | /   
//              ----------------------------/

// Each GeomNode represents a renderable object that exists within an OctNode
// A GeomNode may span accross multiple OctNodes in space.  We don't want
// to split the polys since we want to keep the database in sync with the
// exact way the data is represented in MAX nor do we wish to generate additional
// geometry, therefore duplicates are allowed.  

// To prevent redraw, we flag the geom piece with the octants that it exists in
// When rendering a piece of geom within a specific octant if any duplicate flags
// are set the data gets added to a list of 

// Octant Rendering Proceedure Pseudocode
// for each of 8 octants
//	 for each octant GeomNode
//		

struct GeomNode
{
	int dupeFlags;
};

struct OctNode
{
	Vector   center[3];	// Center position of this octent cube
	float    size;		// The volume of the cube / 3 since width = height = depth
	
	OctNode* node[8];
	OctNode* parent;	// Parent OctNode that contains this one (NULL if root)
						// necessary for minimizing redraw
};

// Looking up the closest octant via binary mapping
// It will be necessary to quickly locate an OctNode of interest given just a position
// in space with minimal searching.  To do this we can map the bits representing the X, Y and Z
// positions of the spatial coordinates into bit trees.

class WorldDB
{
	LinkList<NxMaterial> mtlList;
	LinkList<NxObject>   objList;
	//LinkList<OctNode*> octListX;

	bool        m_sky;		// Loaded scene is sky

	bool		LoadMaterials(FILE* fp, int version);
	bool		LoadObjects(FILE* fp, int version);

public:
	WorldDB();
	~WorldDB();

	void        AddMaterial(NxMaterial* pMtl);
	NxMaterial* GetMaterial(unsigned long crc);

	bool		LoadScene(char* filename);
	bool		SaveScene(char* filename);
};

#endif
