#include <stdlib.h>
#include <fstream.h>

#include <core\defines.h>
#include <core\crc.h>
#include <gel\scripting\tokens.h>
#include <gel/scripting/init.h>
#include <gel/scripting/script.h>
#include <gel/scripting/symboltable.h>
#include <gel/scripting/parse.h>
#include <gel/scripting/checksum.h>
#include <gel/scripting/struct.h>
#include <gel/scripting/array.h>
#include <gel\scripting\scriptcache.h>
#include <gel\scripting\utils.h>

#include <sys/config/config.h>

#include "Genlib/FileDatabase.h"
#include "Genlib/Utility.h"
#include "Genlib/VirtualFile.h"
#include "terrain.h"
#include "particle.h"
#include "anims.h"
#include "models.h"
#include "cameras.h"
#include "profiles.h"
#include "levelassetlister.h"

// TODO:  which files is the level pre file dependent on?
// (like qcompass)
// and only run those files if it's changed
// main runs file database only once, then batches

#define PED_PROFILE_LIST_FILE "\\bin\\win32\\pedprofilelist.txt"
#define PED_ASSET_LIST_FILE "\\bin\\win32\\pedassetlist.txt"

#define vMAJOR_REV 2
#define vMINOR_REV 0

using namespace Script;

#include <gel/scripting/skiptoken.cpp>

char QNFileName[_MAX_PATH];
char OutFileName[_MAX_PATH];
char OutPedPartsFileName[_MAX_PATH];
char OutBspFileName[_MAX_PATH];
char OutColFileName[_MAX_PATH];
char DataPath[_MAX_PATH];
char LevelPath[_MAX_PATH];
char SkyPath[_MAX_PATH];
char ShellPath[_MAX_PATH];
char ModelPathName[_MAX_PATH];

#define MAX_FILES 256
char OutputtedFile[MAX_FILES][_MAX_PATH];
int NumOutputtedFiles;

int platform = Utils::vPLATFORM_NONE;
int GetPlatform()
{
	return platform;
}

char platform_string[32];

IoUtils::CVirtualOutputFile theOutputFile;
IoUtils::CVirtualOutputFile theOutputPedFile;

IoUtils::CFileDatabase thePermAnimDatabase;
IoUtils::CFileDatabase theFileDatabase;

inline void GetOutputFileName( char* pTarget, const char* pBaseName, const char* pExt )
{
	strcpy( pTarget, pBaseName );
	strcat( pTarget, pExt );
	strcat( pTarget, "." );
	strcat( pTarget, Utils::GetPlatformExt(platform) );
}

const char *GetPlatformExtension()
{
	strcpy( platform_string, "." );
	strcat( platform_string, Utils::GetPlatformExt( platform ) ); 

	return platform_string;
}

bool MyScriptExists(uint32 scriptName)
{
	Script::CSymbolTableEntry *p_entry=Script::Resolve(scriptName);
	if (p_entry && p_entry->mType==ESYMBOLTYPE_QSCRIPT)
	{
		return true;
	}
	return false;
}

bool ScriptExists(const char *p_scriptName)
{
	return MyScriptExists(Script::GenerateCRC(p_scriptName));
}

bool TestExistence(char *filename, int line, bool assertOnFail)
{
	// there are some cases where the file MUST exist
	// and others where we just want to check what
	// kind of file it is (SKIN or MDL).  In the cases
	// where it must exist, we write it out anyway,
	// and if the file doesn't exist, the MakePRE
	// utility will handle the error message...
	// this will speed things up considerably
	// (Ideally, the MakePRE would handle ALL the 
	// existence checking, since it has to build a database
	// of filetimes anyway)

#if 1
	if ( assertOnFail )
	{
		return true;
	}
#endif

/*
	if ( theFileDatabase.FileExists(filename) )
	{
		return true;
	}
*/
	int timeStamp = Utils::GetTimeStamp(filename);

/*
	theFileDatabase.AddFile( filename, timeStamp );
*/

	if ( assertOnFail && timeStamp == -1 )
	{
		Utils::Assert( 0, "Couldn't find %s", filename );
//		printf( "%s doesn't exist %d!\n", filename, line );
	}

	return timeStamp != -1;
}

bool ScriptPrintf(Script::CStruct *pParams, Script::CScript *pScript)
{
	const char *p_text="";
	pParams->GetString(NONAME,&p_text);
	printf("PCPrintf: %s\n",p_text);

	return true;
}

bool ScriptInNetGame(Script::CStruct *pParams, Script::CScript *pScript)
{
	return false;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

// @script | PrintStruct | Prints the contents of the passed structure
// @uparm structure | The structure to print
bool ScriptPrintStruct(Script::CStruct *pParams, Script::CScript *pScript)
{
	Script::PrintContents(pParams);
	return true;
}



bool ScriptGetCurrentSkaterProfileIndex(Script::CStruct *pParams, Script::CScript *pScript)
{
	// this just returns a dummy value so that subsequent if-test don't choke
	pScript->GetParams()->AddInteger( "currentSkaterProfileIndex", 0);

	printf( "%s\n", pScript->GetScriptInfo() );

	// GJ:  for some reason, if i return false, then i get an assert
	// that the component still points to some data...  i'll have to
	// ask ken about that sometime.
	return true;
}

// Required because the animload_human script contains a call to this.
bool ScriptGetArraySize(Script::CStruct *pParams, Script::CScript *pScript)
{
	Script::CArray* pArray;
	
	pParams->GetArray( NONAME, &pArray, Script::ASSERT );

	pScript->GetParams()->AddInteger( "array_size", pArray->GetSize() );

	return true;
}

void RegisterCFuncs()
{
	Script::CSymbolTableEntry *p_new=NULL;
	
	p_new=Script::CreateNewSymbolEntry(0x88684899/*LoadTerrainSounds*/);
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptLoadTerrainSounds;

	p_new=Script::CreateNewSymbolEntry(0x2a621572/*LoadSound*/);
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptLoadSound;

	p_new=Script::CreateNewSymbolEntry(0x604053a4/*LoadParticleTexture*/);
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptLoadParticleTexture;

	p_new=Script::CreateNewSymbolEntry(0x89921f5b/*LoadAnim*/);
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptLoadAnim;

	p_new=Script::CreateNewSymbolEntry(0xd0438caa/*LoadAsset*/);
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptLoadAsset;

	p_new=Script::CreateNewSymbolEntry(Script::GenerateCRC("PrintStruct")/*LoadAsset*/);
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptPrintStruct;

	p_new=Script::CreateNewSymbolEntry(0x39b5c4b0/*GetArraySize*/);
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptGetArraySize;

	p_new=Script::CreateNewSymbolEntry(0xa1da6fe9/*InNetGame*/);
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptInNetGame;

	p_new=Script::CreateNewSymbolEntry(0xd05cbe31/*GetCurrentSkaterProfileIndex*/);
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptGetCurrentSkaterProfileIndex;

	p_new=Script::CreateNewSymbolEntry(0xef067f2c/*pcprintf*/);
	p_new->mType=ESYMBOLTYPE_CFUNCTION;
	p_new->mpCFunction=ScriptPrintf;
}

uint8 *LoadFileIntoMemory(const char *p_fileName)
{
	IoUtils::CVirtualInputFile theInputFile;
	int size = theInputFile.Load(p_fileName);
	if (!size)
	{
		//printf("File '%s' does not exist!\n",p_fileName);
		return NULL;
	}
	uint8* p_file = (uint8*)malloc(size);
	theInputFile.Read( (char*)p_file, size );
	return p_file;
}

void LoadQB(const char *p_qb_name)
{
	uint8* p_qb=LoadFileIntoMemory(p_qb_name);
	if (p_qb)
	{
		Script::ParseQB(p_qb_name,p_qb);
		// All the globals and scripts defined in the q file are now loaded, 
		// so no need to keep the qb any more.
		free(p_qb);
	}
}

void LoadRequiredQBFiles(const char *p_level_name)
{
	char p_allanims_qb_name[_MAX_PATH];
	sprintf(p_allanims_qb_name,"%s\\data\\scripts\\allanims.qb",GetProjRoot());

	char p_startup_qb_name[_MAX_PATH];
	sprintf(p_startup_qb_name,"%s\\data\\scripts\\game\\startup.qb",GetProjRoot());

	char p_terrain_qb_name[_MAX_PATH];
	sprintf(p_terrain_qb_name,"%s\\data\\scripts\\game\\terrain.qb", GetProjRoot());


	char p_level_scripts_qb_name[_MAX_PATH];
	sprintf(p_level_scripts_qb_name,"%s\\data\\levels\\%s\\%s_scripts.qb",GetProjRoot(),p_level_name,p_level_name);

	char p_level_sfx_qb_name[_MAX_PATH];
	sprintf(p_level_sfx_qb_name,"%s\\data\\levels\\%s\\%s_sfx.qb",GetProjRoot(),p_level_name,p_level_name);

	char p_level_qb_name[_MAX_PATH];
	sprintf(p_level_qb_name,"%s\\data\\levels\\%s\\%s.qb",GetProjRoot(),p_level_name,p_level_name);


	LoadQB(p_startup_qb_name);
	LoadQB(p_allanims_qb_name);
	LoadQB(p_terrain_qb_name);
	LoadQB(p_level_scripts_qb_name);
	LoadQB(p_level_sfx_qb_name);
	LoadQB(p_level_qb_name);
}

// If the passed anim file is loaded by any of the scripts that are called from
// the load_permanent_anims script, this will return true.
bool AnimFileIsPermanentlyLoaded(const char *p_fileName)
{
	return thePermAnimDatabase.FileExists( p_fileName );
}

bool TestIfAlreadyOutputted(char *ModelFileName)
{
	for (int i = 0; i < NumOutputtedFiles; i++)
	{
		if (strcmp(OutputtedFile[i], ModelFileName) == 0)
			return true;
	}

	return false;
}

void AttemptOutputCameraAnimFile(IoUtils::CVirtualOutputFile* pOutputFile, char *pCameraAnimName)
{
	char pFullCameraAnimFileName[_MAX_PATH];
	
	strcpy(pFullCameraAnimFileName, DataPath);
	strcat(pFullCameraAnimFileName, pCameraAnimName);
	strcat(pFullCameraAnimFileName, GetPlatformExtension());

	if (TestExistence(pFullCameraAnimFileName,__LINE__))
	{
		pOutputFile->WriteString( pFullCameraAnimFileName, true );
	}
}

void AttemptOutputFile(IoUtils::CVirtualOutputFile* pOutputFile, const char *pModelName)
{
	if ( !strcmp(pModelName,"none") )
	{
		return;
	}

	//printf("pModelName=%s\n",pModelName);

	// compute the full name of the model file, minus extension
	char ModelFileName[_MAX_PATH];
	strcpy(ModelFileName, ModelPathName);
	// ModelFileName will now be "c:\skate4\data\Models\"

	if ( strstr( pModelName, "/" ) || strstr( pModelName, "\\" ) )
	{
		// It's the new format, where the name will be followed by an extension.
		char *p_extension=strstr( pModelName, "." );
		if (p_extension==NULL)
		{
			printf("Filename is missing extension: %s\n",pModelName);
			Utils::Assert( 0, "Error!" ); 
		}
		strcat(ModelFileName, pModelName);	
		// Remove the extension by turning the . into a terminator.
		*strstr( ModelFileName, "." )=0;
	}
	else
	{
		// It's the old format, in which the name is the name of the subdirectory
		// as well as the of the files within in.
		strcat(ModelFileName, pModelName);
		strcat(ModelFileName, "\\");
		strcat(ModelFileName, pModelName);
	}

	//printf("AttemptOutputFile %s\n", ModelFileName);
										
	if (TestIfAlreadyOutputted(ModelFileName))
	{
		//printf("already outputted model %s\n", ModelFileName);
		return;
	}

	// see if SKN exists, and output its name if so
	int skinExists = false;
	char ModelSkinFileName[_MAX_PATH];
	GetOutputFileName(ModelSkinFileName, ModelFileName, ".skin");
	if (TestExistence(ModelSkinFileName,__LINE__))
	{
		skinExists = true;
		pOutputFile->WriteString( ModelSkinFileName, true );
	}

	if ( skinExists )
	{
		// see if CAS exists, and output its name if so
		char ModelCasFileName[_MAX_PATH];
		GetOutputFileName(ModelCasFileName, ModelFileName, ".cas");
		if (TestExistence(ModelCasFileName,__LINE__,true))
		{
			pOutputFile->WriteString( ModelCasFileName, true );
		}
	}

	if ( !skinExists )
	{
		// see if QB exists, and output its name if so
		char ModelQbFileName[_MAX_PATH];
		strcpy(ModelQbFileName,ModelFileName);
		strcat(ModelQbFileName,".qb");
		if (TestExistence(ModelQbFileName,__LINE__))
		{
			pOutputFile->WriteString( ModelQbFileName, true );
		}

		char *pMdlExtension="";
		if( platform == Utils::vPLATFORM_PS2 )
		{
			pMdlExtension=".geom.ps2";
		}
		else if( platform == Utils::vPLATFORM_XBOX )
		{
			pMdlExtension=".mdl.xbx";
		}
		else if( platform == Utils::vPLATFORM_NGC )
		{
			pMdlExtension=".mdl.ngc";
		}

		// see if MDL exists, and output its name if so
		char ModelDffFileName[_MAX_PATH];
		strcpy(ModelDffFileName, ModelFileName);
		strcat(ModelDffFileName, pMdlExtension);
		if (TestExistence(ModelDffFileName,__LINE__,true))
		{
			pOutputFile->WriteString( ModelDffFileName, true );
		}
	}
	
	// see if TEX exists, and output its name if so
	char ModelTexFileName[_MAX_PATH];
	GetOutputFileName(ModelTexFileName, ModelFileName, ".tex");
	if (TestExistence(ModelTexFileName,__LINE__,true))
	{
		pOutputFile->WriteString( ModelTexFileName, true );
	}

	// see if COL exists, and output its name if so
	char ModelColFileName[_MAX_PATH];
	GetOutputFileName(ModelColFileName, ModelFileName, ".col");
	if (TestExistence(ModelColFileName,__LINE__))
	{
		pOutputFile->WriteString( ModelColFileName, true );
	}


	strcpy(OutputtedFile[NumOutputtedFiles++], ModelFileName);
	
/*
	GJ:  Models don't have anims (for now), so we 
	can shave off some time by not calling the following...

#if USE_VIRTUAL_FILE
	AttemptOutputModelAnims(pOutputFile, pModelName);
#else
	AttemptOutputModelAnims(out, pModelName);
#endif
*/
	
	//printf("outputted model: %s\n", ModelFileName);
}

struct SPedProfileListEntry
{
	uint32 mProfileChecksum;
	uint32 mPartChecksum;
	uint32 mDescriptionChecksum;
	bool mChooseRandomized;
};

#define PED_PROFILE_LIST_SIZE 900
SPedProfileListEntry gpPedProfileList[PED_PROFILE_LIST_SIZE];
uint32 gPedProfileListNumEntries=0;

#define MAX_PED_ASSET_FILE_NAME_CHARS 100
struct SPedAssetListEntry
{
	uint32 mPartChecksum;
	uint32 mDescriptionChecksum;
	char mpAssetFileName[MAX_PED_ASSET_FILE_NAME_CHARS];
	bool mNoRandom;
	bool mAddedToPreAlready; // Used to ensure the same files aren't listed twice.
};

#define PED_ASSET_LIST_SIZE 500
SPedAssetListEntry gpPedAssetList[PED_ASSET_LIST_SIZE];
uint32 gPedAssetListNumEntries=0;

char *SkipWhiteSpace(char *p_ch)
{
	Utils::Assert(p_ch!=NULL,"NULL p_ch");

	while (*p_ch==' ' || *p_ch=='\t' || *p_ch==0x0d || *p_ch==0x0a)
	{
		++p_ch;
	}

	return p_ch;
}

char *ReadChecksum(char *p_ch, uint32 *p_checksum)
{
	Utils::Assert(p_ch!=NULL,"NULL p_ch");
	Utils::Assert(p_checksum!=NULL,"NULL p_checksum");

	Utils::Assert(p_ch[0]=='0' && p_ch[1]=='x',"Bad characters '%c%c' in hex value, expected '0x'",p_ch[0],p_ch[1]);
	p_ch+=2;

	*p_checksum=0;
	for (int i=0; i<8; ++i)
	{
		*p_checksum<<=4;

		if (*p_ch>='0' && *p_ch<='9')
		{
			*p_checksum+=*p_ch-'0';
		}
		else if (*p_ch>='a' && *p_ch<='f')
		{
			*p_checksum+=10+*p_ch-'a';
		}
		else if (*p_ch>='A' && *p_ch<='F')
		{
			*p_checksum+=10+*p_ch-'A';
		}
		else
		{
			Utils::Assert(0,"Bad character '%c' in hex value",*p_ch);
		}

		++p_ch;
	}

	return p_ch;
}

char *ReadBool(char *p_ch, bool *p_bool)
{
	Utils::Assert(p_ch!=NULL,"NULL p_ch");
	Utils::Assert(p_bool!=NULL,"NULL p_bool");

	if (*p_ch=='0')
	{
		*p_bool=false;
	}
	else if (*p_ch=='1')
	{
		*p_bool=true;
	}
	else
	{
		Utils::Assert(0,"Bad character '%c' in bool value, expected '0' or '1'",*p_ch);
	}

	++p_ch;
	return p_ch;
}

char *ReadString(char *p_ch, char *p_dest, uint32 dest_buffer_size)
{
	Utils::Assert(p_ch!=NULL,"NULL p_ch");
	Utils::Assert(p_dest!=NULL,"NULL p_dest");

	uint32 bytes_written=0;
	while (true)
	{
		if (*p_ch==0 || *p_ch==' ' || *p_ch=='\t' || *p_ch==0x0d || *p_ch==0x0a)
		{
			break;
		}

		Utils::Assert(bytes_written<dest_buffer_size-1,"Too many chars in string! Increase MAX_PED_ASSET_FILE_NAME_CHARS in LevelAssetLister.cpp");
		*p_dest++=*p_ch++;
		++bytes_written;
	}
	*p_dest=0;

	return p_ch;
}

void LoadPedProfileAndAssetLists( const char* proj_root )
{
	gPedProfileListNumEntries=0;
	gPedAssetListNumEntries=0;

	// Open pedprofilelist.txt
	char profileListName[_MAX_PATH];
	sprintf( profileListName, "%s%s", proj_root, PED_PROFILE_LIST_FILE );
	ifstream profile_txt_file(profileListName, ios::in | ios::binary | ios::nocreate);
	if ( !profile_txt_file.is_open() )
	{
		Utils::Assert( 0, "Couldn't open %s", profileListName );
		return;
	}

	profile_txt_file.seekg(0, ios::end);
	int txt_size = profile_txt_file.tellg();
	if (txt_size)
	{
		// The file exists, so load it into memory and stick a 0 terminator on the
		// end for ease of parsing.
		profile_txt_file.seekg(0, ios::beg);
		char *p_ped_profile_list_txt = (char*)malloc(txt_size+1); // +1 for terminator
		profile_txt_file.read(p_ped_profile_list_txt, txt_size);
		profile_txt_file.close();
		p_ped_profile_list_txt[txt_size]=0;

		// Parse the file, filling in the gpPedProfileList array
		char *p_ch=p_ped_profile_list_txt;
		while (*p_ch)
		{
			if (gPedProfileListNumEntries >= PED_PROFILE_LIST_SIZE) 
			{
				Utils::Assert( 0, "Too many ped profiles (increase PED_PROFILE_LIST_SIZE from %d)", PED_PROFILE_LIST_SIZE );
			}

			p_ch=ReadChecksum(p_ch,&gpPedProfileList[gPedProfileListNumEntries].mProfileChecksum);
			p_ch=SkipWhiteSpace(p_ch);
			p_ch=ReadChecksum(p_ch,&gpPedProfileList[gPedProfileListNumEntries].mPartChecksum);
			p_ch=SkipWhiteSpace(p_ch);
			p_ch=ReadChecksum(p_ch,&gpPedProfileList[gPedProfileListNumEntries].mDescriptionChecksum);
			p_ch=SkipWhiteSpace(p_ch);
			p_ch=ReadBool(p_ch,&gpPedProfileList[gPedProfileListNumEntries].mChooseRandomized);
			p_ch=SkipWhiteSpace(p_ch);

			++gPedProfileListNumEntries;
		}

		free (p_ped_profile_list_txt);
	}
	else
	{
		profile_txt_file.close();
	}

	
	// Open pedassetlist.txt
	char assetListName[_MAX_PATH];
	sprintf( assetListName, "%s%s", proj_root, PED_ASSET_LIST_FILE );
	ifstream asset_txt_file(assetListName, ios::in | ios::binary | ios::nocreate);
	if ( !asset_txt_file.is_open() )
	{
		Utils::Assert( 0, "Couldn't open %s", assetListName );
		return;
	}
	asset_txt_file.seekg(0, ios::end);
	txt_size = asset_txt_file.tellg();
	if (txt_size)
	{
		// The file exists, so load it into memory and stick a 0 terminator on the
		// end for ease of parsing.
		asset_txt_file.seekg(0, ios::beg);
		char *p_ped_asset_list_txt = (char*)malloc(txt_size+1); // +1 for terminator
		asset_txt_file.read(p_ped_asset_list_txt, txt_size);
		asset_txt_file.close();
		p_ped_asset_list_txt[txt_size]=0;

		// Parse the file, filling in the gpPedAssetList array
		char *p_ch=p_ped_asset_list_txt;
		while (*p_ch)
		{
			if (gPedAssetListNumEntries >= PED_ASSET_LIST_SIZE) 
			{
				Utils::Assert( 0, "Too many ped assets (increase PED_ASSET_LIST_SIZE from %d)", PED_ASSET_LIST_SIZE );
			}

			p_ch=ReadChecksum(p_ch,&gpPedAssetList[gPedAssetListNumEntries].mPartChecksum);
			p_ch=SkipWhiteSpace(p_ch);
			p_ch=ReadChecksum(p_ch,&gpPedAssetList[gPedAssetListNumEntries].mDescriptionChecksum);
			p_ch=SkipWhiteSpace(p_ch);
			p_ch=ReadString(p_ch,gpPedAssetList[gPedAssetListNumEntries].mpAssetFileName,MAX_PED_ASSET_FILE_NAME_CHARS);
			p_ch=SkipWhiteSpace(p_ch);
			p_ch=ReadBool(p_ch,&gpPedAssetList[gPedAssetListNumEntries].mNoRandom);
			p_ch=SkipWhiteSpace(p_ch);

			gpPedAssetList[gPedAssetListNumEntries].mAddedToPreAlready=false;
			++gPedAssetListNumEntries;
		}

		free (p_ped_asset_list_txt);
	}
	else
	{
		asset_txt_file.close();
	}
}

void OutputProfile(IoUtils::CVirtualOutputFile* pOutputFile, uint32 profile_checksum)
{
	SPedProfileListEntry *p_profile_entry=NULL;
	for (uint32 p=0; p<gPedProfileListNumEntries; ++p)
	{
		if (gpPedProfileList[p].mProfileChecksum==profile_checksum)
		{
			if (!gpPedProfileList[p].mChooseRandomized)
			{
				// Run through the ped asset list adding all the skin and tex
				// files for the part/descriptions that match those for the profile.
				for (uint32 a=0; a<gPedAssetListNumEntries; ++a)
				{
					if (gpPedAssetList[a].mPartChecksum==gpPedProfileList[p].mPartChecksum &&
						gpPedAssetList[a].mDescriptionChecksum==gpPedProfileList[p].mDescriptionChecksum &&
						!gpPedAssetList[a].mAddedToPreAlready)
					{
						char p_file_name[_MAX_PATH];
						strcpy(p_file_name,gpPedAssetList[a].mpAssetFileName);
						strcat(p_file_name,".skin.");
						strcat(p_file_name,Utils::GetPlatformExt(platform));
						if (TestExistence(p_file_name,__LINE__,true))
						{
							pOutputFile->WriteString( p_file_name, true );
						}

						strcpy(p_file_name,gpPedAssetList[a].mpAssetFileName);
						strcat(p_file_name,".tex.");
						strcat(p_file_name,Utils::GetPlatformExt(platform));
						if (TestExistence(p_file_name,__LINE__,true))
						{
							pOutputFile->WriteString( p_file_name, true );
						}

						strcpy(p_file_name,gpPedAssetList[a].mpAssetFileName);
						strcat(p_file_name,".cas.");
						strcat(p_file_name,Utils::GetPlatformExt(platform));
						if (TestExistence(p_file_name,__LINE__,true))
						{
							pOutputFile->WriteString( p_file_name, true );
						}

						gpPedAssetList[a].mAddedToPreAlready=true;
					}
				}
			}
			else
			{
				// Run through the ped asset list adding all the skin and tex
				// files for the parts matching those for the profile, and for which mNoRandom is false.
				for (uint32 a=0; a<gPedAssetListNumEntries; ++a)
				{
					if (gpPedAssetList[a].mPartChecksum==gpPedProfileList[p].mPartChecksum &&
						!gpPedAssetList[a].mNoRandom &&
						!gpPedAssetList[a].mAddedToPreAlready)
					{
						char p_file_name[1000];
						strcpy(p_file_name,gpPedAssetList[a].mpAssetFileName);
						strcat(p_file_name,".skin.");
						strcat(p_file_name,Utils::GetPlatformExt(platform));
						if (TestExistence(p_file_name,__LINE__,true))
						{
							pOutputFile->WriteString( p_file_name, true );
						}

						strcpy(p_file_name,gpPedAssetList[a].mpAssetFileName);
						strcat(p_file_name,".tex.");
						strcat(p_file_name,Utils::GetPlatformExt(platform));
						if (TestExistence(p_file_name,__LINE__,true))
						{
							pOutputFile->WriteString( p_file_name, true );
						}

						strcpy(p_file_name,gpPedAssetList[a].mpAssetFileName);
						strcat(p_file_name,".cas.");
						strcat(p_file_name,Utils::GetPlatformExt(platform));
						if (TestExistence(p_file_name,__LINE__,true))
						{
							pOutputFile->WriteString( p_file_name, true );
						}

						gpPedAssetList[a].mAddedToPreAlready=true;
					}
				}
			}
		}
	}
}

void OutputPedFile(IoUtils::CVirtualOutputFile* pOutputFile, const char *p_model_name)
{
	Utils::Assert(p_model_name!=NULL,"NULL p_model_name");

	// compute the full name of the model file, minus extension
	char ModelFileName[_MAX_PATH];
	strcpy(ModelFileName, ModelPathName);

	bool got_full_filename=false;

	char p_temp[1000];
	strcpy(p_temp,p_model_name);
	char *p_scan=p_temp;
	while (*p_scan)
	{
		if (*p_scan=='.')
		{
			*p_scan=0;
			got_full_filename=true;
			break;
		}
		++p_scan;
	}

	if (got_full_filename)
	{
		strcat(ModelFileName, p_temp);
		//printf("OutputPedFile: New: %s\n",ModelFileName);
	}
	else
	{
		strcat(ModelFileName, p_temp);
		strcat(ModelFileName, "\\");
		strcat(ModelFileName, p_temp);
		//printf("OutputPedFile: Old: %s\n",ModelFileName);
	}
	
	if (TestIfAlreadyOutputted(ModelFileName))
	{
		return;
	}

	char p_file_name[1000];
	GetOutputFileName(p_file_name,ModelFileName,".skin");
	if (TestExistence(p_file_name,__LINE__))
	{
		pOutputFile->WriteString( p_file_name, true );
	}

	GetOutputFileName(p_file_name,ModelFileName,".tex");
	if (TestExistence(p_file_name,__LINE__,true))
	{
		pOutputFile->WriteString( p_file_name, true );
	}

	GetOutputFileName(p_file_name,ModelFileName,".cas");
	if (TestExistence(p_file_name,__LINE__))
	{
		pOutputFile->WriteString( p_file_name, true );
	}

	GetOutputFileName(p_file_name,ModelFileName,".mdl");
	if (TestExistence(p_file_name,__LINE__))
	{
		pOutputFile->WriteString( p_file_name, true );
	}

	GetOutputFileName(p_file_name,ModelFileName,".col");
	if (TestExistence(p_file_name,__LINE__))
	{
		pOutputFile->WriteString( p_file_name, true );
	}

	strcpy(OutputtedFile[NumOutputtedFiles++], ModelFileName);
}

// Returns true if the name is that of some pedestrian
bool IsPed(const char *p_name)
{
	Utils::Assert(p_name!=NULL,"NULL p_name");

	const char *p_scan=p_name;
	const char *p_filename=NULL;
	while (*p_scan)
	{
		if (*p_scan=='/' || *p_scan=='\\')
		{
			p_filename=p_scan+1;
		}
		++p_scan;
	}


	if (p_filename != NULL)
	{
		// p_name is in the new format, where it is the full filename of the model, 
		// including the extension on the end.
		// p_filename is now the bit following the final \, ie the actual file name.

		if ((p_filename[0]=='p' || p_filename[0]=='P') &&
			(p_filename[1]=='e' || p_filename[1]=='E') &&
			(p_filename[2]=='d' || p_filename[2]=='D'))
		{
			return true;
		}
	}
	else
	{
		// p_name is in the old format, where it is a single word, no extension.
		if ((p_name[0]=='p' || p_name[0]=='P') &&
			(p_name[1]=='e' || p_name[1]=='E') &&
			(p_name[2]=='d' || p_name[2]=='D'))
		{
			return true;
		}
	}

	return false;
}

// Returns true if the name is that of some animal
bool IsAnimal(const char *p_name)
{
	Utils::Assert(p_name!=NULL,"NULL p_name");

	const char *p_scan=p_name;
	const char *p_filename=NULL;
	while (*p_scan)
	{
		if (*p_scan=='/' || *p_scan=='\\')
		{
			p_filename=p_scan+1;
		}
		++p_scan;
	}


	if (p_filename != NULL)
	{
		// p_name is in the new format, where it is the full filename of the model, 
		// including the extension on the end.
		// p_filename is now the bit following the final \, ie the actual file name.

		if ((p_filename[0]=='a' || p_filename[0]=='A') &&
			(p_filename[1]=='n' || p_filename[1]=='N') &&
			(p_filename[2]=='l' || p_filename[2]=='L'))
		{
			return true;
		}
	}
	else
	{
		// p_name is in the old format, where it is a single word, no extension.
		if ((p_name[0]=='a' || p_name[0]=='A') &&
			(p_name[1]=='n' || p_name[1]=='N') &&
			(p_name[2]=='l' || p_name[2]=='L'))
		{
			return true;
		}
	}

	return false;
}




void print_title()
{
#ifdef _DEBUG
	printf( "\nLevelAssetLister v%d.%d (Debug) Built %s %s\n", vMAJOR_REV, vMINOR_REV ,__DATE__, __TIME__ );
#else
	printf( "\nLevelAssetLister v%d.%d (Release) Built %s %s\n", vMAJOR_REV, vMINOR_REV ,__DATE__, __TIME__ );
#endif
	printf( "Neversoft Entertainment\n" );
}

void print_usage()
{
	print_title();

	printf( "Usage: LevelAssetLister -p[p: Playstation/g: Gamecube/x: Xbox] -f[f: LevelName]\n" );
	printf( "Example: LevelAssetLister -pp -fAlc\n" );
}

const char *GetProjRoot()
{
	static const char *sp_proj_root = NULL;
	if (!sp_proj_root)
	{
		sp_proj_root=getenv( "PROJ_ROOT" );
		Utils::Assert(sp_proj_root!=NULL,"You must first define your PROJ_ROOT environment variable" );
	}
	return sp_proj_root;
}

bool UpToDate(const char *p_levelName)
{
	// Removed optimization for the moment since there were problems with it not running
	// when the ped parts q files were changed.
	return false;

	if (Utils::GetTimeStamp(OutPedPartsFileName) == -1 ||
		Utils::GetTimeStamp(OutBspFileName) == -1 ||
		Utils::GetTimeStamp(OutColFileName) == -1)
	{
		return false;
	}

	int pre_txt_time_stamp=Utils::GetTimeStamp(OutFileName);
	if (pre_txt_time_stamp== -1)
	{
		return false;
	}

	char p_qb_name[_MAX_PATH];
	sprintf(p_qb_name,"%s\\data\\scripts\\allanims.qb",GetProjRoot());
	if (Utils::GetTimeStamp(p_qb_name) >= pre_txt_time_stamp)
	{
		return false;
	}

	sprintf(p_qb_name,"%s\\data\\scripts\\game\\startup.qb",GetProjRoot());
	if (Utils::GetTimeStamp(p_qb_name) >= pre_txt_time_stamp)
	{
		return false;
	}

	sprintf(p_qb_name,"%s\\data\\scripts\\game\\terrain.qb", GetProjRoot());
	if (Utils::GetTimeStamp(p_qb_name) >= pre_txt_time_stamp)
	{
		return false;
	}

	sprintf(p_qb_name,"%s\\data\\levels\\%s\\%s_scripts.qb",GetProjRoot(),p_levelName,p_levelName);
	if (Utils::GetTimeStamp(p_qb_name) >= pre_txt_time_stamp)
	{
		return false;
	}

	sprintf(p_qb_name,"%s\\data\\levels\\%s\\%s_sfx.qb",GetProjRoot(),p_levelName,p_levelName);
	if (Utils::GetTimeStamp(p_qb_name) >= pre_txt_time_stamp)
	{
		return false;
	}

	sprintf(p_qb_name,"%s\\data\\levels\\%s\\%s.qb",GetProjRoot(),p_levelName,p_levelName);
	if (Utils::GetTimeStamp(p_qb_name) >= pre_txt_time_stamp)
	{
		return false;
	}

	return true;
}

int main(int argc, char* argv[])
{
//	print_title();

	Spt::SingletonPtr< Script::CScriptCache> script_cache( true );

	const char* proj_path = GetProjRoot();

	char* pPlatform;
	pPlatform = Utils::TestArg( argc, argv, 'P' );

	if ( pPlatform )
	{
		switch ( *pPlatform )
		{
			case 'X':
			case 'x':
				platform = Utils::vPLATFORM_XBOX;
				break;
			case 'G':
			case 'g':
				platform = Utils::vPLATFORM_NGC;
				break;
			case 'P':
			case 'p':
				platform = Utils::vPLATFORM_PS2;
				break;
		}
	}

	if ( platform == Utils::vPLATFORM_NONE )
	{
		// no platform was specified
		print_usage();
		exit(1);
	}

	char* pLevelName = Utils::TestArg( argc, argv, 'F' );
	if ( !pLevelName )
	{
		print_usage();
		exit(1);
	}

	//printf("Running LevelAssetLister on '%s'\n",pLevelName);

	char DataPath2[_MAX_PATH];

	sprintf( DataPath, "%s\\data\\", proj_path );
	sprintf( DataPath2, "%s\\data", proj_path );
	sprintf( LevelPath, "%s\\Levels\\%s", DataPath2, pLevelName );
	sprintf( SkyPath, "%s\\Levels\\%s_sky", DataPath2, pLevelName );
	sprintf( ShellPath, "%s\\Levels\\%s_shell", DataPath2, pLevelName );
	sprintf( QNFileName, "%s\\%s.qn", LevelPath, pLevelName );
	sprintf( ModelPathName, "%s\\Models\\", DataPath2 );

	if( platform == Utils::vPLATFORM_PS2 )
	{
		sprintf( OutFileName, "%spre.txt", LevelPath );
		sprintf( OutPedPartsFileName, "%spedpre.txt", LevelPath );

		strcpy(OutBspFileName, proj_path);
		strcat(OutBspFileName, "\\data\\Levels\\");
		strcat(OutBspFileName, pLevelName);
		strcat(OutBspFileName, "SCNPRE.txt");

		sprintf(OutColFileName, "%s\\data\\Levels\\%sCOLPRE.txt", proj_path, pLevelName);
	}
	else if( platform == Utils::vPLATFORM_XBOX )
	{
		sprintf( OutFileName, "%sprx.txt", LevelPath );
		sprintf( OutPedPartsFileName, "%spedprx.txt", LevelPath );

		strcpy(OutBspFileName, proj_path);
		strcat(OutBspFileName, "\\data\\Levels\\");
		strcat(OutBspFileName, pLevelName);
		strcat(OutBspFileName, "SCNPRX.txt");

		sprintf(OutColFileName, "%s\\data\\Levels\\%sCOLPRX.txt", proj_path, pLevelName);
	}
	else if( platform == Utils::vPLATFORM_NGC )
	{
		sprintf( OutFileName, "%sprg.txt", LevelPath );
		sprintf( OutPedPartsFileName, "%spedprg.txt", LevelPath );

		strcpy(OutBspFileName, proj_path);
		strcat(OutBspFileName, "\\data\\Levels\\");
		strcat(OutBspFileName, pLevelName);
		strcat(OutBspFileName, "SCNPRG.txt");

		sprintf(OutColFileName, "%s\\data\\Levels\\%sCOLPRG.txt", proj_path, pLevelName);
	}

//	theFileDatabase.RefreshFileList( DataPath2 );

	theOutputFile.Init( 2 * 1024 * 1024 );
	theOutputPedFile.Init( 2 * 1024 * 1024 );
	
	NumOutputtedFiles = 0;


	if (UpToDate(pLevelName))
	{
		// Nothing to do!
		return 0;
	}

	printf("Listing assets for level '%s'\n",pLevelName);

	// A few essential calls for the script code. These are done after the call to UpToDate rather
	// than at the start of main, because the allocation of the pools takes a while to execute, 
	// which would cause an unnecessary pause in the build process.
	Config::Init(0,NULL);
	Mem::Manager::sSetUp();

	Script::CreateSymbolHashTable();
	Script::AllocatePools();
	RegisterCFuncs();

	LoadRequiredQBFiles(pLevelName);

	// Load in the names of all the permanent anims into thePermAnimDatabase

	LoadAnimCommandsShouldWriteTo(PERM_ANIM_DATABASE);
	Script::RunScript("load_permanent_anims");

	// Output all the terrain sounds into the pre txt file
	Script::RunScript("LoadTerrain");

	// Existence check is just to avoid a warning printf
	if (ScriptExists("LoadAllParticleTextures"))
	{
		// Output all the particle textures into the pre txt file.
		Script::RunScript("LoadAllParticleTextures");
	}

	Script::RunScript("LoadCameras");
	Script::RunScript("LoadObjectAnims");

	// Load Gary's pedprofilelist.txt and pedassetlist.txt
	LoadPedProfileAndAssetLists( proj_path );

	// Output all the model files, as defined by the Model="" parameters in the
	// node array.
	WriteModelFileNames();

	// Output all the profiles, as defined by the Profile="" parameters in the
	// node array.
	WriteProfiles();

	LoadAnimCommandsShouldWriteTo(LEVEL_PRE_TXT_FILE);
	Script::RunScript("load_level_anims");


	// Make sure that both the male & female bits are always included no matter what,
	// because the game code pre-loads them.
	OutputProfile(&theOutputPedFile, 0x977b8a26); // random_male_profile
	OutputProfile(&theOutputPedFile, 0x7908a803); // random_female_profile

	// GJ:  As long as we're here, output the bozo ped, 
	// which is programmatically loaded in single-player games
	// (ideally, this wouldn't be hardcoded...)
	char BozoModelFileName[_MAX_PATH];
	strcpy(BozoModelFileName, ModelPathName);
	strcat(BozoModelFileName, "peds\\PedPros\\PedPro_Neversoft\\PedPro_Neversoft.skin");
	theOutputPedFile.WriteString( BozoModelFileName, false );
	theOutputPedFile.WriteString( GetPlatformExtension(), true );
	strcpy(BozoModelFileName, ModelPathName);
	strcat(BozoModelFileName, "peds\\PedPros\\PedPro_Neversoft\\PedPro_Neversoft.tex");
	theOutputPedFile.WriteString( BozoModelFileName, false );
	theOutputPedFile.WriteString( GetPlatformExtension(), true );
	strcpy(BozoModelFileName, ModelPathName);
	strcat(BozoModelFileName, "peds\\PedPros\\PedPro_Neversoft\\PedPro_Neversoft.cas");
	theOutputPedFile.WriteString( BozoModelFileName, false );
	theOutputPedFile.WriteString( GetPlatformExtension(), true );

	// GJ:  As long as we're here, output the crown model
	// which is programmatically loaded in multi-player games
	// (ideally, this wouldn't be hardcoded...)
	char CrownModelFileName[_MAX_PATH];
	strcpy(CrownModelFileName, ModelPathName);
	if( platform == Utils::vPLATFORM_PS2 )
	{
		strcat(CrownModelFileName, "crown\\crown.geom");
	}
	else
	{
		strcat(CrownModelFileName, "crown\\crown.mdl");
	}
	theOutputPedFile.WriteString( CrownModelFileName, false );
	theOutputPedFile.WriteString( GetPlatformExtension(), true );
	strcpy(CrownModelFileName, ModelPathName);
	strcat(CrownModelFileName, "crown\\crown.tex");
	theOutputPedFile.WriteString( CrownModelFileName, false );
	theOutputPedFile.WriteString( GetPlatformExtension(), true );
	strcpy(CrownModelFileName, ModelPathName);
	strcat(CrownModelFileName, "crown\\crown.col");
	theOutputPedFile.WriteString( CrownModelFileName, false );
	theOutputPedFile.WriteString( GetPlatformExtension(), true );
	// Find startup script LevName_Startup and scan it for instances of LoadSound
	char desiredScriptName[_MAX_PATH];
	strcpy(desiredScriptName, pLevelName);
	strcat(desiredScriptName, "_Startup");
	// Existence check is just to avoid a warning printf
	if (ScriptExists(desiredScriptName))
	{
		Script::RunScript(desiredScriptName);
	}

	// Level QB
	// Not on the PS2, as we don't have enough memory to load from .PRE, due to large temp buffers.
	if( platform != Utils::vPLATFORM_PS2 )
	{
		char msg[_MAX_PATH];
		strcpy(msg,LevelPath);
		strcat(msg,"\\");
		strcat(msg,pLevelName);
		strcat(msg,".qb");
		theOutputFile.WriteString( msg, true );
	}

	// Level scripts QB
	char p_scripts_file_name[_MAX_PATH];
	strcpy(p_scripts_file_name,LevelPath);
	strcat(p_scripts_file_name,"\\");
	strcat(p_scripts_file_name,pLevelName);
	strcat(p_scripts_file_name,"_scripts.qb");
	if (TestExistence(p_scripts_file_name,__LINE__))
	{
		theOutputFile.WriteString( p_scripts_file_name, true );
	}

	// Level sfx QB
	char p_sfx_file_name[_MAX_PATH];
	strcpy(p_sfx_file_name,LevelPath);
	strcat(p_sfx_file_name,"\\");
	strcat(p_sfx_file_name,pLevelName);
	strcat(p_sfx_file_name,"_sfx.qb");
	if (TestExistence(p_sfx_file_name,__LINE__))
	{
		theOutputFile.WriteString( p_sfx_file_name, true );
	}



	if ( !theOutputFile.Save( OutFileName ) )
	{
		Utils::Assert(0, "couldn't open file %s\n", OutFileName);
	}

	if ( !theOutputPedFile.Save( OutPedPartsFileName ) )
	{
		Utils::Assert(0, "couldn't open file %s\n", OutPedPartsFileName);
	}
	
	IoUtils::CVirtualOutputFile theOutputBSPFile;
	theOutputBSPFile.Init( 2 * 1024 * 1024 );

	for (int pass = 0;pass<3;pass++)
	{

		char	RootName[_MAX_PATH];
		if (pass == 0)
		{
			strcpy(RootName,LevelPath);
			strcat(RootName,"\\");
			strcat(RootName, pLevelName);
//			strcat(RootName,"\\");
//			strcat(RootName, pLevelName);
		}
		else if (pass == 1)
		{
			strcpy(RootName,SkyPath);
			strcat(RootName,"\\");
			strcat(RootName, pLevelName);
			strcat(RootName,"_sky");
//			strcat(RootName,"\\");
//			strcat(RootName, pLevelName);
//			strcat(RootName,"_sky");
		}
		else if (pass == 2)
		{
			strcpy(RootName,ShellPath);
			strcat(RootName,"\\");
			strcat(RootName, pLevelName);
			strcat(RootName,"_shell");
//			strcat(RootName,"\\");
//			strcat(RootName, pLevelName);
//			strcat(RootName,"_sky");
		}

		// Output Level SCN name, if file exists
		char BspName[_MAX_PATH];
		strcpy(BspName, RootName);
		bool scnExists = false;
		if( platform == Utils::vPLATFORM_PS2 )
		{
			strcat(BspName, ".geom.ps2NOT");   // Not outputting the .geom, as not enough memory
		}
		else if( platform == Utils::vPLATFORM_XBOX )
		{
			strcat(BspName, ".scn.xbx");
			scnExists = true;
		}
		else if( platform == Utils::vPLATFORM_NGC )
		{
			strcat(BspName, ".scn.ngc");
			scnExists = true;
		}
		if (scnExists && TestExistence(BspName,__LINE__,false))
		{
			theOutputBSPFile.WriteString( BspName, true );
		}

		// Output Level TEX name, if file exists
		char TexName[_MAX_PATH];
		strcpy(TexName,RootName);
		strcat(TexName,".tex.");
		strcat(TexName,Utils::GetPlatformExt(platform));
		if (TestExistence(TexName,__LINE__,false))
		{
			theOutputBSPFile.WriteString( TexName, true );
		}
	}

	if ( !theOutputBSPFile.Save( OutBspFileName ) )
	{
		Utils::Assert(0, "couldn't open file %s\n", OutBspFileName);
	}

	// Now the collision	

	IoUtils::CVirtualOutputFile theOutputColFile;
	theOutputColFile.Init( 2 * 1024 * 1024 );

	// Output Level col name, if file exists
	char ColName[_MAX_PATH];
	strcpy(ColName,LevelPath);
	strcat(ColName,"\\");
	strcat(ColName,pLevelName);
	strcat(ColName,".col.");
	strcat(ColName,Utils::GetPlatformExt(platform));	
	if (TestExistence(ColName,__LINE__))
	{
		theOutputColFile.WriteString( ColName, true );
	}
	
	if ( !theOutputColFile.Save( OutColFileName ) )
	{
		Utils::Assert(0, "couldn't open file %s\n", OutFileName);
	}

	return 0;
}















// UNUSED FUNCTIONS:

#if 0

#if USE_VIRTUAL_FILE
void AttemptOutputModelAnims(IoUtils::CVirtualOutputFile* pOutputFile, char *pModelName)
#else
void AttemptOutputModelAnims(fstream &out, char *pModelName)
#endif
{
	// compute the full name of the model file, minus extension
	char AnimsQbFile[_MAX_PATH];
	strcpy(AnimsQbFile, DataPath);
	strcat(AnimsQbFile, "scripts\\");
	strcat(AnimsQbFile, pModelName);
	strcat(AnimsQbFile, ".qb");

//	don't need to test existence, since we're going to open the fstream anyway
//	if (!TestExistence(AnimsQbFile))
//		return;

	ifstream qb(AnimsQbFile, ios::in | ios::binary | ios::nocreate);
	if (!qb.is_open())
	{
		return;
		Utils::Assert(0, "couldn't open file %s\n", AnimsQbFile);
	}
	qb.seekg(0, ios::end);
	int qb_size = qb.tellg();
	qb.seekg(0, ios::beg);
	char *qbBuf = new char[qb_size];
	qb.read(qbBuf, qb_size);
	qb.close();
	//printf("opened file %s, size %d\n", QbFileName, qb_size);
	
	uint8 *pToken = (uint8 *) qbBuf;
	
	// Scan through the qb file. (ModelName.qb)
	while (*pToken!=Script::ESCRIPTTOKEN_ENDOFFILE)
	{
			// Look for the name 'name', followed by equals, followed by string.
        
			if (*pToken==Script::ESCRIPTTOKEN_NAME && *(uint32*)(pToken+1)==0xa1dc81f9) // Checksum of 'name'
			{
					pToken=SkipToken(pToken);
					if (*pToken==Script::ESCRIPTTOKEN_EQUALS)
					{
							pToken=SkipToken(pToken);
							if (*pToken==Script::ESCRIPTTOKEN_STRING)
							{
									const char *pAnimName=(const char *)(pToken+5);
									// pAnimName is now a pointer to the animation name

									// ... we should now have the name of an animation
									char SkaFile[_MAX_PATH];
									strcpy(SkaFile, DataPath);
									strcat(SkaFile, pAnimName);

									if (TestExistence(SkaFile,__LINE__))
									{
#if USE_VIRTUAL_FILE
										Utils::Assert( 0, "Ska file %s not written out (see Gary)!", SkaFile );
										pOutputFile->WriteString( SkaFile, true );
#else
										out << SkaFile << "\n";
										out.flush();
#endif
									}
							}
					}
			}                                                       
                                
			pToken=SkipToken(pToken);
	}       
}

#endif