/*
	InterMaxLink.cpp
	This DLL provides functions for facilitating access to a shared memory
	block between InterMax.gup and other Windows applications

	aml - 9-3-03
*/

#include "InterMaxLink.h"
#include <windows.h>
#define  SHARED_MEM_SIZE  (1024 * 1024)
#define  CLIENT_MAX       (10)

enum
{
	CREC_INACTIVE        = 0x0001,	// Client record was unregistered (slot may be used by new clients registering)
	CREC_WAITINGREAD     = 0x0002,	// Client has not yet read the shared memory space after an update
	CREC_BROADCASTSENT   = 0x0004,	// Client has recieved a broadcast notification
};

struct ClientRecord
{
	char name[256];									// Name of the client application

	DWORD idProcess;								// Handle to the Process this client runs in
	int  (*cbNotify)(void*,DWORD,DWORD,int,int);	// Function called to inform data is ready to be retrieved
													// IML will block write attempts until all clients acknowledge
													// reading the packet

	int   broadcastID;								// notifyCode sent from a broadcast
	int   bufferSize;								// size of buffer to be retrieved
	int   arg1;										// User defined argument for message 1
	int   arg2;										// User defined argument for message 2
	int   nBlocksRemaining;							// Number of blocks remaining in last multi block transfer

	void* blockMem;									// Reconstruct memory for multiblock transfer
													// (WARNING!  This memory should only be accessed from the Process
													//  recieving the multiblock transfer)

	int   multiblockID;								// Multiblock notify code
	int   multiBufferSize;							// Size of the full multibuffer

	DWORD flags;

	/*
	ClientRecord()
	{
		blockMem = NULL;
	}

	~ClientRecord()
	{
		if (blockMem)
			free(blockMem);
	}
	*/
};

bool gbMultiBlockAckWait = false;				// True if multiblock transfer initiator is waiting for an ack
												// processing should not continue on sender side until ack is recieved

bool gbMultiBlockRecv = false;					// True if this client is in the middle of recieving a multiblock packet



#pragma data_seg(".shared")

ClientRecord crec[CLIENT_MAX] = {""};			// Need to assign this something or else the compiler doesn't
												// add it to the .shared segment for some reason

unsigned char shared[SHARED_MEM_SIZE] = {0};	// 1 MB block of shared memory
int lastClientID = 0;
int memOwnerID = -1;							// ID of client that currently has memory locked

bool bMultiBlockTransLock = false;				// True if a multiblock transfer is currently occurring
												// only 1 multiblock transfer can go on at any given time

int gThisClient;								// Defines the ID (from FindClient for the attached client making)
												// this call into the DLL

#pragma data_seg()

// Ensures that any apps that disconnect from the DLL, are no longer active in our client list
void ClientDisconnect()
{
	DWORD idProcess = GetCurrentProcessId();

	for(int i = 0; i < lastClientID; i++)
	{
		if (!(crec[i].flags & CREC_INACTIVE))
		{
			if (crec[i].idProcess == idProcess)
			{
				crec[i].flags |= CREC_INACTIVE;
				OutputDebugString("ClientDisconnect:  Forced disconnect from InterMaxLink.dll\n");
			}
		}
	}
}

BOOL WINAPI DllMain(HINSTANCE hInst, DWORD fdwReason, LPVOID lpvReserved)
{
	switch(fdwReason)
	{
	case DLL_PROCESS_ATTACH:
		//GetCurrentProcess()
		break;

	case DLL_PROCESS_DETACH:
		ClientDisconnect();
		break;

	case DLL_THREAD_ATTACH:

		break;

	case DLL_THREAD_DETACH:

		break;
	}

	return TRUE;
}

int FindClient()
{
	DWORD idProcess = GetCurrentProcessId();

	for(int i = 0; i < lastClientID; i++)
	{
		if (crec[i].idProcess == idProcess)
			return i;
	}

	return -1;
}

unsigned int imlGetBlockSize()
{
	return SHARED_MEM_SIZE;
}

bool imlGetBlockOwner(char* str, int size)
{
	if (memOwnerID == -1)
	{
		OutputDebugString("imlGetBlockOwner: Failed.  Memory is not currently locked\n");
		return false;
	}

	strncpy(str, crec[memOwnerID].name, size);
	return true;
}

int imlRegisterClient(char* name, int (*cbNotify)(void*,DWORD,DWORD,int,int))
{
	// Scan for any inactive clients that we can overtake
	int insertID = -1;

	if (FindClient() != -1)
	{
		OutputDebugString("imlRegisterClient: Failed.  Only one client may be registered per Process\n");
		return -1;
	}

	for(int i = 0; i < lastClientID; i++)
	{
		if (crec[i].flags & CREC_INACTIVE)
		{
			insertID = i;
			break;
		}
	}

	// If we were unable to find an inactive client to take over add a new one
	if (insertID == -1)
	{
		if (lastClientID == CLIENT_MAX)
		{
			OutputDebugString("imlRegisterClient: Failed.  Maximum clients reached\n");
			return -1;
		}

		insertID = lastClientID;
		lastClientID++;
	}

	strcpy(crec[insertID].name, name);
	crec[insertID].cbNotify         = cbNotify;
	crec[insertID].idProcess        = GetCurrentProcessId();
	crec[insertID].broadcastID      = 0;
	crec[insertID].bufferSize       = 0;
	crec[insertID].arg1             = 0;
	crec[insertID].arg2             = 0;
	crec[insertID].nBlocksRemaining = 0;
	crec[insertID].flags            = 0;
	crec[insertID].blockMem         = NULL;
	crec[insertID].multiblockID     = 0;
	crec[insertID].multiBufferSize  = 0;

	gThisClient = insertID;
	return insertID;
}

bool imlUnregisterClient()
{
	DWORD hCurProcess = GetCurrentProcessId();

	for(int i = 0; i < lastClientID; i++)
	{
		if (crec[i].idProcess == hCurProcess)
		{
			if (crec[i].flags & CREC_INACTIVE)
			{
				OutputDebugString("imlUnregisterClient: Failed.  Client already unregistered\n");
				return false;
			}

			crec[i].flags |= CREC_INACTIVE;
			return true;
		}
	}

	return false;
}

bool imlUnregisterClientByID(int ID)
{
	if (ID >= lastClientID)
	{
		OutputDebugString("imlUnregisterClientByID: Failed.  Client not registered\n");
		return false;
	}

	if (crec[ID].flags & CREC_INACTIVE)
	{
		OutputDebugString("imlUnregisterClientByID: Failed.  Client already unregistered\n");
		return false;
	}

	crec[ID].flags |= CREC_INACTIVE;
	return true;
}

bool imlUnregisterClientByName(char* name)
{
	for(int i = 0; i < lastClientID; i++)
	{
		if (strcmp(name, crec[i].name) == 0)
		{
			if (!(crec[i].flags & CREC_INACTIVE))
			{
				crec[i].flags |= CREC_INACTIVE;
				return true;
			}
		}
	}

	OutputDebugString("imlUnregisterClientByName: Failed.  Client name not registered\n");
	return false;
}

bool imlIsClientRegistered(char* name)
{
	for(int i = 0; i < lastClientID; i++)
	{
		if (strcmp(name, crec[i].name) == 0)
		{
			if (!(crec[i].flags & CREC_INACTIVE))
				return true;
		}
	}

	return false;
}

// Note: this will stall until all clients have processed the last message with imlUpdate
void imlBroadcastClients(DWORD notifyCode, DWORD argSize, int arg1, int arg2)
{
	int clientMe = FindClient();

	for(int i = 0; i < lastClientID; i++)
	{
		// Don't send broadcast messages to ourself
		if (i != clientMe)
		{
			if (!(crec[i].flags & CREC_INACTIVE))
			{
				// Wait for any existing broadcast message to be processed on this client first
				while(crec[i].flags & CREC_BROADCASTSENT) {}

				crec[i].broadcastID = notifyCode;
				crec[i].bufferSize  = argSize;
				crec[i].arg1        = arg1;
				crec[i].arg2        = arg2;
				crec[i].flags |= CREC_BROADCASTSENT;
			}
		}
	}
}

void* imlLockSection(int clientID)
{
	if (clientID == -1)
	{
		clientID = FindClient();

		if (clientID == -1)
			OutputDebugString("imlLockSection: Failed. Client did not register\n");
	}

	if (memOwnerID != -1)
	{
		OutputDebugString("imlLockSection: Failed. Another client has the memory locked\n");
		return NULL;
	}

	// Ensure that all clients have read the memory before allowing another lock
	for(int i = 0; i < lastClientID; i++)
	{
		if (!crec[lastClientID].flags & CREC_INACTIVE)
		{
			if (crec[lastClientID].flags & CREC_WAITINGREAD)
			{
				OutputDebugString("imlLockSection: Failed. Not all clients have read the current memory block\n");
				return NULL;
			}
		}
	}

	memOwnerID = clientID;
	return shared;
}

// NotifyCode defines the type of data in the shared mem section and is broadcast to clients
bool imlUnlockSection(DWORD notifyCode, DWORD size, int arg1, int arg2)
{
	if (size > SHARED_MEM_SIZE)
	{
		OutputDebugString("imlUnlockSection: Failed mem size exceeds allowable size (MEMORY OVERRUN!!)\n");
		return false;
	}

	// Mark connected clients into waiting read state
	int initiatingClient = FindClient();
	int i;

	if (initiatingClient != memOwnerID)
	{
		OutputDebugString("imlUnlockSection: Failed.  Calling Process does not own locked memory\n");
		return false;
	}

	for(i = 0; i < lastClientID; i++)
	{
		if (i != memOwnerID)
			crec[i].flags |= CREC_WAITINGREAD;
	}

	// Signal active connected clients that memory is ready to be accessed
	imlBroadcastClients(notifyCode, size, arg1, arg2);
	memOwnerID = -1;
	return true;
}

bool imlAreClientsUpdated()
{
	for(int i = 0; i < lastClientID; i++)
		if (crec[i].flags & CREC_WAITINGREAD)
			return false;

	return true;
}

// Called when a client has finished reading the memory block
bool imlBlockAck()
{
	int clientID = FindClient();

	if (clientID == -1)
	{
		OutputDebugString("imlBlockAck: Failed.  Client not registered.\n");
		return false;
	}

	if (!(crec[clientID].flags & CREC_WAITINGREAD))
	{
		OutputDebugString("imlBlockAck: Warning!: Already got Ack for this Process.\n");
		return false;
	}

	crec[clientID].flags &= ~CREC_WAITINGREAD;

	if (gbMultiBlockRecv)
	{
		// If we just recieved a multiblock an acknowledgement must be returned
		imlBroadcastClients(MULTIBLOCK_TRANS_ACK_ID, 0, 0, 0);
	}

	return true;
}

bool imlWaitForReads(int timeout)
{
	if (timeout == 0)
	{
		// Block until all clients have read data
		while(!imlAreClientsUpdated()) {}
		return true;
	}
	else
	{
		DWORD dwTimeout = GetTickCount() + timeout;

		while(!imlAreClientsUpdated() && GetTickCount() < dwTimeout) {}

		if (GetTickCount() >= dwTimeout)
			return false;

		return true;
	}
}

// This function is called periodically by each client to check if another client is signaling it
// since the client initiating a broadcast may not have read access to another client's memory space
// to call its notify callback
void imlUpdate()
{
	// Only process updates on this instance
	int i = gThisClient;

	//for(int i = 0; i < lastClientID; i++)
	{
		if (!(crec[i].flags & CREC_INACTIVE))
		{
			// If a broadcast message was sent call the client's notify function
			if (crec[i].flags & CREC_BROADCASTSENT)
			{
				// Handle any multiblock transfers
				switch(crec[i].broadcastID)
				{
				case MULTIBLOCK_TRANS_HEADER_ID:
					{
						if (crec[i].nBlocksRemaining > 0 || gbMultiBlockRecv)
						{
							OutputDebugString("imlUpdate: Warning!  Two multiblock transfers are occurring simultaneously SHOULDN'T BE POSSIBLE HALT\n");
							//__asm int 3;
						}
						else
						{
							gbMultiBlockRecv = true;

							crec[i].nBlocksRemaining = crec[i].arg1;
							crec[i].multiblockID     = crec[i].arg2;

							// Allocate the final size of the buffer
							if (crec[i].blockMem)
								free(crec[i].blockMem);

							crec[i].blockMem = malloc(crec[i].bufferSize);
						}

						imlBlockAck();
					}
					break;

				case MULTIBLOCK_TRANS_BLOCK_ID:
					{
						if (crec[i].nBlocksRemaining == 0)
						{
							OutputDebugString("imlUpdate: Warning!  multiblock recieved past expected number of blocks SHOULDN'T BE POSSIBLE HALT\n");
							//__asm int 3;
						}

						// Copy the shared memory into this block
						// arg1 is the block ID
						unsigned char* pMem = (((unsigned char*)crec[i].blockMem) + crec[i].arg1 * SHARED_MEM_SIZE);
						memcpy(pMem, shared, SHARED_MEM_SIZE);

						crec[i].nBlocksRemaining--;
						if (crec[i].nBlocksRemaining == 0)
						{
							crec[i].cbNotify(crec[i].blockMem,crec[i].multiblockID, crec[i].multiBufferSize, 0, 0);
						}
						else
							imlBlockAck();
					}
					break;

				case MULTIBLOCK_TRANS_ACK_ID:
					{
						gbMultiBlockAckWait = false;
					}
					break;

				default:
					{
						crec[i].cbNotify(shared,crec[i].broadcastID, crec[i].bufferSize, crec[i].arg1, crec[i].arg2);
						crec[i].flags &= ~CREC_BROADCASTSENT;
					}
					break;
				}
			}
		}
	}
}

// imlTransferBlock is responsible for sending a block of data (possibly larger than the buffer size)
// to all connected clients
bool imlTransferBlock(int ID, void* pData, int size)
{
	if (bMultiBlockTransLock)
	{
		OutputDebugString("imlTransferBlock: Failed!  A multiblock transfer is currently underway\n");
		return false;
	}

	bMultiBlockTransLock = true;

	if (size < SHARED_MEM_SIZE)
	{
		// Simple transfer/broadcast  (all clients should already be registered)
		unsigned char* mem;

		// Wait to get a lock on shared memory (could be locked by another client)
		while(mem = (unsigned char*)imlLockSection()) {}
		memcpy(mem, pData, size);

		imlUnlockSection(ID, size);
	}
	else
	{
		int   nBlocks = size / SHARED_MEM_SIZE;
		float fBlocks = (float)size / (float)SHARED_MEM_SIZE;

		// Roundup number of required blocks
		if (fBlocks > (float)nBlocks)
			nBlocks++;

		// Send simple broadcast to inform of multiblock recv
		imlBroadcastClients(MULTIBLOCK_TRANS_HEADER_ID, size, nBlocks, ID);

		// Split the data into chunks sending it a meg at a time
		// imlUpdate is
		unsigned char* srcMem = (unsigned char*)pData;
		int nBytesRemaining = size;							// Bytes remaining in transfer
		int blockSize;
		unsigned char* mem;

		for(int i = 0; i < nBlocks; i++)
		{
			// Wait to get lock
			while(mem = (unsigned char*)imlLockSection())

			if (nBytesRemaining < SHARED_MEM_SIZE)
				blockSize = nBytesRemaining;
			else
				blockSize = SHARED_MEM_SIZE;

			memcpy(mem, srcMem, SHARED_MEM_SIZE);

			imlUnlockSection(MULTIBLOCK_TRANS_BLOCK_ID, blockSize, i, size);

			srcMem += SHARED_MEM_SIZE;
			nBytesRemaining -= SHARED_MEM_SIZE;
		}
	}

	gbMultiBlockAckWait = true;
	while(gbMultiBlockAckWait) { imlUpdate(); }

	bMultiBlockTransLock = false;
	return true;
}

void imlEnumClients(bool (*func)(char* name))
{
	for(int i = 0; i < lastClientID; i++)
		if (!func(crec[i].name))
			return;
}
