/*
	DebugLink.cpp
	DLL for facilitating communication of small shared memory data
	between applications

	aml - 8-14-03
*/

#include <windows.h>
#include <stdio.h>
#include "DebugLink.h"
#include "stack.h"

#define DBGREC_MAX 10000

HINSTANCE ghInstance;
DebugState dbs;

enum DebugType
{
	DBGTYPE_STATE,
	DBGTYPE_MSG,
	DBGTYPE_EXITSTATE,
};

struct DebugRecord
{
	DebugType type;
	
	union
	{
		DebugState state;
		char       message[256];
	};
};

DebugRecord dbr;

#pragma data_seg(".shared")
// Note for some reason VC seems to require that each element be initialized
// or it doesn't actually get placed into the defined data segment

CRITICAL_SECTION gcs = { 0,0,0,0,0 };
DebugState gdbgState = dbs;
char       gMessage[2048] = "";
bool       gbDbgInit = false;
void (*UpdateState)(DebugState state) = NULL;
void (*UpdateMsg)(char* msg) = NULL;
int  refCount = 0;
DebugRecord dbgList[DBGREC_MAX] = {DBGTYPE_STATE};
int dbgRecStart = 0;
int dbgRecEnd   = 0;
bool dbgRecLooped = false;
#pragma data_seg()

BOOL WINAPI DllMain(HINSTANCE hInst, DWORD fdwReason, LPVOID lpvReserved)
{
	switch(fdwReason)
	{
	case DLL_PROCESS_ATTACH:
		if (refCount == 0)
			InitializeCriticalSection(&gcs);
			
		refCount++;
		break;

	case DLL_PROCESS_DETACH:
		refCount--;

		if (refCount == 0)
			DeleteCriticalSection(&gcs);
		break;
	}

	ghInstance = hInst;
	return TRUE;
}

void PostDebugState(DebugState* state)
{
	// Must ensure that app completes doing its stuff before
	// it gets the next update
	//EnterCriticalSection(&gcs);

	gdbgState = *state;

	dbgRecEnd++;
	if (dbgRecEnd == DBGREC_MAX)
	{
		dbgRecEnd   = 0;
		dbgRecLooped = true;
	}

	if (dbgRecLooped)
	{
		if (dbgRecEnd + 1 == DBGREC_MAX)
			dbgRecStart = 0;

		dbgRecStart = dbgRecEnd + 1;
	}

	dbgList[dbgRecEnd].type  = DBGTYPE_STATE;
	dbgList[dbgRecEnd].state = *state;

	//if (UpdateState)
	//	UpdateState(gdbgState);

	//LeaveCriticalSection(&gcs);
}

void PostExitDebugState(DebugState* state)
{
	// Must ensure that app completes doing its stuff before
	// it gets the next update
	//EnterCriticalSection(&gcs);

	gdbgState = *state;

	dbgRecEnd++;
	if (dbgRecEnd == DBGREC_MAX)
	{
		dbgRecEnd   = 0;
		dbgRecLooped = true;
	}

	if (dbgRecLooped)
	{
		if (dbgRecEnd + 1 == DBGREC_MAX)
			dbgRecStart = 0;

		dbgRecStart = dbgRecEnd + 1;
	}

	dbgList[dbgRecEnd].type  = DBGTYPE_EXITSTATE;
	dbgList[dbgRecEnd].state = *state;

	//if (UpdateState)
	//	UpdateState(gdbgState);

	//LeaveCriticalSection(&gcs);
}

DebugState GetDebugState()
{
	return gdbgState;
}

void PostDebugMessage(char* str)
{
	//EnterCriticalSection(&gcs);
	strcpy(gMessage, str);

	dbgRecEnd++;
	if (dbgRecEnd == DBGREC_MAX)
	{
		dbgRecEnd   = 0;
		dbgRecLooped = true;
	}

	if (dbgRecLooped)
	{
		if (dbgRecEnd + 1 == DBGREC_MAX)
			dbgRecStart = 0;

		dbgRecStart = dbgRecEnd + 1;
	}

	dbgList[dbgRecEnd].type  = DBGTYPE_MSG;
	strcpy(dbgList[dbgRecEnd].message, gMessage);

	//if (UpdateMsg)
	//	UpdateMsg(gMessage);

	//LeaveCriticalSection(&gcs);
}

char* GetDebugMessage()
{
	return gMessage;
}

void SetDebugInit(bool bVal)
{
	gbDbgInit = bVal;
}

bool IsDebugInit()
{
	return gbDbgInit;
}

bool SetUpdateStateCB(void (*func)(DebugState))
{
	if (func != NULL && UpdateState)
		return false;

	UpdateState = func;	
	return true;
}

bool SetUpdateMsgCB(void (*func)(char* msg))
{
	if (func != NULL && UpdateMsg)
		return false;

	UpdateMsg = func;
	return true;
}

bool DumpLog(char* filename)
{
	EnterCriticalSection(&gcs);

	char file_name[1024];

	if (filename)
		strcpy(file_name, filename);
	else
		strcpy(file_name, "c:\\DebugLink.log");

	FILE* fp = fopen(file_name, "w");

	if (!fp)
		return false;

	int listPos = dbgRecStart;

	do
	{
		switch(dbgList[listPos].type)
		{
		case DBGTYPE_STATE:
			fprintf(fp, "%s(%i): Entered %s\n", dbgList[listPos].state.lastFile,
				                                dbgList[listPos].state.lastLine,
											    dbgList[listPos].state.lastFunction);
			break;

		case DBGTYPE_MSG:
			fprintf(fp, "MSG: %s\n", dbgList[listPos].message);
			break;

		case DBGTYPE_EXITSTATE:
			fprintf(fp, "%s(%i): Exited %s\n", dbgList[listPos].state.lastFile,
				                               dbgList[listPos].state.lastLine,
										       dbgList[listPos].state.lastFunction);
			break;
		}

		listPos++;

		// Wrap list
		if (listPos == DBGREC_MAX)
			listPos = 0;
	}
	while( listPos != dbgRecEnd + 1);

	fclose(fp);
	memset(dbgList, 0, sizeof(DebugRecord) * DBGREC_MAX);
	dbgRecStart = 0;
	dbgRecEnd   = 0;

	LeaveCriticalSection(&gcs);
	return true;
}

void dbgPrintf(const char* text, ...)
{
	char buf[2048];

	va_list args;
	va_start( args, text );
	vsprintf( buf, text, args);
	va_end( args );

	PostDebugMessage(buf);
}
