/*
	Waveform to PS2 VAG format converter
	AML - 3-19-01
*/

#include <windows.h>
#include <conio.h>
#include <mmreg.h>
#include <msacm.h>
#include <stdio.h>
#include <direct.h>
#include <io.h>
#include <process.h>
#include <sys/stat.h>
#include <fstream.h>
#include "wavio.h"
#include "vagio.h"
#include "GenCrc.h"

#define MM_OK     0
#define MM_ERROR -1

struct FindData
{
	char* srcdir;
	char* dstdir;
};

#define PLATFORM_PS2	0
#define PLATFORM_XBOX	1
#define PLATFORM_NGC	2

bool bLooped=false;			// True if the wave file is looped

// Command line Switches
bool 	bRecurse=false;
bool 	bNameResample=false;
bool 	bLoop=false;					// Force looped vag
bool 	bNoLoop=false;					// Force not looped vag
bool 	bWaveLoop=false;				// loop if the wave is looped
bool 	bVerbose=false;
bool 	bBitdepth=false;				// true if custom force bit conversion
bool 	bResample=false;				// true if custom force resample
bool 	bMono=false;					// true if custom force to 1 channel
bool 	bStereo=false;					// true if custom force to 2 channels
bool 	bNoVAG=false;					// true if wave should be converted to another wav (not vag)
bool 	bCheckDate=false;				// true if the date of the file should be checked
bool 	bWarnRate=false;				// true if user should be warned if not sampled at given rate
bool 	bUseDefaultRate=false;			// true if default sample rate should be used (-o) when option -s is used
bool 	bOutputWMA=false;				// true if wav->wma conversion is required over the default (wav->pcm). Only valid for Xbox
bool 	bOutputAllChunks=false;			// true if all chunks should be copied from the wav to the pcm file (for sampler chunk for looping). Only valid for Xbox
bool 	bBatchMode=false;				// true if running from a batch file
bool	bSaveLipSyncData=false;			// true if we want to save out average waveform amplitude data for lipsync


int  	platform=PLATFORM_PS2;			// Platform to use based on -p parameter.
int  	defaultRate;					// Default rate to use if -s and -o are specified
int  	custSampleRate;					// custom sample rate
int  	custBitDepth;					// custom bit depth
int  	WarnRate;						// sample rate that all samples should be if using -w option
int  	vagMode=VAGMODE_STD;			// VAG compression mode

char*	pWMAEncodingParams=NULL;		// Optional ecoding patameters for the WMA encoder (Xbox specific)
char*	pXboxADPCMEncodingParams=NULL;	// Optional ecoding patameters for the Xbox ADPCM encoder (Xbox specific)

// Is file 2 newer than file one
bool FileIsNewer( char* file1, char* file2 )
{
	HANDLE file_handle_1, file_handle_2;
	WIN32_FIND_DATA find_data_1, find_data_2;
	
	file_handle_1 = FindFirstFile( file1, &find_data_1 );  
	if( file_handle_1 == INVALID_HANDLE_VALUE )
	{
		return true;
	}

	file_handle_2 = FindFirstFile( file2, &find_data_2 );  
	if( file_handle_2 == INVALID_HANDLE_VALUE )
	{
		FindClose( file_handle_1 );
		return false;
	}
	
	FindClose( file_handle_1 );
	FindClose( file_handle_2 );
	return( CompareFileTime( &find_data_2.ftLastWriteTime, &find_data_1.ftLastWriteTime ) > 0 );
}

void LCase(char* str)
{
	int len=strlen(str);

	for(int i=0;i<len;i++)
		str[i]=tolower(str[i]);
}

// Makes directories passed in as string hierarchy
void MDir(char* dirs, bool check)
{
	char curpath[1024];
	char curdir[1024];
	char drive[80];

	_getcwd(curpath,1023);
	if (check && (_chdir(dirs) == 0))
	{
		_chdir(curpath);
		return;
	}

	_splitpath(dirs,drive,NULL,NULL,NULL);
	strcat(drive,"\\");
	_chdir(drive);
	
	char* pos;
	char* dpos=dirs;

	while((pos=strstr(dpos,"\\"))!=0)
	{
		char* cpos = dpos;
		int   i    = 0;

		while(cpos!=pos)
		{
			curdir[i++]=*cpos;
			cpos++;
		}

		curdir[i]=0;

		// Make dir
		if (strstr(curdir,":")==0)
		{
			_mkdir(curdir);
			_chdir(curdir);
		}

		dpos = pos+1;
	}

	// Make final dir (dpos)
	_mkdir(dpos);
	_chdir(curpath);
}

void Find(char* path,char* prefix,void (*fpCallback)(char* filename,void* data),void* pData)
{
	_finddata_t fdata;

	int success = _chdir(path);
	if( success == -1 )
	{
		// Failed to switch to teh target path, so certainly no files to find...
		return;
	}

	char WorkingDir[256];
	_getcwd(WorkingDir,255);

	long hFindFirst=_findfirst("*.*",&fdata);

	if (hFindFirst==-1)
	{
		_findclose( hFindFirst );
		return;
	}

	int  hFile=0;

	while(hFile!=-1)
	{
		if (fdata.attrib & _A_SUBDIR)
		{
			if (strcmp(fdata.name,".")==0 ||
				strcmp(fdata.name,"..")==0)
			{
				hFile=_findnext(hFindFirst,&fdata);
				continue;
			}

			Find(fdata.name,prefix,fpCallback,pData);
			_chdir(WorkingDir);
		}

		char strLName[256];
		char strLPrefix[256];

		strcpy(strLName,fdata.name);
		strcpy(strLPrefix,prefix);

		LCase(strLName);
		LCase(strLPrefix);

		if (strstr(strLName,strLPrefix))
			fpCallback(fdata.name,pData);

		hFile=_findnext(hFindFirst,&fdata);
	}
	_findclose( hFindFirst );
}

bool MakeWaveformAmplitudeChecksumFile(char *srcFamFilename)
{
	// Open source file
	FILE* finp=fopen(srcFamFilename,"rb");
	if (!finp)
	{
		printf("ERROR: Failed to open '%s'.\n",srcFamFilename);
		return false;
	}

	// Generate new filename and dir
	char destFile[256];
	char destDir[256];
	unsigned int destFileCRC = 0;

	char *p_env_path = getenv( "PROJ_ROOT" );
	if( p_env_path == NULL )
	{
		printf( "MakeWaveformAmplitudeChecksumFile requires that the environment variable PROJ_ROOT is set.\n" );
		return false;
	}
	strcpy(destDir, p_env_path);
	strcat(destDir, "\\Data\\Fam\\");

	// Get the filename
	strcpy(destFile, srcFamFilename);
	for (int idx = strlen(srcFamFilename) - 1; idx >=0; idx--)
	{
		// Find start of filename
		if ((srcFamFilename[idx] == '\\') || (srcFamFilename[idx] == '/'))
		{
			strcpy(destFile, &srcFamFilename[idx + 1]);	
			break;
		}
	}

	// Find end and checksum
	char *p_endFile = strrchr(destFile, '.');
	if (p_endFile)
	{
		*p_endFile = '\0';
	}
	destFileCRC = GenerateCRC(destFile);
	sprintf(destFile,"%s%08x.fam", destDir, destFileCRC);

	// Open new file (and make sure directory is made)
	_mkdir(destDir);
	FILE* foutp=fopen(destFile,"wb");
	if (!foutp)
	{
		printf("ERROR: Failed to open '%s'.\n",destFile);
		fclose(finp);
		return false;
	}

	// Copy data
	int bytesRead;
	unsigned char buffer[4096];
	while (bytesRead = fread(buffer, 1, 4096, finp))
	{
		//printf("Writing %d bytes\n", bytesRead);
		fwrite(buffer, 1, bytesRead, foutp);
	}

	fclose(finp);
	fclose(foutp);
	return true;
}

#define FRAMES_PER_SEC	(60)		// Somehow, we need to change this for PAL
bool GenerateWaveformAmplitudeData(char *wavFilename, WAVEFORMATEX *p_wavfmt, void *p_wavdata, int data_size, int frames_per_second)
{
	bool makeChecksumFile = false;

	if (p_wavfmt->wFormatTag != WAVE_FORMAT_PCM)
	{
		printf("GenerateWaveformAmplitudeData(): %s isn't a PCM Format waveform.\n");
		return false;
	}

	if (p_wavfmt->wBitsPerSample != 16)
	{
		printf("GenerateWaveformAmplitudeData(): %s PCM data isn't 16-bit\n");
		return false;
	}

	// Generate new filename and dir
	char destFile[256];
	char destDir[256];

	strcpy(destFile, wavFilename);

	// Change wav directory to fam
	char *famdir = strstr(destFile, "\\wav\\");
	if (!famdir) famdir = strstr(destFile, "\\Wav\\");
	if (!famdir) famdir = strstr(destFile, "\\WAV\\");
	if (famdir)
	{
		famdir[1] = 'f';
		famdir[2] = 'a';
		famdir[3] = 'm';
		makeChecksumFile = true;
	}

	// And change filetype
	destFile[strlen(destFile) - 3] = 'f';
	destFile[strlen(destFile) - 2] = 'a';
	destFile[strlen(destFile) - 1] = 'm';

	if (bVerbose)
	{
		printf("Making fam file %s\n", destFile);
	}

	// Create directory string
	strcpy(destDir, destFile);
	char *dirend = strrchr(destDir, '\\');
	if (dirend)
	{
		*dirend = '\0';
	}

	// Open new file (and make sure directory is made)
	MDir(destDir, true);
	FILE* fp=fopen(destFile,"wb");
	if (!fp)
	{
		printf("ERROR: Failed to write '%s'.\n",destFile);
		return false;
	}

	int totalSamples = data_size / sizeof(short);
	double fSamplesPerFrame = ((double) p_wavfmt->nSamplesPerSec) / frames_per_second;	// Get float amount so we don't drift
	int samplesPerFrame = (int) fSamplesPerFrame;

	int numFrames = (totalSamples + (double) (samplesPerFrame * p_wavfmt->nChannels) - 1.0f ) /	// Round up
		            (double) (samplesPerFrame * p_wavfmt->nChannels);

	int samplesProcessed = 0;

	if (bVerbose)
	{
		printf("Samples per sec: %d; Num channels: %d\n", p_wavfmt->nSamplesPerSec, p_wavfmt->nChannels);
		printf("Total samples: %d; Samples per frame: %f (%d); Num frames: %d\n", totalSamples / p_wavfmt->nChannels, fSamplesPerFrame, samplesPerFrame, numFrames);
	}

	//short *p_frame_samples = (short *) p_wavdata;
	short *p_sample = (short *) p_wavdata;

	for (int frame = 0; frame < numFrames; frame++)
	{
		unsigned int sample_total = 0;
		int samplesThisFrame = samplesPerFrame;

		// Go through all the samples for a frame
		for (int i = 0; i < samplesPerFrame; i++)
		{
			for (int ch = 0; ch < p_wavfmt->nChannels; ch++)
			{
				sample_total += abs(*(p_sample++));
				samplesProcessed++;
			}

			// if last frame, make sure it doesn't go over the buffer size
			if (samplesProcessed >= totalSamples)
			{
				if (frame != (numFrames - 1))
				{
					printf("GenerateWaveformAmplitudeData(): PCM data smaller than expected.  Aborted.\n");
					fclose(fp);
					return false;
				}
				samplesThisFrame = i + 1;
				break;
			}
		}

		// Find average amplitude and write
		unsigned int average = sample_total / (samplesThisFrame * p_wavfmt->nChannels);
		average = average >> 7;	// It could be up to 15 bits, but want to shrink it down to 8
		unsigned char c_avg = average;
#if 1
		fputc(c_avg, fp);
#else
		if ((frame & 0x3) == 0)
		{
			fputc(c_avg, fp);
		}
#endif

		// Check for drift and adjust
		int drift = ((int) (fSamplesPerFrame * (double) (frame + 1))) - samplesProcessed;
		if ((frame != (numFrames - 1)) && drift)
		{
			if (bVerbose)
			{
				printf("Drift in frame %d: adjusting by %d samples\n", frame, drift / p_wavfmt->nChannels);
			}
			p_sample += drift;
			samplesProcessed += drift;
		}
	}

	fclose(fp);

	// Make checksum version of file
	if (makeChecksumFile)
	{
		MakeWaveformAmplitudeChecksumFile(destFile);
	}

	return true;
}

bool ConvertWAV(char* srcFilename, char* dstFilename)
{
	WAVEFORMATEX* srcfmt;
	SamplerChunk* schunk=NULL;
	int           schunkSize;				// Size of the sampler chunk including extended data
	bool          bHasSamplerChunk = false;
	bool		  saveThisLipSyncData = bSaveLipSyncData;
	void*         srcdata;
	void*		  dstdata;
	int           srcsize;
	DWORD		  dstsize;
	
	char *p_env_path = getenv( "PROJ_ROOT" );
	if( p_env_path == NULL )
	{
		printf( "NSWav2Vag requires that the environment variable PROJ_ROOT is set.\n" );
		exit( 1 );
	}

	// Check for lipsync name
	char lowercaseName[256];
	int sidx = 0;
	do
	{
		lowercaseName[sidx] = tolower(srcFilename[sidx]);
	} while (srcFilename[sidx++]);
	if (strstr(lowercaseName,"_lip.") || strstr(lowercaseName,"_lip_"))
	{
		saveThisLipSyncData = true;
	}

	if( platform == PLATFORM_XBOX )
	{
		// Spawn out to xbadpcmencode, or wm8eutil, depending on the file type.
		static char *args[8];
		char executable_path[256];

		strcpy( executable_path, p_env_path );
		
		if( bOutputWMA )
		{
			strcat( executable_path, "\\bin\\win32\\wm8eutil.exe" );
		
			args[0] = executable_path;
			args[1] = "-input";
			args[2] = srcFilename;
			args[3] = "-output";
			args[4] = dstFilename;
			args[5] = "-a_setting";
			args[6] = (	pWMAEncodingParams ) ? pWMAEncodingParams : "48_44_1";
			args[7] = NULL;
			spawnvp( P_WAIT, args[0], &args[0] );
			if (!saveThisLipSyncData)
				return true;
		}
		else
		{
			strcat( executable_path, "\\bin\\win32\\xbadpcmencode.exe" );
		
			if( pXboxADPCMEncodingParams )
			{
				args[0] = executable_path;

				int	nextarg = 1;
//				args[nextarg++] = "-Ob";			// Optimal per-block encoding.
				if( bOutputAllChunks )
					args[nextarg++] = "-c";			// Output all chunks.
				args[nextarg++] = pXboxADPCMEncodingParams;
				args[nextarg++] = srcFilename;
				args[nextarg++] = dstFilename;
				args[nextarg++] = NULL;
			}
			else 
			{
				args[0] = executable_path;

				int	nextarg = 1;
//				args[nextarg++] = "-Ob";			// Optimal per-block encoding.
				if( bOutputAllChunks )
					args[nextarg++] = "-c";			// Output all chunks.
				args[nextarg++] = srcFilename;
				args[nextarg++] = dstFilename;
				args[nextarg++] = NULL;
			}
			spawnvp( P_WAIT, args[0], &args[0] );
			if (!saveThisLipSyncData)
				return true;
		}
	}
	
	bLooped=false;

	// Read sampler chunk
	bHasSamplerChunk = GetSamplerChunk(srcFilename,&schunk,&schunkSize);

	if (bWaveLoop && !bLoop)
		if (!WAVLoops(srcFilename,&bLooped))
		{
			printf("ERROR: Couldn't open %s (to determine loop status)\n",srcFilename);
			
			if (bHasSamplerChunk)
				free(schunk);

			return false;
		}

	if (bLoop)
		bLooped=true;

	if (bNoLoop)
		bLooped=false;

	if (bVerbose)
	{
		if (bLooped)
			printf("Looped wave detected.\n");
		else
			printf("Non-looped wave detected.\n");
	}

	if (!LoadWAV(srcFilename,&srcfmt,&srcdata,&srcsize))
	{
		printf("ERROR: Couldn't open %s\n",srcFilename);

		if (bHasSamplerChunk)
			free(schunk);
		
		return false;
	}


	if (bVerbose)
	{
		printf("Source File: %s\n",srcFilename);
		printf("Dest File: %s\n",dstFilename);
		printf("Source format:\n");
		DispWAVInfo(srcfmt);
	}

	if (bWarnRate)
	{
		if (srcfmt->nSamplesPerSec != (UINT)WarnRate)
		{
			char cwd[1024];
			_getcwd(cwd,1023);

			printf("HALTED: The file %s\\%s is sampled at %i and should be %i.\n",cwd,srcFilename,srcfmt->nSamplesPerSec,WarnRate);
			exit(-2);
		}
	}

	if (saveThisLipSyncData)
	{
		GenerateWaveformAmplitudeData(srcFilename, srcfmt, srcdata, srcsize, 60);

		if( platform == PLATFORM_XBOX )		// XBox WAV conversion already done, so exit
		{
			if (bHasSamplerChunk)
				free(schunk);

			return true;
		}
	}

	WAVEFORMATEX dstfmt;
	ZeroMemory(&dstfmt,sizeof(WAVEFORMATEX));
	dstfmt.cbSize          = 0;							// No extended data
	dstfmt.wFormatTag      = WAVE_FORMAT_PCM;			// We expect all data to be PCM before passing to VAG
	dstfmt.nSamplesPerSec  = srcfmt->nSamplesPerSec;	// This will be altered if resampling is to occur
	dstfmt.wBitsPerSample  = 16;						// VAG requires all data to be 16 bit
	if (platform == PLATFORM_PS2)
	{
		dstfmt.nChannels       = srcfmt->nChannels;			// Same channels
	}
	else
	{
		// Don't know if the other platforms can handle more than one channel
		dstfmt.nChannels       = 1;
	}

	
	// Perform resampling through the ACM
	if (bNameResample)
	{
		// Resample to 11 KHz
		if (strstr(srcFilename,"_11."))
			dstfmt.nSamplesPerSec = 11025;

		// Resample to 22 KHz
		else if (strstr(srcFilename,"_22."))
			dstfmt.nSamplesPerSec = 22050;

		// Resample to 44 KHz
		else if (strstr(srcFilename,"_44."))
			dstfmt.nSamplesPerSec = 44100;

		else
		{
			if (bUseDefaultRate)
				dstfmt.nSamplesPerSec = defaultRate;
		}
	}

	if (bResample)
		dstfmt.nSamplesPerSec = custSampleRate;

	if (bBitdepth)
		dstfmt.wBitsPerSample = custBitDepth;

	if (bMono)
		dstfmt.nChannels = 1;

	if (bStereo)
		dstfmt.nChannels = 2;

	// Force resample to test
	//dstfmt.wBitsPerSample = 8;
	//dstfmt.nSamplesPerSec = 11025;

	// Compute remaining fields for PCM data
	dstfmt.nBlockAlign     = dstfmt.nChannels * dstfmt.wBitsPerSample / 8;
	dstfmt.nAvgBytesPerSec = dstfmt.nSamplesPerSec * dstfmt.nBlockAlign;

	if (bVerbose)
	{
		printf("Destination format:\n");
		DispWAVInfo(&dstfmt);
	}

	// Check NO_SSRC environment variable
	char *p_env_no_ssrc = getenv( "NO_SSRC" );
	bool use_ssrc = (p_env_no_ssrc == NULL) || (atoi(p_env_no_ssrc) != 1);
	if (bVerbose && use_ssrc)
	{
		printf("Using SSRC for conversion.\n");
	}

	// Downsample using WIN32 calls or SSRC
	if (use_ssrc && (dstfmt.nSamplesPerSec != srcfmt->nSamplesPerSec))
	{
		WAVEFORMATEX* p_dstfmt;
		static char *args[8];
		char executable_path[256];
		char dstWavFilename[256];
		char rate_string[16];

		strcpy( executable_path, p_env_path );
		strcat( executable_path, "\\bin\\win32\\ssrc.exe" );

		sprintf(rate_string, "%d", dstfmt.nSamplesPerSec);

		// Generate new filename
		char newName[128];
		strcpy(dstWavFilename, srcFilename);
		char *p_nameStart = strrchr(dstWavFilename, '\\');
		if (p_nameStart)
		{
			// Preserve path
			p_nameStart++;
			sprintf(newName, "_%d_%s", dstfmt.nSamplesPerSec / 1000, p_nameStart);
			*p_nameStart = '\0';
			strcat(dstWavFilename, newName);
		}
		else
		{
			// No path
			sprintf(newName, "_%d_%s", dstfmt.nSamplesPerSec / 1000, dstWavFilename);
			strcpy(dstWavFilename, newName);
		}

		// Do conversion
		args[0] = executable_path;
		args[1] = "--rate";
		args[2] = rate_string;
		//args[3] = "--twopass";	// Garrett: Twopass seems to not work on some wav files, so I'm disabling it
		args[3] = "--quiet";
		args[4] = srcFilename;
		args[5] = dstWavFilename;
		args[6] = NULL;
		spawnvp( P_WAIT, args[0], &args[0] );

		if (!LoadWAV(dstWavFilename,&p_dstfmt,&dstdata,(int *) &dstsize))
		{
			printf("ERROR: Couldn't open %s\n",dstWavFilename);

			free(srcdata);
			free(srcfmt);	

			if (bHasSamplerChunk)
				free(schunk);
		
			return false;
		}
		
		dstfmt = *p_dstfmt;
		free(p_dstfmt);

		remove(dstWavFilename);
	}
	else
	{
		// Now open an ACM stream
		MMRESULT   mmr;
		HACMSTREAM stream;

		mmr = acmStreamOpen(&stream,
			NULL,		// Use first capable codec
			srcfmt,
			&dstfmt,
			NULL,
			NULL,
			NULL,
			ACM_STREAMOPENF_NONREALTIME);

		if (mmr!=MM_OK)
		{
			printf("ERROR: Failed to find an appropriate conversion codec.\n");

			// Free allocated data
			free(srcdata);
			free(srcfmt);	

			if (bHasSamplerChunk)
				free(schunk);

			return false;
		}

		// Determine the recommended size for our output buffer
		mmr = acmStreamSize(stream,srcsize,&dstsize,ACM_STREAMSIZEF_SOURCE);

		if (mmr!=MM_OK)
		{
			printf("ERROR: Couldn't determine recommended destination buffer size.\n");

			acmStreamClose(stream,0);

			// Free allocated data
			free(srcdata);
			free(srcfmt);		

			if (bHasSamplerChunk)
				free(schunk);

			return false;
		}

		// Allocate destination buffer
		dstdata=malloc(dstsize);

		if (bVerbose)
		{
			printf("Source buffer is %i bytes.\n",srcsize);
			printf("Allocated %i bytes for output buffer.\n",dstsize);
		}

		ACMSTREAMHEADER streamheader;
		ZeroMemory(&streamheader,sizeof(ACMSTREAMHEADER));

		streamheader.cbStruct    = sizeof(ACMSTREAMHEADER);
		streamheader.pbSrc       = (UCHAR*)srcdata;
		streamheader.cbSrcLength = srcsize;
		streamheader.pbDst       = (UCHAR*)dstdata;
		streamheader.cbDstLength = dstsize;

		mmr = acmStreamPrepareHeader(stream,&streamheader,0);

		if (mmr!=MM_OK)
		{
			printf("ERROR: Couldn't prepare stream header.\n");

			acmStreamUnprepareHeader(stream,&streamheader,0);
			acmStreamClose(stream,0);

			// Free allocated data
			free(srcdata);
			free(dstdata);
			free(srcfmt);

			if (bHasSamplerChunk)
				free(schunk);

			return false;
		}

		mmr = acmStreamConvert(stream,&streamheader,0);

		if (mmr!=MM_OK)
		{
			printf("ERROR: Failed to convert stream.\n");

			acmStreamUnprepareHeader(stream,&streamheader,0);
			acmStreamClose(stream,0);

			// Free allocated data
			free(srcdata);
			free(dstdata);
			free(srcfmt);

			if (bHasSamplerChunk)
				free(schunk);

			return false;
		}

		dstsize = streamheader.cbDstLengthUsed;		// In case it has changed


		acmStreamUnprepareHeader(stream,&streamheader,0);
		acmStreamClose(stream,0);
	}

	// Adjust sampler chunk data to correspond to samples per sec
	if (schunk)
	{
		// 22675 is 44.1Khz so we'll compute it from that ratio
		schunk->dwSamplePeriod = 22675 * dstfmt.nSamplesPerSec / 44100;

		// Despite cbSampleLoops containing 0, there usually seems to be a sample tak'd on anyway
		if (schunkSize >= sizeof(SamplerChunk) + sizeof(SampleLoop))
		{
			// We only care about the first loop
			SampleLoop* sloop = (SampleLoop*)(((unsigned char*)schunk) + sizeof(SamplerChunk));
			//sloop->dwFraction = sloop->dwFraction * dstfmt.nSamplesPerSec / srcfmt->nSamplesPerSec;
			//sloop->dwFraction = 244721 / 4;//dstsize - 1000;
			sloop->dwFraction = (dstsize / 2) - 1;
		}
	}

	// Format conversion was completed successfully, now convert to VAG
	// or save out the new wave

	if (bNoVAG)
	{
		// Output a waveform instead
		if (bHasSamplerChunk)
		{
			if (!SaveWAV(dstFilename,&dstfmt,dstdata,dstsize,schunk,schunkSize))
			{
				printf("ERROR: Failed to write wave file %s\n",dstFilename);

				// Free allocated data
				free(srcdata);
				free(dstdata);
				free(srcfmt);

				if (bHasSamplerChunk)
					free(schunk);

				return false;
			}
		}
		else
		{
			if (!SaveWAV(dstFilename,&dstfmt,dstdata,dstsize,NULL))
			{
				printf("ERROR: Failed to write wave file %s\n",dstFilename);

				// Free allocated data
				free(srcdata);
				free(dstdata);
				free(srcfmt);

				if (bHasSamplerChunk)
					free(schunk);

				return false;
			}
		}
	}
	else
	{
		if (!ConvertVAG(&dstfmt,dstFilename,vagMode,(char*)dstdata,dstsize,bLooped))
		{
			// Free allocated data
			free(srcdata);
			free(dstdata);
			free(srcfmt);

			if (bHasSamplerChunk)
				free(schunk);

			return false;
		}
	}

	// Free allocated data
	free(srcdata);
	free(dstdata);
	free(srcfmt);

	if (bHasSamplerChunk)
		free(schunk);

	return true;
}

void ConvertWAVs(char* filename,void* data)
{
	char  path[1024];
	char  dstdir[1024];
	char  dstfile[1024];
	char  dstleftfile[1024];
	bool  dstleft = false;
	char* udir;
	_getcwd(path,1023);

	FindData* fdata=(FindData*)data;

	LCase(path);
	LCase(fdata->dstdir);
	LCase(fdata->srcdir);

	char* sdirpos=strstr(path,fdata->srcdir);

	if (!sdirpos)
	{
		_chdir(fdata->srcdir);
		_getcwd(path,1023);
		sdirpos = strstr(path,fdata->srcdir);
	}

	sdirpos += strlen(fdata->srcdir);

	if (*sdirpos=='\\')
		udir = sdirpos+1;
	else
		udir = sdirpos;

	int i;
	i=0;

	strcpy(dstdir, fdata->dstdir);

	if (dstdir[strlen(dstdir)-1]!='\\')
		strcat(dstdir,"\\");

	strcat(dstdir, udir);
	
	strcpy(dstfile,dstdir);
	
	if (dstfile[strlen(dstfile)-1]!='\\')
		strcat(dstfile,"\\");
	
	strcat(dstfile,filename);

	LCase(dstfile);

	if (!bNoVAG)
	{
		char* pos=strstr(dstfile,".wav");

		if (pos)
		{
			if( platform == PLATFORM_XBOX )
			{
				if( bOutputWMA )
				{
					pos[1]='w';
					pos[2]='m';
					pos[3]='a';
				}
				else
				{
					pos[1]='p';
					pos[2]='c';
					pos[3]='m';
				}
			}
			else
			{
				pos[1]='v';
				pos[3]='g';

				// Also generate left channel vag file
				strcpy(dstleftfile, dstfile);
				char* leftpos=strstr(dstleftfile,".vag");
				if (leftpos)
				{
					*leftpos = '\0';
					strcat(dstleftfile, "l.vag");
					dstleft = true;
				}
			}
		}
	}

	//printf("%s\n",dstfile);
	MDir(dstdir, false);
	
	if (bCheckDate)
	{
		if (!FileIsNewer(filename,dstfile) && !(dstleft && FileIsNewer(filename,dstleftfile)))
		{
			if (ConvertWAV(filename,dstfile))
				printf("Success: %s\n",dstfile);
			else
				printf("Failed: %s\\%s\n",fdata->srcdir,filename);
		}
	}
	else
	{
		if (ConvertWAV(filename,dstfile))
			printf("Success: %s\n",dstfile);
		else
			printf("Failed: %s\\%s\n",fdata->srcdir,filename);
	}
}

int main(int argc,char* argv[])
{
	if (argc<2)
	{
		printf("wav2vag.exe - Wave to PS2 VAG Audio Converter\n");
		printf("Neversoft Entertainment, 2001  aml  Built %s %s\n\n",__DATE__,__TIME__);

		printf("Usage: wav2vag.exe  [sourceFilename/Dir] {-d[destFilename]} {options}\n\n");
		printf("Any .wav file is valid in which there is an appropriate Windows codec installed\n");
		printf("Options:\n");
		printf("-p[platform][file type override]\tOptionally convert for Xbox (x) or Gamecube (g). Optionally output wma(w)\n");
		printf("-r[destdir]\tRecurse directory structure for .wav files 1st param src dir\n");
		printf("\t\tto convert.  destdir defines the destination directory for\n");
		printf("\t\tconverted .vag's\n");
		printf("-b<batchfile>\tProcesses .wav files listed in <batchfile>\n");
		printf("-s\t\tResamples the .wav to correspond with naming conventions:\n");
		printf("\t\t_11 - 11 KHz   _22 - 22 KHz   _44 - 44 KHz\n");
		printf("-l\t\tForce looping .vag\n");
		printf("-nl\t\tForce not looped .vag\n");
		printf("-L\t\tMake .vag loop if .wav is looped\n");
		printf("-c[mode]\tSpecifies VAG compression mode\n");
		printf("\t\t1 - Standard  2- High Band  3- Low Band  4- 4bit straight\n");
		printf("-n\t\tDoesn't convert to .vag just converts the wave\n");
		printf("-v\t\tVerbose mode\n");
		printf("-R[khz]\t\tForce resample at given samples per second.\n");
		printf("\t\tValid values (11025, 22050, 44100, etc.)\n");
		printf("-B[bits]\tForce bit conversion (8, 16)\n");
		printf("-M\t\tConverts to mono (1 channel sound)\n");
		printf("-S\t\tConverts to stereo (2 channel sound)\n");
		printf("-D\t\tCheck file dates (only update if source is newer)\n");
		printf("-w[Khz]\t\tWarn if source not sampled at given rate\n");
		printf("-o[Khz]\t\tSets default sample rate (if -s is used samples that aren't\n");
		printf("\t\tspecified will be set to the sample rate specified here.\n");
		printf("-f\t\tGenerate a .fam file (Frame Amplitude used for lip-sync)\n");
		return -1;
	}

	char* path;
	char* srcFilename = NULL;
	char* dstFilename = NULL;
	char* batchFilename = NULL;

	// Process switches
	for(int i=1;i<argc;i++)
	{
		if ((argv[i][0] != '-') && !srcFilename)
		{
			srcFilename = argv[i];
		}

		if( strstr( argv[i],"-p" ))
		{
			if( argv[i][2] == 'x' )
			{
				platform=PLATFORM_XBOX;

				// Check to see if an output file type override has been added.
				if( strlen( argv[i] ) > 3 )
				{
					if( argv[i][3] == 'w' )
					{
						bOutputWMA = true;

						// Store the pointer to the WMA encoding parameters.
						if( argv[i][4] != 0 )
						{
							pWMAEncodingParams=&argv[i][4];
						}
					}
					else if( argv[i][3] == 'c' )
					{
						bOutputAllChunks = true;

						// Store the pointer to the ADPCM encoding parameters.
						if( argv[i][4] != 0 )
						{
							pXboxADPCMEncodingParams=&argv[i][4];
						}
					}
					else
					{
						// Store the pointer to the ADPCM encoding parameters.
						pXboxADPCMEncodingParams=&argv[i][3];
					}
				}
			}
			else if( argv[i][2] == 'g' )
				platform=PLATFORM_NGC;
		}

		if (strstr(argv[i],"-r"))
		{
			bRecurse=true;
			path=strstr(argv[i],"-r")+2;
		}

		if (strstr(argv[i],"-b"))
		{
			bBatchMode=true;
			batchFilename=strstr(argv[i],"-b")+2;
		}

		if (strstr(argv[i],"-s"))
			bNameResample=true;

		if (strstr(argv[i],"-l"))
			bLoop=true;

		if (strstr(argv[i],"-nl"))
			bNoLoop=true;

		if (strstr(argv[i],"-L"))
			bWaveLoop=true;

		if (strstr(argv[i],"-v"))
			bVerbose=true;

		if (strstr(argv[i],"-d"))
		{
			dstFilename = strstr(argv[i],"-d")+2;
		}

		if (strstr(argv[i],"-M"))
			bMono=true;
		
		if (strstr(argv[i],"-S"))
			bStereo=true;

		if (strstr(argv[i],"-R"))
		{
			bResample=true;
			custSampleRate = atoi(strstr(argv[i],"-R")+2);
		}

		if (strstr(argv[i],"-B"))
		{
			bBitdepth=true;
			custBitDepth = atoi(strstr(argv[i],"-B")+2);
		}

		if (strstr(argv[i],"-n"))
			bNoVAG=true;

		// VAG compression
		if (strstr(argv[i],"-c"))
			vagMode = atoi(strstr(argv[i],"-c")+2);

		if (strstr(argv[i],"-D"))
			bCheckDate=true;

		if (strstr(argv[i],"-w"))
		{
			WarnRate = atoi(strstr(argv[i],"-w")+2);
			bWarnRate=true;
		}

		if (strstr(argv[i],"-o"))
		{
			bUseDefaultRate = true;
			defaultRate = atoi(strstr(argv[i],"-o")+2);
		}

		if (strstr(argv[i],"-f"))
		{
			bSaveLipSyncData = true;
		}
	}

	// Verify any custom wave conversions are valid for VAG
	if (!bNoVAG)
	{
		if (bBitdepth && custBitDepth!=16)
		{
			printf("ABORTED: The VAG format only supports 16-bit samples.\n");
			return -1;
		}

		if (bStereo)
		{
			printf("ABORTED: The VAG format only supports mono samples.\n");
			return -1;
		}
	}

	// Verify destination filename
	if (!bRecurse && !bBatchMode && !dstFilename)
	{
		printf("ABORTED: You must specify an output filename if you don't have the -r or -b flag set.\n");
		return -1;
	}

	if (bBatchMode)
	{
		fstream batchStream( batchFilename, ios::in | ios::nocreate );
		if ( !batchStream.is_open() )
		{
			printf( "Error!  Could not open %s\n", batchFilename );
			batchStream.close();
			exit(1);
		}

		printf( "Running batch mode on %s\n", batchFilename );

		while ( !batchStream.eof() )
		{
			char lineBuf[256];
			char destFile[256];
			char destDir[256];
			batchStream.getline( &lineBuf[0], 256 );
			if ( batchStream.eof() )
			{
				break;
			}
			strcpy(destFile, lineBuf);

			if( platform == PLATFORM_XBOX )
			{
				// Change wav directory to wma/pcm.
				char *pcmdir = strstr(destFile, "\\wav\\");
				if (!pcmdir) pcmdir = strstr(destFile, "\\Wav\\");
				if (!pcmdir) pcmdir = strstr(destFile, "\\WAV\\");
				if( pcmdir )
				{
					if(	bOutputWMA )
					{
						pcmdir[1] = 'w';
						pcmdir[2] = 'm';
						pcmdir[3] = 'a';
					}
					else
					{
						pcmdir[1] = 'p';
						pcmdir[2] = 'c';
						pcmdir[3] = 'm';
					}
				}

				// And change filetype.
				if( bOutputWMA )
				{
					destFile[strlen(destFile) - 3] = 'w';
					destFile[strlen(destFile) - 2] = 'm';
					destFile[strlen(destFile) - 1] = 'a';
				}
				else
				{
					destFile[strlen(destFile) - 3] = 'p';
					destFile[strlen(destFile) - 2] = 'c';
					destFile[strlen(destFile) - 1] = 'm';
				}

				printf( "Making %s: %s\n", ( bOutputWMA ) ? "wma" : "pcm", destFile );
			}
			else
			{
				// Change wav directory to vag
				char *vagdir = strstr(destFile, "\\wav\\");
				if (!vagdir) vagdir = strstr(destFile, "\\Wav\\");
				if (!vagdir) vagdir = strstr(destFile, "\\WAV\\");
				if (vagdir)
				{
					vagdir[1] = 'v';
					vagdir[2] = 'a';
					vagdir[3] = 'g';
				}

				// And change filetype
				destFile[strlen(destFile) - 3] = 'v';
				destFile[strlen(destFile) - 2] = 'a';
				destFile[strlen(destFile) - 1] = 'g';

				printf("Making vag %s\n", destFile);
			}

			// Create directory string
			strcpy(destDir, destFile);
			char *dirend = strrchr(destDir, '\\');
			if (dirend)
			{
				*dirend = '\0';
			}

			MDir(destDir, true);
			if (!ConvertWAV(lineBuf,destFile))
			{
				printf("WAV conversion failed on %s\n", lineBuf);
				batchStream.close();
				exit(1);
			}
		}

		batchStream.close();
	}
	else if (!bRecurse)
	{
		if (bCheckDate)
		{
			if (!FileIsNewer(srcFilename,dstFilename))
			{
				if (ConvertWAV(srcFilename,dstFilename))
					printf("Success: %s\n",dstFilename);
				else
					printf("Failed: %s\n",dstFilename);
			}
			else
				printf("File is up-to-date.\n");
		}
		else
			if (ConvertWAV(srcFilename,dstFilename))
				printf("Success: %s\n",dstFilename);
			else
				printf("Failed: %s\n",dstFilename);
	}
	else
	{
		FindData fdata;
		fdata.srcdir = srcFilename;
		fdata.dstdir = path;

		Find(srcFilename,".wav",ConvertWAVs,&fdata);
	}
}
