#include <stdio.h>
#include <memory.h>
#include <assert.h>
#include <windows.h>


const int PACK_WMA			= 1;
const int PACK_PCM			= 2;
const int READ_BUFFER_SIZE	= 24 * 1024 * 1024;

static unsigned int CRCTable[256] = // CRC polynomial 0xedb88320
{
      0x00000000, 0x77073096, 0xee0e612c, 0x990951ba,
      0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
      0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
      0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
      0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de,
      0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
      0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec,
      0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
      0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
      0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
      0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940,
      0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
      0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116,
      0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
      0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
      0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
      0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a,
      0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
      0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818,
      0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
      0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
      0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
      0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c,
      0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
      0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2,
      0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
      0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
      0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
      0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086,
      0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
      0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4,
      0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
      0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
      0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
      0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8,
      0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
      0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe,
      0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
      0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
      0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
      0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252,
      0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
      0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60,
      0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
      0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
      0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
      0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04,
      0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
      0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a,
      0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
      0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
      0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
      0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e,
      0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
      0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c,
      0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
      0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
      0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
      0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0,
      0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
      0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6,
      0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
      0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
      0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};

struct sDirectoryStack
{
	sDirectoryStack( void );
	~sDirectoryStack( void );

	void		Push( const char* p_dir );
	const char*	Pop( void );
	bool		IsEmpty( void )		{ return ( m_index <= 0 ); }

	char	m_stack[256][256];
	int		m_index;
};


sDirectoryStack::sDirectoryStack( void )
{
	m_index = 0;
}

sDirectoryStack::~sDirectoryStack( void )
{
}


void sDirectoryStack::Push( const char* p_dir )
{
	strcpy( &m_stack[m_index][0], p_dir );
	++m_index;

}



const char *sDirectoryStack::Pop( void )
{
	if( m_index > 0 )
	{
		--m_index;
		return &m_stack[m_index][0];
	}
	return NULL;
}


unsigned int	listing[8192][3];
unsigned int	num_files;
unsigned int	offset;
unsigned char*	p_read_buffer;
sDirectoryStack	directory_stack;


static unsigned int GenerateCRCFromString( const char *pName )
{
    // A checksum of zero is used to mean no name.
    if (!pName) 
	{
		return 0;
	}	

    // Initializing the CRC to all one bits avoids failure of detection
	// should entire data stream get cyclically bit-shifted by one position.
	// The calculation of the probability of this happening is left as
	// an exercise for the reader.
	unsigned int rc = 0xffffffff;
	const char *pCh=pName;
    while (true)
    {
        char ch=*pCh++;
		if (!ch)
		{
			break;
		}
			
        // Convert to lower case.
        if (ch>='A' && ch<='Z') 
		{
			ch='a'+ch-'A';
		}	
		// Convert forward slashes to backslashes, otherwise two filenames which are
		// effectively the same but with different slashes will give different checksums.
		if (ch=='/')
		{
			ch='\\';
		}	
		
        rc = CRCTable[(rc^ch) & 0xff] ^ ((rc>>8) & 0x00ffffff);
    }

	return rc;
}



static bool does_corresponding_wav_exist( char *p_filename )
{
	char wav_filename[256];

	// Convert the filename into the corresponding wav file.
	strcpy( wav_filename, p_filename );

	char* p_replace;

	// Replace wma.
	p_replace	= strstr( wav_filename, "\\wma\\" );
	if( p_replace )
	{
		p_replace[1]	= 'w';
		p_replace[2]	= 'a';
		p_replace[3]	= 'v';

		p_replace		= strstr( wav_filename, ".wma" );
		if( p_replace )
		{
			p_replace[1] = 'w';
			p_replace[2] = 'a';
			p_replace[3] = 'v';
		}
	}

	// Replace pcm.
	p_replace	= strstr( wav_filename, "\\pcm\\" );
	if( p_replace )
	{
		p_replace[1]	= 'w';
		p_replace[2]	= 'a';
		p_replace[3]	= 'v';

		p_replace		= strstr( wav_filename, ".pcm" );
		if( p_replace )
		{
			p_replace[1] = 'w';
			p_replace[2] = 'a';
			p_replace[3] = 'v';
		}
	}
	
	WIN32_FIND_DATA	wav_data;
	HANDLE			wav_file;

	wav_file = FindFirstFile( wav_filename, &wav_data );
	FindClose( wav_file );

	if( wav_file == INVALID_HANDLE_VALUE )
	{
		printf( "%s has no ocrresponding wav file\n", p_filename );
		return false;
	}
	return true;
}







void pack_streams( int pack_type = PACK_WMA )
{
	printf( "Packing streams in %s format... ", ( pack_type == PACK_WMA ) ? "WMA" : "PCM" );
	
	num_files	= 0;
	offset		= 0;
	memset( listing, 0, sizeof( int ) * 4096 * 3 );
	
	// First thing to do is to see whether any .wma files are actually newer than the wad file. If not, there is nothing to do.
	WIN32_FIND_DATA	w_data, h_data;
	HANDLE			w_file, h_file;
	DWORD			bytes_written;
	bool			newer_wad_snippet_exists = false;
	char			current_dir[256];

	char *p_path = getenv( "PROJ_ROOT" );
	if( p_path == NULL )
	{
		fprintf( stderr, "You must first define the PROJ_ROOT environment variable\n" );
		exit( 1 );
	}

	char wad_filename[256];
	char dat_filename[256];
	char root_directory[256];

	strcpy( wad_filename, p_path );
	strcpy( dat_filename, p_path );
	strcpy( root_directory, p_path );

	if( pack_type == PACK_WMA )
	{
		strcat( wad_filename, "\\data\\streams\\wma\\wma.wad" );
		strcat( dat_filename, "\\data\\streams\\wma\\wma.dat" );
		strcat( root_directory, "\\data\\streams\\wma\\*.*" );
	}
	else
	{
		strcat( wad_filename, "\\data\\streams\\pcm\\pcm.wad" );
		strcat( dat_filename, "\\data\\streams\\pcm\\pcm.dat" );
		strcat( root_directory, "\\data\\streams\\pcm\\*.*" );
	}

	w_file = FindFirstFile( wad_filename, &w_data );
	FindClose( w_file );

	if( w_file == INVALID_HANDLE_VALUE )
	{
		// No wad file, so we want to create one.
		newer_wad_snippet_exists = true;
	}
	else
	{
		// Push the root directory onto the stack.
		directory_stack.Push( root_directory );

		while( !directory_stack.IsEmpty())
		{
			strcpy( current_dir, directory_stack.Pop());

			h_file = FindFirstFile( current_dir, &h_data );

			if( h_file != INVALID_HANDLE_VALUE )
			{
				BOOL rv = true;
				while( rv )
				{
					// If this file is a directory, just add it to the stack.
					if( h_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
					{
						if(( strcmp( h_data.cFileName, "." ) != 0 ) && ( strcmp( h_data.cFileName, ".." ) != 0 ))
						{
							char new_dir[256];
							strcpy( new_dir, current_dir );
							strcpy( new_dir + strlen( new_dir ) - 3, h_data.cFileName );
							strcat( new_dir, "\\*.*" );
							directory_stack.Push( new_dir );
						}
					}
					else if((( pack_type == PACK_WMA ) && ( strstr( h_data.cFileName, ".wma" ))) ||
							(( pack_type == PACK_PCM ) && ( strstr( h_data.cFileName, ".pcm" ))))
					{
						// We've found a .wma file, check to see whether it is newer than the wad file.
						if( CompareFileTime( &h_data.ftLastWriteTime, &w_data.ftLastWriteTime ) == 1 )
						{
							// A newer wad snippet exists.
							newer_wad_snippet_exists = true;
							break;
						}
					}
					rv = FindNextFile( h_file, &h_data );
				}
				if( newer_wad_snippet_exists )
				{
					break;
				}
			}
		}
		FindClose( h_file );
	}

	// Make sure the directory stack is empty.
	while( !directory_stack.IsEmpty())
	{
		directory_stack.Pop();
	}
	
	if( !newer_wad_snippet_exists )
	{
		printf( "No newer stream files - nothing to do.\n" );
		return;
	}
	else
	{
		printf( "Newer stream files - proceeding to pack.\n" );
	}
	
	HANDLE of, tf;

	of = CreateFile( wad_filename, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL );
	if( of == INVALID_HANDLE_VALUE )
	{
		fprintf( stderr, "Failed to create %s\n", wad_filename );
		exit( 1 );
	}

	// Push the root directory onto the stack.
	directory_stack.Push( root_directory );

	while( !directory_stack.IsEmpty())
	{
		strcpy( current_dir, directory_stack.Pop());

		h_file = FindFirstFile( current_dir, &h_data );
		if( h_file != INVALID_HANDLE_VALUE )
		{
			BOOL rv = true;
			while( rv )
			{
				// If this file is a directory, just add it to the stack.
				if( h_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
				{
					if(( strcmp( h_data.cFileName, "." ) != 0 ) && ( strcmp( h_data.cFileName, ".." ) != 0 ))
					{
						char new_dir[256];
						strcpy( new_dir, current_dir );
						strcpy( new_dir + strlen( new_dir ) - 3, h_data.cFileName );
						strcat( new_dir, "\\*.*" );
						directory_stack.Push( new_dir );
					}
				}
				else if((( pack_type == PACK_WMA ) && ( strstr( h_data.cFileName, ".wma" ))) ||
						(( pack_type == PACK_PCM ) && ( strstr( h_data.cFileName, ".pcm" ))))
				{
					DWORD bytes_read;
					DWORD size = h_data.nFileSizeLow;

					char filename[256];
					strcpy( filename, current_dir );
					strcpy( filename + strlen( filename ) - 3, h_data.cFileName );
				
					if( does_corresponding_wav_exist( filename ))
					{
						HANDLE h_this_file = CreateFile( filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );

						// Check for this file exceeding the read buffer size.
						DWORD file_size = GetFileSize( h_this_file, NULL );
						if( file_size > READ_BUFFER_SIZE )
						{
							fprintf( stderr, "File: %s exceeds read buffer size.\n", filename );
							exit( 1 );
						}

						ReadFile( h_this_file, p_read_buffer, size, &bytes_read, NULL );

						if( bytes_read != size )
						{
							fprintf( stderr, "File: %s read error.\n", filename );
							exit( 1 );
						}
						CloseHandle( h_this_file );

						// Pad to nearest 8k block (WMA) or 16k block (ADPCM).
						DWORD aligned_size = ( pack_type == PACK_WMA ) ? (( size + 8191 ) & ~8191 ) : (( size + 16383 ) & ~16383 );
						if( size < aligned_size )
						{
							for( unsigned int s = size; s < aligned_size; ++s )
							{
								p_read_buffer[s] = 0;
							}
						}

						WriteFile( of, p_read_buffer, aligned_size, &bytes_written, NULL );

						strcpy( filename, h_data.cFileName );
						filename[strlen( filename ) - 4] = 0;
						
						// Check this file hasn't already been wadded.
						unsigned int checksum = GenerateCRCFromString( filename );
						bool found = false;
						for( unsigned int c = 0; c < num_files; ++c )
						{
							if( listing[c][0] == checksum )
							{
								found = true;
								printf( "Duplicate checksum %x found with filename %s\n", checksum, h_data.cFileName );
								break;
							}
						}
					
						if( !found )
						{
							listing[num_files][0] = checksum;
							listing[num_files][1] = offset;
							listing[num_files][2] = size;
							++num_files;
						}

						offset += aligned_size;
					}
				}
				rv = FindNextFile( h_file, &h_data );
			}
		
		}
	}

	FindClose( h_file );

	CloseHandle( of );
	
	if( num_files > 0 )
	{
		tf = CreateFile( dat_filename, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL );
		if( tf == INVALID_HANDLE_VALUE )
		{
			fprintf( stderr, "Failed to create %s\n", dat_filename );
			exit( 1 );
		}

		WriteFile( tf, &num_files, sizeof( int ), &bytes_written, NULL );
		WriteFile( tf, listing, sizeof( int ) * 3 * num_files, &bytes_written, NULL );
		CloseHandle( tf );
	}
}



void pack_music( int pack_type = PACK_WMA )
{
	printf( "Packing music in %s format... ", ( pack_type == PACK_WMA ) ? "WMA" : "PCM" );
	
	num_files	= 0;
	offset		= 0;
	memset( listing, 0, sizeof( int ) * 4096 * 3 );
	
	// First thing to do is to see whether any .wma files are actually newer than the wad file. If not, there is nothing to do.
	WIN32_FIND_DATA	w_data, h_data;
	HANDLE			w_file, h_file;
	DWORD			bytes_written;
	bool			newer_wad_snippet_exists = false;
	char			current_dir[256];

	char *p_path = getenv( "PROJ_ROOT" );
	if( p_path == NULL )
	{
		fprintf( stderr, "You must first define the PROJ_ROOT environment variable\n" );
		exit( 1 );
	}

	char wad_filename[256];
	char dat_filename[256];
	char root_directory[256];

	strcpy( wad_filename, p_path );
	strcpy( dat_filename, p_path );
	strcpy( root_directory, p_path );

	if( pack_type == PACK_WMA )
	{
		strcat( wad_filename, "\\data\\streams\\wma\\music_wma.wad" );
		strcat( dat_filename, "\\data\\streams\\wma\\music_wma.dat" );
		strcat( root_directory, "\\data\\music\\wma\\*.*" );
	}
	else
	{
		strcat( wad_filename, "\\data\\streams\\pcm\\music_pcm.wad" );
		strcat( dat_filename, "\\data\\streams\\pcm\\music_pcm.dat" );
		strcat( root_directory, "\\data\\music\\pcm\\*.*" );
	}

	w_file = FindFirstFile( wad_filename, &w_data );
	FindClose( w_file );

	if( w_file == INVALID_HANDLE_VALUE )
	{
		// No wad file, so we want to create one.
		newer_wad_snippet_exists = true;
	}
	else
	{
		// Push the root directory onto the stack.
		directory_stack.Push( root_directory );

		while( !directory_stack.IsEmpty())
		{
			strcpy( current_dir, directory_stack.Pop());

			h_file = FindFirstFile( current_dir, &h_data );

			if( h_file != INVALID_HANDLE_VALUE )
			{
				BOOL rv = true;
				while( rv )
				{
					// If this file is a directory, just add it to the stack.
					if( h_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
					{
						if(( strcmp( h_data.cFileName, "." ) != 0 ) && ( strcmp( h_data.cFileName, ".." ) != 0 ))
						{
							char new_dir[256];
							strcpy( new_dir, current_dir );
							strcpy( new_dir + strlen( new_dir ) - 3, h_data.cFileName );
							strcat( new_dir, "\\*.*" );
							directory_stack.Push( new_dir );
						}
					}
					else if((( pack_type == PACK_WMA ) && ( strstr( h_data.cFileName, ".wma" ))) ||
							(( pack_type == PACK_PCM ) && ( strstr( h_data.cFileName, ".pcm" ))))
					{
						// We've found a .wma file, check to see whether it is newer than the wad file.
						if( CompareFileTime( &h_data.ftLastWriteTime, &w_data.ftLastWriteTime ) == 1 )
						{
							// A newer wad snippet exists.
							newer_wad_snippet_exists = true;
							break;
						}
					}
					rv = FindNextFile( h_file, &h_data );
				}
				if( newer_wad_snippet_exists )
				{
					break;
				}
			}
		}
		FindClose( h_file );
	}

	// Make sure the directory stack is empty.
	while( !directory_stack.IsEmpty())
	{
		directory_stack.Pop();
	}
	
	if( !newer_wad_snippet_exists )
	{
		printf( "No newer music files - nothing to do.\n" );
		return;
	}
	else
	{
		printf( "Newer music files - proceeding to pack.\n" );
	}
	
	HANDLE of, tf;

	of = CreateFile( wad_filename, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL );
	if( of == INVALID_HANDLE_VALUE )
	{
		fprintf( stderr, "Failed to create %s\n", wad_filename );
		exit( 1 );
	}

	// Push the root directory onto the stack.
	directory_stack.Push( root_directory );

	while( !directory_stack.IsEmpty())
	{
		strcpy( current_dir, directory_stack.Pop());

		h_file = FindFirstFile( current_dir, &h_data );
		if( h_file != INVALID_HANDLE_VALUE )
		{
			BOOL rv = true;
			while( rv )
			{
				// If this file is a directory, just add it to the stack.
				if( h_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
				{
					if(( strcmp( h_data.cFileName, "." ) != 0 ) && ( strcmp( h_data.cFileName, ".." ) != 0 ))
					{
						char new_dir[256];
						strcpy( new_dir, current_dir );
						strcpy( new_dir + strlen( new_dir ) - 3, h_data.cFileName );
						strcat( new_dir, "\\*.*" );
						directory_stack.Push( new_dir );
					}
				}
				else if((( pack_type == PACK_WMA ) && ( strstr( h_data.cFileName, ".wma" ))) ||
						(( pack_type == PACK_PCM ) && ( strstr( h_data.cFileName, ".pcm" ))))
				{
					DWORD bytes_read;
					DWORD size = h_data.nFileSizeLow;

					char filename[256];
					strcpy( filename, current_dir );
					strcpy( filename + strlen( filename ) - 3, h_data.cFileName );
				
					if( does_corresponding_wav_exist( filename ))
					{
						HANDLE h_this_file = CreateFile( filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );

						// Check for this file exceeding the read buffer size.
						DWORD file_size = GetFileSize( h_this_file, NULL );
						if( file_size > READ_BUFFER_SIZE )
						{
							fprintf( stderr, "File: %s exceeds read buffer size.\n", filename );
							exit( 1 );
						}

						ReadFile( h_this_file, p_read_buffer, size, &bytes_read, NULL );

						if( bytes_read != size )
						{
							fprintf( stderr, "File: %s read error.\n", filename );
							exit( 1 );
						}
						CloseHandle( h_this_file );

						// Pad to nearest 8k block (WMA) or 16k block (ADPCM).
						DWORD aligned_size = ( pack_type == PACK_WMA ) ? (( size + 8191 ) & ~8191 ) : (( size + 16383 ) & ~16383 );
						if( size < aligned_size )
						{
							for( unsigned int s = size; s < aligned_size; ++s )
							{
								p_read_buffer[s] = 0;
							}
						}
					
						WriteFile( of, p_read_buffer, aligned_size, &bytes_written, NULL );

						strcpy( filename, h_data.cFileName );
						filename[strlen( filename ) - 4] = 0;
						
						// Check this file hasn't already been wadded.
						unsigned int checksum = GenerateCRCFromString( filename );
						bool found = false;
						for( unsigned int c = 0; c < num_files; ++c )
						{
							if( listing[c][0] == checksum )
							{
								found = true;
								printf( "Duplicate checksum %x found with filename %s\n", checksum, h_data.cFileName );
								break;
							}
						}
					
						if( !found )
						{
							listing[num_files][0] = checksum;
							listing[num_files][1] = offset;
							listing[num_files][2] = size;
							++num_files;
						}

						offset += aligned_size;
					}
				}
				rv = FindNextFile( h_file, &h_data );
			}
		
		}
	}

	FindClose( h_file );

	CloseHandle( of );
	
	if( num_files > 0 )
	{
		tf = CreateFile( dat_filename, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL );
		if( tf == INVALID_HANDLE_VALUE )
		{
			fprintf( stderr, "Failed to create %s\n", dat_filename );
			exit( 1 );
		}

		WriteFile( tf, &num_files, sizeof( int ), &bytes_written, NULL );
		WriteFile( tf, listing, sizeof( int ) * 3 * num_files, &bytes_written, NULL );
		CloseHandle( tf );
	}
}



int main(int argc, char* argv[])
{
	// Grab a 16mb read buffer.
	p_read_buffer = new unsigned char[READ_BUFFER_SIZE];
	
	printf( "WMAPack - Pack Xbox .wma/.pcm files into 1 single file\n" );
	
	pack_streams( PACK_WMA );
	pack_music( PACK_WMA );
	
	pack_streams( PACK_PCM );
	pack_music( PACK_PCM );

	// Release the read buffer.
	delete [] p_read_buffer;

	return 0;
}

