/*
	TrackUI.cpp
	Track Visualization Graph Control

	This is used in the cutscene exporter to control when cameras
	are switched

	aml - 1-30-03
*/

#include "TrackUI.h"
#include "Resource.h"
#include "../Misc/Saveable.h"
#include "../appdata.h"
#include "next.h"
#include "MemDebug.h"
#include "../PropEdit/ConfigData.h"

extern HINSTANCE  hInstance;
extern Interface* gInterface;

#define CUSTIDC_SHOWHIDE  1
#define CUSTIDC_KEYLEFT   2
#define CUSTIDC_KEYRIGHT  3
#define CUSTIDC_ZOOMIN    4
#define CUSTIDC_ZOOMOUT   5
#define CUSTIDC_RESET     6

void TrackUI::Init()
{
	hFont = CreateFont(14,0,0,0,FW_BOLD,0,0,0,0,0,0,0, VARIABLE_PITCH | FF_SWISS, _T(""));
	minFrame = gInterface->GetAnimRange().Start();
	maxFrame = gInterface->GetAnimRange().End();

	GetClientRect(hwnd, &clientRect);
	curFrame  = 0;
	selTrack  = NULL;
	swapTrack = NULL;
	state     = TAS_FREE;

	fpTimeChangeCB    = NULL;
	pTimeChangeData   = NULL;

	fpDblClick        = NULL;
	pDblClickData     = NULL;

	fpKeyAdded        = NULL;
	pKeyAddedData     = NULL;

	fpKeyRemoved      = NULL;
	pKeyRemovedData   = NULL;

	fpPreKeyMoved     = NULL;
	pPreKeyMovedData  = NULL;

	fpPostKeyMoved    = NULL;
	pPostKeyMovedData = NULL;

	fpKeyChanged      = NULL;
	pKeyChangedData   = NULL;

	saveID = vNAPP_DEFAULT_TRACKKEY_DATA;

	Interval interval = gInterface->GetAnimRange();
	origStartTime = interval.Start() / GetTicksPerFrame();
	origEndTime = interval.End() / GetTicksPerFrame();

	gInterface->RegisterTimeChangeCallback(this);
	
#ifndef DISABLE_NOTIFICATIONS
	RegisterNotification(TimeUnitsChanged,this,NOTIFY_TIMEUNITS_CHANGE);
	RegisterNotification(UnitsChanged,this,NOTIFY_UNITS_CHANGE);
#endif

	graphStartTime = 0;

	iKeySelected   = -1;
	bMayAddKeys    = true;
	bMayRemoveKeys = true;

	hMenu = LoadMenu(hInstance, MAKEINTRESOURCE(IDR_TRACKUIMENU));
	hPopupMenu = GetSubMenu(hMenu,0);

	// Setup the default add key
	defaultKey.time              = 50 * GetTicksPerFrame();
	defaultKey.userData          = 0;
	defaultKey.name              = "NewKey";
	defaultKey.pInternalUserData = NULL;
	defaultKey.iUserDataSize     = 0;
	defaultKey.cDisplayColor     = RGB(255,0,0);
	defaultKey.cHighlightColor   = RGB(0,255,0);
	//defaultKey.ActionCB        = NULL;

	linkUpdateKey = NULL;

	hwndScroll   = NULL;
	hwndZoom     = NULL;
	hwndShowHide = NULL;
	hwndReset    = NULL;

	bUIVisible   = false;

	scrollOffset = 0;
	fZoomFactor  = 1.0f;
}

TrackUI::TrackUI(HINSTANCE hInstance) :
	UIControl(hInstance, CLASS)
{
	Init();
}

TrackUI::TrackUI(HINSTANCE hInstance, HWND hwnd) :
	UIControl(hInstance, CLASS)
{
	Init();
}

TrackUI::TrackUI(TrackUI& right) :
	UIControl(right.hInstance, CLASS)
{
	Init();
	barlist = right.barlist;
}

TrackUI& TrackUI::operator =(TrackUI& right)
{
	barlist = right.barlist;

	return *this;
}

TrackUI::~TrackUI()
{
	gInterface->UnRegisterTimeChangeCallback(this);

	UnRegisterNotification(TimeUnitsChanged,this,NOTIFY_TIMEUNITS_CHANGE);
	UnRegisterNotification(UnitsChanged,this,NOTIFY_UNITS_CHANGE);
	DeleteObject(hFont);

	DestroyMenu(hPopupMenu);
	DestroyMenu(hMenu);

	if (hwndScroll)
		DestroyWindow(hwndScroll);

	if (hwndZoom)
		DestroyWindow(hwndZoom);

	if (hwndShowHide)
		DestroyWindow(hwndShowHide);

	if (hwndReset)
		DestroyWindow(hwndReset);
}

void TrackUI::TimeUnitsChanged(void *param, NotifyInfo* info)
{
	OutputDebugString("HANDLER: TimeUnitsChanged (TrackUI)\n");
	TrackUI* pthis = (TrackUI*)param;
	InvalidateRect(pthis->hwnd, NULL, TRUE);
}

void TrackUI::UnitsChanged(void *param, NotifyInfo* info)
{
	OutputDebugString("HANDLER: UnitsChanged (TrackUI)\n");
	TrackUI* pthis = (TrackUI*)param;
	InvalidateRect(pthis->hwnd, NULL, TRUE);
}

void TrackUI::BuildControlUI()
{
	RECT wndRect;
	GetWindowRect(hwnd, &wndRect);

	unsigned int width  = wndRect.right  - wndRect.left;
	unsigned int height = wndRect.bottom - wndRect.top;

	hwndShowHide = CreateWindow("button",
		                        "#",
								WS_CHILD|WS_VISIBLE,
								width - 15,
								height - 15,
								15,
								15,
								hwnd,
								(HMENU)CUSTIDC_SHOWHIDE,
								hInstance,
								NULL);

	hwndScroll = CreateWindow("scrollbar",
		                        "",
								WS_CHILD|WS_VISIBLE|SBS_HORZ,
								width - 60,
								height - 15,
								30,
								15,
								hwnd,
								(HMENU)CUSTIDC_KEYRIGHT,
								hInstance,
								NULL);

	hwndZoom  = CreateWindow("scrollbar",
		                        "",
								WS_CHILD|WS_VISIBLE|SBS_VERT,
								width  - 15,
								height - 45,
								15,
								30,
								hwnd,
								(HMENU)CUSTIDC_ZOOMIN,
								hInstance,
								NULL);

	hwndReset = CreateWindow("button",
		                     "R",
							 WS_CHILD|WS_VISIBLE,
							 width - 30,
							 height - 15,
							 15,
							 15,
							 hwnd,
							 (HMENU)CUSTIDC_RESET,
							 hInstance,
							 NULL);

	bUIVisible = true;
}

void TrackUI::CycleTracks()
{
	// Cycle through the available tracks
	Link<TrackUIRange>* curTrack = GetPointTrack();

	// If the pointer is not over a track we cycle to the next track
	if (!curTrack)
	{
		if (selTrack)
		{
			selTrack = selTrack->next;

			if (!selTrack)
				selTrack = barlist.GetHead();
		}
		else
			selTrack = barlist.GetHead();

		state = TAS_FREE;
		InvalidateRect(hwnd, NULL, TRUE);
	}
	else
	{
		// If the pointer is on a track we enter swap mode
		if (state == TAS_FREE)
		{
			if (selTrack)
			{
				swapTrack = selTrack;
				state = TAS_SWAPTRACK;
				InvalidateRect(hwnd, NULL, TRUE);
			}
		}
	}
}

LRESULT TrackUI::WndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_MOUSEMOVE:
		{
			mousePos.x = LOWORD(lParam);
			mousePos.y = HIWORD(lParam);
			
			RECT rect;
			GetClientRect(hwnd, &rect);
			rect.left++;
			rect.right--;
			rect.top++;
			rect.bottom--;
			InvalidateRect(hwnd, &rect, TRUE);
		}
		break;

	// Right clicking cycles through available tracks
	case WM_RBUTTONDOWN:
		{
			if (state == TAS_LOCK)
				break;

			if (state == TAS_KEYMOVE)
			{
				if (fpPostKeyMoved)
					fpPostKeyMoved(this, linkUpdateKey, pPostKeyMovedData);

				state = TAS_FREE;
			}

			RECT  rectClient;
			POINT localPt, screenPt;
			GetCursorPos(&localPt);
			GetCursorPos(&screenPt);
			ScreenToClient(hwnd, &localPt);
			GetClientRect(hwnd, &rectClient);

			int clientHeight = rectClient.bottom - rectClient.top;

			// The popup menu should appear if you're over top of a key or
			// your cursor lies in the bottom 4th of the control
			if (iKeySelected > -1 || localPt.y > clientHeight - clientHeight / 4)
			{
				// If a key object is selected in the TrackUI then we need to
				// bring up the key options menu

				if (iKeySelected > -1 && bMayRemoveKeys)
					EnableMenuItem(hPopupMenu, IDM_REMOVEKEY, MF_BYCOMMAND|MF_ENABLED);
				else
					EnableMenuItem(hPopupMenu, IDM_REMOVEKEY, MF_BYCOMMAND|MF_GRAYED);

				if (bMayAddKeys)
					EnableMenuItem(hPopupMenu, IDM_ADDKEY, MF_BYCOMMAND|MF_ENABLED);
				else
					EnableMenuItem(hPopupMenu, IDM_ADDKEY, MF_BYCOMMAND|MF_GRAYED);

				int ID = (int)TrackPopupMenu(hPopupMenu,TPM_LEFTBUTTON|TPM_RETURNCMD,
										     screenPt.x,screenPt.y,
										     0,hwnd,NULL);

				switch(ID)
				{
				case IDM_ADDKEY:
					{
						//linkUpdateKey = trackKeys.AddToTail(&defaultKey);
						linkUpdateKey = AddKey(&defaultKey);
						iKeySelected  = trackKeys.GetSize() - 1;
						RECT rect;
						GetWindowRect(hwnd, &rect);
						ClipCursor(&rect);

						if (fpPreKeyMoved)
							fpPreKeyMoved(this, linkUpdateKey, pPreKeyMovedData);

						state = TAS_KEYMOVE;
					}
					break;

				case IDM_MOVEKEY:
					{
						RECT rect;
						GetWindowRect(hwnd, &rect);
						ClipCursor(&rect);

						if (fpPreKeyMoved)
							fpPreKeyMoved(this, linkUpdateKey, pPreKeyMovedData);

						state = TAS_KEYMOVE;
					}
					break;

				case IDM_REMOVEKEY:
					{
						if (linkUpdateKey)
						{
							RemoveKey(linkUpdateKey);
							linkUpdateKey = NULL;
							iKeySelected = -1;

							state = TAS_FREE;

							InvalidateRect(hwnd, NULL, TRUE);
						}
					}
					break;

				case IDM_CYCLE:
					CycleTracks();
					break;
				}
			}
			else
			{
				CycleTracks();
			}	// end if KeySelected
		}
		break;

	case WM_LBUTTONDBLCLK:
		{
			if (state == TAS_KEYMOVE)
			{
				ClipCursor(NULL);

				if (fpPostKeyMoved)
					fpPostKeyMoved(this, linkUpdateKey, pPostKeyMovedData);

				state = TAS_FREE;
				return 0;
			}

			if (iKeySelected > -1)
			{
				TimeAdjustState prevTimeAdjustState = state;

				if (linkUpdateKey)
					if (fpAction)
						fpAction(linkUpdateKey, pActionData);

					//if (linkUpdateKey->data.ActionCB)
					//	linkUpdateKey->data.ActionCB(linkUpdateKey, linkUpdateKey->data.userData);

				// If the action callback hasn't adjusted the state then we should set the state back to free
				if (prevTimeAdjustState == state)
					state = TAS_FREE;

				return 0;
			}

			if (selTrack != NULL && selTrack != swapTrack)
			{
				if (fpDblClick)
					fpDblClick(this, selTrack, pDblClickData);
			}

			state = TAS_FREE;
			return 0;
		}
		break;

	case WM_LBUTTONDOWN:
		{	
			switch(state)
			{
			case TAS_FREE:
				{
					// Cannot switch to start time unless some track item is selected
					if (selTrack)
					{
						// Should only enter the time move state if the pointer is over a bar
						if (GetPointTrack())
						{
							// Lock the cursor to stay within this section of the screen
							RECT rect;
							GetWindowRect(hwnd, &rect);
							ClipCursor(&rect);

							state = TAS_TIMEMOVE;
						}
					}
				}
				break;

			case TAS_TIMEMOVE:
				{
					// Unlock the cursor
					ClipCursor(NULL);

					state = TAS_FREE;
					PostUpdate();
				}
				break;

			case TAS_SWAPTRACK:
				{
					// Unlock the cursor
					ClipCursor(NULL);

					TrackUIRange tuir;
					tuir.name     = swapTrack->data.name;
					tuir.userData = swapTrack->data.userData;

					swapTrack->data.name     = selTrack->data.name;
					swapTrack->data.userData = selTrack->data.userData;

					selTrack->data.name     = tuir.name;
					selTrack->data.userData = tuir.userData;

					state = TAS_FREE;
					swapTrack = NULL;
					InvalidateRect(hwnd, NULL, TRUE);
					PostUpdate();
				}
				break;

			case TAS_KEYMOVE:
				{
					ClipCursor(NULL);

					if (fpPostKeyMoved)
						fpPostKeyMoved(this, linkUpdateKey, pPostKeyMovedData);

					state = TAS_FREE;
					trackKeys.Sort();

					if (fpKeyChanged)
						fpKeyChanged(this, pKeyChangedData);
				}
				break;
			}
		}
		break;

	case WM_PAINT:
		{
			PAINTSTRUCT ps;
			HDC         hdc;

			hdc = BeginPaint(hwnd, &ps);
			SelectObject(hdc, hFont);
			//BuildTrackRegions();
			DrawGraph(hdc);
			EndPaint(hwnd, &ps);
			return 0;
		}

	case WM_HSCROLL:
		{
			switch(LOWORD(wParam))
			{
			case SB_LINELEFT:
				KeyLeft();
				break;

			case SB_LINERIGHT:
				KeyRight();
				break;
			}

			return 0;
		}

	case WM_VSCROLL:
		{
			switch(LOWORD(wParam))
			{
			case SB_LINEUP:
				ZoomIn();
				break;

			case SB_LINEDOWN:
				ZoomOut();
				break;
			}

			return 0;
		}

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case CUSTIDC_SHOWHIDE:
			ShowHideUI();
			return TRUE;

		case CUSTIDC_KEYLEFT:
			KeyLeft();
			return TRUE;

		case CUSTIDC_KEYRIGHT:
			KeyRight();
			return TRUE;

		case CUSTIDC_ZOOMIN:
			ZoomIn();
			return TRUE;

		case CUSTIDC_ZOOMOUT:
			ZoomOut();
			return TRUE;

		case CUSTIDC_RESET:
			ResetZoomScroll();
			return TRUE;
		}
	}

	return DefWindowProc(hwnd, msg, wParam, lParam);
}

void TrackUI::RegisterDblClickCB(void(*fp)(TrackUI*,Link<TrackUIRange>*,void*), void* pData)
{
	fpDblClick = fp;
	pDblClickData = pData;
}

Link<TrackUIRange>* TrackUI::AddItem(int end, CStr name, DWORD userData, Link<TrackUIRange>* link)
{
	TrackUIRange tuir;
	tuir.end      = end;
	tuir.name     = name;
	tuir.userData = userData;

	if (link)
		return barlist.AddAfterLink(link, &tuir);

	Link<TrackUIRange>* result = barlist.Add(&tuir);
	barlist.Sort();

	InvalidateRect(hwnd, NULL, TRUE);

	return result;
}

Link<TrackUIRange>* TrackUI::GetPointTrack()
{
	Link<TrackUIRange>* curtrack = barlist.GetHead();

	GetClientRect(hwnd, &clientRect);

	Interval animRange = gInterface->GetAnimRange();
	int start = animRange.Start() / GetTicksPerFrame();
	int end   = animRange.End() / GetTicksPerFrame();

	int   width      = clientRect.right - clientRect.left;
	int   height     = clientRect.bottom - clientRect.top;
	float convFactor = (float)width / (float)(end - start);

	POINT ptMouse;
	GetCursorPos(&ptMouse);
	ScreenToClient(hwnd, &ptMouse);

	while(curtrack)
	{
		if (curtrack->prev)
			start = curtrack->prev->data.end;
		else
			start = graphStartTime;

		if (ptMouse.x > start * convFactor + 1		        &&
			ptMouse.x < curtrack->data.end * convFactor + 1 &&
			ptMouse.y > clientRect.bottom / 2 - 20	        &&
			ptMouse.y < clientRect.bottom / 2 - 8)
		{
			return curtrack;
		}

		curtrack = curtrack->next;
	}

	return NULL;
}

void TrackUI::DrawGraph(HDC hdc)
{
	HBRUSH      hBrush     = (HBRUSH)GetStockObject(LTGRAY_BRUSH);
	HBRUSH      hBrushNULL = (HBRUSH)GetStockObject(NULL_BRUSH);
	HPEN        hDashPen   = (HPEN)CreatePen(PS_DASH, 2, 0);
	HPEN        hSolidPen  = (HPEN)CreatePen(PS_SOLID, 0, 0);

	GetClientRect(hwnd, &clientRect);

	// Draw Bar
	SelectObject(hdc, hBrush);
	SelectObject(hdc, hSolidPen);
	SetBkMode(hdc, TRANSPARENT);

	// Draw time tick marks on graph
	Interval animRange = gInterface->GetAnimRange();
	int start = animRange.Start() / GetTicksPerFrame();
	int end   = animRange.End() / GetTicksPerFrame();
	int i;

	int origStart = start;
	int origEnd   = end;

	// Handle zooming
	//start = (float)start * fZoomFactor;
	start *= fZoomFactor;
	end   *= fZoomFactor;
	start += scrollOffset;
	end   += scrollOffset;
	//end   = start + scrollOffset + ((float)(end - start) * fZoomFactor);

	int   width      = clientRect.right - clientRect.left;
	int   height     = clientRect.bottom - clientRect.top;
	int   halfWidth  = width / 2;
	int   halfHeight = height / 2;
	float convFactor = (float)width / (float)(end - start);

	POINT ptMouse;
	GetCursorPos(&ptMouse);
	ScreenToClient(hwnd, &ptMouse);

	// Update selected track
	if (state == TAS_FREE || state == TAS_SWAPTRACK)
	{
		Link<TrackUIRange>* clink = barlist.GetHead();

		while(clink)
		{
			if (state == TAS_FREE ||
				(state == TAS_SWAPTRACK && clink != swapTrack))
			{
				if (ptMouse.x > start * convFactor + 1           &&
					ptMouse.x < clink->data.end * convFactor + 1 &&
					ptMouse.y > clientRect.bottom / 2 - 20       &&
					ptMouse.y < clientRect.bottom / 2 - 8)
				{
					selTrack  = clink;
					break;
				}
			}

			clink = clink->next;
		}
	}

	if (state == TAS_KEYMOVE)
	{
		if (linkUpdateKey)
		{
			linkUpdateKey->data.time = ((float)mousePos.x / (float)width * (float)(end - start) + start) * GetTicksPerFrame();
			//linkUpdateKey->data.time = (int)((float)mousePos.x / (float)width * (float)end) * GetTicksPerFrame();
			//trackKeys.Sort();
		}
	}

	if (state == TAS_TIMEMOVE)
	{
		if (selTrack)
		{
			selTrack->data.end = (int)((float)mousePos.x / (float)width * (float)(end - start) + start);
			
			if (selTrack->data.end < graphStartTime)
				graphStartTime = selTrack->data.end;
		}

		barlist.Sort();
	}


	Link<TrackUIRange>* curlink = barlist.GetHead();

	while(curlink)
	{
		// Ensure that all randomly generated colors are reasonably far from black (our pen color)
		//char r = 128 + rand()%128, g = 128 + rand()%128, b = 128 + rand()%128;

		HBRUSH hBarBrush;
		
		int barstart;

		if (curlink->prev)
			barstart = curlink->prev->data.end;
		else
			barstart = graphStartTime;

		bool bSelected;

		if (state   == TAS_SWAPTRACK &&
			curlink == swapTrack)
		{
			hBarBrush = CreateSolidBrush(SEL_SWAP);
		}
		else if (curlink == selTrack)
		{
			bSelected = true;		
			hBarBrush = CreateSolidBrush(SEL_BARCOLOR);
		}
		else
		{
			bSelected = false;
			hBarBrush = CreateSolidBrush(UNSEL_BARCOLOR);
		}

		SelectObject(hdc, hBarBrush);

		Rectangle(hdc,
				  (barstart - start) * convFactor + 1,
				  clientRect.bottom / 2 - 20,
				  (curlink->data.end - start) * convFactor + 1,
				  clientRect.bottom / 2 - 8);

		DeleteObject(hBarBrush);			

		RECT textRect;
		SetRect(&textRect,
				(barstart - start) * convFactor + 1,
				clientRect.bottom / 2 - 20,
				(curlink->data.end - start) * convFactor + 1,
				clientRect.bottom / 2 - 8);

		DrawText(hdc, curlink->data.name, curlink->data.name.Length(), &textRect, DT_CENTER | DT_VCENTER | DT_SINGLELINE | DT_WORD_ELLIPSIS);

		// Draw the name of the object if appropriate
		if (bSelected)
			TextOut(hdc, mousePos.x + 15, mousePos.y - 20, (char*)curlink->data.name, strlen(curlink->data.name));

		curlink = curlink->next;
	}
 
	// Place marker every 20th of the width as converted to frames
	int incr = (end - start) / 20;
	if (incr <= 1)
		incr = 2;

	int len = 10;

	for(i = start; i <= end; i += incr)
	{
		float XPos = (float)(i - start) * convFactor;

		//if (i % (incr * 2))
		if (len == 20)
			len = 10;
		else
			len = 20;

		MoveToEx(hdc, XPos, 0, NULL);
		LineTo(hdc, XPos, len);

		MoveToEx(hdc, XPos, clientRect.bottom, NULL);
		LineTo(hdc, XPos, clientRect.bottom - len);

		if (len == 20)
		{
			char str[80];
			sprintf(str, "%i", i);
			TextOut(hdc, XPos - 5, clientRect.bottom - len - 10, str, strlen(str));
		}
	}
	
	// Draw a vertical line at the time position
	SelectObject(hdc, hDashPen);
	MoveToEx(hdc, (int)((float)(curFrame - start) * convFactor), 0, NULL);
	LineTo(hdc, (int)((float)(curFrame - start) * convFactor), clientRect.bottom);

	// Draw a vertical line at the mouse position
	SelectObject(hdc, hSolidPen);
	MoveToEx(hdc, mousePos.x, 0, NULL);
	LineTo(hdc, mousePos.x, clientRect.bottom);

	// Draw time of mouse position
	char strPos[80];
	sprintf(strPos, "%i", (int)((float)mousePos.x / (float)width * (float)(end - start) + start));
	RECT capRect;
	SetRect(&capRect,
		    mousePos.x + 15,		// left
			mousePos.y,				// top
			mousePos.x + 15 + 100,	// right
			mousePos.y + 100);		// bottom

	TextOut(hdc, mousePos.x + 15, mousePos.y, strPos, strlen(strPos));

	// If a track is selected put its name in the lower right hand corner
	if (selTrack)
	{
		RECT textRect;
		SetRect(&textRect,
				clientRect.left,
				clientRect.bottom - height / 2.5f,
				clientRect.right - 5,
				clientRect.bottom);
				
		DrawText(hdc, (char*)selTrack->data.name, selTrack->data.name.Length(), &textRect, DT_RIGHT|DT_TOP);
	}

	// Draw Border
	MoveToEx(hdc, 0, 0, NULL);
	LineTo(hdc, clientRect.right-1, 0);
	LineTo(hdc, clientRect.right-1, clientRect.bottom-1);
	LineTo(hdc, 0, clientRect.bottom-1);
	LineTo(hdc, 0, 0);

	DeleteObject(hSolidPen);
	DeleteObject(hDashPen);

	// Draw the track UI keys
	iKeySelected = -1;

	Link<TrackUIKey>* link = trackKeys.GetHead();
	int curKey = 0;

	while(link)
	{
		// We'll draw a small selectable box at each key time
		RECT   rectKeyUI, rectText;

		int xPos = ((link->data.time / GetTicksPerFrame() - start) * convFactor);
		SetRect(&rectKeyUI, xPos, halfHeight - 5, xPos + 5, halfHeight + 5);

		int blockWidth  = rectKeyUI.right  - rectKeyUI.left;
		int blockHeight = rectKeyUI.bottom - rectKeyUI.top;

		SetRect(&rectText, 
				rectKeyUI.left - 40, 
				rectKeyUI.top + 5,
				rectKeyUI.right + 40,
				rectKeyUI.bottom + 10);

		DrawText(hdc, (char*)link->data.name, link->data.name.Length(), &rectText, DT_CENTER|DT_VCENTER);
		
		// The color of the fill brush depends on if its selected or not
		HBRUSH hBrushKeyFill;
			
		if (state == TAS_KEYMOVE || state == TAS_LOCK)
		{
			if (linkUpdateKey == link)
				hBrushKeyFill = CreateSolidBrush(link->data.cHighlightColor);
			else
				hBrushKeyFill = CreateSolidBrush(link->data.cDisplayColor);
		}
		else
		{
			if (PtInRect(&rectKeyUI, ptMouse))
			{
				hBrushKeyFill = CreateSolidBrush(link->data.cHighlightColor);
				iKeySelected = curKey;
				linkUpdateKey = link;
			}
			else
			{
				hBrushKeyFill = CreateSolidBrush(link->data.cDisplayColor);
			}
		}

		FillRect(hdc, &rectKeyUI, hBrushKeyFill);
		DeleteObject(hBrushKeyFill);

		link = link->next;
		curKey++;
	}
}

void TrackUI::TimeChanged(TimeValue t)
{
	GetClientRect(hwnd, &clientRect);
	Interval animRange = gInterface->GetAnimRange();

	int startTime = animRange.Start() / GetTicksPerFrame();
	int endTime   = animRange.End() / GetTicksPerFrame();

	startTime *= fZoomFactor;
	endTime   *= fZoomFactor;
	startTime += scrollOffset;
	endTime   += scrollOffset;

	if (startTime != origStartTime ||
		endTime   != origEndTime)
	{
		origStartTime = startTime;
		origEndTime   = endTime;

		InvalidateRect(hwnd, NULL, TRUE);
		return;
	}

	int   width      = clientRect.right - clientRect.left;
	//float convFactor = (float)width / ((float)animRange.End() / GetTicksPerFrame());
	float convFactor = (float)width / (float)(endTime - startTime);

	RECT rect;
	rect.left   = (curFrame - startTime) * convFactor - 1;
	rect.right  = (curFrame - startTime) * convFactor + 1;
	rect.top    = clientRect.top;
	rect.bottom = clientRect.bottom;

	InvalidateRect(hwnd, &rect, TRUE);
	curFrame = t / GetTicksPerFrame();

	int scrollpt = (endTime - startTime) / 4;

	if (curFrame > endTime - scrollpt)
	{
		scrollOffset += curFrame - endTime + scrollpt;
		InvalidateRect(hwnd, NULL, TRUE);
	}

	if (curFrame < startTime)
	{
		scrollOffset -= startTime - curFrame;
		InvalidateRect(hwnd,NULL,TRUE);
	}

	rect.left   = (curFrame - startTime) * convFactor - 1;
	rect.right  = (curFrame - startTime) * convFactor + 1;
	rect.top    = clientRect.top;
	rect.bottom = clientRect.bottom;

	if (fpTimeChangeCB)
		fpTimeChangeCB(this, pTimeChangeData);

	InvalidateRect(hwnd, &rect, TRUE);
	//InvalidateRect(hwnd, NULL, TRUE);
}

int TrackUI::TrackLength()
{
	Interval animRange = gInterface->GetAnimRange();

	int startTime = animRange.Start() / GetTicksPerFrame();
	int endTime   = animRange.End() / GetTicksPerFrame();

	return endTime - startTime;
}

void TrackUI::ClearBar()
{
	barlist.Clear();
	curFrame  = 0;
	selTrack  = NULL;
	swapTrack = NULL;
	state     = TAS_FREE;

	InvalidateRect(hwnd, NULL, TRUE);
}

void TrackUI::Clear()
{
	barlist.Clear();
	trackKeys.Clear();
	curFrame  = 0;
	selTrack  = NULL;
	swapTrack = NULL;
	state     = TAS_FREE;

	InvalidateRect(hwnd, NULL, TRUE);
}

void TrackUI::PostUpdate()
{
	HWND hwndParent = GetParent(hwnd);
	SendMessage(hwndParent, TRACKUI_UPDATE, 0, (LPARAM)hwnd);
}

void TrackUI::RegisterTimeChangeCB(void(*fp)(TrackUI*, void*), void* pData)
{
	fpTimeChangeCB  = fp;
	pTimeChangeData = pData;
}

void TrackUI::SetStartTime(int time)
{
	graphStartTime = time;
	InvalidateRect(hwnd, NULL, TRUE);
}

Link<TrackUIRange>* TrackUI::FindTrack(TimeValue time)
{
	Link<TrackUIRange>* link = barlist.GetHead();
	int frameTime = time / GetTicksPerFrame();
	int start = graphStartTime;

	while(link)
	{
		if (frameTime >= start && frameTime < link->data.end)
			return link;

		start = link->data.end;
		link = link->next;
	}

	return NULL;
}

void TrackUI::ClearKeys()
{
	trackKeys.Clear();
	Refresh();
}

Link<TrackUIKey>* TrackUI::AddKey(TrackUIKey* trackKey)
{
	Link<TrackUIKey>* link = trackKeys.AddToTail(trackKey);
	
	if (fpKeyAdded)
		fpKeyAdded(this, link, pKeyAddedData);

	if (fpKeyChanged)
		fpKeyChanged(this, pKeyChangedData);

	return link;
}

void TrackUI::RemoveKey(Link<TrackUIKey>* linkTrackKey)
{
	if (fpKeyRemoved)
		fpKeyRemoved(this, linkTrackKey, pKeyRemovedData);

	trackKeys.Remove(linkUpdateKey);

	if (fpKeyChanged)
		fpKeyChanged(this, pKeyChangedData);
}

void TrackUI::Refresh()
{
	InvalidateRect(hwnd, NULL, TRUE);
}

void TrackUI::LockAffectKey(int time)
{
	MayAddTrackKeys(false);
	MayRemoveTrackKeys(false);

	TrackUIKey timeKey;
	timeKey.time = time;
	
	linkUpdateKey = trackKeys.Find(&timeKey);
	state = TAS_LOCK;
}

void TrackUI::LockAffectKey(Link<TrackUIKey>* link)
{
	MayAddTrackKeys(false);
	MayRemoveTrackKeys(false);

	linkUpdateKey = link;
	state = TAS_LOCK;
}

void TrackUI::UnlockAffectKey()
{
	MayAddTrackKeys(true);
	MayRemoveTrackKeys(true);

	linkUpdateKey = NULL;
	state = TAS_FREE;
}

void TrackUI::SetKeyAddedCB(void(*pFunc)(TrackUI*,Link<TrackUIKey>*,void*), void* pData)
{
	fpKeyAdded    = pFunc;
	pKeyAddedData = pData;
}

void TrackUI::SetRemoveKeyCB(void(*pFunc)(TrackUI*,Link<TrackUIKey>*,void*), void* pData)
{
	fpKeyRemoved    = pFunc;
	pKeyRemovedData = pData;
}

void TrackUI::SetPreKeyMovedCB(void(*pFunc)(TrackUI*,Link<TrackUIKey>*,void*), void* pData)
{
	fpPreKeyMoved    = pFunc;
	pPreKeyMovedData = pData;
}

void TrackUI::SetPostKeyMovedCB(void(*pFunc)(TrackUI*,Link<TrackUIKey>*,void*), void* pData)
{
	fpPostKeyMoved    = pFunc;
	pPostKeyMovedData = pData;
}

void TrackUI::SetKeyChangedCB(void(*pFunc)(TrackUI*,void*), void* pData)
{
	fpKeyChanged    = pFunc;
	pKeyChangedData = pData;
}

/////////////////// TrackUIKey
TrackUIKey::TrackUIKey()
{
	time              = 0;
	userData          = 0;
	pInternalUserData = NULL;
	iUserDataSize     = 0;
	cDisplayColor     = RGB(255,0,0);
	cHighlightColor   = RGB(0,255,0);
}

TrackUIKey::TrackUIKey(const TrackUIKey& right)
{
	time              = 0;
	userData          = 0;
	pInternalUserData = NULL;
	iUserDataSize     = 0;
	cDisplayColor     = RGB(255,0,0);
	cHighlightColor   = RGB(0,255,0);

	*this = right;
}

TrackUIKey::~TrackUIKey()
{
	if (pInternalUserData)
		free(pInternalUserData);
}

TrackUIKey& TrackUIKey::operator= (const TrackUIKey& right)
{
	time       = right.time;
	userData   = right.userData;
	name       = right.name;
	userBuffer = right.userBuffer;

	if (right.iUserDataSize > 0)
	{
		AllocUserData(right.iUserDataSize);
		memcpy(pInternalUserData, right.pInternalUserData, right.iUserDataSize);
	}
	else
		pInternalUserData = NULL;

	iUserDataSize = right.iUserDataSize;

	cDisplayColor   = right.cDisplayColor;
	cHighlightColor = right.cHighlightColor;

	return *this;
}

void TrackUIKey::AllocUserData(int size)
{
	if (pInternalUserData)
	{
		VerifyMemory();
		free(pInternalUserData);
		VerifyMemory();
	}

	if (size == 0)
	{
		pInternalUserData = NULL;
		iUserDataSize = 0;
		return;
	}

	pInternalUserData = malloc(size);
	iUserDataSize     = size;

	VerifyMemory();
}

CStr TrackUIKey::GetClass()
{
	CStr strClass;

	if (iUserDataSize > sizeof(DWORD) + sizeof(int) + sizeof(int))
	{
		unsigned char* pos = (unsigned char*)pInternalUserData;
		DWORD version = *((DWORD*)pos);
		pos += sizeof(DWORD);
		GetString(&pos, strClass);
	}

	return strClass;
}

CStr TrackUIKey::GetType()
{
	CStr strType;

	if (iUserDataSize > sizeof(DWORD) + sizeof(int) + sizeof(int))
	{
		unsigned char* pos = (unsigned char*)pInternalUserData;
		DWORD version = *((DWORD*)pos);
		pos += sizeof(DWORD);
		GetString(&pos, strType);	// Actually class but, used to advance pointer
		GetString(&pos, strType);
	}

	return strType;
}

CStr TrackUIKey::GetProp(CStr name)
{
	CStr strClass, strType;
	LinkList<ConfigProp> cprops;

	if (iUserDataSize > sizeof(DWORD) + sizeof(int) + sizeof(int) + sizeof(int))
	{
		unsigned char* pos = (unsigned char*)pInternalUserData;
		DWORD version = *((DWORD*)pos);
		pos += sizeof(DWORD);
		GetString(&pos, strClass);	// Actually class but, used to advance pointer
		GetString(&pos, strType);
		GetList<ConfigProp>(&pos, &cprops);

		// Search for the property and return its value
		Link<ConfigProp>* link = cprops.GetHead();\

		while(link)
		{
			if (link->data.name == name)
				return link->data.value;

			link = link->next;
		}
	}

	return CStr("");
}

int  TrackUIKey::GetSize()
{
	return sizeof(int) +
		   sizeof(DWORD) +
		   GetStringSize(name) +
		   GetStringSize(userBuffer) +
		   sizeof(int) +
		   iUserDataSize +
		   sizeof(COLORREF) +
		   sizeof(COLORREF);
}

void TrackUIKey::Store(void* data)
{
	unsigned char* pos = (unsigned char*)data;

	*((int*)pos) = time;
	pos += sizeof(int);

	*((DWORD*)pos) = userData;
	pos += sizeof(DWORD);
	
	WriteString(&pos, name);
	WriteString(&pos, userBuffer);

	*((int*)pos) = iUserDataSize;
	pos += sizeof(int);

	memcpy(pos, pInternalUserData, iUserDataSize);
	pos += iUserDataSize;

	*((COLORREF*)pos) = cDisplayColor;
	pos += sizeof(COLORREF);
	
	*((COLORREF*)pos) = cHighlightColor;
	pos += sizeof(COLORREF);
}

void TrackUIKey::Retrieve(void* data)
{
	unsigned char* pos = (unsigned char*)data;

	time = *((int*)pos);
	pos += sizeof(int);
	
	userData = *((DWORD*)pos);
	pos += sizeof(DWORD);
	GetString(&pos, name);
	GetString(&pos, userBuffer);
	iUserDataSize = *((int*)pos);
	pos += sizeof(int);

	iUserDataSize     = iUserDataSize;
	AllocUserData(iUserDataSize);
	memcpy(pInternalUserData, pos, iUserDataSize);
	pos += iUserDataSize;

	cDisplayColor   = *((COLORREF*)pos);
	pos += sizeof(COLORREF);

	cHighlightColor = *((COLORREF*)pos);
	pos += sizeof(COLORREF);
}
////////////////////////////////////////////////////

int TrackUI::CalcKeySize()
{
	int size = 0;
	Link<TrackUIKey>* link = trackKeys.GetHead();

	while(link)
	{
		size += link->data.GetSize();
		link = link->next;
	}

	return size;
}

void TrackUI::SaveKeyData(Animatable* animbase)
{
	VerifyMemory();
	Link<TrackUIKey>* link = trackKeys.GetHead();

	int keyMem = CalcKeySize() + sizeof(int);
	int count = trackKeys.GetSize();
	unsigned char* pKeyMem = (unsigned char*)malloc(keyMem);
	unsigned char* pos = pKeyMem;

	*((int*)pos) = count;
	pos += sizeof(int);

	while(link)
	{
		link->data.Store(pos);
		pos += link->data.GetSize();

		link = link->next;
	}

	assert(pos == pKeyMem + keyMem);

	if (animbase->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, saveID))
		animbase->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, saveID);

	animbase->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, saveID, keyMem, pKeyMem);

	VerifyMemory();
}

void TrackUI::LoadKeyData(Animatable* animbase)
{
	VerifyMemory();

	AppDataChunk* appdata = animbase->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, saveID);

	if (appdata && appdata->data)
	{
		trackKeys.Clear();
		unsigned char* pos = (unsigned char*)appdata->data;

		int count = *((int*)pos);
		pos += sizeof(int);

		for(int i = 0; i < count; i++)
		{
			TrackUIKey key;

			key.Retrieve(pos);
			pos += key.GetSize();

			trackKeys.Add(&key);
		}
	}

	VerifyMemory();
}

void TrackUI::LoadKeyData(Animatable* animbase, DWORD id)
{
	SetKeySaveChunk(id);
	LoadKeyData(animbase);
}

void TrackUI::LoadKeyData()
{
	LoadKeyData(saveAnimatable);
}

void TrackUI::SaveKeyData()
{
	SaveKeyData(saveAnimatable);
}

// Intended to be called by a dialog created when called from the KeyAction
// This makes trackUI signal that the key has been changed
void TrackUI::CallKeyUpdatedCB()
{
	VerifyMemory();

	if (fpKeyChanged)
		fpKeyChanged(this, pKeyChangedData);

	VerifyMemory();
}

void TrackUI::ShowHideUI()
{
	if (bUIVisible)
	{
		ShowWindow(hwndScroll, SW_HIDE);
		ShowWindow(hwndZoom,   SW_HIDE);
		ShowWindow(hwndReset,  SW_HIDE);

		bUIVisible = false;
	}
	else
	{
		ShowWindow(hwndScroll, SW_SHOW);
		ShowWindow(hwndZoom,   SW_SHOW);
		ShowWindow(hwndReset,  SW_SHOW);
		
		bUIVisible = true;
	}
}

void TrackUI::KeyLeft()
{
	if (scrollOffset == 0)
		return;

	scrollOffset -= 20 * fZoomFactor;
	
	if (scrollOffset < 0)
		scrollOffset = 0;

	InvalidateRect(hwnd, NULL, TRUE);
}

void TrackUI::KeyRight()
{
	scrollOffset += 20 * fZoomFactor;

	InvalidateRect(hwnd, NULL, TRUE);
}

void TrackUI::ZoomIn()
{
	fZoomFactor += 0.1f;

	InvalidateRect(hwnd, NULL, TRUE);
}

void TrackUI::ZoomOut()
{
	if (fZoomFactor <= 0.2f)
	{
		fZoomFactor = 0.1f;
		InvalidateRect(hwnd, NULL, TRUE);
		return;
	}

	fZoomFactor -= 0.1f;

	InvalidateRect(hwnd, NULL, TRUE);
}

void TrackUI::ResetZoomScroll()
{
	fZoomFactor  = 1.0f;
	scrollOffset = 0;

	InvalidateRect(hwnd, NULL, TRUE);
}
