/*
	SpinEdit.cpp
	A linked spinner Edit Box Control
	4-8-01
*/

#include "SpinEdit.h"
#include "PropListInt.h"
#define SPIN_WIDTH  10

SpinEdit::SpinEdit()
{
	IEdit = NULL;
	ISpin = NULL;
	value = 0.0f;
	incr  = 1.0f;
	min   = 0.0f;
	max   = 99999.0f;
}

SpinEdit::SpinEdit(HWND hwnd,float min,float max)
{
	IEdit = NULL;
	ISpin = NULL;
	value = 0.0f;

	this->hwnd = hwnd;
	this->min  = min;
	this->max  = max;
	this->incr = 1.0f;
}

SpinEdit::SpinEdit(HWND hwnd)
{
	IEdit=NULL;
	ISpin=NULL;

	this->hwnd = hwnd;
}

SpinEdit::~SpinEdit()
{
	if (ISpin)
		ReleaseISpinner(ISpin);

	if (IEdit)
		ReleaseICustEdit(IEdit);
}

SpinEdit::SpinEdit(SpinEdit& right)
{
	IEdit = NULL;
	ISpin = NULL;
	value = right.value;
	hwnd  = NULL;
	min   = right.min;
	max   = right.max;
	incr  = right.incr;
}

SpinEdit& SpinEdit::operator = (SpinEdit& right)
{
	IEdit = NULL;
	ISpin = NULL;
	value = right.value;
	hwnd  = NULL;
	min   = right.min;
	max   = right.max;
	incr  = right.incr;

	return *this;
}


void SpinEdit::Register(HINSTANCE hInstance)
{
	WNDCLASS wndclass;
	ZeroMemory(&wndclass,sizeof(WNDCLASS));

	wndclass.style       = CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc = WndProc;
	wndclass.cbClsExtra=0;
	wndclass.cbWndExtra=0;
	wndclass.hInstance=hInstance;
	wndclass.hIcon=NULL;
	wndclass.hCursor=LoadCursor(NULL,IDC_ARROW);
	wndclass.hbrBackground=(HBRUSH)GetStockObject(WHITE_BRUSH);
	wndclass.lpszMenuName=NULL;
	wndclass.lpszClassName=SPIN_EDIT_NAME;

	RegisterClass(&wndclass);
}

void SpinEdit::UnRegister(HINSTANCE hInstance)
{
	UnregisterClass(SPIN_EDIT_NAME,hInstance);
}

void SpinEdit::BuildControlUI()
{
	RECT wndRect;
	GetClientRect(hwnd,&wndRect);

	// Construct scaled spinner
	hwndSpinner = CreateWindow("SpinnerControl",
		                       "",
				               WS_CHILD,
							   wndRect.right-SPIN_WIDTH,	// x
							   0,							// y
							   SPIN_WIDTH,					// width
							   wndRect.bottom-wndRect.top,	// height
							   hwnd,						// parent
							   NULL,						// no menu
							   hInstance,
							   NULL);

	// Construct edit box
	hwndEdit = CreateWindow("CustEdit",
		                    "",
							WS_CHILD,
							0,								// x
							0,								// y
							wndRect.right-wndRect.left,		// width
							wndRect.bottom-wndRect.top,		// height
							hwnd,							// parent
							NULL,							// no menu
							hInstance,
							NULL);

	IEdit = GetICustEdit(hwndEdit);
	ISpin = GetISpinner(hwndSpinner);

	ISpin->SetLimits(min,max);
	ISpin->SetScale(incr);
	SetSpinnerPrecision(5);

	ISpin->LinkToEdit(hwndEdit, EDITTYPE_FLOAT);
	ISpin->SetValue(0.0f,TRUE);

	//SetValue(value);
	ISpin->SetValue(value,FALSE);

	ShowWindow(hwndEdit, SW_SHOW);
	ShowWindow(hwndSpinner, SW_SHOW);
}

void SpinEdit::DestroyUI()
{
	if (ISpin)
	{
		ReleaseISpinner(ISpin);
		ISpin=NULL;
	}
	
	if (IEdit)
	{
		ReleaseICustEdit(IEdit);
		IEdit=NULL;
	}

	if (hwndSpinner)
	{
		DestroyWindow(hwndSpinner);
		hwndSpinner=NULL;
	}

	if (hwndEdit)
	{
		DestroyWindow(hwndEdit);
		hwndEdit=NULL;
	}
}

void SpinEdit::Resize(int width,int height)
{
	RECT wndRect;
	GetClientRect(hwnd,&wndRect);

	MoveWindow(hwndSpinner,
		       wndRect.right-SPIN_WIDTH,	// x
			   0,							// y
			   SPIN_WIDTH,					// width
			   wndRect.bottom-wndRect.top,	// height
			   TRUE);
			   
	MoveWindow(hwndEdit,
		       0,							// x
			   0,							// y
			   wndRect.right-SPIN_WIDTH,	// width
			   wndRect.bottom-wndRect.top,	// height
			   TRUE);
}

LPARAM SpinEdit::WndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	SpinEdit* pthis = (SpinEdit*)GetWindowLong(hwnd,GWL_USERDATA);
	

	switch(msg)
	{
	case WM_CREATE:
		SetWindowLong(hwnd, GWL_USERDATA, (LONG)((CREATESTRUCT*)lParam)->lpCreateParams);
		pthis = (SpinEdit*)((CREATESTRUCT*)lParam)->lpCreateParams;
		pthis->Attach(hwnd);
		return TRUE;

	// Inform on focus acquisition as well
	case CC_SPINNER_CHANGE:
		{
			HWND hwndParent = GetParent(hwnd);
			int  ID = (int)GetWindowLong(hwnd,GWL_ID);
			SendMessage(hwndParent,WM_CHILDUPDATED,(WPARAM)ID,0);
			SendMessage(hwndParent,WM_CHILDGOTFOCUS,(WPARAM)ID,0);
			return TRUE;
		}

	case WM_SETFOCUS:
		{
			SetFocus(pthis->GetFocusWindow());
			return 0;
		}

	case WM_COMMAND:
		switch(HIWORD(wParam))
		{
		case EN_CHANGE:
			{
				HWND hwndParent = GetParent(hwnd);
				int  ID = (int)GetWindowLong(hwnd,GWL_ID);
				SendMessage(hwndParent,WM_CHILDUPDATED,(WPARAM)ID,0);
				return TRUE;
			}

		case EN_SETFOCUS:
			{
				HWND hwndParent = GetParent(hwnd);
				int  ID = (int)GetWindowLong(hwnd,GWL_ID);
				SendMessage(hwndParent,WM_CHILDGOTFOCUS,(WPARAM)ID,0);
				return TRUE;
			}
		}
		return TRUE;

	case WM_SIZE:
		pthis->Resize(LOWORD(lParam),HIWORD(lParam));
		return TRUE;

	case WM_DESTROY:
		pthis->DestroyUI();
		return TRUE;
	}

	return DefWindowProc(hwnd,msg,wParam,lParam);
}

void SpinEdit::Attach(HWND theHWnd)
{
	hwnd=theHWnd;
	DestroyUI();
	BuildControlUI();
}

void SpinEdit::SetLimits(float min, float max)
{
	this->min =min;
	this->max =max;

	if (ISpin)
		ISpin->SetLimits(min,max);
}

void SpinEdit::SetIncr(float incr)
{
	this->incr=incr;

	if (ISpin)
		ISpin->SetScale(incr);
}

void SpinEdit::SetValue(float val)
{
	if (ISpin)
	{
		ISpin->SetValue(val,TRUE);
	}

	value = val;
}

void SpinEdit::GetValue(CStr& str)
{
	char tmpstr[256];

	if (IEdit)
	{
		IEdit->GetText(tmpstr,255);
		str = CStr(tmpstr);
	}
	else
	{
		sprintf(tmpstr,"%f",value);
		str = tmpstr;
	}
}

float SpinEdit::GetValue()
{
	if (IEdit)
		return IEdit->GetFloat();

	return 0.0f;
}

HWND SpinEdit::GetFocusWindow()
{
	return hwndEdit;
}
