/*
	RichText.cpp
	Support class for RichText Edit Controls

	Adam Lippmann
	12-13-01

	HISTORY:
	5-8-01	:  Stream data is now kept in memory until the callback function says it's
	           completed reading it.  Was getting random crashes as it appears that
			   RichEdit runs the callback in a seperate thread that can still be active
			   after an EM_STREAMIN message
*/

#include "RichText.h"
#include "../PropEdit/ParseFuncs.h"
#include "../Resource.h"

RichText::RichText(HINSTANCE hInstance,HWND hwndRTF)
{
	this->hInstance=hInstance;
	hwnd=hwndRTF;

	// Clear out search notification vars
	fpNotifyComplete =NULL;
	fpNotifyCancelled=NULL;
	pNotifyCompleteData=NULL;
	pNotifyCancelledData=NULL;

	SendMessage(hwndRTF,EM_SETLIMITTEXT,(WPARAM)RICHTEXT_SIZELIMIT,0);
	SendMessage(hwndRTF,EM_SETEVENTMASK,0,(LPARAM)(DWORD)ENM_KEYEVENTS|ENM_MOUSEEVENTS|ENM_CHANGE);
}

RichText::~RichText()
{
	// No longer applicable
	//LocalFree(hMem);
}

CStr RichText::GetWord()
{
	UINT curPos;
	char lineBuf[1024];
	char wordBuf[1024];
	int  line;
	int  lineIndex;
	int  xPos;
	int  wordStart,wordEnd;
	int  lineLen;
	WORD bufSize=512;

	// According to the spec, the first WORD of the buffer specifies it's size
	// Windows then overwrites the data over the word at the beginning of the buffer
	memcpy(lineBuf,&bufSize,sizeof(WORD));

	// Retrieve the cursor position and the line containing the cursor
	SendMessage(hwnd,EM_GETSEL,(WPARAM)&curPos,NULL);
	line=SendMessage(hwnd,EM_LINEFROMCHAR,(WPARAM)curPos,0);

	SendMessage(hwnd,EM_GETLINE,(WPARAM)(int)line,(LPARAM)lineBuf);

	lineLen=strlen(lineBuf);

	// Figure out the x position of the cursor for this line
	lineIndex=SendMessage(hwnd,EM_LINEINDEX,(WPARAM)line,0);
	xPos=curPos-lineIndex-1;


	// Now scan the line to the left for spaces to determine the word start pos
	for(wordStart=xPos;wordStart>0;wordStart--)
		if (lineBuf[wordStart]==' '  ||
			lineBuf[wordStart]=='\t' ||
			lineBuf[wordStart]=='\r' ||
			lineBuf[wordStart]=='\n')
		{
			wordStart++;
			break;
		}

	// Now scan the line to the right for spaces to determine the word end pos
	for(wordEnd=xPos;wordEnd<lineLen;wordEnd++)
		if (lineBuf[wordEnd]==' '  ||
			lineBuf[wordEnd]=='\t' ||
			lineBuf[wordEnd]=='\r' ||
			lineBuf[wordEnd]=='\n')
		{
			//wordEnd--;
			break;
		}

	// Now build the word
	int i=0;

	for(int linePos=wordStart;linePos<wordEnd;linePos++)
	{
		if (i == bufSize - 2)
			break;

		wordBuf[i++]=lineBuf[linePos];
	}

	wordBuf[i]='\0';

	return CStr(wordBuf);
}

CStr RichText::GetLine(int lineNum)
{
	char lineBuf[512];
	WORD bufSize=512;
	int  curPos;
	int  lineLen;

	// According to the spec, the first WORD of the buffer specifies it's size
	// Windows then overwrites the data over the word at the beginning of the buffer
	memcpy(lineBuf,&bufSize,sizeof(WORD));

	if (lineNum==-1)
	{
		// Retrieve the cursor position and the line containing the cursor
		SendMessage(hwnd,EM_GETSEL,(WPARAM)&curPos,NULL);
		lineNum=SendMessage(hwnd,EM_LINEFROMCHAR,(WPARAM)curPos,0);
	}

	SendMessage(hwnd,EM_GETLINE,(WPARAM)lineNum,(LPARAM)lineBuf);
	lineLen=strlen(lineBuf);

	// Strip char 13
	for(int i=0;i<lineLen;i++)
	{
		if (i >= 510)
		{
			lineBuf[i]='\0';
			break;
		}

		if (lineBuf[i]==13)
		{
			lineBuf[i]='\0';
			break;
		}
	}

	return CStr(lineBuf);	
}



void RichText::SetLine(char* newLine,int lineNum,SetLineMode mode)
{
	DWORD oldPosStart,oldPosEnd;			// Start and ending positions of previous text selection
	int   linePos;
	int   lineLen;

	//char buf[1024];
	//sprintf(buf,"Line: %s\n",newLine);
	//OutputDebugString(buf);

	// What we're going to do (Since we can't do a EM_GETHANDLE [Unsupported])
	// and Get/SetWindowText would be way too expensive (it copies the entire text buffer and
	// we'd have to do that every space, up, down, left, right, backspace, etc.)

	// Select the last line of text, and use EM_STREAMIN to pump
	// the newLine content into a temporary selection

	// Turn off selection highlighting and scrolling so there's no flicker

	SendMessage(hwnd,EM_HIDESELECTION,(WPARAM)TRUE,0);
	SendMessage(hwnd,EM_SETOPTIONS,(WPARAM)ECOOP_AND,(LPARAM)~(ECO_AUTOVSCROLL|ECO_AUTOHSCROLL));

	// Retrieve cursor pos
	SendMessage(hwnd,EM_GETSEL,(WPARAM)&oldPosStart,(LPARAM)&oldPosEnd);

	if (lineNum==-1)
	{
		// Retrieve the line containing the cursor
		lineNum=SendMessage(hwnd,EM_LINEFROMCHAR,(WPARAM)oldPosStart,0);
	}

	linePos=SendMessage(hwnd,EM_LINEINDEX,(WPARAM)lineNum,0);
	lineLen=SendMessage(hwnd,EM_LINELENGTH,(WPARAM)linePos,0);

	// Set selection to the specified line
	SendMessage(hwnd,EM_SETSEL,(WPARAM)linePos,(LPARAM)(linePos+lineLen));

	// Replace the line with streamin
	EDITSTREAM* editstream=new EDITSTREAM;
	StreamData* sData=new StreamData;

	sData->pBuf     =new char[strlen(newLine)+1];
	strcpy(sData->pBuf,newLine);

	sData->bufsize  =strlen(newLine);
	sData->pos      =0;
	sData->pRichText=this;

	editstream->dwCookie   =(DWORD)sData;
	editstream->dwError    =0;
	editstream->pfnCallback=EditStreamCB;

	try{
	switch(mode)
	{
	case SETLINE_RTF:
		SendMessage(hwnd,EM_STREAMIN,(WPARAM)(UINT)SF_RTF|SFF_SELECTION,(LPARAM)editstream);
		break;

	case SETLINE_TXT:
		SendMessage(hwnd,EM_STREAMIN,(WPARAM)(UINT)SF_TEXT|SFF_SELECTION,(LPARAM)editstream);
		break;
	}
	}
	catch(...)
	{
		OutputDebugString("Got Exception!");
	}

	// Return to the previous selection
	SendMessage(hwnd,EM_SETSEL,(WPARAM)oldPosStart,(LPARAM)oldPosEnd);

	// Turn on selection highlighting and scrolling again now that we're done
	SendMessage(hwnd,EM_HIDESELECTION,(WPARAM)FALSE,0);
	SendMessage(hwnd,EM_SETOPTIONS,(WPARAM)ECOOP_SET,(LPARAM)ECO_AUTOVSCROLL|ECO_AUTOHSCROLL);
}

int RichText::GetLineNum()
{
	int curPos;

	// Retrieve cursor pos
	SendMessage(hwnd,EM_GETSEL,(WPARAM)&curPos,0);

	// Retrieve line containing cursor
	return (int)SendMessage(hwnd,EM_LINEFROMCHAR,(WPARAM)curPos,0);
}

DWORD CALLBACK RichText::EditStreamCB(DWORD dwCookie,LPBYTE buf,LONG numBytes,LONG* bytesTransf)
{
	StreamData* pData=(StreamData*)dwCookie;
	int bytesToCopy;

	//char dbuf[1024];
	//sprintf(dbuf,"CBLine: %s\n",pData->pBuf);
	//OutputDebugString(dbuf);

	// Check if we're still streaming in even after we're done
	if (pData->pos==pData->bufsize)
	{
		// STOP
		//__asm int 3;
		*bytesTransf=0;

		delete pData->pBuf;
		delete pData;
		return 0;
	}

	if (pData->bufsize<numBytes)
		bytesToCopy=pData->bufsize;		// This should almost always be the case
	else
		bytesToCopy=numBytes;

	// Copy our information into the control's buffer
	for(int i=pData->pos;i<bytesToCopy;i++)
	{
		buf[i]=pData->pBuf[i];
	}

	// Update the position marker in case it didn't fit in their buffer
	pData->pos+=bytesToCopy;

	*bytesTransf=bytesToCopy;
	
	if (bytesToCopy>=pData->bufsize)
	{
		delete pData->pBuf;
		delete pData;
	}

	return 0;
}

BOOL RichText::ProcMessage(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	return FALSE;
}

int RichText::GetCurPos()
{
	int pos;
	SendMessage(hwnd,EM_GETSEL,(WPARAM)&pos,0);

	return pos;
}

POINT RichText::GetScreenPos(int index)
{
	DWORD  coordPos;
	POINT  pos;
	RECT   rect;

	if (index==-1)
		index=GetCurPos();

	coordPos=SendMessage(hwnd,EM_POSFROMCHAR,(WPARAM)index,0);

	pos.x=LOWORD(coordPos);
	pos.y=HIWORD(coordPos);

	GetWindowRect(hwnd,&rect);
	pos.x+=rect.left;
	pos.y+=rect.top;

	return pos;
}

bool RichText::HasText()
{
	int lines=SendMessage(hwnd,EM_GETLINECOUNT,0,0);

	if (lines<2)
	{
		int linelen=SendMessage(hwnd,EM_LINELENGTH,0,0);

		if (linelen==0)
			return false;
	}

	return true;
}

void RichText::Clear()
{
	SendMessage(hwnd,WM_SETTEXT,0,(LPARAM)"");
}

CStr RichText::GetText()
{
	CStr  str;
	char* tmpBuf;
	int   txtLen=GetWindowTextLength(hwnd)+1;

	tmpBuf=(char*)malloc(txtLen);
	GetWindowText(hwnd,tmpBuf,txtLen);
	str=CStr(tmpBuf);
	free(tmpBuf);

	return str;
}

void RichText::GetText(char* str,int count)
{
	GetWindowText(hwnd,str,count);
}

void RichText::SetText(CStr str)
{
	SetWindowText(hwnd,(char*)str);
}

int RichText::Search(CStr strSearch,DWORD flags,BOOL bSearchRange)
{
	DWORD selStart,selEnd;
	SendMessage(hwnd,EM_GETSEL,(WPARAM)&selStart,(LPARAM)&selEnd);

	FINDTEXT findtext;

	if (!bSearchRange)
	{
		findtext.chrg.cpMin=selEnd;
		findtext.chrg.cpMax=-1;
	}
	else
	{
		findtext.chrg.cpMin=selStart;
		findtext.chrg.cpMax=selEnd;
	}

	findtext.lpstrText=(char*)strSearch;

	return (int)SendMessage(hwnd,EM_FINDTEXT,(WPARAM)flags,(LPARAM)&findtext);
}

void RichText::Highlight(int start,int end)
{
	SendMessage(hwnd,EM_SETSEL,(WPARAM)start,(LPARAM)end);
}

void RichText::HighlightLine(int index)
{
	int lineNum=SendMessage(hwnd,EM_LINEFROMCHAR,(WPARAM)index,0);	
	int lineIndex=SendMessage(hwnd,EM_LINEINDEX,(WPARAM)lineNum,0);
	int lineLen=SendMessage(hwnd,EM_LINELENGTH,(WPARAM)index,0);
	int lineEnd=lineIndex+lineLen;

	SendMessage(hwnd,EM_SETSEL,(WPARAM)lineIndex,(LPARAM)lineEnd);
}

void RichText::HighlightLineNum(int linenum)
{
	int lineIndex=SendMessage(hwnd,EM_LINEINDEX,(WPARAM)linenum,0);
	HighlightLine(lineIndex);
}

void RichText::Replace(CStr strReplace)
{
	SendMessage(hwnd,EM_REPLACESEL,(WPARAM)TRUE,(LPARAM)(char*)strReplace);
}

void RichText::ClipboardCut()
{
	// Copy the data to the clipboard
	ClipboardCopy();
	
	// Clear the selected data
	SendMessage(hwnd,EM_REPLACESEL,(WPARAM)TRUE,(LPARAM)"");
}

void RichText::ClipboardCopy()
{
	/*
	HANDLE buffer;
	char*  pBuf;

	if (OpenClipboard(hwnd))
	{
		int bufLen=GetWindowTextLength(hwnd);

		// System owns so we don't have to free
		buffer=GlobalAlloc(GHND|GMEM_SHARE,bufLen);
		
		pBuf=(char*)GlobalLock(buffer);
		SendMessage(hwnd,EM_GETSELTEXT,0,(LPARAM)pBuf);
		GlobalUnlock(buffer);

		EmptyClipboard();

		if (!SetClipboardData(CF_TEXT,buffer))
		{
			// Clipboard set failed, free our data
			GlobalFree(buffer);
		}

		CloseClipboard();
	}
	*/

	SendMessage(hwnd,WM_COPY,0,0);
}

void RichText::ClipboardPaste()
{
	/*
	if (OpenClipboard(hwnd))
	{
		//HANDLE hMem=GetClipboardData(CF_TEXT);
		HANDLE hMem=GetClipboardData(CF_UNICODETEXT);

		if (hMem)
			SendMessage(hwnd,EM_REPLACESEL,(WPARAM)TRUE,(LPARAM)hMem);

		CloseClipboard();
	}
	*/

	SendMessage(hwnd,WM_PASTE,0,0);
}

void RichText::SearchDlg(void(*NotifyComplete)(void*),void* pDataComplete,
						 void(*NotifyCancelled)(void*),void* pDataCancelled)
{
	fpNotifyComplete     = NotifyComplete;
	pNotifyCompleteData  = pDataComplete;
	fpNotifyCancelled    = NotifyCancelled;
	pNotifyCancelledData = pDataCancelled;

	hwndSearch=CreateDialog(hInstance,MAKEINTRESOURCE(IDD_SEARCH),hwnd,RedirectSearchDlgProc);
	SetWindowLong(hwndSearch,GWL_USERDATA,(LONG)this);
	SetWindowPos(hwndSearch,HWND_TOPMOST,0,0,0,0,SWP_NOMOVE|SWP_NOSIZE|SWP_SHOWWINDOW);
	ShowWindow(hwndSearch,SW_SHOW);
}

void RichText::ReplaceDlg(void(*NotifyComplete)(void*),void* pDataComplete,
						  void(*NotifyCancelled)(void*),void* pDataCancelled,
						  void(*NotifyReplaceAll)(void*),void* pDataReplaceAll)
{
	fpNotifyComplete      = NotifyComplete;
	pNotifyCompleteData   = pDataComplete;
	fpNotifyCancelled     = NotifyCancelled;
	pNotifyCancelledData  = pDataCancelled;
	fpNotifyReplaceAll    = NotifyReplaceAll;
	pNotifyReplaceAllData = pDataReplaceAll;

	hwndReplace=CreateDialog(hInstance,MAKEINTRESOURCE(IDD_REPLACE),hwnd,RedirectReplaceDlgProc);
	SetWindowLong(hwndReplace,GWL_USERDATA,(LONG)this);
	SetWindowPos(hwndReplace,HWND_TOPMOST,0,0,0,0,SWP_NOMOVE|SWP_NOSIZE|SWP_SHOWWINDOW);
	ShowWindow(hwndReplace,SW_SHOW);
}

BOOL CALLBACK RichText::RedirectSearchDlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	RichText* thisclass=(RichText*)GetWindowLong(hwnd,GWL_USERDATA);
	return thisclass->SearchDlgProc(hwnd,msg,wParam,lParam);
}

BOOL RichText::SearchDlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	switch(msg)
	{
	case WM_CLOSE:
		EndDialog(hwnd,0);
		return TRUE;

	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			EnableAccelerators();
		else
			DisableAccelerators();

		return TRUE;
		
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_FINDNEXT:
			FindNext();
			return TRUE;

		case IDC_CANCEL:
			if (fpNotifyCancelled)
				fpNotifyCancelled(pNotifyCancelledData);

			EndDialog(hwnd,0);
			return TRUE;
		}

		return TRUE;
	}

	return FALSE;
}

void RichText::FindNext()
{
	DWORD flags=0;
	BOOL  bSearchRange;
	char  buffer[256];

	if (IsDlgButtonChecked(hwndSearch,IDC_WHOLEWORD))
		flags|=RTS_WHOLEWORD;

	if (IsDlgButtonChecked(hwndSearch,IDC_MATCHCASE))
		flags|=RTS_MATCHCASE;

	GetDlgItemText(hwndSearch,IDC_SEARCHEDIT,buffer,256);

	if (IsDlgButtonChecked(hwndSearch,IDC_SEARCHRANGE))
		bSearchRange=TRUE;
	else
		bSearchRange=FALSE;

	int index=Search(buffer,flags,bSearchRange);

	if (index!=-1)
		Highlight(index,index+strlen(buffer));
	else
	{
		if (fpNotifyComplete)
		{
			fpNotifyComplete(pNotifyCompleteData);
			return;
		}

		char errBuf[256];
		sprintf(errBuf,"Couldn't find '%s'.",buffer);
		MessageBox(hwnd,errBuf,"Search",MB_ICONINFORMATION|MB_OK);
	}
}

bool RichText::RFindNext(bool bWarn)
{
	DWORD flags=0;
	char  buffer[256];

	if (IsDlgButtonChecked(hwndReplace,IDC_RWHOLEWORD))
		flags|=RTS_WHOLEWORD;

	if (IsDlgButtonChecked(hwndReplace,IDC_RMATCHCASE))
		flags|=RTS_MATCHCASE;

	GetDlgItemText(hwndReplace,IDC_EDITRSEARCH,buffer,256);

	int index=Search(buffer,flags);

	if (index!=-1)
		Highlight(index,index+strlen(buffer));
	else
	{
		if (bWarn)
		{
			if (fpNotifyComplete)
			{
				fpNotifyComplete(pNotifyCompleteData);
				return false;
			}

			char errBuf[256];
			sprintf(errBuf,"Couldn't find '%s'.",buffer);
			MessageBox(hwnd,errBuf,"Search and Replace",MB_ICONINFORMATION|MB_OK);
		}

		return false;
	}

	return true;
}

bool RichText::Replace(bool bWarn)
{
	char  buffer[256];
	int   startPos,endPos;

	SendMessage(hwnd,EM_GETSEL,(WPARAM)&startPos,(LPARAM)&endPos);

	// Perform replace if there's a current selection
	if (endPos>startPos)
	{
		GetDlgItemText(hwndReplace,IDC_EDITREPLACE,buffer,256);
		SendMessage(hwnd,EM_REPLACESEL,(WPARAM)TRUE,(LPARAM)buffer);
	}

	return RFindNext(bWarn);
}

void RichText::ReplaceAll()
{
	DWORD flags=0;
	char  buffer[256];

	if (IsDlgButtonChecked(hwndReplace,IDC_RWHOLEWORD))
		flags|=RTS_WHOLEWORD;

	if (IsDlgButtonChecked(hwndReplace,IDC_RMATCHCASE))
		flags|=RTS_MATCHCASE;

	GetDlgItemText(hwndReplace,IDC_EDITRSEARCH,buffer,256);

	int index=Search(buffer,flags);

	while(Replace(FALSE));
	Replace(FALSE);

	if (fpNotifyComplete)
	{
		fpNotifyComplete(pNotifyCompleteData);
		return;
	}

	MessageBox(hwnd,"Search Completed.","Search and Replace",MB_ICONINFORMATION|MB_OK);
}

BOOL CALLBACK RichText::RedirectReplaceDlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	RichText* thisclass=(RichText*)GetWindowLong(hwnd,GWL_USERDATA);
	return thisclass->ReplaceDlgProc(hwnd,msg,wParam,lParam);
}

BOOL RichText::ReplaceDlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	switch(msg)
	{
	case WM_CLOSE:
		EndDialog(hwnd,0);
		return TRUE;

	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			EnableAccelerators();
		else
			DisableAccelerators();

		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_RFINDNEXT:
			RFindNext();
			return TRUE;

		case IDC_REPLACE:
			Replace();
			return TRUE;

		case IDC_REPLACEALL:
			if (fpNotifyReplaceAll)
				fpNotifyReplaceAll(pNotifyReplaceAllData);

			ReplaceAll();
			return TRUE;

		case IDC_RCANCEL:
			if (fpNotifyCancelled)
				fpNotifyCancelled(pNotifyCancelledData);

			EndDialog(hwnd,0);
			return TRUE;
		}
	}

	return FALSE;
}

void RichText::CloseSearch()
{
	EndDialog(hwndSearch,0);
}

void RichText::CloseReplace()
{
	EndDialog(hwndReplace,0);
}

void RichText::LoadDlg(char* Title,char* Filter,char* DefaultExt,CStr& Filename)
{
	OPENFILENAME ofn;
	char filename[256]="";

	// Fill out the arduously long openfilename struct
	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=hwnd;
	ofn.hInstance=hInstance;
	ofn.lpstrFilter=Filter;
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=NULL;
	ofn.nMaxFileTitle=0;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle=Title;
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=DefaultExt;
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetOpenFileName(&ofn);

	Filename=CStr(filename);
}

int RichText::SaveDlg(char* Title,char* Filter,char* DefaultExt,CStr& Filename)
{
	OPENFILENAME ofn;
	char filename[256]="";

	// Fill out the arduously long openfilename struct
	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=hwnd;
	ofn.hInstance=hInstance;
	ofn.lpstrFilter=Filter;
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=NULL;
	ofn.nMaxFileTitle=0;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle=Title;
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=DefaultExt;
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetSaveFileName(&ofn);

	Filename=CStr(filename);

	return ofn.nFilterIndex;
}

bool RichText::SaveDlg(char* Title,char* DefaultExt)
{
	CStr Filename;

	int index=SaveDlg(Title,"Script Files (*.q)\0*.q\0Text File (*.txt)\0*.txt\0RTF File (*.rtf)\0*.rtf\0\0",DefaultExt,Filename);
	
	if (index==0)
		return false;

	if (index<3)
		return SaveTXT(Filename);

	return SaveRTF(Filename);
}

DWORD RichText::SaveStreamCB(DWORD dwCookie,LPBYTE buf,LONG numBytes,LONG* bytesTransf)
{
	FILE* fp=(FILE*)dwCookie;
	*bytesTransf=fwrite(buf,1,numBytes,fp);

	if (*bytesTransf>0)
		return 0;

	return -1;
}

DWORD RichText::SaveStreamTextCB(DWORD dwCookie,LPBYTE buf,LONG numBytes,LONG* bytesTransf)
{
	char* tmpbuf;
	CStr* strBuf=(CStr*)dwCookie;
	*bytesTransf=numBytes;

	tmpbuf=(char*)malloc(numBytes+2);

	for(int i=0;i<numBytes;i++)
		tmpbuf[i]=buf[i];
	
	tmpbuf[i]='\0';
	(*strBuf)=(*strBuf)+CStr(tmpbuf);

	free(tmpbuf);

	return 0;
}

bool RichText::SaveTXT(char* Filename)
{
	EDITSTREAM  editstream;
	
	FILE* fp=fopen(Filename,"w");
	CStr  strBuf;

	editstream.dwCookie=(DWORD)&strBuf;
	editstream.dwError =0;
	editstream.pfnCallback=SaveStreamTextCB;

	int numBytes=SendMessage(hwnd,EM_STREAMOUT,(WPARAM)SF_TEXT,(LPARAM)&editstream);
	
	// Fix eol sequence from CRCRLF to CRLF (if that's the case)
	strBuf=ReplaceStr(strBuf,"\r\n","\n");
	fwrite(strBuf,strBuf.Length(),1,fp);
	
	fclose(fp);

	if (numBytes>0)
		return true;

	return false;
}

bool RichText::SaveTXT(FILE* fp)
{
	EDITSTREAM  editstream;

	if (!fp)
		return false;

	CStr  strBuf;

	editstream.dwCookie=(DWORD)&strBuf;
	editstream.dwError =0;
	editstream.pfnCallback=SaveStreamTextCB;

	int numBytes=SendMessage(hwnd,EM_STREAMOUT,(WPARAM)SF_TEXT,(LPARAM)&editstream);
	
	// Fix eol sequence from CRCRLF to CRLF (if that's the case)
	strBuf=ReplaceStr(strBuf,"\r\n","\n");
	fwrite(strBuf,strBuf.Length(),1,fp);

	if (numBytes>0)
		return true;

	return false;
}

bool RichText::SaveRTF(char* Filename)
{
	EDITSTREAM  editstream;
	
	FILE* fp=fopen(Filename,"w");

	editstream.dwCookie=(DWORD)fp;
	editstream.dwError =0;
	editstream.pfnCallback=SaveStreamCB;

	int numBytes=SendMessage(hwnd,EM_STREAMOUT,(WPARAM)SF_RTF,(LPARAM)&editstream);
	fclose(fp);

	if (numBytes>0)
		return true;

	return false;
}

void RichText::GoToEnd()
{
	DWORD curPos;
	int   lineNum,lineIndex,lineLen,newPos;

	// Retrieve the cursor position and the line containing the cursor
	SendMessage(hwnd,EM_GETSEL,(WPARAM)&curPos,NULL);
	lineNum=SendMessage(hwnd,EM_LINEFROMCHAR,(WPARAM)curPos,0);

	lineIndex=SendMessage(hwnd,EM_LINEINDEX,(WPARAM)lineNum,0);
	lineLen=SendMessage(hwnd,EM_LINELENGTH,(WPARAM)lineIndex,0);

	newPos=lineIndex+lineLen;

	SendMessage(hwnd,EM_SETSEL,(WPARAM)(INT)newPos,(LPARAM)(INT)newPos);
}

void RichText::DisableText()
{
	SendMessage(hwnd,EM_SETOPTIONS,(WPARAM)ECOOP_OR,(LPARAM)ECO_READONLY);
}

void RichText::EnableText()
{
	DWORD options = (DWORD)SendMessage(hwnd,EM_GETOPTIONS,0,0);
	options &= ~ECO_READONLY;

	SendMessage(hwnd,EM_SETOPTIONS,(WPARAM)ECOOP_SET,(LPARAM)options);
}
