/*
	MultiList.h
	This is a list control that can contain lists within lists for
	selecting things from a hierarchy
*/

#ifndef __MULTILIST__
#define __MULTILIST__

#include "UIControl.h"
#include "max.h"
#include "../misc/llist.h"

#define MULTILIST_CLASSNAME "MultiList"

#define WM_GETTEXTLAST  WM_USER + 0x0012
#define WM_RESETLAST    WM_USER + 0x0013

#define GetDlgItemTextLast(hwnd,id,string,max) \
	SendMessage(GetDlgItem(hwnd,id),WM_GETTEXTLAST,(WPARAM)max,(LPARAM)string);

#define ResetLastText(hwnd,id) \
	SendMessage(GetDlgItem(hwnd,id),WM_RESETLAST,0,0);

struct TmpMenuRec
{
	CStr name;
	bool bUnique;
};

struct MenuDesc
{
	CStr  menuName;
	HMENU hMenu;

	LinkList<MenuDesc> submenus;

	bool operator== (MenuDesc& right)
	{
		if (menuName == right.menuName)
			return true;

		return false;
	}
};

struct NameDesc
{
	CStr  name;
	CStr  path;
	DWORD data;

	NameDesc()
	{
		data = 0;
	}

	bool operator== (NameDesc& right)
	{
		if (name == right.name &&
			path == right.path)
			return true;

		return false;
	}

	static bool CompareName(NameDesc* pthis, NameDesc* pother)
	{
		if (pthis->name == pother->name)
			return true;

		return false;
	}

	static bool ComparePath(NameDesc* pthis, NameDesc* pother)
	{
		if (pthis->path == pother->path)
			return true;

		return false;
	}
};

class MultiList: public UIControl
{
	HWND  hwndButton;			// UI button that opens the dropdown multilist
	HMENU hMenu;				// Main menu
	HMENU hPopupMenu;			// The menu that is actually displayed and used
	int   lastMenuID;
	int   lastSelID;			// Last selected item ID
	int   oldSelID;				// item ID selected before the last selected item
	bool  bDispFullPath;		// If true the menu path is displayed
	bool  bEvalFullPath;		// If true the menu path is returned in the evaluation
	bool  bAck;					// Acknowledge that changes have been made to control selection
	bool  bGroupText;			// True if []'s should be placed around sel text

	CStr  strValue;				// The value to be returned on GetText messages and GetValue()
	CStr  strDelimOut;			// User defined menu delimiter for output only
	CStr  curValue;
	CStr  lastValue;

	LinkList<MenuDesc>   menus;		// Holds on to the created submenus so everything can be deleted
	LinkList<NameDesc>   names;		// Name list corresponds to assigned item IDs for the entire popup menu
	LinkList<TmpMenuRec> buildcmds;	// List of construction params from AddItem in the event that the MultiList
									// hasn't been created yet we need to hold on to these until the menu is constructed

	void  BuildControlUI();

	LRESULT WndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

	void Destroy();
	bool SplitNamePath(NameDesc& nd, char* str);

	int  AddItemInternal(char* str);
	int  AddItemUniqueInternal(char* str);

public:
	CLASSNAME(MULTILIST_CLASSNAME)

	MultiList(HINSTANCE hInstance);
	MultiList(HINSTANCE hInstance, HWND hwnd);
	~MultiList();

	MultiList(MultiList& right);
	MultiList& operator = (MultiList& right);

	// Items added to the MultiList follow the following syntax
	// "List A/SubList B/SubsubList A/subsubsubList D/...etc.../My Item"
	int   AddItem(char* str);
	int   AddItemUnique(char* str);
	void  Reset();
	void  ResetKeepText();

	void  SetFont(HFONT hFont);
	void  SetGroupText(bool bVal);				// True if []'s should be placed around sel text

	void  OpenMenu();

	CStr  GetValue();
	CStr  GetAbsValue();						// Retrieves absolute value regardless of EvalFullPath
	void  SetValue(CStr val,bool bNotify=true);

	DWORD GetUserData(int id);
	void  SetUserData(int id,DWORD val);

	void  SortMenus();

	void  EvalWindowText();			// Evaluates the text displayed in the window
	void  BuildStoredList();

	inline int GetCurSel() { return SendMessage(hwnd,CB_GETCURSEL,0,0); }
	inline void SetCurSel(int i) { SendMessage(hwnd,CB_SETCURSEL,(WPARAM)i,0); }

	inline void DispFullPath(bool bVal) { bDispFullPath = bVal; }
	inline void EvalFullPath(bool bVal) { bEvalFullPath = bVal; }
	inline void SetOutDelim(CStr delim) { strDelimOut = delim;  }

	inline void DisableAck() { bAck = false; }
	inline void EnableAck()  { bAck = true;  }

	inline bool GetAck()     { return bAck;  }

	inline CStr GetLastValue() { return lastValue; }

	void ResetLast();
};

#endif
