/*
	DisplayDevice.cpp
	Maintains display adapter information
	aml - 11-21-02
*/

#include "DisplayDevice.h"

DisplayDevice::DisplayDevice()
{
	hMonitor   = NULL;
	device     = NULL;
	texDefault = NULL;

	ZeroMemory(&caps, sizeof(D3DCAPS8));
	ZeroMemory(&ident, sizeof(D3DADAPTER_IDENTIFIER8));
	ZeroMemory(&mode, sizeof(D3DDISPLAYMODE));
	ZeroMemory(&d3dpp, sizeof(D3DPRESENT_PARAMETERS));
	
	id                = 0;
	nVertBuffers      = 0;
	nIndexBuffers     = 0;
	bHardwareVBuffers = false;
}

DisplayDevice::~DisplayDevice()
{

}

Link<IndexBuffer>* DisplayDevice::AddIndexBuffer(int size)
{
	Link<IndexBuffer>* link = indexBuffers.Add();
	if (FAILED(device->CreateIndexBuffer(size * sizeof(D3DFace),
			                                0,
											D3DFMT_INDEX16,
											D3DPOOL_DEFAULT,
											&link->data.indexBuffer)))
	{
		indexBuffers.Remove(link);
		return NULL;
	}

    return link;
}

void DisplayDevice::RemoveIndexBuffer(Link<IndexBuffer>* iBuffer)
{
	indexBuffers.Remove(iBuffer);
}

bool DisplayDevice::AssignIndexData(Link<IndexBuffer>* ibuffer, D3DFace* faces, int nFaces)
{
	BYTE* pIndicies;
	if (FAILED(ibuffer->data.indexBuffer->Lock(0, sizeof(D3DFace) * nFaces, &pIndicies, 0)))
		return false;

	memcpy(pIndicies, faces, sizeof(D3DFace) * nFaces);
	ibuffer->data.indexBuffer->Unlock();

	return true;
}

Link<VertexBuffer>* DisplayDevice::AddVertBuffer(int size)
{
	Link<VertexBuffer>* link = vertBuffers.Add();
	if (FAILED(device->CreateVertexBuffer(size * sizeof(D3DVert),
											0,
											D3DVert::Type,
											D3DPOOL_DEFAULT,
											&link->data.vertBuffer)))
	{
		vertBuffers.Remove(link);
		return NULL;
	}

	return link;
}

void DisplayDevice::RemoveVertBuffer(Link<VertexBuffer>* vBuffer)
{
	vertBuffers.Remove(vBuffer);
}

bool DisplayDevice::AssignVertData(Link<VertexBuffer>* vbuffer, D3DVert* verts, int nVerts)
{
	BYTE* pVerts;
	if (FAILED(vbuffer->data.vertBuffer->Lock(0, sizeof(D3DVert) * nVerts, &pVerts, 0)))
		return false;

	memcpy(pVerts, verts, sizeof(D3DVert) * nVerts);
	vbuffer->data.vertBuffer->Unlock();

	return true;
}

Link<Texture>* DisplayDevice::AddTexture(const char* filename)
{
	Link<Texture>* texture = textures.Add();

	if (FAILED(D3DXCreateTextureFromFile( device, filename, &texture->data.texture )))
	{
		textures.Remove(texture);
		return NULL;
	}

	return texture;
}

void DisplayDevice::SetMatrices()
{
	device->SetTransform( D3DTS_WORLD,      &matWorld );
	device->SetTransform( D3DTS_VIEW,       &matView );
	device->SetTransform( D3DTS_PROJECTION, &matProj );
}

void DisplayDevice::FreeTextures()
{
	Link<Texture>* curtex  = textures.GetHead();
	Link<Texture>* nexttex;

	while(curtex)
	{
		nexttex = curtex->next;
		textures.Remove(curtex);
		curtex = nexttex;
	}
}

void DisplayDevice::FreeVertBuffers()
{
	Link<VertexBuffer>* curvbuf = vertBuffers.GetHead();
	Link<VertexBuffer>* nextvbuf;

	while(curvbuf)
	{
		nextvbuf = curvbuf->next;
		vertBuffers.Remove(curvbuf);
		curvbuf = nextvbuf;
	}
}

void DisplayDevice::FreeIndexBuffers()
{
	Link<IndexBuffer>* curibuf = indexBuffers.GetHead();
	Link<IndexBuffer>* nextibuf;

	while(curibuf)
	{
		nextibuf = curibuf->next;
		indexBuffers.Remove(curibuf);
		curibuf = nextibuf;
	}
}

void DisplayDevice::FreeAll()
{
	FreeTextures();
	FreeVertBuffers();
	FreeIndexBuffers();
}
