/*
	NodeCRC.h
	Functions for determining when node data has changed
	and needs updating to speed script export
*/

#ifndef __NODECRC__
#define __NODECRC__

#include "max.h"

#define NODEFLAG_HAS_SCRIPT			0x0001	// Set if the node had a script when parsed
#define NODEFLAG_HAS_STATIC_PROG	0x0002	// Set if the node has a MAXScript attached that may be evaluated just once
                                            // and cached in the node for later fast use if the node hasn't changed

#define NODEFLAG_HAS_VOLATILE_PROG  0x0004  // Set if the node has a MAXScript attached that MUST be evaluated always
											// and should not be cached in the node.  This should be used if the
                                            // MAXScript alters its buffer with respect to solely geometrical changes
                                            // on the object.  As mesh changes will not cause a differing NodeCRC.
#define NODEFLAG_HAS_PTEX			0x0008	// Set if the node has a particle texture

struct NodeCRC
{
	unsigned long nameCRC;		// CRC for the name of the node at time of output
	unsigned long propBufCRC;	// CRC for node property buffer at time of output
	unsigned long propBufCRC2;	// Secondary CRC for property buffer (uses different algorithm)
								// was getting CRC collisions on differing data

	//unsigned long refSum;		// Combined XORs of the node handles for all nodes that referenced
								// this node at time of output (for links)
	Matrix3       TM;			// Node's transform matrix at time of output
	FILETIME      scriptsTime;	// Time of the last scripts.ini update since the node was output
	FILETIME      pluginTime;	// Time of the last plugin (next.gup) update since the node was output

	unsigned long nodeFlags;	// Flags that relate to conditions encountered when the node
	                            // output buffer was originally parsed, here to save reading more chunks
								// (THESE FLAGS ARE NOT USED IN NODE CRC COMPARISONS)

	NodeCRC();
	NodeCRC(INode* node);
	NodeCRC(NodeCRC& right);
	NodeCRC& operator =(NodeCRC& right);
	int operator ==(NodeCRC& right);

	void Store(INode* node);
	bool Retrieve(INode* node);
};

unsigned long GenAdditiveCRC(char* buf);
void AssignNodeOutput(INode* node, NodeCRC* ncrc, char* buf, char* bufScript);
bool GetNodeOutput(INode* node, NodeCRC* ncrc, CStr& out);
bool GetNodeOutputScript(INode* node, NodeCRC* ncrc, CStr& out);

void InitNodeCRCFileTimes();

#endif
