#include <Engine/Engine.h>
#include <Sk/Gamenet/ExportMsg.h>
#include <Gel/Scripting/Script.h>
#include <Gel/Scripting/Struct.h>
#include <Gel/Scripting/Utils.h>
#include <Export/ScriptExport.h>
#include <resource.h>
#include <max.h>
#include "LinkMan.h"
#include "LinkUI.h"
#include "LinkOptions.h"
#include "../Trigger/Trigger.h"
#include <hold.h>

class LinkMan: public ILinkMan
{	
	bool    m_bLocked;

public:
	LinkMan( void );

	// From ILinkMan
	bool    IsLocked( void );		// Returns true if selection related ops should be denied
	void    SetLock( bool bVal );
	void	ManyToOneLink( void );
	void	OneToManyLink( void );
	void	ChainLink( bool closed, bool enumerate = true );
	void	ReverseChainLink( bool closed );
	void	Unlink( void );
	void	SetLinkColor( Color color );
	Color	GetLinkColor( void );
	void	SetDisplayMode( int mode );
	int		GetDisplayMode( void );
	void	SetSelectChildren( bool select );
	bool	GetSelectChildren( void );
	void	SetSelectParents( bool select );
	bool	GetSelectParents( void );
	void	EnableAutoUpdate( bool enable );
	bool	AutoUpdateEnabled( void );

	void	RecalculateReferences( void );
	void    CleanAbandonedLinks( void );

	void	IncrementSystemSelectOpID( void );
	int		GetSystemSelectID( void );

	bool	IsMerging( void );
	void	OnPreMerge( void );
	void	OnPostMerge( void );
	void	OnPostOpen( void );
	void	OnPreSave( void );
	
	void	Link( INode *from, INode *to, class LinkRestoreObj *restore_obj = NULL );	
	bool	AlreadyLinked( INode *from, INode *to, bool &circular_ref );
		
	void	EnumerateSelectedNodes( bool reverse = false );
	bool	CancelledLinkOp( void );
	
	Tab		<INode *> m_SelectedNodes;
	Color	m_Color;
	bool	m_Merging;
	int		m_DisplayMode;
	bool	m_SelectChildren;
	bool	m_SelectParents;
	bool	m_AutoUpdateEnabled;
	int		m_SystemSelectOpID;
};

class LinkRestoreObj : public RestoreObj
{
public:
	LinkRestoreObj( LinkMan *link_man ) : m_LinkMan( link_man ) {}
	void Redo();
	void Restore( int isUndo );
	TSTR Description() { return TSTR(_T("NExtLinkRestore")); }

	LinkMan *m_LinkMan;
};

void LinkRestoreObj::Redo( void )
{	
	m_LinkMan->RecalculateReferences();
}

void LinkRestoreObj::Restore( int isUndo )
{
	m_LinkMan->RecalculateReferences();
}

LinkMan::LinkMan( void )
{
	m_Color = Color( 0.5f, 0.5f, 0.5f );
	m_Merging = false;
	m_DisplayMode = DISPLAY_MODE_ALL;
	m_SelectChildren = false;
	m_SelectParents = false;
	m_AutoUpdateEnabled = false;
	m_SystemSelectOpID = 0;
	m_bLocked = false;
}

void LinkMan::SetLock(bool bVal)
{
	m_bLocked = bVal;
}

Color	LinkMan::GetLinkColor( void )
{
	return m_Color;
}

void	LinkMan::SetLinkColor( Color color )
{
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;
	int num_selected;
	int num_children = root->NumberOfChildren();
	int i;

	num_selected = gInterface->GetSelNodeCount();

	m_Color = m_Color;

	if( num_selected == 1 )
	{
		for( i = 0; i < num_children; i++ )
		{
			node = root->GetChildNode(i);
			obj = node->EvalWorldState(0).obj;

			if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
			{
				LinkObject* link = (LinkObject*) obj;
				if( link->from_node->Selected())
				{
					link->m_Color = color;
				}
			}
		}	
	}
	else if( num_selected > 1 )
	{
		for( i = 0; i < num_children; i++ )
		{
			node = root->GetChildNode(i);
			obj = node->EvalWorldState(0).obj;

			if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
			{
				LinkObject* link = (LinkObject*) obj;
				if( link->from_node->Selected() && link->to_node->Selected())
				{
					link->m_Color = color;
				}
			}
		}
	}

	gInterface->ForceCompleteRedraw();
}

void	LinkMan::EnumerateSelectedNodes( bool reverse )
{
	int i, num_selected;
	INode *sel_node;
	Object *obj;

	m_SelectedNodes.Resize( 0 );
	num_selected = gInterface->GetSelNodeCount();
	for( i = 0; i < num_selected; i++ )
	{
		sel_node = gInterface->GetSelNode( i );
		obj = sel_node->EvalWorldState(0).obj;
		if( obj && obj->ClassID() != vLINK_OBJ_CLASS_ID )
		{
			if( reverse )
			{
				m_SelectedNodes.Insert( 0, 1, &sel_node );
			}
			else
			{
				m_SelectedNodes.Append( 1, &sel_node );
			}
		}
	}	
}

bool	LinkMan::IsLocked( void )
{
	return m_bLocked;
}

void	LinkMan::ManyToOneLink( void )
{	
	int num_selected;
	INode *to, *from;
	int i;
	bool circular_ref, cancelled;
	LinkRestoreObj *restore_obj;

	EnumerateSelectedNodes();

	cancelled = false;
	num_selected = m_SelectedNodes.Count();
	if( num_selected == 1 )
	{
#ifdef LINK_TO_SELF_ENABLED
		// The last one will be considered the target node for the link
		if( theHold.Holding() == false ) 
		{
			theHold.Begin();
		}

		restore_obj = new LinkRestoreObj( this );;
		theHold.Put( restore_obj );
		
		to = m_SelectedNodes[0];
		from = to;
		if( AlreadyLinked( from, to, circular_ref ) == false )
		{
			if( circular_ref )
			{
				if( CancelledLinkOp())
				{
					cancelled = true;		
					break;
				}
			}
			Link( from, to, restore_obj );
		} 		
		if( cancelled )
		{
			theHold.Cancel();
		}
		else
		{
			theHold.Accept(_T( "Many To One Link" ));
		}
		gInterface->RedrawViews( gInterface->GetTime());
#else
		return;
#endif
	}
	else
	{
		if( theHold.Holding() == false ) 
		{
			theHold.Begin();
		}

		restore_obj = new LinkRestoreObj( this );
		theHold.Put( restore_obj );

		to = m_SelectedNodes[num_selected - 1];
		for( i = 0; i < (num_selected - 1); i++ )
		{
			from = m_SelectedNodes[i];
			if( AlreadyLinked( from, to, circular_ref ) == false )
			{
				if( circular_ref )
				{
					if( CancelledLinkOp())
					{
						cancelled = true;
						break;						
					}
				}
				Link( from, to, restore_obj );
			} 
		}

		if( cancelled )
		{
			theHold.Cancel();
		}
		else
		{
			theHold.Accept(_T( "Many To One Link" ));		
		}
		gInterface->RedrawViews( gInterface->GetTime());
	}

	m_SelectedNodes.Resize( 0 );
}

void	LinkMan::OneToManyLink( void )
{
	int num_selected;
	bool circular_ref, cancelled;
	INode *to, *from;
	int i;
	LinkRestoreObj *restore_obj;

	EnumerateSelectedNodes();

	cancelled = false;
	num_selected = m_SelectedNodes.Count();
	if( num_selected == 1 )
	{
#ifdef LINK_TO_SELF_ENABLED
		// The last one will be considered the target node for the link
		if( theHold.Holding() == false ) 
		{
			theHold.Begin();
		}

		restore_obj = new LinkRestoreObj( this );
		theHold.Put( restore_obj );
		from = m_SelectedNodes[0];
		to = from;
		if( AlreadyLinked( from, to, circular_ref ) == false )
		{
			if( circular_ref )
			{
				if( CancelledLinkOp())
				{
					cancelled = true;		
					break;
				}
			}
			Link( from, to, restore_obj );
		}
		if( cancelled )
		{
			theHold.Cancel();
		}
		else
		{		
			theHold.Accept(_T( "One To Many Link" ));
		}
		gInterface->RedrawViews( gInterface->GetTime());	
#else
		return;
#endif
	}
	else
	{
		if( theHold.Holding() == false ) 
		{
			theHold.Begin();
		}

		restore_obj = new LinkRestoreObj( this );
		theHold.Put( restore_obj );

		from = m_SelectedNodes[0];
		for( i = 1; i < num_selected; i++ )
		{
			to = m_SelectedNodes[i];
			if( AlreadyLinked( from, to, circular_ref ) == false )
			{
				if( circular_ref )
				{
					if( CancelledLinkOp())
					{
						cancelled = true;		
						break;
					}
				}
				Link( from, to, restore_obj );
			} 
		}

		if( cancelled )
		{
			theHold.Cancel();
		}
		else
		{
			theHold.Accept(_T( "One To Many Link" ));
		}
		gInterface->RedrawViews( gInterface->GetTime());	
		
	}	

	m_SelectedNodes.Resize( 0 );
}

void	LinkMan::ReverseChainLink( bool closed )
{
	EnumerateSelectedNodes( true );
	ChainLink( closed, false );	
}

void	LinkMan::ChainLink( bool closed, bool enumerate )
{
	int num_selected;
	bool circular_ref, cancelled;
	INode *to, *from;
	int i;
	LinkRestoreObj *restore_obj;

	if( enumerate )
	{
		EnumerateSelectedNodes();
	}

	cancelled = false;
	num_selected = m_SelectedNodes.Count();
	if( num_selected == 1 )
	{
#ifdef LINK_TO_SELF_ENABLED
		if( closed )		
		{
			if( theHold.Holding() == false ) 
			{
				theHold.Begin();
			}

			restore_obj = new LinkRestoreObj( this );
			theHold.Put( restore_obj );

			// The last one will be considered the target node for the link
			from = m_SelectedNodes[0];

			to = from;
			if( AlreadyLinked( from, to, circular_ref ) == false )
			{
				if( circular_ref )
				{
					if( CancelledLinkOp())
					{
						cancelled = true;		
						break;
					}
				}
				Link( from, to, restore_obj );
			} 

			if( cancelled )
			{
				theHold.Cancel();
			}
			else
			{
				theHold.Accept(_T( "Chain Link" ));
			}
			gInterface->RedrawViews( gInterface->GetTime());
			
		}
#else		
		return;		
#endif
	}
	else
	{
		if( theHold.Holding() == false ) 
		{
			theHold.Begin();
		}

		restore_obj = new LinkRestoreObj( this );
		theHold.Put( restore_obj );

		for( i = 0; i < ( num_selected - 1 ); i++ )
		{
			from = m_SelectedNodes[i];
			to = m_SelectedNodes[i + 1];
			if( AlreadyLinked( from, to, circular_ref ) == false )
			{
				if( circular_ref )
				{
					if( CancelledLinkOp())
					{
						cancelled = true;		
						break;
					}
				}
				Link( from, to, restore_obj );
			} 
		}
		if( !cancelled && closed )
		{
			from = m_SelectedNodes[num_selected - 1];
			to = m_SelectedNodes[0];
			if( AlreadyLinked( from, to, circular_ref ) == false )
			{
				if( circular_ref )
				{
					if( CancelledLinkOp())
					{
						cancelled = true;								
					}
				}
				if( !cancelled )
				{
					Link( from, to, restore_obj );
				}
			} 
		}

		if( cancelled )
		{
			theHold.Cancel();
		}
		else
		{
			theHold.Accept(_T( "Chain Link" ));
		}
		gInterface->RedrawViews( gInterface->GetTime());
		
	}

	m_SelectedNodes.Resize( 0 );
}

void	LinkMan::RecalculateReferences( void )
{
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;
	int num_children = root->NumberOfChildren();
	int num_selected = gInterface->GetSelNodeCount();
	int i;
	
	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link_obj = (LinkObject*) obj;
			link_obj->ref_count = 0;			
		}
	}

	for( i = 0; i < num_selected; i++ )
	{
		IncrementSystemSelectOpID();
		node = gInterface->GetSelNode( i );		
		node->NotifyDependents( FOREVER, PART_ALL, REFMSG_SELECT_SURROUNDING_NODES );		
	}

}

void	LinkMan::Unlink( void )
{
	m_bLocked=true;

	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;
	int num_children = root->NumberOfChildren();
	LinkRestoreObj *restore_obj;
	int i;
	
	Tab<INode *> del_list;

	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link = (LinkObject*) obj;
			if( link->from_node && link->to_node )
			{
				if( link->from_node->Selected() && link->to_node->Selected())
				{
					del_list.Append( 1, &node );					
				}
			}
		}
	}	

	theHold.Begin();
	restore_obj = new LinkRestoreObj( this );;
	theHold.Put( restore_obj );
	for( i = 0; i < del_list.Count(); i++ )
	{
		gInterface->DeleteNode( del_list[i], FALSE );
		
	}
	theHold.Accept( _T("Unlink"));
	
	RecalculateReferences();

	gInterface->RedrawViews( gInterface->GetTime());

	m_bLocked=false;
}

void	LinkMan::OnPreSave( void )
{
	int i, num_children;
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;	
	Tab<INode *> del_list;
	LinkOptions options;
	
	num_children = root->NumberOfChildren();	
	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link = (LinkObject*) obj;			
			if(	( link->to_node == NULL ) ||
				( link->from_node == NULL ))
			{
				del_list.Append( 1, &node );
			}
		}		
	}

	for( i = 0; i < del_list.Count(); i++ )
	{
		gInterface->DeleteNode( del_list[i], FALSE );			
	}

	options.m_DisplayMode = m_DisplayMode;
	options.m_SelectChildren = m_SelectChildren;
	options.m_SelectParents = m_SelectParents;
	SetLinkOptions( &options );
}

void	LinkMan::CleanAbandonedLinks()
{
	int i, num_children;
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;	
	Tab<INode *> del_list;

	num_children = root->NumberOfChildren();	
	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link = (LinkObject*) obj;			
			link->link_node = node;
			
			if(	( link->to_node == NULL ) ||
				( link->from_node == NULL ))
			{
				del_list.Append( 1, &node );
			}
		}		
	}

	for( i = 0; i < del_list.Count(); i++ )
	{
		gInterface->DeleteNode( del_list[i], FALSE );			
	}
}

void	LinkMan::OnPostOpen( void )
{
	LinkOptions *options;
	
	CleanAbandonedLinks();

	if( options = GetLinkOptions())
	{
		m_SelectChildren = options->m_SelectChildren;
		m_SelectParents = options->m_SelectParents;
		m_DisplayMode = options->m_DisplayMode;

		if( m_SelectChildren || m_SelectParents )
		{
			if( m_DisplayMode == ILinkMan::DISPLAY_MODE_SYSTEM )
			{
				SetDisplayMode( ILinkMan::DISPLAY_MODE_ALL );
			}
		}
		else
		{
			SetDisplayMode( m_DisplayMode );
		}
		
		LinkUI::SetSelectChildren( m_SelectChildren );
		LinkUI::SetSelectParents( m_SelectParents );
	}
}

bool	LinkMan::IsMerging( void )
{
	return m_Merging;
}

void	LinkMan::OnPreMerge( void )
{
	m_Merging = true;
}

void	LinkMan::OnPostMerge( void )
{
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;
	int num_children = root->NumberOfChildren();
	int i;
	
	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link = (LinkObject*) obj;

			link->link_node = node;
			if( link->to_node == NULL )
			{
				char *node_name;
				
				node_name = GetLinkToObjectName( link->link_node );				
				if( node_name )
				{
					link->to_node = gInterface->GetINodeByName( node_name );
				}
			}

			if( link->from_node == NULL )
			{
				char *node_name;

				node_name = GetLinkFromObjectName( link->link_node );
				if( node_name )
				{
					link->from_node = gInterface->GetINodeByName( node_name );
				}
			}
		}
	}
	
	m_Merging = false;
}

void	LinkMan::IncrementSystemSelectOpID( void )
{
	m_SystemSelectOpID++;
}

int		LinkMan::GetSystemSelectID( void )
{
	return m_SystemSelectOpID;
}

void	LinkMan::SetDisplayMode( int mode )
{
	m_DisplayMode = mode;	
	if( m_DisplayMode == ILinkMan::DISPLAY_MODE_SYSTEM )
	{
		RecalculateReferences();
	}
	gInterface->ForceCompleteRedraw();
}

int		LinkMan::GetDisplayMode( void )
{
	return m_DisplayMode;
}

void	LinkMan::SetSelectChildren( bool select )
{
	m_SelectChildren = select;
	if( m_SelectChildren == false )
	{
		RecalculateReferences();
	}
}

bool	LinkMan::GetSelectChildren( void )
{
	return m_SelectChildren;
}

void	LinkMan::SetSelectParents( bool select )
{
	m_SelectParents = select;
	if( m_SelectParents == false )
	{
		RecalculateReferences();
	}
}

bool	LinkMan::GetSelectParents( void )
{
	return m_SelectParents;
}

class ChangeBool
{
public:
	bool	m_Changed[3];
};

class AutoUpdateObject : public ReferenceMaker
{
public:
	AutoUpdateObject( void );
	~AutoUpdateObject( void );

	RefResult NotifyRefChanged( Interval changeInt,RefTargetHandle hTarget, PartID& partID, RefMessage message );
	int NumRefs( void );
	RefTargetHandle GetReference( int i );
	void SetReference(int i, RefTargetHandle rtarg);

private:
	Tab< RefTargetHandle > refs;
	Tab< int > last_update_time;
	Tab< Point3 > original_position;
	Tab< Point3 > last_position;
	Tab< ChangeBool > element_changes;
	bool	in_drag_op;

};

AutoUpdateObject::AutoUpdateObject( void )
{
	in_drag_op = false;
}

AutoUpdateObject::~AutoUpdateObject( void )
{
	DeleteAllRefsFromMe();
}

int AutoUpdateObject::NumRefs( void )
{
	return refs.Count();
}

RefTargetHandle AutoUpdateObject::GetReference( int i )
{
	return refs[i];
}

void AutoUpdateObject::SetReference(int i, RefTargetHandle rtarg)
{
	int time;
	Point3 position;
	ChangeBool change = { false, false, false };

	time = 0;
	position.x = 0;
	position.y = 0;
	position.z = 0;
	refs.Insert( i, 1, &rtarg );	
	last_update_time.Insert( i, 1, &time );
	original_position.Insert( i, 1, &position );
	last_position.Insert( i, 1, &position );
	element_changes.Insert( i, 1, &change );
}


RefResult	AutoUpdateObject::NotifyRefChanged( Interval changeInt,RefTargetHandle hTarget, 
												PartID& partID, RefMessage message ) 
{ 
	int i, node_index;
	bool found;	
#ifdef OUTPUT_POSITIONAL_DATA
	char msg_str[1024];
#endif

	node_index = 0;
	found = false;
	for( i = 0; i < refs.Count(); i++ )
	{
		if( refs[i] == hTarget )
		{
			node_index = i;
			found = true;
			break;
		}
	}

	if( !found )
	{
		return REF_SUCCEED;
	}

	if( message == REFMSG_MOUSE_CYCLE_STARTED )
	{
		INode* node = (INode*) hTarget;
		
		Control* ctrl, *pos_ctrl;

		ctrl = node->GetTMController();
		pos_ctrl = ctrl->GetPositionController();
		pos_ctrl->GetValue( 0, &original_position[node_index], FOREVER );// GetSetMethod method=CTRL_ABSOLUTE)=0;				
		
#ifdef OUTPUT_POSITIONAL_DATA
		sprintf( msg_str, "CS: %f %f %f\n", original_position[node_index].x, original_position[node_index].y, original_position[node_index].z );
		OutputDebugString( msg_str );
#endif

		element_changes[node_index].m_Changed[0] = false;
		element_changes[node_index].m_Changed[1] = false;
		element_changes[node_index].m_Changed[2] = false;

		in_drag_op = true;
	}
	else if( message == REFMSG_MOUSE_CYCLE_COMPLETED )
	{
		INode* node = (INode*) hTarget;
		Point3 pos;
		Net::MsgDesc msgdesc;
		Script::CStruct* p_struct;
		uint32 buffer[1024];				
		uint32 length;
		Control* ctrl, *pos_ctrl;

		ctrl = node->GetTMController();
		pos_ctrl = ctrl->GetPositionController();
		pos_ctrl->GetValue( 0, &pos, FOREVER );

#ifdef OUTPUT_POSITIONAL_DATA
		sprintf( msg_str, "CM: %f %f %f         %f %f %f\n", pos.x, pos.y, pos.z, original_position[node_index].x, original_position[node_index].y, original_position[node_index].z );
		OutputDebugString( msg_str );
#endif

		last_position[node_index] = pos;
		buffer[0] = Script::GenerateCRC( "MoveNode" );
		p_struct = new Script::CStruct;
		p_struct->AddVector( "pos", pos.x, pos.z, -pos.y );
		p_struct->AddChecksum( "name", Script::GenerateCRC( node->GetName()));
		length = Script::WriteToBuffer( p_struct, (uint8*) &buffer[1], 1020 );
		length += sizeof( uint32 );

		msgdesc.m_Id     = Net::vMSG_ID_RUN_SCRIPT_COMMAND;
		msgdesc.m_Length = length;
		msgdesc.m_Data   = buffer;
		msgdesc.m_Singular = false;

		gClient->EnqueueMessageToServer( &msgdesc );
		//gClient->SendData();
		delete p_struct;

		in_drag_op = false;
	}
	else if( message == REFMSG_CHANGE )
	{
		INode* node = (INode*) hTarget;
		Net::MsgDesc msgdesc;
		Script::CStruct* p_struct;
		uint32 buffer[1024];				
		uint32 length;
		Point3 pos;
		static Point3 last_pos;
		Control* ctrl, *pos_ctrl;
		bool should_update;
		
		ctrl = node->GetTMController();
		pos_ctrl = ctrl->GetPositionController();
		pos_ctrl->GetValue( 0, &pos, FOREVER );


#ifdef OUTPUT_POSITIONAL_DATA		
		sprintf( msg_str, "pre %f %f %f         %f %f %f\n", pos.x, pos.y, pos.z, original_position[node_index].x, original_position[node_index].y, original_position[node_index].z );
		OutputDebugString( msg_str );
#endif
		if( pos.x != original_position[node_index].x )
		{
			element_changes[node_index].m_Changed[0] = true;
		}
		if( pos.y != original_position[node_index].y )
		{
			element_changes[node_index].m_Changed[1] = true;
		}
		if( pos.z != original_position[node_index].z )
		{
			element_changes[node_index].m_Changed[2] = true;
		}

		if( in_drag_op )
		{
			if( element_changes[node_index].m_Changed[0] )
			{
				if( pos.x == original_position[node_index].x )
				{
					return REF_SUCCEED;
				}
			}
			if( element_changes[node_index].m_Changed[1] )
			{
				if( pos.y == original_position[node_index].y )
				{
					return REF_SUCCEED;
				}
			}
			if( element_changes[node_index].m_Changed[2] )
			{
				if( pos.z == original_position[node_index].z )
				{
					return REF_SUCCEED;
				}
			}
		}

		should_update = false;
		if( !in_drag_op )
		{
			should_update = true;
		}
		else
		{
			if( ( pos != last_position[node_index] ) &&
				( pos != original_position[node_index] ))
			{
				should_update = true;
			}
		}

		if( should_update )
		{
#ifdef OUTPUT_POSITIONAL_DATA		
			sprintf( msg_str, "%f %f %f         %f %f %f\n", pos.x, pos.y, pos.z, original_position[node_index].x, original_position[node_index].y, original_position[node_index].z );
			OutputDebugString( msg_str );
#endif
						
			last_position[node_index] = pos;
			buffer[0] = Script::GenerateCRC( "MoveNode" );
			p_struct = new Script::CStruct;
			p_struct->AddVector( "pos", pos.x, pos.z, -pos.y );
			p_struct->AddChecksum( "name", Script::GenerateCRC( node->GetName()));
			length = Script::WriteToBuffer( p_struct, (uint8*) &buffer[1], 1020 );
			length += sizeof( uint32 );

			msgdesc.m_Id     = Net::vMSG_ID_RUN_SCRIPT_COMMAND;
			msgdesc.m_Length = length;
			msgdesc.m_Data   = buffer;
			msgdesc.m_Singular = false;

			gClient->EnqueueMessageToServer( &msgdesc );
			gClient->SendData();
			delete p_struct;
		}
		
	}
	else if( message == REFMSG_TARGET_DELETED )
	{
		refs.Delete( node_index, 1 );
		//refs[node_index] = NULL;
	}

	return REF_SUCCEED; 
}

void	LinkMan::EnableAutoUpdate( bool enable )
{
	static AutoUpdateObject* ref;

	m_AutoUpdateEnabled = enable;
	if( m_AutoUpdateEnabled )
	{
		if( ref == NULL )
		{
			int i, ref_no, num_children;
			INode* root, *node;

			root = gInterface->GetRootNode();
			num_children = root->NumberOfChildren();

			ref = new AutoUpdateObject;
			ref_no = 0;
			
			for( i = 0; i < num_children; i++ )
			{
				node = root->GetChildNode(i);

				Object* obj = node->EvalWorldState(gInterface->GetTime()).obj;
				if( obj && obj->ClassID() == vTRIGGER_CLASS_ID )
				{				
					ref->MakeRefByID( FOREVER, ref_no++, node );
				}
			}
		}
	}
	else
	{
		if( ref )
		{
			delete ref;
			ref = NULL;
		}
	}
}

bool	LinkMan::AutoUpdateEnabled( void )
{
	return m_AutoUpdateEnabled;
}

static LinkMan link_man;
ILinkMan* GetLinkMan( void )
{
	return &link_man;
}

bool	LinkMan::AlreadyLinked( INode *from, INode *to, bool &circular_ref )
{
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;
	int num_children = root->NumberOfChildren();
	int i;

	circular_ref = false;
	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link = (LinkObject*) obj;
			if(( link->to_node == from ) && ( link->from_node == to ))
			{
				circular_ref = true;
			}
			if(( link->from_node == from ) && ( link->to_node == to ))
			{
				return true;
			}
		}
	}

	return false;
}

void	LinkMan::Link( INode *from, INode *to, LinkRestoreObj *restore_obj )
{
	Matrix3 tm;
	Matrix3 link_tm( 1 );
	Point3 from_pos, to_pos, link_pos;
	LinkObject* link;
	Object *to_obj, *from_obj;
	
	to_obj = to->EvalWorldState(0).obj;
	from_obj = from->EvalWorldState(0).obj;

	if( ( to_obj->ClassID() == vLINK_OBJ_CLASS_ID ) ||
		( from_obj->ClassID() == vLINK_OBJ_CLASS_ID ))
	{
		return;
	}

	link = new LinkObject;
	
	link->link_node = gInterface->CreateObjectNode( link );

	// center the new link node between its target references
	tm = from->GetNodeTM( 0 );
	from_pos = tm.GetTrans();

	tm = to->GetNodeTM( 0 );
	to_pos = tm.GetTrans();

	link_pos = ( from_pos + to_pos ) / 2;
	link_tm.SetTrans( link_pos );
	link->link_node->SetNodeTM( 0, link_tm );

	link->from_node = from;
	link->to_node = to;

	gInterface->SelectNode( link->link_node, 0 );

	link->MakeRefByID( FOREVER, LinkObject::REF_ID_FROM, from );
	link->MakeRefByID( FOREVER, LinkObject::REF_ID_TO, to );
	
	RecalculateReferences();
	
	SetLinkFromObjectName( link->link_node, from->GetName());
	SetLinkToObjectName( link->link_node, to->GetName());
}

bool	LinkMan::CancelledLinkOp( void )
{
	return ( MessageBox( NULL, "This operation will create a cyclical reference. Proceed?", "Warning", MB_OKCANCEL ) == IDCANCEL );
}

// This will scan back through the node's reference data looking for the node
// at the start of the link chain
INode* FindLastLink(INode* node)
{
	RefList&     reflist = node->GetRefList();
	RefListItem* refItem = reflist.FirstItem();

	while(refItem)
	{
		if (refItem->maker && refItem->maker->ClassID() == vLINK_OBJ_CLASS_ID)
		{
			LinkObject* linkObj = (LinkObject*)refItem->maker;

			if (linkObj->from_node == node)
			{
				refItem = refItem->next;
				continue;
			}
				
			return linkObj->from_node;
		}

		refItem = refItem->next;
	}

	return NULL;
}

INode* FindNextLink(INode* node)
{
	RefList&     reflist = node->GetRefList();
	RefListItem* refItem = reflist.FirstItem();

	while(refItem)
	{
		if (refItem->maker && refItem->maker->ClassID() == vLINK_OBJ_CLASS_ID)
		{
			LinkObject* linkObj = (LinkObject*)refItem->maker;

			if (linkObj->to_node == node)
			{
				refItem = refItem->next;
				continue;
			}
				
			return linkObj->to_node;
		}

		refItem = refItem->next;
	}

	return NULL;
}

INode* FindStartLink(INode* node)
{
	INode* nodeStart = node;
	INode* lastNode  = NULL;

	while(nodeStart)
	{
		lastNode = nodeStart;
		nodeStart = FindLastLink(lastNode);
	}

	return lastNode;
}